{------------------------------------------------------------------------------}
{                                                                              }
{                               Yuriy Kopnin                                   }
{                                   LGPL                                       }
{                                                                              }
{------------------------------------------------------------------------------}
unit IBXSQLEdits;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, SQLEditor, PropEdits, IBCustomDataSet, IBDatabase;

type

  { TIBXSQLPropertyEditor }

  TIBXSQLPropertyEditor = class(TSQLPropertyEditor)
  protected
    DB: TIBDataBase;
    TR: TIBTransaction;
    TABLES: TIBDataSet;
    PROCS: TIBDataSet;
    PARAMS: TIBDataSet;
    RFields: TIBDataSet;
    RKeyFields: TIBDataSet;
    procedure LoadMetadata(MetaTypes: TMetadataTypes); override;
  protected
  public
    procedure LoadTables(WithView: Boolean);
    procedure LoadProcs;
    constructor Create(Hook: TPropertyEditorHook; APropCount: Integer);
       override;
    destructor Destroy; override;
  end;

  { TIBXProcSQLPropertyEditor }

  TIBXProcSQLPropertyEditor = class(TIBXSQLPropertyEditor)
  protected
    function DefaultMetaTypes: TMetadataTypes; override;
    constructor Create(Hook: TPropertyEditorHook; APropCount: Integer);
       override;
  end;

implementation

uses synhighlightersqlex, ComCtrls, Forms;

{ TIBXProcSQLPropertyEditor }

function TIBXProcSQLPropertyEditor.DefaultMetaTypes: TMetadataTypes;
begin
  Result := [mtStoredProc];
end;

constructor TIBXProcSQLPropertyEditor.Create(Hook: TPropertyEditorHook;
  APropCount: Integer);
begin
  inherited Create(Hook, APropCount);
  HideRadioSqlType := True;
  DefaultIndex:= 5;
end;

{ TIBXSQLPropertyEditor }

procedure TIBXSQLPropertyEditor.LoadMetadata(MetaTypes: TMetadataTypes);
var
  AComponent: TIBCustomDataSet;
begin
  ClearNameList;
  AComponent := TIBCustomDataSet(GetComponent(0));
  if AComponent.Database = nil then Exit;
  DB.Params.Assign(AComponent.Database.Params);
  DB.DatabaseName := AComponent.Database.DatabaseName;
  DB.LibraryName := AComponent.Database.LibraryName;;
  DB.SQLDialect := AComponent.Database.SQLDialect;
  DB.LoginPrompt := False;
  DB.Open;
  if not DB.Connected then Exit;
  try

    if mtTable in MetaTypes then
      LoadTables(mtView in MetaTypes);

    if mtStoredProc in MetaTypes then  LoadProcs;

  finally
    if DB.Connected then DB.Close;
  end;
end;

procedure TIBXSQLPropertyEditor.LoadTables(WithView: Boolean);
var
  TableNode: TTreeNode;
  Primary: Boolean;
  FieldName: string;
begin
  TABLES.SelectSQL.Strings[3] := '(RDB$VIEW_BLR IS NULL)';
  TABLES.Open;
  try
    while not TABLES.EOF do
    begin
      TableNode := AddTable(TABLES.Fields[0].AsString);
      RFields.ParamByName('rName').AsString := TableNode.Text;
      RKeyFields.ParamByName('rName').AsString := TableNode.Text;
      RFields.Open;
      RKeyFields.Open;
      try
        while not RFields.EOF do
        begin
          FieldName := RFields.Fields[0].AsString;
          Primary := RKeyFields.Locate('RDB$FIELD_NAME', FieldName, []);
          AddField(TableNode, FieldName, Primary);
          RFields.Next;
        end;
      finally
        RFields.Close;
        RKeyFields.Close;
      end;
      TABLES.Next;
    end;

    if WithView then
    begin
      TABLES.Close;
      TABLES.SelectSQL.Strings[3] := '(RDB$VIEW_BLR IS NOT NULL)';
      TABLES.Open;
      while not TABLES.EOF do
      begin
        TableNode := AddView(TABLES.Fields[0].AsString);
        RFields.ParamByName('rName').AsString := TableNode.Text;
        RFields.Open;
        try
          while not RFields.EOF do
          begin
            FieldName := RFields.Fields[0].AsString;
            AddField(TableNode, FieldName, False);
            RFields.Next;
          end;
        finally
          RFields.Close;
        end;
        TABLES.Next;
      end;
    end;

  finally
    TABLES.Close;
  end;
end;

procedure TIBXSQLPropertyEditor.LoadProcs;
var
  ProcNode: TTreeNode;
  InParam: Boolean;
begin
  PROCS.Open;
  try
	  while not PROCS.EOF do
	  begin
	    ProcNode := AddProcedure(PROCS.Fields[0].AsString);
	    PARAMS.ParamByName('ProcName').AsString := ProcNode.Text;
	    PARAMS.Open;
	    try
	      while not PARAMS.EOF do
	      begin
          InParam := PARAMS.Fields[1].AsInteger = 0;
          AddProcParam(ProcNode, PARAMS.Fields[0].AsString, InParam);
          PARAMS.Next;
	      end;
	    finally
	        PARAMS.Close;
	    end;
	    PROCS.Next;
	  end;
  finally
    if PROCS.Active then PROCS.Close;
  end;

end;

constructor TIBXSQLPropertyEditor.Create(Hook: TPropertyEditorHook;
  APropCount: Integer);
begin
  inherited Create(Hook, APropCount);
  UseReturningField := True;
  SqlDialect := sqlFirebird;
  DB := TIBDataBase.Create(Application);
  TR := TIBTransaction.Create(Application);
  TR.Params.Add('read_committed');
  TR.Params.Add('rec_version');
  TR.Params.Add('nowait');
  TR.DefaultDatabase := DB;
  DB.DefaultTransaction := TR;
  TABLES := TIBDataSet.Create(Application);
  TABLES.Database := DB;
  TABLES.SelectSQL.Append('select RDB$RELATION_NAME');
  TABLES.SelectSQL.Append('from RDB$RELATIONS');
  TABLES.SelectSQL.Append('where (RDB$SYSTEM_FLAG = 0) AND');
  TABLES.SelectSQL.Append('(RDB$VIEW_BLR is NULL)');
  TABLES.SelectSQL.Append('order by RDB$RELATION_NAME');

  PROCS := TIBDataSet.Create(Application);
  PROCS.Database := DB;
  PROCS.SelectSQL.Append('select rdb$procedure_name as PR_NAME');
  PROCS.SelectSQL.Append('from rdb$procedures');
  PROCS.SelectSQL.Append('order by rdb$procedure_name');


  PARAMS := TIBDataSet.Create(Application);
  PARAMS.Database := DB;
  PARAMS.SelectSQL.Append('select rdb$parameter_name,');
  PARAMS.SelectSQL.Append('rdb$parameter_type');
  PARAMS.SelectSQL.Append('from rdb$procedure_parameters');
  PARAMS.SelectSQL.Append('where rdb$procedure_name = :ProcName');
  PARAMS.SelectSQL.Append('order by rdb$parameter_type, rdb$parameter_number');

  RFields := TIBDataSet.Create(Application);
  RFields.Database := DB;
  RFields.SelectSQL.Append('SELECT RDB$FIELD_NAME FROM RDB$RELATION_FIELDS');
  RFields.SelectSQL.Append('WHERE RDB$RELATION_NAME = :rName');

  RKeyFields := TIBDataSet.Create(Application);
  RKeyFields.Database := DB;
  RKeyFields.SelectSQL.Append('SELECT RDB$FIELD_NAME FROM RDB$RELATION_CONSTRAINTS R LEFT JOIN RDB$INDEX_SEGMENTS I ON I.RDB$INDEX_NAME = R.RDB$INDEX_NAME');
  RKeyFields.SelectSQL.Append('WHERE R.RDB$RELATION_NAME = :RNAME AND R.RDB$CONSTRAINT_TYPE = ''PRIMARY KEY''');
end;

destructor TIBXSQLPropertyEditor.Destroy;
begin
  TABLES.Free;
  PROCS.Free;
  PARAMS.Free;
  RFields.Free;
  RKeyFields.Free;
  TR.Free;
  DB.Free;
  inherited Destroy;
end;

end.

