{------------------------------------------------------------------------------}
{                                                                              }
{                              Yuriy Kopnin                                    }
{                            Package VisuaTech                                 }
{                                 LGPL                                         }
{                                                                              }
{------------------------------------------------------------------------------}

unit XDBGrids;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, Math, FileUtil, DB, Types, Buttons,
  LCLStrConsts, LCLIntf, LCLProc, LCLType, LMessages, LResources,
  Controls, StdCtrls, Graphics, xGrids, Dialogs, Themes, variants,
  Clipbrd, ActnList, ExtCtrls, LazUtilities;

{$if FPC_FULLVERSION<20701}
  {$DEFINE noautomatedbookmark}
{$endif}

type
  TCustomXDBGrid = class;
  TxColumn = class;
  EInvalidGridOperation = class(Exception);

  TxGridStyle = (xgsNormal, xgsClassic);
  TCellCursorPos = (ccpTop, ccpBottom);


  TxDBGridOption = (
    xdgEditing,                          // Ya
    xdgTitles,                           // Ya
    xdgIndicator,                        // Ya
    xdgColumnResize,                     // Ya
    xdgColumnMove,                       // Ya
    xdgColLines,                         // Ya
    xdgRowLines,                         // Ya
    xdgTabs,                             // Ya
    xdgAlwaysShowEditor,                 // Ya
    xdgRowSelect,                        // Ya
    xdgAlwaysShowSelection,              // Ya
    xdgConfirmDelete,
    xdgCancelOnExit,                     // Ya
    xdgMultiselect,                      // Ya
    xdgHeaderHotTracking,
    xdgHeaderPushedLook,
    xdgPersistentMultiSelect,
    xdgAutoSizeColumns,
    xdgAnyButtonCanSelect,               // any mouse button can move selection
    xdgDisableDelete,                    // disable deleting records with Ctrl+Delete
    xdgDisableInsert,                    // disable inserting (or append) records
    xdgCellHints,                        // show individual cell hints
    xdgTruncCellHints,                   // show cell hints if cell text is too long
    xdgCellEllipsis,                      // show ... if cell text is truncated
    xdgAutoLocate,
    xdgAutoSortIcon,
    xdgRowHighLight
    );
  TxDbGridOptions = set of TxDBGridOption;

  TxDbGridExtraOption = (
    dgeAutoColumns,       // if uncustomized columns, add them anyway?
    dgeCheckboxColumn     // enable the use of checkbox in columns
    );
  TxDbGridExtraOptions = set of TxDbGridExtraOption;

  TxDbGridStatusItem = (gsUpdatingData, gsAddingAutoColumns,
    gsRemovingAutoColumns, gsAutoSized, gsStartEditing);
  TxDbGridStatus = set of TxDbGridStatusItem;

  TxDataSetScrolledEvent =
    procedure(DataSet: TDataSet; Distance: integer) of object;

  TxDBGridClickEvent =
    procedure(Column: TxColumn; CellPos: TCellCursorPos) of object;

  TxMovedEvent =
    procedure(Sender: TObject; FromIndex, ToIndex: integer) of object;

  TxDrawColumnCellEvent =
    procedure(Sender: TObject; const Rect: TRect; DataCol: integer;
    Column: TxColumn; State: TxGridDrawState) of object;

  TxGetDbEditMaskEvent =
    procedure(Sender: TObject; const Field: TField;
    var Value: string) of object;

  TxDbGridSelEditorEvent =
    procedure(Sender: TObject; Column: TxColumn;
    var Editor: TWinControl) of object;

  TxPrepareDbGridCanvasEvent =
    procedure(Sender: TObject; DataCol: integer; Column: TxColumn;
    AState: TxGridDrawState) of object;

  TxDbGridCheckBoxBitmapEvent =
    procedure(Sender: TObject; const CheckedState: TCheckboxState;
    var ABitmap: TBitmap) of object;

  TxDbGridCheckboxStateEvent =
    procedure(Sender: TObject; Column: TxColumn;
    var AState: TCheckboxState) of object;

  TxDbGridCellHintEvent =
    procedure(Sender: TObject; Column: TxColumn; var AText: string) of object;

type
  { TxBookmarkList }

  TxBookmarkList = class
  private
    FList: TFPList; // list of TBookmark
    FGrid: TCustomXDBGrid;
    FDataset: TDataset;
    FUseCompareBookmarks: boolean;
    FCanDoBinarySearch: boolean;
    function GetCount: integer;
    function GetCurrentRowSelected: boolean;
    function GetItem(AIndex: Integer): TBookmark;
    procedure SetCurrentRowSelected(const AValue: boolean);
    procedure CheckActive;
  public
    constructor Create(AGrid: TCustomXDBGrid);
    destructor Destroy; override;

    procedure Clear;
    procedure Delete;
    function  Find(const Item: TBookmark; var AIndex: Integer): boolean;
    function  IndexOf(const Item: TBookmark): Integer;
    function  Refresh: boolean;

    property Count: integer read GetCount;
    property CurrentRowSelected: boolean
      read GetCurrentRowSelected write SetCurrentRowSelected;
    property Items[AIndex: Integer]: TBookmark read GetItem; default;
  end;

  { TxComponentDataLink }

  TxComponentDataLink = class(TDatalink)
  private
    FDataSet: TDataSet;
    FDataSetName: string;
    FModified: boolean;
    FOnDatasetChanged: TDatasetNotifyEvent;
    fOnDataSetClose: TDataSetNotifyEvent;
    fOnDataSetOpen: TDataSetNotifyEvent;
    FOnDataSetScrolled: TxDataSetScrolledEvent;
    FOnEditingChanged: TDataSetNotifyEvent;
    fOnInvalidDataSet: TDataSetNotifyEvent;
    fOnInvalidDataSource: TDataSetNotifyEvent;
    FOnLayoutChanged: TDataSetNotifyEvent;
    fOnNewDataSet: TDataSetNotifyEvent;
    FOnRecordChanged: TFieldNotifyEvent;
    FOnUpdateData: TDataSetNotifyEvent;

    function GetDataSetName: string;
    function GetFields(Index: integer): TField;
    procedure SetDataSetName(const AValue: string);
  protected
    procedure RecordChanged(Field: TField); override;
    procedure DataSetChanged; override;
    procedure ActiveChanged; override;
    procedure LayoutChanged; override;
    procedure DataSetScrolled(Distance: integer); override;
    procedure FocusControl(Field: TFieldRef); override;
    // Testing Events
    procedure CheckBrowseMode; override;
    procedure EditingChanged; override;
    procedure UpdateData; override;
    function MoveBy(Distance: integer): integer; override;
    property Modified: boolean read FModified write FModified;
  public
    property OnRecordChanged: TFieldNotifyEvent
      read FOnRecordChanged write FOnRecordChanged;
    property OnDataSetChanged: TDatasetNotifyEvent
      read FOnDatasetChanged write FOnDataSetChanged;
    property OnNewDataSet: TDataSetNotifyEvent read fOnNewDataSet write fOnNewDataSet;
    property OnDataSetOpen: TDataSetNotifyEvent read fOnDataSetOpen write fOnDataSetOpen;
    property OnInvalidDataSet: TDataSetNotifyEvent
      read fOnInvalidDataSet write fOnInvalidDataSet;
    property OnInvalidDataSource: TDataSetNotifyEvent
      read fOnInvalidDataSource write fOnInvalidDataSource;
    property OnLayoutChanged: TDataSetNotifyEvent
      read FOnLayoutChanged write FOnLayoutChanged;
    property OnDataSetClose: TDataSetNotifyEvent
      read fOnDataSetClose write fOnDataSetClose;
    property OnDataSetScrolled: TxDataSetScrolledEvent
      read FOnDataSetScrolled write FOnDataSetScrolled;
    property OnEditingChanged: TDataSetNotifyEvent
      read FOnEditingChanged write FOnEditingChanged;
    property OnUpdateData: TDataSetNotifyEvent read FOnUpdateData write FOnUpdateData;
    property DataSetName: string read GetDataSetName write SetDataSetName;
    property Fields[Index: integer]: TField read GetFields;
    property VisualControl;
  end;

  { TxColumn }

  { TxColumnTitle }

  TxColumnTitle = class(TxGridColumnTitle)
  private

  protected
    function GetDefaultCaption: string; override;
    function GetxDefaultCaption: string; override;

  published

  end;

  TxSortOrder = (soxNone, soxAscending, soxDescending);
  TImagesIndexes = class(TStringList);

  { TxColumn }

  TxColumn = class(TxGridColumn)
  private
    FDisplayFormat: string;
    FDisplayFormatChanged: boolean;
    FFieldName: string;
    FField: TField;
    FIsAutomaticColumn: boolean;
    FDesignIndex: integer;

    //************
    FDrawImage: boolean;
    FxDrawImage: boolean;
    FxDisplayFormat: string;
    FxDisplayFormatChanged: boolean;
    FxFieldName: string;
    FxField: TField;
    FxAlignment: ^TAlignment;
    FWordWrap: boolean;
    FxWordWrap: boolean;
    FPrcWidth: integer;
    FSorted: TxSortOrder;
    FxSorted: TxSortOrder;
    FImagesIndexes: TImagesIndexes;
    FxImagesIndexes: TImagesIndexes;
    procedure SetImagesIndexes(AValue: TImagesIndexes);
    procedure SetPrcWidth(AValue: integer);
    procedure SetSorted(AValue: TxSortOrder);
    procedure SetWordWrap(AValue: boolean);
    procedure SetxImagesIndexes(AValue: TImagesIndexes);
    procedure SetxSorted(AValue: TxSortOrder);
    procedure SetxWordWrap(AValue: boolean);
    procedure xApplyDisplayFormat;
    function GetxDisplayFormat: string;
    function GetxField: TField;
    procedure SetxDisplayFormat(const AValue: string);
    procedure SetxField(const AValue: TField);
    procedure SetxFieldName(const AValue: string);
    function IsxDisplayFormatStored: boolean;
    function GetxAlignment: TAlignment;
    procedure SetxAlignment(const AValue: TAlignment);
    function IsxAlignmentStored: boolean;
    //************

    procedure ApplyDisplayFormat;
    function GetDataSet: TDataSet;
    function GetDisplayFormat: string;
    function GetField: TField;
    function GetIsDesignColumn: boolean;
    function IsDisplayFormatStored: boolean;
    procedure SetDisplayFormat(const AValue: string);
    procedure SetField(const AValue: TField);
    procedure SetFieldName(const AValue: string);
  protected
    function CreateTitle: TxGridColumnTitle; override;
    function GetDefaultAlignment: TAlignment; override;
    function GetDefaultDisplayFormat: string;
    function GetDefaultValueChecked: string; override;
    function GetDefaultValueUnchecked: string; override;
    function GetDefaultVisible: boolean; override;
    function GetDisplayName: string; override;
    function GetDefaultReadOnly: boolean; override;
    function GetDefaultWidth: integer; override;
    function GetPickList: TStrings; override;
    property IsAutomaticColumn: boolean read FIsAutomaticColumn;
    property IsDesignColumn: boolean read GetIsDesignColumn;
    procedure LinkField;

    //************
    function GetxDefaultDisplayFormat: string;
    procedure xLinkField;
    function GetxColumnTitle: TxColumnTitle;
    function GetxDefaultAlignment: TAlignment; virtual;
    procedure SetDrawImage(AValue: boolean);
    procedure SetxDrawImage(AValue: boolean);
    //************

  public
    constructor Create(ACollection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    function IsDefault: boolean; override;
    property DesignIndex: integer read FDesignIndex;
    property Field: TField read GetField write SetField;

    //************
    property xField: TField read GetxField write SetxField;
    property Sorted: TxSortOrder read FSorted write SetSorted;
    property xSorted: TxSortOrder read FxSorted write SetxSorted;
    //************

  published
    property FieldName: string read FFieldName write SetFieldName;
    property DisplayFormat: string read GetDisplayFormat
      write SetDisplayFormat stored IsDisplayFormatStored;

    //**************
    property xFieldName: string read FxFieldName write SetxFieldName;
    property xDisplayFormat: string read GetxDisplayFormat
      write SetxDisplayFormat stored IsxDisplayFormatStored;
    //property ColumnTitle: TxColumnTitle read GetxColumnTitle;
    property xAlignment: TAlignment
      read GetxAlignment write SetxAlignment stored IsxAlignmentStored;
    property DrawImage: boolean read FDrawImage write SetDrawImage;
    property xDrawImage: boolean read FxDrawImage write SetxDrawImage;
    property WordWrap: boolean read FWordWrap write SetWordWrap;
    property xWordWrap: boolean read FxWordWrap write SetxWordWrap;
    property PrcWidth: integer read FPrcWidth write SetPrcWidth;
    property ImagesIndexes: TImagesIndexes read FImagesIndexes write SetImagesIndexes;
    property xImagesIndexes: TImagesIndexes read FxImagesIndexes write SetxImagesIndexes;
    //***************

  end;

  TxColumnOrder = (coDesignOrder, coFieldIndexOrder);

  { TxDBGridColumns }
  TxDBGridColumns = class(TxGridColumns)
  private
    function GetColumn(Index: integer): TxColumn;
    procedure SetColumn(Index: integer; Value: TxColumn);
  protected
    procedure Update(Item: TCollectionItem); override;
    function ColumnFromField(Field: TField): TxColumn;
    function HasAutomaticColumns: boolean;
    function HasDesignColumns: boolean;
    procedure RemoveAutoColumns;
  public
    procedure ResetSorted;
    function Add: TxColumn;
    procedure LinkFields;
    procedure ResetColumnsOrder(ColumnOrder: TxColumnOrder);
    property Items[Index: integer]: TxColumn read GetColumn write SetColumn; default;
  end;

  {TRowsHighLightItem}
  TRowsHighLightItem = class(TCollectionItem)
  private
    FFieldName: string;
    FFieldValue: string;
    FColor: TColor;
    FFontColor: TColor;
    FCellOnly: Boolean;
  protected
  public
    constructor Create(ACollection: TCollection); override;
  published
    procedure Assign(Source: TPersistent); override;
    property FieldName: string read FFieldName write FFieldName;
    property FieldValue: string read FFieldValue write FFieldValue;
    property Color: TColor read FColor write FColor;
    property FontColor: TColor read FFontColor write FFontColor;
    property CellOnly: Boolean read FCellOnly write FCellOnly default False;
  end;

  { TRowsHightLightItems }

  TRowsHightLightItems = class(TCollection)
  private
    FGrid: TCustomXGrid;
    function GetRowHLItem(Index: Integer): TRowsHighLightItem;
    procedure SetRowHLItem(Index: Integer; AValue: TRowsHighLightItem);
  protected
    procedure Update(Item: TCollectionItem); override;
  public
    constructor Create(AGrid: TCustomXGrid; aItemClass: TCollectionItemClass);
    property Items[Index: Integer]: TRowsHighLightItem read GetRowHLItem write SetRowHLItem; default;
  end;

  { TCustomXDBGrid }

  TAutoSortDataSet = procedure (DataSet: TDataSet; Column: TxColumn; CellCursorPos: TCellCursorPos);
  TGetDataSortField = procedure (DataSet: TDataSet; var SortedFieldName: string; var Desc: Boolean);

  TOnGetImageIndexFromField = function(F: TField; ColIndex: integer): integer of object;
  TOnEditButtonClick =
    procedure(Sender: TObject; Column: TxColumn; CCPos: TCellCursorPos) of object;

  TCustomXDBGrid = class(TCustomXGrid)
  private
    FDataLink: TxComponentDataLink;
    FExtraOptions: TxDbGridExtraOptions;
    FOnCellClick: TxDBGridClickEvent;
    FOnColEnter, FOnColExit: TNotifyEvent;
    FOnColumnMoved: TxMovedEvent;
    FOnColumnSized: TNotifyEvent;
    FOnDrawColumnCell: TxDrawColumnCellEvent;
    FOnFieldEditMask: TxGetDbEditMaskEvent;
    FOnTitleClick: TxDBGridClickEvent;
    FOnSelectEditor: TxDbGridSelEditorEvent;
    FOnCheckboxBitmap: TxDbGridCheckBoxBitmapEvent;
    FOnCheckboxState: TxDbGridCheckboxStateEvent;
    FOptions: TxDbGridOptions;
    FReadOnly: boolean;
    FColEnterPending: boolean;
    FLayoutChangedCount: integer;
    FDblClicked: Boolean;

    FDrawingActiveRecord: boolean;
    FDrawingMultiSelRecord: boolean;
    FDrawingEmptyDataset: boolean;

    FOldPosition: integer;
    FDefaultColWidths: boolean;
    FGridStatus: TxDbGridStatus;
    FOldControlStyle: TControlStyle;
    FSelectedRows: TxBookmarkList;
    FOnPrepareCanvas: TxPrepareDbGridCanvasEvent;
    FKeyBookmark: TBookmark;
    FKeySign: integer;
    FSavedRecord: integer;
    FOnGetCellHint: TxDbGridCellHintEvent;
    FOnChangeLocateStr: TNotifyEvent;

    //***************
    FxStyle: TxGridStyle;
    FxLineColor: TColor;
    FxSelectedFontColor: TColor;
    FDoubleRowHeight: boolean;
    FCellCursorPos: TCellCursorPos;
    FOnImageIndexFromField: TOnGetImageIndexFromField;
    FIncRowHeight: integer;
    FMidleLineColor: TColor;
    FDrawMidleLine: boolean;
    FOnEdButtonClick: TOnEditButtonClick;
    FThumbTrack: boolean;
    FInsertActionLink: TActionLink;
    FInsertAction: TBasicAction;
    FEditActionLink: TActionLink;
    FEditAction: TBasicAction;
    FSelectActionLink: TActionLink;
    FSelectAction: TBasicAction;
    FDeleteActionLink: TActionLink;
    FDeleteAction: TBasicAction;
    FAutoSort: Boolean;
    FLocateStr: string;
    FLocateInfoPanel: TPanel;
    FUseAutoFilter: Boolean;
    FRowsHighLights: TRowsHightLightItems;

    function GetDeleteAction: TBasicAction;
    function GetDoubleRowHeight: boolean;
    function GetEditAction: TBasicAction;
    function GetInsertAction: TBasicAction;
    function GetRowHighLights: TRowsHightLightItems;
    function GetSelectAction: TBasicAction;
    function GetSortedField: string;
    procedure SetDeleteAction(AValue: TBasicAction);
    procedure SetDrawMidleLine(AValue: boolean);
    procedure SetEditAction(AValue: TBasicAction);
    procedure SetIncRowHight(AValue: integer);
    procedure SetInsertAction(AValue: TBasicAction);
    procedure SetLocateInfoPanel(AValue: TPanel);
    procedure SetLocateStr(AValue: string);
    procedure SetMidleLineColor(AValue: TColor);
    procedure SetRowHighLights(AValue: TRowsHightLightItems);
    procedure SetSelectAction(AValue: TBasicAction);
    procedure xSwapCheckBox;
    //***************

    procedure EmptyGrid;
    function GetColumns: TxDBGridColumns;
    function GetCurrentColumn: TxColumn;
    function GetCurrentField: TField;
    function GetDataSource: TDataSource;
    function GetRecordCount: integer;
    function GetSelectedFieldRect: TRect;
    function GetSelectedIndex: integer;
    function GetThumbTracking: boolean;
    procedure OnRecordChanged(Field: TField);
    procedure OnDataSetChanged(aDataSet: TDataSet);
    procedure OnDataSetOpen(aDataSet: TDataSet);
    procedure OnDataSetClose(aDataSet: TDataSet);
    procedure OnEditingChanged(aDataSet: TDataSet);
    procedure OnInvalidDataSet(aDataSet: TDataSet);
    procedure OnInvalidDataSource(aDataSet: TDataset);
    procedure OnLayoutChanged(aDataSet: TDataSet);
    procedure OnNewDataSet(aDataSet: TDataset);
    procedure OnDataSetScrolled(aDataSet: TDataSet; Distance: integer);
    procedure OnUpdateData(aDataSet: TDataSet);
    procedure SetColumns(const AValue: TxDBGridColumns);
    //procedure ReadColumns(Reader: TReader);
    //procedure SetColumns(const AValue: TxDBGridColumns);
    procedure SetCurrentField(const AValue: TField);
    procedure SetDataSource(const AValue: TDataSource);
    procedure SetExtraOptions(const AValue: TxDbGridExtraOptions);
    procedure SetOptions(const AValue: TxDbGridOptions);
    procedure SetSelectedIndex(const AValue: integer);
    procedure SetThumbTracking(const AValue: boolean);
    procedure UpdateBufferCount;

    // Temporal
    function GetColumnCount: integer;

    function DefaultFieldColWidth(F: TField): integer;

    procedure UpdateGridColumnSizes;
    procedure UpdateScrollbarRange;
    procedure DoLayoutChanged;
    //procedure WriteColumns(Writer: TWriter);

    procedure RestoreEditor;
    function ISEOF: boolean;
    function ValidDataSet: boolean;
    function InsertCancelable: boolean;

    procedure SwapCheckBox;
    procedure ToggleSelectedRow;
    procedure SelectRecord(AValue: boolean);
    procedure GetScrollbarParams(out aRange, aPage, aPos: integer);
    procedure CMGetDataLink(var Message: TLMessage); message CM_GETDATALINK;
    procedure ClearSelection(selCurrent: boolean = False);
    function NeedAutoSizeColumns: boolean;
    procedure RenewColWidths;
  protected
    FEditingColumn: integer;
    FTempText: string;
    FiltterField: string;
    FilterValue: string;
    FilterCaseInsensitive: Boolean;
    FilterContaining: Boolean;
    SaveOldFilterEvent: TFilterRecordEvent;
    function RowHighLightUse: Boolean; override;
    procedure StartUpdating;
    procedure EndUpdating;
    function UpdatingData: boolean;
    procedure AddAutomaticColumns;
    procedure AssignTo(Dest: TPersistent); override;
    procedure BeforeMoveSelection(const DCol, DRow: integer); override;
    procedure BeginLayout;
    procedure CellClick(const aCol, aRow: integer; const Button: TMouseButton;
      P: TPoint); override;
    procedure InvalidateSizes;
    procedure ColRowMoved(IsColumn: boolean; FromIndex, ToIndex: integer); override;
    function ColumnEditorStyle(aCol: integer; F: TField): TxColumnButtonStyle;
    function CreateColumns: TxGridColumns; override;
    procedure CreateWnd; override;
    procedure DblClick; override;
    procedure DefineProperties(Filer: TFiler); override;
    procedure DefaultDrawCell(aCol, aRow: integer; aRect: TRect; aState: TxGridDrawState);
    function DefaultEditorStyle(const Style: TxColumnButtonStyle;
      const F: TField): TxColumnButtonStyle;
    procedure DoEditorShow; override;
    procedure DoExit; override;
    function DoMouseWheelDown(Shift: TShiftState; MousePos: TPoint): boolean; override;
    function DoMouseWheelUp(Shift: TShiftState; MousePos: TPoint): boolean; override;
    procedure DoOnChangeBounds; override;
    procedure DoPrepareCanvas(aCol, aRow: integer; aState: TxGridDrawState); override;
    procedure DrawAllRows; override;
    procedure DrawFocusRect(aCol, aRow: integer; ARect: TRect); override;
    procedure DrawRow(ARow: integer); override;
    procedure DrawCell(aCol, aRow: integer; aRect: TRect; aState: TxGridDrawState);
      override;
    procedure DrawCheckboxBitmaps(aCol: integer; aRect: TRect; F: TField);
    procedure DrawFixedText(aCol, aRow: integer; aRect: TRect; aState: TxGridDrawState);
    procedure DrawColumnText(aCol, aRow: integer; aRect: TRect;
      aState: TxGridDrawState); override;
    procedure DrawIndicator(ACanvas: TCanvas; R: TRect; Opt: TDataSetState;
      MultiSel: boolean); virtual;
    procedure EditingColumn(aCol: integer; Ok: boolean);
    procedure EditorCancelEditing;
    procedure EditorDoGetValue; override;
    function EditorCanAcceptKey(const ch: TUTF8Char): boolean; override;
    function EditorIsReadOnly: boolean; override;
    procedure EditorTextChanged(const aCol, aRow: integer; const aText: string); override;
    procedure EndLayout;
    function FieldIndexFromGridColumn(AGridCol: integer): integer;
    function FirstGridColumn: integer; override;
    function GetBufferCount: integer;
    function GetCellHintText(aCol, aRow: integer): string; override;
    function GetDefaultColumnAlignment(Column: integer): TAlignment; override;
    function GetDefaultColumnWidth(Column: integer): integer; override;
    function GetDefaultColumnReadOnly(Column: integer): boolean; override;
    function GetDefaultColumnTitle(Column: integer): string; override;
    function GetDefaultRowHeight: integer; override;
    function GetDsFieldFromGridColumn(Column: integer): TField;
    function GetEditMask(aCol, aRow: longint): string; override;
    function GetEditText(aCol, aRow: longint): string; override;
    function GetFieldFromGridColumn(Column: integer): TField;
    function GetGridColumnFromField(F: TField): integer;
    function GetImageForCheckBox(const aCol, aRow: integer;
      CheckBoxView: TCheckBoxState): TBitmap; override;
    function GetIsCellSelected(aCol, aRow: integer): boolean; override;
    function GetIsCellTitle(aCol, aRow: integer): boolean; override;
    procedure GetSelectedState(AState: TxGridDrawState; out IsSelected: boolean);
      override;
    function GetTruncCellHintText(aCol, aRow: integer): string; override;
    function GridCanModify: boolean;
    function GridCanDelete: boolean;
    procedure GetSBVisibility(out HsbVisible, VsbVisible: boolean); override;
    procedure GetSBRanges(const HsbVisible, VsbVisible: boolean;
      out HsbRange, VsbRange, HsbPage, VsbPage, HsbPos, VsbPos: integer); override;
    procedure HeaderClick(IsColumn: boolean; index: integer; P: TPoint); override;
    procedure HeaderSized(IsColumn: boolean; Index: integer); override;
    function IsValidChar(AField: TField; AChar: TUTF8Char): boolean;
    procedure KeyDown(var Key: word; Shift: TShiftState); override;
    procedure LinkActive(Value: boolean); virtual;
    procedure LayoutChanged; virtual;
    procedure Loaded; override;
    procedure MoveSelection; override;
    function MouseButtonAllowed(Button: TMouseButton): boolean; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
      override;
    procedure DoRowHighLight(aCol, aRow: Integer; aState: TxGridDrawState);
  override;
    procedure PrepareCanvas(aCol, aRow: integer; aState: TxGridDrawState); override;
    procedure PrepareCellHints(aCol, aRow: integer); override;
    procedure RemoveAutomaticColumns;
    procedure ResetSizes; override;
    procedure SelectEditor; override;
    procedure SetEditText(ACol, ARow: longint; const Value: string); override;
    procedure SetFixedCols(const AValue: integer); override;
    function SelectCell(aCol, aRow: integer): boolean; override;
    procedure UnprepareCellHints; override;
    procedure UpdateActive; virtual;
    procedure UpdateAutoSizeColumns;
    procedure UpdateData; virtual;
    function UpdateGridCounts: integer;
    procedure WMVScroll(var Message: TLMVScroll); message LM_VScroll;
    procedure WndProc(var TheMessage: TLMessage); override;
    procedure DoAutoLocate;
    procedure UTF8KeyPress(var UTF8Key: TUTF8Char); override;
    //procedure KeyPress(var Key: char); override;
    //*****************
    procedure SetDoubleRowHeight(AValue: boolean);
    procedure SetXStyle(Value: TxGridStyle);
    procedure xDrawCheckboxBitmaps(aCol, aRow: integer; aRect: TRect;
      F: TField; ChckAlignment: TAlignment);
    procedure xDrawGridCheckboxBitmaps(const aCol, aRow: integer;
      const aRect: TRect; const aState: TCheckboxState;
      bmpAlign: TAlignment);
    procedure DrawCellGrid(aCol, aRow: integer; aRect: TRect;
      aState: TxGridDrawState); override;
    procedure SetxLineColor(AValue: TColor);
    function GetxFieldFromGridColumn(ACol: integer): TField;
    function IsxFieldExistFromGridColumn(ACol: integer): boolean;
    procedure DrawCellText(aCol, aRow: integer; aRect: TRect;
      aState: TxGridDrawState; aText: string); override;
    procedure xDrawCellText(aCol, aRow: integer; aRect: TRect;
      aState: TxGridDrawState; aText: string; WordWrap: boolean);
    procedure DrawCellImage(ARect: TRect; AColumn: TxColumn; F: TField);
    procedure xDrawCellImage(ARect: TRect; AColumn: TxColumn; F: TField);
    procedure DoEditButtonClick(const ACol, ARow: integer); override;
    function IsExistXFields: boolean;
    procedure FontChanged(Sender: TObject); override;
    procedure IncFontValue(AValue: integer);
    procedure Notification(AComponent: TComponent; Operation: TOperation);
      override;
    procedure SortDataSet(Column: TxColumn; CellCursorPos: TCellCursorPos);
    procedure OnAutoFilter(DataSet: TDataSet; var Accept: Boolean);
    //*****************

    property Columns: TxDBGridColumns read GetColumns write SetColumns;
    property GridStatus: TxDbGridStatus read FGridStatus write FGridStatus;
    property Datalink: TxComponentDataLink read FDatalink;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property Options: TxDbGridOptions read FOptions write SetOptions default
      [xdgColumnResize, xdgColumnMove, xdgTitles, xdgIndicator, xdgRowLines,
      xdgColLines, xdgConfirmDelete, xdgCancelOnExit, xdgTabs, xdgEditing,
      xdgAlwaysShowSelection];
    property OptionsExtra: TxDbGridExtraOptions
      read FExtraOptions write SetExtraOptions default [dgeAutoColumns, dgeCheckboxColumn];
    property ReadOnly: boolean read FReadOnly write FReadOnly default False;
    property SelectedRows: TxBookmarkList read FSelectedRows;

    property OnCellClick: TxDBGridClickEvent read FOnCellClick write FOnCellClick;
    property OnColEnter: TNotifyEvent read FOnColEnter write FOnColEnter;
    property OnColExit: TNotifyEvent read FOnColExit write FOnColExit;
    property OnColumnMoved: TxMovedEvent read FOnColumnMoved write FOnColumnMoved;
    property OnColumnSized: TNotifyEvent read FOnColumnSized write FOnColumnSized;
    property OnDrawColumnCell: TxDrawColumnCellEvent
      read FOnDrawColumnCell write FOnDrawColumnCell;
    property OnFieldEditMask: TxGetDbEditMaskEvent
      read FOnFieldEditMask write FOnFieldEditMask;
    property OnGetCellHint: TxDbGridCellHintEvent
      read FOnGetCellHint write FOnGetCellHint;
    property OnPrepareCanvas: TxPrepareDbGridCanvasEvent
      read FOnPrepareCanvas write FOnPrepareCanvas;
    property OnSelectEditor: TxDbGridSelEditorEvent
      read FOnSelectEditor write FOnSelectEditor;
    property OnTitleClick: TxDBGridClickEvent read FOnTitleClick write FOnTitleClick;
    property OnUserCheckboxBitmap: TxDbGridCheckBoxBitmapEvent
      read FOnCheckboxBitmap write FOnCheckboxBitmap;
    property OnUserCheckboxState: TxDbGridCheckboxStateEvent
      read FOnCheckboxState write FOnCheckboxState;
    property InsertActionLink: TActionLink read FInsertActionLink write FInsertActionLink;
    property EditActionLink: TActionLink read FEditActionLink write FEditActionLink;
    property SelectActionLink: TActionLink read FSelectActionLink write FSelectActionLink;
    property DeleteActionLink: TActionLink read FDeleteActionLink write FDeleteActionLink;
  public
    constructor Create(AOwner: TComponent); override;
    procedure ShowFilterForm(F: TField; ACaption, InputStr: string);
    procedure ActivateFilter(FieldName, AValue: string; IsCaseInsensitive: Boolean = False; AContainingValue: Boolean = True);
    procedure ResetFilter;

    procedure AutoSizeColumns;
    procedure CheckSorted;
    procedure InitiateAction; override;
    procedure DefaultDrawColumnCell(const Rect: TRect; DataCol: integer;
      Column: TxColumn; State: TxGridDrawState);
    function EditorByStyle(Style: TxColumnButtonStyle): TWinControl; override;
    procedure ResetColWidths;
    function IndicatorWidth: integer;
    destructor Destroy; override;

    //*******************
    function EditorRect(aCol, aRow: integer): TRect; override;
    function SelectedEditorRect: TRect;
    procedure CheckSortingImage;
    property AutoSort: Boolean read FAutoSort write FAutoSort;
    property CellCursorPos: TCellCursorPos read FCellCursorPos;
    property DoubleRowHeight: boolean read GetDoubleRowHeight write SetDoubleRowHeight;
    property xStyle: TxGridStyle read FxStyle write SetXStyle;
    property xLineColor: TColor read FxLineColor write SetxLineColor;
    property OnImageIndexFromField: TOnGetImageIndexFromField
      read FOnImageIndexFromField write FOnImageIndexFromField;
    property IncRowHight: integer read FIncRowHeight write SetIncRowHight;
    property MidleLineColor: TColor read FMidleLineColor write SetMidleLineColor;
    property DrawMidleLine: boolean read FDrawMidleLine write SetDrawMidleLine;
    property OnEditButtonClick: TOnEditButtonClick
      read FOnEdButtonClick write FOnEdButtonClick;
    property ThumbTrack: boolean read FThumbTrack write FThumbTrack;
    //*******************

    property SelectedField: TField read GetCurrentField write SetCurrentField;
    property SortedField: string read GetSortedField;
    property SelectedIndex: integer read GetSelectedIndex write SetSelectedIndex;
    property SelectedColumn: TxColumn read GetCurrentColumn;
    property SelectedFieldRect: TRect read GetSelectedFieldRect;
    property ThumbTracking: boolean read GetThumbTracking write SetThumbTracking;
    property InsertAction: TBasicAction read GetInsertAction write SetInsertAction;
    property EditAction: TBasicAction read GetEditAction write SetEditAction;
    property SelectAction: TBasicAction read GetSelectAction write SetSelectAction;
    property DeleteAction: TBasicAction read GetDeleteAction write SetDeleteAction;
    property LocateStr: string read FLocateStr write SetLocateStr;
    property OnChangeLocateStr: TNotifyEvent read FOnChangeLocateStr write FOnChangeLocateStr;
    property LocateInfoPanel: TPanel read FLocateInfoPanel write SetLocateInfoPanel;
    property UserAutoFilter: Boolean read FUseAutoFilter write FUseAutoFilter default True;
    property RowsHighLights: TRowsHightLightItems read GetRowHighLights write SetRowHighLights stored RowHighLightUse;
  end;

  TxDBGrid = class(TCustomXDBGrid)
  public
    property BorderColor;
    property Canvas;
    property DefaultTextStyle;
    property EditorBorderStyle;
    property EditorMode;
    property ExtendedColSizing;
    property FastEditing;
    property FocusColor;
    property FocusRectVisible;
    property GridLineColor;
    property GridLineStyle;
    property SelectedRows;
  published
    property AutoSort;
    property Align;
    property AlternateColor;
    property Anchors;
    property AutoAdvance default aaRightDown;
    property AutoEdit;
    property AutoFillColumns;
    property BiDiMode;
    property BorderSpacing;
    property BorderStyle;
    property CellHintPriority;
    property Color;
    property Columns; // stored false;
    property Constraints;
    property DataSource;
    property DefaultDrawing;
    property ImageList;
    property TitleImageList;
    property DragCursor;
    property DragMode;
    property Enabled;
    property FixedColor;
    property FixedCols;
    property FixedHotColor;
    property Flat;
    property Font;
    property HeaderHotZones;
    property HeaderPushZones;
    property LocateInfoPanel;
    //property ImeMode;
    //property ImeName;
    property Options;
    property OptionsExtra;
    property ParentBiDiMode;
    property ParentColor default False;
    property ParentFont;
    //property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property Scrollbars default ssBoth;
    property SelectedColor;
    property SelectedUnfocusedColor;
    property RowHighLightColor;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property TitleFont;
    property TitleStyle;
    property ThumbTrack;
    property UseXORFeatures;
    property Visible;
    property UserAutoFilter;
    property OnCellClick;
    property OnChangeLocateStr;
    property OnColEnter;
    property OnColExit;
    property OnColumnMoved;
    property OnColumnSized;
    property OnContextPopup;
    property OnDrawColumnCell;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEditButtonClick;
    property OnEditingDone;
    //property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnFieldEditMask;
    property OnGetCellHint;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseEnter;
    property OnMouseLeave;
    property OnMouseMove;
    property OnMouseUp;
    property OnPrepareCanvas;
    property OnSelectEditor;
    //property OnStartDock;
    property OnStartDrag;
    property OnTitleClick;
    property OnUserCheckboxBitmap;
    property OnUserCheckboxState;
    property OnUTF8KeyPress;

    //******************
    property DoubleRowHeight;
    property xStyle;
    property xLineColor;
    property OnImageIndexFromField;
    property IncRowHight;
    property MidleLineColor;
    property DrawMidleLine;
    property InsertAction;
    property EditAction;
    property SelectAction;
    property DeleteAction;
    property RowsHighLights;
  end;

var
  AutoSortDataSet: TAutoSortDataSet;
  GetDataSortField: TGetDataSortField;

implementation

uses
  LConvEncoding, GridFilter, Forms, LazUTF8;

function CalcCanvasCharWidth(Canvas: TCanvas): integer;
begin
  if Canvas.HandleAllocated then
    Result := Canvas.TextWidth('MX') div 2
  else
    Result := 8;
end;

function CalcColumnFieldWidth(Canvas: TCanvas; hasTitle: boolean;
  aTitle: string; aTitleFont: TFont; Field: TField): integer;
var
  aCharWidth: integer;
  aFont: TFont;
  UseTitleFont: boolean;
begin
  if (Field = nil) or (Field.DisplayWidth = 0) then
    Result := DEFCOLWIDTH
  else
  begin

    aCharWidth := CalcCanvasCharWidth(Canvas);
    if Field.DisplayWidth > UTF8Length(aTitle) then
      Result := aCharWidth * Field.DisplayWidth
    else
      Result := aCharWidth * UTF8Length(aTitle);

    if HasTitle then
    begin
      UseTitleFont :=
        (Canvas.Font.Size <> aTitleFont.Size) or
        (Canvas.Font.Style <> aTitleFont.Style) or
        (Canvas.Font.CharSet <> aTitleFont.CharSet) or
        (Canvas.Font.Name <> aTitleFont.Name);
      if UseTitleFont then
      begin
        aFont := TFont.Create;
        aFont.Assign(Canvas.Font);
        Canvas.Font := aTitleFont;
      end;
      try
        aCharWidth := Canvas.TextWidth(ATitle) + 6;
        if aCharWidth > Result then
          Result := aCharWidth;
      finally
        if UseTitleFont then
        begin
          Canvas.Font := aFont;
          aFont.Free;
        end;
      end;
    end; // if HasTitle ...

  end; // if (Field=nil) or (Field.DisplayWidth=0) ...
end;

var
  LookupTmpSetActive: boolean;
  LookupBookMark: TBookmark;

procedure LookupGetBookMark(ALookupField: TField);
begin
  LookupTmpSetActive := not ALookupField.LookupDataSet.Active;
  if LookupTmpSetActive then
    ALookupField.LookupDataSet.Active := True
  else
  begin
    LookupBookMark := ALookupField.LookupDataSet.GetBookmark;
    ALookupField.LookupDataSet.DisableControls;
  end;
end;

procedure LookupGotoBookMark(ALookupField: TField);
begin
  if LookupTmpSetActive then
  begin
    ALookupField.LookupDataSet.Active := False;
    LookupTmpSetActive := False;
  end
  else
    try
      ALookupField.LookupDataSet.GotoBookmark(LookupBookMark);
      ALookupField.LookupDataSet.FreeBookmark(LookupBookMark);
    finally
      ALookupField.LookupDataSet.EnableControls;
    end;
end;

{ TRowsHighLightItem }

constructor TRowsHighLightItem.Create(ACollection: TCollection);
begin
  inherited Create(ACollection);
  Color := clNone;
  FCellOnly := False;
end;

procedure TRowsHighLightItem.Assign(Source: TPersistent);
begin
  if Source is TRowsHighLightItem then
  begin
    //DebugLn('Assigning TxColumn[',dbgs(Index),'] a TxColumn')
    Collection.BeginUpdate;
    try
      inherited Assign(Source);
      FieldName := TRowsHighLightItem(Source).FieldName;
      FieldValue := TRowsHighLightItem(Source).FieldValue;
      Color := TRowsHighLightItem(Source).Color;;
    finally
      Collection.EndUpdate;
    end;
  end
  else
    inherited Assign(Source);
end;

{ TRowsHightLightItems }

function TRowsHightLightItems.GetRowHLItem(Index: Integer): TRowsHighLightItem;
begin
  Result := TRowsHighLightItem(inherited Items[Index]);
end;

procedure TRowsHightLightItems.SetRowHLItem(Index: Integer;
  AValue: TRowsHighLightItem);
begin
  Items[Index].Assign(AValue);
end;

procedure TRowsHightLightItems.Update(Item: TCollectionItem);
begin
  if (FGrid <> nil) and not (csLoading in FGrid.ComponentState) then
    TCustomXDBGrid(FGrid).LayoutChanged;
end;

constructor TRowsHightLightItems.Create(AGrid: TCustomXGrid;
  aItemClass: TCollectionItemClass);
begin
  inherited Create(aItemClass);
  FGrid := AGrid;
end;

{ TCustomXDBGrid }

procedure TCustomXDBGrid.OnRecordChanged(Field: TField);
var
  c: integer;
begin
  {$IfDef dbgDBGrid}
  DBGOut('(' + Name + ') ', 'TCustomDBGrid.OnRecordChanged(Field=');
  if Field = nil then
    DebugLn('nil)')
  else
    DebugLn(Field.FieldName, ')');
  {$Endif}
  if Field = nil then
    UpdateActive
  else
  begin
    c := GetGridColumnFromField(Field);
    if c > 0 then
      InvalidateCell(C, Row)
    else
      UpdateActive;
  end;
end;

function TCustomXDBGrid.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

function TCustomXDBGrid.GetRecordCount: integer;
begin
  Result := FDataLink.DataSet.RecordCount;
end;

function TCustomXDBGrid.GetSelectedFieldRect: TRect;
begin
  Result := CellRect(Col, Row);
end;

function TCustomXDBGrid.GetSelectedIndex: integer;
begin
  if Columns.Enabled then
    Result := ColumnIndexFromGridColumn(Col)
  else
    Result := FieldIndexFromGridColumn(Col);
end;

function TCustomXDBGrid.GetThumbTracking: boolean;
begin
  Result := xgoThumbTracking in inherited Options;
end;

function TCustomXDBGrid.GetDoubleRowHeight: boolean;
begin
  Result := FDoubleRowHeight;
  if not Result then
  begin
    Result := IsExistXFields;
  end;
end;

function TCustomXDBGrid.GetDeleteAction: TBasicAction;
begin
  if DeleteActionLink <> nil then
    Result := DeleteActionLink.Action
  else
    Result := nil;
end;

function TCustomXDBGrid.GetEditAction: TBasicAction;
begin
  if EditActionLink <> nil then
    Result := EditActionLink.Action
  else
    Result := nil;
end;

function TCustomXDBGrid.GetInsertAction: TBasicAction;
begin
  if InsertActionLink <> nil then
    Result := InsertActionLink.Action
  else
    Result := nil;
end;

function TCustomXDBGrid.GetRowHighLights: TRowsHightLightItems;
begin
  Result := FRowsHighLights;
end;

function TCustomXDBGrid.GetSelectAction: TBasicAction;
begin
  if SelectActionLink <> nil then
    Result := SelectActionLink.Action
  else
    Result := nil;
end;

function TCustomXDBGrid.GetSortedField: string;
var
  I: Integer;
begin
  Result := '';
  for I := 0 to Columns.Count - 1 do
  begin
    if Columns.Items[I].Sorted <> soxNone then
    begin
      Result := Columns.Items[I].FieldName;
      Break;
    end
    else
    if Columns.Items[I].xSorted <> soxNone then
    begin
      Result := Columns.Items[I].xFieldName;
      Break;
    end;
  end;
end;

procedure TCustomXDBGrid.SetDeleteAction(AValue: TBasicAction);
begin
  if AValue = nil then
  begin
    DeleteActionLink.Free;
    DeleteActionLink := nil;
  end
  else
  begin
    if DeleteActionLink = nil then
      DeleteActionLink := TActionLink.Create(Self);
    DeleteActionLink.Action := AValue;
    AValue.FreeNotification(Self);
  end;
end;

procedure TCustomXDBGrid.SetDrawMidleLine(AValue: boolean);
begin
  if FDrawMidleLine = AValue then
    Exit;
  FDrawMidleLine := AValue;
  Invalidate;
end;

procedure TCustomXDBGrid.SetEditAction(AValue: TBasicAction);
begin
  if AValue = nil then
  begin
    EditActionLink.Free;
    EditActionLink := nil;
  end
  else
  begin
    if EditActionLink = nil then
      EditActionLink := TActionLink.Create(Self);
    EditActionLink.Action := AValue;
    AValue.FreeNotification(Self);
  end;
end;

procedure TCustomXDBGrid.SetIncRowHight(AValue: integer);
begin
  if FIncRowHeight <> AValue then
  begin
    FIncRowHeight := AValue;
    DefaultRowHeight := GetDefaultRowHeight;
    VisualChange;
  end;
end;

procedure TCustomXDBGrid.SetInsertAction(AValue: TBasicAction);
begin
  if AValue = nil then
  begin
    InsertActionLink.Free;
    InsertActionLink := nil;
  end
  else
  begin
    if InsertActionLink = nil then
      InsertActionLink := TActionLink.Create(Self);
    InsertActionLink.Action := AValue;
    AValue.FreeNotification(Self);
  end;
end;

procedure TCustomXDBGrid.SetLocateInfoPanel(AValue: TPanel);
begin
  if FLocateInfoPanel <> AValue then
  begin
    FLocateInfoPanel:=AValue;
    if FLocateInfoPanel <> nil then
      FLocateInfoPanel.FreeNotification(Self);
  end;
end;

procedure TCustomXDBGrid.SetLocateStr(AValue: string);
begin
  if FLocateStr=AValue then Exit;
  FLocateStr:=AValue;
  if Assigned(FLocateInfoPanel) then FLocateInfoPanel.Caption := FLocateStr;
  if Assigned(FOnChangeLocateStr) then
      FOnChangeLocateStr(Self);
end;

procedure TCustomXDBGrid.SetMidleLineColor(AValue: TColor);
begin
  if FMidleLineColor = AValue then
    Exit;
  FMidleLineColor := AValue;
  Invalidate;
end;

procedure TCustomXDBGrid.SetRowHighLights(AValue: TRowsHightLightItems);
begin
  FRowsHighLights.Assign(AValue);
end;

procedure TCustomXDBGrid.SetSelectAction(AValue: TBasicAction);
begin
  if AValue = nil then
  begin
    SelectActionLink.Free;
    SelectActionLink := nil;
  end
  else
  begin
    if SelectActionLink = nil then
      SelectActionLink := TActionLink.Create(Self);
    SelectActionLink.Action := AValue;
    AValue.FreeNotification(Self);
  end;
end;

procedure TCustomXDBGrid.SetXStyle(Value: TxGridStyle);
begin
  if FxStyle <> Value then
  begin
    FxStyle := Value;
    Invalidate;
  end;
end;

procedure TCustomXDBGrid.xDrawCheckboxBitmaps(aCol, aRow: integer;
  aRect: TRect; F: TField; ChckAlignment: TAlignment);
var
  AState: TCheckboxState;
begin
  if (aCol = Col) and (aRow = Row) then
  begin
    // show checkbox only if overriden editor is hidden
    if EditorMode then
      exit;
  end;

  // by SSY
  if (F <> nil) then
    if F.DataType = ftBoolean then
      if F.IsNull then
        AState := cbGrayed
      else
      if F.AsBoolean then
        AState := cbChecked
      else
        AState := cbUnChecked
    else
    if F.AsString = ColumnFromGridColumn(aCol).ValueChecked then
      AState := cbChecked
    else

    if F.AsString = ColumnFromGridColumn(aCol).ValueUnChecked then
      AState := cbUnChecked
    else
      AState := cbGrayed
  else
    AState := cbGrayed;

  if assigned(OnUserCheckboxState) then
    OnUserCheckboxState(Self, TxColumn(ColumnFromGridColumn(aCol)), AState);

  xDrawGridCheckboxBitmaps(aCol, Row{dummy}, ARect, AState, ChckAlignment);
end;

procedure TCustomXDBGrid.xDrawGridCheckboxBitmaps(const aCol, aRow: integer;
  const aRect: TRect; const aState: TCheckboxState; bmpAlign: TAlignment);
const
  arrtb: array[TCheckboxState] of TThemedButton =
    (tbCheckBoxUncheckedNormal, tbCheckBoxCheckedNormal, tbCheckBoxMixedNormal);
var
  ChkBitmap: TBitmap;
  XPos, YPos: integer;
  details: TThemedElementDetails;
  PaintRect: TRect;
  CSize: TSize;
begin
  if (TitleStyle = tsxNative) and not assigned(OnUserCheckboxBitmap) then
  begin
    Details := ThemeServices.GetElementDetails(arrtb[AState]);
    CSize := ThemeServices.GetDetailSize(Details);
    with PaintRect do
    begin
      case bmpAlign of
        taCenter: Left := Trunc((aRect.Left + aRect.Right - CSize.cx) / 2);
        taLeftJustify: Left := ARect.Left + xconstCellPadding;
        taRightJustify: Left := ARect.Right - CSize.Cx - xconstCellPadding - 1;
      end;
      Top := Trunc((aRect.Top + aRect.Bottom - CSize.cy) / 2);
      PaintRect := Bounds(Left, Top, CSize.cx, CSize.cy);
    end;
    ThemeServices.DrawElement(Canvas.Handle, Details, PaintRect, nil);
  end
  else
  begin
    ChkBitmap := GetImageForCheckBox(aCol, aRow, AState);
    if ChkBitmap <> nil then
    begin
      case bmpAlign of
        taCenter: XPos := Trunc((aRect.Left + aRect.Right - ChkBitmap.Width) / 2);
        taLeftJustify: XPos := ARect.Left + xconstCellPadding;
        taRightJustify: XPos := ARect.Right - ChkBitmap.Width - xconstCellPadding - 1;
      end;
      YPos := Trunc((aRect.Top + aRect.Bottom - ChkBitmap.Height) / 2);
      Canvas.Draw(XPos, YPos, ChkBitmap);
    end;
  end;
end;

procedure TCustomXDBGrid.DrawCellGrid(aCol, aRow: integer; aRect: TRect;
  aState: TxGridDrawState);
var
  ALeft, Y: integer;
  xF: boolean;
begin
  xF := IsxFieldExistFromGridColumn(aCol);
  if xF then
    xF := DrawMidleLine;
  if xStyle = xgsClassic then
  begin
    inherited DrawCellGrid(aCol, aRow, aRect, aState);
    if xf then
    begin
      with Canvas, ARect do
      begin
        ALeft := Left;
        Y := Top + (Bottom - Top) div 2 - 1;
        if xgdSelected in aState then
          Canvas.Pen.Color := clWhite
        else
          Canvas.Pen.Color := FMidleLineColor;
        MoveTo(ALeft + 2, Y);
        LineTo(Right - 2, Y);
      end;
    end;
    Exit;
  end;

  with Canvas, aRect do
  begin
    Pen.Style := GridLineStyle;
    Pen.Color := xLineColor;
    Pen.Width := GridLineWidth;

    ALeft := Left;
    if aCol > 0 then
    begin
      Inc(ALeft, 1);
      MoveTo(ALeft, Top);
      LineTo(ALeft, Bottom);
      Pen.Color := clWindow;
      MoveTo(ALeft - 1, Top);
      LineTo(ALeft - 1, Bottom);
    end
    else
      Dec(ALeft, 1);

    Pen.Color := xLineColor;
    MoveTo(ALeft, Bottom - 1);
    LineTo(Right, Bottom - 1);
    if UseRightToLeftAlignment then
    begin
      MoveTo(ALeft, Top);
      LineTo(ALeft, Bottom);
    end
    else
    begin
      MoveTo(Right - 1, Top);
      LineTo(Right - 1, Bottom);
    end;

    if xF then
    begin
      Y := Top + (Bottom - Top) div 2 - 1;
      if xgdSelected in aState then
        Canvas.Pen.Color := clWhite
      else
        Canvas.Pen.Color := FMidleLineColor;
      MoveTo(ALeft + 2, Y);
      LineTo(Right - 2, Y);
    end;
  end;
end;

procedure TCustomXDBGrid.SetxLineColor(AValue: TColor);
begin
  if FxLineColor <> AValue then
  begin
    FxLineColor := AValue;
    if xStyle = xgsNormal then
      Invalidate;
  end;
end;

function TCustomXDBGrid.GetxFieldFromGridColumn(ACol: integer): TField;
var
  i: integer;
begin
  Result := nil;
  if Columns.Enabled then
  begin
    i := ColumnIndexFromGridColumn(ACol);
    if i >= 0 then
      Result := TxDBGridColumns(Columns)[i].FxField
    else
      Result := nil;
  end;
  {else
    result := GetDsFieldFromGridColumn(Column);}
end;

function TCustomXDBGrid.IsxFieldExistFromGridColumn(ACol: integer): boolean;
var
  i: integer;
begin
  Result := False;
  if Columns.Enabled then
  begin
    i := ColumnIndexFromGridColumn(ACol);
    if i >= 0 then
    begin
      Result := TxDBGridColumns(Columns)[i].FxFieldName <> '';
    end;
  end;
end;

procedure TCustomXDBGrid.DrawCellText(aCol, aRow: integer; aRect: TRect;
  aState: TxGridDrawState; aText: string);
var
  txtStyle: TTextStyle;
  F: TField;
begin
  if DoubleRowHeight then
  begin
    with ARect do
    begin
      Dec(Right, xconstCellPadding);
      case Canvas.TextStyle.Alignment of
        Classes.taLeftJustify: Inc(Left, xconstCellPadding);
        Classes.taRightJustify: Dec(Right, 1);
      end;
      case Canvas.TextStyle.Layout of
        tlTop: Inc(Top, xconstCellPadding);
        tlBottom: Dec(Bottom, xconstCellPadding);
      end;

      if Right < Left then
        Right := Left;
      if Left > Right then
        Left := Right;
      if Bottom < Top then
        Bottom := Top;
      if Top > Bottom then
        Top := Bottom;

      F := GetxFieldFromGridColumn(aCol);
      if F = nil then
        txtStyle.SingleLine := False
      else
        txtStyle.SingleLine := True;

      txtStyle.RightToLeft := Canvas.TextStyle.RightToLeft;
      txtStyle.Wordbreak := True;
      txtStyle.Alignment := Canvas.TextStyle.Alignment;
      txtStyle.Layout := tlCenter;
      txtStyle.EndEllipsis := False;
      //txtStyle.Opaque := False;

      if (Left <> Right) and (Top <> Bottom) then
        Canvas.TextRect(aRect, Left, Top, aText, txtStyle);
    end;
  end
  else
    inherited DrawCellText(aCol, aRow, aRect, aState, aText);
end;

procedure TCustomXDBGrid.xDrawCellText(aCol, aRow: integer; aRect: TRect;
  aState: TxGridDrawState; aText: string; WordWrap: boolean);
var
  txtStyle: TTextStyle;
begin
  with ARect do
  begin
    Dec(Right, xconstCellPadding);
    case Canvas.TextStyle.Alignment of
      Classes.taLeftJustify: Inc(Left, xconstCellPadding);
      Classes.taRightJustify: Dec(Right, 1);
    end;
    case Canvas.TextStyle.Layout of
      tlTop: Inc(Top, xconstCellPadding);
      tlBottom: Dec(Bottom, xconstCellPadding);
    end;

    if Right < Left then
      Right := Left;
    if Left > Right then
      Left := Right;
    if Bottom < Top then
      Bottom := Top;
    if Top > Bottom then
      Top := Bottom;

    txtStyle.SingleLine := not WordWrap;
    txtStyle.RightToLeft := Canvas.TextStyle.RightToLeft;
    txtStyle.Wordbreak := True;
    txtStyle.Alignment := Canvas.TextStyle.Alignment;
    txtStyle.Layout := tlCenter;
    txtStyle.EndEllipsis := False;
    txtStyle.SystemFont := False;
    txtStyle.Opaque := False;

    if (Left <> Right) and (Top <> Bottom) then
      Canvas.TextRect(aRect, Left, Top, aText, txtStyle);
  end;
end;

procedure TCustomXDBGrid.DrawCellImage(ARect: TRect; AColumn: TxColumn;
  F: TField);
var
  ImgIndex: integer;
  w, h: integer;
  needStretch: boolean;
  S: string;
  R: TRect;
begin
  if ImageList = nil then
    Exit;
  if F = nil then
    Exit;

  if Assigned(FOnImageIndexFromField) then
    ImgIndex := FOnImageIndexFromField(F, AColumn.Index)
  else
  if AColumn.ImagesIndexes.Count > 0 then
  begin
    ImgIndex := -1;
    S := AColumn.ImagesIndexes.Values[F.AsString];
    if S <> '' then ImgIndex := StrToInt(S);
  end
  else
  if F is TNumericField then
    ImgIndex := F.AsInteger
  else
    ImgIndex := -1;

  if not InRange(ImgIndex, 0, ImageList.Count - 1) then
    exit;

  needStretch := False;

  w := ImageList.Width;
  h := ImageList.Height;
  R := ARect;
  R.Left := ARect.Left + (ARect.Right - ARect.Left - w) div 2;
  R.Top := ARect.Top + (ARect.Bottom - ARect.Top - h) div 2;

  if R.Left < ARect.Left then
  begin
    needStretch := True;
    R.Left := ARect.Left + 1;
    R.Right := ARect.Right - 1;
  end;

  if R.Top < ARect.Top then
  begin
    needStretch := True;
    R.Top := ARect.Top + 1;
    R.Bottom := ARect.Bottom - 1;
  end;
  needStretch := False;
  if needStretch then
    ImageList.StretchDraw(Canvas, ImgIndex, R)
  else
    ImageList.Draw(Canvas, r.Left, r.Top, ImgIndex);

end;

procedure TCustomXDBGrid.xDrawCellImage(ARect: TRect; AColumn: TxColumn;
  F: TField);
var
  ImgIndex: integer;
  w, h: integer;
  needStretch: boolean;
  S: string;
  R: TRect;
begin
  if ImageList = nil then
    Exit;
  if F = nil then
    Exit;

  if Assigned(FOnImageIndexFromField) then
    ImgIndex := FOnImageIndexFromField(F, AColumn.Index)
  else
  if AColumn.xImagesIndexes.Count > 0 then
  begin
    ImgIndex := -1;
    S := AColumn.xImagesIndexes.Values[F.AsString];
    if S <> '' then ImgIndex := StrToInt(S);
  end
  else
  if F is TNumericField then
    ImgIndex := F.AsInteger
  else
    ImgIndex := -1;

  if not InRange(ImgIndex, 0, ImageList.Count - 1) then
    exit;

  needStretch := False;

  w := ImageList.Width;
  h := ImageList.Height;
  R := ARect;
  R.Left := ARect.Left + (ARect.Right - ARect.Left - w) div 2;
  R.Top := ARect.Top + (ARect.Bottom - ARect.Top - h) div 2;

  if R.Left < ARect.Left then
  begin
    needStretch := True;
    R.Left := ARect.Left + 1;
    R.Right := ARect.Right - 1;
  end;

  if R.Top < ARect.Top then
  begin
    needStretch := True;
    R.Top := ARect.Top + 1;
    R.Bottom := ARect.Bottom - 1;
  end;
  needStretch := False;
  if needStretch then
    ImageList.StretchDraw(Canvas, ImgIndex, R)
  else
    ImageList.Draw(Canvas, r.Left, r.Top, ImgIndex);
end;

procedure TCustomXDBGrid.DoEditButtonClick(const ACol, ARow: integer);
begin
  if Assigned(FOnEdButtonClick) then
    FOnEdButtonClick(Self, SelectedColumn, CellCursorPos);

  inherited DoEditButtonClick(ACol, ARow);

  EditordoSetValue;
end;

function TCustomXDBGrid.IsExistXFields: boolean;
var
  I: integer;
begin
  Result := False;
  for I := 0 to Columns.Count - 1 do
  begin
    if Columns.Items[I].xFieldName <> '' then
    begin
      Result := True;
      Break;
    end;
  end;
end;

procedure TCustomXDBGrid.FontChanged(Sender: TObject);
begin
  inherited FontChanged(Sender);
  DefaultRowHeight := GetDefaultRowHeight;
end;

procedure TCustomXDBGrid.IncFontValue(AValue: integer);
var
  I: Integer;
begin
  if (Font.Size = 0) and (AValue = 1) then
    Font.Size := 10
  else
  begin
    I := Font.Size + AValue;
    if I < 10 then Font.Size:= 0
    else
      Font.Size:= I;
  end;


end;

procedure TCustomXDBGrid.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  if (Operation = opRemove) then
  begin
    if AComponent is TBasicAction then
    begin
      if AComponent = InsertAction then InsertAction := nil
      else
      if AComponent = EditAction then EditAction := nil
      else
      if AComponent = SelectAction then SelectAction := nil
      else
      if AComponent = DeleteAction then DeleteAction := nil;
    end
    else
    if AComponent = LocateInfoPanel then LocateInfoPanel := nil;
  end;
  inherited Notification(AComponent, Operation);
end;

procedure TCustomXDBGrid.SortDataSet(Column: TxColumn;
  CellCursorPos: TCellCursorPos);
begin
  if Assigned(AutoSortDataSet) and Assigned(DataSource.DataSet)
    and not (csDesigning in ComponentState) then
    begin
      CheckSortingImage;
      AutoSortDataSet(DataSource.DataSet, Column, CellCursorPos);
    end;
end;

procedure TCustomXDBGrid.xSwapCheckBox;
var
  SelField: TField;
  TempColumn: TxColumn;
begin
  if not GridCanModify then
    exit;

  SelField := nil;
  TempColumn := SelectedColumn;
  if TempColumn <> nil then
    SelField := TempColumn.xField;
  //  TempColumn := TColumn(ColumnFromGridColumn(Col));
  if (SelField <> nil) and (TempColumn <> nil) and not TempColumn.ReadOnly and
    Datalink.Edit then
  begin
    if SelField.DataType = ftBoolean then
      SelField.AsBoolean := not SelField.AsBoolean
    else
    begin
      if TempColumn.ValueChecked = SelField.AsString then
        SelField.AsString := TempColumn.ValueUnchecked
      else
        SelField.AsString := TempColumn.ValueChecked;
    end;
    InvalidateCell(Col, Row);
  end;
end;

procedure TCustomXDBGrid.EmptyGrid;
var
  OldFixedCols, OldFixedRows: integer;
begin
  OldFixedCols := FixedCols;
  OldFixedRows := FixedRows;
  Clear;
  RowCount := OldFixedRows + 1;
  ColCount := OldFixedCols + 1;
  if xdgIndicator in Options then
    ColWidths[0] := IndicatorWidth;
end;

function TCustomXDBGrid.GetColumns: TxDBGridColumns;
begin
  Result := TxDBGridColumns(inherited Columns);
end;

procedure TCustomXDBGrid.InvalidateSizes;
begin
  GridFlags := GridFlags + [gfxVisualChange];
end;

function TCustomXDBGrid.GetCurrentColumn: TxColumn;
begin
  if Columns.Enabled then
    Result := TxColumn(Columns[SelectedIndex])
  else
    Result := nil;
end;

function TCustomXDBGrid.GetCurrentField: TField;
var
  CIndex: integer;
  Column: TxColumn;
begin
  Column := nil;
  if SelectedColumn <> nil then
    Column := SelectedColumn
  else
  begin
    CIndex := ColumnIndexFromGridColumn(Col);
    if CIndex >= 0 then
      Column := Columns[CIndex];
  end;

  if Column <> nil then
  begin
    if CellCursorPos = ccpTop then
      Result := Column.Field
    else
      Result := Column.xField;
  end
  else
    Result := GetFieldFromGridColumn(Col);
end;

procedure TCustomXDBGrid.OnDataSetChanged(aDataSet: TDataSet);
begin
  {$Ifdef dbgDBGrid}
  DebugLn('(%s) TCustomDBDrid.OnDataSetChanged(aDataSet=%s) INIT',
    [Name, dbgsname(ADataset)]);
  {$endif}
  if not (gsStartEditing in FGridStatus) then
  begin
    if EditorMode then
      EditorMode := False;
    LayoutChanged;
  end;
  UpdateActive;
  if not (gsStartEditing in FGridStatus) then
  begin
    SelectEditor;
    if (xdgAlwaysShowEditor in Options) and not EditorMode then
      EditorMode := True;
  end;
  {$Ifdef dbgDBGrid}
  DebugLn('(%s) TCustomDBDrid.OnDataSetChanged(aDataSet=%s) DONE',
    [Name, dbgsname(ADataset)]);
  {$endif}
end;

procedure TCustomXDBGrid.OnDataSetOpen(aDataSet: TDataSet);

begin
  {$Ifdef dbgDBGrid}
  DebugLn('(%s) TCustomDBDrid.OnDataSetOpen INIT', [Name]);
  {$endif}
  RenewColWidths;
  LinkActive(True);
  UpdateActive;
  SelectEditor;
  CheckSorted;
  {$Ifdef dbgDBGrid}
  DebugLn('(%s) TCustomDBDrid.OnDataSetOpen DONE', [Name]);
  {$endif}
end;

procedure TCustomXDBGrid.OnDataSetClose(aDataSet: TDataSet);
begin
  {$ifdef dbgDBGrid}
  DebugLn('(', Name, ') ', 'TCustomDBGrid.OnDataSetClose');
  {$endif}
  LinkActive(False);
end;

procedure TCustomXDBGrid.OnEditingChanged(aDataSet: TDataSet);
begin
  {$ifdef dbgDBGrid}
  DebugLn('(', Name, ') ', 'TCustomDBGrid.OnEditingChanged');
  if aDataSet <> nil then
  begin
    DebugLn(['Editing=', dsEdit = aDataSet.State]);
    DebugLn(['Inserting=', dsInsert = aDataSet.State]);
  end
  else
    DebugLn('Dataset=nil');
  {$endif}
  FDataLink.Modified := False;
  UpdateActive;
end;

procedure TCustomXDBGrid.OnInvalidDataSet(aDataSet: TDataSet);
begin
  {$ifdef dbgDBGrid}
  DebugLn('(', Name, ') ', 'TCustomDBGrid.OnInvalidDataSet');
  {$endif}
  LinkActive(False);
end;

procedure TCustomXDBGrid.OnInvalidDataSource(aDataSet: TDataset);
begin
  {$ifdef dbgDBGrid}
  DebugLn('(', Name, ') ', 'TCustomDBGrid.OnInvalidDataSource');
  {$endif}
  LinkActive(False);
end;

procedure TCustomXDBGrid.OnLayoutChanged(aDataSet: TDataSet);
begin
  {$ifdef dbgDBGrid}
  DebugLn('(', Name, ') ', 'TCustomDBGrid.OnLayoutChanged');
  {$endif}
  LayoutChanged;
end;

procedure TCustomXDBGrid.OnNewDataSet(aDataSet: TDataset);

begin
  {$Ifdef dbgDBGrid}
  DebugLn('(%s) TCustomDBDrid.OnNewDataset INIT', [Name]);
  {$endif}
  RenewColWidths;
  LinkActive(True);
  UpdateActive;
  SelectEditor;
  CheckSorted;
  {$Ifdef dbgDBGrid}
  DebugLn('(%s) TCustomDBDrid.OnNewDataset DONE', [Name]);
  {$endif}
end;

procedure TCustomXDBGrid.OnDataSetScrolled(aDataSet: TDataSet; Distance: integer
  );
var
  OldEditorMode: boolean;
begin
  {$ifdef dbgDBGrid}
  DebugLn(ClassName, ' (', Name, ')', '.OnDataSetScrolled(', IntToStr(Distance), ')');
  Debugln('Dataset.RecordCount=', IntToStr(aDataSet.RecordCount));
  {$endif}
  UpdateScrollBarRange;
  // todo: Use a fast interface method to scroll a rectangular section of window
  //       if distance=+, Row[Distance] to Row[RowCount-2] UP
  //       if distance=-, Row[FixedRows+1] to Row[RowCount+Distance] DOWN

  OldEditorMode := EditorMode;
  if OldEditorMode then
    EditorMode := False;

  if Distance <> 0 then
  begin
    Row := FixedRows + FDataLink.ActiveRecord;
    Invalidate;
  end
  else
    UpdateActive;

  if OldEditorMode and (xdgAlwaysShowEditor in Options) then
    EditorMode := True;
end;

procedure TCustomXDBGrid.OnUpdateData(aDataSet: TDataSet);
begin
  UpdateData;
end;

procedure TCustomXDBGrid.SetColumns(const AValue: TxDBGridColumns);
begin
  inherited Columns := TxGridColumns(AValue);
end;

{
procedure TCustomXDBGrid.ReadColumns(Reader: TReader);
begin
  Columns.Clear;
  Reader.ReadValue;
  Reader.ReadCollection(Columns);
end;
procedure TCustomXDBGrid.SetColumns(const AValue: TxDBGridColumns);
begin
  Columns.Assign(AValue);
end;
}
procedure TCustomXDBGrid.SetCurrentField(const AValue: TField);
var
  i: integer;
begin
  if Avalue <> SelectedField then
  begin
    i := GetGridColumnFromField(AValue);
    if (i >= FirstGridColumn) and (i >= FixedCols) then
      Col := i;
  end;
end;

procedure TCustomXDBGrid.SetDataSource(const AValue: TDataSource);
begin
  if AValue = FDatalink.Datasource then
    Exit;
  RenewColWidths;
  FDataLink.DataSource := AValue;
  UpdateActive;
end;

procedure TCustomXDBGrid.SetExtraOptions(const AValue: TxDbGridExtraOptions);
var
  OldOptions: TxDbGridExtraOptions;

  function IsOptionChanged(Op: TxDbGridExtraOption): boolean;
  begin
    Result := ((op in OldOptions) and not (op in AValue)) or
      (not (op in OldOptions) and (op in AValue));
  end;

begin
  if FExtraOptions = AValue then
    exit;
  OldOptions := FExtraOptions;
  FExtraOptions := AValue;

  if IsOptionChanged(dgeCheckboxColumn) then
    Invalidate;

  if IsOptionChanged(dgeAutoColumns) then
  begin
    if dgeAutoColumns in aValue then
      AddAutomaticColumns
    else if TxDBGridColumns(Columns).HasAutomaticColumns then
      RemoveAutomaticColumns;
    UpdateActive;
  end;

end;

procedure TCustomXDBGrid.SetOptions(const AValue: TxDbGridOptions);
var
  OldOptions: TxGridOptions;
  ChangedOptions: TxDbGridOptions;
  MultiSel: boolean;
begin
  if FOptions <> AValue then
  begin
    MultiSel := xdgMultiselect in FOptions;
    ChangedOptions := (FOptions - AValue) + (AValue - FOptions);
    FOptions := AValue;
    OldOptions := inherited Options;

    if xdgRowSelect in FOptions then
    begin
      FOptions := FOptions - [xdgEditing, xdgAlwaysShowEditor];
      Include(OldOptions, xgoRelaxedRowSelect);
    end;


    BeginLayout;

    if xdgRowLines in fOptions then
      Include(OldOptions, xgoHorzLine)
    else
      Exclude(OldOptions, xgoHorzLine);

    if xdgColLines in fOptions then
      Include(OldOptions, xgoVertLine)
    else
      Exclude(OldOptions, xgoVertLine);

    if xdgColumnResize in fOptions then
      Include(OldOptions, xgoColSizing)
    else
      Exclude(OldOptions, xgoColSizing);

    if xdgColumnMove in fOptions then
      Include(OldOptions, xgoColMoving)
    else
      Exclude(OldOptions, xgoColMoving);

    if xdgAlwaysShowEditor in FOptions then
      Include(OldOptions, xgoAlwaysShowEditor)
    else
      Exclude(OldOptions, xgoAlwaysShowEditor);

    if xdgRowSelect in FOptions then
      Include(OldOptions, xgoRowSelect)
    else
      Exclude(OldOptions, xgoRowSelect);

    if xdgRowHighLight in FOptions then
      Include(OldOptions, xgoRowHightLight)
    else
      Exclude(OldOptions, xgoRowHightLight);

    if xdgEditing in FOptions then
      Include(OldOptions, xgoEditing)
    else
      Exclude(OldOptions, xgoediting);

    if xdgTabs in FOptions then
      Include(OldOptions, xgoTabs)
    else
      Exclude(OldOptions, xgoTabs);

    if xdgHeaderHotTracking in FOptions then
      Include(OldOptions, xgoHeaderHotTracking)
    else
      Exclude(OldOptions, xgoHeaderHotTracking);

    if xdgHeaderPushedLook in FOptions then
      Include(OldOptions, xgoHeaderPushedLook)
    else
      Exclude(OldOptions, xgoHeaderPushedLook);

    if xdgCellHints in FOptions then
      Include(OldOptions, xgoCellHints)
    else
      Exclude(OldOptions, xgoCellHints);

    if xdgTruncCellHints in FOptions then
      Include(OldOptions, xgoTruncCellHints)
    else
      Exclude(OldOptions, xgoTruncCellHints);

    if xdgCellEllipsis in FOptions then
      Include(OldOptions, xgoCellEllipsis)
    else
      Exclude(OldOptions, xgoCellEllipsis);

    if (xdgIndicator in ChangedOptions) then
    begin
      if (xdgIndicator in FOptions) then
        FixedCols := FixedCols + 1
      else
        FixedCols := FixedCols - 1;
    end;

    if (xdgTitles in ChangedOptions) then
    begin
      if xdgTitles in FOptions then
        FixedRows := FixedRows + 1
      else
        FixedRows := FixedRows - 1;
    end;

    if (xdgAutoSizeColumns in ChangedOptions) then
    begin
      Exclude(FGridStatus, gsAutoSized);
    end;

    inherited Options := OldOptions;

    if MultiSel and not (xdgMultiselect in FOptions) then
    begin
      FSelectedRows.Clear;
      FKeyBookmark := nil;
    end;

    EndLayout;
  end;
end;

procedure TCustomXDBGrid.SetSelectedIndex(const AValue: integer);
begin
  Col := FirstGridColumn + AValue;
end;

procedure TCustomXDBGrid.SetThumbTracking(const AValue: boolean);
begin
  BeginUpdate;
  if Avalue then
    inherited Options := inherited Options + [xgoThumbTracking]
  else
    inherited Options := inherited Options - [xgoThumbTracking];
  EndUpdate(False);
end;

procedure TCustomXDBGrid.UpdateBufferCount;
var
  BCount: integer;
begin
  if FDataLink.Active then
  begin
    BCount := GetBufferCount;
    if BCount < 1 then
      BCount := 1;
    FDataLink.BufferCount := BCount;
    {$ifdef dbgDBGrid}
    DebugLn('%s (%s), FDatalink.BufferCount=%d', [ClassName, Name, FDataLink.BufferCount]);
    {$endif}
  end;
end;

procedure TCustomXDBGrid.UpdateData;
var
  selField, edField: TField;
  LookupKeyValues: variant;
  Column: TxColumn;
  ColIndex: integer;
begin
  // get Editor text and update field content
  if not UpdatingData and (FEditingColumn > -1) and FDatalink.Editing then
  begin
    ColIndex := ColumnIndexFromGridColumn(Col);
    Column := Columns[ColIndex];
    if (Column.xField <> nil) and (CellCursorPos = ccpBottom) then
    begin
      selField := Column.xField;
      edField := GetxFieldFromGridColumn(FEditingColumn);
    end
    else
    begin
      SelField := SelectedField;
      edField := GetFieldFromGridColumn(FEditingColumn);

    end;

    if (edField <> nil) and (edField = SelField) then
    begin
      {$ifdef dbgDBGrid}
      DebugLn('---> UpdateData: Field[', edField.Fieldname,
        '(', edField.AsString, ')]=', FTempText, ' INIT');
      {$endif}

      StartUpdating;
      edField.Text := FTempText;
      if edField.Lookup then
      begin
        LookupKeyValues := Null;
        if edField.LookupCache then
          LookupKeyValues := edField.LookupList.FirstKeyByValue(FTempText)
        else
        begin
          LookupGetBookMark(edField);
          try
            if edField.LookupDataSet.Locate(edField.LookupResultField,
              VarArrayOf([FTempText]), []) then
              LookupKeyValues :=
                edField.LookupDataSet.FieldValues[edField.LookupKeyFields];
          finally
            LookupGotoBookMark(edField);
          end;
        end;
        edField.DataSet.FieldValues[edField.KeyFields] := LookupKeyValues;
      end;
      EndUpdating;

      EditingColumn(FEditingColumn, False);
      {$ifdef dbgDBGrid}
      DebugLn('<--- UpdateData: Chk: Field:=', edField.AsString, ' END');
      {$endif}
    end;

  end;
end;

{$ifdef dbgDBGrid}
function SBCodeToStr(Code: integer): string;
begin
  case Code of
    SB_LINEUP: Result := 'SB_LINEUP';
    SB_LINEDOWN: Result := 'SB_LINEDOWN';
    SB_PAGEUP: Result := 'SB_PAGEUP';
    SB_PAGEDOWN: Result := 'SB_PAGEDOWN';
    SB_THUMBTRACK: Result := 'SB_THUMBTRACK';
    SB_THUMBPOSITION: Result := 'SB_THUMBPOSITION';
    SB_ENDSCROLL: Result := 'SB_SCROLLEND';
    SB_TOP: Result := 'SB_TOP';
    SB_BOTTOM: Result := 'SB_BOTTOM';
    else
      Result := IntToStr(Code) + ' -> ?';
  end;
end;

{$endif}

procedure TCustomXDBGrid.WMVScroll(var Message: TLMVScroll);
var
  IsSeq: boolean;
  aPos, aRange, aPage: integer;
  DeltaRec: integer;

  function MaxPos: integer;
  begin
    if IsSeq then
      Result := GetRecordCount - 1
    else
      Result := 4;
  end;

  procedure DsMoveBy(Delta: integer);
  begin
    FDataLink.MoveBy(Delta);
    GetScrollbarParams(aRange, aPage, aPos);
  end;

  procedure DsGoto(BOF: boolean);
  begin
    if BOF then
      FDatalink.DataSet.First
    else
      FDataLink.DataSet.Last;
    GetScrollbarParams(aRange, aPage, aPos);
  end;

begin
  if not FDatalink.Active then
    exit;

  {$ifdef dbgDBGrid}
  DebugLn('VSCROLL: Code=', SbCodeToStr(Message.ScrollCode),
    ' Position=', dbgs(Message.Pos), ' OldPos=', Dbgs(FOldPosition));
  {$endif}

  IsSeq := FDatalink.DataSet.IsSequenced;
  case Message.ScrollCode of
    SB_TOP:
      DsGoto(True);
    SB_BOTTOM:
      DsGoto(False);
    SB_PAGEUP:
      DsMoveBy(-VisibleRowCount);
    SB_LINEUP:
      DsMoveBy(-1);
    SB_LINEDOWN:
      DsMoveBy(1);
    SB_PAGEDOWN:
      DsMoveBy(VisibleRowCount);
    SB_THUMBPOSITION, SB_THUMBTRACK:
    begin
      if (Message.ScrollCode = SB_THUMBTRACK) then
      begin
        if not ThumbTrack then Exit;
      end;

      aPos := Message.Pos;
      if aPos = FOldPosition then
        exit;
      if aPos >= MaxPos then
        dsGoto(False)
      else if aPos <= 0 then
        dsGoto(True)
      else if IsSeq then
        FDatalink.DataSet.RecNo := aPos + 1
      else
      begin
        DeltaRec := Message.Pos - FOldPosition;
        if DeltaRec = 0 then
          exit
        else if DeltaRec < -1 then
          DsMoveBy(-VisibleRowCount)
        else if DeltaRec > 1 then
          DsMoveBy(VisibleRowCount)
        else
          DsMoveBy(DeltaRec);
      end;
    end;
    else
      Exit;
  end;

  ScrollBarPosition(SB_VERT, aPos);
  FOldPosition := aPos;

  if EditorMode then
    RestoreEditor;
  {$ifdef dbgDBGrid}
  DebugLn('---- Diff=', dbgs(DeltaRec), ' FinalPos=', dbgs(aPos));
  {$endif}
end;

procedure TCustomXDBGrid.WndProc(var TheMessage: TLMessage);
begin
  if (TheMessage.Msg = LM_SETFOCUS) and (gsUpdatingData in FGridStatus) then
  begin
    {$ifdef dbgGrid}
    DebugLn('DBGrid.LM_SETFOCUS while updating');
{$endif}
    if EditorMode then
    begin
      LCLIntf.SetFocus(Editor.Handle);
      EditorSelectAll;
    end;
    exit;
  end;
  inherited WndProc(TheMessage);
end;

procedure TCustomXDBGrid.DoAutoLocate;
var
  C: TxColumn;
  FieldName, AStr: string;
begin
  if Assigned(Datalink.DataSet) then
  begin

    FieldName:= SortedField;

    if FieldName = '' then
    begin
      C := GetCurrentColumn;
      if C <> nil then
      begin
        if CellCursorPos = ccpTop then
          FieldName:= C.FieldName
        else
          FieldName:= C.xFieldName;
      end;
    end;

    if (FieldName <> '') and (FLocateStr <> '') then
    begin
      AStr := UTF8Copy(FLocateStr, 1, UTF8Length(FLocateStr));
      Datalink.DataSet.Locate(FieldName, AStr, [loCaseInsensitive, loPartialKey]);
    end;
  end;
end;

procedure TCustomXDBGrid.UTF8KeyPress(var UTF8Key: TUTF8Char);
var
  S: string;
  AnsStr: AnsiString;

  function IsDigit(C: Char): Boolean;
  begin
    Result := False;
    if C <> '' then
      if C in ['0'..'9'] then Result := True;
  end;

  function IsLetter(C: Char): Boolean;
  var
    I: Integer;
  begin
    Result := False;
    if C <> '' then
    begin
      I := Ord(C);
      if (I = 32) or ((I >= 65) and (I <= 90)) or ((I >= 97) and (I < 122))
      or ((I >= 192) and (I <= 255)) or (I = 168) or (I = 184) then Result := True;
    end;
  end;

  function IsSimbol(C: Char): Boolean;
  var
    I: Integer;
  begin
    Result := False;
    if C <> '' then
    begin
      I := Ord(C);
      if I in [33..47, 58..64, 91..96, 123..126, 128..191] then Result := True;
    end;
  end;

begin
  inherited UTF8KeyPress(UTF8Key);
  if (xdgAutoLocate in Options) and (xdgRowSelect in Options) then
  begin
    S := UTF8Key;
    AnsStr := UTF8ToCP1251(S);
    if IsDigit(AnsStr[1]) or IsLetter(AnsStr[1]) or IsSimbol(AnsStr[1]) then
    begin
      LocateStr := FLocateStr + UTF8Key;
      DoAutoLocate;
    end;
  end;
end;

{procedure TCustomXDBGrid.KeyPress(var Key: char);
begin
  inherited KeyPress(Key);
  if (Key = ' ') and (xdgAutoLocate in Options) and (xdgRowSelect in Options) then
  begin
    LocateStr := FLocateStr + Key;
    DoAutoLocate;
  end;
end;}

procedure TCustomXDBGrid.SetDoubleRowHeight(AValue: boolean);
var
  ExF: boolean;
begin
  if FDoubleRowHeight <> AValue then
  begin
    ExF := IsExistXFields;
    if ExF then
    begin
      if (ExF <> FDoubleRowHeight) then
      begin
        FDoubleRowHeight := True;
        DefaultRowHeight := GetDefaultRowHeight;
        VisualChange;
      end;
    end
    else
    begin
      FDoubleRowHeight := AValue;
      DefaultRowHeight := GetDefaultRowHeight;
      VisualChange;
    end;
  end;
end;

function TCustomXDBGrid.DefaultFieldColWidth(F: TField): integer;
begin
  if not HandleAllocated or (F = nil) then
    Result := DefaultColWidth
  else
  begin
    if F.DisplayWidth = 0 then
      if Canvas.HandleAllocated then
        Result := Canvas.TextWidth(F.DisplayName) + 3
      else
        Result := DefaultColWidth
    else
      Result := F.DisplayWidth * CalcCanvasCharWidth(Canvas);
  end;
end;

function TCustomXDBGrid.GetColumnCount: integer;
var
  i: integer;
  F: TField;
begin
  Result := 0;
  if Columns.Enabled then
    Result := Columns.VisibleCount
  else
  if (dgeAutoColumns in OptionsExtra) and FDataLink.Active then
    for i := 0 to FDataLink.DataSet.FieldCount - 1 do
    begin
      F := FDataLink.DataSet.Fields[i];
      if (F <> nil) and F.Visible then
        Inc(Result);
    end;
end;

// Get the visible field (from dataset fields) that corresponds to given column
function TCustomXDBGrid.GetDsFieldFromGridColumn(Column: integer): TField;
var
  i: integer;
begin
  i := FieldIndexFromGridColumn(Column);
  if i >= 0 then
    Result := FDataLink.DataSet.Fields[i]
  else
    Result := nil;
end;

function TCustomXDBGrid.FirstGridColumn: integer;
begin
  if (xdgIndicator in Options) then
    Result := 1
  else
    Result := 0;
end;

procedure TCustomXDBGrid.PrepareCellHints(aCol, aRow: integer);
begin
  if not DataLink.Active then
    Exit;
  FSavedRecord := DataLink.ActiveRecord;
  DataLink.ActiveRecord := ARow - FixedRows;
end;

procedure TCustomXDBGrid.UnprepareCellHints;
begin
  if not DataLink.Active then
    Exit;
  DataLink.ActiveRecord := FSavedRecord;
end;

function TCustomXDBGrid.GetCellHintText(aCol, aRow: integer): string;
var
  C: TxColumn;
begin
  Result := '';
  if (ARow < FixedRows) then
    exit;
  if Assigned(FOnGetCellHint) then
  begin
    C := ColumnFromGridColumn(ACol) as TxColumn;
    FOnGetCellHint(self, C, Result);
  end;

end;

function TCustomXDBGrid.GetTruncCellHintText(aCol, aRow: integer): string;
var
  F: TField;
begin
  Result := '';
  if ARow < FixedRows then
    exit;
  F := GetFieldFromGridColumn(ACol);
  if (F <> nil) then
    if (F.DataType <> ftBlob) then
      Result := F.DisplayText
    else
      Result := '(blob)';
end;

// obtain the field either from a Db column or directly from dataset fields
function TCustomXDBGrid.GetFieldFromGridColumn(Column: integer): TField;
var
  i: integer;
begin
  if Columns.Enabled then
  begin
    i := ColumnIndexFromGridColumn(Column);
    if i >= 0 then
      Result := TxDBGridColumns(Columns)[i].FField
    else
      Result := nil;
  end
  else
    Result := GetDsFieldFromGridColumn(Column);
end;

// obtain the corresponding grid column for the given field
function TCustomXDBGrid.GetGridColumnFromField(F: TField): integer;
var
  i: integer;
begin
  Result := -1;
  for i := FirstGridColumn to ColCount - 1 do
  begin
    if GetFieldFromGridColumn(i) = F then
    begin
      Result := i;
      break;
    end;
  end;
end;

function TCustomXDBGrid.GetImageForCheckBox(const aCol, aRow: integer;
  CheckBoxView: TCheckBoxState): TBitmap;
begin
  Result := inherited GetImageForCheckBox(aCol, aRow, CheckBoxView);
  if Assigned(OnUserCheckboxBitmap) then
    OnUserCheckboxBitmap(Self, CheckBoxView, Result);
end;

// obtain the visible field index corresponding to the grid column index
function TCustomXDBGrid.FieldIndexFromGridColumn(AGridCol: integer): integer;
var
  i: integer;
  Column: TxColumn;
begin
  Result := -1;
  if not FDatalink.Active then
    exit;

  if Columns.Enabled then
  begin
    Column := TxColumn(ColumnFromGridColumn(AGridCol));
    if (Column <> nil) and (Column.Field <> nil) and Column.Field.Visible then
      Result := FDatalink.Dataset.Fields.IndexOf(Column.Field);
  end
  else
  begin
    AGridCol := AGridCol - FirstGridColumn;
    i := 0;
    while (AGridCol >= 0) and (i < FDatalink.DataSet.FieldCount) do
    begin
      if FDatalink.Fields[i].Visible then
      begin
        Dec(AGridCol);
        if AGridCol < 0 then
        begin
          Result := i;
          break;
        end;
      end;
      Inc(i);
    end;
  end;
end;

function TCustomXDBGrid.GetBufferCount: integer;
begin
  Result := ClientHeight div DefaultRowHeight;
  if xdgTitles in Options then
    Dec(Result, 1);
end;

procedure TCustomXDBGrid.UpdateGridColumnSizes;
var
  i: integer;
begin
  if FDefaultColWidths then
  begin
    if xdgIndicator in Options then
      ColWidths[0] := IndicatorWidth;
    if NeedAutoSizeColumns then
      UpdateAutoSizeColumns
    else
      for i := FirstGridColumn to ColCount - 1 do
        ColWidths[i] := GetColumnWidth(i);
  end;
end;

procedure TCustomXDBGrid.UpdateScrollbarRange;
var
  aRange, aPage, aPos: integer;
  ScrollInfo: TScrollInfo;
begin
  if not HandleAllocated then
    exit;

  GetScrollBarParams(aRange, aPage, aPos);

  FillChar(ScrollInfo, SizeOf(ScrollInfo), 0);
  ScrollInfo.cbSize := SizeOf(ScrollInfo);

  {TODO: try to move this out}
  {$ifdef WINDOWS}
  ScrollInfo.fMask := SIF_ALL or SIF_DISABLENOSCROLL;
  ScrollInfo.ntrackPos := 0;
  {$else}
  ScrollInfo.fMask := SIF_ALL or SIF_UPDATEPOLICY;
  //ScrollInfo.ntrackPos := SB_POLICY_CONTINUOUS;
  ScrollInfo.ntrackPos := SB_POLICY_DISCONTINUOUS;
  {$endif}
  ScrollInfo.nMin := 0;
  ScrollInfo.nMax := aRange;
  ScrollInfo.nPos := Min(aPos, aRange - aPage);
  ScrollInfo.nPage := aPage;
  // the redraw argument of SetScrollInfo means under gtk
  // if the scrollbar is visible or not, in windows it
  // seems to mean if the scrollbar is redrawn or not
  // to reflect the scrollbar changes made
  SetScrollInfo(Handle, SB_VERT, ScrollInfo,
    (ScrollBars in [ssBoth, ssVertical]) or
    ((Scrollbars in [ssAutoVertical, ssAutoBoth]) and (aRange > aPAge))
    );
  FOldPosition := aPos;
  {$ifdef dbgDBGrid}
  DebugLn('UpdateScrollBarRange: Handle=', IntToStr(Handle),
    ' aRange=', IntToStr(aRange),
    ' aPage=', IntToStr(aPage), ' aPos=', IntToStr(aPos));
  {$endif}
end;

procedure TCustomXDBGrid.DoLayoutChanged;
begin
  if csDestroying in ComponentState then
    exit;
  {$ifdef dbgDBGrid}
  DebugLn('doLayoutChanged INIT');
{$endif}
  BeginUpdate;
  if UpdateGridCounts = 0 then
    EmptyGrid;
  EndUpdate;
  UpdateScrollbarRange;
  {$ifdef dbgDBGrid}
  DebugLn('doLayoutChanged FIN');
{$endif}
end;

{
procedure TCustomXDBGrid.WriteColumns(Writer: TWriter);
begin
  if Columns.IsDefault then
    Writer.WriteCollection(nil)
  else
    Writer.WriteCollection(Columns);
end;
}
procedure TCustomXDBGrid.RestoreEditor;
begin
  if EditorMode then
  begin
    EditorMode := False;
    EditorMode := True;
  end;
end;

function TCustomXDBGrid.ISEOF: boolean;
begin
  with FDatalink do
    Result :=
      Active and DataSet.EOF;
end;

function TCustomXDBGrid.ValidDataSet: boolean;
begin
  Result := FDatalink.Active and (FDatalink.DataSet <> nil);
end;

function TCustomXDBGrid.InsertCancelable: boolean;
begin
  with FDatalink.DataSet do
    Result := (State = dsInsert) and not (Modified or FDataLink.FModified);
end;

procedure TCustomXDBGrid.StartUpdating;
begin
  if not UpdatingData then
  begin
    {$ifdef dbgDBGrid}
    DebugLn('DBGrid.StartUpdating');
{$endif}
    Include(FGridStatus, gsUpdatingData);
    FOldControlStyle := ControlStyle;
    ControlStyle := ControlStyle + [csActionClient];
    LockEditor;
  end
  else
    {$ifdef dbgDBGrid}
    DebugLn('WARNING: multiple call to StartUpdating');
{$endif}
end;

procedure TCustomXDBGrid.EndUpdating;
begin
  {$ifdef dbgDBGrid}
  DebugLn('DBGrid.EndUpdating');
{$endif}
  Exclude(FGridStatus, gsUpdatingData);
  ControlStyle := FOldControlStyle;
  UnLockEditor;
  if csActionClient in ControlStyle then
    DebugLn('WARNING: still got csActionClient');
end;

function TCustomXDBGrid.UpdatingData: boolean;
begin
  Result := gsUpdatingData in FGridStatus;
end;

procedure TCustomXDBGrid.AddAutomaticColumns;
var
  i: integer;
  F: TField;
begin
  // add as many columns as there are fields in the dataset
  // do this only at runtime.
  if (csDesigning in ComponentState) or not FDatalink.Active or
    (gsRemovingAutoColumns in FGridStatus) or not
    (dgeAutoColumns in OptionsExtra) then
    exit;
  Include(FGridStatus, gsAddingAutoColumns);
  try
    for i := 0 to FDataLink.DataSet.FieldCount - 1 do
    begin

      F := FDataLink.DataSet.Fields[i];

      if TxDBGridColumns(Columns).ColumnFromField(F) <> nil then
        // this field is already in the collection. This could only happen
        // if AddAutomaticColumns was called out of LayoutChanged.
        // to avoid duplicate columns skip this field.
        continue;

      if (F <> nil) then
      begin
        with TxDBGridColumns(Columns).Add do
        begin
          FIsAutomaticColumn := True;
          Field := F;
          Visible := F.Visible;
        end;
      end;

    end;
    // honor the field.index
    TxDBGridColumns(Columns).ResetColumnsOrder(coFieldIndexOrder);
  finally
    Exclude(FGridStatus, gsAddingAutoColumns);
  end;

end;

procedure TCustomXDBGrid.AssignTo(Dest: TPersistent);
begin
  if Dest is TCustomXDBGrid then
  begin
    // TODO
  end
  else
    inherited AssignTo(Dest);
end;

procedure TCustomXDBGrid.UpdateAutoSizeColumns;
var
  ACol, EndCol, w: integer;
  ColWidth, AllWidth: integer;
  C: TxColumn;
begin

  if gsAutoSized in GridStatus then
    exit;

    w := ClientWidth;
    EndCol := Columns.Count - 1;
    AllWidth := 0;
    if xdgIndicator in Options then
      AllWidth := IndicatorWidth;
    for ACol := 0 to EndCol do
    begin
      C := Columns[ACol];
      if C.PrcWidth > 0 then
      begin
        if (ACol = EndCol) and (AllWidth < W) then
          ColWidth := W - AllWidth
        else
        begin
          ColWidth := RoundToInt(W / 100 * C.PrcWidth);
          AllWidth := AllWidth + ColWidth;
        end;
        C.Width := ColWidth;
      end;
    end;

  {BeginLayout;

  DatalinkActive := FDatalink.Active;
  if DatalinkActive then
    CurActiveRecord := FDatalink.ActiveRecord;

  tmpCanvas := GetWorkingCanvas(Canvas);
  try
    for aCol:=FixedCols to ColCount-1 do begin

      Field := GetFieldFromGridColumn(ACol);
      C := ColumnFromGridColumn(ACol);

      if (C<>nil) and (C.Title<>nil) then begin
        tmpCanvas.Font := C.Title.Font;
        ColWidth := tmpCanvas.TextWidth(trim(C.Title.Caption));
        tmpCanvas.Font := C.Font;
      end else begin
        if (Field<>nil) then begin
          tmpCanvas.Font := TitleFont;
          ColWidth := tmpCanvas.TextWidth(Field.FieldName);
        end
        else
          ColWidth := 0;
        tmpCanvas.Font := Font;
      end;

      if (Field<>nil) and DatalinkActive then
        for ARow := FixedRows to RowCount-1 do begin

          FDatalink.ActiveRecord := ARow - FixedRows;

          if Field.dataType<>ftBlob then
            s := trim(Field.DisplayText)
          else
            s := '(blob)';
          w := tmpCanvas.TextWidth(s);
          if w>ColWidth then
            ColWidth := w;

        end;

      if ColWidth=0 then
        ColWidth := GetColumnWidth(ACol);

      ColWidths[ACol] := ColWidth + 15;
    end;
  finally
    if TmpCanvas<>Canvas then
      FreeWorkingCanvas(tmpCanvas);

    if DatalinkActive then
      FDatalink.ActiveRecord := CurActiveRecord;

    include(FGridStatus, gsAutoSized);

    EndLayout;
  end;}

end;

procedure TCustomXDBGrid.SwapCheckBox;
var
  SelField: TField;
  TempColumn: TxColumn;
begin
  if not GridCanModify then
    exit;

  SelField := SelectedField;
  TempColumn := TxColumn(ColumnFromGridColumn(Col));
  if (SelField <> nil) and (TempColumn <> nil) and not TempColumn.ReadOnly and
    FDatalink.Edit then
  begin
    if SelField.DataType = ftBoolean then
    begin
      SelField.AsBoolean := not SelField.AsBoolean;
    end
    else
    begin
      if TempColumn.ValueChecked = SelField.AsString then
        SelField.AsString := TempColumn.ValueUnchecked
      else
        SelField.AsString := TempColumn.ValueChecked;

    end;
    InvalidateCell(Col, Row);
  end;
end;

procedure TCustomXDBGrid.ToggleSelectedRow;
begin
  SelectRecord(not FSelectedRows.CurrentRowSelected);
end;

procedure TCustomXDBGrid.LinkActive(Value: boolean);
begin
  if not Value then
  begin
    FSelectedRows.Clear;
    RemoveAutomaticColumns;
  end;
  LayoutChanged;
end;

procedure TCustomXDBGrid.LayoutChanged;
begin
  if csDestroying in ComponentState then
    exit;
  if FLayoutChangedCount = 0 then
  begin
    BeginLayout;
    if Columns.Count > 0 then
    begin
      TxDBGridColumns(Columns).LinkFields;
    end
    else if not FDataLink.Active then
      FDataLink.BufferCount := 0
    else
      AddAutomaticColumns;
    EndLayout;
  end;
end;

procedure TCustomXDBGrid.Loaded;
begin
  LayoutChanged;
  inherited Loaded;
end;

type
  TProtFields = class(TFields)
  end;

procedure TCustomXDBGrid.ColRowMoved(IsColumn: boolean; FromIndex, ToIndex: integer);
var
  F: TField;
begin
  if IsColumn then
  begin
    if Columns.Enabled then
      inherited ColRowMoved(IsColumn, FromIndex, ToIndex)
    else if FDatalink.Active and (FDataLink.DataSet <> nil) then
    begin
      F := GetDsFieldFromGridColumn(FromIndex);
      if F <> nil then
      begin
        TProtFields(FDatalink.DataSet.Fields).SetFieldIndex(
          F, ToIndex - FirstGridColumn);
      end;
    end;
    if Assigned(OnColumnMoved) then
      OnColumnMoved(Self, FromIndex, ToIndex);
  end;
end;

function TCustomXDBGrid.ColumnEditorStyle(aCol: integer; F: TField): TxColumnButtonStyle;
var
  gridcol: TxGridColumn;
begin
  Result := cbsxAuto;
  gridcol := ColumnFromGridColumn(aCol);
  if Columns.Enabled and assigned(gridcol) then
    Result := gridcol.ButtonStyle;

  Result := DefaultEditorStyle(Result, F);
end;

function TCustomXDBGrid.CreateColumns: TxGridColumns;
begin
  Result := TxDBGridColumns.Create(Self, TxColumn);
end;

procedure TCustomXDBGrid.CreateWnd;
begin
  inherited CreateWnd;
  LayoutChanged;
  if Scrollbars in [ssBoth, ssVertical, ssAutoBoth, ssAutoVertical] then
    ScrollBarShow(SB_VERT, True);
end;

procedure TCustomXDBGrid.DblClick;
begin
  {if (SelectAction <> nil) and (SelectAction.OnExecute <> nil)
    and (FDataLink.DataSet <> nil) and not FDataLink.DataSet.IsEmpty then
    SelectAction.Execute
  else
  if (EditAction <> nil) and (FDataLink.DataSet <> nil) then
      EditAction.Execute
  else}
  inherited DblClick;
end;

procedure TCustomXDBGrid.DefineProperties(Filer: TFiler);
  {
  function HasColumns: boolean;
  var
    C: TxGridColumns;
  begin
    if Filer.Ancestor <> nil then
      C := TxCustomGrid(Filer.Ancestor).Columns
    else
      C := Columns;
    if C<>nil then
      result := not C.IsDefault
    else
      result := false;
  end;
  }
begin
  // simply avoid to call TCustomXGrid.DefineProperties method
  // which defines ColWidths,Rowheights,Cells
  //Filer.DefineProperty('Columns',  @ReadColumns,  @WriteColumns,  HasColumns);
end;

procedure TCustomXDBGrid.DefaultDrawCell(aCol, aRow: integer; aRect: TRect;
  aState: TxGridDrawState);

var
  S: string;
  F, ExF: TField;
  cbs, xcbs: TxColumnButtonStyle;
  R, ExR: TRect;
  ExColumn: TxColumn;
  ColIndex: integer;
  ts: TTextStyle;
  DrawImage, xDrawImage: boolean;
  WordWrap, xWordWrap: boolean;
begin
  R := aRect;
  DrawImage := False;
  xDrawImage := False;
  WordWrap := False;
  xWordWrap := False;
  if (xStyle = xgsNormal) and (aCol > 0) then
    Inc(R.Left, 1);
  // background
  if ((xgdFixed in aState) and
     ((aRow = 0) or ((aCol = 0) and (xdgIndicator in Options)))) and
     (TitleStyle = tsxNative) then
  begin
    R := aRect;
    if xStyle = xgsNormal then
      Inc(R.Right);
    DrawThemedCell(aCol, aRow, R, aState);
  end
  else
  begin
    Canvas.FillRect(aRect);
  end;

  if (xgdFixed in aState) and
     (((aCol = 0) and (xdgIndicator in Options)) or (aRow = 0)) then
    DrawFixedText(aCol, aRow, aRect, aState)
  else
  if not FDrawingEmptyDataset then
  begin
    ColIndex := ColumnIndexFromGridColumn(aCol);
    ExColumn := nil;
    F := nil;
    ExF := nil;
    if ColIndex >= 0 then
    begin
      ExColumn := Columns[ColIndex];
      F := ExColumn.Field;
      ExF := ExColumn.xField;
      WordWrap := ExColumn.WordWrap;
      DrawImage := ExColumn.DrawImage;
    end
    else
    begin
      F := GetFieldFromGridColumn(aCol);
    end;

    if ExF <> nil then
    begin
      ExR := R;
      R.Bottom := R.Top + (R.Bottom - R.Top) div 2;
      ExR.Top := R.Bottom;
      xcbs := ColumnEditorStyle(aCol, ExF);
      if ExColumn <> nil then
      begin
        xDrawImage := ExColumn.xDrawImage;
        xWordWrap := ExColumn.xWordWrap;
      end;
    end;

    if DrawImage or xDrawImage then
    begin
      if DrawImage then
        DrawCellImage(R, ExColumn, F);
      if xDrawImage and Assigned(ExF) then
        xDrawCellImage(ExR, ExColumn, ExF);
    end
    else
    begin

      cbs := ColumnEditorStyle(aCol, F);
      case cbs of
        cbsxCheckBoxColumn:
          DrawCheckBoxBitmaps(aCol, R, F);
        else
        begin

          if cbs = cbsxButtonColumn then
            DrawButtonCell(aCol, aRow, R, aState);

          {$ifdef dbggridpaint}
          DbgOut(' Col=%d', [ACol]);
          {$endif}
          if F <> nil then
          begin
            {$ifdef dbgGridPaint}
            DbgOut(' Field=%s', [F.FieldName]);
            {$endif}
            if F.dataType <> ftBlob then
              S := F.DisplayText
            else
              S := '(blob)';
          end
          else
            S := '';
           {$ifdef dbggridpaint}
          DbgOut(' Value=%s ', [S]);
            {$endif}

          xDrawCellText(aCol, aRow, R, aState, S, WordWrap);
        end;
      end;
    end;
    if ExF <> nil then
    begin
      if xDrawImage then
      begin
        DrawCellImage(ExR, ExColumn, ExF);
      end
      else
      begin
        case xcbs of
          cbsxCheckBoxColumn:
            xDrawCheckBoxBitmaps(aCol, aRow, ExR, ExF, ExColumn.xAlignment);
          else
          begin

            if cbs = cbsxButtonColumn then
              DrawButtonCell(aCol, aRow, ExR, aState);

            {$ifdef dbggridpaint}
            DbgOut(' Col=%d', [ACol]);
            {$endif}
            if ExF <> nil then
            begin
              {$ifdef dbgGridPaint}
              DbgOut(' Field=%s', [ExF.FieldName]);
              {$endif}
              if ExF.dataType <> ftBlob then
                S := ExF.DisplayText
              else
                S := '(blob)';
            end
            else
              S := '';

            {$ifdef dbggridpaint}
            DbgOut(' Value=%s ', [S]);
            {$endif}

            ts := Canvas.TextStyle;
            ts.Alignment := ExColumn.xAlignment;
            Canvas.TextStyle := ts;
            xDrawCellText(aCol, aRow, ExR, aState, S, xWordWrap);
          end;
        end;

      end;
    end;
  end;
end;

function TCustomXDBGrid.DefaultEditorStyle(const Style: TxColumnButtonStyle;
  const F: TField): TxColumnButtonStyle;
begin
  Result := Style;
  if (Result = cbsxAuto) and (F <> nil) then
    case F.DataType of
      ftBoolean: Result := cbsxCheckBoxColumn;
    end;
  if (Result = cbsxCheckBoxColumn) and not (dgeCheckboxColumn in FExtraOptions) then
    Result := cbsxAuto;
end;

procedure TCustomXDBGrid.DoEditorShow;
var
  F: TField;
  ACol: TxGridColumn;
  R: Boolean;
begin
  inherited DoEditorShow;
  F := GetCurrentField;

  R := F.ReadOnly;
  if not R then
  begin
    ACol := GetCurrentColumn;
    R := ACol.ReadOnly;
  end;
  if F <> nil then
  begin
    if Editor is TxStringCellEditor  then
      TxStringCellEditor(Editor).ReadOnly := R
    else
    if Editor is TxCompositeCellEditor then
      TxCompositeCellEditor(Editor).ReadOnly := R
    else
    if Editor is TxPickListCellEditor then
      TxPickListCellEditor(Editor).ReadOnly := R;
  end;
end;


procedure TCustomXDBGrid.DoOnChangeBounds;
begin
  BeginUpdate;
  inherited DoOnChangeBounds;
  if HandleAllocated then
    LayoutChanged;
  EndUpdate;
end;

procedure TCustomXDBGrid.DoPrepareCanvas(aCol, aRow: integer; aState: TxGridDrawState);
var
  DataCol: integer;
begin
  if Assigned(OnPrepareCanvas) and (ARow >= FixedRows) then
  begin
    DataCol := ColumnIndexFromGridColumn(aCol);
    if DataCol >= 0 then
      OnPrepareCanvas(Self, DataCol, TxColumn(Columns[DataCol]), aState);
  end;
end;

procedure TCustomXDBGrid.BeforeMoveSelection(const DCol, DRow: integer);
begin
  {$ifdef dbgDBGrid}
  DebugLn('DBGrid.BefMovSel INIT');
{$endif}
  inherited BeforeMoveSelection(DCol, DRow);
  if DCol <> Col then
  begin
    if assigned(OnColExit) then
      OnColExit(Self);
    FColEnterPending := True;
  end;
{$ifdef dbgDBGrid}
  DebugLn('DBGrid.BefMovSel END');
{$endif}
end;

procedure TCustomXDBGrid.HeaderClick(IsColumn: boolean; index: integer; P: TPoint);
var
  Column: TxColumn;
  CP: TCellCursorPos;
  R: TRect;
begin
  if (Assigned(OnTitleClick) and IsColumn)
    or (FAutoSort and Assigned(AutoSortDataSet)) then
  begin
    Column := TxColumn(ColumnFromGridColumn(Index));
    if Column <> nil then
    begin
      CP := ccpTop;
      if Column.xFieldName <> '' then
      begin
        R := CellRect(index, 0);
        R.Top := R.Top + (R.Bottom - R.Top) div 2;
        if PtInRect(R, P) then
          CP := ccpBottom;
      end;
      if Assigned(OnTitleClick) then
         OnTitleClick(Column, CP);
      if FAutoSort then
        SortDataSet(Column, CP);
    end;
  end;
end;

procedure TCustomXDBGrid.KeyDown(var Key: word; Shift: TShiftState);
type
  TOperation = (opMoveBy, opCancel, opAppend, opInsert, opDelete);
var
  DeltaCol, DeltaRow, AColIndex, L: integer;
  xColumn: TxColumn;
  AReadOnly: Boolean;

  procedure DoOnKeyDown;
  begin
    {$ifdef dbgGrid}
    DebugLn('DoOnKeyDown INIT');
{$endif}
    if Assigned(OnKeyDown) then
      OnKeyDown(Self, Key, Shift);
    {$ifdef dbgGrid}
    DebugLn('DoOnKeyDown FIN');
{$endif}
  end;

  {$ifdef dbgGrid}
  function OperToStr(AOper: TOperation): string;
  begin
    case AOper of
      opMoveBy: Result := 'opMoveBy';
      opCancel: Result := 'opCancel';
      opAppend: Result := 'opAppend';
      opInsert: Result := 'opInsert';
      opDelete: Result := 'opDelete';
    end;
  end;

  {$endif}
  procedure DoOperation(AOper: TOperation; Arg: integer = 0);
  begin
    {$IfDef dbgGrid}
    DebugLn('KeyDown.DoOperation(%s,%d) INIT', [OperToStr(AOper), arg]);
{$Endif}
    GridFlags := GridFlags + [gfxEditingDone];
    case AOper of
      opMoveBy:
        FDatalink.MoveBy(Arg);
      opCancel:
      begin
        if EditorMode then
          EditorCancelEditing;
        FDatalink.Dataset.Cancel;

      end;
      opAppend:
        FDatalink.Dataset.Append;
      opInsert:
        FDatalink.Dataset.Insert;
      opDelete:
        FDatalink.Dataset.Delete;
    end;
    GridFlags := GridFlags - [gfxEditingDone];
    {$IfDef dbgGrid}
    DebugLn('KeyDown.DoOperation(%s,%d) DONE', [OperToStr(AOper), arg]);
{$Endif}
  end;

  procedure SelectNextRec(const AStart, ADown: boolean);
  var
    N: integer;
    CurBookmark: TBookmark;
  begin
    if xdgPersistentMultiSelect in Options then
      exit;

    if (ssShift in Shift) then
    begin

      CurBookmark := FDatalink.DataSet.GetBookmark;
      if FKeyBookmark = nil then
        FKeyBookmark := CurBookmark;

      if (FKeyBookmark = CurBookmark) then
      begin
        if AStart then
        begin
          SelectRecord(True);
          if ADown then
            FKeySign := 1
          else
            FKeySign := -1;
          exit;
        end;
        FKeySign := 0;
      end;

      n := 4 * Ord(FKeySign >= 0) + 2 * Ord(ADown) + 1 * Ord(AStart);
      case n of
        0, 6, 8..11:
        begin
          SelectRecord(True);
        end;
        3, 5:
        begin
          SelectRecord(False);
        end;
      end;
    end
    else
      ClearSelection(True);
  end;

  function doVKDown: boolean;
  var
    xColumn: TxColumn;
    ColIndex: integer;
    F: TField;
    ApplyPos: boolean;
  begin
    {$ifdef dbgGrid}
    DebugLn('DoVKDown INIT');
{$endif}
    ColIndex := ColumnIndexFromGridColumn(Col);
    if ColIndex >= 0 then
    begin
      xColumn := Columns[ColIndex];
      F := xColumn.xField;
    end;

    if (F <> nil) and (CellCursorPos = ccpTop) then
    begin
      ApplyPos := True;
      GridFlags := GridFlags + [gfxEditingDone];
      if EditorMode then
      begin

        if not EditorGetValue(True) then
          ApplyPos := False;
      end;
      if ApplyPos then
      begin
        FCellCursorPos := ccpBottom;
        InvalidateCell(Col, Row);
        SelectEditor;
        SetFocus;
      end;
      GridFlags := GridFlags - [gfxEditingDone];
    end
    else
    begin
      //if F <> nil then


      if InsertCancelable then
      begin
        if IsEOF then
          Result := True
        else
        begin
          doOperation(opCancel);
          Result := False;
        end;
      end
      else
      begin
        Result := False;
        SelectNextRec(True, True);
        doOperation(opMoveBy, 1);
        if GridCanModify and FDataLink.EOF then
        begin
          if not (xdgDisableInsert in Options) then
            doOperation(opAppend);
        end
        else
          SelectNextRec(False, True);
      end;
      FCellCursorPos := ccpTop;
    end;
    {$ifdef dbgGrid}
    DebugLn('DoVKDown FIN');
{$endif}
  end;

  function DoVKUP: boolean;
  var
    Column: TxColumn;
    ColIndex: integer;
    F: TField;
    ApplyPos: boolean;
  begin
    {$ifdef dbgGrid}
    DebugLn('DoVKUP INIT');
{$endif}
    ColIndex := ColumnIndexFromGridColumn(Col);
    if ColIndex >= 0 then
    begin
      Column := Columns[ColIndex];
      F := Column.xField;
    end;

    if (F <> nil) and (CellCursorPos = ccpBottom) then
    begin
      GridFlags := GridFlags + [gfxEditingDone];
      ApplyPos := True;
      if EditorMode then
      begin
        if not EditorGetValue(True) then
          ApplyPos := False;
      end;

      if ApplyPos then
      begin
        FCellCursorPos := ccpTop;
        InvalidateCell(Col, Row);
        SelectEditor;
        Result := FDatalink.DataSet.BOF;
        SetFocus;
      end;
      GridFlags := GridFlags - [gfxEditingDone];
    end
    else
    begin

      if InsertCancelable then
        doOperation(opCancel)
      else
      begin
        SelectNextRec(True, False);
        doOperation(opMoveBy, -1);
        SelectNextRec(False, False);
      end;

      if F <> nil then
        FCellCursorPos := ccpBottom
      else
        FCellCursorPos := ccpTop;
    end;
    Result := FDatalink.DataSet.BOF;
    {$ifdef dbgGrid}
    DebugLn('DoVKUP FIN');
{$endif}
  end;

  procedure MoveSel(AReset: boolean);
  begin
    if (DeltaCol <> 0) or (DeltaRow <> 0) then
    begin
      if DeltaRow > 0 then
      begin
        if doVKDown then;
          //DeltaCol:=0; // tochk: strict? already in EOF, don't change column
      end
      else
      if DeltaRow < 0 then
      begin
        if doVKUP then;
          //DeltaCol:=0; // tochk: strict? already in BOF, don't change column
      end;
      GridFlags := GridFlags + [gfxEditingDone];
      if (DeltaCol <> 0) then
      begin
        Col := Col + DeltaCol;
        FCellCursorPos := ccpTop;
      end;
      GridFlags := GridFlags - [gfxEditingDone];
    end
    else
    if AReset then
      ResetEditor;
  end;

  procedure DoValueToClipboard;
  var
    CS: string;
  begin
    Key := 0;
    if Datalink.Active and (SelectedField <> nil) then
    begin
      try
        CS := SelectedField.AsString;
        Clipboard.AsText := Trim(CS);
      except
        ShowMessage('Не удалось скопировать в буфер');
      end;
    end;

  end;

begin
  {$IfDef dbgGrid}
  DebugLn('DBGrid.KeyDown %s INIT Key=%d', [Name, Key]);
{$Endif}
  case Key of

    VK_ADD:
    begin
      if ssCtrl in Shift then
        IncFontValue(1)
      else
        inherited KeyDown(Key, Shift);
    end;

    VK_SUBTRACT:
    begin
      if (ssCtrl in Shift) and (Font.Size > 0) then
        IncFontValue(-1)
      else
        inherited KeyDown(Key, Shift);
    end;

    VK_TAB:
    begin
      doOnKeyDown;
      if (Key <> 0) and ValidDataset then
      begin
        if xdgTabs in Options then
        begin

          if ((ssShift in shift) and (Col <= GetFirstVisibleColumn) and
            (Row <= GetFirstVisibleRow)) then
          begin
            if EditorKey then
              GridFlags := GridFlags + [gfxRevEditorTab];
            exit;
          end;

          GetDeltaMoveNext(ssShift in Shift, DeltaCol, DeltaRow);

          if (not (ssShift in Shift)) and (Row >= GetLastVisibleRow) and
            (DeltaRow > 0) and (Col = GetLastVisibleColumn) and
            (FDatalink.Editing or not GridCanModify) then
          begin
            exit;
          end;

          MoveSel(False);
          Key := 0;
        end;
      end;
    end;

    VK_RETURN:
    begin
      doOnKeyDown;
      if (Key <> 0) and ValidDataset then
      begin
        key := 0;
        if (xdgEditing in Options) and not EditorMode then
          EditorMode := True
        else
        begin
          AColIndex := ColumnIndexFromGridColumn(Col);
          if AColIndex >= 0 then
            xColumn := Columns[AColIndex];

          if ssCtrl in Shift then
          begin
            if (xColumn <> nil) and ((xColumn.ButtonStyle = cbsxEllipsis) or
              (xColumn.ButtonStyle = cbsxButton)) then
            begin
              DoEditButtonClick(Col, Row);
            end;
          end
          else
          begin
            if (SelectAction <> nil) and (SelectAction.OnExecute <> nil)
              and (FDataLink.DataSet <> nil) and not FDataLink.DataSet.IsEmpty then
              SelectAction.Execute
            else
            if EditorMode then
            if (xColumn <> nil) and (xColumn.xField <> nil) and
              (CellCursorPos = ccpTop) then
            begin
              GridFlags := GridFlags + [gfxEditingDone];
              if EditorGetValue(True) then
              begin
                FCellCursorPos := ccpBottom;
                SelectEditor;
                InvalidateCell(Col, Row);
                SetFocus;
              end;
              GridFlags := GridFlags - [gfxEditingDone];
            end
            else
            begin
              GetDeltaMoveNext(ssShift in Shift, DeltaCol, DeltaRow);
              MoveSel(True);
            end;
          end;
        end;
      end;
    end;

    VK_DELETE:
    begin
      doOnKeyDown;
      if (Key <> 0) and (ssCtrl in Shift) and GridCanDelete then
      begin
        if DeleteAction <> nil then
        begin
          DeleteAction.Execute;
          Key := 0;
        end
        else
        if not (xdgConfirmDelete in Options) or
          (MessageDlg(rsDeleteRecord, mtConfirmation, mbOKCancel, 0) <> mrCancel)
        then
        begin
          doOperation(opDelete);
          key := 0;
        end;

      end;
    end;

    VK_DOWN:
    begin
      DoOnKeyDown;
      if (Key <> 0) and ValidDataset then
      begin
        doVKDown;
        Key := 0;
      end;
    end;

    VK_UP:
    begin
      doOnKeyDown;
      if (Key <> 0) and ValidDataset then
      begin
        doVKUp;
        key := 0;
      end;
    end;

    VK_RIGHT, VK_LEFT:
    begin
      FCellCursorPos := ccpTop;
      if not EditorMode then
        inherited KeyDown(Key, Shift);
    end;

    VK_NEXT:
    begin
      doOnKeyDown;
      if (Key <> 0) and ValidDataset then
      begin
        doOperation(opMoveBy, VisibleRowCount);
        ClearSelection(True);
        Key := 0;
      end;
    end;

    VK_PRIOR:
    begin
      doOnKeyDown;
      if (Key <> 0) and ValidDataset then
      begin
        doOperation(opMoveBy, -VisibleRowCount);
        ClearSelection(True);
        key := 0;
      end;
    end;

    VK_ESCAPE:
    begin
      doOnKeyDown;
      if (Key <> 0) and ValidDataset then
      begin
        if EditorMode then
        begin
          GridFlags := GridFlags + [gfxEditingDone];
          EditorCancelEditing;
          if FDatalink.Active and not FDatalink.Dataset.Modified then
            FDatalink.Modified := False;
          Key := 0;
          GridFlags := GridFlags - [gfxEditingDone];
        end
        else
        if FDataLink.Active then
          doOperation(opCancel);
      end;
    end;

    VK_INSERT:
    begin
      doOnKeyDown;
      if (Key <> 0) and (Shift = []) then
        if (InsertAction <> nil) and (FDataLink.DataSet <> nil) then
        begin
          Key := 0;
          InsertAction.Execute;
        end
        else
        if not (xdgDisableInsert in Options) and GridCanModify then
        begin
          doOperation(opInsert);
          Key := 0;
        end
        else inherited KeyDown(Key, Shift);
    end;

    VK_F4:
    begin
      DoOnKeyDown;
      if (Key <> 0) and (Shift = []) then
      begin
        if (EditAction <> nil) and (Datalink.DataSet <> nil) then EditAction.Execute
        else inherited KeyDown(Key, Shift);
      end;
    end;

    VK_HOME:
    begin
      doOnKeyDown;
      if Key <> 0 then
      begin
        if FDatalink.Active then
        begin
          GridFlags := GridFlags + [gfxEditingDone];
          if ssCTRL in Shift then
            FDataLink.DataSet.First
          else
            MoveNextSelectable(False, FixedCols, Row);
          GridFlags := GridFlags - [gfxEditingDone];
          ClearSelection(True);
          Key := 0;
        end;
      end;
    end;

    VK_END:
    begin
      doOnKeyDown;
      if Key <> 0 then
      begin
        if FDatalink.Active then
        begin
          GridFlags := GridFlags + [gfxEditingDone];
          if ssCTRL in shift then
            FDatalink.DataSet.Last
          else
            MoveNextSelectable(False, ColCount - 1, Row);
          GridFlags := GridFlags - [gfxEditingDone];
          ClearSelection(True);
          Key := 0;
        end;
      end;
    end;

    VK_SPACE:
    begin
      doOnKeyDown;
      if (Key <> 0) and ValidDataset then
      begin
        if GridCanModify and (ColumnEditorStyle(Col, SelectedField) = cbsxCheckBoxColumn) then
        begin
          SwapCheckBox;
          Key := 0;
        end;
      end;
    end;

    VK_MULTIPLY:
    begin
      doOnKeyDown;
      if (Key <> 0) and ValidDataset and (ssCtrl in Shift) then
        ToggleSelectedRow;
    end;

    VK_BACK:
    begin
      DoOnKeyDown;
      if (xdgAutoLocate in Options) and (xdgRowSelect in Options) then
      begin
        if FLocateStr <> '' then
        begin
          L := UTF8Length(FLocateStr);
          LocateStr := UTF8Copy(FLocateStr, 1, L - 1);
        end;
      end;
    end;

    67, 83: {Ctrl+C}
    begin
      if (Shift = [ssCtrl]) and not EditorMode then
      begin
        DoValueToClipboard;
        Key := 0;
      end
      else
        inherited KeyDown(Key, Shift);
    end;

    70, 65: {Ctrl+F}
    begin
      if FUseAutoFilter and (Shift = [ssCtrl]) and not EditorMode then
      begin
        ShowFilterForm(nil, '', '');
        Key := 0;
      end
      else
        inherited KeyDown(Key, Shift);
    end

    else
    begin
      inherited KeyDown(Key, Shift);
      if EditorMode and (FDataLink.FDataSet <> nil) and not ReadOnly then
      begin
        AReadOnly:= EditorIsReadOnly;
        if (FDataLink.DataSet.State = dsBrowse) and AReadOnly = False then
          FDataLink.DataSet.Edit;
      end;
    end;
  end;
  {$IfDef dbgGrid}
  DebugLn('DBGrid.KeyDown END Key= %d', [Key]);
{$Endif}
end;

procedure TCustomXDBGrid.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: integer);
var
  Gz: TxGridZone;
  P: TPoint;
  F: TField;
  R, ExR: TRect;
  NewCellCursorPos: TCellCursorPos;
  ApplyNewPos, AFocused: boolean;

  procedure doMouseDown;
  begin
    if assigned(OnMouseDown) then
      OnMouseDown(Self, Button, Shift, X, Y);
  end;

  procedure doInherited;
  begin
    inherited MouseDown(Button, Shift, X, Y);
  end;

  procedure doMoveBy;
  begin
    {$IfDef dbgGrid}
    DebugLn('DBGrid.MouseDown MoveBy INIT');
{$Endif}
    FDatalink.MoveBy(P.Y - Row);
    {$IfDef dbgGrid}
    DebugLn('DBGrid.MouseDown MoveBy END');
{$Endif}
  end;

  procedure doMoveToColumn;
  begin
    {$IfDef dbgGrid}
    DebugLn('DBGrid.MouseDown MoveToCol INIT Col=', IntToStr(P.X));
{$Endif}
    Col := P.X;
    {$IfDef dbgGrid}
    DebugLn('DBGrid.MouseDown MoveToCol END');
{$Endif}
  end;

  procedure DoCancel;
  begin
    {$IfDef dbgGrid}
    DebugLn('DBGrid.MouseDown Dataset.CANCEL INIT');
{$Endif}
    if EditorMode then
      EditorCancelEditing;
    FDatalink.Dataset.cancel;
    {$IfDef dbgGrid}
    DebugLn('DBGrid.MouseDown Dataset.CANCEL FIN');
{$Endif}
  end;

  procedure DoAcceptValue;
  begin
    if EditorMode and FDatalink.FModified then
      EditorMode := False;
  end;

  function DoAutoEdit: boolean;
  begin
    Result := AutoEdit and EditingAllowed(Col) and (GCache.ClickCell.X = Col) and
      (GCache.ClickCell.Y = Row);
    if Result then
    begin
      SelectEditor;
      EditorShow(True);
    end;
  end;

begin
  {$ifdef dbgDBGrid}
  DebugLn('DBGrid.mousedown - INIT');
{$endif}

  if (csDesigning in componentState) {or not GCache.ValidGrid } then
    exit;

  if UpdatingData then
  begin
    {$ifdef dbgDBGrid}
    DebugLn('DBGrid.MouseDown - UpdatingData');
{$endif}
    exit;
  end;

  FDblClicked := False;
  if not MouseButtonAllowed(Button) then
  begin
    doInherited;
    exit;
  end;

  {$IfDef dbgGrid}
  DebugLn('DBGrid.MouseDown INIT');
{$Endif}
  Gz := MouseToGridZone(X, Y);
  CacheMouseDown(X, Y);
  case Gz of
    xgzFixedCells, xgzFixedCols:
      doInherited;
    else
    begin

      FKeyBookmark := nil; // force new keyboard selection start
      SetFocus;

      P := MouseToCell(Point(X, Y));
      NewCellCursorPos := CellCursorPos;

      if Gz = xgzNormal then
      begin
        if ssDouble in Shift then FDblClicked:= True;

        F := GetxFieldFromGridColumn(GCache.ClickCell.x);
        if F <> nil then
        begin
          R := CellRect(GCache.ClickCell.x, GCache.ClickCell.y);
          ExR := R;
          ExR.Top := R.Top + (R.Bottom - R.Top) div 2 - 1;
          R.Bottom := R.Top + 1;

          if PtInRect(ExR, GCache.ClickMouse) then
            NewCellCursorPos := ccpBottom
          else
            NewCellCursorPos := ccpTop;

          if (Col = GCache.ClickCell.x) and (Row = GCache.ClickCell.y) then
          begin
            if NewCellCursorPos <> CellCursorPos then
            begin
              ApplyNewPos := True;
              if EditorMode then
              begin
                if not EditorGetValue(True) then
                  ApplyNewPos := False;
              end;
              if ApplyNewPos then
              begin
                FCellCursorPos := NewCellCursorPos;
                InvalidateCell(GCache.ClickCell.x, GCache.ClickCell.y);
              end;
              Exit;
            end;
          end;
        end
        else
          NewCellCursorPos := ccpTop;
      end;

      if Gz = xgzFixedRows then
        P.X := Col;

      if P.Y = Row then
      begin
        //doAcceptValue;

        if ssCtrl in Shift then
        begin
          doMouseDown;
          ToggleSelectedRow;
        end
        else
        begin
          if Button = mbLeft then
            ClearSelection(True);
          if gz = xgzFixedRows then
            doMouseDown
          else
            doInherited;
        end;

      end
      else
      begin
        doMouseDown;
        if ValidDataSet then
        begin
          if InsertCancelable and IsEOF then
            doCancel;
          doMoveBy;
        end;
        if ssCtrl in Shift then
          ToggleSelectedRow
        else
        begin
          if Button = mbLeft then
            ClearSelection(True);
          doMoveToColumn;
        end;
      end;

      if FCellCursorPos <> NewCellCursorPos then
        FCellCursorPos := NewCellCursorPos;
    end;
  end;
  {$IfDef dbgGrid}
  DebugLn('DBGrid.MouseDown END');
{$Endif}
end;

procedure TCustomXDBGrid.MouseMove(Shift: TShiftState; X, Y: integer);
begin
  if fGridState = xgsSelecting then
    exit
  else
    inherited MouseMove(Shift, X, Y);
end;

procedure TCustomXDBGrid.MouseUp(Button: TMouseButton; Shift: TShiftState; X,
  Y: Integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
  if FDblClicked then
  begin
    FDblClicked := False;
    if (SelectAction <> nil) and (SelectAction.OnExecute <> nil)
    and (FDataLink.DataSet <> nil) and not FDataLink.DataSet.IsEmpty then
      SelectAction.Execute
    else
    if (EditAction <> nil) and (FDataLink.DataSet <> nil) then
        EditAction.Execute
  end;
end;

procedure TCustomXDBGrid.DoRowHighLight(aCol, aRow: Integer;
  aState: TxGridDrawState);
var
  F, ColumnField, xColumnField: TField;
  I: Integer;
  RHItem: TRowsHighLightItem;
  S: string;
  ApplyHL, ApplyCL: Boolean;
begin
  if FDataLink.Active and (FRowsHighLights.Count > 0) then
  begin
    ColumnField := GetFieldFromGridColumn(aCol);
    xColumnField := GetxFieldFromGridColumn(aCol);
    for I := 0 to FRowsHighLights.Count - 1 do
    begin
      ApplyHL := False;
      RHItem := FRowsHighLights.Items[I];
      F := nil;
      if not RHItem.CellOnly then
      begin
        F := FDataLink.DataSet.FindField(RHItem.FieldName);
        if F <> nil then
        begin
          S := RHItem.FieldValue;
          if S = 'NULL' then
          begin
            if F.IsNull then ApplyHL := True;
          end
          else
          if SameText(F.AsString, S) then
          begin
            ApplyHL := True;
          end;
          if ApplyHL then
          begin
            if RHItem.Color <> clNone then
                Canvas.Brush.Color := RHItem.Color;
            if RHItem.FontColor <> clNone then
                Canvas.Font.Color := RHItem.FontColor;
          end;
        end;
      end;
    end;

    for I := 0 to FRowsHighLights.Count - 1 do
    begin
      RHItem := FRowsHighLights.Items[I];
      F := nil;
      if RHItem.CellOnly then
      begin
        F := ColumnField;
        if Assigned(F) and not SameText(F.FieldName, RHItem.FieldName) then
        begin
          F := xColumnField;
          if Assigned(F) and not SameText(F.FieldName, RHItem.FieldName) then
            F := nil;
        end;

        if F <> nil then
        begin
          ApplyCL := False;
          S := RHItem.FieldValue;
          if (S = 'NULL') and F.IsNull then
            ApplyCL := True
          else
          if SameText(F.AsString, S) then
            ApplyCL := True;

          if ApplyCL then
          begin
            if RHItem.Color <> clNone then
                Canvas.Brush.Color := RHItem.Color;
            if RHItem.FontColor <> clNone then
                Canvas.Font.Color := RHItem.FontColor;
          end;
        end;
      end;
    end;
  end;
end;

procedure TCustomXDBGrid.PrepareCanvas(aCol, aRow: integer; aState: TxGridDrawState);
var
  CurrentTextStyle: TTextStyle;
begin


  inherited PrepareCanvas(aCol, aRow, aState);

  if xgdFixed in aState then
  begin
    if ((aCol = 0) and (xdgIndicator in Options)) or (aRow = 0) then
    begin
      if xgdHot in aState then
        Canvas.Brush.Color := FixedHotColor
      else
        Canvas.Brush.Color := GetColumnColor(aCol, xgdFixed in AState);
    end
    else
    begin
      SetCanvasFont(GetColumnFont(aCol, False));
      if aRow = Row then
      begin
        if Focused then
        begin
          Canvas.Brush.Color:= SelectedColor;
          Canvas.Font.Color:= FxSelectedFontColor;
        end
        else
        begin
          Canvas.Brush.Color:= SelectedUnfocusedColor;
          Canvas.Font.Color:= clBlack;
        end;
      end
      else
        Canvas.Brush.Color := GetColumnColor(aCol, False);

      CurrentTextStyle := DefaultTextStyle;
      CurrentTextStyle.Alignment := BidiFlipAlignment(GetColumnAlignment(aCol, False), UseRightToLeftAlignment);
      CurrentTextStyle.Layout := GetColumnLayout(aCol, False);
      CurrentTextStyle.ShowPrefix := False;
      CurrentTextStyle.EndEllipsis := (xgoCellEllipsis in inherited Options);
      Canvas.TextStyle := CurrentTextStyle;
    end;
  end;

  if (not FDatalink.Active) and ((xgdSelected in aState) or (xgdFocused in aState)) then
    Canvas.Brush.Color := Self.Color;

end;

procedure TCustomXDBGrid.RemoveAutomaticColumns;
begin
  if not (csDesigning in ComponentState) then
    TxDBGridColumns(Columns).RemoveAutoColumns;
end;

procedure TCustomXDBGrid.ResetSizes;
begin
  LayoutChanged;
  inherited ResetSizes;
end;

procedure TCustomXDBGrid.SelectEditor;
var
  aEditor: TWinControl;
begin
  {$ifdef dbgDBGrid}
  DebugLn('TCustomDBGrid.SelectEditor INIT Editor=', dbgsname(editor));
  {$endif}
  if (FDatalink <> nil) and FDatalink.Active then
  begin
    inherited SelectEditor;

    if (SelectedField is TStringField) then
    begin
      if (Editor is TCustomEdit) then
        TCustomEdit(Editor).MaxLength := SelectedField.Size
      else
      if (Editor is TxCompositeCellEditor) then
        TxCompositeCellEditor(Editor).MaxLength := SelectedField.Size;
    end;

    if Assigned(OnSelectEditor) then
    begin
      aEditor := Editor;
      OnSelectEditor(Self, SelectedColumn, aEditor);
      Editor := aEditor;
    end;
  end
  else
    Editor := nil;
  {$ifdef dbgDBGrid}
  DebugLn('TCustomDBGrid.SelectEditor DONE Editor=', dbgsname(editor));
  {$endif}
end;

procedure TCustomXDBGrid.SetEditText(ACol, ARow: longint; const Value: string);
begin
  FTempText := Value;
end;

procedure TCustomXDBGrid.SetFixedCols(const AValue: integer);
begin
  if (FixedCols = AValue) or (AValue < FirstGridColumn) then
    exit;
  if (AValue = 0) and (xdgIndicator in Options) then FixedCols:= 1;
  inherited SetFixedCols(AValue);
end;

function TCustomXDBGrid.SelectCell(aCol, aRow: integer): boolean;
begin
  Result := (ColWidths[aCol] > 0) and (RowHeights[aRow] > 0);
end;

procedure TCustomXDBGrid.BeginLayout;
begin
  Inc(FLayoutChangedCount);
end;

procedure TCustomXDBGrid.EditingColumn(aCol: integer; Ok: boolean);
begin
  {$ifdef dbgDBGrid}
  DebugLn(['DBGrid.EditingColumn INIT aCol=', aCol, ' Ok=', ok]);
{$endif}
  if Ok then
  begin
    FEditingColumn := aCol;
    FDatalink.Modified := True;
  end
  else
    FEditingColumn := -1;
  {$ifdef dbgDBGrid}
  DebugLn('DBGrid.EditingColumn END');
{$endif}
end;

procedure TCustomXDBGrid.EditorCancelEditing;
begin
  EditingColumn(FEditingColumn, False); // prevents updating the value
  if EditorMode then
  begin
    EditorMode := False;
    if xdgAlwaysShowEditor in Options then
      EditorMode := True;
  end;
end;

procedure TCustomXDBGrid.EditorDoGetValue;
begin
  {$ifdef dbgDBGrid}
  DebugLn('DBGrid.EditorDoGetValue INIT');
{$endif}
  inherited EditordoGetValue;
  UpdateData;
  {$ifdef dbgDBGrid}
  DebugLn('DBGrid.EditorDoGetValue FIN');
{$endif}
end;

procedure TCustomXDBGrid.CellClick(const aCol, aRow: integer;
  const Button: TMouseButton; P: TPoint);
var
  CP: TCellCursorPos;
  C: TxColumn;
  I: integer;
  R: TRect;
begin

  if Button <> mbLeft then
    exit;

  if (aCol >= FirstGridColumn) and (aRow >= FixedRows) then
  begin
    if ColumnEditorStyle(ACol, SelectedField) = cbsxCheckBoxColumn then
    begin
      if (Editor = nil) or not EditorMode then
        SwapCheckBox;
    end;
  end;

  if Assigned(OnCellClick) then
  begin
    I := ColumnIndexFromGridColumn(ACol);
    if I >= 0 then
    begin
      CP := ccpTop;
      C := Columns[I];
      if C.xFieldName <> '' then
      begin
        R := CellRect(aCol, aRow);
        R.Top := R.Top + (R.Bottom - R.Top) div 2;
        if PtInRect(R, P) then
          CP := ccpBottom;
      end;
      OnCellClick(C, CP);
    end;

  end;
end;

procedure TCustomXDBGrid.EndLayout;
begin
  Dec(FLayoutChangedCount);
  if FLayoutChangedCount = 0 then
    DoLayoutChanged;
end;

function TCustomXDBGrid.GetDefaultColumnAlignment(Column: integer): TAlignment;
var
  F: TField;
begin
  F := GetDsFieldFromGridColumn(Column);
  if F <> nil then
    Result := F.Alignment
  else
    Result := taLeftJustify;
end;

function TCustomXDBGrid.GetDefaultColumnWidth(Column: integer): integer;
begin
  Result := DefaultFieldColWidth(GetDsFieldFromGridColumn(Column));
end;

function TCustomXDBGrid.GetDefaultColumnReadOnly(Column: integer): boolean;
var
  F: Tfield;
begin
  Result := True;
  if not Self.ReadOnly and (FDataLink.Active and not FDatalink.ReadOnly) then
  begin
    F := GetDsFieldFromGridColumn(Column);
    Result := (F = nil) or F.ReadOnly;
  end;
end;

function TCustomXDBGrid.GetDefaultColumnTitle(Column: integer): string;
var
  F: Tfield;
begin
  F := GetDsFieldFromGridColumn(Column);
  if F <> nil then
    Result := F.DisplayName
  else
    Result := '';
end;

function TCustomXDBGrid.GetDefaultRowHeight: integer;
begin
  Result := inherited GetDefaultRowHeight;
  {$IfDef WINDOWS}
  Dec(Result, 2); // a litle smaller for dbgrid
  {$EndIf}
  Inc(Result, FIncRowHeight);
  if DoubleRowHeight then
    Result := Result * 2;
end;

procedure TCustomXDBGrid.DoExit;
begin
  {$ifdef dbgDBGrid}
  DebugLn('DBGrid.DoExit INIT');
{$Endif}
  if ValidDataSet and (xdgCancelOnExit in Options) and InsertCancelable then
  begin
    FDataLink.DataSet.Cancel;
    EditingColumn(FEditingColumn, False);
  end;
  inherited DoExit;
  {$ifdef dbgDBGrid}
  DebugLn('DBGrid.DoExit FIN');
{$Endif}
end;

function TCustomXDBGrid.DoMouseWheelDown(Shift: TShiftState;
  MousePos: TPoint): boolean;
begin
  Result := False;
  if ssCtrl in Shift then
  begin
    Result := True;
    IncFontValue(-1);
  end
  else
  if Assigned(OnMouseWheelDown) then
    OnMouseWheelDown(Self, Shift, MousePos, Result);
  if not Result and FDatalink.Active then
  begin
    FDatalink.MoveBy(1);
    Result := True;
  end;
end;

function TCustomXDBGrid.DoMouseWheelUp(Shift: TShiftState; MousePos: TPoint): boolean;
begin
  Result := False;
  if ssCtrl in Shift then
  begin
    Result := True;
    IncFontValue(1);
  end
  else
  if Assigned(OnMouseWheelUp) then
    OnMouseWheelUp(Self, Shift, MousePos, Result);

  if not Result and FDatalink.Active then
  begin
    FDatalink.MoveBy(-1);
    Result := True;
  end;
end;

function TCustomXDBGrid.GetEditMask(aCol, aRow: longint): string;
var
  aField: TField;
begin
  Result := '';
  if FDataLink.Active then
  begin
    aField := GetFieldFromGridColumn(aCol);
    if (aField <> nil) then
    begin
      {$ifdef EnableFieldEditMask}
      // enable following line if TField gets in the future a MaskEdit property
      //Result := aField.EditMask;
      if assigned(OnFieldEditMask) then
        OnFieldEditMask(Self, AField, Result);
      {$endif}
    end;
  end;
end;

function TCustomXDBGrid.GetEditText(aCol, aRow: longint): string;
var
  aField: TField;
begin
  if DoubleRowHeight and (CellCursorPos = ccpBottom) then
  begin
    if DataLink.Active then
    begin
      aField := GetxFieldFromGridColumn(aCol);
      if (aField <> nil) then
        Result := aField.Text;
    end;
  end
  else
  if FDataLink.Active then
  begin
    aField := GetFieldFromGridColumn(aCol);
    if aField <> nil then
    begin
      Result := aField.Text;
    end;
  end;
end;

function TCustomXDBGrid.GetIsCellSelected(aCol, aRow: integer): boolean;
begin
  Result := inherited GetIsCellSelected(aCol, aRow) or FDrawingMultiSelRecord;
end;

function TCustomXDBGrid.GetIsCellTitle(aCol, aRow: integer): boolean;
begin
  Result := (FixedRows > 0) and (aRow = 0);
  if Result and Columns.Enabled then
    Result := (aCol >= FirstGridColumn);
end;

procedure TCustomXDBGrid.GetSelectedState(AState: TxGridDrawState;
  out IsSelected: boolean);
begin
  inherited GetSelectedState(AState, IsSelected);
  if IsSelected and not Self.Focused and not (xdgAlwaysShowSelection in Options) then
    IsSelected := False;
end;

function TCustomXDBGrid.GridCanModify: boolean;
begin
  Result := not ReadOnly and (xdgEditing in Options) and not
    FDataLink.ReadOnly and FDataLink.Active and FDatalink.DataSet.CanModify;
end;

function TCustomXDBGrid.GridCanDelete: boolean;
begin
  Result := not ReadOnly and not (xdgDisableDelete in Options)
    and not FDataLink.ReadOnly and FDataLink.Active and FDatalink.DataSet.CanModify
    and not FDataLink.DataSet.IsEmpty;
end;

procedure TCustomXDBGrid.GetSBVisibility(out HsbVisible, VsbVisible: boolean);
var
  aRange, aPage, aPos: integer;
begin
  inherited GetSBVisibility(HsbVisible, VsbVisible);
  VSbVisible := (ScrollBars in [ssVertical, ssBoth]);
  if not VSbVisible and ScrollBarAutomatic(ssVertical) then
  begin
    GetScrollbarParams(aRange, aPage, aPos);
    if ARange > aPage then
      VSbVisible := True;
  end;
end;

procedure TCustomXDBGrid.GetSBRanges(const HsbVisible, VsbVisible: boolean;
  out HsbRange, VsbRange, HsbPage, VsbPage, HsbPos, VsbPos: integer);
begin
  inherited GetSBRanges(HsbVisible, VsbVisible, HsbRange, VsbRange,
    HsbPage, VsbPage, HsbPos, VsbPos);
  if VSbVisible then
    GetScrollbarParams(VsbRange, VsbPage, VsbPos)
  else
  begin
    VsbRange := 0;
    VsbPage := 0;
    VsbPos := 0;
  end;
end;

procedure TCustomXDBGrid.MoveSelection;
begin
  {$ifdef dbgDBGrid}
  DebugLn('DBGrid.MoveSelection INIT');
{$Endif}
  inherited MoveSelection;
  if FColEnterPending and Assigned(OnColEnter) then
  begin
    OnColEnter(Self);
  end;
  FColEnterPending := False;
  UpdateActive;
  {$ifdef dbgDBGrid}
  DebugLn('DBGrid.MoveSelection FIN');
{$Endif}
end;

function TCustomXDBGrid.MouseButtonAllowed(Button: TMouseButton): boolean;
begin
  Result := (Button = mbLeft) or (xdgAnyButtonCanSelect in Options);
end;

procedure TCustomXDBGrid.DrawAllRows;
var
  CurActiveRecord: integer;
begin
  if FDataLink.Active then
  begin
    {$ifdef dbgGridPaint}
    DebugLn;
    DebugLn('%s DrawAllRows: Link.ActiveRecord=%d, Row=%d', [Name,
      FDataLink.ActiveRecord, Row]);
    {$endif}
    CurActiveRecord := FDataLink.ActiveRecord;
    FDrawingEmptyDataset := FDatalink.DataSet.IsEmpty;
  end
  else
    FDrawingEmptyDataset := True;
  try
    inherited DrawAllRows;
  finally
    if FDataLink.Active then
    begin
      FDataLink.ActiveRecord := CurActiveRecord;
      {$ifdef dbgGridPaint}
      DebugLn('%s DrawAllRows END Link.ActiveRecord=%d, Row=%d', [Name,
        FDataLink.ActiveRecord, Row]);
      DebugLn;
      {$endif}
    end;
  end;
end;

procedure TCustomXDBGrid.DrawFocusRect(aCol, aRow: integer; ARect: TRect);
var
  F: TField;
begin
  // Draw focused cell if we have the focus
  Dec(ARect.Bottom);

  if Self.Focused and (xdgAlwaysShowSelection in Options) and
    Datalink.Active and DefaultDrawing then
  begin
    if xStyle = xgsClassic then
      Dec(ARect.Left, 2);
    F := GetxFieldFromGridColumn(aCol);
    if F <> nil then
    begin
      if CellCursorPos = ccpTop then
        ARect.Bottom := ARect.Top + (ARect.Bottom - ARect.Top) div 2 - 1
      else
        ARect.Top := ARect.Top + (ARect.Bottom - ARect.Top) div 2 + 1;
    end;
    if aCol > 0 then
      Inc(ARect.Left, 2);
    Dec(ARect.Right);
    DrawRubberRect(Canvas, aRect, FocusColor);
  end;
end;


procedure TCustomXDBGrid.DrawRow(ARow: integer);
begin
  if (ARow >= FixedRows) and FDataLink.Active then
  begin
    //if (Arow>=FixedRows) and FCanBrowse then
    FDataLink.ActiveRecord := ARow - FixedRows;
    FDrawingActiveRecord := ARow = Row;
    FDrawingMultiSelRecord := (xdgMultiselect in Options) and
      SelectedRows.CurrentRowSelected;
  end
  else
  begin
    FDrawingActiveRecord := False;
    FDrawingMultiSelRecord := False;
  end;
  {$ifdef dbgGridPaint}
  DbgOut('DrawRow Row=', IntToStr(ARow), ' Act=', dbgs(FDrawingActiveRecord));
  {$endif}
  inherited DrawRow(ARow);
  {$ifdef dbgGridPaint}
  DebugLn(' End Row');
  {$endif}
end;

procedure TCustomXDBGrid.DrawCell(aCol, aRow: integer; aRect: TRect;
  aState: TxGridDrawState);
var
  DataCol: integer;
begin
  PrepareCanvas(aCol, aRow, aState);

  {$ifdef dbgGridPaint}
  DbgOut(' ', IntToStr(aCol));
  if xgdSelected in aState then
    DbgOut('S');
  if gdFocused in aState then
    DbgOut('*');
  if xgdFixed in aState then
    DbgOut('F');
  {$endif dbgGridPaint}

  if xgdSelected in aState then
  begin
    if Focused then
      Canvas.Font.Color := FxSelectedFontColor
    else
      Canvas.Font.Color := clBlack;
  end;

  if (ARow >= FixedRows) and Assigned(OnDrawColumnCell) and not
    (csDesigning in ComponentState) then
  begin
    DataCol := ColumnIndexFromGridColumn(aCol);
    if DataCol >= 0 then
      OnDrawColumnCell(Self, aRect, DataCol, TxColumn(Columns[DataCol]), aState);
  end;

  if (xgdFixed in aState) or DefaultDrawing then
    DefaultDrawCell(aCol, aRow, aRect, aState);

  DrawCellGrid(aCol, aRow, aRect, aState);
end;

procedure TCustomXDBGrid.DrawCheckboxBitmaps(aCol: integer; aRect: TRect; F: TField);
var
  AState: TCheckboxState;
begin
  if (aCol = Col) and FDrawingActiveRecord then
  begin
    // show checkbox only if overriden editor is hidden
    if EditorMode then
      exit;
  end;

  // by SSY
  if (F <> nil) then
    if F.DataType = ftBoolean then
      if F.IsNull then
        AState := cbGrayed
      else
      if F.AsBoolean then
        AState := cbChecked
      else
        AState := cbUnChecked
    else
    if F.AsString = ColumnFromGridColumn(aCol).ValueChecked then
      AState := cbChecked
    else

    if F.AsString = ColumnFromGridColumn(aCol).ValueUnChecked then
      AState := cbUnChecked
    else
      AState := cbGrayed
  else
    AState := cbGrayed;

  if assigned(OnUserCheckboxState) then
    OnUserCheckboxState(Self, TxColumn(ColumnFromGridColumn(aCol)), AState);

  DrawGridCheckboxBitmaps(aCol, Row{dummy}, ARect, AState);
end;

procedure TCustomXDBGrid.DrawFixedText(aCol, aRow: integer; aRect: TRect;
  aState: TxGridDrawState);

  function GetDatasetState: TDataSetState;
  begin
    if FDatalink.Active then
      Result := FDataLink.DataSet.State
    else
      Result := dsInactive;
  end;

begin
  if (ACol = 0) and (xdgIndicator in Options) and FDrawingActiveRecord then
  begin
    DrawIndicator(Canvas, aRect, GetDataSetState, FDrawingMultiSelRecord);
    {$ifdef dbgGridPaint}
    dbgOut('>');
    {$endif}
  end
  else
  if (ACol = 0) and (xdgIndicator in Options) and FDrawingMultiSelRecord then
    DrawIndicator(Canvas, aRect, dsCurValue{dummy}, True)
  else
    DrawColumnText(aCol, aRow, aRect, aState);
end;

procedure TCustomXDBGrid.DrawColumnText(aCol, aRow: integer; aRect: TRect;
  aState: TxGridDrawState);
var
  F, ExF: TField;
  xFieldName: string;
  Column: TxColumn;
  ColIndex: integer;
  R: TRect;
  ColTitle: TxColumnTitle;
begin
  ColIndex := ColumnIndexFromGridColumn(aCol);
  if ColIndex >= 0 then
  begin
    Column := Columns[ColIndex];
    F := Column.Field;
    ExF := Column.xField;
    xFieldName := Column.xFieldName;
    if GetIsCellTitle(aCol, aRow) then
    begin
      if xFieldName <> '' then
      begin
        R := aRect;
        R.Bottom := R.Top + (R.Bottom - R.Top) div 2 - 1;
        if (Column.Title.ImageIndex >= 0) and (TitleImageList <> nil) then
          DrawColumnTitleImage(R, aCol, Column.Title.ImageIndex);
        ColTitle := TxColumnTitle(Column.Title);
        xDrawCellText(aCol, aRow, R, aState, ColTitle.Caption, ColTitle.WordWrap);
        R.Top := R.Bottom + 1;
        R.Bottom := aRect.Bottom;
        if (Column.Title.xImageIndex >= 0) and (TitleImageList <> nil) then
          DrawColumnTitleImage(R, aCol, Column.Title.xImageIndex);
        xDrawCellText(aCol, aRow, R, aState, ColTitle.xCaption, ColTitle.xWordWrap);
      end
      else
      begin
        ColTitle := TxColumnTitle(Column.Title);
        if (Column.Title.ImageIndex >= 0) and (TitleImageList <> nil) then
          DrawColumnTitleImage(aRect, aCol, Column.Title.ImageIndex);
        xDrawCellText(aCol, aRow, aRect, aState, GetColumnTitle(aCol), ColTitle.WordWrap);
      end;
    end
    else
    begin
      if ExF <> nil then
      begin
        R := aRect;
        R.Bottom := R.Top + (R.Bottom - R.Top) div 2 - 1;
        if F <> nil then
          xDrawCellText(aCol, aRow, R, aState, F.DisplayText, ColTitle.WordWrap);
        R.Top := R.Bottom + 1;
        R.Bottom := aRect.Bottom;
        xDrawCellText(aCol, aRow, R, aState, ExF.DisplayText, ColTitle.xWordWrap);
      end
      else
      begin
        if F <> nil then
          xDrawCellText(aCol, aRow, aRect, aState, F.DisplayText, ColTitle.WordWrap);
      end;
    end;
  end
  else
  if GetIsCellTitle(aCol, aRow) then
    inherited DrawColumnText(aCol, aRow, aRect, aState)
  else
  begin
    F := GetFieldFromGridColumn(aCol);
    if F <> nil then
      xDrawCellText(aCol, aRow, aRect, aState, F.DisplayText, False);
  end;
end;

procedure TCustomXDBGrid.DrawIndicator(ACanvas: TCanvas; R: TRect;
  Opt: TDataSetState; MultiSel: boolean);
var
  dx, dy, x, y: integer;

  procedure CenterY;
  begin
    y := R.Top + (R.Bottom - R.Top) div 2;
  end;

  procedure CenterX;
  begin
    X := R.Left + (R.Right - R.Left) div 2;
  end;

  procedure DrawEdit(clr: Tcolor);
  begin
    ACanvas.Pen.Color := clr;
    CenterY;
    CenterX;
    ACanvas.MoveTo(X - 2, Y - Dy);
    ACanvas.LineTo(X + 3, Y - Dy);
    ACanvas.MoveTo(X, Y - Dy);
    ACanvas.LineTo(X, Y + Dy);
    ACanvas.MoveTo(X - 2, Y + Dy);
    ACanvas.LineTo(X + 3, Y + Dy);
  end;

begin
  dx := 6;
  dy := 6;
  case Opt of
    dsBrowse:
    begin
      ACanvas.Brush.Color := clBlack;
      ACanvas.Pen.Color := clBlack;
      CenterY;
      x := R.Left + 3;
      if MultiSel then
      begin
        ACanvas.Polyline([point(x, y - dy), point(x + dx, y), point(x, y + dy),
          point(x, y + dy - 1)]);
        ACanvas.Polyline([point(x, y - dy + 1), point(x + dx - 1, y), point(x, y + dy - 1),
          point(x, y + dy - 2)]);
        CenterX;
        Dec(X, 3);
        ACanvas.Ellipse(Rect(X - 2, Y - 2, X + 2, Y + 2));
      end
      else
        ACanvas.Polygon([point(x, y - dy), point(x + dx, y), point(x, y + dy), point(x, y - dy)]);
    end;
    dsEdit:
      DrawEdit(clBlack);
    dsInsert:
      DrawEdit(clGreen);
    else
      if MultiSel then
      begin
        ACanvas.Brush.Color := clBlack;
        ACanvas.Pen.Color := clBlack;
        CenterX;
        CenterY;
        ACanvas.Ellipse(Rect(X - 3, Y - 3, X + 3, Y + 3));
      end;
  end;
end;

function TCustomXDBGrid.EditorCanAcceptKey(const ch: TUTF8Char): boolean;
var
  aField: TField;
begin
  Result := False;
  if FDataLink.Active then
  begin
    if ch = #8 then
      Result := True
    else
    begin
	    aField := SelectedField;
	    if aField <> nil then
	    begin
	        Result := IsValidChar(AField, Ch) and not aField.Calculated and
	          (aField.DataType <> ftAutoInc) and (not aField.Lookup) and not aField.IsBlob;
      end;
    end;
  end;
end;

function TCustomXDBGrid.EditorIsReadOnly: boolean;
var
  AField: TField;
  FieldList: TList;
  I: integer;
begin
  Result := inherited EditorIsReadOnly;
  if not Result then
  begin

    AField := GetFieldFromGridColumn(Col);
    if assigned(AField) then
    begin

      // if field can't be modified, it's assumed readonly
      Result := not AField.CanModify;

      // if field is readonly, check if it's a lookup field
      if Result and AField.Lookup then
      begin
        FieldList := TList.Create;
        try
          AField.DataSet.GetFieldList(FieldList, AField.KeyFields);
          // check if any keyfields are there
          Result := (FieldList.Count = 0); // if not simply is still readonly
          // if yes assumed keyfields are modifiable
          for I := 0 to FieldList.Count - 1 do
            if not TField(FieldList[I]).CanModify then
            begin
              Result := True; // at least one keyfield is readonly
              break;
            end;
        finally
          FieldList.Free;
        end;
      end;

      // if it's not readonly and is not already editing, start editing.
      if not Result and not FDatalink.Editing then
      begin
        Include(FGridStatus, gsStartEditing);
        Result := not FDataLink.Edit;
        Exclude(FGridStatus, gsStartEditing);
      end;

    end
    else
      Result := True;  // field is nil so it's readonly

    EditingColumn(Col, not Result);
  end;
end;

procedure TCustomXDBGrid.EditorTextChanged(const aCol, aRow: integer;
  const aText: string);
begin
  if not EditorIsReadonly then
    SetEditText(aCol, aRow, aText);
end;

procedure TCustomXDBGrid.HeaderSized(IsColumn: boolean; Index: integer);
var
  i: integer;
begin
  if IsColumn then
  begin
    if Columns.Enabled then
    begin
      i := ColumnIndexFromGridColumn(Index);
      if i >= 0 then
        Columns[i].Width := ColWidths[Index];
    end;
    FDefaultColWidths := False;
    if Assigned(OnColumnSized) then
      OnColumnSized(Self);
  end;
end;

function TCustomXDBGrid.IsValidChar(AField: TField; AChar: TUTF8Char): boolean;
begin
  Result := False;

  if Length(AChar) > 1 then
  begin
    // problem: AField should validate a unicode char, but AField has no
    //          such facility, ask the user, if user is not interested
    //          do ansi convertion and try with field.

    { TODO: is this really necessary?
    if Assigned(FOnValidateUTF8Char) then begin
      result := true;
      OnValidateUT8Char(Self, AField, AChar, Result)
      exit;
    end else
    }
    AChar := UTF8ToSys(AChar);
  end
  else
  if Length(AChar) = 0 then
    exit;

  Result := AField.IsValidChar(AChar[1]);
end;

procedure TCustomXDBGrid.UpdateActive;
var
  PrevRow: integer;
begin
  if (csDestroying in ComponentState) or (FDatalink = nil) or
    (not FDatalink.Active) or (FDatalink.ActiveRecord < 0) then
    exit;
  {$IfDef dbgDBGrid}
  DebugLn(Name, '.UpdateActive: ActiveRecord=', dbgs(FDataLink.ActiveRecord),
    ' FixedRows=', dbgs(FixedRows), ' Row=', dbgs(Row));
  {$endif}
  PrevRow := Row;
  Row := FixedRows + FDataLink.ActiveRecord;
  if PrevRow <> Row then
  begin
    if FixedCols > 1 then
      InvalidateRow(PrevRow)
    else
      InvalidateCell(0, PrevRow);
  end;
  InvalidateRow(Row);
end;

function TCustomXDBGrid.UpdateGridCounts: integer;
var
  RecCount: integer;
  FRCount, FCCount: integer;
begin
  // find out the column count, if result=0 then
  // there are no visible columns defined or dataset is inactive
  // or there are no visible fields, ie the grid is blank
  {$IfDef dbgDBGrid}
  DebugLn('TCustomDbgrid.UpdateGridCounts INIT');
{$endif}
  BeginUpdate;
  try
    Result := GetColumnCount;
    if Result > 0 then
    begin

      if xdgTitles in Options then
        FRCount := 1
      else
        FRCount := 0;
      if xdgIndicator in Options then
        FCCount := 1
      else
        FCCount := 0;
      InternalSetColCount(Result + FCCount);

      if FDataLink.Active then
      begin
        UpdateBufferCount;
        RecCount := FDataLink.RecordCount;
        if RecCount < 1 then
          RecCount := 1;
      end
      else
      begin
        RecCount := 0;
        if FRCount = 0 then
          // need to be large enough to hold indicator
          // if there is one, and if there are no titles
          RecCount := FCCount;
      end;

      Inc(RecCount, FRCount);

      RowCount := RecCount;
      FixedRows := FRCount;

      UpdateGridColumnSizes;

      if FDatalink.Active and (FDatalink.ActiveRecord >= 0) then
        AdjustEditorBounds(Col, FixedRows + FDatalink.ActiveRecord);
    end;
  finally
    EndUpdate;
  end;
  {$IfDef dbgDBGrid}
  DebugLn('TCustomDbgrid.UpdateGridCounts END');
{$endif}
end;

constructor TCustomXDBGrid.Create(AOwner: TComponent);
begin
  FUseAutoFilter := True;
  FEditingColumn := -1;
  DragDx := 5;
  inherited Create(AOwner);
  FLocateStr := '';
  FAutoSort := True;
  FDataLink := TxComponentDataLink.Create;//(Self);
  FDataLink.OnRecordChanged := @OnRecordChanged;
  FDataLink.OnDatasetChanged := @OnDataSetChanged;
  FDataLink.OnDataSetOpen := @OnDataSetOpen;
  FDataLink.OnDataSetClose := @OnDataSetClose;
  FDataLink.OnNewDataSet := @OnNewDataSet;
  FDataLink.OnInvalidDataSet := @OnInvalidDataset;
  FDataLink.OnInvalidDataSource := @OnInvalidDataSource;
  FDataLink.OnDataSetScrolled := @OnDataSetScrolled;
  FDataLink.OnLayoutChanged := @OnLayoutChanged;
  FDataLink.OnEditingChanged := @OnEditingChanged;
  FDataLink.OnUpdateData := @OnUpdateData;
  FDataLink.VisualControl := True;

  FRowsHighLights := TRowsHightLightItems.Create(Self, TRowsHighLightItem);

  FSelectedRows := TxBookmarkList.Create(Self);
  FIncRowHeight := 0;

  RenewColWidths;

  FOptions := [xdgAnyButtonCanSelect, xdgColumnResize, xdgColumnMove, xdgTitles,
    xdgIndicator, xdgRowLines, xdgColLines, xdgConfirmDelete,
    xdgEditing, xdgAlwaysShowSelection, xdgAutoSizeColumns, xdgAutoLocate];

  inherited Options :=
    [xgoFixedVertLine, xgoFixedHorzLine, xgoVertLine, xgoHorzLine,
    xgoSmoothScroll, xgoColMoving, xgoEditing, xgoDrawFocusSelected,
    xgoColSizing];

  FExtraOptions := [dgeAutoColumns, dgeCheckboxColumn];

  AutoAdvance := aaRightDown;

  // What a dilema!, we need ssAutoHorizontal and ssVertical!!!
  ScrollBars := ssBoth;

  FCellCursorPos := ccpTop;
  FxStyle := xgsNormal;
  FxLineColor := clBtnShadow;
  FxSelectedFontColor := clNavy;
  SelectedColor := clSkyBlue;
  //SelectedColor := RGBToColor(213, 232, 252);
  FDoubleRowHeight := False;
  FMidleLineColor := clSilver;
  FDrawMidleLine := True;
  FThumbTrack := True;
  FLocateInfoPanel := nil;
  FilterContaining := True;
  SaveOldFilterEvent:= nil;
end;

procedure TCustomXDBGrid.ActivateFilter(FieldName, AValue: string;
  IsCaseInsensitive: Boolean; AContainingValue: Boolean);
var
  SaveCursor: TCursor;
begin
  if Datalink.DataSet <> nil then
  begin
      FilterCaseInsensitive := IsCaseInsensitive;
      FilterContaining := AContainingValue;
      if not IsCaseInsensitive then
        FilterValue := UTF8LowerCase(AValue)
      else
        FilterValue := AValue;
      FiltterField := FieldName;
      if Assigned(DataSource.DataSet.OnFilterRecord) then
        SaveOldFilterEvent := DataSource.DataSet.OnFilterRecord;
      DataSource.DataSet.OnFilterRecord := @OnAutoFilter;
      SaveCursor := Screen.Cursor;
      Screen.Cursor := crHourGlass;
      DataSource.DataSet.Filtered := True;
      Screen.Cursor := SaveCursor;
  end;
end;

procedure TCustomXDBGrid.ResetFilter;
begin
  if DataSource <> nil then
    if DataSource.DataSet <> nil then
    begin
      DataSource.DataSet.Filtered:= False;
      if SaveOldFilterEvent <> nil then
        DataSource.DataSet.OnFilterRecord := SaveOldFilterEvent
      else
        DataSource.DataSet.OnFilterRecord:= nil;
    end;
end;

procedure TCustomXDBGrid.OnAutoFilter(DataSet: TDataSet; var Accept: Boolean);
var
  S: string;
begin
  if FilterCaseInsensitive then
    S := DataSet.FieldByName(FiltterField).AsString
  else
    S := UTF8LowerCase(DataSet.FieldByName(FiltterField).AsString);

  if FilterValue = '' then
  begin
    if S = '' then Accept := True
    else Accept :=  False;
  end
  else
  if FilterContaining then
  begin
    if Pos(FilterValue, S) > 0 then
      Accept := True
    else
      Accept := False;
  end
  else
  begin
    if Pos(FilterValue, S) = 1 then
      Accept := True
    else
      Accept := False;
  end;
end;

procedure TCustomXDBGrid.ShowFilterForm(F: TField; ACaption, InputStr: string);
begin
  if F = nil then
  begin
    if CellCursorPos = ccpBottom then
    begin
      F := SelectedColumn.xField;
      ACaption := SelectedColumn.Title.xCaption;
    end
    else
    begin
      F := SelectedColumn.Field;
      ACaption := SelectedColumn.Title.Caption;
    end;
  end;

  if Assigned(F) then
  begin
    GridFilterForm := TGridFilterForm.Create(Self);
    GridFilterForm.Font.Assign(Font);
    GridFilterForm.Font.Color:= clBlack;
    GridFilterForm.Panel1.Font.Style := [fsBold];
    GridFilterForm.Field := F;
    GridFilterForm.Panel1.Caption := ACaption;
    GridFilterForm.Edit1.Text := InputStr;
    GridFilterForm.ShowModal;
  end;
end;

procedure TCustomXDBGrid.AutoSizeColumns;
begin
  RenewColWidths;
  LayoutChanged;
end;

procedure TCustomXDBGrid.CheckSorted;
var
  S: string;
  Desc: Boolean;
  I: Integer;
  aDataset: TDataSet;
begin
  aDataset := Datalink.DataSet;
  if (xdgAutoSortIcon in Options) and (aDataSet <> nil) and aDataSet.Active
    and Assigned(GetDataSortField) then
  begin
    GetDataSortField(aDataSet, S, Desc);
    if S <> '' then
    begin
      CheckSortingImage;
      for I := 0 to Columns.Count - 1 do
      begin
        if (Columns.Items[I].FieldName = S) then
        begin
          if Desc then
            Columns.Items[I].Sorted := soxDescending
          else
            Columns.Items[I].Sorted := soxAscending;
          Break;
        end
        else
        if (Columns.Items[I].xFieldName = S) then
        begin
          if Desc then
            Columns.Items[I].xSorted := soxDescending
          else
            Columns.Items[I].xSorted := soxAscending;
          Break;
        end;
      end;
    end;
  end;
end;

procedure TCustomXDBGrid.InitiateAction;
begin
  {$ifdef dbgDBGrid}
  DebugLn('===> DBGrid.InitiateAction INIT');
{$endif}
  inherited InitiateAction;
  if (gsUpdatingData in FGridStatus) then
  begin
    EndUpdating;
    {
    if EditorMode then begin
      Editor.SetFocus;
      EditorSelectAll;
    end;
    }
  end;
  {$ifdef dbgDBGrid}
  DebugLn('<=== DBGrid.InitiateAction FIN');
{$endif}
end;

procedure TCustomXDBGrid.DefaultDrawColumnCell(const Rect: TRect;
  DataCol: integer; Column: TxColumn; State: TxGridDrawState);
var
  S: string;
  F: TField;
  DataRow: integer;
begin

  F := Column.Field;

  DataCol := GridColumnFromColumnIndex(DataCol);
  if FDataLink.Active then
    DataRow := FixedRows + FDataLink.ActiveRecord
  else
    DataRow := 0;

  if DataCol >= FirstGridColumn then
    case ColumnEditorStyle(DataCol, F) of

      cbsxCheckBoxColumn:
        DrawCheckBoxBitmaps(DataCol, Rect, F);

      else
      begin
        if F <> nil then
        begin
          if F.dataType <> ftBlob then
            S := F.DisplayText
          else
            S := '(blob)';
        end
        else
          S := '';
        DrawCellText(DataCol, DataRow, Rect, State, S);
      end;

    end;
end;

function TCustomXDBGrid.EditorByStyle(Style: TxColumnButtonStyle): TWinControl;
begin
  // we want override the editor style if it is cbsxAuto because
  // field.datatype might be ftBoolean or some other cases
  if Style = cbsxAuto then
    Style := ColumnEditorStyle(Col, SelectedField);

  Result := inherited EditorByStyle(Style);
end;

procedure TCustomXDBGrid.ResetColWidths;
begin
  if not FDefaultColWidths then
  begin
    RenewColWidths;
    LayoutChanged;
  end;
end;

function TCustomXDBGrid.IndicatorWidth: integer;
begin
  Result := 12;
end;

procedure TCustomXDBGrid.SelectRecord(AValue: boolean);
begin
  if xdgMultiselect in Options then
    FSelectedRows.CurrentRowSelected := AValue;
end;

procedure TCustomXDBGrid.GetScrollbarParams(out aRange, aPage, aPos: integer);
begin
  if (FDatalink <> nil) and FDatalink.Active then
  begin
    if FDatalink.dataset.IsSequenced then
    begin
      aRange := GetRecordCount + VisibleRowCount - 1;
      aPage := VisibleRowCount;
      if aPage < 1 then
        aPage := 1;
      if FDatalink.BOF then
        aPos := 0
      else
      if FDatalink.EOF then
        aPos := aRange
      else
        aPos := FDataLink.DataSet.RecNo - 1; // RecNo is 1 based
      if aPos < 0 then
        aPos := 0;
    end
    else
    begin
      aRange := 6;
      aPage := 2;
      if FDatalink.EOF then
        aPos := 4
      else
      if FDatalink.BOF then
        aPos := 0
      else
        aPos := 2;
    end;
  end
  else
  begin
    aRange := 0;
    aPage := 0;
    aPos := 0;
  end;
end;

procedure TCustomXDBGrid.CMGetDataLink(var Message: TLMessage);
begin
  Message.Result := PtrUInt(FDataLink);
end;

procedure TCustomXDBGrid.ClearSelection(selCurrent: boolean = False);
begin
  if [xdgMultiselect, xdgPersistentMultiSelect] * Options = [xdgMultiselect] then
  begin
    if SelectedRows.Count > 0 then
      SelectedRows.Clear;
    if SelCurrent then
      SelectRecord(True);
  end;
  FKeyBookmark := nil;
end;

function TCustomXDBGrid.NeedAutoSizeColumns: boolean;
begin
  Result := (xdgAutoSizeColumns in Options)
  //and (HandleAllocated)
  ;
end;

procedure TCustomXDBGrid.RenewColWidths;
begin
  FDefaultColWidths := True;
  exclude(FGridStatus, gsAutoSized);
end;

function TCustomXDBGrid.RowHighLightUse: Boolean;
begin
  Result := FRowsHighLights.Count > 0;
end;

destructor TCustomXDBGrid.Destroy;
begin
  FSelectedRows.Free;
  FDataLink.OnDataSetChanged := nil;
  FDataLink.OnRecordChanged := nil;
  FDataLink.Free;
  FreeAndNil(FRowsHighLights);
  FreeThenNil(FInsertActionLink);
  FreeThenNil(FEditActionLink);
  FreeThenNil(FSelectActionLink);
  FreeThenNil(FDeleteActionLink);
  inherited Destroy;
end;

function TCustomXDBGrid.EditorRect(aCol, aRow: integer): TRect;
var
  F: TField;
  Column: TxColumn;
  ColIndex: integer;
begin
  Result := inherited EditorRect(aCol, aRow);
  if xStyle = xgsNormal then
    Inc(Result.Left, 1);
  ColIndex := ColumnIndexFromGridColumn(aCol);
  Column := nil;
  if ColIndex >= 0 then
    Column := Columns[ColIndex];
  //else Dec(Result.Bottom);

  if Column <> nil then
  begin
    F := Column.xField;
    if (F <> nil) then
    begin
      if (CellCursorPos = ccpBottom) then
        Result.Top := Result.Top + (Result.Bottom - Result.Top) div 2 - 1
      else
        Result.Bottom := Result.Top + (Result.Bottom - Result.Top) div 2 + 1;
    end;
    if Column.ButtonStyle = cbsxEllipsis then
      Dec(Result.Bottom);
  end;
end;

function TCustomXDBGrid.SelectedEditorRect: TRect;
begin
  Result := EditorRect(Col, Row);
end;

procedure TCustomXDBGrid.CheckSortingImage;
var
  Bitmap: TPortableNetworkGraphic;
begin
  if TitleImageList = nil then
    TitleImageList := TImageList.Create(Self);

  if AscImgInd = -1 then
  begin
    Bitmap := TPortableNetworkGraphic.Create;
    try
      Bitmap.LoadFromResourceName(hInstance, 'sortasc');
      AscImgInd := TitleImageList.Add(Bitmap, nil);
      Bitmap.LoadFromResourceName(hInstance, 'sortdesc');
      DescImgInd := TitleImageList.Add(Bitmap, nil);
    finally
      Bitmap.Free;
    end;
  end;
end;

{ TxComponentDataLink }

function TxComponentDataLink.GetFields(Index: integer): TField;
begin
  if (index >= 0) and (index < DataSet.FieldCount) then
    Result := DataSet.Fields[index];
end;

function TxComponentDataLink.GetDataSetName: string;
begin
  Result := FDataSetName;
  if DataSet <> nil then
    Result := DataSet.Name;
end;

procedure TxComponentDataLink.SetDataSetName(const AValue: string);
begin
  if FDataSetName <> AValue then
    FDataSetName := AValue;
end;

procedure TxComponentDataLink.RecordChanged(Field: TField);
begin
  {$ifdef dbgDBGrid}
  DebugLn('TComponentDataLink.RecordChanged');
  {$endif}
  if Assigned(OnRecordChanged) then
    OnRecordChanged(Field);
end;

procedure TxComponentDataLink.DataSetChanged;
begin
  {$ifdef dbgDBGrid}
  DebugLn('TComponentDataLink.DataSetChanged, FirstRecord=', dbgs(FirstRecord));
  {$Endif}
  // todo: improve this routine, for example: OnDatasetInserted
  if Assigned(OnDataSetChanged) then
    OnDataSetChanged(DataSet);
end;

procedure TxComponentDataLink.ActiveChanged;
begin
  {$ifdef dbgDBGrid}
  DebugLnEnter('TComponentDataLink.ActiveChanged INIT');
  {$endif}
  if Active then
  begin
    fDataSet := DataSet;
    if DataSetName <> fDataSetName then
    begin
      fDataSetName := DataSetName;
      if Assigned(fOnNewDataSet) then
        fOnNewDataSet(DataSet);
    end
    else
    if Assigned(fOnDataSetOpen) then
      fOnDataSetOpen(DataSet);
  end
  else
  begin
    BufferCount := 0;
    if (DataSource = nil) then
    begin
      if Assigned(fOnInvalidDataSource) then
        fOnInvalidDataSource(fDataSet);
      fDataSet := nil;
      fDataSetName := '[???]';
    end
    else
    begin
      if (DataSet = nil) or (csDestroying in DataSet.ComponentState) then
      begin
        if Assigned(fOnInvalidDataSet) then
          fOnInvalidDataSet(fDataSet);
        fDataSet := nil;
        fDataSetName := '[???]';
      end
      else
      begin
        if Assigned(fOnDataSetClose) then
          fOnDataSetClose(DataSet);
        if DataSet <> nil then
          FDataSetName := DataSetName;
      end;
    end;
  end;
  {$ifdef dbgDBGrid}
  DebugLnExit('TComponentDataLink.ActiveChanged DONE');
  {$endif}
end;

procedure TxComponentDataLink.LayoutChanged;
begin
  {$ifdef dbgDBGrid}
  DebugLn('TComponentDataLink.LayoutChanged');
  {$Endif}
  if Assigned(OnLayoutChanged) then
    OnLayoutChanged(DataSet);
end;

procedure TxComponentDataLink.DataSetScrolled(Distance: integer);
begin
  {$ifdef dbgDBGrid}
  DebugLn('TComponentDataLink.DataSetScrolled(', IntToStr(Distance), ')');
  {$endif}
  if Assigned(OnDataSetScrolled) then
    OnDataSetScrolled(DataSet, Distance);
end;

procedure TxComponentDataLink.FocusControl(Field: TFieldRef);
begin
  {$ifdef dbgDBGrid}
  DebugLn('TComponentDataLink.FocusControl');
  {$endif}
end;

procedure TxComponentDataLink.CheckBrowseMode;
begin
  {$ifdef dbgDBGrid}
  DebugLn(ClassName, '.CheckBrowseMode');
  {$endif}
  inherited CheckBrowseMode;
end;

procedure TxComponentDataLink.EditingChanged;
begin
  {$ifdef dbgDBGrid}
  DebugLn(ClassName, '.EditingChanged');
  {$endif}
  if Assigned(OnEditingChanged) then
    OnEditingChanged(DataSet);
end;

procedure TxComponentDataLink.UpdateData;
begin
  {$ifdef dbgDBGrid}
  DebugLn(ClassName, '.UpdateData');
  {$endif}
  if Assigned(OnUpdatedata) then
    OnUpdateData(DataSet);
end;

function TxComponentDataLink.MoveBy(Distance: integer): integer;
begin
  (*
  {$ifdef dbgDBGrid}
  DebugLn(ClassName,'.MoveBy  INIT: Distance=',Distance);
  {$endif}
  *)
  Result := inherited MoveBy(Distance);
  (*
  {$ifdef dbgDBGrid}
  DebugLn(ClassName,'.MoveBy  END: Distance=',Distance);
  {$endif}
  *)
end;

{ TxDBGridColumns }

function TxDBGridColumns.GetColumn(Index: integer): TxColumn;
begin
  Result := TxColumn(inherited Items[Index]);
end;

procedure TxDBGridColumns.SetColumn(Index: integer; Value: TxColumn);
begin
  Items[Index].Assign(Value);
end;

procedure TxDBGridColumns.Update(Item: TCollectionItem);
begin
  if (Grid <> nil) and not (csLoading in Grid.ComponentState) then
    TCustomXDBGrid(Grid).LayoutChanged;
end;

function TxDBGridColumns.ColumnFromField(Field: TField): TxColumn;
var
  i: integer;
begin
  if Field <> nil then
    for i := 0 to Count - 1 do
    begin
      Result := Items[i];
      if (Result <> nil) and (Result.Field = Field) then
        exit;
    end;
  Result := nil;
end;

function TxDBGridColumns.HasAutomaticColumns: boolean;
var
  i: integer;
begin
  Result := False;
  for i := 0 to Count - 1 do
    if Items[i].IsAutomaticColumn then
    begin
      Result := True;
      break;
    end;
end;

function TxDBGridColumns.HasDesignColumns: boolean;
var
  i: integer;
begin
  Result := False;
  for i := 0 to Count - 1 do
    if Items[i].IsDesignColumn then
    begin
      Result := True;
      break;
    end;
end;

procedure TxDBGridColumns.RemoveAutoColumns;
var
  i: integer;
  G: TCustomXDBGrid;
begin
  if HasAutomaticColumns then
  begin
    G := TCustomXDBGrid(Grid);
    G.GridStatus := G.GridStatus + [gsRemovingAutoColumns];
    BeginUpdate;
    try
      for i := Count - 1 downto 0 do
        if Items[i].IsAutomaticColumn then
          Delete(i);
    finally
      EndUpdate;
      G.GridStatus := G.GridStatus - [gsRemovingAutoColumns];
    end;
  end;
end;

procedure TxDBGridColumns.ResetSorted;
var
  I: Integer;
begin
  for I := 0 to Count -1 do
  begin
    Items[I].FSorted := soxNone;
    Items[I].Title.ImageIndex := Items[I].Title.FOldImageIndex;
    Items[I].FxSorted := soxNone;
    Items[I].Title.xImageIndex := Items[I].Title.FOldxImageIndex;
  end;
end;

function CompareFieldIndex(P1, P2: Pointer): integer;
begin
  if P1 = P2 then
    Result := 0
  else if (P1 = nil) or (TxColumn(P1).Field = nil) then
    Result := 1
  else if (P2 = nil) or (TxColumn(P2).Field = nil) then
    Result := -1
  else
    Result := TxColumn(P1).Field.Index - TxColumn(P2).Field.Index;
end;

function CompareDesignIndex(P1, P2: Pointer): integer;
begin
  Result := TxColumn(P1).DesignIndex - TxColumn(P2).DesignIndex;
end;

procedure TxDBGridColumns.ResetColumnsOrder(ColumnOrder: TxColumnOrder);
var
  L: TList;
  i: integer;
begin

  L := TList.Create;
  try

    for i := 0 to Count - 1 do
      L.Add(Items[i]);

    case ColumnOrder of
      coDesignOrder:
      begin
        if not HasDesignColumns then
          exit;
        L.Sort(@CompareDesignIndex);
      end;
      coFieldIndexOrder:
        L.Sort(@CompareFieldIndex);
      else
        exit;
    end;

    for i := 0 to L.Count - 1 do
      TxColumn(L.Items[i]).Index := i;

  finally
    L.Free;
  end;
end;

function TxDBGridColumns.Add: TxColumn;
var
  G: TCustomXDBGrid;
begin
  G := TCustomXDBGrid(Grid);
  if G <> nil then
  begin
    // remove automatic columns before adding user columns
    if not (gsAddingAutoColumns in G.GridStatus) then
      RemoveAutoColumns;
  end;
  Result := TxColumn(inherited add);
end;

procedure TxDBGridColumns.LinkFields;
var
  i: integer;
  G: TCustomXDBGrid;
begin
  G := TCustomXDBGrid(Grid);
  if G <> nil then
    G.BeginLayout;
  for i := 0 to Count - 1 do
  begin
    Items[i].LinkField;
    Items[i].xLinkField;
  end;
  if G <> nil then
    G.EndLayout;
end;

{ TxColumn }

function TxColumn.GetField: TField;
begin
  if (FFieldName <> '') and (FField <> nil) then
    LinkField;
  Result := FField;
end;

function TxColumn.GetIsDesignColumn: boolean;
begin
  Result := (DesignIndex >= 0) and (DesignIndex < 10000);
end;

function TxColumn.GetPickList: TStrings;
begin
  Result := inherited GetPickList;
  if (Field <> nil) and FField.Lookup then
  begin
    if FField.LookupCache then
      FField.LookupList.ValuesToStrings(Result)
    else
    begin
      Result.Clear;
      LookupGetBookMark(FField);
      try
        with FField.LookupDataSet do
        begin
          First;
          while not EOF do
          begin
            Result.Add(FieldByName(FField.LookupResultField).AsString);
            Next;
          end;
        end;
      finally
        LookupGotoBookMark(FField);
      end;
    end;
  end;
end;

procedure TxColumn.xApplyDisplayFormat;
begin
  if (FxField <> nil) and FxDisplayFormatChanged then
  begin
    if (FxField is TNumericField) then
      TNumericField(FxField).DisplayFormat := xDisplayFormat
    else if (FxField is TDateTimeField) then
      TDateTimeField(FxField).DisplayFormat := xDisplayFormat;
  end;
end;

procedure TxColumn.SetWordWrap(AValue: boolean);
begin
  if FWordWrap <> AValue then
  begin
    FWordWrap := AValue;
    ColumnChanged;
  end;
end;

procedure TxColumn.SetxImagesIndexes(AValue: TImagesIndexes);
begin
  FxImagesIndexes.Assign(AValue);
  Grid.Invalidate;
end;

procedure TxColumn.SetxSorted(AValue: TxSortOrder);
begin
  if FxSorted=AValue then Exit;
  TxDBGridColumns(Collection).ResetSorted;
  FxSorted := AValue;
  case AValue of
    soxNone: Title.xImageIndex := Title.FOldxImageIndex;
    soxAscending: Title.xImageIndex:= Grid.AscImgInd;
    soxDescending: Title.xImageIndex:= Grid.DescImgInd;
  end;
  ColumnChanged;

end;

procedure TxColumn.SetPrcWidth(AValue: integer);
begin
  if FPrcWidth = AValue then
    Exit;
  if (AValue < 0) or (AValue > 100) then
    Exception.Create('Invalid Value (procent)');
  FPrcWidth := AValue;
  ColumnChanged;
end;

procedure TxColumn.SetImagesIndexes(AValue: TImagesIndexes);
begin
  FImagesIndexes.Assign(AValue);
  Grid.Invalidate;
end;

procedure TxColumn.SetSorted(AValue: TxSortOrder);
begin
  if FSorted=AValue then Exit;
  //TxDBGridColumns(Collection).Grid;
  TxDBGridColumns(Collection).ResetSorted;
  FSorted:= AValue;
  case AValue of
    soxNone: Title.ImageIndex := Title.FOldImageIndex;
    soxAscending: Title.ImageIndex:= Grid.AscImgInd;
    soxDescending: Title.ImageIndex:= Grid.DescImgInd;
  end;
  ColumnChanged;
end;

procedure TxColumn.SetxWordWrap(AValue: boolean);
begin
  if FxWordWrap <> AValue then
  begin
    FxWordWrap := AValue;
    ColumnChanged;
  end;
end;

function TxColumn.GetxDisplayFormat: string;
begin
  if not FxDisplayFormatChanged then
    Result := GetxDefaultDisplayFormat
  else
    Result := FxDisplayFormat;
end;

function TxColumn.GetxField: TField;
begin
  if (FxFieldName <> '') and (FxField <> nil) then
    xLinkField;
  Result := FxField;
end;

procedure TxColumn.SetxDisplayFormat(const AValue: string);
begin
  if (not FxDisplayFormatChanged) or not SameText(AValue, FxDisplayFormat) then
  begin
    FxDisplayFormat := AValue;
    if FxDisplayFormat = '' then
      FxDisplayFormatChanged := False
    else
      FxDisplayFormatChanged := True;
    ColumnChanged;
  end;
end;

procedure TxColumn.SetxField(const AValue: TField);

begin
  if FxField <> AValue then
  begin
    FxField := AValue;
    if FxField <> nil then
      FxFieldName := FxField.FieldName;
    ColumnChanged;

  end;
end;

procedure TxColumn.SetxFieldName(const AValue: string);
var
  AGrid: TCustomXDBGrid;
begin
  if FxFieldName = AValue then
    exit;
  FxFieldName := AValue;
  if FxFieldName <> '' then
  begin
    AGrid := TCustomXDBGrid(TxDBGridColumns(Collection).Grid);
    AGrid.DoubleRowHeight := True;
  end;
  xLinkField;
  ColumnChanged;
end;

function TxColumn.IsxDisplayFormatStored: boolean;
begin
  Result := FxDisplayFormatChanged;
end;

function TxColumn.GetxAlignment: TAlignment;
begin
  if FxAlignment = nil then
    Result := GetxDefaultAlignment
  else
    Result := FxAlignment^;
end;

procedure TxColumn.SetxAlignment(const AValue: TAlignment);
begin
  if FxAlignment = nil then
  begin
    if AValue = GetxDefaultAlignment then
      exit;
    New(FxAlignment);
  end
  else
  if FxAlignment^ = AValue then
    exit;

  FxAlignment^ := AValue;
  ColumnChanged;
end;

function TxColumn.IsxAlignmentStored: boolean;
begin
  Result := FxAlignment <> nil;
end;

procedure TxColumn.ApplyDisplayFormat;
begin
  if (FField <> nil) and FDisplayFormatChanged then
  begin
    if (FField is TNumericField) then
      TNumericField(FField).DisplayFormat := DisplayFormat
    else if (FField is TDateTimeField) then
      TDateTimeField(FField).DisplayFormat := DisplayFormat;
  end;
end;

function TxColumn.GetDisplayFormat: string;
begin
  if not FDisplayFormatChanged then
    Result := GetDefaultDisplayFormat
  else
    Result := FDisplayFormat;
end;

function TxColumn.IsDisplayFormatStored: boolean;
begin
  Result := FDisplayFormatChanged;
end;

procedure TxColumn.SetDisplayFormat(const AValue: string);
begin
  if (not FDisplayFormatChanged) or (CompareText(AValue, FDisplayFormat) <> 0) then
  begin
    FDisplayFormat := AValue;
    if FDisplayFormat = '' then
      FDisplayFormatChanged := False
    else
      FDisplayFormatChanged := True;
    ColumnChanged;
  end;
end;

procedure TxColumn.SetField(const AValue: TField);
begin
  if FField <> AValue then
  begin
    FField := AValue;
    if FField <> nil then
      FFieldName := FField.FieldName;
    ColumnChanged;
  end;
end;

procedure TxColumn.SetFieldName(const AValue: string);
begin
  if FFieldName = AValue then
    exit;
  FFieldName := AValue;
  LinkField;
  ColumnChanged;
end;

function TxColumn.GetDataSet: TDataSet;
var
  AGrid: TCustomXDBGrid;
begin
  AGrid := TCustomXDBGrid(Grid);
  if (AGrid <> nil) then
    Result := AGrid.FDataLink.DataSet
  else
    Result := nil;
end;

procedure TxColumn.Assign(Source: TPersistent);
begin
  if Source is TxColumn then
  begin
    //DebugLn('Assigning TxColumn[',dbgs(Index),'] a TxColumn')
    Collection.BeginUpdate;
    try
      inherited Assign(Source);
      FieldName := TxColumn(Source).FieldName;
      DisplayFormat := TxColumn(Source).DisplayFormat;
      ValueChecked := TxColumn(Source).ValueChecked;
      ValueUnchecked := TxColumn(Source).ValueUnchecked;
    finally
      Collection.EndUpdate;
    end;
  end
  else
    inherited Assign(Source);
end;

function TxColumn.GetDefaultWidth: integer;
var
  AGrid: TCustomXDBGrid;
  tmpCanvas: TCanvas;
begin
  AGrid := TCustomXDBGrid(Grid);
  if AGrid <> nil then
  begin

    tmpCanvas := GetWorkingCanvas(aGrid.Canvas);
    tmpCanvas.Font := aGrid.Font;

    if FField <> nil then
      Result := CalcColumnFieldWidth(tmpCanvas, xdgTitles in
        aGrid.Options, Title.Caption, Title.Font, FField)
    else
      Result := AGrid.DefaultColWidth;

    if tmpCanvas <> AGrid.Canvas then
      FreeWorkingCanvas(tmpCanvas);

  end
  else
    Result := DEFCOLWIDTH;
end;

function TxColumn.CreateTitle: TxGridColumnTitle;
begin
  Result := TxColumnTitle.Create(Self);
end;

constructor TxColumn.Create(ACollection: TCollection);
var
  AGrid: TCustomXGrid;
begin
  inherited Create(ACollection);
  FPrcWidth := 0;
  FImagesIndexes := TImagesIndexes.Create;
  FxImagesIndexes := TImagesIndexes.Create;
  if ACollection is TxDBGridColumns then
  begin
    AGrid := TxDBGridColumns(ACollection).Grid;
    if (AGrid <> nil) and (csLoading in AGrid.ComponentState) then
      FDesignIndex := Index
    else
      FDesignIndex := 10000;
  end;
  Title.Alignment := taCenter;
end;

destructor TxColumn.Destroy;
begin
  FImagesIndexes.Free;
  FxImagesIndexes.Free;
  inherited Destroy;
end;

function TxColumn.IsDefault: boolean;
begin
  Result := not FDisplayFormatChanged and not FxDisplayFormatChanged and (inherited IsDefault());

end;

procedure TxColumn.LinkField;
var
  AGrid: TCustomXDBGrid;
begin
  AGrid := TCustomXDBGrid(Grid);
  if (AGrid <> nil) and AGrid.FDatalink.Active then
  begin
    Field := AGrid.FDataLink.DataSet.FindField(FFieldName);
    ApplyDisplayFormat;
  end
  else
    Field := nil;
end;

function TxColumn.GetxDefaultDisplayFormat: string;
begin
  Result := '';
  if FxField <> nil then
  begin
    if FxField is TNumericField then
      Result := TNumericField(FxField).DisplayFormat
    else if FxField is TDateTimeField then
      Result := TDateTimeField(FxField).DisplayFormat;
  end;
end;

procedure TxColumn.xLinkField;
var
  AGrid: TCustomxDBGrid;
begin
  AGrid := TCustomxDBGrid(Grid);
  if (AGrid <> nil) and AGrid.Datalink.Active then
  begin
    xField := AGrid.DataLink.DataSet.FindField(FxFieldName);
    xApplyDisplayFormat;
  end
  else
    xField := nil;
end;

function TxColumn.GetxColumnTitle: TxColumnTitle;
begin
  Result := TxColumnTitle(Title);
end;

function TxColumn.GetxDefaultAlignment: TAlignment;
var
  Bs: set of TxColumnButtonStyle;
begin
  bs := [buttonStyle];
  if FxField <> nil then
  begin
    if Grid <> nil then
      Include(bs, TCustomxDbGrid(Grid).DefaultEditorStyle(ButtonStyle, FxField));

    if bs * [cbsxCheckBoxColumn, cbsxButtonColumn] <> [] then
      Result := taCenter
    else
      Result := FxField.Alignment;
  end
  else
    Result := taLeftJustify;

end;

procedure TxColumn.SetDrawImage(AValue: boolean);
begin
  if FDrawImage <> AValue then
  begin
    FDrawImage := AValue;
    ColumnChanged;
  end;
end;

procedure TxColumn.SetxDrawImage(AValue: boolean);
begin
  if FxDrawImage <> AValue then
  begin
    FxDrawImage := AValue;
    ColumnChanged;
  end;
end;

function TxColumn.GetDefaultDisplayFormat: string;
begin
  Result := '';
  if FField <> nil then
  begin
    if FField is TNumericField then
      Result := TNumericField(FField).DisplayFormat
    else if FField is TDateTimeField then
      Result := TDateTimeField(FField).DisplayFormat;
  end;
end;

function TxColumn.GetDefaultValueChecked: string;
begin
  if (FField <> nil) and (FField.Datatype = ftBoolean) then
    Result := BoolToStr(True)
  else
    Result := '1';
end;

function TxColumn.GetDefaultValueUnchecked: string;
begin
  if (FField <> nil) and (FField.DataType = ftBoolean) then
    Result := BoolToStr(False)
  else
    Result := '0';
end;

function TxColumn.GetDefaultReadOnly: boolean;
var
  AGrid: TCustomXDBGrid;
begin
  AGrid := TCustomXDBGrid(Grid);
  Result := ((AGrid <> nil) and (AGrid.ReadOnly)); // or ((FField <> nil) and (FField.ReadOnly));
end;

function TxColumn.GetDefaultVisible: boolean;
begin
  if FField <> nil then
    Result := FField.Visible
  else
    Result := True;
end;

function TxColumn.GetDisplayName: string;
begin
  if FFieldName <> '' then
    Result := FFieldName
  else
    Result := inherited GetDisplayName;
end;

function TxColumn.GetDefaultAlignment: TAlignment;
var
  Bs: set of TxColumnButtonStyle;
begin
  bs := [buttonStyle];
  if Grid <> nil then
    Include(bs, TCustomXDBGrid(Grid).DefaultEditorStyle(ButtonStyle, FField));
  if bs * [cbsxCheckBoxColumn, cbsxButtonColumn] <> [] then
    Result := taCenter
  else
  if FField <> nil then
    Result := FField.Alignment
  else
    Result := taLeftJustify;
end;

{ TxColumnTitle }

function TxColumnTitle.GetDefaultCaption: string;
begin
  with (Column as TxColumn) do
  begin
    if FieldName <> '' then
    begin
      if FField <> nil then
        Result := Field.DisplayName
      else
        Result := Fieldname;
    end
    else
      Result := inherited GetDefaultCaption;
  end;
end;

function TxColumnTitle.GetxDefaultCaption: string;
begin
  with (Column as TxColumn) do
  begin
    if xFieldName <> '' then
    begin
      if FxField <> nil then
        Result := xField.DisplayName
      else
        Result := xFieldName;
    end
    else
      Result := '';
  end;
end;

{ TxBookmarkList }

function TxBookmarkList.GetCount: integer;
begin
  {$ifdef dbgDBGrid}
  DebugLn('%s.GetCount FList.Count=%d',[ClassName, FList.Count]);
  {$endif}
  result := FList.Count;
end;

function TxBookmarkList.GetCurrentRowSelected: boolean;
var
  Bookmark: TBookmark;
begin
  CheckActive;
  Bookmark := FDataset.GetBookmark;
  Result := IndexOf(Bookmark)>=0;
  FDataset.FreeBookmark(Bookmark);
end;

function TxBookmarkList.GetItem(AIndex: Integer): TBookmark;
begin
  Result := TBookmark(FList[AIndex]);
end;

procedure TxBookmarkList.SetCurrentRowSelected(const AValue: boolean);
var
  Bookmark: pointer;
  Index: Integer;
begin
  CheckActive;

  Bookmark := nil;
  TBookmark(Bookmark) := FDataset.GetBookmark; // fetch and increase reference count
  if Bookmark = nil then
    Exit;

  if Find(Bookmark, Index) then begin
    FDataset.FreeBookmark(Bookmark);
    {$ifndef noautomatedbookmark}
    SetLength(TBookmark(Bookmark),0); // decrease reference count
    {$endif noautomatedbookmark}
    if not AValue then begin
      FDataset.FreeBookmark(Pointer(Items[Index]));
      {$ifndef noautomatedbookmark}
      Bookmark := FList[Index];
      SetLength(TBookmark(Bookmark),0); // decrease reference count
      {$endif noautomatedbookmark}
      FList.Delete(Index);
      FGrid.Invalidate;
    end;
  end else begin
    if AValue then begin
      // the reference count of Bookmark was increased above, so it is save to
      // store it here as pointer
      FList.Insert(Index, Bookmark);
      FGrid.Invalidate;
    end else
      FDataset.FreeBookmark(Bookmark);
  end;
end;

procedure TxBookmarkList.CheckActive;
begin
  {$ifdef dbgDBGrid}
  DebugLn('%s.CheckActive', [ClassName]);
  {$endif}
  if not FGrid.FDataLink.Active then
    raise EInvalidGridOperation.Create('Dataset Inactive');

  if FGrid.DataSource.DataSet=FDataset then
    exit;
  FDataset := FGrid.DataSource.DataSet;

  // Note.
  //
  // Some dataset descendants do not implement CompareBookmarks, for these we
  // use MyCompareBookmarks in the hope the allocated bookmark memory is used
  // to hold some kind of record index.
  FUseCompareBookmarks := TMethod(@FDataset.CompareBookmarks).Code<>pointer(@TDataset.CompareBookmarks);

  // Note.
  //
  // fpc help say CompareBookmarks should return -1, 0 or 1 ... which imply that
  // bookmarks should be a sorted array (or list). In this scenario binary search
  // is the prefered method for finding a bookmark.
  //
  // The problem here is that TBufDataset and TSQLQuery (and thus TCustomSQLQuery
  // and TCustomBufDataset) CompareBookmarks only return 0 or -1 (some kind of
  // is this a valid bookmark or not), the result is that it appears as an unsorted
  // list (or array) and binary search should not be used.
  //
  // The weird thing is that if we use MyCompareBookmarks which deals with comparing
  // the memory reserved for bookmarks in the hope bookmarks are just some kind of
  // reocord indexes, currently work fine for TCustomBufDataset derived datasets.
  // however using CompareBookmarks is always the right thing to use where implemented.
  //
  // As Dbgrid should be TDataset implementation agnostic this is a way I found
  // to know if the dataset is derived from TCustomBufDataset or not.
  // Once TCustomBufDataset is fixed, remove this ugly note & hack.
  case FDataset.ClassName of
    'TSQLQuery','TBufDataset','TCustomSQLQuery','TCustomBufDataset':
      FCanDoBinarySearch := false;
    else
      FCanDoBinarySearch := true;
  end;
end;

constructor TxBookmarkList.Create(AGrid: TCustomXDBGrid);
begin
  inherited Create;
  FGrid := AGrid;
  FList := TFPList.Create;
end;

destructor TxBookmarkList.Destroy;
begin
  Clear;
  FreeAndNil(FList);
  inherited Destroy;
end;

procedure TxBookmarkList.Clear;
var
  i: Integer;
  {$ifndef noautomatedbookmark}
  Bookmark: Pointer;
  {$endif}
begin
  for i:=0 to FList.Count-1 do
  begin
    {$ifdef dbgDBGrid}
    DebugLn('%s.Clear', [ClassName]);
    {$endif}
    FDataset.FreeBookmark(Items[i]);
    {$ifndef noautomatedbookmark}
    Bookmark := FList[i];
    SetLength(TBookmark(Bookmark),0); // decrease reference count
    {$endif noautomatedbookmark}
  end;
  FList.Clear;
  FGrid.Invalidate;
end;

procedure TxBookmarkList.Delete;
var
  i: Integer;
  {$ifndef noautomatedbookmark}
  Bookmark: Pointer;
  {$endif}
begin
  {$ifdef dbgDBGrid}
  DebugLn('%s.Delete', [ClassName]);
  {$endif}
  for i := FList.Count-1 downto 0 do begin
    FDataset.GotoBookmark(Items[i]);
    {$ifndef noautomatedbookmark}
    Bookmark := FList[i];
    SetLength(TBookmark(Bookmark),0); // decrease reference count
    {$else}
    FDataset.FreeBookmark(Items[i]);
    {$endif noautomatedbookmark}
    FDataset.Delete;
    FList.Delete(i);
  end;
end;

type
  TDs=class(TDataset)
  end;

function MyCompareBookmarks(ds:Tdataset; b1,b2:pointer): Integer;
begin
  if b1=b2 then
    result := 0
  else
  if b1=nil then
    result := -1
  else
  if b2=nil then
    result := 1
  else begin
    // Note: Tds(ds).bookmarksize is set at creation of TDataSet and does not change
    result := CompareMemRange(b1,b2,Tds(ds).bookmarksize);
  end;
end;

function TxBookmarkList.Find(const Item: TBookmark; var AIndex: Integer): boolean;
var
  L, R, I: Integer;
  CompareRes: Integer;

  procedure BinarySearch;
  begin
    L := 0;
    R := FList.Count - 1;
    while (L <= R) do
    begin
      I := L + (R - L) div 2;
      if FUseCompareBookmarks then
        CompareRes := FDataset.CompareBookmarks(Item, TBookmark(FList[I]))
      else
        CompareRes := MyCompareBookmarks(FDataset, pointer(Item), FList[I]);
      if (CompareRes > 0) then
        L := I + 1
      else
      begin
        R := I - 1;
        if (CompareRes = 0) then
        begin
           Result := True;
           L := I;
        end;
      end;
    end;
    AIndex := L;
  end;

  procedure VisitAll;
  begin
    AIndex := 0;
    i := 0;
    while i<FList.Count do begin
      if FUseCompareBookmarks then
        CompareRes := FDataset.CompareBookmarks(Item, TBookmark(FList[I]))
      else
        CompareRes := MyCompareBookmarks(FDataset, pointer(Item), FList[I]);
      if CompareRes=0 then begin
        result := true;
        AIndex := i;
        exit;
      end;
      inc(i);
    end;
  end;

begin
  {$ifdef dbgDBGrid}
  DebugLn('%s.Find', [ClassName]);
  {$endif}

  Result := False;
  if Item=nil then
    Exit;
  if FCanDoBinarySearch then
    BinarySearch
  else
    VisitAll;
end;

function TxBookmarkList.IndexOf(const Item: TBookmark): Integer;
begin
  {$ifdef dbgDBGrid}
  DebugLn('%s.IndexOf', [ClassName]);
  {$endif}
  if not Find(Item, Result) then
    Result := -1;
end;

function TxBookmarkList.Refresh: boolean;
var
  i: LongInt;
begin
  {$ifdef dbgDBGrid}
  DebugLn('%s.Refresh', [ClassName]);
  {$endif}
  Result := False;
  for i := FList.Count - 1 downto 0 do
    if not FDataset.BookmarkValid(TBookMark(Items[i])) then begin
      Result := True;
      Flist.Delete(i);
    end;
  if Result then
    FGrid.Invalidate;
end;

initialization
  AutoSortDataSet := nil;

end.
