{------------------------------------------------------------------------------}
{                                                                              }
{                              Yuriy Kopnin                                    }
{                            Package VisuaTech                                 }
{                                 LGPL                                         }
{                                                                              }
{------------------------------------------------------------------------------}

unit xDBControls;

{$mode objfpc}{$H+}

interface

uses
  Buttons, DBCtrls,
  Types, Classes, SysUtils, Math, LCLType, LCLProc, LCLIntf,
  GraphType, Graphics, ImgList, ActnList, Controls, StdCtrls, LMessages, Forms,
  Themes, Menus, LResources;

type

  { TxBitBtn }

  TxBitBtn = class(TBitBtn)
  protected
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  end;

  { TEmbedBtnActionLink }

  TEmbedBtnActionLink = class(TControlActionLink)
  protected
    procedure AssignClient(AClient: TObject); override;
    procedure SetGroupIndex(Value: Integer); override;
    procedure SetChecked(Value: Boolean); override;
  public
    function IsCheckedLinked: Boolean; override;
    function IsGroupIndexLinked: Boolean; override;
  end;

  { TEmbededButton }

  TEmbededButton = class(TGraphicControl)
  private
    FGroupIndex: integer;
    FLastDrawDetails: TThemedElementDetails;
    FLayout: TButtonLayout;
    FMargin: integer;
    FSpacing: integer;
    FShortcut: TShortCut;
    FShowAccelChar: boolean;
    FShowCaption: boolean;
    FAllowAllUp: boolean;
    FDown: boolean;
    FDownLoaded: boolean;// value of Down set during loading
    FDragging: boolean;
    FFlat: boolean;
    FMouseInControl: boolean;
    FImageList: TCustomImageList;
    FImageIndex: TImageIndex;
    FDownImageIndex: TImageIndex;
    procedure SetShowCaption(const AValue: boolean);
    procedure UpdateExclusive;
    procedure SetAllowAllUp(Value: boolean);
    procedure SetLayout(const Value: TButtonLayout);
    procedure SetShowAccelChar(Value: boolean);
    procedure CMButtonPressed(var Message: TLMessage); message CM_BUTTONPRESSED;
    procedure CMEnabledChanged(var Message: TLMessage); message CM_ENABLEDCHANGED;
  private
    procedure DoMouseUp(var Message: TLMMouse; Button: TMouseButton);
    function GetDownImageIndex: TImageIndex;
    function GetImageIndex: TImageIndex; virtual;
    function GetImageList: TCustomImageList;
    procedure SetDownImageIndex(AValue: TImageIndex);
    procedure SetImageIndex(AValue: TImageIndex); virtual;
    procedure SetImageList(AValue: TCustomImageList);
    procedure WMLButtonDown(var Message: TLMLButtonDown); message LM_LBUTTONDOWN;
    procedure WMLButtonUp(var Message: TLMLButtonUp); message LM_LBUTTONUP;
    procedure WMLButtonDBLCLK(var Message: TLMLButtonDblClk); message LM_LBUTTONDBLCLK;
  protected
    FState: TButtonState;
    class procedure WSRegisterClass; override;
    function DialogChar(var Message: TLMKey): boolean; override;
    procedure CalculatePreferredSize(var PreferredWidth, PreferredHeight: integer;
      WithThemeSpace: boolean); override;
    procedure MeasureDraw(Draw: boolean; PaintRect: TRect;
      out PreferredWidth, PreferredHeight: integer);
    procedure MouseEnter; override;
    procedure MouseLeave; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: integer); override;
    procedure Paint; override;
    procedure PaintBackground(var PaintRect: TRect); virtual;
    procedure SetDown(Value: boolean);
    procedure SetGroupIndex(const Value: integer);
    procedure SetFlat(const Value: boolean);
    procedure SetMargin(const Value: integer);
    procedure SetSpacing(const Value: integer);
    procedure RealSetText(const Value: TCaption); override;
    procedure UpdateState(InvalidateOnChange: boolean); virtual;
    function GetDrawDetails: TThemedElementDetails; virtual;
    property MouseInControl: boolean read FMouseInControl;
    procedure ActionChange(Sender: TObject; CheckDefaults: boolean); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    class function GetControlClassDefaultSize: TSize; override;
    procedure Loaded; override;
  protected
    function GetTextSize(Drawing: boolean; PaintRect: TRect): TSize; virtual;
    function DrawGlyph(ACanvas: TCanvas; const AClient: TRect;
      const AOffset: TPoint; AState: TButtonState; BiDiFlags: longint): TRect; virtual;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
       override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function FindDownButton: TEmbededButton;
    procedure Click; override; // make Click public
  public
    property AllowAllUp: boolean read FAllowAllUp write SetAllowAllUp default False;
    property Color default clBtnFace;
    property Down: boolean read FDown write SetDown default False;
    property Flat: boolean read FFlat write SetFlat default False;
    property GroupIndex: integer read FGroupIndex write SetGroupIndex default 0;
    property Layout: TButtonLayout read FLayout write SetLayout default blGlyphLeft;
    property Margin: integer read FMargin write SetMargin default -1;
    property ShowAccelChar: boolean
      read FShowAccelChar write SetShowAccelChar default True;
    property ShowCaption: boolean read FShowCaption write SetShowCaption default True;
    property Spacing: integer read FSpacing write SetSpacing default 4;
    property ImageList: TCustomImageList read GetImageList write SetImageList;
    property ImageIndex: TImageIndex read GetImageIndex write SetImageIndex default -1;
    property DownImageIndex: TImageIndex read GetDownImageIndex write SetDownImageIndex default -1;
  end;

  {TxSpeedBtn}
  TxSpeedBtn = class(TEmbededButton)
  published
    property ImageList;
    property ImageIndex;
    property DownImageIndex;

    property Action;
    property Align;
    property AllowAllUp;
    property Anchors;
    property AutoSize;
    property BidiMode;
    property BorderSpacing;
    property Constraints;
    property Caption;
    property Color;
    property Down;
    property Enabled;
    property Flat;
    property Font;
    property GroupIndex;
    property Layout;
    property Margin;
    property Spacing;
    //property Transparent;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseEnter;
    property OnMouseLeave;
    property OnMouseMove;
    property OnMouseUp;
    property OnPaint;
    property OnResize;
    property OnChangeBounds;
    property ShowCaption;
    property ShowHint;
    property ParentBidiMode;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
  end;

  { TEmbededLabel }

  TEmbededLabel = class(TLabel)
  protected
    procedure SetVisible(Value: Boolean); override;
  end;

  { TxCustomLabeledEdit }

  TxCustomLabeledEdit = class(TEdit)
  private
    FLabel: TEmbededLabel;
    FLabelAncorCompanion: TAnchorKind;
    FLabelAlignment: TAlignment;
    function GetLabelCaption: TCaption;
    function GetLabelVisible: Boolean;
    procedure SetLabelAlignment(AValue: TAlignment);
    procedure SetLabelAnchorCompoanion(AValue: TAnchorKind);
    procedure SetLabelCaption(AValue: TCaption);
    procedure SetLabelVisible(AValue: Boolean);
  protected
    procedure DoPositionLabel; virtual;
    procedure CMBiDiModeChanged(var Message: TLMessage); message CM_BIDIMODECHANGED;
    procedure Loaded; override;
    procedure SetParent(NewParent: TWinControl); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
       override;
    procedure CMVisibleChanged(var Msg: TLMessage); message CM_VISIBLECHANGED;
  public
    constructor Create(AOwner: TComponent); override;
    procedure FontChanged(Sender: TObject); override;
    property LabelVisible: Boolean read GetLabelVisible write SetLabelVisible;
    property LabelCaption: TCaption read GetLabelCaption write SetLabelCaption;
    property LabelAncorCompanion: TAnchorKind read FLabelAncorCompanion write SetLabelAnchorCompoanion;
    property LabelAlignment: TAlignment read FLabelAlignment write SetLabelAlignment;
  end;

  TxLabeledEdit = class(TxCustomLabeledEdit)
  published
    property LabelAlignment;
    property LabelCaption;
    property LabelVisible;
    property LabelAncorCompanion;
  end;

  { TxCustomEditButton }

  TxCustomEditButton = class(TxCustomLabeledEdit)
  private
    FButton: TEmbededButton;
    FButtonNeedsFocus: boolean;
    FDirectInput: boolean;
    FIsReadOnly: boolean;
    FOnButtonClick: TNotifyEvent;
    FCanceledFlat: Boolean;
    function GetButtonAction: TBasicAction;
    function GetButtonHint: TTranslateString;
    function GetButtonLayot: TButtonLayout;
    function GetButtonSpacing: Integer;
    function GetButtonWidth: integer;
    function GetButtonCaption: TCaption;
    function GetDirectInput: boolean;
    function GetDownImageIndex: TImageIndex;
    function GetFlat: boolean;
    function GetImageIndex: TImageIndex;
    function GetImageList: TCustomImageList;
    procedure SetButtonAction(AValue: TBasicAction);
    procedure SetButtonHint(const AValue: TTranslateString);
    procedure SetButtonLayot(AValue: TButtonLayout);
    procedure SetButtonSpacing(AValue: Integer);
    procedure SetCanceledFlat(AValue: Boolean);
    procedure SetButtonCaption(AValue: TCaption);
    procedure SetDownImageIndex(AValue: TImageIndex);
    procedure SetImageIndex(AValue: TImageIndex);
    procedure SetButtonNeedsFocus(const AValue: boolean);
    procedure SetButtonWidth(const AValue: integer);
    procedure SetDirectInput(const AValue: boolean);
    procedure SetFlat(const AValue: boolean);
    function GetMinHeight: integer;
    procedure WMSetFocus(var Message: TLMSetFocus); message LM_SETFOCUS;
    procedure WMKillFocus(var Message: TLMKillFocus); message LM_KILLFOCUS;
    function IsImageListStored: boolean;
  protected
    SaveFlat: Boolean;
    procedure CheckButtonVisible;
    function CalcButtonVisible: boolean; virtual;
    function GetReadOnly: boolean; override;
    procedure SetParent(AParent: TWinControl); override;
    procedure SetReadOnly(AValue: boolean); override;
    procedure DoPositionButton; virtual;
    procedure DoButtonClick(Sender: TObject); virtual;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure CMVisibleChanged(var Msg: TLMessage); message CM_VISIBLECHANGED;
    procedure CMEnabledChanged(var Msg: TLMessage); message CM_ENABLEDCHANGED;
    procedure CMBiDiModeChanged(var Message: TLMessage); message CM_BIDIMODECHANGED;
    procedure SetImageList(const AValue: TCustomImageList);
    procedure KeyDown(var Key: word; Shift: TShiftState); override;
    procedure DoEnter; override;
    procedure DoExit; override;
    // New properties.
    property DirectInput: boolean read GetDirectInput write SetDirectInput default True;
    property OnButtonClick: TNotifyEvent read FOnButtonClick write FOnButtonClick;
    property Button: TEmbededButton read FButton;
    property ButtonHint: TTranslateString read GetButtonHint write SetButtonHint;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure FontChanged(Sender: TObject); override;
    procedure Assign(Source: TPersistent); override;
    property Flat: boolean read GetFlat write SetFlat default False;
    property ButtonOnlyWhenFocused: boolean read FButtonNeedsFocus
      write SetButtonNeedsFocus default False;
    property ImageList: TCustomImageList read GetImageList write SetImageList;
    property ImageIndex: TImageIndex read GetImageIndex write SetImageIndex default -1;
    property DownImageIndex: TImageIndex read GetDownImageIndex write SetDownImageIndex default -1;
    property ButtonWidth: integer read GetButtonWidth write SetButtonWidth;
    property ButtonCaption: TCaption read GetButtonCaption write SetButtonCaption;
    property ButtonSpacing: Integer read GetButtonSpacing write SetButtonSpacing;
    property ButtonLayot: TButtonLayout read GetButtonLayot write SetButtonLayot;
    property ButtonAction: TBasicAction read GetButtonAction write SetButtonAction;
    property CanceledFlat: Boolean read FCanceledFlat write SetCanceledFlat;
  end;

  TxEditButton = class(TxCustomEditButton)
  public
    property Button;
  published
    property ButtonOnlyWhenFocused;
    property ButtonWidth;
    property ButtonHint;
    property ButtonSpacing;
    property ButtonLayot;

    property CanceledFlat;
    property ButtonCaption;
    property Flat;
    property ImageList;
    property ImageIndex;
    property DownImageIndex;

    property LabelAlignment;
    property LabelCaption;
    property LabelVisible;
    property LabelAncorCompanion;

    property OnButtonClick;
  end;

  { TxCustomLabeledCombo}
  TxCustomLabeledCombo = class(TComboBox)
  private
    FLabel: TEmbededLabel;
    FLabelAncorCompanion: TAnchorKind;
    FLabelAlignment: TAlignment;
    function GetLabelCaption: TCaption;
    function GetLabelVisible: Boolean;
    procedure SetLabelAlignment(AValue: TAlignment);
    procedure SetLabelAnchorCompoanion(AValue: TAnchorKind);
    procedure SetLabelCaption(AValue: TCaption);
    procedure SetLabelVisible(AValue: Boolean);
  protected
    procedure DoPositionLabel; virtual;
    procedure CMBiDiModeChanged(var Message: TLMessage); message CM_BIDIMODECHANGED;
    procedure Loaded; override;
    procedure SetParent(NewParent: TWinControl); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
       override;
    procedure CMVisibleChanged(var Msg: TLMessage); message CM_VISIBLECHANGED;
  public
    constructor Create(AOwner: TComponent); override;
    procedure FontChanged(Sender: TObject); override;
    property LabelVisible: Boolean read GetLabelVisible write SetLabelVisible;
    property LabelCaption: TCaption read GetLabelCaption write SetLabelCaption;
    property LabelAncorCompanion: TAnchorKind read FLabelAncorCompanion write SetLabelAnchorCompoanion;
    property LabelAlignment: TAlignment read FLabelAlignment write SetLabelAlignment;
  end;

  TxLabeledCombo = class(TxCustomLabeledCombo)
    property LabelAlignment;
    property LabelCaption;
    property LabelVisible;
    property LabelAncorCompanion;
  end;

  { TCustomLabeledDBEdit }

  TCustomLabeledDBEdit = class(TDBEdit)
  private
    FLabel: TEmbededLabel;
    FLabelAncorCompanion: TAnchorKind;
    FLabelAlignment: TAlignment;
    function GetLabelCaption: TCaption;
    function GetLabelVisible: Boolean;
    procedure SetLabelAlignment(AValue: TAlignment);
    procedure SetLabelAnchorCompoanion(AValue: TAnchorKind);
    procedure SetLabelCaption(AValue: TCaption);
    procedure SetLabelVisible(AValue: Boolean);
  protected
    procedure DoPositionLabel; virtual;
    procedure CMBiDiModeChanged(var Message: TLMessage); message CM_BIDIMODECHANGED;
    procedure Loaded; override;
    procedure SetParent(NewParent: TWinControl); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
       override;
    procedure CMVisibleChanged(var Msg: TLMessage); message CM_VISIBLECHANGED;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure FontChanged(Sender: TObject); override;
    property LabelVisible: Boolean read GetLabelVisible write SetLabelVisible;
    property LabelCaption: TCaption read GetLabelCaption write SetLabelCaption;
    property LabelAncorCompanion: TAnchorKind read FLabelAncorCompanion write SetLabelAnchorCompoanion;
    property LabelAlignment: TAlignment read FLabelAlignment write SetLabelAlignment;
  end;

  TLabeledDBEdit = class(TCustomLabeledDBEdit)
  published
    property LabelAlignment;
    property LabelCaption;
    property LabelVisible;
    property LabelAncorCompanion;
  end;

  {TCustomLabeledDBCombo}

  TCustomLabeledDBCombo = class(TDBComboBox)
  private
    FLabel: TEmbededLabel;
    FLabelAncorCompanion: TAnchorKind;
    FLabelAlignment: TAlignment;
    function GetLabelCaption: TCaption;
    function GetLabelVisible: Boolean;
    procedure SetLabelAlignment(AValue: TAlignment);
    procedure SetLabelAnchorCompoanion(AValue: TAnchorKind);
    procedure SetLabelCaption(AValue: TCaption);
    procedure SetLabelVisible(AValue: Boolean);
  protected
    procedure DoPositionLabel; virtual;
    procedure CMBiDiModeChanged(var Message: TLMessage); message CM_BIDIMODECHANGED;
    procedure Loaded; override;
    procedure SetParent(NewParent: TWinControl); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
       override;
    procedure CMVisibleChanged(var Msg: TLMessage); message CM_VISIBLECHANGED;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure FontChanged(Sender: TObject); override;
    property LabelVisible: Boolean read GetLabelVisible write SetLabelVisible;
    property LabelCaption: TCaption read GetLabelCaption write SetLabelCaption;
    property LabelAncorCompanion: TAnchorKind read FLabelAncorCompanion write SetLabelAnchorCompoanion;
    property LabelAlignment: TAlignment read FLabelAlignment write SetLabelAlignment;
  end;

  TLabeledDBCombo = class(TCustomLabeledDBCombo)
  published
    property LabelAlignment;
    property LabelCaption;
    property LabelVisible;
    property LabelAncorCompanion;
  end;

  {TCustomLabeledDBLookupCombo}

  TCustomLabeledDBLookupCombo = class(TDBLookupComboBox)
  private
    FLabel: TEmbededLabel;
    FLabelAncorCompanion: TAnchorKind;
    FLabelAlignment: TAlignment;
    function GetLabelCaption: TCaption;
    function GetLabelVisible: Boolean;
    procedure SetLabelAlignment(AValue: TAlignment);
    procedure SetLabelAnchorCompoanion(AValue: TAnchorKind);
    procedure SetLabelCaption(AValue: TCaption);
    procedure SetLabelVisible(AValue: Boolean);
  protected
    procedure DoPositionLabel; virtual;
    procedure CMBiDiModeChanged(var Message: TLMessage); message CM_BIDIMODECHANGED;
    procedure Loaded; override;
    procedure SetParent(NewParent: TWinControl); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
       override;
    procedure CMVisibleChanged(var Msg: TLMessage); message CM_VISIBLECHANGED;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure FontChanged(Sender: TObject); override;
    property LabelVisible: Boolean read GetLabelVisible write SetLabelVisible;
    property LabelCaption: TCaption read GetLabelCaption write SetLabelCaption;
    property LabelAncorCompanion: TAnchorKind read FLabelAncorCompanion write SetLabelAnchorCompoanion;
    property LabelAlignment: TAlignment read FLabelAlignment write SetLabelAlignment;
  end;

  TLabeledDBLookupCombo = class(TCustomLabeledDBLookupCombo)
  published
    property LabelAlignment;
    property LabelCaption;
    property LabelVisible;
    property LabelAncorCompanion;
  end;

  { TCustomDBEditButton }

  TCustomDBEditButton = class(TCustomLabeledDBEdit)
  private
    FButton: TEmbededButton;
    FButtonNeedsFocus: boolean;
    FDirectInput: boolean;
    FIsReadOnly: boolean;
    FOnButtonClick: TNotifyEvent;
    FCanceledFlat: Boolean;
    function GetButtonAction: TBasicAction;
    function GetButtonHint: TTranslateString;
    function GetButtonLayot: TButtonLayout;
    function GetButtonShowHint: Boolean;
    function GetButtonSpacing: Integer;
    function GetButtonWidth: integer;
    function GetButtonCaption: TCaption;
    function GetDirectInput: boolean;
    function GetDownImageIndex: TImageIndex;
    function GetFlat: boolean;
    function GetImageIndex: TImageIndex;
    function GetImageList: TCustomImageList;
    procedure SetButtonAction(AValue: TBasicAction);
    procedure SetButtonHint(const AValue: TTranslateString);
    procedure SetButtonLayot(AValue: TButtonLayout);
    procedure SetButtonShowHint(AValue: Boolean);
    procedure SetButtonSpacing(AValue: Integer);
    procedure SetCanceledFlat(AValue: Boolean);
    procedure SetButtonCaption(AValue: TCaption);
    procedure SetDownImageIndex(AValue: TImageIndex);
    procedure SetImageIndex(AValue: TImageIndex);
    procedure SetButtonNeedsFocus(const AValue: boolean);
    procedure SetButtonWidth(const AValue: integer);
    procedure SetDirectInput(const AValue: boolean);
    procedure SetFlat(const AValue: boolean);
    function GetMinHeight: integer;
    procedure WMSetFocus(var Message: TLMSetFocus); message LM_SETFOCUS;
    procedure WMKillFocus(var Message: TLMKillFocus); message LM_KILLFOCUS;
    function IsImageListStored: boolean;
  protected
    SaveFlat: Boolean;
    procedure CheckButtonVisible;
    function CalcButtonVisible: boolean; virtual;
    function GetReadOnly: boolean; override;
    procedure SetParent(AParent: TWinControl); override;
    procedure SetReadOnly(AValue: boolean); override;
    procedure DoPositionButton; virtual;
    procedure DoButtonClick(Sender: TObject); virtual;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure CMVisibleChanged(var Msg: TLMessage); message CM_VISIBLECHANGED;
    procedure CMEnabledChanged(var Msg: TLMessage); message CM_ENABLEDCHANGED;
    procedure CMBiDiModeChanged(var Message: TLMessage); message CM_BIDIMODECHANGED;
    procedure SetImageList(const AValue: TCustomImageList);
    procedure KeyDown(var Key: word; Shift: TShiftState); override;
    procedure DoEnter; override;
    procedure DoExit; override;
    // New properties.
    property DirectInput: boolean read GetDirectInput write SetDirectInput default True;
    property OnButtonClick: TNotifyEvent read FOnButtonClick write FOnButtonClick;
    property Button: TEmbededButton read FButton;
    property ButtonHint: TTranslateString read GetButtonHint write SetButtonHint;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure FontChanged(Sender: TObject); override;
    procedure Assign(Source: TPersistent); override;
    property Flat: boolean read GetFlat write SetFlat default False;
    property ButtonOnlyWhenFocused: boolean read FButtonNeedsFocus
      write SetButtonNeedsFocus default False;
    property ImageList: TCustomImageList read GetImageList write SetImageList;
    property ImageIndex: TImageIndex read GetImageIndex write SetImageIndex default -1;
    property DownImageIndex: TImageIndex read GetDownImageIndex write SetDownImageIndex default -1;
    property ButtonWidth: integer read GetButtonWidth write SetButtonWidth;
    property ButtonCaption: TCaption read GetButtonCaption write SetButtonCaption;
    property ButtonSpacing: Integer read GetButtonSpacing write SetButtonSpacing;
    property ButtonLayot: TButtonLayout read GetButtonLayot write SetButtonLayot;
    property ButtonAction: TBasicAction read GetButtonAction write SetButtonAction;
    property ButtonShowHint: Boolean read GetButtonShowHint write SetButtonShowHint;
    property CanceledFlat: Boolean read FCanceledFlat write SetCanceledFlat;
  end;

  TDBEditButton = class(TCustomDBEditButton)
  public
    property Button;
  published
    property ButtonOnlyWhenFocused;
    property ButtonWidth;
    property ButtonHint;
    property ButtonSpacing;
    property ButtonLayot;
    property ButtonShowHint;

    property CanceledFlat;
    property ButtonCaption;
    property Flat;
    property ImageList;
    property ImageIndex;
    property DownImageIndex;

    property LabelAlignment;
    property LabelCaption;
    property LabelVisible;
    property LabelAncorCompanion;

    property OnButtonClick;
  end;

  { TxDBCheckBox }

  TxDBCheckBox = class(TDBCheckBox)
  public
    constructor Create(TheOwner: TComponent); override;
  published
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property TabStop default True;
  end;

implementation

uses Dialogs, db;

{TEmbededButton}

const
  UpState: array[boolean] of TButtonState =
    (
    {False} bsUp, // mouse in control = false
    {True } bsHot // mouse in contorl = true
    );

{ TxBitBtn }

procedure TxBitBtn.ActionChange(Sender: TObject; CheckDefaults: Boolean);
var
  Actn: TCustomAction;
begin
  inherited ActionChange(Sender, CheckDefaults);

  if Sender is TCustomAction then
  begin
    Actn := TCustomAction(Sender);
    if (Actn.ActionList <> nil) then
    begin
      if Actn.ActionList.Images <> nil then
        Images := Actn.ActionList.Images;
      ImageIndex:= Actn.ImageIndex;
    end;
  end;
end;

{ TxCustomLabeledCombo }

function TxCustomLabeledCombo.GetLabelCaption: TCaption;
begin
  Result := FLabel.Caption;
end;

function TxCustomLabeledCombo.GetLabelVisible: Boolean;
begin
  Result := FLabel.Visible;
end;

procedure TxCustomLabeledCombo.SetLabelAlignment(AValue: TAlignment);
begin
  if FLabelAlignment=AValue then Exit;
  FLabelAlignment:=AValue;
  if (FLabelAncorCompanion <> akLeft) and (FLabelAncorCompanion <> akRight) then
    FLabel.Alignment:= FLabelAlignment;
end;

procedure TxCustomLabeledCombo.SetLabelAnchorCompoanion(AValue: TAnchorKind);
begin
  if FLabelAncorCompanion=AValue then Exit;
  FLabelAncorCompanion := AValue;
  if AValue = akRight then
  begin
    FLabel.Alignment := taRightJustify;
    FLabel.AutoSize := True;
    FLabel.Layout := tlCenter;
  end
  else
  begin
    FLabel.Alignment := FLabelAlignment;
    FLabel.Layout := tlTop;
  end;
  DoPositionLabel;
end;

procedure TxCustomLabeledCombo.SetLabelCaption(AValue: TCaption);
begin
  FLabel.Caption := AValue;
end;

procedure TxCustomLabeledCombo.SetLabelVisible(AValue: Boolean);
begin
  FLabel.Visible := AValue;
end;

procedure TxCustomLabeledCombo.DoPositionLabel;
var
  L: Integer;
begin
  if (FLabel <> nil) and (FLabel.Visible) then
  begin
    FLabel.Parent := Parent;
    L := 2;
    if (FLabelAncorCompanion = akRight) or (FLabelAncorCompanion = akLeft) then
    begin
      L := 4;
      FLabel.Layout:= tlCenter;
    end
    else
    begin
      FLabel.Layout:= tlTop;
      if FLabelAncorCompanion = akTop then L := 0;
    end;

    FLabel.AnchorToCompanion(FLabelAncorCompanion, L, Self);
  end;
end;

procedure TxCustomLabeledCombo.CMBiDiModeChanged(var Message: TLMessage);
begin
  inherited;
  DoPositionLabel;
end;

procedure TxCustomLabeledCombo.Loaded;
begin
  inherited Loaded;
  DoPositionLabel;
end;

procedure TxCustomLabeledCombo.SetParent(NewParent: TWinControl);
begin
  inherited SetParent(NewParent);
  if FLabel <> nil then  DoPositionLabel;
end;

procedure TxCustomLabeledCombo.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then
  begin
    if AComponent = FLabel then
      FLabel := nil;
  end;
end;

procedure TxCustomLabeledCombo.CMVisibleChanged(var Msg: TLMessage);
begin
  inherited CMVisibleChanged(Msg);
  FLabel.Visible := Visible;
end;

constructor TxCustomLabeledCombo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLabelAlignment:= taLeftJustify;
  FLabelAncorCompanion := akBottom;
  FLabel := TEmbededLabel.Create(Self);
  FLabel.AutoSize := True;
  FLabel.ControlStyle := FLabel.ControlStyle + [csNoDesignSelectable];
  FLabel.Cursor := crDefault;
  FLabel.FreeNotification(Self);
  FLabel.Caption := 'Catption';
  Width := 80;
end;

procedure TxCustomLabeledCombo.FontChanged(Sender: TObject);
begin
  inherited FontChanged(Sender);
  FLabel.Font.Assign(Font);
end;

{ TxDBCheckBox }

constructor TxDBCheckBox.Create(TheOwner: TComponent);
begin
  inherited Create(TheOwner);
  TabStop := True;
end;

{ TxCustomLabeledEdit }

function TxCustomLabeledEdit.GetLabelCaption: TCaption;
begin
  Result := FLabel.Caption;
end;

function TxCustomLabeledEdit.GetLabelVisible: Boolean;
begin
  Result := FLabel.Visible;
end;

procedure TxCustomLabeledEdit.SetLabelAlignment(AValue: TAlignment);
begin
  if FLabelAlignment=AValue then Exit;
  FLabelAlignment:=AValue;
  if (FLabelAncorCompanion <> akLeft) and (FLabelAncorCompanion <> akRight) then
    FLabel.Alignment:= FLabelAlignment;
end;

procedure TxCustomLabeledEdit.SetLabelAnchorCompoanion(AValue: TAnchorKind);
begin
  if FLabelAncorCompanion=AValue then Exit;
  FLabelAncorCompanion := AValue;
  if AValue = akRight then
  begin
    FLabel.Alignment := taRightJustify;
    FLabel.AutoSize := True;
    FLabel.Layout := tlCenter;
  end
  else
  begin
    FLabel.Alignment := FLabelAlignment;
    FLabel.Layout := tlTop;
  end;
  DoPositionLabel;
end;

procedure TxCustomLabeledEdit.SetLabelCaption(AValue: TCaption);
begin
  FLabel.Caption := AValue;
end;

procedure TxCustomLabeledEdit.SetLabelVisible(AValue: Boolean);
begin
  FLabel.Visible := AValue;
end;

procedure TxCustomLabeledEdit.DoPositionLabel;
var
  L: Integer;
begin
  if (FLabel <> nil) and (FLabel.Visible) then
  begin
    FLabel.Parent := Parent;
    L := 2;
    if (FLabelAncorCompanion = akRight) or (FLabelAncorCompanion = akLeft) then
    begin
      L := 4;
      FLabel.Layout:= tlCenter;
    end
    else
    begin
      FLabel.Layout:= tlTop;
      if FLabelAncorCompanion = akTop then L := 0;
    end;

    FLabel.AnchorToCompanion(FLabelAncorCompanion, L, Self);
  end;
end;

procedure TxCustomLabeledEdit.CMBiDiModeChanged(var Message: TLMessage);
begin
  inherited;
  DoPositionLabel;
end;

procedure TxCustomLabeledEdit.Loaded;
begin
  inherited Loaded;
  DoPositionLabel;
end;

procedure TxCustomLabeledEdit.SetParent(NewParent: TWinControl);
begin
  inherited SetParent(NewParent);
  if FLabel <> nil then  DoPositionLabel;
end;

procedure TxCustomLabeledEdit.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then
  begin
    if AComponent = FLabel then
      FLabel := nil;
  end;
end;

procedure TxCustomLabeledEdit.CMVisibleChanged(var Msg: TLMessage);
begin
  inherited CMVisibleChanged(Msg);
  FLabel.Visible := Visible;
end;

constructor TxCustomLabeledEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLabelAlignment:= taLeftJustify;
  FLabelAncorCompanion := akBottom;
  FLabel := TEmbededLabel.Create(Self);
  FLabel.AutoSize := True;
  FLabel.ControlStyle := FLabel.ControlStyle + [csNoDesignSelectable];
  FLabel.Cursor := crDefault;
  FLabel.FreeNotification(Self);
  FLabel.Caption := 'Caption';
end;

procedure TxCustomLabeledEdit.FontChanged(Sender: TObject);
begin
  inherited FontChanged(Sender);
  FLabel.Font.Assign(Font);
end;

{ TEmbedBtnActionLink }

procedure TEmbedBtnActionLink.AssignClient(AClient: TObject);
begin
  inherited AssignClient(AClient);
  FClient := AClient as TEmbededButton;
end;

procedure TEmbedBtnActionLink.SetGroupIndex(Value: Integer);
begin
  if IsGroupIndexLinked then TEmbededButton(FClient).GroupIndex := Value;
end;

procedure TEmbedBtnActionLink.SetChecked(Value: Boolean);
begin
  if IsCheckedLinked then TEmbededButton(FClient).Down := Value;
end;

function TEmbedBtnActionLink.IsCheckedLinked: Boolean;
var
  Btn: TEmbededButton;
begin
  Btn := TEmbededButton(FClient);
  Result := inherited IsCheckedLinked
            and (Btn.GroupIndex <> 0)
            and Btn.AllowAllUp
            and (Btn.Down = (Action as TCustomAction).Checked);
end;

function TEmbedBtnActionLink.IsGroupIndexLinked: Boolean;
var
  Btn: TEmbededButton;
begin
  Btn:=TEmbededButton(FClient);
  Result := (Btn is TEmbededButton) and
    (Btn.GroupIndex = (Action as TCustomAction).GroupIndex);
end;

{ TCustomLabeledDBEdit }

function TCustomLabeledDBEdit.GetLabelCaption: TCaption;
begin
  Result := FLabel.Caption;
end;

function TCustomLabeledDBEdit.GetLabelVisible: Boolean;
begin
  Result := FLabel.Visible;
end;

procedure TCustomLabeledDBEdit.SetLabelAlignment(AValue: TAlignment);
begin
  if FLabelAlignment=AValue then Exit;
  FLabelAlignment:=AValue;
  if (FLabelAncorCompanion <> akLeft) and (FLabelAncorCompanion <> akRight) then
    FLabel.Alignment:= FLabelAlignment;
end;

procedure TCustomLabeledDBEdit.SetLabelAnchorCompoanion(AValue: TAnchorKind);
begin
  if FLabelAncorCompanion=AValue then Exit;
  FLabelAncorCompanion := AValue;
  if AValue = akRight then
  begin
    FLabel.Alignment := taRightJustify;
    FLabel.AutoSize := True;
    FLabel.Layout := tlCenter;
  end
  else
  begin
    FLabel.Alignment := FLabelAlignment;
    FLabel.Layout := tlTop;
  end;
  DoPositionLabel;
end;

procedure TCustomLabeledDBEdit.SetLabelCaption(AValue: TCaption);
begin
  FLabel.Caption := AValue;
end;

procedure TCustomLabeledDBEdit.SetLabelVisible(AValue: Boolean);
begin
  FLabel.Visible := AValue;
end;

procedure TCustomLabeledDBEdit.DoPositionLabel;
var
  L: Integer;
begin
  if (FLabel <> nil) and (FLabel.Visible) then
  begin
    FLabel.Parent := Parent;
    L := 2;
    if (FLabelAncorCompanion = akRight) or (FLabelAncorCompanion = akLeft) then
    begin
      L := 4;
      FLabel.Layout:= tlCenter;
    end
    else
    begin
      FLabel.Layout:= tlTop;
      if FLabelAncorCompanion = akTop then L := 0;
    end;

    FLabel.AnchorToCompanion(FLabelAncorCompanion, L, Self);
  end;
end;

procedure TCustomLabeledDBEdit.CMBiDiModeChanged(var Message: TLMessage);
begin
  inherited;
  DoPositionLabel;
end;

procedure TCustomLabeledDBEdit.Loaded;
begin
  inherited Loaded;
  DoPositionLabel;
end;

procedure TCustomLabeledDBEdit.SetParent(NewParent: TWinControl);
begin
  inherited SetParent(NewParent);
  if FLabel <> nil then  DoPositionLabel;
end;

procedure TCustomLabeledDBEdit.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then
  begin
    if AComponent = FLabel then
      FLabel := nil;
  end;
end;

procedure TCustomLabeledDBEdit.CMVisibleChanged(var Msg: TLMessage);
begin
  inherited CMVisibleChanged(Msg);
  FLabel.Visible := Visible;
end;

procedure TCustomLabeledDBEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  if (Key = VK_DELETE) and (Shift = [ssCtrl]) then
  begin
    if EditCanModify then
    begin
      Key := 0;
      if Field.DataSet.State = dsBrowse then Field.DataSet.Edit;
      Field.Clear;
    end;
  end
  else
  inherited KeyDown(Key, Shift);
end;

constructor TCustomLabeledDBEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLabelAlignment:= taLeftJustify;
  FLabelAncorCompanion := akBottom;
  FLabel := TEmbededLabel.Create(Self);
  FLabel.AutoSize := True;
  FLabel.ControlStyle := FLabel.ControlStyle + [csNoDesignSelectable];
  FLabel.Cursor := crDefault;
  FLabel.FreeNotification(Self);
  FLabel.Caption := 'Caption';
end;

procedure TCustomLabeledDBEdit.FontChanged(Sender: TObject);
begin
  inherited FontChanged(Sender);
  FLabel.Font.Assign(Font);
end;

{ TEmbededLabel }

procedure TEmbededLabel.SetVisible(Value: Boolean);
begin
  inherited SetVisible(Value);
  if Visible then
    Parent := TControl(Owner).Parent
  else
    Parent := nil
end;

{ TCustomLabeledDBCombo }

function TCustomLabeledDBCombo.GetLabelCaption: TCaption;
begin
  Result := FLabel.Caption;
end;

function TCustomLabeledDBCombo.GetLabelVisible: Boolean;
begin
  Result := FLabel.Visible;
end;

procedure TCustomLabeledDBCombo.SetLabelAlignment(AValue: TAlignment);
begin
  if FLabelAlignment=AValue then Exit;
  FLabelAlignment:=AValue;
  if (FLabelAncorCompanion <> akLeft) and (FLabelAncorCompanion <> akRight) then
    FLabel.Alignment:= FLabelAlignment;
end;

procedure TCustomLabeledDBCombo.SetLabelAnchorCompoanion(AValue: TAnchorKind);
begin
  if FLabelAncorCompanion=AValue then Exit;
  FLabelAncorCompanion := AValue;
  if AValue = akRight then
  begin
    FLabel.Alignment := taRightJustify;
    FLabel.AutoSize := True;
    FLabel.Layout := tlCenter;
  end
  else
  begin
    FLabel.Alignment := FLabelAlignment;
    FLabel.Layout := tlTop;
  end;
  DoPositionLabel;
end;

procedure TCustomLabeledDBCombo.SetLabelCaption(AValue: TCaption);
begin
  FLabel.Caption := AValue;
end;

procedure TCustomLabeledDBCombo.SetLabelVisible(AValue: Boolean);
begin
  FLabel.Visible := AValue;
end;

procedure TCustomLabeledDBCombo.DoPositionLabel;
var
  L: Integer;
begin
  if (FLabel <> nil) and (FLabel.Visible) then
  begin
    FLabel.Parent := Parent;
    L := 2;
    if (FLabelAncorCompanion = akRight) or (FLabelAncorCompanion = akLeft) then
    begin
      L := 4;
      FLabel.Layout:= tlCenter;
    end
    else
    begin
      FLabel.Layout:= tlTop;
      if FLabelAncorCompanion = akTop then L := 0;
    end;

    FLabel.AnchorToCompanion(FLabelAncorCompanion, L, Self);
  end;
end;

procedure TCustomLabeledDBCombo.CMBiDiModeChanged(var Message: TLMessage);
begin
  inherited;
  DoPositionLabel;
end;

procedure TCustomLabeledDBCombo.Loaded;
begin
  inherited Loaded;
  DoPositionLabel;
end;

procedure TCustomLabeledDBCombo.SetParent(NewParent: TWinControl);
begin
  inherited SetParent(NewParent);
  if FLabel <> nil then  DoPositionLabel;
end;

procedure TCustomLabeledDBCombo.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then
  begin
    if AComponent = FLabel then
      FLabel := nil;
  end;
end;

procedure TCustomLabeledDBCombo.CMVisibleChanged(var Msg: TLMessage);
begin
  inherited CMVisibleChanged(Msg);
  FLabel.Visible := Visible;
end;

procedure TCustomLabeledDBCombo.KeyDown(var Key: Word; Shift: TShiftState);
begin
  if (Key = VK_DELETE) and (Shift = [ssCtrl]) then
  begin
    if Field.CanModify then
    begin
      Key := 0;
      if Field.DataSet.State = dsBrowse then Field.DataSet.Edit;
      Field.Clear;
    end;
  end
  else
  inherited KeyDown(Key, Shift);
end;

constructor TCustomLabeledDBCombo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLabelAlignment:= taLeftJustify;
  FLabelAncorCompanion := akBottom;
  FLabel := TEmbededLabel.Create(Self);
  FLabel.AutoSize := True;
  FLabel.ControlStyle := FLabel.ControlStyle + [csNoDesignSelectable];
  FLabel.Cursor := crDefault;
  FLabel.FreeNotification(Self);
  FLabel.Caption := 'Catption';
  Width := 80;
end;

procedure TCustomLabeledDBCombo.FontChanged(Sender: TObject);
begin
  inherited FontChanged(Sender);
  FLabel.Font.Assign(Font);
end;

{ TCustomLabeledDBLookupCombo }

function TCustomLabeledDBLookupCombo.GetLabelCaption: TCaption;
begin
  Result := FLabel.Caption;
end;

function TCustomLabeledDBLookupCombo.GetLabelVisible: Boolean;
begin
  Result := FLabel.Visible;
end;

procedure TCustomLabeledDBLookupCombo.SetLabelAlignment(AValue: TAlignment);
begin
  if FLabelAlignment=AValue then Exit;
  FLabelAlignment:=AValue;
  if (FLabelAncorCompanion <> akLeft) and (FLabelAncorCompanion <> akRight) then
    FLabel.Alignment:= FLabelAlignment;
end;

procedure TCustomLabeledDBLookupCombo.SetLabelAnchorCompoanion(AValue: TAnchorKind);
begin
  if FLabelAncorCompanion=AValue then Exit;
  FLabelAncorCompanion := AValue;
  if AValue = akRight then
  begin
    FLabel.Alignment := taRightJustify;
    FLabel.AutoSize := True;
    FLabel.Layout := tlCenter;
  end
  else
  begin
    FLabel.Alignment := FLabelAlignment;
    FLabel.Layout := tlTop;
  end;
  DoPositionLabel;
end;

procedure TCustomLabeledDBLookupCombo.SetLabelCaption(AValue: TCaption);
begin
  FLabel.Caption := AValue;
end;

procedure TCustomLabeledDBLookupCombo.SetLabelVisible(AValue: Boolean);
begin
  FLabel.Visible := AValue;
end;

procedure TCustomLabeledDBLookupCombo.DoPositionLabel;
var
  L: Integer;
begin
  if (FLabel <> nil) and (FLabel.Visible) then
  begin
    FLabel.Parent := Parent;
    L := 2;
    if (FLabelAncorCompanion = akRight) or (FLabelAncorCompanion = akLeft) then
    begin
      L := 4;
      FLabel.Layout:= tlCenter;
    end
    else
    begin
      FLabel.Layout:= tlTop;
      if FLabelAncorCompanion = akTop then L := 0;
    end;

    FLabel.AnchorToCompanion(FLabelAncorCompanion, L, Self);
  end;
end;

procedure TCustomLabeledDBLookupCombo.CMBiDiModeChanged(var Message: TLMessage);
begin
  inherited;
  DoPositionLabel;
end;

procedure TCustomLabeledDBLookupCombo.Loaded;
begin
  inherited Loaded;
  DoPositionLabel;
end;

procedure TCustomLabeledDBLookupCombo.SetParent(NewParent: TWinControl);
begin
  inherited SetParent(NewParent);
  if FLabel <> nil then  DoPositionLabel;
end;

procedure TCustomLabeledDBLookupCombo.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then
  begin
    if AComponent = FLabel then
      FLabel := nil;
  end;
end;

procedure TCustomLabeledDBLookupCombo.CMVisibleChanged(var Msg: TLMessage);
begin
  inherited CMVisibleChanged(Msg);
  FLabel.Visible := Visible;
end;

procedure TCustomLabeledDBLookupCombo.KeyDown(var Key: Word; Shift: TShiftState
  );
begin
  if (Key = VK_DELETE) and (Shift = [ssCtrl]) then
  begin
    if Field.CanModify then
    begin
      Key := 0;
      if Field.DataSet.State = dsBrowse then Field.DataSet.Edit;
      Field.Clear;
    end;
  end
  else
  inherited KeyDown(Key, Shift);
end;

constructor TCustomLabeledDBLookupCombo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLabelAlignment:= taLeftJustify;
  FLabelAncorCompanion := akBottom;
  FLabel := TEmbededLabel.Create(Self);
  FLabel.AutoSize := True;
  FLabel.ControlStyle := FLabel.ControlStyle + [csNoDesignSelectable];
  FLabel.Cursor := crDefault;
  FLabel.FreeNotification(Self);
  FLabel.Caption := 'Catption';
  Width := 80;
end;

procedure TCustomLabeledDBLookupCombo.FontChanged(Sender: TObject);
begin
  inherited FontChanged(Sender);
  FLabel.Font.Assign(Font);
end;

{------------------------------------------------------------------------------
  Method:  TEmbededButton.Create
  Params:  none
  Returns: Nothing

  Constructor for the class.
 ------------------------------------------------------------------------------}
constructor TEmbededButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  with GetControlClassDefaultSize do
    SetInitialBounds(0, 0, CX, CY);
  ControlStyle := ControlStyle + [csCaptureMouse] -
    [csSetCaption, csClickEvents, csOpaque];
  FLayout := blGlyphLeft;
  FAllowAllUp := False;
  FMouseInControl := False;
  FDragging := False;
  FShowAccelChar := True;
  FSpacing := 4;
  FMargin := -1;
  Color := clBtnFace;
  FShowCaption := True;
  FImageIndex := -1;
  FDownImageIndex := -1;
end;

destructor TEmbededButton.Destroy;
begin
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.FindDownButton: TEmbededButton;

  Searches the speed button with Down=true and the same GroupIndex.
 ------------------------------------------------------------------------------}
function TEmbededButton.FindDownButton: TEmbededButton;

  function FindDown(AWinControl: TWinControl): TEmbededButton;
  var
    i: integer;
    Child: TControl;
    Button: TEmbededButton;
  begin
    if AWinControl = nil then
      Exit(nil);
    for i := 0 to AWinControl.ControlCount - 1 do
    begin
      Child := AWinControl.Controls[i];
      if Child is TEmbededButton then
      begin
        Button := TEmbededButton(Child);
        if (Button.GroupIndex = GroupIndex) and (Button.Down) then
          Exit(Button);
      end;
      if Child is TWinControl then
      begin
        Result := FindDown(TWinControl(Child));
        if Result <> nil then
          Exit;
      end;
    end;
    Result := nil;
  end;

begin
  if Down or (GroupIndex = 0) then
    exit(Self);
  Result := FindDown(GetFirstParentForm(Self));
end;

procedure TEmbededButton.Click;
begin
  inherited Click;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.SetAllowAllUp
  Params: Value:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.SetAllowAllUp(Value: boolean);
begin
  if FAllowAllUp <> Value then
  begin
    FAllowAllUp := Value;
    UpdateExclusive;
  end;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.SetDown
  Params: Value:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.SetDown(Value: boolean);
var
  OldState: TButtonState;
  OldDown: boolean;
begin
  //since Down needs GroupIndex, then we need to wait that all properties
  //loaded before we continue
  if (csLoading in ComponentState) then
  begin
    FDownLoaded := Value;
    exit;
  end
  else
  begin
    if FGroupIndex = 0 then
      Value := False;
    if FDown <> Value then
    begin
      if FDown and not FAllowAllUp then
        Exit;
      OldDown := FDown;
      FDown := Value;
      OldState := FState;
      if FDown then
        FState := bsExclusive
      else
        FState := UpState[FMouseInControl];
      if (OldDown <> FDown) or (OldState <> FState) then
        Invalidate;
      if Value then
        UpdateExclusive;
    end;
  end;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.SetFlat
  Params: Value:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.SetFlat(const Value: boolean);
begin
  if FFlat <> Value then
  begin
    FFlat := Value;
    Invalidate;
  end;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.SetGroupIndex
  Params: Value:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.SetGroupIndex(const Value: integer);
begin
  if FGroupIndex <> Value then
  begin
    FGroupIndex := Value;
    UpdateExclusive;
  end;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.SetMargin
  Params: Value:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.SetMargin(const Value: integer);
begin
  if FMargin <> Value then
  begin
    FMargin := Value;
    Invalidate;
  end;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.SetSpacing
  Params: Value:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.SetSpacing(const Value: integer);
begin
  if FSpacing <> Value then
  begin
    FSpacing := Value;
    Invalidate;
  end;
end;

{------------------------------------------------------------------------------
  procedure TEmbededButton.RealSetText(const Value: TCaption);
 ------------------------------------------------------------------------------}
procedure TEmbededButton.RealSetText(const Value: TCaption);
begin
  if Caption = Value then
    Exit;
  inherited RealSetText(Value);

  Invalidate;
end;

{------------------------------------------------------------------------------
  procedure TEmbededButton.UpdateState(InvalidateOnChange: boolean);
 ------------------------------------------------------------------------------}
procedure TEmbededButton.UpdateState(InvalidateOnChange: boolean);
var
  OldState: TButtonState;
begin
  OldState := FState;
  if not IsEnabled then
  begin
    FState := bsDisabled;
    FDragging := False;
  end
  else
  begin
    if FState = bsDisabled then
    begin
      if FDown and (GroupIndex <> 0) then
        FState := bsExclusive
      else
        FState := UpState[FMouseInControl];
    end
    else
    if (FState in [bsHot, bsDown]) and (not FMouseInControl) and
      (not FDragging) and (not FDown) then
    begin
      // return to normal
      FState := bsUp;
    end
    else
    if (FState = bsUp) and FMouseInControl then
      FState := bsHot;
  end;
  if FState <> OldState then
    if (Action is TCustomAction) then
      TCustomAction(Action).Checked := FState = bsDown;
  //if InvalidateOnChange then DebugLn(['TxCustomSpeedButton.UpdateState ',DbgSName(Self),' InvalidateOnChange=',InvalidateOnChange,' StateChange=',FState<>OldState]);
  if InvalidateOnChange and ((FState <> OldState) or not
    ThemedElementDetailsEqual(FLastDrawDetails, GetDrawDetails)) then
    Invalidate;
end;

{------------------------------------------------------------------------------
  function TEmbededButton.GetDrawDetails: TThemedElementDetails;
 ------------------------------------------------------------------------------}
function TEmbededButton.GetDrawDetails: TThemedElementDetails;

  function ButtonPart: TThemedButton;
  begin
    // tbPushButtonNormal, tbPushButtonHot, tbPushButtonPressed,
    // tbPushButtonDisabled, tbPushButtonDefaulted

    // no check states available
    Result := tbPushButtonNormal;
    if not IsEnabled then
      Result := tbPushButtonDisabled
    else
    if FState in [bsDown, bsExclusive] then
      Result := tbPushButtonPressed
    else
    if FState = bsHot then
      Result := tbPushButtonHot
    else
      Result := tbPushButtonNormal;
  end;

  function ToolButtonPart: TThemedToolBar;
  begin
    // ttbButtonNormal, ttbButtonHot, ttbButtonPressed, ttbButtonDisabled
    // ttbButtonChecked, ttbButtonCheckedHot
    if not IsEnabled then
      Result := ttbButtonDisabled
    else
    begin
      if Down then
      begin // checked states
        if FMouseInControl then
          Result := ttbButtonCheckedHot
        else
          Result := ttbButtonChecked;
      end
      else
      begin
        if FState in [bsDown, bsExclusive] then
          Result := ttbButtonPressed
        else
        if FState = bsHot then
          Result := ttbButtonHot
        else
          Result := ttbButtonNormal;
      end;
    end;
  end;

begin
  if Flat then
    Result := ThemeServices.GetElementDetails(ToolButtonPart)
  else
    Result := ThemeServices.GetElementDetails(ButtonPart);
end;

procedure TEmbededButton.ActionChange(Sender: TObject; CheckDefaults: boolean);
var
  Actn: TCustomAction;
begin
  inherited ActionChange(Sender, CheckDefaults);

  if Sender is TCustomAction then
  begin
    Actn := TCustomAction(Sender);
    if CheckDefaults or (Self.GroupIndex = 0) then
        Self.GroupIndex := Actn.GroupIndex;
    if (Actn.ActionList <> nil) then
    begin
      if Actn.ActionList.Images <> nil then
      begin
        if ImageList = nil then ImageList := Actn.ActionList.Images;
      end;

      ImageIndex:= Actn.ImageIndex;
    end;
  end;
end;

function TEmbededButton.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TEmbedBtnActionLink;
end;

class function TEmbededButton.GetControlClassDefaultSize: TSize;
begin
  Result.CX := 23;
  Result.CY := 22;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.UpdateExclusive
  Params: none
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.UpdateExclusive;
var
  msg: TLMessage;
begin
  if (FGroupIndex <> 0) and (Parent <> nil) and (not (csLoading in ComponentState)) then
  begin
    Msg.Msg := CM_ButtonPressed;
    Msg.WParam := FGroupIndex;
    Msg.LParam := PtrInt(Self);
    Msg.Result := 0;
    Parent.Broadcast(Msg);
  end;
end;

{------------------------------------------------------------------------------
  Function: TEmbededButton.GetGlyph
  Params: none
  Returns:  The bitmap

 ------------------------------------------------------------------------------}
procedure TEmbededButton.SetShowCaption(const AValue: boolean);
begin
  if FShowCaption = AValue then
    exit;
  FShowCaption := AValue;
  invalidate;
end;

function TEmbededButton.DialogChar(var Message: TLMKey): boolean;
begin
  Result := False;
  // Sometimes LM_CHAR is received instead of LM_SYSCHAR, maybe intentionally
  // (LCL handles it) or maybe sent by mistake. In either case exit.
  if (Message.Msg <> LM_SYSCHAR) or not FShowAccelChar then
    Exit;
  if IsAccel(Message.CharCode, Caption) then
  begin
    Result := True;
    if GroupIndex <> 0 then
      SetDown(not FDown);
    Click;
  end
  else
    Result := inherited DialogChar(Message);
end;

procedure TEmbededButton.CalculatePreferredSize(
  var PreferredWidth, PreferredHeight: integer; WithThemeSpace: boolean);
var
  r: TRect;
begin
  r := Rect(0, 0, 0, 0);
  MeasureDraw(False, r, PreferredWidth, PreferredHeight);
end;

procedure TEmbededButton.MeasureDraw(Draw: boolean; PaintRect: TRect;
  out PreferredWidth, PreferredHeight: integer);
var
  GlyphWidth, GlyphHeight: integer;
  Offset, OffsetCap: TPoint;
  ClientSize, TotalSize, TextSize, GlyphSize: TSize;
  M, S: integer;
  SIndex: longint;
  TMP: string;
  TextFlags: integer;
  DrawDetails: TThemedElementDetails;
  FixedWidth: boolean;
  FixedHeight: boolean;
  TextRect: TRect;
  HasGlyph: boolean;
  HasText: boolean;
  CurLayout: TButtonLayout;
begin
  DrawDetails := GetDrawDetails;

  PreferredWidth := 0;
  PreferredHeight := 0;
  HasGlyph := False;

  if Draw then
  begin
    FLastDrawDetails := DrawDetails;
    PaintBackground(PaintRect);
    FixedWidth := True;
    FixedHeight := True;
  end
  else
  begin
    FixedWidth := WidthIsAnchored;
    FixedHeight := HeightIsAnchored;
  end;
  ClientSize.cx := PaintRect.Right - PaintRect.Left;
  ClientSize.cy := PaintRect.Bottom - PaintRect.Top;

  GlyphSize.cx := 16;
  GlyphSize.cy := 16;

  if ImageList <> nil then
  begin
    GlyphSize.cx := ImageList.Width;
    GlyphSize.cy := ImageList.Height;
    if (ImageIndex >= 0) and (ImageIndex < ImageList.Count) then HasGlyph := True ;
  end;
  GlyphWidth := GlyphSize.CX;
  GlyphHeight := GlyphSize.CY;

  CurLayout := BidiAdjustButtonLayout(UseRightToLeftReading, Layout);
  if ShowCaption and (Caption <> '') then
  begin
    TextRect := PaintRect;
    // for wordbreak compute the maximum size for the text
    if Margin > 0 then
      InflateRect(TextRect, -Margin, -Margin);
    if HasGlyph then
    begin
      if (Spacing >= 0) then
        if CurLayout in [blGlyphLeft, blGlyphRight] then
          Dec(TextRect.Right, Spacing)
        else
          Dec(TextRect.Bottom, Spacing);
      if CurLayout in [blGlyphLeft, blGlyphRight] then
        Dec(TextRect.Right, GlyphWidth)
      else
        Dec(TextRect.Bottom, GlyphHeight);
    end;
    if not FixedWidth then
    begin
      TextRect.Left := 0;
      TextRect.Right := High(TextRect.Right) div 2;
    end;
    if not FixedHeight then
    begin
      TextRect.Top := 0;
      TextRect.Bottom := High(TextRect.Bottom) div 2;
    end;
    TextSize := GetTextSize(Draw, TextRect);
  end
  else
  begin
    TextSize.cx := 0;
    TextSize.cy := 0;
  end;
  HasText := (TextSize.cx <> 0) or (TextSize.cy <> 0);

  if Caption <> '' then
  begin
    TMP := Caption;
    SIndex := DeleteAmpersands(TMP);
    if SIndex > 0 then
      if SIndex <= Length(TMP) then
      begin
        FShortcut := Ord(TMP[SIndex]);
      end;
  end;

  if HasGlyph and HasText then
    S := Spacing
  else
    S := 0;

  M := Margin;
  if not Draw then
  begin
    if M < 0 then
      M := 2;
    if S < 0 then
      S := M;
  end;

  // Calculate caption and glyph layout
  if M = -1 then
  begin
    // auto compute margin to center content
    if S = -1 then
    begin
      // use the same value for Spacing and Margin
      TotalSize.cx := TextSize.cx + GlyphWidth;
      TotalSize.cy := TextSize.cy + GlyphHeight;
      if Layout in [blGlyphLeft, blGlyphRight] then
        M := (ClientSize.cx - TotalSize.cx) div 3
      else
        M := (ClientSize.cy - TotalSize.cy) div 3;
      S := M;
    end
    else
    begin
      // fixed Spacing and center content
      TotalSize.cx := GlyphWidth + S + TextSize.cx;
      TotalSize.cy := GlyphHeight + S + TextSize.cy;
      if Layout in [blGlyphLeft, blGlyphRight] then
        M := (ClientSize.cx - TotalSize.cx) div 2
      else
        M := (ClientSize.cy - TotalSize.cy) div 2;
    end;
  end
  else
  begin
    // fixed Margin
    if S = -1 then
    begin
      // use the rest for Spacing between Glyph and Caption
      TotalSize.cx := ClientSize.cx - (Margin + GlyphWidth);
      TotalSize.cy := ClientSize.cy - (Margin + GlyphHeight);
      if Layout in [blGlyphLeft, blGlyphRight] then
        S := (TotalSize.cx - TextSize.cx) div 2
      else
        S := (TotalSize.cy - TextSize.cy) div 2;
    end;
  end;

  //debugln(['TxCustomSpeedButton.MeasureDraw AAA3 ',DbgSName(Self),' PaintRect=',dbgs(PaintRect),' GlyphSize=',GlyphWidth,'x',GlyphHeight,' TextSize=',TextSize.cx,'x',TextSize.cy,' S=',S,' M=',M]);

  if Draw then
  begin
    case CurLayout of
      blGlyphLeft:
      begin
        Offset.X := M;
        Offset.Y := (ClientSize.cy - GlyphHeight) div 2;
        OffsetCap.X := Offset.X + GlyphWidth + S;
        OffsetCap.Y := (ClientSize.cy - TextSize.cy) div 2 - 1;
        if (FState = bsDown) or (Down = true) then
        begin
          OffsetCap.Y := OffsetCap.Y + 1;
          OffsetCap.X := OffsetCap.X + 1;
        end;
      end;
      blGlyphRight:
      begin
        Offset.X := ClientSize.cx - M - GlyphWidth;
        Offset.Y := (ClientSize.cy - GlyphHeight) div 2;
        OffsetCap.X := Offset.X - S - TextSize.cx;
        OffsetCap.Y := (ClientSize.cy - TextSize.cy) div 2 - 1;
        if (FState = bsDown) or (Down = true) then
        begin
          OffsetCap.Y := OffsetCap.Y + 1;
          OffsetCap.X := OffsetCap.X + 1;
        end;
      end;
      blGlyphTop:
      begin
        Offset.X := (ClientSize.cx - GlyphWidth) div 2;
        Offset.Y := M;
        OffsetCap.X := (ClientSize.cx - TextSize.cx) div 2;
        OffsetCap.Y := Offset.Y + GlyphHeight + S;
      end;
      blGlyphBottom:
      begin
        Offset.X := (ClientSize.cx - GlyphWidth) div 2;
        Offset.Y := ClientSize.cy - M - GlyphHeight;
        OffsetCap.X := (ClientSize.cx - TextSize.cx) div 2;
        OffsetCap.Y := Offset.Y - S - TextSize.cy;
      end;
    end;

    DrawGlyph(Canvas, PaintRect, Offset, FState, 0);

    if FShowCaption and (Caption <> '') then
    begin
      with PaintRect, OffsetCap do
      begin
        Left := Left + X;
        Top := Top + Y;

      end;

      TextFlags := DT_LEFT or DT_TOP;
      if UseRightToLeftReading then
        TextFlags := TextFlags or DT_RTLREADING;

      if Draw then
        ThemeServices.DrawText(Canvas, DrawDetails, Caption, PaintRect,
          TextFlags, 0);
    end;
  end
  else
  begin
    // measuring, not drawing
    case CurLayout of
      blGlyphLeft, blGlyphRight:
      begin
        PreferredWidth := 2 * M + S + GlyphWidth + TextSize.cx;
        PreferredHeight := 2 * M + Max(GlyphHeight, TextSize.cy);
      end;
      blGlyphTop, blGlyphBottom:
      begin
        PreferredWidth := 2 * M + Max(GlyphWidth, TextSize.cx);
        PreferredHeight := 2 * M + S + GlyphHeight + TextSize.cy;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.Paint
  Params: none
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.Paint;
var
  PaintRect: TRect;
  PreferredWidth: integer;
  PreferredHeight: integer;
begin
  UpdateState(False);
  //if FGlyph = nil then exit;

  PaintRect := ClientRect;
  MeasureDraw(True, PaintRect, PreferredWidth, PreferredHeight);

  inherited Paint;
end;

procedure TEmbededButton.PaintBackground(var PaintRect: TRect);
begin
  if ThemeServices.HasTransparentParts(FLastDrawDetails) then
  begin
    Canvas.Brush.Color := Color;
    Canvas.FillRect(PaintRect);
  end;
  ThemeServices.DrawElement(Canvas.Handle, FLastDrawDetails, PaintRect);
  //PaintRect := ThemeServices.ContentRect(Canvas.Handle, FLastDrawDetails, PaintRect);
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.MouseDown
  Params: Button:
          Shift:
          X, Y:
  Returns:  nothing
 ------------------------------------------------------------------------------}
procedure TEmbededButton.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: integer);
begin
  inherited MouseDown(Button, Shift, X, Y);
  if csDesigning in ComponentState then
    exit;

  if (Button = mbLeft) and IsEnabled then
  begin
    if not FDown then
    begin
      FState := bsDown;
      if (Action is TCustomAction) then
        TCustomAction(Action).Checked := False;
      Invalidate;
    end;
    FDragging := True;
  end;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.MouseMove
  Params: Shift:
          X, Y:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.MouseMove(Shift: TShiftState; X, Y: integer);
var
  NewState: TButtonState;
begin
  inherited MouseMove(Shift, X, Y);
  if csDesigning in ComponentState then
    exit;

  if FDragging then
  begin
    //DebugLn('Trace:FDragging is true');
    if FDown then
      NewState := bsExclusive
    else
    begin
      if (X >= 0) and (X < Width) and (Y >= 0) and (Y < Height) then
        NewState := bsDown
      else
        NewState := UpState[FMouseInControl];
    end;

    if NewState <> FState then
    begin
      //debugln(['TxCustomSpeedButton.MouseMove ',DbgSName(Self),' fState=',ord(fstate),' NewState=',ord(NewState)]);
      FState := NewState;
      Invalidate;
    end;
  end;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.MouseUp
  Params: Button:
          Shift:
          X, Y:
  Returns:  nothing
 ------------------------------------------------------------------------------}
procedure TEmbededButton.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
end;

{------------------------------------------------------------------------------
       TEmbededButton DoMouseUp  "Event Handler"
------------------------------------------------------------------------------}
procedure TEmbededButton.DoMouseUp(var Message: TLMMouse; Button: TMouseButton);
begin
  if not (csNoStdEvents in ControlStyle) then
    with Message do
      MouseUp(Button, KeysToShiftState(Keys), XPos, YPos);
end;

function TEmbededButton.GetDownImageIndex: TImageIndex;
begin
  Result := FDownImageIndex;
end;

function TEmbededButton.GetImageIndex: TImageIndex;
begin
  Result := FImageIndex;
end;

function TEmbededButton.GetImageList: TCustomImageList;
begin
  Result := FImageList;
end;

procedure TEmbededButton.SetDownImageIndex(AValue: TImageIndex);
begin
  if FDownImageIndex <> AValue then
    FDownImageIndex := AValue;
end;

procedure TEmbededButton.SetImageIndex(AValue: TImageIndex);
begin
  if FImageIndex = AValue then
    Exit;
  FImageIndex := AValue;
  Invalidate;
end;

procedure TEmbededButton.SetImageList(AValue: TCustomImageList);
begin
  if FImageList <> AValue then
  begin
    FImageList := AValue;
    if FImageList <> nil then FImageList.FreeNotification(Self);
    Invalidate;
  end;
end;

procedure TEmbededButton.WMLButtonDown(var Message: TLMLButtonDown);
begin
  inherited;

  // because csClickEvents is not set no csClicked is set in the inherited method
  Include(FControlState, csClicked);
end;

procedure TEmbededButton.WMLButtonDBLCLK(var Message: TLMLButtonDblClk);
begin
  inherited;
  // if in a group, raise dblclick event, otherwise translate to click event
  if Down then
    DblClick
  else
    Click;
end;

class procedure TEmbededButton.WSRegisterClass;
begin
  inherited WSRegisterClass;
  //RegisterCustomXSpeedButton;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.WMLButtonUp
  Params: Message
  Returns: Nothing

  Mouse event handler
 ------------------------------------------------------------------------------}
procedure TEmbededButton.WMLButtonUp(var Message: TLMLButtonUp);
var
  OldState: TButtonState;
  NeedClick: boolean;
begin
  //DebugLn('TxCustomSpeedButton.WMLButtonUp A ',DbgSName(Self),' csCaptureMouse=',DbgS(csCaptureMouse in ControlStyle),' csClicked=',DbgS(csClicked in ControlState));
  if (csCaptureMouse in ControlStyle) and (mbLeft in CaptureMouseButtons) then
  begin
    {$IFDEF VerboseMouseCapture}
    DebugLn('TxCustomSpeedButton.WMLButtonUp ', Name, ':', ClassName);
    {$ENDIF}
    MouseCapture := False;
  end;

  NeedClick := False;

  if not (csDesigning in ComponentState) and FDragging then
  begin
    OldState := FState;
    FDragging := False;

    if FGroupIndex = 0 then
    begin
      FState := UpState[FMouseInControl];
      if OldState <> FState then
        Invalidate;
    end
    else
    if (Message.XPos >= 0) and (Message.XPos < Width) and (Message.YPos >= 0) and
      (Message.YPos < Height) then
    begin
      SetDown(not FDown);
      NeedClick := True;
    end;
  end;

  DoMouseUp(Message, mbLeft);

  if csClicked in ControlState then
  begin
    Exclude(FControlState, csClicked);
    //DebugLn('TxCustomSpeedButton.WMLButtonUp B ',dbgs(ClientRect.Left),',',dbgs(ClientRect.Top),',',dbgs(ClientRect.Right),',',dbgs(ClientRect.Bottom),' ',dbgs(Message.Pos.X),',',dbgs(Message.Pos.Y));
    if PtInRect(ClientRect, SmallPointToPoint(Message.Pos)) then
    begin
      //DebugLn('TxCustomSpeedButton.WMLButtonUp C');
      // Important: Calling Click can invoke modal dialogs, so call this as last
      NeedClick := False;
      Click;
    end;
  end;

  if NeedClick then
    Click;
  //DebugLn('TxCustomSpeedButton.WMLButtonUp END');
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.SetLayout
  Params: Value: new layout value
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.SetLayout(const Value: TButtonLayout);
begin
  if Value <> FLayout then
  begin
    FLayout := Value;
    Invalidate;
  end;
end;

procedure TEmbededButton.SetShowAccelChar(Value: boolean);
begin
  if FShowAccelChar <> Value then
  begin
    FShowAccelChar := Value;
    Invalidate;
  end;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.CMButtonPressed
  Params: Message:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.CMButtonPressed(var Message: TLMessage);
var
  Sender: TEmbededButton;
begin
  if csDestroying in ComponentState then
    exit;
  if Message.WParam = WParam(FGroupIndex) then
  begin
    Sender := TEmbededButton(Message.LParam);
    if Sender <> Self then
    begin
      if Sender.Down and FDown then
      begin
        FDown := False;
        FState := UpState[FMouseInControl];
        Invalidate;
      end;
      FAllowAllUp := Sender.AllowAllUp;
    end;
  end;
end;

procedure TEmbededButton.Loaded;
begin
  inherited Loaded;
  UpdateExclusive;
  if FDownLoaded then
    SetDown(FDownLoaded);
end;

function TEmbededButton.GetTextSize(Drawing: boolean; PaintRect: TRect): TSize;
var
  TMP: string;
  TXTStyle: TTextStyle;
  Flags: cardinal;
begin
  if FShowCaption and (Caption <> '') then
  begin
    TMP := Caption;
    TXTStyle := Canvas.TextStyle;
    TXTStyle.Opaque := False;
    TXTStyle.Clipping := True;
    TXTStyle.ShowPrefix := ShowAccelChar;
    TXTStyle.Alignment := taLeftJustify;
    TXTStyle.Layout := tlTop;
    TXTStyle.RightToLeft := UseRightToLeftReading;
    TXTStyle.SystemFont := Canvas.Font.IsDefault;//Match System Default Style
    DeleteAmpersands(TMP);

    Flags := DT_CalcRect;
    if not TXTStyle.SingleLine then
      Inc(Flags, DT_WordBreak);

    DrawText(Canvas.Handle, PChar(TMP), Length(TMP), PaintRect, Flags);
    Result.CY := PaintRect.Bottom - PaintRect.Top;
    Result.CX := PaintRect.Right - PaintRect.Left;
  end
  else
  begin
    Result.CY := 0;
    Result.CX := 0;
  end;
end;

function TEmbededButton.DrawGlyph(ACanvas: TCanvas; const AClient: TRect;
  const AOffset: TPoint; AState: TButtonState; BiDiFlags: longint): TRect;
var
  ImgIndex: integer;
begin
  Result := AClient;
  if ImageList = nil then
    Exit;

  if FState = bsDown then
  begin
    if (DownImageIndex >= 0) and (DownImageIndex < ImageList.Count) then
      ImgIndex := DownImageIndex
    else
      ImgIndex := ImageIndex;
  end
  else
    ImgIndex := ImageIndex;

  if (ImgIndex >= 0) and (ImgIndex < ImageList.Count) then
  begin
    if (AState = bsDown) or (Down = true) then
      ImageList.Draw(ACanvas, AOffset.x + 1, AOffset.y + 1, ImgIndex, Enabled)
    else
     ImageList.Draw(ACanvas, AOffset.x, AOffset.y, ImgIndex, Enabled);
  end;
  {if Assigned(FGlyph) then
  begin
    if (AState = bsDown) or (Down = true) then
      Result := FGlyph.Draw(ACanvas, AClient, point(AOffset.x + 1, AOffset.y + 1), AState, ATransparent, BiDiFlags)
    else
      Result := FGlyph.Draw(ACanvas, AClient, AOffset, AState, ATransparent, BiDiFlags);
  end;}
end;

procedure TEmbededButton.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
  begin
    if AComponent = FImageList then
      ImageList := nil;
  end;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.CMEnabledChanged
  Params: Message:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.CMEnabledChanged(var Message: TLMEssage);
begin
  //Should create a new glyph based on the new state
  UpdateState(True);
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.MouseEnter
  Params:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.MouseEnter;
begin
  if csDesigning in ComponentState then
    exit;
  if not FMouseInControl and IsEnabled and (GetCapture = 0) then
  begin
    FMouseInControl := True;
    UpdateState(True);
    inherited MouseEnter;
  end;
end;

{------------------------------------------------------------------------------
  Method: TEmbededButton.MouseLeave
  Params:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TEmbededButton.MouseLeave;
begin
  if csDesigning in ComponentState then
    exit;
  ///DebugLn(['TxCustomSpeedButton.MouseLeave ',DbgSName(Self),' FMouseInControl=',FMouseInControl,' FDragging=',FDragging]);
  if FMouseInControl then
  begin
    FMouseInControl := False;
    if IsEnabled then
    begin
      if FDragging and (not MouseCapture) then
      begin
        // something fetched our mouse capture
        FDragging := False;
      end;
      UpdateState(True);
      inherited MouseLeave;
    end;
  end;
end;

{ TCustomEditButton }

constructor TCustomDBEditButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCanceledFlat := False;
  FDirectInput := True;
  FButton := TEmbededButton.Create(Self);
  FButton.Width := Self.Height;
  FButton.Height := Self.Height;
  FButton.FreeNotification(Self);
  CheckButtonVisible;
  FButton.OnClick := @DoButtonClick;
  FButton.Cursor := crDefault;
  FButton.ControlStyle := FButton.ControlStyle + [csNoDesignSelectable];
  //ControlStyle := ControlStyle - [csSetCaption];
  {FLabelAncorCompanion := akBottom;
  FLabel := TEmbededLabel.Create(Self);
  FLabel.AutoSize := True;
  FLabel.ControlStyle := FLabel.ControlStyle + [csNoDesignSelectable];
  FLabel.Cursor := crArrow;
  FLabel.FreeNotification(Self);
  FLabel.Caption := 'Catption';}
end;

destructor TCustomDBEditButton.Destroy;
begin
  FreeAndNil(FButton);
  inherited Destroy;
end;

procedure TCustomDBEditButton.FontChanged(Sender: TObject);
begin
  inherited FontChanged(Sender);
  FButton.Font.Assign(Font);
end;

procedure TCustomDBEditButton.Assign(Source: TPersistent);
begin
  inherited Assign(Source);
  ImageList := TCustomDBEditButton(Source).ImageList;
  ImageIndex := TCustomDBEditButton(Source).ImageIndex;
end;

function TCustomDBEditButton.GetButtonWidth: integer;
begin
  Result := FButton.Width;
end;

function TCustomDBEditButton.GetButtonCaption: TCaption;
begin
  Result := FButton.Caption;
end;

function TCustomDBEditButton.GetButtonHint: TTranslateString;
begin
  Result := FButton.Hint;
end;

function TCustomDBEditButton.GetButtonAction: TBasicAction;
begin
  Result := FButton.Action;
end;

function TCustomDBEditButton.GetButtonLayot: TButtonLayout;
begin
  Result := FButton.Layout;
end;

function TCustomDBEditButton.GetButtonShowHint: Boolean;
begin
  Result := FButton.ShowHint;
end;

function TCustomDBEditButton.GetButtonSpacing: Integer;
begin
  Result := FButton.Spacing;
end;

function TCustomDBEditButton.GetDirectInput: boolean;
begin
  Result := FDirectInput;
end;

function TCustomDBEditButton.GetDownImageIndex: TImageIndex;
begin
  Result := FButton.DownImageIndex;
end;

function TCustomDBEditButton.GetFlat: boolean;
begin
  if Assigned(FButton) then
    Result := FButton.Flat
  else
    Result := False;
end;

function TCustomDBEditButton.GetImageIndex: TImageIndex;
begin
  Result := FButton.ImageIndex;
end;

function TCustomDBEditButton.GetImageList: TCustomImageList;
begin
  Result := FButton.ImageList;
end;

procedure TCustomDBEditButton.SetButtonAction(AValue: TBasicAction);
begin
  FButton.Action := AValue;
end;

function TCustomDBEditButton.CalcButtonVisible: boolean;
begin
  Result := (csdesigning in ComponentState) or
    (Visible and (Focused or not FButtonNeedsFocus));
end;

procedure TCustomDBEditButton.CheckButtonVisible;
begin
  if Assigned(FButton) then
    FButton.Visible := CalcButtonVisible;
end;

procedure TCustomDBEditButton.SetButtonHint(const AValue: TTranslateString);
begin
  FButton.Hint := AValue;
end;

procedure TCustomDBEditButton.SetButtonLayot(AValue: TButtonLayout);
begin
  FButton.Layout := AValue;
end;

procedure TCustomDBEditButton.SetButtonShowHint(AValue: Boolean);
begin
  FButton.ShowHint := AValue;
end;

procedure TCustomDBEditButton.SetButtonSpacing(AValue: Integer);
begin
  FButton.Spacing := AValue;
end;

procedure TCustomDBEditButton.SetCanceledFlat(AValue: Boolean);
begin
  if FCanceledFlat=AValue then Exit;
  FCanceledFlat:=AValue;
end;

procedure TCustomDBEditButton.SetButtonCaption(AValue: TCaption);
begin
  FButton.Caption := AValue;
end;

procedure TCustomDBEditButton.SetDownImageIndex(AValue: TImageIndex);
begin
  FButton.DownImageIndex := AValue;
end;

procedure TCustomDBEditButton.SetImageIndex(AValue: TImageIndex);
begin
  FButton.ImageIndex := AValue;
end;

procedure TCustomDBEditButton.SetButtonNeedsFocus(const AValue: boolean);
begin
  if FButtonNeedsFocus <> AValue then
  begin
    FButtonNeedsFocus := AValue;
    CheckButtonVisible;
  end;
end;

procedure TCustomDBEditButton.SetButtonWidth(const AValue: integer);
begin
  if AValue < Self.Height then
    FButton.Width := Self.Height
  else
    FButton.Width := AValue;
end;

procedure TCustomDBEditButton.SetDirectInput(const AValue: boolean);
begin
  FDirectInput := AValue;
  inherited SetReadOnly((not FDirectInput) or (FIsReadOnly));
end;

procedure TCustomDBEditButton.SetFlat(const AValue: boolean);
begin
  if Assigned(FButton) then
    FButton.Flat := AValue;
end;

procedure TCustomDBEditButton.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then
  begin
    if AComponent = FButton then
      FButton := nil
  end;
end;

procedure TCustomDBEditButton.CMVisibleChanged(var Msg: TLMessage);
begin
  inherited CMVisibleChanged(Msg);
  CheckButtonVisible;
end;

procedure TCustomDBEditButton.CMEnabledChanged(var Msg: TLMessage);
begin
  inherited CMEnabledChanged(Msg);
  if (FButton <> nil) then
    FButton.Enabled := Enabled;
end;

procedure TCustomDBEditButton.CMBiDiModeChanged(var Message: TLMessage);
begin
  inherited;
  DoPositionButton;
end;

procedure TCustomDBEditButton.SetImageList(const AValue: TCustomImageList);
begin
  FButton.ImageList := AValue;
end;

procedure TCustomDBEditButton.KeyDown(var Key: word; Shift: TShiftState);
begin
  if (Key = VK_RETURN) and (ssCtrl in Shift) and Assigned(FOnButtonClick) then
    FButton.Click
  else
  if (Key = VK_DELETE) and (Shift = [ssCtrl]) then
  begin
    if EditCanModify then
    begin
      Key := 0;
      if Field.DataSet.State = dsBrowse then Field.DataSet.Edit;
      Field.Clear;
    end;
  end
  else
    inherited KeyDown(Key, Shift);
end;

procedure TCustomDBEditButton.DoEnter;
begin
  if FCanceledFlat then
  begin
    SaveFlat := Flat;
    if Flat then Flat:= False;
  end;
  inherited DoEnter;
end;

procedure TCustomDBEditButton.DoExit;
begin
  if FCanceledFlat then Flat:= SaveFlat;
  inherited DoExit;
end;

function TCustomDBEditButton.GetMinHeight: integer;
begin
  Result := 23;
end;

procedure TCustomDBEditButton.DoButtonClick(Sender: TObject);
begin
  if not Focused then SetFocus;

  if Assigned(FOnButtonClick) then
    FOnButtonClick(Self);
end;

procedure TCustomDBEditButton.Loaded;
begin
  inherited Loaded;
  DoPositionButton;
  CheckButtonVisible;
end;

procedure TCustomDBEditButton.WMKillFocus(var Message: TLMKillFocus);
begin
  CheckButtonVisible;
  inherited;
end;

function TCustomDBEditButton.IsImageListStored: boolean;
begin
  Result := FButton.ImageList <> nil;
end;

function TCustomDBEditButton.GetReadOnly: boolean;
begin
  Result := FIsReadOnly;
end;

procedure TCustomDBEditButton.SetParent(AParent: TWinControl);
begin
  inherited SetParent(AParent);
  if FButton <> nil then
  begin
    DoPositionButton;
    CheckButtonVisible;
  end;
end;

procedure TCustomDBEditButton.SetReadOnly(AValue: boolean);
begin
  FIsReadOnly := AValue;
  if Assigned(FButton) then
    FButton.Enabled := Enabled;
  inherited SetReadOnly(FIsReadOnly or (not DirectInput));
end;

procedure TCustomDBEditButton.DoPositionButton;
begin
  if FButton <> nil then
  begin
    FButton.Parent := Parent;
    if BiDiMode = bdLeftToRight then
      FButton.AnchorToCompanion(akLeft, 0, Self)
    else
      FButton.AnchorToCompanion(akRight, 0, Self);
  end;
end;

procedure TCustomDBEditButton.WMSetFocus(var Message: TLMSetFocus);
begin
  CheckButtonVisible;
  inherited;
end;

{ TxCustomEditButton }

constructor TxCustomEditButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCanceledFlat := False;
  FDirectInput := True;
  FButton := TEmbededButton.Create(Self);
  FButton.Width := Self.Height;
  FButton.Height := Self.Height;
  FButton.FreeNotification(Self);
  CheckButtonVisible;
  FButton.OnClick := @DoButtonClick;
  FButton.Cursor := crDefault;
  FButton.ControlStyle := FButton.ControlStyle + [csNoDesignSelectable];
  //ControlStyle := ControlStyle - [csSetCaption];
  {FLabelAncorCompanion := akBottom;
  FLabel := TEmbededLabel.Create(Self);
  FLabel.AutoSize := True;
  FLabel.ControlStyle := FLabel.ControlStyle + [csNoDesignSelectable];
  FLabel.Cursor := crArrow;
  FLabel.FreeNotification(Self);
  FLabel.Caption := 'Catption';}
end;

destructor TxCustomEditButton.Destroy;
begin
  FreeAndNil(FButton);
  inherited Destroy;
end;

procedure TxCustomEditButton.FontChanged(Sender: TObject);
begin
  inherited FontChanged(Sender);
  FButton.Font.Assign(Font);
end;

procedure TxCustomEditButton.Assign(Source: TPersistent);
begin
  inherited Assign(Source);
  ImageList := TxCustomEditButton(Source).ImageList;
  ImageIndex := TxCustomEditButton(Source).ImageIndex;
end;

function TxCustomEditButton.GetButtonWidth: integer;
begin
  Result := FButton.Width;
end;

function TxCustomEditButton.GetButtonCaption: TCaption;
begin
  Result := FButton.Caption;
end;

function TxCustomEditButton.GetButtonHint: TTranslateString;
begin
  Result := FButton.Hint;
end;

function TxCustomEditButton.GetButtonAction: TBasicAction;
begin
  Result := FButton.Action;
end;

function TxCustomEditButton.GetButtonLayot: TButtonLayout;
begin
  Result := FButton.Layout;
end;

function TxCustomEditButton.GetButtonSpacing: Integer;
begin
  Result := FButton.Spacing;
end;

function TxCustomEditButton.GetDirectInput: boolean;
begin
  Result := FDirectInput;
end;

function TxCustomEditButton.GetDownImageIndex: TImageIndex;
begin
  Result := FButton.DownImageIndex;
end;

function TxCustomEditButton.GetFlat: boolean;
begin
  if Assigned(FButton) then
    Result := FButton.Flat
  else
    Result := False;
end;

function TxCustomEditButton.GetImageIndex: TImageIndex;
begin
  Result := FButton.ImageIndex;
end;

function TxCustomEditButton.GetImageList: TCustomImageList;
begin
  Result := FButton.ImageList;
end;

procedure TxCustomEditButton.SetButtonAction(AValue: TBasicAction);
begin
  FButton.Action := AValue;
end;

function TxCustomEditButton.CalcButtonVisible: boolean;
begin
  Result := (csdesigning in ComponentState) or
    (Visible and (Focused or not FButtonNeedsFocus));
end;

procedure TxCustomEditButton.CheckButtonVisible;
begin
  if Assigned(FButton) then
    FButton.Visible := CalcButtonVisible;
end;

procedure TxCustomEditButton.SetButtonHint(const AValue: TTranslateString);
begin
  FButton.Hint := AValue;
end;

procedure TxCustomEditButton.SetButtonLayot(AValue: TButtonLayout);
begin
  FButton.Layout := AValue;
end;

procedure TxCustomEditButton.SetButtonSpacing(AValue: Integer);
begin
  FButton.Spacing := AValue;
end;

procedure TxCustomEditButton.SetCanceledFlat(AValue: Boolean);
begin
  if FCanceledFlat=AValue then Exit;
  FCanceledFlat:=AValue;
end;

procedure TxCustomEditButton.SetButtonCaption(AValue: TCaption);
begin
  FButton.Caption := AValue;
end;

procedure TxCustomEditButton.SetDownImageIndex(AValue: TImageIndex);
begin
  FButton.DownImageIndex := AValue;
end;

procedure TxCustomEditButton.SetImageIndex(AValue: TImageIndex);
begin
  FButton.ImageIndex := AValue;
end;

procedure TxCustomEditButton.SetButtonNeedsFocus(const AValue: boolean);
begin
  if FButtonNeedsFocus <> AValue then
  begin
    FButtonNeedsFocus := AValue;
    CheckButtonVisible;
  end;
end;

procedure TxCustomEditButton.SetButtonWidth(const AValue: integer);
begin
  if AValue < Self.Height then
    FButton.Width := Self.Height
  else
    FButton.Width := AValue;
end;

procedure TxCustomEditButton.SetDirectInput(const AValue: boolean);
begin
  FDirectInput := AValue;
  inherited SetReadOnly((not FDirectInput) or (FIsReadOnly));
end;

procedure TxCustomEditButton.SetFlat(const AValue: boolean);
begin
  if Assigned(FButton) then
    FButton.Flat := AValue;
end;

procedure TxCustomEditButton.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then
  begin
    if AComponent = FButton then
      FButton := nil
  end;
end;

procedure TxCustomEditButton.CMVisibleChanged(var Msg: TLMessage);
begin
  inherited CMVisibleChanged(Msg);
  CheckButtonVisible;
end;

procedure TxCustomEditButton.CMEnabledChanged(var Msg: TLMessage);
begin
  inherited CMEnabledChanged(Msg);
  if (FButton <> nil) then
    FButton.Enabled := Enabled;
end;

procedure TxCustomEditButton.CMBiDiModeChanged(var Message: TLMessage);
begin
  inherited;
  DoPositionButton;
end;

procedure TxCustomEditButton.SetImageList(const AValue: TCustomImageList);
begin
  FButton.ImageList := AValue;
end;

procedure TxCustomEditButton.KeyDown(var Key: word; Shift: TShiftState);
begin
  if (Key = VK_RETURN) and (ssCtrl in Shift) and Assigned(FOnButtonClick) then
    FButton.Click
  else
    inherited KeyDown(Key, Shift);
end;

procedure TxCustomEditButton.DoEnter;
begin
  if FCanceledFlat then
  begin
    SaveFlat := Flat;
    if Flat then Flat:= False;
  end;
  inherited DoEnter;
end;

procedure TxCustomEditButton.DoExit;
begin
  if FCanceledFlat then Flat:= SaveFlat;
  inherited DoExit;
end;

function TxCustomEditButton.GetMinHeight: integer;
begin
  Result := 23;
end;

procedure TxCustomEditButton.DoButtonClick(Sender: TObject);
begin
  if not Focused then SetFocus;

  if Assigned(FOnButtonClick) then
    FOnButtonClick(Self);
end;

procedure TxCustomEditButton.Loaded;
begin
  inherited Loaded;
  DoPositionButton;
  CheckButtonVisible;
end;

procedure TxCustomEditButton.WMKillFocus(var Message: TLMKillFocus);
begin
  CheckButtonVisible;
  inherited;
end;

function TxCustomEditButton.IsImageListStored: boolean;
begin
  Result := FButton.ImageList <> nil;
end;

function TxCustomEditButton.GetReadOnly: boolean;
begin
  Result := FIsReadOnly;
end;

procedure TxCustomEditButton.SetParent(AParent: TWinControl);
begin
  inherited SetParent(AParent);
  if FButton <> nil then
  begin
    DoPositionButton;
    CheckButtonVisible;
  end;
end;

procedure TxCustomEditButton.SetReadOnly(AValue: boolean);
begin
  FIsReadOnly := AValue;
  if Assigned(FButton) then
    FButton.Enabled := Enabled;
  inherited SetReadOnly(FIsReadOnly or (not DirectInput));
end;

procedure TxCustomEditButton.DoPositionButton;
begin
  if FButton <> nil then
  begin
    FButton.Parent := Parent;
    if BiDiMode = bdLeftToRight then
      FButton.AnchorToCompanion(akLeft, 0, Self)
    else
      FButton.AnchorToCompanion(akRight, 0, Self);
  end;
end;

procedure TxCustomEditButton.WMSetFocus(var Message: TLMSetFocus);
begin
  CheckButtonVisible;
  inherited;
end;

end.
