{------------------------------------------------------------------------------}
{                                                                              }
{                           Author Yuriy Kopnin                                }
{                            Package VisuaTech                                 }
{                                 LGPL                                         }
{                                                                              }
{------------------------------------------------------------------------------}

unit FormDesig;

{$mode objfpc}{$H+}
{$M+}

interface

uses
  Classes, SysUtils, Forms, Controls, LMessages, lclintf, LCLType, DesSelection,
  Graphics, PropEdits, PropEditUtils, ObjectInspector, Menus, types,
  StdCtrls, LazUTF8, LCLProc, TypInfo, SuFrame, ComponentEditors, ExtCtrls;

type
  TGrabberMarker = (gmNone, gmLeftTop, gmLeftBottom, gmRightTop, gmRightBottom,
    gmLeft, gmTop, gmBottom, gmRight);

  TDesignerManager = class;

  TComponentCaption = class(TLabel)
  protected
    procedure SetBounds(aLeft, aTop, aWidth, aHeight: integer); override;
  public
    constructor Create(TheOwner: TComponent); override;
  end;

  { TNonVisual }

  TNonVisual = class(TCustomControl)
  private
    FComponent: TComponent;
    FCaption: TComponentCaption;
    FDesignerManager: TDesignerManager;
  protected
    Bmp: TBitmap;
    procedure WMMove(var Message: TLMMove); message LM_MOVE;
    procedure WMSize(var Message: TLMSize); message LM_SIZE;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure SetComponent(Value: TComponent);
    procedure SetParent(NewParent: TWinControl); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
      override;
    procedure Paint; override;
    procedure SetLabelBounds(ALeft: Integer; ATop: Integer);
  public
    constructor Create(AOwner: TComponent; DesManager: TDesignerManager);
    destructor Destroy; override;
    procedure UpdateCaption;
    property Component: TComponent read FComponent write SetComponent;
    property DesignerManager: TDesignerManager read FDesignerManager;
  end;

  TJITMethods = class;

  { TJITMethod }

  TJITMethod = class
  private
    FMethod: TMethod;
    FOwner: TJITMethods;
    FTheMethodName: shortstring;
  public
    TypeData: PTypeData;

    constructor Create(AnOwner: TJITMethods; const aMethodName: shortstring; ATypeData: PTypeData);
    destructor Destroy; override;
    property Method: TMethod read FMethod;
    property TheMethodName: shortstring read FTheMethodName;
    property Owner: TJITMethods read FOwner;
  end;


  { TJITMethods }

  TJITMethods = class
  private
    fClearing: boolean;
    fMethods: TList;
    procedure InternalAdd(const AMethod: TJITMethod);
    procedure InternalRemove(const AMethod: TJITMethod);
  public
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    function Count: Integer;
    function Get(Index: Integer): TJITMethod;
    function Add(const aMethodName: shortstring; ATypeData: PTypeData): TJITMethod;
    function Find(const aMethodName: shortstring): TJITMethod;
    function Delete(aMethod: TJITMethod): boolean;
    function Delete(const aMethodName: shortstring): boolean;
    function Rename(const OldMethodName, NewMethodName: shortstring): boolean;
  end;

  { TDesignerHook }
  TMoveObjectMethod =(movKontur, movObject);

  TDesignerHook = class(TComponentEditorDesigner)
  private
     FDM: TDataModule;
     FFrame: TFrame;
     FDesignerSelections: TDesignerSelections;
     FDesignerManager: TDesignerManager;
     FMarkerBmp: TBitmap;
     FDrawMarkers: Boolean;
     FMovSelection: Boolean;
     FSizeOperation: TGrabberMarker;
     FMovePos: TPoint;
     FModified: Boolean;
     FReadComponentList: TFPList;
     FHintTimer: TTimer;
     FHintWIndow: THintWindow;
     Canvas: TCanvas;
     FMoveRectList: TList;
     FTempMoveKontur: Boolean;
     procedure OnHintTimer(Sender : TObject);
  protected
     LastFormCursor: TCursor;
     LastMouseMovePos: TPoint;
     SizingControl: TControl;
     //ShiftKeyState: TShiftState;
     FReading: TComponent;
     FCR: Boolean;
     CaptureRect: TRect;
     CaptControl: TWinControl;
     OperationControl: TControl;
     MouseDownShift: TShiftState;
     MouseDownComponent: TComponent;
     procedure ClearMoveRect;
     function GetPropertyEditorHook: TPropertyEditorHook; override;
     procedure SetForm(AForm: TCustomForm);
     function PaintControl(Sender: TControl; TheMessage: TLMPaint): Boolean;
     procedure PaintSelections;
     procedure DrawCaptureRect;
     procedure PaintSelectionRect;
     procedure MoveSelectionToNewPos;
     procedure InvalidateMarkers(EnableAfter: Boolean);
     function GetMarkerRect(AComponent: TComponent; MItems: TGrabberMarker): TRect;
     procedure SetTempCursor(ARoot: TWinControl; ACursor: TCursor);
     function  HandleSetCursor(var TheMessage: TLMessage): boolean;
     procedure CreateMoveRect;
     function GetSelectionIndex(AControl: TPersistent): Integer;
     procedure MouseDownOnControl(Sender: TControl; var TheMessage: TLMMouse);
     procedure MouseMoveOnControl(Sender: TControl; var TheMessage: TLMMouse);
     procedure MouseUpOnControl(Sender: TControl; var TheMessage: TLMMouse);
     procedure GetMouseMsgShift(TheMessage: TLMMouse; var Shift: TShiftState;
                               var Button: TMouseButton);
     procedure KeyDownOnControl(Sender: TControl; var TheMessage: TLMKeyDown);
     procedure UTF8KeyPress(var UTF8Key: TUTF8Char); override;
     function SizeControl(Sender: TControl; TheMessage: TLMSize): Boolean;
     function MoveControl(Sender: TControl; TheMessage: TLMMove): Boolean;
     function OnFormCloseQuery: boolean;
     function GetGrabberMarker(C: TControl; P: TPoint): TGrabberMarker;
     procedure MoveSelection(DX, DY: Integer);
     procedure ResizeControl(DX, DY: Integer);
     procedure UpNonVisual;
     procedure RectToFormRect(PR: PRect; WControl: TWinControl);
     function GetRoot: TComponent;
     procedure SetRoot(AComponent: TComponent);
     function OnFormActivate(Sender: TControl; var TheMessage: TLMActivate): Boolean;
     function DoCopySelectionToClipboard: boolean;
     function DoPasteSelectionFromClipboard(PasteFlags: TComponentPasteSelectionFlags): boolean;

     function DoInsertFromStream(s: TStream; PasteParent: TWinControl;
                                PasteFlags: TComponentPasteSelectionFlags): Boolean;
     function GetPasteParent: TWinControl;
     procedure OnDesignerPasteComponents(TxtCompStream: TStream;
        ParentControl: TWinControl;  var NewComponents: TFPList);
     procedure ReadingComponentnSetName(Reader: TReader; Component: TComponent;
        var AName: string);
     procedure ReadComp(Component: TComponent);
     procedure DoDoubleClickOnComponent(Component: TComponent);
     procedure HandlePopupMenu(Sender: TControl; var Message: TLMContextMenu);
  public
    MethodsHandlers: TJITMethods;
    NonVisualSet: Boolean;
    constructor Create(Manager: TDesignerManager; JITMethods: TJITMethods = nil);
    destructor Destroy; override;
    function DeleteSelection: Boolean; override;
    procedure ValidateRename(AComponent: TComponent; const CurName,
       NewName: string); override;

    function CanUndo: Boolean; override;
    function CanRedo: Boolean; override;
    function Undo: Boolean; override;
    function Redo: Boolean; override;
    procedure UpdateCaption;
    function IsUndoLocked: boolean; override;
    function AddUndoAction(const aPersistent: TPersistent;
      aOpType: TUndoOpType; IsSetNewId: boolean; aFieldName: string;
      const aOldVal, aNewVal: variant): boolean; override;
    procedure ComponentRenamed(AComponent: TComponent);

    //******************************************************
    function IsDesignMsg(Sender: TControl; var TheMessage: TLMessage): Boolean; override;
    //*******************************************************

    function CreateUniqueComponentName(const AClassName: string): string; override;
    procedure DesignClose;
    procedure DesignOpen;
    procedure SetNonVisual;
    procedure HideHint;
    function CopySelection: boolean; override;
    function CutSelection: boolean; override;
    function CanPaste: Boolean; override;
    function DesignSelectable(AComponent: TComponent): Boolean;
    function PasteSelection(Flags: TComponentPasteSelectionFlags): boolean;
       override;
    function CopySelectionToStream(AllComponentsStream: TStream): boolean; override;
    procedure AddSelection(Control: TControl);
    procedure Modified; override;
    function IsModified: Boolean;
    procedure ResetModified;
    procedure BringToFront;
    procedure SendToBack;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure PaintGrid; override;
    function GetShiftState: TShiftState; override;
    procedure SelectOnlyThisComponent(AComponent: TComponent); override;
    function UniqueName(const BaseName: string): string; override;
    function ControlAtPos(Sender: TControl; P: TPoint): TControl;
    procedure AddComponent(Sender: TControl; X, Y: Integer);
    procedure AfterAddComponent(AComponent: TComponent);
    function ClearSelection: boolean; override;
    procedure GetSelectionFromCaptureRect;
    function GetNonVisualOfObject(AObject: TPersistent): TNonVisual;
    function MousePosToGridPos(P: TPoint): TPoint;
    property Form: TCustomForm read FForm write SetForm;
    property DesignerSelections: TDesignerSelections read FDesignerSelections;
    property Root: TComponent read GetRoot write SetRoot;
    property DesignManager: TDesignerManager read FDesignerManager;
  end;

  TIsObjNameExist = function(AName: string): Boolean of object;
  TOnAfterAddComponent = procedure (AComponent, ARoot: TComponent) of object;
  TOnDeleteComponent = procedure (AComponent, ARoot: TComponent) of object;
  TOnDobleClickComponent = procedure (AComponent, ARoot: TComponent) of object;
  TOnGetIcon = procedure (Source: TComponent; var Bmp: TBitmap) of object;
  TIsVisualComponent = function (AComponent: TComponent): Boolean of object;
  TOnFormActivate = procedure (ARoot: TComponent; AForm: TCustomForm) of object;
  TOnGetClassResurce = function (AClassName: string): string of object;
  {TOnValidateRename = procedure (AComponent: TComponent;
    const CurName, NewName: string) of object;}

  {TDesignPopupMenu}

  TDesignPopupMenu = class(TPopupMenu)
  private
    FControlItems: TList;
  protected
    EditorMenuList: TList;
    procedure AddMenuItem(MenuItem: TMenuItem);
    procedure MenuClick(Sender: TObject);
    procedure BringToFrontClick(Sender: TObject);
    procedure SendToBackClick(Sender: TObject);
    procedure CopyToClipboard(Sender: TObject);
    procedure CtuToClipboard(Sender: TObject);
    procedure PasteFromClipboard(Sender: TObject);
    procedure DeleteSelection(Sender: TObject);
    procedure CreateItems;
  public
    DesComponent: TComponent;
    Editor: TBaseComponentEditor;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ControlItemShow;
    procedure ControlItemHide;
    procedure ClearMenuList;
    procedure SetEditor(AComponent: TComponent; AEditor: TBaseComponentEditor);
  end;

  TOnGetComponentClass = function (AClassName: string): TComponentClass of object;
  TOnGetReadOnly = function (Sender: TComponent) : Boolean of object;
  TIsIdeSystemKey = function (Key: Word): Boolean of object;

  { TDesignerManager }

  TDesignerManager = class(TComponent)
  private
    FDesignObjects: TList;
    FGridStep: Integer;
    FObjInspector: TObjectInspectorDlg;
    FIsObjNameExist: TIsObjNameExist;
    FOnAfterAddComponent: TOnAfterAddComponent;
    FOnDeleteComponent: TOnDeleteComponent;
    FOnDobleClickComponent: TOnDobleClickComponent;
    FOnGetIcon: TOnGetIcon;
    FActive: Boolean;
    FOnFormActivate: TOnFormActivate;
    FOnGetClassResurce: TOnGetClassResurce;
    FDesignPopupMenu: TDesignPopupMenu;
    FOnGetComponentClass: TOnGetComponentClass;
    FNewDesignPosTop: Integer;
    FNewDesignPosLeft: Integer;
    FNewDesignWidth: Integer;
    FNewDesignHeight: Integer;
    FShowHints: Boolean;
    FOnGetReadOnly: TOnGetReadOnly;
    FMoveObjectMethod: TMoveObjectMethod;
    FShowGrid: Boolean;
    FControlKeyMove: TShiftStateEnum;
    FControlKeyResize: TShiftStateEnum;
    FIsIdeSysKey: TIsIdeSystemKey;
    FIsVisualCompoent: TIsVisualComponent;
    FFirstSelectThenDrag: Boolean;
    function GetReadOnly: Boolean;
    procedure SetControlKeyMove(AValue: TShiftStateEnum);
    procedure SetControlKeyResize(AValue: TShiftStateEnum);
    procedure SetShowGrid(AValue: Boolean);
    //FOnValidateRename: TOnValidateRename;
  protected
    {$IfDef WINDOWS}
    FSaveLeft: Integer;
    FSaveTop: Integer;
    FSaveWidth: Integer;
    FSaveHeight: Integer;
    {$EndIf}
    LoadHandlers: TJITMethods;
    {procedure Notification(AComponent: TComponent; Operation: TOperation);
       override;}
    procedure SetGridStep(Value: Integer);
    procedure SetObjInspector(AValue: TObjectInspectorDlg);
    function ObjNameExist(AName: string): Boolean;
    procedure WriteMethodProp(Writer: TWriter; Instance: TPersistent;
      PropInfo: PPropInfo; const MethodValue, DefMethodValue: TMethod;
      var Handled: boolean);
    procedure ReaderSetMethodProp(Reader: TReader; Instance: TPersistent;
      PropInfo: PPropInfo; const TheMethodName: string;
      var Handled: boolean);
    function GetDesignObjectsCount: Integer;
    function GetItems(Index: Integer): TDesignerHook;
    function GetItemsRoot(Index: Integer): TComponent;
    function GetItemsDesignForm(Index: Integer): TCustomForm;
    procedure SetActive(AValue: Boolean);
    procedure ReaderReadStringProp(Sender:TObject;
        const Instance: TPersistent; PropInfo: PPropInfo;
        var Content:string);
    procedure InitialDesignForm(AComponent: TComponent; AForm: TCustomForm);
    function GetPropertyEditorHook: TPropertyEditorHook;
    procedure DesignPopupMenu(Pos: TPoint; ASelections: TDesignerSelections);

  public
    //PropEditHook: TPropertyEditorHook;
    AddComponentClass: TComponentClass;
    CurrentDesignForm: TCustomForm;
    CurrentDesignRoot: TComponent;
    AddFrame: string;
    WriteFrame: TFrame;
    ControlInSelection: Boolean;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Clear;
    function DesignSelectable(AComponent: TComponent): Boolean;
    procedure UpNonVisual;
    procedure BringToFront;
    procedure SendToBack;
    procedure DeleteComponent(AComponent, ARoot: TComponent);
    procedure DeleteAndFreeByRoot(ARoot: TComponent);
    procedure DeleteAndFree(Index: Integer);
    procedure PaintCurentSelection;
    procedure ClearMarkers;
    procedure PersistentAdded(APersistent: TPersistent; Select: boolean);
    procedure PersistentDeleting(APersistent: TPersistent);
    procedure AfterAddComponent(AComponent, ARoot: TComponent);
    procedure SetComponentName(AComponent: TComponent; BaseClassName: string);
    function GetDesignForm(APersistent: TPersistent): TCustomForm;
    function CreateDesignerForComponent(AComponent: TComponent;
      SetUpNonVisual: Boolean = False; AJITMethods: TJITMethods = nil): TCustomForm;
    procedure ChangeSelection(Root: TComponent; Selections: TDesignerSelections; DesForm: TCustomForm);
    procedure OnExtChangeSelection(Selection: TPersistentSelectionList);
    procedure SelectOnlyThisComponent(ACompoenent: TComponent);
    procedure InvalidateAll;
    function GetDesignObjectByName(AName: string): TComponent;
    function GetDesignFormFromRoot(ARoot: TComponent): TCustomForm;
    procedure NewDataModule(var DM: TDataModule; var F: TCustomForm);
    function NewForm: TCustomForm;
    function NewDesForm: TCustomForm;
    procedure ShowForm(Form: TCustomForm);
    procedure NewFrame(var Frame: TFrame; var F: TCustomForm);
    procedure AncestorNotFound(Reader: TReader; const ComponentName: string;
    ComponentClass: TPersistentClass; var Component: TComponent);
    {$IfDef WINDOWS}
    procedure OnCreateComponent(Reader: TReader;
      ComponentClass: TComponentClass; var Component: TComponent);
    {$EndIf}
    function ObjToText(Instance: Tcomponent): string;
    procedure TextToDesignObj(s: string; var Root: TComponent;
      var DesignForm: TCustomForm; InitializForm: Boolean = True);
    procedure TextToObject(S: string; var Root: TComponent);
    procedure SetFrameMethodProp(Reader: TReader; Instance: TPersistent;
      PropInfo: PPropInfo; const TheMethodName: string;
      var Handled: boolean);
    procedure CopyToClipboard;
    procedure CutToClipboard;
    procedure PasterFromClipboard;
    procedure DeleteSelections;
    procedure HideDesignHint;
    procedure TextToFrame(S: string; var Frame: TFrame);
    function GetRootClassName: string;
    property DesignObjects: TList read FDesignObjects;
    property Active: Boolean read FActive write SetActive;
    property DesignCount: Integer read GetDesignObjectsCount;
    property Items[Index: Integer]: TDesignerHook read GetItems;
    property ItemsRoot[Index: Integer]: TComponent read GetItemsRoot;
    property ItemsDesignForm[Index: Integer]: TCustomForm read GetItemsDesignForm;
    property PropEditHook: TPropertyEditorHook read GetPropertyEditorHook;
    property ReadOnly: Boolean read GetReadOnly;
  published
    property GridStep: Integer read FGridStep write SetGridStep;
    property ObjInspector: TObjectInspectorDlg
             read FObjInspector write SetObjInspector;
    property IsObjNameExist: TIsObjNameExist
             read FIsObjNameExist write FIsObjNameExist;
    property OnAfterAddComponent: TOnAfterAddComponent
             read FOnAfterAddComponent write FOnAfterAddComponent;
    property OnDeleteComponent: TOnDeleteComponent
             read FOnDeleteComponent write FOnDeleteComponent;
    property OnDobleClickComponent: TOnDobleClickComponent
             read FOnDobleClickComponent write FOnDobleClickComponent;
    property OnGetIcon: TOnGetIcon read FOnGetIcon write FOnGetIcon;
    property OnFormActivate: TOnFormActivate
             read FOnFormActivate write FOnFormActivate;
    property OnGetClassResurce: TOnGetClassResurce
             read FOnGetClassResurce  write FOnGetClassResurce;
    property OnGetComponentClass: TOnGetComponentClass
             read FOnGetComponentClass write FOnGetComponentClass;
    property NewDesignPosTop: Integer read FNewDesignPosTop
             write FNewDesignPosTop;
    property NewDesignPosLeft: Integer read FNewDesignPosLeft
             write FNewDesignPosLeft;
    property NewDesignWdith: Integer read FNewDesignWidth
             write FNewDesignWidth;
    property NewDesignHeight: Integer read FNewDesignHeight
             write FNewDesignHeight;
    property ShowHints: Boolean read FShowHints write FShowHints;
    property OnGetReadOnly: TOnGetReadOnly read FOnGetReadOnly write FOnGetReadOnly;
    property MoveObjectMethod: TMoveObjectMethod
             read FMoveObjectMethod write FMoveObjectMethod;
    property ShowGrid: Boolean read FShowGrid write SetShowGrid;
    property ControlKeyMove: TShiftStateEnum read FControlKeyMove
             write SetControlKeyMove;
    property ControlKeyResize: TShiftStateEnum read FControlKeyResize
             write SetControlKeyResize;
    property IsIdeSysKey: TIsIdeSystemKey read FIsIdeSysKey write FIsIdeSysKey;
    property IsVisualComponent: TIsVisualComponent
             read FIsVisualCompoent write FIsVisualCompoent;
    property FirstSelectThenDrag: Boolean read FFirstSelectThenDrag write FFirstSelectThenDrag;
  end;

function LeftFromDesignInfo(ADesignInfo: LongInt): SmallInt; inline;
function TopFromDesignInfo(ADesignInfo: LongInt): SmallInt; inline;
function LeftTopToDesignInfo(const ALeft, ATop: SmallInt): LongInt; inline;
procedure DesignInfoToLeftTop(ADesignInfo: LongInt; out ALeft, ATop: SmallInt); inline;
procedure SetComponentDesignMode(AComponent: TComponent; Value: Boolean);
procedure SetComponentDesignInstanceMode(AComponent: TComponent; Value: Boolean);
procedure SetComponentInlineMode(AComponent: TComponent; Value: Boolean);
procedure SetComponentAncestorMode(AComponent: TComponent; Value: Boolean);

function IsJITMethod(const aMethod: TMethod): boolean;
function GetJITMethod(const aMethod: TMethod; out aJITMethod: TJITMethod): boolean;
function CompareJITMethod(Data1, Data2: Pointer): integer;

function CalculateTypeDataSize(PropInfoCount: integer): integer;
function CalculateTypeInfoSize(const AClassName: shortstring;
                               PropInfoCount: integer): integer;
function GetTypeDataPropCountAddr(TypeData: PTypeData): PWord;

function GetParentFormRelativeTopLeft(Component: TComponent): TPoint;

implementation

uses LCLClasses, LResources, Dialogs, ActnList, DB,
  Clipbrd, Math, ComCtrls, ObjInspStrConsts;

const
  vmtInstanceSizeNeg = vmtInstanceSize+sizeof(ptrint);

  lisUnableToStreamSelectedComponents = 'Unable to stream selected components';
  lisThereWasAnErrorDuringWritingTheSelectedComponent = 'There was an error '
    +'during writing the selected component %s:%s:%s%s';
  lisUnableConvertBinaryStreamToText = 'Unable convert binary stream to text';
  lisThereWasAnErrorWhileConvertingTheBinaryStreamOfThe = 'There was an error '
    +'while converting the binary stream of the selected component %s:%s:%s%s';
  lisUnableCopyComponentsToClipboard = 'Unable copy components to clipboard';
  lisThereWasAnErrorWhileCopyingTheComponentStreamToCli = 'There was an error '
    +'while copying the component stream to clipboard:%s%s';

type
  TSetDesigningComponent = class(TComponent)
  public
    class procedure SetDesigningOfComponent(AComponent: TComponent; Value: Boolean);
    class procedure SetDesignInstanceOfComponent(AComponent: TComponent; Value: Boolean);
    class procedure SetInlineOfComponent(AComponent: TComponent; Value: Boolean);
  end;

 { function GetParentFormRelativeTopLeft(Component: TComponent): TPoint;
  var
    FormOrigin: TPoint;
    ParentForm: TCustomForm;
    Parent: TWinControl;
    p: TPoint;
  begin
    if Component is TControl then
    begin
      ParentForm := GetParentForm(TControl(Component));
      Parent := TControl(Component).Parent;
      if (Parent = nil) or (ParentForm = nil) then
      begin
        Result := Point(0, 0);
      end else
      begin
        Result := Parent.ClientOrigin;
        FormOrigin := ParentForm.ClientOrigin;
        //DebugLn(['GetParentFormRelativeTopLeft Component=',dbgsName(Component),' Parent=',dbgsName(Parent),' ',dbgs(Result),' ParentForm=',dbgsName(ParentForm),' ',dbgs(FormOrigin)]);
        Result.X := Result.X - FormOrigin.X + TControl(Component).Left;
        Result.Y := Result.Y - FormOrigin.Y + TControl(Component).Top;
      end;
    end else
    begin
      Result.X := LeftFromDesignInfo(Component.DesignInfo);
      Result.Y := TopFromDesignInfo(Component.DesignInfo);
      if Component.Owner is TWinControl then
      begin
        Parent:=TWinControl(Component.Owner);
        ParentForm := GetParentForm(Parent);
        if (ParentForm<>nil) and (ParentForm<>Parent) then
        begin
          p:=Parent.ClientOrigin;
          FormOrigin := ParentForm.ClientOrigin;
          inc(Result.X,p.X-FormOrigin.X);
          inc(Result.Y,p.Y-FormOrigin.Y);
        end;
      end;
    end;
  end;

function GetParentFormRelativeBounds(Component: TControl): TRect;
var
    CTopLeft: TPoint;
begin
    CTopLeft := GetParentFormRelativeTopLeft(Component);
    Result.Left := CTopLeft.X;
    Result.Top := CTopLeft.Y;
    Result.Right := Result.Left + Component.Width;
    Result.Bottom := Result.Top + Component.Height;
end;}


procedure SetComponentDesignMode(AComponent: TComponent; Value: Boolean);
begin
  TSetDesigningComponent.SetDesigningOfComponent(AComponent, Value);
end;

procedure SetComponentDesignInstanceMode(AComponent: TComponent; Value: Boolean);
begin
  TSetDesigningComponent.SetDesignInstanceOfComponent(AComponent, Value);
end;

procedure SetComponentInlineMode(AComponent: TComponent; Value: Boolean);
begin
  TSetDesigningComponent.SetInlineOfComponent(AComponent, Value);
end;

procedure SetComponentAncestorMode(AComponent: TComponent; Value: Boolean);
begin
  TSetDesigningComponent(AComponent).SetAncestor(Value);
end;

function GetComponentLeft(AComponent: TComponent): integer;
begin
  if AComponent is TControl then
    Result := TControl(AComponent).Left
  else
    Result := LeftFromDesignInfo(AComponent.DesignInfo);
end;

function GetComponentTop(AComponent: TComponent): integer;
begin
  if AComponent is TControl then
    Result := TControl(AComponent).Top
  else
    Result := TopFromDesignInfo(AComponent.DesignInfo);
end;

{ TDesignPopupMenu }

procedure TDesignPopupMenu.AddMenuItem(MenuItem: TMenuItem);
var
  I: Integer;
begin
  I := EditorMenuList.Add(MenuItem);
  Items.Insert(I, MenuItem);
end;

procedure TDesignPopupMenu.MenuClick(Sender: TObject);
begin
  if Editor = nil then Exit;
  Editor.ExecuteVerb(TMenuItem(Sender).MenuIndex);
  ClearMenuList;
  Application.ProcessMessages;
  TDesignerManager(Owner).PaintCurentSelection;
end;

procedure TDesignPopupMenu.BringToFrontClick(Sender: TObject);
begin
  TDesignerManager(Owner).BringToFront;
end;

procedure TDesignPopupMenu.SendToBackClick(Sender: TObject);
begin
  TDesignerManager(Owner).SendToBack;
end;

procedure TDesignPopupMenu.CopyToClipboard(Sender: TObject);
begin
  TDesignerManager(Owner).CopyToClipboard;
end;

procedure TDesignPopupMenu.CtuToClipboard(Sender: TObject);
begin
  TDesignerManager(Owner).CutToClipboard;
end;

procedure TDesignPopupMenu.PasteFromClipboard(Sender: TObject);
begin
  TDesignerManager(Owner).PasterFromClipboard;
end;

procedure TDesignPopupMenu.DeleteSelection(Sender: TObject);
begin
  TDesignerManager(Owner).DeleteSelections;
end;

procedure TDesignPopupMenu.CreateItems;
var
  MI: TMenuItem;
begin
  MI := TMenuItem.Create(Self);
  MI.Caption:= oisCopyComponents;
  MI.OnClick:= @CopyToClipboard;
  Items.Add(MI);

  MI := TMenuItem.Create(Self);
  MI.Caption:= oisCutComponents;
  MI.OnClick:= @CtuToClipboard;
  Items.Add(MI);

  MI := TMenuItem.Create(Self);
  MI.Caption:= oisPasteComponents;
  MI.OnClick:= @PasteFromClipboard;
  Items.Add(MI);

  MI := TMenuItem.Create(Self);
  MI.Caption:= '-';
  Items.Add(MI);

  MI := TMenuItem.Create(Self);
  MI.Caption:= oisDeleteComponents;
  MI.OnClick:= @DeleteSelection;
  Items.Add(MI);

  MI := TMenuItem.Create(Self);
  MI.Caption:= '-';
  Items.Add(MI);

  MI := TMenuItem.Create(Self);
  MI.Caption := oisOrderMoveToFront;
  MI.OnClick:= @BringToFrontClick;
  Items.Add(MI);
  FControlItems.Add(MI);
  MI := TMenuItem.Create(Self);
  MI.Caption:= oisOrderMoveToBack;
  MI.OnClick:= @SendToBackClick;
  Items.Add(MI);
  FControlItems.Add(MI);
end;

constructor TDesignPopupMenu.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  EditorMenuList := TList.Create;
  FControlItems := TList.Create;
  CreateItems;
end;

destructor TDesignPopupMenu.Destroy;
begin
  inherited Destroy;
  EditorMenuList.Free;
  FControlItems.Free;
end;

procedure TDesignPopupMenu.ControlItemShow;
var
  I: Integer;
begin
  for I := 0 to FControlItems.Count - 1 do
    TMenuItem(FControlItems.Items[I]).Visible := True;
end;

procedure TDesignPopupMenu.ControlItemHide;
var
  I: Integer;
begin
  for I := 0 to FControlItems.Count - 1 do
    TMenuItem(FControlItems.Items[I]).Visible := False;
end;

procedure TDesignPopupMenu.ClearMenuList;
var
  I: Integer;
  MI: TMenuItem;
begin
  for I := 0 to EditorMenuList.Count - 1 do
  begin
    MI := TMenuItem(EditorMenuList.Items[I]);
    Items.Delete(MI.MenuIndex);
    MI.Free;
  end;
  EditorMenuList.Clear;
end;

procedure TDesignPopupMenu.SetEditor(AComponent: TComponent;
  AEditor: TBaseComponentEditor);
var
  I: Integer;
  MI: TMenuItem;
begin
  ClearMenuList;
  Editor := AEditor;
  if AEditor <> nil then
  begin
    for I := 0 to AEditor.GetVerbCount - 1 do
    begin
      MI := TMenuItem.Create(Self);
      MI.Caption := AEditor.GetVerb(I);
      MI.OnClick := @MenuClick;
      AddMenuItem(MI);
    end;
    if AEditor.GetVerbCount > 0 then
    begin
      MI := TMenuItem.Create(Self);
      MI.Caption := '-';
      AddMenuItem(MI);
    end;
  end;

  DesComponent := AComponent;

  if DesComponent is TControl then
    ControlItemShow
  else
    ControlItemHide;
end;

{TSetDesigningComponent}

class procedure TSetDesigningComponent.SetDesigningOfComponent(
  AComponent: TComponent; Value: Boolean);
begin
  TSetDesigningComponent(AComponent).SetDesigning(Value, True);
end;

class procedure TSetDesigningComponent.SetDesignInstanceOfComponent(
  AComponent: TComponent; Value: Boolean);
begin
  // requires fpc >= 2.2.1
  TSetDesigningComponent(AComponent).SetDesignInstance(Value);
end;

class procedure TSetDesigningComponent.SetInlineOfComponent(
  AComponent: TComponent; Value: Boolean);
begin
  // requires fpc >= 2.2.1
  TSetDesigningComponent(AComponent).SetInline(Value);
end;

function LeftFromDesignInfo(ADesignInfo: LongInt): SmallInt;
begin
  Result := LazLongRec(ADesignInfo).Lo;
end;

function TopFromDesignInfo(ADesignInfo: LongInt): SmallInt;
begin
  Result := LazLongRec(ADesignInfo).Hi;
end;

function LeftTopToDesignInfo(const ALeft, ATop: SmallInt): LongInt;
begin
  LazLongRec(Result).Lo:=ALeft;
  LazLongRec(Result).Hi:=ATop;
end;

procedure DesignInfoToLeftTop(ADesignInfo: LongInt; out ALeft, ATop: SmallInt);
begin
  ALeft := LazLongRec(ADesignInfo).Lo;
  ATop := LazLongRec(ADesignInfo).Hi;
end;

function ClassNameToComponentName(const AClassName: string): string;
begin
  Result:=AClassName;
  if (length(Result)>2) and (Result[1] in ['T','t'])
  and (not (Result[2] in ['0'..'9'])) then
    System.Delete(Result,1,1);
end;

function GetFormRelativeMousePosition(Form: TCustomForm): TPoint;
var
  FormClientOrigin: TPoint;
begin
  Result.X:=0;
  Result.Y:=0;
  GetCursorPos(Result);
  FormClientOrigin:=Form.ClientOrigin;
  dec(Result.X,FormClientOrigin.X);
  dec(Result.Y,FormClientOrigin.Y);
end;

{ TJITMethod }

function IsJITMethod(const aMethod: TMethod): boolean;
begin
  Result:=(aMethod.Data<>nil) and (aMethod.Code=nil)
      and (TObject(aMethod.Data).ClassType=TJITMethod);
end;

function GetJITMethod(const aMethod: TMethod; out aJITMethod: TJITMethod
  ): boolean;
begin
  if IsJITMethod(aMethod) then begin
    Result:=true;
    aJITMethod:=TJITMethod(aMethod.Data);
  end else begin
    Result:=false;
    aJITMethod:=nil;
  end;
end;

function CompareJITMethod(Data1, Data2: Pointer): integer;
var
  JITMethod1: TJITMethod absolute Data1;
  JITMethod2: TJITMethod absolute Data2;
begin
  Result:=CompareText(JITMethod1.TheMethodName,JITMethod2.TheMethodName);
end;

function CalculateTypeDataSize(PropInfoCount: integer): integer;
begin
  Result := SizeOf(TTypeData) + 2; // TTypeData + one word for new prop count
  // Actually the size depends on the UnitName. But SizeOf(TTypeData) already
  // uses the maximum size of the shortstring.
  {$ifdef FPC_REQUIRES_PROPER_ALIGNMENT}
  if Result and (SizeOf(Pointer) - 1) <> 0 then
    Inc(Result, SizeOf(Pointer)); // a few bytes too much, but at least enough
  {$endif}
  inc(Result,PropInfoCount*SizeOf(TPropInfo));
end;

function CalculateTypeInfoSize(const AClassName: shortstring;
  PropInfoCount: integer): integer;
begin
  Result := SizeOf(TTypeKind) + 1 + length(AClassName)
           + CalculateTypeDataSize(PropInfoCount);
  {$warnings off}
  if SizeOf(TTypeKind)<>1 then
    raise Exception.Create('CalculateTypeInfoSize SizeOf(TTypeInfo^.Kind)<>1');
  {$warnings on}
end;

function GetTypeDataPropCountAddr(TypeData: PTypeData): PWord;
begin
  Result:=PWord(PByte(@TypeData^.UnitName)+Length(TypeData^.UnitName)+1);
  {$ifdef FPC_REQUIRES_PROPER_ALIGNMENT}
  Result := Align(Result, SizeOf(Pointer));
  {$endif}
end;

function GetParentFormRelativeTopLeft(Component: TComponent): TPoint;
var
  FormOrigin: TPoint;
  ParentForm: TCustomForm;
  Parent: TWinControl;
  p: TPoint;
begin
  if Component is TControl then
  begin
    ParentForm := GetParentForm(TControl(Component));
    Parent := TControl(Component).Parent;
    if (Parent = nil) or (ParentForm = nil) then
    begin
      Result := Point(0, 0);
    end else
    begin
      Result := Parent.ClientOrigin;
      FormOrigin := ParentForm.ClientOrigin;
      //DebugLn(['GetParentFormRelativeTopLeft Component=',dbgsName(Component),' Parent=',dbgsName(Parent),' ',dbgs(Result),' ParentForm=',dbgsName(ParentForm),' ',dbgs(FormOrigin)]);
      Result.X := Result.X - FormOrigin.X + TControl(Component).Left;
      Result.Y := Result.Y - FormOrigin.Y + TControl(Component).Top;
    end;
  end else
  begin
    Result.X := LeftFromDesignInfo(Component.DesignInfo);
    Result.Y := TopFromDesignInfo(Component.DesignInfo);
    if Component.Owner is TWinControl then
    begin
      Parent:=TWinControl(Component.Owner);
      ParentForm := GetParentForm(Parent);
      if (ParentForm<>nil) and (ParentForm<>Parent) then
      begin
        p:=Parent.ClientOrigin;
        FormOrigin := ParentForm.ClientOrigin;
        inc(Result.X,p.X-FormOrigin.X);
        inc(Result.Y,p.Y-FormOrigin.Y);
      end;
    end;
  end;
end;

constructor TJITMethod.Create(AnOwner: TJITMethods;
  const aMethodName: shortstring; AtypeData: PTypeData);
begin
  FMethod.Data:=Self;
  FMethod.Code := nil;
  fTheMethodName:=aMethodName;
  FOwner:=AnOwner;
  TypeData := AtypeData;
  Owner.InternalAdd(Self);
end;

destructor TJITMethod.Destroy;
begin
  if Owner<>nil then
    Owner.InternalRemove(Self);
  inherited Destroy;
end;

{ TJITMethods }

procedure TJITMethods.InternalAdd(const AMethod: TJITMethod);
begin
  fMethods.Add(AMethod);
  AMethod.fOwner:=Self;
end;

procedure TJITMethods.InternalRemove(const AMethod: TJITMethod);
begin
  AMethod.fOwner:=nil;
  if not fClearing then
    fMethods.Remove(AMethod);
end;

constructor TJITMethods.Create;
begin
  fMethods:=TList.Create;
end;

destructor TJITMethods.Destroy;
begin
  Clear;
  FreeAndNil(fMethods);
  inherited Destroy;
end;

procedure TJITMethods.Clear;
begin
  fClearing:=true;
  while fMethods.Count > 0 do
  begin
    TJITMethod(fMethods.Items[0]).Free;
    fMethods.Delete(0);
  end;
  fClearing:=false;
end;

function TJITMethods.Count: Integer;
begin
  Result := fMethods.Count;
end;

function TJITMethods.Get(Index: Integer): TJITMethod;
begin
  Result := TJITMethod(fMethods.Items[Index]);
end;

function TJITMethods.Add(const aMethodName: shortstring; ATypeData: PTypeData): TJITMethod;
begin
  Result:=Find(aMethodName);
  if Result=nil then begin
    Result:=TJITMethod.Create(Self, aMethodName, ATypeData);
  end;
end;

function TJITMethods.Find(const aMethodName: shortstring): TJITMethod;
var
  CurMethod: TJITMethod;
  Comp: LongInt;
  I: Integer;
begin
  Result := nil;
  for I := 0 to fMethods.Count - 1 do
  begin
    CurMethod := TJITMethod(fMethods.Items[I]);
    Comp:=CompareText(aMethodName,CurMethod.TheMethodName);
    if Comp = 0 then
    begin
      Result := CurMethod;
      Break;
    end;
  end;

end;

function TJITMethods.Delete(aMethod: TJITMethod): boolean;
begin
  if (aMethod=nil) then
    Result:=false
  else if aMethod.Owner<>Self then
    RaiseGDBException('TJITMethods.DeleteJITMethod')
  else begin
    Result:=true;
    InternalRemove(aMethod);
    aMethod.Free;
  end;
end;

function TJITMethods.Delete(const aMethodName: shortstring): boolean;
var
  CurMethod: TJITMethod;
begin
  CurMethod:=Find(aMethodName);
  if CurMethod=nil then begin
    Result:=false;
  end else begin
    Result:=true;
    InternalRemove(CurMethod);
    CurMethod.Free;
  end;
end;

function TJITMethods.Rename(const OldMethodName,
  NewMethodName: shortstring): boolean;
var
  CurMethod: TJITMethod;
begin
  CurMethod:=Find(OldMethodName);
  if CurMethod=nil then begin
    Result:=false;
  end else begin
    Result:=true;
    //DebugLn(['TJITMethods.Rename Class=',DbgSName(aClass),' Old=',CurMethod.TheMethodName,' New=',NewMethodName]);
    fMethods.Remove(CurMethod);
    CurMethod.fTheMethodName:=NewMethodName;
    fMethods.Add(CurMethod);
  end;
end;

{TComponentCaption}

constructor TComponentCaption.Create(TheOwner: TComponent);
begin
  inherited Create(TheOwner);
  ParentFont := False;
  //Anchors := [akTop, akTop];
end;

procedure TComponentCaption.SetBounds(aLeft, aTop, aWidth, aHeight: integer);
var
  I, T, L, W, H: Integer;
  C: TCanvas;
begin
  C := TCanvas.Create;
  C.Handle := GetDC(0);
  try
    C.Font := Self.Font;
    W := C.TextWidth(Caption);
    H := C.TextHeight('W');
    I := (W - 32) div 2;
    L := TNonVisual(Owner).Left - I;
    T := TNonVisual(Owner).Top + 34;
    inherited SetBounds(L, T, W, H);
  finally
    ReleaseDC(0, C.Handle);
    C.Free;
  end;
end;

{TNonVisual}

constructor TNonVisual.Create(AOwner: TComponent; DesManager: TDesignerManager);
begin
  inherited Create(AOwner);
  FDesignerManager := DesManager;
  FComponent := nil;
  FCaption := TComponentCaption.Create(Self);
  Width := 32;
  Height := 32;
  ControlStyle := ControlStyle - [csAcceptsControls];
  Bmp := TBitmap.Create;
  Bmp.Transparent := True;
  TabStop := False;
  //SetupInternalLabel;
  //Anchors := [akLeft, akTop];
end;

destructor TNonVisual.Destroy;
begin
  Bmp.Free;
  inherited Destroy;
end;

procedure TNonVisual.UpdateCaption;
begin
  if FComponent <> nil then
  begin
    if FComponent.Name <> FCaption.Caption then
    begin
      FCaption.Caption := FComponent.Name;
      SetLabelBounds(Left, Top);
    end;
  end;
end;

procedure TNonVisual.SetComponent(Value: TComponent);
begin
  FComponent := Value;
  FComponent.FreeNotification(Self);
  if Value <> nil then FCaption.Caption := FComponent.Name;
  if Assigned(FDesignerManager.FOnGetIcon) then
  begin
      FDesignerManager.OnGetIcon(FComponent, Bmp);
      Repaint;
  end;
end;

procedure TNonVisual.SetParent(NewParent: TWinControl);
begin
  inherited SetParent(NewParent);
  FCaption.Parent := NewParent;
end;

procedure TNonVisual.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
  begin
    WindowClass.Style := not (CS_HREDRAW or not CS_VREDRAW);
    ExStyle := ExStyle or WS_EX_TOPMOST;
    ExStyle := ExStyle and not WS_EX_CONTROLPARENT;
    ExStyle := ExStyle or WS_EX_NOPARENTNOTIFY;

    Style := Style and not WS_TABSTOP;
    Style := Style and not WS_BORDER;
  end;
end;

procedure TNonVisual.WMSize(var Message: TLMSize);
begin
  Width := 32;
  Height := 32;
  Message.Result := 0;
end;

procedure TNonVisual.WMMove(var Message: TLMMove);
begin
  FComponent.DesignInfo := LeftTopToDesignInfo(Message.XPos, Message.YPos);
  SetLabelBounds(Message.XPos, Message.YPos);
  //SetLabelBounds(Left, Top);
end;

procedure TNonVisual.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
  begin
    if AComponent = FCaption then FCaption := nil
    else
    if AComponent = FComponent then
    begin
      Owner.RemoveComponent(Self);
      FComponent := nil;
      Free;
    end;
  end;
end;

procedure TNonVisual.Paint;
var
  R: TRect;
  X, Y: Integer;
begin
  R := ClientRect;
  Canvas.Brush.Color := clBtnFace;
  Canvas.FillRect(R);


  if Bmp <> nil then
  begin
    X := (32 - Bmp.Width) div 2;
    Y := (32 - Bmp.Height) div 2;
    //Bmp.TransparentColor := cl;
    //Bmp.Transparent := False;

    Canvas.Draw(X, Y, Bmp);
  end;

  Canvas.Pen.Color := clWhite;
  Canvas.MoveTo(0,31);
  Canvas.LineTo(0, 0);
  Canvas.LineTo(31, 0);
  Canvas.Pen.Color := clGray;
  Canvas.MoveTo(31, 1);
  Canvas.LineTo(31, 31);
  Canvas.LineTo(1, 31);
  if FComponent <> nil then
  begin
    if FComponent.Name <> FCaption.Caption then
    begin
      FCaption.Caption := FComponent.Name;
      SetLabelBounds(Left, Top);
    end;

  end;
end;

procedure TNonVisual.SetLabelBounds(ALeft: Integer; ATop: Integer);
var I: Integer;
begin
  if FCaption = nil then Exit;
  I := (FCaption.Width - 32) div 2;
  FCaption.SetBounds(ALeft - I, ATop + 34, FCaption.Width, FCaption.Height);
end;

{TDesignerHook}

constructor TDesignerHook.Create(Manager: TDesignerManager; JITMethods: TJITMethods = nil);
begin
  inherited Create;
  FDesignerManager := Manager;
  Form := nil;
  LastFormCursor := crDefault;
  FDesignerSelections := TDesignerSelections.Create;
  FMarkerBmp := TBitmap.Create;
  FMarkerBmp.Width := 5;
  FMarkerBmp.Height := 5;
  FDrawMarkers := True;
  FMovSelection := False;
  FSizeOperation := gmNone;
  FMoveRectList := TList.Create;
  FCR := False;
  FTempMoveKontur:= False;
  NonVisualSet := False;
  if JITMethods <> nil then
    MethodsHandlers := JITMethods
  else
    MethodsHandlers := TJITMethods.Create;

  FHintTimer := TTimer.Create(nil);
  FHintTimer.Interval := 500;
  FHintTimer.Enabled := False;
  FHintTimer.OnTimer := @OnHintTimer;

  FHintWindow := THintWindow.Create(nil);
  FHintWIndow.DoubleBuffered := True;
  FHIntWindow.Visible := False;
  FHintWIndow.Color := clInfoBk;
  FHintWindow.HideInterval := 5000;
  FHintWindow.AutoHide := True;
  OperationControl := nil;
  PropertyEditorHook.AddHandlerComponentRenamed(@ComponentRenamed);
end;

destructor TDesignerHook.Destroy;
begin
  if Canvas <> nil then
  begin
    Canvas.Handle:= 0;
    Canvas.Free;
  end;
  FDesignerSelections.Free;
  FMarkerBmp.Free;
  MethodsHandlers.free;
  FreeAndNil(FHintWIndow);
  FreeAndNil(FHintTimer);
  ClearMoveRect;
  FMoveRectList.Free;
  PropertyEditorHook.RemoveHandlerComponentRenamed(@ComponentRenamed);
  inherited Destroy;
end;

function TDesignerHook.GetMarkerRect(AComponent: TComponent; MItems: TGrabberMarker): TRect;
var
  R: TRect;
  MultiSelect: Boolean;
  P: TPoint;
  AControl: TControl;
begin
  MultiSelect := False;
  Result := Rect(0, 0, 0, 0);
  //if not (AControl is TControl) then Exit;
  AControl := TControl(AComponent);
  if DesignerSelections.Count > 1 then MultiSelect := True;
  R := AControl.BoundsRect;
  if AControl.Parent <> Form then
  begin
    P := R.TopLeft;
    P := AControl.Parent.ClientToScreen(P);
    P := Form.ScreenToClient(P);
    R.TopLeft := P;
    P := R.BottomRight;
    P := AControl.Parent.ClientToScreen(P);
    P := Form.ScreenToClient(P);
    R.BottomRight := P;
  end;
  if not MultiSelect then InflateRect(R, 2, 2);
  case MItems of
    gmLeftTop:
    begin
      Result.Left := R.Left;
      Result.Top := R.Top;
      Result.Right := Result.Left + FMarkerBmp.Width;
      Result.Bottom := Result.Top + FMarkerBmp.Height;
    end;
    gmLeft:
    begin
      Result.Left := R.Left;
      Result.Top := R.Top + (R.Bottom - R.Top) div 2 - FMarkerBmp.Height div 2;
      Result.Right := Result.Left + FMarkerBmp.Width;
      Result.Bottom := Result.Top + FMarkerBmp.Height;
    end;
    gmLeftBottom:
    begin
      Result.Left := R.Left;
      Result.Top := R.Bottom - FMarkerBmp.Height;
      Result.Right := Result.Left + FMarkerBmp.Width;
      Result.Bottom := Result.Top + FMarkerBmp.Height;
    end;
    gmTop:
    begin
      Result.Left := R.Left + (R.Right - R.Left) div 2 - FMarkerBmp.Width div 2;
      Result.Top := R.Top;
      Result.Right := Result.Left + FMarkerBmp.Width;
      Result.Bottom := Result.Top + FMarkerBmp.Height;
    end;
    gmBottom:
    begin
      Result.Left := R.Left + (R.Right - R.Left) div 2 - FMarkerBmp.Width div 2;
      Result.Top := R.Bottom - FMarkerBmp.Height;
      Result.Right := Result.Left + FMarkerBmp.Width;
      Result.Bottom := Result.Top + FMarkerBmp.Height;
    end;
    gmRightTop:
    begin
      Result.Left := R.Right - FMarkerBmp.Width;
      Result.Top := R.Top;
      Result.Right := Result.Left + FMarkerBmp.Width;
      Result.Bottom := Result.Top + FMarkerBmp.Height;
    end;
    gmRight:
    begin
      Result.Left := R.Right - FMarkerBmp.Width;
      Result.Top := R.Top + (R.Bottom - R.Top) div 2 - FMarkerBmp.Height div 2;
      Result.Right := Result.Left + FMarkerBmp.Width;
      Result.Bottom := Result.Top + FMarkerBmp.Height;
    end;
    gmRightBottom:
    begin
      Result.Left := R.Right - FMarkerBmp.Width;
      Result.Top := R.Bottom - FMarkerBmp.Height;
      Result.Right := Result.Left + FMarkerBmp.Width;
      Result.Bottom := Result.Top + FMarkerBmp.Height;
    end;
  end;
end;

procedure TDesignerHook.InvalidateMarkers(EnableAfter: Boolean);
var
  R: TRect;
  C: TControl;
  Obj: TPersistent;
  I: Integer;
begin
  if (DesignerSelections.Count = 1) and (DesignerSelections.Items[0] = Form) then Exit;
  FDrawMarkers := False;
  for I := 0 to DesignerSelections.Count - 1 do
  begin
    Obj := DesignerSelections.Items[I];
    if Obj is TControl then
    begin
        C := TControl(DesignerSelections.Items[I]);
        R := GetMarkerRect(C, gmLeftTop);
        InvalidateRect(FForm.Handle, @R, False);
        R := GetMarkerRect(C, gmLeftBottom);
        InvalidateRect(FForm.Handle, @R, False);
        R := GetMarkerRect(C, gmRightTop);
        InvalidateRect(FForm.Handle, @R, False);
        R := GetMarkerRect(C, gmRightBottom);
        InvalidateRect(FForm.Handle, @R, False);
        if DesignerSelections.Count = 1 then
        begin
          R := GetMarkerRect(C, gmLeft);
          InvalidateRect(FForm.Handle, @R, False);
          R := GetMarkerRect(C, gmTop);
          InvalidateRect(FForm.Handle, @R, False);
          R := GetMarkerRect(C, gmBottom);
          InvalidateRect(FForm.Handle, @R, False);
          R := GetMarkerRect(C, gmRight);
          InvalidateRect(FForm.Handle, @R, False);
        end;
    end;
  end;
  FDrawMarkers := EnableAfter;
end;

procedure TDesignerHook.PaintSelections;
var
  I: Integer;
  C: TControl;
  R: TRect;
  MultiSelect: Boolean;
begin
  if not FDrawMarkers or FMovSelection then Exit;
  if DesignerSelections.Count = 0 then Exit
  else
  if DesignerSelections.Count = 1 then
  begin
    if (FFrame <> nil) and (DesignerSelections.Items[0] = FFrame) then Exit;

    if (DesignerSelections.Items[0] = FForm) then Exit;
  end;

  FMarkerBmp.Canvas.Brush.Color := clLime;
  FMarkerBmp.Canvas.Pen.Color:= clBlack;
  MultiSelect := False;
  if DesignerSelections.Count > 1 then
  begin
    FMarkerBmp.Canvas.Brush.Color := clBlue;
    MultiSelect := True;
  end;
  FMarkerBmp.Canvas.Rectangle(0, 0, FMarkerBmp.Width, FMarkerBmp.Height);

  Canvas.Handle := GetDesignerDC(FForm.Handle);
  try
    for I := 0 to DesignerSelections.Count - 1 do
    begin
      if DesignerSelections.Items[I] is TControl then
      begin
        C := TControl(DesignerSelections.Items[I]);
        R := GetMarkerRect(C, gmLeftTop);
        Canvas.Draw(R.Left, R.Top, FMarkerBmp);
        R := GetMarkerRect(C, gmLeftBottom);
        Canvas.Draw(R.Left, R.Top, FMarkerBmp);
        R := GetMarkerRect(C, gmRightTop);
        Canvas.Draw(R.Left, R.Top, FMarkerBmp);
        R := GetMarkerRect(C, gmRightBottom);
        Canvas.Draw(R.Left, R.Top, FMarkerBmp);
        if DesignerSelections.Count = 1 then
        begin
          R := GetMarkerRect(C, gmLeft);
          Canvas.Draw(R.Left, R.Top, FMarkerBmp);
          R := GetMarkerRect(C, gmTop);
          Canvas.Draw(R.Left, R.Top, FMarkerBmp);
          R := GetMarkerRect(C, gmBottom);
          Canvas.Draw(R.Left, R.Top, FMarkerBmp);
          R := GetMarkerRect(C, gmRight);
          Canvas.Draw(R.Left, R.Top, FMarkerBmp);
        end;
      end;
    end;

  finally
    Canvas.Handle := 0;
  end;
end;

procedure TDesignerHook.DrawCaptureRect;
begin
  if (CaptureRect.Left = CaptureRect.Right)
    and (CaptureRect.Top = CaptureRect.Bottom) then Exit;

  Canvas.Handle := GetDesignerDC(FForm.Handle);
  try
    //Canvas.Pen.Color := clGray;
    //Canvas.DrawFocusRect(CaptureRect);

    Canvas.Brush.Color:= FForm.Color;
    Canvas.Brush.Style:= bsClear;
    Canvas.Pen.Color:= clGray;
    Canvas.Pen.Mode:= pmXor;
    Canvas.Pen.Style:= psSolid;
    Canvas.Pen.Width:= 1;
    Canvas.Rectangle(CaptureRect);
  finally
    Canvas.Handle := 0;
  end;
end;

procedure TDesignerHook.PaintSelectionRect;
var
  I: Integer;
  PR: PRect;
begin

  Canvas.Handle := GetDesignerDC(FForm.Handle);
  try

    Canvas.Brush.Color:= FForm.Color;
    Canvas.Brush.Style:= bsClear;
    Canvas.Pen.Color:= clGray;
    Canvas.Pen.Mode:= pmXor;
    Canvas.Pen.Style:= psSolid;
    Canvas.Pen.Width:= 2;

    for I := 0 to FMoveRectList.Count - 1 do
    begin
      PR := PRect(FMoveRectList.Items[I]);
      Canvas.Rectangle(PR^);
    end;

  finally
    Canvas.Handle := 0;
  end;

end;

procedure TDesignerHook.MoveSelectionToNewPos;
var
  I: Integer;
  PR: PRect;
  P1, P2: TPoint;
  C: TControl;
begin
  Modified;
  for I := 0 to DesignerSelections.Count - 1 do
  begin
    C := TControl(DesignerSelections.Items[I]);
    PR := PRect(FMoveRectList.Items[I]);
    if C.Parent <> FForm then
    begin
      P1 := PR^.TopLeft;
      P1 := FForm.ClientToScreen(P1);
      P1 := C.Parent.ScreenToClient(P1);
      PR^.TopLeft := P1;

      P2 := PR^.BottomRight;
      P2 := FForm.ClientToScreen(P2);
      P2 := C.Parent.ScreenToClient(P2);
      PR^.BottomRight := P2;
    end;

    C.BoundsRect := PR^;
  end;
  if (FDesignerManager.ObjInspector <> nil) then
  begin
    if (FDesignerManager.PropEditHook.LookupRoot = Root) then
    begin
      FDesignerManager.ObjInspector.RefreshPropertyValues;
    end;
  end;
end;

function TDesignerHook.PaintControl(Sender: TControl; TheMessage: TLMPaint): Boolean;
begin
  Result:=true;
  Sender.Dispatch(TheMessage);
  PaintSelections;
end;

function TDesignerHook.DeleteSelection: Boolean;
var
  I: Integer;
  OBJList: TList;
  AComponent: TComponent;

  procedure AddControlToDelList(AControl: TControl);
  var
    WC: TWinControl;
    N: Integer;
    CS: TControlStyle;
    Control: TControl;
  begin
    OBJList.Add(AControl);
    if AControl is TWinControl then
    begin
      WC := TWinControl(AControl);
      CS := WC.ControlStyle;
      if (csAcceptsControls in CS) and not (csOwnedChildrenNotSelectable in CS) then
        for N := 0 to WC.ControlCount - 1 do
        begin
          Control := WC.Controls[N];
          CS := Control.ControlStyle;
          if not (csNoDesignSelectable in CS) then
            AddControlToDelList(Control);
        end;
    end;
  end;

begin
  Result := False;
  if FDesignerManager.ReadOnly then Exit;

  if (DesignerSelections.Count = 1) and
     (DesignerSelections.Items[0] = FForm) then Exit;

  InvalidateMarkers(False);
  OBJList := TList.Create;
  try
    for I := 0 to DesignerSelections.Count - 1 do
    begin
      AddControlToDelList(TControl(DesignerSelections.Items[I]));
    end;
    DesignerSelections.Clear;
    DesignerSelections.Add(FForm);
    FDesignerManager.ChangeSelection(Root, DesignerSelections, FForm);
    FDrawMarkers := True;
    for I := OBJList.Count - 1 downto 0 do
    begin
      AComponent := TComponent(OBJList.Items[I]);
      if AComponent is TNonVisual then AComponent := TNonVisual(AComponent).Component;
      FDesignerManager.DeleteComponent(AComponent, Root);
      AComponent.Free;
    end;
    Modified;
  finally
    OBJList.Free;
  end;
  FForm.Invalidate;
  Result := True;
end;

procedure TDesignerHook.KeyDownOnControl(Sender: TControl; var TheMessage: TLMKeyDown);
var
  Shift: TShiftState;
  ResizeKey, MoveKey: TShiftStateEnum;

  procedure CustomResize(DX, DY: Integer);
  var
    I: Integer;
    C: TControl;
  begin
    if DesignerSelections.Count = 0 then Exit;

    if (DesignerSelections.Count = 1)
      and (DesignerSelections.Items[0] = FForm) then Exit;

    InvalidateMarkers(False);
    for I := 0 to DesignerSelections.Count -1 do
    begin
      if DesignerSelections.Items[I] is TControl then
      begin;
        C := TControl(DesignerSelections.Items[I]);
        if Shift = [MoveKey] then
        begin
          C.Left := C.Left + DX;
          C.Top := C.Top + DY;
        end
        else
        if Shift = [ResizeKey] then
        begin
          C.Width := C.Width + DX;
          C.Height := C.Height + DY;
        end;
      end;
    end;
    FDrawMarkers := True;
    Application.ProcessMessages;
    PaintSelections;
  end;

begin
  if FDesignerManager.ReadOnly then Exit;

  Shift := KeyDataToShiftState(TheMessage.KeyData);
  ResizeKey:= FDesignerManager.ControlKeyResize;
  MoveKey:= FDesignerManager.ControlKeyMove;

  case TheMessage.CharCode of
    VK_DELETE: DeleteSelection;
    VK_UP:
    begin
      if (Shift = [MoveKey]) or (Shift = [ResizeKey]) then CustomResize(0, -1);
    end;
    VK_DOWN:
    begin
      if (Shift = [MoveKey]) or (Shift = [ResizeKey]) then CustomResize(0, 1);
    end;
    VK_LEFT:
    begin
      if (Shift = [MoveKey]) or (Shift = [ResizeKey]) then CustomResize(-1, 0);
    end;
    VK_RIGHT:
    begin
      if (Shift = [MoveKey]) or (Shift = [ResizeKey]) then CustomResize(1, 0);
    end;
    VK_C, 83:
    begin
      if Shift = [ssCtrl] then CopySelection;
    end;
    VK_V, 77:
    begin
      if Shift = [ssCtrl] then PasteSelection([cpsfFindUniquePositions]);
    end;
    VK_X:
    begin
      if Shift = [ssCtrl] then CutSelection;
    end;
  end;
end;

procedure TDesignerHook.UTF8KeyPress(var UTF8Key: TUTF8Char);
begin

end;

function TDesignerHook.SizeControl(Sender: TControl; TheMessage: TLMSize): Boolean;
var
  P: TPoint;
  ASender: TControl;
begin
  Result := True;
  //if FSizeOperation = gmNone then InvalidateMarkers(False);
  Sender.Dispatch(TheMessage);
  Modified;
  ASender := Sender;
  if Sender = FForm then
  begin
    if FDM <> nil then
    begin
      P.x:= FForm.Width;
      P.y:= FForm.Height;
      FDM.DesignSize := P;
    end
    else
    if FFrame <> nil then
    begin
      ASender := FFrame;
      FFrame.Width := FForm.ClientWidth;
      FFrame.Height := FForm.ClientHeight;
    end;
  end
  else
  begin
    if (FFrame <> nil) and (Sender = FFrame) then
    begin
      Form.ClientWidth := FFrame.Width;
      Form.ClientHeight := FFrame.Height;
    end;
  end;

  if (Sender = FForm) and (FDesignerManager.ObjInspector <> nil) then
  begin
    if (FDesignerManager.PropEditHook.LookupRoot = Root) and
      (DesignerSelections.ControlInSelection(ASender)) then
    begin
      FDesignerManager.ObjInspector.RefreshPropertyValues;
    end;
  end;
end;

function TDesignerHook.MoveControl(Sender: TControl; TheMessage: TLMMove): Boolean;
begin
  Result := True;
  Sender.Dispatch(TheMessage);
  Modified;
  if Sender = FForm then
  begin
    if FDM <> nil then
    begin
      FDM.DesignInfo := LeftTopToDesignInfo(FForm.Left, FForm.Top);
    end;
  end;
  if (FDesignerManager.ObjInspector <> nil)and (Sender = FForm) then
  begin
    if (FDesignerManager.PropEditHook.LookupRoot = Root)
      and (DesignerSelections.ControlInSelection(Sender)) then
    begin
      FDesignerManager.ObjInspector.RefreshPropertyValues;
    end;
  end;
end;

function TDesignerHook.OnFormCloseQuery: boolean;
begin
  Result := True;
end;

procedure TDesignerHook.AddSelection(Control: TControl);
begin
  DesignerSelections.Add(Control);
  Application.ProcessMessages;
  //PaintSelections;
  FDesignerManager.ChangeSelection(Root, DesignerSelections, FForm);
end;

function TDesignerHook.IsDesignMsg(Sender: TControl; var TheMessage: TLMessage): Boolean;
begin
  Result := False;
  if not FDesignerManager.Active then Exit;

  if csDesigning in Sender.ComponentState then
  begin
    Result := True;
    case TheMessage.msg of
      LM_PAINT: Result := PaintControl(Sender, TLMPaint(TheMessage));
      LM_MOUSEMOVE: MouseMoveOnControl(Sender, TLMMouse(TheMessage));
      LM_LBUTTONDOWN,
      LM_RBUTTONDOWN,
      LM_LBUTTONDBLCLK:  MouseDownOnControl(Sender,TLMMouse(TheMessage));
      LM_LBUTTONUP,
      LM_RBUTTONUP:   MouseUpOnControl(Sender, TLMMouse(TheMessage));
      LM_SETCURSOR:   Result:=HandleSetCursor(TheMessage);
      CN_KEYDOWN, CN_SYSKEYDOWN:
      begin
        if Assigned(FDesignerManager.FIsIdeSysKey) then
          Result := not FDesignerManager.FIsIdeSysKey(TLMKeyDown(TheMessage).CharCode);

        if Result then
          KeyDownOnControl(Sender, TLMKeyDown(TheMessage));
      end;
      LM_SIZE: Result := SizeControl(Sender, TLMSize(TheMessage));
      LM_MOVE: Result := MoveControl(Sender, TLMMove(TheMessage));
      LM_CLOSEQUERY: Result := OnFormCloseQuery;
      LM_ACTIVATE: Result := OnFormActivate(Sender, TLMActivate(TheMessage));
      LM_CONTEXTMENU: HandlePopupMenu(Sender, TLMContextMenu(TheMessage));
    else
      Result := False;
    end;
    //Form.Canvas.TextOut(10,10, GetMessageName(TheMessage.msg) + '   ' + IntToStr(TheMessage.msg) + '        ');
  end;
end;

procedure TDesignerHook.SetNonVisual;
var
  I: Integer;
  DI: LongInt;
  ALeft, ATop: SmallInt;
  N: TNonVisual;
  Cm: TComponent;
 begin
  I := 0;
  while I < Root.ComponentCount do
  begin
    Cm := Root.Components[I];
    if Assigned(FDesignerManager.IsVisualComponent) then
    begin
      if not FDesignerManager.IsVisualComponent(Cm) then
      begin
        Inc(I);
        Continue;
      end;
    end
    else
    if (Cm is TMenuItem) or (Cm is TCustomAction) or (Cm is TField) then
    begin
        Inc(I);
        Continue;
    end;

    if not (Root.Components[I] is TControl) then
    begin
      N := TNonVisual.Create(Root.Components[I], FDesignerManager);
      N.Component := Root.Components[I];
      //FFormDesigner.GetImage(N.Component, N.Bmp);
      if FFrame = nil then
        N.Parent := FForm
      else
        N.Parent := FFrame;
      DI := Root.Components[I].DesignInfo;

      DesignInfoToLeftTop(DI, ALeft, ATop);
      N.Left := ALeft;
      N.Top := ATop;
      //SetWindowPos(N.Handle, HWND_TOPMOST, N.Left, N.Top, N.Width, N.Height, SWP_SHOWWINDOW or SWP_NOSIZE);
      N.SetZOrder(True);
    end;
    Inc(I);
  end;
  NonVisualSet := True;
end;

procedure TDesignerHook.HideHint;
begin
  FHintTimer.Enabled := False;
  FHintWIndow.Visible := False;
end;

function TDesignerHook.CopySelection: boolean;
begin
  Result := DoCopySelectionToClipboard;
end;

function TDesignerHook.CutSelection: boolean;
begin
  Result := False;
  if CopySelection then Result := DeleteSelection;
end;

function TDesignerHook.CanPaste: Boolean;
begin
  Result:= Assigned(Form) and
           Assigned(Root) and
           not (csDestroying in Root.ComponentState);
end;

function TDesignerHook.DesignSelectable(AComponent: TComponent): Boolean;
var
  I: Integer;
  N: TNonVisual;
  FormComponent: TComponent;
begin
  Result := False;
  I := 0;
  while I < FForm.ControlCount do
  begin
    if FForm.Controls[I] is TNonVisual then
    begin
      N := TNonVisual(FForm.Controls[I]);
      FormComponent := N.Component;
    end
    else
      FormComponent := FForm.Controls[I];

    if FormComponent = AComponent then
    begin
      Result := True;
      Break;
    end;
    Inc(I);
  end;
end;

function TDesignerHook.DoPasteSelectionFromClipboard(
  PasteFlags: TComponentPasteSelectionFlags): boolean;
var
  AllComponentText: string;
  CurTextCompStream: TMemoryStream;
begin
  Result:=false;
  if not CanPaste then exit;
  // read component stream from clipboard
  AllComponentText:=ClipBoard.AsText;
  if AllComponentText='' then exit;
  CurTextCompStream:=TMemoryStream.Create;
  try
    CurTextCompStream.Write(AllComponentText[1],length(AllComponentText));
    CurTextCompStream.Position:=0;
    if not DoInsertFromStream(CurTextCompStream,nil,PasteFlags) then
      exit;
  finally
    CurTextCompStream.Free;
  end;
  Result:=true;
end;

procedure TDesignerHook.ReadingComponentnSetName(Reader: TReader;
  Component: TComponent; var AName: string);
var
  I: Integer;
  B: Boolean;
begin
  B := False;
  for I := 0 to Root.ComponentCount - 1 do
  begin
    if SameText(Root.Components[I].Name, AName) then
    begin
      B := True;
      Break;
    end;
  end;
  if B then AName:= UniqueName(Component.ClassName);
end;

function TDesignerHook.DoInsertFromStream(s: TStream; PasteParent: TWinControl;
  PasteFlags: TComponentPasteSelectionFlags): Boolean;
var
  NewSelection: TDesignerSelections;
  NewComponents: TFPList;

  procedure FindUniquePosition(AComponent: TComponent);
  var
    P: TPoint;
    AControl: TControl;
    AParent: TWinControl;
    i, NonVisualCompWidth: Integer;
    OverlappedControl: TControl;

  begin
    NonVisualCompWidth := 32;
    if AComponent is TControl then
    begin
      AControl:=TControl(AComponent);
      AParent:=AControl.Parent;
      if AParent=nil then exit;
      P:=Point(AControl.Left,AControl.Top);
      i:=AParent.ControlCount-1;
      while i>=0 do
      begin
        OverlappedControl:=AParent.Controls[i];
        if (NewComponents.IndexOf(OverlappedControl)<0)
        and (OverlappedControl.Left=P.X)
        and (OverlappedControl.Top=P.Y) then
        begin
          inc(P.X,NonVisualCompWidth);
          inc(P.Y,NonVisualCompWidth);
          if (P.X>AParent.ClientWidth-AControl.Width)
          or (P.Y>AParent.ClientHeight-AControl.Height) then
            break;
          i:=AParent.ControlCount-1;
        end else
          dec(i);
      end;
      P.x:=Max(0,Min(P.x,AParent.ClientWidth-AControl.Width));
      P.y:=Max(0,Min(P.y,AParent.ClientHeight-AControl.Height));
      AControl.SetBounds(P.x,P.y,AControl.Width,AControl.Height);
      if AControl is TNonVisual then
      begin
        TNonVisual(AControl).SetLabelBounds(AControl.Left, AControl.Top);
      end;
    end;
    {else
    begin
      P:=GetParentFormRelativeTopLeft(AComponent);
      repeat
        OverlappedComponent:= nil; //NonVisualComponentAtPos(P.x,P.y);
        if (OverlappedComponent=nil) then break;
        inc(P.X,NonVisualCompWidth);
        inc(P.Y,NonVisualCompWidth);
        if (P.X+NonVisualCompWidth>Form.ClientWidth)
        or (P.Y+NonVisualCompWidth>Form.ClientHeight) then
          break;
      until false;
      AComponent.DesignInfo := LeftTopToDesignInfo(
        SmallInt(Max(0, Min(P.x, Form.ClientWidth - NonVisualCompWidth))),
        SmallInt(Max(0, Min(P.y, Form.ClientHeight - NonVisualCompWidth))));
    end;}
  end;

var
  i: Integer;
  NewComponent, AComp: TComponent;
  NV: TNonVisual;
begin
  Result:=false;
  //debugln('TDesigner.DoInsertFromStream A');
  if (cpsfReplace in PasteFlags) and (not DeleteSelection) then exit;

  //debugln('TDesigner.DoInsertFromStream B s.Size=',dbgs(s.Size),' S.Position=',dbgs(S.Position));
  if PasteParent=nil then PasteParent := GetPasteParent;
  NewSelection:=TDesignerSelections.Create;
  NewComponents:=TFPList.Create;
  try
    Form.DisableAutoSizing;
    try

      // read component stream from clipboard
      if (s.Size<=S.Position) then begin
        debugln('TDesigner.DoInsertFromStream Stream Empty s.Size=',dbgs(s.Size),' S.Position=',dbgs(S.Position));
        exit;
      end;

      // create components and add to LookupRoot
      OnDesignerPasteComponents(s, PasteParent, NewComponents);

      // add new component to new selection
      for i:=0 to NewComponents.Count-1 do
      begin
        NewComponent:=TComponent(NewComponents[i]);
        NewComponent.FreeNotification(Form);
        AComp := NewComponent;
        if not (NewComponent is TControl) then
        begin
          NV := TNonVisual.Create(NewComponent, FDesignerManager);
          NV.Parent := FForm;
          NV.Component := NewComponent;
          NV.Left := LeftFromDesignInfo(NewComponent.DesignInfo);
          NV.Top := TopFromDesignInfo(NewComponent.DesignInfo);
          NV.SetZOrder(True);
          NewComponent := NV;
        end;
        NewSelection.Add(NewComponent);
        // set new nice bounds
        if cpsfFindUniquePositions in PasteFlags then
          FindUniquePosition(NewComponent);
        // finish adding component
        AfterAddComponent(AComp);
        Modified;
      end;

      {if NewSelection.Count>0 then
        FOnPastedComponents(Self,FLookupRoot);}

    finally
      Form.EnableAutoSizing;
    end;
  finally
    NewComponents.Free;
    if NewSelection.Count>0 then
    begin
      ClearSelection;
      DesignerSelections.Assign(NewSelection);
      PaintSelections;
      FDesignerManager.ChangeSelection(Root, DesignerSelections, Form);
    end;
    NewSelection.Free;
  end;
  Result:=true;
end;

function TDesignerHook.GetPasteParent: TWinControl;
begin
  Result := Form;
  if FFrame <> nil then Result := FFrame;
  if DesignerSelections.Count = 1 then
  begin
    if DesignerSelections.Items[0] is TWinControl then
    begin
      Result := TWinControl(DesignerSelections.Items[0]);
      if not (csAcceptsControls in Result.ControlStyle) then
        Result := Result.Parent;
    end
    else
    if DesignerSelections.Items[0] is TControl then
      Result := TControl(DesignerSelections.Items[0]).Parent;
  end;
end;

procedure TDesignerHook.OnDesignerPasteComponents(TxtCompStream: TStream;
  ParentControl: TWinControl; var NewComponents: TFPList);
var
  BinCompStream: TMemoryStream;
  c: Char;
  Reader: TReader;
begin
  DebugLn('OnDesignerPasteComponent A');

  BinCompStream:=TMemoryStream.Create;
  try
    try
      LRSObjectTextToBinary(TxtCompStream,BinCompStream);
      // always append an "object list end"
      c:=#0;
      BinCompStream.Write(c,1);
    except
      on E: Exception do
      begin
        ShowMessage('Error to convert Text To Object');
        exit;
      end;
    end;
    BinCompStream.Position:=0;
    Reader := TReader.Create(BinCompStream, 4096);
    Reader.Root := Root;
    Reader.OnSetName := @ReadingComponentnSetName;
    FDesignerManager.LoadHandlers := MethodsHandlers;
    Reader.OnSetMethodProperty := @FDesignerManager.ReaderSetMethodProp;
    FReadComponentList := NewComponents;
    try
      Reader.ReadComponents(Root, ParentControl, @ReadComp);
    finally
      Reader.Free;
    end;
    // create the component
    if NewComponents.Count=0 then
    begin
      DebugLn('TMainIDE.OnDesignerPasteComponent FAILED FormEditor1.CreateChildComponentFromStream');
      exit;
    end;

  finally
    BinCompStream.Free;
  end;
end;

procedure TDesignerHook.ReadComp(Component: TComponent);
begin
  FReadComponentList.Add(Component);
end;

procedure TDesignerHook.DoDoubleClickOnComponent(Component: TComponent);
var
  BaseCompEditor: TBaseComponentEditor;
  CompEditorDesigner: TComponentEditorDesigner;
begin
  CompEditorDesigner := TComponentEditorDesigner(Self);
  BaseCompEditor := GetComponentEditor(Component, CompEditorDesigner);

  FMovSelection:= False;
  FCR:= False;
  Application.ProcessMessages;
  if BaseCompEditor <> nil then BaseCompEditor.Edit;
  //PaintSelections;
  //Application.ProcessMessages;
  if Assigned(FDesignerManager.OnDobleClickComponent) then
    FDesignerManager.OnDobleClickComponent(Component, Root);
end;

procedure TDesignerHook.HandlePopupMenu(Sender: TControl;
  var Message: TLMContextMenu);
begin
  if (DesignerSelections.Count > 0) then
  begin
    if DesignerSelections.Items[0] is TComponent then
    begin
      FDesignerManager.DesignPopupMenu(Message.Pos, DesignerSelections);
      Message.Result := 0;
    end;
  end;
  Message.Result := 1;
end;

function TDesignerHook.PasteSelection(Flags: TComponentPasteSelectionFlags
  ): boolean;
begin
  Result:=DoPasteSelectionFromClipboard(Flags);
end;

function TDesignerHook.CopySelectionToStream(AllComponentsStream: TStream): boolean;
var
  i: Integer;
  BinCompStream: TMemoryStream;
  TxtCompStream: TMemoryStream;
  CurComponent: TComponent;
  DestroyDriver: Boolean;
  Writer: TWriter;
begin
  Result:=false;
  if (DesignerSelections.Count=0) then exit;

  for i:=0 to DesignerSelections.Count-1 do
  begin
    if not (DesignerSelections.Items[i] is TComponent) then continue;

    BinCompStream:=TMemoryStream.Create;
    TxtCompStream:=TMemoryStream.Create;
    try
      // write component binary stream
      try
        CurComponent := TComponent(DesignerSelections.Items[i]);
        if CurComponent is TNonVisual then
           CurComponent := TNonVisual(CurComponent).Component;

        DestroyDriver:=false;
        Writer := CreateLRSWriter(BinCompStream,DestroyDriver);
        try
          Writer.OnWriteMethodProperty:=@FDesignerManager.WriteMethodProp;
          Writer.Root:= Root;
          Writer.WriteComponent(CurComponent);
        finally
          if DestroyDriver then Writer.Driver.Free;
          Writer.Destroy;
        end;
      except
        on E: Exception do
        begin
          MessageDlg(lisUnableToStreamSelectedComponents,
            Format(lisThereWasAnErrorDuringWritingTheSelectedComponent, [
              CurComponent.Name, CurComponent.ClassName, #13, E.Message]),
            mtError,[mbOk],0);
          exit;
        end;
      end;
      BinCompStream.Position:=0;
      // convert binary to text stream
      try
        LRSObjectBinaryToText(BinCompStream,TxtCompStream);
      except
        on E: Exception do
        begin
          MessageDlg(lisUnableConvertBinaryStreamToText,
            Format(lisThereWasAnErrorWhileConvertingTheBinaryStreamOfThe, [
              CurComponent.Name, CurComponent.ClassName, #13, E.Message]),
            mtError,[mbOk],0);
          exit;
        end;
      end;
      // add text stream to the all stream
      TxtCompStream.Position:=0;
      AllComponentsStream.CopyFrom(TxtCompStream,TxtCompStream.Size);
    finally
      BinCompStream.Free;
      TxtCompStream.Free;
    end;
  end;
  Result:=true;
end;

procedure TDesignerHook.Modified;
begin
  FModified := True;
end;

function TDesignerHook.IsModified: Boolean;
begin
  Result := FModified;
end;

procedure TDesignerHook.ResetModified;
begin
  FModified := False;
end;

procedure TDesignerHook.BringToFront;
var
  I: Integer;
begin
  if DesignerSelections.Count = 0 then Exit;
  I := 0;
  while I < DesignerSelections.Count do
  begin
    if DesignerSelections.Items[I] is TControl then
      TControl(DesignerSelections.Items[I]).BringToFront;
    Inc(I);
  end;
  UpNonVisual;
  PaintSelections;
end;

procedure TDesignerHook.SendToBack;
var
  I: Integer;
begin
  if DesignerSelections.Count = 0 then Exit;
  I := 0;
  while I < DesignerSelections.Count do
  begin
    if DesignerSelections.Items[I] is TControl then
      TControl(DesignerSelections.Items[I]).SendToBack;
    Inc(I);
  end;
  UpNonVisual;
  PaintSelections;
end;

procedure TDesignerHook.Notification(AComponent: TComponent;
        Operation: TOperation);
var
  ARoot: TComponent;
begin
  if Operation = opRemove then
  begin
    if AComponent.Owner <> nil then ARoot := AComponent.Owner
    else ARoot := AComponent;

    if ARoot = Root then
    begin
      FDesignerManager.DeleteComponent(AComponent, Root);
      Modified;
    end;
  end;
end;

procedure TDesignerHook.PaintGrid;
var
  R: TRect;
begin
  if not FDesignerManager.ShowGrid then Exit;
  {$IFDEF UNIX}
  R := FForm.ClientRect;
  {$ELSE}
  R := FForm.Canvas.ClipRect;
  {$ENDIF}
    DrawGrid(FForm.Canvas.Handle, R, FDesignerManager.GridStep, FDesignerManager.GridStep);
  //DrawGrid(FForm.Canvas.Handle, FForm.Canvas.ClipRect, FDesignerManager.GridStep, FDesignerManager.GridStep);
end;

procedure TDesignerHook.ValidateRename(AComponent: TComponent;
      const CurName, NewName: string);
begin

end;

function TDesignerHook.CanUndo: Boolean;
begin
  Result := False;
end;

function TDesignerHook.CanRedo: Boolean;
begin
  Result := False;
end;

function TDesignerHook.Undo: Boolean;
begin
  Result := False;
end;

function TDesignerHook.Redo: Boolean;
begin
  Result := False;
end;

procedure TDesignerHook.UpdateCaption;
begin
  Form.Caption := Root.Name;
end;

function TDesignerHook.IsUndoLocked: boolean;
begin
  Result := false;
end;

function TDesignerHook.AddUndoAction(const aPersistent: TPersistent;
  aOpType: TUndoOpType; IsSetNewId: boolean; aFieldName: string; const aOldVal,
  aNewVal: variant): boolean;
begin
  Result := False;
end;

procedure TDesignerHook.ComponentRenamed(AComponent: TComponent);
var
  NV: TNonVisual;
begin
  if (AComponent is TDataModule) or (AComponent is TCustomFrame) then
  begin
    if AComponent = Root then UpdateCaption;
  end
  else
  if not (AComponent is TControl) then
  begin
    NV := GetNonVisualOfObject(AComponent);
    if NV <> nil then NV.UpdateCaption;
  end;
end;

function TDesignerHook.GetShiftState: TShiftState;
begin

end;

procedure TDesignerHook.SelectOnlyThisComponent(AComponent: TComponent);
begin
  ClearSelection;
  DesignerSelections.Add(AComponent);
  FDesignerManager.ChangeSelection(Root, DesignerSelections, Form);
end;

function TDesignerHook.UniqueName(const BaseName: string): string;
var
  I: Integer;
  S: String;
begin
  Result := ClassNameToComponentName(BaseName);
  for I := 1 to 1000 do
  begin
    S := Result + IntToStr(I);
    if Root.FindComponent(S) = nil then
    begin
      Result := S;
      Break;
    end;
  end;
end;

function TDesignerHook.CreateUniqueComponentName(const AClassName: string): string;
begin
  Result := UniqueName(AClassName);
end;

procedure TDesignerHook.DesignClose;
begin
  Form.Designer := nil;
  SetComponentDesignMode(Form, False);
  if PropertyEditorHook.LookupRoot = Root then
    FDesignerManager.ChangeSelection(nil, DesignerSelections, nil);
end;

procedure TDesignerHook.DesignOpen;
begin
  Form.Designer := Self;
  SetComponentDesignMode(Form, True);
  if Root <> Form then
  begin
    SetComponentDesignMode(Root, True);
    if not NonVisualSet then
    begin
      FDesignerManager.InitialDesignForm(Root, Form);
      SetNonVisual;
    end;
  end;
  if not NonVisualSet then DesignerSelections.Add(Root);
end;

procedure TDesignerHook.SetForm(AForm: TCustomForm);
begin
  if FForm <> AForm then
  begin
    FForm := AForm;
    if FForm <> nil then
    begin
      if FForm.Designer = nil then FForm.Designer := Self;
      //if not (Root is TDataModule) then
        SetComponentDesignMode(FForm, True);
      Canvas := TCanvas.Create;
      Canvas.Handle := GetDesignerDC(FForm.Handle);
      Canvas.Handle := 0;
      //PaintGrid;
    end;
  end;
end;

procedure TDesignerHook.OnHintTimer(Sender: TObject);

  function GetComponentHintText(AComponent: TComponent): String;
  const
    HintNameStr = '%s: %s';
    HintPositionStr = 'Position: %d, %d';
    HintSizeStr = 'Size: %d x %d';
    HintTabStr = 'TabStop: %s; TabOrder: %d';
  var
    AControl: TControl absolute AComponent;
    AWinControl: TWinControl absolute AComponent;
    AComponentEditor: TBaseComponentEditor;
    S: String;
  begin
    // component name and classname
    Result := Format(HintNameStr, [AComponent.Name, AComponent.ClassName]);
    // component position
    Result := Result + LineEnding + Format(HintPositionStr, [GetComponentLeft(AComponent), GetComponentTop(AComponent)]);
    if AComponent is TControl then // more info for controls
    begin
      // size
      Result := Result + '; ' + Format(HintSizeStr, [AControl.Width, AControl.Height]);
      // and TabStop, TabOrder for TWinControl
      if (AComponent is TWinControl) and not (AComponent = Form) then
        Result := Result + LineEnding + Format(HintTabStr, [BoolToStr(AWinControl.TabStop, True), AWinControl.TabOrder]);
    end;
    AComponentEditor := GetComponentEditor(AComponent, Self);
    if Assigned(AComponentEditor) then
    begin
      S := AComponentEditor.GetCustomHint;
      if S <> '' then
        Result := Result + LineEnding + S;
      AComponentEditor.Free;
    end;
  end;

var
  Rect: TRect;
  AHint, OldHint: String;
  Position, ClientPos: TPoint;
  AWinControl: TWinControl;
  AComponent: TComponent;
  Contr: TControl;
  I: Integer;
begin
  FHintTimer.Enabled := False;
  if (not FDesignerManager.ShowHints) then Exit;

  Position := Mouse.CursorPos;
  AWinControl := FindLCLWindow(Position);
  if not (Assigned(AWinControl)) then Exit;
  if GetDesignerForm(AWinControl) <> Form then exit;

   ClientPos := Form.ScreenToClient(Position);
  if FMovSelection then
  begin
      if (OperationControl <> nil) then
      begin
        if (FDesignerManager.MoveObjectMethod = movKontur) or FTempMoveKontur then
        begin
          I := GetSelectionIndex(OperationControl);
          Rect := PRect(FMoveRectList.Items[I])^;
          if OperationControl.Parent <> FForm then
          begin
            Rect.TopLeft := FForm.ClientToScreen(Rect.TopLeft);
            Rect.TopLeft := OperationControl.Parent.ScreenToClient(Rect.TopLeft);
          end;
          AHint := IntToStr(Rect.Left) + ', ' +  IntToStr(Rect.Top);
        end
        else
          AHint := IntToStr(OperationControl.Left) + ', ' +  IntToStr(OperationControl.Top);
      end;
  end
  else
  if FSizeOperation <> gmNone then
  begin
    if (OperationControl <> nil) and not(OperationControl is TNonVisual) then
    begin
      if (FDesignerManager.MoveObjectMethod = movKontur) or FTempMoveKontur then
        begin
          I := GetSelectionIndex(OperationControl);
          Rect := PRect(FMoveRectList.Items[I])^;
          if OperationControl.Parent <> FForm then
          begin
            Rect.TopLeft := FForm.ClientToScreen(Rect.TopLeft);
            Rect.TopLeft := OperationControl.Parent.ScreenToClient(Rect.TopLeft);
          end;
          AHint := IntToStr(Rect.Right - Rect.Left) + ' x ' +  IntToStr(Rect.Bottom - Rect.Top);
        end
        else
         AHint:= IntToStr(OperationControl.Width) + ' x ' +  IntToStr(OperationControl.Height);;
    end;
  end
  else
  begin
      if not PtInRect(Form.ClientRect, ClientPos) then Exit;
      Contr := ControlAtPos(Form, ClientPos);
      if Contr is TComponentCaption then Contr := TControl(Contr.Owner);
      AComponent := Contr;
      if Contr is TNonVisual then AComponent := TNonVisual(Contr).Component;
      if (csNoDesignSelectable in Contr.ControlStyle) then Contr := Contr.Parent
      else
      if csNoDesignVisible in Contr.ControlStyle then Contr := Contr.Parent;
      //AComponent := ComponentAtPos(ClientPos.X,ClientPos.Y,true,true);
      if not Assigned(AComponent) then
        AComponent := AWinControl;
      if AComponent = nil then exit;
      AHint := GetComponentHintText(AComponent);
  end;

  if AHint = '' then Exit;
  OldHint:= FHintWIndow.Caption;
  Rect := FHintWindow.CalcHintRect(0, AHint, nil);  //no maxwidth
  Rect.Left := Position.X + 15;
  Rect.Top := Position.Y + 20;
  Rect.Right := Rect.Left + Rect.Right;
  Rect.Bottom := Rect.Top + Rect.Bottom;
  FHintWIndow.Caption:= AHint;
  FHintWindow.ActivateHint(Rect, AHint);
  if OldHint <> AHint then FHintWIndow.Invalidate;
end;

procedure TDesignerHook.ClearMoveRect;
var
  I: Integer;
  P: PRect;
begin
  for I := 0 to FMoveRectList.Count - 1 do
  begin
    P := PRect(FMoveRectList.Items[I]);
    Dispose(P);
  end;
  FMoveRectList.Clear;
end;

function TDesignerHook.GetPropertyEditorHook: TPropertyEditorHook;
begin
  Result := FDesignerManager.PropEditHook;
end;

procedure TDesignerHook.SetTempCursor(ARoot: TWinControl; ACursor: TCursor);

  procedure Traverse(ARoot: TWinControl);
  var
    i: integer;
  begin
    for i := 0 to ARoot.ControlCount - 1 do
    begin
      ARoot.Controls[i].SetTempCursor(ACursor);
      if ARoot.Controls[i] is TWinControl then
        Traverse(TWinControl(ARoot.Controls[i]));
    end;
  end;

begin
  Traverse(ARoot);
  ARoot.SetTempCursor(ACursor);
end;

function TDesignerHook.HandleSetCursor(var TheMessage: TLMessage): boolean;
begin
  Result := Lo(DWord(TheMessage.LParam)) = HTCLIENT;
  if Result then
  begin
    SetTempCursor(Form, LastFormCursor);
    TheMessage.Result := 1;
  end;
end;

procedure TDesignerHook.CreateMoveRect;
var
  I: Integer;
  C: TControl;
  PR: PRect;
  AParent: TWinControl;
begin
  AParent := FForm;
  if FFrame <> nil then AParent := FFrame;
  for I := 0 to DesignerSelections.Count - 1 do
  begin
    C := TControl(DesignerSelections.Items[I]);
    New(PR);
    PR^:= C.BoundsRect;
    if C.Parent <> AParent then
      RectToFormRect(PR, C.Parent);
    FMoveRectList.Add(PR);
  end;
  PaintSelectionRect;
end;

function TDesignerHook.GetSelectionIndex(AControl: TPersistent): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FDesignerSelections.Count -1 do
  begin
    if FDesignerSelections.Items[I] = AControl then
    begin
      Result := I;
      Break;
    end;
  end;
end;

procedure TDesignerHook.GetMouseMsgShift(TheMessage: TLMMouse;
  var Shift: TShiftState; var Button: TMouseButton);
begin
  Shift := [];
  if (TheMessage.Keys and MK_Shift) = MK_Shift then
    Include(Shift, ssShift);
  if (TheMessage.Keys and MK_Control) = MK_Control then
    Include(Shift, ssCtrl);

  if GetKeyState(VK_MENU) < 0 then Include(Shift, ssAlt);
  if (GetKeyState(VK_LWIN) < 0) or (GetKeyState(VK_RWIN) < 0) then Include(Shift, ssMeta);

  case TheMessage.Msg of
  LM_LBUTTONUP,LM_LBUTTONDBLCLK,LM_LBUTTONTRIPLECLK,LM_LBUTTONQUADCLK:
    begin
      Include(Shift, ssLeft);
      Button := mbLeft;
    end;
  LM_MBUTTONUP,LM_MBUTTONDBLCLK,LM_MBUTTONTRIPLECLK,LM_MBUTTONQUADCLK:
    begin
      Include(Shift, ssMiddle);
      Button := mbMiddle;
    end;
  LM_RBUTTONUP,LM_RBUTTONDBLCLK,LM_RBUTTONTRIPLECLK,LM_RBUTTONQUADCLK:
    begin
      Include(Shift, ssRight);
      Button := mbRight;
    end;
  else
    if (TheMessage.Keys and MK_MButton) <> 0 then
    begin
      Include(Shift, ssMiddle);
      Button := mbMiddle;
    end;
    if (TheMessage.Keys and MK_RButton) <> 0 then
    begin
      Include(Shift, ssRight);
      Button := mbRight;
    end;
    if (TheMessage.Keys and MK_LButton) <> 0 then
    begin
      Include(Shift, ssLeft);
      Button := mbLeft;
    end;
    if (TheMessage.Keys and MK_XBUTTON1) <> 0 then
    begin
      Include(Shift, ssExtra1);
      Button := mbExtra1;
    end;
    if (TheMessage.Keys and MK_XBUTTON2) <> 0 then
    begin
      Include(Shift, ssExtra2);
      Button := mbExtra2;
    end;
  end;

  case TheMessage.Msg of
  LM_LBUTTONDBLCLK,LM_MBUTTONDBLCLK,LM_RBUTTONDBLCLK,LM_XBUTTONDBLCLK:
    Include(Shift, ssDouble);
  LM_LBUTTONTRIPLECLK,LM_MBUTTONTRIPLECLK,LM_RBUTTONTRIPLECLK,LM_XBUTTONTRIPLECLK:
    Include(Shift, ssTriple);
  LM_LBUTTONQUADCLK,LM_MBUTTONQUADCLK,LM_RBUTTONQUADCLK,LM_XBUTTONQUADCLK:
    Include(Shift, ssQuad);
  end;
end;

function TDesignerHook.ControlAtPos(Sender: TControl; P: TPoint): TControl;
var
  Control: TControl;
  WControl: TWinControl;

  function GetControlAt(AControl: TWinControl): TControl;
  var
    I: Integer;
    R: TRect;
    C: TControl;
  begin
    Result := nil;
    I := AControl.ControlCount - 1;
    while I >= 0 do
    begin
      C := AControl.Controls[I];
      if (C is TWinControl) and not (csNoDesignVisible in C.ControlStyle)
      and not (csNoDesignSelectable in C.ControlStyle)
      and not (csSubComponent in C.ComponentStyle) then
      begin
        R := C.BoundsRect;
        if PtInRect(R, P) then
        begin
            Result := C;
            Break;
        end;
      end;
      Dec(I);
    end;
    if Result = nil then
    begin
      I := AControl.ControlCount - 1;
      while I >= 0 do
      begin
        C := AControl.Controls[I];
        if not (C is TWinControl) and not (csNoDesignVisible in C.ControlStyle)
        and not (csNoDesignSelectable in C.ControlStyle) then
        begin
          R := C.BoundsRect;
          if PtInRect(R, P) then
          begin
            Result := C;
            Break;
          end;
        end;
        Dec(I);
      end;
    end;
  end;

begin
  Result := Sender;

  if Sender is TWinControl then
  begin
    WControl := TWinControl(Sender);
    if (csAcceptsControls in WControl.ControlStyle) or (WControl.FCompStyle = csNotebook) then
    begin
      Control := GetControlAt(WControl);
      if (Control is TFrame) and (Control.Owner <> nil) then
      begin
        Result := Control;
      end
      else
      if (Control <> nil) then
      begin
        P := Sender.ClientToScreen(P);
        P := Control.ScreenToClient(P);
        Result := ControlAtPos(Control, P);
      end;
    end
    else Result := WControl;

  end;

end;

function TDesignerHook.GetGrabberMarker(C: TControl; P: TPoint): TGrabberMarker;

  function PosInGrabber(MI: TGrabberMarker): Boolean;
  var R: TRect;
  begin
    Result := False;
    R := GetMarkerRect(C, MI);
    if PtInRect(R, P) then Result := True;
  end;

begin
    Result := gmNone;
    if PosInGrabber(gmLeftTop) then Result := gmLeftTop
    else
    if PosInGrabber(gmLeft) then Result := gmLeft
    else
    if PosInGrabber(gmLeftBottom) then Result := gmLeftBottom
    else
    if PosInGrabber(gmTop) then Result := gmTop
    else
    if PosInGrabber(gmBottom) then Result := gmBottom
    else
    if PosInGrabber(gmRightTop) then Result := gmRightTop
    else
    if PosInGrabber(gmRight) then Result := gmRight
    else
    if PosInGrabber(gmRightBottom) then Result := gmRightBottom;
end;

procedure TDesignerHook.MouseMoveOnControl(Sender: TControl;
  var TheMessage: TLMMouse);
var
  Button: TMouseButton;
  ACursor: TCursor;
  C: TControl;
  P, LastMousePos: TPoint;
  GM: TGrabberMarker;
  DX, DY: Integer;
  Shift: TShiftState;

  function HasButtonInSelection: Boolean;
  var
    I: Integer;
  begin
    Result := False;
    for I := 0 to FDesignerSelections.Count - 1 do
    begin
      if FDesignerSelections.Items[I] is TCustomButton then
      begin
        Result := True;
        Break;
      end;
    end;
  end;

begin
  GetMouseMsgShift(TheMessage, Shift, Button);
  LastMousePos := GetFormRelativeMousePosition(FForm);
  if not FMovSelection and (FSizeOperation = gmNone) then
  begin
    FHintTimer.Enabled := False;
    FHintTimer.Enabled := True;
    FHintWIndow.Visible := False;
  end;

  if FCR then
  begin
    FHintTimer.Enabled := False;
    DrawCaptureRect;
    DX := LastMousePos.X;
    DY := LastMousePos.Y;
    if DX <= FMovePos.x then
    begin
      CaptureRect.Left := DX;
      CaptureRect.Right := FMovePos.x;
    end
    else
    begin
      CaptureRect.Left := FMovePos.x;
      CaptureRect.Right := DX;
    end;
    if DY <= FMovePos.y then
    begin
      CaptureRect.Top := DY;
      CaptureRect.Bottom:= FMovePos.y;
    end
    else
    begin
      CaptureRect.Top := FMovePos.y;
      CaptureRect.Bottom := DY;
    end;

    DrawCaptureRect;
  end
  else
  if FSizeOperation <> gmNone then
  begin
    FHintTimer.Enabled := False;
    P.x := LastMousePos.X;
    P.y := LastMousePos.Y;
    if not (ssAlt in Shift) then  P := MousePosToGridPos(P);
    DX := P.x - FMovePos.x;
    DY := P.y - FMovePos.y;
    if not (ssAlt in Shift) then
    begin
      DX := DX div FDesignerManager.GridStep * FDesignerManager.GridStep;
      DY := DY div FDesignerManager.GridStep * FDesignerManager.GridStep;
    end;
    if (DX <> 0) or (DY <> 0) then
    begin
      Exclude(MouseDownShift, ssDouble);
      ResizeControl(DX, DY);
      OnHintTimer(FHintTimer);
    end;
    FMovePos := P;
    ACursor := crDefault;
    case FSizeOperation of
      gmLeftTop: ACursor := crSizeNW;
      gmLeft, gmRight: ACursor := crSizeWE;
      gmLeftBottom: ACursor := crSizeNESW;
      gmTop, gmBottom: ACursor := crSizeNS;
      gmRightTop: ACursor := crSizeNE;
      gmRightBottom: ACursor:= crSizeSE;
    end;
    if LastFormCursor <> ACursor then
    begin
      SetTempCursor(FForm, ACursor);
      LastFormCursor := ACursor;
    end;
  end
  else
  if FMovSelection then
  begin
    P.x := LastMousePos.X;
    P.y := LastMousePos.Y;
    if not (ssAlt in Shift) then  P := MousePosToGridPos(P);
    DX := P.x - FMovePos.x;
    DY := P.y - FMovePos.y;
    if not (ssAlt in Shift) then
    begin
      DX := DX div FDesignerManager.GridStep * FDesignerManager.GridStep;
      DY := DY div FDesignerManager.GridStep * FDesignerManager.GridStep;
    end;
    if (DX <> 0) or (DY <> 0) then
    begin
      Exclude(MouseDownShift, ssDouble);
      MoveSelection(DX, DY);
      OnHintTimer(FHintTimer);
    end;
    FMovePos := P;
  end
  else
  if (DesignerSelections.Count = 1) and
     (DesignerSelections.Items[0] <> FForm) and
     (DesignerSelections.Items[0] is TControl) then
  begin
    FHintTimer.Enabled := False;
    FHintTimer.Enabled := True;
    FHintWindow.Visible := False;

    C := TControl(DesignerSelections.Items[0]);
    if not (C is TNonVisual) then
    begin
      P.x := LastMousePos.X;
      P.y := LastMousePos.Y;
      GM := GetGrabberMarker(C, P);
      ACursor := crDefault;
      case GM of
        gmLeftTop: ACursor := crSizeNW;
        gmLeft, gmRight: ACursor := crSizeWE;
        gmLeftBottom: ACursor := crSizeNESW;
        gmTop, gmBottom: ACursor := crSizeNS;
        gmRightTop: ACursor := crSizeNE;
        gmRightBottom: ACursor:= crSizeSE;
      end;
      if LastFormCursor <> ACursor then
      begin
        SetTempCursor(FForm, ACursor);
        LastFormCursor := ACursor;
      end;
    end;
  end
  else
    Exclude(MouseDownShift, ssDouble);

  //костыль для кнопок, приводит к пожиранию памяти
  if not FMovSelection and (FSizeOperation = gmNone)
     and HasButtonInSelection then PaintSelections;
end;

procedure TDesignerHook.ResizeControl(DX, DY: Integer);
var
  L, ATop: Integer;
  R: TRect;
  PR: PRect;
  AParent: TWinControl;
begin
  if FDesignerManager.ReadOnly then Exit;


  if (FDesignerManager.MoveObjectMethod = movObject) and not FTempMoveKontur then
  begin
    R := SizingControl.BoundsRect;
    InvalidateMarkers(False);
    Modified;

    case FSizeOperation of
      gmRight:
      begin
        Inc(R.Right, DX);
        if R.Right < R.Left then
        begin
          L := R.Left;
          R.Left := R.Right;
          R.Right := L;
          FSizeOperation := gmLeft;
        end;
        SizingControl.BoundsRect := R;
        if FDrawMarkers then PaintSelections;
      end;
      gmLeft:
      begin
       Inc(R.Left, DX);
       if R.Left > R.Right then
       begin
          L := R.Right;
          R.Right := R.Left;
          R.Left := L;
          FSizeOperation := gmRight;
        end;
        SizingControl.BoundsRect := R;
        if FDrawMarkers then PaintSelections;
      end;
      gmTop:
      begin
       Inc(R.Top, DY);
       if R.Top > R.Bottom then
       begin
          L := R.Bottom;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmBottom;
        end;
        L := SizingControl.Height;
        ATop := SizingControl.Top;
        SizingControl.BoundsRect := R;
        if SizingControl.Height = L then SizingControl.Top := ATop;
        if FDrawMarkers then PaintSelections;
      end;
      gmBottom:
      begin
       Inc(R.Bottom, DY);
       if R.Bottom < R.Top then
       begin
          L := R.Top;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmTop;
        end;
        SizingControl.BoundsRect := R;
        if FDrawMarkers then PaintSelections;
      end;
      gmRightBottom:
      begin
        Inc(R.Bottom, DY);
        Inc(R.Right, DX);
        if R.Bottom < R.Top then
        begin
          L := R.Top;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmLeftTop;
        end;
        if R.Right < R.Left then
        begin
          L := R.Left;
          R.Left := R.Right;
          R.Right := L;
          FSizeOperation := gmLeftTop;
        end;
        SizingControl.BoundsRect := R;
        if FDrawMarkers then PaintSelections;
      end;
      gmLeftTop:
      begin
        Inc(R.Top, DY);
        Inc(R.Left, DX);
        if R.Top > R.Bottom then
        begin
          L := R.Bottom;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmRightBottom;
        end;
        if R.Left > R.Right then
        begin
          L := R.Right;
          R.Right := R.Left;
          R.Left := L;
          FSizeOperation := gmRightBottom;
        end;
        ATop := SizingControl.Top;
        L := SizingControl.Height;
        SizingControl.BoundsRect := R;
        if SizingControl.Height = L then SizingControl.Top := ATop;
        if FDrawMarkers then PaintSelections;
      end;
      gmLeftBottom:
      begin
        Inc(R.Bottom, DY);
        Inc(R.Left, DX);
        if R.Bottom < R.Top then
        begin
          L := R.Top;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmRightTop;
        end;
        if R.Left > R.Right then
        begin
          L := R.Right;
          R.Right := R.Left;
          R.Left := L;
          FSizeOperation := gmRightTop;
        end;
        SizingControl.BoundsRect := R;
        if FDrawMarkers then PaintSelections;
      end;
      gmRightTop:
      begin
        Inc(R.Top, DY);
        Inc(R.Right, DX);
        if R.Top > R.Bottom then
        begin
          L := R.Bottom;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmLeftBottom;
        end;
        if R.Top > R.Bottom then
        begin
          L := R.Bottom;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmLeftBottom;
        end;
        ATop := SizingControl.Top;
        L := SizingControl.Height;
        SizingControl.BoundsRect := R;
        if SizingControl.Height = L then SizingControl.Top := ATop;
        if FDrawMarkers then PaintSelections;
      end;
    end;
    if FDesignerManager.ObjInspector <> nil then
    FDesignerManager.ObjInspector.RefreshPropertyValues;
  end
  else
  begin
    AParent := FForm;
    if FFrame <> nil then AParent := FFrame;
    if FMoveRectList.Count = 0 then
    begin
      New(PR);
      PR^ := SizingControl.BoundsRect;
      if SizingControl.Parent <> AParent then
        RectToFormRect(PR, SizingControl.Parent);
      FMoveRectList.Add(PR);
    end
    else
    begin
      PaintSelectionRect;
      PR := PRect(FMoveRectList.Items[0]);
    end;

    R := PR^;

    case FSizeOperation of
      gmRight:
      begin
        Inc(R.Right, DX);
        if R.Right < R.Left then
        begin
          L := R.Left;
          R.Left := R.Right;
          R.Right := L;
          FSizeOperation := gmLeft;
        end;
      end;
      gmLeft:
      begin
       Inc(R.Left, DX);
       if R.Left > R.Right then
       begin
          L := R.Right;
          R.Right := R.Left;
          R.Left := L;
          FSizeOperation := gmRight;
        end;
      end;
      gmTop:
      begin
       Inc(R.Top, DY);
       if R.Top > R.Bottom then
       begin
          L := R.Bottom;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmBottom;
        end;
      end;
      gmBottom:
      begin
       Inc(R.Bottom, DY);
       if R.Bottom < R.Top then
       begin
          L := R.Top;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmTop;
        end;
      end;
      gmRightBottom:
      begin
        Inc(R.Bottom, DY);
        Inc(R.Right, DX);
        if R.Bottom < R.Top then
        begin
          L := R.Top;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmLeftTop;
        end;
        if R.Right < R.Left then
        begin
          L := R.Left;
          R.Left := R.Right;
          R.Right := L;
          FSizeOperation := gmLeftTop;
        end;
      end;
      gmLeftTop:
      begin
        Inc(R.Top, DY);
        Inc(R.Left, DX);
        if R.Top > R.Bottom then
        begin
          L := R.Bottom;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmRightBottom;
        end;
        if R.Left > R.Right then
        begin
          L := R.Right;
          R.Right := R.Left;
          R.Left := L;
          FSizeOperation := gmRightBottom;
        end;
      end;
      gmLeftBottom:
      begin
        Inc(R.Bottom, DY);
        Inc(R.Left, DX);
        if R.Bottom < R.Top then
        begin
          L := R.Top;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmRightTop;
        end;
        if R.Left > R.Right then
        begin
          L := R.Right;
          R.Right := R.Left;
          R.Left := L;
          FSizeOperation := gmRightTop;
        end;
        SizingControl.BoundsRect := R;
        if FDrawMarkers then PaintSelections;
      end;
      gmRightTop:
      begin
        Inc(R.Top, DY);
        Inc(R.Right, DX);
        if R.Top > R.Bottom then
        begin
          L := R.Bottom;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmLeftBottom;
        end;
        if R.Top > R.Bottom then
        begin
          L := R.Bottom;
          R.Top := R.Bottom;
          R.Bottom := L;
          FSizeOperation := gmLeftBottom;
        end;
      end;
    end;

    PR^ := R;
    PaintSelectionRect;
  end;

end;

procedure TDesignerHook.MoveSelection(DX, DY: Integer);
var
  I: Integer;
  C: TControl;
  PR: PRect;
  TopLeft: TPoint;
  AParent: TWinControl;
begin
  if not FMovSelection or FDesignerManager.ReadOnly then Exit;


  if (FDesignerManager.MoveObjectMethod = movObject) and not FTempMoveKontur then
  begin
    if FDrawMarkers then InvalidateMarkers(False);
    for I := 0 to DesignerSelections.Count - 1 do
    begin
      C := TControl(DesignerSelections.Items[I]);
      TopLeft := C.BoundsRect.TopLeft;
      Inc(TopLeft.x, DX);
      Inc(TopLeft.y, DY);
      C.Left:= TopLeft.x;
      C.Top := TopLeft.y;
    end;
  end
  else
  begin
    if FMoveRectList.Count = 0 then
    begin
      AParent := FForm;
      if FFrame <> nil then AParent := FFrame;
      if FDrawMarkers then InvalidateMarkers(False);
      for I := 0 to DesignerSelections.Count - 1 do
      begin
        C := TControl(DesignerSelections.Items[I]);
        New(PR);
        PR^:= C.BoundsRect;
        Inc(PR^.Left, DX);
        Inc(PR^.Right, DX);
        Inc(PR^.Top, DY);
        Inc(PR^.Bottom, DY);
        if C.Parent <> AParent then
          RectToFormRect(PR, C.Parent);
        FMoveRectList.Add(PR);
      end;
      PaintSelectionRect;
    end
    else
    begin
      PaintSelectionRect;
      FForm.InvalidateClientRectCache(True);
      for I := 0 to FMoveRectList.Count - 1 do
      begin
        PR := PRect(FMoveRectList.Items[I]);
        Inc(PR^.Left, DX);
        Inc(PR^.Right, DX);
        Inc(PR^.Top, DY);
        Inc(PR^.Bottom, DY);
      end;
      PaintSelectionRect;
    end;
  end;
  Modified;
  if FDesignerManager.ObjInspector <> nil then
    FDesignerManager.ObjInspector.RefreshPropertyValues;
end;

procedure TDesignerHook.MouseDownOnControl(Sender: TControl;
  var TheMessage: TLMMouse);
var
  Button: TMouseButton;
  Control, AControl: TControl;
  SelControl: TControl;
  P, MouseDownPos: TPoint;
  ACursor: TCursor;
  ShiftKeyState: TShiftState;
begin
  if (FSizeOperation <> gmNone) or FMovSelection then  Exit;

  SetCaptureControl(nil);
  FHintTimer.Enabled := False;
  FHintWindow.Visible := False;
  OperationControl := nil;
  FTempMoveKontur := False;

  GetMouseMsgShift(TheMessage, ShiftKeyState, Button);
  MouseDownPos := GetFormRelativeMousePosition(Form);

  if FDesignerManager.PropEditHook.LookupRoot <> Root then
      FDesignerManager.ChangeSelection(Root, DesignerSelections, FForm);


  MouseDownShift := ShiftKeyState;
  if (FDesignerManager.AddComponentClass <> nil) or (FSizeOperation <> gmNone)
    or (ssShift in MouseDownShift) then Exclude(MouseDownShift, ssDouble);
  MouseDownComponent := nil;

  if (Button = mbLeft) or (Button = mbRight) then
  begin
    if Button = mbLeft then SetCaptureControl(FForm);;
    AControl := Form;
    if FFrame <> nil then  AControl := FFrame;
    Control := ControlAtPos(AControl, MouseDownPos);

    if Control is TComponentCaption then Control := TControl(Control.Owner);

    if csNoDesignSelectable in Control.ControlStyle then
      Control := Control.Parent;

    MouseDownComponent := Control;
    if Control is TNonVisual then
      MouseDownComponent := TNonVisual(Control).Component;

    if (FDesignerManager.AddComponentClass <> nil) and (Root is TDataModule) then
    begin
      if FDesignerManager.AddComponentClass.InheritsFrom(TControl) then
      begin
        FDesignerManager.AddComponentClass := nil;
        if Assigned(FDesignerManager.OnAfterAddComponent) then
          FDesignerManager.OnAfterAddComponent(nil, Root);
      end;
    end;

    if FDesignerManager.AddComponentClass <> nil then Exit;
    if FDesignerManager.AddFrame <> '' then Exit;

    FSizeOperation := gmNone;
    if (DesignerSelections.Count = 1) and (DesignerSelections.Items[0] <> FForm)
        and (DesignerSelections.Items[0] is TControl) then
    begin
      SelControl := TControl(DesignerSelections.Items[0]);
      if not (SelControl is TNonVisual) then
      begin
        P.x := MouseDownPos.X;
        P.y := MouseDownPos.Y;
        FSizeOperation := GetGrabberMarker(SelControl, P);
        ACursor := crDefault;
        case FSizeOperation of
          gmLeftTop: ACursor := crSizeNW;
          gmLeft, gmRight: ACursor := crSizeWE;
          gmLeftBottom: ACursor := crSizeNESW;
          gmTop, gmBottom: ACursor := crSizeNS;
          gmRightTop: ACursor := crSizeNE;
          gmRightBottom: ACursor:= crSizeSE;
        end;
        if LastFormCursor <> ACursor then
        begin
          SetTempCursor(FForm, ACursor);
        end;
      end;
    end;

    if (FSizeOperation <> gmNone) and (Button = mbLeft) then
    begin
      OperationControl := SelControl;
      SizingControl := SelControl;
      FMovePos := P;
      if OperationControl is TCustomLabel then FTempMoveKontur:= True;
      if (FDesignerManager.MoveObjectMethod = movKontur) or FTempMoveKontur then
      begin
        InvalidateMarkers(False);
        Application.ProcessMessages;
        CreateMoveRect;
      end;
    end
    else
    begin
      if FFrame <> nil then
        AControl := FFrame
      else AControl := Form;

      if not (Control is TNonVisual) and (ssCtrl in ShiftKeyState) then
      begin
        if Control is TWinControl then
          CaptControl := TWinControl(Control)
        else
          CaptControl := Control.Parent;
        CaptureRect.Left := MouseDownPos.X;
        CaptureRect.Top := MouseDownPos.Y;
        CaptureRect.Right:= CaptureRect.Left;
        CaptureRect.Bottom:= CaptureRect.Top;
        FMovePos.x := MouseDownPos.X;
        FMovePos.y := MouseDownPos.Y;
        FCR := True;
      end
      else
      if DesignerSelections.ControlInSelection(Control)
          and (Control <> AControl) then
      begin
        if (Button = mbLeft) and not (ssCtrl in ShiftKeyState)
            and not (ssShift in ShiftKeyState) and not (ssDouble in ShiftKeyState) then
        begin
          OperationControl := Control;
          FMovSelection := True;
          FMovePos.x := MouseDownPos.X;
          FMovePos.y := MouseDownPos.Y;
          if OperationControl is TToolButton then FTempMoveKontur := True;
          if (FDesignerManager.MoveObjectMethod = movKontur) or FTempMoveKontur then
          begin
            InvalidateMarkers(False);
            Application.ProcessMessages;
            CreateMoveRect;
          end;
        end;
      end
      else
      begin
        if DesignerSelections.Count > 0 then SelControl := TControl(DesignerSelections.Items[0]);

        if not (ssShift in ShiftKeyState) or (Control = FForm)// or (Button = mbRight)
           or ((DesignerSelections.Count = 1) and (SelControl = FForm))
           or (SelControl.Parent <> Control.Parent)  then
             ClearSelection;

        if (DesignerSelections.Count = 1) then InvalidateMarkers(True);

        AddSelection(Control);

        if (DesignerSelections.Count = 1) and not (FDesignerManager.FirstSelectThenDrag) then
          if (Button = mbLeft) and not (ssShift in ShiftKeyState)
             and not (ssCtrl in ShiftKeyState)  and not (ssDouble in ShiftKeyState)  then
          begin
            if Control <> AControl then
            begin
                FMovSelection := True;
                OperationControl := Control;
                FMovePos.x := MouseDownPos.X;
                FMovePos.y := MouseDownPos.Y;
                if OperationControl is TToolButton then FTempMoveKontur := True;
                if (FDesignerManager.MoveObjectMethod = movKontur) or FTempMoveKontur then
                begin
                  InvalidateMarkers(False);
                  Application.ProcessMessages;
                  CreateMoveRect;
                end;
            end;
          end
      end;
    end;
  end;
  TheMessage.Result := 1;
end;

function TDesignerHook.MousePosToGridPos(P: TPoint): TPoint;
begin
  Result.X := Round(P.X / FDesignerManager.GridStep) * FDesignerManager.GridStep;
  Result.Y := Round(P.Y / FDesignerManager.GridStep) * FDesignerManager.GridStep;
end;

function TDesignerHook.ClearSelection: boolean;
begin
  Result := False;
  if FDesignerManager.ControlInSelection then InvalidateMarkers(True);
  FDesignerManager.ControlInSelection := False;
  DesignerSelections.Clear;
  Result := True;
end;

procedure TDesignerHook.GetSelectionFromCaptureRect;
var
  R, CR: TRect;
  P: TPoint;
  I: Integer;
  B: Boolean;
  List: TList;
begin
  P := CaptureRect.TopLeft;
  P := Form.ClientToScreen(P);
  P := CaptControl.ScreenToClient(P);
  if P.x < 0 then P.x := 0;
  if P.y < 0 then P.y := 0;
  R.TopLeft := P;

  P := CaptureRect.BottomRight;
  P := Form.ClientToScreen(P);
  P := CaptControl.ScreenToClient(P);
  R.BottomRight := P;

  List := TList.Create;
  try
    for I := 0 to CaptControl.ControlCount - 1 do
    begin
      CR := CaptControl.Controls[I].BoundsRect;
      B := False;
      if PtInRect(R, CR.TopLeft) then B := True
      else
      if PtInRect(R, CR.BottomRight) then B := True
      else
      begin
        P.x := CR.Left;
        P.y := CR.Bottom;
        if PtInRect(R, P) then B := True
        else
        begin
          P.x := CR.Right;
          P.y := CR.Top;
          if PtInRect(R, P) then B := True
        end;
      end;
      if B then List.Add(CaptControl.Controls[I]);
    end;

    if List.Count > 0 then
    begin
      ClearSelection;
      Application.ProcessMessages;
    end;

    for I := 0 to List.Count - 1 do DesignerSelections.Add(TControl(List.Items[I]));
    Application.ProcessMessages;
    PaintSelections;
    FDesignerManager.ChangeSelection(Root, DesignerSelections, FForm);
  finally
     List.Free;
  end;
end;

function TDesignerHook.GetNonVisualOfObject(AObject: TPersistent): TNonVisual;
var
  I: Integer;
begin
  Result := nil;
  for I := 0 to FForm.ControlCount - 1 do
  begin
    if FForm.Controls[I] is TNonVisual then
    begin
      if TNonVisual(FForm.Controls[I]).Component = AObject then
      begin
        Result := TNonVisual(FForm.Controls[I]);
        Break;
      end;
    end;
  end;
end;

procedure TDesignerHook.AddComponent(Sender: TControl; X, Y: Integer);
var
  AComponent, OriginalComponent: TComponent;
  AFrame: TFrame;
  AControl, AParentControl: TControl;
  AParent: TWinControl;
  P, MouseDownPos: TPoint;
  NV: TNonVisual;
  S: string;
begin
  if FDesignerManager.ReadOnly then Exit;

  MouseDownPos := GetFormRelativeMousePosition(Form);

  AFrame := nil;
  if FDesignerManager.AddFrame <> '' then
  begin
    S := FDesignerManager.AddFrame;
    FDesignerManager.TextToFrame(S, AFrame);
    AComponent := AFrame;
  end
  else
    AComponent := FDesignerManager.AddComponentClass.Create(Root);

   AComponent.FreeNotification(Form);
   //AComponent.FreeNotification(Root);

  OriginalComponent := AComponent;
  AComponent.Name := UniqueName(AComponent.ClassName);

  FDesignerManager.AddComponentClass := nil;
  FDesignerManager.AddFrame := '';
  if not (AComponent is TControl) then
  begin
    NV := TNonVisual.Create(AComponent, FDesignerManager);
    NV.Component := AComponent;
    AComponent := NV;
  end;

  if AComponent is TControl then
  begin
    AControl := TControl(AComponent);
    P.x := X;
    P.y := Y;
    if AComponent is TNonVisual then
    begin
      AParent := FForm;
      if FFrame <> nil then AParent := FFrame;
    end
    else
    begin
      AParent := Form;
      if FFrame <> nil then AParent := FFrame;
      AParentControl := ControlAtPos(AParent, MouseDownPos);
      if (AParentControl is TWinControl) and (csAcceptsControls in AParentControl.ControlStyle)  then
        AParent := TWinControl(AParentControl)
      else
        AParent := AParentControl.Parent;
    end;

    if Sender <> AParent then
    begin
      P := Sender.ClientToScreen(P);
      P := AParent.ScreenToClient(P);
    end;

    P := MousePosToGridPos(P);
    AControl.Parent := AParent;
    AControl.Left := P.X;
    AControl.Top := P.Y;
    ClearSelection;
    FDrawMarkers := True;
    if AControl is TWinControl then UpNonVisual;

    AfterAddComponent(OriginalComponent);
    AddSelection(AControl);
  end;

end;

procedure TDesignerHook.AfterAddComponent(AComponent: TComponent);
begin
  FDesignerManager.AfterAddComponent(AComponent, Root);
  Modified;
end;

procedure TDesignerHook.MouseUpOnControl(Sender : TControl; var TheMessage:TLMMouse);
var
  Button: TMouseButton;
  Acomponent: TComponent;
  P: TPoint;
  Shift: TShiftState;
begin
  SetCaptureControl(nil);
  GetMouseMsgShift(TheMessage, Shift, Button);
  if  (FMovSelection or (FSizeOperation <> gmNone)) and (FMoveRectList.Count > 0) then
  begin
    PaintSelectionRect;
    MoveSelectionToNewPos;
  end;
  ClearMoveRect;
  FTempMoveKontur:=False;
  OperationControl := nil;
  FHintWIndow.Visible := False;
  FMovSelection := False;
  FSizeOperation := gmNone;
  FDrawMarkers := True;
  //if FDesignerManager.ReadOnly then Exit;

  if FCR then
  begin
    DrawCaptureRect;
    FCR := False;
    GetSelectionFromCaptureRect;
  end
  else
  if Button = mbLeft then
  begin
    if (ssDouble in MouseDownShift) then
    begin
      if (DesignerSelections.Count > 0) then
      begin
        Acomponent := MouseDownComponent;
        MouseDownComponent := nil;
        if Acomponent <> nil then
        begin
          PaintSelections;
          FHintWIndow.Visible:= False;
          FHintTimer.Enabled := False;
          DoDoubleClickOnComponent(Acomponent);
        end;
      end;
    end
    else
    if FDesignerManager.AddComponentClass <> nil then
    begin
      AddComponent(Sender, TheMessage.XPos, TheMessage.YPos);
    end
    else
    if FDesignerManager.AddFrame <> '' then
    begin
      AddComponent(Sender, TheMessage.XPos, TheMessage.YPos);
    end
    else
    begin
      //FForm.Invalidate;
      PaintSelections;
    end;
    TheMessage.Result:=1;
  end
  else
  begin
    PaintSelections;

  end;

end;

procedure TDesignerHook.UpNonVisual;
var
  I, DI: Integer;
  N: TNonVisual;
begin
  I := 0;
  while I < FForm.ControlCount do
  begin
    if FForm.Controls[I] is TNonVisual then
    begin
      N := TNonVisual(FForm.Controls[I]);
      N.TabOrder := FForm.ComponentCount - 1;
      DI := N.FComponent.DesignInfo;
      N.Left := LongRec(DI).Lo;
      N.Top := LongRec(DI).Hi;
      N.BringToFront;
      //SetWindowPos(N.Handle, HWND_TOP, N.Left, N.Top, N.Width, N.Height, SWP_SHOWWINDOW or SWP_NOSIZE);
    end;
    Inc(I);
  end;
end;

procedure TDesignerHook.RectToFormRect(PR: PRect; WControl: TWinControl);
var
  P1, P2: TPoint;
begin
  if WControl = nil then Exit;
  P1 := PR^.TopLeft;
  P1 := WControl.ClientToScreen(P1);
  P1 := FForm.ScreenToClient(P1);

  P2 := PR^.BottomRight;
  P2 := WControl.ClientToScreen(P2);
  P2 := FForm.ScreenToClient(P2);

  PR^.TopLeft := P1;
  PR^.BottomRight := P2;
end;

function TDesignerHook.GetRoot: TComponent;
begin
  Result := FForm;
  if FDM <> nil then Result := FDM
  else
  if FFrame <> nil then Result := FFrame;
end;

procedure TDesignerHook.SetRoot(AComponent: TComponent);
begin
  if AComponent is TDataModule then
  begin
    FDM := TDataModule(AComponent);
    SetComponentDesignMode(AComponent, True);
    FLookupRoot := FDM;
  end
  else
  if AComponent is TFrame then
  begin
    FFrame := TFrame(AComponent);
    SetComponentDesignMode(AComponent, True);
    FLookupRoot := FFrame;
  end
  else
  if AComponent is TCustomForm then
  begin
    FLookupRoot := FForm;
    FForm := TCustomForm(AComponent);
  end
  else
    FLookupRoot := nil;
end;

function TDesignerHook.OnFormActivate(Sender: TControl;
   var TheMessage: TLMActivate): Boolean;
begin
  Result := True;
  if Sender = FForm then
  begin

    if TheMessage.Active = WA_INACTIVE then
      FHintWIndow.Visible := False
    else
    begin
      if Assigned(FDesignerManager.OnFormActivate) then
        FDesignerManager.OnFormActivate(Root, Form);

      if FDesignerManager.PropEditHook.LookupRoot <> Root then
        FDesignerManager.ChangeSelection(Root, DesignerSelections, FForm);
    end
  end;
end;

function TDesignerHook.DoCopySelectionToClipboard: boolean;
var
  AllComponentsStream: TMemoryStream;
  AllComponentText: string;
begin
  Result := false;
  if DesignerSelections.Count = 0 then exit;
  if DesignerSelections.Items[0] = Root then exit;
  //if DesignerSelections.OnlyInvisiblePersistentsSelected then exit;

  AllComponentsStream:=TMemoryStream.Create;
  try
    // copy components to stream
    if not CopySelectionToStream(AllComponentsStream) then exit;
    SetLength(AllComponentText,AllComponentsStream.Size);
    if AllComponentText<>'' then
    begin
      AllComponentsStream.Position:=0;
      AllComponentsStream.Read(AllComponentText[1],length(AllComponentText));
    end;

    // copy to clipboard
    try
      ClipBoard.AsText:=AllComponentText;
    except
      on E: Exception do begin
        MessageDlg(lisUnableCopyComponentsToClipboard,
          Format(lisThereWasAnErrorWhileCopyingTheComponentStreamToCli, [#13,
            E.Message]),
          mtError,[mbOk],0);
        exit;
      end;
    end;
  finally
    AllComponentsStream.Free;
  end;
  Result:=true;
end;

{TDesignerManager}

constructor TDesignerManager.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  AddComponentClass := nil;
  FDesignObjects := TList.Create;
  FGridStep := 8;
  FActive := True;
  FDesignPopupMenu := TDesignPopupMenu.Create(Self);
  FNewDesignPosLeft := 300;
  FNewDesignPosTop := 150;
  FNewDesignWidth := 600;
  FNewDesignHeight := 500;
  FShowHints := True;
  FShowGrid := True;
  FMoveObjectMethod:= movObject;
  FControlKeyMove := ssCtrl;
  FControlKeyResize := ssShift;
  FFirstSelectThenDrag := False;
  ControlInSelection := False;
  if not (csDesigning in componentState) then
  begin
    GlobalDesignHook:=TPropertyEditorHook.Create(nil);
    PropEditHook.AddHandlerPersistentAdded(@PersistentAdded);
    PropEditHook.AddHandlerPersistentDeleting(@PersistentDeleting);
  end;
  if not (csDesigning in ComponentState) then
    OnGetDesignerForm := @GetDesignForm;
end;

destructor TDesignerManager.Destroy;
begin
  FActive := False;
  Clear;
  FDesignObjects.Free;
  FDesignPopupMenu.Free;
  if not (csDesigning in ComponentState) then
  begin
    PropEditHook.RemoveHandlerPersistentAdded(@PersistentAdded);
    PropEditHook.RemoveHandlerPersistentDeleting(@PersistentDeleting);
    GlobalDesignHook.Free;
    GlobalDesignHook := nil;
  end;
  if not (csDesigning in ComponentState) then
    OnGetDesignerForm := nil;
  inherited Destroy;
end;

function TDesignerManager.ObjNameExist(AName: string): Boolean;
var
  I: Integer;
begin
  Result := False;
  for I := 0 to DesignCount - 1 do
  begin
    if SameText(ItemsRoot[I].Name, AName) then
    begin
      Result := True;
      Break;
    end;
  end;
  if not Result then
  begin
    if Assigned(FIsObjNameExist) then Result := FIsObjNameExist(AName);
  end;
end;

procedure TDesignerManager.SetComponentName(AComponent: TComponent; BaseClassName: string);
var
  AName, BaseName: string;
  I: Integer;
begin
  BaseName := ClassNameToComponentName(BaseClassName);
  for I := 1 to 1000 do
  begin
    AName := BaseName + IntToStr(I);
    if not ObjNameExist(AName) then
    begin
      AComponent.Name:= AName;
      Break;
    end;
  end;
end;

function TDesignerManager.GetDesignForm(APersistent: TPersistent): TCustomForm;
var
  Control: TComponent;
  AOwner: TComponent;
begin
  Result := nil;
  if APersistent is TComponent then
  begin
    if APersistent is TNonVisual then
      Control := TNonVisual(APersistent).Component
    else
      Control := TComponent(APersistent);
    AOwner := Control;
    if Control.Owner <> nil then
      AOwner := Control.Owner;

    Result := GetDesignFormFromRoot(AOwner);
  end;
end;

procedure TDesignerManager.PaintCurentSelection;
begin
  if CurrentDesignForm <> nil then
  begin
    TDesignerHook(CurrentDesignForm.Designer).PaintSelections;
  end;
end;

procedure TDesignerManager.ClearMarkers;
begin
  if CurrentDesignForm <> nil then
  begin
    TDesignerHook(CurrentDesignForm.Designer).InvalidateMarkers(True);
  end;
end;

procedure TDesignerManager.AfterAddComponent(AComponent, ARoot: TComponent);
begin
  //UnComment in Lazarus 2.2
  if FObjInspector <> nil then FObjInspector.FillComponentList(True);
  //Comment next row in Lazarus 2.2
  //if FObjInspector <> nil then FObjInspector.FillComponentList;
  if Assigned(FOnAfterAddComponent) then FOnAfterAddComponent(AComponent, ARoot);
  if CurrentDesignForm <> nil then
  begin
    TDesignerHook(CurrentDesignForm.Designer).Modified;
  end;
end;

procedure TDesignerManager.InitialDesignForm(AComponent: TComponent; AForm: TCustomForm);
var
  Aleft, ATop: Smallint;
  W, H, I: Integer;
begin
  if AComponent is TFrame then
  begin
    W := TFrame(AComponent).Width;
    H := TFrame(AComponent).Height;
    AForm.ClientWidth :=  W;
    AForm.ClientHeight := H;
    TFrame(AComponent).Parent := AForm;
    TDesignerHook(AForm.Designer).UpNonVisual;
    for I := 0 to AComponent.ComponentCount - 1 do
    begin
      if not (AComponent.Components[I] is TNonVisual) then
        AComponent.Components[I].FreeNotification(AForm);
    end;
  end
  else
  if AComponent is TDataModule then
  begin
    AForm.Caption:= AComponent.Name;
    AForm.Color := clWhite;
    AForm.Width := TDataModule(AComponent).DesignSize.x;
    AForm.Height := TDataModule(AComponent).DesignSize.y;
    DesignInfoToLeftTop(TDataModule(AComponent).DesignInfo, Aleft, ATop);
    AForm.Left := Aleft;
    AForm.Top := ATop;
    for I := 0 to AComponent.ComponentCount - 1 do
    begin
      if not (AComponent.Components[I] is TNonVisual) then
        AComponent.Components[I].FreeNotification(AForm);
    end;
  end;
end;

function TDesignerManager.GetPropertyEditorHook: TPropertyEditorHook;
begin
  Result := GlobalDesignHook;
end;

procedure TDesignerManager.DesignPopupMenu(Pos: TPoint;
  ASelections: TDesignerSelections);
var
  BaseCompEditor: TBaseComponentEditor;
  CompEditorDesigner: TComponentEditorDesigner;
  AComponent: TComponent;
begin
  if (ASelections = nil) or ReadOnly then Exit;
  AComponent := TComponent(ASelections.Items[0]);
  BaseCompEditor := nil;
  if AComponent is TNonVisual then AComponent := TNonVisual(AComponent).Component;
  if ASelections.Count = 1 then
  begin
    CompEditorDesigner := TComponentEditorDesigner(CurrentDesignForm.Designer);
    BaseCompEditor := GetComponentEditor(AComponent, CompEditorDesigner);
  end;

  FDesignPopupMenu.SetEditor(AComponent, BaseCompEditor);
  FDesignPopupMenu.PopUp;
end;

procedure TDesignerManager.PersistentAdded(APersistent: TPersistent;
  Select: boolean);
var
  AComponent: TComponent;
  Editor: TBaseComponentEditor;
begin
  if APersistent is TComponent then
  begin
    AComponent := TComponent(APersistent);
    if AComponent.Owner = GlobalDesignHook.LookupRoot then
    begin
      AComponent.FreeNotification(CurrentDesignForm);
      AfterAddComponent(AComponent, AComponent.Owner);
      if Select then
        SelectOnlyThisComponent(AComponent);
    end;
  end;
end;

procedure TDesignerManager.PersistentDeleting(APersistent: TPersistent);
var
  AComponent: TComponent;
begin
  if APersistent is TComponent then
  begin
    AComponent := TComponent(APersistent);
    if AComponent.Owner = GlobalDesignHook.LookupRoot then
      DeleteComponent(AComponent, AComponent.Owner);
  end;
end;

function TDesignerManager.CreateDesignerForComponent(AComponent: TComponent;
   SetUpNonVisual: Boolean = False; AJITMethods: TJITMethods = nil): TCustomForm;
var
  D: TDesignerHook;
begin
  if AComponent is TCustomForm then
    Result := TCustomForm(AComponent)
  else
  begin
    Result := NewForm;
    Result.Caption:= AComponent.Name;
  end;

  if Result.Designer <> nil then
    D := TDesignerHook(Result.Designer)
  else
    D := TDesignerHook.Create(Self, AJITMethods);

  Result.Designer := D;
  D.Form := Result;
  D.Root := AComponent;
  FDesignObjects.Add(D);

  {if SetUpNonVisual then
  begin
    D.SetNonVisual;
    D.AddSelection(Result);
  end;}
end;

function TDesignerManager.NewForm: TCustomForm;
begin
  Result := TForm.CreateNew(nil);
  Result.Visible := False;
  Result.DoubleBuffered := True;
  Result.Left := FNewDesignPosLeft;
  Result.Top:= FNewDesignPosTop;
  Result.Width := FNewDesignWidth;
  Result.Height := FNewDesignHeight;
end;

function TDesignerManager.NewDesForm: TCustomForm;
begin
  Result := NewForm;
  Result.DoubleBuffered := True;
  SetComponentName(Result, 'TForm');
  CreateDesignerForComponent(Result, False);
end;

procedure TDesignerManager.ShowForm(Form: TCustomForm);
var
  V: Boolean;
begin
  if Form <> nil then
  begin
    {$IFDEF WINDOWS}
    SetWindowPos(Form.Handle, HWND_TOP, Form.Left, Form.Top, Form.Width, Form.Height, SWP_SHOWWINDOW);
    {$ELSE}
    V := Form.Visible;
    Form.Visible := True;
    Form.Show;
    Form.Visible := V;
    {$ENDIF}
  end;
end;

procedure TDesignerManager.NewDataModule(var DM: TDataModule; var F: TCustomForm);
begin
  DM := TDataModule.CreateNew(nil);
  SetComponentName(DM, 'TDataModule');
  F := CreateDesignerForComponent(DM, False);
  DM.DesignInfo := LeftTopToDesignInfo(F.Left, F.Top);
  DM.DesignSize := Point(F.Width, F.Height);
  F.Caption := DM.Name;
  F.Color := clWhite;
end;

procedure TDesignerManager.NewFrame(var Frame: TFrame; var F: TCustomForm);
begin
  Frame := TSuFrame.Create(nil);
  Frame.Width := 500;
  Frame.Height := 400;
  SetComponentName(Frame, 'TFrame');
  F := CreateDesignerForComponent(Frame, False);
  Frame.Parent := F;
  Frame.Width := F.ClientWidth;
  Frame.Height := F.ClientHeight;
  F.Caption := Frame.Name;
  //InitialDesignForm(Frame, F);
end;

function TDesignerManager.GetDesignObjectByName(AName: string): TComponent;
var
  I: Integer;
begin
  Result := nil;
  I := 0;
  while I < DesignCount do
  begin
    if SameText(ItemsRoot[I].Name, AName) then
    begin
      Result := ItemsRoot[I];
      Break;
    end;
    Inc(I);
  end;
end;

function TDesignerManager.GetDesignFormFromRoot(ARoot: TComponent): TCustomForm;
var
  I: Integer;
begin
  Result := nil;
  I := 0;
  while I < DesignCount do
  begin
    if ItemsRoot[I] = ARoot then
    begin
      Result := ItemsDesignForm[I];
      Break;
    end;
    Inc(I);
  end;
end;

procedure TDesignerManager.InvalidateAll;
var
  I: Integer;
begin
  for I := 0 to DesignCount - 1 do
  begin
    ItemsDesignForm[I].Invalidate;
  end;
end;

procedure TDesignerManager.SetObjInspector(AValue: TObjectInspectorDlg);
begin
  if FObjInspector <> AValue then
  begin
    FObjInspector := AValue;
    FObjInspector.PropertyEditorHook := PropEditHook;
  end;
end;

procedure TDesignerManager.ChangeSelection(Root: TComponent;
     Selections: TDesignerSelections; DesForm: TCustomForm);
var
  PSList: TPersistentSelectionList;
  I: Integer;
begin
  if FObjInspector = nil then Exit;
  if Root = nil then
  begin
    ControlInSelection := False;
    PropEditHook.LookupRoot := nil;
    ObjInspector.Selection.Clear;
    ObjInspector.RefreshSelection;
    CurrentDesignForm := nil;
    CurrentDesignRoot := nil;
    Exit;
  end;

  CurrentDesignForm := DesForm;
  CurrentDesignRoot := Root;
  PSList := TPersistentSelectionList.Create;
  try

    for I := 0 to Selections.Count - 1 do
    begin
      if Selections.Items[I] is TControl then ControlInSelection := True;

      if Selections.Items[I] = DesForm then
        PSList.Add(Root)
      else
      if Selections.Items[I] is TNonVisual then
          PSList.Add(TNonVisual(Selections.Items[I]).Component)
      else
          PSList.Add(Selections.Items[I]);
    end;
    PropEditHook.LookupRoot := Root;
    ObjInspector.Selection := PSList;
    //PropEditHook.SetSelection(PSList);

    //FObjInspector.ComponentTree.RebuildComponentNodes;
  finally
      PSList.Free;
  end;
end;

procedure TDesignerManager.OnExtChangeSelection(Selection: TPersistentSelectionList);
var
  DH: TDesignerHook;
  I: Integer;
  C: TComponent;
  NV: TNonVisual;
  PSList: TPersistentSelectionList;
  B: Boolean;
begin
  if (FObjInspector = nil) or (CurrentDesignForm = nil) then Exit;

  DH := TDesignerHook(CurrentDesignForm.Designer);



  PSList := TPersistentSelectionList.Create;
  try

    for I := 0 to DH.DesignerSelections.Count - 1 do
    begin
      if DH.DesignerSelections.Items[I] = CurrentDesignForm then
        PSList.Add(CurrentDesignRoot)
      else
      if DH.DesignerSelections.Items[I] is TNonVisual then
          PSList.Add(TNonVisual(DH.DesignerSelections.Items[I]).Component)
        else
          PSList.Add(DH.DesignerSelections.Items[I]);
    end;
    B := Selection.IsEqual(PSList);
  finally
      PSList.Free;
  end;

  if not B then
  begin
    DH.ClearSelection;
    for I := 0 to Selection.Count - 1 do
    begin
      C := nil;
      if Selection.Items[I] = CurrentDesignRoot then
        C := CurrentDesignForm
      else
      if not (Selection.Items[I] is TControl) then
      begin
        NV := DH.GetNonVisualOfObject(Selection.Items[I]);
        if NV <> nil then C := NV;
      end
      else
        C := TComponent(Selection.Items[I]);
      if C <> nil then  DH.DesignerSelections.Add(C);
    end;

    if (DH.DesignerSelections.Count > 0) or (Selection.Count = 0) then
    begin
      DH.PaintSelections;
      ChangeSelection(DH.Root, DH.DesignerSelections, DH.Form);
    end;
  end;
end;

procedure TDesignerManager.SelectOnlyThisComponent(ACompoenent: TComponent);
var
  DH: TDesignerHook;
begin
  if CurrentDesignForm <> nil then
  begin
    DH := TDesignerHook(CurrentDesignForm.Designer);
    DH.SelectOnlyThisComponent(ACompoenent);
    GlobalDesignHook.SelectOnlyThis(ACompoenent);
  end;
end;

function TDesignerManager.GetReadOnly: Boolean;
begin
  Result := False;
  if Assigned(FOnGetReadOnly) then
    Result := FOnGetReadOnly(Self);
end;

procedure TDesignerManager.SetControlKeyMove(AValue: TShiftStateEnum);
begin
  if FControlKeyMove=AValue then Exit;
  if AValue in [ssCtrl, ssAlt, ssShift] then
  begin
    FControlKeyMove:=AValue;
    if FControlKeyMove = FControlKeyResize then
    begin
      if FControlKeyMove = ssCtrl then
        FControlKeyResize:= ssAlt
      else
        FControlKeyResize:= ssCtrl;
    end;
  end;
end;

procedure TDesignerManager.SetControlKeyResize(AValue: TShiftStateEnum);
begin
  if FControlKeyResize=AValue then Exit;
  if AValue in [ssCtrl, ssAlt, ssShift] then
  begin
    FControlKeyResize:=AValue;
    if FControlKeyResize = FControlKeyMove then
    begin
      if FControlKeyResize = ssShift then
        FControlKeyMove:= ssCtrl
      else
        FControlKeyMove:= ssShift;
    end;
  end;
end;

procedure TDesignerManager.SetShowGrid(AValue: Boolean);
begin
  if FShowGrid <> AValue then
  begin
    FShowGrid:=AValue;
    InvalidateAll;
  end;
end;

procedure TDesignerManager.SetGridStep(Value: Integer);
begin
  if (FGridStep <> Value) and (Value > 0) then
  begin
    FGridStep := Value;
    InvalidateAll;
  end;
end;

{procedure TDesignerManager.Notification(AComponent: TComponent; Operation: TOperation);
begin
  //inherited Notification(AComponent, Operation);
  if Operation = opRemove then
  begin
    if AComponent.Owner <> nil then
    begin
      if AComponent.Owner = CurrentDesignRoot then
      begin
        if Assigned(OnDeleteComponent) then
           OnDeleteComponent(AComponent, AComponent.Owner);
      end;
    end;
  end;
end;}

procedure TDesignerManager.WriteMethodProp(Writer: TWriter; Instance: TPersistent;
      PropInfo: PPropInfo; const MethodValue, DefMethodValue: TMethod;
      var Handled: boolean);
var
  JMethod: TJITMethod;
  PrpName: string;
begin
  Handled := False;
  if MethodValue.Data <> nil then
  begin
    JMethod := TJITMethod(MethodValue.Data);
    PrpName:= PropInfo^.Name;
    if Instance is TComponent then
    begin
      if csSubComponent in TComponent(Instance).ComponentStyle then
          PrpName := TComponent(Instance).Name + '.' + PrpName;
    end;
    Writer.Driver.BeginProperty(PrpName);
    Writer.Driver.WriteMethodName(JMethod.FTheMethodName);
    Writer.Driver.EndProperty;
    Handled := True;
  end;
end;

function TDesignerManager.ObjToText(Instance: Tcomponent): string;
var
  StrStream: TStringStream;
  BinStream: TMemoryStream;
  Writer: TWriter;
  S: string;
begin
  BinStream := TMemoryStream.Create;
  try
    StrStream := TStringStream.Create(s);
    try
      Writer := TWriter.Create(BinStream, 4096);
      Writer.OnWriteMethodProperty:= @WriteMethodProp;
      try
        Writer.WriteDescendent(Instance, nil);
      finally
        Writer.Free;
      end;
      BinStream.Seek(0, soFromBeginning);
      ObjectBinaryToText(BinStream, StrStream, oteLFM);
      StrStream.Seek(0, soFromBeginning);
      Result:= StrStream.DataString;
    finally
      StrStream.Free;
    end;
  finally
    BinStream.Free;
  end;
end;

procedure TDesignerManager.ReaderSetMethodProp(Reader: TReader;
  Instance: TPersistent; PropInfo: PPropInfo; const TheMethodName: string;
  var Handled: boolean);
var
  JMethod: TJITMethod;
  ATypeData: PTypeData;
begin
  Handled := True;
  ATypeData := GetTypeData(PropInfo^.PropType);
  JMethod := LoadHandlers.Add(TheMethodName, ATypeData);
  SetMethodProp(Instance, PropInfo, JMethod.FMethod);
end;

function TDesignerManager.GetDesignObjectsCount: Integer;
begin
  Result := FDesignObjects.Count;
end;

function TDesignerManager.GetItems(Index: Integer): TDesignerHook;
begin
  Result := TDesignerHook(FDesignObjects.Items[Index]);
end;

function TDesignerManager.GetItemsRoot(Index: Integer): TComponent;
begin
  Result := Items[Index].Root;
end;

function TDesignerManager.GetItemsDesignForm(Index: Integer): TCustomForm;
begin
  Result := Items[Index].Form;
end;

procedure TDesignerManager.SetActive(AValue: Boolean);
begin
  if FActive <> AValue then
  begin
    FActive:= AValue;
    if not FActive then
    begin
      if ObjInspector <> nil then
      begin
        ObjInspector.Selection := nil;
      end;
      PropEditHook.LookupRoot := nil;
    end;
  end;
end;

procedure TDesignerManager.DeleteAndFree(Index: Integer);
var
  DH: TDesignerHook;
begin
  DH := Items[Index];
  if PropEditHook.LookupRoot = DH.Root then
  begin
    ChangeSelection(nil, DH.DesignerSelections, nil);
  end;
  if DH.Root <> DH.Form then
  begin
    DH.Root.Free;
    DH.Form.Free;
  end
  else
     DH.Root.Free;
  DH.Free;
  FDesignObjects.Delete(Index);
end;

procedure TDesignerManager.DeleteComponent(AComponent, ARoot: TComponent);
begin
  if Assigned(FOnDeleteComponent) then
    FOnDeleteComponent(AComponent, ARoot);
  //Uncomment in Lazrus 2.2
  if FObjInspector <> nil then FObjInspector.FillComponentList(False);
  //Commment next row in Lazaurs 2.2
  //if FObjInspector <> nil then FObjInspector.FillComponentList;
  if CurrentDesignForm <> nil then TDesignerHook(CurrentDesignForm.Designer).Modified;
end;

procedure TDesignerManager.DeleteAndFreeByRoot(ARoot: TComponent);
var
  I: Integer;
begin
  I := 0;
  while I < DesignCount do
  begin
    if ItemsRoot[I] = ARoot then
    begin
      DeleteAndFree(I);
      Break;
    end;
    Inc(I);
  end;
end;

procedure TDesignerManager.Clear;
var
  SaveActive: Boolean;
begin
  //PropEditHook.LookupRoot := nil;
  SaveActive := Active;
  Active := False;
  while DesignCount > 0 do DeleteAndFree(0);
  Active := SaveActive;
end;

function TDesignerManager.DesignSelectable(AComponent: TComponent): Boolean;
var
  DH: TDesignerHook;
begin
  Result := False;
  if CurrentDesignForm <> nil then
  begin
    DH := TDesignerHook(CurrentDesignForm.Designer);
    Result := DH.DesignSelectable(AComponent);
  end;
end;

procedure TDesignerManager.UpNonVisual;
var
  DH: TDesignerHook;
begin
  if CurrentDesignForm <> nil then
  begin
    DH := TDesignerHook(CurrentDesignForm.Designer);
    DH.UpNonVisual;
  end;
end;

procedure TDesignerManager.BringToFront;
var
  D: TDesignerHook;
begin
  if CurrentDesignForm = nil then Exit;
  D := TDesignerHook(CurrentDesignForm.Designer);
  D.BringToFront;
end;

procedure TDesignerManager.SendToBack;
var
  D: TDesignerHook;
begin
  if CurrentDesignForm = nil then Exit;
  D := TDesignerHook(CurrentDesignForm.Designer);
  D.SendToBack;
end;

procedure TDesignerManager.ReaderReadStringProp(Sender:TObject;
        const Instance: TPersistent; PropInfo: PPropInfo;
        var Content:string);
var
  S: string;
  Frame: TFrame;
begin
  if (Instance is TSuFrame) and (TComponent(Instance).Owner <> nil) then
  begin
    if PropInfo^.Name = 'VirtualClassName' then
    begin
      if Assigned(FOnGetClassResurce) then
      begin
        S := FOnGetClassResurce(Content);
        if S <> '' then
        begin
          Frame := TFrame(Instance);
          TextToFrame(S, Frame);
        end;
      end;
    end;
  end;
end;

procedure TDesignerManager.AncestorNotFound(Reader: TReader;
  const ComponentName: string; ComponentClass: TPersistentClass;
  var Component: TComponent);
begin
  ShowMessage(ComponentName);
end;

{$IFDEF WINDOWS}
procedure TDesignerManager.OnCreateComponent(Reader: TReader;
  ComponentClass: TComponentClass; var Component: TComponent);
begin
  if ComponentClass.ClassNameIs('TMainMenu') and (Reader.Root is TCustomForm) then
  begin
    FSaveLeft := TCustomForm(Reader.Root).Left;
    FSaveTop := TCustomForm(Reader.Root).Top;
    FSaveHeight := TCustomForm(Reader.Root).Height;
    FSaveWidth := TCustomForm(Reader.Root).Width;
  end;
end;
{$ENDIF}


procedure TDesignerManager.TextToDesignObj(s: string; var Root: TComponent;
  var DesignForm: TCustomForm; InitializForm: Boolean);
var
  StrStream:TStringStream;
  BinStream: TMemoryStream;
  Reader: TReader;
  LFMType, LFMComponentName, LFMClassName: String;
  TypeClass: TComponentClass;
begin
  ReadLFMHeader(S, LFMType, LFMComponentName, LFMClassName);

  if (LFMClassName <> '') and (LFMComponentName <> '') then
  begin

    if LFMClassName = 'TForm' then
      TypeClass:= TForm
    else
    if LFMClassName = 'TDataModule' then
      TypeClass := TDataModule
    else
    if LFMClassName = 'TSuFrame' then
      TypeClass := TSuFrame
    else
    if LFMClassName = 'TFrame' then
      TypeClass := TFrame;

    LoadHandlers := TJITMethods.Create;
    Root := TComponent(TypeClass.newinstance);
    SetComponentDesignMode(Root, True);
    Root.Create(nil);
    DesignForm := CreateDesignerForComponent(Root, InitializForm, LoadHandlers);
    Root.Name := LFMComponentName;
    {$IfDef WINDOWS}
    FSaveLeft := 0;
    FSaveTop := 0;
    FSaveWidth := 0;
    FSaveHeight := 0;
    {$EndIf}
    StrStream := TStringStream.Create(s);
    try
       BinStream := TMemoryStream.Create;
       try
          ObjectTextToBinary(StrStream, BinStream);
          BinStream.Seek(0, soFromBeginning);
          Reader := TReader.Create(BinStream, 4096);
          Reader.OnSetMethodProperty := @ReaderSetMethodProp;
          Reader.OnReadStringProperty := @ReaderReadStringProp;
          Reader.OnAncestorNotFound:= @AncestorNotFound;
          {$IfDef WINDOWS}
          Reader.OnCreateComponent := @OnCreateComponent;
          {$EndIf}
          try
            Reader.ReadRootComponent(Root);
          finally
            Reader.Free;
          end;
       finally
       BinStream.Free;
       end;
    finally
       StrStream.Free;
    end;
    {$IfDef WINDOWS}
    if (FSaveLeft > 0) and (DesignForm.Left <> FSaveLeft) then
    begin
      DesignForm.Left := FSaveLeft;
      DesignForm.Top := FSaveTop;
      DesignForm.Height := FSaveHeight;
      DesignForm.Width := FSaveWidth;
    end;
    {$EndIf}
    if InitializForm then
    begin
      InitialDesignForm(Root, DesignForm);
      TDesignerHook(DesignForm.Designer).SetNonVisual;
      TDesignerHook(DesignForm.Designer).AddSelection(DesignForm);
      TDesignerHook(DesignForm.Designer).ResetModified;
    end;
  end;
end;

procedure TDesignerManager.TextToObject(S: string; var Root: TComponent);
var
  StrStream:TStringStream;
  BinStream: TMemoryStream;
  Reader: TReader;
begin
  StrStream := TStringStream.Create(s);
  try
    BinStream := TMemoryStream.Create;
    try
          ObjectTextToBinary(StrStream, BinStream);
          BinStream.Seek(0, soFromBeginning);
          Reader := TReader.Create(BinStream, 4096);
          try
            Reader.ReadRootComponent(Root);
          finally
             Reader.Free;
          end;
    finally
      BinStream.Free;
    end;
  finally
    StrStream.Free;
  end;
end;

procedure TDesignerManager.SetFrameMethodProp(Reader: TReader;
  Instance: TPersistent; PropInfo: PPropInfo; const TheMethodName: string;
  var Handled: boolean);
  begin
    Handled := True;
  end;

procedure TDesignerManager.CopyToClipboard;
begin
  if CurrentDesignForm <> nil then
    TDesignerHook(CurrentDesignForm.Designer).CopySelection;
end;

procedure TDesignerManager.CutToClipboard;
begin
  if CurrentDesignForm <> nil then
  begin
    TDesignerHook(CurrentDesignForm.Designer).CutSelection;
  end;
end;

procedure TDesignerManager.PasterFromClipboard;
begin
  if CurrentDesignForm <> nil then
  begin
    TDesignerHook(CurrentDesignForm.Designer).PasteSelection([cpsfFindUniquePositions]);
  end;
end;

procedure TDesignerManager.DeleteSelections;
begin
  if CurrentDesignForm <> nil then
  begin
    TDesignerHook(CurrentDesignForm.Designer).DeleteSelection;
  end;
end;

procedure TDesignerManager.HideDesignHint;
begin
  if CurrentDesignForm <> nil then
  begin
    TDesignerHook(CurrentDesignForm.Designer).HideHint;
  end;
end;

procedure TDesignerManager.TextToFrame(S: string; var Frame: TFrame);
var
  StrStream:TStringStream;
  BinStream: TMemoryStream;
  Reader: TReader;
  R: TRect;
  Align: TAlign;
  SetPos: Boolean;
begin
  SetPos := False;
  if Frame = nil then
    Frame := TSuFrame.Create(TComponent(PropEditHook.LookupRoot))
  else
  begin
    R := Frame.BoundsRect;
    Align := Frame.Align;
    SetPos := True;
    //AParent := Frame.Parent;
  end;
  StrStream := TStringStream.Create(s);
  try
     BinStream := TMemoryStream.Create;
     try
        ObjectTextToBinary(StrStream, BinStream);
        BinStream.Seek(0, soFromBeginning);
        Reader := TReader.Create(BinStream, 4096);
        Reader.OnSetMethodProperty := @SetFrameMethodProp;
        try
          Reader.ReadRootComponent(Frame);
        finally
          Reader.Free;
        end;
     finally
       BinStream.Free;
     end;
  finally
   StrStream.Free;
  end;
  if SetPos then
  begin
    Frame.Align := Align;
    Frame.BoundsRect := R;
  end;
end;

function TDesignerManager.GetRootClassName: string;
begin
  Result := 'T' + TComponent(PropEditHook.LookupRoot).Name;
end;

end.

