unit dpi_parser;

{$mode objfpc}{$H+}

interface

uses
  SysUtils, Classes, CodeProcs;

const
  msEOF      = Char(0);
  msSymbol   = Char(1);
  msString   = Char(2);
  msInteger  = Char(3);
  msHexInt   = Char(4);
  msFloat    = Char(5);
  msText     = Char(6);
  msChar     = Char(7);
  msComent   = Char(8);
  msUnknSym  = Char(9);

type
  TDpString = string;


  TIdentToken = (
    ID_EOF,
    ID_Symbol,
    ID_NewLine,
    ID_RetCaret,
    ID_Comment,
    ID_Identifier,
    ID_SemiColon,
    ID_Comma,
    ID_Period,
    ID_Colon,
    ID_OpenRound,
    ID_CloseRound,
    ID_OpenBlock,
    ID_CloseBlock,
    ID_Assignment,
    ID_Inc,
    ID_Dec,
    ID_Equal,
    ID_NotEqual,
    ID_Greater,
    ID_GreaterEqual,
    ID_Less,
    ID_LessEqual,
    ID_Plus,
    ID_Minus,
    ID_Divide,
    ID_Multiply,
    ID_Integer,
    ID_Float,
    ID_String,
    ID_Char,
    ID_HexInt,
    ID_AddressOf,
    ID_Dereference,
    ID_Dots,
    ID_and,
    ID_array,
    ID_begin,
    ID_case,
    ID_const,
    ID_div,
    ID_do,
    ID_downto,
    ID_else,
    ID_elseif,
    ID_end,
    ID_for,
    ID_function,
    ID_if,
    ID_in,
    ID_mod,
    ID_not,
    ID_of,
    ID_or,
    ID_procedure,
    ID_method,
    ID_program,
    ID_read,
    ID_repeat,
    ID_record,
    ID_set,
    ID_shl,
    ID_shr,
    ID_then,
    ID_to,
    ID_type,
    ID_until,
    ID_Break,
    ID_Continue,
    ID_Exit,
    ID_uses,
    ID_var,
    ID_while,
    ID_with,
    ID_write,
    ID_xor,
    ID_class,
    ID_constructor,
    ID_destructor,
    ID_inherited,
    ID_private,
    ID_public,
    ID_published,
    ID_protected,
    ID_property,
    ID_virtual,
    ID_override,
    ID_overload,
    ID_As,
    ID_Is,
    ID_Unit,
    ID_Try,
    ID_Except,
    ID_On,
    ID_Finally,
    ID_External,
    ID_Forward,
    ID_Export,
    ID_Label,
    ID_Goto,
    ID_Chr,
    ID_Ord,
    ID_Interface,
    ID_Implementation,
    ID_initialization,
    ID_finalization,
    ID_out,
    ID_nil,
    ID_False,
    ID_True,
    ID_Unknown
    );

  PMemoToken = ^TMemoToken;
  TMemoToken = record
    TokenName: string;
    TokenID: TIdentToken;
    Col: Integer;
    Row: Integer;
    EndCol: Integer;
    HighLight: Boolean;
    KeyConst: Integer;
    IsEnd: Boolean;
  end;

  { TMemoTokens }

  TMemoTokens = class
  private
    FList: TList;
    FPos: Integer;
    function Get(Index: Integer): PMemoToken;
    function GetToken: PMemoToken;
    function GetPreviewNext: PMemoToken;
  protected
  public
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    function Count: Integer;
    procedure Next;
    procedure NextWithSkipComment;
    procedure PrevWithSkipComment;
    procedure Prev;
    procedure First;
    procedure Last;
    function Add(TokenName: string; TokenID: TIdentToken;
      Col, Row, EndCol, KeyConst: Integer; IsEnd: Boolean): Integer;
    property Pos: Integer read FPos write FPos;
    property Items[Index: Integer]: PMemoToken read Get;
    property Token: PMemoToken read GetToken;
    property PreviewNext: PMemoToken read GetPreviewNext;
  end;

  { TSuMemoParser }

  TSuMemoParser = class
  private
    FString: TDpString;
    FStartPos: PChar;
    FEndPos: PChar;
    FRow: Longint;
    FCol: Longint;
    FStartCol: PChar;
    FTokenPos: Integer;
    FTokenEnd: Integer;
    FCommentOpen: Boolean;
    FCommentOpen2: Boolean;
    BlockCount: Integer;
    FInClass: Boolean;
    FLanguage: Integer;
    FPropOpen: Boolean;
    function GetCol: Integer;
    function GetColEnd: Integer;
  protected
    function TokenString: TDpString;
    function TokenText: TDpString;
    function TokenFloat: Extended;
    function TokenInt: Integer;
  public
    //function PreviewToken: Char;
    constructor Create;
    destructor Destroy; override;
    function IsIdentChar(p: PChar): boolean;
    function NextToken: Char; virtual;
    procedure SetParseStr(S, ALanguange: string);
    procedure GetTokensList(Tokens: TMemoTokens);
    property Row: LongInt read FRow;
    property Col: LongInt read GetCol;
    property ColEnd: LongInt read GetColEnd;
    property TokenPos: Integer read FTokenPos;
    property TokenEnd: Integer read FTokenEnd;
    property AsString: TDpString read TokenString;
    property AsFloat: Extended read TokenFloat;
    property AsInteger: Integer read TokenInt;
    property AsText: TDpString read TokenText;
  end;

var
  KeyAnd, KeyArray, KeyBegin, KeyEnd, KeyAs, KeyCase, KeyChr,
  KeyConst, KeyClass, KeyConstr, KeyDestr, KeyDiv, KeyDo, KeyDownto,
  KeyElse, KeyElseif, KeyExcept, KeyExit, KeyExport, KeyExternal,
  KeyFinaliz, KeyFinally, KeyFor, KeyForward, KeyFunction, KeyProcedure,
  KeyMethod, KeyIf, KeyImplement, KeyIn, KeyInherited, KeyInitializ,
  KeyInterface, KeyIs, KeyLabel, KeyMod, KeyNil, KeyNot, KeyOf, KeyOr,
  KeyOrd, KeyOut, KeyOverride, KeyOverload, KeyPrivate, KeyProgram,
  KeyProperty, KeyProtected, KeyPublic, KeyPublished, KeyRecord,
  KeyRepeat, KeySet, KeyShl, KeyShr, KeyThen, KeyTo, KeyTry, KeyType,
  KeyUnit, KeyUntil, KeyUses, KeyVar, KeyVirtual, KeyWhile, KeyWith,
  KeyXor, KeyBreak, KeyContinue, KeyTrue, KeyFalse, KeyString,
  KeyOn, KeyRead, KeyWrite: String;

  KeyResult, KeySelf, KeyCreate: String;


implementation

uses Forms, dpUtils, character, LazUTF8;

var
  HashAnd, HashArray, HashBegin, HashEnd, HashAs, HashCase, HashChr,
  HashConst, HashClass, HashConstr, HashDestr, HashDiv, HashDo, HashDownto,
  HashElse, HashElseif, HashExcept, HashExit, HashExport, HashExternal,
  HashFinaliz, HashFinally, HashFor, HashForward, HashFunction, HashProcedure,
  HashMethod, HashIf, HashImplement, HashIn, HashInherited, HashInitializ,
  HashInterface, HashIs, HashLabel, HashMod, HashNil, HashNot, HashOf, HashOr,
  HashOrd, HashOut, HashOverride, HashOverload, HashPrivate, HashProgram, HashProperty,
  HashProtected, HashPublic, HashPublished, HashRecord, HashRepeat, HashSet,
  HashShl, HashShr, HashThen, HashTo, HashTry, HashType, HashUnit, HashUntil,
  HashUses, HashVar, HashVirtual, HashWhile, HashWith, HashXor, HashBreak,
  HashContinue, HashTrue, HashFalse, HashString, HashOn, HashRead, HashWrite: Cardinal;

var
  LHashAnd, LHashArray, LHashBegin, LHashEnd, LHashAs, LHashCase, LHashChr,
  LHashConst, LHashClass, LHashConstr, LHashDestr, LHashDiv, LHashDo, LHashDownto,
  LHashElse, LHashElseif, LHashExcept, LHashExit, LHashExport, LHashExternal,
  LHashFinaliz, LHashFinally, LHashFor, LHashForward, LHashFunction, LHashProcedure,
  LHashMethod, LHashIf, LHashImplement, LHashIn, LHashInherited, LHashInitializ,
  LHashInterface, LHashIs, LHashLabel, LHashMod, LHashNil, LHashNot, LHashOf, LHashOr,
  LHashOrd, LHashOut, LHashOverride, LHashOverload, LHashPrivate, LHashProgram,
  LHashProperty, LHashProtected, LHashPublic, LHashPublished, LHashRecord,
  LHashRepeat, LHashSet, LHashShl, LHashShr, LHashThen, LHashTo, LHashTry, LHashType,
  LHashUnit, LHashUntil, LHashUses, LHashVar, LHashVirtual, LHashWhile, LHashWith,
  LHashXor, LHashBreak, LHashContinue, LHashTrue, LHashFalse, LHashString,
  LHashOn, LHashRead, LHashWrite: Cardinal;

  LHashResult, LHashSelf, LHashCreate: Cardinal;

{TSuMemoParser}

constructor TSuMemoParser.Create;
begin
  inherited Create;
  FCommentOpen := False;
  FCommentOpen2 := False;
  FPropOpen := False;
end;

destructor TSuMemoParser.Destroy;
begin
  //FTokens.Free;
  inherited Destroy;
end;

function TSuMemoParser.IsIdentChar(p: PChar): boolean;
var
  u: UnicodeString;
  i: Integer;
  L: SizeUInt;
begin
  Result := False;
  Result := p^ in ['a'..'z','A'..'Z','0'..'9','_'];
  if Result then exit;

  if p^ <= #127 then exit;
  i := UTF8CharacterLength(p);
  SetLength(u, i);
  // wide chars of UTF-16 <= bytes of UTF-8 string
  if ConvertUTF8ToUTF16(PWideChar(u), i + 1, p, i, [toInvalidCharToSymbol], L) = trNoError then
  begin
    SetLength(u, L - 1);
    if L > 1 then
      Result := TCharacter.IsLetterOrDigit(u, 1);
  end;
end;

procedure TSuMemoParser.SetParseStr(S, ALanguange: string);
var
  I: Integer;
begin
  FLanguage := 0;
  if ALanguange = 'PASCAL' then FLanguage := 1;
  FCommentOpen := False;
  FCommentOpen2 := False;
  I := Length(S);
  FString := '';
  FRow := 0;
  FCol := 0;
  FTokenPos := 0;
  FTokenEnd := 0;

  if I > 0 then
  begin
    FString := S;
    I := Length(FString);
    Inc(I);
    SetLength(FString, I);
    FString[I] := #0;
    FStartPos := @FString[1];
    FEndPos := FStartPos;
    FStartCol := FStartPos;
  end
  else
  begin
    FStartPos := nil;
    FEndPos := nil;
    FStartCol := nil;
  end;
end;

function TSuMemoParser.NextToken: Char;
var
  P: PChar;
  L: Integer;

  procedure SkipBlanks;
  begin
    case P^ of
      ' ', #8:
      begin
        while (P^ = ' ') or (P^ = #8) do
        begin
          Inc(P);
        end;
      end;

      '/':
      begin
        Inc(P, 2);
        while not (P^ in [#0, #13, #10]) do Inc(P);
      end;

      '{':
      begin
        Inc(P);
        while P^ <> msEOF do
        begin
          case P^ of
            #10:
            begin
              FStartCol := P;
              Inc(FRow);
            end;
            '}':
            begin
              Inc(P);
              Break;
            end;
          end;
          Inc(P);
        end;
      end;

      '(':
      begin
        Inc(P, 2);
        while P^ <> msEOF do
        begin
          case P^ of
            #10:
            begin
              Inc(FRow);
              FStartCol := P;
            end;
            '*':
            begin
              if (P + 1)^ = ')' then
              begin
                Inc(P, 2);
                Break;
              end;
            end;
          end;

          Inc(P);
        end;
      end;

      #13:
      begin
        //Inc(FRow);
        Inc(P);
      end;

      #10:
      begin
        Inc(FRow);
        Inc(P);
        FStartCol := P;
      end;
    end;

  end;

begin
  if FStartPos = nil then
  begin
    Result := msEOF;
    Exit;
  end;
  FStartPos := FEndPos;
  P := FStartPos;

  if P^ = #0 then
  begin
    Result := msEOF;
    Exit;
  end;

  while (P^ = ' ') or (P^ = '{') or (P^ = #8)
    or (P^ = #13) or (P^ = #10)
    or (((P^ = '/') and ((P + 1)^ = '/')))
    or ((P^ = '(') and ((P + 1)^ = '*')) do
  begin
    SkipBlanks;
  end;

  FStartPos := P;

  if ((P^ = '(') and ((P + 1)^ = '*')) or FCommentOpen then
  begin
    FCommentOpen := True;
    while True do
    begin
      if (P^ = '*') and ((P + 1)^ = ')') then
      begin
        Inc(P, 2);
        FCommentOpen := False;
        Break;
      end
      else
      if CharInSet(P^, [#13, #10, #0]) then Break
      else Inc(P);
    end;
    Result := msComent;
    FEndPos := P;
    Exit;
  end;

  if (P^ = '{') or FCommentOpen2 then
  begin
    FCommentOpen2 := True;
    while True do
    begin
      if (P^ = '}') then
      begin
        Inc(P);
        FCommentOpen2 := False;
        Break;
      end
      else
      if CharInSet(P^, [#13, #10, #0]) then Break
      else Inc(P);
    end;
    Result := msComent;
    FEndPos := P;
    Exit;
  end;

  if (P^ = '/') and ((P + 1)^ = '/') then
  begin
    while True do
    begin
      Inc(P);
      if CharInSet(P^, [#13, #10, #0]) then Break;
    end;
    Result := msComent;
    FEndPos := P;
    Exit;
  end;

  if P^ = #0 then
  begin
      Result := msEOF;
  end
  else
  if P^ in ['0'..'9'] then
  begin
      Result := toInteger;
      Inc(P);
      while CharInSet(P^, ['0'..'9', '.', 'E', 'e']) do
      begin
        if (P^ = '.') then
        begin
          if CharInSet((P + 1)^, ['0'..'9']) then
            Result := msFloat
          else Break;
        end
        else
        if (P^ = 'e') or (P^ = 'E') then
        begin
          Result := msFloat;
          if ((P + 1)^ = '+') or ((P + 1)^ = '-') then Inc(P);
        end;
        Inc(P);
      end;
  end
  else
  if IsIdentChar(P) then
  begin
    L := UTF8CharacterLength(P);
    Inc(P, L);
    while IsIdentChar(P) or (P^ = '$') do
    begin
      L := UTF8CharacterLength(P);
      Inc(P, L);
    end;
    Result := msString;
  end
  else
  if P^ = '#' then
  begin
      FStartPos := P;
      Inc(P);
      while CharInSet(P^, ['0'..'9']) do Inc(P);
      if (FStartPos + 1) = P then
        Result := msUnknSym
      else
        Result := msChar;
  end
  else
  if P^ =  '$' then
  begin
      Inc(P);
      while CharInSet(P^, ['0'..'9', 'A'..'F', 'a'..'f']) do Inc(P);
      Result := msHexInt;
  end
  else
  if P^ =  '''' then
  begin
      Inc(P);
      while (P^ <> #0) and (P^ <> #10) and (P^ <> #13) do
      begin
        if P^ = '''' then
        begin
          if (P + 1)^ = '''' then
          begin
            Inc(P, 2);
            Continue;
          end
          else
          begin
            Inc(P);
            Break;
          end;
        end;
        Inc(P);
      end;
      Result := msText;
  end
  else
  begin
    if (P^ = '.') and ((P + 1)^ = '.') then Inc(P);
    L := UTF8CharacterLength(p);
    Inc(P, L);
    Result := msSymbol;
  end;
  FEndPos := P;
end;

function TSuMemoParser.TokenString: TDpString;
var
  I: Integer;
begin
  if (FStartPos <> nil) or (FEndPos <> nil) then
  begin
    I := FEndPos - FStartPos;
    FTokenPos := FTokenEnd;
    FTokenEnd := FTokenPos + I;
    SetString(Result, FStartPos, I);
  end;
end;

function TSuMemoParser.TokenText: TDpString;
var
  I: Integer;
begin
  if (FStartPos <> nil) or (FEndPos <> nil) then
  begin
    I := FEndPos - FStartPos;
    FTokenPos := FTokenEnd;
    FTokenEnd := FTokenPos + I;
    SetString(Result, FStartPos + 1, I - 2);
    I := Length(Result);
    while I > 0 do
    begin
      if (I > 1) and (Result[I] = '''') then
      begin
        if Result[I - 1] = '''' then
        begin
          Delete(Result, I, 1);
          Dec(I);
        end;
      end;
      Dec(I);
    end;
  end;
end;

function TSuMemoParser.TokenFloat: Extended;
begin
  Result := StrToFloat(AsString);
end;

function TSuMemoParser.TokenInt: Integer;
begin
  Result := StrToInt(AsString);
end;

function TSuMemoParser.GetCol: Integer;
var
  I: Integer;
begin
  I := FStartPos - FStartCol;
  Result := UTF8Length(FStartCol, I);
end;

function TSuMemoParser.GetColEnd: Integer;
var
  I: Integer;
begin
  I := FEndPos - FStartCol;
  Result := UTF8Length(FStartCol, I);
end;

procedure TSuMemoParser.GetTokensList(Tokens: TMemoTokens);
var
  C: Char;
  S, LS: string;
  H: Cardinal;
  I: Integer;
begin
  Tokens.Clear;
  BlockCount := 0;
  FInClass := False;
  FRow := 0;
  if FString = '' then
  begin
    Tokens.Add('', ID_EOF, 0, 0, 0, 0, False);
    Exit;
  end;
  C := NextToken;
  while C <> msEOF do
  begin
    case C of

      msSymbol:
      begin
        S := string(AsString);
        if S = '++' then Tokens.Add(S, ID_Inc, Col, Row, ColEnd, 2, False)
        else
        if S = '--' then Tokens.Add(S, ID_Dec, Col, Row, ColEnd, 2, False)
        else
        if S = '==' then Tokens.Add(S, ID_Equal, Col, Row, ColEnd, 2, False)
        else
        if S = '>=' then Tokens.Add(S, ID_GreaterEqual, Col, Row, ColEnd, 2, False)
        else
        if S = '>' then Tokens.Add(S, ID_Greater, Col, Row, ColEnd, 2, False)
        else
        if S = '<' then Tokens.Add(S, ID_Less, Col, Row, ColEnd, 2, False)
        else
        if S = '<=' then Tokens.Add(S, ID_LessEqual, Col, Row, ColEnd, 2, False)
        else
        if S = '<>' then Tokens.Add(S, ID_NotEqual, Col, Row, ColEnd, 2, False)
        else
        if S = '(' then Tokens.Add(S, ID_OpenRound, Col, Row, ColEnd, 2, False)
        else
        if S = ')' then Tokens.Add(S, ID_CloseRound, Col, Row, ColEnd, 2, False)
        else
        if S = '[' then Tokens.Add(S, ID_OpenBlock, Col, Row, ColEnd, 2, False)
        else
        if S = ']' then Tokens.Add(S, ID_CloseBlock, Col, Row, ColEnd, 2, False)
        else
        if S = ',' then Tokens.Add(S, ID_Comma, Col, Row, ColEnd, 2, False)
        else
        if S = '.' then Tokens.Add(S, ID_Period, Col, Row, ColEnd, 2, False)
        else
        if S = '..' then Tokens.Add(S, ID_Dots, Col, Row, ColEnd, 2, False)
        else
        if S = '@' then Tokens.Add(S, ID_AddressOf, Col, Row, ColEnd, 2, False)
        else
        if S = '^' then Tokens.Add(S, ID_Dereference, Col, Row, ColEnd, 2, False)
        else
        if S = ';' then
        begin
          Tokens.Add(S, ID_SemiColon, Col, Row, ColEnd, 2, False);
          FPropOpen := False;
        end
        else
        if S = ':' then Tokens.Add(S, ID_Colon, Col, Row, ColEnd, 2, False)
        else
        if S = '=' then Tokens.Add(S, ID_Assignment, Col, Row, ColEnd, 2, False)
        else
        if S = '+' then Tokens.Add(S, ID_Plus, Col, Row, ColEnd, 2, False)
        else
        if S = '-' then Tokens.Add(S, ID_Minus, Col, Row, ColEnd, 2, False)
        else
        if S = '*' then Tokens.Add(S, ID_Multiply, Col, Row, ColEnd, 2, False)
        else
        if S = '/' then Tokens.Add(S, ID_Divide, Col, Row, ColEnd, 2, False)
        else
          Tokens.Add(S, ID_Unknown, Col, Row, ColEnd, 2, False);

      end;
      msChar:
      begin
        S := string(AsString);
        Tokens.Add(S, ID_Char, Col, Row, ColEnd, 0, False);
      end;
      msString:
      begin
        S := AsString;
        LS := UTF8LowerCase(S);
        H := Hash(LS);
        I := -1;
        if (H = HashAnd) and (LS = 'and') then
          I := Tokens.Add(S, ID_and, Col, Row, ColEnd, 1, False)
        else
        if (H = HashArray) and (LS = 'array') then
          I := Tokens.Add(S, ID_array, Col, Row, ColEnd, 1, False)
        else
        if (H = HashAs) and (LS = 'as') then
          I := Tokens.Add(S, ID_As, Col, Row, ColEnd, 1, False)
        else
        if (H = HashBegin) and (LS = 'begin') then
        begin
          Inc(BlockCount);
          I := Tokens.Add(S, ID_begin, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashCase) and (LS = 'case') then
        begin
          Inc(BlockCount);
          I := Tokens.Add(S, ID_case, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashChr) and (LS = 'chr') then
          I := Tokens.Add(S, ID_Chr, Col, Row, ColEnd, 1, False)
        else
        if (H = HashClass) and (LS = 'class') then
        begin
          Inc(BlockCount);
          FInClass := True;
          I := Tokens.Add(S, ID_class, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashConst) and (LS = 'const') then
          I := Tokens.Add(S, ID_const, Col, Row, ColEnd, 1, False)
        else
        if (H = HashConstr) and (LS = 'constructor') then
        begin
          if not FInClass then BlockCount := 0;
          I := Tokens.Add(S, ID_constructor, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashDestr) and (LS = 'destructor') then
        begin
          if not FInClass then BlockCount := 0;
          I := Tokens.Add(S, ID_destructor, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashDiv) and (LS = 'div') then
          I := Tokens.Add(S, ID_div, Col, Row, ColEnd, 1, False)
        else
        if (H = HashDo) and (LS = 'do') then
          I := Tokens.Add(S, ID_do, Col, Row, ColEnd, 1, False)
        else
        if (H = HashDownto) and (LS = 'downto') then
          I := Tokens.Add(S, ID_downto, Col, Row, ColEnd, 1, False)
        else
        if (H = HashElse) and (LS = 'else') then
          I := Tokens.Add(S, ID_else, Col, Row, ColEnd, 1, False)
        else
        if (H = HashElseif) and (LS = 'elseif') then
          I := Tokens.Add(S, ID_elseif, Col, Row, ColEnd, 1, False)
        else
        if (H = HashEnd) and (LS = 'end') then
        begin
          Dec(BlockCount);
          if FInClass then FInClass := False;
          I := Tokens.Add(S, ID_end, Col, Row, ColEnd, 1, BlockCount = 0)
        end
        else
        if (H = HashExcept) and (LS = 'except') then
          I := Tokens.Add(S, ID_Except, Col, Row, ColEnd, 1, False)
        else
        if (H = HashOn) and (LS = 'on') then
          I := Tokens.Add(S, ID_On, Col, Row, ColEnd, 1, False)
        else
        if (H = HashExit) and (LS = 'exit') then
          I := Tokens.Add(S, ID_exit, Col, Row, ColEnd, 0, False)
        else
        if (H = HashExport) and (LS = 'export') then
          I := Tokens.Add(S, ID_Export, Col, Row, ColEnd, 1, False)
        else
        if (H = HashExternal) and (LS = 'external') then
          I := Tokens.Add(S, ID_External, Col, Row, ColEnd, 1, False)
        else
        if (H = HashFinaliz) and (LS = 'finalization') then
          I := Tokens.Add(S, ID_finalization, Col, Row, ColEnd, 1, False)
        else
        if (H = HashFinally) and (LS = 'finally') then
          I := Tokens.Add(S, ID_Finally, Col, Row, ColEnd, 1, False)
        else
        if (H = HashFor) and (LS = 'for') then
        begin
          if FLanguage = 0 then Inc(BlockCount);
          I := Tokens.Add(S, ID_for, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashForward) and (LS = 'forward') then
          I := Tokens.Add(S, ID_Forward, Col, Row, ColEnd, 1, False)
        else
        if (H = HashFunction) and (LS = 'function') then
        begin
          if not FInClass then BlockCount := 0;
          I := Tokens.Add(S, ID_function, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashProcedure) and (LS = 'procedure') then
        begin
          if not FInClass then BlockCount := 0;
          I := Tokens.Add(S, ID_function, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashMethod) and (LS = 'method') then
        begin
          if not FInClass then BlockCount := 0;
          I := Tokens.Add(S, ID_method, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashIf) and (LS = 'if') then
        begin
          if FLanguage = 0 then Inc(BlockCount);
          I := Tokens.Add(S, ID_if, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashImplement) and (LS = 'implementation') then
          I := Tokens.Add(S, ID_Implementation, Col, Row, ColEnd, 1, False)
        else
        if (H = HashIn) and (LS = 'in') then
          I := Tokens.Add(S, ID_in, Col, Row, ColEnd, 1, False)
        else
        if (H = HashInherited) and (LS = 'inherited') then
          I := Tokens.Add(S, ID_inherited, Col, Row, ColEnd, 1, False)
        else
        if (H = HashInitializ) and (LS = 'initialization') then
          I := Tokens.Add(S, ID_initialization, Col, Row, ColEnd, 1, False)
        else
        if (H = HashInterface) and (LS = 'interface') then
          I := Tokens.Add(S, ID_Interface, Col, Row, ColEnd, 1, False)
        else
        if (H = HashIs) and (LS = 'is') then
          I := Tokens.Add(S, ID_Is, Col, Row, ColEnd, 1, False)
        else
        if (H = HashLabel) and (LS = 'label') then
          I := Tokens.Add(S, ID_Label, Col, Row, ColEnd, 1, False)
        else
        if (H = HashMod) and (LS = 'mod') then
          I := Tokens.Add(S, ID_mod, Col, Row, ColEnd, 1, False)
        else
        if (H = HashNil) and (LS = 'nil') then
          I := Tokens.Add(S, ID_nil, Col, Row, ColEnd, 1, False)
        else
        if (H = HashNot) and (LS = 'not') then
          I := Tokens.Add(S, ID_not, Col, Row, ColEnd, 1, False)
        else
        if (H = HashOf) and (LS = 'of') then
          I := Tokens.Add(S, ID_of, Col, Row, ColEnd, 1, False)
        else
        if (H = HashOr) and (LS = 'or') then
          I := Tokens.Add(S, ID_or, Col, Row, ColEnd, 1, False)
        else
        if (H = HashOrd) and (LS = 'ord') then
          I := Tokens.Add(S, ID_Ord, Col, Row, ColEnd, 1, False)
        else
        if (H = HashOut) and (LS = 'out') then
          I := Tokens.Add(S, ID_out, Col, Row, ColEnd, 1, False)
        else
        if (H = HashOverride) and (LS = 'override') then
          I := Tokens.Add(S, ID_override, Col, Row, ColEnd, 1, False)
        else
        if (H = HashOverload) and (LS = 'overload') then
          I := Tokens.Add(S, ID_overload, Col, Row, ColEnd, 1, False)
        else
        if (H = HashPrivate) and (LS = 'private') then
          I := Tokens.Add(S, ID_private, Col, Row, ColEnd, 1, False)
        else
        if (H = HashProgram) and (LS = 'program') then
          I := Tokens.Add(S, ID_program, Col, Row, ColEnd, 1, False)
        else
        if (H = HashProperty) and (LS = 'property') then
        begin
          FPropOpen := True;
          I := Tokens.Add(S, ID_property, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashProtected) and (LS = 'protected') then
          I := Tokens.Add(S, ID_protected, Col, Row, ColEnd, 1, False)
        else
        if (H = HashPublic) and (LS = 'public') then
          I := Tokens.Add(S, ID_public, Col, Row, ColEnd, 1, False)
        else
        if (H = HashPublished) and (LS = 'published') then
          I := Tokens.Add(S, ID_published, Col, Row, ColEnd, 1, False)
        else
        if (H = HashRecord) and (LS = 'record') then
        begin
          Inc(BlockCount);
          FInClass := True;
          I := Tokens.Add(S, ID_class, Col, Row, ColEnd, 1, False)
        end
        else
        if (H = HashRepeat) and (LS = 'repeat') then
          I := Tokens.Add(S, ID_repeat, Col, Row, ColEnd, 1, False)
        else
        if (H = HashSet) and (LS = 'set') then
          I := Tokens.Add(S, ID_set, Col, Row, ColEnd, 1, False)
        else
        if (H = HashShl) and (LS = 'shl') then
          I := Tokens.Add(S, ID_shl, Col, Row, ColEnd, 1, False)
        else
        if (H = HashShr) and (LS = 'shr') then
          I := Tokens.Add(S, ID_shr, Col, Row, ColEnd, 1, False)
        else
        if (H = HashThen) and (LS = 'then') then
          I := Tokens.Add(S, ID_then, Col, Row, ColEnd, 1, False)
        else
        if (H = HashTo) and (LS = 'to') then
          I := Tokens.Add(S, ID_to, Col, Row, ColEnd, 1, False)
        else
        if (H = HashTry) and (LS = 'try') then
        begin
            Inc(BlockCount);
            I := Tokens.Add(S, ID_Try, Col, Row, ColEnd, 1, False)
        end
        else
        if (H = HashType) and (LS = 'type') then
            I := Tokens.Add(S, ID_type, Col, Row, ColEnd, 1, False)
        else
        if (H = HashUnit) and (LS = 'unit') then
            I := Tokens.Add(S, ID_Unit, Col, Row, ColEnd, 1, False)
        else
        if (H = HashUntil) and (LS = 'until') then
            I := Tokens.Add(S, ID_until, Col, Row, ColEnd, 1, False)
        else
        if (H = HashUses) and (LS = 'uses') then
            I := Tokens.Add(S, ID_uses, Col, Row, ColEnd, 1, False)
        else
        if (H = HashVar) and (LS = 'var') then
            I := Tokens.Add(S, ID_var, Col, Row, ColEnd, 1, False)
        else
        if (H = HashVirtual) and (LS = 'virtual') then
           I := Tokens.Add(S, ID_virtual, Col, Row, ColEnd, 1, False)
        else
        if (H = HashWhile) and (LS = 'while') then
        begin
            if FLanguage = 0 then Inc(BlockCount);
            I := Tokens.Add(S, ID_while, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashWith) and (LS = 'with') then
        begin
          if FLanguage = 0 then Inc(BlockCount);
          I := Tokens.Add(S, ID_with, Col, Row, ColEnd, 1, False);
        end
        else
        if (H = HashXor) and (LS = 'xor') then
            I := Tokens.Add(S, ID_xor, Col, Row, ColEnd, 1, False)
        else
        if (H = HashBreak) and (LS = 'break') then
           I := Tokens.Add(S, ID_Break, Col, Row, ColEnd, 0, False)
        else
        if (H = HashContinue) and (LS = 'continue') then
            I := Tokens.Add(S, ID_Continue, Col, Row, ColEnd, 0, False)
        else
        if (H = HashTrue) and (LS = 'true') then
            I := Tokens.Add(S, ID_True, Col, Row, ColEnd, 0, False)
        else
        if (H = HashFalse) and (LS = 'false') then
            I := Tokens.Add(S, ID_False, Col, Row, ColEnd, 0, False)
        else
        if (H = HashString) and (LS = 'string') then
            I := Tokens.Add(S, ID_Identifier, Col, Row, ColEnd, 1, False)
        else
        if (H = HashRead) and (LS = 'read') then
        begin
          if FPropOpen then
            I := Tokens.Add(S, ID_read, Col, Row, ColEnd, 1, False)
          else
            I := Tokens.Add(S, ID_Identifier, Col, Row, ColEnd, 0, False)
        end
        else
        if (H = HashWrite) and (LS = 'write') then
        begin
           if FPropOpen then
             I := Tokens.Add(S, ID_write, Col, Row, ColEnd, 1, False)
           else
             I := Tokens.Add(S, ID_Identifier, Col, Row, ColEnd, 0, False)
        end
        else
        if (H = LHashAnd) and (LS = KeyAnd) then
              Tokens.Add(S, ID_and, Col, Row, ColEnd, 1, False)
        else
          if (H = LHashArray) and (LS = KeyArray) then
            Tokens.Add(S, ID_array, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashAs) and (LS = KeyAs) then
            Tokens.Add(S, ID_As, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashBegin) and (LS = KeyBegin) then
          begin
            Inc(BlockCount);
            Tokens.Add(S, ID_begin, Col, Row, ColEnd, 1, False);
          end
          else
          if (H = LHashCase) and (LS = KeyCase) then
          begin
            Inc(BlockCount);
            Tokens.Add(S, ID_case, Col, Row, ColEnd, 1, False);
          end
          else
          if (H = LHashClass) and (LS = KeyClass) then
          begin
            Inc(BlockCount);
            FInClass := True;
            Tokens.Add(S, ID_class, Col, Row, ColEnd, 1, False);
          end
          else
          if (H = LHashConst) and (LS = KeyConst) then
            Tokens.Add(S, ID_const, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashConstr) and (LS = KeyConstr) then
          begin
            if not FInClass then BlockCount := 0;
            Tokens.Add(S, ID_constructor, Col, Row, ColEnd, 1, False);
          end
          else
          if (H = LHashDestr) and (LS = KeyDestr) then
          begin
            if not FInClass then BlockCount := 0;
            Tokens.Add(S, ID_destructor, Col, Row, ColEnd, 1, False);
          end
          else
          if (H = LHashDo) and (LS = KeyDo) then
            Tokens.Add(S, ID_do, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashDownto) and (LS = KeyDownto) then
            Tokens.Add(S, ID_downto, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashElse) and (LS = KeyElse) then
            Tokens.Add(S, ID_else, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashElseif) and (LS = KeyElseif) then
            Tokens.Add(S, ID_elseif, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashEnd) and (LS = KeyEnd) then
          begin
            Dec(BlockCount);
            if FInClass then FInClass := False;
            Tokens.Add(S, ID_end, Col, Row, ColEnd, 1, BlockCount = 0)
          end
          else
          if (H = LHashExcept) and (LS = KeyExcept) then
            Tokens.Add(S, ID_Except, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashOn) and (LS = KeyOn) then
            Tokens.Add(S, ID_On, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashExit) and (LS = KeyExit) then
            Tokens.Add(S, ID_exit, Col, Row, ColEnd, 0, False)
          else
          if (H = LHashFinaliz) and (LS = KeyFinaliz) then
            Tokens.Add(S, ID_finalization, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashFinally) and (LS = KeyFinally) then
            Tokens.Add(S, ID_Finally, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashFor) and (LS = KeyFor) then
          begin
            if FLanguage = 0 then Inc(BlockCount);
            Tokens.Add(S, ID_for, Col, Row, ColEnd, 1, False);
          end
          else
          if (H = LHashForward) and (LS = KeyForward) then
            Tokens.Add(S, ID_Forward, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashFunction) and (LS = KeyFunction) then
          begin
            if not FInClass then BlockCount := 0;
            Tokens.Add(S, ID_function, Col, Row, ColEnd, 1, False);
          end
          else
          if (H = LHashProcedure) and (LS = KeyProcedure) then
          begin
            if not FInClass then BlockCount := 0;
            Tokens.Add(S, ID_procedure, Col, Row, ColEnd, 1, False);
          end
          else
          if (H = LHashMethod) and (LS = KeyMethod) then
          begin
            if not FInClass then BlockCount := 0;
            Tokens.Add(S, ID_method, Col, Row, ColEnd, 1, False);
          end
          else
          if (H = LHashIf) and (LS = KeyIf) then
          begin
            if FLanguage = 0 then Inc(BlockCount);
            Tokens.Add(S, ID_if, Col, Row, ColEnd, 1, False);
          end
          else
          if (H = LHashImplement) and (LS = KeyImplement) then
            Tokens.Add(S, ID_Implementation, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashIn) and (LS = KeyIn) then
            Tokens.Add(S, ID_in, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashInherited) and (LS = KeyInherited) then
            Tokens.Add(S, ID_inherited, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashInitializ) and (LS = KeyInitializ) then
            Tokens.Add(S, ID_initialization, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashInterface) and (LS = KeyInterface) then
            Tokens.Add(S, ID_Interface, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashIs) and (LS = KeyIs) then
            Tokens.Add(S, ID_Is, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashMod) and (LS = KeyMod) then
            Tokens.Add(S, ID_mod, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashNil) and (LS = KeyNil) then
            Tokens.Add(S, ID_nil, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashNot) and (LS = KeyNot) then
            Tokens.Add(S, ID_not, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashOf) and (LS = KeyOf) then
            Tokens.Add(S, ID_of, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashOr) and (LS = KeyOr) then
            Tokens.Add(S, ID_or, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashPrivate) and (LS = KeyPrivate) then
            Tokens.Add(S, ID_private, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashProgram) and (LS = KeyProgram) then
            Tokens.Add(S, ID_program, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashProtected) and (LS = KeyProtected) then
            Tokens.Add(S, ID_protected, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashPublic) and (LS = KeyPublic) then
            Tokens.Add(S, ID_public, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashRepeat) and (LS = KeyRepeat) then
            Tokens.Add(S, ID_repeat, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashSet) and (LS = KeySet) then
            Tokens.Add(S, ID_set, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashThen) and (LS = KeyThen) then
            Tokens.Add(S, ID_then, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashTo) and (LS = KeyTo) then
            Tokens.Add(S, ID_to, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashTry) and (LS = KeyTry) then
          begin
            Inc(BlockCount);
            Tokens.Add(S, ID_Try, Col, Row, ColEnd, 1, False)
          end
          else
          if (H = LHashType) and (LS = KeyType) then
            Tokens.Add(S, ID_type, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashUnit) and (CompareText(S, KeyUnit) = 0) then
            Tokens.Add(S, ID_Unit, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashUntil) and (LS = KeyUntil) then
            Tokens.Add(S, ID_until, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashUses) and (LS = KeyUses) then
            Tokens.Add(S, ID_uses, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashVar) and (LS = KeyVar) then
            Tokens.Add(S, ID_var, Col, Row, ColEnd, 1, False)
          else
          if (H = LHashWhile) and (LS = KeyWhile) then
          begin
            if FLanguage = 0 then Inc(BlockCount);
            Tokens.Add(S, ID_while, Col, Row, ColEnd, 1, False);
          end
          else
          if (H = LHashBreak) and (LS = KeyBreak) then
            Tokens.Add(S, ID_Break, Col, Row, ColEnd, 0, False)
          else
          if (H = LHashContinue) and (LS = KeyContinue) then
            Tokens.Add(S, ID_Continue, Col, Row, ColEnd, 0, False)
          else
          if (H = LHashTrue) and (LS = KeyTrue) then
            Tokens.Add(S, ID_True, Col, Row, ColEnd, 0, False)
          else
          if (H = LHashFalse) and (LS = KeyFalse) then
            Tokens.Add(S, ID_False, Col, Row, ColEnd, 0, False)
          else
          if (H = LHashWith) and (LS = KeyWith) then
          begin
            if FLanguage = 0 then Inc(BlockCount);
            Tokens.Add(S, ID_with, Col, Row, ColEnd, 1, False)
          end
          else
          if (H = LHashProperty) and (LS = KeyProperty) then
          begin
            FPropOpen := True;
            Tokens.Add(S, ID_property, Col, Row, ColEnd, 1, False)
          end
          else
          if (H = LHashRead) and (LS = KeyRead) then
          begin
            if FPropOpen then
              Tokens.Add(S, ID_read, Col, Row, ColEnd, 1, False)
            else
              Tokens.Add(S, ID_Identifier, Col, Row, ColEnd, 0, False)
          end
          else
          if (H = LHashWrite) and (LS = KeyWrite) then
          begin
            if FPropOpen then
              Tokens.Add(S, ID_write, Col, Row, ColEnd, 1, False)
            else
              Tokens.Add(S, ID_Identifier, Col, Row, ColEnd, 0, False)
          end
          else
            Tokens.Add(S, ID_Identifier, Col, Row, ColEnd, 0, False);
      end;
      msComent:
      begin
        S := string(AsString);
        Tokens.Add(S, ID_Comment, Col, Row, ColEnd, 0, False);
      end;
      msInteger:
      begin
        S := string(AsString);
        Tokens.Add(S, ID_Integer, Col, Row, ColEnd, 0, False);
      end;
      msHexInt:
      begin
        S := string(AsString);
        Tokens.Add(S, ID_Integer, Col, Row, ColEnd, 0, False);
      end;
      msFloat:
      begin
        S := string(AsString);
        Tokens.Add(S, ID_Float, Col, Row, ColEnd, 0, False);
      end;
      msText:
      begin
        S := string(AsString);
        Tokens.Add(S, ID_String, Col, Row, ColEnd, 0, False);
      end;
      msUnknSym:
      begin
        S := string(AsString);
        Tokens.Add(S, ID_Unknown, Col, Row, ColEnd, 0, False);
      end;
    end;
    C := NextToken;
  end;
  Tokens.Add('', ID_EOF, Col, Row, ColEnd, 0, False);
end;

{TMemoTokens}

constructor TMemoTokens.Create;
begin
  FList := TList.Create;
  FPos := 0;
end;

destructor TMemoTokens.Destroy;
begin
  Clear;
  FList.Free;
  inherited Destroy;
end;

function TMemoTokens.Count: Integer;
begin
  Result := FList.Count;
end;

procedure TMemoTokens.Clear;
var
  PR: PMemoToken;
  I: Integer;
begin
  FPos := 0;
  I := 0;
  while I < FList.Count do
  begin
    PR := PMemoToken(FList.Items[I]);
    Dispose(PR);
    FList.Items[I] := nil;
    Inc(I);
  end;
  FList.Clear;
end;

function TMemoTokens.Add(TokenName: string; TokenID: TIdentToken; Col, Row,
  EndCol, KeyConst: Integer; IsEnd: Boolean): Integer;
var
  PR: PMemoToken;
begin
  New(PR);
  PR^.HighLight := False;
  PR^.TokenName := TokenName;
  PR^.TokenID := TokenID;
  PR^.Col := Col;
  PR^.EndCol := EndCol;
  PR^.Row := Row;
  PR^.KeyConst := KeyConst;
  PR^.IsEnd := IsEnd;
  Result := FList.Add(PR);
end;

function TMemoTokens.Get(Index: Integer): PMemoToken;
begin
  if (Index < 0) or (Index >= FList.Count) then Index := FList.Count  - 1;

  Result := PMemoToken(FList.Items[Index]);
end;

function TMemoTokens.GetToken: PMemoToken;
begin
  Result := Get(FPos);
end;

function TMemoTokens.GetPreviewNext: PMemoToken;
begin
  Result := Get(FPos + 1);
end;

procedure TMemoTokens.Next;
begin
  Inc(FPos);
end;

procedure TMemoTokens.NextWithSkipComment;
begin
  Inc(FPos);
  if Token^.TokenID = ID_Comment then
    while Token^.TokenID = ID_Comment do Inc(FPos);
end;

procedure TMemoTokens.PrevWithSkipComment;
begin
  Dec(FPos);
  if Token^.TokenID = ID_Comment then
    while Token^.TokenID = ID_Comment do Dec(FPos);
end;

procedure TMemoTokens.Prev;
begin
  Dec(FPos);
end;

procedure TMemoTokens.First;
begin
  FPos := 0;
end;

procedure TMemoTokens.Last;
begin
  FPos := Count - 1;
end;

initialization
  HashAnd := Hash('and');
  HashBegin := Hash('begin');
  HashEnd := Hash('end');
  HashArray := Hash('array');
  HashAs := Hash('as');
  HashCase := Hash('case');
  HashChr := Hash('chr');
  HashConst := Hash('const');
  HashClass := Hash('class');
  HashConstr := Hash('constructor');
  HashDestr := Hash('destructor');
  HashDiv := Hash('div');
  HashDo := Hash('do');
  HashDownto := Hash('downto');
  HashElse := Hash('else');
  HashElseif := Hash('elseif');
  HashExcept := Hash('except');
  HashExit := Hash('exit');
  HashExport := Hash('export');
  HashExternal := Hash('external');
  HashFinaliz := Hash('finalization');
  HashFinally := Hash('finally');
  HashFor := Hash('for');
  HashForward := Hash('forward');
  HashFunction := Hash('function');
  HashProcedure := Hash('procedure');
  HashMethod := Hash('method');
  HashIf := Hash('if');
  HashImplement := Hash('implementation');
  HashIn := Hash('in');
  HashInherited := Hash('inherited');
  HashInitializ := Hash('initialization');
  HashInterface := Hash('interface');
  HashIs := Hash('is');
  HashLabel := Hash('label');
  HashMod := Hash('mod');
  HashNil := Hash('nil');
  HashNot := Hash('not');
  HashOf := Hash('of');
  HashOr := Hash('or');
  HashOrd := Hash('ord');
  HashOut := Hash('out');
  HashOverride := Hash('override');
  HashOverload := Hash('overload');
  HashPrivate := Hash('private');
  HashProtected := Hash('protected');
  HashProgram := Hash('program');
  HashProperty := Hash('property');
  HashPublic := Hash('public');
  HashPublished := Hash('published');
  HashRecord := Hash('record');
  HashRepeat := Hash('repeat');
  HashSet := Hash('set');
  HashShl := Hash('shl');
  HashShr := Hash('shr');
  HashThen := Hash('then');
  HashTo := Hash('to');
  HashTry := Hash('try');
  HashType := Hash('type');
  HashUnit := Hash('unit');
  HashUntil := Hash('until');
  HashUses := Hash('uses');
  HashVar := Hash('var');
  HashVirtual := Hash('virtual');
  HashWhile := Hash('while');
  HashWith := Hash('with');
  HashXor := Hash('xor');
  HashBreak := Hash('break');
  HashContinue := Hash('continue');
  HashTrue := Hash('true');
  HashFalse := Hash('false');
  HashString := Hash('string');
  HashOn := Hash('on');
  HashRead := Hash('read');
  HashWrite := Hash('write');

  KeyAnd := 'и';
  KeyArray := 'массив';
  KeyBegin := 'начало';
  KeyEnd := 'конец';
  KeyAs := 'как';
  KeyCase := 'выбор';
  KeyChr := 'символ';
  KeyConst := 'конст';
  KeyClass := 'класс';
  KeyConstr := 'конструктор';
  KeyDestr := 'деструктор';
  KeyDiv := 'деление_цел';
  KeyDo := 'выполнять';
  KeyDownto := 'вниздо';
  KeyElse := 'иначе';
  KeyElseif := 'иначеесли';
  KeyExcept := 'исключ';
  KeyExit := 'выход';
  KeyExport := 'экспорт';
  KeyExternal := 'внешний';
  KeyFinaliz := 'завершение';
  KeyFinally := 'завершить';
  KeyFor := 'для';
  KeyForward := 'опережающее';
  KeyFunction := 'функция';
  KeyProcedure := 'процедура';
  KeyMethod := 'метод';
  KeyIf := 'если';
  KeyImplement := 'реализация';
  KeyIn := 'естьв';
  KeyInherited := 'родительский';
  KeyInitializ := 'инициализация';
  KeyInterface := 'интерфейс';
  KeyIs := 'есть';
  KeyLabel := 'метка';
  KeyMod := 'делениеост';
  KeyNil := 'пустой';
  KeyNot := 'не';
  KeyOf := 'из';
  KeyOr := 'или';
  //KeyOrd := 'перечисл_знач';
  KeyOut := 'возвр';
  KeyOverride := 'переопределен';
  KeyOverload := 'перегружен';
  KeyPrivate := 'закрытые';
  KeyProtected := 'защищенные';
  KeyPublic := 'открытые';
  KeyPublished := 'опубликованные';
  KeyProgram := 'программа';
  KeyProperty := 'свойство';
  KeyRecord := 'запись';
  KeyRepeat := 'повторять';
  KeySet := 'множество';
  KeyShl := 'сдвигвлево';
  KeyShr := 'сдвигвправо';
  KeyThen := 'тогда';
  KeyTo := 'до';
  KeyTry := 'проба';
  KeyType := 'тип';
  KeyUnit := 'модуль';
  KeyUntil := 'повтордо';
  KeyUses := 'подключить';
  KeyVar := 'перем';
  KeyVirtual := 'виртуальный';
  KeyWhile := 'пока';
  KeyWith := 'вместе_с';
  KeyXor := 'и_или';
  KeyBreak := 'прервать';
  KeyContinue := 'продолжить';
  KeyTrue := 'правда';
  KeyFalse := 'ложь';
  KeyString := 'строка';
  KeyOn := 'в';
  KeyRead := 'читать';
  KeyWrite := 'писать';

  KeyResult := 'результат';
  KeySelf := 'этот';
  KeyCreate := 'создать';

  LHashAnd := Hash(KeyAnd);
  LHashArray := Hash(KeyArray);
  LHashBegin := Hash(KeyBegin);
  LHashEnd := Hash(KeyEnd);
  LHashAs := Hash(KeyAs);
  LHashCase := Hash(KeyCase);
  LHashChr := Hash(KeyChr);
  LHashConst := Hash(KeyConst);
  LHashClass := Hash(KeyClass);
  LHashConstr := Hash(KeyConstr);
  LHashDestr := Hash(KeyDestr);
  LHashDiv := Hash(KeyDiv);
  LHashDo := Hash(KeyDo);
  LHashDownto := Hash(KeyDownto);
  LHashElse := Hash(KeyElse);
  LHashElseif := Hash(KeyElseif);
  LHashExcept := Hash(KeyExcept);
  LHashExit := Hash(KeyExit);
  LHashExport := Hash(KeyExport);
  LHashExternal := Hash(KeyExternal);
  LHashFinaliz := Hash(KeyFinaliz);
  LHashFinally := Hash(KeyFinally);
  LHashFor := Hash(KeyFor);
  LHashForward := Hash(KeyForward);
  LHashFunction := Hash(KeyFunction);
  LHashProcedure := Hash(KeyProcedure);
  LHashMethod := Hash(KeyMethod);
  LHashIf := Hash(KeyIf);
  LHashImplement := Hash(KeyImplement);
  LHashIn := Hash(KeyIn);
  LHashInherited := Hash(KeyInherited);
  LHashInitializ := Hash(KeyInitializ);
  LHashInterface := Hash(KeyInterface);
  LHashIs := Hash(KeyIs);
  LHashLabel := Hash(KeyLabel);
  LHashMod := Hash(KeyMod);
  LHashNil := Hash(KeyNil);
  LHashNot := Hash(KeyNot);
  LHashOf := Hash(KeyOf);
  LHashOr := Hash(KeyOr);
  //LHashOrd := Hash(KeyOrd);
  LHashOut := Hash(KeyOut);
  LHashOverride := Hash(KeyOverride);
  LHashOverload := Hash(KeyOverload);
  LHashPrivate := Hash(KeyPrivate);
  LHashProtected := Hash(KeyProtected);
  LHashPublic := Hash(KeyPublic);
  LHashPublished := Hash(KeyPublished);
  LHashProgram := Hash(KeyProgram);
  LHashProperty := Hash(KeyProperty);
  LHashRecord := Hash(KeyRecord);
  LHashRepeat := Hash(KeyRepeat);
  LHashSet := Hash(KeySet);
  LHashShl := Hash(KeyShl);
  LHashShr := Hash(KeyShr);
  LHashThen := Hash(KeyThen);
  LHashTo := Hash(KeyTo);
  LHashTry := Hash(KeyTry);
  LHashType := Hash(KeyType);
  LHashUnit := Hash(KeyUnit);
  LHashUntil := Hash(KeyUntil);
  LHashUses := Hash(KeyUses);
  LHashVar := Hash(KeyVar);
  LHashVirtual := Hash(KeyVirtual);
  LHashWhile := Hash(KeyWhile);
  LHashWith := Hash(KeyWith);
  LHashXor := Hash(KeyXor);
  LHashBreak := Hash(KeyBreak);
  LHashContinue := Hash(KeyContinue);
  LHashTrue := Hash(KeyTrue);
  LHashFalse := Hash(KeyFalse);
  LHashString := Hash(KeyString);
  LHashOn := Hash(KeyOn);
  LHashRead := Hash(KeyRead);
  LHashWrite := Hash(KeyWrite);

  LHashCreate := Hash(KeyCreate);
  LHashSelf := Hash(KeySelf);
  LHashResult := Hash(KeyResult);
end.

