#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################
# Tool to mount crw-daq.su to local /mnt/ssh/crw-daq.su.
##########################################################

################################
# Current script identification.
################################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

####################
# mount information.
####################
readonly local_dir="/mnt/ssh/crw-daq.su";
readonly remote_account_url="kouriakine@crw-daq.su";
readonly remote_dir="/home/users/k/kouriakine/domains";

#############################################################################################################################
# https://stackoverflow.com/questions/1763891/is-it-possible-to-make-stdout-and-stderr-output-be-of-different-colors-in-xterm
#############################################################################################################################
function colorize_norm()( set -o pipefail; ( "$@" 2>&1>&3 | sed $'s,.*,\e[31m&\e[m,'   >&2 ) 3>&1 | sed $'s,.*,\e[32m&\e[m,'   );
function colorize_bold()( set -o pipefail; ( "$@" 2>&1>&3 | sed $'s,.*,\e[1;31m&\e[m,' >&2 ) 3>&1 | sed $'s,.*,\e[1;32m&\e[m,' );
function colorize_head()( set -o pipefail; ( "$@" 2>&1>&3 | sed $'s,.*,\e[1;35m&\e[m,' >&2 ) 3>&1 | sed $'s,.*,\e[1;37m&\e[m,' );
function colorize_none()( "$@"; );

#################################
# print $1 to stderr without EOL.
#################################
function print_stderr(){
 1>&2 echo -ne "$1";
};

###############################
# fatal error message and exit.
###############################
function fatal(){
 if [ -t 1 ] && [ -t 2 ]; then
  colorize_bold print_stderr "\n\n$2\n\n";
 else
  colorize_none print_stderr "\n\n$2\n\n";
 fi;
 exit $1;
};

#######################################
# allow other users to use fuse mounts.
#######################################
function allow_other_fuse(){
 if [ -n "$(cat /etc/fuse.conf | grep -P '^\s*#\s*user_allow_other\b')" ]; then
  sudo perl -p -i.bak -e 's/^\s*#\s*user_allow_other\b/user_allow_other/' /etc/fuse.conf;
  cat /etc/fuse.conf;
 fi;
 if [ -n "$(cat /etc/fuse.conf | grep -P '^\s*user_allow_other\b')" ]; then
  echo "Option user_allow_other is ON.";
 fi;
};

#########################
# get list of all groups.
#########################
function list_all_groups(){
 cat /etc/group | cut -d: -f1;
};

#########################################
### check and create group 'fuse' if need
#########################################
function check_group_fuse(){
 if [ -z "$(list_all_groups | grep 'fuse')" ]; then
  sudo groupadd fuse;
 fi;
 if [ -n "$(list_all_groups | grep 'fuse')" ]; then
  echo "Group 'fuse' exists.";
 fi;
};

################################################
### add user to group 'fuse' if one not in group
################################################
function ensure_user_fuse(){
 if [ -z "$(groups | grep 'fuse')" ]; then
  sudo usermod -aG fuse $USER;
 fi;
 if [ -n "$(groups | grep 'fuse')" ]; then
  echo "User $USER is member of group 'fuse'.";
 fi;
};

##########################################
# make directory and allow read/write into
##########################################
function ensure_dir(){
 if [ ! -d "$1" ]; then
  if sudo mkdir -p "$1" &&  sudo chown $2 "$1" && sudo chmod $3 "$1"; then
   echo "Directory $1 created.";
   return 0;
  else
   fatal 1 "Error: could not create $1";
  fi;
 fi;
 if [ -d "$1" ]; then
  echo "Directory $1 exists.";
 fi;
};


########################################
# wait until input termination chars $*.
########################################
function wait_termitation(){
 if [ $# -gt 0 ]; then
  local terminated=0;
  while [ $terminated -eq 0 ]; do
   local cinp="";
   if read -n 1 -t 10 cinp; then
    for term in $*; do
     if [ "$cinp" = "$term" ]; then
      let terminated=1;
      break;
     fi;
    done;
    echo -ne "\b";
   fi;
  done;
 fi;
};

############################
# mount crw-daq.su via sshfs
############################
function ssh_mount_crw_daq_su(){
 allow_other_fuse;
 check_group_fuse;
 ensure_user_fuse;
 ensure_dir /mnt/ssh            root:root 755;
 ensure_dir /mnt/ssh/crw-daq.su root:root 755;
 #############
 # mount sshfs
 #############
 if sudo sshfs -o allow_other $remote_account_url:$remote_dir $local_dir; then
  colorize_bold echo "Connection eastablished.";
  cat /etc/mtab | grep 'ssh'; ls $local_dir;
  colorize_bold echo "Press Q/E/D to Quit/Exit/Disconnect.";
  wait_termitation Q E D; echo ""; sleep 2;
  sudo umount "$local_dir";
  if [ "$(ls $local_dir | wc -w)" -eq 0 ]; then
   colorize_bold echo "Disconnected.";
  else
   colorize_bold echo "Still connected. Should be unmounted later.";
  fi;
 else
  fatal 1 "Error: could not connect $remote_account_url.";
 fi;
 colorize_bold echo -ne "\n\n Done. \n\n"
};

###############
# main function
###############
function main(){
 ssh_mount_crw_daq_su $*;
};

main "$@";

##############
## END OF FILE
##############
