////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020 Alexey Kuryakin kouriakine@mail.ru under LGPL license.  //
////////////////////////////////////////////////////////////////////////////////

unit dpVbs; // Diesel Pascal VBScript - style constants and functions.

interface

uses dpCmdArgs,dpSystem,dpSysUtils;

// Environment variables.
function WshGetEnv(Name:String):String;
function WshSetEnv(Name,Value:String):Boolean;
function WshExpEnv(Name:String):String;
function WshSpecialFolders(Name:String):String;

// Registry routines.
// Name is KEY\Path\Param or KEY\Path\ for default param.
// KEY is HKCU=HKEY_CURRENT_USER, HKLM=HKEY_LOCAL_MACHINE,
// HKCR=HKEY_CLASSES_ROOT, HKU=HKEY_USERS, HKCC=HKEY_CURRENT_CONFIG
// DataType:'REG_SZ','REG_EXPAND_SZ','REG_DWORD','REG_BINARY'.
function WshRegRead(Name:String):Variant;
function WshRegWrite(Name:String; Value:Variant; DataType:String):Boolean;
function WshRegDelete(Name:String):Boolean;

// Processes and execution.
function WshAppActivate(Title:String):Boolean;
function WshRun(CmdLine:String; Mode:Integer; Wait:Boolean):Integer;
function WmiCountProcessesByPidExe(Pid:Cardinal; Exe:String):Integer;

// Show popup window with Text, Title, timeout (SecondsToWait) and Mode defined buttons.
function WshPopup(Text:String; SecondsToWait:Integer; Title:String; Mode:Integer):Integer;

const                       // Popup constants:
 vbOKOnly           = 0;    // Выводится кнопка ОК.
 vbOKCancel         = 1;    // Выводятся кнопки ОК и Отмена (Cancel).
 vbAbortRetryIgnore = 2;    // Выводятся кнопки Стоп (Abort), Повтор (Retry) и Пропустить (Ignore).
 vbYesNoCancel      = 3;    // Выводятся кнопки Да (Yes), Нет (No) и Отмена (Cancel).
 vbYesNo            = 4;    // Выводятся кнопки Да (Yes) и Нет (No).
 vbRetryCancel      = 5;    // Выводятся кнопки Повтор (Retry) и Отмена (Cancel).
 vbCritical         = 16;   // Выводится значок Stop Mark.
 vbQuestion         = 32;   // Выводится значок Question Mark.
 vbExclamation      = 48;   // Выводится значок Exclamation Mark.
 vbInformation      = 64;   // Выводится значок Information Mark.
 vbDefaultButton1   = 0;    // По умолчанию в окне выбирается первая кнопка.
 vbDefaultButton2   = 256;  // По умолчанию в окне выбирается вторая кнопка.
 vbDefaultButton3   = 512;  // По умолчанию в окне выбирается третья кнопка.
 vbDefaultButton4   = 768;  // По умолчанию в окне выбирается четвёртая кнопка.
 vbApplicationModal = 0;    // Диалоговое окно выводится в модальном режиме.
 vbSystemModal      = 4096; // Диалоговое окно выводится в модальном режиме и располагается сверху всех запускаемых приложений.
 vbOK               = 1;    // Нажата кнопка ОК.
 vbCancel           = 2;    // Нажата кнопка Отмена (Cancel).
 vbAbort            = 3;    // Нажата кнопка Стоп (Abort).
 vbRetry            = 4;    // Нажата кнопка Повтор (Retry).
 vbIgnore           = 5;    // Нажата кнопка Пропустить (Ignore).
 vbYes              = 6;    // Нажата кнопка Да (Yes).
 vbNo               = 7;    // Нажата кнопка Нет (No).
 vbTimeout          = -1;   // Popup timeout

function vbCr:String;       // Chr(13) Возврат каретки.
function vbLf:String;       // Chr(10) Перевод строки.
function vbTab:String;      // Chr(9) Символ табуляции.
function vbFormFeed:String; // Chr(12) Перевод страницы.
function vbNullChar:String; // Chr(0) Символ с нулевым кодом.
function vbVerticalTab:String; // Chr(11) Символ вертикальной табуляции.
function VbCrLf:String;     // Chr(13) & Chr(10) Возврат каретки и перевод строки.
function vbNewLine:String;  // Chr(13) & Chr(10) or Chr(10) Определяемый платформой (подходящий) символ новой строки.

implementation

function WshGetEnv(Name:String):String;
var WshShell:Variant;
begin
 Result:='';
 if IsWindows then
 if (Name<>'') then
 try
  WshShell:=CreateOleObject('WScript.Shell');
  try
   Result:=WshShell.Environment('Process').Item(Name);
  finally
   WshShell:=Unassigned;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'WshGetEnv');
   Result:='';
  end;
 end;
end;

function WshSetEnv(Name,Value:String):Boolean;
var WshShell:Variant;
begin
 Result:=false;
 if IsWindows then
 if (Name<>'') then
 try
  WshShell:=CreateOleObject('WScript.Shell');
  try
   WshShell.Environment('Process').Item(Name):=Value;
   Result:=true;
  finally
   WshShell:=Unassigned;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'WshSetEnv');
   Result:=false;
  end;
 end;
end;

function WshExpEnv(Name:String):String;
var WshShell:Variant;
begin
 Result:='';
 if IsWindows then
 if (Name<>'') then
 try
  WshShell:=CreateOleObject('WScript.Shell');
  try
   Result:=WshShell.ExpandEnvironmentStrings(Name);
  finally
   WshShell:=Unassigned;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'WshExpEnv');
   Result:='';
  end;
 end;
end;

function WshSpecialFolders(Name:String):String;
var WshShell:Variant;
begin
 Result:='';
 if IsWindows then
 if (Name<>'') then
 try
  WshShell:=CreateOleObject('WScript.Shell');
  try
   Result:=WshShell.SpecialFolders(Name);
  finally
   WshShell:=Unassigned;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'WshSpecialFolders');
   Result:='';
  end;
 end;
end;

function WshRegRead(Name:String):Variant;
var WshShell:Variant;
begin
 Result:=Unassigned;
 if IsWindows then
 if (Name<>'') then
 try
  WshShell:=CreateOleObject('WScript.Shell');
  try
   Result:=WshShell.RegRead(Name);
  finally
   WshShell:=Unassigned;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'WshRegRead');
   Result:=Unassigned;
  end;
 end;
end;

function WshRegWrite(Name:String; Value:Variant; DataType:String):Boolean;
var WshShell:Variant;
begin
 Result:=false;
 if IsWindows then
 if (Name<>'') then
 try
  WshShell:=CreateOleObject('WScript.Shell');
  try
   WshShell.RegWrite(Name,Value,DataType);
   Result:=true;
  finally
   WshShell:=Unassigned;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'WshRegWrite');
   Result:=false;
  end;
 end;
end;

function WshRegDelete(Name:String):Boolean;
var WshShell:Variant;
begin
 Result:=false;
 if IsWindows then
 if (Name<>'') then
 try
  WshShell:=CreateOleObject('WScript.Shell');
  try
   WshShell.RegDelete(Name);
   Result:=true;
  finally
   WshShell:=Unassigned;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'WshRegDelete');
   Result:=false;
  end;
 end;
end;

function WshAppActivate(Title:String):Boolean;
var WshShell:Variant;
begin
 Result:=false;
 if IsWindows then
 if (Title<>'') then
 try
  WshShell:=CreateOleObject('WScript.Shell');
  try
   Result:=WshShell.AppActivate(Title);
  finally
   WshShell:=Unassigned;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'WshAppActivate');
   Result:=false;
  end;
 end;
end;

function WshRun(CmdLine:String; Mode:Integer; Wait:Boolean):Integer;
var WshShell:Variant;
begin
 Result:=-1;
 if IsWindows then
 if (CmdLine<>'') then
 try
  WshShell:=CreateOleObject('WScript.Shell');
  try
   Result:=WshShell.Run(CmdLine,Mode,Wait);
  finally
   WshShell:=Unassigned;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'WshRun');
   Result:=-1;
  end;
 end;
end;

function WmiCountProcessesByPidExe(Pid:Cardinal; Exe:String):Integer;
var objLocator,objWMIService,colItems:Variant; Query:String;
begin
 Result:=0;
 if IsWindows then
 if (Pid<>0) and (Exe<>'') then
 try
  objLocator:=CreateOleObject('WbemScripting.SWbemLocator');
  try
   objWMIService:=objLocator.ConnectServer('.','root\cimv2');
   try
    Query:='Select * from Win32_Process';
    Query:=Query+' where ProcessID = '+AnsiQuotedStr(IntToStr(Pid),'"');
    Query:=Query+' and Name = '+AnsiQuotedStr(Exe,'"');
    colItems:=objWMIService.ExecQuery(Query);
    try
     Result:=colItems.Count;
    finally
     colItems:=Unassigned;
    end;
   finally
    objWMIService:=Unassigned;
   end;
  finally
   objLocator:=Unassigned;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'WmiCountProcessesByPidExe');
   Result:=0;
  end;
 end;
end;

function WshPopup(Text:String; SecondsToWait:Integer; Title:String; Mode:Integer):Integer;
var WshShell:Variant;
begin
 Result:=-1;
 try
  WshShell:=CreateOleObject('WScript.Shell');
  try
   Result:=WshShell.Popup(Text,SecondsToWait,Title,Mode);
  finally
   WshShell:=Unassigned;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'WshPopup');
   Result:=-1;
  end;
 end;
end;

function vbCr:String;          begin Result:=#13; end;
function vbLf:String;          begin Result:=#10; end;
function vbTab:String;         begin Result:=#9;  end;
function vbFormFeed:String;    begin Result:=#12; end;
function vbNullChar:String;    begin Result:=#0; end;
function vbVerticalTab:String; begin Result:=#11; end;
function VbCrLf:String;        begin Result:=#13+#10; end;
function vbNewLine:String;     begin Result:=LineEnding; end;

end.
