////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020 Alexey Kuryakin kouriakine@mail.ru under LGPL license.  //
////////////////////////////////////////////////////////////////////////////////

unit dpCrwDaqSys; // Diesel Pascal CRW-DAQ specific functions.

interface

uses dpCmdArgs,dpSystem,dpSysUtils,dpVbs,dpWindows;

const // List of CRW-DAQ environment variables.
 CRW_DAQ_ENV_VAR_NAMES = 'CRW_DAQ_CONFIG_BASE_TIME,CRW_DAQ_CONFIG_DATA_PATH,'
 +'CRW_DAQ_CONFIG_FILE,CRW_DAQ_CONFIG_HOME_DIR,CRW_DAQ_CONFIG_LOAD_TIME,'
 +'CRW_DAQ_CONFIG_PATH,CRW_DAQ_CONFIG_START_TIME,CRW_DAQ_CONFIG_TEMP_PATH,'
 +'CRW_DAQ_CONFIG_TIME_UNITS,CRW_DAQ_INCLUDE_PATH,CRW_DAQ_SYS_EXE_FILE,'
 +'CRW_DAQ_SYS_EXE_PID,CRW_DAQ_SYS_EXE_TID,CRW_DAQ_SYS_HOME_DIR,'
 +'CRW_DAQ_SYS_INIT_PATH,CRW_DAQ_SYS_INI_FILE,CRW_DAQ_SYS_LANG,'
 +'CRW_DAQ_SYS_MESSAGE,CRW_DAQ_SYS_PATH,CRW_DAQ_SYS_VERSION,'
 +'CRW_DAQ_SYS_CLASS,CRW_DAQ_SYS_TITLE,CRW_DAQ_SYS_HANDLE';

// Read CRW-DAQ environment.
function CRW_DAQ_ENVIRONMENT_LIST:TStringList; // List of CRW-DAQ variables
function CRW_DAQ_READ_ENVIRONMENT:Boolean;     // Read CRW-DAQ variables
function CRW_DAQ_ENVIRONMENT_OK:Boolean;       // Valid?

// CRW-DAQ environment variables.
function CRW_DAQ_CONFIG_BASE_TIME  : Double;   // DAQ System base time
function CRW_DAQ_CONFIG_DATA_PATH  : String;   // DAQ System DATA directory
function CRW_DAQ_CONFIG_FILE       : String;   // DAQ System main config file
function CRW_DAQ_CONFIG_HOME_DIR   : String;   // DAQ System config directory
function CRW_DAQ_CONFIG_LOAD_TIME  : Double;   // DAQ System load time
function CRW_DAQ_CONFIG_PATH       : String;   // DAQ System search path list
function CRW_DAQ_CONFIG_START_TIME : Double;   // DAQ System start time
function CRW_DAQ_CONFIG_TEMP_PATH  : String;   // DAQ System TEMP directory
function CRW_DAQ_CONFIG_TIME_UNITS : Double;   // DAQ System time units
function CRW_DAQ_INCLUDE_PATH      : String;   // DAQ System INCLUDE path list
function CRW_DAQ_SYS_EXE_FILE      : String;   // CRW-DAQ filename
function CRW_DAQ_SYS_EXE_PID       : DWORD;    // CRW-DAQ ProcessID
function CRW_DAQ_SYS_EXE_TID       : DWORD;    // CRW-DAQ ThreadID
function CRW_DAQ_SYS_HOME_DIR      : String;   // CRW-DAQ directory
function CRW_DAQ_SYS_INIT_PATH     : String;   // Initial PATH at CRW-DAQ start
function CRW_DAQ_SYS_INI_FILE      : String;   // CRW-DAQ main INI filename
function CRW_DAQ_SYS_LANG          : String;   // CRW-DAQ language params
function CRW_DAQ_SYS_MESSAGE       : String;   // System message ID
function CRW_DAQ_SYS_PATH          : String;   // CRW-DAQ search path
function CRW_DAQ_SYS_CLASS         : String;   // CRW-DAQ window class
function CRW_DAQ_SYS_TITLE         : String;   // CRW-DAQ window title
function CRW_DAQ_SYS_HANDLE        : DWORD;    // CRW-DAQ window handle
function CRW_DAQ_SYS_VERSION       : String;   // CRW-DAQ version
function CRW_DAQ_SYS_LANGUAGE      : String;   // UNKNOWN/ENGLISH/RUSSIAN
function CRW_DAQ_SYS_LANGUAGEID    : Integer;  // lng_XXX language ID

function IsValidHandle(h:THandle):Boolean;
function OpenProcessByPid(aPid:DWORD):THandle;
function ProcessIsRunning(hProcess:THandle):Boolean;

procedure OpenParentCrwDaqProcess;
procedure CloseParentCrwDaqProcess;
function  ParentCrwDaqProcessHandle:THandle;

// Parent CRW-DAQ window Title.
function ParentCrwDaqTitle:String;

// Parent CRW-DAQ is running?
function ParentCrwDaqIsRunning(Mode:Integer=0):Boolean;

// Get/set right CRW-DAQ child title.
function GetCrwDaqChildTitle:String;
function SetCrwDaqChildTitle(Title:String):String;

// Constants uses by Send2CrwDaq function to send WM_COPYDATA.
const guidTFormCrw32_Hex = 'DD26786B81B2A94BB32FAF78A24DEDA6';
const guidSysConsole_Hex = '3AA1D831607962419158272335902CF6';
const guidTFormCrw32_Str = '{6B7826DD-B281-4BA9-B32F-AF78A24DEDA6}';
const guidSysConsole_Str = '{31D8A13A-7960-4162-9158-272335902CF6}';

// Find main CRW-DAQ window handle by Pid
function FindCrwDaqWindowHandle(Pid:Cardinal=0):HWND;

// Send Data message to CRW-DAQ Main Console.
function Send2CrwDaq(Pid:Cardinal; Data:String; Sender:HWND):Integer;

implementation

var _CRW_DAQ_ENVIRONMENT_LIST_ : TStringList = nil;
var _CRW_DAQ_ENVIRONMENT_OK_   : Boolean     = false;
var _CRW_DAQ_SYS_EXE_PID_      : DWORD       = 0;
var _CRW_DAQ_SYS_EXE_TID_      : DWORD       = 0;
var _CRW_DAQ_SYS_HANDLE_       : DWORD       = 0;

function CRW_DAQ_ENVIRONMENT_LIST:TStringList;
begin
 if (_CRW_DAQ_ENVIRONMENT_LIST_=nil) then begin
 _CRW_DAQ_ENVIRONMENT_LIST_:=TStringList.Create;
  _CRW_DAQ_ENVIRONMENT_LIST_.Sorted:=true;
  CRW_DAQ_READ_ENVIRONMENT;
 end;
 Result:=_CRW_DAQ_ENVIRONMENT_LIST_;
end;

function CRW_DAQ_READ_ENVIRONMENT:Boolean;
var List:TStringList; i:Integer; Name,Value:String;
begin
 Result:=false;
 _CRW_DAQ_SYS_EXE_PID_:=0;
 _CRW_DAQ_SYS_EXE_TID_:=0;
 _CRW_DAQ_ENVIRONMENT_OK_:=false;
 if (_CRW_DAQ_ENVIRONMENT_LIST_=nil) then Exit;
 _CRW_DAQ_ENVIRONMENT_LIST_.Clear;
 try
  List:=TStringList.Create;
  try
   List.Sorted:=true;
   List.CommaText:=CRW_DAQ_ENV_VAR_NAMES;
   for i:=0 to List.Count-1 do begin
    Name:=Trim(List.Strings[i]);  if (Name='') then Continue;
    Value:=GetEnv(Name); if (Value='') then Continue;
    _CRW_DAQ_ENVIRONMENT_LIST_.Values[Name]:=Value;
   end;
   _CRW_DAQ_SYS_EXE_PID_ := StrToDWordDef(_CRW_DAQ_ENVIRONMENT_LIST_.Values['CRW_DAQ_SYS_EXE_PID'],0);
   _CRW_DAQ_SYS_EXE_TID_ := StrToDWordDef(_CRW_DAQ_ENVIRONMENT_LIST_.Values['CRW_DAQ_SYS_EXE_TID'],0);
   _CRW_DAQ_SYS_HANDLE_  := StrToDWordDef(_CRW_DAQ_ENVIRONMENT_LIST_.Values['CRW_DAQ_SYS_HANDLE'],0);
   LanguageId:=CRW_DAQ_SYS_LANGUAGEID;
   if (CRW_DAQ_SYS_EXE_FILE='') or not FileExists(CRW_DAQ_SYS_EXE_FILE) then Exit;
   if (CRW_DAQ_SYS_INI_FILE='') or not FileExists(CRW_DAQ_SYS_INI_FILE) then Exit;
   if (CRW_DAQ_SYS_HOME_DIR='') or not DirectoryExists(CRW_DAQ_SYS_HOME_DIR) then Exit;
   if (CRW_DAQ_SYS_EXE_PID=0) then Exit;
   if (CRW_DAQ_SYS_EXE_TID=0) then Exit;
   if (CRW_DAQ_SYS_MESSAGE='') then Exit;
   if (CRW_DAQ_SYS_VERSION='') then Exit;
   if (CRW_DAQ_SYS_HANDLE=0) then Exit;
   if (CRW_DAQ_SYS_TITLE='') then Exit;
   if (CRW_DAQ_SYS_CLASS='') then Exit;
   if (CRW_DAQ_SYS_PATH='') then Exit;
   if (CRW_DAQ_SYS_INIT_PATH='') then Exit;
   _CRW_DAQ_ENVIRONMENT_OK_:=true;
   Result:=true;
  finally
   List.Free;
  end;
 except
  on E:Exception do BugReport(E,nil,'CRW_DAQ_READ_ENVIRONMENT');
 end;
end;

function CRW_DAQ_ENVIRONMENT_OK:Boolean;
begin
 Result:=_CRW_DAQ_ENVIRONMENT_OK_;
end;

function CRW_DAQ_CONFIG_BASE_TIME:Double;
begin
 Result:=StrToFloatDef(CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_CONFIG_BASE_TIME'],0.0);
end;

function CRW_DAQ_CONFIG_DATA_PATH:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_CONFIG_DATA_PATH'];
end;

function CRW_DAQ_CONFIG_FILE:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_CONFIG_FILE'];
end;

function CRW_DAQ_CONFIG_HOME_DIR:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_CONFIG_HOME_DIR'];
end;

function CRW_DAQ_CONFIG_LOAD_TIME:Double;
begin
 Result:=StrToFloatDef(CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_CONFIG_LOAD_TIME'],0.0);
end;

function CRW_DAQ_CONFIG_PATH:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_CONFIG_PATH'];
end;

function CRW_DAQ_CONFIG_START_TIME:Double;
begin
 Result:=StrToFloatDef(CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_CONFIG_START_TIME'],0.0);
end;

function CRW_DAQ_CONFIG_TEMP_PATH:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_CONFIG_TEMP_PATH'];
end;

function CRW_DAQ_CONFIG_TIME_UNITS:Double;
begin
 Result:=StrToFloatDef(CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_CONFIG_TIME_UNITS'],0.0);
end;

function CRW_DAQ_INCLUDE_PATH:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_INCLUDE_PATH'];
end;

function CRW_DAQ_SYS_EXE_FILE:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_SYS_EXE_FILE'];
end;

function CRW_DAQ_SYS_EXE_PID:DWORD;
begin
 Result:=_CRW_DAQ_SYS_EXE_PID_;
end;

function CRW_DAQ_SYS_EXE_TID:DWORD;
begin
 Result:=_CRW_DAQ_SYS_EXE_TID_;
end;

function CRW_DAQ_SYS_INIT_PATH:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_SYS_INIT_PATH'];
end;

function CRW_DAQ_SYS_INI_FILE:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_SYS_INI_FILE'];
end;

function CRW_DAQ_SYS_HOME_DIR:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_SYS_HOME_DIR'];
end;

function CRW_DAQ_SYS_LANG:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_SYS_LANG'];
end;

function CRW_DAQ_SYS_MESSAGE:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_SYS_MESSAGE'];
end;

function CRW_DAQ_SYS_PATH:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_SYS_PATH'];
end;

function CRW_DAQ_SYS_CLASS:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_SYS_CLASS'];
end;

function CRW_DAQ_SYS_TITLE:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_SYS_TITLE'];
end;

function CRW_DAQ_SYS_HANDLE:DWORD;
begin
 Result:=_CRW_DAQ_SYS_HANDLE_;
end;

function CRW_DAQ_SYS_VERSION:String;
begin
 Result:=CRW_DAQ_ENVIRONMENT_LIST.Values['CRW_DAQ_SYS_VERSION'];
end;

function CRW_DAQ_SYS_LANGUAGE:String;
begin
 Result:=ExtractWord(1,CRW_DAQ_SYS_LANG,',;');
 if (Result='') then Result:='UNKNOWN';
end;

function CRW_DAQ_SYS_LANGUAGEID:Integer;
var S:String;
begin
 S:=CRW_DAQ_SYS_LANGUAGE;
 if SameText(S,'RUSSIAN') then Result:=lng_RUSSIAN else
 if SameText(S,'ENGLISH') then Result:=lng_ENGLISH else
 Result:=lng_UNKNOWN;
end;

var
 ChildTitle : String = 'Unknown';

function ParentCrwDaqTitle:String;
begin
 Result:=CRW_DAQ_SYS_TITLE;
end;

var
 nParentMode    : Integer = 0; // 0:Default; 1:PsAPI, 2:WMIC, 3:Win32API
 hParentProcess : THandle = 0; // Handle of parent CRW-DAQ process

function IsValidHandle(h:THandle):Boolean;
begin
 Result:=(h<>0) and (h<>INVALID_HANDLE_VALUE);
end;

function OpenProcessByPid(aPid:DWORD):THandle;
begin
 Result:=INVALID_HANDLE_VALUE;
 if IsWindows and (aPid<>0) and (aPid<>WinApi.GetCurrentProcessId) then
 try
  Result:=WinApi.OpenProcess(PROCESS_QUERY_INFORMATION or SYNCHRONIZE,False,aPid);
 except
  on E:Exception do BugReport(E,Application,'OpenProcessByPid');
 end;
end;

function ProcessIsRunning(hProcess:THandle):Boolean;
begin
 if IsValidHandle(hProcess)
 then Result:=(WinApi.WaitForSingleObject(hProcess,0)<>WAIT_OBJECT_0)
 else Result:=False;
end;

function ParentCrwDaqProcessHandle:THandle;
begin
 if not IsValidHandle(hParentProcess) then begin
  if ParentCrwDaqIsRunning(1) then begin
   hParentProcess:=OpenProcessByPid(CRW_DAQ_SYS_EXE_PID);
   if IsValidHandle(hParentProcess) then begin
    LogEvents(Now,FormatVar('Opened Parent PID %u',CRW_DAQ_SYS_EXE_PID)
                 +FormatVar(' with Handle %u',hParentProcess));
    nParentMode:=3;
   end;
  end;
 end;
 Result:=hParentProcess;
end;

procedure OpenParentCrwDaqProcess;
begin
 ParentCrwDaqProcessHandle;
end;

procedure CloseParentCrwDaqProcess;
begin
 try
  if IsValidHandle(hParentProcess) then begin
   LogEvents(Now,FormatVar('Closed Parent PID %u',CRW_DAQ_SYS_EXE_PID)
                +FormatVar(' with Handle %u',hParentProcess));
   WinApi.CloseHandle(hParentProcess);
  end;
  hParentProcess:=0;
  nParentMode:=0;
 except
  on E:Exception do BugReport(E,Application,'CloseParentCrwDaqProcess');;
 end;
end;

function ParentCrwDaqIsRunning(Mode:Integer=0):Boolean;
const CrwDaqExe='Crw32.exe';
begin
 Result:=false;
 if (Mode=0) then Mode:=nParentMode;
 if (Mode=3) then begin
  if not IsValidHandle(hParentProcess) then begin
   nParentMode:=0;
   Mode:=0;
  end;
 end;
 if (Mode<1) or (Mode>3) then Mode:=1;
 case Mode of
  1: Result:=(GetListOfProcesses(CRW_DAQ_SYS_EXE_PID,0,CrwDaqExe)<>'');
  2: Result:=WmiCountProcessesByPidExe(CRW_DAQ_SYS_EXE_PID,CrwDaqExe)>0;
  3: Result:=ProcessIsRunning(hParentProcess);
 end;
end;

function GetCrwDaqChildTitle:String;
begin
 if (CRW_DAQ_SYS_EXE_PID=0)
 then Result:=GetScriptName+' - '+ChildTitle
 else Result:=ParentCrwDaqTitle+' - '+ChildTitle;
end;

function SetCrwDaqChildTitle(Title:String):String;
begin
 Title:=Trim(Title);
 if (Title<>'') then ChildTitle:=Title;
 Result:=GetCrwDaqChildTitle;
end;

function IsValidToSend(Data:String):Boolean;
begin
 Result:=(Data<>'');
end;

function ConvertToHex(Data:String):String;
begin
 Result:=Utf8ToAnsiToHex(Data);
 //Result:=RecodeStr(Data,'Utf8,Ansi,Hex');
end;

function Send2CrwDaq(Pid:Cardinal; Data:String; Sender:HWND):Integer;
var CmdLine,Dir,Txt,Opt,Head:String; Target:HWND;
begin
 Result:=-1;
 Data:=Trim(Data);
 if (Data<>'') then
 try
  if (Pid=0) then Pid:=CRW_DAQ_SYS_EXE_PID;
  if (Pid=0) then Exit; // Undefined target
  Head:=guidTFormCrw32_Hex+guidSysConsole_Hex+VarToHex(GetProcessId);
  Target:=FindCrwDaqWindowHandle(Pid);
  //LogEvents(Now,FormatVar('Send2CrwDaq Target Window %d',Target));
  if (Target<>0) and (Length(Head)=72) and IsValidToSend(Data) then begin
   if (WinApi.SendWmCopyData(Target,Sender,Pid,Head+ConvertToHex(Data+LineEnding),true)>0)
   then Result:=0 else Result:=1;
   Exit;
  end;
  // Fallback method uses unix send2crwdaq.exe utiluty
  Dir:=AddPathDelim(GetEnv('Temp'))+'VisualTech\DieselPascal\Temporary';
  if not DirectoryExists(Dir) then MakeDir(Dir);
  Txt:=AddPathDelim(Dir)+'Message_'+IntToStr(GetTickCount)+'.txt';
  if DirectoryExists(Dir) then
  if FileWriteAsText(Txt,Data+LineEnding,ioModeAnsi) then
  try
   if (Pid>0) then Opt:=' -p '+IntToStr(Pid) else Opt:='';
   CmdLine:=AnsiQuotedStr(GetComSpec,'"')+' /c type '+AnsiQuotedStr(Txt,'"');
   CmdLine:=CmdLine+' | unix send2crwdaq'+Opt;
   Result:=WshRun(CmdLine,0,true);
  finally
   DeleteFile(Txt);
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'Send2CrwDaq');
   Result:=-1;
  end;
 end;
end;

var myCrwDaqSysWnd:HWND=0;

function CrwDaqHandleMatch(Pid:Cardinal; hWin:HWND):Boolean;
begin
 Result:=false;
 if (hWin=0) then Exit;
 if (Pid=0) then Pid:=CRW_DAQ_SYS_EXE_PID;
 if (Pid=0) then Exit; // Undefined target
 if not WinApi.IsWindow(hWin) then Exit;
 if (Pid<>CRW_DAQ_SYS_EXE_PID) then Exit;
 if (WinApi.GetWindowProcessId(hWin)<>Pid) then Exit;
 if (WinApi.GetClassName(hWin)<>CRW_DAQ_SYS_CLASS) then Exit;
 if (WinApi.GetWindowText(hWin)<>CRW_DAQ_SYS_TITLE) then Exit;
 Result:=true;
end;

function FindCrwDaqWindowHandle(Pid:Cardinal=0):HWND;
var wClass,wTitle:String; hWin:HWND;
begin
 Result:=0;
 if (Pid=0) then Pid:=CRW_DAQ_SYS_EXE_PID;
 if (Pid=0) then Exit; // Undefined target
 if (Pid=CRW_DAQ_SYS_EXE_PID) then begin // It's parent PID
  // Check if previousely saved handle myCrwDaqSysWnd is Ok
  if CrwDaqHandleMatch(Pid,myCrwDaqSysWnd) then begin
   Result:=myCrwDaqSysWnd; // Handle is Ok
   Exit;
  end;
  hWin:=CRW_DAQ_SYS_HANDLE;
  if CrwDaqHandleMatch(Pid,hWin) then begin
   myCrwDaqSysWnd:=hWin; // Save for future
   Result:=hWin;         // Handle is Ok
   Exit;
  end;
  if (CRW_DAQ_SYS_CLASS<>'') and (CRW_DAQ_SYS_TITLE<>'') then
  Result:=WinApi.FindWindow(CRW_DAQ_SYS_CLASS,CRW_DAQ_SYS_TITLE);
  if (WinApi.GetWindowProcessId(Result)<>Pid) then Result:=0;
  if (Result<>0) then myCrwDaqSysWnd:=Result; // For future
  if (Result<>0) then Exit;
 end;
 // Fallback procedure
 wTitle:=FormatVar('CRW-DAQ/%d@',Pid)+GetEnv('ComputerName');
 wClass:=CRW_DAQ_SYS_CLASS; if (wClass='') then wClass:='TFormCrw32';
 Result:=StrToIntDef(ExtractWord(1,WinApi.GetListOfWindows(Pid,wClass,wTitle),', '),0);
end;

initialization

 ChildTitle:=GetScriptName;
 CRW_DAQ_ENVIRONMENT_LIST;

end.
