////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020 Alexey Kuryakin kouriakine@mail.ru under LGPL license.  //
////////////////////////////////////////////////////////////////////////////////

unit dpColors; // Diesel Pascal named color tables and routines.

interface

uses dpSystem,dpSysUtils;

 // Convert ColorCode to hex $GGBBRR code of RGB
function ColorToHexString(Color:TColor):String;

 // Convert ColorCode to known color name or $GGBBRR code of RGB
function ColorToString(ColorCode:TColor; Prefix:String=''):String;

 // Convert ColorName (known color name or $GGBBRR code) to color code
 // Return DefaultColor if convertion failed
function StringToColor(ColorName:String; DefaultColor:TColor=clNone):TColor;

 // Find nearest known color by given color.
function FindNearestKnownColor(aColor:TColor; Def:TColor=clDefault):TColor;

 // List of known colors: 0:All,1:ByName,2:ByCode,3:ByName(),4:ByCode()
function GetListOfKnownColors(Mode:Integer=0):String;

procedure InitKnownColorsList; // Initialize list of known colors
procedure FreeKnownColorsList; // Finalize   list of known colors

var KnownColorsList       : TStringList = nil; // List of known colors (names & codes)
var KnownColorsListByName : TStringList = nil; // List of known colors indexed by name
var KnownColorsListByCode : TStringList = nil; // List of known colors indexed by code

var DefaultFaceColor : TColor = $F0F0F0;       // BtnFace for W10LTSC = gray94
var DefaultTextColor : TColor = $000000;       // BtnText for W10LTSC = black

implementation

function ColorToHexString(Color:TColor):String;
begin
 Result:=FormatVar('$%.6x',Color);
end;

function ColorToString(ColorCode:TColor; Prefix:String=''):String;
var i:Integer;
begin
 Result:=ColorToHexString(ColorCode);
 if (KnownColorsListByCode<>nil) then
 try
  i:=KnownColorsListByCode.IndexOf(Result);
  if (i>=0) then begin
   i:=Integer(KnownColorsListByCode.Objects[i])-1;
   if (i>=0) and (i<=KnownColorsList.Count-1)
   then Result:=Prefix+KnownColorsList.Strings[i];
  end;
 except
  on E:Exception do BugReport(E,nil,'ColorToString');
 end;
end;

function StringToColor(ColorName:String; DefaultColor:TColor=clNone):TColor;
var i:Integer;
begin
 Result:=DefaultColor;
 ColorName:=Trim(ColorName);
 if (ColorName<>'') then
 try
  if (KnownColorsListByName<>nil) then begin
   i:=KnownColorsListByName.IndexOf(ColorName);
   if (i>=0) then begin
    Result:=TColor(KnownColorsListByName.Objects[i]);
    Exit;
   end;
  end;
  Result:=StrToIntDef(ColorName,DefaultColor);
 except
  on E:Exception do BugReport(E,nil,'StringToColor');
 end;
end;

function FindNearestKnownColor(aColor:TColor; Def:TColor=clDefault):TColor;
var i,d0,d1,d2,dd:Integer; Color,a,b:TColor; Diff,MinDiff:Double;
begin
 Result:=aColor;
 if (KnownColorsList<>nil) then
 try
  Result:=Def; MinDiff:=MaxInt;
  for i:=0 to KnownColorsList.Count-1 do begin
   Color:=TColor(KnownColorsList.Objects[i]);
   a:=Color; b:=aColor;
   if (a=b) then begin Result:=Color; Break; end;
   d0:=(a and $FF)-(b and $FF); a:=a shr 8; b:=b shr 8;
   d1:=(a and $FF)-(b and $FF); a:=a shr 8; b:=b shr 8;
   d2:=(a and $FF)-(b and $FF); dd:=d0*d0+d1*d1+d2*d2;
   Diff:=sqrt(dd);
   if (Diff<MinDiff) then begin
    Result:=Color;
    MinDiff:=Diff;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'FindNearestKnownColor');
 end;
end;

function GetListOfKnownColors(Mode:Integer=0):String;
var List:TStringList; Line,Name:String; Color:TColor; i:Integer;
begin
 Result:='';
 if (KnownColorsList<>nil) then
 try
  List:=TStringList.Create;
  try
   case Mode of
    0:
    for i:=0 to KnownColorsList.Count-1 do begin
     Line:=KnownColorsList.Strings[i];
     Color:=TColor(KnownColorsList.Objects[i]);
     Line:=Line+' = '+ColorToHexString(Color);
     List.Add(Line);
    end;
    1:
    for i:=0 to KnownColorsListByName.Count-1 do begin
     Line:=KnownColorsListByName.Strings[i];
     Color:=TColor(KnownColorsListByName.Objects[i]);
     Line:=Line+' = '+ColorToHexString(Color);
     List.Add(Line);
    end;
    2:
    for i:=0 to KnownColorsListByCode.Count-1 do begin
     Line:=KnownColorsListByCode.Strings[i];
     Name:=ColorToString(StrToIntDef(Line,clNone));
     if Pos('$',Name)>0 then Name:='';
     Line:=Line+' = '+Name;
     List.Add(Line);
    end;
    3:
    for i:=0 to KnownColorsListByName.Count-1 do begin
     Line:=KnownColorsListByName.Strings[i];
     Color:=StringToColor(Line,clNone);
     if (Color<>clNone) then Line:=Line+' = '+ColorToHexString(Color);
     List.Add(Line);
    end;
    4:
    for i:=0 to KnownColorsListByCode.Count-1 do begin
     Line:=KnownColorsListByCode.Strings[i];
     Name:=ColorToString(StrToIntDef(Line,clNone));
     Line:=Line+' = '+Name;
     List.Add(Line);
    end;
   end;
   Result:=List.Text;
  finally
   List.Free;
  end;
 except
  on E:Exception do BugReport(E,nil,'GetListOfKnownColors');
 end;
end;

procedure InitKnownColorsList;
var i,k:Integer; n,h:String; c:TColor;
 procedure AddColor(Name:String; Color:TColor);
 begin
  if (KnownColorsList<>nil) then begin
   KnownColorsList.AddObject(Name,TObject(Color));
  end;
 end;
begin
 try
  if (KnownColorsList=nil) then begin
   KnownColorsList:=TStringList.Create;
   KnownColorsList.Sorted:=false;
   //
   // Vga colors
   //
   AddColor('Black',$000000);
   AddColor('Maroon',$000080);
   AddColor('Green',$008000);
   AddColor('Olive',$008080);
   AddColor('Navy',$800000);
   AddColor('Purple',$800080);
   AddColor('Teal',$808000);
   AddColor('Gray',$808080);
   AddColor('Silver',$C0C0C0);
   AddColor('Red',$0000FF);
   AddColor('Lime',$00FF00);
   AddColor('Yellow',$00FFFF);
   AddColor('Blue',$FF0000);
   AddColor('Fuchsia',$FF00FF);
   AddColor('Aqua',$FFFF00);
   AddColor('White',$FFFFFF);
   //
   // ColWin.crc
   //
   AddColor('LtGray',$C0C0C0);
   AddColor('DkGray',$808080);
   AddColor('LightGray',$C0C0C0);
   AddColor('DarkGray',$808080);
   //
   // Windows system colors
   //////////////////////////////////////////////////////////////////////// Win-XP W10LTSC
   //AddColor('ScrollBar',           ColorToRgb(clScrollBar));           // $C8D0D4 $C8C8C8
   //AddColor('Background',          ColorToRgb(clBackground));          // $000000 $000000
   //AddColor('ActiveCaption',       ColorToRgb(clActiveCaption));       // $6A240A $D1B499
   //AddColor('InactiveCaption',     ColorToRgb(clInactiveCaption));     // $808080 $DBCDBF
   //AddColor('Menu',                ColorToRgb(clMenu));                // $C8D0D4 $F0F0F0
   //AddColor('Window',              ColorToRgb(clWindow));              // $FFFFFF $FFFFFF
   //AddColor('WindowFrame',         ColorToRgb(clWindowFrame));         // $000000 $646464
   //AddColor('MenuText',            ColorToRgb(clMenuText));            // $000000 $000000
   //AddColor('WindowText',          ColorToRgb(clWindowText));          // $000000 $000000
   //AddColor('CaptionText',         ColorToRgb(clCaptionText));         // $FFFFFF $000000
   //AddColor('ActiveBorder',        ColorToRgb(clActiveBorder));        // $C8D0D4 $B4B4B4
   //AddColor('InactiveBorder',      ColorToRgb(clInactiveBorder));      // $C8D0D4 $FCF7F4
   //AddColor('AppWorkSpace',        ColorToRgb(clAppWorkSpace));        // $808080 $ABABAB
   //AddColor('Highlight',           ColorToRgb(clHighlight));           // $6A240A $D77800
   //AddColor('HighlightText',       ColorToRgb(clHighlightText));       // $FFFFFF $FFFFFF
   //AddColor('BtnFace',             ColorToRgb(clBtnFace));             // $C8D0D4 $F0F0F0
   //AddColor('BtnShadow',           ColorToRgb(clBtnShadow));           // $808080 $A0A0A0
   //AddColor('GrayText',            ColorToRgb(clGrayText));            // $808080 $6D6D6D
   //AddColor('BtnText',             ColorToRgb(clBtnText));             // $000000 $000000
   //AddColor('InactiveCaptionText', ColorToRgb(clInactiveCaptionText)); // $C8D0D4 $000000
   //AddColor('BtnHighlight',        ColorToRgb(clBtnHighlight));        // $FFFFFF $FFFFFF
   //AddColor('3DDkShadow',          ColorToRgb(cl3DDkShadow));          // $404040 $696969
   //AddColor('3DLight',             ColorToRgb(cl3DLight));             // $C8D0D4 $E3E3E3
   //AddColor('InfoText',            ColorToRgb(clInfoText));            // $000000 $000000
   //AddColor('InfoBk',              ColorToRgb(clInfoBk));              // $E1FFFF $E1FFFF
   ////////////////////////////////////////////////////////////////////////////////////////
   //
   // ColWeb.crc
   //
   AddColor('IndianRed',$5C5CCD);
   AddColor('LightCoral',$8080F0);
   AddColor('Salmon',$7280FA);
   AddColor('DarkSalmon',$7A96E9);
   AddColor('LightSalmon',$7AA0FF);
   AddColor('Crimson',$3C14DC);
   AddColor('Red',$0000FF);
   AddColor('FireBrick',$2222B2);
   AddColor('DarkRed',$00008B);
   AddColor('Pink',$CBC0FF);
   AddColor('LightPink',$C1B6FF);
   AddColor('HotPink',$B469FF);
   AddColor('DeepPink',$9314FF);
   AddColor('MediumVioletRed',$8515C7);
   AddColor('PaleVioletRed',$9370DB);
   AddColor('LightSalmon',$7AA0FF);
   AddColor('Coral',$507FFF);
   AddColor('Tomato',$4763FF);
   AddColor('OrangeRed',$0045FF);
   AddColor('DarkOrange',$008CFF);
   AddColor('Orange',$00A5FF);
   AddColor('Gold',$00D7FF);
   AddColor('Yellow',$00FFFF);
   AddColor('LightYellow',$E0FFFF);
   AddColor('LemonChiffon',$CDFAFF);
   AddColor('LightGoldenrodYellow',$D2FAFA);
   AddColor('PapayaWhip',$D5EFFF);
   AddColor('Moccasin',$B5E4FF);
   AddColor('PeachPuff',$B9DAFF);
   AddColor('PaleGoldenrod',$AAE8EE);
   AddColor('Khaki',$8CE6F0);
   AddColor('DarkKhaki',$6BB7BD);
   AddColor('Lavender',$FAE6E6);
   AddColor('Thistle',$D8BFD8);
   AddColor('Plum',$DDA0DD);
   AddColor('Violet',$EE82EE);
   AddColor('Orchid',$D670DA);
   AddColor('Fuchsia',$FF00FF);
   AddColor('Magenta',$FF00FF);
   AddColor('MediumOrchid',$D355BA);
   AddColor('MediumPurple',$DB7093);
   AddColor('Amethyst',$CC6699);
   AddColor('BlueViolet',$E22B8A);
   AddColor('DarkViolet',$D30094);
   AddColor('DarkOrchid',$CC3299);
   AddColor('DarkMagenta',$8B008B);
   AddColor('Purple',$800080);
   AddColor('Indigo',$82004B);
   AddColor('SlateBlue',$CD5A6A);
   AddColor('DarkSlateBlue',$8B3D48);
   AddColor('MediumSlateBlue',$EE687B);
   AddColor('RebeccaPurple',$993366);
   AddColor('GreenYellow',$2FFFAD);
   AddColor('Chartreuse',$00FF7F);
   AddColor('LawnGreen',$00FC7C);
   AddColor('Lime',$00FF00);
   AddColor('LimeGreen',$32CD32);
   AddColor('PaleGreen',$98FB98);
   AddColor('LightGreen',$90EE90);
   AddColor('MediumSpringGreen',$9AFA00);
   AddColor('SpringGreen',$7FFF00);
   AddColor('MediumSeaGreen',$71B33C);
   AddColor('SeaGreen',$578B2E);
   AddColor('ForestGreen',$228B22);
   AddColor('Green',$008000);
   AddColor('DarkGreen',$006400);
   AddColor('YellowGreen',$32CD9A);
   AddColor('OliveDrab',$238E6B);
   AddColor('Olive',$008080);
   AddColor('DarkOliveGreen',$2F6B55);
   AddColor('MediumAquamarine',$AACD66);
   AddColor('DarkSeaGreen',$8FBC8F);
   AddColor('LightSeaGreen',$AAB220);
   AddColor('DarkCyan',$8B8B00);
   AddColor('Teal',$808000);
   AddColor('Aqua',$FFFF00);
   AddColor('Cyan',$FFFF00);
   AddColor('LightCyan',$FFFFE0);
   AddColor('PaleTurquoise',$EEEEAF);
   AddColor('Aquamarine',$D4FF7F);
   AddColor('Turquoise',$D0E040);
   AddColor('MediumTurquoise',$CCD148);
   AddColor('DarkTurquoise',$D1CE00);
   AddColor('CadetBlue',$A09E5F);
   AddColor('SteelBlue',$B48246);
   AddColor('LightSteelBlue',$DEC4B0);
   AddColor('PowderBlue',$E6E0B0);
   AddColor('LightBlue',$E6D8AD);
   AddColor('SkyBlue',$EBCE87);
   AddColor('LightSkyBlue',$FACE87);
   AddColor('DeepSkyBlue',$FFBF00);
   AddColor('DodgerBlue',$FF901E);
   AddColor('CornflowerBlue',$ED9564);
   AddColor('MediumSlateBlue',$EE687B);
   AddColor('RoyalBlue',$E16941);
   AddColor('Blue',$FF0000);
   AddColor('MediumBlue',$CD0000);
   AddColor('DarkBlue',$8B0000);
   AddColor('Navy',$800000);
   AddColor('MidnightBlue',$701919);
   AddColor('Cornsilk',$DCF8FF);
   AddColor('BlanchedAlmond',$CDEBFF);
   AddColor('Bisque',$C4E4FF);
   AddColor('NavajoWhite',$ADDEFF);
   AddColor('Wheat',$B3DEF5);
   AddColor('BurlyWood',$87B8DE);
   AddColor('Tan',$8CB4D2);
   AddColor('RosyBrown',$8F8FBC);
   AddColor('SandyBrown',$60A4F4);
   AddColor('Goldenrod',$20A5DA);
   AddColor('DarkGoldenrod',$0B86B8);
   AddColor('Peru',$3F85CD);
   AddColor('Chocolate',$1E69D2);
   AddColor('SaddleBrown',$13458B);
   AddColor('Sienna',$2D52A0);
   AddColor('Brown',$2A2AA5);
   AddColor('Maroon',$000080);
   AddColor('White',$FFFFFF);
   AddColor('Snow',$FAFAFF);
   AddColor('Honeydew',$F0FFF0);
   AddColor('MintCream',$FAFFF5);
   AddColor('Azure',$FFFFF0);
   AddColor('AliceBlue',$FFF8F0);
   AddColor('GhostWhite',$FFF8F8);
   AddColor('WhiteSmoke',$F5F5F5);
   AddColor('Seashell',$EEF5FF);
   AddColor('Beige',$DCF5F5);
   AddColor('OldLace',$E6F5FD);
   AddColor('FloralWhite',$F0FAFF);
   AddColor('Ivory',$F0FFFF);
   AddColor('AntiqueWhite',$D7EBFA);
   AddColor('Linen',$E6F0FA);
   AddColor('LavenderBlush',$F5F0FF);
   AddColor('MistyRose',$E1E4FF);
   AddColor('Gainsboro',$DCDCDC);
   AddColor('LightGray',$D3D3D3);
   AddColor('LightGrey',$D3D3D3);
   AddColor('Silver',$C0C0C0);
   AddColor('DarkGray',$A9A9A9);
   AddColor('DarkGrey',$A9A9A9);
   AddColor('Gray',$808080);
   AddColor('Grey',$808080);
   AddColor('DimGray',$696969);
   AddColor('DimGrey',$696969);
   AddColor('LightSlateGray',$998877);
   AddColor('LightSlateGrey',$998877);
   AddColor('SlateGray',$908070);
   AddColor('SlateGrey',$908070);
   AddColor('DarkSlateGray',$4F4F2F);
   AddColor('DarkSlateGrey',$4F4F2F);
    //
    // ColX11.crc
    //
   AddColor('snow',$FAFAFF);
   AddColor('ghostwhite',$FFF8F8);
   AddColor('whitesmoke',$F5F5F5);
   AddColor('gainsboro',$DCDCDC);
   AddColor('floralwhite',$F0FAFF);
   AddColor('oldlace',$E6F5FD);
   AddColor('linen',$E6F0FA);
   AddColor('antiquewhite',$D7EBFA);
   AddColor('papayawhip',$D5EFFF);
   AddColor('blanchedalmond',$CDEBFF);
   AddColor('bisque',$C4E4FF);
   AddColor('peachpuff',$B9DAFF);
   AddColor('navajowhite',$ADDEFF);
   AddColor('moccasin',$B5E4FF);
   AddColor('cornsilk',$DCF8FF);
   AddColor('ivory',$F0FFFF);
   AddColor('lemonchiffon',$CDFAFF);
   AddColor('seashell',$EEF5FF);
   AddColor('honeydew',$F0FFF0);
   AddColor('mintcream',$FAFFF5);
   AddColor('azure',$FFFFF0);
   AddColor('aliceblue',$FFF8F0);
   AddColor('lavender',$FAE6E6);
   AddColor('lavenderblush',$F5F0FF);
   AddColor('mistyrose',$E1E4FF);
   AddColor('white',$FFFFFF);
   AddColor('black',$000000);
   AddColor('darkslategray',$4F4F2F);
   AddColor('darkslategrey',$4F4F2F);
   AddColor('dimgray',$696969);
   AddColor('dimgrey',$696969);
   AddColor('slategray',$908070);
   AddColor('slategrey',$908070);
   AddColor('lightslategray',$998877);
   AddColor('lightslategrey',$998877);
   AddColor('gray',$BEBEBE);
   AddColor('grey',$BEBEBE);
   AddColor('lightgrey',$D3D3D3);
   AddColor('lightgray',$D3D3D3);
   AddColor('midnightblue',$701919);
   AddColor('navy',$800000);
   AddColor('navyblue',$800000);
   AddColor('cornflowerblue',$ED9564);
   AddColor('darkslateblue',$8B3D48);
   AddColor('slateblue',$CD5A6A);
   AddColor('mediumslateblue',$EE687B);
   AddColor('lightslateblue',$FF7084);
   AddColor('mediumblue',$CD0000);
   AddColor('royalblue',$E16941);
   AddColor('blue',$FF0000);
   AddColor('dodgerblue',$FF901E);
   AddColor('deepskyblue',$FFBF00);
   AddColor('skyblue',$EBCE87);
   AddColor('lightskyblue',$FACE87);
   AddColor('steelblue',$B48246);
   AddColor('lightsteelblue',$DEC4B0);
   AddColor('lightblue',$E6D8AD);
   AddColor('powderblue',$E6E0B0);
   AddColor('paleturquoise',$EEEEAF);
   AddColor('darkturquoise',$D1CE00);
   AddColor('mediumturquoise',$CCD148);
   AddColor('turquoise',$D0E040);
   AddColor('cyan',$FFFF00);
   AddColor('lightcyan',$FFFFE0);
   AddColor('cadetblue',$A09E5F);
   AddColor('mediumaquamarine',$AACD66);
   AddColor('aquamarine',$D4FF7F);
   AddColor('darkgreen',$006400);
   AddColor('darkolivegreen',$2F6B55);
   AddColor('darkseagreen',$8FBC8F);
   AddColor('seagreen',$578B2E);
   AddColor('mediumseagreen',$71B33C);
   AddColor('lightseagreen',$AAB220);
   AddColor('palegreen',$98FB98);
   AddColor('springgreen',$7FFF00);
   AddColor('lawngreen',$00FC7C);
   AddColor('green',$00FF00);
   AddColor('chartreuse',$00FF7F);
   AddColor('mediumspringgreen',$9AFA00);
   AddColor('greenyellow',$2FFFAD);
   AddColor('limegreen',$32CD32);
   AddColor('yellowgreen',$32CD9A);
   AddColor('forestgreen',$228B22);
   AddColor('olivedrab',$238E6B);
   AddColor('darkkhaki',$6BB7BD);
   AddColor('khaki',$8CE6F0);
   AddColor('palegoldenrod',$AAE8EE);
   AddColor('lightgoldenrodyellow',$D2FAFA);
   AddColor('lightyellow',$E0FFFF);
   AddColor('yellow',$00FFFF);
   AddColor('gold',$00D7FF);
   AddColor('lightgoldenrod',$82DDEE);
   AddColor('goldenrod',$20A5DA);
   AddColor('darkgoldenrod',$0B86B8);
   AddColor('rosybrown',$8F8FBC);
   AddColor('indianred',$5C5CCD);
   AddColor('saddlebrown',$13458B);
   AddColor('sienna',$2D52A0);
   AddColor('peru',$3F85CD);
   AddColor('burlywood',$87B8DE);
   AddColor('beige',$DCF5F5);
   AddColor('wheat',$B3DEF5);
   AddColor('sandybrown',$60A4F4);
   AddColor('tan',$8CB4D2);
   AddColor('chocolate',$1E69D2);
   AddColor('firebrick',$2222B2);
   AddColor('brown',$2A2AA5);
   AddColor('darksalmon',$7A96E9);
   AddColor('salmon',$7280FA);
   AddColor('lightsalmon',$7AA0FF);
   AddColor('orange',$00A5FF);
   AddColor('darkorange',$008CFF);
   AddColor('coral',$507FFF);
   AddColor('lightcoral',$8080F0);
   AddColor('tomato',$4763FF);
   AddColor('orangered',$0045FF);
   AddColor('red',$0000FF);
   AddColor('hotpink',$B469FF);
   AddColor('deeppink',$9314FF);
   AddColor('pink',$CBC0FF);
   AddColor('lightpink',$C1B6FF);
   AddColor('palevioletred',$9370DB);
   AddColor('maroon',$6030B0);
   AddColor('mediumvioletred',$8515C7);
   AddColor('violetred',$9020D0);
   AddColor('magenta',$FF00FF);
   AddColor('violet',$EE82EE);
   AddColor('plum',$DDA0DD);
   AddColor('orchid',$D670DA);
   AddColor('mediumorchid',$D355BA);
   AddColor('darkorchid',$CC3299);
   AddColor('darkviolet',$D30094);
   AddColor('blueviolet',$E22B8A);
   AddColor('purple',$F020A0);
   AddColor('mediumpurple',$DB7093);
   AddColor('thistle',$D8BFD8);
   AddColor('snow1',$FAFAFF);
   AddColor('snow2',$E9E9EE);
   AddColor('snow3',$C9C9CD);
   AddColor('snow4',$89898B);
   AddColor('seashell1',$EEF5FF);
   AddColor('seashell2',$DEE5EE);
   AddColor('seashell3',$BFC5CD);
   AddColor('seashell4',$82868B);
   AddColor('AntiqueWhite1',$DBEFFF);
   AddColor('AntiqueWhite2',$CCDFEE);
   AddColor('AntiqueWhite3',$B0C0CD);
   AddColor('AntiqueWhite4',$78838B);
   AddColor('bisque1',$C4E4FF);
   AddColor('bisque2',$B7D5EE);
   AddColor('bisque3',$9EB7CD);
   AddColor('bisque4',$6B7D8B);
   AddColor('PeachPuff1',$B9DAFF);
   AddColor('PeachPuff2',$ADCBEE);
   AddColor('PeachPuff3',$95AFCD);
   AddColor('PeachPuff4',$65778B);
   AddColor('NavajoWhite1',$ADDEFF);
   AddColor('NavajoWhite2',$A1CFEE);
   AddColor('NavajoWhite3',$8BB3CD);
   AddColor('NavajoWhite4',$5E798B);
   AddColor('LemonChiffon1',$CDFAFF);
   AddColor('LemonChiffon2',$BFE9EE);
   AddColor('LemonChiffon3',$A5C9CD);
   AddColor('LemonChiffon4',$70898B);
   AddColor('cornsilk1',$DCF8FF);
   AddColor('cornsilk2',$CDE8EE);
   AddColor('cornsilk3',$B1C8CD);
   AddColor('cornsilk4',$78888B);
   AddColor('ivory1',$F0FFFF);
   AddColor('ivory2',$E0EEEE);
   AddColor('ivory3',$C1CDCD);
   AddColor('ivory4',$838B8B);
   AddColor('honeydew1',$F0FFF0);
   AddColor('honeydew2',$E0EEE0);
   AddColor('honeydew3',$C1CDC1);
   AddColor('honeydew4',$838B83);
   AddColor('LavenderBlush1',$F5F0FF);
   AddColor('LavenderBlush2',$E5E0EE);
   AddColor('LavenderBlush3',$C5C1CD);
   AddColor('LavenderBlush4',$86838B);
   AddColor('MistyRose1',$E1E4FF);
   AddColor('MistyRose2',$D2D5EE);
   AddColor('MistyRose3',$B5B7CD);
   AddColor('MistyRose4',$7B7D8B);
   AddColor('azure1',$FFFFF0);
   AddColor('azure2',$EEEEE0);
   AddColor('azure3',$CDCDC1);
   AddColor('azure4',$8B8B83);
   AddColor('SlateBlue1',$FF6F83);
   AddColor('SlateBlue2',$EE677A);
   AddColor('SlateBlue3',$CD5969);
   AddColor('SlateBlue4',$8B3C47);
   AddColor('RoyalBlue1',$FF7648);
   AddColor('RoyalBlue2',$EE6E43);
   AddColor('RoyalBlue3',$CD5F3A);
   AddColor('RoyalBlue4',$8B4027);
   AddColor('blue1',$FF0000);
   AddColor('blue2',$EE0000);
   AddColor('blue3',$CD0000);
   AddColor('blue4',$8B0000);
   AddColor('DodgerBlue1',$FF901E);
   AddColor('DodgerBlue2',$EE861C);
   AddColor('DodgerBlue3',$CD7418);
   AddColor('DodgerBlue4',$8B4E10);
   AddColor('SteelBlue1',$FFB863);
   AddColor('SteelBlue2',$EEAC5C);
   AddColor('SteelBlue3',$CD944F);
   AddColor('SteelBlue4',$8B6436);
   AddColor('DeepSkyBlue1',$FFBF00);
   AddColor('DeepSkyBlue2',$EEB200);
   AddColor('DeepSkyBlue3',$CD9A00);
   AddColor('DeepSkyBlue4',$8B6800);
   AddColor('SkyBlue1',$FFCE87);
   AddColor('SkyBlue2',$EEC07E);
   AddColor('SkyBlue3',$CDA66C);
   AddColor('SkyBlue4',$8B704A);
   AddColor('LightSkyBlue1',$FFE2B0);
   AddColor('LightSkyBlue2',$EED3A4);
   AddColor('LightSkyBlue3',$CDB68D);
   AddColor('LightSkyBlue4',$8B7B60);
   AddColor('SlateGray1',$FFE2C6);
   AddColor('SlateGray2',$EED3B9);
   AddColor('SlateGray3',$CDB69F);
   AddColor('SlateGray4',$8B7B6C);
   AddColor('LightSteelBlue1',$FFE1CA);
   AddColor('LightSteelBlue2',$EED2BC);
   AddColor('LightSteelBlue3',$CDB5A2);
   AddColor('LightSteelBlue4',$8B7B6E);
   AddColor('LightBlue1',$FFEFBF);
   AddColor('LightBlue2',$EEDFB2);
   AddColor('LightBlue3',$CDC09A);
   AddColor('LightBlue4',$8B8368);
   AddColor('LightCyan1',$FFFFE0);
   AddColor('LightCyan2',$EEEED1);
   AddColor('LightCyan3',$CDCDB4);
   AddColor('LightCyan4',$8B8B7A);
   AddColor('PaleTurquoise1',$FFFFBB);
   AddColor('PaleTurquoise2',$EEEEAE);
   AddColor('PaleTurquoise3',$CDCD96);
   AddColor('PaleTurquoise4',$8B8B66);
   AddColor('CadetBlue1',$FFF598);
   AddColor('CadetBlue2',$EEE58E);
   AddColor('CadetBlue3',$CDC57A);
   AddColor('CadetBlue4',$8B8653);
   AddColor('turquoise1',$FFF500);
   AddColor('turquoise2',$EEE500);
   AddColor('turquoise3',$CDC500);
   AddColor('turquoise4',$8B8600);
   AddColor('cyan1',$FFFF00);
   AddColor('cyan2',$EEEE00);
   AddColor('cyan3',$CDCD00);
   AddColor('cyan4',$8B8B00);
   AddColor('DarkSlateGray1',$FFFF97);
   AddColor('DarkSlateGray2',$EEEE8D);
   AddColor('DarkSlateGray3',$CDCD79);
   AddColor('DarkSlateGray4',$8B8B52);
   AddColor('aquamarine1',$D4FF7F);
   AddColor('aquamarine2',$C6EE76);
   AddColor('aquamarine3',$AACD66);
   AddColor('aquamarine4',$748B45);
   AddColor('DarkSeaGreen1',$C1FFC1);
   AddColor('DarkSeaGreen2',$B4EEB4);
   AddColor('DarkSeaGreen3',$9BCD9B);
   AddColor('DarkSeaGreen4',$698B69);
   AddColor('SeaGreen1',$9FFF54);
   AddColor('SeaGreen2',$94EE4E);
   AddColor('SeaGreen3',$80CD43);
   AddColor('SeaGreen4',$578B2E);
   AddColor('PaleGreen1',$9AFF9A);
   AddColor('PaleGreen2',$90EE90);
   AddColor('PaleGreen3',$7CCD7C);
   AddColor('PaleGreen4',$548B54);
   AddColor('SpringGreen1',$7FFF00);
   AddColor('SpringGreen2',$76EE00);
   AddColor('SpringGreen3',$66CD00);
   AddColor('SpringGreen4',$458B00);
   AddColor('green1',$00FF00);
   AddColor('green2',$00EE00);
   AddColor('green3',$00CD00);
   AddColor('green4',$008B00);
   AddColor('chartreuse1',$00FF7F);
   AddColor('chartreuse2',$00EE76);
   AddColor('chartreuse3',$00CD66);
   AddColor('chartreuse4',$008B45);
   AddColor('OliveDrab1',$3EFFC0);
   AddColor('OliveDrab2',$3AEEB3);
   AddColor('OliveDrab3',$32CD9A);
   AddColor('OliveDrab4',$228B69);
   AddColor('DarkOliveGreen1',$70FFCA);
   AddColor('DarkOliveGreen2',$68EEBC);
   AddColor('DarkOliveGreen3',$5ACDA2);
   AddColor('DarkOliveGreen4',$3D8B6E);
   AddColor('khaki1',$8FF6FF);
   AddColor('khaki2',$85E6EE);
   AddColor('khaki3',$73C6CD);
   AddColor('khaki4',$4E868B);
   AddColor('LightGoldenrod1',$8BECFF);
   AddColor('LightGoldenrod2',$82DCEE);
   AddColor('LightGoldenrod3',$70BECD);
   AddColor('LightGoldenrod4',$4C818B);
   AddColor('LightYellow1',$E0FFFF);
   AddColor('LightYellow2',$D1EEEE);
   AddColor('LightYellow3',$B4CDCD);
   AddColor('LightYellow4',$7A8B8B);
   AddColor('yellow1',$00FFFF);
   AddColor('yellow2',$00EEEE);
   AddColor('yellow3',$00CDCD);
   AddColor('yellow4',$008B8B);
   AddColor('gold1',$00D7FF);
   AddColor('gold2',$00C9EE);
   AddColor('gold3',$00ADCD);
   AddColor('gold4',$00758B);
   AddColor('goldenrod1',$25C1FF);
   AddColor('goldenrod2',$22B4EE);
   AddColor('goldenrod3',$1D9BCD);
   AddColor('goldenrod4',$14698B);
   AddColor('DarkGoldenrod1',$0FB9FF);
   AddColor('DarkGoldenrod2',$0EADEE);
   AddColor('DarkGoldenrod3',$0C95CD);
   AddColor('DarkGoldenrod4',$08658B);
   AddColor('RosyBrown1',$C1C1FF);
   AddColor('RosyBrown2',$B4B4EE);
   AddColor('RosyBrown3',$9B9BCD);
   AddColor('RosyBrown4',$69698B);
   AddColor('IndianRed1',$6A6AFF);
   AddColor('IndianRed2',$6363EE);
   AddColor('IndianRed3',$5555CD);
   AddColor('IndianRed4',$3A3A8B);
   AddColor('sienna1',$4782FF);
   AddColor('sienna2',$4279EE);
   AddColor('sienna3',$3968CD);
   AddColor('sienna4',$26478B);
   AddColor('burlywood1',$9BD3FF);
   AddColor('burlywood2',$91C5EE);
   AddColor('burlywood3',$7DAACD);
   AddColor('burlywood4',$55738B);
   AddColor('wheat1',$BAE7FF);
   AddColor('wheat2',$AED8EE);
   AddColor('wheat3',$96BACD);
   AddColor('wheat4',$667E8B);
   AddColor('tan1',$4FA5FF);
   AddColor('tan2',$499AEE);
   AddColor('tan3',$3F85CD);
   AddColor('tan4',$2B5A8B);
   AddColor('chocolate1',$247FFF);
   AddColor('chocolate2',$2176EE);
   AddColor('chocolate3',$1D66CD);
   AddColor('chocolate4',$13458B);
   AddColor('firebrick1',$3030FF);
   AddColor('firebrick2',$2C2CEE);
   AddColor('firebrick3',$2626CD);
   AddColor('firebrick4',$1A1A8B);
   AddColor('brown1',$4040FF);
   AddColor('brown2',$3B3BEE);
   AddColor('brown3',$3333CD);
   AddColor('brown4',$23238B);
   AddColor('salmon1',$698CFF);
   AddColor('salmon2',$6282EE);
   AddColor('salmon3',$5470CD);
   AddColor('salmon4',$394C8B);
   AddColor('LightSalmon1',$7AA0FF);
   AddColor('LightSalmon2',$7295EE);
   AddColor('LightSalmon3',$6281CD);
   AddColor('LightSalmon4',$42578B);
   AddColor('orange1',$00A5FF);
   AddColor('orange2',$009AEE);
   AddColor('orange3',$0085CD);
   AddColor('orange4',$005A8B);
   AddColor('DarkOrange1',$007FFF);
   AddColor('DarkOrange2',$0076EE);
   AddColor('DarkOrange3',$0066CD);
   AddColor('DarkOrange4',$00458B);
   AddColor('coral1',$5672FF);
   AddColor('coral2',$506AEE);
   AddColor('coral3',$455BCD);
   AddColor('coral4',$2F3E8B);
   AddColor('tomato1',$4763FF);
   AddColor('tomato2',$425CEE);
   AddColor('tomato3',$394FCD);
   AddColor('tomato4',$26368B);
   AddColor('OrangeRed1',$0045FF);
   AddColor('OrangeRed2',$0040EE);
   AddColor('OrangeRed3',$0037CD);
   AddColor('OrangeRed4',$00258B);
   AddColor('red1',$0000FF);
   AddColor('red2',$0000EE);
   AddColor('red3',$0000CD);
   AddColor('red4',$00008B);
   AddColor('DeepPink1',$9314FF);
   AddColor('DeepPink2',$8912EE);
   AddColor('DeepPink3',$7610CD);
   AddColor('DeepPink4',$500A8B);
   AddColor('HotPink1',$B46EFF);
   AddColor('HotPink2',$A76AEE);
   AddColor('HotPink3',$9060CD);
   AddColor('HotPink4',$623A8B);
   AddColor('pink1',$C5B5FF);
   AddColor('pink2',$B8A9EE);
   AddColor('pink3',$9E91CD);
   AddColor('pink4',$6C638B);
   AddColor('LightPink1',$B9AEFF);
   AddColor('LightPink2',$ADA2EE);
   AddColor('LightPink3',$958CCD);
   AddColor('LightPink4',$655F8B);
   AddColor('PaleVioletRed1',$AB82FF);
   AddColor('PaleVioletRed2',$9F79EE);
   AddColor('PaleVioletRed3',$8968CD);
   AddColor('PaleVioletRed4',$5D478B);
   AddColor('maroon1',$B334FF);
   AddColor('maroon2',$A730EE);
   AddColor('maroon3',$9029CD);
   AddColor('maroon4',$621C8B);
   AddColor('VioletRed1',$963EFF);
   AddColor('VioletRed2',$8C3AEE);
   AddColor('VioletRed3',$7832CD);
   AddColor('VioletRed4',$52228B);
   AddColor('magenta1',$FF00FF);
   AddColor('magenta2',$EE00EE);
   AddColor('magenta3',$CD00CD);
   AddColor('magenta4',$8B008B);
   AddColor('orchid1',$FA83FF);
   AddColor('orchid2',$E97AEE);
   AddColor('orchid3',$C969CD);
   AddColor('orchid4',$89478B);
   AddColor('plum1',$FFBBFF);
   AddColor('plum2',$EEAEEE);
   AddColor('plum3',$CD96CD);
   AddColor('plum4',$8B668B);
   AddColor('MediumOrchid1',$FF66E0);
   AddColor('MediumOrchid2',$EE5FD1);
   AddColor('MediumOrchid3',$CD52B4);
   AddColor('MediumOrchid4',$8B377A);
   AddColor('DarkOrchid1',$FF3EBF);
   AddColor('DarkOrchid2',$EE3AB2);
   AddColor('DarkOrchid3',$CD329A);
   AddColor('DarkOrchid4',$8B2268);
   AddColor('purple1',$FF309B);
   AddColor('purple2',$EE2C91);
   AddColor('purple3',$CD267D);
   AddColor('purple4',$8B1A55);
   AddColor('MediumPurple1',$FF82AB);
   AddColor('MediumPurple2',$EE799F);
   AddColor('MediumPurple3',$CD6889);
   AddColor('MediumPurple4',$8B475D);
   AddColor('thistle1',$FFE1FF);
   AddColor('thistle2',$EED2EE);
   AddColor('thistle3',$CDB5CD);
   AddColor('thistle4',$8B7B8B);
   AddColor('gray0',$000000);
   AddColor('grey0',$000000);
   AddColor('gray1',$030303);
   AddColor('grey1',$030303);
   AddColor('gray2',$050505);
   AddColor('grey2',$050505);
   AddColor('gray3',$080808);
   AddColor('grey3',$080808);
   AddColor('gray4',$0A0A0A);
   AddColor('grey4',$0A0A0A);
   AddColor('gray5',$0D0D0D);
   AddColor('grey5',$0D0D0D);
   AddColor('gray6',$0F0F0F);
   AddColor('grey6',$0F0F0F);
   AddColor('gray7',$121212);
   AddColor('grey7',$121212);
   AddColor('gray8',$141414);
   AddColor('grey8',$141414);
   AddColor('gray9',$171717);
   AddColor('grey9',$171717);
   AddColor('gray10',$1A1A1A);
   AddColor('grey10',$1A1A1A);
   AddColor('gray11',$1C1C1C);
   AddColor('grey11',$1C1C1C);
   AddColor('gray12',$1F1F1F);
   AddColor('grey12',$1F1F1F);
   AddColor('gray13',$212121);
   AddColor('grey13',$212121);
   AddColor('gray14',$242424);
   AddColor('grey14',$242424);
   AddColor('gray15',$262626);
   AddColor('grey15',$262626);
   AddColor('gray16',$292929);
   AddColor('grey16',$292929);
   AddColor('gray17',$2B2B2B);
   AddColor('grey17',$2B2B2B);
   AddColor('gray18',$2E2E2E);
   AddColor('grey18',$2E2E2E);
   AddColor('gray19',$303030);
   AddColor('grey19',$303030);
   AddColor('gray20',$333333);
   AddColor('grey20',$333333);
   AddColor('gray21',$363636);
   AddColor('grey21',$363636);
   AddColor('gray22',$383838);
   AddColor('grey22',$383838);
   AddColor('gray23',$3B3B3B);
   AddColor('grey23',$3B3B3B);
   AddColor('gray24',$3D3D3D);
   AddColor('grey24',$3D3D3D);
   AddColor('gray25',$404040);
   AddColor('grey25',$404040);
   AddColor('gray26',$424242);
   AddColor('grey26',$424242);
   AddColor('gray27',$454545);
   AddColor('grey27',$454545);
   AddColor('gray28',$474747);
   AddColor('grey28',$474747);
   AddColor('gray29',$4A4A4A);
   AddColor('grey29',$4A4A4A);
   AddColor('gray30',$4D4D4D);
   AddColor('grey30',$4D4D4D);
   AddColor('gray31',$4F4F4F);
   AddColor('grey31',$4F4F4F);
   AddColor('gray32',$525252);
   AddColor('grey32',$525252);
   AddColor('gray33',$545454);
   AddColor('grey33',$545454);
   AddColor('gray34',$575757);
   AddColor('grey34',$575757);
   AddColor('gray35',$595959);
   AddColor('grey35',$595959);
   AddColor('gray36',$5C5C5C);
   AddColor('grey36',$5C5C5C);
   AddColor('gray37',$5E5E5E);
   AddColor('grey37',$5E5E5E);
   AddColor('gray38',$616161);
   AddColor('grey38',$616161);
   AddColor('gray39',$636363);
   AddColor('grey39',$636363);
   AddColor('gray40',$666666);
   AddColor('grey40',$666666);
   AddColor('gray41',$696969);
   AddColor('grey41',$696969);
   AddColor('gray42',$6B6B6B);
   AddColor('grey42',$6B6B6B);
   AddColor('gray43',$6E6E6E);
   AddColor('grey43',$6E6E6E);
   AddColor('gray44',$707070);
   AddColor('grey44',$707070);
   AddColor('gray45',$737373);
   AddColor('grey45',$737373);
   AddColor('gray46',$757575);
   AddColor('grey46',$757575);
   AddColor('gray47',$787878);
   AddColor('grey47',$787878);
   AddColor('gray48',$7A7A7A);
   AddColor('grey48',$7A7A7A);
   AddColor('gray49',$7D7D7D);
   AddColor('grey49',$7D7D7D);
   AddColor('gray50',$7F7F7F);
   AddColor('grey50',$7F7F7F);
   AddColor('gray51',$828282);
   AddColor('grey51',$828282);
   AddColor('gray52',$858585);
   AddColor('grey52',$858585);
   AddColor('gray53',$878787);
   AddColor('grey53',$878787);
   AddColor('gray54',$8A8A8A);
   AddColor('grey54',$8A8A8A);
   AddColor('gray55',$8C8C8C);
   AddColor('grey55',$8C8C8C);
   AddColor('gray56',$8F8F8F);
   AddColor('grey56',$8F8F8F);
   AddColor('gray57',$919191);
   AddColor('grey57',$919191);
   AddColor('gray58',$949494);
   AddColor('grey58',$949494);
   AddColor('gray59',$969696);
   AddColor('grey59',$969696);
   AddColor('gray60',$999999);
   AddColor('grey60',$999999);
   AddColor('gray61',$9C9C9C);
   AddColor('grey61',$9C9C9C);
   AddColor('gray62',$9E9E9E);
   AddColor('grey62',$9E9E9E);
   AddColor('gray63',$A1A1A1);
   AddColor('grey63',$A1A1A1);
   AddColor('gray64',$A3A3A3);
   AddColor('grey64',$A3A3A3);
   AddColor('gray65',$A6A6A6);
   AddColor('grey65',$A6A6A6);
   AddColor('gray66',$A8A8A8);
   AddColor('grey66',$A8A8A8);
   AddColor('gray67',$ABABAB);
   AddColor('grey67',$ABABAB);
   AddColor('gray68',$ADADAD);
   AddColor('grey68',$ADADAD);
   AddColor('gray69',$B0B0B0);
   AddColor('grey69',$B0B0B0);
   AddColor('gray70',$B3B3B3);
   AddColor('grey70',$B3B3B3);
   AddColor('gray71',$B5B5B5);
   AddColor('grey71',$B5B5B5);
   AddColor('gray72',$B8B8B8);
   AddColor('grey72',$B8B8B8);
   AddColor('gray73',$BABABA);
   AddColor('grey73',$BABABA);
   AddColor('gray74',$BDBDBD);
   AddColor('grey74',$BDBDBD);
   AddColor('gray75',$BFBFBF);
   AddColor('grey75',$BFBFBF);
   AddColor('gray76',$C2C2C2);
   AddColor('grey76',$C2C2C2);
   AddColor('gray77',$C4C4C4);
   AddColor('grey77',$C4C4C4);
   AddColor('gray78',$C7C7C7);
   AddColor('grey78',$C7C7C7);
   AddColor('gray79',$C9C9C9);
   AddColor('grey79',$C9C9C9);
   AddColor('gray80',$CCCCCC);
   AddColor('grey80',$CCCCCC);
   AddColor('gray81',$CFCFCF);
   AddColor('grey81',$CFCFCF);
   AddColor('gray82',$D1D1D1);
   AddColor('grey82',$D1D1D1);
   AddColor('gray83',$D4D4D4);
   AddColor('grey83',$D4D4D4);
   AddColor('gray84',$D6D6D6);
   AddColor('grey84',$D6D6D6);
   AddColor('gray85',$D9D9D9);
   AddColor('grey85',$D9D9D9);
   AddColor('gray86',$DBDBDB);
   AddColor('grey86',$DBDBDB);
   AddColor('gray87',$DEDEDE);
   AddColor('grey87',$DEDEDE);
   AddColor('gray88',$E0E0E0);
   AddColor('grey88',$E0E0E0);
   AddColor('gray89',$E3E3E3);
   AddColor('grey89',$E3E3E3);
   AddColor('gray90',$E5E5E5);
   AddColor('grey90',$E5E5E5);
   AddColor('gray91',$E8E8E8);
   AddColor('grey91',$E8E8E8);
   AddColor('gray92',$EBEBEB);
   AddColor('grey92',$EBEBEB);
   AddColor('gray93',$EDEDED);
   AddColor('grey93',$EDEDED);
   AddColor('gray94',$F0F0F0);
   AddColor('grey94',$F0F0F0);
   AddColor('gray95',$F2F2F2);
   AddColor('grey95',$F2F2F2);
   AddColor('gray96',$F5F5F5);
   AddColor('grey96',$F5F5F5);
   AddColor('gray97',$F7F7F7);
   AddColor('grey97',$F7F7F7);
   AddColor('gray98',$FAFAFA);
   AddColor('grey98',$FAFAFA);
   AddColor('gray99',$FCFCFC);
   AddColor('grey99',$FCFCFC);
   AddColor('gray100',$FFFFFF);
   AddColor('grey100',$FFFFFF);
   AddColor('darkgrey',$A9A9A9);
   AddColor('darkgray',$A9A9A9);
   AddColor('darkblue',$8B0000);
   AddColor('darkcyan',$8B8B00);
   AddColor('darkmagenta',$8B008B);
   AddColor('darkred',$00008B);
   AddColor('lightgreen',$90EE90);
  end;
  if (KnownColorsListByName=nil) then begin
   KnownColorsListByName:=TStringList.Create;
   KnownColorsListByName.Sorted:=true;
   for i:=0 to KnownColorsList.Count-1 do begin
    n:=KnownColorsList.Strings[i];
    c:=TColor(KnownColorsList.Objects[i]); h:='cl'+n;
    if KnownColorsListByName.IndexOf(n)<0 then KnownColorsListByName.AddObject(n,TObject(c));
    if KnownColorsListByName.IndexOf(h)<0 then KnownColorsListByName.AddObject(h,TObject(c));
   end;
  end;
  if (KnownColorsListByCode=nil) then begin
   KnownColorsListByCode:=TStringList.Create;
   KnownColorsListByCode.Sorted:=true;
   for i:=0 to KnownColorsList.Count-1 do begin
    n:=KnownColorsList.Strings[i]; k:=i+1;
    c:=TColor(KnownColorsList.Objects[i]); h:=ColorToHexString(c);
    if KnownColorsListByCode.IndexOf(h)<0 then KnownColorsListByCode.AddObject(h,TObject(k));
   end;
  end;
  DefaultFaceColor:=FindNearestKnownColor(ColorToRgb(clBtnFace));
  DefaultTextColor:=FindNearestKnownColor(ColorToRgb(clBtnText));
 except
  on E:Exception do BugReport(E,nil,'InitKnownColorsList');
 end;
end;

procedure FreeKnownColorsList;
begin
 try
  if (KnownColorsList<>nil) then begin
   KnownColorsList.Free;
   KnownColorsList:=nil;
  end;
  if (KnownColorsListByName<>nil) then begin
   KnownColorsListByName.Free;
   KnownColorsListByName:=nil;
  end;
  if (KnownColorsListByCode<>nil) then begin
   KnownColorsListByCode.Free;
   KnownColorsListByCode:=nil;
  end;
 except
  on E:Exception do BugReport(E,nil,'FreeKnownColorsList');
 end;
end;

initialization

 InitKnownColorsList;

// finalization

// FreeKnownColorsList;

end.
