/////////////////////////////////////////////////////////////////////
// gRun Copyright(c) 2018 Alexey Kuryakin RU kouriakine@mail.ru
//
// Purposes:
// gRun - free Win32 utility to run program in specified display
//        mode (hidden,normal,minimized,maximized;no/wait).
//
// License for use and distribution: GNU LGPL (see below).
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU Lesser General Public License for more details.
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330,
// Boston, MA  02111-1307  USA
/////////////////////////////////////////////////////////////////////
program grun;

{$APPTYPE GUI}

{$ALIGN             OFF}
{$BOOLEVAL          OFF}
{$ASSERTIONS        ON}
{$DEBUGINFO         ON}
{$DEFINITIONINFO    ON}
{$EXTENDEDSYNTAX    ON}
{$LONGSTRINGS       ON}
{$HINTS             ON}
{$IOCHECKS          OFF}
{$WRITEABLECONST    ON}
{$LOCALSYMBOLS      ON}
{$MINENUMSIZE       1}
{$OPENSTRINGS       ON}
{$OPTIMIZATION      ON}
{$OVERFLOWCHECKS    OFF}
{$RANGECHECKS       OFF}
{$REALCOMPATIBILITY OFF}
{$STACKFRAMES       OFF}
{$TYPEDADDRESS      OFF}
{$TYPEINFO          ON}
{$VARSTRINGCHECKS   OFF}
{$WARNINGS          ON}

{$RESOURCE grun.res}

uses windows,shellapi;

const CRLF = #13#10;
const About =
     'Help on gRun.exe, v1.5:'+CRLF
    +''+CRLF
    +'Copyright(c) 2018 Alexey Kuryakin RU kouriakine@mail.ru.'+CRLF
    +'gRun - Win32 utility to run program in specified display'+CRLF
    +'       mode (hidden,normal,minimized,maximized;no/wait).'+CRLF
    +'       It works like analog of Linux GTK utility gRun.'+CRLF
    +'Use this program free under the terms of LGPL license.'+CRLF
    +''+CRLF
    +'Usage: grun [-opt] cmdline'+CRLF
    +''+CRLF
    +'Options (started from - char):'+CRLF
    +' -v  - verbose mode: MessageBox on errors (default is silent)'+CRLF
    +'       it''s better to place this option as 1st option in list'+CRLF
    +' -w  - wait while cmdline running, default is no waits'+CRLF
    +'       in wait mode gRun return cmdline''s exit code'+CRLF
    +' -h  - run cmdline in hidden window (SW_HIDE), default is visible (SW_SHOW)'+CRLF
    +' -0..9 run with display mode assigned to 0=SW_HIDE(like -h), 1=SW_SHOWNORMAL,'+CRLF
    +'       2=SW_SHOWMINIMIZED, 3=SW_SHOWMAXIMIZED, 4=SW_SHOWNOACTIVATE,'+CRLF
    +'       5=SW_SHOW(default), 6=SW_MINIMIZE, 7=SW_SHOWMINNOACTIVE,'+CRLF
    +'       8=SW_SHOWNA, 9=SW_RESTORE'+CRLF
    +' -i  - run with IDLE_PRIORITY_CLASS         ( 4  )'+CRLF
    +' -b  - run with BELOW_NORMAL_PRIORITY_CLASS ( 6  )'+CRLF
    +' -n  - run with NORMAL_PRIORITY_CLASS       ( 8  )'+CRLF
    +' -a  - run with ABOVE_NORMAL_PRIORITY_CLASS ( 10 )'+CRLF
    +' -g  - run with HIGH_PRIORITY_CLASS         ( 13 )'+CRLF
    +' -r  - run with REALTIME_PRIORITY_CLASS     ( 24 )'+CRLF
    +' -c  - run with ComSpec /c ... (not compatible with -ksopmxy)'+CRLF
    +' -k  - run with ComSpec /k ... (not compatible with -csopmxy)'+CRLF
    +' -s  - run with ShellExecuteEx(default) instead of CreateProcess'+CRLF
    +' -o  - run with ShellExecuteEx(open) instead of CreateProcess'+CRLF
    +' -p  - run with ShellExecuteEx(print) instead of CreateProcess'+CRLF
    +' -m  - run with ShellExecuteEx(edit) instead of CreateProcess'+CRLF
    +' -x  - run with ShellExecuteEx(explore) instead of CreateProcess'+CRLF
    +' -y  - run with ShellExecuteEx(properties) instead of CreateProcess'+CRLF
    +' -e  - run with ELEVATED (ADMIN) rights via ShellExecuteEx(runas)'+CRLF
    +' -?  - run this help screen'+CRLF
    +''+CRLF
    +'Exit code returned by gRun:'+CRLF
    +' 0   - OK, cmdline was successfully started'+CRLF
    +' 1   - invalid options specified (unsupported opt)'+CRLF
    +' 2   - no arguments specified (empty command line)'+CRLF
    +' 3   - could not create process (invalid cmdline or access)'+CRLF
    +' 4   - exception happened during execution'+CRLF
    +' n   - in wait mode (-w) gRun return cmdline''s exit code'+CRLF
    +''+CRLF
    +'Examples:'+CRLF
    +' grun firefox.exe - run Firefox in default mode, no wait'+CRLF
    +' grun -cwh myprog - run myprog in hidden CMD console, wait'+CRLF
    +' grun -b calc     - run calculator with low priority, no wait'+CRLF
    +' grun -vw notepad - run notepad in normal mode, verbose, wait'+CRLF
    +' grun -w7 myprog  - run myprog in minimized noactive mode, wait'+CRLF
    +' grun -hw myprog  - run myprog in hidden mode, wait'+CRLF
    +' grun -e7 myprog  - run myprog ELEVATED, minimized noactive mode'+CRLF
    +' grun -o help.htm - open document help.htm'+CRLF
    +' grun -m help.txt - edit document help.txt'+CRLF
    ;

// Special chars:
/////////////////
const SpaceChar                      = ' ';
const TabChar                        = #9;
const DoubleQuota                    = '"';

// Utility functions for CmdLine parsing:
/////////////////////////////////////////
function IsSameChar(a,b:Char):Boolean; begin Result:=(a=b); end;
function IsSameCharOrNul(a,b:Char):Boolean; begin Result:=IsSameChar(a,b) or IsSameChar(a,#0); end;
function TabToSpace(c:Char):Char; begin if(c=TabChar) then Result:=SpaceChar else Result:=c; end;
function IsSpaceChar(c:Char):Boolean; begin Result:=IsSameChar(c,SpaceChar); end;
function IsSpaceCharOrTab(c:Char):Boolean; begin Result:=IsSpaceChar(TabToSpace(c)); end;
function IsSpaceCharOrNul(c:Char):Boolean; begin Result:=IsSameCharOrNul(c,SpaceChar); end;
function IsSpaceCharOrTabOrNul(c:Char):Boolean; begin Result:=IsSameCharOrNul(TabToSpace(c),SpaceChar); end;
function IsOptionChar(c:Char):Boolean; begin Result:=IsSameChar(c,'-') or IsSameChar(c,'/'); end;

// Exit codes returned by gRun:
//////////////////////////////////
const ecOK                          = 0;
const ecBadOpt                      = 1;
const ecNoArgs                      = 2;
const ecFailStart                   = 3;
const ecException                   = 4;

const BELOW_NORMAL_PRIORITY_CLASS   = $4000;     // Not defined in Windows.pas
const ABOVE_NORMAL_PRIORITY_CLASS   = $8000;     // Not defined in Windows.pas

function IntToStr(i:Integer):AnsiString;
begin
 Str(i,Result);
end;

function Max(a,b:Integer):Integer;
begin
 if(a>b) then Result:=a else Result:=b;
end;

// To be called on fatal errors:
////////////////////////////////
function Fatal(exitcode:Integer; verb:Boolean; Title:PChar; Content:PChar):Integer;
var LastError,len:Integer; msg,tmp:AnsiString;
begin
 if ( verb ) then
 case ( exitcode ) of
  ecOK        : MessageBox(0,Content,Title,MB_OK+MB_ICONINFORMATION);
  ecBadOpt    : MessageBox(0,Content,Title,MB_OK+MB_ICONWARNING);
  ecNoArgs    : MessageBox(0,Content,Title,MB_OK+MB_ICONWARNING);
  ecFailStart : begin
   LastError := GetLastError(); SetLength(tmp,16*1024);
   len := FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM or FORMAT_MESSAGE_ARGUMENT_ARRAY, nil, LastError, 0, PChar(tmp), Length(tmp), nil);
   while ( (len > 0) and (tmp[len-1] <= SpaceChar)) do Dec(len); SetLength(tmp,Max(0,len-1));
   if (Length(tmp)=0) and (LastError=193) then tmp:='That is not a valid Win32 application.'; if (Length(tmp)=0) then tmp:='???';
   msg:='Win32 error '+IntToStr(LastError)+' occured:'+CRLF+CRLF+'   "'+tmp+'"'+CRLF+CRLF+'in command line:'+CRLF+CRLF+'   '+GetCommandLine()+CRLF+CRLF+Content;
   MessageBox(0,PChar(msg),Title,MB_OK+MB_ICONERROR);
  end;
 end;
 Result:=exitcode;
end;

function strlen(s:PChar):Integer;
begin
 Result:=0;
 if Assigned(s) then
 while s[0] <> #0 do begin inc(s); inc(Result); end;
end;
function strcpy(d,s:PChar):PChar;
begin
 Result:=d;
 if Assigned(d) then begin
  if Assigned(s) then
  while s[0] <> #0 do begin d[0]:=s[0]; inc(d); inc(s); end;
  d[0]:=#0;
 end;
end;
function strcat(d,s:PChar):PChar;
begin
 Result:=d;
 if Assigned(d) then begin
  while d[0] <> #0 do inc(d);
  strcpy(d,s);
 end;
end;

const
 DOMAIN_ALIAS_RID_ADMINS     = $00000220;
 SECURITY_BUILTIN_DOMAIN_RID = $00000020;
 SECURITY_NT_AUTHORITY : TSidIdentifierAuthority = (Value:(0,0,0,0,0,5));
 SEE_MASK_NOCLOSEPROCESS     = $00000040; // Use to indicate that the hProcess member receives the process handle.
 SEE_MASK_CONNECTNETDRV      = $00000080; // Validate the share and connect to a drive letter. This enables reconnection of disconnected network drives. The lpFile member is a UNC path of a file on a network.
 SEE_MASK_NOASYNC            = $00000100; // Wait for the execute operation to complete before returning. Applications that exit immediately after calling ShellExecuteEx should specify this flag.
 SEE_MASK_FLAG_NO_UI         = $00000400; // Do not display an error message box if an error occurs.
 
function CheckWin32Version(AMajor,AMinor:Cardinal):Boolean;
var osvi:TOSVersionInfo;
begin
 osvi.dwOSVersionInfoSize:=SizeOf(osvi);
 if GetVersionEx(osvi)
 then Result:=(osvi.dwMajorVersion>AMajor) or ((osvi.dwMajorVersion=AMajor) and (osvi.dwMinorVersion>=AMinor))
 else Result:=false;
end;

function IsUACAvailable:Boolean;
begin
 Result:=CheckWin32Version(6,0);  // Windows Vista and above
end;

function CheckTokenMembership(TokenHandle:THANDLE; SidToCheck:PSID; IsMember:PBOOL):BOOL; stdcall;
const _CheckTokenMembership:function(TokenHandle:THANDLE;SidToCheck:PSID;IsMember:PBOOL):BOOL stdcall = nil;
var hModule:THandle;
begin
 if not Assigned(_CheckTokenMembership) then begin
  hModule:=GetModuleHandle('advapi32.dll');
  if hModule<>0 then @_CheckTokenMembership:=GetProcAddress(hModule,'CheckTokenMembership');
 end;
 if not Assigned(_CheckTokenMembership) then begin
  SetLastError(ERROR_CALL_NOT_IMPLEMENTED);
  Result:=FALSE;
 end else
 Result:=_CheckTokenMembership(TokenHandle,SidToCheck,IsMember);
end;

function IsMemberOfGroup(rid:DWORD):Boolean;
var sid:PSID; IsMember:BOOL;
begin
 if AllocateAndInitializeSid(SECURITY_NT_AUTHORITY,2,SECURITY_BUILTIN_DOMAIN_RID,rid,0,0,0,0,0,0,sid) then begin
  if CheckTokenMembership(0,sid,@IsMember) then Result:=IsMember else Result:=FALSE;
  FreeSid(sid);
 end else Result:=FALSE;
end;

function IsAdministrator:Boolean;
begin
 Result:=IsMemberOfGroup(DOMAIN_ALIAS_RID_ADMINS);
end;

function IsElevated: Boolean;
const TokenElevation=TTokenInformationClass(20);
type TOKEN_ELEVATION = record TokenIsElevated: DWORD; end;
var TokenHandle:THandle; ResultLength:Cardinal; ATokenElevation:TOKEN_ELEVATION; HaveToken:Boolean;
begin
 if IsUACAvailable then begin
   TokenHandle:=0;
   HaveToken:=OpenThreadToken(GetCurrentThread,TOKEN_QUERY,True,TokenHandle);
   if (not HaveToken) and (GetLastError = ERROR_NO_TOKEN) then
   HaveToken:=OpenProcessToken(GetCurrentProcess,TOKEN_QUERY,TokenHandle);
   if HaveToken then begin
    ResultLength := 0;
    if GetTokenInformation(TokenHandle,TokenElevation,@ATokenElevation,SizeOf(ATokenElevation),ResultLength)
    then Result:=ATokenElevation.TokenIsElevated<>0
    else Result:=False;
    CloseHandle(TokenHandle);
   end else Result := False;
 end else Result:=IsAdministrator;
end;

function CheckRange(x,a,b:Integer):BOOL;
begin
 Result:=(a<=x) and (x<=b);
end;

procedure Main;
const MAX_CMD = 1024*64;
var si:STARTUPINFO; pi:PROCESS_INFORMATION;
 see:TShellExecuteInfo; WaitTimeOut,PriorityClass,DisplayMode:DWORD;
 CreateProcessResult,VerboseMode,ShellEx,seeElevate,seeOpen,seeEdit:BOOL;
 seePrint,seeExplore,seeProperties,ComSpecModeC,ComSpecModeK:BOOL;
 lpszCmdLine,pwd,eOpt,ComSpec,SelfExe,eCmdLine:PChar; stopchar:char;
 procedure Initialize;
 begin
  ZeroMemory(@si,sizeof(si));
  ZeroMemory(@pi,sizeof(pi));
  WaitTimeOut:=0;
  VerboseMode:=FALSE;
  stopchar:=SpaceChar;
  DisplayMode:=SW_SHOW;
  CreateProcessResult:=FALSE;
  lpszCmdLine:=nil;
  PriorityClass:=0;
  ShellEx:=FALSE;
  seeElevate:=FALSE;
  seeOpen:=FALSE;
  seeEdit:=FALSE;
  seePrint:=FALSE;
  seeExplore:=FALSE;
  seeProperties:=FALSE;
  ZeroMemory(@see,sizeof(see));
  pwd:=nil;
  eOpt:=nil;
  ComSpec:=nil;
  SelfExe:=nil;
  eCmdLine:=nil;
  ComSpecModeC:=FALSE;
  ComSpecModeK:=FALSE;
 end;
 procedure Cleanup;
 begin
  if Assigned(pwd) then FreeMem(pwd);
  if Assigned(eOpt) then FreeMem(eOpt);
  if Assigned(ComSpec) then FreeMem(ComSpec);
  if Assigned(SelfExe) then FreeMem(SelfExe);
  if Assigned(eCmdLine) then FreeMem(eCmdLine);
 end;
begin
 try
  Initialize;
  try
    // Initialize.
    exitcode := ecOK;
   
    lpszCmdLine := GetCommandLine();

    // Skip leading space & tab chars.
    while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) do Inc(lpszCmdLine);

    // Skip executable file name, maybe double quoted.
    if ( IsSameChar(lpszCmdLine[0],DoubleQuota) ) then begin stopchar := DoubleQuota; Inc(lpszCmdLine); end;
    while ( not IsSameCharOrNul(TabToSpace(lpszCmdLine[0]),stopchar) ) do Inc(lpszCmdLine);
    if ( IsSameChar(lpszCmdLine[0],stopchar) ) then Inc(lpszCmdLine);

    // Skip leading space & tab chars.
    while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) do Inc(lpszCmdLine);

    // Check options /opt or -opt.
    while ( IsOptionChar(lpszCmdLine[0]) ) do begin
        Inc(lpszCmdLine);
        while ( not IsSpaceCharOrTabOrNul(lpszCmdLine[0]) ) do begin
            case ( lpszCmdLine[0] ) of
                'W' ,  // Set waiting timeout.
                'w' :  WaitTimeOut := INFINITE;
                'H' ,  // Set hidden mode.
                'h' :  DisplayMode := SW_HIDE;
                '0' :  DisplayMode := SW_HIDE;
                '1' :  DisplayMode := SW_SHOWNORMAL;
                '2' :  DisplayMode := SW_SHOWMINIMIZED;
                '3' :  DisplayMode := SW_SHOWMAXIMIZED;
                '4' :  DisplayMode := SW_SHOWNOACTIVATE;
                '5' :  DisplayMode := SW_SHOW;
                '6' :  DisplayMode := SW_MINIMIZE;
                '7' :  DisplayMode := SW_SHOWMINNOACTIVE;
                '8' :  DisplayMode := SW_SHOWNA;
                '9' :  DisplayMode := SW_RESTORE;
                'V' ,  // Set verbose mode.
                'v' :  VerboseMode := TRUE;
                'I' ,  // Set IDLE_PRIORITY_CLASS
                'i' :  PriorityClass := IDLE_PRIORITY_CLASS;
                'B' ,  // Set BELOW_NORMAL_PRIORITY_CLASS
                'b' :  PriorityClass := BELOW_NORMAL_PRIORITY_CLASS;
                'N' ,  // Set NORMAL_PRIORITY_CLASS
                'n' :  PriorityClass := NORMAL_PRIORITY_CLASS;
                'A' ,  // Set ABOVE_NORMAL_PRIORITY_CLASS
                'a' :  PriorityClass := ABOVE_NORMAL_PRIORITY_CLASS;
                'G' ,  // Set HIGH_PRIORITY_CLASS
                'g' :  PriorityClass := HIGH_PRIORITY_CLASS;
                'R' ,  // Set REALTIME_PRIORITY_CLASS
                'r' :  PriorityClass := REALTIME_PRIORITY_CLASS;
                'C' ,  // Set ComSpec mode C
                'c' :  ComSpecModeC := TRUE;
                'K' ,  // Set ComSpec mode K
                'k' :  ComSpecModeK := TRUE;
                'S' ,  // ShellExecuteEx:
                's' :  ShellEx := TRUE;
                'E' ,  // ShellEx+Elevate.
                'e' :  seeElevate := TRUE;
                'O' ,  // ShellEx Open.
                'o' :  seeOpen := TRUE;
                'M' ,  // ShellEx Edit.
                'm' :  seeEdit := TRUE;
                'P' ,  // ShellEx Print.
                'p' :  seePrint := TRUE;
                'X' ,  // ShellEx Explore.
                'x' :  seeExplore := TRUE;
                'Y' ,  // ShellEx Properties.
                'y' :  seeProperties := TRUE;
                '?' :  begin ExitCode:=Fatal(ecOK,TRUE,'gRun -> Help.',About); Exit; end;
                else   begin ExitCode:=Fatal(ecBadOpt,VerboseMode,'gRun -> Invalid option found.',About); Exit; end;
            end;
            Inc(lpszCmdLine);
        end;
        // Skip space & tab chars.
        while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) do Inc(lpszCmdLine);
    end;

    // Skip space & tab chars.
    while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) do Inc(lpszCmdLine);

    // No arguments?
    if (lpszCmdLine[0] = #0) then begin ExitCode:=Fatal(ecNoArgs,TRUE,'gRun -> No arguments found.',About); Exit; end;

    // Elevation required?
    if IsElevated {or not IsUACAvailable} then seeElevate:=FALSE;

    // ShellExecute required?
    if seeElevate then ShellEx:=TRUE;
    if seeOpen then ShellEx:=TRUE;
    if seeEdit then ShellEx:=TRUE;
    if seePrint then ShellEx:=TRUE;
    if seeExplore then ShellEx:=TRUE;
    if seeProperties then ShellEx:=TRUE;

    if ShellEx then begin
        // ShellExecuteEx branch:
        
        GetMem(pwd,MAX_PATH);
        GetMem(eOpt,MAX_PATH);
        GetMem(SelfExe,MAX_PATH);
        GetMem(ComSpec,MAX_PATH);
        GetMem(eCmdLine,MAX_CMD);
        ZeroMemory(@see, sizeof(see));
        see.cbSize:=sizeof(see);
        see.fMask:=see.fMask or SEE_MASK_NOASYNC;
        see.fMask:=see.fMask or SEE_MASK_FLAG_NO_UI;
        see.fMask:=see.fMask or SEE_MASK_CONNECTNETDRV;
        see.fMask:=see.fMask or SEE_MASK_NOCLOSEPROCESS;
        strcpy(pwd,'');
        if CheckRange(GetCurrentDirectory(MAX_PATH,pwd),1,MAX_PATH-1)
        then see.lpDirectory:=pwd;
        if seeElevate then begin
            see.lpVerb:='runas';
            strcpy(ComSpec,'');
            if not CheckRange(GetEnvironmentVariable('ComSpec',ComSpec,MAX_PATH),1,MAX_PATH-1)
            then strcpy(ComSpec,'cmd.exe'); // Fallback ComSpec
            see.lpFile:=ComSpec;
            if not CheckRange(GetModuleFileName(0,SelfExe,MAX_PATH),1,MAX_PATH-1) then begin
                ExitCode:=Fatal(ecFailStart,VerboseMode,'gRun -> GetModuleFileName failed.','');
                Exit;
            end;
            strcpy(eOpt,'');
            if VerboseMode then strcat(eOpt,'v');
            if ComSpecModeC then strcat(eOpt,'c');
            if ComSpecModeK then strcat(eOpt,'k');
            if WaitTimeOut<>0 then strcat(eOpt,'w');
            if DisplayMode=SW_HIDE then strcat(eOpt,'0');
            if DisplayMode=SW_SHOWNORMAL then strcat(eOpt,'1');
            if DisplayMode=SW_SHOWMINIMIZED then strcat(eOpt,'2');
            if DisplayMode=SW_SHOWMAXIMIZED then strcat(eOpt,'3');
            if DisplayMode=SW_SHOWNOACTIVATE then strcat(eOpt,'4');
            if DisplayMode=SW_SHOW then strcat(eOpt,'5');
            if DisplayMode=SW_MINIMIZE then strcat(eOpt,'6');
            if DisplayMode=SW_SHOWMINNOACTIVE then strcat(eOpt,'7');
            if DisplayMode=SW_SHOWNA then strcat(eOpt,'8');
            if DisplayMode=SW_RESTORE then strcat(eOpt,'9');
            if PriorityClass=IDLE_PRIORITY_CLASS then strcat(eOpt,'i');
            if PriorityClass=BELOW_NORMAL_PRIORITY_CLASS then strcat(eOpt,'b');
            if PriorityClass=NORMAL_PRIORITY_CLASS then strcat(eOpt,'n');
            if PriorityClass=ABOVE_NORMAL_PRIORITY_CLASS then strcat(eOpt,'a');
            if PriorityClass=HIGH_PRIORITY_CLASS then strcat(eOpt,'g');
            if PriorityClass=REALTIME_PRIORITY_CLASS then strcat(eOpt,'r');
            strcpy(eCmdLine,'/c ');
            strcat(eCmdLine,'pushd "');
            strcat(eCmdLine,pwd);
            strcat(eCmdLine,'" & ');
            strcat(eCmdLine,'"');
            strcat(eCmdLine,SelfExe);
            strcat(eCmdLine,'" ');
            if strlen(eOpt)>0 then begin
                strcat(eCmdLine,' -');
                strcat(eCmdLine,eOpt);
                strcat(eCmdLine,' ');
            end;
            if strlen(eCmdLine)+strlen(lpszCmdLine)>=MAX_CMD then begin
                ExitCode:=Fatal(ecFailStart,VerboseMode,'gRun -> Command line too long.','');
                Exit;
            end;
            strcat(eCmdLine,lpszCmdLine);
            see.lpParameters:=eCmdLine;
            see.nShow:=SW_SHOWMINNOACTIVE;
        end else begin
            if seeProperties then see.lpVerb:='properties';
            if seeExplore then see.lpVerb:='explore';
            if seePrint then see.lpVerb:='print';
            if seeEdit then see.lpVerb:='edit';
            if seeOpen then see.lpVerb:='open';
            stopchar:=SpaceChar;
            see.lpFile:=eCmdLine;
            while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) do Inc(lpszCmdLine);
            if ( IsSameChar(lpszCmdLine[0],DoubleQuota) ) then begin stopchar := DoubleQuota; see.lpFile[0]:=lpszCmdLine[0]; Inc(see.lpFile); Inc(lpszCmdLine); end;
            while ( not IsSameCharOrNul(TabToSpace(lpszCmdLine[0]),stopchar) ) do begin see.lpFile[0]:=lpszCmdLine[0]; Inc(see.lpFile); Inc(lpszCmdLine); end;
            if ( IsSameChar(lpszCmdLine[0],stopchar) ) then begin if not IsSpaceCharOrTab(stopchar) then see.lpFile[0]:=lpszCmdLine[0]; Inc(see.lpFile); Inc(lpszCmdLine); end;
            while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) do Inc(lpszCmdLine);
            see.lpFile[0]:=#0;
            see.lpFile:=eCmdLine;
            see.lpParameters:=lpszCmdLine;
            see.nShow:=DisplayMode;
        end;
        if ShellExecuteEx(@see) then begin
            if see.hProcess<>0 then begin
                if ( WaitForSingleObject(see.hProcess, WaitTimeOut) = WAIT_OBJECT_0 )
                    then GetExitCodeProcess(see.hProcess, DWORD(exitcode));
                CloseHandle( see.hProcess );
            end else begin
                if see.hInstApp <= 32 then begin
                    ExitCode:=Fatal(ecFailStart,VerboseMode,'gRun -> ShellExecuteEx failed.','');
                    Exit;
                end;
            end;
        end else begin
            ExitCode:=Fatal(ecFailStart,VerboseMode,'gRun -> ShellExecuteEx failed.','');
            Exit;
        end;
    end else begin
        // Create Process branch:

        // Run with ComSpec
        if ComSpecModeC or ComSpecModeK then begin
            GetMem(ComSpec,MAX_PATH);
            GetMem(eCmdLine,MAX_CMD);
            strcpy(ComSpec,'');
            if not CheckRange(GetEnvironmentVariable('ComSpec',ComSpec,MAX_PATH),1,MAX_PATH-1)
            then strcpy(ComSpec,'cmd.exe'); // Fallback ComSpec
            strcpy(eCmdLine,'"');
            strcat(eCmdLine,ComSpec);
            if ComSpecModeC
            then strcat(eCmdLine,'" /c ')
            else strcat(eCmdLine,'" /k ');
            if strlen(eCmdLine)+strlen(lpszCmdLine)>=MAX_CMD then begin
                ExitCode:=Fatal(ecFailStart,VerboseMode,'gRun -> Command line too long.','');
                Exit;
            end;
            strcat(eCmdLine,lpszCmdLine);
            lpszCmdLine:=eCmdLine;
        end;

        // Initialize startup info.
        ZeroMemory(@si, sizeof(si));
        si.cb := sizeof(si);
        si.lpDesktop := nil;
        si.dwFlags := STARTF_USESHOWWINDOW;
        si.wShowWindow := DisplayMode;

        // Create process with new console & specified display mode.
        // Uses tail of original command line to create new process.
        CreateProcessResult := CreateProcess( nil, lpszCmdLine, nil, nil, FALSE, CREATE_NEW_CONSOLE or PriorityClass, nil, nil, si, pi );

        // Wait while child running (if /w option specified). Take child exit code.
        if( CreateProcessResult ) then begin
            if ( WaitForSingleObject(pi.hProcess, WaitTimeOut) = WAIT_OBJECT_0 )
                then GetExitCodeProcess(pi.hProcess, DWORD(exitcode));
            CloseHandle( pi.hProcess );
            CloseHandle( pi.hThread );
        end else begin
            ExitCode:=Fatal(ecFailStart,VerboseMode,'gRun -> CreateProcess failed.','');
            Exit;
        end;
    end;
    // Done!
  finally
   Cleanup;
  end;
 except
  ExitCode:=Fatal(ecException,VerboseMode,'gRun -> Exception happened.','');
 end;
end;

begin
 Main;
end.
