'****************************************************************
'** Copyright (c) 2021 Alexey Kuryakin kouriakine@mail.ru      **
'** Under MIT License, see https://opensource.org/licenses/MIT **
'****************************************************************
'** photoviewerc.vbs - call Windows PhotoViewer (console mode) **
'** photoviewerw.vbs - call Windows PhotoViewer (windows mode) **
'** By default try to find PhotoViewer.dll, default for Win10. **
'** As fallback uses shimgvw.dll (old version of PhotoViewer). **
'** Usage:                                                     **
'**  cscript //i //nologo photoviewerc.vbs targret_image.png   **
'**  wscript //i //nologo photoviewerw.vbs targret_image.png   **
'****************************************************************

'******************************************************************************************************
'** Sample1 (WinXP):                                                                                 **
'** rundll32.exe %SystemRoot%\System32\shimgvw.dll,ImageView_Fullscreen file.bmp                     **
'** Sample2 (Win10):                                                                                 **
'** rundll32.exe "%ProgramFiles%\Windows Photo Viewer\PhotoViewer.dll",ImageView_Fullscreen file.bmp **
'******************************************************************************************************

Option Explicit

'******* System objects
'**********************
dim StdIn  : set StdIn  = WScript.StdIn
dim StdOut : set StdOut = WScript.StdOut
dim StdErr : set StdErr = WScript.StdErr
dim Shell  : set Shell  = WScript.CreateObject("WScript.Shell")
dim FSO    : set FSO    = WScript.CreateObject("Scripting.FileSystemObject")

'******* Windows Script Host detection
'*************************************

function WScriptBaseName '*** return "cscript" or "wscript"
 WScriptBaseName=lcase(FSO.GetBaseName(WScript.FullName))
end function

dim IsCScript : IsCScript = false
dim IsWScript : IsWScript = false

sub DetectScriptHost
 select case WScriptBaseName
  case "cscript" : IsCScript=true
  case "wscript" : IsWScript=true
  case else      : WScript.Echo "Error: Unknown WScript host: "+WScriptBaseName+"." : WScript.Quit(3)
 end select
end sub

call DetectScriptHost

'******* Utility routines
'************************

function IsEmptyStr(s)
 IsEmptyStr=not(Trim(s)<>"")
end function

sub PrintStdOut(line)
 if IsCScript then StdOut.WriteLine(line) else call WScript.Echo(line) end if
end sub

sub PrintStdErr(line)
 if IsCScript then StdErr.WriteLine(line) else call WScript.Echo(line) end if
end sub

sub WriteLn(line)
 PrintStdOut(line)
end sub

sub Abort(ExitCode,ErrorMessage)
 PrintStdErr(ErrorMessage)
 WScript.Quit(ExitCode)
end sub

sub Assert(Condition,ExitCode,ErrorMessage)
 if not cbool(Condition) then
  call Abort(ExitCode,ErrorMessage)
 end if
end sub

function atoldef(s,def)
 On Error Resume Next
 dim l : l=clng(s)
 if (Err.Number<>0) then : l=def : end if : Err.Clear
 On Error Goto 0
 atoldef=l
end function

function GetBaseName(path)
 GetBaseName = FSO.GetBaseName(path)
end function

function FileExists(path)
 FileExists = FSO.FileExists(path)
end function

function ExpandEnvironmentStrings(str)
 ExpandEnvironmentStrings = Shell.ExpandEnvironmentStrings(str)
end function

function ShellRun(cmd,show,wait)
 ShellRun = Shell.Run(cmd,show,wait)
end function

function QuotedIfNeed(arg)
 if (InStr(arg," ")>0) and (InStr(arg,"""")=0) then arg=""""+arg+"""" end if
 QuotedIfNeed=arg
end function

'******* Print Version
'*********************

function GetVersion(name)
 GetVersion=ucase(name)+" version 1.0"
end function

sub PrintVersion(name)
 Writeln(GetVersion(name))
end sub

'******* Print help screen
'*************************

sub PrintHelp(name)
 Writeln(GetVersion(name)+vbCrLf+_
  "Utility to call Windows PhotoViewer."+vbCrLf+_
  "Copyright (c) 2021 Alexey Kuryakin kouriakine@mail.ru"+vbCrLf+_
  "Under MIT License, see https://opensource.org/licenses/MIT"+vbCrLf+_
  "Help on "+ucase(name)+":"+vbCrLf+_
  " =================> Syntax:"+vbCrLf+_
  "  "+ucase(name)+" [Options] [Arguments]"+vbCrLf+_
  " =================> Options:"+vbCrLf+_
  "   --            => options ended, next is params"+vbCrLf+_
  "   --version     => print program version and exit"+vbCrLf+_
  "   -h,--help     => print this help screen and exit"+vbCrLf+_
  "   -d,--delay n  => delay [ms] after show each image"+vbCrLf+_
  " ===================> Arguments:"+vbCrLf+_
  "   FileName      => image file name(s) to view"+vbCrLf+_
  " =================> Exit Code:"+vbCrLf+_
  "  0 = Success, else some error found"+vbCrLf+_
  " =================> Examples:"+vbCrLf+_
  "  "+name+" --help"+vbCrLf+_
  "  "+name+" --version"+vbCrLf+_
  "  "+name+" flower.jpg"+vbCrLf+_
  "  "+name+" -d 1000 cat.png dog.bmp bird.jpg")
end sub

'******* Program data and options
'********************************

dim argnum   : argnum   = 0
dim ExitCode : ExitCode = 0
dim Delay    : Delay    = 0

'******* Program routines
'************************

sub SetDelay(arg)
 Delay=atoldef(arg,0)
end sub

function GetViewerDll
 dim dll : dll=ExpandEnvironmentStrings("%ProgramFiles%\Windows Photo Viewer\PhotoViewer.dll")
 if (InStr(GetBaseName(WScript.ScriptName),"photoviewer")=0) then dll="" end if '*** use old style viewer?
 if IsEmptyStr(dll) or not FileExists(dll) then dll=ExpandEnvironmentStrings("%SystemRoot%\System32\shimgvw.dll") end if
 if IsEmptyStr(dll) or not FileExists(dll) then dll="shimgvw.dll" end if '*** it`s fallback solution
 GetViewerDll=dll
end function

function GetViewerCmd(arg)
 GetViewerCmd=trim("rundll32.exe "+QuotedIfNeed(GetViewerDll)+",ImageView_Fullscreen"+" "+arg)
end function

function RunViewerCmd(arg)
 dim result : result=-1
 if not IsEmptyStr(arg) and FileExists(arg) then result=ShellRun(GetViewerCmd(arg),1,false) end if
 RunViewerCmd=result
end function

'******* Handle arguments
'************************

sub HandleArgs(arg)
 argnum=argnum+1
 if (argnum>1) and (delay>0) then WScript.Sleep(Delay) end if
 dim code : code=RunViewerCmd(arg)
 if (code<0) then ExitCode=1 end if
 if IsCScript then
  if (code>=0) then
   PrintStdOut("View: "+arg)
  else
   PrintStdErr("Fail: "+arg)
  end if
 end if
end sub

'******* Command line parsing
'****************************

dim i,arg,opt,isopt : arg="" : opt="" : isopt=true
for i=0 to WScript.Arguments.Count-1
 arg=WScript.Arguments(i)
 if (left(arg,1)="-") and isopt and (opt="") then
  select case arg
   case "--"                    : isopt=false ' end of options, interpret next arguments as params
   case "--version"             : PrintVersion(GetBaseName(WScript.ScriptName)) : WScript.Quit(0)
   case "-h","--help"           : PrintHelp(GetBaseName(WScript.ScriptName))    : WScript.Quit(0)
   case "-d","--delay"          : opt = arg
   case else                    : call Abort(2,"Error: unknown option "+arg+". Call "+GetBaseName(WScript.ScriptName)+" --help.")
  end select
 else
  select case opt
   case ""                      : call HandleArgs(arg) : isopt=false
   case "-d","--delay"          : call SetDelay(arg)
   case else                    : call Abort(2,"Error: unknown option "+opt+". Call "+GetBaseName(WScript.ScriptName)+" --help.")
  end select
  opt=""
 end if
next

'******* Task execution
'**********************

if IsCScript and (WScript.Arguments.Count=0) then : PrintHelp(GetBaseName(WScript.ScriptName)) : WScript.Quit(0) : end if

WScript.Quit(ExitCode)

'******* Done
'************
