'****************************************************************
'** Copyright (c) 2021 Alexey Kuryakin kouriakine@mail.ru      **
'** Under MIT License, see https://opensource.org/licenses/MIT **
'****************************************************************
'** Command line tool to open dialog to ask confirm.           **
'** Usage: call ask-confirm "Title" "Text" && command          **
'****************************************************************

Option Explicit

'******* System objects
'**********************

dim StdIn  : set StdIn  = WScript.StdIn
dim StdOut : set StdOut = WScript.StdOut
dim StdErr : set StdErr = WScript.StdErr
dim Shell  : set Shell  = WScript.CreateObject("WScript.Shell")
dim FSO    : set FSO    = WScript.CreateObject("Scripting.FileSystemObject")

'******* Utility routines
'************************

sub WriteLn(line)
 StdOut.WriteLine(line)
end sub

sub PrintStdErr(line)
 StdErr.WriteLine(line)
end sub

sub Abort(ExitCode,ErrorMessage)
 PrintStdErr(ErrorMessage)
 WScript.Quit(ExitCode)
end sub

sub Assert(Condition,ExitCode,ErrorMessage)
 if not cbool(Condition) then
  call Abort(ExitCode,ErrorMessage)
 end if
end sub

function atoldef(s,def)
 On Error Resume Next
 dim l : l=clng(s)
 if (Err.Number<>0) then : l=def : end if : Err.Clear
 On Error Goto 0
 atoldef=l
end function

function GetEnv(VarName)
 GetEnv = Shell.Environment("PROCESS").Item(VarName)
end function

function GetBaseName(path)
 GetBaseName = FSO.GetBaseName(path)
end function

function MessageBox(Caption,Message,Buttons,Timeout)
 if IsEmpty(Timeout) then
  MessageBox = MsgBox(Message,Buttons,Caption)
 else
  MessageBox = Shell.Popup(Message,Timeout,Caption,Buttons)
 end if
end function

'******* Print Version
'*********************

sub PrintVersion(name)
 WriteLn(ucase(name)+" version 1.0")
end sub

'******* Print help screen
'*************************

sub PrintHelp(name)
 PrintVersion(name)
 WriteLn("Command to ask confirmation by YesNoCancel dialog.")
 WriteLn("Copyright (c) 2021 Alexey Kuryakin kouriakine@mail.ru")
 WriteLn("Under MIT License, see https://opensource.org/licenses/MIT")
 WriteLn("Help on "+ucase(name)+":")
 WriteLn(" =================> Syntax:")
 WriteLn("  "+ucase(name)+" [Arguments] [Options] ")
 WriteLn(" =================> Options:")
 WriteLn("   --            => options ended, next is params")
 WriteLn("   --version     => print program version and exit")
 WriteLn("   -h,--help     => print this help screen and exit")
 WriteLn("   -t t          => set timeout t seconds to cancel")
 WriteLn("   --timeout t   => set timeout t seconds to cancel")
 WriteLn(" =================> Arguments:")
 WriteLn("   Title         => Window title for confirmation dialog")
 WriteLn("   Message       => Message text for confirmation dialog")
 WriteLn(" =================> Exit Code:")
 WriteLn("   0             => Yes button clicked (operation confirmed)")
 WriteLn("   1             => No, Cancel clicked (operation cancelled)")
 WriteLn("   2             => some error found (bad arguments/options)")
 WriteLn("   else          => some error found (internal script error)")
 WriteLn(" =================> Note:")
 WriteLn("   Environment variable (ask-confirm-timeout) may be used to set timeout.")
 WriteLn("   If any button was not clicked until timeout, the operations cancelled.")
 WriteLn(" =================> Examples:")
 WriteLn("  call "+name+" --help")
 WriteLn("  call "+name+" --version")
 WriteLn("  call "+name+" ""Confirm"" ""Apply operation?"" && cmd /c echo Applied.")
 WriteLn("  call "+name+" ""Confirm"" ""Delete file?"" -t 30 && cmd /c echo File deleted.")
 WriteLn("  call "+name+" ""Confirm please"" ""Do it now?"" --timeout 10 && cmd /c echo Done.")
end sub

'******* Program data and options
'********************************

dim argnum  : argnum  = 0
dim Title   : Title   = ""
dim Message : Message = ""
dim Timeout : Timeout = Empty

'******* Program routines
'************************

sub SetTimeout(s)
 if (s="") then : exit sub : end if
 dim t : t=atoldef(s,0)
 if (t>0) then : Timeout=t : end if
end sub

'******* Handle arguments
'************************

sub HandleArgs(arg)
 argnum=argnum+1
 select case argnum
  case 1    : Title=arg
  case 2    : Message=arg
  case else : call Abort(2,"Error: excess argument found. See --help.")
 end select
end sub

'******* Command line parsing
'****************************

dim i,arg,opt,isopt : arg="" : opt="" : isopt=true
for i=0 to WScript.Arguments.Count-1
 arg=WScript.Arguments(i)
 if (left(arg,1)="-") and isopt and (opt="") then
  select case arg
   case "--"                    : isopt=false ' end of options, interpret next arguments as params
   case "--version"             : PrintVersion(GetBaseName(WScript.ScriptName)) : WScript.Quit(0)
   case "-h","--help"           : PrintHelp(GetBaseName(WScript.ScriptName))    : WScript.Quit(0)
   case "-t","--timeout"        : opt=arg
   case else                    : call Abort(2,"Error: unknown option "+arg+". See --help.")
  end select
 else
  select case opt
   case "-t","--timeout"        : call SetTimeout(arg)
   case ""                      : call HandleArgs(arg)
   case else                    : call Abort(2,"Error: unknown option "+opt+". See --help.")
  end select
  opt=""
 end if
next

'******* Task execution
'**********************

if (WScript.Arguments.Count=0) then : PrintHelp(GetBaseName(WScript.ScriptName)) : WScript.Quit(0) : end if

call Assert(WScript.Interactive,2,"Error: "+WScript.Name+" is not Interactive. Use //I option.")

call Assert(Title<>"",2,"Error: Title is not specified. See --help.")
call Assert(Message<>"",2,"Error: Message is not specified. See --help.")

if IsEmpty(Timeout) then : call SetTimeout(GetEnv("ask-confirm-timeout")) : end if

select case MessageBox(Title,Message,vbYesNoCancel+vbQuestion,Timeout)
 case vbOk     : WScript.Quit(0)
 case vbCancel : WScript.Quit(1)
 case vbAbort  : WScript.Quit(1)
 case vbRetry  : WScript.Quit(0)
 case vbIgnore : WScript.Quit(0)
 case vbYes    : WScript.Quit(0)
 case vbNo     : WScript.Quit(1)
 case -1       : WScript.Quit(1)
 case else     : WScript.Quit(1)
end select

WScript.Quit(0)

'******* Done
'************
