unit uhistory;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, StdCtrls, lazUTF8;

type

  PStep = ^TStep;

  TStep = record           
    BytePos  : SizeInt;
    SelStart : SizeInt;
    InsStr   : string;
    DelStr   : string;
  end;

  { THistory }

  THistory = class
  private
    FMemo        : TMemo;
    FSteps       : TList;         // History Records
    FIndex       : Integer;       // Based 0
    FOldOnChange : TNotifyEvent;
    FInEdit      : Boolean;
    FPrevContent : string;

    function    GetStep(AIndex: Integer): PStep;  
    function    CurStep: PStep;

    procedure   AddStep(ABytePos, ASelStart: SizeInt; AInsStr, ADelStr: string);
    procedure   DelStep(Index: Integer);

    procedure   MemoOnChange(Sender: TObject);

    function    StrDiff(out ABytePos, ASelStart: SizeInt; out AInsStr, ADelStr: string): Boolean;
  public
    constructor Create(AMemo: TMemo);
    destructor  Destroy; override;

    function    CanUndo: Boolean;
    function    CanRedo: Boolean;
    procedure   Undo;
    procedure   Redo;
  end;

implementation

{ THistory }

function THistory.GetStep(AIndex: Integer): PStep;
begin
  Result := PStep(FSteps[AIndex]);
end;

function THistory.CurStep: PStep;
begin
  Result := GetStep(FIndex);
end;
      
procedure THistory.AddStep(ABytePos, ASelStart: SizeInt; AInsStr, ADelStr: string);
begin
  DelStep(FIndex + 1);

  FSteps.Add(new(PStep));
  Inc(FIndex);
  with CurStep^ do begin
    BytePos  := ABytePos;
    SelStart := ASelStart;
    InsStr   := AInsStr;
    DelStr   := ADelStr;
  end;
end;

procedure THistory.DelStep(Index: Integer);
var
  i: Integer;
begin
  for i := FSteps.Count - 1 downto Index do begin
    GetStep(i)^.InsStr := '';
    GetStep(i)^.DelStr := '';
    dispose(GetStep(i));
    FSteps.Delete(i);
  end;
  FIndex := Index - 1;
end;

procedure THistory.MemoOnChange(Sender: TObject);
var
  InsStr, DelStr: string;
  BytePos, SelStart: SizeInt;
  // TickCount: Int64;
begin
  // TickCount := GetTickCount64;
  if FInEdit and StrDiff(BytePos, SelStart, InsStr, DelStr) then begin
    // write(GetTickCount64 - TickCount, ' + ');
    // TickCount := GetTickCount64;
    AddStep(BytePos, SelStart, InsStr, DelStr);
    FPrevContent := FMemo.Text;
  end;           
  // writeln(GetTickCount64 - TickCount);

  if Assigned(FOldOnChange) then
    FOldOnChange(Sender);
end;

function THistory.StrDiff(out ABytePos, ASelStart: SizeInt; out AInsStr, ADelStr: string): Boolean;
var
  CurContent, CurPos, CurEnd, PrevPos: PChar;
  CurLen, PrevLen, DiffLen: SizeInt;
  CharLen: Integer;
begin
  CurContent := PChar(FMemo.Text);
  CurPos := CurContent;
  PrevPos := PChar(FPrevContent);

  CurLen := Length(CurPos);
  PrevLen := Length(PrevPos);
  DiffLen := CurLen - PrevLen;

  if DiffLen < 0 then
    CurEnd := CurPos + CurLen - 1
  else if DiffLen > 0 then
    CurEnd := CurPos + PrevLen - 1
  else begin
    Result := False;
    Exit;
  end;

  while (CurPos <= CurEnd) do begin
    if CurPos^ <> PrevPos^ then
      Break;
    Inc(CurPos);
    Inc(PrevPos);
  end;
  Utf8TryFindCodepointStart(CurContent, CurPos, CharLen);
  ABytePos := CurPos - CurContent + 1;
  if DiffLen > 0 then begin
    AInsStr := Copy(FMemo.Text, ABytePos, DiffLen);
    ADelStr := '';
  end else if DiffLen < 0 then begin
    AInsStr := '';
    ADelStr := Copy(FPrevContent, ABytePos, -DiffLen);
  end;  
  ASelStart := FMemo.SelStart;

  Result := True;
end;

constructor THistory.Create(AMemo: TMemo);
begin
  inherited Create;
  FSteps := TList.Create;
  FMemo := AMemo;
  FOldOnChange := FMemo.OnChange;
  FMemo.OnChange := @MemoOnChange; 
  FPrevContent := FMemo.Text;
  FIndex := -1;
  FInEdit := True;
end;

destructor THistory.Destroy;
begin
  FInEdit := False;
  FMemo.OnChange := FOldOnChange;
  FMemo := nil;
  FPrevContent := '';
  DelStep(0);
  FSteps.Free;
  inherited Destroy;
end;

function THistory.CanUndo: Boolean;
begin
  Result := FIndex >= 0;
end;

function THistory.CanRedo: Boolean;
begin
  Result := FIndex < FSteps.Count - 1;
end;

procedure THistory.Undo;
var
  NewSelStart: SizeInt;
begin
  if FIndex < 0 then Exit;
  FInEdit := False;

  // FPrevContent == FMemo.Text
  with CurStep^ do begin
    if InsStr <> '' then begin
      Delete(FPrevContent, BytePos, Length(InsStr));
      NewSelStart := SelStart - UTF8LengthFast(InsStr);
    end else if DelStr <> '' then begin
      Insert(DelStr, FPrevContent, BytePos);
      NewSelStart := SelStart + UTF8LengthFast(DelStr);
    end else begin
      writeln('Invalid history data!');
    end;
  end;

  FMemo.Text := FPrevContent; 
  FMemo.SelStart := NewSelStart;
  FInEdit := True;

  Dec(FIndex);
end;

procedure THistory.Redo;
var
  NewSelStart: SizeInt;
begin            
  if FIndex >= FSteps.Count - 1 then Exit;
  FInEdit := False;
                  
  Inc(FIndex);
  // FPrevContent == FMemo.Text
  with CurStep^ do begin
    if InsStr <> '' then begin
      Insert(InsStr, FPrevContent, BytePos);
      NewSelStart := SelStart + UTF8LengthFast(InsStr);
    end else if DelStr <> '' then begin
      Delete(FPrevContent, BytePos, Length(DelStr));
      NewSelStart := SelStart - UTF8LengthFast(DelStr);
    end else begin
      writeln('Invalid history data!');
    end;
  end;

  FMemo.Text := FPrevContent;
  FMemo.SelStart := NewSelStart;
  FInEdit := True;
end;

end.

