////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Text editor window.                                                        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231121 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_texteditor; // Form Text Editor

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ClipBrd, ImgList,
 printer4lazarus, PrintersDlgs,
 lcltype, lclintf, Printers,
 Form_CrwDaqSysChild,
 Form_ListBoxSelection,
 _crw_alloc, _crw_str, _crw_eldraw, _crw_dynar,
 _crw_guard, _crw_fio, _crw_rtc, _crw_ef,
 _crw_sect, ptembed, _crw_utf8, _crw_memo,
 _crw_crypt, _crw_hash, _crw_colors,
 _crw_gloss, _crw_flgrd, _crw_sesman,
 _crw_appforms, _crw_apptools, _crw_apputils;

type

  { TFormTextEditor }

  TFormTextEditor = class(TFormCrwDaqSysChild)
    Editor: TMemo;
    FontDialog: TFontDialog;
    PrintDialog: TPrintDialog;
    ActionEditReadOnly: TAction;
    ActionEditDelete: TAction;
    ActionEditCut: TAction;
    ActionEditCopy: TAction;
    ActionEditPaste: TAction;
    ActionEditUndo: TAction;
    ActionEditRedo: TAction;
    ActionEditOblivion: TAction;
    ActionEditFind: TAction;
    ActionEditReplace: TAction;
    ActionEditAgain: TAction;
    ActionEditSelectAll: TAction;
    ActionEditSelectNone: TAction;
    ActionEditGoToStart: TAction;
    ActionEditGoToEnd: TAction;
    ActionEditGoToLineN: TAction;
    ActionEditRefresh: TAction;
    ActionEditReopen: TAction;
    ActionEditCheckFileChanged: TAction;
    ActionEditFont: TAction;
    ActionEditWordWrap: TAction;
    ActionEditReadTable: TAction;
    ActionEditInsertTemplate: TAction;
    ActionEditShiftLeft: TAction;
    ActionEditShiftRight: TAction;
    MenuEdit: TMenuItem;
    MenuEditReadOnly: TMenuItem;
    MenuEditReadOnlySeparator: TMenuItem;
    MenuEditDelete: TMenuItem;
    MenuEditCut: TMenuItem;
    MenuEditCopy: TMenuItem;
    MenuEditPaste: TMenuItem;
    MenuEditUndo: TMenuItem;
    MenuEditRedo: TMenuItem;
    MenuEditOblivion: TMenuItem;
    MenuEditFindSeparator: TMenuItem;
    MenuEditFind: TMenuItem;
    MenuEditReplace: TMenuItem;
    MenuEditAgain: TMenuItem;
    MenuEditSelectAllSeparator: TMenuItem;
    MenuEditSelectAll: TMenuItem;
    MenuEditSelectNone: TMenuItem;
    MenuEditGoToStartSeparator: TMenuItem;
    MenuEditGoToStart: TMenuItem;
    MenuEditGoToEnd: TMenuItem;
    MenuEditGoToLineN: TMenuItem;
    MenuEditRefreshSeparator: TMenuItem;
    MenuEditRefresh: TMenuItem;
    MenuEditReopen: TMenuItem;
    MenuEditFont: TMenuItem;
    MenuEditWordWrap: TMenuItem;
    MenuEditWordWrapSeparator: TMenuItem;
    MenuEditReadTable: TMenuItem;
    MenuEditShiftLeft: TMenuItem;
    MenuEditShiftRight: TMenuItem;
    MenuEditShiftRightSeparator: TMenuItem;
    MenuEditInsertTemplate: TMenuItem;
    ToolButtonFilePrint: TToolButton;
    ToolButtonFileSave: TToolButton;
    ToolButtonFileSaveSeparator: TToolButton;
    ToolButtonEditReadOnly: TToolButton;
    ToolButtonEditReadOnlySeparator: TToolButton;
    ToolButtonEditDelete: TToolButton;
    ToolButtonEditCut: TToolButton;
    ToolButtonEditCopy: TToolButton;
    ToolButtonEditPaste: TToolButton;
    ToolButtonEditUndo: TToolButton;
    ToolButtonEditRedo: TToolButton;
    ToolButtonEditOblivionSeparator: TToolButton;
    ToolButtonEditOblivion: TToolButton;
    ToolButtonEditFindSeparator: TToolButton;
    ToolButtonEditFind: TToolButton;
    ToolButtonEditReplace: TToolButton;
    ToolButtonEditAgain: TToolButton;
    ToolButtonEditSelectAllSeparator: TToolButton;
    ToolButtonEditSelectAll: TToolButton;
    ToolButtonEditSelectNone: TToolButton;
    ToolButtonEditGoToStartSeparator: TToolButton;
    ToolButtonEditGoToStart: TToolButton;
    ToolButtonEditGoToEnd: TToolButton;
    ToolButtonEditGoToLineN: TToolButton;
    ToolButtonEditRefreshSeparator: TToolButton;
    ToolButtonEditReopen: TToolButton;
    ToolButtonEditWordWrap: TToolButton;
    ToolButtonEditFont: TToolButton;
    ToolButtonEditReadTableSeparator: TToolButton;
    ToolButtonEditReadTable: TToolButton;
    ToolButtonEditInsertTemplateSeparator: TToolButton;
    ToolButtonEditInsertTemplate: TToolButton;
    ToolButtonEditShiftLeft: TToolButton;
    ToolButtonEditShiftRight: TToolButton;
    PopupMenuEditReadOnly: TMenuItem;
    PopupMenuFile: TMenuItem;
    PopupMenuFileSave: TMenuItem;
    PopupMenuFilePrint: TMenuItem;
    PopupMenuEdit: TMenuItem;
    PopupMenuEditCut: TMenuItem;
    PopupMenuEditCopy: TMenuItem;
    PopupMenuEditPaste: TMenuItem;
    PopupMenuEditUndo: TMenuItem;
    PopupMenuEditRedo: TMenuItem;
    PopupMenuEditOblivion: TMenuItem;
    PopupMenuEditSelectAll: TMenuItem;
    PopupMenuEditSelectNone: TMenuItem;
    PopupMenuFind: TMenuItem;
    PopupMenuEditFind: TMenuItem;
    PopupMenuEditReplace: TMenuItem;
    PopupMenuEditAgain: TMenuItem;
    PopupMenuEditGoToLineN: TMenuItem;
    PopupMenuEditGoToStart: TMenuItem;
    PopupMenuEditGoToEnd: TMenuItem;
    PopupMenuTool: TMenuItem;
    PopupMenuEditRefresh: TMenuItem;
    PopupMenuEditReopen: TMenuItem;
    PopupMenuEditFont: TMenuItem;
    PopupMenuEditWordWrap: TMenuItem;
    PopupMenuEditReadTable: TMenuItem;
    PopupMenuEditShiftLeft: TMenuItem;
    PopupMenuEditShiftRight: TMenuItem;
    PopupMenuEditInsertTemplate: TMenuItem;
    PopupMenuFileSaveAs: TMenuItem;
    PopupMenuFilePrinterSetup: TMenuItem;
    PopupMenuEditDelete: TMenuItem;
    procedure FormActivate(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormDeactivate(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure ActionEditReadOnlyExecute(Sender: TObject);
    procedure ActionEditDeleteExecute(Sender: TObject);
    procedure ActionEditCutExecute(Sender: TObject);
    procedure ActionEditCopyExecute(Sender: TObject);
    procedure ActionEditPasteExecute(Sender: TObject);
    procedure ActionEditUndoExecute(Sender: TObject);
    procedure ActionEditRedoExecute(Sender: TObject);
    procedure ActionEditOblivionExecute(Sender: TObject);
    procedure ActionEditFindExecute(Sender: TObject);
    procedure ActionEditReplaceExecute(Sender: TObject);
    procedure ActionEditAgainExecute(Sender: TObject);
    procedure ActionEditSelectAllExecute(Sender: TObject);
    procedure ActionEditSelectNoneExecute(Sender: TObject);
    procedure ActionEditGoToStartExecute(Sender: TObject);
    procedure ActionEditGoToEndExecute(Sender: TObject);
    procedure ActionEditGoToLineNExecute(Sender: TObject);
    procedure ActionEditRefreshExecute(Sender: TObject);
    procedure ActionEditReopenExecute(Sender: TObject);
    procedure ActionEditFontExecute(Sender: TObject);
    procedure ActionEditWordWrapExecute(Sender: TObject);
    procedure ActionEditReadTableExecute(Sender: TObject);
    procedure ActionEditInsertTemplateExecute(Sender: TObject);
    procedure ActionEditShiftLeftExecute(Sender: TObject);
    procedure ActionEditShiftRightExecute(Sender: TObject);
    procedure ActionEditCheckFileChangedExecute(Sender: TObject);
  private
    { Private declarations }
    myFindDialog : TMasterForm;
    myReplaceDialog : TMasterForm;
    myLastFind : (lfNone,lfFind,lfReplace);
    myPathName : LongString;
    myLinePos  : TPoint;
    myFileSize : Int64;
    myFileDate : Int64;
    myLastSel  : packed record
     Start,Len : Integer;
     HashCode  : QWord;
     TimeStamp : QWord;
    end;
    myAnnals   : TUndoHistory;
    function  GetPathName:LongString;
    procedure SetPathName(const aPathName:LongString);
    function  CheckFileChanged(Mode:Integer):Boolean;
    function  GetIsReadOnly:Boolean;
    procedure SetIsReadOnly(aReadOnly:Boolean);
    function  EvalHash(const S:LongString):QWord;
    procedure ZeroLastSelection;
    procedure SaveLastSelection;
    procedure BackLastSelection;
    procedure SetFocusToEditor(PM:Boolean=true);
    function  GetAnnals:TUndoHistory;
  public
    { Public declarations }
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
    procedure UpdateCommands; override;
    procedure PrintView; override;
    procedure Open(const AFileName: LongString);
    procedure FileSave; override;
    procedure FileSaveAs; override;
    procedure Reopen; virtual;
    property  PathName:LongString read GetPathName write SetPathName;
    property  IsReadOnly:Boolean read GetIsReadOnly write SetIsReadOnly;
    procedure FindText(const aFindText:LongString; aOptions:Integer);
    procedure ReplaceText(const aFindText,aReplaceText:LongString; aOptions:Integer);
    procedure TextShiftExecute(Shift:Integer);
    procedure GoToRowCol(aRow,aCol:Integer);
    procedure DefaultReadOnly;
    function  HasAnnals:Boolean;
    property  Annals:TUndoHistory read GetAnnals;
  private
    function  GetPerformReadOnly:Boolean;
    procedure SetPerformReadOnly(aReadOnly:Boolean);
    function  GetPerformModified:Boolean;
    procedure SetPerformModified(aModified:Boolean);
    function  GetPerformSelLength:Integer;
    procedure SetPerformSelLength(aLength:Integer);
    function  GetPerformSelText:LongString;
    procedure SetPerformSelText(const aText:LongString);
    function  GetPerformText:LongString;
    procedure SetPerformText(const aText:LongString);
    function  GetPerformWideText:WideString;
    procedure SetPerformWideText(const aText:WideString);
    function  GetPerformLinesText:LongString;
    procedure SetPerformLinesText(const aText:LongString);
    function  GetPerformSelStart:Integer;
    procedure SetPerformSelStart(aSelStart:Integer);
    function  GetPerformCaretPos:TPoint;
    procedure SetPerformCaretPos(const aCaretPos:TPoint);
    function  GetPerformWordWrap:Boolean;
    procedure SetPerformWordWrap(aWordWrap:Boolean);
  public
    procedure PerformUndo;
    procedure PerformRedo;
    function  PerformCanUndo:Boolean;
    function  PerformCanRedo:Boolean;
    procedure PerformClearSelection;
    procedure PerformCutToClipboard;
    procedure PerformCopyToClipboard;
    procedure PerformPasteFromClipboard;
    procedure PerformPasteText(const aText:LongString);
    procedure PerformOblivion(Mode:Integer=3);
    procedure PerformSelectAll;
    procedure PerformSelectNone;
    procedure PerformGotoStart;
    procedure PerformGotoEnd;
    procedure PerformGoToRowCol(aRow,aCol:Integer);
    procedure PerformSetFocus(PM:Boolean=true);
  public
    property  PerformText:LongString read GetPerformText write SetPerformText;
    property  PerformWideText:WideString read GetPerformWideText write SetPerformWideText;
    property  PerformSelText:LongString read GetPerformSelText write SetPerformSelText;
    property  PerformSelLength:Integer read GetPerformSelLength write SetPerformSelLength;
    property  PerformLinesText:LongString read GetPerformLinesText write SetPerformLinesText;
    property  PerformSelStart:Integer read GetPerformSelStart write SetPerformSelStart;
    property  PerformModified:Boolean read GetPerformModified write SetPerformModified;
    property  PerformReadOnly:Boolean read GetPerformReadOnly write SetPerformReadOnly;
    property  PerformCaretPos:TPoint read GetPerformCaretPos write SetPerformCaretPos;
    property  PerformWordWrap:Boolean read GetPerformWordWrap write SetPerformWordWrap;
  public
    function  ReadLastSelection(out aSelStart,aSelLength:Integer; out aSelText:LongString):Boolean;
    function  IsFileProtected(Mode:Integer=0):Boolean;
    function  EditorPosParams:LongString;
  public
    class function NoNamePath:LongString;
    class function IsNoNamePath(const aPath:LongString):Boolean;
  end;

type
  TTextEditorList = class(TObjectStorage)
  private
    function   GetEditor(i:Integer):TFormTextEditor;
    procedure  SetEditor(i:Integer; aEditor:TFormTextEditor);
  public
    property   Editor[i:Integer]:TFormTextEditor read GetEditor write SetEditor; default;
    procedure  SaveAll;
  end;

function  NewTextEditor(const FileName: LongString=''):TFormTextEditor;
procedure Kill(var TheObject:TFormTextEditor); overload;

procedure Timer_MonitorTextEditor;

function  NewTextEditorList(aOwnsObjects : Boolean = true;
                            aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                            aStep        : LongInt = DefaultTObjectStorageStep
                                       ) : TTextEditorList;
procedure Kill(var TheObject:TTextEditorList); overload;

function  ActiveTextEditor:TFormTextEditor;
function  FindTextEditor(const FileName:LongString;
                               CreateNew:Boolean=false;
                               BringToFront:Boolean=false):TFormTextEditor;

const
 FullTextEditorList : TTextEditorList = nil;
 DefaultEditorFont : TFontParams = (
  CharSet : RUSSIAN_CHARSET;
  Color   : clBlack;
  Height  : -13;
  Name    : 'PT Mono';
  Pitch   : fpFixed;
  Style   : [];
 );

const
 TextEditorsPalette : packed record
  MainBackColor : TColor;
  MainTextColor : TColor;
  ReadOnlyColor : TColor;
 end = (
  MainBackColor : clWhite;
  MainTextColor : clBlack;
  ReadOnlyColor : clInfoBk;
 );

procedure Init_TextEditorsPalette;

const
 fr_WholeWord       = $01;
 fr_MatchCase       = $02;
 fr_PromptOnReplace = $04;
 fr_Replace         = $08;
 fr_ReplaceAll      = $10;

implementation

uses
 Form_CrwDaq,
 Form_CurveWindow,
 Form_TextEditorTemplates,
 Form_TextReplaceDialog,
 Form_TextFindDialog;

{$R *.lfm}

 {
 ******************************
 TTextEditorList implementation
 ******************************
 }
function   TTextEditorList.GetEditor(i:Integer):TFormTextEditor;
begin
 Result:=TFormTextEditor(Items[i]);
end;

procedure  TTextEditorList.SetEditor(i:Integer; aEditor:TFormTextEditor);
begin
 Items[i]:=aEditor;
end;

procedure do_save_editor(Index:LongInt; const aObject:TObject; var Terminate:Boolean; CustomData:Pointer);
begin
 if aObject is TFormTextEditor then
 with aObject as TFormTextEditor do begin
  // Do not autosave NoName documents
  if IsNoNamePath(PathName) then begin
   PerformModified:=false;
   PerformReadOnly:=true;
  end;
  // Autosave modified documents if one exists
  if PerformModified and not PerformReadOnly then begin
   if FileExists(PathName) then FileSave;
   PerformModified:=false;
  end;
 end;
end;

procedure TTextEditorList.SaveAll;
begin
 if Ok then
 try
  ForEach(do_save_editor,nil);
 except
  on E:Exception do BugReport(E,Self,'SaveAll');
 end;
end;

function  NewTextEditorList(aOwnsObjects : Boolean = true;
                            aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                            aStep        : LongInt = DefaultTObjectStorageStep
                                       ) : TTextEditorList;
begin
 Result:=nil;
 try
  Result:=TTextEditorList.Create(aOwnsObjects,aCapacity,aStep);
 except
  on E:Exception do BugReport(E,nil,'NewTextEditorList');
 end;
end;

procedure Kill(var TheObject:TTextEditorList); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end;
end;

 {
 ************************
 General purpose routines
 ************************
 }
const
 NoNameList='noname,безымянный';

class function TFormTextEditor.NoNamePath:LongString;
begin
 Result:=RusEng(ExtractWord(2,NoNameList,ScanSpaces),ExtractWord(1,NoNameList,ScanSpaces));
end;

class function TFormTextEditor.IsNoNamePath(const aPath:LongString):Boolean;
var Name:LongString;
begin
 if (DefaultSystemCodePage<>CP_UTF8)
 then Name:=LowerCase(ExtractFileName(aPath))
 else Name:=utf8_lowercase(ExtractFileName(aPath));
 Result:=(WordIndex(Name,NoNameList,ScanSpaces)>0);
end;

function NewTextEditor(const FileName:LongString=''):TFormTextEditor;
begin
 Result:=nil;
 try
  Result:=TFormTextEditor.Create(Application.MainForm);
  if IsNonEmptyStr(FileName) then begin
   Result.Open(UnifyFileAlias(FileName));
   Result.DefaultReadOnly;
  end;
 except
  on E:Exception do BugReport(E,nil,'NewTextEditor');
 end;
end;

procedure Kill(var TheObject:TFormTextEditor); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

function ActiveTextEditor:TFormTextEditor;
var Child:TForm;
begin
 Result:=nil;
 try
  if SdiMan.FindActiveChild(Child,sf_SdiChild,sf_SdiControl) then
  if (Child is TFormTextEditor) then Result:=TFormTextEditor(Child);
 except
  on E:Exception do BugReport(E,nil,'ActiveTextEditor');
 end;
end;

type
 TFindRec = packed record
  Match    : TFormTextEditor;
  PathName : LongString;
 end;

procedure FindForm(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 with TFindRec(Custom^) do
 if (Form is TFormTextEditor) then
 if IsSameFileName(TFormTextEditor(Form).PathName,PathName) then begin
  Match:=TFormTextEditor(Form);
  Terminate:=true;
 end;
end;

function  FindTextEditor(const FileName:LongString;
                               CreateNew:Boolean=false;
                               BringToFront:Boolean=false):TFormTextEditor;
var
 FindRec:TFindRec;
begin
 Result:=nil;
 try
  FindRec:=Default(TFindRec);
  try
   FindRec.Match:=nil;
   FindRec.PathName:=StringBuffer(FileName);
   if (FileName<>'') then SdiMan.ForEachChild(FindForm,@FindRec);
   Result:=FindRec.Match;
   if CreateNew and not Assigned(Result) then Result:=NewTextEditor(FileName);
   if Assigned(Result) and BringToFront then begin
    Result.Show;
    Result.WindowState:=wsNormal;
    SdiMan.ActivateChild(Result);
   end;
  finally
   FindRec.PathName:='';
  end;
 except
  on E:Exception do BugReport(E,nil,'FindTextEditor');
 end;
end;

procedure Timer_MonitorTextEditor;
const LastActiveForm:TForm=nil; var Form:TForm;
begin
 try
  if (Application=nil) then Exit;
  if not Application.Active then begin LastActiveForm:=nil; Exit; end;
  if (Application.MainForm=nil) then Exit;
  Form:=SdiMan.ActiveChild;
  if (Form=nil) then begin LastActiveForm:=nil; Exit; end;
  if not Form.Active then begin LastActiveForm:=nil; Exit; end;
  if (LastActiveForm=Form) then Exit else LastActiveForm:=Form;
  if (Form is TFormTextEditor) then TFormTextEditor(Form).ActionEditCheckFileChanged.Execute;
 except
  on E:Exception do BugReport(E,nil,'Timer_MonitorTextEditor');
 end;
end;

 {
 ******************************
 TFormTextEditor implementation
 ******************************
 }
function  TFormTextEditor.GetPathName:LongString;
begin
 if Assigned(Self) then Result:=myPathName else Result:='';
end;

procedure TFormTextEditor.SetPathName(const aPathName:LongString);
begin
 if Assigned(Self) then begin
  if IsNoNamePath(aPathName)
  then myPathName:=NoNamePath
  else myPathName:=UnifyFileAlias(aPathName,ua_RealPath);
  Caption:=myPathName;
 end;
end;

function TFormTextEditor.GetIsReadOnly:Boolean;
begin
 Result:=PerformReadOnly;
end;

procedure TFormTextEditor.SetIsReadOnly(aReadOnly:Boolean);
begin
 PerformReadOnly:=aReadOnly;
end;

function TFormTextEditor.GetAnnals:TUndoHistory;
begin
 if Assigned(Self)
 then Result:=myAnnals
 else Result:=nil;
end;

function TFormTextEditor.HasAnnals:Boolean;
begin
 if Assigned(Self)
 then Result:=Assigned(myAnnals)
 else Result:=false;
end;

function TFormTextEditor.EvalHash(const S:LongString):QWord;
begin
 if (S='') then Exit(0);
 if IsCpu64
 then Result:=Hash64_RS(PChar(S),Length(S))
 else Result:=Hash32_RS(PChar(S),Length(S));
end;

procedure TFormTextEditor.ZeroLastSelection;
begin
 if Assigned(Self) then begin
  myLastSel.Len:=0;
  myLastSel.Start:=0;
  myLastSel.HashCode:=0;
  myLastSel.TimeStamp:=0;
 end;
end;

procedure TFormTextEditor.SaveLastSelection;
begin
 if Assigned(Self) then begin
  ZeroLastSelection;
  if (PerformSelLength>0) then begin
   myLastSel.Len:=PerformSelLength;
   myLastSel.Start:=PerformSelStart;
   myLastSel.TimeStamp:=GetTickCount64;
   myLastSel.HashCode:=EvalHash(PerformLinesText);
  end;
 end;
end;

procedure TFormTextEditor.BackLastSelection;
var Hash:QWord;
begin
 if Assigned(Self) then begin
  if (myLastSel.Len>0) then begin
   Hash:=EvalHash(PerformLinesText);
   if (Hash=myLastSel.HashCode) then begin
    PerformSelStart:=myLastSel.Start;
    PerformSelLength:=myLastSel.Len;
   end;
  end;
  ZeroLastSelection;
 end;
end;

function TFormTextEditor.ReadLastSelection(out aSelStart,aSelLength:Integer;
                                           out aSelText:LongString):Boolean;
var Hash:QWord;
begin
 Result:=false;
 aSelStart:=0; aSelLength:=0; aSelText:='';
 if Assigned(Self) then begin
  if (myLastSel.Len>0) then begin
   Hash:=EvalHash(PerformLinesText);
   if (Hash=myLastSel.HashCode) then begin
    aSelStart:=myLastSel.Start;
    aSelLength:=myLastSel.Len;
    aSelText:=WideToStr(Copy(PerformWideText,1+aSelStart,aSelLength));
    Result:=true;
   end;
  end;
 end;
end;

function TFormTextEditor.EditorPosParams:LongString;
begin
 if Ok then Result:=ControlPosParams(Editor) else Result:='';
end;

procedure TFormTextEditor.SetFocusToEditor(PM:Boolean=true);
begin
 if Assigned(Self) then PerformSetFocus(PM);
end;

procedure TFormTextEditor.FormCreate(Sender: TObject);
begin
 inherited;
 myAnnals:=NewUndoHistory(Editor);
 myAnnals.Master:=@myAnnals;
 myLinePos:=Point(1,1);
 myFindDialog:=nil;
 myReplaceDialog:=nil;
 PathName:=NoNamePath;
 myLastFind:=lfNone;
 RestoreFont(Editor.Font,DefaultEditorFont);
 ActionEditRefreshExecute(Self);
 UpdateMenu(MenuEdit,
            RusEng('Правка','Edit')+MenuRightSpace,
            RusEng('Меню для редактирования.','Editor operations.'),
            0);
 UpdateMenu(MenuEditReadOnly,
            RusEng('Редактор\запретить','Edit\disable'),
            RusEng('Разрешить\запретить редактирование текста.','Enable\disable edit text.'),
            ShortCut(VK_F4,[]));
 UpdateMenu(MenuEditDelete,
            RusEng('Удалить','Delete'),
            RusEng('Удалить выделенный фрагмент.','Delete selected region.'),
            ShortCut(VK_DELETE,[ssCtrl]));
 UpdateMenu(MenuEditCut,
            RusEng('Вырезать','Cut'),
            RusEng('Вырезать в буфер обмена.','Cut to clipboard.'),
            ShortCut(Word('X'),[ssCtrl]));
 UpdateMenu(MenuEditCopy,
            RusEng('Копировать','Copy'),
            RusEng('Копировать в буфер обмена.','Copy to clipboard.'),
            ShortCut(Word('C'),[ssCtrl]));
 UpdateMenu(MenuEditPaste,
            RusEng('Вставить','Paste'),
            RusEng('Вставить из буфера обмена.','Paste from clipboard.'),
            ShortCut(Word('V'),[ssCtrl]));
 UpdateMenu(MenuEditUndo,
            RusEng('Отменить','Undo'),
            RusEng('Отменить последнюю операцию.','Undo last operation.'),
            ShortCut(Word('Z'),[ssCtrl]));
 UpdateMenu(MenuEditRedo,
            RusEng('Вернуть','Redo'),
            RusEng('Вернуть (повторить) последнюю операцию.','Redo last operation.'),
            ShortCut(Word('Y'),[ssCtrl]));
 UpdateMenu(MenuEditOblivion,
            RusEng('Сброс истории','Reset Undo'),
            RusEng('Сброс (очистка) истории изменений.','Reset (clear) operation history.'),
            0);
 UpdateMenu(MenuEditFind,
            RusEng('Найти …','Find …'),
            RusEng('Поиск.','Find.'),
            ShortCut(Word('F'),[ssCtrl]));
 UpdateMenu(MenuEditReplace,
            RusEng('Заменить …','Replace …'),
            RusEng('Замена.','Replace.'),
            ShortCut(Word('R'),[ssCtrl]));
 UpdateMenu(MenuEditAgain,
            RusEng('Найти еще …','Find next …'),
            RusEng('Повторить поиск/замену.','Repeat last Find/Replace.'),
            ShortCut(VK_F3,[]));
 UpdateMenu(MenuEditSelectAll,
            RusEng('Выделить все','Select all'),
            RusEng('Выделить все.','Select all.'),
            ShortCut(Word('A'),[ssCtrl]));
 UpdateMenu(MenuEditSelectNone,
            RusEng('Снять выделение','Select none'),
            RusEng('Отменить выделение блока.','Cancel text selection.'),
            ShortCut(Word('0'),[ssCtrl]));
 UpdateMenu(MenuEditGoToStart,
            RusEng('Иди к началу','Go to start'),
            RusEng('Идти к началу текста.','Go to start of text.'),
            ShortCut(VK_PRIOR,[ssCtrl]));
 UpdateMenu(MenuEditGoToLineN,
            RusEng('Иди к строке №','Go to line №'),
            RusEng('Поиск строки № в тексте.','Find line of text by №.'),
            ShortCut(Word('G'),[ssCtrl]));
 UpdateMenu(MenuEditGoToEnd,
            RusEng('Иди к концу','Go to end'),
            RusEng('Идти к концу текста.','Go to end of text.'),
            ShortCut(VK_NEXT,[ssCtrl]));
 UpdateMenu(MenuEditWordWrap,
            RusEng('Перенос','Word wrap'),
            RusEng('Автоматический перенос слов.','Automatic word wrap.'),
            ShortCut(Word('W'),[ssCtrl]));
 UpdateMenu(MenuEditRefresh,
            RusEng('Обновить','Refresh'),
            RusEng('Обновление (перерисовка) содержимого окна.','Refresh (redraw) window contents.'),
            ShortCut(VK_F5,[]));
 UpdateMenu(MenuEditReopen,
            RusEng('ПереОткрыть файл','ReOpen file'),
            RusEng('ПереОткрыть (заново прочитать из файла) редактируемый текст.','ReOpen (reload from file) editing text.'),
            0);
 UpdateMenu(MenuEditFont,
            RusEng('Выбрать фонт …','Choose font …'),
            RusEng('Диалог выбора фонта.','Choose font dialog.'),
            0);
 UpdateMenu(MenuEditReadTable,
            RusEng('Чтение таблицы','Read table'),
            RusEng('Чтение таблицы кривых.','Read table of curves.'),
            ShortCut(Word('T'),[ssCtrl]));
 UpdateMenu(MenuEditInsertTemplate,
            RusEng('Вставить шаблон','Insert template'),
            RusEng('Вставить текст из библиотеки шаблонов.','Insert text from template library.'),
            ShortCut(Word('I'),[ssCtrl]));
 UpdateMenu(MenuEditShiftLeft,
            RusEng('Сдвиг <--','Shift <--'),
            RusEng('Сдвиг фрагмента текста влево <--.','Text selection shift left <--.'),
            ShortCut(VK_HOME,[ssCtrl]));
 UpdateMenu(MenuEditShiftRight,
            RusEng('Сдвиг -->','Shift -->'),
            RusEng('Сдвиг фрагмента текста вправо -->.','Text selection shift right -->.'),
            ShortCut(VK_END,[ssCtrl]));
 PopupMenuFile.Caption:=RusEng('Файл …','File …');
 PopupMenuEdit.Caption:=RusEng('Правка …','Edit …');
 PopupMenuFind.Caption:=RusEng('Поиск …','Find …');
 PopupMenuTool.Caption:=RusEng('Утилиты …','Utility …');
 PerformReadOnly:=PerformReadOnly; // To update colors
 ZeroLastSelection;
 myFileSize:=0;
 myFileDate:=0;
end;

procedure TFormTextEditor.FormDestroy(Sender: TObject);
begin
 Kill(myReplaceDialog);
 Kill(myFindDialog);
 Kill(myAnnals);
 myPathName:='';
 inherited;
end;

procedure TFormTextEditor.AfterConstruction;
begin
 inherited AfterConstruction;
 FullTextEditorList.Add(Self);
 AddonSdiFlags(sf_SdiTextEdit);
end;

procedure TFormTextEditor.BeforeDestruction;
begin
 FullTextEditorList.Remove(Self);
 inherited BeforeDestruction;
end;

procedure TFormTextEditor.FormActivate(Sender: TObject);
begin
 inherited;
 BackLastSelection;
end;

procedure TFormTextEditor.FormDeactivate(Sender: TObject);
begin
 inherited;
 SaveLastSelection;
end;

procedure TFormTextEditor.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
 if PerformModified and not PerformReadOnly then begin
  case YesNoCancel(RusEng('Сохранить изменения в ','Save changes to ')+PathName+'?') of
   mrYes    : FileSave;
   mrCancel : CanClose := False;
  end;
 end;
 if CanClose then begin
  Kill(myReplaceDialog);
  Kill(myFindDialog);
 end;
end;

procedure TFormTextEditor.UpdateCommands;
var Exposed,HasSelection,CanEdit,CanCut,CanUndo,CanRedo,CanPaste:Boolean;
var m,w:Boolean; p:TPoint; l,z,n:Integer; sr,sc,sl,sm,sw,sz:LongString;
begin
 try
  inherited UpdateCommands;
  Exposed:=FormIsExposed(Self);
  CanEdit:=Exposed and not PerformReadOnly;
  HasSelection:=Exposed and (PerformSelLength>0);
  CanCut:=Exposed and HasSelection and CanEdit;
  CanUndo:=Exposed and CanEdit and PerformCanUndo;
  CanRedo:=Exposed and CanEdit and PerformCanRedo;
  CanPaste:=Exposed and CanEdit and Clipboard.HasFormat(CF_TEXT);
  ActionFilePrint.Enabled:=Exposed;
  ActionEditFind.Enabled:=Exposed;
  ActionEditAgain.Enabled:=Exposed;
  ActionEditSelectAll.Enabled:=Exposed;
  ActionEditGoToStart.Enabled:=Exposed;
  ActionEditGoToEnd.Enabled:=Exposed;
  ActionEditRefresh.Enabled:=Exposed;
  ActionEditFont.Enabled:=Exposed;
  ActionEditWordWrap.Enabled:=Exposed;
  ActionFileSave.Enabled:=CanEdit;
  ActionFileSaveAs.Enabled:=CanEdit;
  ActionEditReplace.Enabled:=CanEdit;
  ActionEditCopy.Enabled:=HasSelection;
  ActionEditSelectNone.Enabled:=HasSelection;
  ActionEditReadTable.Enabled:=HasSelection;
  ActionEditDelete.Enabled:=CanCut;
  ActionEditCut.Enabled:=CanCut;
  ActionEditUndo.Enabled:=CanUndo;
  ActionEditRedo.Enabled:=CanRedo;
  ActionEditOblivion.Enabled:=HasAnnals and CanUndo or CanRedo;
  ActionEditPaste.Enabled:=CanPaste;
  ActionEditInsertTemplate.Enabled:=CanEdit;
  ActionEditShiftLeft.Enabled:=CanCut;
  ActionEditShiftRight.Enabled:=CanCut;
  ActionEditReopen.Enabled:=CanEdit;
  {}
  p:=PerformCaretPos;
  m:=PerformModified;
  w:=PerformWordWrap;
  l:=PerformSelLength;
  z:=Annals.Size;
  sr:=RusEng('Стр:','Row:')+IntToStr(p.y+1);
  sc:=RusEng('Кол:','Col:')+IntToStr(p.x+1);
  sl:=RusEng('Выб:','Sel:')+IntToStr(l);
  sz:=RusEng('Пам:','Mem:')+IntToStr(z);
  if CanEdit then sm:=RusEng('Правка','Edit ') else sm:=RusEng('Чтение','View ');
  if m then sm:=sm+RusEng(' Изменён',' Modified');
  if w then sw:=RusEng('Переносы','Wrap') else sw:='';
  if (StatusBar.Panels[0].Text<>sr)
  or (StatusBar.Panels[1].Text<>sc)
  or (StatusBar.Panels[2].Text<>sl)
  or (StatusBar.Panels[3].Text<>sm)
  or (StatusBar.Panels[4].Text<>sw)
  or (StatusBar.Panels[5].Text<>sz)
  then begin
   StatusBar.Panels[0].Text:=sr;
   StatusBar.Panels[1].Text:=sc;
   StatusBar.Panels[2].Text:=sl;
   StatusBar.Panels[3].Text:=sm;
   StatusBar.Panels[4].Text:=sw;
   StatusBar.Panels[5].Text:=sz;
   StatusBar.Update;
  end;
  if HasAnnals then begin
   n:=IfThen(Annals.Errors=0,306,307);
   if (ActionEditOblivion.ImageIndex<>n)
   then ActionEditOblivion.ImageIndex:=n;
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateCommands');
 end;
end;

procedure TFormTextEditor.PrintView;
var Bmp:TBitmap; Key:Integer; mm:TMainMenu; tb,sb:Boolean;
var Params:LongString;
begin
 Params:='@set Panel.Font   Name:PT_Mono\Size:12\Color:Navy\Style:[Bold]'+EOL
        +'@set ListBox.Font Name:PT_Mono\Size:12\Color:Black\Style:[Bold]'+EOL
        +EditorPosParams;
 Key:=ListBoxMenu(RusEng('Файл\Печать','File\Print'),
                  RusEng('Как печатать','How to print'),
                  RusEng('Скопировать изображение в Буфер Обмена (цвет.)'+EOL+
                         'Скопировать изображение в Буфер Обмена (серый)'+EOL+
                         'Напечатать изображение на Принтере '+Printer.PrinterName+EOL+
                         'Напечатать как текст на Принтере '+Printer.PrinterName,
                         'Copy bitmap to Clipboard (color)'+EOL+
                         'Copy bitmap to Clipboard (gray)'+EOL+
                         'Print picture on Printer '+Printer.PrinterName+EOL+
                         'Print as text on Printer '+Printer.PrinterName),
                         myPrintViewKey,Params);
 if (Key>=0) then
 try
  myPrintViewKey:=Key;
  tb:=ToolBar.Visible;
  sb:=StatusBar.Visible;
  mm:=Menu;
  try
   if HasFlags(HidesOnPrint,hop_ToolBar) then ToolBar.Visible:=False;
   if HasFlags(HidesOnPrint,hop_StatusBar) then StatusBar.Visible:=False;
   if HasFlags(HidesOnPrint,hop_Menu) then Menu:=nil;
   if HasFlags(HidesOnPrint,hop_DrawView) then DrawView;
   if HasFlags(HidesOnPrint,hop_ProcMess) then SafeApplicationProcessMessages;
   case Key of
    0,1: begin
          Bmp:=GetPrintableImage;
          if Assigned(Bmp) then
          try
           if (Key=1) then ConvertBmpToBlackAndWhite(Bmp,clWindow,clBtnFace,3);
           CopyFormBmpToClipboard(Bmp,true);
          finally
           Kill(Bmp);
          end;
         end;
    2:   if HasPrintersDialog then Print;
    3:   if HasPrintersDialog then Editor.Print(PathName);
   end;
  finally
   if HasFlags(HidesOnPrint,hop_Menu) then Menu:=mm;
   if HasFlags(HidesOnPrint,hop_ToolBar) then ToolBar.Visible:=tb;
   if HasFlags(HidesOnPrint,hop_StatusBar) then StatusBar.Visible:=sb;
  end;
  if HasFlags(HidesOnPrint,hop_DrawView) then DrawView;
  if HasFlags(HidesOnPrint,hop_ProcMess) then SafeApplicationProcessMessages;
 except
  on E:Exception do BugReport(E,Self,'PrintView');
 end;
end;

function TFormTextEditor.CheckFileChanged(Mode:Integer):Boolean;
var aFileSize,aFileDate:Int64;
begin
 Result:=false;
 if (Self=nil) then Exit;
 if IsNoNamePath(PathName) then Exit;
 if IsNonEmptyStr(PathName) and FileExists(PathName) then begin
  aFileSize:=GetFileSize64(PathName);
  aFileDate:=GetFileDate(PathName);
 end else begin
  aFileSize:=0;
  aFileDate:=0;
 end;
 Result:=(aFileSize<>myFileSize) or (aFileDate<>myFileDate);
 if HasFlags(Mode,1) then begin
  myFileSize:=aFileSize;
  myFileDate:=aFileDate;
 end;
end;

procedure TFormTextEditor.Open(const AFileName: LongString);
begin
 if Ok then
 try
  if IsEmptyStr(AFileName) then begin
   PathName:=NoNamePath;
   with Editor do begin
    Lines.Clear;
    SelStart:=0;
    SelLength:=0;
    Modified:=False;
    PerformOblivion;
   end;
   ActionEditRefreshExecute(Self);
  end else begin
   PathName:=UnifyFileAlias(AFileName);
   with Editor do begin
    Lines.LoadFromFile(PathName);
    SelStart:=0;
    SelLength:=0;
    Modified:=False;
    PerformOblivion;
   end;
   CheckFileChanged(1);
   ActionEditRefreshExecute(Self);
   SendToMainConsole('@silent @integrity load.txt '+PathName+EOL);
  end;
 except
  on E:Exception do BugReport(E,Self,'Open');
 end;
end;

procedure TFormTextEditor.FileSave;
begin
 if Ok then
 try
  if IsNoNamePath(PathName)
  then FileSaveAs
  else begin
   Editor.Lines.SaveToFile(PathName);
   PerformModified:=false; CheckFileChanged(1);
   SendToMainConsole('@silent @integrity save.txt '+PathName+EOL);
  end;
 except
  on E:Exception do BugReport(E,Self,'FileSave');
 end;
end;

procedure TFormTextEditor.FileSaveAs;
var Ext,FName,Params:LongString;
begin
 if Ok then
 try
  SaveDialog.Title:=RusEng('Файл\Сохранить','File\Save');
  SaveDialog.Filter:=RusEng({ 1} 'Текстовые файлы       (*.txt)|*.txt|'+
                            { 2} 'Файлы калибровок      (*.cal)|*.cal|'+
                            { 3} 'Файлы конфигурации    (*.cfg)|*.cfg|'+
                            { 4} 'Файлы мнемосхем       (*.crc)|*.crc|'+
                            { 5} 'Файлы инициализации   (*.ini)|*.ini|'+
                            { 6} 'Файлы Pascal кодов    (*.pas)|*.pas|'+
                            { 7} 'Файлы включений       (*.inc)|*.inc|'+
                            { 8} 'Файлы журналов        (*.log)|*.log|'+
                            { 9} 'Проекты Delphi        (*.dpr)|*.dpr|'+
                            {10} 'Проекты Lazarus       (*.lpr)|*.lpr|'+
                            {11} 'Файлы макросов        (*.mac)|*.mac|'+
                            {12} 'Файлы HTML            (*.htm)|*.htm|'+
                            {13} 'Файлы скриптов NSIS   (*.nsi)|*.nsi|'+
                            {14} 'Все остальные файлы     (*.*)|*.*|',
                            { 1} 'Text files            (*.txt)|*.txt|'+
                            { 2} 'Calibration files     (*.cal)|*.cal|'+
                            { 3} 'Configuration files   (*.cfg)|*.cfg|'+
                            { 4} 'Circuit files         (*.crc)|*.crc|'+
                            { 5} 'Initialization files  (*.ini)|*.ini|'+
                            { 6} 'Pascal source files   (*.pas)|*.pas|'+
                            { 7} 'Include files         (*.inc)|*.inc|'+
                            { 8} 'Log files             (*.log)|*.log|'+
                            { 9} 'Delphi project files  (*.dpr)|*.dpr|'+
                            {10} 'Lazarus project files (*.lpr)|*.lpr|'+
                            {11} 'Macro files           (*.mac)|*.mac|'+
                            {12} 'HTML files            (*.htm)|*.htm|'+
                            {13} 'NSIS scripting files  (*.nsi)|*.nsi|'+
                            {14} 'All other files         (*.*)|*.*|');
  SaveDialog.FileName:=PathName;
  if IsEmptyStr(SaveDialog.FileName) or IsNoNamePath(SaveDialog.FileName)
  then OpenDialogSelectType(SaveDialog,'*.txt')
  else OpenDialogSelectType(SaveDialog,SaveDialog.FileName);
  Params:=ControlPosParams(ToolBar,'LB');
  if ExecuteFileDialog(GuardOpenDialog(SaveDialog),Params) then begin
   FName:=UnifyFileAlias(SaveDialog.FileName);
   Ext:=UnifyAlias(SaveDialog.DefaultExt);
   if IsEmptyStr(Ext) or IsWildCard(Ext)
   then FName:=UnifyFileAlias(FName,ua_FileDefLow)
   else FName:=UnifyFileAlias(ForceExtension(FName,Ext),ua_FileDefLow);
   if FileExists(FName) and
      (YesNo(RusEng('Файл существует!'+EOL+'Хотите перезаписать файл?',
                    'File exists!'+EOL+'Do you want overwrite this file?')+EOL+FName)<>mrYes)
   then FName:='';
   if (FName<>'') and not IsNoNamePath(FName) then begin
    PathName:=FName;
    FileSave;
   end;
   if IsNoNamePath(FName) then begin
    Echo(RusEng('Не могу сохранить "безымянный" файл '+FName+EOL
               +'Пожалуйста, дайте файлу другое имя.',
                'Could not save "noname" file '+FName+EOL
               +'Please assign a name to file.'));
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'FileSaveAs');
 end;
end;

procedure TFormTextEditor.FindText(const aFindText:LongString; aOptions:Integer);
var Options:TSearchTypes; MatchPos,StartPos,TailSize:Integer;
begin
 if Ok then
 if (aFindText<>'') then
 try
  SetFocusToEditor;
  myLastFind:=lfFind; Options:=[];
  if HasFlags(aOptions,fr_WholeWord) then include(Options,stWholeWord);
  if HasFlags(aOptions,fr_MatchCase) then include(Options,stMatchCase);
  StartPos:=PerformSelStart+PerformSelLength;
  TailSize:=Length(StrToWide(PerformText))-StartPos;
  MatchPos:=Editor.FindText(aFindText,StartPos,TailSize,Options);
  if (MatchPos>=0) then begin
   SmartFocus(Editor);
   PerformSelStart:=MatchPos;
   PerformSelLength:=Length(StrToWide(aFindText));
  end else begin
   Information(Format(RusEng('Не могу найти строку "%s".','Could not find text "%s".'),[aFindText]));
  end;
 except
  on E:Exception do BugReport(E,Self,'FindText');
 end;
end;

procedure TFormTextEditor.ReplaceText(const aFindText,aReplaceText:LongString; aOptions:Integer);
var Options:TSearchTypes; MatchPos,StartPos,TailSize:Integer;
 function Prompt:Boolean;
 var S,L:Integer;
 begin
  if HasFlags(aOptions,fr_PromptOnReplace) then begin
   S:=PerformSelStart; L:=PerformSelLength;
   Result:=(YesNo(Format(RusEng('Заменить строку%s"%s"на строку%s"%s"?',
                                'Replace string%s"%s"to string%s"%s"?'),
                                [EOL,aFindText,EOL,aReplaceText]))=mrYes);
   PerformSelStart:=S; PerformSelLength:=L;
  end else Result:=true;
 end;
begin
 if Ok then
 if (aFindText<>'') then
 if not PerformReadOnly then
 try
  SetFocusToEditor;
  myLastFind:=lfReplace; Options:=[];
  if HasFlags(aOptions,fr_WholeWord) then include(Options,stWholeWord);
  if HasFlags(aOptions,fr_MatchCase) then include(Options,stMatchCase);
  if HasFlags(aOptions,fr_ReplaceAll) then begin
   while true do begin
    StartPos:=PerformSelStart+PerformSelLength;
    TailSize:=Length(StrToWide(PerformText))-StartPos;
    MatchPos:=Editor.FindText(aFindText,StartPos,TailSize,Options);
    if (MatchPos>=0) then begin
     SmartFocus(Editor);
     PerformSelStart:=MatchPos;
     PerformSelLength:=Length(StrToWide(aFindText));
     if Prompt then begin
      PerformSelText:=aReplaceText;
      PerformSelStart:=MatchPos+Length(StrToWide(aReplaceText));
      PerformSelLength:=0;
     end else Break;
    end else Break;
   end;
  end else
  if HasFlags(aOptions,fr_Replace) then begin
   if (PerformSelLength>0) then begin
    StartPos:=PerformSelStart;
    TailSize:=PerformSelLength;
    MatchPos:=Editor.FindText(aFindText,StartPos,TailSize,Options);
    if (MatchPos=StartPos) then begin
     if Prompt then begin
      PerformSelText:=aReplaceText;
      PerformSelStart:=MatchPos+Length(StrToWide(aReplaceText));
      PerformSelLength:=0;
     end;
    end;
   end;
   StartPos:=PerformSelStart+PerformSelLength;
   TailSize:=Length(StrToWide(PerformText))-StartPos;
   MatchPos:=Editor.FindText(aFindText,StartPos,TailSize,Options);
   if (MatchPos>=0) then begin
    SmartFocus(Editor);
    PerformSelStart:=MatchPos;
    PerformSelLength:=Length(StrToWide(aFindText));
   end else begin
    Information(Format(RusEng('Не могу найти строку "%s".','Could not find text "%s".'),[aFindText]));
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ReplaceText');
 end;
end;

function TFormTextEditor.IsFileProtected(Mode:Integer):Boolean;
var msg:LongString;
begin
 Result:=false;
 if Assigned(Self) then begin
  if IsNoNamePath(PathName) then Exit(false);
  Result:=FileGuard.ProtectedReadOnly(PathName);
  if Result and HasFlags(Mode,1+2) then begin
   msg:=ExtractFileNameExt(PathName);
   msg:=RusEng('Файл защищен: ','File protected: ')+msg;
   if HasFlags(Mode,1) then Echo(StdDateTimePrompt(msecnow)+msg);
   msg:=SessionManager.TitlePidAtHost+': '+msg;
   msg:='@silent @tooltip text '+QArg(msg);
   msg:=msg+' preset stdWarning delay 15000';
   if HasFlags(Mode,2) then SendToMainConsole(msg+EOL);
  end;
 end;
end;

procedure TFormTextEditor.ActionEditReadOnlyExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditReadOnly)<0 then Exit;
 inherited;
 if Ok then
 try
  PerformReadOnly:=IsFileProtected(1+2) or not PerformReadOnly;
  if not PerformReadOnly then ActionEditCheckFileChanged.Execute;
 except
  on E:Exception do BugReport(E,Self,'ActionEditReadOnlyExecute');
 end;
end;

procedure TFormTextEditor.ActionEditDeleteExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditDelete)<0 then Exit;
 if Ok then PerformClearSelection;
end;

procedure TFormTextEditor.ActionEditCutExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditCut)<0 then Exit;
 if Ok then PerformCutToClipboard;
end;

procedure TFormTextEditor.ActionEditCopyExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditCopy)<0 then Exit;
 if Ok then PerformCopyToClipboard;
end;

procedure TFormTextEditor.ActionEditPasteExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditPaste)<0 then Exit;
 if Ok then PerformPasteFromClipboard;
end;

procedure TFormTextEditor.ActionEditUndoExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditUndo)<0 then Exit;
 if Ok and PerformCanUndo then PerformUndo;
end;

procedure TFormTextEditor.ActionEditRedoExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditRedo)<0 then Exit;
 if Ok and PerformCanRedo then PerformRedo;
end;

function ConfirmOblivion(const par:LongString):Boolean;
var ask:LongString;
begin
 ask:=RusEng('Вы хотите очистить историю изменений редактора?',
             'Do you want to clear editor UnDo/ReDo history?');
 Result:=(YesNo(ask,par)=mrYes);
end;

procedure TFormTextEditor.ActionEditOblivionExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditOblivion)<0 then Exit;
 if Ok and ConfirmOblivion('') then PerformOblivion;
end;

procedure TFormTextEditor.ActionEditFindExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditFind)<0 then Exit;
 if Ok then
 try
  if myReplaceDialog.Ok then myReplaceDialog.Hide;
  if not myFindDialog.Ok then begin
   myFindDialog:=NewFormTextFindDialog(Self,EditorPosParams);
   myFindDialog.Master:=@myFindDialog;
  end;
  if myFindDialog.Ok then begin
   myFindDialog.Show;
   myFindDialog.WindowState:=wsNormal;
   myFindDialog.BringToFront;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditFindExecute');
 end;
end;

procedure TFormTextEditor.ActionEditReplaceExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditReplace)<0 then Exit;
 if Ok then
 try
  if myFindDialog.Ok then myFindDialog.Hide;
  if not myReplaceDialog.Ok then begin
   myReplaceDialog:=NewFormTextReplaceDialog(Self,EditorPosParams);
   myReplaceDialog.Master:=@myReplaceDialog;
  end;
  if myReplaceDialog.Ok then begin
   myReplaceDialog.Show;
   myReplaceDialog.WindowState:=wsNormal;
   myReplaceDialog.BringToFront;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditReplaceExecute');
 end;
end;

procedure TFormTextEditor.ActionEditAgainExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditAgain)<0 then Exit;
 if Ok then
 case myLastFind of
  lfFind    : ActionEditFindExecute(Sender);
  lfReplace : ActionEditReplaceExecute(Sender);
 end;
end;

procedure TFormTextEditor.ActionEditSelectAllExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditSelectAll)<0 then Exit;
 if Ok then PerformSelectAll;
end;

procedure TFormTextEditor.ActionEditSelectNoneExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditSelectNone)<0 then Exit;
 if Ok then PerformSelectNone;
end;

procedure TFormTextEditor.ActionEditGoToStartExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditGoToStart)<0 then Exit;
 if Ok then PerformGoToStart;
end;

procedure TFormTextEditor.ActionEditGoToEndExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditGoToEnd)<0 then Exit;
 if Ok then PerformGoToEnd;
end;

procedure TFormTextEditor.ActionEditRefreshExecute(Sender: TObject);
var aModified : Boolean;
begin
 if Ok then
 try
  inherited;
  aModified:=PerformModified;
  RefreshWindow;
  PerformModified:=aModified;
 except
  on E:Exception do BugReport(E,Self,'ActionEditRefreshExecute');
 end;
end;

procedure TFormTextEditor.Reopen;
begin
 try
  if (PathName<>'') and FileExists(PathName) then Open(PathName);
 except
  on E:Exception do BugReport(E,Self,'Reopen');
 end;
end;

procedure TFormTextEditor.ActionEditReopenExecute(Sender: TObject);
var Msg:LongString;
begin
 if Guard.CheckAction(ga_Guest,ActionEditReopen)<0 then Exit;
 if PerformModified then begin
  Msg:=RusEng('Редактируемый файл:'+EOL+PathName+EOL
             +'был изменен в редакторе. Эти изменения пропадут при чтении файла.'+EOL
             +'Вы хотите загрузить текст из файла?',
              'Editing file:'+EOL+PathName+EOL
             +'was changed in editor. This changes will be lost when read file.'+EOL
             +'Do you want to load text from file?');
  if not CanShowModal then Exit;
  if (YesNo(Msg,EditorPosParams)<>mrYes) then Exit;
 end;
 inherited;
 Reopen;
end;

procedure TFormTextEditor.ActionEditFontExecute(Sender: TObject);
var aModified:Boolean;
begin
 if Guard.CheckAction(ga_Guest,ActionEditFont)<0 then Exit;
 if Ok then
 try
  ScreenFontsUpdate;
  FontDialog.Font.Assign(Editor.Font);
  if FontDialog.Execute then begin
   aModified:=PerformModified;
   Editor.Font.Assign(FontDialog.Font);
   ActionEditRefreshExecute(Sender);
   PerformModified:=aModified;
   Echo(StdDateTimePrompt+'Font selected:'+EOL
       +Trim(GetFontAsText(Editor.Font)));
   ClipBoard.AsText:=GetFontAsText(Editor.Font);
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditFontExecute');
 end;
end;

procedure TFormTextEditor.ActionEditWordWrapExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditWordWrap)<0 then Exit;
 if Ok then
 try
  PerformWordWrap:=not PerformWordWrap;
  if (Editor.ScrollBars<>ssAutoBoth) then begin
   if PerformWordWrap
   then Editor.ScrollBars:=ssVertical
   else Editor.ScrollBars:=ssBoth;
  end;
  MenuEditWordWrap.Checked:=PerformWordWrap;
 except
  on E:Exception do BugReport(E,Self,'ActionEditWordWrapExecute');
 end;
end;

procedure TFormTextEditor.ActionEditReadTableExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditReadTable)<0 then Exit;
 if Ok then
 try
  NewCurveReadTextTable(Self);
 except
  on E:Exception do BugReport(E,Self,'ActionEditReadTableExecute');
 end;
end;

procedure TFormTextEditor.ActionEditInsertTemplateExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditInsertTemplate)<0 then Exit;
 if Ok then
 try
  PerformSelLength:=0;
  Clipboard.AsText:=FormTextEditorTemplatesExecute(RusEng('Вставить из шаблона …','Insert template …'), PerformCaretPos.X, true,EditorPosParams);
  if Clipboard.HasFormat(CF_TEXT) then begin
   PerformPasteFromClipboard;
   ActionEditRefresh.Execute;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditInsertTemplateExecute');
 end;
end;

procedure TFormTextEditor.GoToRowCol(aRow,aCol:Integer);
begin
 if Assigned(Self) then PerformGoToRowCol(aRow,aCol);
end;

procedure TFormTextEditor.DefaultReadOnly;
var b:Boolean;
begin
 if Ok then
 try
  b:=false;
  if ReadIniFileBoolean(SysIniFile,SectSystem,'OpenTextAsReadOnly%b',b) then
  if b then PerformReadOnly:=true;
  if IsFileProtected then PerformReadOnly:=true;
 except
  on E:Exception do BugReport(E,Self,'DefaultReadOnly');
 end;
end;

procedure TFormTextEditor.PerformUndo;
begin
 if Assigned(Self) then
 try
  if HasAnnals
  then Annals.Undo
  else Editor.Undo;
 except
  on E:Exception do BugReport(E,Self,'PerformUndo');
 end;
end;

procedure TFormTextEditor.PerformRedo;
begin
 if Assigned(Self) then
 try
  if HasAnnals
  then Annals.Redo;
 except
  on E:Exception do BugReport(E,Self,'PerformRedo');
 end;
end;

function TFormTextEditor.PerformCanUndo:Boolean;
begin
 Result:=false;
 if Assigned(Self) then
 try
  if HasAnnals
  then Result:=Annals.CanUndo
  else Result:=Editor.CanUndo;
 except
  on E:Exception do BugReport(E,Self,'PerformCanUndo');
 end;
end;

function TFormTextEditor.PerformCanRedo:Boolean;
begin
 Result:=false;
 if Assigned(Self) then
 try
  if HasAnnals
  then Result:=Annals.CanRedo;
 except
  on E:Exception do BugReport(E,Self,'PerformCanRedo');
 end;
end;

procedure TFormTextEditor.PerformClearSelection;
begin
 if Assigned(Self) then
 try
  if HasAnnals
  then Annals.ClearSelection
  else Editor.ClearSelection;
 except
  on E:Exception do BugReport(E,Self,'PerformClearSelection');
 end;
end;

procedure TFormTextEditor.PerformCutToClipboard;
begin
 if Assigned(Self) then
 try
  if HasAnnals
  then Annals.CutToClipboard
  else Editor.CutToClipboard;
 except
  on E:Exception do BugReport(E,Self,'PerformCutToClipboard');
 end;
end;

procedure TFormTextEditor.PerformCopyToClipboard;
begin
 if Assigned(Self) then
 try
  Editor.CopyToClipboard;
 except
  on E:Exception do BugReport(E,Self,'PerformCopyToClipboard');
 end;
end;

procedure TFormTextEditor.PerformPasteFromClipboard;
begin
 if Assigned(Self) then
 try
  if HasAnnals
  then Annals.PasteFromClipboard
  else Editor.PasteFromClipboard;
 except
  on E:Exception do BugReport(E,Self,'PerformPasteFromClipboard');
 end;
end;

procedure TFormTextEditor.PerformPasteText(const aText:LongString);
begin
 if Assigned(Self) then
 try
  if HasAnnals
  then Annals.PasteText(aText)
  else Editor.SelText:=aText;
 except
  on E:Exception do BugReport(E,Self,'PerformPasteText');
 end;
end;

procedure TFormTextEditor.PerformOblivion(Mode:Integer);
begin
 if Assigned(Self) then
 try
  if HasFlags(Mode,1) then Editor.Modified:=false;
  if HasFlags(Mode,2) then if HasAnnals then Annals.Reset;
 except
  on E:Exception do BugReport(E,Self,'PerformOblivion');
 end;
end;

procedure TFormTextEditor.PerformSelectAll;
begin
 if Assigned(Self) then
 try
  Editor.SelectAll;
 except
  on E:Exception do BugReport(E,Self,'PerformSelectAll');
 end;
end;

procedure TFormTextEditor.PerformSelectNone;
begin
 if Assigned(Self) then
 try
  Editor.SelLength:=0;
 except
  on E:Exception do BugReport(E,Self,'PerformSelectNone');
 end;
end;

procedure TFormTextEditor.PerformGoToStart;
begin
 if Assigned(Self) then
 try
  Editor.SelLength:=0;
  Editor.SelStart:=0;
  PerformGoToRowCol(1,1);
 except
  on E:Exception do BugReport(E,Self,'PerformGoToStart');
 end;
end;

procedure TFormTextEditor.PerformGoToEnd;
begin
 if Assigned(Self) then
 try
  Editor.SelLength:=0;
  Editor.SelStart:=MaxInt;
  PerformGoToRowCol(Editor.Lines.Count,1);
 except
  on E:Exception do BugReport(E,Self,'PerformGoToEnd');
 end;
end;

procedure TFormTextEditor.PerformGoToRowCol(aRow,aCol:Integer);
begin
 if Assigned(Self) then
 try
  Editor.ApplyGoToRowCol(aRow,aCol);
 except
  on E:Exception do BugReport(E,Self,'PerformGoToRowCol');
 end;
end;

function TFormTextEditor.GetPerformReadOnly:Boolean;
begin
 if Assigned(Self)
 then Result:=Editor.ReadOnly
 else Result:=false;
end;

procedure TFormTextEditor.SetPerformReadOnly(aReadOnly:Boolean);
begin
 if Assigned(Self) then
 try
  Editor.ReadOnly:=aReadOnly;
  if Editor.ReadOnly
  then Editor.Color:=TextEditorsPalette.ReadOnlyColor
  else Editor.Color:=TextEditorsPalette.MainBackColor;
  Editor.Font.Color:=TextEditorsPalette.MainTextColor;
 except
  on E:Exception do BugReport(E,Self,'SetPerformReadOnly');
 end;
end;

function TFormTextEditor.GetPerformModified:Boolean;
begin
 if Assigned(Self)
 then Result:=Editor.Modified
 else Result:=false;
end;

procedure TFormTextEditor.SetPerformModified(aModified:Boolean);
begin
 if Assigned(Self) then
 try
  Editor.Modified:=aModified;
  if not aModified then PerformOblivion;
 except
  on E:Exception do BugReport(E,Self,'SetPerformModified');
 end;
end;

function TFormTextEditor.GetPerformSelStart:Integer;
begin
 if Assigned(Self)
 then Result:=Editor.SelStart
 else Result:=0;
end;

procedure TFormTextEditor.SetPerformSelStart(aSelStart:Integer);
begin
 if Assigned(Self) then Editor.SelStart:=aSelStart;
end;

function TFormTextEditor.GetPerformSelLength:Integer;
begin
 if Assigned(Self)
 then Result:=Editor.SelLength
 else Result:=0;
end;

procedure TFormTextEditor.SetPerformSelLength(aLength:Integer);
begin
 if Assigned(Self) then Editor.SelLength:=aLength;
end;

function TFormTextEditor.GetPerformSelText:LongString;
begin
 if Assigned(Self)
 then Result:=Editor.SelText
 else Result:='';
end;

procedure TFormTextEditor.SetPerformSelText(const aText:LongString);
begin
 if Assigned(Self) then PerformPasteText(aText);
end;

function TFormTextEditor.GetPerformText:LongString;
begin
 if Assigned(Self)
 then Result:=Editor.Text
 else Result:='';
end;

procedure TFormTextEditor.SetPerformText(const aText:LongString);
begin
 if Assigned(Self) then begin
  Editor.Text:=aText;
  PerformOblivion;
 end;
end;

function TFormTextEditor.GetPerformWideText:WideString;
begin
 if Assigned(Self)
 then Result:=StrToWide(Editor.Text)
 else Result:='';
end;

procedure TFormTextEditor.SetPerformWideText(const aText:WideString);
begin
 if Assigned(Self) then begin
  Editor.Text:=WideToStr(aText);
  PerformOblivion;
 end;
end;

function TFormTextEditor.GetPerformLinesText:LongString;
begin
 if Assigned(Self)
 then Result:=Editor.Lines.Text
 else Result:='';
end;

procedure TFormTextEditor.SetPerformLinesText(const aText:LongString);
begin
 if Assigned(Self) then begin
  Editor.Lines.Text:=aText;
  PerformOblivion;
 end;
end;

procedure TFormTextEditor.PerformSetFocus(PM:Boolean=true);
begin
 if Assigned(Self) then begin
  if Editor.CanSetFocus and not Editor.Focused then begin
   Editor.SetFocus; if PM then SafeApplicationProcessMessages;
  end;
 end;
end;

function  TFormTextEditor.GetPerformCaretPos:TPoint;
begin
 if Assigned(Self)
 then Result:=Editor.CaretPos
 else Result:=Point(0,0);
end;

procedure TFormTextEditor.SetPerformCaretPos(const aCaretPos:TPoint);
begin
 if Assigned(Self) then begin
  if MemoCaretPosNeedBugFix
  then PerformGoToRowCol(aCaretPos.Y+1,aCaretPos.X+1)
  else Editor.CaretPos:=aCaretPos;
 end;
end;

function  TFormTextEditor.GetPerformWordWrap:Boolean;
begin
 if Assigned(Self)
 then Result:=Editor.WordWrap
 else Result:=false;
end;

procedure TFormTextEditor.SetPerformWordWrap(aWordWrap:Boolean);
begin
 if Assigned(Self) then Editor.WordWrap:=aWordWrap;
end;

function LineShift(const S:LongString; Shift:Integer):LongString;
begin
 Result:=S;
 if (Shift>0) then Result:=StringOfChar(' ',Shift)+Result;
 if (Shift<0) then Result:=WideToStr(Copy(StrToWide(Result),1-Shift,MaxInt));
end;

function TextShift(const S:LongString; Shift:Integer; TrimEOL:Boolean):LongString;
var i:Integer; List:TStringList;
begin
 Result:=S;
 if (Shift<>0) then
 try
  List:=TStringList.Create;
  try
   List.Text:=S;
   for i:=0 to List.Count-1 do List[i]:=LineShift(List[i],Shift);
   Result:=List.Text;
   if TrimEOL then begin
    if EndsStr(EOL,Result)
    then Result:=Copy(Result,1,Length(Result)-Length(EOL));
   end;
  finally
   Kill(List);
  end;
 except
  on E:Exception do BugReport(E,nil,'TextShift');
 end;
end;

procedure TFormTextEditor.TextShiftExecute(Shift:Integer);
var TheStart:Integer; TheBlock:LongString; NeedTrim:Boolean;
begin
 if Assigned(Self) then
 try
  if (PerformSelLength>0) then begin
   TheStart:=PerformSelStart;
   TheBlock:=PerformSelText;
   NeedTrim:=not EndsStr(EOL,TheBlock);
   TheBlock:=TextShift(TheBlock,Shift,NeedTrim);
   PerformSelText:=TheBlock;
   PerformSelStart:=TheStart;
   PerformSelLength:=Length(StrToWide(TheBlock));
  end;
 except
  on E:Exception do BugReport(E,nil,'TextShiftExecute');
 end;
end;

procedure TFormTextEditor.ActionEditShiftLeftExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditShiftLeft)<0 then Exit;
 if Ok then TextShiftExecute(-1);
end;

procedure TFormTextEditor.ActionEditShiftRightExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditShiftRight)<0 then Exit;
 if Ok then TextShiftExecute(+1);
end;

procedure TFormTextEditor.ActionEditGoToLineNExecute(Sender: TObject);
var s:LongString; const Delims=JustSpaces+[',',';','=',':','\','/'];
begin
 if Guard.CheckAction(ga_Guest,ActionEditGoToLineN)<0 then Exit;
 inherited;
 if Ok then
 try
  s:=Format('%d : %d',[myLinePos.Y,myLinePos.X]);
  try
   if InputQuery(RusEng('Иди на строку № …','Go to line № …'),
                 RusEng('Ввести номер строки:колонки','Enter row:column number'),s)
   then begin
    myLinePos.Y:=StrToIntDef(ExtractWord(1,s,Delims),0);
    myLinePos.X:=StrToIntDef(ExtractWord(2,s,Delims),1);
    if (myLinePos.Y>0) and (myLinePos.X>0) then begin
     GoToRowCol(myLinePos.y,myLinePos.x);
     if (PerformCaretPos.Y+1<>myLinePos.Y)
     then Warning(Format(RusEng('Не могу найти строку "%s".',
                                'Could not find line "%s".'),[s]));
    end else Warning(Format(RusEng('Неверный номер строки "%s".',
                                   'Invalid line number "%s".'),[s]));
   end;
  finally
   s:='';
  end;  
 except
  on E:Exception do BugReport(E,Self,'ActionEditGoToLineNExecute');
 end;
end;

procedure TFormTextEditor.ActionEditCheckFileChangedExecute(Sender: TObject);
var Msg:LongString;
begin
 inherited;
 try
  if not PerformReadOnly then
  if CheckFileChanged(0) then begin
   Msg:=RusEng('Редактируемый файл:'+EOL+PathName+EOL
              +'был изменен другим (внешним) редактором.'+EOL
              +'Вы хотите загрузить текст из файла?',
               'Editing file:'+EOL+PathName+EOL
              +'was changed by another (external) editor.'+EOL
              +'Do you want to load text from file?');
   if CanShowModal then if (YesNo(Msg,EditorPosParams)=mrYes) then ActionEditReopen.Execute;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditCheckFileChangedExecute');
 end;
end;

procedure Init_TextEditorsPalette;
var s:LongString;
begin
 try
  s:='';
  if SysGlossary.ReadIniAlpha(SysIniFile,SectTextEditorsPalette,'MainBackColor',s)
  then TextEditorsPalette.MainBackColor:=StringToColor(s,clWhite);
  if SysGlossary.ReadIniAlpha(SysIniFile,SectTextEditorsPalette,'MainTextColor',s)
  then TextEditorsPalette.MainTextColor:=StringToColor(s,clBlack);
  if SysGlossary.ReadIniAlpha(SysIniFile,SectTextEditorsPalette,'ReadOnlyColor',s)
  then TextEditorsPalette.ReadOnlyColor:=StringToColor(s,clInfoBk);
  DefaultEditorFont.Color:=TextEditorsPalette.MainTextColor;
 except
  on E:Exception do BugReport(E,nil,'Init_TextEditorsPalette');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_texteditor;
begin
 FullTextEditorList:=NewTextEditorList(false);
 FullTextEditorList.Master:=@FullTextEditorList;
end;

procedure Free_form_texteditor;
begin
 ResourceLeakageLog(Format('%-60s = %d',['FullTextEditorList.Count', FullTextEditorList.Count]));
 Kill(FullTextEditorList);
end;

initialization

 Init_form_texteditor;

finalization

 Free_form_texteditor;

end.

//////////////
// END OF FILE
//////////////

