////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Spectr DAQ Window.                                                    //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231203 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_spectrdaqwindow; // Form Spectr DAQ Window

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild, Form_SpectrWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_ef,
 _crw_ee, _crw_zm, _crw_curves, _crw_riff,
 _crw_sort, _crw_calib, _crw_daqsys, _crw_daqdev,
 _crw_appforms, _crw_apptools, _crw_apputils;

 {
 Модуль реализует спектрометрическое окно для системы CRW-DAQ с калибровками
 }
type

  { TFormSpectrDaqWindow }

  TFormSpectrDaqWindow = class(TFormSpectrWindow)
    ActionSpectrEditCalibrationEn: TAction;
    ActionSpectrEditCalibrationHw: TAction;
    ToolButtonSpectrEditEnCalSeparator: TToolButton;
    ToolButtonSpectrEditCalibrationEn: TToolButton;
    ToolButtonSpectrEditCalibrationHw: TToolButton;
    MenuSpectrEditCalibration: TMenuItem;
    MenuSpectrEditCalibrationEn: TMenuItem;
    MenuSpectrEditCalibrationHw: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure ActionSpectrEditCalibrationEnExecute(Sender: TObject);
    procedure ActionSpectrEditCalibrationHwExecute(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure StatusBarDblClick(Sender: TObject);
  private
    { Private declarations }
    myDeviceName  : LongString;    { имя устройства - спектрометра     }
    mySpectrometr : TDaqDevice;    { спектрометр где брать калибровки  }
    myEnCalNum    : Integer;       { номер калибровки энергии          }
    myHwCalNum    : Integer;       { номер калибровки полуширины       }
    myMarkerLabel : packed record  { наименования меток для MarkerView }
     Ch           : PureString;    { канал                             }
     Cnt          : PureString;    { счет                              }
     En           : PureString;    { энергия                           }
     Hw           : PureString;    { полуширина                        }
     RoiC         : PureString;    { область интереса в каналах        }
     RoiE         : PureString;    { область интереса в keV            }
     Int0         : PureString;    { интеграл 0                        }
     Int1         : PureString;    { интеграл 1                        }
     Int2         : PureString;    { интеграл 2                        }
    end;
    function  GetSpectrometr:TDaqDevice;
    function  GetEnCalNum:Integer;
    function  GetHwCalNum:Integer;
  public
    { Public declarations }
    property  Spectrometr : TDaqDevice read GetSpectrometr;
    property  EnCalNum    : Integer    read GetEnCalNum;
    property  HwCalNum    : Integer    read GetHwCalNum;
    procedure Config(CfgFile,Section:LongString); override;
    procedure Animate; override;
    function  FormatMarker(aMarker:Integer):LongString; override;
    function  EnCalibrExist:Boolean; override;
    function  EnCalibr(Chan:Double):Double; override;
    function  HwCalibrExist:Boolean; override;
    function  HwCalibr(Chan:Double):Double; override;
    function  SaveCalibrations(FileName : LongString):Boolean; override;
    function  LoadCalibrations(FileName : LongString):Boolean; override;
  end;

const
 regSpecDaqWin = 'SpecDaqWin';
 fmtMarkerLabela = '%a;%a;%a;%a;%a;%a;%a;%a;%a';
 fmtMarkerLabels = '%s,%s,%s,%s,%s,%s,%s,%s,%s';

function  NewSpecDaqWin(const aCaption    : LongString;
                              aSpectr     : TCurve;
                              aOwnsSpectr : Boolean):TFormSpectrWindow;
procedure Kill(var TheObject:TFormSpectrDaqWindow); overload;

implementation

{$R *.lfm}

 {
 *******************************************************************************
 General purpose utilites
 *******************************************************************************
 }
function  NewSpecDaqWin(const aCaption    : LongString;
                              aSpectr     : TCurve;
                              aOwnsSpectr : Boolean ) : TFormSpectrWindow;
begin
 Application.CreateForm(TFormSpectrDaqWindow, Result);
 if Result.Ok then
 with Result do
 try
  LockDraw;
  if aCaption=''
  then Caption:=RusEng('Спектрометрическое окно','Spectrometry window')
  else Caption:=aCaption;
  AssignSpectr(aSpectr,aOwnsSpectr);
 finally
  UnlockDraw;
 end;
end;

procedure Kill(var TheObject:TFormSpectrDaqWindow); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

 {
 *******************************************************************************
 TFormSpectrDaqWindow implementation
 *******************************************************************************
 }
function  TFormSpectrDaqWindow.GetSpectrometr:TDaqDevice;
begin
 if Assigned(Self) then Result:=mySpectrometr else Result:=nil;
end;

function  TFormSpectrDaqWindow.GetEnCalNum:Integer;
begin
 if Assigned(Self) then Result:=myEnCalNum else Result:=-1;
end;

function  TFormSpectrDaqWindow.GetHwCalNum:Integer;
begin
 if Assigned(Self) then Result:=myHwCalNum else Result:=-1;
end;

procedure TFormSpectrDaqWindow.FormCreate(Sender: TObject);
begin
 inherited;
 myDeviceName:='';
 mySpectrometr:=nil;
 myEnCalNum:=-1;
 myHwCalNum:=-1;
 myMarkerLabel.Ch:=RusEng('КАНАЛ','CHANNEL');
 myMarkerLabel.Cnt:=RusEng('СЧЕТ','COUNT');
 myMarkerLabel.En:='';
 myMarkerLabel.Hw:='';
 myMarkerLabel.RoiC:='';
 myMarkerLabel.RoiE:='';
 myMarkerLabel.Int0:='';
 myMarkerLabel.Int1:='';
 myMarkerLabel.Int2:='';
 UpdateMenu(MenuSpectrEditCalibration,
            RusEng('Калибровка','Calibration'),
            RusEng('Калибровки энергии и полуширины.','Energy and FWHM calibrations.'),
            0);
 UpdateMenu(MenuSpectrEditCalibrationEn,
            RusEng('Канал-Энергия','Chan-Energy'),
            RusEng('Калибровка канал - энергия.','Energy vs channel calibration.'),
            0);
 UpdateMenu(MenuSpectrEditCalibrationHw,
            RusEng('Канал-Ширина','Chan-FWHM'),
            RusEng('Калибровка канал - полуширина на полувысоте.','FWHM vs channel calibration.'),
            0);
end;

procedure TFormSpectrDaqWindow.FormDestroy(Sender: TObject);
begin
 myDeviceName:='';
 myMarkerLabel.Ch:='';
 myMarkerLabel.Cnt:='';
 myMarkerLabel.En:='';
 myMarkerLabel.Hw:='';
 myMarkerLabel.RoiC:='';
 myMarkerLabel.RoiE:='';
 myMarkerLabel.Int0:='';
 myMarkerLabel.Int1:='';
 myMarkerLabel.Int2:='';
 inherited;
end;

procedure TFormSpectrDaqWindow.Config(CfgFile,Section:LongString);
begin
 CfgFile:=UnifyFileAlias(CfgFile);
 inherited Config(CfgFile,Section);
 ReadIniFileRecord(CfgFile,Section,'MarkerLabel'+fmtMarkerLabela,myMarkerLabel);
 ReadIniFileAlpha(CfgFile,Section,'Spectrometr%a',myDeviceName);
 ReadIniFileInteger(CfgFile,Section,'EnCalibration%i',myEnCalNum);
 ReadIniFileInteger(CfgFile,Section,'HwCalibration%i',myHwCalNum);
end;

procedure TFormSpectrDaqWindow.Animate;
begin
 inherited Animate;
 mySpectrometr:=FullDaqDeviceList.Find(myDeviceName);
 if IsNonEmptyStr(myDeviceName) then
 if not Spectrometr.Ok then
 Daq.AddWarning('['+Caption+'] -> invalid device '+myDeviceName);
 EnCalibrChanged;
 HwCalibrChanged;
end;

function TFormSpectrDaqWindow.FormatMarker(aMarker:Integer):LongString;
 function Present(const What:LongString):Boolean;
 begin
  Result:=(Length(What)>0) and not IsSameText(What,'*');
 end;
begin
 Result:='';
 if (SpectrSize>0) then begin
  Result:=Format('  %s:',[SpectrName]);
  if Present(myMarkerLabel.Ch)
  then Result:=Result+Format('  %s=%d',[myMarkerLabel.Ch, aMarker]);
  if Present(myMarkerLabel.Cnt)
  then Result:=Result+Format('  %s=%.0f',[myMarkerLabel.Cnt, SpectrValue[aMarker]]);
  if Present(myMarkerLabel.En) and EnCalibrExist
  then Result:=Result+Format('  %s=%.7g',[myMarkerLabel.En, EnCalibr(aMarker)]);
  if Present(myMarkerLabel.Hw) and HwCalibrExist
  then Result:=Result+Format('  %s=%.7g',[myMarkerLabel.Hw, HwCalibr(aMarker)]);
  if Present(myMarkerLabel.RoiC)
  then Result:=Result+Format('  %s=[%d,%d]',[myMarkerLabel.RoiC,ROIL,ROIR]);
  if Present(myMarkerLabel.RoiE)
  then Result:=Result+Format('  %s=[%.7g,%.7g]',[myMarkerLabel.RoiE,EnCalibr(ROIL)*ord(ROIL<ROIR),
                                                                    EnCalibr(ROIR)*ord(ROIL<ROIR)]);
  if Present(myMarkerLabel.Int0)
  then Result:=Result+Format('  %s=%g',[myMarkerLabel.Int0,GetRoiIntegral(ROIL,ROIR,0)]);
  if Present(myMarkerLabel.Int1)
  then Result:=Result+Format('  %s=%g',[myMarkerLabel.Int1,GetRoiIntegral(ROIL,ROIR,1)]);
  if Present(myMarkerLabel.Int2)
  then Result:=Result+Format('  %s=%g',[myMarkerLabel.Int2,GetRoiIntegral(ROIL,ROIR,2)]);
 end;
end;

function  TFormSpectrDaqWindow.EnCalibrExist:Boolean;
begin
 EnCalibrExist:=Spectrometr.Ok and
                (FullDaqDeviceList.IndexOf(Spectrometr)>=0) and
                (EnCalNum>=0) and (EnCalNum<Spectrometr.NumCalibrations);
end;

function  TFormSpectrDaqWindow.EnCalibr(Chan:Double):Double;
begin
 Result:=Spectrometr.Transform(EnCalNum,Chan,0);
end;

function  TFormSpectrDaqWindow.HwCalibrExist:Boolean;
begin
 HwCalibrExist:=Spectrometr.Ok and
                (FullDaqDeviceList.IndexOf(Spectrometr)>=0) and
                (HwCalNum>=0) and (HwCalNum<Spectrometr.NumCalibrations);
end;

function  TFormSpectrDaqWindow.HwCalibr(Chan:Double):Double;
begin
 HwCalibr:=Spectrometr.Transform(HwCalNum,Chan,0);
end;

function TFormSpectrDaqWindow.SaveCalibrations(FileName:LongString):Boolean;
var NameX,NameY:LongString;
begin
 Result:=false;
 try
  FileName:=UnifyFileAlias(FileName,ua_FileDefLow);
  if EnCalibrExist then
  if Spectrometr.Calibration[EnCalNum].Ok then begin
   NameX:=Spectrometr.Calibration[EnCalNum].NameX;
   NameY:=Spectrometr.Calibration[EnCalNum].NameY;
   try
    Spectrometr.Calibration[EnCalNum].NameX:='Chan';
    Spectrometr.Calibration[EnCalNum].NameY:='En';
    if not Spectrometr.Calibration[EnCalNum].SaveToFile(FileName,'',true)
    then Raise EWriteError.CreateFmt(RusEng('Ошибка записи калибровки En "%s".',
                                            'Error writing calibration En "%s".'),[FileName]);
   finally
    Spectrometr.Calibration[EnCalNum].NameX:=NameX;
    Spectrometr.Calibration[EnCalNum].NameY:=NameY;
   end;
  end;
  if HwCalibrExist then
  if Spectrometr.Calibration[HwCalNum].Ok then begin
   NameX:=Spectrometr.Calibration[HwCalNum].NameX;
   NameY:=Spectrometr.Calibration[HwCalNum].NameY;
   try
    Spectrometr.Calibration[HwCalNum].NameX:='Chan';
    Spectrometr.Calibration[HwCalNum].NameY:='Hw';
    if not Spectrometr.Calibration[HwCalNum].SaveToFile(FileName,'',true)
    then Raise EWriteError.CreateFmt(RusEng('Ошибка записи калибровки Hw "%s".',
                                            'Error writing calibration Hw "%s".'),[FileName]);
   finally
    Spectrometr.Calibration[HwCalNum].NameX:=NameX;
    Spectrometr.Calibration[HwCalNum].NameY:=NameY;
   end;
  end;
  Result:=true;
 except
  on E:Exception do BugReport(E,Self,'SaveCalibrations');
 end;
end;

function TFormSpectrDaqWindow.LoadCalibrations(FileName:LongString):Boolean;
var
 Cal : TPolynomCalibration;
begin
 Result:=false;
 try
  FileName:=UnifyFileAlias(FileName);
  if EnCalibrExist then begin
   if (Length(ExtractTextSection(FileName,'[Chan-En calibration]',efConfig))=0)
   then Raise EReadError.CreateFmt(RusEng('В файле "%s"'+EOL+'отсутствует калибровка "%s".',
                                          'File "%s"'+EOL+'do''nt contains calibration "%s".'),
                                          [FileName,'Chan-En']);
   Cal:=NewCalibrationLoadFromFile('Chan','En','',FileName,'');
   try
    if not Cal.Ok
    then Raise EReadError.CreateFmt(RusEng('Ошибка чтения калибровки "%s"'+EOL+'из файла "%s".',
                                           'Error read calibration "%s"'+EOL+'from file "%s".'),
                                           ['Chan-En',FileName]);
    if not Spectrometr.Calibration[EnCalNum].Ok then begin
     Spectrometr.InitCalibration(EnCalNum);
     Spectrometr.Calibration[EnCalNum].NameX:='Chan';
     Spectrometr.Calibration[EnCalNum].NameY:='En';
     Spectrometr.Calibration[EnCalNum].BoundA:=0;
     Spectrometr.Calibration[EnCalNum].BoundB:=SpectrSize-1;
    end;
    if Spectrometr.Calibration[EnCalNum].Ok then Spectrometr.Calibration[EnCalNum].Assign(Cal);
    EnCalibrChanged;
   finally
    Kill(Cal);
   end;
  end;
  if HwCalibrExist then begin
   if (Length(ExtractTextSection(FileName,'[Chan-Hw calibration]',efConfig))=0)
   then Raise EReadError.CreateFmt(RusEng('В файле "%s"'+EOL+'отсутствует калибровка "%s"',
                                          'File "%s"'+EOL+'do''nt contains calibration "%s"'),
                                          [FileName,'Chan-Hw']);
   Cal:=NewCalibrationLoadFromFile('Chan','Hw','',FileName,'');
   try
    if not Cal.Ok
    then Raise EReadError.CreateFmt(RusEng('Ошибка чтения калибровки "%s"'+EOL+'из файла "%s".',
                                           'Error read calibration "%s"'+EOL+'from file "%s".'),
                                           ['Chan-Hw',FileName]);
    if not Spectrometr.Calibration[HwCalNum].Ok then begin
     Spectrometr.InitCalibration(HwCalNum);
     Spectrometr.Calibration[HwCalNum].NameX:='Chan';
     Spectrometr.Calibration[HwCalNum].NameY:='Hw';
     Spectrometr.Calibration[HwCalNum].BoundA:=0;
     Spectrometr.Calibration[HwCalNum].BoundB:=SpectrSize-1;
    end;
    if Spectrometr.Calibration[HwCalNum].Ok then Spectrometr.Calibration[HwCalNum].Assign(Cal);
    HwCalibrChanged;
   finally
    Kill(Cal);
   end;
  end;
  Result:=true;
 except
  on E:Exception do BugReport(E,Self,'LoadCalibrations');
 end;
end;

procedure TFormSpectrDaqWindow.ActionSpectrEditCalibrationEnExecute(Sender: TObject);
begin
 inherited;
 if Spectrometr.Ok and (EnCalNum>=0) then begin
  Spectrometr.EditCalibration(EnCalNum);
  EnCalibrChanged;
 end;
end;

procedure TFormSpectrDaqWindow.ActionSpectrEditCalibrationHwExecute(Sender: TObject);
begin
 inherited;
 if Spectrometr.Ok and (HwCalNum>=0) then begin
  Spectrometr.EditCalibration(HwCalNum);
  HwCalibrChanged;
 end;
end;

procedure TFormSpectrDaqWindow.StatusBarDblClick(Sender: TObject);
var s:LongString; Buff:TParsingBuffer;
begin
 inherited;
 with myMarkerLabel do
 s:=Format(fmtMarkerLabels,[Ch,Cnt,En,Hw,RoiC,RoiE,Int0,Int1,Int2]);
 if InputQuery(RusEng('Формат статуса','Format status line'),
               RusEng('Задайте ','Setup ')+'Ch,Cnt,En,HW,RoiCh,RoiEn,Int,Gnd,Peak',s)
 then begin
  ScanVarRecord(svAsIs,StrCopyBuff(Buff,s),fmtMarkerLabela,myMarkerLabel);
  UpdateMarkerView(Marker);
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_spectrdaqwindow;
begin
 RegisterSpectrWindowConstructor(NewSpecDaqWin, regSpecDaqWin);
end;

procedure Free_form_spectrdaqwindow;
begin
end;

initialization

 Init_form_spectrdaqwindow;

finalization

 Free_form_spectrdaqwindow;

end.

//////////////
// END OF FILE
//////////////

