////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Lazarus Project Editor.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241009 - Created from Delphi Project Editor (A.K.)                       //
////////////////////////////////////////////////////////////////////////////////

unit form_lazarusprojecteditor; // Form Lazarus Project Editor

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 Form_ListBoxSelection,
 Form_TextEditor, Form_CurveWindow,
 Form_EditDataAnalysisPluginArguments,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_syscal,
 _crw_crwapi, _crw_crwapiserver,
 _crw_sect, _crw_sesman, _crw_fpcup,
 _crw_appforms, _crw_apptools, _crw_apputils;

type

  { TFormLazarusProjectEditor }

  TFormLazarusProjectEditor = class(TFormTextEditor)
    ActionEditCompile: TAction;
    ActionEditRun: TAction;
    ActionEditApiHelp: TAction;
    ActionEditCrwLibHelp: TAction;
    MenuEditCompile: TMenuItem;
    MenuEditRun: TMenuItem;
    MenuEditApiHelp: TMenuItem;
    MenuEditCrwLibHelp: TMenuItem;
    MenuEditDeleteSeparator: TMenuItem;
    ToolButtonEditCompileSeparator: TToolButton;
    ToolButtonEditCompile: TToolButton;
    ToolButtonEditRun: TToolButton;
    ToolButtonEditApiHelp: TToolButton;
    ToolButtonEditCrwLibHelp: TToolButton;
    PopupMenuEditCompile: TMenuItem;
    PopupMenuEditRun: TMenuItem;
    PopupMenuEditApiHelp: TMenuItem;
    PopupMenuEditCrwLibHelp: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure ActionEditCompileExecute(Sender: TObject);
    procedure ActionEditRunExecute(Sender: TObject);
    procedure ActionEditApiHelpExecute(Sender: TObject);
    procedure ActionEditCrwLibHelpExecute(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    procedure UpdateCommands; override;
  end;

function  NewLazarusProjectEditor(aProject:LongString):TFormLazarusProjectEditor;
function  FindLazarusProjectEditor(FileName:LongString;
                                   CreateNew:Boolean=false;
                                   BringToFront:Boolean=false):TFormLazarusProjectEditor;
procedure Kill(var TheObject:TFormLazarusProjectEditor); overload;

implementation

{$R *.lfm}

function NewLazarusProjectEditor(aProject:LongString):TFormLazarusProjectEditor;
begin
 Result:=nil;
 try
  aProject:=UnifyFileAlias(aProject);
  if FileExists(aProject) then begin
   Application.CreateForm(TFormLazarusProjectEditor, Result);
   if Result.Ok then begin
    Result.Open(aProject);
    Result.DefaultReadOnly;
   end;
  end else Echo(Format(RusEng('Не найден проект: "%s".',
                              'Not found project: "%s".'),[aProject]));
 except
  on E:Exception do BugReport(E,nil,'NewLazarusProjectEditor');
 end;
end;

type
 TFindRec = packed record
  Match    : TFormLazarusProjectEditor;
  PathName : LongString;
 end;

procedure FindForm(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 with TFindRec(Custom^) do
 if Form is TFormLazarusProjectEditor then
 if IsSameFileName((Form as TFormLazarusProjectEditor).PathName,PathName) then begin
  Match:=(Form as TFormLazarusProjectEditor);
  Terminate:=true;
 end;
end;

function FindLazarusProjectEditor(FileName:LongString;
                                  CreateNew:Boolean=false;
                                  BringToFront:Boolean=false):TFormLazarusProjectEditor;
var FindRec:TFindRec;
begin
 Result:=nil;
 try
  FindRec:=Default(TFindRec);
  try
   FileName:=UnifyFileAlias(FileName);
   FindRec.Match:=nil;
   FindRec.PathName:=StringBuffer(FileName);
   SdiMan.ForEachChild(FindForm,@FindRec);
   Result:=FindRec.Match;
   if CreateNew and not Assigned(Result) then Result:=NewLazarusProjectEditor(FileName);
   if Assigned(Result) and BringToFront then begin
    Result.Show;
    Result.WindowState:=wsNormal;
    SdiMan.ActivateChild(Result);
   end;
  finally
   FindRec.PathName:='';
  end;
 except
  on E:Exception do BugReport(E,nil,'FindLazarusProjectEditor');
 end;
end;

procedure Kill(var TheObject:TFormLazarusProjectEditor); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

procedure TFormLazarusProjectEditor.FormCreate(Sender: TObject);
begin
 inherited;
 UpdateMenu(MenuEditCompile,
            RusEng('Компиляция','Compile'),
            RusEng('Компилировать Lazarus программу.','Compile Lazarus program source.'),
            ShortCut(VK_F9,[ssCtrl]));
 UpdateMenu(MenuEditRun,
            RusEng('Выполнение','Run'),
            RusEng('Выполнить Lazarus программу.','Execute Lazarus program.'),
            0);
 UpdateMenu(MenuEditApiHelp,
            RusEng('Справка по API','API help'),
            RusEng('Вызов справки по API.','Call API help window.'),
            ShortCut(VK_F1, [ssCtrl]));
 UpdateMenu(MenuEditCrwLibHelp,
            RusEng('Справка по CrwLib','CrwLib help'),
            RusEng('Вызов справки по CrwLib.','Call CrwLib help window.'),
            0);
end;

procedure TFormLazarusProjectEditor.UpdateCommands;
var Busy:Boolean;
begin
 try
  inherited UpdateCommands;
  Busy:=Fpcup.LazBuildBusy;
  ActionEditRun.Enabled:=not Busy;
  ActionEditCompile.Enabled:=not Busy;
  ActionEditApiHelp.Enabled:=not Busy;
 except
  on E:Exception do BugReport(E,Self,'UpdateCommands');
 end;
end;

procedure FpcupGoToError(Form:TFormLazarusProjectEditor);
var i,p,row,col:Integer; line,sc,sr,fn:LongString;
begin
 if SdiMan.IsChild(Form) then begin
  for i:=0 to Fpcup.Errors.Count-1 do begin
   line:=Fpcup.Errors[i]; p:=Pos(') Error: ',line);
   if (p<=0) then continue; Dec(p); sc:=''; sr:='';
   while IsLexeme(StrFetch(line,p),lex_digit) do begin sc:=StrFetch(line,p)+sc; Dec(p); end;
   if (StrFetch(line,p)=',') then Dec(p) else continue;
   while IsLexeme(StrFetch(line,p),lex_digit) do begin sr:=StrFetch(line,p)+sr; Dec(p); end;
   if (StrFetch(line,p)='(') then Dec(p) else continue;
   fn:=Copy(line,1,p); col:=StrToIntDef(sc,0); row:=StrToIntDef(sr,0);
   if IsSameFileName(fn,Form.PathName) and (row>0) and (col>0) then begin
    SdiMan.ActivateChild(Form);
    Form.GoToRowCol(row,col);
    Exit;
   end;
  end;
 end;
end;

const TheLazProjEditor:TFormLazarusProjectEditor=nil;

procedure Timer_LazProjEditorReporter;
var State:TFpcupStateCode;
begin
 State:=Fpcup.LazBuildState;
 if Fpcup.LazBuildBusy then Exit;
 if (State in [scStandby,scRunning]) then Exit;
 SecondActions.Remove(Timer_LazProjEditorReporter);
 if SdiMan.IsChild(TheLazProjEditor) then begin
  Echo(StdDateTimePrompt(msecnow)
      +RusEng('Завершение компиляции ',
              'Done compile ')+Fpcup.LazBuildLpr);
  Fpcup.DoReading;
  Echo(Fpcup.ReportLazBuild);
  FpcupGoToError(TheLazProjEditor);
 end;
 Fpcup.ResetLazBuild;
 TheLazProjEditor:=nil;
end;

procedure TFormLazarusProjectEditor.ActionEditCompileExecute(Sender:TObject);
var msg:LongString;
begin
 if Guard.CheckAction(ga_Root,ActionEditCompile)<0 then Exit;
 inherited;
 try
  if PerformModified then FileSave;
  if not Fpcup.LazarusInstalled then begin
   msg:=RusEng('Компилятор FpcupDeluxe/Lazarus не найден.',
               'Compiler FpcupDeluxe/Lazarus is not found.');
   SendToMainConsole(Format('@silent @tooltip text "%s: %s" preset stdError delay 60000',
                    [SessionManager.TitlePidAtHost,msg])+EOL);
   Echo(msg);
   Exit;
  end;
  if (Fpcup.LazBuildState in [scRunning]) then begin
   Echo(RusEng('Компилятор сейчас занят. Попробуйте позже.',
               'Compiler is busy now. Try copmpile later.'));
   Exit;
  end;
  if Fpcup.RunCompile(PathName) then begin
   Echo(StdDateTimePrompt(msecnow)
       +RusEng('Пуск компиляции ',
               'Start compile ')+Fpcup.LazBuildLpr);
   TheLazProjEditor:=Self;
   SecondActions.Add(Timer_LazProjEditorReporter);
  end else begin
   Echo(StdDateTimePrompt(msecnow)
       +RusEng('Сбой запуска компиляции ',
               'Could not start compile ')+Fpcup.LazBuildLpr);
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditCompileExecute');
 end;
end;

procedure TFormLazarusProjectEditor.ActionEditRunExecute(Sender: TObject);
var CrwApi:TCrwApiServer; WinStat:TWindowState;
var Argums,DllPath,ExePath,Params,Msg:LongString;
var aName,aSource,aBinary:LongString;
const MinimizeOnRun:Boolean=true;
begin
 if Guard.CheckAction(ga_User,ActionEditRun)<0 then Exit;
 inherited;
 Argums:=''; Msg:='';
 WinStat:=WindowState;
 Params:=ControlPosParams(Editor,'LT');
 if MinimizeOnRun then begin
  WindowState:=wsMinimized;
  SafeApplicationProcessMessages;
 end;
 try
  try
   DllPath:=ForceExtension(PathName,'.dll');
   ExePath:=ForceExtension(PathName,'.exe');
   DllPath:=UnifyFileAlias(AdaptDllFileName(DllPath),ua_RealPath);
   ExePath:=UnifyFileAlias(AdaptExeFileName(ExePath),ua_RealPath);
   if PerformModified then begin
    Msg:=Format(RusEng('Код поекта %s обновлен. Запущена компиляция.',
                       'Project %s is modified. Compilation started.'),
               [ExtractFileNameExt(PathName)]);
    SendToMainConsole(Format('@silent @tooltip text "%s: %s" preset stdNotify delay 15000',
                            [SessionManager.TitlePidAtHost,Msg])+EOL);
    ActionEditCompile.Execute;
    Echo(Msg);
    Exit;
   end;
   if not (FileExists(DllPath) or FileExists(ExePath)) then begin
    Msg:=Format(RusEng('Исполняемый файл проекта %s не найден. Запущена компиляция.',
                       'Executable file of project %s not found. Compilation started.'),
               [ExtractFileNameExt(PathName)]);
    SendToMainConsole(Format('@silent @tooltip text "%s: %s" preset stdNotify delay 15000',
                            [SessionManager.TitlePidAtHost,Msg])+EOL);
    ActionEditCompile.Execute;
    Echo(Msg);
    Exit;
   end;
   if not ActiveCurveWindow.CheckIsDataProtected then begin
    if FileExists(DllPath) then begin
     if ActiveCurveWindow.Ok
     then Params:=ControlPosParams(ActiveCurveWindow,'LT',0,GetWindowCaptionHeight);
     if ExecuteFormEditDataAnalysisArguments(Self,Argums,Params)=mrOk then begin
      aSource:=PathName; aBinary:=DllPath;
      aName:=LowerCase(ExtractBaseName(aSource));
      CrwApi:=NewCrwApiServer(ForDataAnalysis,aName,aSource,aBinary);
      if Assigned(CrwApi) then
      try
       CrwApi.ExecuteDataAnalysis(DllPath,ActiveCurveWindow,Argums);
      finally
       CrwApi.Free;
      end;
     end;
    end else
    if FileExists(ExePath) then begin
     SendToMainConsole(Format('@silent @term -ca task "%s"',[ExePath])+EOL);
    end;
   end;
  finally
   WindowState:=WinStat; if (WinStat=wsNormal) then SdiMan.ActivateChild(Self);
   Argums:='';
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditRunExecute');
 end;
end;

procedure AddFileToList(const FileName:LongString; const FileDetails:TSearchRec;
                        SubDirLevel:Integer; var Terminate:Boolean; CustomData:Pointer);
begin
 if FileDetails.Attr and ({faVolumeID+}faDirectory)=0 then
 with TText(CustomData) do Addln(MakeRelativePath(FileName,Line[0]));
end;

procedure TFormLazarusProjectEditor.ActionEditApiHelpExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditApiHelp)<0 then Exit;
 inherited;
 with Fpcup do LazHelp(IniFileLazIndex,'');
end;

procedure TFormLazarusProjectEditor.ActionEditCrwLibHelpExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditCrwLibHelp)<0 then Exit;
 inherited;
 with Fpcup do LazHelp(IniFileCrwLibIndex,'');
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_lazarusprojecteditor;
begin
end;

procedure Free_form_lazarusprojecteditor;
begin
end;

initialization

 Init_form_lazarusprojecteditor;

finalization

 Free_form_lazarusprojecteditor;

end.

//////////////
// END OF FILE
//////////////

