////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Unit System Console.                                                       //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231211 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit unit_systemconsole; // Unit System Console

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Printers,
 Grids, Spin,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 Form_TextEditor, Form_CurveWindow,
 Form_ConsoleWindow, Form_TermWindow,
 Form_TabWindow, Form_CircuitWindow,
 Form_NsisWindow, Unit_ResourceMonitorConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_spk, _crw_guard,
 _crw_ef, _crw_ee, _crw_pio, _crw_environ,
 _crw_polling, _crw_task, _crw_proc, _crw_assoc,
 _crw_curves, _crw_riff, _crw_couple, _crw_cmdargs,
 _crw_pipe, _crw_pipeio,  _crw_tcp, _crw_lm,
 _crw_crypt, _crw_base64, _crw_meta, _crw_prmstrlst,
 _crw_fonts, ptembed, _crw_dcc32, _crw_utf8,
 _crw_hash, _crw_hl, _crw_colors, _crw_sect,
 _crw_fsm, _crw_wine, _crw_vbox, _crw_dbapi,
 _crw_daqtags, _crw_lttb, _crw_syscal, _crw_uart,
 _crw_spcfld, _crw_sysid, _crw_wmctrl, _crw_bmpcache,
 _crw_mimeapps, _crw_gloss, _crw_sesman, _crw_uri,
 _crw_geoid, _crw_lngid, _crw_dbglog, _crw_fpcup,
 _crw_netif, _crw_flgrd, _crw_atomic, _crw_syslog,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
 ESystemConsole = class(EEchoException);
 TSystemConsole = class(TMasterObject)
 private
  myForm          : TFormConsoleWindow;
  myEchoFile      : LongString;
  myFileLimit     : LongInt;
  myLines         : TText;
  myTtyLatch      : TLatch;
  myTtyPort0      : LongInt;
  myTtyPorts      : TObjectStorage;
  myTtyShield     : Boolean;
  myTtyVerbose    : Boolean;
  myIntegrityFile : LongString;
  myDelayedEe     : TExpressionEvaluator; // Uses to perform delayed evaluation of expressions
  myDelayedFlag   : Boolean;              // without SystemCalculator locks to avoid deadlock.
  function  ShouldBeDelayed:Boolean;
  function  DelayedEvaluate(const Expression : LongString;
                              var Answer     : Double;
                              var Comment    : LongString) : Integer;
  function  GetForm:TFormConsoleWindow;
  function  GetEchoFile:LongString;
  procedure SetEchoFile(const aEchoFile:LongString);
  function  GetFileLimit:LongInt;
  procedure SetFileLimit(const aFileLimit:LongInt);
  function  TtyFind(aPort:Integer):TTcpServer;
  function  GetTtyShield:Boolean;
  procedure SetTtyShield(aShield:Boolean);
  function  GetTtyVerbose:Boolean;
  procedure SetTtyVerbose(aVerbose:Boolean);
  procedure RegisterActions;
  procedure RegisterFunctions;
 public
  constructor Create;
  destructor  Destroy; override;
  procedure   Open;
  procedure   Close(NeedSave:Boolean=true);
  procedure   GoHome;
  procedure   GoTail;
  procedure   Activate;
 public
  property    Form      : TFormConsoleWindow read GetForm;
  property    EchoFile  : LongString         read GetEchoFile  write SetEchoFile;
  property    FileLimit : LongInt            read GetFileLimit write SetFileLimit;
 public
  function  TtyListenCount(aPort:Integer):LongInt;
  function  TtyListen(aPort:Integer):LongInt;
  function  TtyClose(aPort:Integer):LongInt;
  function  TtyPortList:LongString;
  procedure TtyCloseAll;
  procedure TtyPolling;
  function  IntegrityEventsLog(const aLine:LongString):Integer;
 public
  property TtyShield    : Boolean            read GetTtyShield  write SetTtyShield;
  property TtyVerbose   : Boolean            read GetTtyVerbose write SetTtyVerbose;
 end;

function SystemConsole:TSystemConsole;

procedure Init_System_Console;
procedure Done_System_Console;
procedure ExecuteSystemConsoleMonitoring(Update:Boolean=false);

const CompileDprVerbose : Boolean = false;
const CompileLprVerbose : Boolean = false;

implementation

uses
 {$IFDEF Poligon}poligon,{$ENDIF Poligon}
 _crw_DaqSys,
 _crw_DaqDev,
 _crw_AdamDev,
 _crw_DaqPascalDevice,
 _crw_CrwApiServer,
 form_daqcontroldialog,
 Form_SpectrWindow,
 Form_CrwDaqLogo,
 Form_CrwDaq;

const
 TcpParamN  = 0;
 TcpMinPort = 1024;
 TcpMaxPort = 65535;
 DefaultTtyPorts = 8;
 DefaultTtyPipes = 8;

function PortInRange(p,a,b:Integer):Boolean;
begin
 Result:=(p>=a) and (p<=b);
end;

 //////////////////////////////////////////
 // Private Hasher for fast string parsing.
 //////////////////////////////////////////
type
 TStringIdentifier = (
  sid_Unknown,
  // @daq
  sid_Compile,
  sid_DevSend,
  sid_DevMsg,
  sid_DevSendMsg,
  sid_DevPost,
  sid_DevPostMsg,
  sid_OpenConsole,
  sid_OpenEditor,
  sid_TopList,
  sid_LoadWindowsDialog,
  sid_KillWindowsDialog,
  sid_Info,
  // end
  sid_Unused
 );

const
 Hasher:THashList=nil;

procedure FreeHasher;
begin
 Kill(Hasher);
end;

procedure InitHasher;
 procedure AddSid(const key:LongString; sid:TStringIdentifier);
 begin
  if (key<>'') then
  if (sid<>sid_Unused) then
  if (sid<>sid_Unknown) then
  Hasher.KeyedLinks[key]:=Ord(sid);
 end;
begin
 if (Hasher<>nil) then Exit;
 Hasher:=NewHashList(false,HashList_DefaultHasher);
 Hasher.Master:=@Hasher;
 ////////////////////////////////////////////
 // Hash List for fast strings identification
 ////////////////////////////////////////////
 AddSid( 'Compile'           , sid_Compile);
 AddSid( 'DevSend'           , sid_DevSend);
 AddSid( 'DevMsg'            , sid_DevMsg);
 AddSid( 'DevSendMsg'        , sid_DevSendMsg);
 AddSid( 'DevPost'           , sid_DevPost);
 AddSid( 'DevPostMsg'        , sid_DevPostMsg);
 AddSid( 'OpenConsole'       , sid_OpenConsole);
 AddSid( 'OpenEditor'        , sid_OpenEditor);
 AddSid( 'TopList'           , sid_TopList);
 AddSid( 'LoadWindowsDialog' , sid_LoadWindowsDialog);
 AddSid( 'KillWindowsDialog' , sid_KillWindowsDialog);
 AddSid( 'Info'              , sid_Info);
end;

function Identify(const key:LongString):TStringIdentifier;
var sid:Integer;
begin
 if (Hasher=nil) then InitHasher;
 sid:=Hasher.KeyedLinks[key];
 if (sid>=Ord(Low(TStringIdentifier))) and (sid<=Ord(High(TStringIdentifier)))
 then Result:=TStringIdentifier(sid)
 else Result:=sid_Unknown;
end;

 {
 *******************************************************************************
 Процедуры для организации системной консоли
 *******************************************************************************
 }
const
 SilentInp : Boolean = true;
 SilentOut : Boolean = true;
 IsActEcho : Boolean = true;

procedure ActEcho(const s:LongString);
begin
 if IsActEcho then Echo(s);
end;
 
function act_silent(ee:TExpressionEvaluator; const args:LongString):double; forward;

procedure SystemInputFilter(aConsole:TFormConsoleWindow; var aText:LongString);
var i:Integer; Answer:Double; Line,Comment:LongString; Silent,cond:Boolean;
begin
 try
  with SystemConsole do
  if Form.Ok and (Form=aConsole) and (Length(aText)>0) then begin
   if SystemCalculator.Ok then begin
    myLines.Text:=aText;
    for i:=0 to myLines.Count-1 do begin
     Line:=myLines[i];
     if (Line<>'') then begin
      if SysLog.Notable(SeverityOfSysInput)
      then SysLog.INFO(SeverityOfSysInput,sdr_SysInput,Line);
      Silent:=SameText('@silent',ExtractWord(1,Line,ScanSpaces));
      if not Silent or not SilentInp then Form.PutText(Line+EOL);
      if Silent then begin // Skip the @silent
       if WordCount(Line,ScanSpaces)>1 then
       while SameText('@silent',ExtractWord(1,Line,ScanSpaces)) do
       Line:=SkipWords(1,Line,ScanSpaces);
      end;
      cond:=true; Comment:=''; Answer:=0;
      if SameText('@if',ExtractWord(1,Line,JustSpaces)) then begin
       try
        SystemCalculator.Evaluate(ExtractWord(2,Line,JustSpaces),Answer,Comment);
        if IsNan(Answer) or IsInf(Answer) or (Answer=0) then cond:=false;
        Line:=SkipWords(2,Line,JustSpaces);
        if (Line='') then begin Line:='@if'; cond:=true; end; // To get help on @if
        if SameText('then',ExtractWord(1,Line,JustSpaces)) then Line:=SkipWords(1,Line,JustSpaces);
        Comment:=''; Answer:=0;
       finally
        IsActEcho:=true;
       end;
      end;
      if cond then
      try
       myDelayedFlag:=false;
       IsActEcho:=not Silent or not SilentOut;
       SystemCalculator.Evaluate(Line,Answer,Comment);
       if myDelayedFlag then DelayedEvaluate(Line,Answer,Comment);
      finally
       IsActEcho:=true;
      end;
      if not Silent or not SilentOut then
      if (Comment<>'') then Form.PutText(Comment+EOL);
     end else Form.PutText(EOL);
    end;
    myLines.Count:=0;
   end else begin
    Form.PutText(aText);
    Form.PutText(RusEng('Интерпретатор недоступен!','Interpreter fails!')+EOL);
   end;
  end;
 except
  on E:Exception do BugReport(E,aConsole,'SystemInputFilter');
 end;
end;

procedure SystemOutputFilter(aConsole:TFormConsoleWindow; var aText:LongString);
var F:Text; IOR:Integer;
begin
 try
  with SystemConsole do
  if Form.Ok and (Form=aConsole) and (Length(EchoFile)>0) and (Length(aText)>0) then begin
   IOR:=IOResult;
   System.Assign(F,EchoFile);
   if FileExists(EchoFile) then System.Append(F) else System.Rewrite(F);
   try
    System.Write(F,aText);
   finally
    System.Close(F);
    SetInOutRes(IOR);
   end;
   if GetFileSize(EchoFile) shr 20 > FileLimit
   then FileRename(EchoFile,ForceExtension(EchoFile,'.OLD'));
  end;
 except
  on E:Exception do BugReport(E,aConsole,'SystemOutputFilter');
 end;
end;

 {
 *******************************************************************************
 Акции для регистрации в ExpressionEvaluator
 Результат выполнения содержится в переменной actionresult.
 *******************************************************************************
 }
function act_help(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1,TopicText,HelpFile:LongString;
begin
 Result:=0;
 if IsMainThread then
 try
  HelpFile:='';
  TopicText:='';
  arg:=ee.SmartArgs(args);
  w1:=ExtractWord(1,arg,ScanSpaces);
  if w1='' then w1:='@help';
  if SysGlossary.ReadIniPath(SysIniFile,SectSystem,'ConsoleManual',HomeDir,HelpFile) and FileExists(HelpFile)
  then TopicText:=ExtractTextSection(HelpFile,UnifySection('HelpTopic '+RusEng('Russian ','English ')+w1),efAsIs);
  if (TopicText<>'')
  then Echo(TopicText)
  else Echo(RusEng('Не могу найти справку на тему ','Could not find help on ')+w1);
  Result:=Length(TopicText);
  TopicText:='';
 except
  on E:Exception do BugReport(E,ee,'act_help');
 end;
end;

type
 TGetUsersThread = class(TThread)
 protected
  arg: LongString;
  procedure Execute; override;
 public
  constructor Create(const aArg:LongString);
  destructor  Destroy; override;
 end;

constructor TGetUsersThread.Create(const aArg:LongString);
begin
 inherited Create(True);
 FreeOnTerminate:=True;
 Priority:=tpIdle;
 arg:=aArg;
 Suspended:=false; // Insted of obsolete Resume;
end;

destructor TGetUsersThread.Destroy;
begin
 arg:='';
 inherited Destroy;
end;

procedure TGetUsersThread.Execute;
var p:TText;
begin
 try
  p:=NewText;
  try
   p.Text:=GetUserList(URL_Decode(ExtractWord(2,arg,ScanSpaces)),
                      StrToIntDef(ExtractWord(3,arg,ScanSpaces),0),
                      StrToIntDef(ExtractWord(4,arg,ScanSpaces),FILTER_NORMAL_ACCOUNT),
                      StrToIntDef(ExtractWord(5,arg,ScanSpaces),DefNetEnumTimeOut));
   if p.Count=0
   then Echo(RusEng('Список пользователей недоступен.','User list is not available.'))
   else Echo(RusEng('Список пользователей:','User list:')+EOL+p.Text);
  finally
   Kill(p);
  end;
 except
  on E:Exception do BugReport(E,Self,'Execute');
 end;
end;

type
 TGetHostsThread = class(TGetUsersThread)
 protected
  procedure Execute; override;
 end;

procedure TGetHostsThread.Execute;
var p:TText; i,L1,L2,L3:Integer;
begin
 try
  p:=NewText;
  try
   p.Text:=GetHostList(URL_Decode(ExtractWord(2,arg,ScanSpaces)),
                       URL_Decode(ExtractWord(3,arg,ScanSpaces)),
                      StrToIntDef(ExtractWord(4,arg,ScanSpaces),0),
                      StrToIntDef(ExtractWord(5,arg,ScanSpaces),Integer(SV_TYPE_ALL)),
                      StrToIntDef(ExtractWord(6,arg,ScanSpaces),DefNetEnumTimeOut));
   L1:=Max(Length('localhost'),Length(ComputerName));
   L2:=Length('255.255.255.255');
   L3:=Length('localhost');
   for i:=0 to p.Count-1 do begin
    L1:=Max(L1,Length(ExtractWord(1,p[i],ScanSpaces)));
    L2:=Max(L2,Length(ExtractWord(2,p[i],ScanSpaces)));
    L3:=Max(L3,Length(ExtractWord(3,p[i],ScanSpaces)));
   end;
   if StrToIntDef(ExtractWord(4,arg,ScanSpaces),0)>0 then begin
    p.InsLn(0,Format('%-*s %-*s %-*s %s',[L1,'.',L2,GetIpAddress('localhost'),L3,'localhost','loopback']));
    p.InsLn(1,Format('%-*s %-*s %-*s %s',[L1,ComputerName,L2,GetIpAddress(ComputerName),L3,'localhost','this host']));
   end else begin
    p.InsLn(0,'.');
    p.InsLn(1,ComputerName);
   end;
   if p.Count=0
   then Echo(RusEng('Список серверов недоступен.','Host list is not available.'))
   else Echo(RusEng('Список серверов:','Host list:')+EOL+p.Text);
  finally
   Kill(p);
  end;
 except
  on E:Exception do BugReport(E,Self,'Execute');
 end;
end;

type
 TGetDomainsThread = class(TGetUsersThread)
 protected
  procedure Execute; override;
 end;

procedure TGetDomainsThread.Execute;
var p:TText;
begin
 try
  p:=NewText;
  try
   p.Text:=p.Text+GetDomainList(URL_Decode(ExtractWord(2,arg,ScanSpaces)),
                               StrToIntDef(ExtractWord(3,arg,ScanSpaces),0),True,
                               StrToIntDef(ExtractWord(4,arg,ScanSpaces),DefNetEnumTimeOut));
   if p.Count=0
   then Echo(RusEng('Список доменов недоступен.','Domain list is not available.'))
   else Echo(RusEng('Список доменов:','Domain list:')+EOL+p.Text);
  finally
   Kill(p);
  end;
 except
  on E:Exception do BugReport(E,Self,'Execute');
 end;
end;

function act_list(ee:TExpressionEvaluator; const args:LongString):double;
var arg,TopicText,w1,w2,s:LongString; List:TStringList;
var pid,tid,m:Integer; p:TText;
begin
 Result:=0;
 if IsMainThread then
 try
  TopicText:='';
  arg:=ee.SmartArgs(args);
  w1:=ExtractWord(1,arg,ScanSpaces);
  w2:=ExtractWord(2,arg,ScanSpaces);
  if IsSameText(w1,'pids') then begin
   Result:=SendToMainConsole('@pid list'+EOL);
   Exit;
  end else
  if IsSameText(w1,'threads') then begin
   Result:=SendToMainConsole('@polling list'+EOL);
   Exit;
  end else
  if IsSameText(w1,'modules') then begin
   pid:=StrToIntDef(ExtractWord(2,arg,ScanSpaces),GetCurrentProcessId);
   List:=GetListOfModules(TStringList.Create,pid);
   try
    TopicText:=Format('Process %d uses %d modules.',[pid,List.Count])+EOL+List.Text;
    Result:=List.Count;
   finally
    Kill(List);
   end;
  end else
  if IsSameText(w1,'menus') then begin
   Result:=SendToMainConsole('@menu list'+EOL);
   Exit;
  end else
  if IsSameText(w1,'keys') then begin
   Result:=SendToMainConsole('@menu list keys'+EOL);
   Exit;
  end else
  if IsSameText(w1,'views') then begin
   Result:=SendToMainConsole('@view list'+EOL);
   Exit;
  end else
  if IsSameText(w1,'forms') then begin
   Result:=SendToMainConsole('@view -f list'+EOL);
   Exit;
  end else
  if IsSameText(w1,'vars') then begin
   p:=ee.VarList.GetText(NewText);
   try
    if p.Count=0
    then TopicText:=RusEng('Список переменных пуст.','Variable list empty.')+EOL
    else TopicText:=RusEng('Список переменных:','Variable list:')+EOL+p.Text;
    Result:=p.Count;
   finally
    Kill(p);
   end;
  end else
  if IsSameText(w1,'cons') then begin
   p:=ee.ConstList.GetText(NewText);
   try
    if p.Count=0
    then TopicText:=RusEng('Список констант пуст.','Constant list empty.')+EOL
    else TopicText:=RusEng('Список констант:','Constant list:')+EOL+p.Text;
    Result:=p.Count;
   finally
    Kill(p);
   end;
  end else
  if IsSameText(w1,'funs') then begin
   p:=ee.FuncList.GetText(NewText);
   try
    if p.Count=0
    then TopicText:=RusEng('Список функций пуст.','Function list empty.')+EOL
    else TopicText:=RusEng('Список функций:','Function list:')+EOL+p.Text;
    Result:=p.Count;
   finally
    Kill(p);
   end;
  end else
  if IsSameText(w1,'acts') then begin
   p:=ee.ActionList.GetText(NewText);
   try
    if p.Count=0
    then TopicText:=RusEng('Список акций пуст.','Action list empty.')+EOL
    else TopicText:=RusEng('Список акций:','Action list:')+EOL+p.Text;
    Result:=p.Count;
   finally
    Kill(p);
   end;
  end else
  if IsSameText(w1,'environs') then begin
   if (w2='')
   then TopicText:=ParamStrListOf('environs')
   else TopicText:=ParamStrListOf('environs where name='+w2);
   Result:=WordCount(TopicText,EolnDelims);
   Echo(SysUtils.TrimRight(TopicText));
   Exit;
  end else
  if IsSameText(w1,'groups') then begin
   if (w2='')
   then TopicText:=ParamStrListOf('groups')
   else TopicText:=ParamStrListOf('groups where name='+w2);
   Result:=WordCount(TopicText,ScanSpaces);
   Echo(SysUtils.TrimRight(TopicText));
   Exit;
  end else
  if IsSameText(w1,'ftypes') then begin
   if (w2='')
   then TopicText:=ParamStrListOf('ftypes')
   else TopicText:=ParamStrListOf('ftypes where name='+w2);
   Result:=WordCount(TopicText,EolnDelims);
   Echo(SysUtils.TrimRight(TopicText));
   Exit;
  end else
  if IsSameText(w1,'assocs') then begin
   if (w2='')
   then TopicText:=ParamStrListOf('assocs')
   else TopicText:=ParamStrListOf('assocs where name='+w2);
   Result:=WordCount(TopicText,EolnDelims);
   Echo(SysUtils.TrimRight(TopicText));
   Exit;
  end else
  if IsSameText(w1,'users') then begin
   if IsUnix then begin
    if (w2='')
    then TopicText:=ParamStrListOf('users')
    else TopicText:=ParamStrListOf('users where name='+w2);
    Result:=WordCount(TopicText,ScanSpaces);
    Echo(SysUtils.TrimRight(TopicText));
    Exit;
   end;
   TopicText:=' Search users: Please wait...';
   TGetUsersThread.Create(arg);
   Result:=1;
  end else
  if IsSameText(w1,'hosts') then begin
   if IsUnix then begin
    if (w2='')
    then TopicText:=ParamStrListOf('hosts')
    else TopicText:=ParamStrListOf('hosts where name='+w2);
    Result:=WordCount(TopicText,ScanSpaces);
    Echo(SysUtils.TrimRight(TopicText));
    Exit;
   end;
   TopicText:=' Search hosts: Please wait...';
   TGetHostsThread.Create(arg);
   Result:=1;
  end else
  if IsSameText(w1,'domains') then begin
   TopicText:=' Search domains: Please wait...';
   TGetDomainsThread.Create(arg);
   Result:=1;
  end else
  if IsSameText(w1,'specfolders') then begin
   if (w2='') or (w2='*')
   then TopicText:=CSIDL_ListAllAsText
   else TopicText:=CSIDL_FolderByName(w2,'')+EOL;
   Result:=CountPos(EOL,TopicText);
  end else
  if IsSameText(w1,'windows') then begin
   List:=TStringList.Create;
   try
    List.Text:=GetListOfWindows(SkipWords(1,arg,ScanSpaces));
    Echo(SysUtils.Trim(List.Text));
    Result:=List.Count;
   finally
    Kill(List);
   end;
   Exit;
  end else
  if IsSameText(w1,'tasks') then begin
   List:=TStringList.Create;
   try
    for tid:=task_ref_min to task_ref_max do
    if (task_ref(tid)=nil) then continue else begin
     pid:=task_pid(tid);
     if (List.Count=0) then begin
      w2:=Format('%-4s  %-5s  %-7s  %-8s  %s',['Task','PID','Status','Priority','CmdLine']);
      List.Add(w2);
     end;
     w2:=Format('%-4d  %-5d  %-7s  %-8s  %s',[tid,pid,ExtractWord(1+Ord(task_wait(tid,0)),'Stopped,Running',ScanSpaces),
               task_ctrl(tid,'ProcessPriority'),task_ctrl(tid,'CmdLine')]);
     List.Add(w2);
    end;
    ActEcho(SysUtils.Trim(List.Text));
    ActEcho(Format('Task(s) found: %d',[Max(0,List.Count-1)]));
    Result:=Max(0,List.Count-1);
   finally
    Kill(List);
   end;
   Exit;
  end else
  if IsSameText(w1,'netifs') then begin
   m:=StrToIntDef(w2,-1);
   s:=GetListOfNetworkInterfaces(m);
   Result:=ForEachStringLine(s,nil,nil);
   Echo(s);
   Exit;
  end else
  if IsSameText(w1,'colors') then begin
   m:=WordIndex(w2,'pas,crc,htm',ScanSpaces);
   case m of
    0,1,2: Echo(KnownColorsListAsText(m));
    3:     begin
            w2:=ExtractWord(3,arg,ScanSpaces);
            if (w2='') then
            if not SysGlossary.ReadIniPath(SysIniFile,SectSystem,'crw-daq-colors.htm',HomeDir,w2)
            then w2:='resource\manual\crw-daq-colors.htm';
            w2:=DefaultPath(w2,HomeDir);
            if (w2='') then Exit;
            if FileExists(w2) then DeleteFile(w2);
            TopicText:=KnownColorsListAsText(m);
            if (WriteBufferToFile(w2,TopicText)>0) then begin
             if IsEmptyStr(ExtractWord(3,arg,ScanSpaces))
             then SendToMainConsole('@run HtmlBrowser '+w2+EOL);
            end else Echo('Error writing '+w2);
            TopicText:='';
           end;
   end;
   Exit;
  end else
  if SameText(w1,'OleDbProviderNames') then begin
   OleDbProviderNames.Text:='';
   Echo(OleDbProviderNames.Text);
   Exit;
  end else
  if SameText(w1,'OdbcDriverNames') then begin
   OdbcDriverNames.Text:='';
   Echo(OdbcDriverNames.Text);
   Exit;
  end else
  TopicText:='';
  if (TopicText<>'') then Echo(SysUtils.TrimRight(TopicText)) else begin
   Echo('Syntax:');
   Echo(' @list vars            - list all variables');
   Echo(' @list cons            - list all constants');
   Echo(' @list funs            - list all functions');
   Echo(' @list acts            - list all @actions');
   Echo(' @list pids            - list all running processes');
   Echo(' @list threads         - list all running threads');
   Echo(' @list modules         - list all uses modules exe,dll');
   Echo(' @list modules pid     - list all uses modules by process pid');
   Echo(' @list tasks           - list all running tasks');
   Echo(' @list netifs          - list all network interfaces');
   Echo(' @list menus           - list all menu commands');
   Echo(' @list keys            - list all menu hot keys');
   Echo(' @list views           - list all visual components');
   Echo(' @list forms           - list all forms (windows)');
   Echo(' @list specfolders     - list all special folders');
   Echo(' @list specfolders id  - list special folder by identifier (id)');
   Echo(' @list users h l f t   - list users: host, level, filter, timeout ');
   Echo(' @list hosts h d l f t - list hosts: host, domain, l, filter, timeout');
   Echo(' @list domains h l t   - list domains: host, level, timeout');
   Echo('  host,domain are URL-encoded; level 0/1=simple/detail; timeout in ms');
   Echo('  users filters:');
   Echo('   $01 Enumerates local user account data on a domain controller.');
   Echo('   $02 Enumerates global user account data on a computer.');
   Echo('   $04 Enumerates proxy account data.');
   Echo('   $08 Enumerates domain trust account data on a domain controller.');
   Echo('   $10 Enumerates workstation or member server account data on a domain controller.');
   Echo('   $20 Enumerates domain controller account data on a domain controller.');
   Echo('  hosts filters:');
   Echo('   $00000001 	All LAN Manager workstations.');
   Echo('   $00000002 	All LAN Manager servers.');
   Echo('   $00000004	Any server running with Microsoft SQL Server.');
   Echo('   $00000008	Primary domain controller.');
   Echo('   $00000010	Backup domain controller.');
   Echo('   $00000020	Server running the Timesource service.');
   Echo('   $00000040	Apple File Protocol servers.');
   Echo('   $00000080	Novell servers.');
   Echo('   $00000100	LAN Manager 2.x Domain Member.');
   Echo('   $40000000	Servers maintained by the browser.');
   Echo('   $00000200	Server sharing print queue.');
   Echo('   $00000400	Server running dial-in service.');
   Echo('   $00000800	Xenix server.');
   Echo('   $00004000	Microsoft File and Print for Netware.');
   Echo('   $00001000	Windows NT (either Workstation or Server).');
   Echo('   $00002000	Server running Windows for Workgroups.');
   Echo('   $00008000	Windows NT Non-DC server.');
   Echo('   $00010000	Server that can run the Browser service.');
   Echo('   $00020000	Server running a Browser service as backup.');
   Echo('   $00040000	Server running the master Browser service.');
   Echo('   $00080000	Server running the domain master Browser.');
   Echo('   $80000000	Primary Domain.');
   Echo('   $00400000	Windows 95 or later.');
   Echo('   $FFFFFFFF	All servers.');
   Echo('  defaults are:');
   Echo('   @list users   . 0 2 10000');
   Echo('   @list hosts   . + 0 -1 10000');
   Echo('   @list domains . 0 10000');
   Echo(' @list windows p c t   - list all windows with pid p, class "c", title "t"');
   Echo(' @list colors      - print list of known colors (names only)');
   Echo(' @list colors pas  - print list of known colors (pas format)');
   Echo(' @list colors crc  - print list of known colors (crc format)');
   Echo(' @list colors htm  - print list of known colors (htm format)');
   Echo(' @list OleDbProviderNames - list installed OLEDB/ADO provider names');
   Echo(' @list OdbcDriverNames    - list installed ODBC driver names');
   Echo(' @list groups             - list user membership groups');
   Echo(' @list groups g           - user membership for group g');
   Echo(' @list environs           - list environment variables');
   Echo(' @list environs e         - get environment variable e');
   Echo(' @list ftypes             - list ftypes/mimes');
   Echo(' @list ftypes f           - get ftype/mime by name');
   Echo(' @list assocs             - list assoc (file ext to ftypes/mimes)');
   Echo(' @list assocs e           - get ftype/mime by file extension e');
  end;
  TopicText:='';
 except
  on E:Exception do BugReport(E,ee,'act_list');
 end;
end;

function act_sysinfo(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1,tail,ver:LongString; valid,detail:Boolean;
begin
 Result:=0;
 if IsMainThread then
 try
  valid:=false;
  detail:=false;
  arg:=Trim(args);
  if IsNonEmptyStr(arg) then begin
   w1:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(w1) then begin // Options started from -
    if IsOption(w1,'--help') then begin
     Echo('Syntax:');
     Echo(' @SysInfo                   - get brief system information');
     Echo(' @SysInfo --help            - get this help');
     Echo(' @SysInfo --file-version    - get the program version information');
     Echo(' @SysInfo --file-version f  - get "f" program version information');
     Echo(' @SysInfo --detail          - get more detail system information');
     Result:=1;
     Exit;
    end;
    if IsOption(w1,'--file-version') then begin
     tail:=Trim(SkipWords(1,arg,ScanSpaces));
     if IsEmptyStr(tail) then tail:=ParamStr(0) else tail:=SmartFileRef(tail);
     if not FileExists(tail) then RAISE ESystemConsole.CreateFmt('File not found %s',[SingleQuotedStr(tail)]);
     ver:=GetFileVersionInfoAsText(tail);
     if (ver<>'')
     then Echo(Format('File Version Info on %s:%s%s',[SingleQuotedStr(tail),EOL,ver]))
     else Echo(Format('Could not get file version info on %s',[SingleQuotedStr(tail)]));
     Result:=Length(ver);
     Exit;
    end;
    if IsOption(w1,'--detail') then begin
     detail:=true;
     valid:=true;
    end;
   end;
   if not valid then begin
    Echo('Syntax error. Use @SysInfo --help for help.');
    Exit;
   end;
  end;
  if not detail
  then EchoBriefSystemInfo
  else EchoBriefSystemInfoEx(true);
  Result:=1;
 except
  on E:Exception do BugReport(E,ee,'act_sysinfo');
 end;
end;

function act_polling(ee:TExpressionEvaluator; const args:LongString):double;
const
 s1 : array[Boolean] of PChar=('LinY','LogY');
 s2 : array[Boolean] of PChar=('','Lg');
 First : Boolean = True;
var
 f:Text; c:TCurve; x,y:Double; i,j,td:Integer; lg:Boolean; p:TPolling;
 fn,w1,w2,w3,w4,arg:LongString; wt:TFormTextEditor; wc:TFormCurveWindow;
 tp:TThreadPriority;
begin
 Result:=0;
 if IsMainThread then
 try
  wc:=nil;
  wt:=nil;
  lg:=false;
  arg:=ee.SmartArgs(args);
  w1:=ExtractWord(1,arg,ScanSpaces);
  w2:=ExtractWord(2,arg,ScanSpaces);
  w3:=ExtractWord(3,arg,ScanSpaces);
  w4:=ExtractWord(4,arg,ScanSpaces);
  fn:=SessionManager.VarTmpFile('polling.log');
  // @polling monitor
  if IsSameText(w1,'Monitor') then begin
   if IsNonEmptyStr(w2) then ResMonStatFlags:=iValDef(w2,ResMonStatDefault);
   OpenResourceMonitorConsole(false);
   Result:=ResMonStatFlags;
   Exit;
  end;
  // @polling list
  if IsSameText(w1,'List') then begin
   for i:=0 to FullPollingList.Count-1 do begin
    p:=TPolling(FullPollingList[i]);
    if TObject(p) is TPolling
    then Echo(Format(' %-32s = %4d, %s',[p.Name,p.Delay,GetPriorityName(p.Priority)]));
    Result:=Result+1;
   end;
   Exit;
  end;
  // @polling clear
  if IsSameText(w1,'Clear') then begin
   for i:=0 to FullPollingList.Count-1 do begin
    p:=TPolling(FullPollingList[i]);
    if TObject(p) is TPolling then
    if IsSameText(w2,'All') or IsSameText(w2,p.Name) then begin
     ActEcho(' Clear '+p.Name);
     Result:=Result+1;
     p.HistClear;
    end;
   end;
   if IsSameText(w2,'All') then First:=true;
   if IsSameText(w2,'Log') then First:=true;
   Exit;
  end;
  // @polling thread name [delay priority]
  if IsSameText(w1,'Thread') then begin
   for i:=0 to FullPollingList.Count-1 do begin
    p:=TPolling(FullPollingList[i]);
    if (TObject(p) is TPolling) then
    if IsSameText(w2,p.Name) then begin
     td:=StrToIntDef(w3,0);
     tp:=GetPriorityByName(w4,p.Priority);
     if InRange(td,1,1000) then p.Delay:=td;
     if (tp<>p.Priority) then p.Priority:=tp;
     Echo(Format(' %s = %d, %s',[p.Name,p.Delay,GetPriorityName(p.Priority)]));
     Result:=Result+1;
    end;
   end;
   Exit;
  end;
  // @polling priorityclass class period
  if IsSameText(w1,'PriorityClass') then begin
   if Length(w2)>0
   then ForcePriorityClass(GetPriorityClassByName(w2),StrToIntDef(w3,PeriodPriorityClass));
   i:=ProcessPriorityToClass(GetProcessPriority(GetCurrentProcessId));
   ActEcho(Format(' Process priority class is %s (%d).',[GetPriorityClassName(i),GetPriorityClassLevel(i)]));
   Result:=GetPriorityClassLevel(i);
   Exit;
  end;
  // @polling plot
  if IsSameText(w1,'Plot') then begin
   lg:=IsSameText(w2,'LogY');
   wc:=NewCurveWindow('Thread polling histograms ('+s1[lg]+' scale)',
                    ^C'Thread call count vs poll period'+ASCII_CR+^L'  '+s2[lg]+'[Call Count]',
                    ^R'Poll period,[ms]->  '+ASCII_CR+^C'Thread polling histogram.');
  end;
  // @polling text
  if IsSameText(w1,'Text') then begin
   lg:=IsSameText(w2,'LogY');
   if First or not FileExists(fn) then begin
    System.Assign(f,fn);
    SetInOutRes(0);
    try
     System.Rewrite(f);
     System.Writeln(f,'Thread polling log file.');
     System.Writeln(f,'************************');
     First:=false;
    finally
     System.Close(f);
     SetInOutRes(0);
    end;
   end;
   wt:=FindTextEditor(fn,true,true);
  end;
  // Fill histograms
  if wc.Ok or wt.Ok then begin
   if wt.Ok then wt.Editor.Lines.Add(EOL+'Thread polling log at '+StdDateTimeStr(msecnow)+':');
   for i:=0 to FullPollingList.Count-1 do begin
    p:=TPolling(FullPollingList[i]);
    if TObject(p) is TPolling then begin
     c:=nil;
     if wc.Ok then ActEcho(' Plot '+p.Name);
     if wt.Ok then ActEcho(' Text '+p.Name);
     if wc.Ok then c:=NewCurve(0,p.Name,clBlack,$1F);
     if wt.Ok then wt.Editor.Lines.Add(p.Name+' polling histogram ('+s1[lg]+' scale):');
     if wt.Ok then wt.Editor.Lines.Add(Format(' DevicePolling = %d, %s',[p.Delay,GetPriorityName(p.Priority)]));
     for j:=0 to p.HistSize-1 do begin
      x:=j*p.HistStep;
      y:=p.HistData[j];
      if wc.Ok then begin
       if lg
       then c.AddPoint(x,Log10(Max(y,0.1)))
       else c.AddPoint(x,y);
      end;
      if wt.Ok and (y>0) then begin
       if lg
       then wt.Editor.Lines.Add(Format(' %-5.0f %5.2f',[x,Log10(Max(y,0.1))]))
       else wt.Editor.Lines.Add(Format(' %-5.0f %1.0f',[x,y]));
      end;
     end;
     if wc.Ok then wc.AddCurve(c);
     if wt.Ok then wt.FileSave;
     Result:=Result+1;
    end;
   end;
  end else begin
   Echo('Syntax:');
   Echo(' @polling list       - list thread polling histograms');
   Echo(' @polling clear xxx  - clear xxx thread polling histogram');
   Echo(' @polling clear all  - clear all thread polling histograms');
   Echo(' @polling clear log  - clear thread polling histograms log file');
   Echo(' @polling plot liny  - plot thread polling histograms in lin scale');
   Echo(' @polling plot logy  - plot thread polling histograms in log scale');
   Echo(' @polling text liny  - show thread polling histograms as text in lin scale');
   Echo(' @polling text logy  - show thread polling histograms as text in log scale');
   Echo(' @polling PriorityClass     - show process priority class');
   Echo(' @polling PriorityClass c p - force process priority class');
   Echo('  c is (Idle,Lower,Normal,Higher,High,RealTime)=(4,6,8,10,13,24)');
   Echo('  p is 0 or period to check and force process priority class, ms');
   Echo(' @polling thread n [t p] - query polling thread name (n) time & priority');
   Echo('                           optional (t,p) uses to assign time & priority');
   Echo(' @polling monitor m  - open RESOURCE MONITOR CONSOLE window');
   Echo('                     - to view polling with mode m bit flags:');
   Echo('                     - 1:objects,2:memory,4:handles,8:threads,');
   Echo('                     - *:defaults, -1:all.');
   Echo('Examples:');
   Echo(' @polling list');
   Echo(' @polling clear all');
   Echo(' @polling plot logy');
   Echo(' @polling monitor -1');
   Echo(' @polling PriorityClass 8 1000');
   Echo(' @polling thread System.Dispatcher');
   Echo(' @polling thread System.Sound 10 tpNormal');
  end;
 except
  on E:Exception do BugReport(E,ee,'act_polling');
 end;
end;

function act_adam(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1,w2:LongString;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=ee.SmartArgs(args);
  w1:=ExtractWord(1,arg,ScanSpaces);
  w2:=ExtractWord(2,arg,ScanSpaces);
  // @adam Console
  if IsSameText(w1,'Console') then begin
   OpenAdamConsole;
   Result:=1;
   Exit;
  end;
  // @adam DebugMode
  // @adam DebugMode n
  if IsSameText(w1,'DebugMode') then begin
   if (w2<>'') then AdamParam.DebugMode:=StrToIntDef(w2,0);
   ActEcho(Format(' AdamDebugMode = %d ($%x).',[AdamParam.DebugMode,AdamParam.DebugMode]));
   Result:=AdamParam.DebugMode;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @adam Console     - open ADAM DEBUG CONSOLE window');
  Echo(' @adam DebugMode   - get debug mode');
  Echo(' @adam DebugMode n - set debug mode');
  Echo('   DebugMode Bit 0 - full I/O protocol');
  Echo('   DebugMode Bit 1 - warning on TimeOut');
  Echo('   DebugMode Bit 2 - warning on format error');
  Echo('   DebugMode Bit 3 - show time');
  Echo(' DO NOT FORGET "@adam DebugMode 0" TO SWITCH OFF DEBUG!');
  Echo('Examples:');
  Echo(' @adam debugmode 15');
  Echo(' @adam console');
  Echo(' @adam debugmode 0');
 except
  on E:Exception do BugReport(E,ee,'act_adam');
 end;
end;

function act_pid(ee:TExpressionEvaluator; const args:LongString):double;
var i,n,pid,cod,chn,nl:Integer; list:TRunningProcessList;
var arg,w1,w2,w3,w4,ps:LongString;
 procedure Head(s:LongString);
 begin
  if (s='') then Exit;
  Echo(s+EOL+StringOfChar('#',Length(s)));
 end;
begin
 Result:=0;
 try
  arg:=ee.SmartArgs(args);
  w1:=ExtractWord(1,arg,ScanSpaces);
  w2:=ExtractWord(2,arg,ScanSpaces);
  w3:=ExtractWord(3,arg,ScanSpaces);
  w4:=ExtractWord(4,arg,ScanSpaces);
  ps:='';
  // @pid Self
  // @pid Current
  if IsSameText(w1,'Self')
  or IsSameText(w1,'Curr')
  or IsSameText(w1,'Current') then begin
   Result:=GetCurrentProcessId;
   Exit;
  end;
  // @pid Parent
  if IsSameText(w1,'Parent') then begin
   Result:=GetParentProcessId;
   Exit;
  end;
  // @pid List
  if IsSameText(w1,'List') then begin
   case StrToIntDef(w2,4) of
    0:   ps:=GetListOfProcesses(0,0,'');
    1:   ps:=GetListOfProcesses(0,0,'',true,glops_CmdLine or glops_FixName);
    2:   ps:=GetListOfProcesses(0,0,'',true,glops_Threads or glops_FixName);
    3:   ps:=GetListOfProcesses(0,0,'',true,glops_CmdLine or glops_Threads or glops_FixName);
    else ps:='';
   end;
   if IsNonEmptyStr(ps) then begin
    Result:=GetTextNumLines(ps);
    Echo(ValidateEOL(ps,-1));
   end else begin
    list:=NewRunningProcessList(glops_CmdLine or glops_FixName);
    try
     nl:=15; for i:=0 to list.Count-1 do  nl:=Max(nl,Length(list.TaskName[i]));
     Echo(Format(' %-11s  %-11s  %8s  %-*s  %-7s  %s',
                ['Pid','ParentPid','Priority',nl,'Name','Threads','CmdLine']));
     with list do
     for i:=0 to Count-1 do
     Echo(Format(' %-11d  %-11d  %-8d  %-*s  %-7d  %s',
                [Pid[i],ParentPid[i],Priority[i],nl,TaskName[i],Threads[i],CmdLine[i]]));
     Result:=List.Count;
    finally
     Kill(list);
    end;
   end;
   Exit;
  end;
  // @pid find exe
  if IsSameText(w1,'Find') then begin
   if (w2='') then Exit;
   list:=NewRunningProcessList(glops_FixName);
   try
    with list do
    for i:=0 to Count-1 do if (Pid[i]<>0) then begin
     if IsSameText(w2,TaskName[i]) or (StrToIntDef(w2,0)=Integer(Pid[i])) then Result:=Pid[i];
     if (Result<>0) then break;
    end;
    if (Result=0)
    then ActEcho(Format('Could not find process %s',[w2]))
    else ActEcho(Format('Found process %s PID %g',[w2,Result]));
   finally
    Kill(list);
   end;
   Exit;
  end;
  // @pid enum exe
  if IsSameText(w1,'Enum') then begin
   if (w2='') then Exit;
   list:=NewRunningProcessList(glops_FixName);
   try
    with list do
    for i:=0 to Count-1 do if (Pid[i]<>0) then begin
     if IsSameText(w2,TaskName[i]) or (StrToIntDef(w2,0)=Integer(Pid[i])) then begin
      ActEcho(Format('Found process %s PID %d',[w2,Pid[i]]));
      Result:=Result+1;
     end;
    end;
    if (Result=0) then ActEcho(Format('Could not find process %s',[w2]));
   finally
    Kill(list);
   end;
   Exit;
  end;
  // @pid Kill pid exitcode childs
  // @pid Kill file exitcode childs
  if IsSameText(w1,'Kill') then begin
   n:=0;
   if Length(w2)>0 then begin
    pid:=StrToIntDef(w2,0);
    cod:=StrToIntDef(w3,0);
    chn:=StrToIntDef(w4,0);
    if pid<>0 then Inc(n,KillProcessTree(pid,cod,chn)) else begin
     list:=NewRunningProcessList(glops_FixName);
     try
      with list do
      for i:=0 to Count-1 do
      if IsSameText(w2,TaskName[i])
      then Inc(n,KillProcessTree(Pid[i],cod,chn));
     finally
      Kill(list);
     end;
    end;
   end;
   ActEcho(Format(' %d pid(s) killed.',[n]));
   Result:=n;
   Exit;
  end;
  // @pid Prio tabs
  if IsSameText(w1,'Prio') then begin
   if IsSameText(w2,'tabs') then begin
    Head('Linux NiceToProcessPriority:'); ps:=''; n:=0;
    for i:=MIN_NICE to MAX_NICE do begin
     ps:=ps+Format('%3d %-8s  ',[i,ProcessPriorityToString(NiceToProcessPriority(i))]);
     if ((n mod 10)=9) then begin Echo(ps); ps:=''; end;
     inc(n);
    end;
    if (ps<>'') then Echo(ps+EOL);
    Head('Windows Level To ProcessPriority:'); ps:=''; n:=0;
    for i:=0 to 31 do begin
     ps:=ps+Format('%2d %-8s  ',[i,ProcessPriorityToString(LevelToProcessPriority(i))]);
     if ((n mod 8)=7) then begin Echo(ps); ps:=''; end;
     inc(n);
    end;
    if (ps<>'') then Echo(ps+EOL);
    Head('Windows Process/ThreadPriority To Level:');
    Echo(PriorityTableAsText(WindowsPriorityToLevel));
    Head('Linux SchedOther Process/ThreadPriority To RLimit:');
    Echo(PriorityTableAsText(SchedOtherPriorityToRLimit));
    Head('Linux SchedOther Process/ThreadPriority To Nice:');
    Echo(PriorityTableAsText(SchedOtherPriorityToNice));
    Result:=1;
    Exit;
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @pid self       - return current process ID');
  Echo(' @pid current    - return current process ID');
  Echo(' @pid parent     - return parent  process ID');
  Echo(' @pid list m     - print list of running processes, mode m=0..4');
  Echo(' @pid find p     - find (first) process with name/pid p, return pid');
  Echo(' @pid enum p     - enumerate processes with name/pid p, return count');
  Echo(' @pid kill p e c - kill process (and maybe his childs)');
  Echo('  p - may be a number, Process Id, pid');
  Echo('  p - may be *.exe file name of process to kill');
  Echo('  e - terminated process exit code, 0 by default');
  Echo('  c - num. of child processes levels you wants to kill, 0 by default');
  Echo(' BE CAREFUL! YOU MAY DAMAGE SYSTEM!');
  Echo(' @pid prio tabs  - print process/thread priority tables');
  Echo('Examples:');
  Echo(' @pid list');
  Echo(' @pid find cmd.exe');
  Echo(' @pid enum cmd exe');
  Echo(' @pid kill 123 1 0');
  Echo(' @pid kill dns.exe');
 except
  on E:Exception do BugReport(E,ee,'act_pid');
 end;
end;

procedure SockReporter(Pipe:TSocketPipe; When:Double; What:PChar; Code:Integer); forward;
procedure PipeReporter(Pipe:TPipe; When:Double; const What:LongString; Code:Integer); forward;

function GetTermOptStr(opt:Integer):LongString;
begin
 Result:='';
 if HasFlags(opt,to_Oem2AnsiRx+to_Ansi2OemTx) then Result:=Result+'a';
 if HasFlags(opt,to_AutoClose)                then Result:=Result+'c';
 if HasFlags(opt,to_CheckSum)                 then Result:=Result+'s';
 if HasFlags(opt,to_Display)                  then Result:=Result+'d';
 if HasFlags(opt,to_Verbose)                  then Result:=Result+'v';
 if HasFlags(opt,to_HexDump)                  then Result:=Result+'x';
 if HasFlags(opt,to_HexConv)                  then Result:=Result+'h';
 if HasFlags(opt,to_UrlConv)                  then Result:=Result+'u';
 if Length(Result)>0 then Result:=' -'+Result;
end;

function GetTermOptInt(const opt:LongString; def:Integer=0):Integer;
begin
 Result:=def;
 if Assigned(StrIPos(PChar(opt),'a')) then Result:=Result or to_Oem2AnsiRx;
 if Assigned(StrIPos(PChar(opt),'a')) then Result:=Result or to_Ansi2OemTx;
 if Assigned(StrIPos(PChar(opt),'c')) then Result:=Result or to_AutoClose;
 if Assigned(StrIPos(PChar(opt),'s')) then Result:=Result or to_CheckSum;
 if Assigned(StrIPos(PChar(opt),'d')) then Result:=Result or to_Display;
 if Assigned(StrIPos(PChar(opt),'v')) then Result:=Result or to_Verbose;
 if Assigned(StrIPos(PChar(opt),'x')) then Result:=Result or to_HexDump;
 if Assigned(StrIPos(PChar(opt),'h')) then Result:=Result or to_HexConv;
 if Assigned(StrIPos(PChar(opt),'u')) then Result:=Result or to_UrlConv;
end;

function CrwDaqSysPath:LongString;
begin
 Result:=SysUtils.Trim(GetEnv('CRW_DAQ_SYS_PATH'));
 if (Result<>'') and (StrFetch(Result,Length(Result))<>PathSep) then Result:=Result+PathSep;
 Result:=Result+Trim(GetEnv('PATH'));
end;

 // @run kate
 // @run -con bash
 // @run cmd
function act_run(ee:TExpressionEvaluator; const args:LongString):double;
var win:TFormTermWindow; tid,trm,pip,swm,p,npr,opt_con:Integer;
var cmd,exe,arg,par,dir,pat,app,opt:LongString;
var isopt,opt_wsh,opt_expand:Boolean;
 procedure SetEnvVar(const aName,aValue:LongString);
 begin
  if SetEnv(aName,aValue)
  then DebugOut(stdfDebug,Format('%s=%s',[aName,GetEnv(aName)]));
 end;
begin
 Result:=0;
 try
  dir:=HomeDir; app:=''; trm:=0;
  pat:=StringReplace(DefaultPathVarStr,';','+',[rfReplaceAll]);
  arg:=ee.SmartArgs(args,1);
  if IsNonEmptyStr(arg) then begin
   swm:=1;
   pip:=0;
   opt_wsh:=false; opt_con:=0; npr:=1;
   p:=GetPriorityClassLevel(GetPriorityClassByName('Normal'));
   cmd:=Trim(arg);
   isopt:=true; opt_expand:=false;
   while IsOption(cmd) and isopt do begin
    opt:=ExtractFirstParam(cmd);
    if IsOption(opt,'-hide') then swm:=0;
    if IsOption(opt,'-sw0')  then swm:=0;
    if IsOption(opt,'-sw1')  then swm:=1;
    if IsOption(opt,'-sw2')  then swm:=2;
    if IsOption(opt,'-sw3')  then swm:=3;
    if IsOption(opt,'-sw4')  then swm:=4;
    if IsOption(opt,'-sw5')  then swm:=5;
    if IsOption(opt,'-sw6')  then swm:=6;
    if IsOption(opt,'-sw7')  then swm:=7;
    if IsOption(opt,'-sw8')  then swm:=8;
    if IsOption(opt,'-sw9')  then swm:=9;
    if IsOption(opt,'-sw10') then swm:=10;
    if IsOption(opt,'-ini')  then SetEnvVar('CRW_DAQ_SYS_PATH',CRW_DAQ_SYS_PATH);
    if IsOption(opt,'-idle')
    or IsOption(opt,'-lower')
    or IsOption(opt,'-normal')
    or IsOption(opt,'-high')
    or IsOption(opt,'-realtime')
    then p:=GetPriorityClassLevel(GetPriorityClassByName(Copy(opt,2,255)));
    if IsOption(opt,'-cd','-homedir') then begin
     cmd:=SkipFirstParam(cmd);
     opt:=ExtractFirstParam(cmd);
     dir:=opt;
     if MaybeEnvStr(dir) then dir:=ExpEnv(dir);
     if MaybeTildeStr(dir) then dir:=SmartFileRef(dir);
    end;
    if IsOption(opt,'-app','-getapppath') then begin
     cmd:=SkipFirstParam(cmd);
     opt:=ExtractFirstParam(cmd);
     app:=GetAppPath(opt,false,false,nil,nil,nil,CrwDaqSysPath,'');
     if IsEmptyStr(app) or not FileExists(app) then app:='';
    end;
    if IsOption(opt,'-path') then begin
     cmd:=SkipFirstParam(cmd);
     opt:=ExtractFirstParam(cmd);
     pat:=opt;
    end;
    if IsOption(Copy(opt,1,6),'-term:') then begin
     trm:=GetTermOptInt(Copy(opt,7,Length(opt)-6),trm);
     pip:=1;
    end;
    if IsOption(opt,'-wsh') then begin
     opt_wsh:=true;
    end;
    if IsOption(opt,'-con','--con') or IsOption(opt,'-console','--console') then begin
     opt_con:=1;
    end;
    if IsOption(opt,'-e','--expand') or IsOption(opt,'-exp','-expand') then begin
     opt_expand:=true;
    end;
    if IsOption(opt,'--') then begin
     isopt:=false;
    end;
    cmd:=SkipFirstParam(cmd);
   end;
   par:='';
   if IsEmptyStr(cmd) then Exit;
   exe:=ExtractFirstParam(cmd);
   if (app<>'') then begin
    exe:=AnsiQuotedIfNeed(app)+' '+AnsiQuotedIfNeed(exe);
    npr:=2;
   end else begin
    if MaybeEnvStr(exe) then exe:=ExpEnv(exe);
    if (WordIndex(exe,'%ComSpec%,$SHELL',ScanSpaces)>0) then exe:=GetComSpec else
    if IsRelativePath(exe) then begin
     par:=Trim(SmartFileSearch(exe,URL_Decode(pat)));
     if IsNonEmptyStr(par) then exe:=AnsiQuotedIfNeed(par);
    end;
    if not FileExists(AnsiDequotedStr(exe,QuoteMark)) then
    if SysGlossary.ReadIniPath(SysIniFile,SectAtRun(1),'@'+exe,HomeDir,par,false)                          // alias like @unix
    or SysGlossary.ReadIniPath(SysIniFile,SectAtRun(1),'@'+DefaultExtension(exe,'.exe'),HomeDir,par,false) // alias like @unix.exe
    or SysGlossary.ReadIniPath(SysIniFile,SectAtRun(1),exe,HomeDir,par,false)                              // name  like unix
    or SysGlossary.ReadIniPath(SysIniFile,SectAtRun(1),DefaultExtension(exe,'.exe'),HomeDir,par,false)     // exe   like unix.exe
    then begin
     if IsRelativePath(par) then par:=AddBackSlash(HomeDir)+Trim(par);
     par:=Trim(FExpand(par));
     if IsNonEmptyStr(par) and FileExists(par) then par:=GetRealFilePathName(par) else par:='';
     if IsNonEmptyStr(par) then exe:=AnsiQuotedIfNeed(par);
    end;
    if IsNonEmptyStr(exe) and HasExtension(exe) then
    if not HasListedExtension(AnsiDequotedStr(exe,QuoteMark),GetEnv('PathExt')+PathSep+DefaultPathExt,PathSep) then begin
     if SysGlossary.ReadIniPath(SysIniFile,SectAtRun(1),ExtractFileExt(exe),HomeDir,par,false) and FileExists(par)
     then par:=GetRealFilePathName(Trim(par)) else par:='';
     if IsEmptyStr(par) then par:=Trim(GetExeByFile(AnsiDequotedStr(exe,QuoteMark)));
     if FileExists(par) and IsNonEmptyStr(par) then
     if not IsSameText(ExtractFileExt(AnsiDequotedStr(exe,QuoteMark)),
                       ExtractFileExt(AnsiDequotedStr(par,QuoteMark)))
     then begin
      exe:=AnsiQuotedIfNeed(par)+' '+AnsiQuotedIfNeed(exe);
      npr:=2;
     end;
    end;
   end;
   if (npr=1) then exe:=AnsiQuotedIfNeed(exe);
   cmd:=Trim(exe+' '+SkipFirstParam(cmd));
   if opt_expand then cmd:=ExpEnv(cmd);
   if pip=0 then begin
    if opt_wsh then begin
     Result:=WScriptShellRun(cmd,swm,false);
     if (Result<>-1)
     then ActEcho(Format(' Started command: %s',[cmd]))
     else Echo(Format(' Could not start: %s',[cmd]));
     Exit;
    end;
    tid:=task_init(cmd);
    if tid<>0 then
    try
     task_ctrl(tid,Format('Display=%d',[swm]));
     task_ctrl(tid,Format('HomeDir=%s',[dir]));
     task_ctrl(tid,Format('ProcessPriority=%d',[p]));
     if (opt_con<>0) then task_ctrl(tid,Format('ConsoleMode=%d',[opt_con]));
     if task_run(tid)
     then Echo(Format('Child PID %d started: %s',[task_pid(tid),cmd]))
     else Echo(Format('Could not execute: %s',[cmd]));
     Result:=task_pid(tid);
    finally
     task_free(tid);
    end;
   end else begin
    win:=nil;
    if HasFlags(trm,to_Verbose)
    then pip:=pipe_init('task '+cmd,PipeReporter,SockReporter)
    else pip:=pipe_init('task '+cmd);
    if (pip<>0) then begin
     pipe_ctrl(pip,Format('HomeDir=%s',[dir]));
     pipe_ctrl(pip,Format('ProcessPriority=%d',[p]));
     win:=NewConsolePipeWindow('@term'+GetTermOptStr(trm)+' task '+Trim(cmd),pip,trm);
     if pipe_run(pip)
     then Echo(Format('Child PID %d started: %s',[pipe_pid(pip),cmd]))
     else begin Echo(Format('Could not execute: %s',[cmd])); Kill(win); end;
    end;
    if win.Ok
    then Result:=pipe_pid(pip)
    else Echo('Could not execute @term'+GetTermOptStr(trm)+' task '+cmd);
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @run [-opt] cmdline - run program');
  Echo('  cmdline       - program command line');
  Echo('  -opt          - options, one of next:');
  Echo('   -hide        - run hidden (Windows)');
  Echo('   -sw0..-sw10  - run with ShowWindow mode 0..10 (Windows)');
  Echo('   - 0=HIDE,1=NORMAL,3=MIN,4=SHOWNA,5=SHOW,6=MAX,7=MINNA,8=SHOWNA,9=RESTORE,10=DEFAULT');
  Echo('   -con         - run in ConsoleMode, i.e. in X terminal console (Unix)');
  Echo('   -ini         - (re)initialize CRW_DAQ_SYS_PATH environ by crwdaq.ini [@run] section');
  Echo('   -idle        - run with Idle     priority class 4');
  Echo('   -lower       - run with Low      priority class 6');
  Echo('   -normal      - run with Normal   priority class 8');
  Echo('   -high        - run with High     priority class 13');
  Echo('   -realtime    - run with RealTime priority class 24');
  Echo('   -cd dir      - run with start directory dir, default dir='+dir);
  Echo('   -homedir dir - run with start directory dir, default dir='+dir);
  Echo('                  ~ or ~~ means user or program homedir');
  Echo('   -path pat    - set search path URLDecode(pat), default pat='+pat);
  Echo('   -wsh         - try to run with WScript.Shell.Run COM object');
  Echo('   -term:opt    - run with StdIn,StdOut redirected to terminal with options opt, see @term');
  Echo('   -app app doc - open document (doc) with application (app) specified by (full) exe filename,');
  Echo('                  registered EXE name, extension or filetype, like "firefox.exe .htm htmlfile"');
  Echo('   -e cmdline   - expand cmdline with environment variables before run');
  Echo('                  also -exp,-expand,--expand synonyms uses');
  Echo('Examples:');
  Echo(' @run did.exe                   -- run did.exe');
  Echo(' @run -hide dns.exe             -- run dns.exe in hidden mode');
  Echo(' @run -sw7  cmd.exe             -- run cmd.exe in minimized no active mode');
  Echo(' @run -idle cmd.exe             -- run cmd.exe with Idle priority');
  Echo(' @run -cd ~~ cmd.exe            -- run cmd.exe at program homedir');
  Echo(' @run -cd c:\temp cmd.exe       -- run cmd.exe at c:\temp folder');
  Echo(' @run -cd %temp% cmd.exe        -- run cmd.exe at temporary %temp% folder');
  Echo(' @run %comspec% /c notepad.exe  -- run Notepad via cmd.exe');
  Echo(' @run -term:a cmd /c dir c:\    -- run DIR command in terminal window');
  Echo(' @run -term:a wo.bat x*.exe     -- run "wo x*.exe" for file search in terminal window');
  Echo(' @run -app "firefox.exe .htm" crw-daq.htm -- View help on CRW-DAQ with Firefox prefer');
  Echo(' @run crw-daq-tools.htm         -- View help on tools library');
  Echo(' @run crw-daq.htm               -- View help on CRW-DAQ');
  Echo(' @run -expand lister %CRW_DAQ_SYS_HOME_DIR%\CrwDaq.ini');
 except
  on E:Exception do BugReport(E,ee,'act_run');
 end;
end;

 /////////////////////////////////////////////////////////////////////////////////////////
 // @GetAppPath firefox.exe chrome.exe -t htmlfile .html --run Resource\Manual\crw-daq.htm
 /////////////////////////////////////////////////////////////////////////////////////////
function act_getapppath(ee:TExpressionEvaluator; const args:LongString):double;
var ExitCode:Integer; arg,cmd:LongString;
begin
 Result:=0;
 try
  ExitCode:=0;
  arg:=ee.SmartArgs(args,1);
  if Length(arg)>0 then begin
   cmd:=GetAppPath(args,false,false,@ExitCode,SystemEchoProcedure,SystemEchoProcedure,CrwDaqSysPath,'');
   if (cmd<>'') and (WordIndex('--run',arg,JustSpaces)+WordIndex('--start',arg,JustSpaces)>0)
   then SendToMainConsole('@silent @run '+cmd+EOL);
   Result:=ExitCode;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @getapppath [-opt] [args] [--run|--start] [params]');
  Echo(' call @getapppath --help for details.');
 except
  on E:Exception do BugReport(E,ee,'act_getapppath');
 end;
end;

function FilterLines(Lines,Pattern:LongString):LongString;
var List:TStringList; i:Integer;
begin
 Result:=Lines;
 Pattern:=Trim(Pattern);
 if IsEmptyStr(Pattern) then Exit;
 List:=TStringList.Create;
 try
  List.Text:=Lines;
  for i:=List.Count-1 downto 0 do begin
   if (PosI(Pattern,List.Strings[i])=0)
   then List.Delete(i);
  end;
  Result:=List.Text;
 finally
  Kill(List);
 end;
end;

 ///////////////////////////////////////////////////////////////////////////////
 // @env X=Y
 ///////////////////////////////////////////////////////////////////////////////
function act_env(ee:TExpressionEvaluator; const args:LongString):double;
var env,val,arg,opt:LongString; p,c:Integer;
begin
 Result:=0;
 try
  arg:=ee.SmartArgs(args,1);
  if IsNonEmptyStr(arg) then begin
   opt:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(opt) then begin
    if IsOption(opt,'-c') then begin
     env:=ExtractWord(2,arg,ScanSpaces);
     if (env<>'') then begin
      val:=GetEnv(env);
      if (val<>'') then Result:=1;
     end;
     Exit;
    end else
    if IsOption(opt,'-v') then begin
     env:=ExtractWord(2,arg,ScanSpaces);
     if (env<>'') then begin
      val:=ValidatePathList(GetEnv(env));
      if SetEnv(env,val) then Result:=Length(val);
     end;
     Exit;
    end else
    if IsOption(opt,'-t') then begin
     env:=ExtractWord(2,arg,ScanSpaces);
     if (env<>'') then begin
      val:=Trim(SkipWords(2,arg,ScanSpaces));
      if (val<>'') then val:=SmartFileRef(val);
      if (val<>'') then if Pos(AnsiUpperCase(PathSep+val+PathSep),AnsiUpperCase(PathSep+GetEnv(env)+PathSep))>0 then val:='';
      if (val<>'') then if DirExists(val) then begin
       val:=ValidatePathList(GetEnv(env)+PathSep+val);
       if SetEnv(env,val) then Result:=Length(val);
      end;
     end;
     Exit;
    end else
    if IsOption(opt,'-h') then begin
     env:=ExtractWord(2,arg,ScanSpaces);
     if (env<>'') then begin
      val:=Trim(SkipWords(2,arg,ScanSpaces));
      if (val<>'') then val:=SmartFileRef(val);
      if (val<>'') then if Pos(AnsiUpperCase(PathSep+val+PathSep),AnsiUpperCase(PathSep+GetEnv(env)+PathSep))>0 then val:='';
      if (val<>'') then if DirExists(val) then begin
       val:=ValidatePathList(val+PathSep+GetEnv(env));
       if SetEnv(env,val) then Result:=Length(val);
      end;
     end;
     Exit;
    end else
    if IsOption(opt,'-d') then begin
     env:=ExtractWord(2,arg,ScanSpaces);
     if (env<>'') then begin
      val:=Trim(SkipWords(2,arg,ScanSpaces));
      if (val<>'') then val:=SmartFileRef(val);
      if (val<>'') then if Pos(AnsiUpperCase(PathSep+val+PathSep),AnsiUpperCase(PathSep+GetEnv(env)+PathSep))=0 then val:='';
      if (val<>'') then if DirExists(val) then begin
       val:=ValidatePathList(StringReplace(PathSep+GetEnv(env)+PathSep,PathSep+val+PathSep,PathSep,[rfIgnoreCase,rfReplaceAll]));
       if SetEnv(env,val) then Result:=Length(val);
      end;
     end;
     Exit;
    end else
    if IsOption(opt,'-l') or IsOption(opt,'-list','--list') then begin
     env:=ExtractWord(2,arg,ScanSpaces);
     val:=EnvironmentVariableList.Text;
     val:=SortTextLines(val);
     val:=FilterLines(val,env);
     Echo(val);
     Exit;
    end else
    if IsOption(opt,'-app') or IsOption(opt,'-getapppath') then begin
     env:=ExtractWord(2,arg,ScanSpaces);
     if (env<>'') then begin
      val:=Trim(SkipWords(2,arg,ScanSpaces));
      if (val<>'') then val:=GetAppPath(val,false,false,@c,nil,nil,CrwDaqSysPath,'');
      if (val<>'') and (c=0) then begin
       if SetEnv(env,val) then Result:=Length(val);
      end;
     end;
     Exit;
    end;
    Exit;
   end;
   p:=ExtractNameValuePair(arg,env,val);
   if (p>0) then begin
    if MaybeEnvStr(val)
    then val:=ExpEnv(val);
   end;
   if (env<>'') then begin
    if (p>0) then SetEnv(env,val);
    ActEcho(Format(' %s=%s',[env,GetEnvVar(env)]));
    Result:=Length(GetEnvVar(env));
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @env name          - get   environment variable');
  Echo(' @env name=         - clear environment variable');
  Echo(' @env name=value    - set   environment variable');
  Echo(' @env -c name       - check environment variable name is exist');
  Echo(' @env -v PATH       - validates the PATH variable search list');
  Echo(' @env -t PATH dir   - add dir to tail of the PATH search list');
  Echo(' @env -h PATH dir   - add dir to head of the PATH search list');
  Echo(' @env -d PATH dir   - delete dir from the PATH search list');
  Echo(' @env -l            - list environment strings');
  Echo(' @env -app name=app - set name = application (app), see @GetAppPath');
  Echo(' Note: supports environment variables (e.g. %temp%).');
  Echo('Examples:');
  Echo(' @env -app WebBrowser=firefox.exe chrome.exe .html -t htmlfile');
  Echo(' @env -t PATH %CRW_DAQ_SYS_HOME_DIR%\resource\shared');
  Echo(' @env dim_dns_node=%ComputerName%');
  Echo(' @env dim_dns_node=main');
  Echo(' @env dim_dns_node=');
  Echo(' @env dim_dns_node');
 except
  on E:Exception do BugReport(E,ee,'act_env');
 end;
end;

function act_reg(ee:TExpressionEvaluator; const args:LongString):double;
var arg,opt,key,path,name,value:LongString; root:HKey;
begin
 Result:=0;
 try
  arg:=ee.SmartArgs(args,1);
  if IsNonEmptyStr(arg) then begin
   if IsOption(arg) then begin
    opt:=ExtractWord(1,arg,ScanSpaces);
    if IsOption(opt,'-r') then begin
     path:=url_decode(ExtractWord(2,arg,ScanSpaces));
     root:=GetRootKeyByName(ExtractWord(1,path,['\','/']));
     key:=SkipWords(1,path,['\','/']);
     name:=Trim(url_decode(ExtractWord(3,arg,ScanSpaces)));
     value:=ReadRegistryString(root,key,name);
     Result:=Length(value);
     if (Result>0)
     then ActEcho(AddBackSlash(path)+name+' = '+value)
     else ActEcho(RusEng('Ключ реестра не найден.','Registry key not found.'))
    end else
    if IsOption(opt,'-a') then begin
     name:=ExtractWord(2,arg,ScanSpaces);
     value:=GetSystemAssoc(name);
     Result:=Length(value);
     if (Result>0)
     then ActEcho(name+' = '+value)
     else ActEcho(RusEng('Ключ реестра не найден.','Registry key not found.'))
    end else
    if IsOption(opt,'-f') then begin
     name:=ExtractWord(2,arg,ScanSpaces);
     value:=GetSystemFType(name);
     Result:=Length(value);
     if (Result>0)
     then ActEcho(name+' = '+value)
     else ActEcho(RusEng('Ключ реестра не найден.','Registry key not found.'))
    end else
    if IsOption(opt,'-e') then begin
     name:=ExtractWord(2,arg,ScanSpaces); value:='';
     if HasSystemAssocExe(name) then value:=GetSystemAssocExe(name) else
     if HasSystemFTypeExe(name) then value:=GetSystemFTypeExe(name);
     Result:=Length(value);
     if (Result>0)
     then ActEcho(name+' = '+value)
     else ActEcho(RusEng('Ключ реестра не найден.','Registry key not found.'))
    end;
   end;
   Exit;
  end;
  Result:=Guard.Level;
  Echo('Syntax:');
  Echo(' @reg -r key name        - read registry key,name value (key,name is URL-encoded)');
  Echo(' @reg -f ftype           - read command line associated with file type id ftype');
  Echo(' @reg -a ext             - read file type id associated with file extension ext');
  Echo(' @reg -e ext             - read *.exe file associated with file extension ext');
  Echo('Examples:');
  Echo(' @reg -r HKLM\SOFTWARE\Microsoft\Windows+NT\CurrentVersion\Fonts PT+Mono+(TrueType)');
  Echo(' @reg -f VisualTech.DieselPascal.CrossMachine');
  Echo(' @reg -a .lm9');
  Echo(' @reg -e .htm');
 except
  on E:Exception do BugReport(E,ee,'act_reg');
 end;
end;

function act_guard(ee:TExpressionEvaluator; const args:LongString):double;
var i:Integer; arg:LongString;
begin
 Result:=0;
 try
  arg:=ee.SmartArgs(args);
  if Length(arg)>0 then begin
   if Str2Int(arg,i) then Guard.Level:=i else
   for i:=ga_Lock to ga_Root do
   if IsSameText(Guard.LevelName[i],Trim(arg)) then Guard.Level:=i;
  end;
  Result:=Guard.Level;
 except
  on E:Exception do BugReport(E,ee,'act_guard');
 end;
end;

function FullComponentName(aComponent:TComponent):LongString;
begin
 Result:='';
 try
  if Assigned(aComponent) then
  if Length(aComponent.Name)>0 then begin
   Result:=FullComponentName(aComponent.Owner);
   if Length(Result)>0 then Result:=Result+'.';
   Result:=Result+aComponent.Name;
  end;
 except
  on E:Exception do BugReport(E,SystemConsole,'FullComponentName');
 end;
end;

function MendComponentName(const Name:LongString):LongString;
const OldForm='FormCrw32'; NewForm='FormCrwDaq';
begin
 Result:=Name;
 if (PosI(OldForm,Result)>0)
 then Result:=StringReplace(Result,OldForm,NewForm,[rfReplaceAll,rfIgnoreCase]);
end;

procedure do_list_components(Component:TComponent; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 if not Assigned(Custom) then Exit;
 if not Assigned(Component) then Exit;
 if (Component is TComponent) and (Length(Component.Name)>0)
 then TStringList(Custom).AddObject(FullComponentName(Component),Component);
 if (Component is TForm) then ForEachComponent(Component,do_list_components,Custom);
end;

procedure do_list_forms(Component:TComponent; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 if not Assigned(Custom) then Exit;
 if not Assigned(Component) then Exit;
 if (Component is TForm) and (Length(Component.Name)>0)
 then TStringList(Custom).AddObject(FullComponentName(Component),Component);
end;

function act_menu(ee:TExpressionEvaluator; const args:LongString):double;
var List,Keys:TStringList; i:Integer; arg,w1,w2:LongString;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=ee.SmartArgs(args);
  w1:=ExtractWord(1,arg,ScanSpaces);
  w2:=ExtractWord(2,arg,ScanSpaces);
  if IsNonEmptyStr(arg) then begin
   List:=TStringList.Create; List.Duplicates:=dupIgnore; List.Sorted:=true;
   Keys:=TStringList.Create; Keys.Duplicates:=dupIgnore; Keys.Sorted:=true;
   try
    ForEachComponent(Application,do_list_components,List);
    if IsSameText('List',w1) then begin
     for i:=0 to List.Count-1 do
     if IsNonEmptyStr(List[i]) then
     if List.Objects[i] is TCustomAction then
     with (List.Objects[i] as TCustomAction) do begin
      if IsSameText('Keys',w2) then begin
       if (ShortCut<>0) and Assigned(Owner)
       then Keys.Add(Format(' %-15s - %s.%s - "%s"',[ShortCutToText(ShortCut),Owner.ClassName,Name,Hint]));
      end else begin
       if (ShortCut=0)
       then Echo(Format(' %s - "%s"',[List[i],Hint]))
       else Echo(Format(' %s - "%s" - "%s"',[List[i],Hint,ShortCutToText(ShortCut)]));
      end;
      Result:=Result+1;
     end;
     for i:=0 to Keys.Count-1 do Echo(Keys[i]);
     Exit;
    end;
    if IsSameText('Run',w1) then begin
     i:=List.IndexOf(MendComponentName(w2));
     if (i>=0) then begin
      if (List.Objects[i] is TCustomAction) then
      with (List.Objects[i] as TCustomAction) do
      if Enabled then Result:=ord(Execute);
     end;
     Exit;
    end;
   finally
    Kill(List);
    Kill(Keys);
   end;
  end;
  Echo('Syntax:');
  Echo(' @menu list      - get list of all menu items');
  Echo(' @menu list keys - get list of all hot keys');
  Echo(' @menu run n     - run menu item "n"');
  Echo('Examples:');
  Echo(' @menu list');
  Echo(' @menu run FormCrwDaq.ActionFileExit');
 except
  on E:Exception do BugReport(E,ee,'act_menu');
 end;
end;

 {
 @view show FormCrwDaq
 }
function act_view(ee:TExpressionEvaluator; const args:LongString):double;
var rcWork:TRect; MonInfo:TMonitorInfo; currMonitor,primMonitor:TMonitor;
var List:TStringList; arg,w1,w2,wn,wt,wp,bmp,fn,sn,s:LongString;
var isopt,opt_expand,opt_forms,opt_title,opt_xor:Boolean;
var ev:TExpressionEvaluator; aChild:TFormCrwDaqSysChild;
var i,n,ic,iv,l1,l2:Integer; MForm:TMasterForm;
var icObj:TObject;
 function Exp2Int(ev:TExpressionEvaluator; const st:LongString; var iv:Integer):Boolean;
 var buf:TParsingBuffer;
 begin
  if IsEmptyStr(st) or IsSameText('*',st) then Result:=false else
  Result:=(ev.EvaluateExpression(StrCopyBuff(buf,st))=ee_OK);
  if Result then if IsNanOrInf(ev.Answer) then Result:=false;
  if Result then iv:=Round(ev.Answer);
 end;
 function FindTarget(List:TStringList; const aName,aTitle:LongString):Integer;
 var i:Integer;
 begin
  Result:=-1;
  if Assigned(List) then begin
   if (aName<>'') then begin
    Result:=List.IndexOf(aName);
    if (Result>=0) then begin
     if opt_forms and not (List.Objects[Result] is TForm) then Result:=-1;
     if (Result>=0) then Exit;
    end;
   end;
   if (aTitle<>'') then begin
    for i:=0 to List.Count-1 do
    if (List.Objects[i] is TForm) then begin
     if IsSameText(aTitle,(List.Objects[i] as TForm).Caption) then begin
      Result:=i;
      Break;
     end;
    end;
   end;
  end;
 end;
 function MatchLine(const Line,Incl,Excl:LongString):Boolean;
 begin
  if IsEmptyStr(Line) then Exit(false);
  if (Excl<>'') and StartsText(Excl,Line) then Exit(false);
  if (Incl<>'') and not StartsText(Incl,Line) then Exit(false);
  Result:=true;
 end;
 function FilterLines(const Data,Incl,Excl:LongString):LongString;
 var Lines:TStringList; i:Integer; Line:LongString;
 begin
  Result:='';
  Lines:=TStringList.Create;
  if Assigned(Lines) then
  try
   Lines.Text:=Data;
   for i:=Lines.Count-1 downto 0 do begin
    Line:=Trim(Lines.Strings[i]);
    if MatchLine(Line,Incl,Excl) then begin
     if (Lines.Strings[i]<>Line)
     then Lines.Strings[i]:=Line;
    end else Lines.Delete(i);
   end;
   Result:=Lines.Text;
  finally
   Kill(Lines);
  end;
 end;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=ee.SmartArgs(args);
  {
  Check options: -e, -f, -t, --
  }
  icObj:=nil;
  aChild:=nil; MForm:=nil; iv:=0;
  w1:=''; w2:=''; bmp:=''; s:='';
  wn:=''; wt:=''; wp:=''; fn:='';
  isopt:=true; opt_expand:=false;
  ev:=nil; List:=nil; opt_xor:=false;
  opt_title:=false; opt_forms:=false;
  while IsOption(arg) and isopt do begin
   w1:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(w1,'-e')
   or IsOption(w1,'-exp','--exp')
   or IsOption(w1,'-expand','--expand') then begin
    opt_expand:=true;
   end else
   if IsOption(w1,'-x')
   or IsOption(w1,'-xor','--xor') then begin
    opt_xor:=true;
   end else
   if IsOption(w1,'-f')
   or IsOption(w1,'-form','--form')
   or IsOption(w1,'-forms','--forms') then begin
    opt_forms:=true;
   end else
   if IsOption(w1,'-t')
   or IsOption(w1,'-tit','--tit')
   or IsOption(w1,'-title','--title') then begin
    opt_title:=true;
    opt_forms:=true;
   end else
   if IsOption(w1,'--') then begin
    isopt:=false;
   end;
   arg:=SkipWords(1,arg,ScanSpaces);
  end;
  if opt_expand and MaybeEnvStr(arg) then arg:=ExpEnv(arg);
  {}
  w1:=ExtractWord(1,arg,ScanSpaces);
  w2:=ExtractWord(2,arg,ScanSpaces);
  {
  @view logo ...
  }
  if SameText(w1,'Logo') then begin
   if SystemConsole.ShouldBeDelayed then Exit;
   if SameText(w2,'show') then begin
    w2:=SkipWords(2,arg,JustSpaces); bmp:='';
    if IsOption(ExtractWord(1,w2,JustSpaces),'-bmp') then begin
     bmp:=DefaultExtension(ExtractWord(2,w2,JustSpaces),'.bmp');
     bmp:=AdaptFileName(DefaultPath(ExpEnv(bmp),HomeDir));
     w2:=SkipWords(2,w2,JustSpaces);
    end;
    if (bmp='') or not FileExists(bmp)
    then bmp:=AdaptFileName(FormCrwDaq.DesktopLogoFile);
    if IsOption(ExtractWord(1,w2,JustSpaces),'-daq') then begin
     w2:=ExtractFirstParam(SkipWords(1,w2,JustSpaces));
     if (w2<>'') then Daq.ShowLogo(w2);
    end else begin
     w1:='CRW-DAQ LOGO'; w2:=ExtractFirstParam(w2);
     if (w2<>'') then ShowFormCrwDaqLogo(w1,w2,bmp,true);
    end;
    Exit;
   end;
   if SameText(w2,'hide') then begin
    HideFormCrwDaqLogo;
    Exit;
   end;
  end;
  {
  @view screen
  }
  if IsSameText('Screen',w1) then begin
   ee.SetConst('screen_width',         Screen.Width);
   ee.SetConst('screen_height',        Screen.Height);
   ee.SetConst('screen_desktopleft',   Screen.DesktopLeft);
   ee.SetConst('screen_desktoptop',    Screen.DesktopTop);
   ee.SetConst('screen_desktopwidth',  Screen.DesktopWidth);
   ee.SetConst('screen_desktopheight', Screen.DesktopHeight);
   ee.SetConst('screen_pixelsperinch', Screen.PixelsPerInch);
   ee.SetConst('screen_monitorcount',  Screen.MonitorCount);
   currMonitor:=nil; primMonitor:=nil;
   for i:=0 to Screen.MonitorCount-1 do begin
    s:=Format('screen_monitor%dleft',  [i]);  ee.SetConst(PChar(s),Screen.Monitors[i].Left);
    s:=Format('screen_monitor%dtop',   [i]);  ee.SetConst(PChar(s),Screen.Monitors[i].Top);
    s:=Format('screen_monitor%dwidth', [i]);  ee.SetConst(PChar(s),Screen.Monitors[i].Width);
    s:=Format('screen_monitor%dheight',[i]);  ee.SetConst(PChar(s),Screen.Monitors[i].Height);
    MonInfo.cbSize:=sizeof(MonInfo);
    if GetMonitorInfo(Screen.Monitors[i].Handle,@MonInfo) then begin
     s:=Format('screen_monitor%dworkleft',  [i]);  ee.SetConst(PChar(s),MonInfo.rcWork.Left);
     s:=Format('screen_monitor%dworktop',   [i]);  ee.SetConst(PChar(s),MonInfo.rcWork.Top);
     s:=Format('screen_monitor%dworkwidth', [i]);  ee.SetConst(PChar(s),MonInfo.rcWork.Right-MonInfo.rcWork.Left);
     s:=Format('screen_monitor%dworkheight',[i]);  ee.SetConst(PChar(s),MonInfo.rcWork.Bottom-MonInfo.rcWork.Top);
     s:=Format('screen_monitor%dflags',     [i]);  ee.SetConst(PChar(s),MonInfo.dwFlags);
     if MonInfo.dwFlags and MONITORINFOF_PRIMARY <> 0 then primMonitor:=Screen.Monitors[i];
    end;
   end;
   if SystemParametersInfo(SPI_GETWORKAREA, 0, @rcWork, 0) then begin
    ee.SetConst('screen_workleft',   rcWork.Left);
    ee.SetConst('screen_worktop',    rcWork.Top);
    ee.SetConst('screen_workwidth',  rcWork.Right-rcWork.Left);
    ee.SetConst('screen_workheight', rcWork.Bottom-rcWork.Top);
   end;
   if Application.MainForm is TForm then currMonitor:=Application.MainForm.Monitor;
   if currMonitor is TMonitor then ee.SetConst('screen_monitornumber',currMonitor.MonitorNum);
   if primMonitor is TMonitor then ee.SetConst('screen_monitorprimary',primMonitor.MonitorNum);
   with GetAppFormBounds do begin
    ee.SetConst('screen_appformleft',Left);
    ee.SetConst('screen_appformtop',Top);
    ee.SetConst('screen_appformright',Right);
    ee.SetConst('screen_appformbottom',Bottom);
    ee.SetConst('screen_appformwidth',Right-Left);
    ee.SetConst('screen_appformheight',Bottom-Top);
   end;
   with GetAppClientBounds do begin
    ee.SetConst('screen_appclientleft',Left);
    ee.SetConst('screen_appclienttop',Top);
    ee.SetConst('screen_appclientright',Right);
    ee.SetConst('screen_appclientbottom',Bottom);
    ee.SetConst('screen_appclientwidth',Right-Left);
    ee.SetConst('screen_appclientheight',Bottom-Top);
   end;
   ee.SetConst('screen_colordepth',GetScreenColorDepth);
   ee.SetConst('systemdefaultuilanguage',GetSystemDefaultUILanguage);
   ee.SetConst('userdefaultuilanguage',GetUserDefaultUILanguage);
   ee.SetConst('threaduilanguage',GetThreadUILanguage);
   ee.SetConst('systemdefaultlcid',GetSystemDefaultLCID);
   ee.SetConst('userdefaultlcid',GetUserDefaultLCID);
   ee.SetConst('threadlocale',GetThreadLocale);
   ee.SetConst('oemcodepage',GetOEMCP);
   ee.SetConst('ansicodepage',GetACP);
   Result:=Screen.MonitorCount;
   Exit;
  end;
  {
  @view list
  @view show        n
  @view hide        n
  @view visible     n
  @view enable      n
  @view disable     n
  @view enabled     n
  @view min         n
  @view max         n
  @view norm        n
  @view WindowState n
  @view title       n t
  @view pos         n [x y w h]
  @view move        n [dx dy dw dh]
  @view ShowFront   n
  @view ShowBack    n
  @view Activate    n
  @view params      n @set ...
  @view params      n [section]
  @view params      n inifile [section]
  }
  if IsNonEmptyStr(arg) then begin
   List:=TStringList.Create; List.Duplicates:=dupIgnore; List.Sorted:=true;
   try
    if opt_forms
    then ForEachComponent(Application,do_list_forms,List)
    else ForEachComponent(Application,do_list_components,List);
    if IsSameText('List',w1) then begin
     l1:=1; l2:=1;
     for n:=1 to 2 do
     for ic:=0 to List.Count-1 do
     if IsNonEmptyStr(List[ic]) then begin
      icObj:=List.Objects[ic];
      if (icObj is TControl) then begin
       if opt_forms and not (icObj is TForm) then continue;
       if (w2<>'') then begin
        s:=List[ic]+' '+icObj.ClassName;
        if (icObj is TForm) then s:=s+' '+(icObj as TForm).Caption;
        if (PosI(w2,s)=0) then continue;
       end;
       if (n=1) then begin
        l1:=Max(l1,Length(List[ic]));
        l2:=Max(l2,Length(icObj.ClassName));
       end else begin
        if (icObj is TForm) then wt:=(icObj as TForm).Caption else wt:='';
        Echo(Format(' %-*s | %-*s | %s',[l1,List[ic],l2,icObj.ClassName,wt]));
        Result:=Result+1;
       end;
      end;
     end;
     Exit;
    end;
    n:=0;
    if IsSameText('Show',w1)        then n:=1;
    if IsSameText('Hide',w1)        then n:=2;
    if IsSameText('Visible',w1)     then n:=3;
    if IsSameText('Enable',w1)      then n:=4;
    if IsSameText('Disable',w1)     then n:=5;
    if IsSameText('Enabled',w1)     then n:=6;
    if IsSameText('Min',w1)         then n:=7;
    if IsSameText('Max',w1)         then n:=8;
    if IsSameText('Norm',w1)        then n:=9;
    if IsSameText('WindowState',w1) then n:=10;
    if IsSameText('Title',w1)       then n:=11;
    if IsSameText('Pos',w1)         then n:=12;
    if IsSameText('Move',w1)        then n:=13;
    if IsSameText('ShowFront',w1)   then n:=14;
    if IsSameText('ShowBack',w1)    then n:=15;
    if IsSameText('Activate',w1)    then n:=16;
    if IsSameText('Params',w1)      then n:=17;
    if IsSameText('Handle',w1)      then n:=18;
    if IsSameText('WmWnd',w1)       then n:=19;
    if (n=16) then begin
     wt:=Trim(SkipWords(1,arg,ScanSpaces));
     if IsLexeme(wt,lex_Quotes)
     then wt:=ExtractPhrase(1,wt,ScanSpaces);
     if IsMainConsoleCaption(wt) then begin
      SystemConsole.Activate;
      Result:=1;
      Exit;
     end else
     if IsDaqControlDialogCaption(wt) then begin
      SdiMan.ActivateChild(FormDaqControlDialog);
      Result:=1;
      Exit;
     end else begin
      for ic:=0 to SdiMan.ChildCount-1 do begin
       aChild:=SdiMan.ChildList[ic];
       if not Assigned(aChild) then Continue;
       if IsSameText(aChild.Caption,wt) then begin
        SdiMan.ActivateChild(aChild);
        Result:=1;
        Exit;
       end;
      end;
     end;
    end;
    if (n>0) then begin
     wt:=Trim(SkipWords(1,arg,ScanSpaces));
     wp:=Trim(SkipPhrases(1,wt,ScanSpaces));
     if IsLexeme(wt,lex_Quotes) or (n in [11,12,13,17])
     then wt:=ExtractPhrase(1,wt,ScanSpaces);
     wn:=Trim(MendComponentName(wt));
     if IsMainConsoleCaption(wt) then wt:=GetMainConsoleCaption;
     if IsDaqControlDialogCaption(wt) then wt:=GetDaqControlDialogCaption;
     ic:=FindTarget(List,IfThen(opt_title,'',wn),IfThen(opt_title,wt,''));
     if (ic<0) and not opt_xor
     then ic:=FindTarget(List,IfThen(opt_title,wn,''),IfThen(opt_title,'',wt));
     if (ic>=0) then begin
      icObj:=List.Objects[ic];
      if (icObj is TControl) then
      with (icObj as TControl) do
      case n of
       1 : begin Show; Result:=BoolToInt(Visible); end;
       2 : begin Hide; Result:=BoolToInt(Visible); end;
       3 : Result:=BoolToInt(Visible);
       4 : begin Enabled:=true;  Result:=BoolToInt(Enabled); end;
       5 : begin Enabled:=false; Result:=BoolToInt(Enabled); end;
       6 : Result:=BoolToInt(Enabled);
      end;
      if (icObj is TMainMenu) then
      if IsSameText(List[ic],Application.MainForm.Name+'.MainMenu') then
      case n of
       1 : begin Application.MainForm.Menu:=(icObj as TMainMenu); Result:=1; end;
       2 : begin Application.MainForm.Menu:=nil; Result:=0; end;
       3 : Result:=BoolToInt(Assigned(Application.MainForm.Menu));
      end;
      if (icObj is TForm) then
      with (icObj as TForm) do
      case n of
       7 : begin Show; WindowState:=wsMinimized; Result:=ord(WindowState); end;
       8 : begin Show; WindowState:=wsMaximized; Result:=ord(WindowState); end;
       9 : begin Show; WindowState:=wsNormal;    Result:=ord(WindowState); end;
       10: Result:=ord(WindowState);
       11: begin
            s:=Trim(wp);
            if (icObj=FormCrwDaq) and (s<>'') then FormCrwDaq.UpdateTitle(s) else
            if Length(s)>0 then Caption:=s;
            Result:=Length(Caption);
           end;
       12: begin
            ee.SetConst('view_pos_left',Left);
            ee.SetConst('view_pos_top',Top);
            ee.SetConst('view_pos_width',Width);
            ee.SetConst('view_pos_height',Height);
            ev:=NewExpressionEvaluator;
            try
             if Exp2Int(ev,ExtractWord(1,wp,ScanSpaces-[',','=']),iv) then Left:=iv;
             if Exp2Int(ev,ExtractWord(2,wp,ScanSpaces-[',','=']),iv) then Top:=iv;
             if Exp2Int(ev,ExtractWord(3,wp,ScanSpaces-[',','=']),iv) then Width:=iv;
             if Exp2Int(ev,ExtractWord(4,wp,ScanSpaces-[',','=']),iv) then Height:=iv;
            finally
             Kill(ev);
            end;
            Result:=Handle;
           end;
       13: begin
            ee.SetConst('view_pos_left',Left);
            ee.SetConst('view_pos_top',Top);
            ee.SetConst('view_pos_width',Width);
            ee.SetConst('view_pos_height',Height);
            ev:=NewExpressionEvaluator;
            try
             if Exp2Int(ev,ExtractWord(1,wp,ScanSpaces-[',','=']),iv) then Left:=Left+iv;
             if Exp2Int(ev,ExtractWord(2,wp,ScanSpaces-[',','=']),iv) then Top:=Top+iv;
             if Exp2Int(ev,ExtractWord(3,wp,ScanSpaces-[',','=']),iv) then Width:=Width+iv;
             if Exp2Int(ev,ExtractWord(4,wp,ScanSpaces-[',','=']),iv) then Height:=Height+iv;
            finally
             Kill(ev);
            end;
            Result:=Handle;
           end;
       16,
       14: begin
            Show;
            if (icObj is TForm) then begin
             if SdiMan.IsChild(icObj) then begin
              if (TForm(icObj).WindowState=wsMinimized)
              then TForm(icObj).WindowState:=wsNormal;
              SdiMan.ActivateChild(TForm(icObj));
             end;
             BringToFront;
            end else begin
             BringToFront;
            end;
            Result:=BoolToInt(Visible);
           end;
       15: begin Show; SendToBack; Result:=BoolToInt(Visible); end;
       17: if (icObj is TMasterForm) then begin
            MForm:=(icObj as TMasterForm);
            if Assigned(MForm) then begin
             if IsLexeme(wp,lex_AtCmnd) then begin
              Result:=MForm.ApplyParams(FilterLines(wp,'@set',''));
              Exit;
             end;
             fn:=SysIniFile; sn:=wp;
             if IsLexeme(sn,lex_Section) then begin
              s:=ExtractTextSection(fn,sn,svConfigNC);
              if opt_expand and MaybeEnvStr(s) then s:=ExpEnv(s);
              Result:=MForm.ApplyParams(FilterLines(s,'@set',''));
              Exit;
             end;
             fn:=SmartFileRef(ExtractWord(1,wp,ScanSpaces));
             sn:=SkipWords(1,wp,ScanSpaces);
             if IsSameText(ExtractFileExt(fn),'.ini') then
             if FileIsReadable(fn) and IsLexeme(sn,lex_Section) then begin
              s:=ExtractTextSection(fn,sn,svConfigNC);
              if opt_expand and MaybeEnvStr(s) then s:=ExpEnv(s);
              Result:=MForm.ApplyParams(FilterLines(s,'@set',''));
              Exit;
             end;
            end;
           end;
       18: if (icObj is TWinControl) then Result:=(icObj as TWinControl).Handle;
       19: if (icObj is TMasterForm) then Result:=(icObj as TMasterForm).WmWnd;
      end;
     end;
     Exit;
    end;
   finally
    Kill(List);
   end;
  end;
  Echo('Syntax:');
  Echo(' @view -e arg         - enable expand environ variables with arg');
  Echo('                        also -exp,-expand,--expand synonyms uses');
  Echo(' @view -f arg         - view only forms, skip other control type');
  Echo('                        also a -form,-forms,--form synonyms uses');
  Echo(' @view -t arg         - use title to find a form instead of name');
  Echo('                        also a -tit,-title,--title synonyms uses');
  Echo(' @view -x arg         - exclusive search only by name | by title');
  Echo('                        also a longopt -xor,--xor synonyms uses');
  Echo(' @view list           - get list of views');
  Echo(' @view -f list        - get list of forms only');
  Echo(' @view list kw        - get list of views by keyword (kw)');
  Echo(' @view show n         - show view item "n"');
  Echo(' @view hide n         - hide view item "n"');
  Echo(' @view showfront n    - show view item "n" and bring to front');
  Echo(' @view showback n     - show view item "n" and send to back');
  Echo(' @view visible n      - check state of item "n"');
  Echo(' @view enable  n      - enable  item "n"');
  Echo(' @view disable n      - disable item "n"');
  Echo(' @view enabled n      - check state of item "n"');
  Echo(' @view min n          - minimize  item "n"');
  Echo(' @view max n          - maximize  item "n"');
  Echo(' @view norm n         - normalize item "n"');
  Echo(' @view WindowState n  - check state of item "n"');
  Echo(' @view title n t      - set new title "t" of item "n"');
  Echo(' @view pos n l t w h  - set left/top/width/height of item "n"');
  Echo(' @view move n l t w h - move/grow left/top/width/height of item "n"');
  Echo(' @view params @set …  - apply parameters specified by @set … command');
  Echo(' @view params [s]     - apply parameters from crwdaq.ini section [s]');
  Echo(' @view params ini [s] - apply parameters read from ini file section [s]');
  Echo(' @view handle n       - find view (TWinControl), return his Handle');
  Echo(' @view wmwnd n        - find view (TMasterForm), return window WmWnd');
  Echo(' @view screen         - readout screen parameters to screen_xxx constants');
  Echo(' @view logo show "t"  - show LOGO screen with text (t) in double quotes');
  Echo('                        -daq option after show cause DAQ system logo');
  Echo('                        -bmp logo.bmp option uses to set bitmap file');
  Echo(' @view logo hide      - hide LOGO screen');
  Echo('Notes:');
  Echo(' Controls and forms has (internal) Name to identify');
  Echo(' Forms also have Title (window caption) to identify');
  Echo(' FormCrwDaq           is Name of Application`s main form');
  Echo(' FormConsoleWindow    is Name of "MAIN CONSOLE" form');
  Echo(' FormDaqControlDialog is Name of "DAQ-SYSTEM" form');
  Echo('Examples:');
  Echo(' @view list');
  Echo(' @view show FormCrwDaq.ToolBar');
  Echo(' @view move FormCrwDaq 100 * * *');
  Echo(' @view showfront FormConsoleWindow');
  Echo(' @view pos FormCrwDaq 100 200 740 560');
  Echo(' @view pos FormConsoleWindow 100 120');
  Echo(' @view -t pos "главная консоль" 100 120');
  Echo(' @view -e title FormCrwDaq %CRW_DAQ_SYS_TITLE%');
  Echo(' @view pos FormCrwDaq %screen_width/4 %screen_height/4 %screen_width/2 %screen_height/2');
  Echo(' @view -e params FormConsoleWindow @set Form.Top 0 relative "$CRW_DAQ_SYS_TITLE" StatusBar.Bottom');
  Echo(' @view params FormConsoleWindow @set Form.Top 0 relative TFormCrwDaq::FormCrwDaq StatusBar.Bottom');
  Echo(' @view -t params "главная консоль" @set Form.Top 0 relative *::FormCrwDaq StatusBar.Bottom');
  Echo(' @view logo show -daq "Идет загрузка DAQ..."');
  Echo(' @view logo show "Wait a minute please..."');
  Echo(' @view logo hide');
 except
  on E:Exception do BugReport(E,ee,'act_view');
 end;
end;

function act_memory(ee:TExpressionEvaluator; const args:LongString):double;
var ex:TExpressionEvaluator; Buff:TParsingBuffer; t:TText; te:TFormTextEditor;
var arg,w1,w2,s:LongString; n:Integer; {$IFDEF WINDOWS} a,b:PtrUInt; {$ENDIF}
begin
 Result:=0;
 try
  arg:=ee.SmartArgs(args);
  w1:=ExtractWord(1,arg,ScanSpaces);
  w2:=ExtractWord(2,arg,ScanSpaces);
  ex:=NewExpressionEvaluator;
  try
   s:='';
   if IsSameText('Load',w1)          then begin Result:=ReadProcMemInfo('MemLoad'); Exit; end;
   if IsSameText('TotalPhys',w1)     then begin Result:=ReadProcMemInfo('MemTotal'); Exit; end;
   if IsSameText('AvailPhys',w1)     then begin Result:=ReadProcMemInfo('MemAvailable'); Exit; end;
   if IsSameText('TotalPageFile',w1) then begin Result:=ReadProcMemInfo('SwapTotal'); Exit; end;
   if IsSameText('AvailPageFile',w1) then begin Result:=ReadProcMemInfo('SwapFree'); Exit; end;
   if IsSameText('AllocMemSize',w1)  then begin Result:=GetAllocMemSize;    Exit; end;
   if IsSameText('AllocMemCount',w1) then begin Result:=GetAllocMemCount;   Exit; end;
   if IsSameText('StdInFifoSize',w1)   then begin Result:=StdInputFifo.Size;  Exit; end;
   if IsSameText('StdInFifoLost',w1)   then begin Result:=StdInputFifo.Lost;  Exit; end;
   if IsSameText('StdInFifoCount',w1)  then begin Result:=StdInputFifo.Count;  Exit; end;
   if IsSameText('StdInFifoSpace',w1)  then begin Result:=StdInputFifo.Space;  Exit; end;
   if IsSameText('StdInFifoLimit',w1)  then begin Result:=StdInputFifo.GrowLimit;  Exit; end;
   if IsSameText('StdInFifoFactor',w1) then begin Result:=StdInputFifo.GrowFactor;  Exit; end;
   if IsSameText('StdOutFifoSize',w1)   then begin Result:=StdOutputFifo.Size;  Exit; end;
   if IsSameText('StdOutFifoLost',w1)   then begin Result:=StdOutputFifo.Lost;  Exit; end;
   if IsSameText('StdOutFifoCount',w1)  then begin Result:=StdOutputFifo.Count;  Exit; end;
   if IsSameText('StdOutFifoSpace',w1)  then begin Result:=StdOutputFifo.Space;  Exit; end;
   if IsSameText('StdOutFifoLimit',w1)  then begin Result:=StdOutputFifo.GrowLimit;  Exit; end;
   if IsSameText('StdOutFifoFactor',w1) then begin Result:=StdOutputFifo.GrowFactor;  Exit; end;
   if IsSameText('SysCalFifoSize',w1)   then begin Result:=SystemCalculator.Fifo.Size;  Exit; end;
   if IsSameText('SysCalFifoLost',w1)   then begin Result:=SystemCalculator.Fifo.Lost;  Exit; end;
   if IsSameText('SysCalFifoCount',w1)  then begin Result:=SystemCalculator.Fifo.Count;  Exit; end;
   if IsSameText('SysCalFifoSpace',w1)  then begin Result:=SystemCalculator.Fifo.Space;  Exit; end;
   if IsSameText('SysCalFifoLimit',w1)  then begin Result:=SystemCalculator.Fifo.GrowLimit;  Exit; end;
   if IsSameText('SysCalFifoFactor',w1) then begin Result:=SystemCalculator.Fifo.GrowFactor;  Exit; end;
   if IsSameText('min',w1) then begin
    if IsNonEmptyStr(w2) then
    {$IFDEF WINDOWS}
    a:=0; b:=0;
    if ex.EvaluateExpression(StrCopyBuff(Buff,w2))=ee_Ok then
    if GetProcessWorkingSetSize(GetCurrentProcess,a,b) then
    if not IsNan(ex.Answer) and not IsInf(ex.Answer) and (ex.Answer>=0) and (Round(ex.Answer)<>a)
    then SetProcessWorkingSetSize(GetCurrentProcess,Round(ex.Answer),b);
    if GetProcessWorkingSetSize(GetCurrentProcess,a,b) then Result:=a;
    {$ENDIF ~WINDOWS}
    Exit;
   end;
   if IsSameText('max',w1) then begin
    {$IFDEF WINDOWS}
    a:=0; b:=0;
    if IsNonEmptyStr(w2) then
    if ex.EvaluateExpression(StrCopyBuff(Buff,w2))=ee_Ok then
    if GetProcessWorkingSetSize(GetCurrentProcess,a,b) then
    if not IsNan(ex.Answer) and not IsInf(ex.Answer) and (ex.Answer>=0) and (Round(ex.Answer)<>b)
    then SetProcessWorkingSetSize(GetCurrentProcess,a,Round(ex.Answer));
    if GetProcessWorkingSetSize(GetCurrentProcess,a,b) then Result:=b;
    {$ENDIF ~WINDOWS}
    Exit;
   end;
   if IsSameText('CfgCacheSize',w1) then begin
    if IsNonEmptyStr(w2) then
    if ex.EvaluateExpression(StrCopyBuff(Buff,w2))=ee_Ok then
    if not IsNan(ex.Answer) and not IsInf(ex.Answer) and (ex.Answer>=0) then
    FreeConfigCache(Round(ex.Answer));
    Result:=FreeConfigCache(MaxInt);
    Exit;
   end;
   if IsSameText('CfgCacheLim',w1) then begin
    if IsNonEmptyStr(w2) then
    if ex.EvaluateExpression(StrCopyBuff(Buff,w2))=ee_Ok then
    if not IsNan(ex.Answer) and not IsInf(ex.Answer) and (ex.Answer>=0) then
    ConfigCacheLimit:=Round(ex.Answer);
    Result:=ConfigCacheLimit;
    Exit;
   end;
   if IsSameText('CfgCachePoll',w1) then begin
    if IsNonEmptyStr(w2) then
    if ex.EvaluateExpression(StrCopyBuff(Buff,w2))=ee_Ok then
    if not IsNan(ex.Answer) and not IsInf(ex.Answer) and (ex.Answer>=0) then
    ConfigCacheCheck:=Round(ex.Answer);
    Result:=ConfigCacheCheck;
    Exit;
   end;
   if IsSameText('CfgCacheHold',w1) then begin
    if IsNonEmptyStr(w2) then
    if ex.EvaluateExpression(StrCopyBuff(Buff,w2))=ee_Ok then
    if not IsNan(ex.Answer) and not IsInf(ex.Answer) and (ex.Answer>=0) then
    ConfigCacheHoldingTime:=Round(ex.Answer);
    Result:=ConfigCacheHoldingTime;
    Exit;
   end;
   if IsSameText('CfgCacheHash',w1) then begin
    if IsNonEmptyStr(w2) then
    if ex.EvaluateExpression(StrCopyBuff(Buff,w2))=ee_Ok then
    if not IsNan(ex.Answer) and not IsInf(ex.Answer) then begin
     ConfigCacheHasherMethod:=FindIndexOfHasher(GetHasherByIndex(Round(ex.Answer)));
     ResetConfigCache;
    end;
    Result:=ConfigCacheHasherMethod;
    Exit;
   end;
   if IsSameText('CfgCacheShot',w1) then begin
    te:=NewTextEditor;
    if Assigned(te) then begin
     te.PerformLinesText:=ConfigCacheSnapshot(StrToIntDef(w2,0));
     te.PathName:=SessionManager.VarTmpFile('configcacheshapshot.lst');
     te.FileSave; te.PerformModified:=false; te.PerformReadOnly:=true;
    end;
    Result:=FreeConfigCache(MaxInt);
    Exit;
   end;
   if IsSameText('CfgCacheLook',w1) then begin
    if IsNonEmptyStr(w2) then begin
     if FileExists(SmartFileRef(w2)) then begin
      t:=ConstructFullConfig(SmartFileRef(w2));
      try
       te:=NewTextEditor;
       if Assigned(te) then begin
        te.PerformLinesText:=t.text;
        te.PathName:=SessionManager.VarTmpFile(ExtractFileName(SmartFileRef(w2))+'.lst');
        te.FileSave; te.PerformModified:=false; te.PerformReadOnly:=true;
       end;
      finally
       Kill(t);
      end;
     end else Echo('File not exist: '+w2);
    end;
    Result:=FreeConfigCache(MaxInt);
    Exit;
   end;
   if IsSameText(w1,'BmpCacheCount') then begin
    Result:=BmpCache.Count;
    Exit;
   end;
   if IsSameText(w1,'BmpCacheFiles') then begin
    s:=BmpCache.BmpFileNamesAsText(StrToIntDef(w2,0));
    Result:=WordCount(s,EolnDelims);
    Echo(Trim(s));
    Exit;
   end;
   if IsSameText(w1,'BmpCacheClear') then begin
    Result:=BoolToInt(BmpCache.Clear);
    Exit;
   end;
   if IsSameText(w1,'BmpCacheCaptureCount') then begin
    Result:=BmpCache.CaptureCount;
    Exit;
   end;
   {
   if IsSameText(w1,'BmpCacheCapture') then begin
    Result:=BmpCache.Capture;
    Exit;
   end;
   if IsSameText(w1,'BmpCacheUncapture') then begin
    Result:=BmpCache.Uncapture;
    Exit;
   end;
   }
   if IsSameText(w1,'BmpCacheStat') then begin
    Echo(Trim(BmpCache.GetBmpStatInfoAsText));
    Result:=BmpCache.Count;
    Exit;
   end;
   if IsSameText(w1,'BmpCacheTest') then begin
    if IsNonEmptyStr(w2) then w2:=SmartFileRef(w2);
    BmpCache.SelfTest(Output,TrimDef(w2,SmartFileRef('~~\demo')));
    Result:=BmpCache.Count;
    Exit;
   end;
   if IsSameText(w1,'BmpCacheLoad') then begin
    if IsNonEmptyStr(w2) then w2:=SmartFileRef(w2) else Exit;
    if DirExists(w2) then BmpCache.LoadFromDir(w2) else
    if FileExists(w2) then BmpCache.Find(w2,True);
    Result:=BmpCache.Count;
    Exit;
   end;
   if IsSameText(w1,'Clear') then begin
    s:=Trim(SkipWords(1,arg,ScanSpaces));
    if WordIndex('All',s,ScanSpaces)>0 then begin
     s:='CfgCache,BmpCache';
    end;
    if WordIndex('CfgCache',s,ScanSpaces)>0 then begin
     n:=SendToMainConsole('@silent @memory CfgCacheSize 0'+EOL);
     Result:=Result+n;
    end;
    if WordIndex('BmpCache',s,ScanSpaces)>0 then begin
     n:=SendToMainConsole('@silent @memory BmpCacheClear'+EOL);
     Result:=Result+n;
    end;
    Exit;
   end;
  finally
   Kill(ex);
  end;
  Echo('Syntax:');
  Echo(' @memory Load            - physical memory utilization, %');
  Echo(' @memory TotalPhys       - total physical memory in bytes');
  Echo(' @memory AvailPhys       - avail physical memory in bytes');
  Echo(' @memory TotalPageFile   - total page file size in bytes');
  Echo(' @memory AvailPageFile   - avail page file size in bytes');
  Echo(' @memory TotalVirtual    - total virtual memory in bytes');
  Echo(' @memory AvailVirtual    - avail virtual memory in bytes');
  Echo(' @memory AllocMemSize    - allocated memory size, bytes');
  Echo(' @memory AllocMemCount   - allocated memory count, blocks');
  Echo(' @memory min             - get min WorkingSetSize in bytes');
  Echo(' @memory min n           - set min WorkingSetSize = n bytes');
  Echo(' @memory max             - get max WorkingSetSize in bytes');
  Echo(' @memory max n           - set max WorkingSetSize = n bytes');
  Echo(' @memory CfgCacheSize    - get config file cache size');
  Echo(' @memory CfgCacheSize n  - set config file cache size');
  Echo(' @memory CfgCacheLim     - get config file cache size limit');
  Echo(' @memory CfgCacheLim n   - set config file cache size limit');
  Echo(' @memory CfgCachePoll    - get config file cache check period, ms');
  Echo(' @memory CfgCachePoll n  - set config file cache check period, ms');
  Echo(' @memory CfgCacheHold    - get config file cache holding time, ms');
  Echo(' @memory CfgCacheHold n  - set config file cache holding time, ms');
  Echo(' @memory CfgCacheHash    - get config file cache hashing method, 0..'+IntToStr(High(THash32MethodId)));
  Echo(' @memory CfgCacheHash n  - set config file cache hashing method and reset');
  Echo(' @memory CfgCacheShot m  - view configuration file cache shapshot with verbosity mode m=0..2');
  Echo(' @memory CfgCacheLook f  - look configuration file cache');
  Echo(' @memory StdInFifoSize   - StdIn fifo buffer size, bytes');
  Echo(' @memory StdInFifoLost   - StdIn fifo buffer lost, bytes');
  Echo(' @memory StdInFifoCount  - StdIn fifo buffer count, bytes');
  Echo(' @memory StdInFifoSpace  - StdIn fifo buffer space, bytes');
  Echo(' @memory StdInFifoLimit  - StdIn fifo buffer limit, bytes');
  Echo(' @memory StdInFifoFactor - StdIn fifo buffer factor, times');
  Echo('  and also similar variables for StdOut and SysCal fifos:');
  Echo('  StdOutFifoSize, SysCalFifoSize etc.');
  Echo(' @memory BmpCacheCount   - return BmpCache list count');
  Echo(' @memory BmpCacheFiles n - print BmpCache files with bitness n=[1,4,8,15,16,24,32]');
  Echo(' @memory BmpCacheClear   - clear BmpCache if it is uncaptured');
  Echo(' @memory BmpCacheCaptureCount - BmpCache capture counter');
  Echo(' @memory BmpCacheStat    - print BmpCache stat info');
  Echo(' @memory BmpCacheTest d  - perform BmpCache self test with dir d');
  Echo(' @memory BmpCacheLoad f  - load BmpCache file f or *.bmp from dir f');
  Echo(' @memory Clear l         - clear cache list l=[All,CfgCache,BmpCache]');
  Echo('Examples:');
  Echo(' @memory min 1024*1024*32');
  Echo(' @memory max 1024*1024*128');
  Echo(' @memory CfgCacheSize 1024*1024*2');
  Echo(' @memory Clear CfgCache,BmpCache');
 except
  on E:Exception do BugReport(E,ee,'act_memory');
 end;
end;

function act_sleep(ee:TExpressionEvaluator; const args:LongString):double;
var ms,dt:Double; arg:LongString;
const MaxDelay=1000*10;
begin
 Result:=0;
 try
  ms:=msecnow;
  arg:=ee.SmartArgs(args);
  if Str2Real(arg,dt) then
  if not IsNan(dt) and not IsInf(dt) and (dt>=0) and (dt<=MaxDelay) then begin
   Sleep(Round(dt));
   Result:=msecnow-ms;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @sleep n - sleep program to n milliseconds');
  Echo('            where n is in range 0..'+IntToStr(MaxDelay));
 except
  on E:Exception do BugReport(E,ee,'act_sleep');
 end;
end;

function ListTextAsTable(aText:LongString; Width:Integer=80; Sort:Boolean=false):LongString;
var Lines,Table:TStringList; i,maxlen,cols:Integer; Line:LongString; sw:WideString;
begin
 Result:='';
 try
  Table:=TStringList.Create;
  Lines:=TStringList.Create;
  try
   Lines.Text:=aText; maxlen:=1; Line:=''; if Sort then Lines.Sort;
   for i:=0 to Lines.Count-1 do maxlen:=Max(maxlen,Length(StrToWide(Lines[i])));
   cols:=Max(1,Width div (maxlen+1));
   for i:=0 to Lines.Count-1 do begin
    sw:=StrToWide(Lines[i]);
    while Length(sw)<maxlen do sw:=sw+' ';
    if (Line='') then Line:=WideToStr(sw) else Line:=Line+' '+WideToStr(sw);
    if ((i mod cols)=(cols-1)) or (i=Lines.Count-1) then begin
     Table.Add(Line);
     Line:='';
    end;
   end;
   Result:=Table.Text;
  finally
   Kill(Lines);
   Kill(Table);
  end;
 except
  on E:Exception do BugReport(E,nil,'ListTextAsTable');
 end;
end;

function act_voice(ee:TExpressionEvaluator; const args:LongString):Double;
var arg,opt,s:LongString; n:Integer;
begin
 Result:=0;
 if UsesBlaster then begin
  arg:=Trim(ee.SmartArgs(args));
  if (arg<>'') then begin
   while IsOption(arg) do begin
    opt:=ExtractWord(1,arg,JustSpaces);
    arg:=SkipWords(1,arg,JustSpaces);
    if IsOption(opt,'-f','--full') then begin
     s:=GetSoundLibrary(true);
     Echo('SoundLibrary:');
     Echo(Trim(s));
     n:=WordCount(s,ScanSpaces);
     Echo(IntToStr(n)+' item(s) on sound library found');
     Exit;
    end;
    if IsOption(opt,'-l','--list') then begin
     s:=GetSoundLibrary(false);
     s:=ListTextAsTable(s,80,true);
     Echo('SoundLibrary: '+IntToStr(n)+' items');
     Echo(Trim(s));
     n:=WordCount(s,ScanSpaces);
     Echo(IntToStr(n)+' item(s) on sound library found');
     Exit;
    end;
    if IsOption(opt,'-a','--add') then begin
     arg:=Trim(arg);
     if (arg<>'') then arg:=SmartFileRef(arg,'',ProgName);
     if (arg<>'') and FileExists(AddPathDelim(arg)+'*.wav') then begin
      AddSoundLibrary(arg);
     end else Echo(RusEng('Не могу добавить звуки ','Could not add sounds ')+arg);
     Exit;
    end;
   end;
   Result:=1;
   Voice(arg);
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @voice s    - play sound s');
  Echo(' @voice -l   - list sound library - base names');
  Echo(' @voice -f   - list sound library - full names');
  Echo(' @voice -a d - add sound library in directory d');
 end;
end;

function act_speak(ee:TExpressionEvaluator; const args:LongString):double;
var arg:LongString;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   if (Speaker.Engine>=0) then begin
    if IsWindows then arg:=DefToAnsiCp(arg);
    Speaker.Speak:=arg;
    Result:=1;
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @speak phrase - Speak some "phrase" via Speech API.');
  Echo('Examples:');
  Echo(' @speech engine 1');
  Echo(' @speak Hello, World.');
  Echo('Example of more important SAPI tags:');
  Echo(' @speak \Rst\Reset all tags to default values.');
  Echo(' @speak Pause \Pau=1000\in speech for 1000 milliseconds.');
  Echo(' @speak Speak \Vol=65535\strong (loud) or \Vol=0\weak (silent).');
  Echo(' @speak \Spd=100\Speech speed 100 words per minute.');
  Echo(' @speak Pitch (speech tone) \Pit=100\low \Pit=200\high.');
  Echo(' @speak Speak \Pro=1\with intonation or \Pro=0\without intonation (monotone).');
  Echo(' @speak Speak \RmS=1\by chars \rms=0\by normal words.');
  Echo(' @speak Force \Emp\next word with strong intonation.');
 except
  on E:Exception do BugReport(E,ee,'act_speak');
 end;
end;

function act_speech(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1,w2:LongString; i,j,k:Integer;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=ee.SmartArgs(args);
  w1:=ExtractWord(1,arg,ScanSpaces);
  w2:=ExtractWord(2,arg,ScanSpaces);
  if IsSameText('Engines',w1) then begin
   Result:=Speaker.Engines.Count;
   for i:=0 to Speaker.Engines.Count-1 do
   ActEcho(Format('SpeechEngine[%d] = %s',[i+1,Speaker.Engines[i]]));
   Exit;
  end;
  if IsSameText('Engine',w1) then begin
   if Str2Int(w2,i) then Speaker.Engine:=i-1 else if Length(w2)>0 then begin
    j:=-1;
    k:=Speaker.Engine;
    if Speaker.Engine<0 then Speaker.Engine:=0;
    w2:=Trim(Copy(arg,Pos(w1,arg)+Length(w1),255));
    for i:=0 to Speaker.Engines.Count-1 do
    if IsSameText(w2,Trim(Speaker.Engines[i])) then j:=i;
    if j<0 then Speaker.Engine:=k else Speaker.Engine:=j;
   end;
   Result:=Speaker.Engine+1;
   Exit;
  end;
  if IsSameText('Volume',w1) then begin
   if (Speaker.Engine>=0) then begin
    if Str2Int(w2,i) then Speaker.Volume:=i;
    Result:=Speaker.Volume;
   end;
   Exit;
  end;
  if IsSameText('Pitch',w1) then begin
   if (Speaker.Engine>=0) then begin
    if Str2Int(w2,i) then Speaker.Pitch:=i;
    Result:=Speaker.Pitch;
   end; 
   Exit;
  end;
  if IsSameText('Speed',w1) then begin
   if (Speaker.Engine>=0) then begin
    if Str2Int(w2,i) then Speaker.Speed:=i;
    Result:=Speaker.Speed;
   end;
   Exit;
  end;
  if IsSameText('Pause',w1) then begin
   if (Speaker.Engine>=0) then begin
    if Str2Int(w2,i) then Speaker.Pause:=(i<>0);
    Result:=Ord(Speaker.Pause);
   end;
   Exit;
  end;
  if IsSameText('Wait',w1) then begin
   if (Speaker.Engine>=0) then begin
    if Str2Int(w2,i) then Speaker.Wait:=(i<>0);
    Result:=Ord(Speaker.Wait);
   end;
   Exit;
  end;
  if IsSameText('Stop',w1) then begin
   if (Speaker.Engine>=0) then begin
    Speaker.Stop;
    Result:=1;
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @speech engine   - Get current Speech API engine.');
  Echo(' @speech engine n - Set current Speech API engine n>0.');
  Echo(' @speech engine 0 - Switch off Speech API, disable speaking.');
  Echo(' @speech engines  - List available Speech API engines.');
  Echo(' @speech volume   - Get current Speech API volume.');
  Echo(' @speech volume n - Set current Speech API volume.');
  Echo(' @speech pitch    - Get current Speech API pitch.');
  Echo(' @speech pitch n  - Set current Speech API pitch.');
  Echo(' @speech speed    - Get current Speech API speed.');
  Echo(' @speech speed  n - Set current Speech API speed.');
  Echo(' @speech pause    - Get current Speech API pause state.');
  Echo(' @speech pause  n - Set current Speech API pause state.');
  Echo(' @speech wait     - Get current Speech API wait flag.');
  Echo(' @speech wait n   - Set current Speech API wait flag (0/1).');
  Echo(' @speech stop     - Stop speech.');
  Echo('Examples:');
  Echo(' @speech engine 1');
  Echo(' @speak Hello, World.');
 except
  on E:Exception do BugReport(E,ee,'act_speech');
 end;
end;

procedure SockReporter(Pipe:TSocketPipe; When:Double; What:PChar; Code:Integer);
begin
 Echo(SocketErrorReport(What,Code));
end;

procedure PipeReporter(Pipe:TPipe; When:Double; const What:LongString; Code:Integer);
begin
 Echo(SocketErrorReport(What,Code));
end;

function act_term(ee:TExpressionEvaluator; const args:LongString):double;
var win:TFormTermWindow; arg,w1:LongString; pip,opt:Integer;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   opt:=0;
   arg:=Trim(arg);
   while IsOption(arg) do begin
    w1:=ExtractWord(1,arg,ScanSpaces);
    arg:=SkipWords(1,arg,ScanSpaces);
    opt:=GetTermOptInt(w1,opt);
   end;
   win:=nil;
   if Opt and to_Verbose <> 0
   then pip:=pipe_init(arg,PipeReporter,SockReporter)
   else pip:=pipe_init(arg);
   if pip<>0 then begin
    win:=NewConsolePipeWindow('@term'+GetTermOptStr(opt)+' '+Trim(arg),pip,opt);
    if not pipe_run(pip) then Kill(win);
   end;
   if win.Ok
   then Result:=pipe_pid(pip)
   else Echo('Could not execute @term'+GetTermOptStr(opt)+' '+arg);
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @term -opts args - Open terminal window with given options and arguments.');
  Echo('  args - list of arguments:');
  Echo('  | task c                - Run program c with asynchronous pipe i/o.');
  Echo('  | pipe n                - Open named pipe server with name n.');
  Echo('  | pipe h\n              - Open named pipe client with name n on host h.');
  Echo('  | tcp port n server m   - Open TCP port n server for m clients.');
  Echo('  | tcp port n client h   - Open TCP port n client to connect to host h.');
  Echo('  | com port n baudrate m - Open COM port n with baudrate m.');
  Echo('  -opts - list of options:');
  Echo('  | -a - ANSI: use ANSI/OEM conversion.  Example: @term -a task cmd');
  Echo('  | -d - Display: run task as visible.   Example: @term -d task cmd');
  Echo('  | -c - Close: close when disconnected. Example: @term -ca task cmd');
  Echo('  | -v - Verbose: display debug text.    Example: @term -v tcp port 1234 server 2');
  Echo('  | -x - HEX: view income data in HEX.   Example: @term -hx com port 2 baudrate 115200');
  Echo('  | -h - HEX: HEX_decode outcome data.   Example: @term -hx com port 2 baudrate 115200');
  Echo('  | -u - URL: URL_decode outcome data.   Example: @term -ux com port 2 baudrate 115200');
  Echo('  | -s - CheckSum: use ADAM checksum.    Example: @term -s com port 2 baudrate 115200');
  Echo('Examples:');
  Echo(' @term -ca task cmd                  - run command processor, ANSI/OEM, auto close on exit');
  Echo(' @term -a task ping.exe              - run program ping.exe, ANSI/OEM, not close on exit');
  Echo(' @term pipe exchange                 - run named pipe server');
  Echo(' @term pipe crwbox\exchange          - run named pipe client');
  Echo(' @term tcp port 1234 server 2        - run TCP port 1234 server for 2 clients');
  Echo(' @term tcp port 1234 client crwbox   - run TCP port 1234 client to connect to crwbox');
  Echo(' @term -s com port 2 baudrate 115200 - open COM port 2 with baudrate 115200 and checksum');
  Echo(' @term -hxs com port 2               - open COM port 2 with baudrate 9600,checksum, HEX I/O');
 except
  on E:Exception do BugReport(E,ee,'act_term');
 end;
end;

function dcc32_Compile(arg:LongString):Boolean;
begin
 Result:=false;
 arg:=Trim(arg);
 if (arg='') then Exit;
 arg:=UnifyFileAlias(FExpand(Trim(arg)));
 if CompileDprVerbose
 then Result:=SafeCompileDelphiProject(arg)
 else Result:=dcc32.Compile(arg,ExtractFilePath(arg));
end;

function Fpcup_Compile(arg:LongString):Boolean;
begin
 Result:=false;
 arg:=Trim(arg);
 if (arg='') then Exit;
 arg:=UnifyFileAlias(FExpand(Trim(arg)));
 if CompileLprVerbose
 then Result:=Fpcup.GuiCompile(arg)
 else Result:=Fpcup.RunCompile(arg);
end;

function act_compile(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1,w2:LongString; dev:TDaqDevice;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   arg:=Trim(Arg);
   w1:=ExtractWord(1,arg,ScanSpaces);
   w2:=ExtractWord(2,arg,ScanSpaces);
   if IsSameText(w1,'Device') then begin
    if SystemConsole.ShouldBeDelayed then Exit;
    dev:=FullDaqDeviceList.Find(w2);
    if Assigned(Dev) then begin
     if dev is TProgramDevice then begin
      Result:=Ord((dev as TProgramDevice).ReCompile);
      if Result=0 then Echo(Format('Could not compile device "%s".',[w2]));
     end else Echo(Format('Device "%s" is not Daq Pascal Program.',[w2]));
    end else Echo(Format('Device "%s" is not found.',[w2]));
   end else
   if SameText(w1,'verbose') then begin
    CompileDprVerbose:=true;
    CompileLprVerbose:=true;
   end else
   if SameText(w1,'silent') then begin
    CompileDprVerbose:=false;
    CompileLprVerbose:=false;
   end else
   if FileExists(arg) then begin
    if IsSameText(ExtractFileExt(arg),'.dpr')
    then Result:=ord(dcc32_Compile(arg)) else
    if IsSameText(ExtractFileExt(arg),'.lpr')
    then Result:=ord(Fpcup_Compile(arg)) else
    if IsSameText(ExtractFileExt(arg),'.nsi') then begin
     if FileExists(GetNsisCompilerPath)
     then Result:=ord(SmartExecute(Format('"%s" "%s"',[GetNsisCompilerPath,arg])));
    end;
    if Result=0 then Echo(Format('Could not compile "%s".',[arg]));
   end else Echo(Format('Could not find file "%s".',[arg]));
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @compile device &name - Compile DAQ Pascal Program device &name.');
  Echo(' @compile filename.dpr - Compile Delphi  project.');
  Echo(' @compile filename.lpr - Compile Lazarus project.');
  Echo(' @compile filename.nsi - Compile NSIS project.');
  Echo(' @compile silent       - Switch silent  DPR/LPR compilation (default).');
  Echo(' @compile verbose      - Switch verbose DPR/LPR compilation.');
  Echo('Note:');
  Echo(' Always use full file names, with path and extensions.');
  Echo('Examples:');
  Echo(' @compile device &WebSrv');
  Echo(' @compile c:\demo\test.dpr');
  Echo(' @compile c:\demo\test.lpr');
  Echo(' @compile c:\test\demo.nsi');
 except
  on E:Exception do BugReport(E,ee,'act_compile');
 end;
end;

const biMiniMax=[biSystemMenu, biMinimize, biMaximize];
type  TDaqTopListRec=packed record CurWin,TabWin,CirWin,SpeWin:packed record n,i:Integer; end; end;

procedure DoDaqTopListSdiChildren(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 if Assigned(Form) and Assigned(Custom) then with TDaqTopListRec(Custom^) do begin
  if (CurWin.n>=0) and (Form is TFormCurveWindow) then with (Form as TFormCurveWindow) do begin
   if (DaqRef<>0) and IsFormExposed and (BorderIcons*biMiniMax<>[]) then begin
    inc(CurWin.i); if (CurWin.i>CurWin.n) then WindowState:=wsMinimized;
   end;
  end;
  if (TabWin.n>=0) and (Form is TFormTabWindow) then with (Form as TFormTabWindow) do begin
   if (DaqRef<>0) and IsFormExposed and (BorderIcons*biMiniMax<>[]) then begin
    inc(TabWin.i); if (TabWin.i>TabWin.n) then WindowState:=wsMinimized;
   end;
  end;
  if (CirWin.n>=0) and (Form is TFormCircuitWindow) then with (Form as TFormCircuitWindow) do begin
   if (DaqRef<>0) and IsFormExposed and (BorderIcons*biMiniMax<>[]) then begin
    inc(CirWin.i); if (CirWin.i>CirWin.n) then WindowState:=wsMinimized;
   end;
  end;
  if (SpeWin.n>=0) and (Form is TFormSpectrWindow) then with (Form as TFormSpectrWindow) do begin
   if (DaqRef<>0) and IsFormExposed and (BorderIcons*biMiniMax<>[]) then begin
    inc(SpeWin.i); if (SpeWin.i>SpeWin.n) then WindowState:=wsMinimized;
   end;
  end;
 end;
end;

function DaqTopList(const TopList:LongString):Integer;
var i,n:Integer; w1,w2:LongString; DaqTopListRec:TDaqTopListRec;
begin
 Result:=0;
 if not Daq.Timer.isStart then Exit;
 with DaqTopListRec do begin
  CurWin.n:=-1; CurWin.i:=0;
  TabWin.n:=-1; TabWin.i:=0;
  CirWin.n:=-1; CirWin.i:=0;
  SpeWin.n:=-1; SpeWin.i:=0;
 end;
 i:=1; n:=WordCount(TopList,ScanSpaces);
 while (i<n) do with DaqTopListRec do begin
  w1:=ExtractWord(i,TopList,ScanSpaces); inc(i);
  w2:=ExtractWord(i,TopList,ScanSpaces); inc(i);
  if SameText(w1,wt_Curve_Window)   then CurWin.n:=StrToIntDef(w2,-1);
  if SameText(w1,wt_Tab_Window)     then TabWin.n:=StrToIntDef(w2,-1);
  if SameText(w1,wt_Circuit_Window) then CirWin.n:=StrToIntDef(w2,-1);
  if SameText(w1,wt_Spectr_Window)  then SpeWin.n:=StrToIntDef(w2,-1);
 end;
 with DaqTopListRec do begin
  if (CurWin.n>=0) or (TabWin.n>=0) or (CirWin.n>=0) or (SpeWin.n>=0)
  then SdiMan.ForEachChildInZOrder(DoDaqTopListSdiChildren,@DaqTopListRec);
  Result:=CurWin.i+TabWin.i+CirWin.i+SpeWin.i;
 end;
end;

function act_daq(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1,w2:LongString; dev:TDaqDevice;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   arg:=Trim(Arg);
   w1:=ExtractWord(1,arg,ScanSpaces);
   w2:=ExtractWord(2,arg,ScanSpaces);
   case Identify(w1) of
    sid_Compile:
    begin
     if not Daq.IsSessionStarted then Exit;
     if SystemConsole.ShouldBeDelayed then Exit;
     dev:=FullDaqDeviceList.Find(w2);
     if Assigned(Dev) then begin
      if dev is TProgramDevice then begin
       Result:=Ord((dev as TProgramDevice).ReCompile);
       if Result=0 then Echo(Format('Could not compile device "%s".',[w2]));
      end else Echo(Format('Device "%s" is not Daq Pascal Program.',[w2]));
     end else Echo(Format('Device "%s" is not found.',[w2]));
    end;
    sid_DevSend, sid_DevMsg, sid_DevSendMsg:
    begin
     if not Daq.IsAcquisitionStarted then Exit;
     dev:=FullDaqDeviceList.Find(w2);
     if Assigned(Dev) then begin
      if dev is TProgramDevice then begin
       Result:=Round((dev as TProgramDevice).DaqPascal._devsend(dev.Ref,SkipWords(2,arg,ScanSpaces)+EOL));
       if Result=0 then Echo(Format('Could not send message to device "%s".',[w2]));
      end else Echo(Format('Device "%s" is not Daq Pascal Program.',[w2]));
     end else Echo(Format('Device "%s" is not found.',[w2]));
    end;
    sid_DevPost, sid_DevPostMsg:
    begin
     if not Daq.IsAcquisitionStarted then Exit;
     dev:=FullDaqDeviceList.Find(w2);
     if Assigned(Dev) then begin
      if dev is TProgramDevice then begin
       Result:=Round((dev as TProgramDevice).DaqPascal._devpost(dev.Ref,SkipWords(2,arg,ScanSpaces)+EOL));
       if Result=0 then Echo(Format('Could not post message to device "%s".',[w2]));
      end else Echo(Format('Device "%s" is not Daq Pascal Program.',[w2]));
     end else Echo(Format('Device "%s" is not found.',[w2]));
    end;
    sid_OpenConsole:
    begin
     if not Daq.IsSessionStarted then Exit;
     dev:=FullDaqDeviceList.Find(w2);
     if Assigned(Dev) then begin
      if dev is TProgramDevice then begin
       Result:=Ord((dev as TProgramDevice).OpenConsole);
       if Result=0 then Echo(Format('Could not open console of device "%s".',[w2]));
      end else Echo(Format('Device "%s" is not Daq Pascal Program.',[w2]));
     end else Echo(Format('Device "%s" is not found.',[w2]));
    end;
    sid_OpenEditor:
    begin
     if not Daq.IsSessionStarted then Exit;
     dev:=FullDaqDeviceList.Find(w2);
     if Assigned(Dev) then begin
      if dev is TProgramDevice then begin
       Result:=Ord((dev as TProgramDevice).OpenDaqPascalEditor);
       if Result=0 then Echo(Format('Could not open editor of device "%s".',[w2]));
      end else Echo(Format('Device "%s" is not Daq Pascal Program.',[w2]));
     end else Echo(Format('Device "%s" is not found.',[w2]));
    end;
    sid_TopList:
    begin
     if not Daq.IsSessionStarted then Exit;
     if WordCount(arg,ScanSpaces)>2 then Result:=DaqTopList(SkipWords(1,arg,ScanSpaces));
    end;
    sid_LoadWindowsDialog:
    begin
     if not Daq.IsSessionStarted then Exit;
     Result:=Daq.LoadWindowsDialog;
    end;
    sid_KillWindowsDialog:
    begin
     if not Daq.IsSessionStarted then Exit;
     Result:=Daq.KillWindowsDialog;
    end;
    sid_Info:
    begin
     Result:=BoolToInt(Daq.IsSessionStarted)
            +BoolToInt(Daq.IsAcquisitionStarted)*2;
     ActEcho(Trim(Daq.SessionStatusInfo));
    end;
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @daq compile &name        - Compile DAQ Pascal Program device &name.');
  Echo(' @daq devsend &name msg    - Send message "msg" to device &name.');
  Echo(' @daq devmsg  &name msg    - Send message "msg" to device &name.');
  Echo(' @daq devsendmsg &name msg - Send message "msg" to device &name.');
  Echo(' @daq devpost &name msg    - Post message "msg" to device &name.');
  Echo(' @daq devpostmsg &name msg - Post message "msg" to device &name.');
  Echo(' @daq openconsole &name    - Open console window of DAQ Pascal Program device &name.');
  Echo(' @daq openeditor &name     - Open editor window of DAQ Pascal Program device &name.');
  Echo(' @daq toplist wt n         - minimize all windows of type wt except n toplist.');
  Echo(' @daq LoadWindowsDialog    - open dialog to load [Windows] from DAQ config.');
  Echo(' @daq KillWindowsDialog    - open dialog to kill/close DAQ window(s).');
  Echo(' @daq info                 - print information on DAQ system.');
  Echo('Examples:');
  Echo(' @daq compile &WebSrv');
  Echo(' @daq devsend &CronSrv @Warning Hello');
  Echo(' @daq toplist Curve_Window 5 Tab_Window 4 Circuit_Window 7 Spectr_Window 3');
 except
  on E:Exception do BugReport(E,ee,'act_daq');
 end;
end;

function act_tty(ee:TExpressionEvaluator; const args:LongString):double;
const
 OnOff:array[Boolean] of PChar = ('OFF','ON');
 procedure ReportTtyListen(p:Integer);
 begin
  ActEcho(Format('@TTY listen %d count %d',[p,SystemConsole.TtyListenCount(p)]));
 end;
 procedure ReportTtyVerbose;
 begin
  ActEcho('@TTY verbose '+OnOff[SystemConsole.TtyVerbose]);
 end;
 procedure ReportTtyShield;
 begin
  ActEcho('@TTY shield '+OnOff[SystemConsole.TtyShield]);
 end;
 function StrToBoolDef(const s:LongString; Def:Boolean):Boolean;
 begin
  Result:=Def;
  if IsNonEmptyStr(s) then
  if IsSameText(s,'OFF') then Result:=False else
  if IsSameText(s,'ON') then Result:=True else Result:=(StrToIntDef(s,Ord(Def))<>0);
 end;
var arg,w1,w2:LongString; i,nport,nlisten:Integer; pl:LongString;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   arg:=Trim(Arg);
   w1:=ExtractWord(1,arg,ScanSpaces);
   w2:=ExtractWord(2,arg,ScanSpaces);
   if IsSameText(w1,'Status') then begin
    nlisten:=0;
    pl:=SystemConsole.TtyPortList;
    for i:=1 to WordCount(pl,ScanSpaces) do
    if Str2Int(ExtractWord(i,pl,ScanSpaces),nport) then
    if SystemConsole.TtyListenCount(nport)>0 then begin
     ReportTtyListen(nport);
     Inc(nlisten);
    end;
    if nlisten=0 then ActEcho('@TTY don`t listen ports');
    ReportTtyVerbose;
    ReportTtyShield;
   end else
   if IsSameText(w1,'Listen') then begin
    if Str2Int(w2,nport) then
    if PortInRange(nport,0,0) or PortInRange(nport,TcpMinPort,TcpMaxPort) then begin
     SystemConsole.TtyListen(nport);
     ReportTtyListen(nport);
    end;
   end else
   if IsSameText(w1,'Close') then begin
    if Str2Int(w2,nport) then
    if PortInRange(nport,0,0) or PortInRange(nport,TcpMinPort,TcpMaxPort) then begin
     SystemConsole.TtyClose(nport);
     ReportTtyListen(nport);
    end;
   end else
   if IsSameText(w1,'Verbose') then begin
    if IsNonEmptyStr(w2) then
    SystemConsole.TtyVerbose:=StrToBoolDef(w2,SystemConsole.TtyVerbose);
    ReportTtyVerbose;
   end else
   if IsSameText(w1,'Shield') then begin
    if IsNonEmptyStr(w2) then
    SystemConsole.TtyShield:=StrToBoolDef(w2,SystemConsole.TtyShield);
    ReportTtyShield;
   end else
   if IsSameText(w1,'Panic') then begin
    ActEcho('@TTY panic, close all ports for security.');
    SystemConsole.TtyCloseAll;
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @tty status       - show status of TTY: listen ports, security shield.');
  Echo(' @tty listen 0     - listen port 0 (use "unix send2crwdaq .." to send).');
  Echo(' @tty close 0      - close  port 0 (use "unix send2crwdaq .." to send).');
  Echo(' @tty listen 54321 - listen TCP port 54321  (use "unix nc .." to send).');
  Echo(' @tty close 54321  - close  TCP port 54321  (use "unix nc .." to send).');
  Echo(' @tty shield ON    - safety shield ON, don`t execute received commands.');
  Echo(' @tty shield OFF   - safety shield OFF,allow execute received commands.');
  Echo(' @tty verbose ON   - switch on  verbose mode, print TCP socket details.');
  Echo(' @tty verbose OFF  - switch off verbose mode, no details on TCP socket.');
  Echo(' @tty panic        - security panic: close all ports, force safe state.');
  Echo('Examples:');
  Echo(' @tty status');
  Echo(' @tty listen 0');
  Echo(' @tty listen 54321');
  Echo(' @tty close 0');
  Echo(' @tty close 54321');
 except
  on E:Exception do BugReport(E,ee,'act_tty');
 end;
end;

procedure do_list_usesfonts(Component:TComponent; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 if Component is TTabSheet       then TStringList(Custom).Add((Component as TTabSheet).Font.Name);
 if Component is TSpinEdit       then TStringList(Custom).Add((Component as TSpinEdit).Font.Name);
 if Component is TPageControl    then TStringList(Custom).Add((Component as TPageControl).Font.Name);
 if Component is TBitBtn         then TStringList(Custom).Add((Component as TBitBtn).Font.Name);
 if Component is TSpeedButton    then TStringList(Custom).Add((Component as TSpeedButton).Font.Name);
 if Component is TRadioButton    then TStringList(Custom).Add((Component as TRadioButton).Font.Name);
 if Component is TRadioGroup     then TStringList(Custom).Add((Component as TRadioGroup).Font.Name);
 if Component is TListBox        then TStringList(Custom).Add((Component as TListBox).Font.Name);
 if Component is TCheckBox       then TStringList(Custom).Add((Component as TCheckBox).Font.Name);
 if Component is TComboBox       then TStringList(Custom).Add((Component as TComboBox).Font.Name);
 if Component is TGroupBox       then TStringList(Custom).Add((Component as TGroupBox).Font.Name);
 if Component is TPaintBox       then TStringList(Custom).Add((Component as TPaintBox).Font.Name);
 if Component is TToolBar        then TStringList(Custom).Add((Component as TToolBar).Font.Name);
 if Component is TStatusBar      then TStringList(Custom).Add((Component as TStatusBar).Font.Name);
 if Component is TStaticText     then TStringList(Custom).Add((Component as TStaticText).Font.Name);
 if Component is TLabel          then TStringList(Custom).Add((Component as TLabel).Font.Name);
 if Component is TPanel          then TStringList(Custom).Add((Component as TPanel).Font.Name);
 if Component is TButton         then TStringList(Custom).Add((Component as TButton).Font.Name);
 if Component is TEdit           then TStringList(Custom).Add((Component as TEdit).Font.Name);
 if Component is TMemo           then TStringList(Custom).Add((Component as TMemo).Font.Name);
 if Component is TForm           then TStringList(Custom).Add((Component as TForm).Font.Name);
 if Component is TComponent then ForEachComponent(Component,do_list_usesfonts,Custom);
 //if Component is TComponent then TStringList(Custom).Add(Component.ClassName);
end;

function act_fonts(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1,w2,w3,w4:LongString; i,c,p:Integer; List:TStringList;
 procedure ScreenFontAssign(Font:TFont; const tail:LongString);
 var i,c,p,h:Integer;
 begin
  if IsNonEmptyStr(tail) then
  if Assigned(Font) then with Font do begin
   h:=StrToIntDef(tail,0); c:=-1; p:=-1;
   for i:=ANSI_CHARSET to OEM_CHARSET do if IsSameText(tail,CharsetToString(i)) then c:=i;
   for i:=DEFAULT_PITCH to VARIABLE_PITCH do if IsSameText(tail,PitchToString(i)) then p:=i;
   if IsSameText(tail,'bolditalic') then Font.Style:=Font.Style+[fsBold]+[fsItalic] else
   if IsSameText(tail,'italic') then Font.Style:=Font.Style-[fsBold]+[fsItalic] else
   if IsSameText(tail,'normal') then Font.Style:=Font.Style-[fsBold]-[fsItalic] else
   if IsSameText(tail,'bold') then Font.Style:=Font.Style+[fsBold] else
   if h<>0 then Height:=h else
   if p>=0 then Pitch:=TFontPitch(p) else
   if c>=0 then Charset:=TFontCharset(c) else
   if SystemFontsFound(Charset,Ord(Pitch),tail)>0 then Name:=tail;
   //if Assigned(Application.MainForm) then Application.MainForm.Invalidate;
  end;
 end;
 function FontsFind(const Alias:LongString):Integer;
 var FontName:LongString;
 begin
  Result:=0;
  FontName:=FindFontNameByAlias(Alias);
  if (FontName<>'') then begin
   ActEcho(FontName);
   Result:=1;
  end else ActEcho('Font '+DoubleAngleQuotedStr(Alias)+' is not found.');
 end;
 function ActEchoLines(s:LongString):Integer;
 begin
  ActEcho(s);
  Result:=ForEachStringLine(s,nil,nil);
 end;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   arg:=Trim(Arg);
   w1:=ExtractWord(1,arg,ScanSpaces);
   w2:=ExtractWord(2,arg,ScanSpaces);
   w3:=ExtractWord(3,arg,ScanSpaces);
   w4:=ExtractWord(4,arg,ScanSpaces);
   if IsSameText(w1,'menu') then begin
    ScreenFontAssign(Screen.MenuFont,Trim(SkipWords(1,arg,ScanSpaces)));
   end else
   if IsSameText(w1,'icon') then begin
    ScreenFontAssign(Screen.IconFont,Trim(SkipWords(1,arg,ScanSpaces)));
   end else
   if IsSameText(w1,'hint') then begin
    ScreenFontAssign(Screen.HintFont,Trim(SkipWords(1,arg,ScanSpaces)));
   end else
   if IsSameText(w1,'embed') then begin
    {$IFDEF WINDOWS}
    if IsSameText(w2,'force') then begin
     FullEmbeddedFontList.FontEmbed;
    end else
    if IsSameText(w2,'smart') then begin
     FullEmbeddedFontList.SmartEmbed;
    end else
    if IsSameText(w2,'save') then begin
     ActEcho(Format('%d font(s) saved.',[FullEmbeddedFontList.FontSave(true,true)]));
    end else
    if IsSameText(w2,'free') then begin
     FullEmbeddedFontList.FontFree;
     FullEmbeddedFontList.DataFree;
    end else
    if IsSameText(w2,'info') then begin
     for i:=0 to FullEmbeddedFontList.Count-1 do
     Echo(Format('%s : %-20s : %s : %s : %d embedded, %d found',[
          FullEmbeddedFontList[i].NickName,
          FullEmbeddedFontList[i].FontName,
          FullEmbeddedFontList[i].FontTarget,
          FullEmbeddedFontList[i].FontSource,
          FullEmbeddedFontList[i].Embedded,
          FullEmbeddedFontList[i].Found]));
    end;
    ActEcho(Format('%d font(s) embedded.',[FullEmbeddedFontList.Embedded]));
    {$ENDIF ~WINDOWS}
    {$IFDEF UNIX}
    Echo(MsgNoSupportOnThisPlatform('@fonts embed'));
    {$ENDIF ~UNIX}
   end else
   if IsSameText(w1,'add') then begin
    {$IFDEF WINDOWS}
    ActEcho(Format('%d',[AddSystemFont(w2,true)]));
    {$ENDIF ~WINDOWS}
    {$IFDEF UNIX}
    ActEcho(MsgNoSupportOnThisPlatform('@fonts embed'));
    {$ENDIF ~UNIX}
   end else
   if IsSameText(w1,'kill') then begin
    {$IFDEF WINDOWS}
    ActEcho(Format('%d',[RemoveSystemFont(w2,true)]));
    {$ENDIF ~WINDOWS}
    {$IFDEF UNIX}
    ActEcho(MsgNoSupportOnThisPlatform('@fonts embed'));
    {$ENDIF ~UNIX}
   end else
   if IsSameText(w1,'path') then begin
    Result:=ActEchoLines(GetSystemFontsPath);
   end else
   if IsSameText(w1,'list') then begin
    if IsSameText(w2,'screen') then begin
     ScreenFontsUpdate;
     Result:=ActEchoLines(Screen.Fonts.Text);
    end else
    if IsSameText(w2,'printer') then begin
     Result:=ActEchoLines(Printer.Fonts.Text);
    end else
    if IsSameText(w2,'uses') then begin
     List:=TStringList.Create;
     try
      List.Duplicates:=dupIgnore; List.Sorted:=true;
      ForEachComponent(Application,do_list_usesfonts,List);
      List.Add(Screen.MenuFont.Name);
      List.Add(Screen.IconFont.Name);
      List.Add(Screen.HintFont.Name);
      Result:=ActEchoLines(List.Text);
     finally
      List.Free;
     end;
    end else
    if IsSameText(w2,'system') then begin
     c:=StrToIntDef(w3,DEFAULT_CHARSET);
     p:=StrToIntDef(w4,DEFAULT_PITCH);
     if c in AvailFontCharsets then
     if p in AvailFontPitches then begin
      Result:=ActEchoLines(TheSystemFonts(c,p));
     end;
    end;
   end else
   if IsSameText(w1,'enum') then begin
    c:=StrToIntDef(w2,DEFAULT_CHARSET);
    p:=StrToIntDef(w3,DEFAULT_PITCH);
    if c in AvailFontCharsets then
    if p in AvailFontPitches then begin
     Result:=ActEchoLines(GetSystemFontsAsText(c,p,SkipWords(3,args,ScanSpaces)));
     Exit;
    end;
    Echo('Invalid charset or pitch.');
   end else
   if IsSameText(w1,'find') then begin
    Result:=FontsFind(Trim(SkipWords(1,arg,ScanSpaces)));
   end;
   Exit;
   i:=0; FakeNOP(i); // To supress compiler hints.
  end;
  Echo('Syntax:');
  Echo(' @fonts list uses       - list all uses fonts.');
  Echo(' @fonts list screen     - list all screen fonts.');
  Echo(' @fonts list printer    - list all printer fonts.');
  Echo(' @fonts list system c p - list system fonts by charset, pitch.');
  Echo(' @fonts enum c p n      - enumerate fonts by charset, pitch, name.');
  Echo(' @fonts find a          - find font name by alias (shortcut name) a.');
  Echo(' @fonts kill f          - kill (unregister) font by file f.');
  Echo(' @fonts add f           - add (register) font by file f.');
  Echo(' @fonts embed info      - print info on embedded fonts.');
  Echo(' @fonts embed force     - embed all available fonts.');
  Echo(' @fonts embed smart     - embed only absent fonts.');
  Echo(' @fonts embed free      - free all embedded fonts.');
  Echo(' @fonts embed save      - save embedded fonts to system path.');
  Echo(' @fonts menu n          - set menu font name/height/charset/pitch/style by name/value n.');
  Echo(' @fonts icon n          - set icon font name/height/charset/pitch/style by name/value n.');
  Echo(' @fonts hint n          - set menu font name/height/charset/pitch/style by name/value n.');
  Echo('Examples:');
  Echo(' @fonts list screen');
  Echo(' @fonts list printer');
  Echo(' @fonts list system 204 1');
  Echo(' @fonts add  c:\Windows\xxx.ttf');
  Echo(' @fonts kill c:\Windows\xxx.ttf');
  Echo(' @fonts enum 204 1');
  Echo(' @fonts enum 204 * pt');
  Echo(' @fonts enum * 1 co');
  Echo(' @fonts find ptmono');
  Echo(' @fonts menu RUSSIAN_CHARSET');
  Echo(' @fonts menu FIXED_PITCH');
  Echo(' @fonts menu PT Mono');
  Echo(' @fonts menu -13');
  Echo(' @fonts menu bold');
  Echo(' @fonts menu italic');
  Echo(' @fonts menu normal');
  Echo(' @fonts menu bolditalic');
 except
  on E:Exception do BugReport(E,ee,'act_fonts');
 end;
end;

function act_silent(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=0;
 if IsMainThread then
 try
  Echo('Syntax:');
  Echo(' @silent cmd - execute cmd command in silent mode.');
  Echo('             - skip input echo, skip output print.');
  Echo('             - @silent depends on @silence mode.');
  Echo('Examples:');
  Echo(' @silent setclockres(1)');
  Echo(' @silent @daq devsend &CronSrv @Warning Hello');
 except
  on E:Exception do BugReport(E,ee,'act_silent');
 end;
end;

function act_silence(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1,w2:LongString; i,o:Integer;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   arg:=Trim(Arg);
   w1:=ExtractWord(1,arg,ScanSpaces);
   w2:=ExtractWord(2,arg,ScanSpaces);
   i:=StrToIntDef(w1,Ord(SilentInp));
   o:=StrToIntDef(w2,Ord(SilentOut));
   SilentInp:=(i<>0); SilentOut:=(o<>0);
   if IsSameText(w1,'*') and IsSameText(w2,'*')
   then Echo(Format('@silence %d %d',[i,o]));
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @silence i o - set mode of @silent command for input and output.');
  Echo('Examples:');
  Echo(' @silence * * - print current @silent i/o modes.');
  Echo(' @silence 0 0 - on @silent cmd : print input echo, print result.');
  Echo(' @silence 1 0 - on @silent cmd : skip  input echo, print result.');
  Echo(' @silence 0 1 - on @silent cmd : print input echo, skip  result.');
  Echo(' @silence 1 1 - on @silent cmd : skip  input echo, skip  result.');
  Echo(' @silence 1 1 - that is default mode of @silent command.');
 except
  on E:Exception do BugReport(E,ee,'act_silence');
 end;
end;

 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
 // @tooltip --deaf 15
 // @tooltip --prefer 1
 // @tooltip --broker stop 200
 // @tooltip --broker start Resource\fpquibrk\fpquibrk.exe Normal tpNormal
 // @tooltip text "Error found" preset stdError delay 15000
 // Note: TooltipList and TooltipDeaf uses to apply "deaf time" as simple antispam against tooltip messages
 // which happens too often. Tooltips should not be very detail or very often to avoid system overload.
 // TooltipList keeps last tooltips and timestamp which increase each second. When timestamp over TooltipDeaf,
 // tooltip deletes from list. Timestamp keeps as Pointer (!) in TooltipList.Objects[..] array, BE CARE!!!
 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
const
 TooltipList     : TStringList     = nil;
 TooltipDeaf     : Cardinal        = 15;
 TooltipTask     : TTask           = nil;
 TooltipText     : TText           = nil;
 TooltipPeriod   : Integer         = 10;
 TooltipTimeOut  : Integer         = 100;
 TooltipFifoSize : Integer         = 1024*32;
 TooltipLogFile  : LongString      = '$CRW_DAQ_VAR_TMP_DIR/@tooltip.log';

procedure TooltipTaskFree(TimeOut:DWORD=MAXDWORD); forward;
procedure TooltipTaskFlush; forward;

procedure TooltipInit;
begin
 if not Assigned(TooltipList) then
 try
  TooltipList:=TStringList.Create;
  TooltipList.Duplicates:=dupIgnore;
  TooltipList.Sorted:=true;
 except
  on E:Exception do BugReport(E,SystemConsole,'TooltipInit');
 end;
end;

procedure TooltipFree;
begin
 TooltipTaskFree;
 Kill(TooltipTask);
 Kill(TooltipText);
 Kill(TooltipList);
 TooltipLogFile:='';
end;

procedure TooltipLog(const s:LongString);
begin
 if (TooltipLogFile<>'') and MaybeEnvStr(TooltipLogFile) then begin
  TooltipLogFile:=UnifyFileAlias(ExpEnv(TooltipLogFile));
  if MaybeEnvStr(TooltipLogFile) then TooltipLogFile:='';
 end;
 if (s<>'') and (TooltipLogFile<>'')
 then SendToMainConsole('@silent @log -e '+TooltipLogFile+' '+s+EOL);
end;

procedure TooltipTaskFree(TimeOut:DWORD=MAXDWORD);
begin
 if Assigned(TooltipTask) then
 try
  if (TimeOut=MAXDWORD) then TimeOut:=TooltipTimeOut;
  TooltipTask.StdInpPipeFifoPutText('@exit'+EOL);
  if TooltipTask.Running(TimeOut)
  then TooltipTask.Terminate(1,0,TimeOut);
  TooltipTaskFlush;
  Kill(TooltipTask);
  Kill(TooltipText);
 except
  on E:Exception do BugReport(E,SystemConsole,'TooltipTaskFree');
 end;
end;

procedure TooltipTaskFlush;
var i:Integer;
begin
 if Assigned(TooltipTask) then
 if Assigned(TooltipText) then
 try
  if TooltipTask.StdOutPipeFifoCount>0 then begin
   TooltipText.Text:=TooltipTask.StdOutPipeFifoGetText;
   for i:=0 to TooltipText.Count-1 do TooltipLog(TooltipText[i]);
   TooltipText.Count:=0;
  end;
 except
  on E:Exception do BugReport(E,SystemConsole,'TooltipTaskFlush');
 end;
end;

procedure TooltipTaskInit(exe:LongString; pr:Integer; tp:TThreadPriority);
var d:LongInt; s:LongString;
begin
 try
  TooltipTaskFree; d:=0; s:='';
  if not Assigned(TooltipText) then TooltipText:=NewText(1024);
  if ReadIniFileLongInt(SysIniFile,SectSystem,'TooltipPeriod%d',d) then TooltipPeriod:=d;
  if ReadIniFileLongInt(SysIniFile,SectSystem,'TooltipTimeOut%d',d) then TooltipTimeOut:=d;
  if ReadIniFileLongInt(SysIniFile,SectSystem,'TooltipFifoSize%d',d) then TooltipFifoSize:=1024*Max(Min(d,1024*64),8);
  if SysGlossary.ReadIniPath(SysIniFile,SectSystem,'TooltipLogFile',HomeDir,s) then s:=SmartFileRef(s,'.log') else s:='';
  if (s<>'') then begin s:=MakeRelativePath(s,ProgName); TooltipLogFile:=s; end;
  if IsEmptyStr(exe) or IsWildCard(exe) then SysGlossary.ReadIniPath(SysIniFile,SectSystem,'TooltipBroker',HomeDir,exe);
  if IsEmptyStr(exe) then begin TooltipLog('ERROR: Tooltip Broker file is empty.'); Exit; end;
  exe:=SmartFileRef(AdaptExeFileName(exe),IfThen(IsWindows,'.exe',''));
  if not FileExists(exe) then begin TooltipLog('ERROR: TooltipBroker file not found: '+exe); Exit; end;
  TooltipTask:=NewTask('',exe,HomeDir,'','',SW_HIDE,TooltipFifoSize,TooltipFifoSize);
  if IsWindows then TooltipTask.CodePage:=GetOEMCP;
  if pr>0 then TooltipTask.ProcessPriority:=pr;
  TooltipTask.ThreadPriority:=tp;
  TooltipTask.StdInpPriority:=tp;
  TooltipTask.StdOutPriority:=tp;
  if TooltipTask.Run
  then TooltipLog(Format('Started %s pid %d',[TooltipTask.ExeName,TooltipTask.Pid]))
  else TooltipLog(Format('Failure %s',       [TooltipTask.ExeName]));
 except
  on E:Exception do BugReport(E,SystemConsole,'TooltipTaskInit');
 end;
end;

procedure TooltipSecondsPoll;                       // Callback on every second
var i:Integer; ts:PtrUInt;
const nPoll:Integer=0;
begin
 if Assigned(TooltipList) then
 try
  for i:=TooltipList.Count-1 downto 0 do begin      // Increment timestamp each second
   ts:=PointerToPtrUInt(TooltipList.Objects[i])+1;  // Uses typecast to store timestamp
   TooltipList.Objects[i]:=PtrUIntToPointer(ts);    // as PtrUInt in Pointer variable
   if (ts>=TooltipDeaf) then TooltipList.Delete(i); // Delete item if time over
  end;
 except
  on E:Exception do BugReport(E,SystemConsole,'TooltipSecondsPoll');
 end;
 if Assigned(TooltipTask) then
 try
  nPoll:=((nPoll+1) mod Max(1,TooltipPeriod));
  if (nPoll=0) then begin
   if not TooltipTask.Running then begin
    TooltipTask.Detach;
    if TooltipTask.Run
    then TooltipLog(Format('Started %s pid %d',[TooltipTask.ExeName,TooltipTask.Pid]))
    else TooltipLog(Format('Failure %s',       [TooltipTask.ExeName]));
   end;
  end;
 except
  on E:Exception do BugReport(E,SystemConsole,'TooltipSecondsPoll');
 end;
 TooltipTaskFlush;
end;

function CallTooltipNotifier(const arg:LongString):Integer;
var tip,path:LongString;
begin
 if Assigned(TooltipTask) then begin
  if TooltipTask.StdInpPipeFifoPutText('@Tooltip='+arg+EOL)
  then Result:=Length('@Tooltip=')+Length(arg)+Length(EOL)
  else Result:=0;
  Exit;
 end;
 Result:=0; tip:=''; path:=GetEnv('PATH');
 if IsEmptyStr(tip) then tip:='fpquitip'; if Length(FileSearch(AdaptExeFileName(tip),path))=0 then tip:='';
 if IsEmptyStr(tip) then tip:='unix';     if Length(FileSearch(AdaptExeFileName(tip),path))=0 then tip:='';
 if (tip<>'') then begin
  if SameText(tip,'unix') then begin
   Result:=SendToMainConsole(Format('@silent @run -hide %s tooltip-notifier %s',[tip,arg])+EOL);
  end else
  if SameText(tip,'fpquitip') then begin
   Result:=SendToMainConsole(Format('@silent @run -hide %s %s',[GetComSpecCmd(tip),arg])+EOL);
  end;
 end;
end;

function act_tooltip(ee:TExpressionEvaluator; const args:LongString):double;
var arg,opt:LongString; i,pr:Integer; tp:TThreadPriority;
begin
 Result:=0;
 if IsMainThread then
 try
  if not Assigned(TooltipList) then TooltipInit;
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   arg:=Trim(Arg);
   opt:=ExtractWord(1,arg,ScanSpaces);
   if IsSameText(opt,'--deaf') then begin
    TooltipDeaf:=StrToIntDef(ExtractWord(2,arg,ScanSpaces),TooltipDeaf);
    if (TooltipDeaf=0) then if Assigned(TooltipList) then TooltipList.Clear;
    Result:=TooltipDeaf;
    Exit;
   end;
   if IsSameText(opt,'--prefer') then begin
    if Str2Int(ExtractWord(2,arg,ScanSpaces),i)
    then PreferTooltip:=(i<>0);
    Result:=Ord(PreferTooltip);
    Exit;
   end;
   if IsSameText(opt,'--broker') then begin
    if IsSameText('stop',ExtractWord(2,arg,ScanSpaces)) then begin
     TooltipTaskFree(StrToIntDef(ExtractWord(3,arg,ScanSpaces),TooltipTimeOut));
     Exit;
    end;
    if IsSameText('start',ExtractWord(2,arg,ScanSpaces)) then begin
     pr:=GetPriorityClassLevel(GetPriorityClassByName(ExtractWord(4,arg,ScanSpaces)));
     if pr=0 then pr:=GetPriorityClassLevel(NORMAL_PRIORITY_CLASS);
     tp:=GetPriorityByName(ExtractWord(5,arg,ScanSpaces));
     TooltipTaskInit(ExtractWord(3,arg,ScanSpaces),pr,tp);
     Result:=TooltipTask.Pid;
     Exit;
    end;
    if IsSameText('status',ExtractWord(2,arg,ScanSpaces)) then begin
     if Assigned(TooltipTask) then begin
      ActEcho(Format('@Tooltip: Broker is ON, PID %d, %s.',[TooltipTask.Pid,ExtractWord(1+Ord(Boolean(TooltipTask.Running)),'Stalled Running',ScanSpaces)]));
      ActEcho(Format('@Tooltip: Exe=%s, Log=%s',[MakeRelativePath(TooltipTask.ExeName,ProgName),TooltipLogFile]));
      Result:=TooltipTask.Pid; if not TooltipTask.Running then Result:=-Result;
     end else ActEcho('@Tooltip: Broker is OFF.');
     Exit;
    end;
    Exit;
   end;
   if Assigned(TooltipList) then if (TooltipList.IndexOf(arg)>=0) then Exit;
   Result:=CallTooltipNotifier(arg);
   if (TooltipDeaf>0) then
   if Assigned(TooltipList) then TooltipList.AddObject(arg,Pointer(0));
   if SysLogNotable(SeverityOfTooltips) and IsNonEmptyStr(arg)
   then SysLogNote(0,SeverityOfTooltips,sdr_System,'@Tooltip '+arg);
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @tooltip            - show help');
  Echo(' @tooltip --deaf n   - set n seconds deaf time (antispam).');
  Echo(' @tooltip --prefer n - if n=1, prefer @tooltip for dialogs');
  Echo(' @tooltip --broker start exe pr tp - start broker process.');
  Echo('                         exe - *.exe path or * for default.');
  Echo('                         pr,tp - process/thread priority.');
  Echo(' @tooltip --broker stop t - stop broker process, timeout t.');
  Echo(' @tooltip --broker status - print status of broker process.');
  Echo(' @tooltip arglist    - show tooltip notifier popup window.');
  Echo('  arglist contains pairs of parameter`s names and values:');
  Echo('  text "..."   - text to show, it''s essential parameter.');
  Echo('  preset xxx   - assign predefined set of parameters xxx:');
  Echo('   stdOk,stdNo,stdHelp,stdStop,stdDeny,stdAbort,stdError,');
  Echo('   stdFails,stdSiren,stdAlarm,stdAlert,stdBreak,stdCancel,');
  Echo('   stdNotify,stdTooltip,stdSuccess,stdWarning,stdQuestion,');
  Echo('   stdException,stdAttention,stdInformation,stdExclamation');
  Echo('  delay d      - show tooltip during d msec, then close');
  Echo('  font "name"  - set font name, like "PT Mono Bold"');
  Echo('  fontSize n   - set font size, pt; default is 16');
  Echo('  textColor c  - set text font color; default is black');
  Echo('  bkColor      - set background color; default is blue');
  Echo('  ico f.ico    - set icon file f.ico');
  Echo('  audio f.wav  - set sound file f.wav');
  Echo('  wav f.wav    - set sound file f.wav');
  Echo('  avi f.wav    - set movie file f.avi');
  Echo('  trans n      - set transparency 0..255');
  Echo('  noDouble n   - 0/1 to no/skip dublicates');
  Echo('  onClick ".." - set command to execute on click');
  Echo('  guid  {...}  - set target window GUID for updateable windows');
  Echo('  sure  n      - set GUID createIfNotVisible flag n=0/1');
  Echo('  delete {..}  - delete window with given GUID');
  Echo('  progress p   - set progress bar with p percents');
  Echo('  run cmd      - set command to run on popup window show');
  Echo('  btnN "Label" - set button N=1..9 label text');
  Echo('  cmdN "cmd"   - set button N=1..9 command on press');
  Echo('  xml   "<..>" - set XML expression to send to FP-QUI');
  Echo('Example:');
  Echo(' @tooltip --broker start * Normal tpNormal');
  Echo(' @tooltip text "Error Message" preset stdError delay 15000');
  Echo(' @tooltip text "Log file" preset stdTooltip btn1 "Show" cmd1 "unix grun wintail %CRW_DAQ_SYS_HOME_DIR%\Temp\@IntegrityEvents.log"');
  Echo(' @tooltip guid {AEF251C7-299F-46B1-9DB9-A695AA9F9BF0} preset stdOk text "Progress 10%" progress 10"');
  Echo('Also try: @run fpquitip --help or @run unix tooltip-notifier --help to get more help');
 except
  on E:Exception do BugReport(E,ee,'act_tooltip');
 end;
end;

 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
 // @log --delete filename
 // @log --event  filename message
 // @log --print  filename message
 // @log --write  filename message
 // @log --view   filename
 // @log --tail   filename
 // @log --lister filename
 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
const
 LogText : TText    = nil;
 LogPoll : TPolling = nil;
 LogMinP : LongInt  = 100;
 LogMode : LongInt  = 0;

procedure LogExecute(aPolling:TPolling; var Terminate:Boolean);
var arg,opt,dst,dsp,dir,msg:LongString; f:Text;
 procedure WriteLine(var f:Text; const dst,msg:LongString);
 begin
  IOResult;
  if IsFileClosed(f) or not IsSameText(TTextRec(f).Name,dst) then begin
   SmartFileClose(f); System.Assign(f,dst);
   if FileExists(dst) then System.Append(f) else System.Rewrite(f);
  end;
  if not IsFileClosed(f) and (IOResult=0) then System.Writeln(f,msg);
  IOResult;
 end;
begin
 if Assigned(LogText) then
 if (LogText.Count>0) then
 try
  AssignNull(f);
  try
   IoResult;
   while (LogText.Count>0) do begin
    arg:=Trim(LogText.GetLn(0));
    if (arg<>'') then begin
     opt:=ExtractWord(1,arg,ScanSpaces);
     if (opt<>'') then begin
      dst:=ExtractWord(2,arg,ScanSpaces);
      if (dst<>'') then begin
       if MaybeEnvStr(dst) then dst:=ExpEnv(dst);
       dst:=SmartFileRef(dst,'.log',ProgName);
       dsp:=FileGuard.ProtectHomeDir(dst);
       if (dsp<>dst) then begin
        dir:=ExtractFileDir(dsp);
        if not DirExists(dir)
        then MkDir(dir);
        dst:=dsp;
       end;
       msg:=SkipWords(2,arg,ScanSpaces);
       if IsSameText(opt,'-p') or IsSameText(opt,'--print') then begin
        WriteLine(f,dst,msg);
       end else
       if IsSameText(opt,'-w') or IsSameText(opt,'--write') then begin
        WriteLine(f,dst,msg);
       end else
       if IsSameText(opt,'-e') or IsSameText(opt,'--event') then begin
        WriteLine(f,dst,StdDateTimePrompt+msg);
       end else
       if IsSameText(opt,'-d') or IsSameText(opt,'--delete') then begin
        SmartFileClose(f); if FileExists(dst) then DeleteFile(dst);
       end;
      end;
     end;
    end;
    IoResult;
    LogText.DelLn(0);
   end;
  finally
   SmartFileClose(f);
  end;
 except
  on E:Exception do BugReport(E,SystemConsole,'LogExecute');
 end;
end;

function LogTextAddLn(const arg:LongString):Integer;
const LastAwake:QWord=0; var tick:QWord;
begin
 Result:=Length(arg);
 LogText.AddLn(arg);
 tick:=GetTickCount64;
 if (tick>=LastAwake+LogMinP) then begin
  LastAwake:=tick;
  LogPoll.Awake;
 end;
end;

procedure LogInit;
var Period:Integer; Priority:TThreadPriority;
begin
 if not Assigned(LogText) then begin
  LogText:=NewText;
  LogText.Master:=@LogText;
 end;
 if not Assigned(LogPoll) then begin
  Period:=1000; Priority:=tpNormal;
  if not ReadIniFilePolling(SysIniFile,SectSystem,'LogPolling',Period,Priority) then begin
   Period:=1000; Priority:=tpNormal;
  end;
  if not ReadIniFileLongInt(SysIniFile,SectSystem,'LogMinPoll%d',LogMinP) then LogMinP:=100;
  LogPoll:=NewPolling(LogExecute,Period,Priority,false,'@Log');
  LogPoll.Master:=@LogPoll;
  logPoll.Enabled:=true;
 end;
end;

procedure LogFree;
begin
 Kill(LogPoll);
 Kill(LogText);
end;

function act_log(ee:TExpressionEvaluator; const args:LongString):double;
var arg,opt,dst:LongString;
begin
 Result:=0;
 if IsMainThread then
 try
  if not Assigned(LogText) or not Assigned(LogPoll) then LogInit;
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   arg:=Trim(Arg);
   if IsOption(arg) then begin
    opt:=ExtractWord(1,arg,ScanSpaces);
    if IsOption(opt,'-p','--print') then begin
     Result:=LogTextAddLn(arg);
     Exit;
    end;
    if IsOption(opt,'-w','--write') then begin
     Result:=LogTextAddLn(arg);
     Exit;
    end;
    if IsOption(opt,'-e','--event') then begin
     Result:=LogTextAddLn(arg);
     Exit;
    end;
    if IsOption(opt,'-d','--delete') then begin
     Result:=LogTextAddLn(arg);
     Exit;
    end;
    if IsOption(opt,'-m','--mode') then begin
     if (WordCount(arg,ScanSpaces)>1)
     then LogMode:=StrToIntDef(ExtractWord(2,arg,ScanSpaces),LogMode);
     Result:=LogMode;
     Exit;
    end;
    dst:=ExtractWord(2,arg,ScanSpaces);
    dst:=SmartFileRef(dst,'.log',ProgName);
    dst:=FileGuard.ProtectHomeDir(dst);
    if IsOption(opt,'-t','--tail') then begin
     Result:=SendToMainConsole(Format('@silent @run wintail "%s"',[dst])+EOL);
     Exit;
    end;
    if IsOption(opt,'-v','--view') then begin
     Result:=SendToMainConsole(Format('@silent @run logview "%s"',[dst])+EOL);
     Exit;
    end;
    if IsOption(opt,'-l','--lister') then begin
     Result:=SendToMainConsole(Format('@silent @run -sw7 unix grun lister "%s"',[dst])+EOL);
     Exit;
    end;
   end else begin
    Result:=Length(arg);
    if (LogMode=0) then ActEcho(arg);
    if HasFlags(LogMode,1) then Echo(arg);
    if HasFlags(LogMode,2) then DebugOut(stdfDebug,StdDateTimePrompt+arg);
    Exit;
   end;
  end;
  Echo('Syntax:');
  Echo(' @log                        - show help');
  Echo(' @log -p                     - synonym of --print.');
  Echo(' @log --print  file.log data - print data string to log file.');
  Echo(' @log -w                     - synonym of --write.');
  Echo(' @log --write  file.log data - write data string to log file.');
  Echo(' @log -e                     - synonym of --event.');
  Echo(' @log --event  file.log data - print date-time and data to log file.');
  Echo(' @log -d                     - synonym of --delete.');
  Echo(' @log --delete file.log      - delete specified log file.');
  Echo(' @log -t                     - synonym of --tail.');
  Echo(' @log --tail   file.log      - view log file with wintail command.');
  Echo(' @log -v                     - synonym of --view.');
  Echo(' @log --view   file.log      - view log file with logview command.');
  Echo(' @log -l                     - synonym of --lister.');
  Echo(' @log --lister file.log      - view log file with lister command.');
  Echo(' @log -m                     - synonym of --mode.');
  Echo(' @log --mode m               - logging mode: 0/1/2=silent/echo/debug.');
  Echo(' @log data                   - print data with current logging mode.');
 except
  on E:Exception do BugReport(E,ee,'act_log');
 end;
end;

 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
 // @syslog severity sender - message
 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
function act_syslog(ee:TExpressionEvaluator; const args:LongString):double;
var sel,nop:Integer; arg,opt,sev,sen,mes:LongString; pct:Boolean;
begin
 Result:=0;
 if IsMainThread then
 try
  pct:=True; nop:=0;
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   while IsOption(arg) do begin
    opt:=ExtractWord(1,arg,PosixBlank);
    arg:=SkipWords(1,arg,PosixBlank); inc(nop);
    if IsOption(opt,'-v') or IsOption(opt,'-var','--var') then begin
     pct:=False;
    end else
    if IsOption(opt,'-t') or IsOption(opt,'-trigger','--trigger') then begin
     sev:=ExtractWord(1,arg,PosixBlank);
     if (sev<>'') then begin
      sel:=SysLog.TriggerLevel;
      sel:=SysLog.StringToSeverity(sev,sel);
      if (SysLog.TriggerLevel<>sel)
      then SysLog.TriggerLevel:=sel;
     end;
     Result:=SysLog.TriggerLevel;
     Exit;
    end else
    if IsOption(opt,'-l') or IsOption(opt,'-list','--list') then begin
     sev:=Trim(SysLog.SeverityList);
     Echo('SysLog Severity Levels:'+EOL+sev);
     Result:=WordCount(sev,EolnDelims);
     Exit;
    end;
   end;
   if pct then begin
    if (nop=0)
    then arg:=args
    else arg:=SkipWords(nop,args,PosixBlank);
   end;
   if SysLog.ParseInputLine(arg,sel,sen,mes) then begin
    pct:=pct and IsLexeme(mes,lex_PctChar);
    if pct then mes:=percent_decode(mes);
    if SysLog.Notable(sel)
    then Result:=SysLog.Note(0,sel,sen,mes);
    Exit;
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @syslog -l    - list severity levels');
  Echo(' @syslog -v    - substitute variables: %var');
  Echo(' @syslog -t    - get severity trigger level');
  Echo(' @syslog -t t  - set severity trigger level (t)');
  Echo(' @syslog SEVERITY SENDER - MESSAGE');
  Echo('         SEVERITY - 0..49 or severity name, see @syslog -s');
  Echo('         SENDER   - string to identify source of the message');
  Echo('         MESSAGE  - string which describes what is happened');
  Echo('Note:');
  Echo(' The MESSAGE assume to be pct-encoded except of -v option.');
  Echo('Example:');
  Echo(' @syslog -t info/notify');
  Echo(' @syslog info/notify demo - Test Message.');
 except
  on E:Exception do BugReport(E,ee,'act_syslog');
 end;
end;

 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
 // @textmetadata -a file
 // @textmetadata -c file
 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
function act_textmetadata(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=0;
 if IsMainThread then
 try
  Result:=TextMetaDataUtility(ExtractWord(1,args,ScanSpaces),
                              ExtractWord(2,args,ScanSpaces),
                              ExtractWord(3,args,ScanSpaces));
 except
  on E:Exception do BugReport(E,ee,'act_textmetadata');
 end;
end;

 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
 // @integrity --add   file.cal
 // @integrity --check file.cal
 // @integrity --clear
 // @integrity --load file [section]
 // @integrity --rule Success:[MetaData]:@CheckSum @silent @echo Success integrity $1
 // @integrity --rule Missing:[MetaData]:@CheckSum @silent @echo Missing integrity $1
 // @integrity --rule Invalid:[MetaData]:@CheckSum @silent @echo Invalid integrity $1
 // @integrity --rule Failure:[MetaData]:@CheckSum @silent @echo Failure integrity $1
 // @integrity Success:[MetaData]:@CheckSum c:\Demo.cfg -> @echo Success integrity c:\Demo.cfg
 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
const
 IntegrityRules        : TText      = nil;
 IntegrityQueue        : TText      = nil;
 IntegrityPoll         : TPolling   = nil;
 IntegrityMinP         : LongInt    = 100;
 IntegrityLastSentMsg  : LongString = '';
 IntegrityLastSentTime : Double     = 0;

procedure IntegrityQueueAddLine(const aLine:LongString);
begin
 if (aLine<>'') then IntegrityQueue.AddLn(aLine);
 IntegrityPoll.Awake;
end;

function IntegrityProcessLine(const args:LongString):Integer;
var arg,w1,w2,msg:LongString;
begin
 Result:=0;
 if (args<>'') then
 try
  arg:=Trim(args);
  if IsNonEmptyStr(arg) then begin
   w1:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(w1) then begin // Options started from -
    if IsOption(w1,'-a','--add') then begin
     w2:=SmartFileRef(ExtractWord(2,arg,ScanSpaces),'',ProgName);
     w2:=UnifyFileAlias(w2,ua_FileDefLow);
     msg:=Trim(SkipWords(2,arg,ScanSpaces));
     Result:=FileAddMetaData(w2,URL_Decode(msg));
     SendToMainConsole(Format('@silent @integrity %s:[MetaData]:@CheckSum %s',
     [UpCaseStr(mde_Message(Round(Result))),MakeRelativePath(Trim(w2),ProgName)])+EOL);
     Exit;
    end;
    if IsOption(w1,'-c','--check') then begin
     w2:=SmartFileRef(ExtractWord(2,arg,ScanSpaces),'',ProgName);
     w2:=UnifyFileAlias(w2,ua_FileDefLow);
     Result:=FileCheckMetaData(w2);
     SendToMainConsole(Format('@silent @integrity %s:[MetaData]:@CheckSum %s',
     [UpCaseStr(mde_Message(Round(Result))),MakeRelativePath(Trim(w2),ProgName)])+EOL);
     Exit;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,SystemConsole,'IntegrityProcessLine');
 end;
end;

procedure IntegrityExecute(aPolling:TPolling; var Terminate:Boolean);
begin
 if Assigned(IntegrityQueue) then
 if (IntegrityQueue.Count>0) then
 try
  while (IntegrityQueue.Count>0) do begin
   IntegrityProcessLine(IntegrityQueue[0]);
   IntegrityQueue.DelLn(0);
   aPolling.WdtReset;
  end;
  if (IntegrityQueue.Count>0) then aPolling.Awake;
 except
  on E:Exception do BugReport(E,SystemConsole,'IntegrityExecute');
 end;
end;

procedure IntegrityInit;
var Period:Integer; Priority:TThreadPriority;
begin
 if not Assigned(IntegrityRules) then begin
  IntegrityRules:=NewText;
  IntegrityRules.Master:=@IntegrityRules;
 end;
 if not Assigned(IntegrityQueue) then begin
  IntegrityQueue:=NewText;
  IntegrityQueue.Master:=@IntegrityQueue;
 end;
 if not Assigned(IntegrityPoll) then begin
  Period:=1000; Priority:=tpNormal;
  if not ReadIniFilePolling(SysIniFile,SectSystem,'IntegrityPolling',Period,Priority) then begin
   Period:=1000; Priority:=tpNormal;
  end;
  if not ReadIniFileLongInt(SysIniFile,SectSystem,'IntegrityMinPoll%d',IntegrityMinP) then IntegrityMinP:=100;
  IntegrityPoll:=NewPolling(IntegrityExecute,Period,Priority,false,'@Integrity');
  IntegrityPoll.Master:=@IntegrityPoll;
  IntegrityPoll.Enabled:=true;
 end;
end;

procedure IntegrityFree;
begin
 Kill(IntegrityPoll);
 Kill(IntegrityQueue);
 Kill(IntegrityRules);
 IntegrityLastSentMsg:='';
end;

function IntegrityRulesAdd(const Args:LongString):Integer;
var i:Integer; Rule,Body:LongString;
begin
 Result:=0;
 if Assigned(IntegrityRules) then
 try
  Rule:=ExtractWord(1,Args,ScanSpaces);
  if (Rule<>'') then begin
   Body:=Trim(SkipWords(1,Args,ScanSpaces));
   if (Body='') then begin
    for i:=IntegrityRules.Count-1 downto 0 do
    if IsSameText(Rule,ExtractWord(1,IntegrityRules[i],ScanSpaces)) then IntegrityRules.DelLn(i);
   end else begin
    IntegrityRules.AddLn(Rule+' '+Body);
   end;
   Result:=Length(Args);
  end;
 except
  on E:Exception do BugReport(E,SystemConsole,'IntegrityRulesAdd');
 end;
end;

function IntegrityRulesRun(const Args:LongString):Integer;
var i,n,k:Integer; Rule,Body:LongString; Cmnd:LongString;
const iarg:array[1..9] of String[2]=('$1', '$2', '$3', '$4', '$5', '$6', '$7', '$8', '$9');
const parg:array[1..9] of String[3]=('$p1','$p2','$p3','$p4','$p5','$p6','$p7','$p8','$p9');
const narg:array[1..9] of String[3]=('$n1','$n2','$n3','$n4','$n5','$n6','$n7','$n8','$n9');
const xarg:array[1..9] of String[3]=('$x1','$x2','$x3','$x4','$x5','$x6','$x7','$x8','$x9');
begin
 Result:=0;
 if Assigned(IntegrityRules) then
 try
  n:=0;
  Rule:=ExtractWord(1,Args,ScanSpaces);
  if (Rule<>'') then begin
   Body:=Trim(SkipWords(1,Args,ScanSpaces));
   for i:=0 to IntegrityRules.Count-1 do
   if IsSameText(Rule,ExtractWord(1,IntegrityRules[i],ScanSpaces)) then begin
    Cmnd:=SkipWords(1,IntegrityRules[i],ScanSpaces);
    if (Pos('$',Cmnd)>0) then begin // $* replacements
     if (Pos('$*',Cmnd)>0) then Cmnd:=StringReplace(Cmnd,'$*',Body,[rfReplaceAll,rfIgnoreCase]);
     for k:=1 to 9 do begin
      if (Pos(iarg[k],Cmnd)>0) then Cmnd:=StringReplace(Cmnd,iarg[k],ExtractWord(k,Body,ScanSpaces),[rfReplaceAll,rfIgnoreCase]);
      if (Pos(parg[k],Cmnd)>0) then Cmnd:=StringReplace(Cmnd,parg[k],ExtractFilePath(ExtractWord(k,Body,ScanSpaces)),[rfReplaceAll,rfIgnoreCase]);
      if (Pos(narg[k],Cmnd)>0) then Cmnd:=StringReplace(Cmnd,narg[k],ExtractFileName(ExtractWord(k,Body,ScanSpaces)),[rfReplaceAll,rfIgnoreCase]);
      if (Pos(xarg[k],Cmnd)>0) then Cmnd:=StringReplace(Cmnd,xarg[k],ExtractFileExt(ExtractWord(k,Body,ScanSpaces)),[rfReplaceAll,rfIgnoreCase]);
     end;
    end;
    Result:=SendToMainConsole(Cmnd+EOL);
    inc(n);
   end;
   if (n=0) and not IsSameText(Rule,'*') then {IntegrityRulesRun('* '+Body)};
  end;
 except
  on E:Exception do BugReport(E,SystemConsole,'IntegrityRulesRun');
 end;
end;

function act_integrity(ee:TExpressionEvaluator; const args:LongString):double;
var arg,msg,w1,w2,w3:LongString; ms:Double; i:Integer; t:TText;
const  DeafTime=1000;
begin
 Result:=0;
 if IsMainThread then
 try
  if not Assigned(IntegrityRules) then IntegrityInit;
  if not Assigned(IntegrityQueue) then IntegrityInit;
  if not Assigned(IntegrityPoll)  then IntegrityInit;
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   arg:=Trim(Arg);
   w1:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(w1) then begin // Options started from -
    if IsOption(w1,'--add-sync') then begin
     Result:=IntegrityProcessLine('--add '+SkipWords(1,arg,ScanSpaces));
     Exit;
    end;
    if IsOption(w1,'--check-sync') then begin
     Result:=IntegrityProcessLine('--check '+SkipWords(1,arg,ScanSpaces));
     Exit;
    end;
    if IsOption(w1,'-a','--add') then begin
     IntegrityQueueAddLine(arg);
     Exit;
    end;
    if IsOption(w1,'-c','--check') then begin
     IntegrityQueueAddLine(arg);
     Exit;
    end;
    if IsOption(w1,'--break') then begin
     IntegrityQueue.Count:=0;
     Exit;
    end;
    if IsOption(w1,'--clear') then begin
     IntegrityRules.Count:=0;
     Exit;
    end;
    if IsOption(w1,'-l','--load') then begin
     w2:=SmartFileRef(ExtractWord(2,arg,ScanSpaces),'',ProgName);
     w3:=UnifySection(ExtractWord(3,arg,ScanSpaces));
     if IsNonEmptyStr(w2) and IsNonEmptyStr(w3) then begin
      t:=ExtractListSection(w2,w3,efConfigNC);
      try
       for i:=0 to t.Count-1 do begin
        msg:=Trim(t[i]);
        if IsSameText(Copy(msg,1,1),'@') then
        Result:=Result+SendToMainConsole(msg+EOL);
       end;
      finally
       Kill(t);
      end;
     end;
     Exit;
    end;
    if IsOption(w1,'-r','--rule') then begin
     msg:=Trim(SkipWords(1,arg,ScanSpaces));
     if Length(msg)=0 then begin
      Echo(IntegrityRules.Text);
      Exit;
     end;
     Result:=IntegrityRulesAdd(msg);
     Exit;
    end;
    Exit;
   end;
   ms:=mSecNow;
   msg:=Trim(arg);
   // Skip integrity message dublicates
   if (IntegrityLastSentMsg<>msg) or (ms-IntegrityLastSentTime>DeafTime) then begin
    SystemConsole.IntegrityEventsLog(msg);
    IntegrityLastSentMsg:=msg;
    IntegrityLastSentTime:=ms;
   end;
   Result:=IntegrityRulesRun(arg);
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @integrity                  - show help');
  Echo(' @integrity -a               - synonym of --add');
  Echo(' @integrity --add file data  - add [MetaData] @CheckSum and data to file.');
  Echo(' @integrity -c               - synonym of --check');
  Echo(' @integrity --check file     - check [MetaData] @CheckSum and send result');
  Echo(' @integrity --add-sync f d   - synchronous version of --add ...');
  Echo(' @integrity --check-sync f   - synchronous version of --check ...');
  Echo(' @integrity --break          - break all pending add/check operations');
  Echo(' @integrity --clear          - clear all rules');
  Echo(' @integrity -r               - synonym of --rule');
  Echo(' @integrity --rule           - view list of integrity notification rules.');
  Echo(' @integrity --rule cmd       - delete rule with given name=cmd with body.');
  Echo(' @integrity --rule cmd arg   - add rule with given name=cmd and body=arg.');
  Echo(' @integrity cmd arg          - call Data Integrity Control Center (DICC).');
  Echo('  DICC will find rule=cmd and put rule`s body=arg to main console to run.');
  Echo('  Also $1..$9, $* macro replacement uses (before body copied to console).');
  Echo('  Also $p1,$n1,$x1 means path,name and extension of 1st argument.');
  Echo('  cmd is command=rule name, like save.crw or Success:[MetaData]:@CheckSum');
  Echo('  arg is arguments, $1,$2..$9 like file name list etc; also $* means all.');
  Echo('Examples:');
  Echo(' @integrity --rule save.crw                           - clear rule save.crw');
  Echo(' @integrity --rule save.crw @echo $x1 file saved: $1  - set new rule command');
  Echo(' @integrity save.crw c:\demo.crw                      - run save.crw rule.');
 except
  on E:Exception do BugReport(E,ee,'act_integrity');
 end;
end;

 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
 // @OnException --raise EDivByZero
 // @OnException Exception «EDivByZero» in CrwDaq.exe PID 4736
 // @OnException --info EDivByZero in CrwDaq.exe at 0022A2DA «Division by zero»
 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
type ETestException=class(Exception);
function act_onexception(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1,w2,tail:LongString; n:Integer;
begin
 Result:=0;
 if IsMainThread then
 try
  n:=0;
  arg:=Trim(args);
  if IsNonEmptyStr(arg) then begin
   w1:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(w1) then begin // Options started from -
    if IsOption(w1,'--raise') then begin
     w2:=ExtractWord(2,arg,ScanSpaces);
     tail:=Trim(SkipWords(2,arg,ScanSpaces));
     if IsSameText(w2,'EAccessViolation') then begin
      n:=PInteger(PtrIntToPointer(-1))^; Echo(IntToStr(n));
     end else
     if IsSameText(w2,'EDivByZero') then begin
      n:=1 div n; Echo(IntToStr(n));
     end else
     if IsSameText(w2,'ENiceException') then begin
      if IsEmptyStr(tail)
      then Raise ENiceException.Create('Nice exception')
      else Raise ENiceException.Create(tail);
     end else
     if IsSameText(w2,'ESoftException') then begin
      if IsEmptyStr(tail)
      then Raise ESoftException.Create('Soft exception')
      else Raise ESoftException.Create(tail);
     end else
     if IsSameText(w2,'EEchoException') then begin
      if IsEmptyStr(tail)
      then Raise EEchoException.Create('Echo exception')
      else Raise EEchoException.Create(tail);
     end else
     if IsEmptyStr(w2)
     then Raise ETestException.Create('Test exception')
     else Raise ETestException.Create(SkipWords(1,arg,ScanSpaces));
     Result:=1;
     Exit;
    end else
    if IsOption(w1,'--info') then begin
     tail:=Trim(SkipWords(1,arg,ScanSpaces));
     Result:=SendToMainConsole('@Silent @Integrity @OnExceptionInfo '+tail+EOL);
     Exit;
    end;
   end else
   Result:=SendToMainConsole('@Silent @Integrity @OnException '+args+EOL);
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @OnException                  - show help');
  Echo(' @OnException Msg              - got exception with message Msg');
  Echo(' @OnException --info Inf       - got more detail exception information Inf');
  Echo(' @OnException --raise E Msg    - raise CPU exception E={EAccessViolation,EDivByZero}');
  Echo('                                 or software {ENiceException, ESoftException,EEchoException}');
  Echo(' @OnException --raise Msg      - raise test exception to check exception handler');
  Echo('Note:');
  Echo(' In current version sends message @Silent @Integrity @OnException Msg');
  Echo(' So «@Integrity --rule @OnException ...» rule should be defined.');
  Echo(' Also sends message @Silent @Integrity @OnExceptionInfo Inf');
  Echo(' So «@Integrity --rule @OnExceptionInfo ...» rule should be defined.');
  Echo('Example:');
  Echo(' @OnException --raise EDivByZero');
  Echo(' @OnException Exception «EDivByZero» in CrwDaq.exe PID 4736');
  Echo(' @OnException --info EDivByZero in CrwDaq.exe at 0022A2DA «Division by zero»');
 except
  on E:Exception do BugReport(E,ee,'act_onexception');
 end;
end;

 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
 // @TestBench ping-pong start 10 tpNormal 10 tpNormal
 // @TestBench ping-pong stop
 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
const ping:TPolling=nil; const pong:TPolling=nil;
procedure PingPong(aPolling:TPolling; var Terminate:Boolean);
begin
 if aPolling=ping then if Assigned(pong) then pong.Awake;
 if aPolling=pong then if Assigned(ping) then ping.Awake;
end;
procedure nopPolling(aPolling:TPolling; var Terminate:Boolean);
begin
end;

procedure testbench_ufn(dl:Double=200); forward;
procedure testbench_draft(arg:LongString); forward;
function memory_manager_self_test:LongString; forward;
function BenchmarkAtomicCounters:LongString; forward;
function BenchmarkCriticalSections:LongString; forward;
function BenchmarkGetListOfProcesses:LongString; forward;

const MultiSmartBitmaps:TObjectStorage=nil;

function act_testbench(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1,w2,w3,w4,w5,w6:LongString; n,i:Integer;
const multi_polling:array of TPolling=nil;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=Trim(args);
  if IsNonEmptyStr(arg) then begin
   w1:=ExtractWord(1,arg,ScanSpaces);
   if IsSameText(w1,'ping-pong') then begin
    w2:=ExtractWord(2,arg,ScanSpaces);
    if IsSameText(w2,'start') then begin
     w3:=ExtractWord(3,arg,ScanSpaces);
     w4:=ExtractWord(4,arg,ScanSpaces);
     w5:=ExtractWord(5,arg,ScanSpaces);
     w6:=ExtractWord(6,arg,ScanSpaces);
     if not Assigned(ping) then begin
      ping:=NewPolling(PingPong,10,tpNormal,false,'@TestBench.Ping');
      ping.Master:=@ping;
     end;
     if not Assigned(pong) then begin
      pong:=NewPolling(PingPong,10,tpNormal,false,'@TestBench.Pong');
      pong.Master:=@pong;
     end;
     ping.Delay:=StrToIntDef(w3,10);
     ping.Priority:=GetPriorityByName(w4);
     pong.Delay:=StrToIntDef(w5,10);
     pong.Priority:=GetPriorityByName(w6);
     ping.Enabled:=true;
     pong.Enabled:=true;
     Result:=Ord(Assigned(ping) and Assigned(pong));
     Exit;
    end else
    if IsSameText(w2,'stop') then begin
     ping.Enabled:=false;
     pong.Enabled:=false;
     Kill(ping);
     Kill(pong);
     Result:=1;
     Exit;
    end;
   end else
   if IsSameText(w1,'multi-polling') then begin
    w2:=ExtractWord(2,arg,ScanSpaces);
    if IsSameText(w2,'start') then begin
     w3:=ExtractWord(3,arg,ScanSpaces);
     w4:=ExtractWord(4,arg,ScanSpaces);
     w5:=ExtractWord(5,arg,ScanSpaces);
     n:=StrToIntDef(w3,1);
     if n>0 then begin
      for i:=0 to Length(multi_polling)-1 do Kill(TObject(multi_polling[i]));
      SetLength(multi_polling,n);
      for i:=0 to Length(multi_polling)-1 do
      multi_polling[i]:=NewPolling(nopPolling,StrToIntDef(w4,10),GetPriorityByName(w5),false,Format('@TestBench.Multi.%d',[i]));
      for i:=0 to Length(multi_polling)-1 do multi_polling[i].Enabled:=true;
     end;
     Result:=Length(multi_polling);
     Exit;
    end else
    if IsSameText(w2,'stop') then begin
     for i:=0 to Length(multi_polling)-1 do Kill(TObject(multi_polling[i]));
     SetLength(multi_polling,0);
     Result:=1;
     Exit;
    end;
   end else
   if IsSameText(w1,'RTC') then begin
    w2:=ExtractWord(2,arg,ScanSpaces);
    n:=StrToIntDef(w2,1000*1000*10);
    Echo(Benchmark_RTC(n,true));
    Echo(Benchmark_RTC(n,false));
   end else
   if IsSameText(w1,'Downsampling') then begin
    Echo('Downsampling test.'+EOL+Test_LTTB(14,6));
   end else
   if IsSameText(w1,'fsm_self_test') then begin
    Echo(fsm_self_test);
   end  else
   if IsSameText(w1,'DemoDebugLog') then begin
    Echo('DebugLog test.');
    Test_DemoDebugLog;
   end else
   if IsSameText(w1,'AdaptFileName') then begin
    Echo(AdaptFileName(SkipWords(1,arg,JustSpaces)));
   end  else
   if IsSameText(w1,'AdaptExeFileName') then begin
    Echo(AdaptExeFileName(SkipWords(1,arg,JustSpaces)));
   end  else
   if IsSameText(w1,'AdaptDllFileName') then begin
    Echo(AdaptDllFileName(SkipWords(1,arg,JustSpaces)));
   end  else
   if IsSameText(w1,'AtomicCounters') then begin
    w2:=ExtractWord(2,arg,ScanSpaces);
    w3:=ExtractWord(3,arg,ScanSpaces);
    Echo(Test_AtomicCounters(StrToIntDef(w2,0),StrToIntDef(w3,0)));
    Echo(BenchmarkAtomicCounters);
    Echo(Test_LockedCounters(StrToIntDef(w2,0),StrToIntDef(w3,0)));
    Echo(BenchmarkCriticalSections);
   end else
   if IsSameText(w1,'GetListOfProcesses') then begin
    n:=ord(glops_prefer_proc_pid_stat);
    i:=ord(glops_prefer_wmi_wql_query);
    w2:=ExtractWord(2,arg,ScanSpaces);
    glops_prefer_proc_pid_stat:=StrToIntDef(w2,n)<>0;
    glops_prefer_wmi_wql_query:=StrToIntDef(w2,i)<>0;
    Echo(BenchmarkGetListOfProcesses);
    glops_prefer_proc_pid_stat:=(n<>0);
    glops_prefer_wmi_wql_query:=(i<>0);
   end else
   if IsSameText(w1,'multi-smartbitmaps') then begin
    w2:=ExtractWord(2,arg,ScanSpaces);
    w3:=ExtractWord(3,arg,ScanSpaces);
    if IsLexeme(w2,lex_digit) then begin
     Kill(MultiSmartBitmaps);
     n:=StrToIntDef(w2,0);
     if (n>0) and FileExists(w3)  then begin
      MultiSmartBitmaps:=NewObjectStorage(True);
      MultiSmartBitmaps.Master:=@MultiSmartBitmaps;
      for i:=0 to n-1 do MultiSmartBitmaps.Add(NewSmartBitmap(w3));
     end;
    end;
    n:=0;
    for i:=0 to MultiSmartBitmaps.Count-1 do begin
     if MultiSmartBitmaps[i] is TSmartBitmap
     then inc(n,BoolToInt((MultiSmartBitmaps[i] as TSmartBitmap).HasBitmap));
    end;
    Echo(Format('MultiSmartBitmaps: %d of %d loaded',[n,MultiSmartBitmaps.Count]));
   end else
   if IsSameText(w1,'memory_manager_self_test') then begin
    Echo(memory_manager_self_test);
   end else
   if IsSameText(w1,'file-ufn') then begin
    w2:=ExtractWord(2,arg,ScanSpaces);
    testbench_ufn(StrToIntDef(w2,200));
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @TestBench                             - show help');
  Echo(' @TestBench ping-pong start t1 p1 t2 p2 - ping-pong test, polling period t, priority p');
  Echo(' @TestBench ping-pong stop              - ping-pong test stop');
  Echo(' @TestBench multi-polling start n t p   - create n polling threads period t, priority p');
  Echo(' @TestBench multi-polling start n t p   - multi-polling test stop');
  Echo(' @TestBench RTC n - benchmark RTC (realtime clocks) with n iterations (default 10000000)');
  Echo(' @TestBench fsm_self_test               - self test of FSM API');
  Echo(' @TestBench DownSampling                - test downsampling algorithm');
  Echo(' @TestBench DemoDebugLog                - test demo for DebugLog');
  Echo(' @TestBench AdaptFileName s             - print AdaptFileName(s)');
  Echo(' @TestBench AdaptExeFileName s          - print AdaptExeFileName(s)');
  Echo(' @TestBench AdaptDllFileName s          - print AdaptDllFileName(s)');
  Echo(' @TestBench AtomicCounters a b          - test AtomicCounters');
  Echo(' @TestBench GetListOfProcesses          - test GetListOfProcesses');
  Echo(' @TestBench Multi-SmartBitmaps n f      - test n-SmartBitmaps with file f');
  Echo(' @TestBench memory_manager_self_test    - Memory Manager test');
  Echo(' @TestBench file-ufn                    - test filename functions');
  Echo('Example:');
  Echo(' @TestBench ping-pong start 10 tpNormal 10 tpNormal');
  Echo(' @TestBench ping-pong stop');
  Echo(' @TestBench multi-polling start 20 10 tpNormal');
  Echo(' @TestBench multi-polling stop');
  testbench_draft(arg);
 except
  on E:Exception do BugReport(E,ee,'act_testbench');
 end;
end;

procedure testbench_ufn(dl:Double=200);
var Files:TStringList; fn,fl,fx,fs,fp:LongString; n,i,nb,ns,hl:SizeInt;
var ms,dt,ct:Double; ob:TObject; bm,bm1,bm2:TBitmap;
 procedure PrintHead(h:LongString);
 begin
  write(Format('%-48s  ',[h]));
 end;
 procedure PrintReport;
 begin
  ct:=1e3*dt/n;
  writeln('Done ',n:10,' calls, ',ns:10,' succeed, ',dt:4:0,' ms elapsed, ',ct:6:3,' mks per call');
 end;
begin
 writeln('Test UFN (unified file name).');
 writeln('System: '+GetOSVersionString);
  fn:=''; fl:=''; fx:=''; fs:=''; fp:='';
 Files:=TStringList.Create;
 try
  Files.Sorted:=True;
  Files.CaseSensitive:=False;
  Files.Duplicates:=dupIgnore;
  Files.Text:=FindAllFilesAsText(HomeDir,AnyFileMask);
  fl:=Files.Text;
  hl:=hashlist_init(False,HashList_DefaultHasher);
  for i:=0 to Files.Count-1 do hashlist_setlink(hl,Files[i],1);
  //
  Files.Clear; Files.UseLocale:=False;
  nb:=100; fn:=SysIniFile; Files.Text:=fn;
  PrintHead('Benchmark TStringList.IndexOf(0,'+IntToStr(Files.Count)+'):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if Files.IndexOf(fn)>=0 then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  Files.Clear; Files.UseLocale:=False;
  nb:=100; fn:=SysIniFile; Files.Text:=fl;
  PrintHead('Benchmark TStringList.IndexOf(0,'+IntToStr(Files.Count)+'):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if Files.IndexOf(fn)>=0 then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  Files.Clear; Files.UseLocale:=True;
  nb:=100; fn:=SysIniFile; Files.Text:=fn;
  PrintHead('Benchmark TStringList.IndexOf(1,'+IntToStr(Files.Count)+'):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if Files.IndexOf(fn)>=0 then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  Files.Clear; Files.UseLocale:=True;
  nb:=100; fn:=SysIniFile; Files.Text:=fl;
  PrintHead('Benchmark TStringList.IndexOf(1,'+IntToStr(Files.Count)+'):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if Files.IndexOf(fn)>=0 then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  PrintHead('Benchmark THashList.IndexOf ('+IntToStr(Files.Count)+' items):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if hashlist_IndexOf(hl,fn)>=0 then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  PrintHead('Benchmark SysUtils.FileExists:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if SysUtils.FileExists(fn) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  PrintHead('Benchmark Fio.FileExists:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if FileExists(fn) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  PrintHead('Benchmark FileIsReadable:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if FileIsReadable(fn) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  PrintHead('Benchmark Fio.FileGetAttr:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if SysUtils.FileGetAttr(fn)<>-1 then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  PrintHead('Benchmark UnifyFileAlias:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if IsNonEmptyStr(UnifyFileAlias(fn)) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fs:=SectSystem; fp:='HelpFile%a';
  PrintHead('Benchmark ReadIniFileAlpha(read):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if ReadIniFileAlpha(fn,fs,fp,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fs:=SectSystem; fp:='-HelpFile%a';
  PrintHead('Benchmark ReadIniFileAlpha(fail):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if not ReadIniFileAlpha(fn,fs,fp,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fx:=StringBuffer(fn);
  PrintHead('Benchmark (S1=S2):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if (fn=fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fx:=StringBuffer(fn);
  PrintHead('Benchmark SysUtils.SameStr(true):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if SysUtils.SameStr(fn,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fx:=StringBuffer(fn);
  PrintHead('Benchmark IsSameStr(true):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if IsSameStr(fn,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fx:=fn; fx:=StringBuffer(fx);
  PrintHead('Benchmark SameText(same):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if SameText(fn,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fx:=fn; fx:=UpCaseStr(fx);
  PrintHead('Benchmark SameText(true):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if SameText(fn,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fx:=fn; fx[Length(fx)]:='x'; fx:=UpCaseStr(fx);
  PrintHead('Benchmark SameText(false):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if not SameText(fn,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fx:=fn; fx:=StringBuffer(fx);
  PrintHead('Benchmark IsSameText(same):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if IsSameText(fn,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fx:=fn; fx:=UpCaseStr(fx);
  PrintHead('Benchmark IsSameText(true):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if IsSameText(fn,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fx:=fn; fx[Length(fx)]:='x'; fx:=UpCaseStr(fx);
  PrintHead('Benchmark IsSameText(false):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if not IsSameText(fn,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fx:=fn; fn:=fn+'да'; fx:=fx+'ДА';
  PrintHead('Benchmark IsSameText(true,UTF8):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if IsSameText(fn,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fx:=fn; fn:=fn+#128; fx:=fx+#128;
  PrintHead('Benchmark IsSameText(true,non-UTF8):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if IsSameText(fn,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile; fx:=fn; fn:=fn+#128; fx:=fx+#129;
  PrintHead('Benchmark IsSameText(false,non-UTF8):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if not IsSameText(fn,fx) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  PrintHead('Benchmark IsLexeme(lex_Ascii,True):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if IsLexeme(fn,lex_Ascii) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile+'Нет';
  PrintHead('Benchmark IsLexeme(lex_Ascii,False):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if not IsLexeme(fn,lex_Ascii) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  PrintHead('Benchmark IsTextAscii(True):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if IsTextAscii(fn) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile+'Нет';
  PrintHead('Benchmark IsTextAscii(False):');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if not IsTextAscii(fn) then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  bm:=TBitmap.Create;
  bm1:=BmpCache.Find(SmartFileRef('resource/daqsite/stdlib/bitmaps/_led10g.bmp'),True);
  bm2:=BmpCache.Find(SmartFileRef('resource/daqsite/stdlib/bitmaps/_led20g.bmp'),True);
  PrintHead('Benchmark TBitmap.Assign:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if Assigned(bm1) then
    if Assigned(bm2) then
    if Assigned(bm) then begin
     if Odd(i)
     then bm.Assign(bm1)
     else bm.Assign(bm2);
     inc(ns);
    end;
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  Kill(bm);
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  bm:=TBitmap.Create;
  bm1:=BmpCache.Find(SmartFileRef('resource/daqsite/stdlib/bitmaps/_led10g.bmp'),True);
  bm2:=BmpCache.Find(SmartFileRef('resource/daqsite/stdlib/bitmaps/_led20g.bmp'),True);
  PrintHead('Benchmark TBitmap.Assign/Clear:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if Assigned(bm1) then
    if Assigned(bm2) then
    if Assigned(bm) then begin
     if Odd(i)
     then bm.Assign(bm1)
     else bm.Assign(bm2);
     //bm.Assign(nil);
     bm.Clear;
     inc(ns);
    end;
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  Kill(bm);
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  PrintHead('Benchmark TBitmap.Create/Free:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    ob:=TBitmap.Create;
    if Assigned(ob) then
    try
     inc(ns);
    finally
     ob.Free;
    end;
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  PrintHead('Benchmark TStringList.Create/Free:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    ob:=TStringList.Create;
    if Assigned(ob) then
    try
     inc(ns);
    finally
     ob.Free;
    end;
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  PrintHead('Benchmark TText.Create/Free:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    ob:=NewText;
    if Assigned(ob) then
    try
     inc(ns);
    finally
     ob.Free;
    end;
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  fx:='~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_200_15_24_white.bmp';
  nb:=10; fn:=SmartFileRef('demo/demo_em2rs/circuits/em2rs_parm.crc');
  PrintHead('Benchmark SmartFileRef:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    fs:=SmartFileRef(fx,'.bmp',fn);
    inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  fx:='[EM2RS.PAR.CFGINPUT7]';
  nb:=1; fn:=SmartFileRef('demo/demo_em2rs/circuits/em2rs_parm.crc');
  PrintHead('Benchmark NewSensorFromCrc/Free:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    ob:=NewCircuitSensorFromCrcFile(fn,fx);
    if Assigned(ob) then
    try
     inc(ns);
    finally
     ob.Free;
    end;
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
  nb:=100; fn:=SysIniFile;
  PrintHead('Benchmark EmptyLoop:');
  n:=0; ms:=mSecNow; dt:=0; ns:=0;
  while (dt<dl) do begin
   for i:=1 to nb do begin
    if (fn<>'') then inc(ns);
    inc(n);
   end;
   dt:=mSecNow-ms;
  end;
  PrintReport;
  //
 finally
  Kill(Files);
  hashlist_free(hl);
 end;
end;

function memory_manager_self_test:LongString;
type PTestData=^TTestData; TTestData=record d:array[0..9] of Double; end;
var px:array[0..9] of Pointer; i,n1,n2,n3,n4,n5,n6,n7,n8,n9,n10,n11:SizeInt;
var s:LongString;
begin
 Result:=''; s:='';
 n1:=GetAllocMemSize;
 for i:=Low(px) to High(px) do px[i]:=GetMem(100);
 n2:=GetAllocMemSize;
 for i:=Low(px) to High(px) do FreeMem(px[i]);
 n3:=GetAllocMemSize;
 for i:=Low(px) to High(px) do px[i]:=TObject.Create;
 n4:=GetAllocMemSize;
 for i:=Low(px) to High(px) do FreeAndNil(px[i]);
 n5:=GetAllocMemSize;
 for i:=Low(px) to High(px) do px[i]:=New(PTestData);
 n6:=GetAllocMemSize;
 for i:=Low(px) to High(px) do Dispose(PTestData(px[i]));
 n7:=GetAllocMemSize;
 for i:=Low(px) to High(px) do s:=s+'0123456789';
 n8:=GetAllocMemSize;
 s:='';
 n9:=GetAllocMemSize;
 for i:=Low(px) to High(px) do px[i]:=AllocMem(100);
 n10:=GetAllocMemSize;
 for i:=Low(px) to High(px) do FreeMem(px[i],100);
 n11:=GetAllocMemSize;
 Result:=Format('Get/FreeMem %d %d %d %d %d',[n1,n2,n2-n1,n3,n3-n1])+EOL
        +Format('New/FreeObj %d %d %d %d %d',[n3,n4,n4-n3,n5,n5-n3])+EOL
        +Format('New/Dispose %d %d %d %d %d',[n5,n6,n6-n5,n7,n7-n5])+EOL
        +Format('Set/LengthS %d %d %d %d %d',[n7,n8,n8-n7,n9,n9-n7])+EOL
        +Format('De/AllocMem %d %d %d %d %d',[n9,n10,n10-n9,n11,n11-n9])+EOL
        +Format('Balance     %d %d %d %d %d %d %d',[n1,n3,n5,n7,n9,n11,n11-n1])+EOL;
end;

procedure testbench_draft(arg:LongString);
var s,w1:LongString; i,n:Integer;  ms:QWord; sb,b:Boolean;
 procedure TestCond(s:LongString; cond,want:Boolean);
 begin
  writeln(Format('%-20s %s %s',[s,BoolToStr(cond,'1','0'),BoolToStr(cond=want,'OK','FAIL')]));
 end;
begin
 Exit;
 Echo('SeverityList:'+EOL+SysLog.SeverityList);
 SysLog.INFO('SysCon','TestBench action.');
 s:='abcd';             TestCond(s,IsLexeme(s,lex_PctChar),false);
 s:='ab%d';             TestCond(s,IsLexeme(s,lex_PctChar),true);
 s:='abcd';             TestCond(s,IsLexeme(s,lex_PctCode),true);
 s:='a%0Fd';            TestCond(s,IsLexeme(s,lex_PctCode),true);
 s:='ad%0F';            TestCond(s,IsLexeme(s,lex_PctCode),true);
 s:='adF%0';            TestCond(s,IsLexeme(s,lex_PctCode),false);
 s:='ad0F%';            TestCond(s,IsLexeme(s,lex_PctCode),false);
 s:='a%xFd';            TestCond(s,IsLexeme(s,lex_PctCode),false);
 s:='a%Fxd';            TestCond(s,IsLexeme(s,lex_PctCode),false);
 s:='a%FAd';            TestCond(s,IsLexeme(s,lex_PctCode),true);
 s:='a%FAd';            TestCond(s,IsLexeme(s,lex_PctData),true);
 s:='a%FAd!';           TestCond(s,IsLexeme(s,lex_PctData),false);
 s:='a%FAd';            TestCond(s,IsLexeme(s,lex_PctNeed),true);
 s:='a/FAd';            TestCond(s,IsLexeme(s,lex_PctNeed),true);
 s:='axFAd';            TestCond(s,IsLexeme(s,lex_PctNeed),false);
 s:='sc://user@host/dbname?a=b&c=d#frag'; TestCond(s,IsLexeme(s,lex_UriAddr),true);
 s:='a._c.de';          TestCond(s,IsLexeme(s,lex_DotName),true);
 s:='a.b..de';          TestCond(s,IsLexeme(s,lex_DotName),false);
 s:='a.bc.de';          TestCond(s,IsLexeme(s,lex_DotName),true);
 s:='192.168.1.1';      TestCond(s,IsLexeme(s,lex_Ip4Addr),true);
 s:='127.0.0.1';        TestCond(s,IsLexeme(s,lex_Ip4Addr),true);
 s:='0.0.0.0';          TestCond(s,IsLexeme(s,lex_Ip4Addr),true);
 s:='255.255.255.255';  TestCond(s,IsLexeme(s,lex_Ip4Addr),true);
 s:='256.256.256.256';  TestCond(s,IsLexeme(s,lex_Ip4Addr),false);
 s:='999.999.999.999';  TestCond(s,IsLexeme(s,lex_Ip4Addr),false);
 s:='1.2.3';            TestCond(s,IsLexeme(s,lex_Ip4Addr),false);
 s:='1.2.3.4';          TestCond(s,IsLexeme(s,lex_Ip4Addr),true);
 s:='0001.2.3.4';       TestCond(s,IsLexeme(s,lex_Ip4Addr),false);
 s:='1.2.3.5.6.7.8.9';  TestCond(s,IsLexeme(s,lex_Ip4Addr),false);
 s:='1.2.3.5.00000001'; TestCond(s,IsLexeme(s,lex_Ip4Addr),false);
 s:='::1';                        TestCond(s,IsLexeme(s,lex_Ip6Addr),true);
 s:='fe80::6990:c8b:b658:d179%8'; TestCond(s,IsLexeme(s,lex_Ip6Addr),true);
 s:='fec0:0:0:ffff::1%1';         TestCond(s,IsLexeme(s,lex_Ip6Addr),true);
 s:='2a:72:b3:c9:53:44';          TestCond(s,IsLexeme(s,lex_Ip6Addr),false);
 s:='fe80::800:27ff:fe00:0';      TestCond(s,IsLexeme(s,lex_Ip6Addr),true);
 w1:='alex_1968$';    Echo('adduser '+w1+' '+BoolToStr(IsLexeme(w1,lex_adduser),'1','0'));
 w1:='alex-_1968';    Echo('adduser '+w1+' '+BoolToStr(IsLexeme(w1,lex_adduser),'1','0'));
 w1:='-alex_1968';    Echo('adduser '+w1+' '+BoolToStr(IsLexeme(w1,lex_adduser),'1','0'));
 w1:='alex_-1968';    Echo('domuser '+w1+' '+BoolToStr(IsLexeme(w1,lex_domuser),'1','0'));
 w1:='alex_1968$';    Echo('domuser '+w1+' '+BoolToStr(IsLexeme(w1,lex_domuser),'1','0'));
 w1:='alex@mail.ru';  Echo('usrhost '+w1+' '+BoolToStr(IsLexeme(w1,lex_usrhost),'1','0'));
 Echo(Test_ForEachQuotedPhrase);
 w1:='123 456 "alpha beta" gamma "delta" ';
 sb:=UsesPhraseIterator;
 for b:=false to true do begin
  UsesPhraseIterator:=b;
  Echo('UsesPhraseIterator = '+IntToStr(Ord(UsesPhraseIterator)));
  n:=PhraseCount(w1,ScanSpaces); Echo(IntToStr(n)+' '+w1);
  Echo('Extract'); for i:=1 to n do Echo(' '+ExtractPhrase(i,w1,ScanSpaces));
  Echo('Skip'); for i:=1 to n do Echo(' '+SkipPhrases(i,w1,ScanSpaces));
  Echo('List'); Echo(PhraseListToTextLines(w1,ScanSpaces));
 end;
 UsesPhraseIterator:=sb;
 Echo(Format('Test %%g pi %g %d %s %g',[pi,1,'ok',pi]));
 Echo(MimeAppsListSearchFiles(EOL));
 Echo(MimeAppsListAsText);
 Echo(GetMimeAppsAllHandlers('text/plain'));
 Echo(GetMimeAppsAllHandlers('text/html'));
 Echo(GetMimeAppsAllHandlers('application/x-diesel'));
 Echo(GetMimeAppsDefHandler('text/plain'));
 Echo(GetMimeAppsDefHandler('text/html'));
 Echo(GetMimeAppsDefHandler('application/x-diesel'));
 Echo(FindMimeAppsDefDesktopFile('text/plain'));
 Echo(FindMimeAppsDefDesktopFile('text/html'));
 Echo(FindMimeAppsDefDesktopFile('application/x-diesel'));
 Echo(FindMimeAppsDefExecEntry('text/plain'));
 Echo(FindMimeAppsDefExecEntry('text/html'));
 Echo(FindMimeAppsDefExecEntry('application/x-diesel'));
 Echo(FindMimeAppsDefExecFile('text/plain'));
 Echo(FindMimeAppsDefExecFile('text/html'));
 Echo(FindMimeAppsDefExecFile('application/x-diesel'));
 Echo(FindMimeAppsDefExecCommand('text/plain','arg1 arg2 arg3'));
 Echo(FindMimeAppsDefExecCommand('text/html','arg1 arg2 arg3'));
 Echo(FindMimeAppsDefExecCommand('application/x-diesel','arg1 arg2 arg3'));
 Echo(FindMimeTypeByFileName('.txt'));
 Echo(FindMimeTypeByFileName('.txt'));
 Echo(GetSystemAssoc('.lm9')+' '+GetSystemAssocExe('.lm9'));
 Echo(iso3166.CsvText+EOL+iso3166.CsvTable+EOL);
 Echo(IntToStr(iso3166.IndexOf('ru')));
 Echo(IntToStr(iso3166.IndexOf('rus')));
 {}
 ms:=GetTickCount64; n:=0; w1:=StringOfChar(' ',1000);
 while (GetTickCount64<ms+1000) do begin
  for i:=1 to 10 do begin Hash32_RS(PChar(w1),Length(w1)); inc(n); end;
 end;
 ms:=GetTickCount64-ms;
 Echo(Format('Hash32_RS takes %7g ns per char',[ms*1e6/(n*Length(w1))]));
 {}
 ms:=GetTickCount64; n:=0; w1:=StringOfChar(' ',1000);
 while (GetTickCount64<ms+1000) do begin
  for i:=1 to 10 do begin Hash64_RS(PChar(w1),Length(w1)); inc(n); end;
 end;
 ms:=GetTickCount64-ms;
 Echo(Format('Hash64_RS takes %7g ns per char',[ms*1e6/(n*Length(w1))]));
 {}
 ms:=dump2q('SignFIFO'); Echo(Format('SignFIFO = %x',[ms]));
end;

{$PUSH}{$I _crw_align_abi.inc}
function BenchmarkAtomicCounters:LongString;
const ic:SizeInt=0; ac:TAtomicCounter=nil;
var t1,t0,dt:QWord; n:SizeInt; p1,p2:Double;
 procedure ido; inline;
 begin
  LockedInc(ic); LockedDec(ic);
 end;
 procedure ado; inline;
 begin
  LockedInc(ac); LockedDec(ac);
 end;
begin
 Result:='Benchmark AtomicCounters:'+EOL;
 LockedInit(ic);
 t0:=GetTickCount64; t1:=t0; n:=0; dt:=1000;
 while (t1-t0<dt) do begin
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  inc(n,100); t1:=GetTickCount64;
 end;
 LockedFree(ic);
 p1:=1e6*(t1-t0)/n;
 Result:=Result+Format('LockedInc/Dec(Counter) takes %d ms per %d ops, performance %1.3f ns per op',[t1-t0,n,p1])+EOL;
 LockedInit(ac);
 t0:=GetTickCount64; t1:=t0; n:=0; dt:=1000;
 while (t1-t0<dt) do begin
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  inc(n,100); t1:=GetTickCount64;
 end;
 LockedFree(ac);
 p2:=1e6*(t1-t0)/n;
 Result:=Result+Format('AtomicInc/Dec(Counter) takes %d ms per %d ops, performance %1.3f ns per op',[t1-t0,n,p2])+EOL;
 Result:=Result+Format('Difference: %1.3f %%',[1e2*abs(p2-p1)/p1])+EOL;
end;
{$POP}

{$PUSH}{$I _crw_align_abi.inc}
function BenchmarkCriticalSections:LongString;
var ic:TRTLCriticalSection=({%H-}); var ac:TSysCriticalSection=nil;
var t1,t0,dt:QWord; n:SizeInt; p1,p2:Double;
 procedure ido; inline;
 begin
  System.EnterCriticalSection(ic); System.LeaveCriticalSection(ic);
 end;
 procedure ado; inline;
 begin
  EnterCriticalSection(ac); EnterCriticalSection(ac);
 end;
begin
 Result:='Benchmark CriticalSections:'+EOL;
 System.InitCriticalSection(ic);
 t0:=GetTickCount64; t1:=t0; n:=0; dt:=1000;
 while (t1-t0<dt) do begin
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  ido;ido;ido;ido;ido;ido;ido;ido;ido;ido;
  inc(n,100); t1:=GetTickCount64;
 end;
 System.DoneCriticalSection(ic);
 p1:=1e6*(t1-t0)/n;
 Result:=Result+Format('RtlEnter/Leave(CritSect) takes %d ms per %d ops, performance %1.3f ns per op',[t1-t0,n,p1])+EOL;
 InitCriticalSection(ac);
 t0:=GetTickCount64; t1:=t0; n:=0; dt:=1000;
 while (t1-t0<dt) do begin
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  ado;ado;ado;ado;ado;ado;ado;ado;ado;ado;
  inc(n,100); t1:=GetTickCount64;
 end;
 DoneCriticalSection(ac);
 p2:=1e6*(t1-t0)/n;
 Result:=Result+Format('SysEnter/Leave(CritSect) takes %d ms per %d ops, performance %1.3f ns per op',[t1-t0,n,p2])+EOL;
 Result:=Result+Format('Difference: %1.3f %%',[1e2*abs(p2-p1)/p1])+EOL;
end;
{$POP}

function BenchmarkGetListOfProcesses:LongString;
var t0,t1,dt,n,np:QWord; p1,p2:Double; ps:LongString;
begin
 Result:='Benchmark GetListOfProcesses:'+EOL;
 t0:=GetTickCount64; t1:=t0; n:=0; dt:=1000;
 while (t1-t0<dt) do begin
  ps:=GetListOfProcesses(0,0,''); ps:=GetListOfProcesses(0,0,'');
  ps:=GetListOfProcesses(0,0,''); ps:=GetListOfProcesses(0,0,'');
  ps:=GetListOfProcesses(0,0,''); ps:=GetListOfProcesses(0,0,'');
  ps:=GetListOfProcesses(0,0,''); ps:=GetListOfProcesses(0,0,'');
  ps:=GetListOfProcesses(0,0,''); ps:=GetListOfProcesses(0,0,'');
  inc(n,10); t1:=GetTickCount64;
 end;
 Result:=Result+ValidateEol(ps,1);
 p1:=1e3*(t1-t0)/n;
 Result:=Result+Format('GetListOfProcesses(all) takes %d ms per %d ops, performance %1.3f mks per op',[t1-t0,n,p1])+EOL;
 np:=ForEachStringLine(ps,nil,nil);
 if (np>0) then Result:=Result+Format('Performance %1.3f mks per process',[p1/np])+EOL;
 t0:=GetTickCount64; t1:=t0; n:=0; dt:=1000;
 while (t1-t0<dt) do begin
  ps:=GetListOfProcesses(GetProcessId,0,''); ps:=GetListOfProcesses(GetProcessId,0,'');
  ps:=GetListOfProcesses(GetProcessId,0,''); ps:=GetListOfProcesses(GetProcessId,0,'');
  ps:=GetListOfProcesses(GetProcessId,0,''); ps:=GetListOfProcesses(GetProcessId,0,'');
  ps:=GetListOfProcesses(GetProcessId,0,''); ps:=GetListOfProcesses(GetProcessId,0,'');
  ps:=GetListOfProcesses(GetProcessId,0,''); ps:=GetListOfProcesses(GetProcessId,0,'');
  inc(n,10); t1:=GetTickCount64;
 end;
 Result:=Result+ValidateEol(ps,1);
 p2:=1e3*(t1-t0)/n;
 Result:=Result+Format('GetListOfProcesses(pid) takes %d ms per %d ops, performance %1.3f mks per op',[t1-t0,n,p2])+EOL;
end;

 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
 // @geo -l       @geo --list
 // @geo -c ru    @geo --code ru
 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
function act_geo(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1,w2:LongString; i:Integer;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=Trim(args);
  if IsNonEmptyStr(arg) then begin
   w1:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(w1,'-l') or IsOption(w1,'-list','--list') then begin
    ActEcho(iso3166.CsvText);
    Result:=iso3166.Count;
    Exit;
   end else
   if IsOption(w1,'-t') or IsOption(w1,'-table','--table') then begin
    ActEcho(iso3166.CsvTable);
    Result:=iso3166.Count;
    Exit;
   end else
   if IsOption(w1,'-c') or IsOption(w1,'-code','--code') then begin
    w2:=ExtractWord(2,arg,ScanSpaces);
    i:=iso3166.IndexOf(w2);
    if (i>=0)
    then ActEcho(iso3166[i].SimpleTable)
    else ActEcho('Not found information on country: '+w2);
    if (i>=0) then Result:=StrToIntDef(iso3166.Items[i].CountryCode,0);
    Exit;
   end;
  end;
  Echo('Syntax:');
  Echo(' @geo        - show help');
  Echo(' @geo -l     - list  of ISO-3166 coutry codes (also -list, --list)');
  Echo(' @geo -t     - table of ISO-3166 coutry codes (also -table, --table)');
  Echo(' @geo -c c   - information on country code c  (also -code, --code)');
  Echo('Example:');
  Echo(' @geo -l');
  Echo(' @geo -c ru');
 except
  on E:Exception do BugReport(E,ee,'act_geo');
 end;
end;

 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
 // @lng -l       @lng --list
 // @lng -c ru    @lng --code ru
 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
function act_lng(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1,w2:LongString; i:Integer;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=Trim(args);
  if IsNonEmptyStr(arg) then begin
   w1:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(w1,'-l') or IsOption(w1,'-list','--list') then begin
    ActEcho(iso639.CsvText);
    Result:=iso639.Count;
    Exit;
   end else
   if IsOption(w1,'-t') or IsOption(w1,'-table','--table') then begin
    ActEcho(iso639.CsvTable);
    Result:=iso639.Count;
    Exit;
   end else
   if IsOption(w1,'-c') or IsOption(w1,'-code','--code') then begin
    w2:=ExtractWord(2,arg,ScanSpaces);
    i:=iso639.IndexOf(w2);
    if (i>=0)
    then ActEcho(iso639[i].SimpleTable)
    else ActEcho('Not found information on language: '+w2);
    if (i>=0) then Result:=StrToIntDef(iso639.Items[i].LanguageCode,0);
    Exit;
   end;
  end;
  Echo('Syntax:');
  Echo(' @lng        - show help');
  Echo(' @lng -l     - list  of ISO-639 language codes (also -list, --list)');
  Echo(' @lng -t     - table of ISO-639 language codes (also -table, --table)');
  Echo(' @lng -c c   - information on language code c  (also -code, --code)');
  Echo('Example:');
  Echo(' @lng -l');
  Echo(' @lng -c ru');
 except
  on E:Exception do BugReport(E,ee,'act_lng');
 end;
end;

 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
 // @file check x.txt
 // @file exist x.txt
 // @file chdir c:\xx
 // @file cd    c:\xx
 // @file size  x.txt
 // @file kill  x.txt
 // @file sects x.ini
 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
function act_file(ee:TExpressionEvaluator; const args:LongString):double;
var w1,wt,arg,s:LongString;
begin
 Result:=0;
 try
  s:='';
  arg:=ee.SmartArgs(args);
  if MaybeEnvStr(arg) then arg:=ExpEnv(arg);
  w1:=ExtractWord(1,arg,ScanSpaces);
  wt:=Trim(SkipWords(1,arg,ScanSpaces));
  // @file Exist filename
  // @file Check filename
  if IsSameText(w1,'Exist') or IsSameText(w1,'Check') then begin
   if (wt='') then Exit;
   Result:=Ord(FileExists(SmartFileRef(wt)));
   if (Result<>0)
   then ActEcho(Format('File exists: %s',[GetRealFilePathName(SmartFileRef(wt))]))
   else ActEcho(Format('File missed: %s',[GetRealFilePathName(SmartFileRef(wt))]));
   Exit;
  end;
  // @file cd dirname
  // @file chdir dirname
  if IsSameText(w1,'cd') or IsSameText(w1,'chdir') then begin
   if IsNonEmptyStr(wt) then Result:=Ord(SetCurrentDir(GetRealFilePathName(SmartFileRef(wt))));
   ActEcho(Format('Current Directory: %s',[GetRealFilePathName(DropBackSlash(GetCurrDir))]));
   Exit;
  end;
  // @file size filename
  if IsSameText(w1,'size') then begin
   if (wt='') then Exit;
   Result:=Ord(GetFileSize64(SmartFileRef(wt)));
   ActEcho(Format('Found size %g of file %s',[Result,GetRealFilePathName(SmartFileRef(wt))]));
   Exit;
  end;
  // @file kill filename
  if IsSameText(w1,'kill') then begin
   if (wt='') then Exit;
   Result:=Ord(FileErase(SmartFileRef(wt)));
   if (Result<>0)
   then ActEcho(Format('File killed: %s',[GetRealFilePathName(SmartFileRef(wt))]))
   else ActEcho(Format('Failed kill: %s',[GetRealFilePathName(SmartFileRef(wt))]));
   Exit;
  end;
  // @file sects filename
  if IsSameText(w1,'sects') then begin
   if (wt='') then Exit;
   wt:=SmartFileRef(wt);
   s:=ExtractListOfSections(wt);
   Result:=WordCount(s,EolnDelims);
   if IsEmptyStr(s) then ActEcho('No sections in file '+wt) else begin
    ActEcho(Format('Section list of file %s:',[wt]));
    ActEcho(Trim(s));
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @file check f   - check file f existance');
  Echo(' @file exist f   - check file f existance');
  Echo(' @file chdir d   - change current directory d');
  Echo(' @file cd d      - change current directory d');
  Echo('                   ~ or ~~ means user or program homedir');
  Echo(' @file size f    - return size of file f or -1');
  Echo(' @file kill f    - kill (delete) file f if one exists');
  Echo(' @file sects f   - print list of sections in file f');
  Echo('Examples:');
  Echo(' @file sects crwdaq.ini');
  Echo(' @file size release.txt');
  Echo(' @file exist release.txt');
  Echo(' @file kill $CRW_DAQ_VAR_TMP_DIR/debug.log');
 except
  on E:Exception do BugReport(E,ee,'act_file');
 end;
end;

 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
 // @note Message with date-time stamp.
 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
function act_note(ee:TExpressionEvaluator; const args:LongString):double;
var arg,w1:LongString; isopt,opt_expand:Boolean;
begin
 Result:=0;
 try
  arg:=ee.SmartArgs(args);
  { Check options }
  isopt:=true; opt_expand:=false;
  while IsOption(arg) and isopt do begin
   w1:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(w1,'-e','--expand') or IsOption(w1,'-exp','-expand') then begin
    opt_expand:=true;
   end else
   if IsOption(w1,'--') then begin
    isopt:=false;
   end;
   arg:=SkipWords(1,arg,ScanSpaces);
  end;
  if opt_expand then arg:=ExpEnv(arg);
  {}
  if IsNonEmptyStr(arg) then begin
   Echo(StdDateTimePrompt+arg);
   Result:=Length(arg);
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @note              - print this help');
  Echo(' @note -e arg       - enable expand environ variables with arg');
  Echo('                      also -exp,-expand,--expand synonyms uses');
  Echo(' @note arg          - print date-time stamp and message (arg)');
  Echo('Example:');
  Echo(' @note -e Start %UserName%@%ComputerName% session.');
 except
  on E:Exception do BugReport(E,ee,'act_note');
 end;
end;

 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
 // @if cond cmd
 // @if cond then cmd
 //////////////////////////////////////////////////////////////////////////////////////////////////////////////
function act_if(ee:TExpressionEvaluator; const args:LongString):double;
var arg:LongString;
begin
 Result:=0;
 try
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @if                - print this help');
  Echo(' @if cond cmd       - equivalent to expression: @if cond then cmd.');
  Echo(' @if cond then cmd  - execute command (cmd) if condition (cond) is true (i.e. non-zero).');
  Echo('                      cond should be simple expression (literal, constant or variable).');
  Echo('Example:');
  Echo(' flag=@file exist Temp\demo.log');
  Echo(' @if flag then @echo File exist');
 except
  on E:Exception do BugReport(E,ee,'act_if');
 end;
end;

 /////////////////////
 // @colorinfo silver
 // @colorinfo $C1A4A4
 /////////////////////
function act_colorinfo(ee:TExpressionEvaluator; const args:LongString):double;
var arg,msg,sn:LongString; cn,cs:TColor;
begin
 Result:=clNone;
 try
  msg:=''; sn:='';
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   if IsOption(arg) then begin
    if IsOption(arg,'-l') or IsOption(arg,'-list','--list') then begin
     Echo(KnownColorsListAsText(1));
     Result:=KnownColorsCount;
    end else
    if IsOption(arg,'-t') or IsOption(arg,'-table','--table') then begin
     if SysGlossary.ReadIniPath(SysIniFile,SectSystem,'crw-daq-colors.htm',HomeDir,sn)
     then Result:=SendToMainConsole('@run WebBrowser '+sn+EOL);
    end else
    Echo('Unknown option '+arg);
   end else begin
    cn:=FindNearestKnownColor(arg);
    cs:=StringToColor(arg);
    sn:=ColorToString(cn);
    if (cn=clNone) then begin
     msg:=arg+' - '+RusEng('неизвестный цвет','unknown color');
     Echo(msg);
     Exit;
    end;
    if (cn<>cs) then begin
     msg:=RusEng('Цвет ','Color ')+arg;
     msg:=msg+' = '+ColorToHexNum(cs)+' = '+ColorToWebNum(cs);
     msg:=msg+' = '+IntToStr(cs)+' = RGB('+ColorToRgbNum(cs)+')';
     msg:=msg+EOL+RusEng('Ближайший известный именованный цвет:','Nearest known named color:')+EOL;
    end;
    msg:=msg+sn;
    msg:=msg+' = '+ColorToHexNum(cn)+' = '+ColorToWebNum(cn);
    msg:=msg+' = '+IntToStr(cn)+' = RGB('+ColorToRgbNum(cn)+')';
    Echo(msg);
    Result:=cn;
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @colorinfo -l      - print list of colors');
  Echo(' @colorinfo --list  - print list of colors');
  Echo(' @colorinfo -t      - open color table HTML');
  Echo(' @colorinfo --table - open color table HTML');
  Echo(' @colorinfo c       - print information on color c (name/code)');
  Echo('Example:');
  Echo(' @colorinfo -l');
  Echo(' @colorinfo -t');
  Echo(' @colorinfo silver');
  Echo(' @colorinfo $C1A4A4');
 except
  on E:Exception do BugReport(E,ee,'act_colorinfo');
 end;
end;

function DlfIterator(n:SizeInt; Line:LongString; Custom:Pointer):Boolean;
begin
 Result:=true;
 if IsLexeme(Line,lex_AtCmnd) then SendToMainConsole(Line+EOL);
end;

 //////////////////////////////////////////////////////
 // @DebugLogCtrl -l n  - list n=1/2=channels/filters
 // @DebugLogCtrl -c n  - clear n=1/2/4=channels/filters/defaults
 // @DebugLogCtrl -i f  - set IncludeFilter f
 // @DebugLogCtrl -e f  - set ExcludeFilter f
 // @DebugLogCtrl -r c  - register channel c
 //////////////////////////////////////////////////////
function act_debuglogctrl(ee:TExpressionEvaluator; const args:LongString):Double;
const DlfSection='[System.StartupScript.DebugLog.Filters.Default]';
var arg,opt,par,buf:LongString; nc:Integer;
begin
 Result:=0;
 try
  arg:=ee.SmartArgs(args);
  opt:=ExtractWord(1,arg,ScanSpaces);
  par:=Trim(SkipWords(1,arg,ScanSpaces));
  if IsOption(opt) then begin
   if IsOption(opt,'-l') or IsOption(opt,'-list','--list') then begin
    Result:=ForEachStringLine(ListDebugLogChannels(1),nil,nil);
    ActEcho(ListDebugLogChannels(StrToIntDef(par,-1)));
   end else
   if IsOption(opt,'-c') or IsOption(opt,'-clear','--clear') then begin
    nc:=StrToIntDef(par,0); if HasFlags(nc,4) then LiftFlags(nc,2);
    Result:=ClearDebugLogChannels(nc);
    if HasFlags(nc,4) then begin
     buf:=ExtractTextSection(SysIniFile,DlfSection,efConfigNC);
     ForEachStringLine(buf,DlfIterator,nil);
    end;
   end else
   if IsOption(opt,'-i') or IsOption(opt,'-include','--include') then begin
    Result:=Ord(DebugLogAddIncludeFilter(par));
   end else
   if IsOption(opt,'-e') or IsOption(opt,'-exclude','--exclude') then begin
    Result:=Ord(DebugLogAddExcludeFilter(par));
   end else
   if IsOption(opt,'-r') or IsOption(opt,'-register','--register') then begin
    Result:=RegisterDebugLogChannel(par);
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @debuglogctrl -l   - list channels/filters of DebugLog');
  Echo(' @debuglogctrl -r c - register channel (c) for DebugLog');
  Echo(' @debuglogctrl -i f - add --include filter for DebugLog');
  Echo(' @debuglogctrl -e f - add --exclude filter for DebugLog');
  Echo(' @debuglogctrl -c 1 - clear channels for DebugLog');
  Echo(' @debuglogctrl -c 2 - clear filters  for DebugLog');
  Echo(' @debuglogctrl -c 4 - clear filters, set defaults');
  Echo('Example:');
  Echo(' @debuglogctrl -l');
  Echo(' @debuglogctrl -r Test');
  Echo(' @debuglogctrl -e Главная Консоль');
  Echo(' @debuglogctrl -i /_DrawView.*Console/i');
  Echo(' @debuglogctrl -c 1');
  Echo(' @debuglogctrl -c 2');
 except
  on E:Exception do BugReport(E,ee,'act_debuglogctrl');
 end;
end;

 ///////////////////////
 // @trayicon -visible
 // @trayicon -visible 0
 // @trayicon -visible 1
 ///////////////////////
function act_trayicon(ee:TExpressionEvaluator; const args:LongString):double;
var arg,opt:LongString; iv:Integer;
begin
 Result:=0;
 try
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   opt:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(opt) then begin // Options started from -
    if IsOption(opt,'-visible','--visible') then begin
     iv:=StrToIntDef(ExtractWord(2,arg,ScanSpaces),-1);
     if not FormCrwDaq.Ok then Exit;
     case iv of
      0:  Result:=Ord(FormCrwDaq.ActionTrayTurnOff.Execute);
      +1: Result:=Ord(FormCrwDaq.ActionTrayTurnOn.Execute);
      -1: Result:=Ord(FormCrwDaq.TrayIconMain.Visible);
     end;
    end;
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @trayicon -visible   - tray icon status');
  Echo(' @trayicon -visible 0 - hide tray icon');
  Echo(' @trayicon -visible 1 - show tray icon');
  Echo('Example:');
  Echo(' @trayicon -visible');
  Echo(' @trayicon -visible 1');
  Echo(' @trayicon -visible 0');
 except
  on E:Exception do BugReport(E,ee,'act_trayicon');
 end;
end;

 ////////////////////
 // @fpcup -help
 // @fpcup -check
 // @fpcup -build lpr
 ////////////////////
function act_fpcup(ee:TExpressionEvaluator; const args:LongString):double;
var arg,opt,lpr:LongString;
begin
 Result:=0;
 try
  arg:=ee.SmartArgs(args);
  if IsNonEmptyStr(arg) then begin
   opt:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(opt) then begin // Options started from -
    if IsOption(opt,'-check','--check') then begin
     if Fpcup.LazarusInstalled then begin
      ActEcho(ExtractPhrase(1,Fpcup.LazarusCmdLine,JustSpaces));
      ActEcho(RusEng('FpcupDeluxe установлен.','FpcupDeluxe installed.'));
      Result:=1;
     end else begin
      ActEcho(RusEng('FpcupDeluxe не найден.','FpcupDeluxe is not found.'));
     end;
     if Assigned(FormCrwDaq) then begin
      FormCrwDaq.ActionHelpFpcup.Enabled:=(Result<>0);
     end;
     Exit;
    end;
    if IsOption(opt,'-build','--build') then begin
     lpr:=Trim(SkipWords(1,arg,JustSpaces));
     Result:=Ord(Fpcup.RunCompile(lpr));
     Exit;
    end;
    if IsOption(opt,'-busy','--busy') then begin
     Result:=Ord(Fpcup.LazBuildBusy);
     if (Result<>0)
     then ActEcho(RusEng('Fpcup ЗАНЯТ.','Fpcup is BUSY.'))
     else ActEcho(RusEng('Fpcup ГОТОВ.','Fpcup is READY.'));
     Exit;
    end;
    if IsOption(opt,'-help','--help') then begin
     Result:=Ord(Fpcup.LazHelp(Fpcup.IniFileLazIndex,''));
     Exit;
    end;
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @fpcup -help      - open lazarus help');
  Echo(' @fpcup -busy      - check if Fpcup busy (running)');
  Echo(' @fpcup -check     - check if Fpcup compiler installed');
  Echo(' @fpcup -build lpr - build project (lpr)');
  Echo('Example:');
  Echo(' @fpcup -help');
  Echo(' @fpcup -busy');
  Echo(' @fpcup -check');
  Echo(' @fpcup -build /opt/demo/project.lpr');
 except
  on E:Exception do BugReport(E,ee,'act_fpcup');
 end;
end;

 /////////////////////////////////////////////////////
 // @wmctrl -query ListWindows
 // @wmctrl -query KillWindow -title "KCalc" -sig TERM
 /////////////////////////////////////////////////////
function act_wmctrl(ee:TExpressionEvaluator; const args:LongString):double;
var arg,opt,ans,query,num,msg:LongString; d,wc,savefmt,savemod:LongInt;
const AnsLim=32;
begin
 Result:=0;
 try
  arg:=ee.SmartArgs(args);
  savefmt:=wmctrl.DefWndFmt;
  savemod:=wmctrl.DefListMode;
  if IsNonEmptyStr(arg) then
  try
   wmctrl.DefWndFmt:=2;
   wmctrl.DefListMode:=5;
   opt:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(opt) then begin // Options started from -
    if IsOption(opt,'-query','--query') then begin
     query:=SkipWords(1,arg,ScanSpaces);
     if IsEmptyStr(query) then Exit(0);
     ans:=wmctrl_query(Trim(query));
     if IsEmptyStr(ans) then begin
      ActEcho(RusEng('Неудачно: ','Failed: ')+query);
      Exit(0);
     end;
     wc:=WordCount(ans,JustSpaces);
     if (wc=1) then num:=Trim(ans) else num:='';
     if StartsText('0x',num) and (Length(num)<=AnsLim)
     and IsLexeme(DropNLeadStr(num,2),lex_xdigit)
     then num:='$'+DropNLeadStr(num,2);
     if (PosEol(ans)>0) then begin // It's list
      Result:=GetTextNumLines(ans);
      ActEcho(TrimRight(ans));
     end else begin
      if (Length(num)<=AnsLim) and Str2Long(num,d) then begin
       msg:=RusEng(' - Успешно: ',' - Succeed: ')+query;
       ActEcho(TrimRight(ans)+msg);
       Result:=d;
      end else begin
       ActEcho(TrimRight(ans));
       Result:=Length(ans);
      end;
     end;
    end;
   end;
   Exit;
  finally
   wmctrl.DefWndFmt:=savefmt;
   wmctrl.DefListMode:=savemod;
  end;
  Echo('Syntax:');
  Echo('@wmctrl -query DisplayInfo -mode m');
  Echo('@wmctrl -query XlibInfo -mode m');
  Echo('@wmctrl -query ListModes');
  Echo('@wmctrl -query ListWindows -pid p -class "c" -title "t" -mode m');
  Echo('@wmctrl -query FindWindow -pid p -class "c" -title "t" -index i');
  Echo('@wmctrl -query WindowManagerName');
  Echo('@wmctrl -query DesktopCount');
  Echo('@wmctrl -query ActiveDesktop');
  Echo('@wmctrl -query SwitchToDesktop -desktop d');
  Echo('@wmctrl -query ActiveWindow');
  Echo('@wmctrl -query ActivateWindow -wnd w -switch-desktop s');
  Echo('@wmctrl -query ShowWindow -wnd w -sw n');
  Echo('@wmctrl -query ShowWindow -wnd w');
  Echo('@wmctrl -query HideWindow -wnd w');
  Echo('@wmctrl -query MinimizeWindow -wnd w');
  Echo('@wmctrl -query MaximizeWindow -wnd w');
  Echo('@wmctrl -query WindowPid -wnd w');
  Echo('@wmctrl -query WindowDesktop -wnd w');
  Echo('@wmctrl -query RestoreWindow -wnd w');
  Echo('@wmctrl -query WindowHost -wnd w');
  Echo('@wmctrl -query WindowTitle -wnd w');
  Echo('@wmctrl -query WindowClass -wnd w');
  Echo('@wmctrl -query WindowBounds -wnd w');
  Echo('@wmctrl -query WindowStateList -wnd w');
  Echo('@wmctrl -query WindowTypeList -wnd w');
  Echo('@wmctrl -query WindowStateFlags -wnd w');
  Echo('@wmctrl -query WindowTypeFlags -wnd w');
  Echo('@wmctrl -query SetWindowStateList -wnd w -wsc c -state-list l');
  Echo('@wmctrl -query SetWindowStateFlags -wnd w -wsc c -state-flags f');
  Echo('@wmctrl -query SetWindowDesktop -wnd w -desktop d');
  Echo('@wmctrl -query SetWindowTitle -wnd w -title "t"');
  Echo('@wmctrl -query SupportedList');
  Echo('@wmctrl -query MoveResizeWindow -wnd w -geom g');
  Echo('@wmctrl -query CloseWindow -wnd w -timeout t');
  Echo('@wmctrl -query KillWindow -wnd w -sig s -timeout t');
  Echo('@wmctrl -query IcccmClass -inst i -class c');
  Echo('@wmctrl -query ListDesktopManagers');
  Echo('@wmctrl -query DesktopManager -index i');
  Echo('@wmctrl -query ListTerminals');
  Echo('@wmctrl -query Terminal -index i');
  Echo('Notes:');
  Echo(' 1) Option -wnd w mostly can be replaced to combination of');
  Echo('    other options: -pid p -class "c" -title "t" to find window.');
  Echo('    This feature uses to work with pid/class/title instead of wnd.');
  Echo(' 2) Short option parameters explanation:');
  Echo('    -mode m           - operation mode, depends on command');
  Echo('    -index i          - index of window to find, 0-based');
  Echo('    -desktop d        - desktop number to switch');
  Echo('    -switch-desktop s - flag (0/1) to switch desktop on activate window');
  Echo('    -wnd w            - target window handle');
  Echo('    -win w            - target window handle');
  Echo('    -pid p            - target window process ID to find');
  Echo('    -class "c"        - target window class name to find');
  Echo('    -title "t"        - target window title text to find');
  Echo('    -sw n             - show window command n where n is 0..10 const:');
  Echo('                        HIDE,SHOWNORMAL,SHOWMINIMIZED,SHOWMAXIMIZED,');
  Echo('                        SHOWNOACTIVATE,SHOW,MINIMIZE,SHOWMINNOACTIVE,');
  Echo('                        SHOWNA,RESTORE,SHOWDEFAULT');
  Echo('    -state-list l     - window state list, subset of items:');
  Echo('                        MODAL,STICKY,MAXIMIZED_VERT,MAXIMIZED_HORZ,');
  Echo('                        SHADED,SKIP_TASKBAR,SKIP_PAGER,HIDDEN,FULLSCREEN,');
  Echo('                        ABOVE,BELOW,DEMANDS_ATTENTION,FOCUSED');
  Echo('    -state-flags f    - bit flags corresponded to state-list');
  Echo('    -wsc c            - window state command 0/1/2=remove/add/toggle');
  Echo('    -geom g           - geometry for MoveResize: g,x,y,w,h');
  Echo('                        g:gravity=0, x,y=left,top, w,h=width,height');
  Echo('    -timeout t        - timeout for close/kill window, ms');
  Echo('    -sig s            - signal number or name to kill window');
  Echo('                        by default uses -sig TERM');
  Echo('    -wndfmt f         - format of wnd to print: f=(0/1/2/3)=(d/u/x/$)');
  Echo('                        0:"%d",  1:"%u",  2:"0x%.8x", 3:"$%.8x"');
  Echo('    -inst i           - instance name for IcccmClass');
  Echo('Example:');
  Echo(' @wmctrl -query ListWindows -mode 1');
  Echo(' @wmctrl -query ShowWindow -title "KCalc"');
  Echo(' @wmctrl -query HideWindow -title "KCalc"');
  Echo(' @wmctrl -query ActivateWindow -title "KCalc"');
  Echo(' wnd=@wmctrl -query FindWindow -title "KCalc"');
  Echo(' @wmctrl -query KillWindow -wnd %wnd -sig TERM');
  Echo(' @wmctrl -query KillWindow -title "KCalc" -sig TERM');
  Echo(' @wmctrl -query FindWindow -title "KCalc" -wndfmt x');
  Echo(' @wmctrl -query MoveResizeWindow -title "KCalc" -geom 0,100,200');
  Echo(' @wmctrl -query SetWindowStateList -title "KCalc" -wsc add -state-list HIDDEN');
 except
  on E:Exception do BugReport(E,ee,'act_wmctrl');
 end;
end;

 //////////////////
 // @sig -code TERM
 //////////////////
function act_sig(ee:TExpressionEvaluator; const args:LongString):double;
var arg,opt,par,lst,cmd:LongString; sig,pid:Integer;
begin
 Result:=0;
 try
  arg:=ee.SmartArgs(args);
  opt:='';par:='';lst:='';cmd:='';
  if IsNonEmptyStr(arg) then begin
   opt:=ExtractWord(1,arg,ScanSpaces);
   par:=ExtractWord(2,arg,ScanSpaces);
   if IsOption(opt) then begin // Options started from -
    if IsOption(opt,'-list','--list') then begin
     lst:=GetListOfUnixSignals(Ord(IsUnix));
     Result:=GetTextNumLines(lst);
     ActEcho(lst);
    end else
    if IsOption(opt,'-code','--code') then begin
     sig:=StringToSigCode(par);
     ActEcho(IntToStr(sig));
     Result:=sig;
    end else
    if IsOption(opt,'-name','--name') then begin
     sig:=StringToSigCode(par);
     ActEcho(SigCodeToString(sig));
     Result:=sig;
    end else
    if IsOption(opt,'-update','--update') then begin
     cmd:=TrimDef(SkipWords(1,arg,ScanSpaces),GetShellCmd('kill -L'));
     if IsUnix and RunCommand(cmd,lst) then
     if (WordCount(lst,ScanSpaces)>=SIGSYS*2) then begin
      Result:=SigListUpdateByTable(lst);
      ActEcho(GetListOfUnixSignals(Ord(IsUnix)));
     end;
    end else
    if IsOption(opt,'-kill','--kill') then begin
     pid:=StrToIntDef(par,0);
     if (pid=0) and (par<>'') then begin
      lst:=GetListOfProcesses(0,0,par,true,glops_FixName);
      if (GetTextNumLines(lst)=0) then begin
       ActEcho(RusEng('Не найден процесс ','Could not find process ')+par);
       Exit;
      end;
      if (GetTextNumLines(lst)>1) then begin
       ActEcho(RusEng('Найдено несколько процессов ','Found several processes ')+par);
       ActEcho(RusEng('Задайте процесс точнее.','Specify process precisely.'));
       ActEcho(lst);
       Exit;
      end;
      pid:=StrToIntDef(ExtractWord(1,lst,ScanSpaces),0);
     end;
     sig:=StringToSigCode('TERM',SIGTERM);
     if IsOption(ExtractWord(3,arg,ScanSpaces),'-s','--s')
     or IsOption(ExtractWord(3,arg,ScanSpaces),'-n','--n')
     or IsOption(ExtractWord(3,arg,ScanSpaces),'-num','--num')
     or IsOption(ExtractWord(3,arg,ScanSpaces),'-sig','--sig')
     or IsOption(ExtractWord(3,arg,ScanSpaces),'-number','--number')
     or IsOption(ExtractWord(3,arg,ScanSpaces),'-signal','--signal')
     then sig:=StringToSigCode(ExtractWord(4,arg,ScanSpaces),SIGTERM);
     if (pid<>0) then begin
      if KillProcess(pid,0,sig) then begin
       ActEcho(RusEng('Послан сигнал ','Sent signal ')+SigCodeToString(sig)
              +RusEng(' процессу PID ',' to process PID ')+IntToStr(pid));
       Result:=pid;
      end else begin
       ActEcho(RusEng('Не могу послать сигнал процессу ',
                      'Could not kill ')+IntToStr(pid));
      end;
     end;
    end;
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @sig -list            - list known signals');
  Echo(' @sig -code sig        - code by signal name');
  Echo(' @sig -name sig        - name by signal code');
  Echo(' @sig -kill pid -s sig - kill pid with signal sig');
  Echo(' @sig -update cmd      - update signals by command cmd');
  Echo(' @sig -update          - update signals by default command');
  Echo('Example:');
  Echo(' @sig -list');
  Echo(' @sig -name 15');
  Echo(' @sig -code TERM');
  Echo(' @sig -update kill -L');
  Echo(' @sig -kill 4321 -s INT');
  Echo(' @sig -kill xterm -s INT');
 except
  on E:Exception do BugReport(E,ee,'act_sig');
 end;
end;

 //////////////////
 // @uart --list
 //////////////////
function act_uart(ee:TExpressionEvaluator; const args:LongString):double;
var arg,opt,msg,avl,act:LongString; txt:TText;
 function FormatAvailComPorts(const avl:LongString):LongString;
 begin
  if IsEmptyStr(avl)
  then Result:=RusEng('Нет наличных COM портов.','No available COM Ports.')
  else Result:=RusEng('Список наличных COM портов: ','List available COM ports: ')+avl;
 end;
 function FormatActiveComPorts(const act:LongString):LongString;
 begin
  if IsEmptyStr(act)
  then Result:=RusEng('Нет активных COM портов.','No active COM Ports.')
  else Result:=RusEng('Список активных COM портов: ','List of active COM Ports: ')+act;
 end;
 function FormatNumComPorts(avl,act:Integer):LongString;
 begin
  Result:=Format(RusEng('COM портов Наличных:%d, Активных:%d',
                        'COM ports Available:%d, Active:%d'),[avl,act]);
 end;
begin
 Result:=0;
 if IsMainThread then
 try
  arg:=ee.SmartArgs(args);
  opt:='';msg:='';avl:='';act:='';
  if IsNonEmptyStr(arg) then begin
   opt:=ExtractWord(1,arg,ScanSpaces);
   if IsOption(opt) then begin // Options started from -
    if IsOption(opt,'-l') or IsOption(opt,'-list','--list') then begin
     act:=uart.ListActivePorts(',');
     avl:=uart.ListAvailPorts(False,',');
     Result:=WordCount(avl,ScanSpaces);
     msg:=FormatNumComPorts(uart.NumAvailPorts,uart.NumActivePorts)+EOL
         +FormatAvailComPorts(avl)+EOL+FormatActiveComPorts(act);
     ActEcho(msg);
    end else
    if IsOption(opt,'-a') or IsOption(opt,'-active','--active') then begin
     act:=uart.ListActivePorts(',');
     Result:=WordCount(act,ScanSpaces);
     msg:=FormatActiveComPorts(act);
     ActEcho(msg);
    end else
    if IsOption(opt,'-m') or IsOption(opt,'-map','--map') then begin
     avl:=uart.ListAvailPorts(False,',');
     Result:=WordCount(avl,ScanSpaces);
     if IsEmptyStr(avl)
     then msg:=FormatAvailComPorts(avl)
     else msg:=uart.ListPortMap;
     ActEcho(msg);
    end else
    if IsOption(opt,'-p') or IsOption(opt,'-prop','--prop') then begin
     act:=uart.ListActivePorts(',');
     Result:=WordCount(act,ScanSpaces);
     txt:=uart.GetProperties(NewText);
     try
      if IsEmptyStr(act)
      then msg:=FormatActiveComPorts(act)
      else msg:=txt.Text;
     finally
      Kill(txt);
     end;
     ActEcho(msg);
    end else
    if IsOption(opt,'-u') or IsOption(opt,'-update','--update') then begin
     if uart.CanUpdatePorts then begin;
      avl:=uart.ListAvailPorts(True,',');
      Result:=WordCount(avl,ScanSpaces);
      msg:=FormatAvailComPorts(avl);
      ActEcho(msg);
     end else begin
      act:=uart.ListActivePorts(',');
      msg:=RusEng('Не могу обновить: заблокирован.',
                  'Could not update: UART blocked.')+EOL
          +FormatActiveComPorts(act);
      ActEcho(msg);
     end;
    end;
   end;
   Exit;
  end;
  Echo('Syntax:');
  Echo(' @uart -map            - print COM ports mapping');
  Echo(' @uart -list           - list available COM ports');
  Echo(' @uart -prop           - print COM ports property');
  Echo(' @uart -active         - list of active COM ports');
  Echo(' @uart -update         - update list of COM ports');
  Echo(' @uart -m,-l,-p,-a,-u  - same as -map,-list,-prop,-active,-update');
  Echo('Example:');
  Echo(' @uart -m');
  Echo(' @uart -list');
  Echo(' @uart -update');
 except
  on E:Exception do BugReport(E,ee,'act_uart');
 end;
end;

 {
 *******************************************************************************
 Функции для регистрации в ExpressionEvaluator
 *******************************************************************************
 }
procedure nargcheck(ee:TExpressionEvaluator; i,j:LongInt);
begin
 if i<>j then begin
  Error(RusEng('Неверное число аргументов функции!','Invalid function argument number!'));
  ee.RuntimeError:=ee_User;
 end;
end;

function fun_detectwine(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 nargcheck(ee,narg,0);
 Result:=detect_wine_number;
end;

function fun_detectvbox(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 nargcheck(ee,narg,0);
 Result:=detect_vbox_number;
end;

function fun_spincountsys(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 nargcheck(ee,narg,0);
 Result:=SystemSpinlockCount;
end;

function fun_spincountdef(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 nargcheck(ee,narg,1);
 Result:=DefaultSpinlockCount;
 DefaultSpinLockCount:=Round(x[0]);
end;

function fun_spincounttag(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 nargcheck(ee,narg,1);
 Result:=SetTagsSpinlockCount(Round(x[0]));
end;

function fun_enablefileguard(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 nargcheck(ee,narg,1);
 Result:=Ord(FileGuard.Enabled);
 if IsNanOrInf(x[0]) then Exit;
 FileGuard.Enabled:=(x[0]<>0);
end;

function fun_enableeditdaqsite(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 nargcheck(ee,narg,1);
 Result:=Ord(TProgramDevice.EnableEditDaqSite);
 if IsNanOrInf(x[0]) then Exit;
 TProgramDevice.EnableEditDaqSite:=(x[0]<>0);
end;

function fun_getfavoriteansicp(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 nargcheck(ee,narg,0);
 Result:=TheFavoriteAnsiCodePage;
end;

function fun_setfavoriteansicp(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 nargcheck(ee,narg,1);
 TheFavoriteAnsiCodePage:=Round(x[0]);
 Result:=TheFavoriteAnsiCodePage;
end;

{$IFDEF Poligon}
function fun_testing(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 nargcheck(ee,narg,1);
 Result:=x[0];
 testpoligon(round(x[0]));
end;
{$ENDIF Poligon}

 {
 *******************************************************************************
 TSystemConsole
 *******************************************************************************
 }
constructor TSystemConsole.Create;
begin
 inherited Create;
 myForm:=nil;
 myEchoFile:='';
 myFileLimit:=32;
 myLines:=NewText;
 myLines.Master:=@myLines;
 myTtyPort0:=0;
 myTtyLatch:=NewLatch;
 myTtyLatch.Master:=@myTtyLatch;
 myTtyPorts:=NewObjectStorage(True,0,DefaultTtyPorts);
 myTtyPorts.Master:=@myTtyPorts;
 myTtyVerbose:=False;
 myTtyShield:=False;
 myIntegrityFile:='';
 myDelayedEe:=NewExpressionEvaluator;
 myDelayedEe.Master:=@myDelayedEe;
 myDelayedFlag:=false;
end;

destructor TSystemConsole.Destroy;
begin
 TtyCloseAll;
 Close(False);
 Kill(myLines);
 Kill(myTtyPorts);
 Kill(myTtyLatch);
 myIntegrityFile:='';
 myEchoFile:='';
 Kill(myDelayedEe);
 inherited Destroy;
end;

function  TSystemConsole.ShouldBeDelayed:Boolean;
begin
 Result:=false;
 if (Self=nil) then Exit;
 Result:=not myDelayedFlag;
 myDelayedFlag:=true;
end;

function  TSystemConsole.DelayedEvaluate(const Expression : LongString;
                                           var Answer     : Double;
                                           var Comment    : LongString) : Integer;
begin
 Answer:=0;
 Comment:='';
 if Assigned(Self) then
 try
  myDelayedEe.Lock;
  try
   Result:=myDelayedEe.EvaluateLine(PChar(Expression));
   if Result=ee_Ok then begin
    if myDelayedEe.MayPrint then Comment:=Format('%g',[myDelayedEe.Answer]);
    Answer:=myDelayedEe.Answer;
   end else begin
    Comment:=ee_ErrorMessage(myDelayedEe.Status)+'.'+EOL+
             StrPas(myDelayedEe.Buffer)+EOL+
             CharStr(myDelayedEe.ErrorPos+1)+'^'+EOL+
             CharStr(myDelayedEe.ErrorPos+1-Length(myDelayedEe.ErrorToken))+StrPas(myDelayedEe.ErrorToken);
   end;
  finally
   myDelayedEe.Unlock;
  end;
 except
  on E:Exception do begin
   BugReport(E,Self);
   Result:=ee_Exception;
   Comment:=ee_ErrorMessage(Result);
  end;
 end else begin
  Result:=ee_NilRef;
  Comment:=ee_ErrorMessage(Result);
 end;
end;

function  TSystemConsole.GetForm:TFormConsoleWindow;
begin
 if Assigned(Self) then Result:=myForm else Result:=nil;
end;

function  TSystemConsole.GetEchoFile:LongString;
begin
 if Assigned(Self) then Result:=myEchoFile else Result:='';
end;

procedure TSystemConsole.SetEchoFile(const aEchoFile:LongString);
begin
 if Assigned(Self) then myEchoFile:=Trim(aEchoFile);
end;

function  TSystemConsole.GetFileLimit:LongInt;
begin
 if Assigned(Self) then Result:=myFileLimit else Result:=0;
end;

procedure TSystemConsole.SetFileLimit(const aFileLimit:LongInt);
begin
 if Assigned(Self) then myFileLimit:=Max(0,aFileLimit);
end;

procedure TtyReporter(Pipe:TSocketPipe; When:Double; What:PChar; Code:Integer);
begin
 if SystemConsole.TtyVerbose then
 Echo(StdDateTimePrompt(When)+SocketErrorReport(What,Code));
end;

function TSystemConsole.TtyFind(aPort:Integer):TTcpServer;
var i:Integer; tcp:TTcpServer;
begin
 Result:=nil;
 if Assigned(Self) then
 if PortInRange(aPort,TcpMinPort,TcpMaxPort) then
 try
  myTtyLatch.Lock;
  try
   for i:=0 to myTtyPorts.Count-1 do begin
    TObject(tcp):=myTtyPorts[i];
    if TObject(tcp) is TTcpServer then
    if tcp.Port=aPort then begin
     Result:=tcp;
     Break;
    end;
   end;
  finally
   myTtyLatch.UnLock;
  end;
 except
  on E:Exception do BugReport(E,Self,'TtyFind');
 end;
end;

function  TSystemConsole.GetTtyVerbose:Boolean;
begin
 if Assigned(Self) then Result:=myTtyVerbose else Result:=False;
end;

procedure  TSystemConsole.SetTtyVerbose(aVerbose:Boolean);
begin
 if Assigned(Self) then myTtyVerbose:=aVerbose;
end;

function  TSystemConsole.GetTtyShield:Boolean;
begin
 if Assigned(Self) then Result:=myTtyShield else Result:=False;
end;

procedure  TSystemConsole.SetTtyShield(aShield:Boolean);
begin
 if Assigned(Self) then myTtyShield:=aShield;
end;

function  TSystemConsole.TtyListenCount(aPort:Integer):LongInt;
var tcp:TTcpServer;
begin
 Result:=0;
 if Assigned(Self) then
 try
  myTtyLatch.Lock;
  try
   if aPort=0 then begin
    Result:=myTtyPort0;
   end else begin
    tcp:=TtyFind(aPort);
    if Assigned(tcp) then Result:=Round(tcp.Polling.LinkParam[TcpParamN]);
   end;
  finally
   myTtyLatch.UnLock;
  end;
 except
  on E:Exception do BugReport(E,Self,'TtyListenCount');
 end;
end;

function TSystemConsole.TtyListen(aPort:Integer):LongInt;
var tcp:TTcpServer;
begin
 Result:=0;
 if Assigned(Self) then
 try
  myTtyLatch.Lock;
  try
   if aPort=0 then begin
    Inc(myTtyPort0);
    Result:=myTtyPort0;
   end else begin
    tcp:=TtyFind(aPort);
    if not Assigned(tcp) then
    if PortInRange(aPort,TcpMinPort,TcpMaxPort) then begin
     tcp:=NewTcpServer(aPort,DefaultTtyPipes,TtyReporter);
     myTtyPorts.Add(tcp);
    end;
    if Assigned(tcp) then begin
     tcp.Polling.LinkParam[TcpParamN]:=tcp.Polling.LinkParam[TcpParamN]+1;
     Result:=Round(tcp.Polling.LinkParam[TcpParamN]);
    end;
   end;
  finally
   myTtyLatch.UnLock;
  end;
 except
  on E:Exception do BugReport(E,Self,'TtyListen');
 end;
end;

function TSystemConsole.TtyClose(aPort:Integer):LongInt;
var tcp:TTcpServer;
begin
 Result:=0;
 if Assigned(Self) then
 try
  myTtyLatch.Lock;
  try
   if aPort=0 then begin
    Result:=myTtyPort0;
    if Result>0 then Dec(Result);
    myTtyPort0:=Result;
   end else begin
    tcp:=TtyFind(aPort);
    if Assigned(tcp) then begin
     Result:=Round(tcp.Polling.LinkParam[TcpParamN]);
     if Result>0 then Dec(Result);
     tcp.Polling.LinkParam[TcpParamN]:=Result;
     if Result=0 then myTtyPorts.Remove(tcp);
    end;
   end;
  finally
   myTtyLatch.UnLock;
  end;
 except
  on E:Exception do BugReport(E,Self,'TtyClose');
 end;
end;

procedure TSystemConsole.TtyCloseAll;
begin
 if Assigned(Self) then
 try
  myTtyLatch.Lock;
  try
   myTtyPort0:=0;
   myTtyPorts.Count:=0;
  finally
   myTtyLatch.UnLock;
  end;
 except
  on E:Exception do BugReport(E,Self,'TtyCloseAll');
 end;
end;

function TSystemConsole.TtyPortList:LongString;
var i:Integer; tcp:TTcpServer;
begin
 Result:='';
 if Assigned(Self) then
 try
  myTtyLatch.Lock;
  try
   if myTtyPort0>0 then Result:='0'+EOL;
   for i:=0 to myTtyPorts.Count-1 do begin
    TObject(tcp):=myTtyPorts[i];                                       
    if TObject(tcp) is TTcpServer then Result:=Result+IntToStr(tcp.Port)+EOL;
   end;
  finally
   myTtyLatch.UnLock;
  end;
 except
  on E:Exception do BugReport(E,Self,'TtyPortList');
 end;
end;

procedure TSystemConsole.TtyPolling;
var i,j:Integer; tcp:TTcpServer; pip:TSocketPipe; data:LongString;
begin
 if Assigned(Self) then
 try
  myTtyLatch.Lock;
  try
   for i:=0 to myTtyPorts.Count-1 do begin
    TObject(tcp):=myTtyPorts[i];
    if TObject(tcp) is TTcpServer then begin
     for j:=0 to tcp.Count-1 do begin
      pip:=tcp.Pipes[j];
      if pip.IsStream then begin
       data:=pip.RxFifo.GetText;
       if Length(data)>0 then data:=ValidateEOL(data,1);
       if not TtyShield then Form.InpFifo.PutText(data);
      end;
     end;
    end;
   end;
  finally
   myTtyLatch.UnLock;
  end;
 except
  on E:Exception do BugReport(E,Self,'TtyPolling');
 end;
end;

function TSystemConsole.IntegrityEventsLog(const aLine:LongString):Integer;
begin
 Result:=0;
 if Assigned(Self) then
 try
  if (myIntegrityFile<>'') then if IsNonEmptyStr(aLine) then
  Result:=SendToMainConsole(Format('@silent @log --event %s %s',[myIntegrityFile,Trim(aLine)])+EOL);
 except
  on E:Exception do BugReport(E,Self,'IntegrityEventsLog');
 end;
end;

procedure SystemConsoleTtyPolling;
begin
 try
  SystemConsole.TtyPolling;
 except
  on E:Exception do BugReport(E,SystemConsole,'TtyPolling');
 end;
end;

procedure SystemConsoleAsync;
begin
 try
  if (SystemCalculator.Fifo.Count>0) then
  SystemConsole.Form.InpFifo.PutText(SystemCalculator.Fifo.GetText);
 except
  on E:Exception do BugReport(E,SystemConsole,'SystemConsoleAsync');
 end;
end;

procedure DropEmptyLines(List:TStrings);
var i:Integer;
begin
 if Assigned(List) then
 try
  for i:=List.Count-1 downto 0 do
  if IsEmptyStr(List[i]) then List.Delete(i);
 except
  on E:Exception do BugReport(E,nil,'DropEmptyLines');
 end;
end;

procedure TSystemConsole.Open;
var M,L:LongInt; s:LongString;
begin
 if Assigned(Self) then
 try
  L:=0; M:=0; s:='';
  if SystemCalculator.Ok then begin
   RegisterActions;
   RegisterFunctions;
   if SysGlossary.ReadIniPath(SysIniFile,SectSystem,'ConsoleConfig',HomeDir,s)
   then SystemCalculator.RestoreVars(s,'[ConsoleVariableList]');
   SystemCalculator.Eval('_Crw_Force_Exit_=0');
   SystemCalculator.Eval('_Daq_Force_Exit_=0');
   SystemCalculator.Eval('_Daq_Force_Stop_=0');
   SystemCalculator.Eval('_Daq_Force_Start_=0');
  end;
  if not myForm.Ok then begin
   if ReadIniFileLongInt(SysIniFile,SectSystem,'ConsoleEchoMode%d',M) and (M in [1..2]) and
      SysGlossary.ReadIniPath(SysIniFile,SectSystem,'ConsoleEchoFile',HomeDir,s)
   then begin
    if M=2 then FileErase(s);
    EchoFile:=s;
    if ReadIniFileLongInt(SysIniFile,SectSystem,'ConsoleFileLimit%d',L) and (L>0)
    then FileLimit:=L;
   end else EchoFile:='';
   myForm:=NewConsoleWindow(GetMainConsoleCaption,
                            StdInputFifo,  false, SystemInputFilter,
                            StdOutputFifo, false, SystemOutputFilter);
   myForm.Master:=@myForm;
   myForm.CloseAction:=caMinimize;
   myForm.GuardInput:=ga_Root;
   myForm.Width:=650;
   myForm.Height:=350;
   myForm.Top:=0;
   myForm.Left:=(Application.MainForm.ClientWidth-Form.Width) div 2;
   myForm.AddonSdiFlags(sf_SdiSysCalc);
   myForm.DefaultMonitor:=dmMainForm;
   myForm.StartMonitoring;
   myForm.PutText(RusEng('Нажмите F1 или введите @help для получения справки'+EOL,
                         'Press F1 or enter @help to get help information'+EOL));
   if SysGlossary.ReadIniPath(SysIniFile,SectSystem,'ConsoleConfig',HomeDir,s)
   then myForm.ComboBoxInput.Items.Text:=ExtractTextSection(s,'[ConsoleInputHistoryList]',efAsIs);
   DropEmptyLines(myForm.ComboBoxInput.Items);
  end;
  if SysGlossary.ReadIniPath(SysIniFile,SectSystem,'IntegrityEventsLog',HomeDir,s)
  then myIntegrityFile:=MakeRelativePath(DefaultExtension(Trim(s),'.log'),ProgName)
  else myIntegrityFile:='';
  Tick55Actions.Add(SystemConsoleAsync);
  Tick55Actions.Add(SystemConsoleTtyPolling);
  SecondActions.Add(TooltipSecondsPoll);
  //IntegrityInit;
  //TooltipInit;
  //LogInit;
 except
  on E:Exception do BugReport(E,Self,'Open');
 end;
end;

procedure TSystemConsole.Close(NeedSave:Boolean);
var s:LongString; F:Text; IOR:Integer;
begin
 if Assigned(Self) then
 try
  s:='';
  if NeedSave then
  if SystemCalculator.Ok then begin
   if SysGlossary.ReadIniPath(SysIniFile,SectSystem,'ConsoleConfig',HomeDir,s)
   then SystemCalculator.SaveVars(s,'[ConsoleVariableList]');
  end;
  if Form.Ok then begin
   Form.StopMonitoring;
   if NeedSave then
   if SysGlossary.ReadIniPath(SysIniFile,SectSystem,'ConsoleConfig',HomeDir,s) then begin
    IOR:=IOResult;
    System.Assign(F,s);
    if FileExists(s) then System.Append(F) else System.Rewrite(F);
    try
     System.Writeln(F,'[ConsoleInputHistoryList]',EOL,Form.ComboBoxInput.Items.Text);
    finally
     System.Close(F);
     SetInOutRes(IOR);
    end;
   end;
  end;
  Kill(myForm);
  EchoFile:='';
  Tick55Actions.Remove(SystemConsoleAsync);
  Tick55Actions.Remove(SystemConsoleTtyPolling);
  SecondActions.Remove(TooltipSecondsPoll);
  IntegrityFree;
  TooltipFree;
  LogFree;
 except
  on E:Exception do BugReport(E,Self,'Close');
 end;
end;

procedure TSystemConsole.GoHome;
const DefSize:TPoint2I=(x:0;y:0);
const DefPos:TPoint2I=(x:0;y:0);
var act:LongString;
begin
 if Assigned(Self) then
 try
  act:='';
  if IsMainThread then
  if Form.Ok then begin
   Form.AdjustDesktopByForm(ApplicationMainForm);
   if not Daq.Timer.isStart then begin
    Form.IsFixedTop:=false;
    Form.IsFixedLeft:=false;
    Form.IsFixedWidth:=false;
    Form.IsFixedHeight:=false;
   end;
   if ReadIniFileString(SysIniFile,SectSystem(1),'Action:FormConsoleWindow.GoHome%s',act,efConfigNC,svConfigNC) then begin
    if IsLexeme(act,lex_AtCmnd) then begin
     SendToMainConsole('@silent '+act+EOL);
     Exit;
    end;
   end;
   if (DefSize.x<=0) or (DefSize.y<=0) then begin
    if not ReadIniFileRecord(SysIniFile,SectSystem(1),'SystemConsoleDefaultPos%i;%i',DefPos)
    or not ReadIniFileRecord(SysIniFile,SectSystem(1),'SystemConsoleDefaultSize%i;%i',DefSize)
    then begin DefPos:=Point2I(167,0); DefSize:=Point2I(650,350); end;
    DefSize.y:=Min(DefSize.y,Screen.Height);
    DefSize.x:=Min(DefSize.x,Screen.Width);
   end;
   if (DefSize.x>0) and (DefSize.y>0) then begin
    Form.Top:=DefPos.y;
    Form.Left:=DefPos.x;
    Form.Width:=DefSize.x;
    Form.Height:=DefSize.y;
   end;
   Exit; // Skip obsolete
   Form.Width:=650; Form.Height:=350; Form.Top:=0;
   Form.Left:=(Application.MainForm.ClientWidth-Form.Width) div 2;
  end;
 except
  on E:Exception do BugReport(E,Self,'GoHome');
 end;
end;

procedure TSystemConsole.GoTail;
begin
 if Assigned(Self) then
 try
  if IsMainThread then
  if Form.Ok then begin
   Form.UpdateScrollBars;
  end;
 except
  on E:Exception do BugReport(E,Self,'GoTail');
 end;
end;

procedure TSystemConsole.Activate;
begin
 if Assigned(Self) then
 try
  if IsMainThread then
  if Form.Ok then begin
   Form.Show;
   Form.WindowState:=wsNormal;
   SdiMan.ActivateChild(Form);
  end;
 except
  on E:Exception do BugReport(E,Self,'Activate');
 end;
end;

procedure TSystemConsole.RegisterActions;
begin
 if Assigned(Self) then
 with SystemCalculator do begin
  SetAction('help',          act_help,         RusEng('Вывод на консоль справки по теме ...',
                                                      'Output help on topic ... to console'));
  SetAction('list',          act_list,         RusEng('Вывод на консоль списка переменных,констант,функций и т.д.',
                                                      'Output list of variables,contants,functions,etc.'));
  SetAction('sysinfo',       act_sysinfo,      RusEng('Краткая информация о системе.',
                                                      'Brief system information.'));
  SetAction('polling',       act_polling,      RusEng('Гистограммы периода опроса потоков.',
                                                      'Thread polling histograms.'));
  SetAction('adam',          act_adam,         RusEng('Отладочные команды для модулей ADAM.',
                                                      'Debug commands for ADAM modules.'));
  SetAction('pid',           act_pid,          RusEng('Работа с процессами (mini Task Manager).',
                                                      'Work with processes (mini Task Manager).'));
  SetAction('run',           act_run,          RusEng('Запуск процесса из командной строки.',
                                                      'Run new process with given cmdline.'));
  SetAction('getapppath',    act_getapppath,   RusEng('GetAppPath - найти или запустить приложение.',
                                                      'GetAppPath - find or start an application.'));
  SetAction('env',           act_env,          RusEng('Прочитать/установить переменную окружения.',
                                                      'Get/Set environment variable.'));
  SetAction('reg',           act_reg,          RusEng('Прочитать/установить переменную (строку) рестра.',
                                                      'Get/Set registry (string) variable.'));
  SetAction('guard',         act_guard,        RusEng('Проверить\изменить уровень прав доступа.',
                                                      'Check\change access rights level.'));
  SetAction('menu',          act_menu,         RusEng('Выполнить команду меню.',
                                                      'Menu command.'));
  SetAction('view',          act_view,         RusEng('Спрятать\показать визуальный элемент.',
                                                      'Show\hide visual element.'));
  SetAction('memory',        act_memory,       RusEng('Чтение\установка параметров памяти.',
                                                      'Get\set memory parametrs.'));
  SetAction('sleep',         act_sleep,        RusEng('Приостановка программы на заданное время.',
                                                      'Sleep program for given time interval.'));
  SetAction('voice',         act_voice,        RusEng('Проигрывание звуковых wav файлов',
                                                      'Play sound wav files'));
  SetAction('speak',         act_speak,        RusEng('Выдать сообщение через речевой синтезатор.',
                                                      'Speak phrase thru speech engine.'));
  SetAction('speech',        act_speech,       RusEng('Проверить\изменить параметры речевого синтезатора.',
                                                      'Get\set speech engine parameters.'));
  SetAction('term',          act_term,         RusEng('Открыть терминальное окно.',
                                                      'Open terminal window.'));
  SetAction('compile',       act_compile,      RusEng('Компилировать указанный в агрументе файл.',
                                                      'Compile specified file.'));
  SetAction('daq',           act_daq,          RusEng('Функции (compile,devsend,..) работы с DAQ-системой.',
                                                      'DAQ-system functions (compile,devsend,...).'));
  SetAction('tty',           act_tty,          RusEng('Работа с виртуальной консолью (TTY): listen,close,status.',
                                                      'Virtual console (TTY) functions: listen,close,status.'));
  SetAction('fonts',         act_fonts,        RusEng('Работа с фонтами: перечисление, регистрация, удаление.',
                                                      'Font tools: enumeration, registration etc.'));
  SetAction('silent',        act_silent,       RusEng('Выполнить команду в тихом режиме, без печати ввода\вывода.',
                                                      'Execute command in silent mode: no echo, no result print.'));
  SetAction('silence',       act_silence,      RusEng('Задает режим работы команды @silent.',
                                                      'Set working mode of @silent command.'));
  SetAction('textmetadata',  act_textmetadata, RusEng('Секция [MetaData]...',
                                                      'Text [MetaData] section...'));
  SetAction('integrity',     act_integrity,    RusEng('Утилита контроля целостности данных.',
                                                      'Data integrity control utility.'));
  SetAction('log',           act_log,          RusEng('Запись данных в журнальный файл.',
                                                      'Event data logger.'));
  SetAction('syslog',        act_syslog,       RusEng('Запись сообщения в Системный Журнал.',
                                                      'Write message to System Log.'));
  SetAction('tooltip',       act_tooltip,      RusEng('Показать всплывающее сообщение.',
                                                      'Show tooltip notifier.'));
  SetAction('onexception',   act_onexception,  RusEng('Обработка исключений.',
                                                      'Exception handler.'));
  SetAction('testbench',     act_testbench,    RusEng('Модуль тестирования.',
                                                      'Testing bench.'));
  SetAction('file',          act_file,         RusEng('Работа с файлами (exist/size/kill).',
                                                      'Work with files (exist/size/kill).'));
  SetAction('note',          act_note,         RusEng('Печать метки времени и сообщения.',
                                                      'Print time stamp and message.'));
  SetAction('if',            act_if,           RusEng('@if a then b - выполнить «b» если условие «a» истинно (отлично от 0).',
                                                      '@if a then b - execute «b» if condition «a» is true (non-zero).'));
  SetAction('geo',           act_geo,          RusEng('Функции геолокации (список кодов стран).',
                                                      'Geolocation functions (list of country codes).'));
  SetAction('lng',           act_lng,          RusEng('Функции языковой поддержки (список кодов языков).',
                                                      'Language functions (list of language codes).'));
  SetAction('colorinfo',     act_colorinfo,    RusEng('Печатает информацию о цвете (код/имя).',
                                                      'Print information about given color (code/name).'));
  SetAction('debuglogctrl',  act_debuglogctrl, RusEng('Управление каналами отладочного вывода DebugLog.',
                                                      'Control of debug log output channels - DebugLog.'));
  SetAction('trayicon',      act_trayicon,     RusEng('Управление состоянием значка в Области Уведомлений.',
                                                      'Control of Application System Tray Icon.'));
  SetAction('fpcup',         act_fpcup,        RusEng('Проверка или вызов компилятора FPC - FpcupDeluxe.',
                                                      'Check or call compiler FPC - FpcupDeluxe.'));
  SetAction('wmctrl',        act_wmctrl,       RusEng('Управление оконным менеджером и операции над окнами.',
                                                      'Window Manager Control: apply operations to windows.'));
  SetAction('sig',           act_sig,          RusEng('Работа с сигналами операционной системы Unix.',
                                                      'Work with Unix signals.'));
  SetAction('uart',          act_uart,         RusEng('Работа с COM портами (UART).',
                                                      'Work with COM ports (UART).'));
 end;
 if Assigned(Self) then
 with myDelayedEe do begin
  SetAction('compile',       act_compile,      RusEng('Компилировать указанный в агрументе файл.',
                                                      'Compile specified file.'));
  SetAction('daq',           act_daq,          RusEng('Функции (compile,devsend,..) работы с DAQ-системой.',
                                                      'DAQ-system functions (compile,devsend,...).'));
  SetAction('view',          act_view,         RusEng('Спрятать\показать визуальный элемент.',
                                                      'Show\hide visual element.'));
 end;
end;

procedure TSystemConsole.RegisterFunctions;
begin
 if Assigned(Self) then
 with SystemCalculator do begin
  SetFunc('detectwine',        0, fun_detectwine,        RusEng('Обнаружение среды исполнения WINE', 'Detecting WINE runtime'));
  SetFunc('detectvbox',        0, fun_detectvbox,        RusEng('Обнаружение среды VirtualBox', 'Detecting VirtualBox runtime'));
  SetFunc('spincountsys',      0, fun_spincountsys,      RusEng('Системный счетчик спин-блокировок', 'System Spinlock Count'));
  SetFunc('spincountdef',      1, fun_spincountdef,      RusEng('Счетчик спин-блокировок по умолчанию', 'Default Spinlock Count'));
  SetFunc('spincounttag',      1, fun_spincounttag,      RusEng('Счетчик спин-блокировки для тегов', 'Spinlock Count for tags'));
  SetFunc('enablefileguard',   1, fun_enablefileguard,   RusEng('Разрешить защиту файлов FileGuard', 'Enable file protection FileGuard'));
  SetFunc('enableeditdaqsite', 1, fun_enableeditdaqsite, RusEng('Разрешить редакцию файлов daqsite/*', 'Enable edit files in daqsite/*'));
  SetFunc('getfavoriteansicp', 0, fun_getfavoriteansicp, RusEng('Узнать TheFavoriteAnsiCodePage', 'Get TheFavoriteAnsiCodePage'));
  SetFunc('setfavoriteansicp', 1, fun_setfavoriteansicp, RusEng('Задать TheFavoriteAnsiCodePage', 'Set TheFavoriteAnsiCodePage'));
  {$IFDEF Poligon}
  SetFunc('testing', 1, fun_testing, RusEng('Тестирование', 'Testing'));
  {$ENDIF Poligon}
 end;
end;

 {
 *******************************************************************************
 Utility functions
 *******************************************************************************
 }
procedure Init_System_Console;
begin
 try
  SystemConsole.Open;
 except
  on E:Exception do BugReport(E,nil,'Init_System_Console');
 end;
end;

procedure Done_System_Console;
begin
 try
  SystemConsole.Close;
 except
  on E:Exception do BugReport(E,nil,'Done_System_Console');
 end;
end;

procedure ExecuteSystemConsoleMonitoring(Update:Boolean=false);
begin
 try
  if IsMainThread then
  if SystemConsole.Form.Ok then begin
   SystemConsole.Form.Monitoring;
   if Update then SystemConsole.Form.Update;
  end;
 except
  on E:Exception do BugReport(E,SystemConsole,'ExecuteSystemConsoleMonitoring');
 end;
end;

 {
 *******************************************************************************
 TSystemConsole
 *******************************************************************************
 }
const
 mySystemConsole : TSystemConsole = nil;

function SystemConsole:TSystemConsole;
begin
 if not Assigned(mySystemConsole) then begin
  mySystemConsole:=TSystemConsole.Create;
  mySystemConsole.Master:=@mySystemConsole;
 end;
 Result:=mySystemConsole;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_unit_systemconsole;
begin
 InitHasher;
 SystemConsole.Ok;
end;

procedure Free_unit_systemconsole;
begin
 LogFree;
 TooltipFree;
 IntegrityFree;
 Kill(TObject(mySystemConsole));
 FreeHasher;
end;

initialization

 Init_unit_systemconsole;

finalization

 Free_unit_systemconsole;

end.

//////////////
// END OF FILE
//////////////

