////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form DAQ Pascal Editor.                                                    //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231209 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_daqpascaleditor; // Form DAQ Pascal Editor

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_TextEditor,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_sesman,
 _crw_task, _crw_regexp, _crw_hl, _crw_gloss,
 _crw_daqsys, _crw_daqdev, _crw_assoc, _crw_sect,
 _crw_appforms, _crw_apptools, _crw_apputils;

type

  { TFormDaqPascalEditor }

  TFormDaqPascalEditor = class(TFormTextEditor)
    ActionEditCompile: TAction;
    ActionEditHelpDaqPascalApi: TAction;
    ActionViewBarFxShow: TAction;
    ActionViewBarFxHide: TAction;
    ActionViewBarFxInc: TAction;
    ActionViewBarFxDec: TAction;
    ActionViewBarFxFont: TAction;
    ActionViewOpenDeviceConsole: TAction;
    MenuEditDeleteSeparator: TMenuItem;
    MenuEditCompile: TMenuItem;
    MenuEditHelpDaqPascalApi: TMenuItem;
    MenuViewBarFx: TMenuItem;
    MenuViewBarFxShow: TMenuItem;
    MenuViewBarFxHide: TMenuItem;
    MenuViewBarFxInc: TMenuItem;
    MenuViewBarFxDec: TMenuItem;
    MenuViewBarFxFont: TMenuItem;
    MenuViewOpenDeviceConsole: TMenuItem;
    ToolButtonEditCompile: TToolButton;
    ToolButtonCompileSeparator: TToolButton;
    ToolButtonEditHelpDaqPascalApi: TToolButton;
    ToolButtonViewBarFxShow: TToolButton;
    ToolButtonViewOpenDeviceConsole: TToolButton;
    PopupMenuEditCompile: TMenuItem;
    PopupMenuEditHelpDaqPascalApi: TMenuItem;
    PanelViewBarFx: TPanel;
    PanelViewBarFxButtons: TPanel;
    BitBtnViewBarFxHide: TBitBtn;
    BitBtnViewBarFxFont: TBitBtn;
    BitBtnViewBarFxInc: TBitBtn;
    BitBtnViewBarFxDec: TBitBtn;
    GroupBoxViewBarFxList: TGroupBox;
    ListBoxViewBarFx: TListBox;
    TimerViewBarFx: TTimer;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ActionEditCompileExecute(Sender: TObject);
    procedure ActionEditHelpDaqPascalApiExecute(Sender: TObject);
    procedure ActionViewBarFxShowExecute(Sender: TObject);
    procedure ActionViewBarFxHideExecute(Sender: TObject);
    procedure ActionViewBarFxIncExecute(Sender: TObject);
    procedure ActionViewBarFxDecExecute(Sender: TObject);
    procedure ActionViewBarFxFontExecute(Sender: TObject);
    procedure ActionViewOpenDeviceConsoleExecute(Sender: TObject);
    procedure ListBoxViewBarFxClick(Sender: TObject);
    procedure TimerViewBarFxTimer(Sender: TObject);
  private
    { Private declarations }
    strFx : LongString;
    rexFx : Integer;
    dicFx : Integer;
    ModFlag : Boolean;
    procedure ClearBarFx(doFree:Boolean);
  public
    { Public declarations }
    LinkedDevice : TDaqDevice;
    procedure UpdateCommands; override;
    procedure Reopen; override;
    procedure UpdateBarFx;
  end;

function NewFormDaqPascalEditor(aDevice   : TDaqDevice;
                          const aFileName : LongString): TFormDaqPascalEditor;
procedure Kill(var TheObject:TFormDaqPascalEditor); overload;

type
 EOpenHtmlBrowser = class(ESoftException);

implementation

{$R *.lfm}

uses
 _crw_daqpascalcompiler,
 _crw_daqpascaldevice;

function NewFormDaqPascalEditor(aDevice   : TDaqDevice;
                          const aFileName : LongString): TFormDaqPascalEditor;
begin
 Result:=nil;
 try
  if (aDevice is TProgramDevice) then
  if IsNonEmptyStr(aFileName) then begin
   Application.CreateForm(TFormDaqPascalEditor, Result);
   if Result.Ok then begin
    Result.Open(UnifyFileAlias(aFileName));
    Result.LinkedDevice:=aDevice;
    Result.DefaultReadOnly;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'NewFormDaqPascalEditor');
 end;
end;

procedure Kill(var TheObject:TFormDaqPascalEditor); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

procedure TFormDaqPascalEditor.FormCreate(Sender: TObject);
begin
 inherited;
 UpdateMenu(MenuEditCompile,
            RusEng('Компиляция','Compile'),
            RusEng('Компилировать DAQ - программу.','Compile DAQ - program source.'),
            ShortCut(VK_F9,[ssCtrl]));
 UpdateMenu(MenuEditHelpDaqPascalApi,
            RusEng('Справка по Daq Pascal','Help on Daq Pascal'),
            RusEng('Вызвать справку по Daq Pascal.','Open Daq Pascal help.'),
            ShortCut(VK_F1,[ssCtrl]));
 UpdateMenu(MenuViewBarFx,
            RusEng('Функции','Functions'),
            RusEng('Меню панели Функции.','Menu of panel Functions.'),
            0);
 UpdateMenu(MenuViewBarFxShow,
            RusEng('Показать Функции','Show Functions'),
            RusEng('Показать панель Функции.','Show panel Functions.'),
            0);
 UpdateMenu(MenuViewBarFxHide,
            RusEng('Убрать Функции','Hide Functions'),
            RusEng('Спрятать панель Функции.','Hide panel Functions.'),
            0);
 UpdateMenu(MenuViewBarFxInc,
            RusEng('Расширить Функции','Grow Functions'),
            RusEng('Расширить панель Функции.','Grow panel Functions.'),
            0);
 UpdateMenu(MenuViewBarFxDec,
            RusEng('Сжать Функции','Shrink Functions'),
            RusEng('Сжать панель Функции.','Shrink panel Functions.'),
            0);
 UpdateMenu(MenuViewBarFxFont,
            RusEng('Шрифт Функций','Font of Functions'),
            RusEng('Изменить шрифт панели Функции.','Change font of panel Functions.'),
            0);
 UpdateMenu(MenuViewOpenDeviceConsole,
            RusEng('Консоль Устройства','Device Concole'),
            RusEng('Открыть Консоль Устройства DAQ Program.','Open DAQ Program Device Console.'),
            0);
 GroupBoxViewBarFxList.Caption:=RusEng(' Функции: ',' Functions: ');
 BitBtnViewBarFxInc.Caption:='';
 BitBtnViewBarFxDec.Caption:='';
 BitBtnViewBarFxHide.Caption:='';
 BitBtnViewBarFxFont.Caption:='';
 PanelViewBarFx.Visible:=false;
 ModFlag:=false;
 strFx:='';
 rexFx:=0;
 dicFx:=0;
end;

procedure TFormDaqPascalEditor.FormDestroy(Sender: TObject);
begin
 inherited;
 ClearBarFx(true);
 strFx:='';
end;

procedure TFormDaqPascalEditor.UpdateCommands;
begin
 inherited;
 try
  ActionEditCompile.Enabled:=(WindowState<>wsMinimized) and not PerformReadOnly;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TFormDaqPascalEditor.Reopen;
begin
 inherited Reopen;
 if PanelViewBarFx.Visible then UpdateBarFx;
end;

procedure TFormDaqPascalEditor.ActionEditCompileExecute(Sender: TObject);
var how:Boolean;
begin
 if Guard.CheckAction(ga_Root,ActionEditCompile)<0 then Exit;
 inherited;
 if Ok then
 try
  FileSave;
  if (LinkedDevice is TProgramDevice) then
  with (LinkedDevice as TProgramDevice) do
  if CanCompileEditor then begin
   how:=CompileEditor;
   NotifyCompile(how);
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditCompileExecute');
 end;
end;

{$IFDEF UNIX}
function GetExeByFile(const FileName:LongString):LongString;
begin
 Result:=GetAppPath('-e firefox -t text/html -a .html');
end;
{$ENDIF ~UNIX}

function OpenExplorer(FileName,Topic:LongString):Boolean;
var tid:Integer; CmdLine,Target:LongString;
begin
 Result:=false;
 try
  FileName:=UnifyFileAlias(FileName);
  if not FileExists(FileName)
  then RAISE EOpenHtmlBrowser.Create(Format('Could not find "%s"',[FileName]));
  CmdLine:=GetExeByFile(FileName);
  if not FileExists(CmdLine)
  then RAISE EOpenHtmlBrowser.Create('Could not find HTML Browser.');
  if IsWindows then CmdLine:=AnsiQuotedIfNeed(CmdLine);
  Topic:=TrimDef(Topic,'index');
  if IsWindows then Target:=AnsiQuotedIfNeed(Format('file:///%s#%s',[FileName,Topic]));
  if IsUnix then Target:=Format('file://%s#%s',[FileName,Topic]);
  tid:=task_init(Trim(CmdLine+' '+Target));
  if (tid<>0) then task_ctrl(tid,'Display=1');
  if (tid<>0) then Result:=task_run(tid);
  if (tid<>0) then task_free(tid);
 except
  on E:Exception do BugReport(E,nil,'OpenExplorer');
 end;
end;

procedure TFormDaqPascalEditor.ActionEditHelpDaqPascalApiExecute(Sender: TObject);
const
 OpChars = '+-*/&|~<>:=()[];{}';
 OpNames = 'plus,minus,multiply,divide,and,or,not,less,greater,colon,equal,brackets,brackets,fetch,fetch,semicolon,comment,comment';
var p:Integer; s:LongString; w:WideString;
begin
 if Guard.CheckAction(ga_Guest,ActionEditHelpDaqPascalApi)<0 then Exit;
 inherited;
 try
  s:='';
  if SysGlossary.ReadIniPath(SysIniFile,SectSystem,'DaqPascalApiIndex',HomeDir,s) and FileExists(s) then begin
   w:=GetWordUnderCursor(PerformWideText,PerformSelStart,StrToWide(OpChars));
   if (Length(w)=1) then begin
    p:=Pos(w[1],OpChars);
    if (p>0) then w:=StrToWide(ExtractWord(p,OpNames,[',']));
   end;
   w:=WideLowerCase(w);
   if not OpenExplorer(s,WideToStr(w)) then Echo(Format('Could not open "%s"',[s]));
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionHelpDaqPascalApiExecute');
 end;
end;

procedure TFormDaqPascalEditor.ActionViewBarFxShowExecute(Sender: TObject);
begin
 inherited;
 PanelViewBarFx.Visible:=true;
 TimerViewBarFx.Enabled:=true;
 UpdateBarFx;
end;

procedure TFormDaqPascalEditor.ActionViewBarFxHideExecute(Sender: TObject);
begin
 inherited;
 PanelViewBarFx.Visible:=false;
 TimerViewBarFx.Enabled:=false;
 UpdateBarFx;
end;

const
 DeltaWidth = 8;

procedure TFormDaqPascalEditor.ActionViewBarFxIncExecute(Sender: TObject);
begin
 inherited;
 if (PanelViewBarFx.Width<PanelViewBarFx.Constraints.MaxWidth)
 then PanelViewBarFx.Width:=PanelViewBarFx.Width+DeltaWidth;
end;

procedure TFormDaqPascalEditor.ActionViewBarFxDecExecute(Sender: TObject);
begin
 inherited;
 if (PanelViewBarFx.Width>PanelViewBarFx.Constraints.MinWidth)
 then PanelViewBarFx.Width:=PanelViewBarFx.Width-DeltaWidth;
end;

procedure TFormDaqPascalEditor.ClearBarFx(doFree:Boolean);
begin
 try
  if doFree then begin
   if (rexFx<>0) then begin regexp_free(rexFx);   rexFx:=0; end;
   if (dicFx<>0) then begin hashlist_free(dicFx); dicFx:=0; end;
  end else begin
   if (rexFx<>0) then regexp_ref(rexFx).Storage.Clear;
   if (dicFx<>0) then hashlist_ref(dicFx).Clear;
  end;
  ListBoxViewBarFx.Clear;
 except
  on E:Exception do BugReport(E,Self,'ClearBarFx');
 end;
end;

function ThePatternFx:LongString;
const PatternFx:LongString='';
var s:LongString;
begin
 s:='';
 if (PatternFx='') and ReadIniFileString(SysIniFile,SectDaqSys,'DaqPascalPatternFx%s',s,efConfigNC) then PatternFx:=Trim(s);
 if (PatternFx='') then PatternFx:='/^\s*(procedure|function)\s+([_a-zA-Z][_a-zA-Z0-9]*)/img'; // Fallback value
 Result:=PatternFx;
end;

procedure TFormDaqPascalEditor.UpdateBarFx;
var i,p,ii:Integer; s:LongString;
begin
 try
  if not PanelViewBarFx.Visible then begin
   ClearBarFx(true);
   Exit;
  end;
  ClearBarFx(false);
  if (rexFx=0) then rexFx:=regexp_init(regexp_def,ThePatternFx);
  if (dicFx=0) then dicFx:=hashlist_init(false,HashList_DefaultHasher);
  if regexp_exec(rexFx,PerformLinesText)>0 then
  try
   ListBoxViewBarFx.Items.BeginUpdate;
   for i:=1 to regexp_matchnum(rexFx,0) do begin
    s:=Trim(regexp_matchstr(rexFx,i,2)); if (s='') then continue;
    p:=regexp_matchpos(rexFx,i,1); if (p<=0) then continue;
    ii:=ListBoxViewBarFx.Items.IndexOf(s);
    if (ii>=0) then ListBoxViewBarFx.Items.Delete(ii);
    ListBoxViewBarFx.Items.Add(s);
    hashlist_setlink(dicFx,s,p);
   end;
   if (strFx<>'') then begin
    ii:=ListBoxViewBarFx.Items.IndexOf(strFx);
    ListBoxViewBarFx.ItemIndex:=ii;
   end;
  finally
   ListBoxViewBarFx.Items.EndUpdate;
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateBarFx');
 end;
end;

procedure TFormDaqPascalEditor.ListBoxViewBarFxClick(Sender: TObject);
var i,p:Integer; s:LongString;
begin
 inherited;
 try
  i:=ListBoxViewBarFx.ItemIndex;
  if (i<0) then Exit;
  s:=Trim(ListBoxViewBarFx.Items[i]);
  if (hashlist_indexof(dicFx,s)<0) then Exit;
  p:=hashlist_getlink(dicFx,s); if (p<=0) then Exit;
  PerformSelStart:=max(0,p-1);
  ActiveControl:=Editor;
  strFx:=s;
 except
  on E:Exception do BugReport(E,Self,'ListBoxViewBarFxClick');
 end;
end;

procedure TFormDaqPascalEditor.ActionViewBarFxFontExecute(Sender: TObject);
const nFont:Integer=0;
begin
 inherited;
 nFont:=(nFont+1) mod 3;
 case nFont of
  0:   RestoreFont(ListBoxViewBarFx.Font,StandardFont);
  1:   RestoreFont(ListBoxViewBarFx.Font,DefaultSansFont);
  2:   RestoreFont(ListBoxViewBarFx.Font,DefaultSansNarrowFont);
  else RestoreFont(ListBoxViewBarFx.Font,StandardFont);
 end;
end;

procedure TFormDaqPascalEditor.ActionViewOpenDeviceConsoleExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewOpenDeviceConsole)<0 then Exit;
 inherited;
 if Ok then
 try
  if (LinkedDevice is TProgramDevice) then
  (LinkedDevice as TProgramDevice).OpenConsole;
 except
  on E:Exception do BugReport(E,Self,'ActionViewOpenDeviceConsoleExecute');
 end;
end;

procedure TFormDaqPascalEditor.TimerViewBarFxTimer(Sender: TObject);
begin
 inherited;
 if (ModFlag<>PerformModified) then begin
  if ModFlag then UpdateBarFx;
  ModFlag:=PerformModified;
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_daqpascaleditor;
begin
end;

procedure Free_form_daqpascaleditor;
begin
end;

initialization

 Init_form_daqpascaleditor;

finalization

 Free_form_daqpascaleditor;

end.

//////////////
// END OF FILE
//////////////

