////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// PCL Device family.                                                         //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231204 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit _crw_pcldev; // PCL Device family

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Printers,
 lcltype, lclintf,
 Form_CrwDaqSysChild, Form_TextEditor, Form_CurveWindow, Form_SurfWindow,
 Form_CircuitWindow, Form_ConsoleWindow, Form_TabWindow, Form_SpectrWindow,
 Form_Calculator, Form_ListBoxSelection, Form_UartTerminal, Form_CalibDialog,
 Form_DaqEditTagDialog,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_ef, _crw_curves, _crw_riff, _crw_pio,
 _crw_calib, _crw_couple, _crw_i8255, _crw_polling,
 _crw_daqtags, _crw_daqevnt, _crw_daqsys, _crw_daqdev,
 _crw_appforms, _crw_apptools, _crw_apputils;

 {
 *******************************************************************************
 Конструктор создает устройства PCL по модели
 *******************************************************************************
 }
function PCLDeviceConstructor(const Name,Family,Model,ConfigFile:LongString):TDaqDevice;

 {
 *******************************************************************************
 Абстрактное PCL-устройство, то есть устройство на ISA шине, с базовым адресом
 Base.
 Каждое устройство выполняется в отдельном потоке Polling. Поток периодически
 вызывает процедуру Poll, в которой по таймеру InquiryTimer вызывается метод
 Handler, который и должен делать основную полезную работу.
 *******************************************************************************
 }
type
 TPclDevice = class(TDaqDevice)
 private
  myPolling   : TPolling;               { поток опроса устройства          }
  function    GetPolling:TPolling;
 protected
  Base        : Word;                   { базовый адрес порта ввода/вывода }
  ad_chan_no  : Integer;                { число электрических A/D каналов  }
  da_chan_no  : Integer;                { число электрических D/A каналов  }
  di_chan_no  : Integer;                { число электрических D/I каналов  }
  do_chan_no  : Integer;                { число электрических D/O каналов  }
 public
  property    Polling : TPolling read GetPolling;
  constructor Create(const aName:LongString; aBase:Word);
  destructor  Destroy; override;
  procedure   Config(FileName:LongString); override;
  function    GetProperty(TheText:TText):TText; override;
  procedure   Poll; override;
  procedure   Handler; virtual;
 end;

 {
 *******************************************************************************
 Список обеспечивает хранение и сканирование устройств типа Pcl.
 Метод EnablePolling разрешает или запрещает опрос всех устройств.
 Метод Start вызывает наследуемый Start и затем разрешает опрос устройств.
 Метод Stop останавливает опрос и затем вызывает наследуемый Stop.
 Метод Poll ничего не делает, так как опрос каждого устройства идет в отдельном
 потоке.
 *******************************************************************************
 }
type
 TPclDeviceList = class(TDaqDeviceList)
 private
  function  GetPclDev(Index:Integer):TPclDevice;
  procedure SetPclDev(Index:Integer; aPclDev:TPclDevice);
 public
  property  SoftDev[i:Integer] : TPclDevice read GetPclDev write SetPclDev; default;
  procedure EnablePolling(aEnable:Boolean);
  function  Start:Boolean; override;
  procedure Stop; override;
  procedure Poll; override;
 end;

function PclDeviceList:TPclDeviceList;


implementation

uses
 _crw_daqdio144device;

 {
 ********************************************
 Конструктор создает устройства PCL по модели
 ********************************************
 }
function PCLDeviceConstructor(const Name,Family,Model,ConfigFile:LongString):TDaqDevice;
var Base:Word;
begin
 Result:=nil;
 if Daq.Ok then
 if SameText(Family,'PCL') then begin
  Base:=0;
  if ReadIniFileWord(ConfigFile,'['+Name+']','Base%w',Base) then begin
   if SameText(Model,'DIO-144') then Result:=TDIO144Device.Create(Name, Base, 6, 'DIO-144');
   if SameText(Model,'DIO-24')  then Result:=TDIO144Device.Create(Name, Base, 1, 'DIO-24');
   if SameText(Model,'PCL-731') then Result:=TDIO144Device.Create(Name, Base, 2, 'PCL-731');
   if SameText(Model,'DIO-48')  then Result:=TDIO144Device.Create(Name, Base, 2, 'DIO-48');
  end else Daq.AddWarning(Name+': Base missed!');
  if not Result.Ok
  then Daq.AddWarning('Error (invalid device) -> '+Name+' = device '+Family+' '+Model);
 end;
end;

 {
 *************************
 TPclDevice implementation
 *************************
 }
procedure DevicePollAction(aPolling:TPolling; var Terminate:Boolean);
var Device:TObject;
begin
 Device:=aPolling.LinkObject;
 if (Device is TPclDevice) then TPclDevice(Device).Poll else Terminate:=true;
end;

function TPclDevice.GetPolling:TPolling;
begin
 if Assigned(Self) then Result:=myPolling else Result:=nil;
end;

constructor TPclDevice.Create(const aName:LongString; aBase:Word);
begin
 inherited Create(aName);
 SetDeviceFamily('PCL');
 myPolling:=NewPolling(DevicePollAction,
                       DefaultDaqPollDelay,  DefaultDaqPollPriority,
                       false, 'Daq.'+Name);
 myPolling.Master:=@myPolling;
 myPolling.LinkObject:=Self;
 Base:=aBase;
 ad_chan_no:=0;
 da_chan_no:=0;
 di_chan_no:=0;
 do_chan_no:=0;
end;

destructor TPclDevice.Destroy;
begin
 Kill(myPolling);
 inherited Destroy;
end;

procedure TPclDevice.Config(FileName:LongString);
var d:Integer; p:TThreadPriority;
begin
 d:=100; p:=tpNormal;
 FileName:=UnifyFileAlias(FileName);
 inherited Config(FileName);
 if ReadIniFilePolling(FileName,   DevSection, 'DevicePolling', d, p)
 or ReadIniFilePolling(FileName,   '[DAQ]',    'DevicePolling', d, p)
 or ReadIniFilePolling(SysIniFile, '[DaqSys]', 'DevicePolling', d, p)
 then begin
  Polling.Delay:=d;
  Polling.Priority:=p;
 end;
end;

function  TPCLDevice.GetProperty(TheText:TText):TText;
begin
 GetProperty:=inherited GetProperty(TheText);
 TheText.Addln(Format('DevicePolling = %d, %s',[Polling.Delay,GetPriorityName(Polling.Priority)]));
 TheText.Addln('Base = $'+hexw(Base));
end;

procedure TPclDevice.Poll;
begin
 if GotEvents then Handler;
end;

procedure TPclDevice.Handler;
begin
end;

 {
 *****************************
 TPclDeviceList implementation
 *****************************
 }
function TPclDeviceList.GetPclDev(Index:Integer):TPclDevice;
begin
 Result:=TPclDevice(Items[Index]);
end;

procedure TPclDeviceList.SetPclDev(Index:Integer; aPclDev:TPclDevice);
begin
 Items[Index]:=aPclDev;
end;

procedure TPclDeviceList.EnablePolling(aEnable:Boolean);
var i:Integer;
 procedure Handle(Item:TPclDevice);
 begin
  if Item.Ok then Item.Polling.Enable(aEnable,DefaultDaqTimeOut);
 end;
begin
 for i:=0 to Count-1 do Handle(Self[i]);
end;

function TPclDeviceList.Start:Boolean;
begin
 EnablePolling(false);
 Result:=inherited Start;
 EnablePolling(true);
end;

procedure TPclDeviceList.Stop;
begin
 EnablePolling(false);
 inherited Stop;
end;

procedure TPclDeviceList.Poll;
begin
 // do nothing, because each device has own thread
end;

function PclDeviceList:TPclDeviceList;
const
 myPclDeviceList : TPclDeviceList = nil;
begin
 if not Assigned(myPclDeviceList) then begin
  myPclDeviceList:=TPclDeviceList.Create(true);
  myPclDeviceList.Master:=@myPclDeviceList;
 end;
 Result:=myPclDeviceList;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_pcldev;
begin
 PclDeviceList.Ok;
end;

procedure Free_crw_pcldev;
begin
 PclDeviceList.Free;
end;

initialization

 Init_crw_pcldev;

finalization

 Free_crw_pcldev;

end.

//////////////
// END OF FILE
//////////////

