////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// DAQ Pascal Device.                                                         //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231209 - Modified for FPC (A.K.)                                         //
// 20240618 - OutFifoCleanRunCount                                            //
// 20241207 - NotifyCompile                                                   //
// 20250129 - Use TAtomicCounter                                              //
// 20250519 - FormatErrorDetails                                              //
////////////////////////////////////////////////////////////////////////////////

unit _crw_daqpascaldevice; // DAQ Pascal Device

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Printers,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 Form_TextEditor, Form_CurveWindow, Form_SurfWindow,
 Form_CircuitWindow, Form_ConsoleWindow, Form_TabWindow,
 Form_SpectrWindow, Form_Calculator, Form_ListBoxSelection,
 Form_UartTerminal, Form_CalibDialog, Form_DaqEditTagDialog,
 Form_DaqPascalEditor,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo, _crw_ef,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut, _crw_pio,
 _crw_dynar, _crw_snd, _crw_guard, _crw_utf8,
 _crw_calib, _crw_curves, _crw_riff, _crw_couple,
 _crw_polling, _crw_proc, _crw_daqtags, _crw_daqevnt,
 _crw_daqsys, _crw_daqdev, _crw_adamdev, _crw_softdev,
 _crw_sect, _crw_gloss, _crw_sesman,
 _crw_daqpascalcompiler,
 _crw_appforms, _crw_apptools, _crw_apputils;

 {
 *******************************************************************************
 Объект:       ProgramDevice
 Объявление:   Device Software Program
 Назначение:   Служит для общей обработки данных при помощи встроенного
               компилятора DAQ PASCAL, который компилирует паскалевский
               файл в промежуточный P-код, выполняемый при помощи
               интерпретатора и достаточно быстрый для подавляющего
               числа задач.
 Входы:        AnalogInput(..)     - определяется конфигурацией.
               DigitalInput(..)    - определяется конфигурацией.
 Выходы:       AnalogOutput(..)    - определяется конфигурацией.
               DigitalOutput(..)   - определяется конфигурацией.
 Описание:     Устройство содержит простой компилятор для выполнения
               программ на языке условно названном DAQ PASCAL.
               Это узко-ориентированный компилятор для встраивания
               в устройства DAQ и проведения простых вычислений в процессе
               измерений.
 Конфигурация: AnalogInputs   = n - задание числа аналоговых входов
               AnalogOutputs  = n - задание числа аналоговых выходов
               DigitalInputs  = n - задание числа цифровых   входов
               DigitalOutputs = n - задание числа цифровых   выходов
               Calibrations   = n - задание числа калибровок
               Calibration#n  = ? - задание калибровки номер n
               DebugMode      = ? - бит0- надо ли выводить диагностику ошибок
                                          интерпретации в файл отладки.
                                    бит1-надо ли открыть консольное окно.
               ProgramSource  = ? - задание имени файла с текстом программы.
               Adam_Slot_Device = n - задание имени n слота для RS-485.
 Пример конфигурации:
               Данный пример генерирует сигнал на аналоговом выходе.
               [DeviceList]
               prog1 = device software program
               ...
               [prog1]
               AnalogInputs = 0
               DigitalInputs = 0
               AnalogOutputs = 3
               DigitalOutputs = 0
               Calibrations = 0
               StdinFifo = 16
               StdoutFifo = 32
               Link AnalogOutput 0 with curve ao0
               Link AnalogOutput 1 with curve ao1
               Link AnalogOutput 2 with curve ao2
               DebugMode = 1
               ProgramSource = prog1.pas

               file prog1.pas:

               program prog1;
               var b:boolean; w,a,n,v,t:real;
               begin
                if getdi_yn(0)<>0 then begin
                 t:=time;
                 w:=getai_yn(0);
                 a:=getai_yn(1);
                 n:=getai_yn(2);
                 v:=a*sin(2*pi*t*w)+n*random(-1,1);
                 b:=putao(0,t,v)
                end;
               end.
 *******************************************************************************
 Описание языка DAQ PASCAL:
 DAQ PASCAL - это узко специализированный компилятор p-кода для задач сбора
 данных (DAQ) в системе CRW-DAQ. Он является расширенным подмножеством языка
 программирования Pascal (это значит, что часть возможностей классического
 Паскаля недоступны, но есть возможности, которых нет в стандартном Паскале).
 Он основан на компиляторе Pascal S Николауса Вирта (Niklaus Wirth) и
 компиляторе Facilis 0.20 взятого из Интернета.

 Так как полное описание языка Паскаль дать довольно затруднительно, здесь
 в основном описаны отличия языка от стандартного Паскаля.

 ДИРЕКТИВЫ КОМПИЛЯТОРА
 $T+ : Запись таблиц сгенерированного p-кода в lst-файл (для отладки)
 $S+ : Запись стека вызовов процедур в lst-файл (для отладки)

 ОБЩАЯ СТРУКТУРА ПРОГРАММЫ
  program xxx(input,output);
  const ...;
  type ...;
  var ...;
   ...procedures  and functions ...
  begin
   ... основная программа...
  end.
  Секция program обязательна,параметры программы необязательны (игнорируются).
  Секции const,type,var необязательны, но при их наличии порядок их должен
  сохраняться, кроме того каждая секция может появляться только один раз.
 КОММЕНТАРИИ как в Borland Pascal.
 ИДЕНТИФИКАТОРЫ имеют 12 значащих символов, заглавный/прописной регистр
  не играет роли.
 ВХОДНОЙ ФАЙЛ ASCII, строки не должны быть длиннее 121 символов.
 МОДУЛИ не поддерживаются.
 ВКЛЮЧАЕМЫЕ ФАЙЛЫ в данной версии не поддерживаются.
 ВЫПОЛНЕНИЕ ПРОГРАММЫ
  Программа DAQ PASCAL описывают действия, которые должны происходить в цикле
  опроса устройств во время работы DAQ. Каждая программа начинает выполнение
  после старта DAQ и постоянно (многократно) вызывается в цикле опроса DAQ.
  Переменные, объявленные на верхнем уровне (то есть не внутри процедур),
  при первом вызове программы все равны нулю, при последующих вызовах
  сохраняют старые значения, пока программа не была перекомпилирована.
  При перекомпиляции все переменные зануляются.
 ПРОЦЕДУРЫ И ФУНКЦИИ Могут объявляться до 7 уровня вложения. Процедурный
  тип не поддерживается.
 УПРАВЛЯЮЩИЕ СТРУКТУРЫ поддерживаются стандартные
  FOR .. TO
  FOR .. DOWNTO
  WHILE .. DO
  REPEAT .. UNTIL
  Процедуры exit,break,continue НЕ ПОДДЕРЖИВАЮТСЯ
 ТИПЫ
  CHAR
  INTEGER  –2147483648..2147483647 (-32768..+32767 в CRW16).
  REAL     double(64 бита) 11 значащих знаков, экспонента -308..+308.
  STRING   до 32748 символов. Все строки динамические, длина до 32K.
  ARRAY    элементы любого типа
  RECORD   элементы любого типа (вариантные записи не поддерживаются)
 НЕ ПОДДЕРЖИВАЮТСЯ
  file, set, диапазон ..., перечислимые типы, указатели, метки, GOTO,
  WITH, FORWARD, процедурные типы, вариантные записи, объекты,
  типы longint,extended,comp,shortint, byte,word и т.д.

 СТРОКИ
  Все строки динамические (в динамической памяти) длины до 32748 символов.
  Память под строки распределяется блоками по 16 байт.
  СТРОКИ, КОТОРЫМ НИЧЕГО НЕ ПРИСВОЕНО, ИСПОЛЬЗОВАТЬ НЕЛЬЗЯ!
  Объявление строки: VAR s: STRING;   присвоение s:='str';
  Для совместимости возможно объявление VAR t: STRING[10];
  При этом константа длины игнорируется. Строки могут быть элементами
  массивов и записей.
  Тип string совместим с char и с array[] of char по присвоению.
  Индексация строк от s[i], i=1..length(s). Индексация за пределами длины
  строки сгенерирует ошибку.
  Строки можно сравнивать операторами =, <>, <=, >=, <, > (сравнение идет
  как ASCII).

 СТАНДАРТНЫЕ ФУНКЦИИ как в стандартном Паскале:
  abs(integer):integer
  abs(real):real
  sqr(integer):integer
  sqr(real):real
  odd(integer):boolean
  chr(integer):char
  ord(char):integer
  succ(char):char
  pred(char):char
  round(real):integer
  trunc(real):integer
  sin(real):real
  cos(real):real
  exp(real):real
  ln(real):real
  sqrt(real):real
  arctan(real):real
 ПРЕДОПРЕДЕЛЕННЫЕ ФУНКЦИИ-КОНСТАНТЫ
  _nan       NAN   нечисловое значение ( типа ln(0) )
  _minusinf  -INF  минус бесконечность ( типа -1/0  )
  _plusinf   +INF  плюс бесконечность  ( типа  1/0  )
  _nil       nil - указатель (для проверок ссылок на кривые,таймеры и т.д.)
  pi:real    число pi=3.14...
  macheps    точность числа real
  maxint     максимальное целое
 СТРОКОВЫЕ ОПЕРАЦИИ:
  maxavail                           доступная память для строк
  +                                  оператор сцепления строк как в примере
                                     s:='one'+'two';
  length(s:string):integer;          длина строки
  copy(s:string; p,n:integer):string копирует подстроку s с позиции p длины n
  pos(s,t:string):integer            положение строки s в строке t
  str(x:integer):string              преобразование числа в строку
  str(x:real):string                 преобразование числа в строку
  strfix(x:real;w,d:integer):string  преобразование числа в строку с фиксированной
                                     шириной w и d точек после запятой
  hexb(x:integer):string             преобразование byte(x) в hex-строку
  hexw(x:integer):string             преобразование word(x) в hex-строку
  hexl(x:real):string                преобразование longint(x) в hex-строку
  val(s:string):integer              преобразование строки в число
  rval(s:string):real                преобразование строки в число
                                     для hex-чисел преобразование делается
                                     по примеру
                                      i:=val('$FF');  i=255
                                      r:=rval('$FF'); r=255.0
   upcasestr(s:string):string       преобразует строку s в верхний регистр
   locasestr(s:string):string       преобразует строку s в нижний регистр
   worddelims(s:string):string      устанавливает новые символы-разделители
                                    слов для функций wordcount,extractword
                                    из строки s (если s-непустая строка)
                                    возвращает бывшие символы-разделители
   wordcount(s:string):integer      возвращает число слов в строке s
   extractword(n:integer;s:string):string выделяет из строки s слово номер n
   EOL:string                       возвращает маркер конца строки
   programe:string                  возвращает имя daq-программы
   devname:string                   возвращает имя устройства
 ВВОД-ВЫВОД
  getfattr(s:string):integer        чтение атрибутов файла с именем s
  setfattr(s:string;a:integer):integer установка атрибутов файла с именем s
                                       в соответствии со значением a
  Атрибуты файлов:
   ReadOnly=$01, Hidden=$02, SysFile=$04, VolumeID=$08, Directory=$10
   Archive=$20
  doserror:integer                  ошибка DOS при последней операции DOS
  mkdir(s:string):boolean           создание файлового каталога s

  Поддерживается только 1 текстовый файл для ввода и 1 текстовый файл
  для вывода.
  eof:boolean                        конец файла ввода
  eoln:boolean                       конец строки ввода
  reset(filename:string):integer     переназначение ввода
  rewrite(filename:string):integer   создание и назначение вывода
  append(filename:string):integer    назначение вывода в конец файла
  fileexists(name:string):boolean    проверка наличия файла
  fileerase(name:string):boolean     удаление файла
  filerename(arg:string):boolean     переименование файла
                                     arg = 'sourcefilename destfilename'
  filecopy(arg:string):boolean       копирование файла
                                     arg = 'sourcefilename destfilename'
  ioresult:integer                   статус операций ввода-вывода как в BP
  read,readln,write,writeln          как в Паскале
  пример:
   if reset(c:\input.txt')=0 then readln(x,y);
   if rewrite('c:\output.txt')=0 then writeln(x:10,y:14:3);
 Задачи ввода-вывода двоичных файлов решаются при помощи другого набора
 процедур.
 В каждой Daq-программе может быть открыт только один двоичный файл.
 Это даже хорошо, так как держать много одновременно открытых файлов нельзя.
 При работе с несколькими файлами они открываются и закрываются попеременно.
 f_reset(n:string;m:integer):integer
  Закрывает файл, если он был открыт.
  Открывает существующий файл с именем n, в режиме m (0-ReadOnly,1-WriteOnly,
  2-Read/Write). Возвращает 0 при успехе или код ошибки IOResult.
 f_rewrite(n:string;m:integer):integer
  Закрывает файл, если он был открыт.
  Создает новый файл именем n, в режиме m (0-ReadOnly,1-WriteOnly,
  2-Read/Write). Возвращает 0 при успехе или код ошибки IOResult.
 f_read(c:integer):string
  Читает c байт в виде строки дампа.
  Значение c не должно быть > 255.
  Возвращает строку, длина которой равна числу прочитанных байт.
  Возвращает статус через функцию IOResult.
 f_write(d:string):integer
  Записывает данные в файл в виде строки дампа d.
  Длина d не должна быть > 255.
  Возвращает число записанных байт.
  Возвращает статус через функцию IOResult.
 f_size:real
  Читает текущий размер открытого файла.
  Возвращает статус через функцию IOResult.
 f_seek(p:real):real
  Позиционирует указатель открытого файла в положение p, если p>=0.
  Возвращает позицию указателя после выполнения операции.
  f_seek(-1) просто вернет текущую позицию указателя.
  Возвращает статус через функцию IOResult.
 f_close:boolean
  Закрывает файл, если он был открыт.
  Возвращает статус через функцию IOResult.
  Файл также автоматоматически закрывается при остановке программы или
  при повторном открытии файла.
 dump(b:boolean):string
 dump(c:char):string
 dump(i:integer):string
 dump(r:real):string
  Возвращает сроку - дамп данных b,c,i или r, то есть их побайтное
  двоичное представление, как они хранятся в памяти. Это позволяет
  записывать переменные в файл в двоичном виде.
  Например,
  dump(true)=chr(1)
  dump('c')='c'
  dump(val('$fedc'))=chr(12)+chr(13)+chr(14)+chr(15)
  dump(0.0)=chr(0)+chr(0)+chr(0)+chr(0)+chr(0)+chr(0)+chr(0)+chr(0)
 dump2b(d:string):boolean
 dump2c(d:string):char
 dump2i(d:string):integer
 dump2r(d:string):real
  Эти функции обратны по смыслу функции dump и возвращают значение
  переменной, представленной строкой дампа d. Это позволяет
  считывать переменные из файла в двоичном виде.
  Если длины строки не хватает, дамп дополняется нулями.
  Если длина строки больше требуемой, ненужные данные игнорируются.
  Пример:
   if f_reset('example.bin',0)=0 then begin
    if f_write(dump(r))<>8 then writeln('WriteError');
    r:=dump2r(f_read(8));
    writeln('filesize=',f_size:10:0);
    writeln('filepos=',f_seek(-1):10:0);
    b:=f_close;
    if IOResult<>0 then writeln('Error!');
   end;
  mime_encode(s:string):string - кодирование данных в MIME,Base64
  mime_decode(s:string):string - раскодирование данных из MIME,Base64
 ФУНКЦИИ DAQ Примечание: так как в данной версии нет типа LongInt и Pointer,
             то битовые операции и ссылки сделаны на основе Real.
             Например, bitand(x,y) вычисляет побитное ИЛИ для округленных
             до LongInt чисел x,y.
             Например, refai(0) возвращает ссылку на кривую AnalogInput(0)
             преобралованную к Real.
  rand:real              случайное число от 0 до 1
  sleep(t:integer):boolean приостанавливает поток на t миллисекунд (Win32)
  mksecnow:real          текущее время в микросекундах от старта программы (Win32)
  msecnow:real           текущее время в миллисекундах
  secnow:real            текущее время в секундах
  getticks:real          текущее время в тиках таймера BIOS
  time:real              локальное время DAQ с начала измерений
  timeunits:real         единицы измерения локального времени DAQ
  numais:integer         число аналоговых входов
  numdis:integer         число цифровых входов
  numaos:integer         число аналоговых выходов
  numdos:integer         число цифровых выходов
  numcals:integer        число калибровок
  deg(x:real):real       перевод радиан в градусы
  rad(x:real):real       перевод градус в радианы
  tan(x:real):real       тангенс
  asin(x:real):real      обратный синус
  acos(x:real):real      обратный косинус
  atan(x:real):real      обратный тангенс
  sinh(x:real):real      гиперболический синус
  cosh(x:real):real      гиперболический косинус
  tanh(x:real):real      гиперболический тангенс
  gamma(n:real):real     гамма-функция: gamma(n) = (n-1)!
  isnan(a:real):boolean  1 если a=NAN или 0
  isinf(a:real):boolean  1 если a=INF или 0
  sign(x:real):integer   знак (+1,0,-1)
  trunc(x:real):integer  округление в сторону нуля
  round(x:real):integer  округление в сторону ближайшего целого
  abs(x)                 модуль
  int(x:real):real       округление в сторону нуля
  frac(x:real):real      дробная часть
  floor(x:real):real     округление в сторону нуля
  ceil(x:real):real      округление в сторону INF
  log(n,x:real):real     логарифм x по основанию n
  hypot(x,y:real):real   sqrt(x^2+y^2)
  random(a,b:real):real  случайное число в интервале (a,b)
  max(x,y:real):real     максимальное из x и y
  min(x,y:real):real     минимальное  из x и y
  eq(x,y:real):boolean   равно true если x=y
  ne(x,y:real):boolean   равно true если x<>y
  lt(x,y:real):boolean   равно true если x<y
  gt(x,y:real):boolean   равно true если x>y
  le(x,y:real):boolean   равно true если x<=y
  ge(x,y:real):boolean   равно true если x>=y
  bitor(x,y:real):real   побитное арифметическое неисключающее или
  bitxor(x,y:real):real  побитное арифметическое исключающее или
  bitand(x,y:real):real  побитное арифметическое  и
  refai(n:integer):real  ссылка на аналоговый вход n или 0 если его нет
  refdi(n:integer):real  ссылка на цифровой вход n или 0 если его нет
  refao(n:integer):real  ссылка на аналоговый выход n или 0 если его нет
  refdo(n:integer):real  ссылка на цифровой выход n или 0 если его нет
  refcalibr(n:integer):real ссылка на калибровку n или 0 если ее нет
  getai_n(n:integer):real   число точек,аналоговый вход номер n
  getdi_n(n:integer):real   число точек,цифровой вход номер n
  crvlen(c:real):real       длина кривой c (c-ссылка на кривую)
  crvx(c,i:real):real       координата x кривой c в точке номер i
  crvy(c,i:real):real       координата y кривой c в точке номер i
  crvget(c,x:real):real     интерполяция кривой c в точке с абсциссой x
  crvwhere(c,x:real):real   индекс ближайшей точки кривой c по абсциссе x
  crvfind(path:string):real поиск кривой по строке path = name win
                            name - имя кривой, win - имя окна
                            если win отсутствует, то поиск в базе данных DAQ
                            если win = *, то поиск в активном окне
                            если win = **, то поиск в верхнем окне для кривых
                            если name = *, то активная кривая в окне
                            возвращает ссылку на кривую
  crvname(c:real):string    имя кривой c
  crvlock(c:real):boolean   блокировка кривой c
  crvunlock(c:real):boolean разблокировка кривой c
  crvins(c,i,x,y:real):real вставка точки в кривую c
  crvdel(c,i,n:real:real    удаление точек из кривой c
  getai_xn(n:integer):real  x последней точки,аналоговый вход номер n
  getai_yn(n:integer):real  y последней точки,аналоговый вход номер n
  getdi_xn(n:integer):real  x последней точки,цифровой вход номер n
  getdi_yn(n:integer):real  y последней точки,цифровой вход номер n
  getai(n:integer;t:real):real    аналоговый вход номер n в момент времени t
  getdi(n:integer;t:real):real    цифровой вход номер n в момент времени t
  getai_xi(n:integer;i:real):real x точки номер i,аналоговый вход номер n
  getai_yi(n:integer;i:real):real y точки номер i,аналоговый вход номер n
  getdi_xi(n:integer;i:real):real x точки номер i,цифровой вход номер n
  getdi_yi(n:integer;i:real):real y точки номер i,цифровой вход номер n
  crvput(c,i,x,y:real):boolean    поместить в кривую c в точке номер i данные x,y
  crvinteg(c,a,b:real):real       интеграл кривой c в интервале a..b
  putev(w,c:integer; t,d0,d1:real):boolean
          генерирует событие: (what,chan,time,data0,data1)
  putao(n:integer;t,d:real):boolean
          генерирует аналоговое событие с номером канала n,
          временем t и данными d.
  putdo(n:integer;t,d:real):boolean
          генерирует цифровое событие с номером канала n,
          временем t и данными d.
  calibr(n:integer;d,p:real):real
          возвращает результат калибровочного преобразования
          n-номер калибровки,d-данные,p-параметр.
  power(x,y:real):real        степень x^y
  bitnot(x:real):real         побитное арифметическое отрицание
  getbitmask(n:integer):real  получить 2^n - то есть бит номер n
  fixerror(n:integer):boolean   фиксирует ошибку номер n
  registererr(s:string):integer регистрирует в системе новый код ошибки
                                с текстом сообщения s
  geterrcount(n:integer):real счетчик ошибок с кодом n
                              n=-2: все ошибки всех устройств
                              n=-1: все ошибки данного устройства
                              n=0..255   - ошибки устройства с кодом n
                              n=256..511 - ошибки всех устройств с кодом n-256
  inportb(n:integer):integer  ввод байта из порта ввода-вывода n
  outportb(n,d:integer):integer вывод байта d в порт ввода-вывода n
  inportw(n:integer):integer  ввод word из порта ввода-вывода n
  outportw(n,d:integer):integer вывод word d в порт ввода-вывода n
  inportl(n:integer):integer  ввод longword из порта ввода-вывода n
  outportl(n,d:integer):integer вывод longword d в порт ввода-вывода n
  isbit(x:real;n:integer):boolean есть ли в x бит номер n
  runcount:real                счетчик вызовов с момента старта DAQ
                               runcount = 1  при старте измерений,
                               runcount = _plusinf при остановке измерений
                               например:
                               if runcount=1 then OnStart else
                               if isinf(runcount) then OnStop else OnIdle;
  tm_new:real;                                        создание таймера
  tm_free(reftm:real):boolean;                        удаление таймера
  tm_gettime(reftm:real):real;                        время со старта таймера
  tm_start(reftm:real):boolean;                       старт таймера
  tm_stop(reftm:real):boolean;                        останов таймера
  tm_isstart(reftm:real):boolean;                     был ли старт таймера
  tm_event(reftm:real):boolean;                       событие таймера
  tm_curint(reftm:real):integer;                      текущий интервал
  tm_numint(reftm:real):integer;                      число интервалов
  tm_addint(reftm,ms:real):boolean;                   добавить интервал
  tm_getint(reftm:real;nint:integer):real;            узнать интервал
  tm_setint(reftm:real;nint:integer;ms:real):boolean; изменить интервал
  aimap(i,n:integer):real       битовая карта наличия кривых на ai(i..i+n-1)
  dimap(i,n:integer):real       битовая карта наличия кривых на di(i..i+n-1)
  aomap(i,n:integer):real       битовая карта наличия кривых на ao(i..i+n-1)
  domap(i,n:integer):real       битовая карта наличия кривых на do(i..i+n-1)
  diword(i,n:integer):real      битовая карта входных битов di(i..i+n-1)
  inot(i:integer):integer       побитная инверсия целого числа (not i)
  ior(i,j:integer):integer      побитное ИЛИ для целых чисел (i or j)
  ixor(i,j:integer):integer     побитное исключающее ИЛИ для целых (i xor j)
  iand(i,j:integer):integer     побитное И для целых чисел (i and j)
  ishift(i,j:integer):integer   сдвиг целых чисел (i*2^j)
  rshift(r:real;j:integer):real сдвиг вещественных чисел (r*2^j)
 АКЦИИ DAQ
  voice(string):boolean        звуковое сообщение в строке arg
  action(string):boolean       вызов метода action для списка устройств arg
  clear(string):boolean        вызов метода clear для списка устройств arg
  cleardevice(string):boolean  вызов метода cleardevice для списка устройств arg
  start(string):boolean        вызов метода start для списка устройств arg
  stop(string):boolean         вызов метода stop для списка устройств arg
  devmsg(string):real          посылка сообщения msg устройству dev
  debugout(string):boolean     вывод сообщения msg в файл отладки
  clearcurve(string):boolean   очистка кривых из списка arg
                               в списке можно задавать длину истории для
                               последующих кривых, например:
                               clearcurve('1000,crv1,2000,crv2,crv3');
  savecrw(string):boolean      выполняет сохранение кривых в файл CRW
  specmark(string):real        выдать Marker  спектрометрич. окна win
  specmarkl(string):real       выдать MarkerL спектрометрич. окна win
  specmarkr(string):real       выдать MarkerR спектрометрич. окна win
  specroil(string):real        выдать RoiL    спектрометрич. окна win
  specroir(string):real        выдать RoiR    спектрометрич. окна win
  windraw(string):boolean      перерисовать окно win
  winshow(string):boolean      показать     окно win
  winhide(string):boolean      спрятать     окно win
  winselect(string):boolean    активизировать окно
  global(string):real          вычисляет выражение expr в общем калькуляторе
 РАБОТА С ПОСЛЕДОВАТЕЛЬНЫМ ПОРТОМ
  comopen(string):boolean      открыть последовательный порт с параметрами из
                               указанной секции в файле конфигурации
                               например OpenCom('[SerialPort-COM2]');
  comclose:boolean             закрыть порт
  comcount:integer             число принятых символов в буфере ввода
                               или -1 если порт не открыт
  comwrite(string):boolean     запись строки в порт
  comread(integer):string      чтение указанного числа символов из порта
  ВНИМАНИЕ! ПРИМЕЧАНИЕ:
   1) Процедура comopen монопольно "захватывает" порт, не проверяя, занят ли
      он другими устройствами. Процедуры comread,comwrite также не проверяют,
      занят ли порт. По этой причине НЕЛЬЗЯ ставить на этот порт другие
      устройства DAQ (будет конфликт).
   2) Поэтому нельзя использовать вышеописанные процедуры для того же порта,
      на котором "повешены" устройства серии ADAM или другое устройство
      ProgramDevice, использующее этот же порт.
   3) Можно бесконфликтно использовать устройства ADAM и ProgramDevice на
      разных портах.
    Для работы в сети RS-485 используются другие 4 функции, так как в этом
  случае нужна специальная дисциплина опроса устройств - устройства должны
  опрашиваться последовательно, а не асинхронно, так как все устройства
  используют один канал связи. При этом в конфигурации должно присутствовать
  устройство типа Adam_Slot, а ссылка Adam_Slot_Device - указывать на это
  устройство. Устройство Adam_Slot служит для того, чтобы через него
  программа получила корректный доступ к RS-485.
  adam_status:integer
   Эта функция возвращает статус устройства в сети RS-485:
   0 = NotAvail   = Сеть RS-485 недоступна. Скорее всего это означает
                    неверную ссылку в переменной Adam_Slot_Device.
   1 = NoRequest  = Запрос не был возбужден. В этом состоянии можно вызывать
                    функцию adam_request для возбуждения запроса.
   2 = WaitQueue  = Запрос возбужден и ожидает очереди. Это значит,что запрос
                    передан в устройство Adam_Slot, но еще не передан в сеть
                    из-за того, что она была занята.
                    В этом состоянии можно безболезненно вызывать функцию
                    adam_request, если надо обновить сообщение.
   3 = WaitAnswer = Запрос возбужден и передан по линии связи, ожидается
                    ответ.
   4 = Answer     = Принят ответ на запрос. Его можно получить вызовом
                    функции adam_get('answer').
   5 = TimeOut    = Ошибка TimeOut. Это значит, что ответ на запрос не был
                    принят в течение установленного времени.
   Если статус был Answer или TimeOut, то после вызова функции статус
   автоматически сбрасывается в NoRequest, чтобы исключить возможность
   ошибочной повторной обработки уже обработанных запросов.
  adam_get(what:string):string
   Эта функция в зависимости от параметра what возвращает:
   'REQUEST'      - Последний возбужденный запрос.
   'ANSWER'       - Ответ на последний возбужденный запрос.
   'ADDRESS'      - Двухсимвольная строка - шестнадцатеричный адрес Adam_Slot.
   'PORT'         - Порт, к которому подключено устройство Adam_Slot.
   'USESCHECKSUM' - Используются ли контрольные суммы в устройстве.
                    Возвращает значение '0' или '1'.
   Регистр символов what не играет роли, однако не должно быть лишних
   пробелов.
  adam_request(request:string;timeout:integer):boolean
   Эта функция возбуждает запрос request, который передается в устройство
   Adam_Slot для последующей передачи в сеть, когда это станет возможным.
   Абсолютное значение timeout определяет время ожидания ответа на запрос
   в миллисекундах, причем если timeout=0, используется стандартное время,
   а если отрицательное - подавляется генерация ошибки TimeOut.
  adam_reqtime:real
   Возвращает время передачи последнего запроса в сеть RS-485.
 ОБРАБОТКА ПРЕРЫВАНИЙ
  irqinit(integer):boolean     устанавливает обработчик прерываний
  irqfree:boolean              снимает обработчик прерываний
  isisrnow:boolean             проверка, находимся ли мы в прерывании или нет
  Функции обработки прерываний требуют комментариев.
  Общий вид программ (в псевдокоде) с прерываниями такой:
   program IsrExample;
   const irq=7;
   begin
    if runcount=1 then begin
     if irqinit(irq) then InitHardware;
    end else
    if isinf(runcount) then begin
     StopHardware;
     irqfree;
    end else
    if isISRnow then begin
     HandleInterrupt;
    end else begin
     HandleIdle;
    end;
   end.
  1) После вызова irqinit инсталируется обработчик прерываний, однако
     прерывания маскируются до выхода из текущего вызова программы.
     В этот момент надо дать аппаратное разрешение прерываний (InitHardware)
  2) Перед вызовом Stop прерывания также маскируются, в этот момент надо
     сделать аппаратный запрет прерываний (StopHardware) и вызвать irqfree
  3) Если программа запущена из прерывания, isISRnow равно true.
     При этом НЕЛЬЗЯ делать критические операции связанные с выделением
     памяти или файловыми операциями: использовать строки в любом их виде,
     вызывать readln/writeln и т.д.
  4) Если irqinit вернул false, прерывание не инсталируется и выполнение
     программы будет продолжено не в режиме прерываний, а в режиме фонового
     опроса.
  5) На одно прерывание, кроме прерывания 0, нельзя установить две или более
     DAQ-программ. На прерывание таймера 0 можно устанавливать любое число
     DAQ-программ, которые будут вызываться каждые 55 миллисекунд.
 КОНФИГУРАЦИИ
  readini(s:string):string     чтение переменной (как строки) из конфигурационного
                               файла s='cfg sect name'
                               cfg  - имя файла конфигурации или пустая
                                      строка для текущей
                               sect - имя секции или пустая строка для
                                      секции текущего устройства
                               name - имя переменной
 КАМАК (интерфейс к библиотеке pkk)
  caminitpkk(p,b,i,c:integer):boolean           InitControllerPkk(p,b,i,c)
                                                p-тип 1-PKK3,2-PKK4,
                                                b-Base, i-Irq,
                                                c-список контроллеров
                                                биты 0..3 соответствуют
                                                контроллеру1..4
  camdonepkk:boolean                            DoneControllerPkk
  camtypepkk:integer                            ControllerPkk
  camnaf(c,n,a,f:integer):boolean               команда CNAF
                                                C-крейт N-станция
                                                A-субадрес F-функция
  camwrite(c,n,a,f,d:integer):boolean           запись 16-битных данных
  camwritelong(c,n,a,f:integer;d:real):boolean  запись 24-битных данных
  camread(c,n,a,f:integer):integer              чтение 16-битных данных
  camreadlong(c,n,a,f:integer):real             чтение 24-битных данных
  camdeclare(c:integer):boolean                 CrateDeclare(c)
  camzero(c:integer):boolean                    CamZero(c)
  camclear(c:integer):boolean                   CamClear(c)
  camsetinhib(c:integer):boolean                SetInhibit(c)
  camclrinhib(c:integer):boolean                ClearInhibit(c)
  camsetlmask(c,m:integer):boolean              SetLamMask(c,m)
  setreqmask(c,m:integer):boolean               SetRequestMask(c,m)
  camgetlmask(c:integer):integer                GetLamMask(c)
  camgetrword(c:integer):integer                GetRequestWord(c)
  camgetbcard(c:integer):integer;               GetBitCard(c)
  camisx(c:integer):boolean                     IsX(c)
  camisxq(c:integer):boolean                    IsXQ(c)
  camgetcsr(c:integer):integer                  GetCSR(c)
  camenabint(c,m:integer):boolean               CamEnableIntr(c,m)
  camdisabint(c,m:integer):boolean              CamDisableInt(c,m)
  camismyreq(c:integer):boolean                 IsMyRequest(c)
  camgetrcode(c:integer):integer                GetRequestCode(c)
  camclrreq(c,m:integer):boolean                ClearRequest(c,m)
  camsaveregs(c:integer):boolean                PkkSaveRegs(c)
  camrestregs(c:integer):boolean                PkkRestRegs(c)
  cammaskofst(c:integer):integer                GetMaskOfStation(c)
  camstoflam(c:integer):integer                 GetStationOfLam(c)
  camgetrmask(c:integer):integer                GetRequestMask(c)
  pkkclrreq(c:integer):boolean                  PkkClearRequest(c)
  pkkenabint(c:integer):boolean                 PkkEnableInterrupt(c)
  pkkdisabint(c:integer):boolean                PkkDisableInterrupt(c)
 ГЛОБАЛЬНЫЕ ПЕРЕМЕННЫЕ (ТЕГИ)
  Для взаимосвязи между программами DAQ Pascal можно передавать данные
  через входы-выходы.  Однако такая передача опосредована через fifo и
  диспетчер событий. В качестве альтернативного пути для передачи параметров
  или мгновенных значений можно использовать глобальные переменные или ТЕГИ,
  доступные через функции, описанные далее.
  Тег идентифицируется целочисленным индексом > 0.
  Значение 0 в результате функции означает,что тег не найден,не создан и т.д.
  Поддерживается 3 типа тегов: 1=INTEGER, 2=REAL, 3=STRING.
  При попытке чтения тега неверного типа возвращается 0 или пустая строка.
  При попытке записи тега неверного типа возвращается false.
  Теги могут быть автоматически созданы при загрузке конфигурации
  в секции [TagList] как в примере:
   [TagList]
   addr = INTEGER $300
   gain = REAL 2.5
   name = STRING PCL-818L
  FindTag(Name:string):integer;              тег по данному имени или 0
  InitTag(Name:string; Typ:integer):integer; инициализация тега или 0
  FreeTag(tag:integer):boolean;              удаление тега если он был
  TypeTag(tag:integer):integer;              тип тега или 0 если нет тега
  NameTag(tag:integer):string;               имя тега или пустая строка
  igetTag(tag:integer):integer;              значение integer-тега или 0
  rgetTag(tag:integer):double;               значение real-тега или 0
  sgetTag(tag:integer):string;               значение string-тега или 0
  isetTag(tag:integer;i:integer):boolean;    запись integer-тега
  rsetTag(tag:integer;r:double):boolean;     запись real-тега
  ssetTag(tag:integer;s:string):boolean;     запись string-тега
 АНИМАЦИЯ МНЕМОСХЕМ
  Программы на DAQ PASCAL можно использовать для написания процедуры
  стратегии (или анимации) мнемосхем. Мнемосхемы содержат СЕНСОРЫ -
  чувствительные области для отображения и управления. Анимация мнемосхемы
  состоит в том, что каждый сенсор связывается с устройством и кривой или
  тегом, а также в назначении процедур, выполняемых по нажатию правой и
  левой кнопок на сенсоре. Сенсор автоматически отображает текущее состояние
  связанной с ним кривой или тега, а устройство реализует алгоритм нажатия.
  Для анимации сенсора через программу (например Prog) на DAQ PASCAL надо:
  1) Завести в секции [TagList] тег (например Btn1) типа INTEGER или REAL,
  2) Связать сенсор (например Button1) с этим тегом через конструкцию
     Link sensor Button1 with device Prog tag Btn1
  3) Описать в программе Prog действия по нажатии кнопок на сенсоре.
     Например, для реализации нажатия кнопки программа модифицирует
     значение тега на инверсное (оно прорисуется автоматически).
  При написании программы обработки нажатий кнопки надо использовать
  следующие функции:
  clickbutton:integer  0 если не было нажатия кнопок
                       1 если нажата левая кнопка, 2 если правая
  clicksensor:string   имя сенсора, который был нажат
  clicktag:integer     тег, связанный с нажатым сенсором, значение которого
                       отображает сенсор или 0 если нет связи
  Для правильной обработки событий сначала надо вызывать clickbutton, и лишь
  при значении > 0 начинать анализ нажатий. Иначе цикл обработки может быть
  неоправданно перегружен.
  Пример (реакция на левую кнопку,нажатие-отпускание кнопки):
   Файл конфигурации:
     [Windows]
     Panel = Circuit_Window
     [Panel]
     Link sensor Button1 with device Prog tag Btn1
     [TagList]
     Btn1 = integer 0
     [DeviceList]
     Prog = device software program
     [Prog]
     ProgramSource = prog.pas
   Файл prog.pas:
    program Prog;
    var b:boolean;
    begin
     if clickbutton=1 then begin
      if (clicksensor='BUTTON1') then begin
       if igettag(clicktag)<>0 then b:=isettag(clicktag,0)
                               else b:=isettag(clicktag,1);
      end;
     end;
    end;
 КАЛЕНДАРЬ:
  Группа функций позволит переводить время, прочитанное функцией msecnow
  в календарную дату и время суток
  ms2year(ms:real):integer    преобразует время msecnow в календарный год
  ms2month(ms:real):integer   преобразует время msecnow в календарный месяц
  ms2day(ms:real):integer     преобразует время msecnow в календарный день
  ms2hour(ms:real):integer    преобразует время msecnow в календарные часы
  ms2min(ms:real):integer     преобразует время msecnow в календарные минуты
  ms2sec(ms:real):integer     преобразует время msecnow в календарные секунды
 ЗАГРУЗКА ДИНАМИЧЕСКИХ БИБЛИОТЕК
   Встроенный язык DAQ PASCAL предназначен для создания сценариев обработки
   данных в относительно медленных измерениях (время реакции - миллисекунды).
   Байт-код, генерируемый компилятором, уступает по производительности коду
   Borland Pascal. Если в программе требуется быстрая обработка данных, есть
   возможность создания и вызова из программ DAQ PASCAL динамических
   библиотек. Сами динамические библиотеки создаются из шаблона, который
   создается через меню Файл|Создать и должны удовлетворять ряду специальных
   договоренностей (см. файл system\_daqevnt.pas):
    * наличие в экспорте библиотеки функции
       CRW_DAQ_DRIVER(MESSAGE:CRW_DAQ_DLL_MESSAGE_PTR):boolean
    * данные передаются в библиотеку и принимаются из библиотеки через
      запись типа CRW_DAQ_DLL_MESSAGE, которая содержит поля:
      1) COMMAND  : integer - код требуемой операции
      2) INSTANCE : integer - код для идентификации DLL
      3) DEVICE   : pointer - указатель вызвавшего устройства
      4) SERVICE  : CRW_DAQ_DLL_SERVICE - таблица функций для обмена данными
                    между DLL и программой и т.д.
   Компилировать и редактировать библиотеки можно также в программе CRW, не
   привлекая других средств.
   Вызов внешних библиотек обеспечивается такими функциями:
  daqdllinit(name:string):integer
   Загружает динамическую библиотеку *.dll с программой обработки данных
   из файла "name".
   Можно указывать имя относительно основного конфигурационного файла.
   Функция возвращает ссылку на библиотеку для доступа к ней или 0 при ошибке.
   Вызов этой функции обязателен перед началом работы с библиотекой.
  daqdllfree(ref:integer):boolean
   Функция удаляет библиотеку "ref", загруженную функцией daqdllinit.
   Вызов этой функции обязателен при завершении работы.
  daqdllcall(ref,command:integer):boolean
   Вызов библиотеки "ref" для выполнения команды "command".
   Все данные в обе стороны передаются через функции SERVICE.
 ПАРАМЕТРЫ СИСТЕМЫ
  paramstr(arg:string):string
   Это мощная функция, через которую можно узнать многие параметры
   системы, а также выполнять некоторые строковые преобразования.
   Пусть строка arg состоит из слов w1,w2,w3, разделенных пробелом или
   знаком табуляции. Регистр символов не играет роли.
   Первое слово w1 задает имя параметра или функции,а w2,w3 - параметры
   этой функции.
   Результат вызова paramstr в зависимости от w1 приведен в таблице:
   '0'..'9'          - возвращается  параметр командной строки
   'PROGNAME'        - возвращается полное имя основной программы CRW
   'DAQPROGRAM'      - возвращается имя программы Daq Pascal
   'HOMEDIR'         - возвращается путь основной программы CRW
   'STARTUPPATH'     - возвращается путь основной программы CRW
   'TEMPPATH'        - возвращается путь временных файлов CRW
   'DAQCONFIGPATH'   - возвращается каталог основного файла конфигурации DAQ
   'DAQCONFIGFILE'   - возвращается имя основного файла конфигурации DAQ
   'DAQDATAPATH'     - возвращается путь каталога данных DAQ
   'DAQBACKUPFILE'   - возвращается имя файла автосохранения DAQ
   'GETCURDIR'       - возвращается текущий каталог
   'DEVICENAME'      - возвращается имя текущего устройства или устройства номер w2, если указано
   'DEVICEMODEL'     - возвращается модель текущего устройства или устройства номер w2, если указано
   'DEVICEFAMILY'    - возвращается семейство текущего устройства или устройства номер w2, если указано
   'CURVENAME'       - возвращается имя кривой номер w2
   'CURWINNAME'      - возвращается имя окна кривых номер w2
   'TABWINNAME'      - возвращается имя окна таблиц номер w2
   'CIRWINNAME'      - возвращается имя окна мнемосхем номер w2
   'SPEWINNAME'      - возвращается имя окна спектров номер w2
   'GETENV'          - возвращается переменная окружения с именем w2
   'FEXPAND'         - возвращается полное имя файла w2
   'EXTRACTFILEPATH' - возвращается путь файла w2
   'EXTRACTFILENAME' - возвращается имя файла w2
   'EXTRACTFILEEXT'  - возвращается расширение файла w2
   'ADDBACKSLASH'    - добавить если надо '\' в конец пути w2
   'MAKERELATIVEPATH'- относительный путь файла w2 относительно файла w3
   'FORCEPATH'       - приставить (насильно)  путь w2 к файлу w3
   'DEFAULTPATH'     - приставить (если надо) путь w3 к файлу w2
   'FORCEEXTENSION'  - приставить (насильно)  к файлу w2 расширение w3
   'DEFAULTEXTENSION'- приставить (если надо) к файлу w2 расширение w3
 УПРАВЛЕНИЕ ЗАДАЧАМИ
  task_init(cmdline:string):integer - создает задачу
  task_free(tid:integer):boolean    - уничтожает задачу (не обязательно процесс)
  task_ref(tid:integer):integer     - ссылка, если <> 0, задача tid существует
  task_pid(tid:integer):integer     - PID процесса или 0 если не был создан
  task_run(tid:integer):boolean     - запуск задачи, создание процесса
  task_wait(tid,timeout:integer):boolean - ожидание завершения
                                           true=задача еще выполняется
  task_send(tid:integer; data:string):integer - послать данные в канал stdin
  task_recv(tid,maxlen:integer):string  - получить данные из канала stdout
  task_txcount(tid:integer):integer - счетчик данных в канале stdin
  task_rxcount(tid:integer):integer - свободное место в канале stdin
  task_txspace(tid:integer):integer - счетчик данных в канале stdout
  task_rxspace(tid:integer):integer - свободное место в канале stdout
  task_result(tid:integer):integer  - код завершения задачи
  task_kill(tid,how,exit_code,timeout:integer):boolean - убить процесс
  task_ctrl(tid:integer; param:string):string - задать параметры процесса
                                    AppName,CmdLine,HomeDir,
                                    StdInPipeSize,StdOutPipeSize,
                                    StdInFileName,StdOutFileName
                                    Display
  getcomspec:string  - командный процессор COMSPEC
  shellexecute(arg:string):integer
   Вызывает внешние программы, открывает файлы, это вызов Windows.ShellExecute.
   arg = cmd | file | cmdline | dir | show
        cmd     - open, print, explore
        file    - имя файла
        cmdline - параметры командной строки
        dir     - каталог запуска
        show    - HIDE,MAXIMIZE,MINIMIZE,RESTORE,SHOW,SHOWDEFAULT,SHOWMAXIMIZED,
                  SHOWMINIMIZED,SHOWMINNOACTIVE,SHOWNA,SHOWNOACTIVATE,SHOWNORMAL
   параметры, кроме cmd, file/dir, могут быть опущены (заменены пробелами).
 ОПЕРАТОРЫ: &, |, ~ - синонимы AND, OR, NOT, соответственно.
 CASE -  не более 30 позиций и без секции else.
 Примечания к версии CRW32:
 *************************
 1). В CRW32 тип Integer имеет 32 bit, а в CRW16 - 16 bit.
 2). В CRW32 заглушены функции
     irqXXX    - прерывания
     pkkXXX    - работа с PKK
     camXXX    - КАМАК через PKK
     daqdllXXX - вызов DLL
     Часть функций может быть восстановлена со временем.
  3) Функции, требующие прорисовок, типа window,winshow,action &etc. выполняются
     не сразу в момент вызова, а помещают команду на выполнение в FIFO, а реально
     выполняют команду в основном потоке в методе Idle. Отсюда следует, что
     во-первых эти функции не занимают время в потоке опроса, а во-вторых что
     они выполняются с задержкой
 *******************************************************************************
 }
type
 TProgramDevice = class(TSoftwareDevice)
 public
  DaqPascal : TDaqPascal;
  DaqEditor : TFormDaqPascalEditor;
  Console   : TFormConsoleWindow;
  constructor Create(const aName:LongString);
  destructor  Destroy; override;
  procedure   Config(FileName:LongString); override;
  procedure   Animate; override;
  function    GetProperty(P:TText):TText; override;
  function    Start:Boolean; override;
  procedure   Stop; override;
  procedure   Idle; override;
  procedure   Handler; override;
  procedure   Watchdog; override;
  function    GotEvents:Boolean; override;
  procedure   RightButtonPropertyDialog; override;
  function    NewPropertyDialog:TDaqDevicePropertyDialog; override;
  procedure   DeferredCommandProblem(const Cmd:LongString); override;
  function    OpenDaqPascalEditor:Boolean;
  procedure   NotifyCompile(how:Boolean);
  function    CanCompileEditor:Boolean;
  function    CompileEditor:Boolean;
  function    ReCompile:Boolean;
  function    WantedConsoleName:LongString;
  function    IsSameConsoleName(aName:LongString):Boolean;
  function    OpenConsole:Boolean;
  procedure   UpdateConsole;
  function    IsOutFifoCleanTime(aReset:Boolean=false):Boolean;
  function    OpenExternEditor(row:Integer=0; col:Integer=0;
                      err:Integer=0; msg:LongString=''; Mode:Integer=1):Boolean;
  function    HandleMessage(const aMsg:LongString; aFlags:Cardinal=hf_Default):Double; override;
  procedure   ExecuteDeferredCommand(var Cmd, Arg:LongString); override;
 private
  myWatchdogLatch    : TLatch;
  myWatchdogStarted  : Double;
  myWatchdogDeadline : Integer;
  myRunningFlag      : Boolean;
  myStopperFlag      : Boolean;
  myUsesExternEditor : Boolean;
  myOutFifoCleanTime : QWord;
 private
  function    GetWatchdogStarted:Double;
  procedure   SetWatchdogStarted(aTime:Double);
  function    GetWatchdogDeadline:Integer;
  procedure   SetWatchdogDeadline(aDeadline:Integer);
  function    GetUsesExternEditor:Boolean;
  procedure   SetUsesExternEditor(aValue:Boolean);
  function    GetUsesInternEditor:Boolean;
  procedure   SetUsesInternEditor(aValue:Boolean);
 public
  property    WatchdogStarted  : Double  read GetWatchdogStarted  write SetWatchdogStarted;
  property    WatchdogDeadline : Integer read GetWatchdogDeadline write SetWatchdogDeadline;
  property    UsesExternEditor : Boolean read GetUsesExternEditor write SetUsesExternEditor;
  property    UsesInternEditor : Boolean read GetUsesInternEditor write SetUsesInternEditor;
 public
  class function PollingViolationCounter:SizeInt;
  class function IncPollingViolationCounter:SizeInt;
 public
  class var EnableEditDaqSite : Boolean;
  class var OutFifoCleanDelay : Integer;
  class var OutFifoCleanRunCount : Double;
 public
  function FormatErrorDetails:LongString;
 end;

implementation

uses
 Form_DaqDeviceControl,
 Form_DaqPascalPropertyDialog;

 {
 *******************************************************************************
 TProgramDevice implementation
 *******************************************************************************
 }
 {
 Create PROGRAM device
 }
constructor TProgramDevice.Create(const aName:LongString);
begin
 inherited Create(aName);
 SetDeviceModel('PROGRAM');
 AnalogFifoSize:=64;
 DigitalFifoSize:=64;
 NumAnalogInputs:=0;
 NumDigitalInputs:=0;
 NumAnalogOutputs:=0;
 NumDigitalOutputs:=0;
 NumCalibrations:=0;
 DaqPascal:=TDaqPascal.Create(Self);
 DaqEditor:=nil;
 Console:=nil;
 myWatchdogLatch:=NewLatch;
 myWatchdogLatch.Master:=@myWatchdogLatch;
 myWatchdogStarted:=0;
 WatchdogDeadline:=DefaultDaqWatchdogDeadline;
 myRunningFlag:=false;
 myStopperFlag:=false;
end;

 {
 *******************************************************************************
 Kill PROGRAM device
 *******************************************************************************
 }
destructor TProgramDevice.Destroy;
begin
 Kill(Console);
 Kill(DaqEditor);
 Kill(DaqPascal);
 Kill(myWatchdogLatch);
 inherited Destroy;
end;

const
 ThePollingViolationCounter : TAtomicCounter = nil;

procedure InitPvCounters;
begin
 LockedInit(ThePollingViolationCounter);
end;

procedure FreePvCounters;
begin
 LockedFree(ThePollingViolationCounter);
end;

class function TProgramDevice.PollingViolationCounter:SizeInt;
begin
 Result:=LockedGet(ThePollingViolationCounter);
end;

class function TProgramDevice.IncPollingViolationCounter:SizeInt;
begin
 Result:=LockedInc(ThePollingViolationCounter);
end;

function TProgramDevice.GetWatchdogStarted:Double;
begin
 Result:=0;
 if Assigned(Self) then
 try
  myWatchdogLatch.Lock;
  try
   Result:=myWatchdogStarted;
  finally
   myWatchdogLatch.UnLock;
  end;
 except
  on E:Exception do BugReport(E,Self,'GetWatchdogStarted');
 end;
end;

procedure TProgramDevice.SetWatchdogStarted(aTime:Double);
begin
 if Assigned(Self) then
 try
  myWatchdogLatch.Lock;
  try
   myWatchdogStarted:=aTime;
  finally
   myWatchdogLatch.UnLock;
  end;
 except
  on E:Exception do BugReport(E,Self,'SetWatchdogStarted');
 end;
end;

function TProgramDevice.GetWatchdogDeadline:Integer;
begin
 if Assigned(Self) then Result:=myWatchdogDeadline else Result:=0;
end;

procedure TProgramDevice.SetWatchdogDeadline(aDeadline:Integer);
begin
 if Assigned(Self) then myWatchdogDeadline:=Max(0,aDeadline);
end;

function TProgramDevice.GetUsesExternEditor:Boolean;
begin
 if Assigned(Self) then Result:=myUsesExternEditor else Result:=false;
end;

procedure TProgramDevice.SetUsesExternEditor(aValue:Boolean);
begin
 if Assigned(Self) then myUsesExternEditor:=aValue;
end;

function TProgramDevice.GetUsesInternEditor:Boolean;
begin
 if Assigned(Self) then Result:=not myUsesExternEditor else Result:=false;
end;

procedure TProgramDevice.SetUsesInternEditor(aValue:Boolean);
begin
 if Assigned(Self) then myUsesExternEditor:=not aValue;
end;

 {
 *******************************************************************************
 Read PROGRAM device configuration, like this one:
 [DeviceName]
 AnalogInputs = 1
 DigitalInputs = 0
 AnalogOutputs = 1
 DigitalOutputs = 0
 Calibrations = 1
 DebugMode = 1
 Calibration#0 = ..\calibr\calib0.cal U(mV) I(A) * Line Line 0 10
 ProgramSource = ..\daqpas\source.pas
 Link AnalogInput  0  with curve u smoothing 0.02 -1 2 2
 Link AnalogOutput 0  with curve i tolerance 0.02 0.01  history 1000
 *******************************************************************************
 }
procedure TProgramDevice.Config(FileName:LongString);
var i:Integer;
begin
 i:=0;
 FileName:=UnifyFileAlias(FileName);
 inherited Config(FileName);
 if ReadIniFileInteger(FileName,DevSection,'AnalogInputs%i',i)
 then NumAnalogInputs:=i
 else NumAnalogInputs:=0;
 if ReadIniFileInteger(FileName,DevSection,'AnalogOutputs%i',i)
 then NumAnalogOutputs:=i
 else NumAnalogOutputs:=0;
 if ReadIniFileInteger(FileName,DevSection,'DigitalInputs%i',i)
 then NumDigitalInputs:=i
 else NumDigitalInputs:=0;
 if ReadIniFileInteger(FileName,DevSection,'DigitalOutputs%i',i)
 then NumDigitalOutputs:=i
 else NumDigitalOutputs:=0;
 if ReadIniFileInteger(FileName,DevSection,'Calibrations%i',i)
 then NumCalibrations:=i
 else NumCalibrations:=0;
 for i:=0 to NumCalibrations-1 do
 ReadCalibration(i,FileName,DevSection,'Calibration#'+d2s(i));
 if DaqPascal.Ok then begin
  if ReadIniFileInteger(FileName,DevSection,'StdInFifo%i',i) then
  if (i>0) and (i<(MaxInt shr 10)) then DaqPascal.InpFifo.Size:=i shl 10;
  if ReadIniFileInteger(FileName,DevSection,'StdOutFifo%i',i) then
  if (i>0) and (i<(MaxInt shr 10)) then DaqPascal.OutFifo.Size:=i shl 10;
  if not ReadIniFileInteger(FileName,DevSection,'DebugMode%i',i) then i:=0;
  DaqPascal.DebugMode:=HasFlags(i,1);
  if HasFlags(i,2) then OpenConsole;
  if ReadIniFilePath(FileName,DevSection,'ProgramSource',
                     ExtractFilePath(FileName),DaqPascal.SrcFile)
  then begin
   DaqPascal.SrcFile:=DefaultExtension(DaqPascal.SrcFile,'.pas');
   DaqPascal.SrcFile:=UnifyFileAlias(DaqPascal.SrcFile); // NB!
   DaqPascal.LstFile:=ForceExtension(DaqPascal.SrcFile,'.lst');
   DaqPascal.LstFile:=ForcePath(Daq.TempPath,DaqPascal.LstFile);
   if FileExists(DaqPascal.SrcFile) then begin
    DaqPascal.Compile(DaqPascal.SrcFile,DaqPascal.LstFile);
    if not DaqPascal.CompileOk
    then Daq.AddWarning(Name+' : Error compile '+DaqPascal.SrcFile);
    SendToMainConsole('@silent @integrity load.pas '+DaqPascal.SrcFile+EOL);
   end else Daq.AddWarning(Name+' : File not found '+DaqPascal.SrcFile);
  end else Daq.AddWarning(Name+' : ProgramSource = ... not found');
 end else Daq.AddWarning(Name+' : Memory error');
 if ReadIniFileInteger(FileName,DevSection,'WatchdogDeadline%i',i)
 or ReadIniFileInteger(FileName,'[DAQ]','WatchdogDeadline%i',i)
 or ReadIniFileInteger(SysIniFile,'[DaqSys]','WatchdogDeadline%i',i)
 then WatchdogDeadline:=Max(i,0);
 if DaqPascal.Ok then begin
  if ReadIniFileInteger(FileName,DevSection,'PostMortalSleepTime%i',i)
  or ReadIniFileInteger(FileName,'[DAQ]','PostMortalSleepTime%i',i)
  or ReadIniFileInteger(SysIniFile,'[DaqSys]','PostMortalSleepTime%i',i)
  then DaqPascal.MortalDelay:=Max(i,0);
 end;
end;

 {
 *******************************************************************************
 PROGRAM device animation: standard links + try to link Adam_Device_Slot
 *******************************************************************************
 }
procedure  TProgramDevice.Animate;
var i,RefCount:Integer; Slot:LongString; Dev:TDaqDevice;
 procedure CheckAdamSlot(Device:TDaqDevice);
 begin
  if (Device is TProgramDevice) then
  if (Device as TProgramDevice).DaqPascal is TDaqPascal then
  if DaqPascal.Adam_Slot=(Device as TProgramDevice).DaqPascal.Adam_Slot then
  inc(RefCount);
 end;
begin
 Slot:='';
 inherited Animate;
 if DaqPascal.Ok then
 if ReadIniFileAlpha(Daq.ConfigFile,DevSection,'Adam_Slot_Device%a',Slot) then begin
  Dev:=FullDaqDeviceList.Find(Slot);
  if (Dev is TAdamSlot) then begin
   DaqPascal.Adam_Slot:=TAdamSlot(Dev);
   RefCount:=0;
   for i:=0 to FullDaqDeviceList.Count-1 do CheckAdamSlot(FullDaqDeviceList[i]);
   if RefCount>1 then begin
    DaqPascal.Adam_Slot:=nil;
    Daq.AddWarning('Device:'+Name+' -> Dublicate Adam_Slot_Device = '+Slot);
   end;
  end else begin
   DaqPascal.Adam_Slot:=nil;
   Daq.AddWarning('Device:'+Name+' -> Invalid Adam_Slot_Device = '+Slot);
  end;
 end;
end;

 {
 *******************************************************************************
 Get PROGRAM device properties as text
 *******************************************************************************
 }
function   TProgramDevice.GetProperty(P:TText):TText;
const SpaceLines = 3;
var
 i : Integer;
begin
 GetProperty:=inherited GetProperty(P);
 for i:=0 to NumCalibrations-1 do
 P.AddLn(FormatCalibration('Calibration#'+d2s(i),Calibration[i]));
 if DaqPascal.Ok then begin
  P.AddLn(Format('StdInFifo = %d KB',[BytesToKb(DaqPascal.InpFifo.Size)]));
  P.AddLn(Format('StdOutFifo = %d KB',[BytesToKb(DaqPascal.OutFifo.Size)]));
  P.AddLn(Format('StdInFifoGrowFactor = %d',[DaqPascal.InpFifo.GrowFactor]));
  P.AddLn(Format('StdOutFifoGrowFactor = %d',[DaqPascal.OutFifo.GrowFactor]));
  P.AddLn(Format('StdInFifoGrowLimit = %d KB',[BytesToKb(DaqPascal.InpFifo.GrowLimit)]));
  P.AddLn(Format('StdOutFifoGrowLimit = %d KB',[BytesToKb(DaqPascal.OutFifo.GrowLimit)]));
  P.Addln('DebugMode = '+d2s(ord(DaqPascal.DebugMode) or 2*ord(Console.Ok)));
  if DaqPascal.Adam_Slot.Ok
  then P.Addln('Adam_Slot_Device = '+DaqPascal.Adam_Slot.Name)
  else P.Addln('Adam_Slot_Device = '+RusEng('(нет)','(none)'));
  if DaqPascal.CompileOk
  then P.Addln(RusEng('ПРОГРАММА УСПЕШНО СКОМПИЛИРОВАНА','PROGRAM COMPILED OK'))
  else P.Addln(RusEng('ОШИБКА КОМПИЛЯЦИИ ','COMPILE ERROR ')
              +d2s(DaqPascal.ExceptResult)+': " '+DaqPascal.ErrorMessage+' "');
  P.Addln('ProgramSource = '+Daq.FileRel(DaqPascal.SrcFile));
  P.Addln('Compiler.itabmax = '+d2s(DaqPascal.itabmax));
  P.Addln('Compiler.btabmax = '+d2s(DaqPascal.btabmax));
  P.Addln('Compiler.atabmax = '+d2s(DaqPascal.atabmax));
  P.Addln('Compiler.rtabmax = '+d2s(DaqPascal.rtabmax));
  P.Addln('Compiler.ctabmax = '+d2s(DaqPascal.ctabmax));
  P.Addln('Compiler.dtabmax = '+d2s(DaqPascal.dtabmax));
  P.Addln('Compiler.stabmax = '+d2s(DaqPascal.stabmax));
  P.Addln('Compiler.slenmax = '+d2s(DaqPascal.slenmax));
  P.AddLn(Format('WatchdogDeadline = %d ms',[WatchdogDeadline]));
  if FileExists(DaqPascal.SrcFile) then begin
   for i:=1 to SpaceLines do P.Addln('');
   P.Addln(RusEng('******************************'));
   P.Addln(RusEng('***** ИСПОЛНЯЕМЫЙ ТЕКСТ ******',
                  '*****  PROGRAM SOURCE   ******'));
   P.Addln(RusEng('******************************'));
   P.ReadFile(DaqPascal.SrcFile,true);
  end;
  if FileExists(DaqPascal.LstFile) then begin
   for i:=1 to SpaceLines do P.Addln('');
   P.Addln(RusEng('******************************'));
   P.Addln(RusEng('***** ОТЛАДОЧНЫЙ ТЕКСТ  ******',
                  '*****  DEBUGGER OUTPUT  ******'));
   P.Addln(RusEng('******************************'));
   P.ReadFile(DaqPascal.LstFile,true);
  end;
 end else P.Addln(RusEng('ОШИБКА ПАМЯТИ КОМПИЛЯТОРА','COMPILER ALLOCATION ERROR'));
end;

 {
 *******************************************************************************
 Start PROGRAM device
 *******************************************************************************
 }
function TProgramDevice.Start;
begin
 Result:=false;
 if inherited Start then begin
  if DaqPascal.Ok then begin
   if not DaqPascal.CompileOk then FixError(ecDaqPascalStart);
   IsOutFifoCleanTime(true);
   DaqPascal.Start;
  end else FixError(ecDaqPascalStart);
  myRunningFlag:=DaqPascal.Ok and DaqPascal.CompileOk;
  myStopperFlag:=false;
  Result:=true; 
 end;
end;

 {
 *******************************************************************************
 Stop PROGRAM device
 Before stop call DAQ program with RunCount=+INF in main VCL thread (!obsolete)
 In new (more correct) version myStopperFlag is set to terminate Polling thread
 and wait for Polling thread termination. Be accurate, it's VIP part of code.
 Also update console.
 *******************************************************************************
 }
procedure TProgramDevice.Stop;
const UseObsolete:Boolean=false;
begin
 if InquiryTimer.isStart and DaqPascal.Ok then begin
  Polling.Enable(false);
  DaqPascal.RunCount:=_plusinf;
  myStopperFlag:=true; // Signal to stop
  if UseObsolete then begin
   Handler;
  end else begin
   Polling.Awake;
   Polling.Enable(true); // NB !!! waiting conditions must be same as in Handler to avoid hanging !!!
   while InquiryTimer.IsStart and DaqPascal.Ok and DaqPascal.CompileOk and myRunningFlag do Sleep(TPolling.DefPollPeriod);
   Polling.Enable(false);
  end;
  UpdateConsole;
  DaqPascal.Stop;
 end;
 inherited Stop;
end;

 {
 *******************************************************************************
 This procedure will called in main VCL thread each 55 ms
 Flush console window fifo
 *******************************************************************************
 }
procedure  TProgramDevice.Idle;
begin
 inherited Idle;
 if DaqPascal.Ok and DaqPascal.CompileOk then UpdateConsole;
end;

 {
 *******************************************************************************
 This procedure will calls in Polling thread.
 If program compiled, run DAQ PASCAL code.
 *******************************************************************************
 }
procedure  TProgramDevice.Handler;
begin
 if InquiryTimer.IsStart then
 if DaqPascal.Ok and DaqPascal.CompileOk then begin
  if myRunningFlag then begin
   WatchdogStarted:=mSecNow;
   if not DaqPascal.Interpret then begin
    FixError(ecDaqPascalHalt);
    if DaqPascal.DebugMode
    then DebugOut(stdfDebug,Name+' -> RunTime Error '+d2s(DaqPascal.ExceptResult)+
                      ' " '+DaqPascal.ErrorMessage+' "');
   end;
   DaqPascal._Flush;
   WatchdogStarted:=0;
   // Control thread violations (if running in wrong thread)
   if (GetCurrentThreadId<>Polling.ThreadID) then IncPollingViolationCounter;
   // Got a signal to stop, so clear myRunningFlag
   if myStopperFlag then begin
    DaqPascal.CleanOnStop;
    myRunningFlag:=false;
    myStopperFlag:=false;
   end;
  end;
 end;
end;

 {
 *******************************************************************************
 This function works in Polling thread. If got events, return true to handle it.
 Possible events are: InquiryTimer, Click, Awake, InpFifo.
 *******************************************************************************
 }
function TProgramDevice.GotEvents:Boolean;
begin
 Result:=(inherited GotEvents) or (DaqPascal.InpFifo.Count>0);
end;

 {
 *******************************************************************************
 Check device deadline
 Executed in Daq.Watchdog thread, so be attentive.
 *******************************************************************************
 }
procedure TProgramDevice.Watchdog;
var WatchdogUptime:Double;
begin
 if InquiryTimer.IsStart then
 if DaqPascal.Ok and DaqPascal.CompileOk then begin
  if (WatchdogDeadline>0) then begin
   WatchdogUptime:=WatchdogStarted;
   if WatchdogUptime>0 then WatchdogUptime:=mSecNow-WatchdogUptime;
   //if DaqPascal.runcount>1 then
   //if not IsInf(DaqPascal.runcount) then
   if (WatchdogUptime>WatchdogDeadline) then begin
    Daq.ConsoleEcho(Format('DAQ SYSTEM WARNING: WATCHDOG DEADLINE DETECTED.'+EOL
     +'No reply from device %s during %1.3f sec. Deadline is %1.3f sec.',
     [Self.Name,WatchdogUptime*1e-3,WatchdogDeadline*1e-3]));
    if SysLogNotable(SeverityOfDaqWatchdog)
    then SysLogNote(0,SeverityOfDaqWatchdog,SysLogSign,
         Format('DAQ Watchdog deadline after %1.3f sec.',[WatchdogUptime*1e-3]));
    DaqPascal.BreakFlag:=ecWatchdogDeadline;
   end;
  end;
 end;
end;

 {
 *******************************************************************************
 Switch off property dialog on right button
 *******************************************************************************
 }
procedure TProgramDevice.RightButtonPropertyDialog;
begin
end;

 {
 *******************************************************************************
 Open property dialog.
 *******************************************************************************
 }
function TProgramDevice.NewPropertyDialog:TDaqDevicePropertyDialog;
begin
 Result:=NewFormDaqPascalPropertyDialog(Self);
 if Result.Ok then begin
  Result.Show;
  Result.Caption:=Model+': '+Name;
 end;
end;

 {
 *******************************************************************************
 Procedure reports about errors in deferred procedures.
 *******************************************************************************
 }
procedure TProgramDevice.DeferredCommandProblem(const Cmd:LongString);
begin
 if Ok then
 if DaqPascal.Ok then
 if DaqPascal.DebugMode then inherited DeferredCommandProblem(Cmd);
end;

 {
 *******************************************************************************
 Open or create new text editor to edit and compile DAQ PASCAL program source.
 Note Master and LinkDevice fields initialization.
 *******************************************************************************
 }
function TProgramDevice.OpenDaqPascalEditor:Boolean;
var
 Form : TFormTextEditor;
begin
 Result:=false;
 if Ok then
 try
  if DaqPascal.Ok then
  if DaqEditor.Ok then begin
   DaqEditor.Show;
   DaqEditor.WindowState:=wsNormal;
   DaqEditor.BringToFront;
   Result:=DaqEditor.Ok;
  end else begin
   Form:=FindTextEditor(DaqPascal.SrcFile);
   if Form is TFormTextEditor then Form.Close;
   DaqEditor:=NewFormDaqPascalEditor(Self,DaqPascal.SrcFile);
   if DaqEditor.Ok then begin
    DaqEditor.Master:=@DaqEditor;
    DaqEditor.LinkedDevice:=Self;
    if IsSameText(Copy(DaqPascal.SrcFile,1,Length(Daq.SitePath)),Daq.SitePath) then begin
     DaqEditor.PerformReadOnly:=true;
     DaqEditor.ActionEditReadOnly.Enabled:=EnableEditDaqSite;
    end;
   end;
   Result:=DaqEditor.Ok;
  end;
 except
  on E:Exception do BugReport(E,Self,'OpenDaqPascalEditor');
 end;
end;

 {
 *******************************************************************************
 Check, if source code in linked editor may be compiled.
 *******************************************************************************
 }
function TProgramDevice.CanCompileEditor:Boolean;
begin
 Result:=false;
 if Ok then
 try
  if Daq.Ok then
  if DaqPascal is TDaqPascal then
  if DaqEditor is TFormDaqPascalEditor then begin
   if IsSameText(DaqPascal.SrcFile,DaqEditor.PathName) then
   if SameText(UnifyAlias(ExtractFileExt(DaqPascal.SrcFile)),'.pas') then
   if FileExists(DaqPascal.SrcFile) then Result:=true;
   if not Result then Daq.ConsoleEcho(RusEng('Не могу компилировать ','Cannot compile ')+DaqPascal.SrcFile);
  end;
 except
  on E:Exception do BugReport(E,Self,'CanCompileEditor');
 end;
end;

 {
 *******************************************************************************
 Compile source file in linked editor.
 For safety, polling disable and stop device acquisition.
 Report compilation status, locate cursor to error position.
 *******************************************************************************
 }
function TProgramDevice.CompileEditor:Boolean;
const
 StepX = 20;
 StepY = 20;
var
 i            : Integer;
 Form         : TFormTextEditor;
 StartedState : Boolean;
 PollingState : Boolean;
 sx           : LongString;
begin
 Result:=false;
 if Ok then
 try
  if Daq.Ok then
  if DaqPascal.Ok then
  if DaqEditor.Ok then
  if CanCompileEditor then begin
   { disable polling }
   PollingState:=Polling.Enabled;
   if PollingState then Polling.Enable(false);
   { stop device }
   StartedState:=InquiryTimer.IsStart;
   if StartedState then Stop;
   { compile source }
   DaqPascal.Compile(DaqPascal.SrcFile,DaqPascal.LstFile);
   { success? }
   if DaqPascal.CompileOk then begin
    Result:=true;
    { report success }
    Daq.OpenConsole(true);
    Daq.ConsoleEcho(StdDateTimePrompt+RusEng('Успешная компиляция','Compile success'));
    Daq.ConsoleEcho(RusEng(' Файл: ',' File: ')+DaqPascal.SrcFile);
   end else begin
    { cursor to error pos }
    DaqEditor.GoToRowCol(DaqPascal.ErrorLine,DaqPascal.ErrorColUtf8);
    { open include files }
    if DaqPascal.PasTop>Low(DaqPascal.PasStack) then
    for i:=Low(DaqPascal.PasStack)+1 to DaqPascal.PasTop do begin
     Form:=FindTextEditor(DaqPascal.PasStack[i].Src,True,True);
     if Form is TFormTextEditor then begin
      if IsSameText(Copy(Form.PathName,1,Length(Daq.SitePath)),Daq.SitePath)
      or not IsSameText(ExtractFilePath(Form.PathName),ExtractFilePath(DaqPascal.SrcFile))
      or not FileExists(Form.PathName) then begin
       Form.PerformModified:=false;
       Form.PerformReadOnly:=true;
       Form.ActionEditReadOnly.Enabled:=false;
      end;
      Form.Top:=DaqEditor.Top+StepY*i;
      Form.Left:=DaqEditor.Left+StepX*i;
      Form.GoToRowCol(DaqPascal.PasStack[i].iln,GetChColUtf8(DaqPascal.PasStack[i].line,DaqPascal.PasStack[i].cc));
     end;
    end;
    { report error }
    Daq.OpenConsole(false);
    Daq.ConsoleEcho(StdDateTimePrompt+RusEng('Ошибка компиляции ','Compile error ')
                   +d2s(DaqPascal.ExceptResult));
    Daq.ConsoleEcho(RusEng(' Файл:    ',' File:   ')+DaqPascal.SrcFile);
    Daq.ConsoleEcho(RusEng(' Причина: ',' Reason: ')+DaqPascal.ErrorMessage);
    Daq.ConsoleEcho(RusEng(' Строка:  ',' Line:   ')+d2s(DaqPascal.ErrorLine));
    Daq.ConsoleEcho(RusEng(' Колонка: ',' Column: ')+d2s(DaqPascal.ErrorColUtf8));
    sx:=FormatErrorDetails; if IsNonEmptyStr(sx) then Daq.ConsoleEcho(TrimRight(sx));
   end;
   UpdateFormDaqDeviceControl(1+2+4);
   if StartedState then Start;
   Polling.Enable(PollingState);
  end;
 except
  on E:Exception do BugReport(E,Self,'CompileEditor');
 end;
end;

 {
 *******************************************************************************
 Show tooltip notification on compile result (how).
 *******************************************************************************
 }
procedure TProgramDevice.NotifyCompile(how:Boolean);
var msg:LongString;
begin
 if Ok then
 try
  msg:=SessionManager.TitlePidAtHost+': ';
  if how
  then msg:=msg+RusEng('Успешная компиляция: ','Compilation succeed: ')
  else msg:=msg+RusEng('Компиляция неудачна: ','Compilation failed: ');
  msg:=msg+Name+'.';
  msg:='@silent @tooltip text '+QArg(msg);
  if how
  then msg:=msg+' preset stdSuccess delay 7000'
  else msg:=msg+' preset stdFails delay 30000';
  SendToMainConsole(msg+EOL);
 except
  on E:Exception do BugReport(E,Self,'NotifyCompile');
 end;
end;

 {
 *******************************************************************************
 ReCompile source file.
 For safety, polling disable and stop device acquisition.
 Report compilation status.
 *******************************************************************************
 }
function TProgramDevice.ReCompile:Boolean;
var StartedState,PollingState:Boolean; sx:LongString;
begin
 Result:=false;
 if Ok then
 try
  if Daq.Ok then
  if DaqPascal.Ok then begin
   { disable polling }
   PollingState:=Polling.Enabled;
   if PollingState then Polling.Enable(false);
   { stop device }
   StartedState:=InquiryTimer.IsStart;
   if StartedState then Stop;
   { compile source }
   DaqPascal.Compile(DaqPascal.SrcFile,DaqPascal.LstFile);
   { success? }
   if DaqPascal.CompileOk then begin
    Result:=true;
    { report success }
    Daq.OpenConsole(true);
    Daq.ConsoleEcho(StdDateTimePrompt+RusEng('Успешная компиляция','Compile success'));
    Daq.ConsoleEcho(RusEng(' Файл: ',' File: ')+DaqPascal.SrcFile);
   end else begin
    { report error }
    Daq.OpenConsole(true);
    Daq.ConsoleEcho(StdDateTimePrompt+RusEng('Ошибка компиляции ','Compile error ')
                   +d2s(DaqPascal.ExceptResult));
    Daq.ConsoleEcho(RusEng(' Файл:    ',' File:   ')+DaqPascal.SrcFile);
    Daq.ConsoleEcho(RusEng(' Причина: ',' Reason: ')+DaqPascal.ErrorMessage);
    Daq.ConsoleEcho(RusEng(' Строка:  ',' Line:   ')+d2s(DaqPascal.ErrorLine));
    Daq.ConsoleEcho(RusEng(' Колонка: ',' Column: ')+d2s(DaqPascal.ErrorColUtf8));
    sx:=FormatErrorDetails; if IsNonEmptyStr(sx) then Daq.ConsoleEcho(TrimRight(sx));
    if UsesExternEditor then OpenExternEditor(DaqPascal.ErrorLine,
                                              DaqPascal.ErrorColUtf8,
                                              DaqPascal.ErrorCode,
                                              DaqPascal.ErrorMessage);
   end;
   UpdateFormDaqDeviceControl(1+2+4);
   if StartedState then Start;
   Polling.Enable(PollingState);
  end;
 except
  on E:Exception do BugReport(E,Self,'ReCompile');
 end;
end;

function TProgramDevice.FormatErrorDetails:LongString;
var i,wc:Integer; sr,sc,sf:LongString;
begin
 Result:='';
 if Ok then
 try
  if Daq.Ok then
  if DaqPascal.Ok then
  if DaqEditor.Ok then
  if CanCompileEditor then begin
   wc:=WordCount(DaqPascal.ErrorDetails,EolnDelims);
   if (wc>1) then begin
    Result:=Result+RusEng('Список Исходных Файлов:','List of Source Files:')+EOL;
    Result:=Result+RusEng(' Строка   Столбец  Файл',' Line     Column   File')+EOL;
    for i:=1 to wc do begin
     sr:=Trim(ExtractPhrase(1,ExtractWord(i,DaqPascal.ErrorDetails,EolnDelims),ScanSpaces));
     sc:=Trim(ExtractPhrase(2,ExtractWord(i,DaqPascal.ErrorDetails,EolnDelims),ScanSpaces));
     sf:=Trim(ExtractPhrase(3,ExtractWord(i,DaqPascal.ErrorDetails,EolnDelims),ScanSpaces));
     Result:=Result+Format(' %-7s  %-7s  %s',[sr,sc,sf])+EOL;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'FormatErrorDetails');
 end;
end;

 {
 *******************************************************************************
 Console input/output filter calls when get text from console input/output FIFO.
 Input fifo echo text to output console and also send to DAQ PASCAL program.
 Output fifo does nothing over default.
 *******************************************************************************
 }
procedure ConsoleInputFilter(aConsole:TFormConsoleWindow; var aText:LongString);
begin
 if aConsole.Ok and (Length(aText)>0) then begin
  if Guard.CheckAction(ga_Root,'TProgramDevice.ConsoleInput')<0 then Exit;
  aConsole.PutText(aText);
  if TObject(aConsole.Custom) is TProgramDevice then
  if (TObject(aConsole.Custom) as TProgramDevice).DaqPascal.Ok then begin
   (TObject(aConsole.Custom) as TProgramDevice).DaqPascal.InpFifo.PutText(aText);
   (TObject(aConsole.Custom) as TProgramDevice).Awake;
  end;
 end;
end;

procedure ConsoleOutputFilter(aConsole:TFormConsoleWindow; var aText:LongString);
begin
end;

procedure LoadFavoriteCommands(Console:TFormConsoleWindow; const IniFile,Section:LongString);
var txt:TText; i:Integer; s:LongString;
begin
 if Console.Ok then
 try
  txt:=ExtractListSection(IniFile,Section,efConfigNC);
  try
   for i:=0 to txt.Count-1 do begin
    s:=Trim(txt[i]); if (Length(s)<3) or (StrFetch(s,1)<>'@') then  continue;
    if (Console.ComboBoxInput.Items.IndexOf(s)<0) then Console.ComboBoxInput.Items.Add(s);
   end;
  finally
   Kill(txt);
  end;
 except
  on E:Exception do BugReport(E,nil,'LoadFavoriteCommands');
 end;
end;

 {
 *******************************************
 Wanted console name, depending on language.
 *******************************************
 }
function TProgramDevice.WantedConsoleName:LongString;
begin
 Result:='';
 if Ok then begin
  Result:=RusEng('КОНСОЛЬ ','CONSOLE ')+Name;
 end;
end;

 {
 ****************************************
 Compare Console.Caption with given name.
 ****************************************
 }
function TProgramDevice.IsSameConsoleName(aName:LongString):Boolean;
var S1,S2:LongString;
begin
 Result:=False;
 if IsNonEmptyStr(aName) then
 if Ok and Console.Ok then begin
  S1:=utf8_uppercase(aName);
  S2:=utf8_uppercase(Console.WinCaption);
  S1:=StringReplace(S1,'КОНСОЛЬ ','CONSOLE ',[rfReplaceAll]);
  S2:=StringReplace(S2,'КОНСОЛЬ ','CONSOLE ',[rfReplaceAll]);
  Result:=IsSameText(S1,S2);
 end;
end;

 {
 *******************************************************************************
 Open or create new console window for device output.
 Note Master and Custom fields initialization.
 *******************************************************************************
 }
function TProgramDevice.OpenConsole:Boolean;
 procedure LoadFavorites;
 var sect:LongString;
 begin
  sect:='';
  if Console.Ok then begin
   if ReadIniFileString(Daq.ConfigFile,DevSection,'FavoriteCommands%s',sect) then begin
    if IsSectionName(sect) then LoadFavoriteCommands(Console,Daq.ConfigFile,sect);
   end else
   LoadFavoriteCommands(Console,SysIniFile,'[DaqSys.DeviceConsole.FavoriteCommands]');
  end;
 end;
begin
 Result:=false;
 if Ok then
 if DaqPascal.Ok then
 try
  if Console.Ok then begin
   Console.Show;
   Console.WindowState:=wsNormal;
   Console.BringToFront;
   Result:=Console.Ok;
  end else begin
   Console:=NewConsoleWindow(WantedConsoleName,
                             NewFifo(InpFifoSize), true, ConsoleInputFilter,
                             NewFifo(OutFifoSize), true, ConsoleOutputFilter);
   Console.InpFifo.GrowFactor:=2; Console.InpFifo.GrowLimit:=DaqPascal.InpFifo.Size*2;
   Console.OutFifo.GrowFactor:=2; Console.OutFifo.GrowLimit:=DaqPascal.OutFifo.Size*2;
   Console.Master:=@Console;
   Console.Custom:=Self;
   Console.StartMonitoring;
   Result:=Console.Ok;
   LoadFavorites;
  end;
 except
  on E:Exception do BugReport(E,Self,'OpenConsole');
 end;
end;

 {
 *******************************************************************************
 Update console window. If has data in DaqPascal.OutFifo, pop and send this data
 to destination console. Destination console is device console or system console,
 if DaqPascal.StdOut has name "CON:".
 *******************************************************************************
 }
procedure  TProgramDevice.UpdateConsole;
var p:TText; i:Integer;
begin
 if Ok then
 if Daq.Ok then
 if DaqPascal.Ok then
 if DaqPascal.CompileOk then
 if DaqPascal.OutFifo.Count>0 then begin
  if SameText(StrPas(TTextRec(DaqPascal.StdOut).Name),'CON:') then begin
   { output to system console, if StdOut has name "CON:" }
   p:=NewText;
   try
    IsOutFifoCleanTime(true);
    p.Text:=DaqPascal.OutFifo.GetText;
    for i:=0 to p.Count-1 do Daq.ConsoleEcho(p[i]);
   finally
    Kill(p);
   end;
  end else begin
   { output to local console otherwise }
   if Console.Ok then begin
    IsOutFifoCleanTime(true);
    Console.PutText(DaqPascal.OutFifo.GetText);
   end else begin
    if IsOutFifoCleanTime
    then DaqPascal.OutFifo.Clear;
   end;
  end;
 end;
end;

function TProgramDevice.IsOutFifoCleanTime(aReset:Boolean=false):Boolean;
var ms:QWord; rc:Double;
begin
 Result:=false;
 if Assigned(Self) then begin
  if aReset then begin
   myOutFifoCleanTime:=0;
   Exit;
  end;
  if Assigned(DaqPascal) then rc:=DaqPascal.RunCount else rc:=0;
  if (rc>OutFifoCleanRunCount) then begin
   myOutFifoCleanTime:=0;
   Result:=true;
   Exit;
  end;
  ms:=GetTickCount64;
  if (myOutFifoCleanTime=0) then myOutFifoCleanTime:=ms;
  if (ms>myOutFifoCleanTime+OutFifoCleanDelay) then begin
   myOutFifoCleanTime:=0;
   Result:=true;
  end;
 end;
end;

 {
 *******************************************************************************
 Open external editor.
 *******************************************************************************
 }
function TProgramDevice.OpenExternEditor(row:Integer=0; col:Integer=0;
                      err:Integer=0; msg:LongString=''; Mode:Integer=1):Boolean;
var cmd,arg:LongString; Form:TForm; Len:Integer;
begin
 Result:=false;
 if Ok and DaqPascal.Ok then
 try
  cmd:=SysGlossary.ReadIniParamDef(SysIniFile,SectDaqSys(1),'OpenDaqPascalEditor','');
  if IsNonEmptyStr(cmd) then begin
   cmd:=StringReplace(cmd,'$SourceFileName',DaqPascal.SrcFile,[rfReplaceAll,rfIgnoreCase]);
   cmd:=StringReplace(cmd,'$DeviceName',Name,[rfReplaceAll,rfIgnoreCase]);
   arg:='--sid '+SessionManager.SessionSign('_');
   if (row>0) then arg:=arg+' --row '+IntToStr(row);
   if (col>0) then arg:=arg+' --col '+IntToStr(col);
   if (err>0) then arg:=arg+' --err '+IntToStr(err);
   if (msg<>'') then arg:=arg+' --msg '+AnsiQuotedStr(msg,QuoteMark);
   Len:=SendToMainConsole(Trim(Trim(cmd)+' '+Trim(arg))+EOL);
   Result:=(Len>0);
   if HasFlags(Mode,1) then begin
    if not SecondActions.HasAction(Timer_MonitorTextEditor) then
    if DaqEditor.Ok then DaqEditor.Close else begin
     Form:=FindTextEditor(DaqPascal.SrcFile);
     if (Form is TFormTextEditor) then Form.Close;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'OpenExternalEditor');
 end;
end;

 {
 *******************************************************************************
 Handle device message. Put message to standard input FIFO.
 Server:  while not eof do begin readln(s); writeln('Received message:',s); end;
 Client:  r:=devmsg('Server Hello world'+EOL);
 *******************************************************************************
 }
function TProgramDevice.HandleMessage(const aMsg:LongString; aFlags:Cardinal=hf_Default):Double;
begin
 Result:=0;
 if Ok then
 if Daq.Ok then
 if DaqPascal.Ok then
 if DaqPascal.CompileOk then begin
  Result:=DaqPascal.InpFifo.Put(@aMsg[1],Length(aMsg));
  if ((aFlags and hf_SkipAwake)=0) then Awake;
 end;
end;

 {
 *******************************************************************************
 Execute in main VCL thread some pending commands...
 *******************************************************************************
 }
procedure TProgramDevice.ExecuteDeferredCommand(var Cmd,Arg:LongString);
 function EditStart(const Arg:LongString):Boolean;
 begin
  EditStart:=false;
  if Ok then
  if Daq.Ok then
  if DaqPascal.Ok then
  if DaqPascal.CompileOk then EditStart:=DaqPascal._EditStart(Arg);
 end;
 function StartPlot3d(const Arg:LongString):Boolean;
 begin
  StartPlot3d:=false;
  if Ok then
  if Daq.Ok then
  if DaqPascal.Ok then
  if DaqPascal.CompileOk then StartPlot3d:=DaqPascal._startplot3d(Arg);
 end;
begin
 inherited ExecuteDeferredCommand(Cmd,Arg);
 if SameText(Cmd,'@EDIT') then begin
  if not EditStart(Arg) then DeferredCommandProblem(Cmd+' '+Arg);
  Cmd:='';
 end else
 if SameText(Cmd,'@PLOT3D') then begin
  if not StartPlot3d(Arg) then DeferredCommandProblem(Cmd+' '+Arg);
  Cmd:='';
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_daqpascaldevice;
begin
 InitPvCounters;
 TProgramDevice.EnableEditDaqSite:=false;
 TProgramDevice.OutFifoCleanDelay:=10000;
 TProgramDevice.OutFifoCleanRunCount:=1e5;
end;

procedure Free_crw_daqpascaldevice;
begin
 ResourceLeakageLog(Format('%-60s = %d',['TProgramDevice.PollingViolationCounter',TProgramDevice.PollingViolationCounter]));
 FreePvCounters;
end;

initialization

 Init_crw_daqpascaldevice;

finalization

 Free_crw_daqpascaldevice;

end.

//////////////
// END OF FILE
//////////////

