////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// CRW DAQ system.                                                            //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231209 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit _crw_crwdaq; // CRW DAQ system

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Printers,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 Form_TextEditor, Form_CurveWindow, Form_SurfWindow,
 Form_CircuitWindow, Form_ConsoleWindow, Form_TabWindow,
 Form_SpectrWindow, Form_Calculator, Form_ListBoxSelection,
 Form_DaqDeviceControl, Form_UartTerminal, Form_CalibDialog,
 Form_DaqControlDialog, Form_CurveRangeSelector,
 Unit_SystemConsole,
 _crw_alloc, _crw_ef, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_pio,
 _crw_pipe, _crw_task, _crw_proc, _crw_syscal,
 _crw_curves, _crw_riff, _crw_calib, _crw_couple,
 _crw_daqtags, _crw_daqevnt, _crw_daqsys, _crw_daqdev,
 _crw_adamdev, _crw_uart, _crw_pcldev, _crw_polling,
 _crw_appforms, _crw_apptools, _crw_apputils;

 {
 *******************************************************************************
 TCrwDaq - конкретная реализация DAQ-подсистемы.
 *******************************************************************************
 }
type
 TCrwDaq = class(TDaqSystem)
 public
  procedure   Idle; override;
  procedure   Poll; override;
  procedure   DispatchEvents; override;
  function    Start:Boolean; override;
  function    Stop:Boolean; override;
  function    Clear:Boolean; override;
  procedure   ReadConfig; override;
  procedure   AnimateDevices; override;
  procedure   AnimateWindows(const aWinList:LongString); override;
  procedure   StartSession; override;
  procedure   StopSession; override;
  function    CommonProperty(P:TText):TText; override;
  procedure   DaqInfo; override;
  procedure   RunPreProcessor; override;
  procedure   RunPostProcessor; override;
 public
  procedure   MinimizeNoRestoreWindows;
 end;

 {
 *******************************************************************************
 Системные функции для "оживления" DAQ-подсистемы.
 *******************************************************************************
 }
procedure Init_Daq_SubSystem;
procedure Done_Daq_SubSystem;
function  ConfirmExit_Daq_SubSystem:Boolean;
procedure Timer_Daq_SubSystem;

implementation

uses
 _crw_softdev,
 Form_CrwDaq,
 Form_CrwDaqLogo;

 {
 *******************************************************************************
 TCrwDaq implementation
 *******************************************************************************
 }
 {
 Процедура фонового опроса по таймеру
 }
procedure TCrwDaq.Idle;
begin
 inherited Idle;
 IdleFormDaqDeviceControl;
 if AcqTimer.IsStart then begin
  SoftwareDeviceList.Idle;
  AdamDeviceList.Idle;
  PclDeviceList.Idle;
 end;
end;

 {
 Основная процедура опроса. !!!!! Выполняется в другом потоке !!!!!
 }
procedure TCrwDaq.Poll;
begin
 inherited Poll;
 if AcqTimer.IsStart then begin
  SoftwareDeviceList.Poll;
  AdamDeviceList.Poll;
  PclDeviceList.Poll;
 end;
end;

 {
 Диспетчеризация событий. !!!!! Выполняется в другом потоке !!!!!
 }
procedure TCrwDaq.DispatchEvents;
begin
 inherited DispatchEvents;
 FullDaqDeviceList.DispatchEvents;
end;

 {
 Пуск измерений
 }
function TCrwDaq.Start:Boolean;
var i:Integer; t:Double;
begin
 Result:=false;
 if inherited Start then begin
  if SolidCrwDaqLogo then
  ShowLogo(RusEng('Подождите, идет старт CRW-DAQ...','Wait while starting CRW-DAQ...'));
  try
   Result:=true;
   t:=AcqTimer.StartTime;
   for i:=0 to Curves.Count-1 do
   Curves[i].Addln(Format('_ACQUISITION_START_TIME_ = %14.0f (%s)', [t, StdDateTimeStr(t)]));
   SetEnv('CRW_DAQ_CONFIG_START_TIME',Format('%g',[t]));
   SoftwareDeviceList.Start;
   AdamDeviceList.Start;
   PclDeviceList.Start;
   OpenFormDaqControlDialog;
   UpdateFormDaqDeviceControl(4);
   Polling.Enable(true,DefaultDaqTimeOut);
   Watchdog.Enable(true,DefaultDaqTimeOut);
  finally
   if (DaqShowLogoDeadline<=0)
   then ShowLogo('');
  end;
 end;
end;

 {
 Останов измерений
 }
function TCrwDaq.Stop:Boolean;
var t:Double; i:Integer;
begin
 Result:=false;
 if inherited Stop then begin
  if SolidCrwDaqLogo then
  ShowLogo(RusEng('Подождите, идет остановка CRW-DAQ...','Wait while stopping CRW-DAQ...'));
  try
   Result:=true;
   SetEnv('CRW_DAQ_CONFIG_START_TIME','0');
   SoftwareDeviceList.Stop;
   AdamDeviceList.Stop;
   PclDeviceList.Stop;
   t:=AcqTimer.GlobalTime;
   for i:=0 to Curves.Count-1 do
   Curves[i].Addln(Format('_ACQUISITION_STOP_TIME_ = %14.0f (%s)', [t, StdDateTimeStr(t)]));
   UpdateFormDaqDeviceControl(4);
  finally
   ShowLogo('');
  end;
 end;
end;

 {
 Очистка всех устройств
 }
function TCrwDaq.Clear:Boolean;
begin
 Result:=false;
 if inherited Clear then begin
  Result:=true;
  SoftwareDeviceList.Clear;
  AdamDeviceList.Clear;
  PclDeviceList.Clear;
 end;
end;

function RunCommandProcessor(const CmdLine,HomeDir:LongString;
 const ProcessPriority,ThreadPriority,StdInPriority,StdOutPriority:LongString;
 Display,PipeSize,WaitTimeOut,KillTimeOut:Integer; ConvertOem:Boolean):Integer;
var
 ms   : Double;
 tid  : Integer;
 Head : LongString;
 Line : LongString;
 Buff : LongString;
 procedure Print(Line:LongString);
 begin
  if ConvertOem then
  if Length(Line)>0 then
  Line:=StrOemToAnsi(Line);
  Daq.ConsoleEcho(Head+Line);
 end;
 function Task_Readln(tid:Integer; var Line,Buff:LongString):Boolean;
 const MaxLeng=KiloByte*16;
 var p:Integer;
 begin
  Line:='';
  Task_Readln:=False;
  if (Task_Pid(tid)<>0) then begin
   if (Length(Buff)<MaxLeng) and (Task_RxCount(tid)>0)
   then Buff:=Buff+Task_Recv(tid,MaxLeng-Length(Buff));
   p:=PosEol(Buff,1,0);
   if (p>0) then begin
    Task_Readln:=True;
    if (p>1) then Line:=Copy(Buff,1,p-1);
    Buff:=Copy(Buff,PosEol(Buff,p,1),MaxInt);
   end else begin
    if (Length(Buff)>=MaxLeng) then begin
     Print('Received line is too long!');
     Buff:='';
    end;
   end;
  end;
 end;
 procedure Task_Process(tid:Integer; var Line,Buff:LongString; Tail:Boolean);
 begin
  while Task_Readln(tid,Line,Buff) do Print(Line);
  if Tail then if Length(Buff)>0 then Print(Buff);
  Daq.ConsoleFlush; //SafeApplicationProcessMessages;
  if Tail then Buff:='';
 end;
begin
 Result:=-1;
 try
  ms:=mSecNow;
  tid:=0; Line:=''; Buff:='';
  if FileExists(ExtractWord(1,CmdLine,ScanSpaces)) then
  try
   tid:=Task_Init(Trim(CmdLine));
   //Task_Ctrl(tid,'AppName='+GetComSpec);
   //Task_Ctrl(tid,'CmdLine=/c '+Trim(CmdLine));
   Task_Ctrl(tid,'HomeDir='+HomeDir);
   Task_Ctrl(tid,'Display='+IntToStr(Display));
   Task_Ctrl(tid,'StdInpPipeSize='+IntToStr(PipeSize));
   Task_Ctrl(tid,'StdOutPipeSize='+IntToStr(PipeSize));
   Task_Ctrl(tid,'ProcessPriority='+ProcessPriority);
   Task_Ctrl(tid,'ThreadPriority='+ThreadPriority);
   Task_Ctrl(tid,'StdInPriority='+StdInPriority);
   Task_Ctrl(tid,'StdOutPriority='+StdOutPriority);
   Head:=ExtractFileNameExt(ExtractWord(1,CmdLine,ScanSpaces));
   if IsWindows then Head:=LoCaseStr(Head);
   Head:=Head+' > ';
   if Task_Run(tid) then begin
    while (mSecNow<=ms+WaitTimeOut) do begin
     if Task_Wait(tid,0)
     then Task_Process(tid,Line,Buff,False)
     else Break;
     Sleep(TPolling.DefPollPeriod);
    end;
    Task_Process(tid,Line,Buff,False);
    if Task_Wait(tid,0) then Task_Kill(tid,0,1,KillTimeOut);
    Task_Process(tid,Line,Buff,True);
    Print('Exit with code '+IntToStr(Task_Result(tid)));
    Print('Execution time '+IntToStr(Round(mSecNow-ms))+' ms');
    Result:=Task_Result(tid);
   end;
  finally
   Task_Free(tid);
   Line:=''; Buff:='';
  end;
 except
  on E:Exception do BugReport(E,nil,'RunCommandProcessor');
 end;
end;

function IsValidAppExt(AppName:LongString):Boolean;
var Ext,PathExt:LongString;
begin
 PathExt:='';
 Ext:=ExtractFileExt(AppName);
 if IsUnix then PathExt:='.sh;.bash';
 if IsWindows then PathExt:=GetEnv('PATHEXT');
 Result:=(WordIndex(Ext,PathExt,ScanSpaces)>0);
end;

 {
 Действия перед загрузкой конфигурации.
 }
procedure TCrwDaq.RunPreProcessor;
var AppName:LongString; RunCode:Integer;
begin
 if Ok then
 try
  with PreProcessor do
  if IsNonEmptyStr(CmdLine) then begin
   AppName:=UnifyFileAlias(FileRef(ExtractWord(1,CmdLine,ScanSpaces)),ua_FileDefLow);
   CmdLine:=StringReplace(CmdLine,ExtractWord(1,CmdLine,ScanSpaces),AppName,[]);
   if FileExists(AppName) then begin
    if IsValidAppExt(AppName) then begin
     ConsoleEcho('');
     ConsoleEcho(StdDateTimePrompt+'PreProcessor: Run '+CmdLine);
     RunCode:=RunCommandProcessor(CmdLine,ExtractFilePath(AppName),
      ExtractWord(1,Priority,ScanSpaces),ExtractWord(2,Priority,ScanSpaces),
      ExtractWord(3,Priority,ScanSpaces),ExtractWord(4,Priority,ScanSpaces),
      Display,PipeSize*1024,TimeOut*1000,1000,True);
     ConsoleEcho(StdDateTimePrompt+'PreProcessor: Exit with code '+IntToStr(RunCode));
     if RunCode<>0 then AddWarning(Format('[DAQ] PreProcessor failed: exit code was %d.',[RunCode]));
    end else begin
     ConsoleEcho('[DAQ] PreProcessor bad extension: '+CmdLine);
     AddWarning('[DAQ] PreProcessor bad extension: '+CmdLine);
    end;
   end else begin
    ConsoleEcho('[DAQ] PreProcessor not found: '+CmdLine);
    AddWarning('[DAQ] PreProcessor not found: '+CmdLine);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'RunPreProcessor');
 end;
end;

 {
 Действия после выгрузки конфигурации.
 }
procedure TCrwDaq.RunPostProcessor;
var AppName:LongString; RunCode:Integer;
begin
 if Ok then
 try
  with PostProcessor do
  if IsNonEmptyStr(CmdLine) then begin
   AppName:=UnifyFileAlias(FileRef(ExtractWord(1,CmdLine,ScanSpaces)),ua_FileDefLow);
   CmdLine:=StringReplace(CmdLine,ExtractWord(1,CmdLine,ScanSpaces),AppName,[]);
   if FileExists(AppName) then begin
    if IsValidAppExt(AppName) then begin
     ConsoleEcho(StdDateTimePrompt+'PostProcessor: Run '+CmdLine);
     RunCode:=RunCommandProcessor(CmdLine,ExtractFilePath(AppName),
      ExtractWord(1,Priority,ScanSpaces),ExtractWord(2,Priority,ScanSpaces),
      ExtractWord(3,Priority,ScanSpaces),ExtractWord(4,Priority,ScanSpaces),
      Display,PipeSize*1024,TimeOut*1000,1000,True);
     ConsoleEcho(StdDateTimePrompt+'PostProcessor: Exit with code '+IntToStr(RunCode));
     if RunCode<>0 then AddWarning(Format('[DAQ] PostProcessor failed: exit code was %d.',[RunCode]));
    end else ConsoleEcho('[DAQ] PostProcessor bad extension: '+CmdLine);
   end else ConsoleEcho('[DAQ] PostProcessor not found: '+CmdLine);
  end;
 except
  on E:Exception do BugReport(E,Self,'RunPostProcessor');
 end;
end;

 {
 Чтение конфигураций в начале сессии DAQ
 }
procedure TCrwDaq.ReadConfig;
begin
 inherited ReadConfig;
 SoftwareDeviceList.Count:=0;
 SoftwareDeviceList.Config(ConfigFile,RusEng('Функциональные устройства','Software devices'),
                           SoftwareDeviceConstructor);
 AdamDeviceList.Count:=0;
 AdamDeviceList.Config(ConfigFile,RusEng('Устройства ADAM','ADAM devices'),
                       AdamDeviceConstructor);
 PclDeviceList.Count:=0;
 PclDeviceList.Config(ConfigFile,RusEng('Устройства PCL','PCL devices'),PclDeviceConstructor);
end;

 {
 Анимация прочитанных объектов
 }
procedure TCrwDaq.AnimateDevices;
begin
 inherited AnimateDevices;
 FullDaqDeviceList.Animate;
 FullDaqDeviceList.CheckCurveLinks;
end;

procedure TCrwDaq.AnimateWindows(const aWinList:LongString);
begin
 inherited AnimateWindows(aWinList);
 AnimateCircuits(DefaultDaqCircuitStrategy,DefaultDaqCircuitMonitor,aWinList);
end;

 {
 Минимизировать скрытые окна.
 }
procedure TCrwDaq.MinimizeNoRestoreWindows;
begin
 if SdiMan.IsSdiMode then SdiMan.ChildList.MinimizeByFlags(sf_SdiNoRestore);
end;

 {
 Дополнительные действия по инициализации сеанса DAQ
 }
procedure TCrwDaq.StartSession;
var i:Integer;
begin
 inherited StartSession;
 for i:=0 to Curves.Count-1 do begin
  Curves[i].Addln(Format('DAQ_START_TIME_ = %14.0f (%s)', [Timer.StartTime, StdDateTimeStr(Timer.StartTime)]));
  Curves[i].Addln(Format('CRW_DAQ_CONFIG_BASE_TIME = %g (%s)', [Timer.StartTime, StdDateTimeStr(Timer.StartTime)]));
  Curves[i].Addln(Format('CRW_DAQ_CONFIG_TIME_UNITS = %g', [Timer.LocalTimeUnits]));
 end;
 SetEnv('CRW_DAQ_CONFIG_BASE_TIME',Format('%g',[Timer.StartTime]));
 SetEnv('CRW_DAQ_CONFIG_TIME_UNITS',Format('%g',[Timer.LocalTimeUnits]));
 SoftwareDeviceList.StartSession;
 AdamDeviceList.StartSession;
 PclDeviceList.StartSession;
 OpenFormDaqDeviceControl;
 HideFormDaqDeviceControl;
 OpenFormDaqControlDialog;
 SdiMan.UpdateAllChilds;
 CurveRangeSelectorDefaults.DAQ.TimeBase:=Timer.StartTime;
 CurveRangeSelectorDefaults.DAQ.TimeUnits:=Timer.LocalTimeUnits;
 MinimizeNoRestoreWindows;
end;

 {
 дополнительные действия по завершению сеанса DAQ
 }
procedure TCrwDaq.StopSession;
var i:Integer; t:Double;
begin
 inherited StopSession;
 t:=Timer.GlobalTime;
 for i:=0 to Curves.Count-1 do
 Curves[i].Addln(Format('DAQ_STOP_TIME_ = %14.0f (%s)', [t, StdDateTimeStr(t)]));
 KillFormDaqDeviceControl;
 SoftwareDeviceList.StopSession;
 SoftwareDeviceList.Count:=0;
 AdamDeviceList.StopSession;
 AdamDeviceList.Count:=0;
 PclDeviceList.StopSession;
 PclDeviceList.Count:=0;
 OpenFormDaqControlDialog;
 CurveRangeSelectorDefaults.DAQ.TimeBase:=0;
 CurveRangeSelectorDefaults.DAQ.TimeUnits:=0;
end;

 {
 Общие свойства системы
 }
function TCrwDaq.CommonProperty(P:TText):TText;
begin
 CommonProperty:=inherited CommonProperty(P);
 AdamDeviceList.CommonProperty(P);
 GetFullPipeListProperties(P);
end;

 {
 Вызвать окно справки
 }
procedure TCrwDaq.DaqInfo;
var Menu:TText; Params:LongString;
 {}
 procedure CalibrInfo;
 var Calib:TPolynomCalibration; Device:TDaqDevice;
 var i,j:Integer; Info:TText; Curve:TCurve;
 begin
  Info:=NewText;
  try
   for i:=0 to FullDaqDeviceList.Count-1 do begin
    Device:=FullDaqDeviceList[i];
    for j:=0 to Device.NumCalibrations-1 do begin
     Calib:=Device.Calibration[j];
     Curve:=Device.AnalogOutputCurve[j];
     if Calib.Ok
     then Info.Addln(FormatCalibration(Format('%-15s %-15s #%-2d',
                     [Device.Name, Curve.Name, j]),Calib));
    end;
   end;
   ListBoxMenu(RusEng('Информация о калибровках','Calibration information'),
               Format('%-15s %-15s #%-2s = %s', ['Device','Curve-AO','N','Calibration']),
               Info.Text,0,Params);
  finally
   Kill(Info);
  end;
 end;
 {}
 procedure LinkInfo;
 var i,j,k:Integer; s:LongString; Inv:Boolean; Info:TText;
 var Curve:TCurve; Device:TDaqDevice;
 begin
  Info:=NewText;
  try
   for i:=0 to FullDaqDeviceList.Count-1 do begin
    Device:=FullDaqDeviceList[i];
    for j:=0 to Device.NumAnalogInputs-1 do begin
     Curve:=Device.AnalogInputCurve[j];
     if Curve.Ok then
     Info.Addln(Format('%-15s %-15s %-13s %-2d', [Device.Name, Curve.Name, 'AnalogInput', j]));
    end;
    for j:=0 to Device.NumDigitalInputs-1 do begin
     Curve:=Device.DigitalInputCurve[j];
     if Curve.Ok then begin
      k:=Device.GetDigitalInputBit(j,Inv);
      if Inv then s:='inverted' else s:='';
      Info.Addln(Format('%-15s %-15s %-13s %-2d %8s bit %-2d',[Device.Name, Curve.Name, 'DigitalInput', j, s, k]));
     end;
    end;
    for j:=0 to Device.NumAnalogOutputs-1 do begin
     Curve:=Device.AnalogOutputCurve[j];
     if Curve.Ok then
     Info.Addln(Format('%-15s %-15s %-13s %-2d', [Device.Name, Curve.Name, 'AnalogOutput', j]));
    end;
    for j:=0 to Device.NumDigitalOutputs-1 do begin
     Curve:=Device.DigitalOutputCurve[j];
     if Curve.Ok then
     Info.Addln(Format('%-15s %-15s %-13s %-2d', [Device.Name, Curve.Name, 'DigitalOutput', j]));
    end;
   end;
   ListBoxMenu(RusEng('Информация о подключениях','Device and curve links'),
               Format('%-15s %-15s %-13s %-2s',['Device','Curve','Link','#N']),
               Info.Text,0,Params);
  finally
   Kill(Info);
  end;
 end;
 {}
 procedure CurveInfo;
 var s:LongString; n,i,j:Integer; Con:Boolean; Info:TText;
 var Curve:TCurve; Device:TDaqDevice;
 const w1=20; w2=7; w3=70;
 begin
  Info:=NewText;
  try
   for n:=0 to Curves.Count-1 do begin
    Curve:=Curves[n];
    if Curve=nil then continue;
    s:=Pad(Curve.Name,w1)+' '+Pad(d2s(Curve.Count),w2)+' '+Pad(d2s(Curve.Step),w2)+' ';
    for i:=0 to FullDaqDeviceList.Count-1 do begin
     if Length(s)>w3 then break;
     Con:=false;
     Device:=FullDaqDeviceList[i];
     if Device.Ok then begin
      for j:=0 to Device.NumAnalogInputs-1 do
      if Curve=Device.AnalogInputCurve[j] then begin Con:=true; break; end;
      for j:=0 to Device.NumDigitalInputs-1  do
      if Curve=Device.DigitalInputCurve[j] then begin Con:=true; break; end;
      for j:=0 to Device.NumAnalogOutputs-1 do
      if Curve=Device.AnalogOutputCurve[j] then begin Con:=true; break; end;
      for j:=0 to Device.NumDigitalOutputs-1 do
      if Curve=Device.DigitalOutputCurve[j] then begin Con:=true; break; end;
      if Con then s:=s+Device.Name+',';
     end;
    end;
    if (StrFetch(s,Length(s))=',') then s:=Copy(s,1,Length(s)-1);
    if (Length(s)>=w3) then s:=s+'...';
    Info.Addln(s);
   end;
   ListBoxMenu(RusEng('Информация о кривых','Information about curves'),
                    Pad(RusEng('Имя Кривой','Curve Name'),w1)+' '+
                    Pad(RusEng('Длина','Length'),w2)+' '+
                    Pad(RusEng('Шаг','Step'),w2)+' '+
                    RusEng('Подключения...','Link to ...'),
                    Info.Text,0,Params);
  finally
   Kill(Info);
  end;
 end;
 {}
 procedure StartingInfo;
 begin
  ListBoxMenu(RusEng('Информация о порядке старта','Starting orger information'),
              RusEng('Порядок старта устройств','Device starting order'),
              SoftwareDeviceList.StartingList+AdamDeviceList.StartingList+PclDeviceList.StartingList,
              0,Params);
 end;
 {}
 procedure StoppingInfo;
 begin
  ListBoxMenu(RusEng('Информация о порядке завершения','Stopping orger information'),
              RusEng('Порядок остановки устройств','Device stopping order'),
              SoftwareDeviceList.StoppingList+AdamDeviceList.StoppingList+PclDeviceList.StoppingList,
              0,Params);
 end;
begin
 if Ok and CheckSessionStarted then begin
  Params:='';
  if FormDaqControlDialog.Ok and FormDaqControlDialog.Active
  then Params:=ControlPosParams(FormDaqControlDialog.ToolButtonDaqInfo,'RT');
  Menu:=NewText;
  try
   Menu.Addln(RusEng('Список всех КАЛИБРОВОК','List of all CALIBRATIONs'));
   Menu.Addln(RusEng('Список всех ПОДКЛЮЧЕНИЙ','List of all LINKs'));
   Menu.Addln(RusEng('Список всех КРИВЫХ','List of all CURVEs'));
   Menu.Addln(RusEng('Список устройств - порядок ПУСК','List of devices - STARTING'));
   Menu.Addln(RusEng('Список устройств - порядок СТОП','List of devices - STOPPING'));
   while true do
   case ListBoxMenu(RusEng('Информация о конфигурации DAQ','DAQ configuration info'),
                    RusEng('Категория информации','Category of information'),
                    Menu.Text,0,Params)
   of
    0 : CalibrInfo;
    1 : LinkInfo;
    2 : CurveInfo;
    3 : StartingInfo;
    4 : StoppingInfo;
    else break;
   end;
  finally
   Kill(Menu);
  end;
 end;
end;

 {
 *******************************************************************************
 Системные функции для "оживления" DAQ-подсистемы.
 *******************************************************************************
 }
 {
 Инициализация DAQ-подсистемы.
 Загрузить DAQ, если в секции crw_run.ini.[Daq].IniFile= … стоит ссылка на
 определенный (не маска) и реально существующий файл конфигурации.
 }
procedure Init_Daq_SubSystem;
 function CheckParamStr:Boolean;
 var i:integer; Path:LongString;
 begin
  Result:=false;
  Exit; // Skip old code!
  for i:=1 to ParamCount do 
  if SameText(UnifyAlias(ExtractFileExt(ParamStr(i))),'.cfg') then begin
   Path:=UnifyFileAlias(ParamStr(i));
   if not IsWildCard(Path) and FileExists(Path) then begin
    Daq.InitSession(Path);
    Result:=true;
    Break;
   end;
  end;
 end;
begin
 try
  if FormCrwDaq.VerboseLogon then begin
   UpdateStatusLine(RusEng('Инициализирую ','Initializing ')+'DAQ …');
   SafeApplicationProcessMessages;
  end;
  Daq:=TCrwDaq.Create;
  if Daq.Ok then begin
   Echo(RusEng('Система DAQ: создана.','DAQ system: Ok.'));
   if not CheckParamStr then
   if not IsWildCard(Daq.ConfigFile) and FileExists(Daq.ConfigFile)
   then Daq.InitSession(Daq.ConfigFile);
  end;
 except
  Kill(TObject(Daq));
 end;
 if not Daq.Ok then Echo(RusEng('Система DAQ: СБОЙ.','DAQ system: FAILS.'));
end;

 {
 Завершение DAQ-подсистемы
 }
procedure Done_Daq_SubSystem;
begin
 if Daq.Ok then if Daq.Timer.IsStart then Daq.DoneSession;
 Kill(TObject(Daq));
end;

 {
 Подтверждение завершения DAQ-подсистемы
 }
function ConfirmExit_Daq_SubSystem:Boolean;
begin
 Result:=true;
 if Daq.Ok then begin
  if Daq.Timer.IsStart then begin
   if (SystemCalculator.Eval('_Daq_Force_Exit_')=1) and Daq.AcqTimer.isStart then Daq.Stop;
   if Daq.CheckAcquisitionStarted then Result:=false else
   if Daq.ConfirmExit then Daq.DoneSession else Result:=false;
  end;
 end;
end;

 {
 Таймерные акции DAQ-подсистемы
 }
procedure Timer_Daq_SubSystem;
const TimeToUpdate:QWord=0; TimeUpdatePeriod=1000;
var TickCount:QWord;
begin
 if Daq.Ok then Daq.Idle;
 if Daq.Timer.IsStart then begin
  TickCount:=GetTickCount64;
  if Daq.Timer.Event then begin
   if (TickCount>TimeToUpdate) then begin
    TimeToUpdate:=TickCount+TimeUpdatePeriod;
    SmartUpdate(FormCrwDaq.StatusBar, sb_DaqTime, Format('%11.4f',[Daq.Timer.LocalTime]));
   end;
  end;
 end else begin
  TickCount:=GetTickCount64;
  if (TickCount>TimeToUpdate) then begin
   TimeToUpdate:=TickCount+TimeUpdatePeriod;
   SmartUpdate(FormCrwDaq.StatusBar, sb_DaqTime, Format('%11.4f',[0.0]));
  end;
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_crwdaq;
begin
end;

procedure Free_crw_crwdaq;
begin
end;

initialization

 Init_crw_crwdaq;

finalization

 Free_crw_crwdaq;

end.

//////////////
// END OF FILE
//////////////

