////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// ADAM Device family.                                                        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231204 - Modified for FPC (A.K.)                                         //
// 20240902 - WatchDogPulser                                                  //
////////////////////////////////////////////////////////////////////////////////

unit _crw_adamdev; // ADAM Device family

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild, Form_TextEditor, Form_CurveWindow, Form_SurfWindow,
 Form_CircuitWindow, Form_ConsoleWindow,  Form_TabWindow, Form_SpectrWindow,
 Form_Calculator, Form_ListBoxSelection, Form_UartTerminal, Form_CalibDialog,
 Form_DaqEditTagDialog, Form_DaqDeviceControl,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_ef, _crw_curves, _crw_riff, _crw_pio,
 _crw_polling, _crw_calib, _crw_couple,
 _crw_daqtags, _crw_daqevnt, _crw_sect,
 _crw_daqsys, _crw_daqdev, _crw_uart,
 _crw_appforms, _crw_apptools, _crw_apputils;

 {
 *******************************************************************************
 Конструктор создает устройства ADAM по модели
 *******************************************************************************
 }
function AdamDeviceConstructor(const Name,Family,Model,ConfigFile:LongString):TDaqDevice;

 {
 *******************************************************************************
 ADAM traffic counters record. Tx=Transmitter, Rx=Receiver.
 GetAdamTraffic returns Tx.Polls Tx.Bytes Rx.Polls Rx.Bytes
 *******************************************************************************
 }
type
 TAdamTrafficRec = packed record
  Tx,Rx:record Polls,Bytes:SizeInt; end;
 end;

function GetAdamTraffic(id:LongString; out traffic:TAdamTrafficRec):Boolean; overload;
function GetAdamTraffic(id:LongString):LongString; overload;

 {
 *******************************************************************************
 TAdamDevice            Абстрактное устройство ADAM. Устройства ADAM работают
                        по схеме запрос-ответ. Чтобы не перегружать цикл Idle
                        пустым ожиданием, при запросе данные пересылаются в
                        порт, а ожидания ответа не происходит - цикл опроса
                        продолжается. При поступлении из порта ответа
                        вызывается обрабочик ответа HandleAnswer.
                        Все возможные типы запросов пронумерованы
                        и циклически перебираются в соответствии с списком
                        опроса RequestList. RequestNumber-текущий номер
                        запроса.
                        Устройства ADAM не могут использоваться независимо
                        так как используют разделяемый рессурс - COM порт.
                        Все ADAM-ы должны опрашиваться только через список
                        TAdamDeviceList, который обеспечивает бесконфликтный
                        опрос.
 Start                  Помимо наследуемого Start инициализирует список
                        опроса RequestList вызовом InitRequestOrder.
 ImmediateAnswerRequest Вызывается для немедленного получения ответа на
                        запрос. Эту процедуру нельзя вызывать после старта
                        измерений ADAM, так как она может 'испортить' цикл
                        опроса. Процедура применяется для разовых операций
                        типа старта,настроек, калибровок и т.д. Для редких
                        операций (которые не слишком перегрузят цикл Idle)
                        можно вызывать ImmediateAnswerRequest в методах
                        GetRequest или Answer, так как внутри этих процедур
                        очередь опроса свободна и может быть временно
                        использована.
                        Процедура автоматически добавляет в строку запроса
                        и удаляет из строки отклика контрольную сумму при
                        сборе с контролем сумм, поэтому пользователь может
                        программировать так, будто контрольные суммы вовсе
                        не используются даже в режиме с контролем.
 GetRequest             Очередной запрос, который надо послать в цикле опроса
                        или пустая строка, если не надо опрашивать. Функция
                        GetRequest использует идентификатор запроса
                        RequestNumber чтобы выбрать тип очередного запроса.
                        При использовании контрольных сумм функция не должна
                        вычислять и добавлять в запрос контрольные суммы,
                        так как это делается на системном уровне,поэтому
                        пользователь может программировать так, будто
                        контрольные суммы вовсе не используются даже в режиме
                        с контролем.
 HandleAnswer           Вызывается, когда получен ответ на запрос.
                        Ответ на запрос содержится в поле Answer обьекта.
                        Интерпретация ответа зависит от типа запроса
                        RequestNumber.
                        При сборе с контролем сумм ответ уже проверен и
                        очищен от контрольной суммы, поэтому пользователь
                        может программировать так, будто контрольные суммы
                        вовсе не используются даже в режиме с контролем.
 GetTimeOut             Выдает требуемый для данного запроса интервал TimeOut
                        в миллисекундах. По умолчанию - 100 ms, для запросов
                        без ответа лучше ставить меньшее время.
 HandleTimeOut          Вызывается при обнаружениии TimeOut.
                        По умолчанию фиксирует ошибку ecAdatTimeout.
                        Перекрывается, если на некоторые запросы не должно
                        быть ответа.
 HandleFormatError      Вызывается при обнаружениии ошибки формата сообщений.
                        По умолчанию фиксирует ошибку ecAdamFormat.
 PrepareNextRequest     Подготовка следующего запроса. Процедура вызывается
                        после того, как прошел цикл запрос-ответ для
                        подготовки устройства к новому запросу.
                        Базовый метод циклически 'прокручивает' запросы
                        RequestNumber из списка RequestList. Перекрытые
                        методы могут делать еще что-то.
                        Например, если есть мультиплексированный АЦП,
                        в этом методе должно происходить переключение
                        на следующий канал.
                        Подчеркнем, что PrepareNextRequest не генерирует
                        запрос, а только подготавливает его заданием
                        номера RequestNumber. Реальный запрос произойдет
                        позже при вызове GetRequest.
 InitRequestList        Виртуальный метод строит список опроса RequestList.
                        При этом надо указать - какие из возможных запросов
                        действительно нужны. Например, запросы не нужны,
                        если к выходу не подключена кривая и данные некуда
                        записывать.
 InitRequestOrder       Инициализирует список опроса RequestList вызовом
                        InitRequestList и затем инициализирует RequestNumber.
                        Вызывается в методах Start, Animate, а также всегда
                        после модификации списка опроса.
 Примечания:
 1. Время события считается по времени возбуждения запроса RequestTime,
    так как ответ может прийти по прерыванию намного раньше следующего
    опроса по Idle.
 2. При использовании контрольных сумм надо учитывать следующее.
    С точки зрения пользователя - на уровне доступных полей и методов
    GetRequest,Request,HandleAnswer,Answer - работа выглядит так, как
    будто никаких сумм нет, хотя на системном уровне - в процедурах
    TAdam.ImmediateAnswerRequest и TAdamList.Idle - в запрос добавляется
    контрольная сумма, а ответ тестируется и очищается от контрольной суммы.
    Для выполнения разовых запросов надо использовать ImmediateAnswerRequest,
    Логику опроса надо строить на GetRequest/HandleAnswer. В обоих случаях
    запрос и ответ - без контрольных сумм, хотя внутри системы контрольные
    суммы используются.
 Дополнительные поля в конфигурации:
  Port = 2                 ; к какому подключен COM-у
  Address = 1              ; адрес устройства
  UsesCheckSum = false     ; использует ли модуль контрольные суммы
 *******************************************************************************
 }
type
 TAdamDevice = class(TDaqDevice)
 protected
  PortN         : Integer;        {номер порта к которому подключен }
  Address       : Integer;        {адрес устройства                 }
  UsesCheckSum  : Boolean;        {используются ли контрольные суммы}
  Request       : LongString;     {какой был последний запрос       }
  Answer        : LongString;     {ответ на запрос от устройства    }
  RequestTime   : Double;         {время-когда был последний запрос }
  RequestNumber : Byte;           {номер-идентификатор типа запроса }
  RequestList   : TByteSet;       {список используемых запросов     }
  Traffic       : TAdamTrafficRec;{счетчики для регистрации траффика}
  function    ImmediateAnswerRequest(const TheRequest:LongString; Attempts:Integer):LongString;
  function    GetRequest:LongString; virtual;
  procedure   HandleAnswer; virtual;
  function    GetTimeOut:Double; virtual;
  procedure   HandleTimeOut; virtual;
  procedure   HandleFormatError; virtual;
  procedure   PrepareNextRequest; virtual;
  procedure   InitRequestList; virtual;
  procedure   InitRequestOrder;
 public
  constructor Create(const aName:LongString);
  destructor  Destroy; override;
  procedure   Config(FileName:LongString); override;
  procedure   Animate; override;
  function    GetProperty(P:TText):TText; override;
  function    Start:Boolean; override;
  function    HandleMessage(const aMsg:LongString; aFlags:Cardinal=hf_Default):Double; override;
  function    GetTraffic:TAdamTrafficRec;
  procedure   ClearTraffic;
 end;

 {
 *******************************************************************************
 Список устройств типа ADAM обеспечивает помимо хранения:
 1. Корректное открытие-закрытие используемых COM-портов.
 2. Непротиворечивый циклический опрос устройств ADAM типа запрос-ответ.
    Для ускорения сделан отдельный цикл опроса для каждого из используемых COM-портов.
 3. Периодическое звуковое оповещение при сбоях на линии.
 4. Опрос идет в отдельном потоке Polling.
 *******************************************************************************
 }
type
 TAdamDeviceList = class(TDaqDeviceList)
 private
  myPolling     : TPolling;
  mySender      : packed array[1..MaxPortNum] of TAdamDevice;
  myUsesPorts   : TByteSet;
  myPollPorts   : LongString;
  myPortLatch   : TSysCriticalSection;
  myLastErrors  : LongInt;
  myCheckTimer  : TIntervalTimer;
  myCheckThresh : Double;
  function    GetAdamDevices(Index:Integer):TAdamDevice;
  procedure   SetAdamDevices(Index:Integer; aAdamDevice:TAdamDevice);
  function    GetPolling:TPolling;
  function    OpenSerialPorts(ConfigFile:LongString):Boolean;
  procedure   CloseSerialPorts;
  procedure   InitSenders;
  function    NextSender(n:Integer):TAdamDevice;
  procedure   CheckConnection;
  procedure   MainLoop;
  function    GetUsesPorts:TByteSet;
  procedure   SetUsesPorts(const Ports:TByteSet);
  function    GetPollPorts:LongString;
  function    GetUsesPort(n:Integer):Boolean;
 protected
  property    UsesPorts : TByteSet read GetUsesPorts write SetUsesPorts;
 public
  constructor Create(aOwns:Boolean);
  destructor  Destroy; override;
 public
  property    AdamDevices[i:Integer] : TAdamDevice read GetAdamDevices write SetAdamDevices; default;
  property    Polling                : TPolling    read GetPolling;
  property    UsesPort[n:Integer]    : Boolean     read GetUsesPort;
  function    Start:Boolean; override;
  procedure   Stop; override;
  procedure   Idle; override;
  procedure   Poll; override;
  procedure   StartSession; override;
  procedure   StopSession; override;
  function    CommonProperty(P:TText):TText;
 end;

 {
 *******************************************************************************
 В этом списке хранятся все устройства типа ADAM.
 *******************************************************************************
 }
function AdamDeviceList:TAdamDeviceList;

 {
 *******************************************************************************
 TAdamXXXXDevice        Базовый объект для устройств ADAM серий:
                         Advantec ADAM серия 4000
                         ICP      ADAM серия 7000
                        и совместимых с ними по командам.
 CheckDevice            Проверка устройства основана на чтении имени модуля.
 ReadModuleName         Читает имя модуля (пустая строка при неудаче)
 ReadModuleFirmware     Читает версию модуля (пустая строка при неудаче)
 ReadStatus             Читает статус модуля
 WriteStatus            Записывает статус модуля
 CheckRangeCode         Проверяет и при необходимости корректирует
                        код диапазона
 CheckRangeCodes        Проверяет и при необходимости корректирует
                        коды диапазонов - только для 7019
 ReadRangeCodes         Читает массив кодов диапазонов - только для 7019
 SetupWatchDog          Установка параметров WatchDog и сброс - стандартная
                        процедура при старте.
 HandleWatchDogStatus   Вызывается в HandleAnswer для проверки статуса
                        WatchDog и при необходимости - сброса.
 AwakeWatchDog          Проверяет, не пора ли послать HostOk, если пора, то
                        включить его в список опроса. Для сокращения времени
                        простоя сообщение HostOk посылается не в каждом цикле
                        опроса, а периодически включается в список опроса, а
                        затем удаляется из него. Таким образом посылка HostOk
                        делается не чаще чем нужно.
 SleepWatchDog          Если HostOk был послан, то исключить его из списка
                        опроса
 WatchDogTime           Если 0, запрет host-сторожа, иначе время host-сторожа
                        в условных единицах равных 0.1 секунды.
 WatchDogPulser         Если >0, задает период посылки импульсов(команд) сброса
                        сторожевого таймера WatchDog, в миллисекундах.
 RangeFactor            Коэффициент для перевода данных в милливольты или 1
                        в зависимости от диапазона
 Start                  Помимо наследуемого Start применяет список StartCommand.
 Stop                   Помимо наследуемого Stop  применяет список StopCommand.
 Дополнительные поля в конфигурации:
  RangeCode = $0E          ; требуемый код диапазона
  RangeCodes = $08, $08, $09, $08, $08, $09, $08, $08 ; коды диапазонов для 7019
  WatchDogTime             ; время host-сторожа в секундах или 0
  WatchDogPulser           ; период сброса Watchdog (мсек) или 0
  StartCommand = XXXXX - посылка команды XXXXX при старте
                         работает замена ?? на адрес устройства, например:
                         StartCommand = $??M
                         StartCommand = $??F
  StopCommand  = XXXXX - посылка команды XXXXX при остановке
                         работает замена ?? на адрес устройства, например:
                         StopCommand = $??M
                         StopCommand = $??F
 *******************************************************************************
 }
type
 TAdamXXXXDevice = class(TAdamDevice)
 protected
  ModelId          : Integer;                       {идентификатор модели}
  RangeCode        : Word;                          {код диапазона}
  RangeCodes       : String[32];                    {коды диапазона для 7019}
  AvailRange       : TByteSet;                      {список допустимых кодов диапазона}
  DataFormat       : Word;                          {код типа шкалы: Engeneering,%,HEX,Ohm + биты FastPoll,Reject50}
  MaskFormat       : Word;                          {маска для проверки кода типа шкалы}
  ad_gate          : TByteSet;                      {карта разрешения аналогового опроса}
  ad_list          : TByteSet;                      {подключенные аналоговые каналы}
  WatchDogSupports : Boolean;                       {поддерживается ли система команд WatchDog}
  WatchDogTime     : Word;                          {0=запрет, иначе время сторожа в усл. ед. }
  WatchDogPulser   : Integer;                       {период импульсов AdamWatchDogPulser, ms }
  ad_chan_no       : Integer;                       {число AD-каналов      }
  ad_gain          : packed array[0..31] of Double; {коэфф. усиления канала}
  ad_zero          : packed array[0..31] of Double; {смещение канала       }
  StartList        : TText;                         {список команд при старте}
  StartAns         : TText;                         {список ответов при старте}
  StopList         : TText;                         {список команд при стопе}
  StopAns          : TText;                         {список ответов при стопе}
  procedure   InitRequestList; override;
  function    ReadModuleName:LongString;
  function    ReadModuleFirmware:LongString;
  function    ReadStatus(out RangeTypeCode  : Byte;
                         out BaudRateCode   : Byte;
                         out DataFormatCode : Byte):Boolean;
  function    WriteStatus(NewAddress     : Byte;
                          RangeTypeCode  : Byte;
                          BaudRateCode   : Byte;
                          DataFormatCode : Byte):Boolean;
  function    GetUnitFormat:Word;
  procedure   SetUnitFormat(value:Word);
  function    GetFastPoll:Boolean;
  procedure   SetFastPoll(value:Boolean);
  function    GetReject50:Boolean;
  procedure   SetReject50(value:Boolean);
  property    UnitFormat:Word  read GetUnitFormat write SetUnitFormat;
  property    FastPoll:Boolean read GetFastPoll   write SetFastPoll;
  property    Reject50:Boolean read GetReject50   write SetReject50;
  function    HasMaskFormat(mask:Word):Boolean;
  procedure   AddMaskFormat(mask:Word);
  function    CheckRangeCodes(n:Integer):Boolean;
  procedure   ReadRangeCodes(FileName:LongString; n:Integer);
  function    CheckRangeCode:Boolean;
  function    CheckDataFormat(Code,Mask:Byte):Boolean;
  function    SetupWatchDog:Boolean;
  procedure   HandleWatchDogStatus;
  procedure   AwakeWatchDog(HostOkRequest:Integer);
  procedure   SleepWatchDog(HostOkRequest:Integer);
  function    ad_transform(chan:Word; data:Double):Double;
  function    RangeFactor:Double;
  function    DoStartCommands:Boolean;
  function    DoStopCommands:Boolean;
  procedure   BugLog(var aErrors:Integer; const aMsg:LongString);
 public
  constructor Create(const aName:LongString; aModelId:Integer);
  destructor  Destroy; override;
  procedure   Config(FileName:LongString); override;
  procedure   Animate; override;
  function    Start:Boolean; override;
  procedure   Stop; override;
  function    GetProperty(P:TText):TText; override;
  function    CheckDevice:Boolean; override;
  function    NewPropertyDialog:TDaqDevicePropertyDialog; override;
  function    UpdatePropertyDialog:Boolean; override;
 end;

 {
 *******************************************************************************
 Объект:       Adam4017
               Adam7017
               Adam4018
               Adam7018
               Adam7019
               Adam7033
               Adam7015
               Adam87013
               Adam87015
               Adam87017
               Adam87018
               На самом деле эти модули реализованы в одном AdamX017
               с вариациями за счет задания ModelId, так как система команд
               общая.
 Объявление:   Device Adam Adam-4017
               Device Adam Adam-7017
               Device Adam Adam-4018
               Device Adam Adam-7018
               Device Adam Adam-7019
               Device Adam Adam-7033
               Device Adam Adam-7015
               Device Adam Adam-87013
               Device Adam Adam-87015
               Device Adam Adam-87017
               Device Adam Adam-87018
 Назначение:   Модули многоканального аналогового ввода.
 Входы:        нет.
 Выходы:       AnalogOutput(0..7) - измерительные каналы (x017,x018).
               AnalogOutput(0..3) - измерительные каналы (87013).
               AnalogOutput(8)    - канал измерения температуры холодного
               спая (только для x018)
 Описание:     АЦП имеет разрядность 16 бит. Модули x017 и x018 отличаются
               тем, что модули x018 ориентированы на термопарные измерения
               и имеют кроме измерительных каналов канал измерения
               температуры холодного спая. Кроме того, есть различия в
               диапазонах.
               Во всех диапазонах приняты единицы измерения (mV,mA,C).
 Конфигурация: без особенностей
 Пример:       [DeviceList]
               Adam1 = device adam adam-7018
               [Adam1]
               Port = 2
               Address = 1
               UsesCheckSum = false
               RangeCode = $0E
               RangeCodes = $08, $09, $08, $09, $08, $09, $08, $09 ; For 7019
               DataFormat = 2
               FastRead = 1
               FastPoll = 0
               Reject50 = 1
               Link AnalogOutput 0 with curve Chan1
               Link AnalogOutput 8 with curve ColdJunction
 *******************************************************************************
 }
type
 TAdamX017Device = class(TAdamXXXXDevice)
 protected
  FastRead    : Boolean;
  function    GetRequest:LongString; override;
  procedure   HandleAnswer; override;
  procedure   InitRequestList; override;
  function    GetTimeOut:Double; override;
  procedure   HandleTimeOut; override;
 public
  constructor Create(const aName:LongString; aModelId:Integer);
  function    Start:Boolean; override;
  procedure   AdvancedPropertyDialog; override;
  procedure   Config(FileName:LongString); override;
  function    GetProperty(P:TText):TText; override;
 end;

 {
 *******************************************************************************
 Объект:       Adam4050  Adam7050  Adam4052  Adam7052  Adam4053  Adam7053
               Adam4060  Adam7060  Adam7063  Adam7065  Adam7066  Adam7067
               Adam7041  Adam7042  Adam7043  Adam7044  Adam87051 Adam87052
               Adam87053 Adam87054 Adam87055 Adam87057 Adam87063 Adam87064
               Adam87065 Adam87066 Adam87068 Adam87069 Adam87058
               На самом деле эти модули реализованы в одном AdamDIO
               с вариациями за счет задания ModelId, так как система команд
               общая.
 Объявление:   Device Adam Adam-4050
               Device Adam Adam-7050
               Device Adam Adam-7051
               Device Adam Adam-4052
               Device Adam Adam-7052
               Device Adam Adam-4053
               Device Adam Adam-7053
               Device Adam Adam-7058
               Device Adam Adam-7059
               Device Adam Adam-4060
               Device Adam Adam-7060
               Device Adam Adam-7063
               Device Adam Adam-7065
               Device Adam Adam-7066
               Device Adam Adam-7067
               Device Adam Adam-7041
               Device Adam Adam-7042
               Device Adam Adam-7043
               Device Adam Adam-7044
               Device Adam Adam-7045
               Device Adam Adam-87051
               Device Adam Adam-87052
               Device Adam Adam-87053
               Device Adam Adam-87054
               Device Adam Adam-87055
               Device Adam Adam-87057
               Device Adam Adam-87058
               Device Adam Adam-87063
               Device Adam Adam-87064
               Device Adam Adam-87065
               Device Adam Adam-87066
               Device Adam Adam-87068
               Device Adam Adam-87069
 Назначение:   Модули цифрового ввода/вывода.
 Входы:        DigitalInput(..)
               Для модулей 4050,7050,4060,7060,7063,7065,7066,7067,
               7042,7043,7044,7045 - равно числу электрических выходов.
               Для модулей 4052,4053,7041 - нет.
 Выходы:       Для модулей 4050,7050,7051,4052,7052,4053,7053,4060,7060,7063,7065,
               7041,7044 - DigitalOutput(1) содержит состояние
               электрических цифровых входов.
               Для модулей 7066,7067,7042,7043,7045 - нет.
 Описание:     Модули отличаются числом электрических входов (DI)
               и выходов (DO):
               4050  - 7  DI, 8  DO
               4052  - 8  DI, 0  DO
               4053  - 16 DI, 0  DO
               4060  - 4  DI, 4  DO
               7041  - 14 DI, 0  DO
               7042  - 0  DI, 13 DO
               7043  - 0  DI, 16 DO
               7044  - 4  DI, 8  DO
               7045  - 0  DI, 16 DO
               7050  - 7  DI, 8  DO
               7051  - 16 DI, 0  DO
               7052  - 8  DI, 0  DO
               7053  - 16 DI, 0  DO
               7058  - 8  DI, 0  DO
               7059  - 8  DI, 0  DO
               7060  - 4  DI, 4  DO
               7063  - 8  DI, 3  DO
               7065  - 4  DI, 5  DO
               7066  - 0  DI, 7  DO
               7067  - 0  DI, 7  DO
               87051 - 16 DI, 0  DO
               87052 - 8  DI, 0  DO
               87053 - 16 DI, 0  DO
               87054 - 8  DI, 8  DO
               87055 - 8  DI, 8  DO
               87057 - 0  DI, 16 DO
               87058 - 8  DI, 0  DO
               87063 - 4  DI, 4  DO
               87064 - 0  DI, 8  DO
               87065 - 0  DI, 8  DO
               87066 - 0  DI, 8  DO
               87068 - 0  DI, 8  DO
               87069 - 0  DI, 8  DO
 Конфигурация: RangeCode должен иметь значение $40
               PowerOnValue - значение выхода после включения модуля
               SafeValue    - значение выхода при сбое host-сторожа
 Пример:       [DeviceList]
               Adam1 = device adam adam-4050
               ...
               [Adam1]
               Port = 2
               Address = 1
               UsesCheckSum = false
               RangeCode = $40
               Link DigitalInput  0 with curve DI1
               Link DigitalOutput 0 with curve DO1
 *******************************************************************************
 }
type
 TAdamDIOsDevice = class(TAdamXXXXDevice)
 protected
  SafeValue    : Word;
  PowerOnValue : Word;
  function    ImmediateOut(w:Word):Boolean;
  function    GetRequest:LongString; override;
  procedure   HandleAnswer; override;
  procedure   PackData(data1,data2:Word);
  procedure   InitRequestList; override;
  function    GetTimeOut:Double; override;
  procedure   HandleTimeOut; override;
 public
  constructor Create(const aName:LongString; aModelId:Integer);
  procedure   Config(FileName:LongString); override;
  function    GetProperty(P:TText):TText; override;
  function    Start:Boolean; override;
 end;

 {
 *******************************************************************************
 Объект:       Adam4021
               Adam7021
               Adam7024
               Adam87024
 Объявление:   Device Adam Adam-4021
               Device Adam Adam-7021
               Device Adam Adam-7024
               Device Adam Adam-87024
 Назначение:   4021,7021 - модуль одноканального аналогового вывода.
                           ЦАП 12 бит.
               7024,87024 - модуль 4-канального аналогового вывода.
                           ЦАП 14 бит.
 Входы:        Для 4021,7021:
               AnalogInput(0)-данные для вывода на ЦАП.
               Для 7024,87024:
               AnalogInput(0..3)-данные для вывода на ЦАПы.
 Выходы:       нет.
 Описание:     Модуль выдает на выходы аналоговый сигнал ЦАП, значение
               которого берет с входов.
               При старте модуль выставляет режимы сторожевого таймера
               и значения выходов 'при включении'-PowerOnValue и
               'безопасный'-SafeValue.
               Во всех диапазонах приняты единицы измерения (mV,mA).
 Конфигурация: Для модуля 4021,7021:
               SafeValue = s0
               PowerOnValue = p0
               Для модуля 7024,87024:
               SafeValue = s0, s1, s2, s3
               PowerOnValue = p0, p1, p2, p3
               SafeValue-'безопасный' режим при сбое Host-компьютера
               PowerOnValue-режим 'при включении'
 Пример:       [DeviceList]
               Adam1 = device adam adam-7021
               Adam2 = device adam adam-7024
               [Adam1]
               Port = 2
               Address = 1
               UsesCheckSum = false
               RangeCode = $32
               Link AnalogInput 0 with curve A1
               WatchDogPulser = 1000
               WatchDogTime = 10
               SafeValue = 0
               PowerOnValue = 0
               [Adam2]
               Port = 2
               Address = 2
               UsesCheckSum = false
               RangeCode = $35
               Link AnalogInput 0 with curve B1
               Link AnalogInput 1 with curve B2
               Link AnalogInput 2 with curve B3
               Link AnalogInput 3 with curve B4
               WatchDogTime = 10
               SafeValue = 0, 0, 0, 0
               PowerOnValue = 0, 0, 0, 0
 *******************************************************************************
 }
type
 TAdamX021Device = class(TAdamXXXXDevice)
 protected
  SafeValue    : packed array[0..3] of Double;
  PowerOnValue : packed array[0..3] of Double;
  Enable16Bit  : LongInt;
  function    DaStr(data:Double):LongString;
  function    GetRequest:LongString; override;
  procedure   HandleAnswer; override;
  function    GetTimeOut:Double; override;
  procedure   HandleTimeOut; override;
  procedure   InitRequestList; override;
 public
  constructor Create(const aName:LongString; aModelId:Integer);
  procedure   Config(FileName:LongString); override;
  function    GetProperty(P:TText):TText; override;
  function    Start:Boolean; override;
 end;

 {
 *******************************************************************************
 Объект:       Adam4011
               Adam7011
               Adam4012
               Adam7012
               Adam4013
               Adam7013
               Adam4014
               Adam7014
               На самом деле эти модули реализованы в одном AdamX011
               с вариациями за счет задания ModelId, так как система команд
               общая.
 Объявление:   Device Adam Adam-4011
               Device Adam Adam-7011
               Device Adam Adam-4012
               Device Adam Adam-7012
               Device Adam Adam-4013
               Device Adam Adam-7013
               Device Adam Adam-4014
               Device Adam Adam-7014
 Назначение:   Все модули этой группы имеют один канал аналогового ввода.
               Кроме того:
               4011,7011-имеют датчик холодного спая.
               4011,7011,4012,7012,4014,7014-имеют 2-уровневый пороговый
               сторож - Alarm, цифровой вход (1 канал), счетчик событий
               (1 канал), цифровой вывод (2 бит), причем цифровой вывод
               управляется с host-компьютера или автономно сторожем Alarm.
               Режим Alarm позволяет делать автономные регуляторы для
               систем повышенной надежности или распределенных систем.
               Модули 7000-серии поддерживают команды Watchdog-таймера.
 Входы:        4011,7011,4012,7012,4014,7014:
                DigitalInput(0..1) - цифровой вывод.
                DigitalInput(2)    - 1/0 - включение/выключение Alarm.
                DigitalInput(3)    - 1/0 - режим Latch/Momentary Alarm.
                AnalogInput(0)     - значение Alarm Lo - нижний  порог.
                AnalogInput(1)     - значение Alarm Hi - верхний порог.
 Выходы:       Все модули:
                AnalogOutput(0)    - измерительный канал.
               4011,7011:
                AnalogOutput(1)    - холодный спай.
                AnalogOutput(2)    - интегральный счетчик событий.
                AnalogOutput(3)    - счетчик частоты счетчик событий.
                DigitalOutput(0)   - цифровой ввод и Alarm
               4012,7012,4014,7014
                AnalogOutput(1)    - интегральный счетчик событий.
                AnalogOutput(2)    - счетчик частоты счетчик событий.
                DigitalOutput(0)   - цифровой ввод и Alarm
 Описание:     АЦП имеет разрядность 16 бит. Модули ADAM-4011,7011
               ориентированы на термопарные измерения и имеют кроме
               измерительных каналов канал измерения температуры холодного
               спая. 4013,7013 ориентированы на RTD-датчик. 4014,7014 имеют
               токовый канал.
               Кроме того, есть различия в диапазонах.
               Во всех диапазонах приняты единицы измерения (mV,mA,C).
               Модули 4011,7011,4012,7012,4014,7014 имеют двухуровневое
               пороговое устройство Alarm, 2 канала цифрового вывода,
               канал цифрового ввода и счетчика.
               Необходимо помнить, что каналы цифрового вывода доступны для
               управления когда выключен Alarm. При включении Alarm значения
               цифровых входов игнорируются, так как цифровой вывод в этом
               режиме определяется пороговым устройством и не зависит от
               сигнала с Host-компьютера. При этом сигнал на DO0(Lo) имеет
               значение 1 если сигнал на входе МЕНЬШЕ нижнего порога AlarmLo,
               а сигнал на DO1(Hi) имеет значение 1 если сигнал на входе
               БОЛЬШЕ верхнего порога AlarmHi. Кроме того, при в режиме Alarm
               игнорируются состояния WatchDog-таймера.
               Alarm имеет два режима - моментальный (Momentary) и с защелкой
               (Latch). В моментальном режиме состояние цифровых выходов
               определяется текущим уровнем на аналоговом входе, а в режиме
               с защелкой цифровые выходы фиксируются при срабатывании
               порогового устройства и далее не зависят от входа до команды
               сброса защелки.
               Цифровой вход модуля используется как для чтения моментального
               значения цифрового уровня, так и для счетчика событий, который
               срабатывает по фронту на цифровом входе и считает число
               импульсов.
               Состояние Alarm, режим Latch, Lo, Hi задаются из конфигурации
               или через входы устройства.
 Конфигурация: PowerOnValue = f      цифровой вывод при включении питания
               SafeValue = f         безопасное состояние цифровых выходов
               AlarmEnable = b       1/0 состояние Alarm вкл/выкл
               AlarmLatch = b        1/0 режим Latch/Momentary
               AlarmLo = f           нижний порог
               AlarmHi = f           верхний порог
               FrequencyPeriod = f   период времени в секундах для подсчета
                                     импульсов для вычисления частоты
               FrequencyScale = f    множитель частоты 1-Гц, 0.001-кГц...
 Пример:       [DeviceList]
               Adam1 = device adam adam-7011
               [Adam1]
               Port = 2
               Address = 1
               UsesCheckSum = false
               RangeCode = $01
               Link AnalogOutput 0 with curve Chan1
               Link AnalogOutput 1 with curve ColdJunction
               PowerOnValue = 0
               SafeValue = 0
               AlarmEnable = 1
               AlarmLatch = 0
               AlarmLo = -10
               AlarmHi = 10
               FrequencyPeriod = 2
               FrequencyScale = 0.001
               FastPoll = 0
               Reject50 = 1
 *******************************************************************************
 }
type
 TAdamX011Device=class(TAdamXXXXDevice)
 protected
  SafeValue    : Word;
  PowerOnValue : Word;
  CounterBase  : Double;
  FreqPeriod   : Double;
  FreqScale    : Double;
  LastCntVal   : Double;
  LastCntTime  : Double;
  AlarmSupports: Boolean;
  AlarmEnable  : Boolean;
  AlarmLatch   : Boolean;
  AlarmLo      : Double;
  AlarmHi      : Double;
  nCJC         : Integer; {номер кривой CJC или -1}
  nEvC         : Integer; {номер кривой EventCounter или -1}
  function    GetRequest:LongString; override;
  procedure   HandleAnswer; override;
  function    GetTimeOut:Double; override;
  procedure   HandleTimeOut; override;
  procedure   InitRequestList; override;
  function    DaStr(data:Double):LongString;
 public
  constructor Create(const aName:LongString; aModelId:Integer);
  procedure   Config(FileName:LongString); override;
  function    GetProperty(P:TText):TText; override;
  function    Start:Boolean; override;
  procedure   AdvancedPropertyDialog; override;
 end;


 {
 *******************************************************************************
 Объект:       Adam4080
               Adam7080
 Объявление:   Device Adam Adam-4080
               Device Adam Adam-7080
 Назначение:   2-канальные модули для счета импульсов или измерения частоты.
               В данной версии Alarm не используется (всегда запрещен).
 Входы:        DigitalInput(0,1) - цифровые выходы DO0/Lo,DO1/Hi
 Выходы:       AnalogOutput(0,1) - частота на входах In0,In1
               AnalogOutput(2,3) - интегральный счет на входах In0,In1
                                   (в режиме RangeCode=$50)
               DigitalOutput(0)  - текущее состояние цифровых выходов
                                   DO0/Lo,DO1/Hi (readback) и Alarm
 Описание:     Реализует 2-канальный счетчик или измеритель частоты.
               Сигнал можно подавать TTL (неизолированный вход) или
               не TTL (изолированный вход от 0 до 24 V)
               Для TTL можно задавать напряжения логического нуля и единицы
               (Trigger). Фильтр позволяет регистрировать сигналы только
               заданного интервала длительности (Filter). Можно также
               запрещать счет внешними воротами (GateMode).
               В данной версии Alarm не используется (всегда запрещен).
 Конфигурация: RangeCode = $50-модуль работает как счетчик
                           $51-модуль работает как частотомер
               InputMode = режим ввода-изолированный или неизолированный вход
                           0 = неизолир.канал 0, неизолир.канал 1
                           1 =   изолир.канал 0,   изолир.канал 1
                           2 = неизолир.канал 0,   изолир.канал 1
                           3 =   изолир.канал 0, неизолир.канал 1
               GateMode = режим ворот
                           0 = ворота с низким уровнем активности
                           1 = ворота с низким уровнем активности
                           2 = ворота запрещены
               Filter = on lo hi
                           on= 0 - фильтр запрещен 1 - фильтр разрешен
                           lo,hi - задают длительность импульсов сигнала
                           в микросекундах от 2 до 65535
               Trigger = lo hi
                           lo,hi от 0 до 50 задают уровни 0 и 1
                           в единицах 0.1 вольт
               FreqPeriod = время интегрирования частотомера в секундах
                            (в режиме $50)
               FreqScale  = 1 масштаб частоты 1=Hz, 0.001=kHz
 Пример:       файл _7080.cfg
 *******************************************************************************
 }
type
 TAdamX080Device = class(TAdamXXXXDevice)
 protected
  InputMode   : Word;
  GateMode    : Word;
  Filter      : packed record OnOff,Lo,Hi : Word; end;
  Trigger     : packed record Lo,Hi : Word; end;
  FreqPeriod  : Double;
  FreqScale   : Double;
  CounterBase : packed array[0..1] of Double;
  LastCntVal  : packed array[0..1] of Double;
  LastCntTime : packed array[0..1] of Double;
  function    GetRequest:LongString; override;
  procedure   HandleAnswer; override;
  function    GetTimeOut:Double; override;
  procedure   HandleTimeOut; override;
  procedure   InitRequestList; override;
 public
  constructor Create(const aName:LongString; aModelId:Integer);
  procedure   Config(FileName:LongString); override;
  function    GetProperty(P:TText):TText; override;
  function    Start:Boolean; override;
  function    HandleMessage(const aMsg:LongString; aFlags:Cardinal=hf_Default):Double; override;
 end;

 {
 *******************************************************************************
 Объект:       RSDAQHost  (RemoteSerialDataAcQuisitionHost)
 Объявление:   Device Adam RSDAQHost
 Назначение:   Для сбора данных с удаленной системы DAQ через RS-232.
               Этот объект позволяет получать на основном компьютере данные
               с автономной программы работающей на другом компьютере и таким
               образом получать строить систему сбора на основе одной мощной
               машины и простых программ сбора на слабых машинах,обслуживающих
               отдельные подсистемы.
 Входы:        DigitalInput(0)-сигнал запрещения сбора.
               Служит для приостановки сбора данных с удаленной машины.
 Выходы:       AnalogOutput(0..255)-данные с удаленной системы DAQ.
               Смысл данных заранее не оговорен и определяется конкретной
               DAQ которая установлена на удаленной машине.
 Описание:     Устройство выдает в цикле опроса запросы типа
               #AANN где AA-адрес NN-номер канала на линию и ждет ответа
               >data (данные data в свободном float-формате типа +3,14159E+0)
               или
               > если данные не готовы.
               Программа на удаленной машине должна принимать команду #AANN и
               после анализа адреса (служащего для различения разных машин)
               выдавать данные из канала номер NN. Если нет данных в канале,
               выдается >. Смысл данных определяется договоренностью между
               программой на удаленной машине и Host-ом в каждом случае.
 Конфигурация: без особенностей.
 *******************************************************************************
 }
type
 TRSDAQHostDevice=class(TAdamDevice)
 protected
  function    GetRequest:LongString; override;
  procedure   HandleAnswer; override;
  procedure   InitRequestList; override;
 public
  constructor Create(const aName:LongString);
 end;

 {
 *******************************************************************************
 Объект:       Balzers TPG 256
               Класс устройств:     MaxiGauge - 6-канальный модуль для
                                    измерения и контроля вакуума
               Фирма-производитель: Balzers Instruments
               Литература:          803088BE.PDF,803086BE.PDF
               Адрес в Интернет:    www.balzers.com
 Объявление:   Device Adam Balzers-TPG256
 Назначение:   6-канальный контроллер датчиков давления вакуума.
 Входы:        нет.
 Выходы:       AnalogOutput(0..5)  - измерительные каналы давления.
                Сюда записывается измеренное давление вакуума, при условии,
                что статус ошибки для данного канала равен 0.
               DigitalOutput(0..5) - статус ошибок в каналах давления.
                0 = измеренные данные в порядке
                1 = давление ниже диапазона
                2 = давление выше диапазона
                3 = ошибка сенсора
                4 = сенсор выключен
                5 = сенсор не подключен
                6 = ошибка идентификации сенсора
               DigitalOutput(6..11) - состояние сенсора включен-выключен.
                0 = сенсор не изменил состояния
                1 = сенсор выключен
                2 = сенсор включен
               DigitalOutput(12..17) - состояние реле
                0 = выключено
                1 = включено
 Описание:     Имеет 6 каналов для подключения датчиков давления, 6 реле для
               управления по давлению от датчиков,а также жидкокристаллический
               дисплей для отображения параметров и измеряемых давлений.
               Связан с PC через RS232 или RS485.
               При связи через RS485 надо выставлять UsesRS485=1 и
               использовать программную переадресацию (7188).
 Конфигурация: При конфигурировании надо иметь в виду, что в зависимости от
               типа сенсоров некоторые значения параметров могут быть
               недопустимы. Подробности описаны в документах фирмы Balzers.
               StartCommand = XXXXX - посылка команды XXXXX при старте
                              формат команды в соответствии с 803088BE.PDF
               StopCommand  = XXXXX - посылка команды XXXXX при остановке
                              формат команды в соответствии с 803088BE.PDF
               UsesRS485    = 0 - RS485 не используется
                              1 - RS485    используется
                              В режиме RS485 перед посылкой команд
                              перед ними вставляется префикс $AA, а из
                              ожидаемого ответа "вычитается" префикс !AA
                              где AA-адрес (hex).
                              Это сделано в расчете на то, что устройство
                              подключается к RS232-порту автономного
                              контроллера типа 7188,  который программно
                              переадресует виртуальный адрес AA на общей
                              шине в реальный порт. Хотя с 7188 Balzers
                              будет обмениваться обычными командами, с точки
                              зрения PC это будет адресуемое устройство.
 Пример:       [DeviceList]
               TPG256    = device adam  Balzers-TPG256
               [TPG256]
               Comment = Balzers TPG256 MaxiGauges unit
               InquiryPeriod = 50
               Link AnalogOutput  0  with curve gauge1
               Link DigitalOutput 0  with curve senerr1
               Link DigitalOutput 6  with curve senon1
               Link DigitalOutput 12 with curve sensps1
               Port = 1
               Address = 1
               UsesCheckSum = false
               UsesRS485 = 0
               StartCommand = LOC,0
               StartCommand = UNI,0
               StopCommand = LOC,0
 **********************
 Список команд TPG-256: [,x] - необязательные параметры, (*) - (default)
 **********************
 SEN[,x,x,x,x,x,x]          Sensor on/off
                            x: 0=NoChange,1=Off,2=On
 SCx[,y,z,x.xEsxx,y.yEsyy]  Sensor control
                            x: Controlled sensor
                                A=sensor1,B=sensor2,C=sensor3,
                                D=sensor4,E=sensor5,F=sensor6
                            y: switch on the controlling source of the sensor
                               0..5=sensor1..6, 6=external control
                               7=manual(*),8=hot start
                            z: switch off the controlling source of the sensor
                               0..5=sensor1..6, 6=external control
                               7=manual(*)
                            x.xEsxx: switching off value
                            y.yEsyy: switching on  value
 PRx                        Status and pressure
                            x: sensor 1..6
                            Answer is x,x.xxxEsx
                            x: status
                               0=Ok, 1=Underrange,2=Overrange,3=Sensor error
                               4=Sensor off,5=NoSensor,6=Identification error
                            x.xxxEsx: measurement value
 DCD[,x]                    Digits: x=2 display x.x, x=3 display x.xx
 CID[,xxxx,xxxx,xxxx,xxxx,xxxx,xxxx]  measurement points names (4 symbols)
 UNI[,x]                    Units of measurement x: 0=mBar(*),1=Torr,2=Pascal
 DCB[,x]                    Bargraph x: 0=Off(*),1=bargraphmeasurement range
                                        2=bargraph 1 decade
 DCC[,xx]                   Contrast x: 0..20,dark..light, default=10
 DCS[,xx]                   Screensave x:0=off, 1..99-hours
 SPx[,y,w.wwEsw,z.zzEsz]    Threshold value setting,allocation
                            x: switching function(Relay) x=1..6->A..F
                            w.wwEsw: lower threshold [mbar], default 1.0E-11
                            z.zzEsz: Upper threshold [mbar], default 9.0E-11
 SPS                        Set point status, answer=x,x,x,x,x,x
                            x: 0=off,1=On
 PUC[,x,x,x,x,x,x]          Underrange control x: 0=deactivate(*),1=activate
 LOC[,x]                    Entry lock x: 0=off(*),1=On
 FIL[,x,x,x,x,x,x]          Filter time constant x:0=fast,1=standard(*),2=slow
 CAx[,x.xxx]                Calibration factor of sensor x=1..6
                            x.xxx = 0.100..9.999 for logarithmic sensor
                                    0.500..2.000 for linear (1=default)
 OFC[,x,x,x,x,x,x]          Offset correction x: 0=off(*),1=activated,
                            2=actual measurement value=offset value
 FSR[,x,x,x,x,x,x]          full scale range x: 0=1,1=10,2=100,3=1000mbar(*),
                            4=2,5=5,6=10,7=50bar
 SAV[,1]                    activate factory settings
 RSX[,x]                    interface x: 0=RS232C(*),1=RS422,2=RS422isol,
                                         3=RS-485isol
 BAU[,x]                    Baudrate 0=300,     2=1200, 2=2400, 3=4800,
                                     4=9600(*), 5=19200
 NAD xx                     RS-485 node address xx=00..31
 ERR                        Error status, answer is xxxxx,yyyyy
                            xxxxx: 0=noerror, other is errors, see doc.
                            yyyyy: 0=noerror, other is errors, see doc.
 PNR                        program version BGxxxxxx-x
 TDI                        display test
 TRA                        RAM test
 TEP                        EPROM test
 TEE                        EEPROM test
 TAI                        test ADC identification inputs
 TAS                        test ADC measurement    inputs
 TID                        sensor identification
                            answer = xxx,xxx,xxx,xxx,xxx,xxx
                            xxx: TPR       (pirani)
                                 IKR9      (cold cathode to 10E-9  mbar)
                                 IKR11     (cold cathode to 10E-11 mbar)
                                 PKR       (fullrange)
                                 AxR       (linear sensor)
                                 IMR       (hot cathode ionization)
                                 No Sensor (no sensor)
                                 No Ident  (no identification)
 *******************************************************************************
 }
type
 TBalzersTPG256 = class(TAdamDevice)
 protected
  StartList : TText;
  StartAns  : TText;
  StopList  : TText;
  StopAns   : TText;
  UsesRS485 : Boolean;
  errors    : Word;
  function    AckEnq(cmd:LongString):LongString;
  function    Send(cmd:LongString; var ans:LongString):Boolean;
  function    Route(s:LongString; mode:Boolean):LongString;
  function    GetRequest:LongString; override;
  procedure   HandleAnswer;  override;
  procedure   HandleTimeOut; override;
  procedure   HandleFormatError; override;
  procedure   InitRequestList; override;
 public
  constructor Create(const aName:LongString);
  destructor  Destroy; override;
  procedure   Config(FileName:LongString); override;
  function    GetProperty(P:TText):TText; override;
  function    Start:Boolean; override;
  procedure   Stop; override;
 end;

 {
 *******************************************************************************
 Объект:       Balzers TPG 252
               Класс устройств:     DualGauge - 2-канальный модуль для
                                    измерения и контроля вакуума
               Фирма-производитель: Balzers Instruments
               Литература:          803098BE.PDF
               Адрес в Интернет:    www.balzers.com
 Объявление:   Device Adam Balzers-TPG252
 Назначение:   2-канальный контроллер датчиков давления вакуума.
 Входы:        нет.
 Выходы:       AnalogOutput(0..1)  - измерительные каналы давления.
                Сюда записывается измеренное давление вакуума, при условии,
                что статус ошибки для данного канала равен 0.
               DigitalOutput(0..1) - статус ошибок в каналах давления.
                0 = измеренные данные в порядке
                1 = давление ниже диапазона
                2 = давление выше диапазона
                3 = ошибка сенсора
                4 = сенсор выключен
                5 = сенсор не подключен (строка 5,2.000E-2)
                6 = ошибка идентификации сенсора
               DigitalOutput(2..3) - состояние сенсора включен-выключен.
                0 = сенсор не изменил состояния или отключен
                1 = сенсор выключен
                2 = automatic (status)
                3 = сенсор включен
               DigitalOutput(4..5) - состояние реле
                0 = выключено
                1 = включено
 Описание:     Имеет 2 канала для подключения датчиков давления, 2 реле для
               управления по давлению от датчиков,а также жидкокристаллический
               дисплей для отображения параметров и измеряемых давлений.
               Связан с PC через RS232 или RS485.
 Конфигурация: При конфигурировании надо иметь в виду, что в зависимости от
               типа сенсоров некоторые значения параметров могут быть
               недопустимы. Подробности описаны в документах фирмы Balzers.
               StartCommand = XXXXX - посылка команды XXXXX при старте
                              формат команды в соответствии с 803098BE.PDF
               StopCommand  = XXXXX - посылка команды XXXXX при остановке
                              формат команды в соответствии с 803098BE.PDF
               UsesRS485 = 0/1 (см. описание Balzers-TPG256)
 Пример:       [DeviceList]
               TPG252    = device adam  Balzers-TPG252
               [TPG252]
               Comment = Balzers TPG252 MaxiGauges unit
               InquiryPeriod = 50
               Link AnalogOutput  0  with curve gauge1
               Link DigitalOutput 0  with curve senerr1
               Link DigitalOutput 6  with curve senon1
               Link DigitalOutput 12 with curve sensps1
               Port = 1
               Address = 1
               UsesCheckSum = false
               StartCommand = LOC,0
               StartCommand = UNI,0
               StopCommand = LOC,0
 **********************
 Список команд TPG-252: [,x] - необязательные параметры, (*) - (default)
 **********************
  Command                    Comment
  SEN [,x,x]                 Sensor 1,2 on/off
                              x: 0=NoChange/NoSensor
                                 1=Off
                                 2=Automatic(Status)
                                 3=Off
  PR1,PR2                     Read status and pressure of sensor 1, sensor 2
                              Answer is x,x.xxxEsxx
                              x: 0=Ok
                                 1=Underrange
                                 2=Overrange
                                 3=SensorError
                                 4=SensorOff
                                 5=NoSensor
                                 6=IdentificationError
                             x.xxxEsxx: pressure[mBar], always exp.format
  PRX                        Read both sensors
                             Answer is x,x.xxxEsxx,y,y.yyyEsyy
  SP1,SP2 [,x.xxEsx,y.yyEsy] Set point
                             x.xxEsxx - lower threshold [mbar]
                             y.yyEsy  - upper threshold [mbar]
  SPS                        Set point status for sensor 1,2.
                             Answer is x,x     x: 0=off, 1=on
  UNI[,x]                    Measurement unit x: 0=mbar(*),1=Torr,2=Pascal
  LOC[,x]                    Parameter setup lock 0=Off(*),1=On
  BAU[,x]                    Baudrate 0=300,1=1200,2=2400,3=4800,4=9600,5=19200
  DIC[,x]                    Display control 0=manual(*),1=automatic
  FIL[,x,x]                  Filter time constant 0=fast,1=normal(*),2=slow
  CAL[,x.xxx,x.xxx]          Calibration factor (1=*)
                             0.100..9.999 logarithmic sensors, 0.5..2 linear
  POC[,x,x]                  PE/Ioni control 0=auto(*),1=manual,2=external
  PUC[,x,x]                  PE underrange control 0=off(*),1=on
  FSR                        full scale range 0=1,1=10,2=100,3=1000[mbar],4=2,5=5,6=10,7=50[bar]
  OFC[,x,x]                  offset correction 0=off(*),1=on,2=automatic(offset measure)
  OFD[,x.xxxEsx,x.xxxEsx]    offset display default=0.000
  TID                        gauge identification, answer=x,x where x is
                             PIR,PE9,PE11,CO9,LIN,ION,noSe,noId
  SAV,x                      save params to EEPROM
                             0=save defaults,1=save user params
  ERR                        error status, answer=xxxx
                             0000=Ok
                             1000=error
                             0100=no hardware installed
                             0010-inadmissible parameter
                             0001-syntax error
  RES                        reset answer is x,x,x,x,x,...
                             0=no errors,else error list
  PNR                        program number (version), answer=BGxxxxxx-x
  DIS                        display test
  RAM                        ram test
  EPR                        EPROM test
  EEP                        EEPROM test
  ADC                        ADC test
  IOT                        IO test
  WDT[,x]                    watchdog control,0=automatic(*),1=manual
  RST                        RS232 test (echo, abortion with ^C)
 *******************************************************************************
 }
type
 TBalzersTPG252 = class(TBalzersTPG256)
 protected
  function    GetRequest:LongString; override;
  procedure   HandleAnswer; override;
  procedure   HandleTimeOut; override;
  procedure   HandleFormatError; override;
  procedure   InitRequestList; override;
 public
  constructor Create(const aName:LongString);
 end;

 {
 *******************************************************************************
 Объект:       Adam_Slot
 Объявление:   Device Adam Adam_Slot
 Назначение:   Для создания драйверов RS-485 на встроенном языке DAQ-PASCAL.
 Входы:        Нет.
 Выходы:       Нет.
 Описание:     Устройство Adam_Slot связано с программой на языке DAQ-PASCAL,
               которая реализует драйвер RS485. Роль Adam_Slot чисто пассивная
               - он обеспечивает простую передачу и прием данных в сети RS485
               и нужен для обеспечения бесконфликтной работы нескольких
               устройств в сети RS-485. Устройство program должно содержать
               ссылку Adam_Slot_Device = ... на устройство Adam_Slot,
               а программа драйвера может использовать функции:
                adam_status:Integer
                adam_get(name:string):string
                adam_request(request:string;timeout:Integer)
                adam_reqtime:real
               Эти функции описаны в устройстве program.
 Конфигурация: Особенностей нет.
 Пример:       [DeviceList]
               &Slot   = device adam     adam_slot
               &Dev485 = device software program
               [&Slot]
               Comment = Slot for RS-485 device &Dev485
               InquiryPeriod = 50
               Port = 1
               Address = 1
               UsesCheckSum = false
               [&Dev485]
               Comment = Device &Dev485 uses device &Slot
               InquiryPeriod = 50
               DebugMode = 1
               ProgramSource = dev485.pas
               Adam_Slot_Device = &Slot
 *******************************************************************************              
 }
type
 TAdamSlot = class(TAdamDevice)
 protected
  Fix      : packed record
   Request : LongString;
   Answer  : LongString;
   ReqTime : Double;
   Status  : Integer;
   TimeOut : Integer;
  end;
  function    GetRequest:LongString; override;
  procedure   HandleAnswer; override;
  function    GetTimeOut:Double; override;
  procedure   HandleTimeOut; override;
  procedure   InitRequestList; override;
  function    SetRequest(const aRequest:LongString; aTimeOut:Integer):Boolean;
 public
  constructor Create(const aName:LongString);
  destructor  Destroy; override;
  function    Start:Boolean; override;
  procedure   Stop; override;
 end;

function _Adam_Get(Slot:TAdamSlot; What:LongString):LongString;
function _Adam_Request(Slot:TAdamSlot; const aRequest:LongString; aTimeOut:Integer):Boolean;
function _Adam_Status(Slot:TAdamSlot):Integer;
function _Adam_ReqTime(Slot:TAdamSlot):Double;

 {
 Константы для функции Adam_Status:
 }
const
 as_NotAvail   = 0;  { Сеть RS-485 недоступна                            }
 as_NoRequest  = 1;  { Запрос не возбуждался после последней очистки     }
 as_WaitQueue  = 2;  { Запрос возбужден, но в сеть еще не передан        }
 as_WaitAnswer = 3;  { Запрос возбужден, передан в сеть, ожидание ответа }
 as_Answer     = 4;  { Пришел ответ на сообщение из сети RS-485          }
 as_TimeOut    = 5;  { TimeOut : ответ не пришел в назначеннное время    }

 {
 BaudRateCodes for Read/Write status for AdamXXXX
 }
const
 br1200   = 3;
 br2400   = 4;
 br4800   = 5;
 br9600   = 6;
 br19200  = 7;
 br38400  = 8;
 br57600  = 9;
 br115200 = 10;

 {
 Data Format codes for AdamXXXX
 }
const
 dfEngineering = 0;     // Use engineering units
 dfPercentFSR  = 1;     // Use % of Full scale range
 dfHexadecimal = 2;     // Use hexadecimal data
 dfOhms        = 3;     // Use ohms for RTD channels
 dfUnitMask    = 3;     // Mask for measurement units
 dfNoCheckSum  = 0;     // Don't use checksum
 dfCheckSum    = $40;   // Use checksum
 dfNormPoll    = 0;     // Use normal poll rate - 10 Hz
 dfFastPoll    = $20;   // Use fast   poll rate - 70 Hz - 7012F,7017F
 dfReject60    = 0;     // Use 60 Hz rejection
 dfReject50    = $80;   // Use 50 Hz rejection

 {
 *******************************************************************************
 DebugMode - отладочный режим
             бит 0 - полный протокол обмена
             бит 1 - протокол Timeout
             бит 2 - протокол ошибок формата
             бит 3 - время
 TimeOut   - 1: Время ожидания ответа на обычные команды
             2: Время ожидания после общих команды без ответа
             3: Время ожидания после специальных команд калибровки
 Attempts  - Число попыток чтения ImmediateAnswerRequest
 *******************************************************************************
 }
const
 AdamParam       : packed record
  DebugMode      : Integer;
  TimeOut        : packed array[1..3] of Integer;
  Attempts       : Integer;
  WatchDogPulser : Integer;
  InitError      : PureString;
  PollError      : PureString;
 end = (
  DebugMode      : 0;
  TimeOut        : (100, 10, 10*1000);
  Attempts       : 3;
  WatchDogPulser : 10000;
  InitError      : 'ошибка';
  PollError      : 'сбойсвязиадам'
 );

procedure Adam_Test;

procedure OpenAdamConsole;
procedure KillAdamConsole;

implementation

const CR=ASCII_CR; LF=ASCII_LF;

function IsValueIn(Value:Integer; const Wanted:TByteSet):Boolean;
var LoByte:Byte;
begin
 LoByte:=Value;
 Result:=(LoByte=Value) and (LoByte in Wanted);
end;

const
 TheAdamConsole : TFormConsoleWindow = nil;

procedure OpenAdamConsole;
begin
 try
  if TheAdamConsole.Ok then begin
   TheAdamConsole.Show;
   TheAdamConsole.WindowState:=wsNormal;
   TheAdamConsole.BringToFront;
  end else begin
   TheAdamConsole:=NewConsoleWindow('ADAM DEBUG CONSOLE', nil, true, nil,
                                    NewFifo(128*1024), true, nil);
   TheAdamConsole.Master:=@TheAdamConsole;
   TheAdamConsole.StartMonitoring;
  end;
 except
  on E:Exception do BugReport(E,nil,'OpenAdamConsole');
 end;
end;

procedure KillAdamConsole;
begin
 Kill(TheAdamConsole);
end;

procedure AdamDebugOut(const s:LongString);
begin
 if (AdamParam.DebugMode=0) then Exit;
 if TheAdamConsole.Ok
 then TheAdamConsole.PutText(s+EOL)
 else DebugOut(stdfDebug,s);
end;

function AdamConvertData(TypeCode:Byte; FormatCode:Byte; const Data:LongString):Double;
 function HexCode2Value(Code:Word; Range:Double):Double;
 begin
  Result:=Range*SmallInt(Code)/$7FFF;
 end;
 function AdcCode2Value(Code:Word; Range:Double):Double;
 begin
  Result:=Max(-Range,Range*SmallInt(Code)/$7FFF);
 end;
 procedure Adc2mV(FSR:Double);
 var d:Double; i:Integer;
 begin
  case FormatCode of
   dfEngineering:
    if (Length(Data)=7) and Str2Real(Data,d)
    then if (FSR<1000) then Result:=d else Result:=d*1000;
   dfPercentFSR:
    if (Length(Data)=7) and Str2Real(Data,d)
    then Result:=d/100*FSR;
   dfHexadecimal:
    if (Length(Data)=4) and StrHex2Long(Data,i)
    then Result:=AdcCode2Value(i,FSR);
  end;
 end;
 procedure Adc2Tc(NFSR,PFSR:Double);
 var d:Double; i:Integer;
 begin
  case FormatCode of
   dfEngineering:
    if IsSameText(Data,'+9999') then Result:=PFSR else
    if IsSameText(Data,'-0000') then Result:=NFSR else
    if IsSameText(Data,'+9999.9') then Result:=PFSR else
    if IsSameText(Data,'-9999.9') then Result:=NFSR else
    if (Length(Data)=7) and Str2Real(Data,d)
    then Result:=d;
   dfPercentFSR:
    if IsSameText(Data,'+9999') then Result:=PFSR else
    if IsSameText(Data,'-0000') then Result:=NFSR else
    if IsSameText(Data,'+999.99') then Result:=PFSR else
    if IsSameText(Data,'-999.99') then Result:=NFSR else
    if (Length(Data)=7) and Str2Real(Data,d)
    then Result:=d/100*max(abs(NFSR),abs(PFSR));
   dfHexadecimal:
    if IsSameText(Data,'7FFF') then Result:=PFSR else
    if IsSameText(Data,'8000') then Result:=NFSR else
    if (Length(Data)=4) and StrHex2Long(Data,i)
    then Result:=HexCode2Value(i,max(abs(NFSR),abs(PFSR)));
  end;
 end;
 procedure Adc2Rtd(NFSR,PFSR,NOHM,POHM:Double);
 var d:Double; i:Integer;
 begin
  case FormatCode of
   dfEngineering:
    if IsSameText(Data,'+9999') then Result:=PFSR else
    if IsSameText(Data,'-0000') then Result:=NFSR else
    if IsSameText(Data,'+999.99') then Result:=PFSR else
    if IsSameText(Data,'-999.99') then Result:=NFSR else
    if (Length(Data)=7) and Str2Real(Data,d)
    then Result:=d;
   dfPercentFSR:
    if IsSameText(Data,'+9999') then Result:=PFSR else
    if IsSameText(Data,'-0000') then Result:=NFSR else
    if IsSameText(Data,'+999.99') then Result:=PFSR else
    if IsSameText(Data,'-999.99') then Result:=NFSR else
    if (Length(Data)=7) and Str2Real(Data,d)
    then Result:=d/100*max(abs(NFSR),abs(PFSR));
   dfHexadecimal:
    if IsSameText(Data,'7FFF') then Result:=PFSR else
    if IsSameText(Data,'8000') then Result:=NFSR else
    if (Length(Data)=4) and StrHex2Long(Data,i)
    then Result:=HexCode2Value(i,max(abs(NFSR),abs(PFSR)));
   dfOhms:
    if IsSameText(Data,'+9999') then Result:=POHM else
    if IsSameText(Data,'-0000') then Result:=NOHM else
    if IsSameText(Data,'+999.99') then Result:=POHM else
    if IsSameText(Data,'-999.99') then Result:=NOHM else
    if IsValueIn(Length(Data),[7,9]) and Str2Real(Data,d)
    then Result:=d;
  end;
 end;
begin
 Result:=_NaN;
 FormatCode:=FormatCode and dfUnitMask;         // Extract code of units
 case TypeCode of
  // ADC mV/mA ranges
  $00: Adc2mV(15);                              //  -15..15  mV ADC input range
  $01: Adc2mV(50);                              //  -50..50  mV ADC input range
  $02: Adc2mV(100);                             // -100..100 mV ADC input range
  $03: Adc2mV(500);                             // -500..500 mV ADC input range
  $04: Adc2mV(1000);                            //   -1..1    V ADC input range
  $05: Adc2mV(2500);                            // -2.5..2.5  V ADC input range
  $06: Adc2mV(20);                              //  -20..20  mA ADC input range
  $08: Adc2mV(10000);                           //  -10..10   V ADC input range
  $09: Adc2mV(5000);                            //   -5..5    V ADC input range
  $0A: Adc2mV(1000);                            //   -1..1    V ADC input range
  $0B: Adc2mV(500);                             // -500..500 mV ADC input range
  $0C: Adc2mV(150);                             // -150..150 mV ADC input range
  $0D: Adc2mV(20);                              //  -20..20  mA ADC input range
  // ADC thermocouple ranges
  $0E: Adc2Tc(-210,760);                        // Type J, -210..760  C
  $0F: Adc2Tc(-270,1372);                       // Type K, -270..1372 C
  $10: Adc2Tc(-270,400);                        // Type T, -270..400  C
  $11: Adc2Tc(-270,1000);                       // Type E, -270..1000 C
  $12: Adc2Tc(   0,1768);                       // Type R,    0..1768 C
  $13: Adc2Tc(   0,1768);                       // Type S,    0..1768 C
  $14: Adc2Tc(   0,1820);                       // Type B,    0..1820 C
  $15: Adc2Tc(-270,1300);                       // Type N, -270..1300 C
  $16: Adc2Tc(   0,2320);                       // Type C,    0..2320 C
  $17: Adc2Tc(-200,800);                        // Type L, -200..800  C
  $18: Adc2Tc(-200,100);                        // Type M, -200..100  C
  $19: Adc2Tc(-200,900);                        // Type L,DIN43710, -200..900  C,
  // RTD ranges
  $20: Adc2Rtd(-100,100,  60.20,138.50);        // Pt100,  -100..100 C, a=0.003850, DIN43760
  $21: Adc2Rtd(   0,100, 100.00,138.50);        // Pt100,     0..100 C, a=0.003850, DIN43760
  $22: Adc2Rtd(   0,200, 100.00,175.84);        // Pt100,     0..200 C, a=0.003850, DIN43760
  $23: Adc2Rtd(   0,600, 100.00,313.65);        // Pt100,     0..600 C, a=0.003850, DIN43760
  $24: Adc2Rtd(-100,100,  59.58,139.16);        // Pt100,  -100..100 C, a=0.003916
  $25: Adc2Rtd(   0,100, 100.00,139.16);        // Pt100,     0..100 C, a=0.003916
  $26: Adc2Rtd(   0,200, 100.00,177.13);        // Pt100,     0..200 C, a=0.003916
  $27: Adc2Rtd(   0,600, 100.00,317.28);        // Pt100,     0..600 C, a=0.003916
  $28: Adc2Rtd( -80,100,  66.60,200.64);        // Ni120,   -80..100 C
  $29: Adc2Rtd(   0,100, 120.60,200.64);        // Ni120,     0..100 C
  $2A: Adc2Rtd(-200,600, 185.20,3137.1);        // Pt1000, -200..600 C, a=0.003850
  $2B: Adc2Rtd( -20,150,  91.56,163.17);        // Cu100,   -20..150 C, a=0.004210
  $2C: Adc2Rtd(   0,200,  90.34,167.75);        // Cu100,     0..200 C, a=0.004210
  $2D: Adc2Rtd( -20,150, 915.60,1631.7);        // Cu1000,  -20..150 C, a=0.004210
  $2E: Adc2Rtd(-200,200,  18.49,175.84);        // Pt100,  -200..200 C, a=0.003850
  $2F: Adc2Rtd(-200,200,  17.14,177.14);        // Pt100,  -200..200 C, a=0.003916
  $80: Adc2Rtd(-200,600,  18.49,313.59);        // Pt100,  -200..600 C, a=0.003850
  $81: Adc2Rtd(-200,600,  17.14,317.28);        // Pt100,  -200..600 C, a=0.003916
  // Thermistor ranges
  $60: Adc2Rtd( -30,240, 173600,539.4);         // -30..240 F, Precon -- ST-A3
  $61: Adc2Rtd( -50,150, 134020,37.2);          // -50..150 C, Type u Fenwell
  $62: Adc2Rtd(   0,150,   6530,37.2);          //   0..150 C, Type u Fenwell
 end;
end;

type
 TAdamAdcData = packed record
  AdcCount : Integer;
  BugCount : Integer;
  AdcValue : packed array[0..7] of Double;
 end;

function AdamConvert8(TypeCode:Byte; FormatCode:Byte; const Prefix:LongString; Data:LongString):TAdamAdcData;
const
 Space4 = '    '; Space7 = '       '; Space9 = '         ';
 Zero4  = '0000'; Zero7  = '+0.0000'; Zero9  = '+0.000000';
var
 i,m,PrefLen,DataLen:Integer;
 procedure ConvertItem(p,n:Integer);
 begin
  with Result do
  if (AdcCount>High(AdcValue)) then Inc(BugCount) else begin
   AdcValue[AdcCount]:=AdamConvertData(TypeCode,FormatCode,Copy(Data,p,n));
   if IsNan(AdcValue[AdcCount]) then Inc(BugCount);
   Inc(AdcCount);
  end;
 end;
begin
 Result.AdcCount:=0;
 Result.BugCount:=0;
 FormatCode:=FormatCode and dfUnitMask; // Extract code of units
 // 7015,7019:disabled channels filled with spaces
 // So we need to replace spaces to digital values
 // Thermistor 60..62 ohm ranges have 9 char width
 if (Pos(' ',Data)>0) then begin
  if (FormatCode=dfHexadecimal)
  then Data:=StringReplace(Data,Space4,Zero4,[rfReplaceAll]) else
  if (TypeCode in [$60..$62]) and (FormatCode=dfOhms)
  then Data:=StringReplace(Data,Space9,Zero9,[rfReplaceAll])
  else Data:=StringReplace(Data,Space7,Zero7,[rfReplaceAll]);
 end;
 DataLen:=Length(Data);
 PrefLen:=Length(Prefix);
 if (DataLen>PrefLen) then
 if IsSameText(Copy(Data,1,PrefLen),Prefix) then begin
  if (FormatCode=dfHexadecimal) then begin
   i:=PrefLen+1;
   while (i<=DataLen) do begin
    ConvertItem(i,4);
    Inc(i,4);
   end;
  end else begin
   m:=0;
   i:=PrefLen+1;
   while (i<=DataLen) do begin
    if (Data[i] in ['+','-']) then begin
     if (m>0) then ConvertItem(m,i-m);
     m:=i;
    end;
    Inc(i);
   end;
   if (m>0) then ConvertItem(m,i-m);
  end;
 end;
end;

procedure Adam_Test;
var
 Errors : Integer;
 procedure Test(T,F:Byte;const Data,Expected:LongString);
 var s:LongString; i:Integer;
 begin
  s:='';
  with AdamConvert8(T,F,'>',Data) do
  for i:=0 to AdcCount-1 do begin
   if (s<>'') then s:=s+',';
   s:=s+Format('%.5g',[AdcValue[i]]);
  end;
  if not IsSameText(s,Expected) then Inc(Errors);
  s:=Format('$%2.2x $%2.2x %-40s -> %s',[T,F,Data,s]);
  DebugOut(stdfDebug,s);
  s:='';
 end;
begin
 Errors:=0;
 DebugOut(stdfDebug,'');
 DebugOut(stdfDebug,'ADAM self testing.');
 DebugOut(stdfDebug,'******************');
 // $00
 Test($00,0,'>-15.000','-15');    Test($00,1,'>-100.00','-15');    Test($00,2,'>8000','-15');
 Test($00,0,'>+15.000','15');     Test($00,1,'>+100.00','15');     Test($00,2,'>7FFF','15');
 Test($00,0,'>+00.000','0');      Test($00,1,'>+000.00','0');      Test($00,2,'>0000','0');
 Test($00,0,'>-15.000+00.000+15.000','-15,0,15');
 Test($00,1,'>-100.00+000.00+100.00','-15,0,15');
 Test($00,2,'>800000007FFF','-15,0,15');
 // $01
 Test($01,0,'>-50.000','-50');    Test($01,1,'>-100.00','-50');    Test($01,2,'>8000','-50');
 Test($01,0,'>+50.000','50');     Test($01,1,'>+100.00','50');     Test($01,2,'>7FFF','50');
 Test($01,0,'>+00.000','0');      Test($01,1,'>+000.00','0');      Test($01,2,'>0000','0');
 Test($01,0,'>-50.000+00.000+50.000','-50,0,50');
 Test($01,1,'>-100.00+000.00+100.00','-50,0,50');
 Test($01,2,'>800000007FFF','-50,0,50');
 // $02
 Test($02,0,'>-100.00','-100');   Test($02,1,'>-100.00','-100');   Test($02,2,'>8000','-100');
 Test($02,0,'>+100.00','100');    Test($02,1,'>+100.00','100');    Test($02,2,'>7FFF','100');
 Test($02,0,'>+000.00','0');      Test($02,1,'>+000.00','0');      Test($02,2,'>0000','0');
 Test($02,0,'>-100.00+000.00+100.00','-100,0,100');
 Test($02,1,'>-100.00+000.00+100.00','-100,0,100');
 Test($02,2,'>800000007FFF','-100,0,100');
 // $03
 Test($03,0,'>-500.00','-500');   Test($03,1,'>-100.00','-500');   Test($03,2,'>8000','-500');
 Test($03,0,'>+500.00','500');    Test($03,1,'>+100.00','500');    Test($03,2,'>7FFF','500');
 Test($03,0,'>+000.00','0');      Test($03,1,'>+000.00','0');      Test($03,2,'>0000','0');
 Test($03,0,'>-500.00+000.00+500.00','-500,0,500');
 Test($03,1,'>-100.00+000.00+100.00','-500,0,500');
 Test($03,2,'>800000007FFF','-500,0,500');
 // $04
 Test($04,0,'>-1.0000','-1000');  Test($04,1,'>-100.00','-1000');  Test($04,2,'>8000','-1000');
 Test($04,0,'>+1.0000','1000');   Test($04,1,'>+100.00','1000');   Test($04,2,'>7FFF','1000');
 Test($04,0,'>+0.0000','0');      Test($04,1,'>+000.00','0');      Test($04,2,'>0000','0');
 Test($04,0,'>-1.0000+000.00+1.0000','-1000,0,1000');
 Test($04,1,'>-100.00+000.00+100.00','-1000,0,1000');
 Test($04,2,'>800000007FFF','-1000,0,1000');
 // $05
 Test($05,0,'>-2.5000','-2500');  Test($05,1,'>-100.00','-2500');  Test($05,2,'>8000','-2500');
 Test($05,0,'>+2.5000','2500');   Test($05,1,'>+100.00','2500');   Test($05,2,'>7FFF','2500');
 Test($05,0,'>+0.0000','0');      Test($05,1,'>+000.00','0');      Test($05,2,'>0000','0');
 Test($05,0,'>-2.5000+000.00+2.5000','-2500,0,2500');
 Test($05,1,'>-100.00+000.00+100.00','-2500,0,2500');
 Test($05,2,'>800000007FFF','-2500,0,2500');
 // $06
 Test($06,0,'>-20.000','-20');    Test($06,1,'>-100.00','-20');    Test($06,2,'>8000','-20');
 Test($06,0,'>+20.000','20');     Test($06,1,'>+100.00','20');     Test($06,2,'>7FFF','20');
 Test($06,0,'>+0.0000','0');      Test($06,1,'>+000.00','0');      Test($06,2,'>0000','0');
 Test($06,0,'>-20.000+000.00+20.000','-20,0,20');
 Test($06,1,'>-100.00+000.00+100.00','-20,0,20');
 Test($06,2,'>800000007FFF','-20,0,20');
 // $08
 Test($08,0,'>-10.000','-10000'); Test($08,1,'>-100.00','-10000'); Test($08,2,'>8000','-10000');
 Test($08,0,'>+10.000','10000');  Test($08,1,'>+100.00','10000');  Test($08,2,'>7FFF','10000');
 Test($08,0,'>+00.000','0');      Test($08,1,'>+000.00','0');      Test($08,2,'>0000','0');
 Test($08,0,'>-10.000+000.00+10.000','-10000,0,10000');
 Test($08,1,'>-100.00+000.00+100.00','-10000,0,10000');
 Test($08,2,'>800000007FFF','-10000,0,10000');
 // $09
 Test($09,0,'>-5.0000','-5000');  Test($09,1,'>-100.00','-5000');  Test($09,2,'>8000','-5000');
 Test($09,0,'>+5.0000','5000');   Test($09,1,'>+100.00','5000');   Test($09,2,'>7FFF','5000');
 Test($09,0,'>+0.0000','0');      Test($09,1,'>+000.00','0');      Test($09,2,'>0000','0');
 Test($09,0,'>-5.0000+000.00+5.0000','-5000,0,5000');
 Test($09,1,'>-100.00+000.00+100.00','-5000,0,5000');
 Test($09,2,'>800000007FFF','-5000,0,5000');
 // $0A
 Test($0A,0,'>-1.0000','-1000');  Test($0A,1,'>-100.00','-1000');  Test($0A,2,'>8000','-1000');
 Test($0A,0,'>+1.0000','1000');   Test($0A,1,'>+100.00','1000');   Test($0A,2,'>7FFF','1000');
 Test($0A,0,'>+0.0000','0');      Test($0A,1,'>+000.00','0');      Test($0A,2,'>0000','0');
 Test($0A,0,'>-1.0000+000.00+1.0000','-1000,0,1000');
 Test($0A,1,'>-100.00+000.00+100.00','-1000,0,1000');
 Test($0A,2,'>800000007FFF','-1000,0,1000');
 // $0B
 Test($0B,0,'>-500.00','-500');   Test($0B,1,'>-100.00','-500');   Test($0B,2,'>8000','-500');
 Test($0B,0,'>+500.00','500');    Test($0B,1,'>+100.00','500');    Test($0B,2,'>7FFF','500');
 Test($0B,0,'>+000.00','0');      Test($0B,1,'>+000.00','0');      Test($0B,2,'>0000','0');
 Test($0B,0,'>-500.00+000.00+500.00','-500,0,500');
 Test($0B,1,'>-100.00+000.00+100.00','-500,0,500');
 Test($0B,2,'>800000007FFF','-500,0,500');
 // $0C
 Test($0C,0,'>-150.00','-150');   Test($0C,1,'>-100.00','-150');   Test($0C,2,'>8000','-150');
 Test($0C,0,'>+150.00','150');    Test($0C,1,'>+100.00','150');    Test($0C,2,'>7FFF','150');
 Test($0C,0,'>+000.00','0');      Test($0C,1,'>+000.00','0');      Test($0C,2,'>0000','0');
 Test($0C,0,'>-150.00+000.00+150.00','-150,0,150');
 Test($0C,1,'>-100.00+000.00+100.00','-150,0,150');
 Test($0C,2,'>800000007FFF','-150,0,150');
 // $0D
 Test($0D,0,'>-20.000','-20');    Test($0D,1,'>-100.00','-20');    Test($0D,2,'>8000','-20');
 Test($0D,0,'>+20.000','20');     Test($0D,1,'>+100.00','20');     Test($0D,2,'>7FFF','20');
 Test($0D,0,'>+0.0000','0');      Test($0D,1,'>+000.00','0');      Test($0D,2,'>0000','0');
 Test($0D,0,'>-20.000+000.00+20.000','-20,0,20');
 Test($0D,1,'>-100.00+000.00+100.00','-20,0,20');
 Test($0D,2,'>800000007FFF','-20,0,20');
 // $0E
 Test($0E,0,'>-210.00+760.00-0000+9999-9999.9+9999.9','-210,760,-210,760,-210,760');
 Test($0E,1,'>-0000-999.99+000.00+100.00+999.99','-210,-210,0,760,760');
 Test($0E,2,'>800000007FFF','-210,0,760');
 // $0F
 Test($0F,0,'>-270.00+1372.0-0000+9999-9999.9+9999.9','-270,1372,-270,1372,-270,1372');
 Test($0F,1,'>-0000-999.99+000.00+100.00+999.99','-270,-270,0,1372,1372');
 Test($0F,2,'>800000007FFF','-270,0,1372');
 // $10
 Test($10,0,'>-270.00+400.00-0000+9999-9999.9+9999.9','-270,400,-270,400,-270,400');
 Test($10,1,'>-0000-999.99+000.00+100.00+999.99','-270,-270,0,400,400');
 Test($10,2,'>800000007FFF','-270,0,400');
 // $11
 Test($11,0,'>-270.00+1000.0-0000+9999-9999.9+9999.9','-270,1000,-270,1000,-270,1000');
 Test($11,1,'>-0000-999.99+000.00+100.00+999.99','-270,-270,0,1000,1000');
 Test($11,2,'>800000007FFF','-270,0,1000');
 // $12
 Test($12,0,'>-0000.0+1768.0-0000+9999-9999.9+9999.9','0,1768,0,1768,0,1768');
 Test($12,1,'>-0000-999.99+000.00+100.00+999.99','0,0,0,1768,1768');
 Test($12,2,'>800000007FFF','0,0,1768');
 // $13
 Test($13,0,'>+0000.0+1768.0-0000+9999-9999.9+9999.9','0,1768,0,1768,0,1768');
 Test($13,1,'>-0000-999.99+000.00+100.00+999.99','0,0,0,1768,1768');
 Test($13,2,'>800000007FFF','0,0,1768');
 // $14
 Test($14,0,'>-0000.0+1820.0-0000+9999-9999.9+9999.9','0,1820,0,1820,0,1820');
 Test($14,1,'>-0000-999.99+000.00+100.00+999.99','0,0,0,1820,1820');
 Test($14,2,'>800000007FFF','0,0,1820');
 // $15
 Test($15,0,'>-270.00+1300.0-0000+9999-9999.9+9999.9','-270,1300,-270,1300,-270,1300');
 Test($15,1,'>-0000-999.99+000.00+100.00+999.99','-270,-270,0,1300,1300');
 Test($15,2,'>800000007FFF','-270,0,1300');
 // $16
 Test($16,0,'>+0000.0+2320.0-0000+9999-9999.9+9999.9','0,2320,0,2320,0,2320');
 Test($16,1,'>-0000-999.99+000.00+100.00+999.99','0,0,0,2320,2320');
 Test($16,2,'>800000007FFF','0,0,2320');
 // $17
 Test($17,0,'>-200.00+800.00-0000+9999-9999.9+9999.9','-200,800,-200,800,-200,800');
 Test($17,1,'>-0000-999.99+000.00+100.00+999.99','-200,-200,0,800,800');
 Test($17,2,'>800000007FFF','-200,0,800');
 // $18
 Test($18,0,'>-200.00+100.00-0000+9999-9999.9+9999.9','-200,100,-200,100,-200,100');
 Test($18,1,'>-0000-999.99+000.00+100.00+999.99','-200,-200,0,200,100');
 Test($18,2,'>800000007FFF','-200,0,100');
 // $19
 Test($19,0,'>-200.00+900.00-0000+9999-9999.9+9999.9','-200,900,-200,900,-200,900');
 Test($19,1,'>-0000-999.99+000.00+100.00+999.99','-200,-200,0,900,900');
 Test($19,2,'>800000007FFF','-200,0,900');
 //
 DebugOut(stdfDebug,Format('Errors=%d',[Errors]));
end;

 {
 *******************************************************************************
 Конструктор создает устройства ADAM по модели
 *******************************************************************************
 }
function AdamDeviceConstructor(const Name,Family,Model,ConfigFile:LongString):TDaqDevice;
begin
 Result:=nil;
 if Daq.Ok then
 if IsSameText(Family,'ADAM') then begin
  if IsSameText(Model,'ADAM-4017')      then Result:=TAdamX017Device.Create(Name,4017);
  if IsSameText(Model,'ADAM-7017')      then Result:=TAdamX017Device.Create(Name,7017);
  if IsSameText(Model,'ADAM-4018')      then Result:=TAdamX017Device.Create(Name,4018);
  if IsSameText(Model,'ADAM-7018')      then Result:=TAdamX017Device.Create(Name,7018);
  if IsSameText(Model,'ADAM-7019')      then Result:=TAdamX017Device.Create(Name,7019);
  if IsSameText(Model,'ADAM-7033')      then Result:=TAdamX017Device.Create(Name,7033);
  if IsSameText(Model,'ADAM-7015')      then Result:=TAdamX017Device.Create(Name,7015);
  if IsSameText(Model,'ADAM-87013')     then Result:=TAdamX017Device.Create(Name,87013);
  if IsSameText(Model,'ADAM-87015')     then Result:=TAdamX017Device.Create(Name,87015);
  if IsSameText(Model,'ADAM-87017')     then Result:=TAdamX017Device.Create(Name,87017);
  if IsSameText(Model,'ADAM-87018')     then Result:=TAdamX017Device.Create(Name,87018);
  if IsSameText(Model,'ADAM-4050')      then Result:=TAdamDIOsDevice.Create(Name,4050);
  if IsSameText(Model,'ADAM-7050')      then Result:=TAdamDIOsDevice.Create(Name,7050);
  if IsSameText(Model,'ADAM-7051')      then Result:=TAdamDIOsDevice.Create(Name,7051);
  if IsSameText(Model,'ADAM-4052')      then Result:=TAdamDIOsDevice.Create(Name,4052);
  if IsSameText(Model,'ADAM-7052')      then Result:=TAdamDIOsDevice.Create(Name,7052);
  if IsSameText(Model,'ADAM-4053')      then Result:=TAdamDIOsDevice.Create(Name,4053);
  if IsSameText(Model,'ADAM-7053')      then Result:=TAdamDIOsDevice.Create(Name,7053);
  if IsSameText(Model,'ADAM-7058')      then Result:=TAdamDIOsDevice.Create(Name,7058);
  if IsSameText(Model,'ADAM-7059')      then Result:=TAdamDIOsDevice.Create(Name,7059);
  if IsSameText(Model,'ADAM-4060')      then Result:=TAdamDIOsDevice.Create(Name,4060);
  if IsSameText(Model,'ADAM-7060')      then Result:=TAdamDIOsDevice.Create(Name,7060);
  if IsSameText(Model,'ADAM-7063')      then Result:=TAdamDIOsDevice.Create(Name,7063);
  if IsSameText(Model,'ADAM-7065')      then Result:=TAdamDIOsDevice.Create(Name,7065);
  if IsSameText(Model,'ADAM-7066')      then Result:=TAdamDIOsDevice.Create(Name,7066);
  if IsSameText(Model,'ADAM-7067')      then Result:=TAdamDIOsDevice.Create(Name,7067);
  if IsSameText(Model,'ADAM-7041')      then Result:=TAdamDIOsDevice.Create(Name,7041);
  if IsSameText(Model,'ADAM-7042')      then Result:=TAdamDIOsDevice.Create(Name,7042);
  if IsSameText(Model,'ADAM-7043')      then Result:=TAdamDIOsDevice.Create(Name,7043);
  if IsSameText(Model,'ADAM-7044')      then Result:=TAdamDIOsDevice.Create(Name,7044);
  if IsSameText(Model,'ADAM-7045')      then Result:=TAdamDIOsDevice.Create(Name,7045);
  if IsSameText(Model,'ADAM-87051')     then Result:=TAdamDIOsDevice.Create(Name,87051);
  if IsSameText(Model,'ADAM-87052')     then Result:=TAdamDIOsDevice.Create(Name,87052);
  if IsSameText(Model,'ADAM-87053')     then Result:=TAdamDIOsDevice.Create(Name,87053);
  if IsSameText(Model,'ADAM-87054')     then Result:=TAdamDIOsDevice.Create(Name,87054);
  if IsSameText(Model,'ADAM-87055')     then Result:=TAdamDIOsDevice.Create(Name,87055);
  if IsSameText(Model,'ADAM-87057')     then Result:=TAdamDIOsDevice.Create(Name,87057);
  if IsSameText(Model,'ADAM-87058')     then Result:=TAdamDIOsDevice.Create(Name,87058);
  if IsSameText(Model,'ADAM-87063')     then Result:=TAdamDIOsDevice.Create(Name,87063);
  if IsSameText(Model,'ADAM-87064')     then Result:=TAdamDIOsDevice.Create(Name,87064);
  if IsSameText(Model,'ADAM-87065')     then Result:=TAdamDIOsDevice.Create(Name,87065);
  if IsSameText(Model,'ADAM-87066')     then Result:=TAdamDIOsDevice.Create(Name,87066);
  if IsSameText(Model,'ADAM-87068')     then Result:=TAdamDIOsDevice.Create(Name,87068);
  if IsSameText(Model,'ADAM-87069')     then Result:=TAdamDIOsDevice.Create(Name,87069);
  if IsSameText(Model,'ADAM-4021')      then Result:=TAdamX021Device.Create(Name,4021);
  if IsSameText(Model,'ADAM-7021')      then Result:=TAdamX021Device.Create(Name,7021);
  if IsSameText(Model,'ADAM-7024')      then Result:=TAdamX021Device.Create(Name,7024);
  if IsSameText(Model,'ADAM-87024')     then Result:=TAdamX021Device.Create(Name,87024);
  if IsSameText(Model,'ADAM-4011')      then Result:=TAdamX011Device.Create(Name,4011);
  if IsSameText(Model,'ADAM-7011')      then Result:=TAdamX011Device.Create(Name,7011);
  if IsSameText(Model,'ADAM-4011D')     then Result:=TAdamX011Device.Create(Name,4011);
  if IsSameText(Model,'ADAM-7011D')     then Result:=TAdamX011Device.Create(Name,7011);
  if IsSameText(Model,'ADAM-4012')      then Result:=TAdamX011Device.Create(Name,4012);
  if IsSameText(Model,'ADAM-7012')      then Result:=TAdamX011Device.Create(Name,7012);
  if IsSameText(Model,'ADAM-7012D')     then Result:=TAdamX011Device.Create(Name,7012);
  if IsSameText(Model,'ADAM-4013')      then Result:=TAdamX011Device.Create(Name,4013);
  if IsSameText(Model,'ADAM-7013')      then Result:=TAdamX011Device.Create(Name,7013);
  if IsSameText(Model,'ADAM-4014D')     then Result:=TAdamX011Device.Create(Name,4014);
  if IsSameText(Model,'ADAM-7014D')     then Result:=TAdamX011Device.Create(Name,7014);
  if IsSameText(Model,'ADAM-4080')      then Result:=TAdamX080Device.Create(Name,4080);
  if IsSameText(Model,'ADAM-7080')      then Result:=TAdamX080Device.Create(Name,7080);
  if IsSameText(Model,'RSDAQHOST')      then Result:=TRSDAQHostDevice.Create(Name);
  if IsSameText(Model,'BALZERS-TPG256') then Result:=TBalzersTPG256.Create(Name);
  if IsSameText(Model,'BALZERS-TPG252') then Result:=TBalzersTPG252.Create(Name);
  if IsSameText(Model,'ADAM_SLOT')      then Result:=TAdamSlot.Create(Name);
  if not Result.Ok
  then Daq.AddWarning('Error (invalid device) -> '+Name+' = device '+Family+' '+Model);
 end;
end;

 {
 *******************************************************************************
 Внутренние утилиты
 *******************************************************************************
 }
function d2s(n:Integer):LongString;
begin
 Result:=Format('%d',[n]);
end;

function f2s(f:Double):LongString;
begin
 Result:=Format('%g',[f]);
end;

var
 LastWatchDog : packed array[1..MaxPortNum] of Double;

procedure SetLastWatchDog(PortN:Integer; t:Double);
begin
 if (PortN>=Low(LastWatchDog)) and (PortN<=High(LastWatchDog))
 then LastWatchDog[PortN]:=t;
end;

function GetLastWatchDog(PortN:Integer):Double;
begin
 if (PortN>=Low(LastWatchDog)) and (PortN<=High(LastWatchDog))
 then Result:=LastWatchDog[PortN]
 else Result:=0;
end;

procedure DefaultAdamParam;
begin
 AdamParam.DebugMode:=0;
 AdamParam.TimeOut[1]:=100;
 AdamParam.TimeOut[2]:=10;
 AdamParam.TimeOut[3]:=10*1000;
 AdamParam.Attempts:=3;
 AdamParam.WatchDogPulser:=10000;
 SafeFillChar(LastWatchDog,sizeof(LastWatchDog),0);
end;

procedure ConfigAdamParam(ConfigFile,Section:LongString);
var i:Integer;
begin
 DefaultAdamParam;
 ConfigFile:=UnifyFileAlias(ConfigFile);
 ReadIniFileInteger(ConfigFile, Section, 'AdamDebugMode%i', AdamParam.DebugMode);
 ReadIniFileRecord(ConfigFile, Section, 'AdamTimeOut%i;%i;%i', AdamParam.TimeOut);
 ReadIniFileInteger(ConfigFile,Section,'AdamAttempts%i', AdamParam.Attempts);
 ReadIniFileString(SysIniFile,SectSysVoice,'AdamInitError%s', AdamParam.InitError);
 ReadIniFileString(SysIniFile,SectSysVoice,'AdamPollError%s', AdamParam.PollError);
 AdamParam.TimeOut[1]:=max(10,AdamParam.TimeOut[1]);
 AdamParam.TimeOut[2]:=max(10,AdamParam.TimeOut[2]);
 AdamParam.TimeOut[3]:=max(10,AdamParam.TimeOut[3]);
 AdamParam.Attempts:=max(1,AdamParam.Attempts);
 i:=0;
 if ReadIniFileInteger(ConfigFile, Section, 'AdamWatchDogPulser%i',i)
 or ReadIniFileInteger(SysIniFile, SectDaqSys, 'AdamWatchDogPulser%i',i)
 then AdamParam.WatchDogPulser:=Max(0,i);
end;

 {
 Здесь мы подменяем процедуры опроса модуля _UART
  1. Для облегчения переносимости
  2. Для вывода протокола обмена в файл в режиме отладки AdamDebugMode
 }
function OpenComPort(PortN:Integer; const ConfigFile,Section:LongString):Boolean;
begin
 Result:=uart[PortN].OpenConfig(ConfigFile,Section);
end;

procedure CloseComPort(PortN:Integer);
begin
 uart[PortN].Close;
end;

function UsesComPort(PortN:Integer):Boolean;
begin
 Result:=uart[PortN].Active;
end;

function RaiseRequest(PortN:Integer; const Request:LongString; TimeOut:Double):Boolean;
begin
 Result:=uart[PortN].RaiseRequest(Request,TimeOut);
end;

function DebugTime:LongString;
begin
 if HasFlags(AdamParam.DebugMode,8)
 then Result:=Format('%14.0f: ',[msecnow])
 else Result:='';
end;

function AnswerRequest(PortN:Integer; var Inquiry,Answer:LongString; var InquiryTime:Double):TRequestStatus;
begin
 Result:=uart[PortN].AnswerRequest(Inquiry,Answer,InquiryTime);
 if HasFlags(AdamParam.DebugMode,1) then
 case Result of
  rs_NoRequest  : ;
  rs_WaitAnswer : ;
  rs_TimeOut    : AdamDebugOut(DebugTime+'COM'+d2s(PortN)+': '+Pad(TrimChars(Inquiry,[CR,LF],[CR,LF]),20)+' = TimeOut');
  rs_Answer     : AdamDebugOut(DebugTime+'COM'+d2s(PortN)+': '+Pad(TrimChars(Inquiry,[CR,LF],[CR,LF]),20)+' = '+Answer);
  else            AdamDebugOut(DebugTime+'COM'+d2s(PortN)+': '+Pad('UNKNOWN ERROR',20)+' = '+d2s(ord(Result)));
 end;
end;

function  ModalAnswerRequest(PortN:Integer; const Inquiry:LongString; TimeOutMs:Double):LongString;
begin
 Result:=uart[PortN].ModalAnswerRequest(Inquiry,TimeoutMs);
 if HasFlags(AdamParam.DebugMode,1) then
 if (Result<>'')
 then AdamDebugOut(DebugTime+'COM'+d2s(PortN)+': '+Pad(TrimChars(Inquiry,[CR,LF],[CR,LF]),20)+' = '+Result)
 else AdamDebugOut(DebugTime+'COM'+d2s(PortN)+': '+Pad(TrimChars(Inquiry,[CR,LF],[CR,LF]),20)+' = TimeOut');
end;

function GetAdamTraffic(id:LongString; out traffic:TAdamTrafficRec):Boolean; overload;
var i,port,n:Integer; ti:TAdamTrafficRec;
begin
 id:=Trim(id);
 Result:=False;
 SafeFillChar(traffic,SizeOf(traffic),0);
 try
  if IsNonEmptyStr(id) then
  for i:=0 to AdamDeviceList.Count-1 do if AdamDeviceList[i].Ok then begin
   n:=0;
   if (n=0) and IsSameText(id,'*') then inc(n);
   if (n=0) and IsSameText(AdamDeviceList[i].Name,id) then inc(n);
   if (n=0) and IsSameText(Copy(id,1,3),'COM') and Str2Int(Copy(id,4,MaxInt),port) then
   if (port in [1..MaxPortNum]) and (port=AdamDeviceList[i].PortN) then inc(n);
   if (n>0) then begin
    ti:=AdamDeviceList[i].GetTraffic;
    traffic.Tx.Polls:=traffic.Tx.Polls+ti.Tx.Polls;
    traffic.Tx.Bytes:=traffic.Tx.Bytes+ti.Tx.Bytes;
    traffic.Rx.Polls:=traffic.Rx.Polls+ti.Rx.Polls;
    traffic.Rx.Bytes:=traffic.Rx.Bytes+ti.Rx.Bytes;
    Result:=True;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'GetAdamTraffic');
 end;
end;

function GetAdamTraffic(id:LongString):LongString; overload;
var traffic:TAdamTrafficRec;
begin
 if GetAdamTraffic(id,traffic)
 then with traffic do Result:=Format('%d %d %d %d',[Tx.Polls,Tx.Bytes,Rx.Polls,Rx.Bytes])
 else Result:='';
end;

 {
 *******************************************************************************
 TAdamDevice implementation
 *******************************************************************************
 }
constructor TAdamDevice.Create(const aName:LongString);
begin
 inherited Create(aName);
 SetDeviceFamily('ADAM');
 PortN:=0;
 Address:=0;
 UsesCheckSum:=false;
 Request:='';
 Answer:='';
 RequestTime:=0;
 RequestNumber:=0;
 RequestList:=[];
 InitRequestOrder;
end;

destructor TAdamDevice.Destroy;
begin
 Answer:='';
 Request:='';
 inherited Destroy;
end;

procedure TAdamDevice.Config(FileName:LongString);
begin
 FileName:=UnifyFileAlias(FileName);
 inherited Config(FileName);
 ReadIniFileInteger(FileName,DevSection,'Port%i',PortN);
 ReadIniFileInteger(FileName,DevSection,'Address%i',Address);
 ReadIniFileBoolean(FileName,DevSection,'UsesCheckSum%b',UsesCheckSum);
 InitRequestOrder;
end;

procedure TAdamDevice.Animate;
begin
 inherited Animate;
 InitRequestOrder;
end;

function  TAdamDevice.GetProperty(P:TText):TText;
begin
 GetProperty:=inherited GetProperty(P);
 P.AddLn('Port = '+d2s(PortN));
 P.AddLn('Address = $'+HexB(Address));
 P.AddLn('UsesCheckSum = '+d2s(ord(UsesCheckSum)));
end;

function TAdamDevice.Start:Boolean;
begin
 Result:=false;
 if inherited Start then begin
  InitRequestOrder;
  Result:=true;
 end;
end;

function TAdamDevice.ImmediateAnswerRequest(const TheRequest:LongString; Attempts:Integer):LongString;
var n:Integer;
begin
 Result:='';
 for n:=1 to Attempts do begin
  if UsesCheckSum
  then Result:=TruncCheckSumCR(ModalAnswerRequest(PortN,
                               CatCheckSumCR(TheRequest),AdamParam.TimeOut[1]))
  else Result:=ModalAnswerRequest(PortN,TheRequest,AdamParam.TimeOut[1]);
  if (Result<>'') then break;
 end;
end;

function  TAdamDevice.GetRequest:LongString;
begin
 Result:='';
end;

procedure TAdamDevice.HandleAnswer;
begin
end;

function  TAdamDevice.GetTimeOut:Double;
begin
 GetTimeOut:=AdamParam.TimeOut[1];
end;

procedure TAdamDevice.HandleTimeOut;
begin
 FixError(ecAdamTimeOut);
 if HasFlags(AdamParam.DebugMode,2)
 then AdamDebugOut(DebugTime+'COM'+d2s(PortN)+': TIMEOUT ON REQUEST '+TrimChars(Request,[CR,LF],[CR,LF]));
end;

procedure TAdamDevice.HandleFormatError;
begin
 FixError(ecAdamFormat);            
 if HasFlags(AdamParam.DebugMode,4)
 then AdamDebugOut(DebugTime+'COM'+d2s(PortN)+': FORMAT ERROR ON REQUEST "'+
               TrimChars(Request,[CR,LF],[CR,LF])+'" , ANSWER= "'+TrimChars(Answer,[CR,LF],[CR,LF])+'"');
end;

procedure TAdamDevice.PrepareNextRequest;
var
 n : Integer;
begin
 for n:=0 to 255 do begin
  inc(RequestNumber);
  if RequestNumber in RequestList then break;
 end;
end;

procedure TAdamDevice.InitRequestList;
begin
 RequestList:=[];
end;

procedure TAdamDevice.InitRequestOrder;
var
 n : Integer;
begin
 InitRequestList;
 RequestNumber:=0;
 for n:=0 to 255 do if n in RequestList then begin
  RequestNumber:=n;
  break;
 end;
end;

function TAdamDevice.HandleMessage(const aMsg:LongString; aFlags:Cardinal=hf_Default):Double;
var inq:LongInt; buf:TParsingBuffer;
begin
 Result:=inherited HandleMessage(aMsg,aFlags); inq:=0;
 if Assigned(ScanVarLongInt(svConfig,StrUpper(StrCopyBuff(buf,aMsg)),'InquiryPeriod%d',inq)) and (inq>=0) then begin
  InquiryTimer.IntervalMs[0]:=inq;
  Result:=Result+1;
 end;
end;

function TAdamDevice.GetTraffic:TAdamTrafficRec;
begin
 Result:=Traffic;
end;

procedure TAdamDevice.ClearTraffic;
begin
 SafeFillChar(Traffic,SizeOf(Traffic),0);
end;

 {
 *******************************************************************************
 TAdamDeviceList implementation
 *******************************************************************************
 }
procedure AdamListPollAction(aPolling:TPolling; var Terminate:Boolean);
begin
 if aPolling.LinkObject is TAdamDeviceList
 then (aPolling.LinkObject as TAdamDeviceList).MainLoop
 else Terminate:=true;
end;

function TAdamDeviceList.GetAdamDevices(Index:Integer):TAdamDevice;
begin
 Result:=TAdamDevice(Items[Index]);
end;

procedure TAdamDeviceList.SetAdamDevices(Index:Integer; aAdamDevice:TAdamDevice);
begin
 Items[Index]:=aAdamDevice;
end;

function TAdamDeviceList.GetPolling:TPolling;
begin
 if Assigned(Self) then Result:=myPolling else Result:=nil;
end;

function TAdamDeviceList.GetUsesPort(n:Integer):Boolean;
begin
 if (Self=nil) then begin Result:=false; Exit; end;
 EnterCriticalSection(myPortLatch);
 try
  Result:=(n in myUsesPorts);
 finally
  LeaveCriticalSection(myPortLatch);
 end;
end;

function TAdamDeviceList.GetUsesPorts:TByteSet;
begin
 if (Self=nil) then begin Result:=[]; Exit; end;
 EnterCriticalSection(myPortLatch);
 try
  Result:=myUsesPorts;
 finally
  LeaveCriticalSection(myPortLatch);
 end;
end;

procedure TAdamDeviceList.SetUsesPorts(const Ports:TByteSet);
var PortNum:Integer;
begin
 if (Self=nil) then Exit;
 EnterCriticalSection(myPortLatch);
 try
  myPollPorts:='';
  myUsesPorts:=Ports;
  for PortNum:=1 to MaxPortNum do
  if (PortNum in myUsesPorts) then myPollPorts:=myPollPorts+Chr(PortNum);
 finally
  LeaveCriticalSection(myPortLatch);
 end;
end;

function TAdamDeviceList.GetPollPorts:LongString;
begin
 Result:='';
 if (Self=nil) then Exit;
 EnterCriticalSection(myPortLatch);
 try
  Result:=myPollPorts;
 finally
  LeaveCriticalSection(myPortLatch);
 end;
end;

constructor TAdamDeviceList.Create(aOwns:Boolean);
begin
 inherited Create(aOwns);
 InitCriticalSection(myPortLatch);
 myUsesPorts:=[]; myPollPorts:='';
 myPolling:=NewPolling(AdamListPollAction,
                       DefaultDaqPollDelay,  DefaultDaqPollPriority,
                       false, 'Daq.Adams');
 myPolling.Master:=@myPolling;
 myPolling.LinkObject:=Self;
 SafeFillChar(mySender,sizeof(mySender),0);
 myLastErrors:=ErrorsCount[ecAdamTimeOut];
 myCheckTimer:=NewIntervalTimer(tmCyclic,NewIntervalMs(60000,1,nil));
 myCheckThresh:=0;
end;

destructor TAdamDeviceList.Destroy;
begin
 Kill(myPolling);
 Kill(myCheckTimer);
 CloseSerialPorts; myPollPorts:='';
 DoneCriticalSection(myPortLatch);
 inherited Destroy;
end;

function TAdamDeviceList.OpenSerialPorts(ConfigFile:LongString):Boolean;
var
 n         : Integer;
 p         : Integer;
 AdamPorts : TByteSet;
 BusyPorts : TByteSet;
 aDelay    : Integer;
 aPriority : TThreadPriority;
begin
 ConfigFile:=UnifyFileAlias(ConfigFile);
 CloseSerialPorts;
 Result:=true;
 BusyPorts:=[];
 for n:=1 to MaxPortNum do if UsesComPort(n) then include(BusyPorts,n);
 AdamPorts:=[];
 for n:=0 to Count-1 do if Self[n].Ok then Include(AdamPorts,Self[n].PortN);
 for n:=1 to MaxPortNum do
 if (n in AdamPorts*BusyPorts) then begin
  Daq.Voice(AdamParam.InitError);
  Daq.ConsoleEcho(RusEng('Порт '+d2s(n)+' занят, устройства ADAM недоступны!',
                         'Port '+d2s(n)+' busy, ADAM devices fails!'));
  AdamPorts:=[];
  Result:=false;
  break;
 end;
 for n:=1 to MaxPortNum do if (n in AdamPorts) then begin
  if not OpenComPort(n, ConfigFile, '[SerialPort-COM'+d2s(n)+']') then begin
   Daq.Voice(AdamParam.InitError);
   Daq.ConsoleEcho(RusEng('Не могу открыть порт '+d2s(n)+', устройства ADAM недоступны!',
                          'Could not open port '+d2s(n)+', ADAM devices fails!'));
   for p:=1 to MaxPortNum do
   if UsesComPort(p) and not (p in BusyPorts) then CloseComPort(p);
   AdamPorts:=[];
   Result:=false;
   break;
  end;
 end;
 UsesPorts:=AdamPorts;
 aDelay:=100; aPriority:=tpNormal;
 if ReadIniFilePolling(ConfigFile, SectDAQ,    'AdamPolling', aDelay, aPriority)
 or ReadIniFilePolling(SysIniFile, SectDaqSys, 'AdamPolling', aDelay, aPriority)
 then begin
  Polling.Delay:=aDelay;
  Polling.Priority:=aPriority;
 end;
end;

procedure TAdamDeviceList.CloseSerialPorts;
var
 n : Integer;
begin
 for n:=1 to MaxPortNum do if UsesPort[n] then CloseComPort(n);
 UsesPorts:=[];
end;

procedure TAdamDeviceList.InitSenders;
var
 n : Integer;
begin
 for n:=1 to MaxPortNum do begin
  mySender[n]:=nil;
  mySender[n]:=NextSender(n);
 end;
end;

function TAdamDeviceList.NextSender(n:Integer):TAdamDevice;
var
 i       : Integer;
 k       : Integer;
 aSender : TAdamDevice;
begin
 Result:=mySender[n];
 k:=IndexOf(mySender[n]);
 for i:=1 to Count do begin
  aSender:=Self[(k+i) mod Count];
  if aSender.Ok and (aSender.PortN=n) then begin
   Result:=aSender;
   break;
  end;
 end;
end;

function  TAdamDeviceList.Start:Boolean;
begin
 myLastErrors:=ErrorsCount[ecAdamTimeOut];
 myCheckTimer.Start;
 OpenSerialPorts(Daq.ConfigFile);
 InitSenders;
 CheckDevice;
 Start:=inherited Start;
 Polling.Enable(true,DefaultDaqTimeOut);
end;

procedure TAdamDeviceList.Stop;
begin
 Polling.Enable(false,DefaultDaqTimeOut);
 inherited Stop;
 myCheckTimer.Stop;
 CloseSerialPorts;
 InitSenders;
end;

procedure TAdamDeviceList.Idle;
begin
 inherited Idle;
 CheckConnection;
end;

procedure TAdamDeviceList.Poll;
begin
 // Poll do nothing, because AdamDeviceList has own Polling thread
end;

procedure TAdamDeviceList.MainLoop;
var i,aLoop,aIter,aPort:Integer; Raised:Boolean; TheTime:Double;
var Inquiry,PollStr:LongString;
const MaxIter=8;
begin
 if Ok and Daq.Ok then
 if (GetCurrentThreadID=Polling.ThreadID) then begin
  PollStr:=GetPollPorts;
  Inquiry:=''; TheTime:=0;
  for i:=1 to Length(PollStr) do begin
   aPort:=Ord(PollStr[i]);
   if (aPort<1) or (aPort>MaxPortNum) then continue;
   if UsesPort[aPort] and mySender[aPort].Ok then begin
    for aIter:=1 to MaxIter do begin
     aLoop:=0;
     with mySender[aPort] do
     if (aPort<>PortN) then FixError(ecAdamRequest) else
     case AnswerRequest(PortN,Inquiry,Answer,TheTime) of
      rs_NoRequest:
       begin
        Request:=GetRequest;
        if (Request='') then begin
         mySender[PortN]:=NextSender(PortN);
         Inc(aLoop);
        end else begin
         if UsesCheckSum
         then Raised:=RaiseRequest(PortN,CatCheckSumCR(Request),GetTimeOut)
         else Raised:=RaiseRequest(PortN,Request,GetTimeOut);
         if Raised
         then RequestTime:=Daq.Timer.LocalTime
         else FixError(ecAdamRequest);
         if Raised then begin
          LockedAdd(Traffic.Tx.Bytes,Length(Request)+2*Ord(UsesCheckSum));
          LockedAdd(Traffic.Tx.Polls,1);
         end;
        end;
       end;
      rs_WaitAnswer :;
      rs_TimeOut :
       begin
        HandleTimeOut;
        PrepareNextRequest;
        mySender[PortN]:=NextSender(PortN);
        Inc(aLoop);
       end;
      rs_Answer :
       begin
        if UsesCheckSum then begin
         if (Inquiry=CatCheckSumCR(Request)) then begin
          Answer:=TruncCheckSumCR(Answer);
          if (Answer<>'') then HandleAnswer else FixError(ecAdamCheckSum);
          if (Answer<>'') then begin
           LockedAdd(Traffic.Rx.Bytes,Length(Answer)+2*Ord(UsesCheckSum));
           LockedAdd(Traffic.Rx.Polls,1);
          end
         end else FixError(ecAdamRequest);
        end else begin
         if (Inquiry=Request) then HandleAnswer else FixError(ecAdamRequest);
         if (Inquiry=Request) then begin
          LockedAdd(Traffic.Rx.Bytes,Length(Answer)+2*Ord(UsesCheckSum));
          LockedAdd(Traffic.Rx.Polls,1);
         end
        end;
        PrepareNextRequest;
        mySender[PortN]:=NextSender(PortN);
        Inc(aLoop);
       end;
      else
       begin
        FixError(ecAdamRequest);
        PrepareNextRequest;
        mySender[PortN]:=NextSender(PortN);
       end;
     end;
     if (aLoop=0) then Break;
    end;
   end;
  end;
 end;
end;

procedure TAdamDeviceList.StartSession;
var
 Data : packed record
  f   : Double;
  t   : Double;
 end;
begin
 inherited StartSession;
 ConfigAdamParam(Daq.ConfigFile,SectDAQ);
 InitSenders; SafeFillChar(Data,SizeOf(Data),0);
 if ReadIniFileRecord(Daq.ConfigFile,SectDAQ,'AdamCheckLine%f;%f',Data) then begin
  myCheckThresh:=Data.f;
  myCheckTimer.IntervalMs[0]:=Data.t*60000;
 end else begin
  myCheckThresh:=0;
  myCheckTimer.IntervalMs[0]:=60000;
 end;
end;

procedure TAdamDeviceList.StopSession;
begin
 inherited StopSession;
 DefaultAdamParam;
 KillAdamConsole;
 InitSenders;
end;

procedure TAdamDeviceList.CheckConnection;
var
 f : Double;
begin
 if (myCheckThresh>0) and myCheckTimer.Event then begin
  f:=(ErrorsCount[ecAdamTimeOut]-myLastErrors)/myCheckTimer.IntervalMs[0]*1000;
  if (f>myCheckThresh) then Daq.Voice(AdamParam.PollError);
  myLastErrors:=ErrorsCount[ecAdamTimeOut];
 end;
end;

function TAdamDeviceList.CommonProperty(P:TText):TText;
begin
 Result:=P;
 if Ok then begin
  P.Addln(SectDAQ);
  P.Addln('AdamDebugMode = $'+HexL(AdamParam.DebugMode));
  P.Addln('AdamTimeOut = '+d2s(AdamParam.TimeOut[1])+', '+d2s(AdamParam.TimeOut[2])+', '+d2s(AdamParam.TimeOut[3]));
  P.Addln('AdamAttempts = '+d2s(AdamParam.Attempts));
  P.Addln('AdamCheckLine = '+f2s(myCheckThresh)+', '+f2s(myCheckTimer.IntervalMs[0]/60000));
  P.Addln(Format('AdamPolling = %d, %s',[Polling.Delay,GetPriorityName(Polling.Priority)]));
  uart.GetProperties(P);
 end;
end;

function AdamDeviceList:TAdamDeviceList;
const
 myAdamDeviceList : TAdamDeviceList = nil;
begin
 if not Assigned(myAdamDeviceList) then begin
  myAdamDeviceList:=TAdamDeviceList.Create(true);
  myAdamDeviceList.Master:=@myAdamDeviceList;
 end;
 Result:=myAdamDeviceList;
end;

 {
 *******************************************************************************
 TAdamXXXXDevice implementation
 *******************************************************************************
 }
constructor TAdamXXXXDevice.Create(const aName:LongString; aModelId:Integer);
var i:Integer;
begin
 inherited Create(aName);
 SetDeviceModel('ADAM-'+d2s(aModelId));
 ModelId:=aModelID;
 RangeCode:=$FFFF;
 RangeCodes:='';
 AvailRange:=[];
 DataFormat:=dfEngineering;
 MaskFormat:=dfUnitMask;
 ad_gate:=[0..255];
 ad_list:=[];
 ad_chan_no:=0;
 for i:=0 to 31 do ad_gain[i]:=1;
 for i:=0 to 31 do ad_zero[i]:=0;
 WatchDogSupports:=false;
 WatchDogPulser:=0;
 WatchDogTime:=0;
 InitRequestOrder;
 StartList:=NewText(0,16);
 StartAns:=NewText(0,16);
 StopList:=NewText(0,16);
 StopAns:=NewText(0,16);
end;

destructor TAdamXXXXDevice.Destroy;
begin
 Kill(StartList);
 Kill(StartAns);
 Kill(StopList);
 Kill(StopAns);
 inherited Destroy;
end;

procedure TAdamXXXXDevice.Config(FileName:LongString);
var i:Integer; f:Double; gate:Boolean; Section:TText;
 procedure ReadLine(const s:LongString);
 var  ss:LongString; buf:TParsingBuffer;
 begin
  ss:='';
  if (s<>'') then begin
   if (ScanVarString(svConfig,StrCopyBuff(buf,UpcaseStr(s)),'StartCommand%s',ss)<>nil)
   then StartList.Addln(TrimChars(StringReplace(ss,'??',HexB(Address),[]),ScanSpaces,ScanSpaces));
   if (ScanVarString(svConfig,StrCopyBuff(buf,UpcaseStr(s)),'StopCommand%s',ss)<>nil)
   then StopList.Addln(TrimChars(StringReplace(ss,'??',HexB(Address),[]),ScanSpaces,ScanSpaces));
  end;
 end;
begin
 FileName:=UnifyFileAlias(FileName);
 inherited Config(FileName);
 ReadIniFileWord(FileName,DevSection,'RangeCode%w',RangeCode);
 ReadIniFileWord(FileName,DevSection,'DataFormat%w',DataFormat);
 DataFormat:=DataFormat and dfUnitMask;
 ad_gate:=[0..255];
 for i:=0 to ad_chan_no-1 do begin
  ReadIniFileDouble(FileName,DevSection,'Gain#'+d2s(i)+'%f',ad_gain[i]);
  ReadIniFileDouble(FileName,DevSection,'Zero#'+d2s(i)+'%f',ad_zero[i]);
  gate:=true;
  ReadIniFileBoolean(FileName,DevSection,'Gate#'+d2s(i)+'%b',gate);
  if not gate then exclude(ad_gate,i);
 end;
 f:=0; i:=0;
 if WatchDogSupports then
 if ReadIniFileDouble(FileName,DevSection,'WatchDogTime%f',f)
 then WatchDogTime:=round(max(0,min(255,f*10)));
 if WatchDogSupports then
 if ReadIniFileInteger(FileName,DevSection,'WatchDogPulser%i',i)
 or ReadIniFileInteger(FileName,DevSection,'AdamWatchDogPulser%i',i)
 then WatchDogPulser:=max(0,i) else WatchDogPulser:=max(0,AdamParam.WatchDogPulser);
 InitRequestOrder;
 StartList.Count:=0;
 StartAns.Count:=0;
 StopList.Count:=0;
 StopAns.Count:=0;
 Section:=ExtractListSection(FileName,DevSection,efConfig);
 try
  for i:=0 to Section.Count-1 do ReadLine(Section[i]);
 finally
  Kill(Section);
 end;
end;

procedure TAdamXXXXDevice.Animate;
begin
 inherited Animate;
 InitRequestOrder;
 ad_gate:=ad_gate*ad_list;
end;

function  TAdamXXXXDevice.GetProperty(P:TText):TText;
var i:Integer; s:LongString;
begin
 GetProperty:=inherited GetProperty(P);
 if (RangeCode in AvailRange) then P.Addln('RangeCode = $'+hexb(RangeCode));
 P.Addln('DataFormat = $'+hexb(UnitFormat));
 if (RangeCodes<>'') then begin
  s:='RangeCodes = ';
  for i:=1 to Length(RangeCodes) do begin
   if (i=1) then s:=s+'$' else s:=s+', $';
   s:=s+hexb(Ord(RangeCodes[i]));
  end;
  P.Addln(s);
 end;
 for i:=0 to ad_chan_no-1 do begin
  P.Addln('Gain#'+d2s(i)+' = '+f2s(ad_gain[i])+', '+
          'Zero#'+d2s(i)+' = '+f2s(ad_zero[i])+', '+
          'Gate#'+d2s(i)+' = '+d2s(ord(i in ad_gate)));
 end;
 P.Addln('WatchDogTime = '+f2s(WatchDogTime/10)+' sec');
 P.Addln('WatchDogPulser = '+d2s(WatchDogPulser)+' ms');
 for i:=0 to StartList.Count-1 do begin
  P.Addln('StartCommand = '+StartList[i]);
  P.Addln('Answer       = '+StartAns[i]);
 end;
 for i:=0 to StopList.Count-1 do begin
  P.Addln('StopCommand  = '+StopList[i]);
  P.Addln('Answer       = '+StopAns[i]);
 end;
end;

function TAdamXXXXDevice.Start:Boolean;
begin
 Result:=false;
 if inherited Start then begin
  StartAns.Count:=0;
  StopAns.Count:=0;
  Result:=true;
 end;
end;

procedure TAdamXXXXDevice.Stop;
var i:Integer; Ask,Ans:LongString; When:Double;
const Tick=10;
begin
 Ask:=''; Ans:=''; When:=0;
 if InquiryTimer.IsStart then begin
  // Wait while last request finished and then apply StopCommand list
  for i:=1 to (AdamParam.TimeOut[1] div Tick) do
  if (AnswerRequest(PortN,Ask,Ans,When)=rs_WaitAnswer) then Sleep(Tick) else Break;
  if not DoStopCommands then FixError(ecAdamStop);
 end;
 inherited Stop;
end;

procedure TAdamXXXXDevice.InitRequestList;
var
 i : Integer;
begin
 inherited InitRequestList;
 ad_list:=[];
 for i:=0 to NumAnalogOutputs-1 do
 if AnalogOutputCurve[i].Ok then include(ad_list,i);
end;

 {
 Проверка наличия устройства - по отклику на команду чтения имени модуля
 Имя модуля не проверяется так как в устройствах серии 7000 имя переменное
 }
function TAdamXXXXDevice.CheckDevice:Boolean;
begin
 CheckDevice:=(ReadModuleName<>'');
end;

function TAdamXXXXDevice.ReadModuleName:LongString;
begin
 Result:=ImmediateAnswerRequest('$'+hexb(Address)+'M'+CR,AdamParam.Attempts);
 if (Copy(Result,1,3)='!'+hexb(Address))
 then Result:=Copy(Result,4,length(Result)-3)
 else Result:='';
end;

function TAdamXXXXDevice.ReadModuleFirmware:LongString;
begin
 Result:=ImmediateAnswerRequest('$'+hexb(Address)+'F'+CR,AdamParam.Attempts);
 if (Copy(Result,1,3)='!'+hexb(Address))
 then Result:=Copy(Result,4,length(Result)-3)
 else Result:='';
end;

function TAdamXXXXDevice.ReadStatus(out RangeTypeCode  : Byte;
                                    out BaudRateCode   : Byte;
                                    out DataFormatCode : Byte):Boolean;
var rtc,brc,dfc:LongInt; Status:LongString;
begin
 Status:=ImmediateAnswerRequest('$'+hexb(Address)+'2'+CR,AdamParam.Attempts);
 if (length(Status)=9) and
    (Copy(Status,1,3)='!'+hexb(Address)) and
    StrHex2Long(Copy(Status,4,2),rtc) and
    StrHex2Long(Copy(Status,6,2),brc) and
    StrHex2Long(Copy(Status,8,2),dfc)
 then begin
  Result:=true;
  RangeTypeCode:=rtc;
  BaudRateCode:=brc;
  DataFormatCode:=dfc;
 end else begin
  Result:=false;
  RangeTypeCode:=0;
  BaudRateCode:=0;
  DataFormatCode:=0;
 end;
end;

function TAdamXXXXDevice.WriteStatus(NewAddress     : Byte;
                                     RangeTypeCode  : Byte;
                                     BaudRateCode   : Byte;
                                     DataFormatCode : Byte):Boolean;
begin
 if (ImmediateAnswerRequest('%'+hexb(Address)+
                                hexb(NewAddress)+
                                hexb(RangeTypeCode)+
                                hexb(BaudRateCode)+
                                hexb(DataFormatCode)+CR,AdamParam.Attempts)
                            ='!'+hexb(NewAddress))
 then begin
  Result:=true;
  Address:=NewAddress;
 end else begin
  Result:=false;
 end;
end;

function TAdamXXXXDevice.GetUnitFormat:Word;
begin
 Result:=(DataFormat and dfUnitMask);
end;

procedure TAdamXXXXDevice.SetUnitFormat(value:Word);
begin
 DataFormat:=(DataFormat and not dfUnitMask) or (value and dfUnitMask);
end;

function TAdamXXXXDevice.GetFastPoll:Boolean;
begin
 Result:=((DataFormat and dfFastPoll)<>0);
end;

procedure TAdamXXXXDevice.SetFastPoll(value:Boolean);
begin
 if value
 then DataFormat:=(DataFormat or dfFastPoll)
 else DataFormat:=(DataFormat and not dfFastPoll);
end;

function TAdamXXXXDevice.GetReject50:Boolean;
begin
 Result:=((DataFormat and dfReject50)<>0);
end;

procedure TAdamXXXXDevice.SetReject50(value:Boolean);
begin
 if value
 then DataFormat:=(DataFormat or dfReject50)
 else DataFormat:=(DataFormat and not dfReject50);
end;

function TAdamXXXXDevice.HasMaskFormat(mask:Word):Boolean;
begin
 Result:=((MaskFormat and mask)<>0);
end;

procedure TAdamXXXXDevice.AddMaskFormat(mask:Word);
begin
 MaskFormat:=MaskFormat or mask;
end;

function TAdamXXXXDevice.CheckRangeCodes(n:Integer):Boolean;
var i,err:Integer; rc:Byte;
begin
 Result:=true; if (RangeCodes='') then exit; // No rangecodes uses
 Result:=false; err:=0;
 for i:=1 to n do begin
  rc:=RangeCode; if (i<=Length(RangeCodes)) then rc:=Ord(RangeCodes[i]);
  if (ImmediateAnswerRequest('$'+hexb(Address)+'8C'+d2s(i-1)+CR,AdamParam.Attempts)
                            ='!'+hexb(Address)+'C'+d2s(i-1)+'R'+hexb(rc))
  then continue; // Range code is already valid
  if (ImmediateAnswerRequest('$'+hexb(Address)+'7C'+d2s(i-1)+'R'+hexb(rc)+CR,AdamParam.Attempts)
                            ='!'+hexb(Address))
  then continue;
  inc(err);
 end;
 for i:=1 to n do begin
  rc:=RangeCode; if (i<=Length(RangeCodes)) then rc:=Ord(RangeCodes[i]);
  if (ImmediateAnswerRequest('$'+hexb(Address)+'8C'+d2s(i-1)+CR,AdamParam.Attempts)
                            ='!'+hexb(Address)+'C'+d2s(i-1)+'R'+hexb(rc))
  then continue; // Range code is valid
  inc(err);
 end;
 if (err<>0) then Exit;
 Result:=true;
end;

procedure TAdamXXXXDevice.ReadRangeCodes(FileName:LongString; n:Integer);
var i:Integer; codes:packed array[1..32] of word; fmt:LongString;
begin
 RangeCodes:='';
 if (n<1) then Exit;
 FileName:=UnifyFileAlias(FileName);
 if (n>High(codes)) then n:=High(codes);
 for i:=Low(codes) to High(codes) do codes[i]:=RangeCode;
 fmt:=''; for i:=1 to n do if (fmt='') then fmt:=fmt+'%w' else fmt:=fmt+';%w';
 ReadIniFileRecord(FileName,DevSection,'RangeCodes'+fmt,codes);
 for i:=1 to n do RangeCodes:=RangeCodes+Chr(codes[i]);
end;

function TAdamXXXXDevice.CheckRangeCode:Boolean;
var rtc,brc,dfc:Byte; Timer:TIntervalTimer;
begin
 Result:=false;
 if not ReadStatus(rtc,brc,dfc) then exit;
 if (RangeCode=$FFFF) then RangeCode:=rtc;
 Result:=true;
 if not (RangeCode in AvailRange) then exit;
 if (rtc=RangeCode) then exit;
 rtc:=RangeCode;
 if WriteStatus(Address,rtc,brc,dfc) then begin
  Timer:=NewIntervalTimer(tmStart+tmCyclic,NewIntervalMs(100,1,nil));
  while Timer.LocalTime<AdamParam.TimeOut[3] do
  if Timer.Event and ReadStatus(rtc,brc,dfc) then break else Sleep(TPolling.DefPollPeriod);
  Kill(Timer);
 end else Result:=false;
end;

function TAdamXXXXDevice.CheckDataFormat(Code,Mask:Byte):Boolean;
var
 rtc   : Byte;
 brc   : Byte;
 dfc   : Byte;
 Timer : TIntervalTimer;
begin
 Result:=true;
 if not ((Code and Mask) = Code) then exit;
 Result:=false;
 if not ReadStatus(rtc,brc,dfc) then exit;
 Result:=true;
 if ((dfc and Mask) = Code) then exit;
 dfc:=(dfc and not Mask) or (Code and Mask);
 if WriteStatus(Address,rtc,brc,dfc) then begin
  Timer:=NewIntervalTimer(tmStart+tmCyclic,NewIntervalMs(100,1,nil));
  while Timer.LocalTime<AdamParam.TimeOut[3] do
  if Timer.Event and ReadStatus(rtc,brc,dfc) then break else Sleep(TPolling.DefPollPeriod);
  Kill(Timer);
 end else Result:=false;
end;


function TAdamXXXXDevice.SetupWatchDog:Boolean;
begin
 Result:=true;
 if WatchDogSupports then begin
  {Watchdog enable/disable}
  if ImmediateAnswerRequest('~'+hexb(Address)+'2'+CR,AdamParam.Attempts)<>
              '!'+hexb(Address)+d2s(ord(WatchDogTime>0))+hexb(WatchDogTime)
  then
  if ImmediateAnswerRequest('~'+hexb(Address)+'3'+d2s(ord(WatchDogTime>0))+
                            hexb(WatchDogTime)+CR,AdamParam.Attempts)<>
                           '!'+hexb(Address)
  then SetupWatchDog:=false;
  {Reset module watchdog state}
  if ImmediateAnswerRequest('~'+hexb(Address)+'1'+CR,AdamParam.Attempts)<>
                            '!'+hexb(Address)
  then SetupWatchDog:=false;
  {сбросить время обновления WatchDog}
  SetLastWatchDog(PortN,0);
 end;
end;

procedure TAdamXXXXDevice.HandleWatchDogStatus;
var d:LongInt;
begin
 if (length(Answer)=5) and (Copy(Answer,1,3)='!'+hexb(Address)) and
    StrHex2Long(Copy(Answer,4,2),d)
 then begin
  if HasFlags(d,$04) then begin {watchdog failure detected, reset}
   FixError(ecAdamWatchDogFail);
   SetupWatchDog;
  end;
 end;
end;

procedure TAdamXXXXDevice.AwakeWatchDog(HostOkRequest:Integer);
var wdt,lim:Double; wp:Integer;
begin
 if WatchDogSupports and (WatchDogTime>0) then begin
  wdt:=Abs(msecnow-GetLastWatchDog(PortN));
  lim:=WatchDogTime*(0.25*0.1*1000);
  wp:=AdamParam.WatchDogPulser; if (wp>0) then lim:=Min(lim,wp);
  wp:=Self.WatchdogPulser;      if (wp>0) then lim:=Min(lim,wp);
  if (wdt>lim) then begin
   SetLastWatchDog(PortN,msecnow);
   include(RequestList,HostOkRequest);
  end;
 end;
end;

procedure TAdamXXXXDevice.SleepWatchDog(HostOkRequest:Integer);
begin
 if (RequestNumber=HostOkRequest) then exclude(RequestList,HostOkRequest);
end;

function TAdamXXXXDevice.ad_transform(chan:Word; data:Double):Double;
begin
 if (word(chan)>=word(ad_chan_no))
 then ad_transform:=data
 else ad_transform:=data/ad_gain[chan]-ad_zero[chan];
end;

function TAdamXXXXDevice.RangeFactor:Double;
begin
 case RangeCode of
  $04,$05,$08,$09,$0A,$32,$33,$34,$35:Result:=1000;
  else Result:=1;
 end;
end;

function TAdamXXXXDevice.NewPropertyDialog:TDaqDevicePropertyDialog;
begin
 Result:=nil;
end;

function TAdamXXXXDevice.UpdatePropertyDialog:Boolean;
begin
 Result:=false;
end;

function TAdamXXXXDevice.DoStartCommands:Boolean;
var
 i,errors : Integer;
 procedure SendCmd(const s:LongString);
 var ss:LongString;
 begin
  if (s<>'') then begin
   ss:=ImmediateAnswerRequest(s+CR,AdamParam.Attempts);
   inc(errors,ord(Length(ss)=0));
   StartAns.Addln(ss);
  end;
 end;
begin
 errors:=0;
 StartAns.Count:=0;
 for i:=0 to StartList.Count-1 do SendCmd(StartList[i]);
 if IsMainThread then UpdateFormDaqDeviceControl(1+2+4);
 Result:=(errors=0);
end;

function TAdamXXXXDevice.DoStopCommands:Boolean;
var
 i,errors : Integer;
 procedure SendCmd(const s:LongString);
 var ss:LongString;
 begin
  if (s<>'') then begin
   ss:=ImmediateAnswerRequest(s+CR,AdamParam.Attempts);
   inc(errors,ord(Length(ss)=0));
   StopAns.Addln(ss);
  end;
 end;
begin
 errors:=0;
 StopAns.Count:=0;
 for i:=0 to StopList.Count-1 do SendCmd(StopList[i]);
 if IsMainThread then UpdateFormDaqDeviceControl(1+2+4);
 Result:=(errors=0);
end;

procedure TAdamXXXXDevice.BugLog(var aErrors:Integer; const aMsg:LongString);
begin
 Inc(aErrors);
 if (AdamParam.DebugMode<>0)
 then AdamDebugOut(DebugTime+Name+' error: '+aMsg);
end;

 {
 *******************************************************************************
 Adam - 4017,4018,7017,7018,7019,7033,7015,87013,87015,87017,87018
 *******************************************************************************
 }
constructor TAdamX017Device.Create(const aName:LongString; aModelId:Integer);
begin
 inherited Create(aName,aModelId);
 AnalogFifoSize:=64;
 DigitalFifoSize:=0;
 NumAnalogInputs:=0;
 NumDigitalInputs:=0;
 NumAnalogOutputs:=0;
 NumDigitalOutputs:=0;
 NumCalibrations:=0;
 FastRead:=false;
 case ModelId of
  4017  : begin NumAnalogOutputs:=8; ad_chan_no:=8; AvailRange:=[$08..$0D]; AddMaskFormat(dfReject50 or dfFastPoll); end;
  4018  : begin NumAnalogOutputs:=9; ad_chan_no:=8; AvailRange:=[$00..$06,$0E..$14]; AddMaskFormat(dfReject50); end;
  7017  : begin NumAnalogOutputs:=8; ad_chan_no:=8; AvailRange:=[$08..$0D]; AddMaskFormat(dfReject50 or dfFastPoll); end;
  7018  : begin NumAnalogOutputs:=9; ad_chan_no:=8; AvailRange:=[$00..$06,$0E..$18]; AddMaskFormat(dfReject50); end;
  7019  : begin NumAnalogOutputs:=9; ad_chan_no:=8; AvailRange:=[$00..$06,$08..$0D,$0E..$19]; end;
  7033  : begin NumAnalogOutputs:=3; ad_chan_no:=3; AvailRange:=[$20..$2A,$2E,$2F,$80,$81]; AddMaskFormat(dfReject50); end;
  7015  : begin NumAnalogOutputs:=6; ad_chan_no:=6; AvailRange:=[$20..$2D,$2E,$2F,$80,$81]; end;
  87013 : begin NumAnalogOutputs:=4; ad_chan_no:=4; AvailRange:=[$20..$2A,$2E,$2F,$80,$81]; AddMaskFormat(dfReject50); end;
  87015 : begin NumAnalogOutputs:=7; ad_chan_no:=7; AvailRange:=[$20..$2D,$2E,$2F,$80,$81]; end;
  87017 : begin NumAnalogOutputs:=8; ad_chan_no:=8; AvailRange:=[$08..$0D]; AddMaskFormat(dfReject50 or dfFastPoll); end;
  87018 : begin NumAnalogOutputs:=9; ad_chan_no:=8; AvailRange:=[$00..$06,$0E..$19]; AddMaskFormat(dfReject50); end;
 end;
 WatchDogSupports:=IsValueIn(ModelId div 1000,[7,87]);
 InitRequestOrder;
end;

procedure TAdamX017Device.Config(FileName:LongString);
var flag:Boolean;
begin
 FileName:=UnifyFileAlias(FileName);
 inherited Config(FileName);
 ReadIniFileBoolean(FileName,DevSection,'FastRead%b',FastRead);
 if HasMaskFormat(dfFastPoll) then begin
  flag:=false;
  ReadIniFileBoolean(FileName,DevSection,'FastPoll%b',flag);
  if flag then FastPoll:=flag;
 end;
 if HasMaskFormat(dfReject50) then begin
  flag:=false;
  ReadIniFileBoolean(FileName,DevSection,'Reject50%b',flag);
  if flag then Reject50:=flag;
 end;
 if (ModelId=7019) then begin
  ReadRangeCodes(FileName,ad_chan_no);
  FastRead:=false; // Not supported
 end;
 if (ModelId=7015) then begin
  ReadRangeCodes(FileName,ad_chan_no);
  FastRead:=false; // Not supported
 end;
 InitRequestOrder;
end;

function  TAdamX017Device.GetProperty(P:TText):TText;
begin
 GetProperty:=inherited GetProperty(P);
 P.AddLn('FastRead = '+d2s(ord(FastRead)));
 if HasMaskFormat(dfFastPoll) then P.AddLn('FastPoll = '+d2s(ord(FastPoll)));
 if HasMaskFormat(dfReject50) then P.AddLn('Reject50 = '+d2s(ord(Reject50)));
end;

 { При старте разрешаем аппаратный опрос используемых каналов }
function TAdamX017Device.Start:Boolean;
var errors:Integer;
 function RequestListMap:Integer;
 var i:Integer;
 begin
  Result:=0;
  for i:=0 to 31 do if (i in ad_list*ad_gate) then Result:=Result or GetBitMask(i);
 end;
 function CheckMap:Boolean;{соответствует ли список опроса маске разрешений}
 begin
  CheckMap:=(ImmediateAnswerRequest('$'+hexb(Address)+'6'+CR,AdamParam.Attempts)
                                   ='!'+hexb(Address)+hexb(RequestListMap));
 end;
 function SetMap:Boolean;{установить маску разрешений}
 begin
  SetMap:=(ImmediateAnswerRequest('$'+hexb(Address)+'5'+
                                      hexb(RequestListMap)+CR,AdamParam.Attempts)
                                 ='!'+hexb(Address));
 end;
 function CheckRequestMap:Boolean;{проверить и установить маску разрешений}
 var Timer:TIntervalTimer;
 begin
  CheckRequestMap:=true;
  if (ModelId=7033) then Exit; // 7033 does not support $AA5,$AA6
  if not CheckMap then begin
   if SetMap then begin
    Timer:=NewIntervalTimer(tmStart+tmCyclic,NewIntervalMs(100,1,nil));
    while Timer.LocalTime<AdamParam.TimeOut[3] do
    if Timer.Event and CheckMap then break else Sleep(TPolling.DefPollPeriod);
    Kill(Timer);
   end else CheckRequestMap:=false;
  end;
 end;
begin
 Start:=false;
 if inherited Start and CheckDevice then begin
  errors:=0;
  if not CheckRangeCode then BugLog(errors,'CheckRangeCode');
  if (RangeCodes<>'') then if not CheckRangeCodes(ad_chan_no) then BugLog(errors,'CheckRangeCodes');
  if not CheckDataFormat(DataFormat and MaskFormat,MaskFormat) then BugLog(errors,'CheckDataFormat');
  if not CheckRequestMap then BugLog(errors,'CheckRequestMap');
  if not SetupWatchDog then BugLog(errors,'SetupWatchDog');
  if not DoStartCommands then BugLog(errors,'DoStartCommands');
  if (errors>0) then FixError(ecAdamStart);
  Start:=true;
 end else begin
  FixError(ecAdamStart);
 end;
end;

function  TAdamX017Device.GetRequest:LongString;
begin
 Result:='';
 if InquiryTimer.Event and (RequestNumber in RequestList) then begin
  AwakeWatchDog(9);
  case RequestNumber of
   0..7 : Result:='#'+hexb(Address)+d2s(RequestNumber)+CR; {AnalogIn}
   8    : Result:='$'+hexb(Address)+'3'+CR;                {CJC in}
   9    : Result:='~**'+CR;                                {WatchDog HostOK}
   10   : Result:='~'+hexb(Address)+'0'+CR;                {WatchDog Read status}
   11   : Result:='#'+hexb(Address)+CR;                    {AnalogIn-All}
  end;
  SleepWatchDog(9);
 end;
end;

procedure TAdamX017Device.HandleAnswer;
var
 f : Double;
 i : Integer;
 function CalcRangeCode:Byte;
 var i:Integer;
 begin
  Result:=RangeCode;
  if (RangeCodes='') then Exit;
  i:=RequestNumber+1; // This code specially for 7019
  if (i<=Length(RangeCodes)) then Result:=Ord(RangeCodes[i]);
 end;
begin
 case RequestNumber of
  0..7:
   with AdamConvert8(CalcRangeCode,UnitFormat,'>',Answer) do
   if (AdcCount=1) and (BugCount=0) then begin
    f:=ad_transform(RequestNumber, AdcValue[0]);
    PutDaqEvent(DaqEvent(evAnalog+evCompress, RequestNumber, RequestTime, f));
   end else HandleFormatError;
  8:
   if (length(Answer)=8) and (StrFetch(Answer,1)='>') and Str2Real(Copy(Answer,2,7),f)
   then PutDaqEvent(DaqEvent(evAnalog+evCompress, RequestNumber, RequestTime, f))
   else HandleFormatError;
  9  : ;
  10 : HandleWatchDogStatus;
  11:
   with AdamConvert8(RangeCode,UnitFormat,'>',Answer) do
   if (AdcCount=ad_chan_no) and (BugCount=0) then begin
    for i:=0 to AdcCount-1 do
    if (i in ad_list*ad_gate) and (AnalogOutputCurve[i]<>nil) then begin
     f:=ad_transform(i, AdcValue[i]);
     PutDaqEvent(DaqEvent(evAnalog+evCompress, i, RequestTime, f));
    end;
   end else HandleFormatError;
 end;
end;

procedure TAdamX017Device.InitRequestList;
var n:Integer;
begin
 inherited InitRequestList;
 for n:=0 to NumAnalogOutputs-1 do
 if (n in ad_list*ad_gate) then
 if (AnalogOutputCurve[n]<>nil) then include(RequestList,n);
 if FastRead then begin
  if (RequestList*[0..ad_chan_no-1]<>[]) then include(RequestList,11);
  RequestList:=RequestList-[0..ad_chan_no-1];
 end;
 if WatchDogSupports and (WatchDogTime>0) then RequestList:=RequestList+[9,10];
end;

function  TAdamX017Device.GetTimeOut:Double;
begin
 case RequestNumber of
  9 : GetTimeOut:=AdamParam.TimeOut[2];
  else GetTimeOut:=inherited GetTimeOut;
 end;
end;

procedure TAdamX017Device.HandleTimeOut;
begin
 case RequestNumber of
  9 : ; {No responce on HostOk command}
  else inherited HandleTimeOut;
 end;
end;

procedure TAdamX017Device.AdvancedPropertyDialog;
begin
end;

 {
 *******************************************************************************
 Adam - 4050,7050,7051,4052,7052,4053,7053,7058,7059,4060,7060,7063,7065,7066,7067,7041,7042
        7043,7044,7045,87051,87052,87053,87054,87055,87057,87063,87064,87065,87066
        87068,87069,87058
 *******************************************************************************
 }
constructor TAdamDIOsDevice.Create(const aName:LongString; aModelId:Integer);
 procedure Num_DOs_DIs(ndos,ndis:integer);
 begin
  NumDigitalInputs:=ndos;
  NumDigitalOutputs:=ord(ndis>0); {all data as bits of single numeral}
 end;
begin
 inherited Create(AName,aModelId);
 AnalogFifoSize:=0;
 DigitalFifoSize:=64;
 NumAnalogInputs:=0;
 NumDigitalInputs:=0;
 NumAnalogOutputs:=0;
 NumDigitalOutputs:=0;
 NumCalibrations:=0;
 case ModelId of
  4050  : Num_DOs_DIs( 8,  7  );
  4052  : Num_DOs_DIs( 0,  8  );
  4053  : Num_DOs_DIs( 0,  16 );
  4060  : Num_DOs_DIs( 4,  4  );
  7041  : Num_DOs_DIs( 0,  14 );
  7042  : Num_DOs_DIs( 13, 0  );
  7043  : Num_DOs_DIs( 16, 0  );
  7044  : Num_DOs_DIs( 8,  4  );
  7045  : Num_DOs_DIs( 16, 0  );
  7050  : Num_DOs_DIs( 8,  7  );
  7051  : Num_DOs_DIs( 0,  16 );
  7052  : Num_DOs_DIs( 0,  8  );
  7053  : Num_DOs_DIs( 0,  16 );
  7058  : Num_DOs_DIs( 0,  8  );
  7059  : Num_DOs_DIs( 0,  8  );
  7060  : Num_DOs_DIs( 4,  4  );
  7063  : Num_DOs_DIs( 3,  8  );
  7065  : Num_DOs_DIs( 5,  4  );
  7066  : Num_DOs_DIs( 7,  0  );
  7067  : Num_DOs_DIs( 7,  0  );
  87051 : Num_DOs_DIs( 0,  16 );
  87052 : Num_DOs_DIs( 0,  8  );
  87053 : Num_DOs_DIs( 0,  16 );
  87054 : Num_DOs_DIs( 8,  8  );
  87055 : Num_DOs_DIs( 8,  8  );
  87057 : Num_DOs_DIs( 16, 0  );
  87058 : Num_DOs_DIs( 0,  8  );
  87063 : Num_DOs_DIs( 4,  4  );
  87064 : Num_DOs_DIs( 8,  0  );
  87065 : Num_DOs_DIs( 8,  0  );
  87066 : Num_DOs_DIs( 8,  0  );
  87068 : Num_DOs_DIs( 8,  0  );
  87069 : Num_DOs_DIs( 8,  0  );
 end;
 AvailRange:=[$40];
 RangeCode:=$40;
 WatchDogSupports:=IsValueIn(ModelId div 1000,[7,87]);
 SafeValue:=0;
 PowerOnValue:=0;
 InitRequestOrder;
end;

procedure TAdamDIOsDevice.Config(FileName:LongString);
begin
 FileName:=UnifyFileAlias(FileName);
 inherited Config(FileName);
 ReadIniFileWord(FileName,DevSection,'SafeValue%w',SafeValue);
 ReadIniFileWord(FileName,DevSection,'PowerOnValue%w',PowerOnValue);
end;

function  TAdamDIOsDevice.GetProperty(P:TText):TText;
begin
 GetProperty:=inherited GetProperty(P);
 P.Addln('SafeValue = $'+hexw(SafeValue));
 P.Addln('PowerOnValue = $'+hexw(PowerOnValue));
end;

function TAdamDIOsDevice.ImmediateOut(w:Word):Boolean;
  procedure Out1;
  begin
   if ImmediateAnswerRequest('#'+hexb(Address)+'00'+hexb(lo(w))+CR,
        AdamParam.Attempts)<>'>'
   then ImmediateOut:=false;
  end;
  procedure Out2;
  begin
   if ImmediateAnswerRequest('#'+hexb(Address)+'0A'+hexb(lo(w))+CR,
        AdamParam.Attempts)<>'>'
   then ImmediateOut:=false;
   if ImmediateAnswerRequest('#'+hexb(Address)+'0B'+hexb(hi(w))+CR,
        AdamParam.Attempts)<>'>'
   then ImmediateOut:=false;
  end;
begin
 ImmediateOut:=true;
 case ModelId of
  4050  : Out1;
  4052  : exit;
  4053  : exit;
  4060  : Out1;
  7041  : exit;
  7042  : Out2;
  7043  : Out2;
  7044  : Out1;
  7045  : Out2;
  7050  : Out1;
  7051  : exit;
  7052  : exit;
  7053  : exit;
  7058  : exit;
  7059  : exit;
  7060  : Out1;
  7063  : Out1;
  7065  : Out1;
  7066  : Out1;
  7067  : Out1;
  87051 : exit;
  87052 : exit;
  87053 : exit;
  87054 : Out1;
  87055 : Out1;
  87057 : Out2;
  87058 : exit;
  87063 : Out1;
  87064 : Out1;
  87065 : Out1;
  87066 : Out1;
  87068 : Out1;
  87069 : Out1;
 end;
end;

function TAdamDIOsDevice.Start:Boolean;
var errors:Integer;
begin
 Start:=false;
 if inherited Start and CheckDevice then begin
  errors:=0;
  if WatchDogSupports then begin
   if not SetupWatchDog then BugLog(errors,'SetupWatchDog');
   {Set power on value,if need}
   if ImmediateAnswerRequest('~'+hexb(Address)+'4P'+CR,AdamParam.Attempts)<>
                             '!'+hexb(Address)+hexw(PowerOnValue)
   then begin
    if ImmediateOut(PowerOnValue)=false then BugLog(errors,'PowerOnValue') else
    if ImmediateAnswerRequest('~'+hexb(Address)+'5P'+CR,AdamParam.Attempts)<>
                              '!'+hexb(Address)
    then BugLog(errors,'SetPowerOnValue');
   end;
   {Set safe value,if need}
   if ImmediateAnswerRequest('~'+hexb(Address)+'4S'+CR,AdamParam.Attempts)<>
                             '!'+hexb(Address)+hexw(SafeValue)
   then begin
    if ImmediateOut(SafeValue)=false then BugLog(errors,'SafeValue') else
    if ImmediateAnswerRequest('~'+hexb(Address)+'5S'+CR,AdamParam.Attempts)<>
                              '!'+hexb(Address)
    then BugLog(errors,'SetSafeValue');
   end;
  end;
  if not DoStartCommands then BugLog(errors,'DoStartCommands');
  if (errors>0) then FixError(ecAdamStart);
  Start:=true;
 end else begin
  FixError(ecAdamStart);
 end;
end;

 {
 Запросы:
 0 - цифровой ввод
 1 - цифровой вывод
 2 - цифровой вывод - порт A
 3 - цифровой вывод - порт B
 4 - Host Ok
 5 - Read watchdog status
 }
function  TAdamDIOsDevice.GetRequest:LongString;
begin
 Result:='';
 if InquiryTimer.Event and (RequestNumber in RequestList) then begin
  AwakeWatchDog(4);
  case RequestNumber of
   0 : Result:='$'+hexb(Address)+'6'+CR;
   1 : Result:='#'+hexb(Address)+'00'+hexb(ConstructDigitalInputWord(0,8))+CR;
   2 : Result:='#'+hexb(Address)+'0A'+hexb(ConstructDigitalInputWord(0,8))+CR;
   3 : Result:='#'+hexb(Address)+'0B'+hexb(ConstructDigitalInputWord(8,8))+CR;
   4 : Result:='~**'+CR;                  {WatchDog HostOK     }
   5 : Result:='~'+hexb(Address)+'0'+CR;  {WatchDog Read status}
  end;
  SleepWatchDog(4);
 end;
end;

procedure TAdamDIOsDevice.HandleAnswer;
var
 d1 : LongInt;
 d2 : LongInt;
begin
 case RequestNumber of
  0    : if (length(Answer)=7) and
            (StrFetch(Answer,1)='!') and
            StrHex2Long(Copy(Answer,2,2),d1) and
            StrHex2Long(Copy(Answer,4,2),d2) and
            (Copy(Answer,6,2)='00')
         then PackData(d1,d2)
         else HandleFormatError;
  1..3 : if (Answer<>'>') then HandleFormatError;
  4    : ;
  5    : HandleWatchDogStatus;
 end;
end;

procedure TAdamDIOsDevice.PackData(data1,data2:Word);
var
 data : Word;
begin
 case ModelId of
  4050  : data:=data2;
  4052  : data:=data1;
  4053  : data:=data1 shl 8 + data2;
  4060  : data:=data2;
  7041  : data:=data1 shl 8 + data2;
  7042  : exit;
  7043  : exit;
  7044  : data:=data2;
  7045  : exit;
  7050  : data:=data2;
  7051  : data:=data1 shl 8 + data2;
  7052  : data:=data1;
  7053  : data:=data1 shl 8 + data2;
  7058  : data:=data1;
  7059  : data:=data1;
  7060  : data:=data2;
  7063  : data:=data2;
  7065  : data:=data2;
  7066  : exit;
  7067  : exit;
  87051 : data:=data1 shl 8 + data2;
  87052 : data:=data1;
  87053 : data:=data1 shl 8 + data2;
  87054 : data:=data2;
  87055 : data:=data2;
  87057 : exit;
  87058 : data:=data1;
  87063 : data:=data2;
  87064 : exit;
  87065 : exit;
  87066 : exit;
  87068 : exit;
  87069 : exit;
  else exit;
 end;
 PutDaqEvent(DaqEvent(evDigital, 0, RequestTime, data));
end;

procedure TAdamDIOsDevice.InitRequestList;
var
 n : Integer;
begin
 inherited InitRequestList;
 if (DigitalOutputCurve[0]<>nil) then include(RequestList,0);
 for n:=0 to NumDigitalInputs-1 do
 if (DigitalInputCurve[n]<>nil) then begin
  case ModelId of
   4050  : include(RequestList,1);
   4052  : ;
   4053  : ;
   4060  : include(RequestList,1);
   7041  : ;
   7042  : begin include(RequestList,2); include(RequestList,3); end;
   7043  : begin include(RequestList,2); include(RequestList,3); end;
   7044  : include(RequestList,1);
   7045  : begin include(RequestList,2); include(RequestList,3); end;
   7050  : include(RequestList,1);
   7051  : ;
   7052  : ;
   7053  : ;
   7058  : ;
   7059  : ;
   7060  : include(RequestList,1);
   7063  : include(RequestList,1);
   7065  : include(RequestList,1);
   7066  : include(RequestList,1);
   7067  : include(RequestList,1);
   87051 : ;
   87052 : ;
   87053 : ;
   87054 : include(RequestList,1);
   87055 : include(RequestList,1);
   87057 : begin include(RequestList,2); include(RequestList,3); end;
   87058 : ;
   87063 : include(RequestList,1);
   87064 : include(RequestList,1);
   87065 : include(RequestList,1);
   87066 : include(RequestList,1);
   87068 : include(RequestList,1);
   87069 : include(RequestList,1);
  end;
 end;
 if WatchDogSupports and (WatchDogTime>0) then RequestList:=RequestList+[4,5];
end;

function  TAdamDIOsDevice.GetTimeOut:Double;
begin
 case RequestNumber of
  4 : GetTimeOut:=AdamParam.TimeOut[2];
  else GetTimeOut:=inherited GetTimeOut;
 end;
end;

procedure TAdamDIOsDevice.HandleTimeOut;
begin
 case RequestNumber of
  4 : ; {No responce on HostOk command}
  else inherited HandleTimeOut;
 end;
end;

 {
 *******************************************************************************
 Adam 4021, 7021, 7024, 87024
 *******************************************************************************
 }
constructor TAdamX021Device.Create(const aName:LongString; aModelId:Integer);
begin
 inherited Create(aName,aModelId);
 AnalogFifoSize:=0;
 DigitalFifoSize:=0;
 NumAnalogInputs:=0;
 NumDigitalInputs:=0;
 NumAnalogOutputs:=0;
 NumDigitalOutputs:=0;
 NumCalibrations:=0;
 case ModelId of
  4021  : begin NumAnalogInputs:=1; AvailRange:=[$30..$32]; end;
  7021  : begin NumAnalogInputs:=1; AvailRange:=[$30..$32]; end;
  7024  : begin NumAnalogInputs:=4; AvailRange:=[$30..$35]; end;
  87024 : begin NumAnalogInputs:=4; AvailRange:=[$30..$35]; end;
 end;
 WatchDogSupports:=IsValueIn(ModelId div 1000,[7,87]);
 SafeValue[0]:=0;
 SafeValue[1]:=0;
 SafeValue[2]:=0;
 SafeValue[3]:=0;
 PowerOnValue[0]:=0;
 PowerOnValue[1]:=0;
 PowerOnValue[2]:=0;
 PowerOnValue[3]:=0;
 InitRequestOrder;
 Enable16Bit:=-1;
end;

procedure TAdamX021Device.Config(FileName:LongString);
begin
 FileName:=UnifyFileAlias(FileName);
 inherited Config(FileName);
 case ModelId of
  4021,7021:
   begin
    ReadIniFileDouble(FileName, DevSection, 'SafeValue%f', SafeValue[0]);
    ReadIniFileDouble(FileName, DevSection, 'PowerOnValue%f', PowerOnValue[0]);
    if (ModelId=7021) then
    ReadIniFileLongInt(FileName, DevSection, 'Enable16Bit%d', Enable16Bit);
   end;
  7024,87024:
   begin
    ReadIniFileRecord(FileName, DevSection, 'SafeValue%f;%f;%f;%f', SafeValue);
    ReadIniFileRecord(FileName, DevSection, 'PowerOnValue%f;%f;%f;%f', PowerOnValue);
   end;
 end;
end;

function  TAdamX021Device.GetProperty(P:TText):TText;
begin
 GetProperty:=inherited GetProperty(P);
 case ModelId of
  4021,7021:
   begin
    P.Addln('SafeValue = '+f2s(SafeValue[0]));
    P.Addln('PowerOnValue = '+f2s(PowerOnValue[0]));
    if (Enable16Bit>=0) then
    P.Addln('Enable16Bit = '+d2s(Sign(Enable16Bit)));
   end;
  7024,87024:
   begin
    P.Addln('SafeValue = '+f2s(SafeValue[0])+', '+f2s(SafeValue[1])+', '+
                           f2s(SafeValue[2])+', '+f2s(SafeValue[3]));
    P.Addln('PowerOnValue = '+f2s(PowerOnValue[0])+', '+f2s(PowerOnValue[1])+', '+
                              f2s(PowerOnValue[2])+', '+f2s(PowerOnValue[3]));
   end;
 end;
end;

 {
 форматирует данные для
 7024 типа +01.234 или -01.234
 7021 типа  01.234 или -01.234
 7021 типа  01.2345 при Enable16Bit
 }
function TAdamX021Device.DaStr(data:Double):LongString;
var i,j:Integer; s:LongString;
begin
 j:=ord(Enable16Bit>0);
 if (j>0) then data:=fabscompress(data,10000) else data:=fabscompress(data,1000);
 str(abs(data):6+j:3+j,s);
 for i:=1 to length(s) do if (s[i]=' ') then s[i]:='0' else break;
 case ModelId of
  4021,7021  : if (data<0) then insert('-',s,1);
  7024,87024 : if (data<0) then insert('-',s,1) else insert('+',s,1);
 end;
 DaStr:=s;
end;

function TAdamX021Device.Start:Boolean;
var i,errors:Integer;
begin
 Start:=false;
 if inherited Start and CheckDevice then begin
  errors:=0;
  if not CheckRangeCode then BugLog(errors,'CheckRangeCode');
  if not CheckDataFormat(dfEngineering,dfUnitMask) then BugLog(errors,'CheckDataFormat');
  {if watchdog supports}
  if WatchDogSupports then begin
   if not SetupWatchDog then BugLog(errors,'SetupWatchDog');
   {Set power on value and safe value}
   case ModelId of
    4021,7021:
     begin
      if (Enable16Bit>=0) then
      if ImmediateAnswerRequest('~'+hexb(Address)+'D'+HexB(8*Sign(Enable16Bit))+CR,
                                AdamParam.Attempts)<>'!'+hexb(Address)
      then BugLog(errors,'PowerOnValue');
      if ImmediateAnswerRequest('#'+hexb(Address)+
                                DaStr(PowerOnValue[0]/RangeFactor)+CR,
                                AdamParam.Attempts)<>'>'
      then BugLog(errors,'SetPowerOnValue') else
      if ImmediateAnswerRequest('$'+hexb(Address)+'4'+CR,
                                AdamParam.Attempts)<>'!'+hexb(Address)
      then BugLog(errors,'SafeValue');
      if ImmediateAnswerRequest('#'+hexb(Address)+
                                DaStr(SafeValue[0]/RangeFactor)+CR,
                                AdamParam.Attempts)<>'>'
      then BugLog(errors,'SetSafeValue') else
      if ImmediateAnswerRequest('~'+hexb(Address)+'4'+CR,
                                AdamParam.Attempts)<>'!'+hexb(Address)+
                                DaStr(SafeValue[0]/RangeFactor)
      then
      if ImmediateAnswerRequest('~'+hexb(Address)+'5'+CR,
                                AdamParam.Attempts)<>'!'+hexb(Address)
      then BugLog(errors,'SafeValue');
     end;
    7024,87024:
     for i:=0 to 3 do begin
      if ImmediateAnswerRequest('#'+hexb(Address)+d2s(i)+
                                DaStr(PowerOnValue[i]/RangeFactor)+CR,
                                AdamParam.Attempts)<>'>'
      then BugLog(errors,'PowerOnValue') else
      if ImmediateAnswerRequest('$'+hexb(Address)+'4'+d2s(i)+CR,
                                AdamParam.Attempts)<>'!'+hexb(Address)
      then BugLog(errors,'SetPowerOnValue');
      if ImmediateAnswerRequest('#'+hexb(Address)+d2s(i)+
                                DaStr(SafeValue[i]/RangeFactor)+CR,
                                AdamParam.Attempts)<>'>'
      then BugLog(errors,'SafeValue') else
      if ImmediateAnswerRequest('~'+hexb(Address)+'4'+d2s(i)+CR,
                                AdamParam.Attempts)<>'!'+hexb(Address)+
                                DaStr(SafeValue[i]/RangeFactor)
      then
      if ImmediateAnswerRequest('~'+hexb(Address)+'5'+d2s(i)+CR,
                                AdamParam.Attempts)<>'!'+hexb(Address)
      then BugLog(errors,'SetSafeValue');
     end;
   end;
  end;
  if not DoStartCommands then BugLog(errors,'DoStartCommands');
  if (errors>0) then FixError(ecAdamStart);
  Start:=true;
 end else begin
  FixError(ecAdamStart);
 end;
end;

 {
 Устройство имеет запросы:
 0..3 - аналоговый вывод 7024
 4    - аналоговый вывод 7021
 5    - Host Ok
 6    - проверка Watchdog
 }
function  TAdamX021Device.GetRequest:LongString;
begin
 Result:='';
 if InquiryTimer.Event and (RequestNumber in RequestList) then begin
  AwakeWatchDog(5);
  case RequestNumber of
   0..3 :
       Result:='#'+hexb(Address)+d2s(RequestNumber)+
       DaStr(SmoothAnalogInputCurve(RequestNumber,Daq.Timer.LocalTime)/RangeFactor)+CR;
   4 : Result:='#'+hexb(Address)+
       DaStr(SmoothAnalogInputCurve(0,Daq.Timer.LocalTime)/RangeFactor)+CR;
   5 : Result:='~**'+CR;                  {WatchDog HostOK     }
   6 : Result:='~'+hexb(Address)+'0'+CR;  {WatchDog Read status}
  end;
  SleepWatchDog(5);
 end;
end;

procedure TAdamX021Device.HandleAnswer;
begin
 case RequestNumber of
  0..4 : if (Answer<>'>') then HandleFormatError;
  5 : ;
  6 : HandleWatchDogStatus;
 end;
end;

function  TAdamX021Device.GetTimeOut:Double;
begin
 case RequestNumber of
  5 : GetTimeOut:=AdamParam.TimeOut[2];
  else GetTimeOut:=inherited GetTimeOut;
 end;
end;

procedure TAdamX021Device.HandleTimeOut;
begin
 case RequestNumber of
  5 : ; {No responce on HostOk command}
  else inherited HandleTimeOut;
 end;
end;

procedure TAdamX021Device.InitRequestList;
var n:Integer;
begin
 inherited InitRequestList;
 case ModelId of
  4021,7021:
   if (AnalogInputCurve[0]<>nil) then include(RequestList,4);
  7024,87024:
   for n:=0 to 3 do
   if (AnalogInputCurve[n]<>nil) then include(RequestList,n);
 end;
 if WatchDogSupports and (WatchDogTime>0) then RequestList:=RequestList+[5,6];
end;

 {
 *******************************************************************************
 Adam 4011,4011D,4012,4013,4014D
 *******************************************************************************
 }
constructor TAdamX011Device.Create(const aName:LongString; aModelId:Integer);
begin
 inherited Create(aName,aModelId);
 AnalogFifoSize:=64;
 DigitalFifoSize:=0;
 NumAnalogInputs:=0;
 NumDigitalInputs:=0;
 NumAnalogOutputs:=0;
 NumDigitalOutputs:=0;
 NumCalibrations:=0;
 case ModelId of
  4011,7011:
   begin
    AddMaskFormat(dfReject50);
    NumDigitalInputs:=4;   {DO0,DO1,AlarmEnable,AlarmLatch}
    NumAnalogInputs:=2;    {AlarmLo,AlarmHi}
    NumAnalogOutputs:=4;   {mV,CJ,EvCnt,EvFreq}
    NumDigitalOutputs:=1;  {di}
    AvailRange:=[$00..$06,$0E..$14];
    if (ModelId=7011) then AvailRange:=[$00..$06,$0E..$18];
    AlarmSupports:=true;
    nCJC:=1;
    nEvC:=2;
   end;
  4012,7012,4014,7014:
   begin
    AddMaskFormat(dfReject50);
    if (ModelId=4012) then AddMaskFormat(dfFastPoll);
    if (ModelId=7012) then AddMaskFormat(dfFastPoll);
    NumDigitalInputs:=4;   {DO0,DO1,AlarmEnable,AlarmLatch}
    NumAnalogInputs:=2;    {AlarmLo,AlarmHi}
    NumAnalogOutputs:=3;   {mV,EvCnt,EvFreq}
    NumDigitalOutputs:=1;  {di}
    AvailRange:=[$08..$0D];
    AlarmSupports:=true;
    nCJC:=-1;
    nEvC:=1;
   end;
  4013,7013:
   begin
    NumAnalogOutputs:=1;
    AvailRange:=[$20..$29,$2E,$2F,$80,$81];
    AlarmSupports:=false;
    nCJC:=-1;
    nEvC:=-1;
   end;
 end;
 AlarmEnable:=false;
 AlarmLatch:=false;
 AlarmLo:=0;
 AlarmHi:=0;
 DigitalFifoSize:=NumDigitalOutputs*64;
 WatchDogSupports:=IsValueIn(ModelId div 1000,[7,87]);
 ad_chan_no:=1;
 CounterBase:=0;
 FreqPeriod:=1;
 FreqScale:=1;
 LastCntVal:=0;
 LastCntTime:=0;
 InitRequestOrder;
end;

procedure TAdamX011Device.Config(FileName:LongString);
var flag:Boolean;
begin
 FileName:=UnifyFileAlias(FileName);
 inherited Config(FileName);
 if HasMaskFormat(dfFastPoll) then begin
  flag:=false;
  ReadIniFileBoolean(FileName,DevSection,'FastPoll%b',flag);
  if flag then FastPoll:=flag;
 end;
 if HasMaskFormat(dfReject50) then begin
  flag:=false;
  ReadIniFileBoolean(FileName,DevSection,'Reject50%b',flag);
  if flag then Reject50:=flag;
 end;
 ReadIniFileWord(FileName,DevSection,'SafeValue%w',SafeValue);
 ReadIniFileWord(FileName,DevSection,'PowerOnValue%w',PowerOnValue);
 ReadIniFileBoolean(FileName,DevSection,'AlarmEnable%b',AlarmEnable);
 ReadIniFileBoolean(FileName,DevSection,'AlarmLatch%b',AlarmLatch);
 ReadIniFileDouble(FileName,DevSection,'AlarmLo%f',AlarmLo);
 ReadIniFileDouble(FileName,DevSection,'AlarmHi%f',AlarmHi);
 ReadIniFileDouble(FileName,DevSection,'FrequencyPeriod%f',FreqPeriod);
 ReadIniFileDouble(FileName,DevSection,'FrequencyScale%f',FreqScale);
 AlarmEnable:=AlarmEnable and AlarmSupports;
end;

function  TAdamX011Device.GetProperty(P:TText):TText;
begin
 GetProperty:=inherited GetProperty(P);
 if HasMaskFormat(dfFastPoll) then P.AddLn('FastPoll = '+d2s(ord(FastPoll)));
 if HasMaskFormat(dfReject50) then P.AddLn('Reject50 = '+d2s(ord(Reject50)));
 P.Addln('SafeValue = $'+hexw(SafeValue));
 P.Addln('PowerOnValue = $'+hexw(PowerOnValue));
 P.Addln('AlarmEnable = '+d2s(ord(AlarmEnable)));
 P.Addln('AlarmLatch = '+d2s(ord(AlarmLatch)));
 P.Addln('AlarmLo = '+f2s(AlarmLo));
 P.Addln('AlarmHi = '+f2s(AlarmHi));
 P.Addln('FrequencyPeriod = '+f2s(FreqPeriod)+' sec');
 P.Addln('FrequencyScale = '+f2s(FreqScale));
end;

function TAdamX011Device.Start:Boolean;
var LM:Char; errors:Integer;
begin
 Start:=false;
 CounterBase:=0;
 LastCntVal:=0;
 LastCntTime:=Daq.Timer.LocalTime;
 if inherited Start and CheckDevice then begin
  errors:=0;
  if not CheckRangeCode then BugLog(errors,'CheckRangeCode');
  if not CheckDataFormat(DataFormat and MaskFormat,MaskFormat) then BugLog(errors,'CheckDataFormat');
  {if watchdog supports}
  if WatchDogSupports then begin
   if not SetupWatchDog then BugLog(errors,'SetupWatchdog');
   {Set power on value and safe value,if need}
   if NumDigitalInputs>0 then
   if ImmediateAnswerRequest('~'+hexb(Address)+'4'+CR,AdamParam.Attempts)<>
      '!'+hexb(Address)+hexb(PowerOnValue)+hexb(SafeValue)
   then begin
    if ImmediateAnswerRequest('~'+hexb(Address)+'5'+
                              hexb(PowerOnValue)+hexb(SafeValue)+CR,
                              AdamParam.Attempts)<>'!'+hexb(Address)
    then BugLog(errors,'SetPowerOnValue');
   end;
  end;
  if AlarmSupports then begin
   {set low alarm}
   if ImmediateAnswerRequest('@'+hexb(Address)+'RL'+CR,
                             AdamParam.Attempts)<>'!'+hexb(Address)+
                             DaStr(AlarmLo/RangeFactor)
   then
   if ImmediateAnswerRequest('@'+hexb(Address)+'LO'+
                             DaStr(AlarmLo/RangeFactor)+CR,
                             AdamParam.Attempts)<>'!'+hexb(Address)
   then BugLog(errors,'AlarmLo');
   {set high alarm}
   if ImmediateAnswerRequest('@'+hexb(Address)+'RH'+CR,
                             AdamParam.Attempts)<>'!'+hexb(Address)+
                             DaStr(AlarmHi/RangeFactor)
   then
   if ImmediateAnswerRequest('@'+hexb(Address)+'HI'+
                             DaStr(AlarmHi/RangeFactor)+CR,
                             AdamParam.Attempts)<>'!'+hexb(Address)
   then BugLog(errors,'AlarmHi');
   {clear event counter}
   if ImmediateAnswerRequest('@'+hexb(Address)+'CE'+CR,
                             AdamParam.Attempts)<>'!'+hexb(Address)
   then BugLog(errors,'ClearEventCounter');
   if AlarmEnable then begin
    {enable alarm}
    if AlarmLatch then LM:='L' else LM:='M';
    if ImmediateAnswerRequest('@'+hexb(Address)+'EA'+LM+CR,
                              AdamParam.Attempts)<>'!'+hexb(Address)
    then BugLog(errors,'EnableAlarm');
   end else begin
    {disable alarm}
    if ImmediateAnswerRequest('@'+hexb(Address)+'DA'+CR,
                             AdamParam.Attempts)<>'!'+hexb(Address)
    then BugLog(errors,'DisableAlarm');
   end;
  end;
  if not DoStartCommands then BugLog(errors,'DoStartCommands');
  if (errors>0) then FixError(ecAdamStart);
  Start:=true;
 end else begin
  FixError(ecAdamStart);
 end;
end;

 {
 Устройство имеет запросы:
 0 - термопарный канал
 1 - холодный спай
 2 - цифровой ввод
 3 - цифровой вывод
 4 - счетчик событий
 }
function  TAdamX011Device.GetRequest:LongString;
begin
 Result:='';
 if InquiryTimer.Event and (RequestNumber in RequestList) then begin
  AwakeWatchDog(1);
  case RequestNumber of
   0 : Result:='#'+hexb(Address)+CR;      {AnalogIn}
   1 : Result:='~**'+CR;                  {WatchDog HostOK}
   2 : Result:='~'+hexb(Address)+'0'+CR;  {WatchDog Read status}
   3 : Result:='$'+hexb(Address)+'3'+CR;  {CJC}
   4 : Result:='@'+hexb(Address)+'DI'+CR; {DigitalIn+AlarmState}
   5 : Result:='@'+hexb(Address)+'DO'+hexb(ConstructDigitalInputWord(0,2))+CR;
   6 : Result:='@'+hexb(Address)+'RE'+CR; {ReadEvent}
   7 : Result:='@'+hexb(Address)+'RL'+CR; {Read Lo alarm}
   8 : Result:='@'+hexb(Address)+'RH'+CR; {Read Hi alarm}
  end;
  SleepWatchDog(1);
 end;
end;

procedure TAdamX011Device.HandleAnswer;
var
 LM : Char;
 f  : Double;
 n  : Integer;
 d  : LongInt;
 d1 : LongInt;
 d2 : LongInt;
 ae : Boolean;
 al : Boolean;
begin
 case RequestNumber of
  0 : with AdamConvert8(RangeCode,UnitFormat,'>',Answer) do
      if (AdcCount=1) and (BugCount=0) then begin
       f:=ad_transform(0,AdcValue[0]);
       PutDaqEvent(DaqEvent(evAnalog+evCompress, 0, RequestTime, f));
      end else HandleFormatError;
  1 : ;
  2 : HandleWatchDogStatus;
  3 : if (length(Answer)=8) and (StrFetch(Answer,1)='>') and Str2Real(Copy(Answer,2,7),f)
      then PutDaqEvent(DaqEvent(evAnalog+evCompress, nCJC, RequestTime, f))
      else HandleFormatError;
  4 : if (length(Answer)=8) and (Copy(Answer,1,3)='!'+hexb(Address)) and
         StrHex2Long(Copy(Answer,7,2),d) and
         StrHex2Long(Copy(Answer,5,2),d1) and
         StrHex2Long(Copy(Answer,4,1),d2)
      then begin
       case d2 of
        0:d2:=0;
        1:d2:=1;
        2:d2:=3;
        else d2:=0;
       end;
       ae:=d2 and 1<>0;
       al:=d2 and 2<>0;
       d:=(d and 1)+((d1 and 3) shl 1)+((d2 and 3) shl 3);
       if (DigitalOutputCurve[0]<>nil)
       then PutDaqEvent(DaqEvent(evDigital, 0, RequestTime, d));
       if (DigitalInputCurve[2]<>nil)
       then AlarmEnable:=ConstructDigitalInputWord(2,1)<>0;
       if (DigitalInputCurve[3]<>nil)
       then AlarmLatch:=ConstructDigitalInputWord(3,1)<>0;
       if (ae<>AlarmEnable) or (al<>AlarmLatch) then begin
        if AlarmEnable then begin
         {enable alarm}
         if AlarmLatch then LM:='L' else LM:='M';
         if ImmediateAnswerRequest('@'+hexb(Address)+'EA'+LM+CR,
                                   AdamParam.Attempts)<>'!'+hexb(Address)
         then HandleFormatError;
         exclude(RequestList,5);
        end else begin
         {disable alarm}
         if ImmediateAnswerRequest('@'+hexb(Address)+'DA'+CR,
                                   AdamParam.Attempts)<>'!'+hexb(Address)
         then HandleFormatError;
         for n:=0 to 1 do
         if (DigitalInputCurve[n]<>nil) then include(RequestList,5);
        end;
       end;
      end else HandleFormatError;
  5 : if (Answer<>'!'+hexb(Address)) then HandleFormatError;
  6 : if (length(Answer)=8) and (Copy(Answer,1,3)='!'+hexb(Address)) and
         Str2Long(Copy(Answer,4,5),d)
      then begin
       if (d=65535) then begin
        CounterBase:=CounterBase+d;
        d:=0;
        ImmediateAnswerRequest('@'+hexb(Address)+'CE'+CR,AdamParam.Attempts);
       end;
       if (AnalogOutputCurve[nEvC]<>nil)
       then PutDaqEvent(DaqEvent(evAnalog+evCompress, nEvC, RequestTime, CounterBase+d));
       if (AnalogOutputCurve[nEvC+1]<>nil) then
       if (RequestTime-LastCntTime)*Daq.Timer.LocalTimeUnits/1000 > FreqPeriod
       then begin
        PutDaqEvent(DaqEvent(evAnalog+evCompress, nEvC+1, RequestTime,
                   (CounterBase+d-LastCntVal)/(RequestTime-LastCntTime)
                   /Daq.Timer.LocalTimeUnits*1000*FreqScale));
        LastCntTime:=RequestTime;
        LastCntVal:=CounterBase+d;
       end;
      end else HandleFormatError;
  7 : begin
       if (AnalogInputCurve[0]<>nil)
       then AlarmLo:=SmoothAnalogInputCurve(0,Daq.Timer.LocalTime);
       if (length(Answer)=10) and (Copy(Answer,1,3)='!'+hexb(Address)) and
          Str2Real(Copy(Answer,4,7),f)
       then begin
        if (abs(AlarmLo/RangeFactor-f)>intpower(10,pos('.',Answer)-length(Answer)))
        then begin
         if ImmediateAnswerRequest('@'+hexb(Address)+'LO'+
                                   DaStr(AlarmLo/RangeFactor)+CR,
                                   AdamParam.Attempts)<>'!'+hexb(Address)
         then HandleFormatError;
        end;
       end else HandleFormatError;
      end;
  8 : begin
       if (AnalogInputCurve[1]<>nil)
       then AlarmHi:=SmoothAnalogInputCurve(1,Daq.Timer.LocalTime);
       if (length(Answer)=10) and (Copy(Answer,1,3)='!'+hexb(Address)) and
          Str2Real(Copy(Answer,4,7),f)
       then begin
        if (abs(AlarmHi/RangeFactor-f)>intpower(10,pos('.',Answer)-length(Answer))*1.9)
        then begin
         if ImmediateAnswerRequest('@'+hexb(Address)+'HI'+
                                   DaStr(AlarmHi/RangeFactor)+CR,
                                   AdamParam.Attempts)<>'!'+hexb(Address)
         then HandleFormatError;
        end;
       end else HandleFormatError;
      end;
 end;
end;

function  TAdamX011Device.GetTimeOut:Double;
begin
 case RequestNumber of
  1 : GetTimeOut:=AdamParam.TimeOut[2];
  else GetTimeOut:=inherited GetTimeOut;
 end;
end;

procedure TAdamX011Device.HandleTimeOut;
begin
 case RequestNumber of
  1 : ; {No responce on HostOk command}
  else inherited HandleTimeOut;
 end;
end;

procedure TAdamX011Device.InitRequestList;
var n:Integer;
begin
 inherited InitRequestList;
 if (0 in ad_list*ad_gate) then
 if (AnalogOutputCurve[0]<>nil) then include(RequestList,0);
 if WatchDogSupports and (WatchDogTime>0) then RequestList:=RequestList+[1,2];
 if (nCJC in ad_list*ad_gate) then
 if (AnalogOutputCurve[nCJC]<>nil) then include(RequestList,3);
 if AlarmSupports then begin
  if not AlarmEnable then
  for n:=0 to 1 do
  if (DigitalInputCurve[n]<>nil) then include(RequestList,5);
  if (nEvC in ad_list*ad_gate) then
  if (AnalogOutputCurve[nEvC]<>nil) or (AnalogOutputCurve[nEvC+1]<>nil)
  then include(RequestList,6);
  if (DigitalOutputCurve[0]<>nil) then include(RequestList,4);
  if (DigitalInputCurve[2]<>nil) then include(RequestList,4);
  if (DigitalInputCurve[3]<>nil) then include(RequestList,4);
  if (AnalogInputCurve[0]<>nil) then include(RequestList,7);
  if (AnalogInputCurve[1]<>nil) then include(RequestList,8);
 end;
end;

function TAdamX011Device.DaStr(data:Double):LongString;
var i,d:Integer; s:LongString;
begin
 case RangeCode of
  $00:d:=3;  $01:d:=3;  $02:d:=2;  $03:d:=2;  $04:d:=4;  $05:d:=4;  $06:d:=3;
  $08:d:=3;  $09:d:=4;  $0A:d:=4;  $0B:d:=2;  $0C:d:=2;  $0D:d:=3;  $0E:d:=2;
  $0F:d:=1;  $10:d:=2;  $11:d:=1;  $12:d:=1;  $13:d:=1;  $14:d:=1;  $15:d:=1;
  $16:d:=1;
  else d:=3;
 end;
 data:=fabscompress(data,intpower(10,d));
 str(abs(data):6:d,s);
 for i:=1 to length(s) do if (s[i]=' ') then s[i]:='0' else break;
 if (data<0) then insert('-',s,1) else insert('+',s,1);
 DaStr:=s;
end;

procedure TAdamX011Device.AdvancedPropertyDialog;
begin
end;

 {
 *******************************************************************************
 Adam 4080,7080
 *******************************************************************************
 }
constructor TAdamX080Device.Create(const aName:LongString; aModelId:Integer);
begin
 inherited Create(aName,aModelId);
 AnalogFifoSize:=64;
 DigitalFifoSize:=64;
 NumAnalogInputs:=0;
 NumDigitalInputs:=2;
 NumAnalogOutputs:=4;
 NumDigitalOutputs:=1;
 NumCalibrations:=0;
 AvailRange:=[$50,$51];
 RangeCode:=$50;
 WatchDogSupports:=IsValueIn(ModelId div 1000,[7,87]);
 InputMode:=0;
 GateMode:=2;
 Filter.OnOff:=0;
 Filter.Lo:=2;
 Filter.Hi:=2;
 Trigger.Lo:=8;  {0.8V}
 Trigger.Hi:=24; {2.4V}
 CounterBase[0]:=0;
 CounterBase[1]:=0;
 FreqPeriod:=1;
 FreqScale:=1;
 LastCntVal[0]:=0;
 LastCntVal[1]:=0;
 LastCntTime[0]:=0;
 LastCntTime[1]:=0;
 InitRequestOrder;
end;

procedure TAdamX080Device.Config(FileName:LongString);
begin
 FileName:=UnifyFileAlias(FileName);
 inherited Config(FileName);
 ReadIniFileWord(FileName,DevSection,'InputMode%w',InputMode);
 ReadIniFileWord(FileName,DevSection,'GateMode%w',GateMode);
 ReadIniFileRecord(FileName,DevSection,'Filter%w;%w;%w',Filter);
 ReadIniFileRecord(FileName,DevSection,'Trigger%w;%w',Trigger);
 ReadIniFileDouble(FileName,DevSection,'FreqPeriod%f',FreqPeriod);
 ReadIniFileDouble(FileName,DevSection,'FreqScale%f',FreqScale);
end;

function  TAdamX080Device.GetProperty(P:TText):TText;
begin
 GetProperty:=inherited GetProperty(P);
 P.Addln('InputMode = '+d2s(InputMode));
 P.Addln('GateMode = '+d2s(GateMode));
 P.Addln('Filter = '+d2s(Filter.OnOff)+','+d2s(Filter.Lo)+','+d2s(Filter.Hi));
 P.Addln('Trigger = '+d2s(Trigger.Lo)+','+d2s(Trigger.Hi));
 P.Addln('FreqPeriod = '+f2s(FreqPeriod));
 P.Addln('FreqScale = '+f2s(FreqScale));
end;

function TAdamX080Device.Start:Boolean;
var n,errors:Integer;
 function w2sz(w,z:Word):LongString;
 var s:LongString;
 begin
  str(w,s);
  while (length(s)<z) do s:='0'+s;
  w2sz:=s;
 end;
begin
 Start:=false;
 CounterBase[0]:=0;
 CounterBase[1]:=0;
 LastCntVal[0]:=0;
 LastCntVal[1]:=0;
 LastCntTime[0]:=Daq.Timer.LocalTime;
 LastCntTime[1]:=Daq.Timer.LocalTime;
 if inherited Start and CheckDevice then begin
  errors:=0;
  {check range code}
  if not CheckRangeCode then BugLog(errors,'CheckRangeCode');
  {watchdog settings}
  if WatchDogSupports then begin
   if not SetupWatchDog then BugLog(errors,'SetupWatchDog');
  end;
  {set input mode}
  if ImmediateAnswerRequest('$'+hexb(Address)+'B'+CR,AdamParam.Attempts)<>
                            '!'+hexb(Address)+d2s(InputMode)
  then
  if ImmediateAnswerRequest('$'+hexb(Address)+'B'+d2s(InputMode)+CR,
       AdamParam.Attempts)<>'!'+hexb(Address)
  then BugLog(errors,'SetInputMode');
  {set trigger Lo}
  if ImmediateAnswerRequest('$'+hexb(Address)+'1L'+CR,AdamParam.Attempts)<>
                            '!'+hexb(Address)+w2sz(Trigger.Lo,2)
  then
  if ImmediateAnswerRequest('$'+hexb(Address)+'1L'+w2sz(Trigger.Lo,2)+CR,
       AdamParam.Attempts)<>'!'+hexb(Address)
  then BugLog(errors,'TriggerLo');
  {set trigger Hi}
  if ImmediateAnswerRequest('$'+hexb(Address)+'1H'+CR,AdamParam.Attempts)<>
                            '!'+hexb(Address)+w2sz(Trigger.Hi,2)
  then
  if ImmediateAnswerRequest('$'+hexb(Address)+'1H'+w2sz(Trigger.Hi,2)+CR,
       AdamParam.Attempts)<>'!'+hexb(Address)
  then BugLog(errors,'TriggerHi');
  {conter settings}
  if (RangeCode=$50) then begin
   {set alarm mode 1}
   if ImmediateAnswerRequest('~'+hexb(Address)+'A1'+CR,AdamParam.Attempts)<>
                             '!'+hexb(Address)
   then BugLog(errors,'AlarmMode');
   {alarm disable}
   if ImmediateAnswerRequest('@'+hexb(Address)+'DA'+CR,AdamParam.Attempts)<>
                             '!'+hexb(Address)
   then BugLog(errors,'AlarmDisable');
   {set filter status}
   if ImmediateAnswerRequest('$'+hexb(Address)+'4'+CR,AdamParam.Attempts)<>
                             '!'+hexb(Address)+d2s(Filter.OnOff)
   then
   if ImmediateAnswerRequest('$'+hexb(Address)+'4'+d2s(Filter.OnOff)+CR,
        AdamParam.Attempts)<>'!'+hexb(Address)
   then BugLog(errors,'FilterStatus');
   {set filter Lo}
   if ImmediateAnswerRequest('$'+hexb(Address)+'0L'+CR,AdamParam.Attempts)<>
                             '!'+hexb(Address)+w2sz(Filter.Lo,5)
   then
   if ImmediateAnswerRequest('$'+hexb(Address)+'0L'+w2sz(Filter.Lo,5)+CR,
        AdamParam.Attempts)<>'!'+hexb(Address)
   then BugLog(errors,'FilterLo');
   {set filter Hi}
   if ImmediateAnswerRequest('$'+hexb(Address)+'0H'+CR,AdamParam.Attempts)<>
                             '!'+hexb(Address)+w2sz(Filter.Hi,5)
   then
   if ImmediateAnswerRequest('$'+hexb(Address)+'0H'+w2sz(Filter.Hi,5)+CR,
        AdamParam.Attempts)<>'!'+hexb(Address)
   then BugLog(errors,'FilterHi');
   {set gate mode}
   if ImmediateAnswerRequest('$'+hexb(Address)+'A'+CR,AdamParam.Attempts)<>
                             '!'+hexb(Address)+d2s(GateMode)
   then
   if ImmediateAnswerRequest('$'+hexb(Address)+'A'+d2s(GateMode)+CR,
        AdamParam.Attempts)<>'!'+hexb(Address)
   then BugLog(errors,'GateMode');
   {set counter max values}
   for n:=0 to 1 do
   if ImmediateAnswerRequest('$'+hexb(Address)+'3'+d2s(n)+CR,AdamParam.Attempts)<>
                             '!'+hexb(Address)+hexl($FFFFFFFF)
   then
   if ImmediateAnswerRequest('$'+hexb(Address)+'3'+d2s(n)+hexl($FFFFFFFF)+CR,
        AdamParam.Attempts)<>'!'+hexb(Address)
   then BugLog(errors,'CounterMax');
   {set counter preset values}
   for n:=0 to 1 do
   if ImmediateAnswerRequest('@'+hexb(Address)+'G'+d2s(n)+CR,
        AdamParam.Attempts)<>'!'+hexb(Address)+hexl(0)
   then
   if ImmediateAnswerRequest('@'+hexb(Address)+'P'+d2s(n)+hexl(0)+CR,
        AdamParam.Attempts)<>'!'+hexb(Address)
   then BugLog(errors,'CounterPreset');
   {go to preset values}
   for n:=0 to 1 do
   if ImmediateAnswerRequest('$'+hexb(Address)+'6'+d2s(n)+CR,
        AdamParam.Attempts)<>'!'+hexb(Address)
   then BugLog(errors,'PresetValues');
   {set counter status = start (enable)}
   for n:=0 to 1 do
   if ImmediateAnswerRequest('$'+hexb(Address)+'5'+d2s(n)+CR,
        AdamParam.Attempts)<>'!'+hexb(Address)+'1'
   then
   if ImmediateAnswerRequest('$'+hexb(Address)+'5'+d2s(n)+'1'+CR,
        AdamParam.Attempts)<>'!'+hexb(Address)
   then BugLog(errors,'CounterStatus');
  end;
  if not DoStartCommands then BugLog(errors,'DoStartCommands');
  if (errors>0) then FixError(ecAdamStart);
  Start:=true;
 end else begin
  FixError(ecAdamStart);
 end;
end;

function  TAdamX080Device.GetRequest:LongString;
begin
 Result:='';
 if InquiryTimer.Event and (RequestNumber in RequestList) then begin
  AwakeWatchDog(2);
  case RequestNumber of
   0 : Result:='#'+hexb(Address)+'0'+CR;  {Read counter or frequency #0}
   1 : Result:='#'+hexb(Address)+'1'+CR;  {Read counter or frequency #1}
   2 : Result:='~**'+CR;                  {WatchDog HostOK}
   3 : Result:='~'+hexb(Address)+'0'+CR;  {WatchDog Read status}
   4 : Result:='$'+hexb(Address)+'70'+CR; {Read counter #0 overflow flag}
   5 : Result:='$'+hexb(Address)+'71'+CR; {Read counter #1 overflow flag}
   6 : Result:='@'+hexb(Address)+'DI'+CR; {DigitalReadback+AlarmState}
   7 : Result:='@'+hexb(Address)+'DO'+hexb(ConstructDigitalInputWord(0,2))+CR;
   8 : Result:='$'+hexb(Address)+'60'+CR; {Clear channel 0}
   9 : Result:='$'+hexb(Address)+'61'+CR; {Clear channel 1}
  end;
  SleepWatchDog(2);
 end;
end;

procedure TAdamX080Device.HandleAnswer;
var
 f    : Double;
 d    : LongInt;
 s    : LongInt;
 chan : Word;
 function u2f(x:Cardinal):Double;
 begin
  Result:=Cardinal(x);
 end;
begin
 case RequestNumber of
  0,1 :
      if (length(Answer)=9) and (StrFetch(Answer,1)='>') and StrHex2Long(Copy(Answer,2,8),d)
      then begin
       f:=u2f(d); {convert unsigned long to Double}
       if (RangeCode=$51) then begin {frequency mode}
        chan:=RequestNumber;
        if (AnalogOutputCurve[chan]<>nil)
        then PutDaqEvent(DaqEvent(evAnalog+evCompress, chan, RequestTime, f));
       end else begin {counter mode}
        {frequency evaluation}
        chan:=RequestNumber;
        if (AnalogOutputCurve[chan]<>nil) then
        if (RequestTime-LastCntTime[chan])*Daq.Timer.LocalTimeUnits/1000>FreqPeriod
        then begin
         PutDaqEvent(DaqEvent(evAnalog+evCompress, chan, RequestTime,
                    (CounterBase[chan]+f-LastCntVal[chan])/
                    (RequestTime-LastCntTime[chan])/
                    Daq.Timer.LocalTimeUnits*1000*FreqScale));
         LastCntTime[chan]:=RequestTime;
         LastCntVal[chan]:=CounterBase[chan]+f;
        end;
        {counter evaluation}
        chan:=RequestNumber+2;
        if (AnalogOutputCurve[chan]<>nil)
        then PutDaqEvent(DaqEvent(evAnalog+evCompress, chan, RequestTime,
                                  CounterBase[RequestNumber]+f));
       end;
      end else HandleFormatError;
  2 : ;
  3 : HandleWatchDogStatus;
  4,5 :
      if (length(Answer)=4) and (Copy(Answer,1,3)='!'+hexb(Address)) and
         Str2Long(Copy(Answer,4,1),d)
      then begin
       if HasFlags(d,1) then begin {counter overflow detected, reset}
        chan:=RequestNumber-4;
        if ImmediateAnswerRequest('$'+hexb(Address)+'6'+d2s(chan)+CR,
             AdamParam.Attempts)<>'!'+hexb(Address)
        then HandleTimeout;
        CounterBase[chan]:=CounterBase[chan]+u2f($FFFFFFFF)+1;
        LastCntVal[chan]:=CounterBase[chan];
       end;
      end else HandleFormatError;
  6 : if (length(Answer)=8) and (Copy(Answer,1,3)='!'+hexb(Address)) and
         StrHex2Long(Copy(Answer,7,2),d) and (d=0) and
         StrHex2Long(Copy(Answer,5,2),d) and
         StrHex2Long(Copy(Answer,4,1),s)
      then begin
       if (DigitalOutputCurve[0]<>nil)
       then PutDaqEvent(DaqEvent(evDigital, 0, RequestTime, (d and 3)+(s and 3) shl 2));
       {we don't use alarm, so alarm must be disabled}
       if (s<>0) then begin
        {set alarm mode 1}
        ImmediateAnswerRequest('~'+hexb(Address)+'A1'+CR,AdamParam.Attempts);
        {alarm disable}
        ImmediateAnswerRequest('@'+hexb(Address)+'DA'+CR,AdamParam.Attempts);
       end;
      end else HandleFormatError;
  7 : if (Answer<>'!'+hexb(Address)) then HandleFormatError;
  8,9 :
      if (Answer<>'!'+hexb(Address)) then HandleFormatError else begin
       Exclude(RequestList,RequestNumber);
       chan:=RequestNumber-8;
       CounterBase[chan]:=0;
       LastCntVal[chan]:=0;
       LastCntTime[chan]:=Daq.Timer.LocalTime;
      end;
 end;
end;

function  TAdamX080Device.GetTimeOut:Double;
begin
 case RequestNumber of
  2 : GetTimeOut:=AdamParam.TimeOut[2];
  else GetTimeOut:=inherited GetTimeOut;
 end;
end;

procedure TAdamX080Device.HandleTimeOut;
begin
 case RequestNumber of
  2 : ; {No responce on HostOk command}
  else inherited HandleTimeOut;
 end;
end;

procedure TAdamX080Device.InitRequestList;
begin
 inherited InitRequestList;
 if (AnalogOutputCurve[0]<>nil) then include(RequestList,0);
 if (AnalogOutputCurve[1]<>nil) then include(RequestList,1);
 if (RangeCode=$50) then begin
  if (AnalogOutputCurve[2]<>nil) then RequestList:=RequestList+[0,4];
  if (AnalogOutputCurve[3]<>nil) then RequestList:=RequestList+[1,5];
 end;
 if WatchDogSupports and (WatchDogTime>0) then RequestList:=RequestList+[2,3];
 if (DigitalOutputCurve[0]<>nil) then include(RequestList,6);
 if (DigitalInputCurve[0]<>nil) then include(RequestList,7);
 if (DigitalInputCurve[1]<>nil) then include(RequestList,7);
end;

function TAdamX080Device.HandleMessage(const aMsg:LongString; aFlags:Cardinal=hf_Default):Double;
var chn:LongInt; buf:TParsingBuffer;
begin
 Result:=inherited HandleMessage(aMsg,aFlags); chn:=0;
 if (RangeCode<>$51) then {ResetCounter disabled in frequency mode}
 if Assigned(ScanVarLongInt(svConfig,StrUpper(StrCopyBuff(buf,aMsg)),'ResetCounter%d',chn)) and (chn in [0..1]) then begin
  Include(RequestList,8+chn);
  Result:=Result+1;
 end;
end;

 {
 *******************************************************************************
 TRSDAQHostDevice implementation
 *******************************************************************************
 }
constructor TRSDAQHostDevice.Create(const aName:LongString);
begin
 inherited Create(aName);
 SetDeviceModel('RSDAQHOST');
 AnalogFifoSize:=1024;
 NumDigitalInputs:=1;
 NumAnalogOutputs:=256;
end;

function  TRSDAQHostDevice.GetRequest:LongString;
begin
 Result:='';
 if InquiryTimer.Event and (RequestNumber in RequestList) then begin
  if (ConstructDigitalInputWord(0,1)=0) then
  case RequestNumber of
   0..255 : Result:='#'+hexb(Address)+hexb(RequestNumber)+CR;
  end;
 end;
end;

procedure TRSDAQHostDevice.HandleAnswer;
var f:Double;
begin
 case RequestNumber of
  0..255 :
   if (Answer='>')
   then {no data ready}
   else {data expected}
   if (length(Answer)>1) and (StrFetch(Answer,1)='>') and
      Str2Real(Copy(Answer,2,length(Answer)-1),f)
   then PutDaqEvent(DaqEvent(evAnalog+evCompress, RequestNumber, RequestTime, f))
   else HandleFormatError;
 end;
end;

procedure TRSDAQHostDevice.InitRequestList;
var i:Integer;
begin
 inherited InitRequestList;
 for i:=0 to 255 do if (AnalogOutputCurve[i]<>nil) then include(RequestList,i);
end;

 {
 *******************************************************************************
 TBalzersTPG256 device implementation
 *******************************************************************************
 }
const          {command char set for Balzers units:        }
//cEXT = #3;   {end of text,^C                             }
  cENQ = #5;   {enquery,request for data transmission      }
  cACK = #6;   {acknowledge,positive report signal         }
  cNAK = #21;  {negative acknowledge,negative report signal}
//cESC = #27;  {escape                                     }
  cINP = false;
  cOUT = true;

constructor TBalzersTPG256.Create(const aName:LongString);
begin
 inherited Create(aName);
 SetDeviceModel('BALZERS-TPG256');
 AnalogFifoSize:=64;
 DigitalFifoSize:=64;
 NumAnalogOutputs:=6;
 NumDigitalOutputs:=18;
 StartList:=NewText(0,16);
 StartAns:=NewText(0,16);
 StopList:=NewText(0,16);
 StopAns:=NewText(0,16);
 UsesRS485:=false;
end;

destructor TBalzersTPG256.Destroy;
begin
 Kill(StartList);
 Kill(StartAns);
 Kill(StopList);
 Kill(StopAns);
 inherited Destroy;
end;

procedure TBalzersTPG256.Config(FileName:LongString);
var i:Integer; Section:TText;
 procedure ReadLine(const s:LongString);
 var ss:LongString; buf:TParsingBuffer;
 begin
  ss:='';
  if (s<>'') then begin
   if (ScanVarString(svConfig,StrCopyBuff(buf,UpcaseStr(s)),'StartCommand%s',ss)<>nil)
   then StartList.Addln(TrimChars(ss,ScanSpaces,ScanSpaces));
   if (ScanVarString(svConfig,StrCopyBuff(buf,UpcaseStr(s)),'StopCommand%s',ss)<>nil)
   then StopList.Addln(TrimChars(ss,ScanSpaces,ScanSpaces));
  end;
 end;
begin
 FileName:=UnifyFileAlias(FileName);
 inherited Config(FileName);
 StartList.Count:=0;
 StartAns.Count:=0;
 StopList.Count:=0;
 StopAns.Count:=0;
 Section:=ExtractListSection(FileName,DevSection,efConfig);
 try
  for i:=0 to Section.Count-1 do ReadLine(Section[i]);
 finally
  Kill(Section);
 end;
 ReadIniFileBoolean(FileName,DevSection,'UsesRS485%b',UsesRS485);
end;

function TBalzersTPG256.GetProperty(P:TText):TText;
var
 i : Integer;
begin
 GetProperty:=inherited GetProperty(P);
 for i:=0 to StartList.Count-1 do begin
  P.Addln('StartCommand = '+StartList[i]);
  P.Addln('Answer       = '+StartAns[i]);
 end;
 for i:=0 to StopList.Count-1 do begin
  P.Addln('StopCommand  = '+StopList[i]);
  P.Addln('Answer       = '+StopAns[i]);
 end;
 P.Addln('UsesRS485 = '+d2s(ord(UsesRS485)));
end;

function    TBalzersTPG256.Start:Boolean;
var
 i : Integer;
 procedure SendCmd(const s:LongString);
 var ss:LongString;
 begin
  if (s<>'') then begin
   ss:=''; Send(s+CR,ss);
   StartAns.Addln(ss);
  end;
 end;
begin
 Start:=false;
 if inherited Start then begin
  StartAns.Count:=0;
  StopAns.Count:=0;
  errors:=0;
  for i:=0 to StartList.Count-1 do SendCmd(StartList[i]);
  if IsMainThread then UpdateFormDaqDeviceControl(1+2+4);
  if (errors>0) then FixError(ecAdamStart);
  Start:=true;
 end;
end;

procedure   TBalzersTPG256.Stop;
var i:Integer; Ask,Ans:LongString; When:Double;
const Tick=10;
 procedure SendCmd(const s:LongString);
 var ss:LongString;
 begin
  if (s<>'') then begin
   ss:=''; Send(s+CR,ss);
   StopAns.Addln(ss);
  end;
 end;
begin
 Ask:=''; Ans:=''; When:=0;
 if InquiryTimer.IsStart then begin
  // Wait while last request finished and then apply StopCommand list
  for i:=1 to AdamParam.TimeOut[1] div Tick do
  if (AnswerRequest(PortN,Ask,Ans,When)=rs_WaitAnswer) then Sleep(Tick) else Break;
  errors:=0;
  StartAns.Count:=0;
  StopAns.Count:=0;
  for i:=0 to StopList.Count-1 do SendCmd(StopList[i]);
  if IsMainThread then UpdateFormDaqDeviceControl(1+2+4);
  if (errors<>0) then FixError(ecAdamStop);
 end;
 inherited Stop;
end;

function TBalzersTPG256.GetRequest:LongString;
begin
 Result:='';
 if InquiryTimer.Event and (RequestNumber in RequestList) then begin
  case RequestNumber of
   0,2,4,6,8,10:
    Result:=Route('PR'+d2s((RequestNumber shr 1) + 1)+CR,cOUT);
   12:
    Result:=Route('SEN'+CR,cOUT);
   14:
    Result:=Route('SPS'+CR,cOUT);
   1,3,5,7,9,11,13,15:
    Result:=Route(cENQ,cOUT);
  end;
 end;
end;

procedure   TBalzersTPG256.HandleAnswer;
var
 i    : Integer;
 err  : Word;
 chan : Word;
 val  : Double;
 sen  : packed array[0..5] of Word;
 sps  : packed array[0..5] of Word;
begin
 Answer:=Route(Answer,cINP);
 case RequestNumber of
  0,2,4,6,8,10,12,14 :
   if (Answer<>cACK) then HandleFormatError;
  1,3,5,7,9,11 :
   if Str2Word(ExtractWord(1,Answer,ScanSpaces),err) and
      Str2Real(ExtractWord(2,Answer,ScanSpaces),val)
   then begin
    chan:=RequestNumber shr 1;
    if (AnalogOutputCurve[chan]<>nil)
    then PutDaqEvent(DaqEvent(evAnalog+evCompress, chan, RequestTime, val));
    if (DigitalOutputCurve[chan]<>nil)
    then PutDaqEvent(DaqEvent(evDigital, chan, RequestTime, err));
   end else HandleFormatError;
  13 :
   if Str2Word(ExtractWord(1,Answer,ScanSpaces),sen[0]) and
      Str2Word(ExtractWord(2,Answer,ScanSpaces),sen[1]) and
      Str2Word(ExtractWord(3,Answer,ScanSpaces),sen[2]) and
      Str2Word(ExtractWord(4,Answer,ScanSpaces),sen[3]) and
      Str2Word(ExtractWord(5,Answer,ScanSpaces),sen[4]) and
      Str2Word(ExtractWord(6,Answer,ScanSpaces),sen[5])
   then begin
    for i:=0 to 5 do begin
     chan:=6+i;
     if (DigitalOutputCurve[chan]<>nil)
     then PutDaqEvent(DaqEvent(evDigital, chan, RequestTime, sen[i]));
    end;
   end else HandleFormatError;
  15 :
   if Str2Word(ExtractWord(1,Answer,ScanSpaces),sps[0]) and
      Str2Word(ExtractWord(2,Answer,ScanSpaces),sps[1]) and
      Str2Word(ExtractWord(3,Answer,ScanSpaces),sps[2]) and
      Str2Word(ExtractWord(4,Answer,ScanSpaces),sps[3]) and
      Str2Word(ExtractWord(5,Answer,ScanSpaces),sps[4]) and
      Str2Word(ExtractWord(6,Answer,ScanSpaces),sps[5])
   then begin
    for i:=0 to 5 do begin
     chan:=12+i;
     if (DigitalOutputCurve[chan]<>nil)
     then PutDaqEvent(DaqEvent(evDigital, chan, RequestTime, sps[i]));
    end;
   end else HandleFormatError;
 end;
end;

procedure   TBalzersTPG256.HandleTimeOut;
begin
 case RequestNumber of
  0,2,4,6,8,10,12,14 : inc(RequestNumber); {we have to pass next request}
 end;
 inherited HandleTimeOut;
end;

procedure   TBalzersTPG256.HandleFormatError;
begin
 case RequestNumber of
  0,2,4,6,8,10,12,14 : inc(RequestNumber); {we have to pass next request}
 end;
 inherited HandleFormatError;
end;

procedure   TBalzersTPG256.InitRequestList;
var n:Integer;
begin
 inherited InitRequestList;
 for n:=0 to 5 do begin
  if (AnalogOutputCurve[n]<>nil) or (DigitalOutputCurve[n]<>nil) then begin
   include(RequestList,n*2);
   include(RequestList,n*2+1);
  end;
  if (DigitalOutputCurve[6+n]<>nil) then RequestList:=RequestList+[12,13];
  if (DigitalOutputCurve[12+n]<>nil) then RequestList:=RequestList+[14,15];
 end;
end;

function TBalzersTPG256.AckEnq(cmd:LongString):LongString;
begin
 cmd:=Route(ImmediateAnswerRequest(Route(cmd,cOUT),AdamParam.Attempts),cINP);
 if (cmd<>cACK) then AckEnq:=cNAK else
 cmd:=Route(ImmediateAnswerRequest(Route(cENQ,cOUT),AdamParam.Attempts),cINP);
 if (cmd='') then AckEnq:=cNAK else AckEnq:=cmd;
end;

function TBalzersTPG256.Send(cmd:LongString; var ans:LongString):Boolean;
begin
 cmd:=AckEnq(cmd);
 if (cmd<>cNAK) then ans:=cmd else begin inc(errors); ans:=''; end;
 Send:=(cmd<>cNAK);
end;

function TBalzersTPG256.Route(s:LongString; mode:Boolean):LongString;
var i:Integer;
begin
 if UsesRS485 then
 case mode of
  cOUT: begin
         s:='$'+hexb(Address)+s;
         if (StrFetch(s,length(s))<>CR) then s:=s+CR;
        end;
  cINP: begin
         i:=pos('!'+hexb(Address),s);
         if (i>0) then system.Delete(s,i,3) else s:='';
        end;
 end;
 Route:=s;
end;

 {
 *******************************************************************************
 TBalzersTPG252 device implementation
 *******************************************************************************
 }
constructor TBalzersTPG252.Create(const aName:LongString);
begin
 inherited Create(aName);
 SetDeviceModel('BALZERS-TPG252');
 AnalogFifoSize:=64;
 DigitalFifoSize:=64;
 NumAnalogOutputs:=2;
 NumDigitalOutputs:=6;
end;

function TBalzersTPG252.GetRequest:LongString;
begin
 Result:='';
 if InquiryTimer.Event and (RequestNumber in RequestList) then begin
  case RequestNumber of
   0,2:
    Result:=Route('PR'+d2s((RequestNumber shr 1) + 1)+CR,cOUT);
   4:
    Result:=Route('SEN'+CR,cOUT);
   6:
    Result:=Route('SPS'+CR,cOUT);
   1,3,5,7:
    Result:=Route(cENQ,cOUT);
  end;
 end;
end;

procedure   TBalzersTPG252.HandleAnswer;
var
 i    : Integer;
 err  : Word;
 chan : Word;
 val  : Double;
 sen  : packed array[0..1] of Word;
 sps  : packed array[0..1] of Word;
begin
 Answer:=Route(Answer,cINP);
 case RequestNumber of
  0,2,4,6 :
   if (Answer<>cACK) then HandleFormatError;
  1,3 :
   if Str2Word(ExtractWord(1,Answer,ScanSpaces),err) and
      Str2Real(ExtractWord(2,Answer,ScanSpaces),val)
   then begin
    chan:=RequestNumber shr 1;
    if (AnalogOutputCurve[chan]<>nil)
    then PutDaqEvent(DaqEvent(evAnalog+evCompress, chan, RequestTime, val));
    if (DigitalOutputCurve[chan]<>nil)
    then PutDaqEvent(DaqEvent(evDigital,chan,RequestTime,err));
   end else HandleFormatError;
  5 :
   if Str2Word(ExtractWord(1,Answer,ScanSpaces),sen[0]) and
      Str2Word(ExtractWord(2,Answer,ScanSpaces),sen[1])
   then begin
    for i:=0 to 1 do begin
     chan:=2+i;
     if (DigitalOutputCurve[chan]<>nil)
     then PutDaqEvent(DaqEvent(evDigital, chan, RequestTime, sen[i]));
    end;
   end else HandleFormatError;
  7 :
   if Str2Word(ExtractWord(1,Answer,ScanSpaces),sps[0]) and
      Str2Word(ExtractWord(2,Answer,ScanSpaces),sps[1])
   then begin
    for i:=0 to 1 do begin
     chan:=4+i;
     if (DigitalOutputCurve[chan]<>nil)
     then PutDaqEvent(DaqEvent(evDigital, chan, RequestTime, sps[i]));
    end;
   end else HandleFormatError;
 end;
end;

procedure   TBalzersTPG252.HandleTimeOut;
begin
 case RequestNumber of
  0,2,4,6 : inc(RequestNumber); {we have to pass next request}
 end;
 FixError(ecAdamTimeOut);           
 if HasFlags(AdamParam.DebugMode,2)
 then AdamDebugOut(DebugTime+'COM'+d2s(PortN)+': TIMEOUT ON REQUEST '+TrimChars(Request,[CR,LF],[CR,LF]));
end;

procedure   TBalzersTPG252.HandleFormatError;
begin
 case RequestNumber of
  0,2,4,6 : inc(RequestNumber); {we have to pass next request}
 end;
 FixError(ecAdamFormat);
 if HasFlags(AdamParam.DebugMode,4)
 then AdamDebugOut(DebugTime+'COM'+d2s(PortN)+': FORMAT ERROR ON REQUEST "'+
               TrimChars(Request,[CR,LF],[CR,LF])+'" , ANSWER= "'+TrimChars(Answer,[CR,LF],[CR,LF])+'"');
end;

procedure   TBalzersTPG252.InitRequestList;
var
 n : Integer;
begin
 RequestList:=[];
 for n:=0 to 1 do begin
  if (AnalogOutputCurve[n]<>nil) or (DigitalOutputCurve[n]<>nil) then begin
   include(RequestList,n*2);
   include(RequestList,n*2+1);
  end;
  if (DigitalOutputCurve[2+n]<>nil) then RequestList:=RequestList+[4,5];
  if (DigitalOutputCurve[4+n]<>nil) then RequestList:=RequestList+[6,7];
 end;
end;

 {
 *******************************************************************************
 TAdamSlot implementation
 *******************************************************************************
 }
constructor TAdamSlot.Create(const aName:LongString);
begin
 inherited Create(AName);
 SetDeviceModel('ADAM_SLOT');
 Fix.Request:='';
 Fix.Answer:='';
 Fix.ReqTime:=0;
 Fix.Status:=as_NotAvail;
 Fix.TimeOut:=0;
end;

destructor TAdamSlot.Destroy;
begin
 Fix.Answer:='';
 Fix.Request:='';
 inherited Destroy;
end;

function TAdamSlot.Start:Boolean;
begin
 Start:=false;
 Lock;
 if inherited Start then begin
  Fix.Request:='';
  Fix.Answer:='';
  Fix.ReqTime:=0;
  if AdamDeviceList.UsesPort[PortN]
  then Fix.Status:=as_NoRequest
  else Fix.Status:=as_NotAvail;
  Fix.TimeOut:=0;
  InitRequestOrder;
  Start:=true;
 end;
 Unlock;
end;

procedure   TAdamSlot.Stop;
begin
 Lock;
 Fix.Request:='';
 Fix.Answer:='';
 Fix.ReqTime:=0;
 Fix.Status:=as_NotAvail;
 Fix.TimeOut:=0;
 Unlock;
 inherited Stop;
end;

function  TAdamSlot.GetRequest:LongString;
begin
 Result:='';
 if InquiryTimer.Event and (RequestNumber in RequestList) then begin
  Lock;
  case RequestNumber of
   0:
   case Fix.Status of
    as_WaitAnswer : HandleTimeOut;
    as_WaitQueue  : begin
                     Result:=Fix.Request;
                     Fix.Status:=as_WaitAnswer;
                    end;
   end;
  end;
  Unlock;
 end;
end;

procedure TAdamSlot.HandleAnswer;
begin
 Lock;
 Fix.Status:=as_Answer;
 Fix.Answer:=Answer;
 Fix.ReqTime:=RequestTime;
 Unlock;
end;

function  TAdamSlot.GetTimeOut:Double;
begin
 Lock;
 if (Fix.TimeOut=0)
 then GetTimeOut:=inherited GetTimeOut
 else GetTimeOut:=abs(Fix.TimeOut);
 Unlock;
end;

procedure TAdamSlot.HandleTimeOut;
begin
 Lock;
 Fix.Status:=as_TimeOut;
 Fix.Answer:='';
 Fix.ReqTime:=RequestTime;
 if (Fix.TimeOut>=0) then inherited HandleTimeOut;
 Unlock;
end;

procedure TAdamSlot.InitRequestList;
begin
 Lock;
 inherited InitRequestList;
 RequestList:=[0];
 Unlock;
end;

function TAdamSlot.SetRequest(const aRequest:LongString; aTimeOut:Integer):Boolean;
begin
 Lock;
 if (Fix.Status in [as_NotAvail,as_WaitAnswer])
 then SetRequest:=false
 else begin
  SetRequest:=true;
  Fix.Request:=aRequest;
  Fix.Answer:='';
  Fix.TimeOut:=aTimeOut;
  if (Fix.Request<>'')
  then Fix.Status:=as_WaitQueue
  else Fix.Status:=as_NoRequest;
 end;
 Unlock;
end;

function _Adam_Get(Slot:TAdamSlot; What:LongString):LongString;
begin
 Result:='';
 if Assigned(Slot) then with Slot do begin
  Lock;
  What:=UpCaseStr(What);
  if SameText(What,'ANSWER')       then Result:=Fix.Answer else
  if SameText(What,'REQUEST')      then Result:=Fix.Request else
  if SameText(What,'ADDRESS')      then Result:=hexb(Address) else
  if SameText(What,'PORT')         then Result:=d2s(PortN) else
  if SameText(What,'USESCHECKSUM') then Result:=d2s(ord(UsesCheckSum));
  Unlock;
 end;
end;

function _Adam_Request(Slot:TAdamSlot; const aRequest:LongString; aTimeOut:Integer):Boolean;
begin
 if Assigned(Slot)
 then Result:=Slot.SetRequest(aRequest,aTimeOut)
 else Result:=false;
end;

function _Adam_Status(Slot:TAdamSlot):Integer;
begin
 if Assigned(Slot) then with Slot do begin
  Lock;
  Result:=Fix.Status;
  case Fix.Status of
   as_NotAvail   : ;
   as_NoRequest  : ;
   as_WaitQueue  : ;
   as_WaitAnswer : ;
   as_Answer     : Fix.Status:=as_NoRequest;
   as_TimeOut    : Fix.Status:=as_NoRequest;
  end;
  Unlock;
 end else Result:=as_NotAvail;
end;

function _Adam_ReqTime(Slot:TAdamSlot):Double;
begin
 if Assigned(Slot) then with Slot do begin
  Lock;
  Result:=Fix.ReqTime;
  Unlock;
 end else Result:=0;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_adamdev;
begin
 AdamDeviceList.Ok;
end;

procedure Free_crw_adamdev;
begin
 AdamDeviceList.Free;
end;

initialization

 Init_crw_adamdev;

finalization

 Free_crw_adamdev;

end.

//////////////
// END OF FILE
//////////////

