////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Unit Read Old CRW Files.                                                   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231124 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit unit_readoldcrwfiles; // Unit Read Old CRW Files

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math, Contnrs,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_ListBoxSelection, Form_CurveWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_curves;

const
 sig_FBPR = $52504246; { 'FBPR' }

function ReadFileSignature(const FileName:LongString):LongInt;
function SelectKeyFromResource(const FileName:LongString):LongString;
function ReadObjectFromResource(const FileName,ObjKey:LongString):TObject;

implementation

function ReadObjectFromStream(S:TStream):TObject; forward;

function ReadByte(S:TStream):Byte;         begin SafeFillChar(Result,SizeOf(Result),0); S.ReadBuffer(Result,sizeof(Result)); end;
function ReadShortInt(S:TStream):ShortInt; begin SafeFillChar(Result,SizeOf(Result),0); S.ReadBuffer(Result,sizeof(Result)); end;
function ReadBoolean(S:TStream):Boolean;   begin SafeFillChar(Result,SizeOf(Result),0); S.ReadBuffer(Result,sizeof(Result)); end;
function ReadWord(S:TStream):Word;         begin SafeFillChar(Result,SizeOf(Result),0); S.ReadBuffer(Result,sizeof(Result)); end;
function ReadSmallInt(S:TStream):SmallInt; begin SafeFillChar(Result,SizeOf(Result),0); S.ReadBuffer(Result,sizeof(Result)); end;
function ReadSingle(S:TStream):Single;     begin SafeFillChar(Result,SizeOf(Result),0); S.ReadBuffer(Result,sizeof(Result)); end;
function ReadLongInt(S:TStream):LongInt;   begin SafeFillChar(Result,SizeOf(Result),0); S.ReadBuffer(Result,sizeof(Result)); end;
function ReadReal48(S:TStream):Real48;     begin SafeFillChar(Result,SizeOf(Result),0); S.ReadBuffer(Result,sizeof(Result)); end;
function ReadDouble(S:TStream):Double;     begin SafeFillChar(Result,SizeOf(Result),0); S.ReadBuffer(Result,sizeof(Result)); end;
function ReadComp(S:TStream):Comp;         begin SafeFillChar(Result,SizeOf(Result),0); S.ReadBuffer(Result,sizeof(Result)); end;
function ReadRect2D(S:TStream):TRect2D;    begin SafeFillChar(Result,SizeOf(Result),0); S.ReadBuffer(Result,sizeof(Result)); end;

function ReadPureString(S:TStream; Convert:Boolean=true):PureString;
begin
 Result:='';
 S.ReadBuffer(Result[0],1);
 if Length(Result)>0 then begin
  S.ReadBuffer(Result[1],Length(Result));
  if Convert then Result:=DosToWinStr(Result);
  if Convert then Result:=ConvertCP(Result,CP_1251,CP_UTF8,CP_NONE);
 end;
end;

function ReadFileSignature(const FileName:LongString):LongInt;
var
 S : TFileStream;
begin
 Result:=0;
 if FileExists(FileName) then
 try
  S:=TFileStream.Create(FileName,fmOpenRead);
  try
   if S.Seek(0,soFromBeginning)<>0 then Raise EReadError.Create('TStream.Seek error!');
   Result:=ReadLongInt(S);
  finally
   Kill(S);
  end;
 except
  on E:Exception do BugReport(E,nil,'ReadFileSignature');
 end;
end;

type
 TResourceItem = class(TObject)
 public
  Pos  : Longint;
  Size : Longint;
  Key  : PureString;
 end;

function ReadResourceItem(S:TStream):TResourceItem;
begin
 Result:=TResourceItem.Create;
 try
  Result.Pos:=ReadLongInt(S);
  Result.Size:=ReadLongInt(S);
  Result.Key:=ReadPureString(S);
 except
  Kill(TObject(Result));
  Raise;
 end;
end;

function ReadResourceItemList(S:TStream):TObjectStorage;
var i,Count:SmallInt;
begin
 Result:=NewObjectStorage;
 try
  Count:=ReadSmallInt(S);
  ReadSmallInt(S);
  ReadSmallInt(S);
  for i:=0 to Count-1 do Result.Add(ReadResourceItem(S));
 except
  Kill(Result);
  Raise;
 end;
end;

function ReadResourceItemListFromResource(S:TStream):TObjectStorage;
var Signature,IndexPos:LongInt;
begin
 Result:=nil;
 try
  if S.Seek(0,soFromBeginning)<>0  then Raise EReadError.Create('TStream.Seek error!');
  Signature:=ReadLongInt(S);
  if Signature <> sig_FBPR then Raise EReadError.CreateFmt('Invalid signature %.8x',[Signature]);
  ReadLongInt(S);
  IndexPos:=ReadLongInt(S);
  if S.Seek(IndexPos,soFromBeginning)<>IndexPos then Raise EReadError.Create('TStream.Seek error!');
  Result:=ReadResourceItemList(S);
 except
  Kill(Result);
  Raise;
 end;
end;

function SelectKeyFromResource(const FileName:LongString):LongString;
var
 P : TText;
 i : Integer;
 L : TObjectStorage;
 S : TFileStream;
begin
 Result:='';
 if FileExists(FileName) then
 try
  P:=nil;
  L:=nil;
  S:=nil;
  try
   P:=NewText;
   S:=TFileStream.Create(FileName,fmOpenRead);
   L:=ReadResourceItemListFromResource(S);
   for i:=0 to L.Count-1 do
   with (L[i] as TResourceItem) do if Length(Key)>0 then P.AddLn(Key);
   if P.Count>0 then begin
    i:=ListBoxMenu(Format(RusEng('Ввод из "%s"','Load from "%s"'),[ExtractFileNameExt(FileName)]),
                   RusEng('Выбрать вводимый объект','Choose object to load'),
                   P.Text);
    if i >= 0 then Result:=P[i];
   end;
  finally
   Kill(P);
   Kill(L);
   Kill(S);
  end;
 except
  on E:Exception do BugReport(E,nil,'SelectKeyFromResource');
 end;
end;

function ReadObjectFromResource(const FileName,ObjKey:LongString):TObject;
var
 i : Integer;
 S : TFileStream;
 L : TObjectStorage;
begin
 Result:=nil;
 if FileExists(FileName) then
 if IsNonEmptyStr(ObjKey) then
 try
  S:=nil;
  L:=nil;
  try
   S:=TFileStream.Create(FileName,fmOpenRead);
   L:=ReadResourceItemListFromResource(S);
   for i:=0 to L.Count-1 do
   with (L[i] as TResourceItem) do
   if Key = ObjKey then begin
    if S.Seek(Pos,soFromBeginning)<>Pos then Raise EReadError.Create('TStream.Seek error!');
    Result:=ReadObjectFromStream(S);
    Break;
   end;
  finally
   Kill(S);
   Kill(L);
  end;
 except
  on E:Exception do BugReport(E,nil,'ReadObjectFromResource');
 end;
end;

function ReadText(S:TStream):TText;
var i,Count:SmallInt;
begin
 Result:=NewText;
 try
  Count:=ReadSmallInt(S);
  ReadSmallInt(S);
  ReadSmallInt(S);
  for i:=0 to Count-1 do Result.Addln(ReadPureString(S));
 except
  Kill(Result);
  Raise;
 end;
end;

function ReadCollection(S:TStream):TObjectStorage;
var i,Count:SmallInt;
begin
 Result:=NewObjectStorage;
 try
  Count:=ReadSmallInt(S);
  ReadSmallInt(S);
  ReadSmallInt(S);
  for i:=0 to Count-1 do Result.Add(ReadObjectFromStream(S));
 except
  Kill(Result);
  Raise;
 end;
end;

function ReadVector(S:TStream):TDoubleVector;
type
 Scalar=(mxByte,mxWord,mxSmallInt,mxLongInt,mxCardinal,mxSingle,mxReal48,
         mxDouble,mxComp,mxExtended,mxChar,mxShortInt);
const
 mxComplex = 1;
 mxSparse  = 2;
 mxScalarSize : packed array[Scalar] of Byte=(1,2,2,4,4,4,6,8,8,10,1,1);
var
 i,SeekPos,ISize,DSize:LongInt; Valid:Boolean;
 mx : packed record Typ:Scalar; State:Word; NR,NC,NZMax:LongInt; end;
begin
 Result:=nil;
 SafeFillChar(mx,sizeof(mx),0);
 try
  mx.Typ:=Scalar(ReadByte(S));
  mx.State:=ReadWord(S);
  mx.NR:=ReadLongInt(S);
  mx.NC:=ReadLongInt(S);
  mx.NZMax:=ReadLongInt(S);
  for i:=1 to 4 do ReadLongInt(S);
  with mx do Valid:=(NR>0) and (NC>0) and (NZmax>0)
                     and (((State and mxSparse=0) and (NZmax=NC*NR)) or (State and mxSparse<>0));
  if not Valid then Raise EReadError.Create('Invalid data structure.');
  Result:=NewDoubleVector(mx.NZMax);
  Result.Exceptions:=false;
  for i:=0 to mx.NZMax-1 do
  case mx.Typ of
   mxDouble      : Result[i]:=ReadDouble(S);
   mxSingle      : Result[i]:=ReadSingle(S);
   mxWord        : Result[i]:=ReadWord(S);
   mxSmallInt    : Result[i]:=ReadSmallInt(S);
   mxLongInt     : Result[i]:=ReadLongInt(S);
   mxComp        : Result[i]:=ReadComp(S);
   mxByte,mxChar : Result[i]:=ReadByte(S);
   mxShortInt    : Result[i]:=ReadShortInt(S);
   else Raise EReadError.CreateFmt('Invalid scalar type %d.',[ord(mx.Typ)]);
  end;
  if (mx.State and mxComplex)<>0 then begin
   DSize:=mxScalarSize[mx.Typ]*mx.NZMax;
   SeekPos:=S.Position+DSize;
   if S.Seek(DSize,soFromCurrent) <> SeekPos then Raise EReadError.Create('TStream.Seek error!');
  end;
  if (mx.State and mxSparse)<>0 then begin
   ISize:=SizeOf(LongInt)*mx.NZmax;
   SeekPos:=S.Position+Isize*2;
   if S.Seek(Isize*2,soFromCurrent) <> SeekPos then Raise EReadError.Create('TStream.Seek error!');
  end;
 except
  Kill(Result);
  Raise;
 end;
end;

function ReadCurve(S:TStream):TCurve;
var
 p       : TPoint2D;
 i       : Integer;
 Lim     : TRect2D;
 XData   : TDoubleVector;
 YData   : TDoubleVector;
 Comment : TText;
 function ExtractVar(const VarName:LongString):LongString;
 begin
  if Comment.Ok then Result:=Comment.GetVar(VarName) else Result:='';
 end;
 procedure DeleteVar(const VarName:LongString);
 var i:Integer;
 begin
  if Comment.Ok then
  while true do begin
   i:=Comment.FindVar(VarName);
   if i<0 then Break;
   Comment.DelLn(i);
  end;
 end;
begin
 Result:=NewCurve;
 try
  XData:=nil;
  YData:=nil;
  Comment:=nil;
  try
   Result.Color:=CgaToRGBColor(ReadByte(S));
   Result.Style:=ReadByte(S);
   Lim:=ReadRect2D(S);
   XData:=ReadObjectFromStream(S) as TDoubleVector;
   YData:=ReadObjectFromStream(S) as TDoubleVector;
   Comment:=ReadObjectFromStream(S) as TText;
   Result.Count:=Max(XData.Length,YData.Length);
   for i:=0 to Result.Count-1 do begin
    if XData.Ok
    then p.x:=XData[i]
    else with Lim do p.x:=a.x+(b.x-a.x)*(i-1)/(Result.Count-1);
    if YData.Ok
    then p.y:=YData[i]
    else with Lim do p.y:=a.y+(b.y-a.y)*(i-1)/(Result.Count-1);
    Result[i]:=p;
   end;
   if Comment.Ok then begin
    Result.Name:=ExtractVar('CurveDescription');
    DeleteVar('CurveDescription');
    if Str2Int(ExtractVar('CurveGrowStep'),i) then Result.Step:=i;
    DeleteVar('CurveGrowStep');
   end;
   Result.Comment.Text:=Comment.Text;
  finally
   Kill(XData);
   Kill(YData);
   Kill(Comment);
  end;
 except
  Kill(Result);
  Raise;
 end;
end;

function ReadCurveBox(S:TStream):TFormCurveWindow;
var
 i       : Integer;
 Limits  : TRect2D;
 DefCrv  : Integer;
 Curves  : TObjectStorage;
 Comment : TText;
begin
 Result:=nil;
 try
  Curves:=nil;
  Comment:=nil;
  try
   Result:=NewCurveWindow;
   Result.WindowState:=wsMinimized;
   Result.Left:=ReadSmallInt(S);
   Result.Top:=ReadSmallInt(S);
   Result.Width:=ReadSmallInt(S)-Result.Left;
   Result.Height:=ReadSmallInt(S)-Result.Top;
   Limits:=ReadRect2D(S);
   DefCrv:=ReadSmallInt(S);
   Result.Caption:=ReadPureString(S);
   Result.Title:=ReadPureString(S);
   Result.Legend:=ReadPureString(S);
   Curves:=ReadObjectFromStream(S) as TObjectStorage;
   Comment:=ReadObjectFromStream(S) as TText;
   Result.Curves.OwnsObjects:=false;
   for i:=0 to Curves.Count-1 do Result.AddCurve(Curves[i] as TCurve);
   Result.Curves.OwnsObjects:=true;
   Curves.OwnsObjects:=false;
   Result.Comment.Text:=Comment.Text;
   Result.World:=Limits;
   Result.DefCurveNum:=DefCrv;
   Result.WindowState:=wsNormal;
  finally
   Kill(Curves);
   Kill(Comment);
  end;
 except
  Kill(Result);
  Raise;
 end;
end;

function ReadObjectFromStream(S:TStream):TObject;
var
 Key : SmallInt;
begin
 Result:=nil;
 Key:=ReadSmallInt(S);
 case Key of
  0   : ;
  50  : Result:=ReadCollection(S);
  702 : Result:=ReadVector(S);
  710 : Result:=ReadText(S);
  773 : Result:=ReadCurve(S);
  776 : Result:=ReadCurveBox(S);
  else Raise EReadError.CreateFmt('Unrecognized object type %d.',[Key]);
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_unit_readoldcrwfiles;
begin
end;

procedure Free_unit_readoldcrwfiles;
begin
end;

initialization

 Init_unit_readoldcrwfiles;

finalization

 Free_unit_readoldcrwfiles;

end.

//////////////
// END OF FILE
//////////////

