////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Curve Tools Run Macro Dialog.                                         //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231130 - Modified for FPC (A.K.)                                         //
// 20240626 - ApplyParams                                                     //
////////////////////////////////////////////////////////////////////////////////

unit form_curvetoolsrunmacrodialog; // Form Curve Tools Run Macro Dialog

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons,
 lcltype, lclintf,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_sort,
 _crw_ef, _crw_zm, _crw_curves, _crw_ee,
 _crw_utf8, _crw_sect,
 _crw_appforms, _crw_apptools, _crw_apputils;

type

  { TFormCurveToolsRunMacroDialog }

  TFormCurveToolsRunMacroDialog = class(TMasterForm)
    LabelMacro: TLabel;
    LabelManual: TLabel;
    MemoMacro: TMemo;
    MemoManual: TMemo;
    PanelMacro: TPanel;
    PanelMacroList: TPanel;
    LabelMacroList: TLabel;
    ListBoxMacroList: TListBox;
    PanelButtons: TPanel;
    BitBtnOk: TBitBtn;
    BitBtnCancel: TBitBtn;
    PanelManual: TPanel;
    PanelRight: TPanel;
    procedure FormCreate(Sender: TObject);
    procedure ListBoxMacroListClick(Sender: TObject);
    procedure ListBoxMacroListDblClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure LabelMacroListClick(Sender: TObject);
    procedure LabelManualClick(Sender: TObject);
    procedure LabelMacroClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    ee:TExpressionEvaluator;
    procedure UpdateMacro;
  end;

const
  FormCurveToolsRunMacroDialog : TFormCurveToolsRunMacroDialog = nil;
  DefaultMacroName = 'шаблон';

function CurveToolsRunMacroDialogExecute(const SrcWin,DstWin:TForm; Macro:LongString=DefaultMacroName; const aParams:LongString=''):Integer;

implementation

{$R *.lfm}

uses
 Form_CrwDaq,
 Form_CurveWindow;

 {
 *******************************************************************************
 Запись для передачи параметров вызова функциям и акциям ExpressionEvaluator
 *******************************************************************************
 }
type
 TMacRec = packed record
  SrcWin : TFormCurveWindow;
  DstWin : TFormCurveWindow;
 end;

 {
 *******************************************************************************
 Акции для регистрации в ExpressionEvaluator
 @caption s     Задает имя окна - приемника
 @title   s     Задает заголовок окна - приемника
 @legend  s     Задает легенду окна - приемника
 @error   s     Сообщение об ошибке и остановка скрипта
 @yesno s       Окно с запросом на подтверждение
 @warning s     Окно с предупреждающим сообщением
 @information s Окно с информационным сообщением
 Результат выполнения содержится в переменной actionresult.
 *******************************************************************************
 }
function UnifyArg(ee:TExpressionEvaluator; const s:LongString):LongString;
begin
 Result:=ee.SmartArgs(ReplaceAlignStr(s,true));
end;

function mac_caption(ee:TExpressionEvaluator; const args:LongString):double;
begin
 with TMacRec(ee.Custom^) do begin
  Result:=ord(DstWin.Ok);
  if DstWin.Ok then DstWin.Caption:=UnifyArg(ee,args);
 end;
end;

function mac_title(ee:TExpressionEvaluator; const args:LongString):double;
begin
 with TMacRec(ee.Custom^) do begin
  Result:=ord(DstWin.Ok);
  if DstWin.Ok then DstWin.Title:=UnifyArg(ee,args);
 end;
end;

function mac_legend(ee:TExpressionEvaluator; const args:LongString):double;
begin
 with TMacRec(ee.Custom^) do begin
  Result:=ord(DstWin.Ok);
  if DstWin.Ok then DstWin.Legend:=UnifyArg(ee,args);
 end;
end;

function mac_error(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=1;
 if IsNonEmptyStr(args) then Error(UnifyArg(ee,args));
 ee.RuntimeError:=ee_User;
end;

function mac_yesno(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=ord(YesNo(UnifyArg(ee,args))=mrYes);
end;

function mac_warning(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=ord(Warning(UnifyArg(ee,args))=mrOk);
end;

function mac_information(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=ord(Information(UnifyArg(ee,args))=mrOk);
end;

function mac_echo(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=1;
 Echo(ee.SmartArgs(args));
end;

procedure RegisterMacroActions(ee:TExpressionEvaluator);
begin
 ee.SetAction('caption',     mac_caption,     RusEng('Задает имя окна',
                                                     'Set window name'));
 ee.SetAction('title',       mac_title,       RusEng('Задает заголовок окна',
                                                     'Set window title'));
 ee.SetAction('legend',      mac_legend,      RusEng('Задает легенду окна',
                                                     'Set window legend'));
 ee.SetAction('error',       mac_error,       RusEng('Сообщение об ошибке и выход из макроса',
                                                     'Show error message end abort macros execution'));
 ee.SetAction('yesno',       mac_yesno,       RusEng('Запрос на подтверждение',
                                                     'Show Yes/No dialog'));
 ee.SetAction('warning',     mac_warning,     RusEng('Предупреждающее сообщение пользователю',
                                                     'Show warning dialog'));
 ee.SetAction('information', mac_information, RusEng('Информационное сообщение пользователю',
                                                     'Show information dialog'));
 ee.SetAction('echo',        mac_echo,        RusEng('Эхо - вывод сообщения в консольное окно',
                                                     'Echo message to console window'));
end;

 {
 *******************************************************************************
 Функции для регистрации в ExpressionEvaluator
 iocount(a)             Число кривых в окне источника(a>0)/приемника(a<0)/буфера(a=0)
 ioselected(a)          Номер выделенной кривой в окне источника(a>0)/приемника(a<0)/буфера(a=0)
 ioselect(a)            Выделить кривую а в окне источника(a>0)/приемника(a<0)
 ioroix(a)              Чтение х-координаты РОИ-маркера номер a
 ioroiy(a)              Чтение y-координаты РОИ-маркера номер a
 iosetroi(a,b,c)        Установить координаты РОИ-маркера номер a равными (b,c)
 iox(a,b)               Координата x точки номер b кривой a в окне источника(a>0)/приемника(a<0)/буфера(a=0)
 ioy(a,b)               Координата y точки номер b кривой a в окне источника(a>0)/приемника(a<0)/буфера(a=0)
 iolen(a)               Длина кривой a в окне источника(a>0)/приемника(a<0)
 iosetlen(a)            Установить длину b кривой a в окне источника(a>0)/приемника(a<0)/буфера(a=0)
 ioadd(a,b,c)           Добавить точку (b,c) к кривой a в окне источника(a>0)/приемника(a<0)/буфера(a=0)
 ioput(a,b,c,d)         Задать точку (c,d) номер b в кривой a в окне источника(a>0)/приемника(a<0)/буфера(a=0)
 iocreate(a,b)          Создать кривую цвета a, стиля b и вернуть ее номер<0
 ioclone(a)             Создать копию кривой a из окна источника(a>0)/приемника(a<0)/буфера(a=0) и вернуть ее номер<0
 iokill(a)              Уничтожить кривую a из окна источника(a>0)/приемника(a<0)/буфера(a=0)
 iosort(a,b,c,d,e)      Сортировка кривой a, возвращает номер кривой <0, b=Flags,c=AbsEps,d=RelEps,e=method
 iosorted(a,b)          Проверяет, отсортирована ли кривая a, а при  b<>0 проверяет также,нет ли дубликатов по X.
 iofindindex(a,b)       Быстро найти номер ближайшей точки кривой a в точке b. Кривая должна быть отсортирована по X.
 iointerpol(a,b)        Линейная интерполяция кривой a в точке b. Кривая должна быть отсортирована по X.
 iosmooth(a,b,c,d,e,f)  Сглаживание кривой a в точке b c окном c степени d и ядром (1-e)^f.Кривая должна быть отсортирована по X.
 iomediana(a,b,c)       Найти индекс медианы массива y-значений кривой a в интервале индексов (b..c).
 *******************************************************************************
 }
function iowin(ee:TExpressionEvaluator; const n:Double):TFormCurveWindow;
begin
 with TMacRec(ee.Custom^) do begin
  if n>0 then Result:=SrcWin else
  if n<0 then Result:=DstWin else Result:=CurveClipboard;
  if not Assigned(Result)
  then mac_error(ee,RusEng('Недопустимый номер кривой ','Invalid curve number ')+d2s(round(n))+'!');
 end;
end;

function iocrv(ee:TExpressionEvaluator; const n:Double):TCurve;
begin
 if n<>0
 then Result:=iowin(ee,n).Curves[abs(round(n))-1]
 else Result:=iowin(ee,n).DefCurve;
 if not Assigned(Result)
 then mac_error(ee,RusEng('Недопустимый номер кривой ','Invalid curve number ')+d2s(round(n))+'!');
end;

procedure iocheck(ee:TExpressionEvaluator; i,j:LongInt);
begin
 if i<>j then mac_error(ee,RusEng('Неверное число аргументов функции!','Invalid function argument number!'));
end;

procedure iocheckindex(ee:TExpressionEvaluator; const n,m:Double);
begin
 if Cardinal(iocrv(ee,n).Count)<=Cardinal(round(m))
 then mac_error(ee,RusEng('Неверный индекс точки кривой!','Invalid curve point index!'));
end;

function mac_iocount(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=iowin(ee,x[0]).Curves.Count;
end;

function mac_ioselected(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=iowin(ee,x[0]).DefCurveNum+1;
end;

function mac_ioselect(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 iowin(ee,x[0]).DefCurveNum:=abs(round(x[0]))-1;
 Result:=iowin(ee,x[0]).DefCurveNum+1;
end;

function mac_ioroix(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 with iowin(ee,x[0]) do
 case abs(round(x[0])) of
  1  : Result:=Roi.A.X;
  2  : Result:=Roi.B.X;
  else Result:=_Nan;
 end;
end;

function mac_ioroiy(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 with iowin(ee,x[0]) do
 case abs(round(x[0])) of
  1  : Result:=Roi.A.Y;
  2  : Result:=Roi.B.Y;
  else Result:=_Nan;
 end;
end;

function mac_iosetroi(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 Result:=1;
 iocheck(ee,narg,3);
 with iowin(ee,x[0]) do
 case abs(round(x[0])) of
  1  : Roi:=Rect2D(x[1], x[2], Roi.B.X, Roi.B.Y);
  2  : Roi:=Rect2D(Roi.A.X, Roi.A.Y, x[1], x[2]);
  else Result:=0;
 end;
end;

function mac_iox(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 iocheckindex(ee,x[0],x[1]);
 Result:=iocrv(ee,x[0])[round(x[1])].x;
end;

function mac_ioy(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 iocheckindex(ee,x[0],x[1]);
 Result:=iocrv(ee,x[0])[round(x[1])].y;
end;

function mac_iolen(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=iocrv(ee,x[0]).Count;
end;

function mac_iosetlen(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 iocrv(ee,x[0]).Count:=round(x[1]);
 Result:=iocrv(ee,x[0]).Count;
end;

function mac_ioadd(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,3);
 iocrv(ee,x[0]).AddPoint(x[1],x[2]);
 Result:=1;
end;

function mac_ioput(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,4);
 iocheckindex(ee,x[0],x[1]);
 iocrv(ee,x[0])[round(x[1])]:=Point2D(x[2],x[3]);
 Result:=1;
end;

function mac_iocreate(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 iowin(ee,-1).AddCurve(NewCurve(0,'',round(x[0]),round(x[1])));
 Result:=-iowin(ee,-1).Curves.Count;
end;

function mac_ioclone(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 iowin(ee,-1).AddCurve(iocrv(ee,x[0]).Clone);
 Result:=-iowin(ee,-1).Curves.Count;
end;

function mac_iokill(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 iowin(ee,x[0]).DeleteCurve(iocrv(ee,x[0]));
 Result:=1;
end;

function mac_iosort(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,5);
 iowin(ee,-1).AddCurve(NewCurveSorted(iocrv(ee,x[0]),round(x[1]),x[2],x[3],TSortMethod(round(x[4]))));
 Result:=-iowin(ee,-1).Curves.Count;
end;

function mac_iosorted(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
const Opt = cfInvalid+cfNoData+cfTooSmall+cfNotSortedX;
begin
 iocheck(ee,narg,2);
 Result:=ord(iocrv(ee,x[0]).Flags and (Opt+cfDuplicatesX*ord(x[1]<>0))=0);
end;

function mac_iofindindex(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=iocrv(ee,x[0]).GetIndexAt(x[1]);
end;

function mac_iointerpol(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=iocrv(ee,x[0]).Interpolate(x[1]);
end;

function mac_iosmooth(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,6);
 Result:=iocrv(ee,x[0]).Smooth(x[1],x[2],round(x[3]),round(x[4]),round(x[5]));
end;

function mac_iominx(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=iocrv(ee,x[0]).Limits.A.X;
end;

function mac_iomaxx(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=iocrv(ee,x[0]).Limits.B.X;
end;

function mac_iominy(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=iocrv(ee,x[0]).Limits.A.Y;
end;

function mac_iomaxy(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=iocrv(ee,x[0]).Limits.B.Y;
end;

function mac_iomediana(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var i:LongInt; v:Double;
begin
 i:=0; v:=0;
 iocheck(ee,narg,3);
 iocheckindex(ee,x[0],x[1]);
 iocheckindex(ee,x[0],x[2]);
 if not iocrv(ee,x[0]).Mediana(round(x[1]),round(x[2]),i,v)
 then mac_error(ee,RusEng('Ошибка вычисления медианы:неверные индексы или мало памяти!',
                          'Median fails:invalid indexes or out of memory!'));
 Result:=i;
end;

procedure RegisterMacroFunctions(ee:TExpressionEvaluator);
begin
 ee.SetFunc('iocount',     1, mac_iocount,     RusEng('Число кривых в окне источника(a>0)/приемника(a<0)/буфера(a=0).',
                                                      'Number of curves in source(a>0)/dest(a<0)/clipboard(a=0) window.'));
 ee.SetFunc('ioselected',  1, mac_ioselected,  RusEng('Номер выделенной кривой в окне источника(a>0)/приемника(a<0)/буфера(a=0).',
                                                      'Index of selected curve in source(a>0)/dest(a<0)/clipboard(a=0) window.'));
 ee.SetFunc('ioselect',    1, mac_ioselect,    RusEng('Выделить кривую а в окне источника(a>0)/приемника(a<0).',
                                                      'Select curve in source(a>0)/dest(a<0) window.'));
 ee.SetFunc('ioroix',      1, mac_ioroix,      RusEng('Чтение x-координаты РОИ-маркера номер a.',
                                                      'Read X-coordinate of ROI marker number a.'));
 ee.SetFunc('ioroiy',      1, mac_ioroiy,      RusEng('Чтение y-координаты РОИ-маркера номер a.',
                                                      'Read Y-coordinate of ROI marker number a.'));
 ee.SetFunc('iosetroi',    3, mac_iosetroi,    RusEng('Установить координаты РОИ-маркера номер a равными (b,c).',
                                                      'Set ROI marker number a to (x,y)=(b,c).'));
 ee.SetFunc('iox',         2, mac_iox,         RusEng('Координата x точки номер b кривой a в окне источника(a>0)/приемника(a<0)/буфера(a=0).',
                                                      'X-coordinate of curve a point number b.'));
 ee.SetFunc('ioy',         2, mac_ioy,         RusEng('Координата y точки номер b кривой a в окне источника(a>0)/приемника(a<0)/буфера(a=0).',
                                                      'Y-coordinate of curve a point number b.'));
 ee.SetFunc('iolen',       1, mac_iolen,       RusEng('Длина кривой a в окне источника(a>0)/приемника(a<0)/буфера(a=0).',
                                                      'Length (number of points) of curve a.'));
 ee.SetFunc('iosetlen',    2, mac_iosetlen,    RusEng('Установить длину b кривой a в окне источника(a>0)/приемника(a<0)/буфера(a=0).',
                                                      'Set new length b of curve a.'));
 ee.SetFunc('ioadd',       3, mac_ioadd,       RusEng('Добавить точку (b,c) к кривой a в окне источника(a>0)/приемника(a<0)/буфера(a=0).',
                                                      'Add point (b,c) to curve number a.'));
 ee.SetFunc('ioput',       4, mac_ioput,       RusEng('Задать точку (c,d) номер b в кривой a в окне источника(a>0)/приемника(a<0)/буфера(a=0).',
                                                      'Put (c,d) to point number b of curve a.'));
 ee.SetFunc('iocreate',    2, mac_iocreate,    RusEng('Создать кривую цвета a, стиля b и вернуть ее номер<0.',
                                                      'Create curve with color a, style b, return curve number.'));
 ee.SetFunc('ioclone',     1, mac_ioclone,     RusEng('Создать копию кривой a из окна источника(a>0)/приемника(a<0)/буфера(a=0) и вернуть ее номер<0.',
                                                      'Create copy of curve a and return number of curve.'));
 ee.SetFunc('iokill',      1, mac_iokill,      RusEng('Уничтожить кривую a из окна источника(a>0)/приемника(a<0)/буфера(a=0).',
                                                      'Destroy curve number a.'));
 ee.SetFunc('iosort',      5, mac_iosort,      RusEng('Сортировка кривой a, возвращает номер кривой <0, b=Flags,c=AbsEps,d=RelEps,e=method.',
                                                      'Sort curve a, b=Flags,c=AbsEps,d=RelEps,e=method, return result curve number.'));
 ee.SetFunc('iosorted',    2, mac_iosorted,    RusEng('Проверяет, отсортирована ли кривая a; при  b<>0 проверяет также,нет ли дубликатов по X.',
                                                      'Check if curve a sorted or not, and if b>0, check dublicate points.'));
 ee.SetFunc('iofindindex', 2, mac_iofindindex, RusEng('Быстро найти номер ближайшей точки кривой a в точке b. Кривая должна быть отсортирована по X.',
                                                      'Quick find index of nearest point of curve a in x=b.'));
 ee.SetFunc('iointerpol',  2, mac_iointerpol,  RusEng('Линейная интерполяция кривой a в точке b. Кривая должна быть отсортирована по X.',
                                                      'Linear interpolation of curve a in point b.'));
 ee.SetFunc('iosmooth',    6, mac_iosmooth,    RusEng('Сглаживание кривой a в точке b c окном c степени d и ядром (1-e)^f.Кривая должна быть отсортирована по X.',
                                                      'Curve a smoothing at point b, window c, polynom power d, kernel (1-e)^f.'));
 ee.SetFunc('iominx',      1, mac_iominx,      RusEng('Минимальное значение x кривой a.',
                                                      'Min. X-coordinate value of curve a.'));
 ee.SetFunc('iomaxx',      1, mac_iomaxx,      RusEng('Максимальное значение x кривой a.',
                                                      'Max. X-coordinate value of curve a.'));
 ee.SetFunc('iominy',      1, mac_iominy,      RusEng('Минимальное значение y кривой a.',
                                                      'Min. Y-coordinate value of curve a.'));
 ee.SetFunc('iomaxy',      1, mac_iomaxy,      RusEng('Максимальное значение y кривой a.',
                                                      'Min. Y-coordinate value of curve a.'));
 ee.SetFunc('iomediana',   3, mac_iomediana,   RusEng('Найти индекс медианы массива y-значений кривой a в интервале индексов (b..c).',
                                                      'Find index of median of curve a y-array in index range (b..c).'));
end;

procedure AddMacroToList(Index:LongInt; const TextLine:LongString; var Terminate:boolean; CustomData:Pointer);
var p:Integer; sn,sv:LongString;
begin
 if not Assigned(CustomData) then Exit;
 p:=ExtractNameValuePair(TextLine,sn,sv);
 if (p>0) and SameText(sn,'MACRO') and (sv<>'') then TText(CustomData).Addln(sv);
end;

function LoadMacroList:LongString;
var Path:LongString; p,m:TText;
begin
 p:=nil;
 m:=NewText; Path:='';
 if ReadIniFilePath(SysIniFile,SectSystem,'CurveToolsMacro',HomeDir,Path)
 then p:=ExtractListSection(Path,UnifySection('[MacroList]'),efAsIs+efDelCom);
 p.ForEach(AddMacroToList,m);
 Result:=m.Text;
 Kill(m);
 Kill(p);
end;

function LoadMacro(const Macro:LongString):LongString;
var Path:LongString;
begin
 Path:='';
 if ReadIniFilePath(SysIniFile,SectSystem,'CurveToolsMacro',HomeDir,Path)
 then Result:=ExtractTextSection(Path,UnifySection('Macro '+Macro),efAsIs);
end;

function LoadManual(const Macro:LongString):LongString;
var Path:LongString;
begin
 Path:='';
 if ReadIniFilePath(SysIniFile,SectSystem,'CurveToolsMacro',HomeDir,Path)
 then Result:=ExtractTextSection(Path,UnifySection('Manual '+Macro),efAsIs);
end;

procedure TFormCurveToolsRunMacroDialog.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 RestoreFont(ListBoxMacroList.Font,DefaultSansFont);
 Caption:=RusEng('Диалог для запуска макроса обработки данных','Dialog to run data processing macro');
 SmartUpdate(LabelMacroList,RusEng('Список макросов:','List of macros:'));
 SmartUpdate(LabelManual,RusEng('Краткое описание макроса:','Short manual:'));
 SmartUpdate(LabelMacro,RusEng('Текст макроса:','Text of macros:'));
 SmartUpdate(BitBtnOk,mrCaption(mrOk));
 SmartUpdate(BitBtnCancel,mrCaption(mrCancel));
 ee:=NewExpressionEvaluator;
 RegisterMacroActions(ee);
 RegisterMacroFunctions(ee);
end;

procedure TFormCurveToolsRunMacroDialog.FormDestroy(Sender: TObject);
begin
 Kill(ee);
end;

procedure TFormCurveToolsRunMacroDialog.UpdateMacro;
var Macro:LongString; p:TText;
begin
 if InRange(ListBoxMacroList.ItemIndex,0,ListBoxMacroList.Items.Count-1)
 then Macro:=UnifyAlias(ListBoxMacroList.Items[ListBoxMacroList.ItemIndex])
 else Macro:='';
 if Macro<>'' then begin
  p:=NewText;
  p.Text:=LoadManual(Macro);
  p.AddLn('');
  p.AddLn(RusEng('Список функций:','List of functions:'));
  p.AddLn(Pad('',length(p[p.Count-1]),'*'));
  ee.FuncList.GetText(p);
  p.AddLn('');
  p.AddLn(RusEng('Список акций:','List of actions:'));
  p.AddLn(Pad('',length(p[p.Count-1]),'*'));
  ee.ActionList.GetText(p);
  p.AddLn('');
  p.AddLn(RusEng('Список констант:','List of constans:'));
  p.AddLn(Pad('',length(p[p.Count-1]),'*'));
  ee.ConstList.GetText(p);
  SmartUpdate(MemoManual,utf8_fixbroken(p.Text));
  Kill(p);
  SmartUpdate(MemoMacro,LoadMacro(Macro));
 end else begin
  SmartUpdate(MemoManual,'?');
  SmartUpdate(MemoMacro,'?');
 end;
end;

function ExecuteMacro(ee:TExpressionEvaluator; const SrcWin,DstWin:TForm; const Macro:LongString):Boolean;
var MacRec:TMacRec;
begin
 Result:=false;
 if (SrcWin is TFormCurveWindow) and (DstWin is TFormCurveWindow) then begin
  MacRec.SrcWin:=SrcWin as TFormCurveWindow;
  MacRec.DstWin:=DstWin as TFormCurveWindow;
  ee.Custom:=@MacRec;
  ee.VarList.Clear;
  ee.Script:=Macro;
  try
   MacRec.DstWin.LockDraw;
   Result:=(ee.RunScript=ee_Ok);
   MacRec.DstWin.AutoRange;
  finally
   MacRec.DstWin.UnlockDraw;
  end;
  if not Result and (ee.Status<>ee_User)
  then Error(ee_ErrorMessage(ee.Status)+'.'+EOL+
             StrPas(ee.Buffer)+EOL+
             CharStr(ee.ErrorPos+1)+'^'+EOL+
             CharStr(ee.ErrorPos+1-Length(ee.ErrorToken))+
             StrPas(ee.ErrorToken));
 end;
end;

procedure TFormCurveToolsRunMacroDialog.ListBoxMacroListClick(Sender: TObject);
begin
 UpdateMacro;
end;

procedure TFormCurveToolsRunMacroDialog.ListBoxMacroListDblClick(Sender: TObject);
begin
 ModalResult:=mrOk;
end;

function CurveToolsRunMacroDialogExecute(const SrcWin,DstWin:TForm; Macro:LongString=DefaultMacroName; const aParams:LongString=''):Integer;
var i:LongInt; apFlags:Integer;
begin
 Result:=mrCancel;
 if CanShowModal(FormCurveToolsRunMacroDialog) then
 if (SrcWin is TFormCurveWindow) and (DstWin is TFormCurveWindow) then
 try
  if not Assigned(FormCurveToolsRunMacroDialog)  then begin
   Application.CreateForm(TFormCurveToolsRunMacroDialog, FormCurveToolsRunMacroDialog);
   FormCurveToolsRunMacroDialog.Master:=@FormCurveToolsRunMacroDialog;
  end;
  if Assigned(FormCurveToolsRunMacroDialog) then with FormCurveToolsRunMacroDialog do begin
   apFlags:=ApplyParams(aParams);
   if not HasFlags(apFlags,apf_FormPos)
   then LocateFormToCenterOfScreen(FormCurveToolsRunMacroDialog);
   ListBoxMacroList.Items.Text:=LoadMacroList;
   ListBoxMacroList.ItemIndex:=Min(0,ListBoxMacroList.Count-1);
   for i:=0 to ListBoxMacroList.Items.Count-1 do
   if SameText(UnifyAlias(ListBoxMacroList.Items[i]),UnifyAlias(Macro)) then begin
    ListBoxMacroList.ItemIndex:=i;
    break;
   end;
   UpdateMacro;
   if mrVoice(ShowModal)=mrOk then begin
    DstWin.Update;
    SrcWin.Update;
    FormCrwDaq.Update;
    if ExecuteMacro(ee,SrcWin,DstWin,MemoMacro.Text) then Result:=mrOk;
    if (Result=mrOk) then DstWin.BringToFront;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'CurveToolsRunMacroDialogExecute'); 
 end;
end;

procedure TFormCurveToolsRunMacroDialog.LabelMacroListClick(Sender: TObject);
begin
 SmartFocus(ListBoxMacroList);
end;

procedure TFormCurveToolsRunMacroDialog.LabelManualClick(Sender: TObject);
begin
 SmartFocus(MemoManual);
end;

procedure TFormCurveToolsRunMacroDialog.LabelMacroClick(Sender: TObject);
begin
 SmartFocus(MemoMacro);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_curvetoolsrunmacrodialog;
begin
end;

procedure Free_form_curvetoolsrunmacrodialog;
begin
end;

initialization

 Init_form_curvetoolsrunmacrodialog;

finalization

 Free_form_curvetoolsrunmacrodialog;

end.

//////////////
// END OF FILE
//////////////

