////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Curve Range Selector.                                                 //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231124 - Modified for FPC (A.K.)                                         //
// 20240626 - ApplyParams                                                     //
////////////////////////////////////////////////////////////////////////////////

unit form_curverangeselector; // Form Curve Range Selector

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons,
 lcltype, lclintf,
 _crw_alloc, _crw_fpu, _crw_str, _crw_rtc, _crw_fifo,
 _crw_eldraw, _crw_fio, _crw_plut, _crw_snd, _crw_ef,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormCurveRangeSelector = class(TMasterForm)
    PanelControls: TPanel;
    PanelButtons: TPanel;
    BitBtnOk: TBitBtn;
    BitBtnCancel: TBitBtn;
    PageControlAxisParams: TPageControl;
    TabSheetAxisLimits: TTabSheet;
    LabelX: TLabel;
    LabelX1: TLabel;
    EditX1: TEdit;
    LabelX2: TLabel;
    EditX2: TEdit;
    LabelY: TLabel;
    LabelY1: TLabel;
    EditY1: TEdit;
    LabelY2: TLabel;
    EditY2: TEdit;
    TabSheetTimeAxisX: TTabSheet;
    LabelTimeBase: TLabel;
    EditTimeBase: TEdit;
    LabelTimeBaseComment: TLabel;
    LabelTimeUnits: TLabel;
    EditTimeUnits: TEdit;
    ButtonTimeUnitsHour: TButton;
    ButtonTimeUnitsMin: TButton;
    ButtonTimeUnitsSec: TButton;
    ButtonTimeUnitsDay: TButton;
    ButtonTimeBaseMidnight: TButton;
    ButtonTimeFromDAQ: TButton;
    ButtonTimeUnitsMsec: TButton;
    ButtonRestore: TButton;
    PanelRangeStatus: TPanel;
    TimerCheckParams: TTimer;
    PanelTimeStatus: TPanel;
    ButtonTimeScaleOff: TButton;
    procedure FormCreate(Sender: TObject);
    procedure LabelX1Click(Sender: TObject);
    procedure LabelY1Click(Sender: TObject);
    procedure LabelX2Click(Sender: TObject);
    procedure LabelY2Click(Sender: TObject);
    procedure ButtonTimeUnitsHourClick(Sender: TObject);
    procedure ButtonTimeUnitsMinClick(Sender: TObject);
    procedure ButtonTimeUnitsSecClick(Sender: TObject);
    procedure ButtonTimeUnitsDayClick(Sender: TObject);
    procedure ButtonTimeBaseMidnightClick(Sender: TObject);
    procedure ButtonTimeFromDAQClick(Sender: TObject);
    procedure ButtonTimeUnitsMsecClick(Sender: TObject);
    procedure ButtonRestoreClick(Sender: TObject);
    procedure LabelTimeUnitsClick(Sender: TObject);
    procedure LabelTimeBaseClick(Sender: TObject);
    procedure LabelTimeBaseCommentClick(Sender: TObject);
    procedure TimerCheckParamsTimer(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure FormDeactivate(Sender: TObject);
    procedure ButtonTimeScaleOffClick(Sender: TObject);
    procedure FormHide(Sender: TObject);
  private
    { Private declarations }
    myStored : record Limits : TRect2d; TimeBase,TimeUnits : Double; end;
    procedure StoreParams(Limits:TRect2D; TimeBase,TimeUnits:Double);
    procedure RestoreParams(Limits:TRect2D; TimeBase,TimeUnits:Double);
    function  DecodeRange(var aRange:TRect2D):Boolean;
    function  DecodeTimeScale(var aTimeBase,aTimeUnits:Double):Boolean;
  public
    { Public declarations }
  end;

function CurveRangeSelectorDialog(var Range:TRect2D; var TimeBase,TimeUnits:Double; const aParams:LongString=''):LongInt;

const
 CurveRangeSelectorDefaults : record
  DAQ : record TimeBase,TimeUnits: Double; end;
 end = (
  DAQ : ( TimeBase:0; TimeUnits: 0)
 );

implementation

uses Form_CurveWindow;

{$R *.lfm}

const
  FormCurveRangeSelector : TFormCurveRangeSelector = nil;

function FormatTimeMs(ms:Double):LongString;
begin
 if IsNanOrInf(ms)
 then Result:=Format('%g',[ms])
 else Result:=GetDateStr(ms,'.',true,false)+'-'+GetTimeStr(ms,':',true);
end;

function DecodeTime(s:LongString):Double;
var year,month,day,hour,min,sec,msec:Integer; Delims:TCharSet;
begin
 Result:=_NaN;
 try
  Delims:=ScanSpaces+['.',':','-'];
  year  :=iValDef(ExtractWord(1,s,Delims),-1); if year<0  then Exit;
  month :=iValDef(ExtractWord(2,s,Delims),-1); if month<0 then Exit;
  day   :=iValDef(ExtractWord(3,s,Delims),-1); if day<0   then Exit;
  hour  :=iValDef(ExtractWord(4,s,Delims),-1); if hour<0  then Exit;
  min   :=iValDef(ExtractWord(5,s,Delims),-1); if min<0   then Exit;
  sec   :=iValDef(ExtractWord(6,s,Delims),-1); if sec<0   then Exit;
  msec  :=iValDef(ExtractWord(7,s,Delims),0);
  Result:=DateTimeToMSec(year,month,day,hour,min,sec,msec);
 except
  on E:Exception do BugReport(E,nil,'DecodeTime');
 end;
end;

function CurveRangeSelectorDialog(var Range:TRect2D; var TimeBase,TimeUnits:Double; const aParams:LongString=''):LongInt;
var R1:TRect2D; Tb,Tu:Double; apFlags:Integer;
begin
 Result:=mrCancel;
 if CanShowModal(FormCurveRangeSelector) then
 try
  if not Assigned(FormCurveRangeSelector) then begin
   Application.CreateForm(TFormCurveRangeSelector, FormCurveRangeSelector);
   FormCurveRangeSelector.Master:=@FormCurveRangeSelector;
  end;
  if Assigned(FormCurveRangeSelector) then begin
   SetStandardFont(FormCurveRangeSelector);
   apFlags:=FormCurveRangeSelector.ApplyParams(aParams);
   if not HasFlags(apFlags,apf_FormPos)
   then LocateFormToCenterOfScreen(FormCurveRangeSelector);
   FormCurveRangeSelector.Caption:=RusEng('Пределы Графика','Plot Ranges');
   FormCurveRangeSelector.TabSheetAxisLimits.Caption:=RusEng('Пределы по X,Y','X,Y Axis Range');
   FormCurveRangeSelector.TabSheetTimeAxisX.Caption:=RusEng('Шкала Времени','Time Scale');
   FormCurveRangeSelector.LabelX.Caption:=RusEng('Пределы по X:','Range along X:');
   FormCurveRangeSelector.LabelY.Caption:=RusEng('Пределы по Y:','Range along Y:');
   FormCurveRangeSelector.LabelX1.Caption:=RusEng('От','From');
   FormCurveRangeSelector.LabelY1.Caption:=RusEng('От','From');
   FormCurveRangeSelector.LabelX2.Caption:=RusEng('До','To');
   FormCurveRangeSelector.LabelY2.Caption:=RusEng('До','To');
   FormCurveRangeSelector.LabelTimeBase.Caption:=RusEng('Базовое Время (начало отсчета)','Time Base (beginning)');
   FormCurveRangeSelector.LabelTimeBaseComment.Caption:=RusEng('Типа: ','Like: ')+FormatTimeMs(mSecNow);
   FormCurveRangeSelector.LabelTimeUnits.Caption:=RusEng('Единицы Времени, мс','Time Units, ms');
   FormCurveRangeSelector.ButtonTimeUnitsDay.Caption:=RusEng('День','Day');
   FormCurveRangeSelector.ButtonTimeUnitsHour.Caption:=RusEng('Час','Hour');
   FormCurveRangeSelector.ButtonTimeUnitsMin.Caption:=RusEng('Мин','Min');
   FormCurveRangeSelector.ButtonTimeUnitsSec.Caption:=RusEng('Сек','Sec');
   FormCurveRangeSelector.ButtonTimeUnitsMsec.Caption:=RusEng('мСек','mSec');
   FormCurveRangeSelector.ButtonTimeBaseMidnight.Caption:=RusEng('Полночь','Midnight');
   FormCurveRangeSelector.ButtonTimeScaleOff.Caption:=RusEng('Отключить Шкалу Времени.','Switch OFF Time Scale.');
   FormCurveRangeSelector.ButtonRestore.Caption:=RusEng('Вернуть','Restore');
   FormCurveRangeSelector.ButtonTimeFromDAQ.Caption:=RusEng('Шкала Времени DAQ','Time from DAQ');
   FormCurveRangeSelector.PageControlAxisParams.ActivePageIndex:=0;
   FormCurveRangeSelector.StoreParams(Range,TimeBase,TimeUnits);
   FormCurveRangeSelector.RestoreParams(Range,TimeBase,TimeUnits);
   R1:=Rect2D(0,0,0,0); Tb:=0; Tu:=0;
   if mrVoice(FormCurveRangeSelector.ShowModal)=mrOk then begin
    if FormCurveRangeSelector.DecodeRange(R1) then begin
     if FormCurveRangeSelector.EditX1.Modified then Range.A.X:=R1.A.X;
     if FormCurveRangeSelector.EditY1.Modified then Range.A.Y:=R1.A.Y;
     if FormCurveRangeSelector.EditX2.Modified then Range.B.X:=R1.B.X;
     if FormCurveRangeSelector.EditY2.Modified then Range.B.Y:=R1.B.Y;
     FormCurveRangeSelector.DecodeTimeScale(Tb,Tu);
     if FormCurveRangeSelector.EditTimeBase.Modified then TimeBase:=Tb;
     if FormCurveRangeSelector.EditTimeUnits.Modified then TimeUnits:=Tu;
     Result:=mrOk;
    end else Error(RusEng('Введен недопустимый Диапазон!','Invalid Range specified!'));
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'CurveRangeSelectorDialog');
 end;
end;

procedure TFormCurveRangeSelector.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 SmartUpdate(BitBtnOk,mrCaption(mrOk));
 SmartUpdate(BitBtnCancel,mrCaption(mrCancel));
end;

procedure TFormCurveRangeSelector.LabelX1Click(Sender: TObject);
begin
 SmartFocus(EditX1);
end;

procedure TFormCurveRangeSelector.LabelY1Click(Sender: TObject);
begin
 SmartFocus(EditY1);
end;

procedure TFormCurveRangeSelector.LabelX2Click(Sender: TObject);
begin
 SmartFocus(EditX2);
end;

procedure TFormCurveRangeSelector.LabelY2Click(Sender: TObject);
begin
 SmartFocus(EditY2);
end;

procedure TFormCurveRangeSelector.LabelTimeBaseClick(Sender: TObject);
begin
 SmartFocus(EditTimeBase);
end;

procedure TFormCurveRangeSelector.LabelTimeBaseCommentClick(Sender: TObject);
begin
 SmartFocus(EditTimeBase);
end;

procedure TFormCurveRangeSelector.LabelTimeUnitsClick(Sender: TObject);
begin
 SmartFocus(EditTimeUnits);
end;

procedure TFormCurveRangeSelector.ButtonTimeUnitsMsecClick(Sender: TObject);
begin
 EditTimeUnits.Text:=Format('%d',[1]);
 EditTimeUnits.Modified:=true;
end;

procedure TFormCurveRangeSelector.ButtonTimeUnitsSecClick(Sender: TObject);
begin
 EditTimeUnits.Text:=Format('%d',[1000]);
 EditTimeUnits.Modified:=true;
end;

procedure TFormCurveRangeSelector.ButtonTimeUnitsMinClick(Sender: TObject);
begin
 EditTimeUnits.Text:=Format('%d',[60*1000]);
 EditTimeUnits.Modified:=true;
end;

procedure TFormCurveRangeSelector.ButtonTimeUnitsHourClick(Sender: TObject);
begin
 EditTimeUnits.Text:=Format('%d',[60*60*1000]);
 EditTimeUnits.Modified:=true;
end;

procedure TFormCurveRangeSelector.ButtonTimeUnitsDayClick(Sender: TObject);
begin
 EditTimeUnits.Text:=Format('%d',[24*60*60*1000]);
 EditTimeUnits.Modified:=true;
end;

procedure TFormCurveRangeSelector.ButtonTimeBaseMidnightClick(Sender: TObject);
begin
 EditTimeBase.Text:=FormatTimeMs(GetMidnightByTimeMs(mSecNow));
 EditTimeBase.Modified:=true;
end;

procedure TFormCurveRangeSelector.ButtonTimeFromDAQClick(Sender: TObject);
begin
 if CurveRangeSelectorDefaults.DAQ.TimeBase>0 then
 if CurveRangeSelectorDefaults.DAQ.TimeUnits>0 then begin
  EditTimeBase.Text:=FormatTimeMs(CurveRangeSelectorDefaults.DAQ.TimeBase);
  EditTimeUnits.Text:=Format('%g',[CurveRangeSelectorDefaults.DAQ.TimeUnits]);
  EditTimeBase.Modified:=true;
  EditTimeUnits.Modified:=true;
 end;
end;

procedure TFormCurveRangeSelector.ButtonTimeScaleOffClick(Sender: TObject);
begin
 EditTimeBase.Text:=Format('%g',[_NaN]);  EditTimeBase.Modified:=true;
 EditTimeUnits.Text:=Format('%g',[_NaN]); EditTimeUnits.Modified:=true;
end;

procedure TFormCurveRangeSelector.ButtonRestoreClick(Sender: TObject);
begin
 RestoreParams(myStored.Limits,myStored.TimeBase,myStored.TimeUnits);
end;

procedure TFormCurveRangeSelector.StoreParams(Limits:TRect2D; TimeBase,TimeUnits:Double);
begin
 myStored.Limits:=Limits; myStored.TimeBase:=TimeBase; myStored.TimeUnits:=TimeUnits;
end;

procedure TFormCurveRangeSelector.RestoreParams(Limits:TRect2D; TimeBase,TimeUnits:Double);
begin
 EditX1.Text:=Format('%g',[Limits.A.X]);       EditX1.Modified:=false;
 EditY1.Text:=Format('%g',[Limits.A.Y]);       EditY1.Modified:=false;
 EditX2.Text:=Format('%g',[Limits.B.X]);       EditX2.Modified:=false;
 EditY2.Text:=Format('%g',[Limits.B.Y]);       EditY2.Modified:=false;
 EditTimeBase.Text:=FormatTimeMs(TimeBase);    EditTimeBase.Modified:=false;
 EditTimeUnits.Text:=Format('%g',[TimeUnits]); EditTimeUnits.Modified:=false;
end;

function TFormCurveRangeSelector.DecodeRange(var aRange:TRect2D):Boolean;
begin
 aRange:=Rect2D(rVal(EditX1.Text),rVal(EditY1.Text),rVal(EditX2.Text),rVal(EditY2.Text));
 Result:=not IsNanOrInf(aRange) and not RectIsEmpty(aRange);
end;

function TFormCurveRangeSelector.DecodeTimeScale(var aTimeBase,aTimeUnits:Double):Boolean;
begin
 aTimeBase:=DecodeTime(EditTimeBase.Text); aTimeUnits:=rVal(EditTimeUnits.Text);
 Result:=IsValidTimeScale(aTimeBase,aTimeUnits);
end;

procedure TFormCurveRangeSelector.TimerCheckParamsTimer(Sender: TObject);
var R:TRect2D; Tb,Tu:Double;
begin
 try
  R:=Rect2D(0,0,0,0);
  if DecodeRange(R) then begin
   PanelRangeStatus.Caption:=RusEng('Пределы X,Y допустимы','X,Y Range is Acceptable.');
   if not PanelRangeStatus.ParentFont then PanelRangeStatus.ParentFont:=true;
  end else begin
   PanelRangeStatus.Caption:=RusEng('Пределы X,Y НЕДОПУСТИМЫ','X,Y Range is INVALID.');
   if not (PanelRangeStatus.Font.Color=clRed) then begin
    PanelRangeStatus.Font.Style:=[fsBold];
    PanelRangeStatus.Font.Color:=clRed;
   end;
  end;
  Tb:=0; Tu:=0;
  if DecodeTimeScale(Tb,Tu) then begin
   PanelTimeStatus.Caption:=RusEng('Шкала Времени Допустима','Time Scale is Acceptable.');
   if not PanelTimeStatus.ParentFont then PanelTimeStatus.ParentFont:=true;
  end else begin
   PanelTimeStatus.Caption:=RusEng('Шкала Времени НЕДОПУСТИМА','Time Scale is INVALID.');
   if not (PanelTimeStatus.Font.Color=clRed) then begin
    PanelTimeStatus.Font.Style:=[fsBold];
    PanelTimeStatus.Font.Color:=clRed;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'TimerCheckParamsTimer');
 end;
end;

procedure TFormCurveRangeSelector.FormActivate(Sender: TObject);
begin
 TimerCheckParams.Enabled:=true;
end;

procedure TFormCurveRangeSelector.FormDeactivate(Sender: TObject);
begin
 TimerCheckParams.Enabled:=false;
end;

procedure TFormCurveRangeSelector.FormHide(Sender: TObject);
begin
 TimerCheckParams.Enabled:=false;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_curverangeselector;
begin
end;

procedure Free_form_curverangeselector;
begin
end;

initialization

 Init_form_curverangeselector;

finalization

 Free_form_curverangeselector;

end.

//////////////
// END OF FILE
//////////////

