////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Curve Edit DownSampling Dialog.                                       //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231124 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_curveeditdownsamplingdialog; // Form Curve Edit DownSampling Dialog

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons,
 lcltype, lclintf,
 _crw_alloc, _crw_ef, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_task, _crw_assoc, _crw_guard, _crw_lttb,
 _crw_appforms, _crw_apptools, _crw_apputils;

type

  { TFormCurveEditDownSamplingDialog }

  TFormCurveEditDownSamplingDialog = class(TMasterForm)
    PanelButtons: TPanel;
    BitBtnOk: TBitBtn;
    BitBtnCancel: TBitBtn;
    PanelControls: TPanel;
    GroupBoxDownSampling: TGroupBox;
    CheckBoxDownSamplingAbsLen: TCheckBox;
    ComboBoxDownSamplingAbsLen: TComboBox;
    CheckBoxDownSamplingPerPix: TCheckBox;
    ComboBoxDownSamplingPerPix: TComboBox;
    ComboBoxDownSamplingMethod: TComboBox;
    ButtonDownSamplingGetDefaults: TButton;
    ButtonDownSamplingSetDefaults: TButton;
    CheckBoxDownSamplingMethod: TCheckBox;
    GroupBoxDownSamplingNote: TGroupBox;
    MemoDownSamplingNote: TMemo;
    BitBtnHelp: TBitBtn;
    CheckBoxDownSamplingTail: TCheckBox;
    ComboBoxDownSamplingTail: TComboBox;
    procedure FormCreate(Sender: TObject);
    procedure CheckBoxDownSamplingMethodClick(Sender: TObject);
    procedure ComboBoxDownSamplingMethodChange(Sender: TObject);
    procedure CheckBoxDownSamplingAbsLenClick(Sender: TObject);
    procedure ComboBoxDownSamplingAbsLenChange(Sender: TObject);
    procedure CheckBoxDownSamplingPerPixClick(Sender: TObject);
    procedure ComboBoxDownSamplingPerPixChange(Sender: TObject);
    procedure ButtonDownSamplingGetDefaultsClick(Sender: TObject);
    procedure ButtonDownSamplingSetDefaultsClick(Sender: TObject);
    procedure CheckBoxDownSamplingTailChange(Sender: TObject);
    procedure ComboBoxDownSamplingTailChange(Sender: TObject);
    procedure BitBtnHelpClick(Sender: TObject);
  private
    { Private declarations }
    myFormLink : TForm;
    myUpdating : Boolean;
    myDownSampling : TDownSamplingParams;
    procedure UpdateDownSampling(Direction:Integer);
    procedure LocateUnderToolButton;
  public
    { Public declarations }
  end;

procedure RunCurveDownSamplingDialog(Form:TForm);

implementation

{$R *.lfm}

uses
 Form_CurveWindow;

const
  FormCurveEditDownSamplingDialog : TFormCurveEditDownSamplingDialog = nil;

procedure RunCurveDownSamplingDialog(Form:TForm);
var SaveDownSampling:TDownSamplingParams;
begin
 if CanShowModal(FormCurveEditDownSamplingDialog) then
 try
  if (Form is TFormCurveWindow) then begin
   if not Assigned(FormCurveEditDownSamplingDialog)  then begin
    Application.CreateForm(TFormCurveEditDownSamplingDialog, FormCurveEditDownSamplingDialog);
    FormCurveEditDownSamplingDialog.Master:=@FormCurveEditDownSamplingDialog;
   end;
   if Assigned(FormCurveEditDownSamplingDialog) then begin
    LocateFormToCenterOfScreen(FormCurveEditDownSamplingDialog);
    FormCurveEditDownSamplingDialog.myFormLink:=Form;
    with TFormCurveWindow(Form) do begin
     SaveDownSampling:=DownSampling;
     FormCurveEditDownSamplingDialog.LocateUnderToolButton;
     FormCurveEditDownSamplingDialog.UpdateDownSampling(+1);
     if (mrVoice(FormCurveEditDownSamplingDialog.ShowModal)=mrOk) then begin
      try
       LockDraw;
       FormCurveEditDownSamplingDialog.UpdateDownSampling(-1);
      finally
       UnlockDraw;
      end;
     end else begin
      try
       LockDraw;
       DownSampling:=SaveDownSampling;
      finally
       UnlockDraw;
      end;
     end;
    end;
    FormCurveEditDownSamplingDialog.myFormLink:=nil;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'RunCurveDownSamplingDialog');
 end;
end;

procedure TFormCurveEditDownSamplingDialog.FormCreate(Sender: TObject);
var SecNotes:LongString;
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 Caption:=RusEng('DownSampling - Диалог выбора параметров Дискретности',
                 'DownSampling - Dialog to modify sampling parameters');
 SmartUpdate(BitBtnOk,mrCaption(mrOk));
 SmartUpdate(BitBtnCancel,mrCaption(mrCancel));
 SmartUpdate(BitBtnHelp,RusEng('Справка','Help'));
 SmartUpdate(GroupBoxDownSampling,RusEng('DownSampling - понижение дискретности','DownSampling for faster drawing'));
 SmartUpdate(CheckBoxDownSamplingMethod,RusEng('Метод понижения дискретности','Method of DownSampling'));
 SmartUpdate(CheckBoxDownSamplingAbsLen,RusEng('Абсолютный порог, точек','Absolute Threshold, points'));
 SmartUpdate(CheckBoxDownSamplingPerPix,RusEng('Число точек на пиксель','Number of Points Per Pixel'));
 SmartUpdate(CheckBoxDownSamplingTail,RusEng('Сохранять хвост, точек ','Keep tail, points'));
 ComboBoxDownSamplingMethod.Items.Text:=dsm_LIST;
 ComboBoxDownSamplingMethod.ItemIndex:=dsm_LTTB;
 MemoDownSamplingNote.WordWrap:=true;
 SecNotes:=RusEng('[DownSampling.Notes.Ru]','[DownSampling.Notes.En]');
 MemoDownSamplingNote.Lines.Text:=ExtractTextSection(SysIniFile,SecNotes,efRTrim);
 if (MemoDownSamplingNote.Lines.Count=0) then
 MemoDownSamplingNote.Lines.Text:=RusEng(
  'DownSampling - понижение дискретности - алгоритм для'+EOL
 +'ускорения отрисовки графиков кривых путем уменьшения'+EOL
 +'числа точек кривой.'+EOL,
  'DownSampling - method to increase performance of data'+EOL
 +'drawing by using less number of data points, i.e.'+EOL
 +'down number of samples.'+EOL);
 SmartUpdate(ButtonDownSamplingGetDefaults,RusEng('Вернуть начальные значения','Return default params'));
 SmartUpdate(ButtonDownSamplingSetDefaults,RusEng('Сохранить как начальные значения','Save as default params'));
 SmartUpdate(GroupBoxDownSamplingNote,RusEng('Справка','Remark'));
 RestoreFont(ButtonDownSamplingGetDefaults.Font,DefaultSansFont);
 RestoreFont(ButtonDownSamplingSetDefaults.Font,DefaultSansFont);
 RestoreFont(CheckBoxDownSamplingMethod.Font,DefaultSansFont);
 RestoreFont(ComboBoxDownSamplingMethod.Font,DefaultSansFont);
 //RestoreFont(CheckBoxDownSamplingAbsLen.Font,DefaultSansFont);
 //RestoreFont(CheckBoxDownSamplingPerPix.Font,DefaultSansFont);
end;

procedure TFormCurveEditDownSamplingDialog.UpdateDownSampling(Direction:Integer);
var iv:Integer; rv:Double;
begin
 if (myFormLink is TFormCurveWindow) then
 with TFormCurveWindow(myFormLink) do
 if (Direction>0) and not myUpdating then begin
  try
   myUpdating:=true;
   myDownSampling:=DownSampling;
   ComboBoxDownSamplingMethod.Items.Text:=dsm_List;
   ComboBoxDownSamplingMethod.ItemIndex:=Min(Max(0,myDownSampling.Method),ComboBoxDownSamplingMethod.Items.Count-1);
   CheckBoxDownSamplingMethod.Checked:=HasFlags(myDownSampling.Mode,dsm_Enable);
   CheckBoxDownSamplingAbsLen.Checked:=HasFlags(myDownSampling.Mode,dsm_AbsLen);
   CheckBoxDownSamplingPerPix.Checked:=HasFlags(myDownSampling.Mode,dsm_PerPix);
   CheckBoxDownSamplingTail.Checked:=HasFlags(myDownSampling.Mode,dsm_Tail);
   ComboBoxDownSamplingAbsLen.Text:=IntToStr(myDownSampling.AbsLen);
   ComboBoxDownSamplingPerPix.Text:=Format('%.7g',[myDownSampling.PerPix]);
   ComboBoxDownSamplingTail.Text:=IntToStr(myDownSampling.Tail);
  finally
   myUpdating:=false;
  end;
 end else
 if (Direction<0) and not myUpdating then begin
  myDownSampling.Method:=Max(0,ComboBoxDownSamplingMethod.ItemIndex);
  LiftFlags(myDownSampling.Mode,dsm_Enable,CheckBoxDownSamplingMethod.Checked);
  LiftFlags(myDownSampling.Mode,dsm_AbsLen,CheckBoxDownSamplingAbsLen.Checked);
  LiftFlags(myDownSampling.Mode,dsm_PerPix,CheckBoxDownSamplingPerPix.Checked);
  LiftFlags(myDownSampling.Mode,dsm_Tail,CheckBoxDownSamplingTail.Checked);
  if Str2Int(Trim(ComboBoxDownSamplingAbsLen.Text),iv)
  then myDownSampling.AbsLen:=Max(0,iv);
  if Str2Real(Trim(ComboBoxDownSamplingPerPix.Text),rv)
  then myDownSampling.PerPix:=Max(0,rv);
  if Str2Int(Trim(ComboBoxDownSamplingTail.Text),iv)
  then myDownSampling.Tail:=Max(0,iv);
  DownSampling:=myDownSampling;
 end;
end;

procedure TFormCurveEditDownSamplingDialog.LocateUnderToolButton;
var P,Delta:TPoint; ToolButton:TToolButton;
begin
 ToolButton:=nil;
 if (myFormLink is TFormCurveWindow) then
 with TFormCurveWindow(myFormLink) do begin
  ToolButton:=ToolButtonCurveEditDownSampling;
  Delta:=Point(-Self.Width,0);
  if (Width<ToolButton.BoundsRect.BottomRight.x) then begin
   ToolButton:=ToolButtonCurveTools;
   Delta:=Point(-ToolButton.Width,0);
  end;
 end;
 if Assigned(ToolButton) then begin
  P:=ToolButton.ClientRect.BottomRight;
  P:=ToolButton.ClientToScreen(P);
  P.X:=P.X+Delta.X;
  Left:=P.X;
  Top:=P.Y;
 end else LocateFormToCenterOfScreen(Self);
end;

procedure TFormCurveEditDownSamplingDialog.CheckBoxDownSamplingMethodClick(Sender: TObject);
begin
 UpdateDownSampling(-1);
end;

procedure TFormCurveEditDownSamplingDialog.ComboBoxDownSamplingMethodChange(Sender: TObject);
begin
 UpdateDownSampling(-1);
end;

procedure TFormCurveEditDownSamplingDialog.CheckBoxDownSamplingAbsLenClick(Sender: TObject);
begin
 UpdateDownSampling(-1);
end;

procedure TFormCurveEditDownSamplingDialog.ComboBoxDownSamplingAbsLenChange(Sender: TObject);
begin
 UpdateDownSampling(-1);
end;

procedure TFormCurveEditDownSamplingDialog.CheckBoxDownSamplingPerPixClick(Sender: TObject);
begin
 UpdateDownSampling(-1);
end;

procedure TFormCurveEditDownSamplingDialog.ComboBoxDownSamplingPerPixChange(Sender: TObject);
begin
 UpdateDownSampling(-1);
end;

procedure TFormCurveEditDownSamplingDialog.CheckBoxDownSamplingTailChange(Sender: TObject);
begin
 UpdateDownSampling(-1);
end;

procedure TFormCurveEditDownSamplingDialog.ComboBoxDownSamplingTailChange(Sender: TObject);
begin
 UpdateDownSampling(-1);
end;

procedure TFormCurveEditDownSamplingDialog.ButtonDownSamplingGetDefaultsClick(Sender: TObject);
begin
 if (myFormLink is TFormCurveWindow) then
 with TFormCurveWindow(myFormLink) do begin
  DownSampling:=DefDownSamplingParams;
  UpdateDownSampling(+1);
 end;
end;

procedure TFormCurveEditDownSamplingDialog.ButtonDownSamplingSetDefaultsClick(Sender: TObject);
var mr:Integer;
begin
 if (myFormLink is TFormCurveWindow) then
 with TFormCurveWindow(myFormLink) do begin
  mr:=YesNo(RusEng('Сохранить текущие параметры DownSampling как начальные значения для всех окон?',
                   'Save current DownSampling parameters as default parameters for all windows?'));
  if (mr=mrYes) then begin
   UpdateDownSampling(-1);
   DefDownSamplingParams:=DownSampling;
  end;
 end;
end;

procedure TFormCurveEditDownSamplingDialog.BitBtnHelpClick(Sender: TObject);
var HelpFile:LongString;
begin
 HelpFile:=HomeDir+'/resource/manual/downsampling/downsampling.md.html';
 HelpFile:=UnifyFileAlias(HelpFile);
 ShellExecuteOpen(HelpFile);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_curveeditdownsamplingdialog;
begin
end;

procedure Free_form_curveeditdownsamplingdialog;
begin
end;

initialization

 Init_form_curveeditdownsamplingdialog;

finalization

 Free_form_curveeditdownsamplingdialog;

end.

//////////////
// END OF FILE
//////////////

