////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Console Window - fast console for text output.                        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231117 - Modified for FPC (A.K.)                                         //
// 20260212 - Modified for LongStrings, add scrolling buttons                 //
////////////////////////////////////////////////////////////////////////////////

unit form_consolewindow; // Form Console Window

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo, _crw_ef,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut, _crw_utf8,
 _crw_dynar, _crw_snd, _crw_guard, _crw_gloss, _crw_sect,
 _crw_sesman, _crw_appforms, _crw_apptools, _crw_apputils;

const
  ConsoleBorderSpace   = 4;            // Space of control to border
  ConsoleVertLineSpace = 2;            // Vertical space between lines
  ConsoleDefLinesCount = 1024;         // Default history length LinesCount
  ConsoleMinLinesCount = 128;          // Minimal history length LinesCount
  ConsoleMaxLinesCount = 1024*1024*4;  // Maximal history length LinesCount
  ConsoleMinHighColumn = 128;          // Minimal value of high visible column
  ConsoleMaxHighColumn = 1024*64;      // Maximal value of high visible column
  ConsoleDefHighColumn = 1024;         // Default value of high visible column
  ConsoleMinTabWidth   = 0;            // Minimal value of TAB width, 0=OFF
  ConsoleMaxTabWidth   = 8;            // Maximal value of TAB width
  ConsoleDefTabWidth   = 4;            // Default value of TAB width

type
  TFormConsoleWindow  = class;
  TConsolePollingProc = procedure(aConsole : TFormConsoleWindow);
  TConsoleFilterProc  = procedure(aConsole : TFormConsoleWindow;
                              var aText    : LongString);

  { TFormConsoleWindow }

  TFormConsoleWindow = class(TFormCrwDaqSysChild)
    MenuViewEditConsoleParams: TMenuItem;
    PanelOutputControls: TPanel;
    PanelOutput: TPanel;
    GroupBoxOutput: TGroupBox;
    PaintBox: TPaintBox;
    ScrollBarH: TScrollBar;
    ScrollBarV: TScrollBar;
    PanelInputControls: TPanel;
    GroupBoxInput: TGroupBox;
    PanelInputCheckBox: TPanel;
    CheckBoxInput: TCheckBox;
    PanelInputComboBox: TPanel;
    ComboBoxInput: TComboBox;
    BitBtnInputToggle: TBitBtn;
    GroupBoxScrollButtons: TGroupBox;
    BitBtnViewScrollDnEnd: TBitBtn;
    BitBtnViewScrollLtEnd: TBitBtn;
    BitBtnViewScrollRtEnd: TBitBtn;
    BitBtnViewScrollUpEnd: TBitBtn;
    ActionEditDelete: TAction;
    ActionEditCut: TAction;
    ActionEditCopy: TAction;
    ActionEditPaste: TAction;
    ActionEditInputEnable: TAction;
    ActionEditInputDisable: TAction;
    ActionEditInputToggle: TAction;
    ActionViewEditConsoleParams: TAction;
    ActionViewScrollDnEnd: TAction;
    ActionViewScrollUpEnd: TAction;
    ActionViewScrollRtEnd: TAction;
    ActionViewScrollLtEnd: TAction;
    ActionViewScrollDnLarge: TAction;
    ActionViewScrollUpLarge: TAction;
    ActionViewScrollDnSmall: TAction;
    ActionViewScrollUpSmall: TAction;
    ActionViewScrollRtSmall: TAction;
    ActionViewScrollLtSmall: TAction;
    MenuEdit: TMenuItem;
    MenuEditDelete: TMenuItem;
    MenuEditCut: TMenuItem;
    MenuEditCopy: TMenuItem;
    MenuEditPaste: TMenuItem;
    MenuInputSeparator: TMenuItem;
    MenuEditInputToggle: TMenuItem;
    MenuViewScrollDnEnd: TMenuItem;
    MenuViewScrollUpEnd: TMenuItem;
    MenuViewScrollRtEnd: TMenuItem;
    MenuViewScrollLtEnd: TMenuItem;
    MenuViewScrollXEnd: TMenuItem;
    PopupMenuFilePrint: TMenuItem;
    PopupMenuFilePrintSeparator: TMenuItem;
    PopupMenuEditDelete: TMenuItem;
    PopupMenuEditCut: TMenuItem;
    PopupMenuEditCopy: TMenuItem;
    PopupMenuEditPaste: TMenuItem;
    PopupMenuEditInputSeparator: TMenuItem;
    PopupMenuEditInputToggle: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure FormResize(Sender: TObject);
    procedure GroupBoxInputDblClick(Sender: TObject);
    procedure GroupBoxOutputDblClick(Sender: TObject);
    procedure PaintBoxMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure PaintBoxMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure PaintBoxMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure PaintBoxPaint(Sender: TObject);
    procedure PanelOutputDblClick(Sender: TObject);
    procedure ScrollBarVChange(Sender: TObject);
    procedure ScrollBarHChange(Sender: TObject);
    procedure ComboBoxInputDblClick(Sender: TObject);
    procedure ComboBoxInputKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure FormActivate(Sender: TObject);
    procedure ActionEditDeleteExecute(Sender: TObject);
    procedure ActionEditCutExecute(Sender: TObject);
    procedure ActionEditCopyExecute(Sender: TObject);
    procedure ActionEditPasteExecute(Sender: TObject);
    procedure ActionEditInputDisableExecute(Sender: TObject);
    procedure ActionEditInputEnableExecute(Sender: TObject);
    procedure ActionEditInputToggleExecute(Sender: TObject);
    procedure ActionViewScrollDnEndExecute(Sender: TObject);
    procedure ActionViewScrollLtEndExecute(Sender: TObject);
    procedure ActionViewScrollRtEndExecute(Sender: TObject);
    procedure ActionViewScrollUpEndExecute(Sender: TObject);
    procedure ActionViewScrollDnLargeExecute(Sender: TObject);
    procedure ActionViewScrollDnSmallExecute(Sender: TObject);
    procedure ActionViewScrollUpLargeExecute(Sender: TObject);
    procedure ActionViewScrollLtSmallExecute(Sender: TObject);
    procedure ActionViewScrollRtSmallExecute(Sender: TObject);
    procedure ActionViewScrollUpSmallExecute(Sender: TObject);
    procedure ActionViewEditConsoleParamsExecute(Sender: TObject);
    procedure CheckBoxInputChange(Sender: TObject);
  private
    { Private declarations }
    myFlatCursor    : Int64;
    myRingCursor    : Integer;
    myFlatLines     : TStringList;
    myTextColor     : TColor;
    myInpFifo       : TFifo;
    myOutFifo       : TFifo;
    myInpOwns       : Boolean;
    myOutOwns       : Boolean;
    myInpFilter     : TConsoleFilterProc;
    myOutFilter     : TConsoleFilterProc;
    myPolling       : TConsolePollingProc;
    myCustom        : Pointer;
    myGuardInput    : Cardinal;
    myCharWidthPx   : Integer;
    myHighColumn    : Integer;
    myTabWidth      : Integer;
    mySelTextRect   : TRect2I;
    mySelTextFlag   : Boolean;
    myXorSelector   : TXorSelector;
    function  GetLinesCount:Integer;
    procedure SetLinesCount(aCount:Integer);
    function  GetHighLines:Integer;
    function  GetFlatCursor:Int64;
    procedure SetFlatCursor(aCursor:Int64);
    function  GetRingCursor:Integer;
    function  FlatToRing(i:Integer):Integer;
    function  GetFlatLines(i:Integer):LongString;
    procedure SetFlatLines(i:Integer; const aLine:LongString);
    function  GetRingLines(i:Integer):LongString;
    procedure SetRingLines(i:Integer; const aLine:LongString);
    function  GetLinesPerPage:Integer;
    function  GetColumnsPerPage:Integer;
    function  LineRect(i:Integer):TRect2I;
    function  GetTextColor:TColor;
    procedure SetTextColor(aColor:TColor);
    function  GetGroundColor:TColor;
    procedure SetGroundColor(aColor:TColor);
    function  GetInpFifo:TFifo;
    function  GetOutFifo:TFifo;
    function  GetInpOwns:Boolean;
    function  GetOutOwns:Boolean;
    function  GetInpFilter:TConsoleFilterProc;
    function  GetOutFilter:TConsoleFilterProc;
    function  GetPolling:TConsolePollingProc;
    procedure SetPolling(aPolling:TConsolePollingProc);
    function  GetCustom:Pointer;
    procedure SetCustom(aCustom:Pointer);
    function  GetGuardInput:Cardinal;
    procedure SetGuardInput(aGuard:Cardinal);
    function  GetMaxLineLength:Integer;
    function  GetCharWidthPx:Integer;
    function  GetHighColumn:Integer;
    procedure SetHighColumn(aColumn:Integer);
    function  GetTabWidth:Integer;
    procedure SetTabWidth(aWidth:Integer);
    function  GetSelTextFlag:Boolean;
    procedure SetSelTextFlag(Flag:Boolean);
    function  GetSelTextRect:TRect2I;
    procedure SetSelTextRect(const R:TRect2I);
    function  GetSelText:LongString;
  public
    { Public declarations }
    property  LinesCount:Integer           read GetLinesCount write SetLinesCount;
    property  HighLines:Integer            read GetHighLines;
    property  FlatCursor:Int64             read GetFlatCursor write SetFlatCursor;
    property  RingCursor:Integer           read GetRingCursor;
    property  FlatLines[i:Integer]:LongString read GetFlatLines write SetFlatLines;
    property  RingLines[i:Integer]:LongString read GetRingLines write SetRingLines;
    property  TextColor:TColor             read GetTextColor   write SetTextColor;
    property  GroundColor:TColor           read GetGroundColor write SetGroundColor;
    property  InpFifo:TFifo                read GetInpFifo;
    property  OutFifo:TFifo                read GetOutFifo;
    property  InpOwns:Boolean              read GetInpOwns;
    property  OutOwns:Boolean              read GetOutOwns;
    property  InpFilter:TConsoleFilterProc read GetInpFilter;
    property  OutFilter:TConsoleFilterProc read GetOutFilter;
    property  Polling:TConsolePollingProc  read GetPolling     write SetPolling;
    property  Custom:Pointer               read GetCustom      write SetCustom;
    property  GuardInput:Cardinal          read GetGuardInput  write SetGuardInput;
    property  MaxLineLength:Integer        read GetMaxLineLength;
    property  CharWidthPx:Integer          read GetCharWidthPx;
    property  LinesPerPage:Integer         read GetLinesPerPage;
    property  ColumnsPerPage:Integer       read GetColumnsPerPage;
    property  HighColumn:Integer           read GetHighColumn write SetHighColumn;
    property  TabWidth:Integer             read GetTabWidth write SetTabWidth;
    property  SelTextFlag:Boolean          read GetSelTextFlag write SetSelTextFlag;
    property  SelTextRect:TRect2I          read GetSelTextRect write SetSelTextRect;
    property  SelText:LongString           read GetSelText;
  public
    procedure UpdateLines(const aText:LongString);
    procedure MoveRingCursor(Step:Integer);
    function  ValidateTab(const Line:LongString):LongString;
    function  TextFromRect(const r:TRect2I):LongString;
    function  PixToPos(Pix:TPoint2I):TPoint2I;
    function  PosToPix(Pos:TPoint2I):TPoint2I;
    procedure DrawSelTextRect;
  public
    procedure AssignFifo(aInpFifo:TFifo; aInpOwns:Boolean; aInpFilter:TConsoleFilterProc;
                         aOutFifo:TFifo; aOutOwns:Boolean; aOutFilter:TConsoleFilterProc);
    procedure UpdateScrollBars(Mode:Integer=0);
    procedure UpdateInputBox(Mode:Integer=0);
    procedure ClearOutput;
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
    procedure UpdateCommands; override;
    procedure DrawView; override;
    procedure PutText(const aText:LongString);  // to output fifo
    procedure StartMonitoring;
    procedure StopMonitoring;
    procedure Monitoring;
  public
    class var AutoFocusInput:Boolean;
    class var AutoEnableInput:Boolean;
  end;
  TConsoleWindowList = class(TObjectStorage)
  private
    function   GetWindow(i:Integer):TFormConsoleWindow;
    procedure  SetWindow(i:Integer; aWindow:TFormConsoleWindow);
  public
    property   Window[i:Integer]:TFormConsoleWindow read GetWindow write SetWindow; default;
  end;

const
  FullConsoleWindowList : TConsoleWindowList = nil;
  ConsoleWindowsMonitor : TConsoleWindowList = nil;

function  NewConsoleWindow(const aCaption   : LongString;
                                 aInpFifo   : TFifo;
                                 aInpOwns   : Boolean;
                                 aInpFilter : TConsoleFilterProc;
                                 aOutFifo   : TFifo;
                                 aOutOwns   : Boolean;
                                 aOutFilter : TConsoleFilterProc):TFormConsoleWindow;
procedure Kill(var TheObject:TFormConsoleWindow); overload;
function  ActiveConsoleWindow:TFormConsoleWindow;

function  NewConsoleWindowList(aOwnsObjects : Boolean = true;
                               aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                               aStep        : LongInt = DefaultTObjectStorageStep
                                          ) : TConsoleWindowList;
procedure Kill(var TheObject:TConsoleWindowList); overload;

implementation

{$R *.lfm}

uses
 form_stringgrideditdialog;

 {
 *********************************
 TConsoleWindowList implementation
 *********************************
 }
function TConsoleWindowList.GetWindow(i:Integer):TFormConsoleWindow;
begin
 Result:=TFormConsoleWindow(Items[i]);
end;

procedure TConsoleWindowList.SetWindow(i:Integer; aWindow:TFormConsoleWindow);
begin
 Items[i]:=aWindow;
end;

function NewConsoleWindowList(aOwnsObjects : Boolean = true;
                              aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                              aStep        : LongInt = DefaultTObjectStorageStep
                                         ) : TConsoleWindowList;
begin
 Result:=nil;
 try
  Result:=TConsoleWindowList.Create(aOwnsObjects,aCapacity,aStep);
 except
  on E:Exception do BugReport(E,nil,'NewConsoleWindowList');
 end;
end;

procedure Kill(var TheObject:TConsoleWindowList); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end;
end;

 {
 *******************************************************************************
 General purpose routines
 *******************************************************************************
 }
function  NewConsoleWindow(const aCaption   : LongString;
                                 aInpFifo   : TFifo;
                                 aInpOwns   : Boolean;
                                 aInpFilter : TConsoleFilterProc;
                                 aOutFifo   : TFifo;
                                 aOutOwns   : Boolean;
                                 aOutFilter : TConsoleFilterProc):TFormConsoleWindow;
begin
 Result:=nil;
 try
  Application.CreateForm(TFormConsoleWindow,Result);
  with Result do if Ok then begin
   try
    LockDraw;
    Caption:=aCaption;
    AssignFifo(aInpFifo, aInpOwns, aInpFilter, aOutFifo, aOutOwns, aOutFilter);
    UpdateScrollBars(1);
    UpdateInputBox(1);
   finally
    UnlockDraw;
   end;
  end else Kill(Result);
 except
  on E:Exception do BugReport(E,nil,'NewConsoleWindow');
 end;
end;

procedure Kill(var TheObject:TFormConsoleWindow); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

function ActiveConsoleWindow:TFormConsoleWindow;
var Child:TForm;
begin
 Result:=nil;
 try
  if Sdiman.FindActiveChild(Child,sf_SdiChild,sf_SdiControl) then
  if (Child is TFormConsoleWindow) then Result:=TFormConsoleWindow(Child);
 except
  on E:Exception do BugReport(E,nil,'ActiveConsoleWindow');
 end;
end;

 {
 *******************************************************************************
 TFormConsoleWindow implementation
 *******************************************************************************
 }
function TFormConsoleWindow.GetLinesCount:Integer;
begin
 if Assigned(Self) and Assigned(myFlatLines)
 then Result:=myFlatLines.Count
 else Result:=0;
end;

procedure TFormConsoleWindow.SetLinesCount(aCount:Integer);
begin
 aCount:=EnsureRange(aCount,ConsoleMinLinesCount,ConsoleMaxLinesCount);
 if Assigned(Self) and Assigned(myFlatLines) then begin
  while (myFlatLines.Count<aCount) do myFlatLines.Add('');
  while (myFlatLines.Count>aCount) do myFlatLines.Delete(myFlatLines.Count-1);
  MoveRingCursor(0);
 end;
end;

function TFormConsoleWindow.GetHighLines:Integer;
begin
 if Assigned(Self) and Assigned(myFlatLines)
 then Result:=myFlatLines.Count-1
 else Result:=0;
end;

function TFormConsoleWindow.GetFlatCursor:Int64;
begin
 if Assigned(Self)
 then Result:=myFlatCursor
 else Result:=0;
end;

procedure TFormConsoleWindow.SetFlatCursor(aCursor:Int64);
begin
 if Assigned(Self) then begin
  myFlatCursor:=aCursor;
  MoveRingCursor(0);
 end;
end;

function TFormConsoleWindow.GetRingCursor:Integer;
begin
 if Assigned(Self)
 then Result:=myRingCursor
 else Result:=0;
end;

function TFormConsoleWindow.FlatToRing(i:Integer):Integer;
begin // FlatToRing uses to perform cyclic ring discipline
 if Assigned(Self) and Assigned(myFlatLines) and (myFlatLines.Count>0)
 then Result:=((myRingCursor+i) mod myFlatLines.Count)
 else Result:=0;
end;

procedure TFormConsoleWindow.MoveRingCursor(Step:Integer);
begin
 if Assigned(Self) and Assigned(myFlatLines) and (myFlatLines.Count>0) then begin
  if (Step<>0) then myFlatCursor:=(myFlatCursor+Step);
  myRingCursor:=(myFlatCursor mod myFlatLines.Count);
  if (Step<>0) then mySelTextFlag:=False;
 end;
end;

function  TFormConsoleWindow.GetFlatLines(i:Integer):LongString;
begin
 if Assigned(Self) and Assigned(myFlatLines) and InRange(i,0,myFlatLines.Count-1)
 then Result:=myFlatLines[i]
 else Result:='';
end;

procedure TFormConsoleWindow.SetFlatLines(i:Integer; const aLine:LongString);
begin
 if Assigned(Self) and Assigned(myFlatLines) and InRange(i,0,myFlatLines.Count-1)
 then myFlatLines[i]:=aLine;
end;

function  TFormConsoleWindow.GetRingLines(i:Integer):LongString;
begin
 if Assigned(Self) and Assigned(myFlatLines) and InRange(i,0,myFlatLines.Count-1)
 then Result:=FlatLines[FlatToRing(i)]
 else Result:='';
end;

procedure TFormConsoleWindow.SetRingLines(i:Integer; const aLine:LongString);
begin
 if Assigned(Self) and Assigned(myFlatLines) and InRange(i,0,myFlatLines.Count-1)
 then FlatLines[FlatToRing(i)]:=aLine;
end;

function TFormConsoleWindow.GetLinesPerPage:Integer;
begin
 if Assigned(Self)
 then Result:=Max(1,PaintBox.Height div Max(1,RectSizeY(LineRect(0))))
 else Result:=0;
end;

function TFormConsoleWindow.GetColumnsPerPage:Integer;
begin
 if Assigned(Self)
 then Result:=Max(1,PaintBox.Width div Max(1,CharWidthPx))
 else Result:=0;
end;

function TFormConsoleWindow.LineRect(i:Integer):TRect2I;
var h:Integer;
begin
 if Assigned(Self) then begin
  h:=abs(PaintBox.Font.Height)+ConsoleVertLineSpace;
  Result.A.X:=0;
  Result.A.Y:=i*h;
  Result.B.X:=PaintBox.Width;
  Result.B.Y:=Result.A.Y+h;
 end else Result:=Rect2I(0,0,0,0);
end;

function  TFormConsoleWindow.GetTextColor:TColor;
begin
 if Ok then Result:=myTextColor else Result:=0;
end;

procedure TFormConsoleWindow.SetTextColor(aColor:TColor);
begin
 if Ok then
 try
  LockDraw;
  myTextColor:=aColor;
 finally
  UnlockDraw;
 end;
end;

function  TFormConsoleWindow.GetGroundColor:TColor;
begin
 if Ok then Result:=PaintBox.Color else Result:=0;
end;

procedure TFormConsoleWindow.SetGroundColor(aColor:TColor);
begin
 if Ok then
 try
  LockDraw;
  PaintBox.Color:=aColor;
 finally
  UnlockDraw;
 end;
end;

function  TFormConsoleWindow.GetInpFifo:TFifo;
begin
 if Assigned(Self) then Result:=myInpFifo else Result:=nil;
end;

function  TFormConsoleWindow.GetOutFifo:TFifo;
begin
 if Assigned(Self) then Result:=myOutFifo else Result:=nil;
end;

function  TFormConsoleWindow.GetInpOwns:Boolean;
begin
 if Assigned(Self) then Result:=myInpOwns else Result:=false;
end;

function  TFormConsoleWindow.GetOutOwns:Boolean;
begin
 if Assigned(Self) then Result:=myOutOwns else Result:=false;
end;

function  TFormConsoleWindow.GetInpFilter:TConsoleFilterProc;
begin
 if Assigned(Self) then Result:=myInpFilter else Result:=nil;
end;

function  TFormConsoleWindow.GetOutFilter:TConsoleFilterProc;
begin
 if Assigned(Self) then Result:=myOutFilter else Result:=nil;
end;

function TFormConsoleWindow.GetPolling:TConsolePollingProc;
begin
 if Assigned(Self) then Result:=myPolling else Result:=nil;
end;

procedure TFormConsoleWindow.SetPolling(aPolling:TConsolePollingProc);
begin
 if Assigned(Self) then myPolling:=aPolling;
end;

function  TFormConsoleWindow.GetCustom;
begin
 if Assigned(Self) then Result:=myCustom else Result:=nil;
end;

procedure TFormConsoleWindow.SetCustom(aCustom:Pointer);
begin
 if Assigned(Self) then myCustom:=aCustom;
end;

function TFormConsoleWindow.GetGuardInput:Cardinal;
begin
 if Assigned(Self) then Result:=myGuardInput else Result:=ga_Lock;
end;

procedure TFormConsoleWindow.SetGuardInput(aGuard:Cardinal);
begin
 if Assigned(Self) then myGuardInput:=Max(ga_Lock,Min(ga_Root,aGuard));
end;

function TFormConsoleWindow.GetHighColumn:Integer;
begin
 if Assigned(Self)
 then Result:=myHighColumn
 else Result:=0;
end;

procedure TFormConsoleWindow.SetHighColumn(aColumn:Integer);
begin
 if Assigned(Self)
 then myHighColumn:=EnsureRange(aColumn,ConsoleMinHighColumn,ConsoleMaxHighColumn);
end;

function TFormConsoleWindow.GetTabWidth:Integer;
begin
 if Assigned(Self)
 then Result:=myTabWidth
 else Result:=0;
end;

procedure TFormConsoleWindow.SetTabWidth(aWidth:Integer);
begin
 if Assigned(Self)
 then myTabWidth:=EnsureRange(aWidth,ConsoleMinTabWidth,ConsoleMaxTabWidth);
end;

function TFormConsoleWindow.GetSelTextFlag:Boolean;
begin
 if Assigned(Self)
 then Result:=mySelTextFlag
 else Result:=False;
end;

procedure TFormConsoleWindow.SetSelTextFlag(Flag:Boolean);
begin
 if Assigned(Self) then mySelTextFlag:=Flag;
end;

function TFormConsoleWindow.GetSelTextRect:TRect2I;
begin
 if Assigned(Self)
 then Result:=mySelTextRect
 else Result:=Default(TRect2I);
end;

procedure TFormConsoleWindow.SetSelTextRect(const R:TRect2I);
begin
 if Assigned(Self) then mySelTextRect:=R;
end;

function TFormConsoleWindow.GetSelText:LongString;
begin
 if Assigned(Self) and SelTextFlag
 then Result:=TextFromRect(SelTextRect)
 else Result:='';
end;

function TFormConsoleWindow.GetMaxLineLength:Integer;
var i,Len:Integer; sLine:LongString;
begin
 Result:=0;
 if Assigned(Self) then begin
  for i:=0 to HighLines do begin
   sLine:=FlatLines[i];
   if (sLine='') then continue;
   if IsTextAscii(sLine) then begin
    Result:=Max(Result,Length(sLine));
    continue;
   end;
   Len:=utf8_length(sLine);
   if (Len=0) then Len:=Length(sLine); // Fallback for broken UTF8
   Result:=Max(Result,Len);
  end;
 end;
end;

function TFormConsoleWindow.GetCharWidthPx:Integer;
begin
 Result:=0;
 if Assigned(Self) then begin
  if (myCharWidthPx<=0) then myCharWidthPx:=PaintBox.Canvas.TextWidth('O');
  Result:=myCharWidthPx;
 end;
 Result:=Max(1,Result);
end;

function TFormConsoleWindow.PixToPos(Pix:TPoint2I):TPoint2I;
var cSize:TPoint2I;
begin
 Result:=Default(TPoint2I);
 if Assigned(Self) then begin
  cSize.x:=CharWidthPx;
  cSize.y:=RectSize(LineRect(0)).y;
  if (cSize.x>0) and (cSize.y>0) then begin
   Result.x:=ScaleValMulDiv(Pix.x,1,cSize.x);
   Result.y:=ScaleValMulDiv(Pix.y,1,cSize.y);
   PointMove(Result,+ScrollBarH.Position,0);
   PointMove(Result,0,+ScrollBarV.Position);
  end;
 end;
end;

function TFormConsoleWindow.PosToPix(Pos:TPoint2I):TPoint2I;
var cSize:TPoint2I;
begin
 Result:=Default(TPoint2I);
 if Assigned(Self) then begin
  cSize.x:=CharWidthPx;
  cSize.y:=RectSize(LineRect(0)).y;
  PointMove(Pos,-ScrollBarH.Position,0);
  PointMove(Pos,0,-ScrollBarV.Position);
  if (cSize.x>0) and (cSize.y>0) then begin
   Result.x:=ScaleValMulDiv(Pos.x,cSize.x,1);
   Result.y:=ScaleValMulDiv(Pos.y,cSize.y,1);
  end;
 end;
end;

function TFormConsoleWindow.TextFromRect(const r:TRect2I):LongString;
var line,s:LongString; n,iy,col,leng:Integer; rc:TRect2I;
begin
 Result:='';
 line:=''; s:=''; n:=0;
 if Assigned(Self) then begin
  if RectIsEmpty(r) then Exit;
  rc:=Rect2I(0,0,HighColumn,HighLines);
  if not RectIsEqual(r,RectIntersection(r,rc)) then Exit;
  leng:=(r.b.x-r.a.x); if (leng<=0) then Exit;
  col:=r.a.x+1; if (col<1) then Exit;
  for iy:=r.a.y to r.b.y-1 do begin
   line:=RingLines[iy];
   if InRange(col,1,utf8_length(line)) then begin
    s:=utf8_copy(line,col,leng);
    if (s<>'') then inc(n);
   end else s:='';
   if (Result='') then Result:=s else Result:=Result+EOL+s;
  end;
  if (n=0) then Result:='';
 end;
end;

procedure TFormConsoleWindow.DrawSelTextRect;
var r,p,rc,pc:TRect2I;
begin
 if Assigned(Self) and SelTextFlag then begin
  p:=SelTextRect; if RectIsEmpty(p) then Exit;
  pc:=Rect2I(0,0,HighColumn,HighLines);
  if not RectIsEqual(p,RectIntersection(p,pc)) then Exit;
  r:=Rect2I(PosToPix(p.a),PosToPix(p.b));
  rc:=Rect2I(0,0,PaintBox.Width,PaintBox.Height);
  if RectIsEmpty(RectIntersection(r,rc)) then Exit;
  RectGrow(r,2,2); // Some grow rect for nice view
  r.a.x:=Max(r.a.x,-1); r.b.x:=Min(r.b.x,PaintBox.Width);
  r.a.y:=Max(r.a.y,-1); r.b.y:=Min(r.b.y,PaintBox.Height);
  if RectIsEmpty(r) then Exit; // Exit if nothing to draw
  DrawRect(PaintBox.Canvas,r,clWhite,psSolid,pmXor,3);
 end;
end;

function TFormConsoleWindow.ValidateTab(const Line:LongString):LongString;
begin
 Result:=Line;
 if not Assigned(Self) then Exit;
 if (TabWidth>0) and (Pos(ASCII_TAB,Line)>0)
 then Result:=Tab2SpaceAdjusted(Line,TabWidth);
end;

function fcw_UpdateLines(n:SizeInt; Line:LongString; Custom:Pointer):Boolean;
var Form:TFormConsoleWindow;
begin
 Result:=True; Form:=TFormConsoleWindow(Custom);
 if (TObject(Custom) is TFormConsoleWindow) then
 with Form do begin
  MoveRingCursor(1);
  RingLines[HighLines]:=ValidateTab(Line);
 end;
end;

procedure TFormConsoleWindow.UpdateLines(const aText:LongString);
begin
 if Ok and (aText<>'') then
 try
  LockDraw;
  try
   ForEachStringLine(aText,fcw_UpdateLines,Self);
  finally
   UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateLines');
 end;
end;

procedure TFormConsoleWindow.AssignFifo(aInpFifo:TFifo; aInpOwns:Boolean; aInpFilter:TConsoleFilterProc;
                                        aOutFifo:TFifo; aOutOwns:Boolean; aOutFilter:TConsoleFilterProc);
begin
 if Ok then
 try
  LockDraw;
  try
   if myInpOwns then Kill(myInpFifo);
   if myOutOwns then Kill(myOutFifo);
   myInpFifo:=aInpFifo;
   myInpOwns:=aInpOwns;
   myInpFilter:=aInpFilter;
   myOutFifo:=aOutFifo;
   myOutOwns:=aOutOwns;
   myOutFilter:=aOutFilter;
   GroupBoxInput.Visible:=InpFifo.Ok;
   GroupBoxOutput.Visible:=OutFifo.Ok;
  finally
   UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self,'AssignFifo');
 end;
end;

procedure TFormConsoleWindow.UpdateScrollBars(Mode:Integer=0);
begin
 if Ok then
 try
  LockDraw;
  try
   myCharWidthPx:=0;{Reset}
   with ScrollBarV do begin
    Min:=0;
    Max:=Math.Max(Min,HighLines-LinesPerPage+1);
    SmallChange:=1;
    LargeChange:=Math.Max(1,ScaleValMulDiv(LinesPerPage,3,4));
    Position:=Max;
   end;
   with ScrollBarH do begin
    Min:=0;
    Max:=Math.Max(Min,HighColumn-ColumnsPerPage+1);
    SmallChange:=1;
    LargeChange:=Math.Max(1,ScaleValMulDiv(ColumnsPerPage,3,4));
    Position:=Min;
   end;
   UpdateInputBox;
  finally
   UnlockDraw;
  end;
  if HasFlags(Mode,1) then begin
   GroupBoxScrollButtons.Width:=BitBtnViewScrollDnEnd.BoundsRect.Right+ConsoleBorderSpace;
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateScrollBars');
 end;
end;

procedure TFormConsoleWindow.UpdateInputBox(Mode:Integer=0);
begin
 if Ok then
 try
  // Update EditInput state
  if CheckBoxInput.Checked then begin
   SmartUpdate(ActionEditInputToggle,RusEng('Запретить Ввод','Disable Input'));
   ActionEditInputToggle.ImageIndex:=177;
   CheckBoxInput.Font.Color:=clBtnText;
  end else begin
   SmartUpdate(ActionEditInputToggle,RusEng('Разрешить Ввод','Enable Input'));
   ActionEditInputToggle.ImageIndex:=176;
   CheckBoxInput.Font.Color:=clDkGray;
  end;
  PopupMenuEditInputToggle.ImageIndex:=ActionEditInputToggle.ImageIndex;
  MenuEditInputToggle.ImageIndex:=ActionEditInputToggle.ImageIndex;
  BitBtnInputToggle.ImageIndex:=ActionEditInputToggle.ImageIndex;
  if HasFlags(Mode,1) then begin
   PanelInputCheckBox.Width:=CheckBoxInput.BoundsRect.Right+ConsoleBorderSpace;
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateInputBox');
 end;
end;

procedure TFormConsoleWindow.ClearOutput;
var i:Integer;
begin
 if Ok then
 try
  LockDraw;
  FlatCursor:=0; SelTextFlag:=False;
  for i:=0 to HighLines do FlatLines[i]:='';
 finally
  UnlockDraw;
 end;
end;

procedure TFormConsoleWindow.UpdateCommands;
var Exposed,CanPaste,CanInput:Boolean;
begin
 try
  inherited UpdateCommands;
  Exposed:=FormIsExposed(Self);
  CanPaste:=Exposed and IsClipboardHasFormat(CF_TEXT);
  CanInput:=CheckBoxInput.Checked;
  ActionFileSave.Enabled:=false;
  ActionFileSaveAs.Enabled:=false;
  ActionFilePrint.Enabled:=Exposed;
  ActionEditDelete.Enabled:=Exposed;
  ActionEditCut.Enabled:=Exposed;
  ActionEditCopy.Enabled:=Exposed;
  ActionEditPaste.Enabled:=CanPaste;
  ComboBoxInput.Enabled:=CanInput;
 except
  on E:Exception do BugReport(E,Self,'UpdateCommands');
 end;
end;

procedure TFormConsoleWindow.DrawView;
var i,cWidth:Integer; R:TRect2I;
begin
 try
  if Ok and IsFormViewable then begin
   DebugLogReport_DrawView;
   cWidth:=Max(1,CharWidthPx);
   for i:=0 to LinesPerPage-1 do begin
    R:=LineRect(i);
    DrawBar(PaintBox.Canvas,R,GroundColor);
    RectMove(R,-ScrollBarH.Position*cWidth,0);
    DrawText(PaintBox.Canvas,R.A,RingLines[ScrollBarV.Position+i],TextColor,GroundColor);
   end;
   DrawBar(PaintBox.Canvas,LineRect(LinesPerPage),GroundColor);
  end;
  if SelTextFlag then DrawSelTextRect;
 except
  on E:Exception do BugReport(E,Self,'DrawView');
 end;
end;

procedure TFormConsoleWindow.PutText(const aText:LongString);
begin
 OutFifo.PutText(aText);
end;

procedure ConsoleWindowMonitor(Index:LongInt; const aObject:TObject; var Terminate:Boolean; CustomData:Pointer);
begin
 if (aObject is TFormConsoleWindow) then
 with TFormConsoleWindow(aObject) do if MonitorEvent then Monitoring;
end;

procedure ConsoleWindowsMonitoring;
begin
 if TMasterForm.IsMonitoringOnPause then Exit;
 ConsoleWindowsMonitor.ForEach(ConsoleWindowMonitor,nil);
end;

procedure TFormConsoleWindow.StartMonitoring;
begin
 try
  if Ok and ConsoleWindowsMonitor.Ok then begin
   if (ConsoleWindowsMonitor.IndexOf(Self)<0) then begin
    ConsoleWindowsMonitor.Add(Self);
    Tick55Actions.Add(ConsoleWindowsMonitoring);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'StartMonitoring');
 end;
end;

procedure TFormConsoleWindow.StopMonitoring;
begin
 try
  if Ok and ConsoleWindowsMonitor.Ok then begin
   if (ConsoleWindowsMonitor.IndexOf(Self)>=0) then begin
    ConsoleWindowsMonitor.Remove(Self);
    if ConsoleWindowsMonitor.Count=0 then
    Tick55Actions.Remove(ConsoleWindowsMonitoring);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'StopMonitoring');
 end;
end;

procedure TFormConsoleWindow.Monitoring;
var aText:LongString;
begin
 if Ok then
 try
  if (InpFifo.Count>0) then begin
   aText:=InpFifo.GetText;
   if Assigned(myInpFilter) then myInpFilter(Self,aText);
   aText:='';
  end;
  if (OutFifo.Count>0) then begin
   aText:=OutFifo.GetText;
   if Assigned(myOutFilter) then myOutFilter(Self,aText);
   if (Length(aText)>0) then UpdateLines(aText);
   aText:='';
  end;
  if Assigned(myPolling) then myPolling(Self);
  if PendingTrigger[ptrg_FormActivate] then begin
   PendingTrigger[ptrg_FormActivate]:=false;
   if AutoFocusInput then PendingTrigger[ptrg_FocusInput]:=true;;
  end;
  if PendingTrigger[ptrg_FocusInput] then begin
   PendingTrigger[ptrg_FocusInput]:=false;
   if Active and not ComboBoxInput.Focused then
   if ComboBoxInput.CanSetFocus then ComboBoxInput.SetFocus;
  end;
 except
  on E:Exception do BugReport(E,Self,'Monitoring');
 end;
end;

procedure TFormConsoleWindow.FormCreate(Sender: TObject);
begin
 inherited;
 myFlatLines:=TStringList.Create;
 LinesCount:=ConsoleDefLinesCount;
 HighColumn:=ConsoleDefHighColumn;
 TabWidth:=ConsoleDefTabWidth;
 myXorSelector:=NewXorSelector;
 myXorSelector.Master:=@myXorSelector;
 myXorSelector.Mode:=sm_Rect;
 myXorSelector.Width:=1;
 ToolBar.Hide;
 StatusBar.Hide;
 myInpFifo:=nil;
 myInpOwns:=false;
 myInpFilter:=nil;
 myOutFifo:=nil;
 myOutOwns:=false;
 myOutFilter:=nil;
 myCustom:=nil;
 myGuardInput:=ga_Guest;
 try
  LockDraw;
  TextColor:=clBtnText;
  GroundColor:=clBtnFace;
  ClearOutput;
  UpdateScrollBars;
 finally
  UnlockDraw;
 end;
 if SdiMan.IsSdiMode then GroupBoxInput.Cursor:=SdiMan.CursorActivateMainSdiForm;
 if SdiMan.IsSdiMode then GroupBoxOutput.Cursor:=SdiMan.CursorActivateMainSdiForm;
 UpdateMenu(MenuEdit,
            RusEng('Правка','Edit')+MenuRightSpace,
            RusEng('Меню для редактирования.','Editor operations.'),
            0);
 UpdateMenu(MenuEditDelete,
            RusEng('Удалить','Delete'),
            RusEng('Удалить выделенный фрагмент.','Delete selected region.'),
            ShortCut(VK_DELETE,[ssCtrl]));
 UpdateMenu(MenuEditCut,
            RusEng('Вырезать','Cut'),
            RusEng('Вырезать в буфер обмена.','Cut to clipboard.'),
            ShortCut(Word('X'),[ssCtrl]));
 UpdateMenu(MenuEditCopy,
            RusEng('Копировать','Copy'),
            RusEng('Копировать в буфер обмена.','Copy to clipboard.'),
            ShortCut(Word('C'),[ssCtrl]));
 UpdateMenu(MenuEditPaste,
            RusEng('Вставить','Paste'),
            RusEng('Вставить из буфера обмена.','Paste from clipboard.'),
            ShortCut(Word('V'),[ssCtrl]));
 UpdateMenu(MenuEditInputToggle,
            RusEng('Ввод','Input'),
            RusEng('Разрежить/Запретить Консольный Ввод.','Enable/Disable Console Input.'),
            ShortCut(Word('I'),[ssCtrl]));
 UpdateMenu(MenuViewScrollXEnd,
            RusEng('Прокрутка …','Scroll to …'),
            RusEng('Меню прокрутки (обзор текста).','Scrolling menu (text observe).'),
            0);
 UpdateMenu(MenuViewScrollLtEnd,
            RusEng('Левый   край','Scroll to Left   End'),
            RusEng('Прокрутка влево до конца.','Scroll to Left End.'),
            0);
 UpdateMenu(MenuViewScrollRtEnd,
            RusEng('Правый  край','Scroll to Right  End'),
            RusEng('Прокрутка вправо до конца.','Scroll to Right End.'),
            0);
 UpdateMenu(MenuViewScrollUpEnd,
            RusEng('Верхний край','Scroll to Top    End'),
            RusEng('Прокрутка вверх до конца.','Scroll to Top End.'),
            0);
 UpdateMenu(MenuViewScrollDnEnd,
            RusEng('Нижний  край','Scroll to Bottom End'),
            RusEng('Прокрутка вниз до конца.','Scroll to Bottom End.'),
            0);
 UpdateMenu(MenuViewEditConsoleParams,
            RusEng('Параметры Консоли …','Console Settings …'),
            RusEng('Редактировать параметры Консоли.','Edit Console Settings.'),
            0);
 GroupBoxInput.Caption:=RusEng('Ввод:','Input:');
 GroupBoxOutput.Caption:=RusEng('Вывод:','Output:');
 GroupBoxInput.Hint:=RusEng('Консольный Ввод.','Console Input.');
 GroupBoxOutput.Hint:=RusEng('Консольный Вывод.','Console Output.');
 CheckBoxInput.Hint:=RusEng('Разрешить/Запретить Ввод.','Enable/Disable Input.');
 GroupBoxScrollButtons.Hint:=RusEng('Управление обзором (прокрутка).','Scrolling controls.');
 GroupBoxScrollButtons.Caption:=RusEng('Обзор:','Scroll:');
 CheckBoxInput.Checked:=AutoEnableInput;
 BitBtnInputToggle.Hint:=CheckBoxInput.Hint;
 BitBtnInputToggle.Caption:='';
 BitBtnViewScrollLtEnd.Caption:='';
 BitBtnViewScrollRtEnd.Caption:='';
 BitBtnViewScrollUpEnd.Caption:='';
 BitBtnViewScrollDnEnd.Caption:='';
 CheckBoxInput.Font.Bold:=True;
 UpdateScrollBars(1);
 UpdateInputBox(1);
end;

procedure TFormConsoleWindow.FormDestroy(Sender: TObject);
begin
 ClearOutput;
 AssignFifo(nil,false,nil,nil,false,nil);
 Kill(myXorSelector);
 Kill(myFlatLines);
 inherited;
end;

procedure TFormConsoleWindow.FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
 function NoneActiveInput:Boolean; inline;
 begin
  Result:=(ActiveControl<>ComboBoxInput);
 end;
 function NoneShift:Boolean; inline;
 begin
  Result:=(Shift=[]);
 end;
 function CtrlShift:Boolean; inline;
 begin
  Result:=(Shift=[ssCtrl]);
 end;
begin
 inherited;
 if Assigned(Self) then begin
  case Key of
   VK_UP: if NoneShift and NoneActiveInput then begin
    ActionViewScrollUpSmall.Execute;
    Key:=VK_UNKNOWN;
   end;
   VK_DOWN: if NoneShift and NoneActiveInput then begin
    ActionViewScrollDnSmall.Execute;
    Key:=VK_UNKNOWN;
   end;
   VK_LEFT: if NoneShift and NoneActiveInput then begin
    ActionViewScrollLtSmall.Execute;
    Key:=VK_UNKNOWN;
   end;
   VK_RIGHT: if NoneShift and NoneActiveInput then begin
    ActionViewScrollRtSmall.Execute;
    Key:=VK_UNKNOWN;
   end;
   VK_PRIOR: if NoneShift or CtrlShift then begin
    if CtrlShift then ActionViewScrollUpEnd.Execute else
    ActionViewScrollUpLarge.Execute;
    Key:=VK_UNKNOWN;
   end;
   VK_NEXT: if NoneShift or CtrlShift then begin
    if CtrlShift then ActionViewScrollDnEnd.Execute else
    ActionViewScrollDnLarge.Execute;
    Key:=VK_UNKNOWN;
   end;
   VK_HOME: if NoneShift and NoneActiveInput then begin
    ActionViewScrollLtEnd.Execute;
    Key:=VK_UNKNOWN;
   end;
   VK_END: if NoneShift and NoneActiveInput then begin
    ActionViewScrollRtEnd.Execute;
    Key:=VK_UNKNOWN;
   end;
  end;
 end;
end;

procedure TFormConsoleWindow.AfterConstruction;
begin
 inherited AfterConstruction;
 FullConsoleWindowList.Add(Self);
 AddonSdiFlags(sf_SdiConsole);
end;

procedure TFormConsoleWindow.BeforeDestruction;
begin
 StopMonitoring;
 FullConsoleWindowList.Remove(Self);
 inherited BeforeDestruction;
end;

procedure TFormConsoleWindow.FormResize(Sender: TObject);
begin
 inherited;
 UpdateScrollBars(1);
 UpdateInputBox(1);
end;

procedure TFormConsoleWindow.GroupBoxInputDblClick(Sender: TObject);
begin
 SdiMan.ActivateMainForm;
end;

procedure TFormConsoleWindow.GroupBoxOutputDblClick(Sender: TObject);
begin
 SdiMan.ActivateMainForm;
end;

procedure TFormConsoleWindow.PaintBoxMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
 procedure SelectTextUnderPix(X,Y:Integer; Timeout:Integer);
 var rp,rc,rs,rb:TRect2I; s,q:LongString; ms:Int64;
 begin
  s:=''; q:='';
  SelTextFlag:=False;
  try
   ms:=GetTickCount64;
   rp.a:=PixToPos(Point2I(x,y));
   rp.b:=rp.a; PointMove(rp.b,1,1);
   rs:=rp; s:=Trim(TextFromRect(rp));
   // With Shift+Left+DblClick the whole line should be selected
   if IsNonEmptyStr(s) and (Shift*[ssCtrl,ssAlt,ssShift]=[ssShift]) then begin
    rs.a.x:=0; rs.b.x:=HighColumn; s:=TextFromRect(rs); rs.b.x:=Length(s);
    if IsNonEmptyStr(s) and not RectIsEmpty(rs) then begin
     SelTextFlag:=True; SelTextRect:=rs;
     Clipboard.AsText:=s;
    end;
    Exit;
   end;
   if IsNonEmptyStr(s) then begin
    rc:=rp; // scan chars before
    while (rc.a.x>0) do begin
     RectMove(rc,-1,0);
     q:=TextFromRect(rc);
     if (GetTickCount64-ms>Timeout) then Exit;
     if IsEmptyStr(q) or HasChars(q,ScanSpaces) then Break;
     rs:=RectUnion(rs,rc);
     s:=q+s;
    end;
   end;
   if IsNonEmptyStr(s) then begin
    rc:=rp; // scan chars after
    while (rc.a.x>0) do begin
     RectMove(rc,+1,0);
     q:=TextFromRect(rc);
     if (GetTickCount64-ms>Timeout) then Exit;
     if IsEmptyStr(q) or HasChars(q,ScanSpaces) then Break;
     rs:=RectUnion(rs,rc);
     s:=s+q;
    end;
   end;
   rb:=Rect2I(0,0,HighColumn,HighLines);
   if RectIsEqual(rs,RectIntersection(rs,rb)) then
   if IsNonEmptyStr(s) and not RectIsEmpty(rs) then begin
    SelTextFlag:=True; SelTextRect:=rs;
    Clipboard.AsText:=s;
   end;
  finally
   LockUnlockDraw;
  end;
 end;
 function MatchWordSelector:Boolean;
 begin
  Result:=(Button=mbLeft) and (ssDouble in Shift);
 end;
 function MatchRectSelector:Boolean;
 begin
  Result:=False;
  if (Button=mbLeft) then
  if not (ssDouble in Shift) then
  if (Shift*[ssCtrl,ssAlt,ssShift]=[])
  or (Shift*[ssCtrl,ssAlt,ssShift]=[ssShift])
  then Result:=True;
 end;
begin
 if Assigned(Self) then
 try
  inherited;
  if (Button=mbLeft) then begin
   if MatchWordSelector then begin
    SelectTextUnderPix(X,Y,1000);
   end else
   if MatchRectSelector then begin
    if myXorSelector.IsStart then myXorSelector.Stop(PaintBox.Canvas);
    myXorSelector.Start(PaintBox.Canvas,Point2I(X,Y));
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'PaintBoxMouseDown');
 end;
end;

procedure TFormConsoleWindow.PaintBoxMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
begin
 if Assigned(Self) then
 try
  inherited;
  if myXorSelector.IsStart then begin
   myXorSelector.ReplaceXY(PaintBox.Canvas,X,Y);
  end;
 except
  on E:Exception do BugReport(E,Self,'PaintBoxMouseMove');
 end;
end;

procedure TFormConsoleWindow.PaintBoxMouseUp(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
 procedure HandleSelection(const Sel:TRect2I);
 var rs,rb:TRect2I; s,q:LongString; i,len:Integer;
 begin
  // Avoid too small rectangles
  if (RectSize(Sel).x<ScaleValMulDiv(CharWidthPx,1,2)) then Exit;
  if (RectSize(Sel).y<ScaleValMulDiv(RectSize(LineRect(0)).y,1,2)) then Exit;
  s:=''; q:='';
  SelTextFlag:=False;
  try
   rs.a:=PixToPos(Sel.a);
   rs.b:=PixToPos(Sel.b);;
   PointMove(rs.b,1,1); len:=0;
   if (ssShift in Shift) then begin
    rs.a.x:=0; rs.b.x:=HighColumn;
   end;
   s:=Trim(TextFromRect(rs));
   for i:=1 to WordCount(s,EolnDelims) do begin
    q:=ExtractWord(i,s,EolnDelims);
    len:=Max(len,utf8_length(q));
   end;
   if (len>0) then rs.b.x:=rs.a.x+len;
   rb:=Rect2I(0,0,HighColumn,HighLines);
   if RectIsEqual(rs,RectIntersection(rs,rb)) then
   if IsNonEmptyStr(s) and not RectIsEmpty(rs) then begin
    SelTextFlag:=True; SelTextRect:=rs;
    Clipboard.AsText:=s;
   end;
  finally
   LockUnlockDraw;
  end;
 end;
begin
 if Assigned(Self) then
 try
  inherited;
  if myXorSelector.IsStart then begin
   myXorSelector.Stop(PaintBox.Canvas);
   HandleSelection(RectValidate(myXorSelector.Selection));
  end;
 except
  on E:Exception do BugReport(E,Self,'PaintBoxMouseUp');
 end;
end;

procedure TFormConsoleWindow.PanelOutputDblClick(Sender: TObject);
begin
 SdiMan.ActivateMainForm;
end;

procedure TFormConsoleWindow.PaintBoxPaint(Sender: TObject);
begin
 inherited;
 try
  LockDraw;
 finally
  UnlockDraw;
 end;
end;

procedure TFormConsoleWindow.ScrollBarVChange(Sender: TObject);
begin
 inherited;
 try
  LockDraw;
 finally
  UnlockDraw;
 end;
end;

procedure TFormConsoleWindow.ScrollBarHChange(Sender: TObject);
begin
 inherited;
 try
  LockDraw;
 finally
  UnlockDraw;
 end;
end;

procedure TFormConsoleWindow.ComboBoxInputDblClick(Sender: TObject);
var History:Integer;
begin
 if Guard.CheckAction(GuardInput,'TFormConsoleWindow.Input')<0 then Exit;
 if Ok then
 try
  inherited;
  History:=SysGlossary.ReadIniLinkDef(SysIniFile,SectSystem,'ConsoleInputHistory',50);
  RecordComboBoxHistory(ComboBoxInput,History);
  InpFifo.PutText(ComboBoxInput.Text+EOL);
  ComboBoxInput.Text:='';
 except
  on E:Exception do BugReport(E,Self,'ComboBoxInputDblClick');
 end;
end;

procedure TFormConsoleWindow.ComboBoxInputKeyDown(Sender: TObject;  var Key: Word; Shift: TShiftState);
 procedure UpDown(step:Integer);
 var i:Integer;
 begin
  if (ComboBoxInput.Items.Count>0) then begin
   i:=EnsureRange(ComboBoxInput.ItemIndex+Step,0,ComboBoxInput.Items.Count-1);
   ComboBoxInput.ItemIndex:=i;
   Key:=0;
  end;
 end;
begin
 inherited;
 case Key of
  VK_RETURN : ComboBoxInputDblClick(Sender);
  VK_DOWN   : UpDown(-1);
  VK_UP     : UpDown(+1);
 end;
end;

procedure TFormConsoleWindow.CheckBoxInputChange(Sender: TObject);
begin
 UpdateInputBox;
 PendingTrigger[ptrg_FocusInput]:=true;
end;

procedure TFormConsoleWindow.ActionEditInputEnableExecute(Sender: TObject);
begin
 if Ok then begin
  CheckBoxInput.Checked:=true;
  UpdateInputBox;
 end;
end;

procedure TFormConsoleWindow.ActionEditInputToggleExecute(Sender: TObject);
begin
 if Ok then begin
  if Guard.CheckAction(ga_Guest,ActionEditInputToggle)<0 then Exit;
  CheckBoxInput.Checked:=not CheckBoxInput.Checked;
  UpdateInputBox;
 end;
end;

procedure TFormConsoleWindow.ActionEditInputDisableExecute(Sender: TObject);
begin
 if Ok then begin
  CheckBoxInput.Checked:=false;
  UpdateInputBox;
 end;
end;

procedure TFormConsoleWindow.FormActivate(Sender: TObject);
begin
 inherited;
 UpdateInputBox;
end;

procedure TFormConsoleWindow.ActionEditDeleteExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditDelete)<0 then Exit;
 inherited;
 ClearOutput;
end;

procedure TFormConsoleWindow.ActionEditCutExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditCut)<0 then Exit;
 inherited;
 ActionEditCopy.Execute;
 ClearOutput;
end;

procedure TFormConsoleWindow.ActionEditCopyExecute(Sender: TObject);
var i,j:Integer; List:TStringList;
begin
 if Guard.CheckAction(ga_Guest,ActionEditCopy)<0 then Exit;
 if Ok then
 try
  inherited;
  if SelTextFlag and (SelText<>'') then begin
   Clipboard.AsText:=SelText;
   Exit;
  end;
  List:=TStringList.Create;
  try
   j:=-1;
   for i:=0 to HighLines do begin
    if (j<0) and (RingLines[i]<>'') then j:=i;
    if (j>=0) then List.Add(RingLines[i]);
   end;
   if (List.Count>0)
   then Clipboard.AsText:=List.Text;
  finally
   Kill(List);
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditCopyExecute');
 end;
end;

procedure TFormConsoleWindow.ActionEditPasteExecute(Sender: TObject);
 procedure PasteText(s:LongString);
 var p,l:Integer; t:LongString;
 begin
  if (s='') then Exit;
  if (PosEol(s)>0) then // Skip empty lines of the text
  while IsEmptyStr(ExtractWord(1,s,EolnDelims)) do begin
   s:=SkipWords(1,s,EolnDelims);
   if IsEmptyStr(s) then Break;
  end;
  // Extract 1st line of clip text, drop selection
  s:=ExtractWord(1,s,EolnDelims); if (s='') then Exit;
  if (ComboBoxInput.SelLength>0) then ComboBoxInput.SelText:='';
  t:=ComboBoxInput.Text;
  p:=ComboBoxInput.SelStart+1;
  if IsSysUtf8 and utf8_valid(t+s) then begin
   t:=utf8_copy(t,1,p-1)+s+utf8_copy(t,p,utf8_length(t)-p+1);
   l:=utf8_length(s);
  end else begin
   t:=copy(t,1,p-1)+s+copy(t,p,length(t)-p+1);
   l:=length(s);
  end;
  ComboBoxInput.Text:=t;
  ComboBoxInput.SelStart:=p-1+l;
 end;
begin
 if Guard.CheckAction(ga_Guest,ActionEditPaste)<0 then Exit;
 if Ok then
 try
  inherited;
  if Clipboard.HasFormat(CF_TEXT) then begin
   if ComboBoxInput.Enabled
   then PasteText(Clipboard.AsText);
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditPasteExecute');
 end;
end;

procedure TFormConsoleWindow.ActionViewScrollLtEndExecute(Sender: TObject);
begin
 if Assigned(Self) then
 try
  ScrollBarH.Position:=ScrollBarH.Min;
 except
  on E:Exception do BugReport(E,Self,'ActionViewScrollLEndExecute');
 end;
end;

procedure TFormConsoleWindow.ActionViewScrollRtEndExecute(Sender: TObject);
var sp:Integer;
begin
 if Assigned(Self) then
 try
  sp:=MaxLineLength-ColumnsPerPage+1;
  ScrollBarH.Position:=EnsureRange(sp,ScrollBarH.Min,ScrollBarH.Max);
 except
  on E:Exception do BugReport(E,Self,'ActionViewScrollREndExecute');
 end;
end;

procedure TFormConsoleWindow.ActionViewScrollUpEndExecute(Sender: TObject);
var i,ib,ie,sp:Integer;
begin
 if Assigned(Self) then
 try
  sp:=ScrollBarV.Min;
  ib:=ScrollBarV.Min;
  ie:=ScrollBarV.Max;
  for i:=ib to ie do begin
   if (RingLines[i]='')
   then continue;
   sp:=i; break;
  end;
  ScrollBarV.Position:=sp;
 except
  on E:Exception do BugReport(E,Self,'ActionViewScrollTEndExecute');
 end;
end;

procedure TFormConsoleWindow.ActionViewScrollDnEndExecute(Sender: TObject);
begin
 if Assigned(Self) then
 try
  ScrollBarV.Position:=ScrollBarV.Max;
 except
  on E:Exception do BugReport(E,Self,'ActionViewScrollBEndExecute');
 end;
end;

procedure TFormConsoleWindow.ActionViewScrollLtSmallExecute(Sender: TObject);
begin
 if Assigned(Self) then
 try
  with ScrollBarH do Position:=EnsureRange(Position-SmallChange,Min,Max);
 except
  on E:Exception do BugReport(E,Self,'ActionViewScrollLtSmallExecute');
 end;
end;

procedure TFormConsoleWindow.ActionViewScrollRtSmallExecute(Sender: TObject);
begin
 if Assigned(Self) then
 try
  with ScrollBarH do Position:=EnsureRange(Position+SmallChange,Min,Max);
 except
  on E:Exception do BugReport(E,Self,'ActionViewScrollRtSmallExecute');
 end;
end;

procedure TFormConsoleWindow.ActionViewScrollUpSmallExecute(Sender: TObject);
begin
 if Assigned(Self) then
 try
  with ScrollBarV do Position:=EnsureRange(Position-SmallChange,Min,Max);
 except
  on E:Exception do BugReport(E,Self,'ActionViewScrollUpSmallExecute');
 end;
end;

procedure TFormConsoleWindow.ActionViewScrollDnSmallExecute(Sender: TObject);
begin
 if Assigned(Self) then
 try
  with ScrollBarV do Position:=EnsureRange(Position+SmallChange,Min,Max);
 except
  on E:Exception do BugReport(E,Self,'ActionViewScrollDnSmallExecute');
 end;
end;

procedure TFormConsoleWindow.ActionViewScrollUpLargeExecute(Sender: TObject);
begin
 if Assigned(Self) then
 try
  with ScrollBarV do Position:=EnsureRange(Position-LargeChange,Min,Max);
 except
  on E:Exception do BugReport(E,Self,'ActionViewScrollUpLargeExecute');
 end;
end;

procedure TFormConsoleWindow.ActionViewScrollDnLargeExecute(Sender: TObject);
begin
 if Assigned(Self) then
 try
  with ScrollBarV do Position:=EnsureRange(Position+LargeChange,Min,Max);
 except
  on E:Exception do BugReport(E,Self,'ActionViewScrollDnLargeExecute');
 end;
end;

procedure TFormConsoleWindow.ActionViewEditConsoleParamsExecute(Sender: TObject);
var Cap,Tab,Params:LongString; iv,nc:Integer;
 procedure AddTabLine(var Tab:LongString; id,sn,sv:LongString);
 begin
  Tab:=Tab+Format('%s|%s|%s',[id,sn,sv])+EOL;
 end;
begin
 if Assigned(Self) then
 try
  Cap:=''; Tab:=''; Params:=''; iv:=0; nc:=0;
  Cap:=SessionManager.SessionHead+' - '+RusEng('Параметры Консоли …','Console Settings …');
  Params:=ControlPosParams(PaintBox);
  AddTabLine(Tab,'LinesCount',RusEng('Длина Истории, строк','History length, lines'),IntToStr(LinesCount));
  AddTabLine(Tab,'HighColumn',RusEng('Ширина строк, сиволов','Line width, chars'),IntToStr(HighColumn));
  AddTabLine(Tab,'TabWidth',RusEng('Ширина TAB, сиволов','TAB width, chars'),IntToStr(TabWidth));
  if ExecuteFormStringGridEditDialog(Cap,Tab,'|',0,2,False,Params)=mrOk then begin
   Tab:=StringReplace(Tab,'|','=',[rfReplaceAll]);
   iv:=StrToIntDef(SkipWords(1,CookieScan(Tab,'LinesCount'),['=']),LinesCount);
   if (iv<>LinesCount) and InRange(iv,ConsoleMinLinesCount,ConsolemaxLinesCount) then begin
    LinesCount:=iv;
    Inc(nc);
   end;
   iv:=StrToIntDef(SkipWords(1,CookieScan(Tab,'HighColumn'),['=']),HighColumn);
   if (iv<>HighColumn) and InRange(iv,ConsoleMinHighColumn,ConsoleMaxHighColumn) then begin
    HighColumn:=iv;
    Inc(nc);
   end;
   iv:=StrToIntDef(SkipWords(1,CookieScan(Tab,'TabWidth'),['=']),TabWidth);
   if (iv<>TabWidth) and InRange(iv,ConsoleMinTabWidth,ConsoleMaxTabWidth) then begin
    TabWidth:=iv;
    Inc(nc);
   end;
   if (nc>0) then begin
    UpdateScrollBars(1);
    UpdateInputBox(1);
    LockUnlockDraw;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditConsoleParamsExecute');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_consolewindow;
begin
 TFormConsoleWindow.AutoFocusInput:=false;
 TFormConsoleWindow.AutoEnableInput:=false;
 FullConsoleWindowList:=NewConsoleWindowList(false);
 FullConsoleWindowList.Master:=@FullConsoleWindowList;
 ConsoleWindowsMonitor:=NewConsoleWindowList(false);
 ConsoleWindowsMonitor.Master:=@ConsoleWindowsMonitor;
end;

procedure Free_form_consolewindow;
begin
 ResourceLeakageLog(Format('%-60s = %d',['FullConsoleWindowList.Count', FullConsoleWindowList.Count]));
 ResourceLeakageLog(Format('%-60s = %d',['ConsoleWindowsMonitor.Count', ConsoleWindowsMonitor.Count]));
 Kill(FullConsoleWindowList);
 Kill(ConsoleWindowsMonitor);
end;

initialization

 Init_form_consolewindow;

finalization

 Free_form_consolewindow;

end.

//////////////
// END OF FILE
//////////////

