////////////////////////////////////////////////////////////////////////////////
//                                                                            //
// Copyright (c) 2021 Alexey Kuryakin kouriakine@mail.ru under LGPL license.  //
//                                                                            //
// Purpose:                                                                   //
//  Header for SMI++ Runtime Library SMIRTL.DLL                               //
//  See http://smi.web.cern.ch                                                //
//                                                                            //
// History:                                                                   //
//  20201123 - 1st release                                                    //
//  20210122 - Update to v56r1                                                //
////////////////////////////////////////////////////////////////////////////////

unit smirtl; // SMI++ runtime library header, see http://smi.web.cern.ch

interface

uses SysUtils;

 //
 // parameters.h
 //
const
  MAXRECL           = 81;
  MAXNAMESIZE       = 33;
  MAXOBJECTLINES    = 5000;
  MAXCLASSLINES     = 100;
  MAXSTATES         = 100;
  MAXATRIBUTES      = 10;
  MAXSUBOBJECTS     = 50;
  MAXACTIONS        = 100;
  MAXINS            = 100;
  MAXPARA           = 5;
  MAXWHENS          = 20;

 ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
 //
 // smirtl.h
 //
 // SMIRTL implements the routines used by a proxy in order to connect,
 //        receive commands and transmit state changes to its associated object in a SM.
 //
 //
 // int smi_attach(obj_name, command_handler)
 // Parameters:
 //     char *obj_name              The name of the associated object to connect to. In the form <domain_name>::<object_name>
 //     void (*command_handler)()   The routine to be called when a command arrives from the SM, handle_command is called with no parameters.
 // Description:
 //     smi_attach should be called by a proxy at startup in order to connect to its associated object in the SM.
 // Return:
 //     0/1 if object not/attached.
 //
 //
 // void smi_volatile()
 // Parameters:
 //     none
 // Description:
 //     Smi_volatile() makes a proxy live only while its associated SMI Domain is active, i.e., a proxy can call
 //     smi_volatile() (normally at start-up) in order to die whenever the SMI domain the proxy is attached to dies.
 //
 //
 // int smi_set_state(state)
 // Parameters:
 //     char *state         The state of the proxy.
 // Description:
 //     Set the initial or new state of the proxy by sending it to its associated object in the SM.
 //     An smi_set_state should be done at start-up in order to communicate the proxy's initial state.
 // Return:
 //     1.
 //
 //
 // smi_terminate_action(state)
 // Parameters:
 //     char *state         The state of the proxy.
 // Description:
 //     Set the new state of the proxy after executing an action, by sending it to its associated object in the SM.
 // Return:
 //     1.
 //
 //
 // int smi_get_action(action, n_params)
 // Parameters:
 //     char *action        The last action received
 //     int *n_params       Number of parameters with this action
 // Description:
 //     When inside the comman_handler the user can use this routine to get the action name and nr. of parameters.
 // Return:
 //     0/1 if action is not/valid.
 //
 //
 // int smi_get_command(cmnd)
 // Parameters:
 //     char *cmnd          The command just received
 //     int *size           The command size (length)
 // Description:
 //     When inside the comman_handler the user can use this routine to get the command, the command is composed
 //     of the action name and any parameters in the form action/par0=val0/par1=val1 .
 // Return:
 //     0/1 if command not/valid.
 //
 //
 // int smi_test_action(action)
 // Parameters:
 //     char *action        Action name to compare with the received action.
 // Description:
 //     Test if the action received is "action". The function returns 1 if it is, 0 if not.
 // Return:
 //     0/1 if action not/match.
 //
 //
 // int smi_get_next_par(param, type, size)
 // Parameters:
 //     char *param         The name of the parameter
 //     int *type           The type of the parameter (STRING, INTEGER, FLOAT)
 //     int *size           The size of the parameter
 // Description:
 //     When inside the command handler this routine can be used to retrieve the parameters present on a received command.
 //     The function will return 0 when all parameters have been read, otherwise it returns the parameter name,
 //     the parameter type and the size necessary to store the parameter.
 // Return:
 //     0/1 if parameter not/retrieved.
 //
 //
 // int smi_get_par_value(param, value)
 // Parameters:
 //     char *param         The name of the parameter
 //     void *value         The parameter value
 // Description:
 //     When inside the command handler this routine can be used to retrieve the parameter value for the parameter
 //     returned by smi_get_next_par. The value argument will be filled with either an integer, a double float or
 //     a string (the size was specified in the previous routine) depending on the parameter type.
 //     This function can also be used directly if the user knows the parameter name, type and size.
 //     The function will return 0 if the specified parameter does not exist.
 // Return:
 //     0/1 if parameter not/retrieved.
 //
 //
 // int smi_set_par(par, value, type)
 // Parameters:
 //     char *par           The parameter name.
 //     void *value         The value of the parameter (passed by reference).
 //     int type            The parameter type: (STRING, INTEGER or FLOAT).
 // Description:
 //     Smi_set_par should be used in order to set object parameters for associated objects.
 //     If an associated objects has parameters, smi_set_par should be called once for each parameter before the first smi_set_state. 
 //     Further parameter setting can be done by the proxy at any time but they will only take effect at the next smi_set_state.
 // Return:
 //     0/1 if parameter was not/set.
 //
 //
 // void smi_register_termination(exit_handler, context)
 // Parameters:
 //     int (*exit_handler)(void *, int *)    
 //     void *context
 // Description:
 //     Internal use.
 //
 ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

const smirtl_dll                    = 'smirtl.dll';                     // SMI runtime DLL
const SMIXX_VERSION                 = 5601;                             // Version of SMI++
const MAX_NAME                      = 132;                              // Max. size of name
const SMI_STRING                    = 0;                                // String  parameter
const SMI_INTEGER                   = 1;                                // Integer parameter
const SMI_FLOAT                     = 2;                                // Float   parameter
type  PAR_TYPES                     = SMI_STRING .. SMI_FLOAT;          // Parameter types
type  smi_cmnd_handler_t            = procedure(); cdecl;               // Command handler
type  smi_exit_handler_t            = function(context:Pointer; var ExitCode:Integer):Integer; cdecl; // Exit handler
 
function  smi_attach(obj_name:PChar; command_handler: smi_cmnd_handler_t): Integer; cdecl;
procedure smi_volatile(); cdecl;
function  smi_set_state(state:PChar): Integer; cdecl;
function  smi_get_state(state:PChar; max_len:Integer): Integer; cdecl;
function  smi_terminate_action(state:PChar): Integer; cdecl;
function  smi_get_action(cmnd:PChar; var n_pars:Integer): Integer; cdecl;
function  smi_get_command(cmnd:PChar; var size:Integer): Integer; cdecl;
function  smi_test_action(cmnd:PChar): Integer; cdecl;
function  smi_get_next_par(param:PChar; var typ:Integer; var size:Integer): Integer; cdecl;
function  smi_get_par_value(param:PChar; value:Pointer): Integer; cdecl;
function  smi_set_par(param:PChar; value:Pointer; typ:Integer): Integer; cdecl;
procedure smi_register_termination(exit_handler:smi_exit_handler_t; context:Pointer); cdecl;

implementation

function  smi_attach;               external smirtl_dll name 'smi_attach';
procedure smi_volatile;             external smirtl_dll name 'smi_volatile';
function  smi_set_state;            external smirtl_dll name 'smi_set_state';
function  smi_get_state;            external smirtl_dll name 'smi_get_state';
function  smi_terminate_action;     external smirtl_dll name 'smi_terminate_action';
function  smi_get_action;           external smirtl_dll name 'smi_get_action';
function  smi_get_command;          external smirtl_dll name 'smi_get_command';
function  smi_test_action;          external smirtl_dll name 'smi_test_action';
function  smi_get_next_par;         external smirtl_dll name 'smi_get_next_par';
function  smi_get_par_value;        external smirtl_dll name 'smi_get_par_value';
function  smi_set_par;              external smirtl_dll name 'smi_set_par';
procedure smi_register_termination; external smirtl_dll name 'smi_register_termination';

end.
