#!/bin/bash

###########################################################
## Copyright (c) 2002-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## _root_hist_1d.sh - make data file histogram with ROOT
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

# Usage: _root_hist_1d.sh  file.dat
# Prepare environment for call ROOT
# Then call ROOT, wait and show GIF

#####################################################################
## Usage: delete_file $1
## Check file $1 and delete if one exists.
#####################################################################
function delete_file() {
 if [ -z "$1" ]; then return 0; fi;
 if [ -e "$1" ]; then rm -fv "$1" | tee -a "$logFile"; fi;
 if [ -e "$1" ]; then return 1; fi;
 return 0;
}

#####################################################################
## Usage: show_usage $1
## Print usage information on program $1.
#####################################################################
function show_usage() {
 echo "Usage: $1 datafile.dat";
 echo "Process datafile.dat via ROOT script $scriptbase.C";
 return 0;
}

#####################################################################
## init_and_check_variables "$1" - initialize & check uses variables
#####################################################################
function init_and_check_variables() {
 if [ $# -eq 0 ]; then
  show_usage "$scriptname";
  fatal 1 "Error: No arguments specified!";
  exit 1;
 fi;
 case $1 in
  *.dat) export datFile="$(realpath "$1")" ;;
  *)     fatal 1 "Error: expected *.dat file."; ;;
 esac;
 local logdir="$CRW_DAQ_VAR_TMP_DIR";
 if [ ! -d "$logdir" ]; then logdir="$TMPDIR"; fi;
 if [ ! -d "$logdir" ]; then logdir="/tmp"; fi;
 export logFile="$logdir/$scriptbase.log";
 export gifFile="$(dirname "$datFile")/$scriptbase.gif";
 delete_file "$logFile";
 delete_file "$gifFile";
 date +%Y.%m.%d-%H.%M.%S:     | tee -a "$logFile";
 echo "User:     $USER"       | tee -a "$logFile";
 echo "Command:  $scriptFILE" | tee -a "$logFile"
 echo "Argument: $1"          | tee -a "$logFile"
 echo "StartDir: $startupdir" | tee -a "$logFile";
 echo "ProgDir:  $scriptHOME" | tee -a "$logFile";
 echo "ProgName: $scriptname" | tee -a "$logFile";
 echo "DataDir:  $datadir"    | tee -a "$logFile";
 echo "logFile:  $logFile"    | tee -a "$logFile";
 echo "datFile:  $datFile"    | tee -a "$logFile";
 echo "gifFile:  $gifFile"    | tee -a "$logFile";
 return 0;
}

declare -i zenPID=0;
#####################################################################
## Usage: progress_bar $1 $2
## Show/hide ($1) progress bar with identifier $2.
#####################################################################
function progress_bar() {
 local mode="$(echo -n "$1" | tr '[:upper:]' '[:lower:]')";
 local title="$(langstr ru "$2: Подождите PID $$ …" en "$2: Wait PID $$ …")";
 local comment="$(langstr ru "$2 выполняет работу, подождите немного …" en "$2 is running, please wait for a while …")";
 if [ "$mode" = "show" ]; then
  zenity --progress --title "$title" --text "$comment" --pulsate --timeout 60000 &
  let "zenPID=$!"; echo "zenity PID $zenPID started." | tee -a "$logFile";
 fi;
 if [ "$mode" = "hide" ]; then
  if [ $zenPID -gt 0 ] && [ -n "$(pgrep -x zenity | grep "$zenPID")" ]; then
   unix closewindow -e zenity "$title" | tee -a "$logFile";
   unix sleep 1;
  fi;
  if [ $zenPID -gt 0 ] && [ -n "$(pgrep -x zenity | grep "$zenPID")" ]; then
   skill -v -p $zenPID | tee -a "$logFile";
   unix sleep 1;
  fi;
  let zenPID=0;
 fi;
 return 0;
}

#####################################################################
## Usage: wait_for_file $1 $2
## Wait until file $1 exist or timeout $2 seconds.
#####################################################################
function wait_for_file() {
 if [ -z "$1" ]; then return 0; fi;       # if empty FileName, return
 if [ -e "$1" ]; then return 0; fi;       # if exist FileName, return
 if [ -z "$2" ]; then return 0; fi;       # if empty Timeout, return
 if [ $2 -le 0 ]; then return 0; fi;      # if Timeout <= 0,  return
 unix sleep 1;                            # sleep 1 second
 wait_for_file "$1" "$(echo $2-1 | bc)";  # recursive call, Timeout-1
 return 0;                                # done
}

#####################################################################
## Usage: show_report_on_success $1
## Show report on success and wait timeout $1 seconds.
#####################################################################
function show_report_on_success() {
 echo ""                 | tee -a "$logFile";
 echo "Result: SUCCESS." | tee -a "$logFile";
 echo ""                 | tee -a "$logFile";
 local phrase="$(langstr ru "Расчет гистограмы завершен успешно." en "The histogram calculation is succeed.")";
 unix grun speak -l $(langstr ru ru en en) -y $(langstr ru Elena en Slt) "$phrase";
 local timeout=30000; if [ -n "$1" ]; then let timeout*=1000; fi;
 echo "@silent @echo $(date +%Y.%m.%d-%H.%M.%S) => $scriptname: Data analysis done - SUCCESS." | unix send2crwdaq;
 echo "@silent @run wintail $logFile" | unix send2crwdaq;
 echo "@silent @tty close 0" | unix send2crwdaq;
 local comment="$(langstr ru "Анализ данных завершен - УСПЕШНО." en "Data analysis done - SUCCESS.")";
 unix grun tooltip-notifier text "$(date +%Y.%m.%d-%H.%M.%S) $scriptname: $comment" preset stdOk delay $timeout;
 return 0;
}

#####################################################################
## Usage: show_report_on_error timeout
## Show report on error and wait timeout sec
#####################################################################
function show_report_on_error() {
 echo ""                  | tee -a "$logFile";
 echo "Result: ERROR."    | tee -a "$logFile";
 echo ""                  | tee -a "$logFile";
 local phrase="$(langstr ru "Расчет гистограмы прошел неудачно." en "The histogram calculation is failed.")";
 unix grun speak -l $(langstr ru ru en en) -y $(langstr ru Elena en Slt) "$phrase";
 local timeout=30000; if [ -n "$1" ]; then let timeout*=1000; fi;
 echo "@silent @echo $(date +%Y.%m.%d-%H.%M.%S) => $scriptname: Data analysis done - FAILURE." | unix send2crwdaq;
 echo "@silent @run wintail $logFile" | unix send2crwdaq;
 echo "@tty close 0" | unix send2crwdaq;
 local comment="$(langstr ru "Анализ данных завершен - ОШИБКА." en "Data analysis done - FAILURE.")";
 unix grun tooltip-notifier text "$(date +%Y.%m.%d-%H.%M.%S) $scriptname: $comment" preset stdError delay $timeout;
 return 0;
}

#####################################################################
## Usage: execute_root FileName
## Call ROOT to execute program filename.C
#####################################################################
function execute_root() {
 if [ -z "$1" ]; then return 0; fi;
 if [ ! -e "$1" ]; then return 0; fi;
 echo ""                         | tee -a "$logFile";
 echo "Now execute ROOT..."      | tee -a "$logFile";
 echo ""                         | tee -a "$logFile";
 unix root -l -b -x -q "$1" 2>&1 | tee -a "$logFile";
 echo ""                         | tee -a "$logFile";
};

#####################################################################
## Usage: view_file filename
## Show specified file by graphics viewer
#####################################################################
function view_file(){
 if [ -e "$1" ]; then
  unix grun lister "$1";
 fi;
};

###################################
# check and rename file.c -> file.C
# ROOT required *.C file extension.
###################################
function check_source_c(){
 if [ -e $scriptbase.C ]; then return 0; fi;
 if [ -e $scriptbase.c ]; then
  mv -fv $scriptbase.c $scriptbase.C | tee -a "$logFile";
 fi;
};

#####################################################################
## Usage: main datfile
## Call ROOT and show result of datfile processing
#####################################################################
function main() {
 init_and_check_variables "$@";
 progress_bar show $scriptname;
 if pushd "$scriptHOME"; then
  echo "WorkDir:  $(pwd)" | tee -a "$logFile";
  check_source_c;
  execute_root "$scriptbase.C";
  wait_for_file "$gifFile" 7;
  if [ -e "$gifFile" ]; then
   view_file "$gifFile";
   unix sleep 1;
   show_report_on_success 60;
   delete_file "$datFile";
  else
   show_report_on_error 60;
   delete_file "$datFile";
  fi;
 fi;
 popd;
 progress_bar hide $scriptname;
 return 0;
};

################
### Main program
################

main "$@";

###############
### Program end
###############
