#!/bin/bash
#########################################################################
# Copyright (c) 2021 Alexey Kuryakin akuryaki@cern.ch, kouriakine@mail.ru
# Script uses to install DIM DNS as systemd service
#########################################################################
# Usage: systemd_dimdns.sh [options]
#########################################################################


DNSEXE="dns";                                   # DIM DNS executable basename
DIMDIR="/opt/dim";                              # DIM DNS directory location
DIMDNS="$DIMDIR/$DNSEXE";                       # DIM DNS executable file
SYSDIR="/etc/systemd/system"                    # systemd config directory
UNITID="dimdns";                                # Service identifier
PIDFILE="/run/$UNITID.pid";                     # PID file 

is_root_user(){
 [ "$USER" == "root" ] || { echo "Error: root required. Use su/sudo to run."; return 1; };
 return 0;
}

print_usage() {
 echo $*;
}

dim_ports_view(){
 iptables -L INPUT;
 iptables -L OUTPUT;
 netstat -ntulp | grep -e '/dns *$';
}

ip_port_enable() {
 local PROT="$1"; # "tcp","udp"
 local PORT="$2"; # port number
 iptables -A INPUT  -p $PROT --dport $PORT -j ACCEPT;
 iptables -A OUTPUT -p $PROT --sport $PORT -j ACCEPT;
}

dim_ports_enable() {
 ip_port_enable  tcp  2505;
 ip_port_enable  tcp  5100;
 ip_port_enable  tcp  5101;
 ip_port_enable  tcp  5102;
}

print_unit() {
 echo "[Unit]";
 echo "Description=DIM DNS service";
 echo "After=network.target";
 echo "After=multi-user.target";
 #echo "After=graphical.target";
 echo "";
 echo "[Service]";
 echo "Type=simple";
 echo "ExecStart=";
 echo "ExecStart=$DIMDNS";
 echo "WorkingDirectory=$DIMDIR";
 echo "PIDFile=$PIDFILE";
 echo "Restart=always";
 #echo "WatchdogSec=30";
 #echo "ProtectHome=true";
 #echo "ProtectSystem=full";
 echo "";
 echo "[Install]";
 echo "Alias=$UNITID.service";
 #echo "WantedBy=multi-user.target";
 echo "";
}

systemd_dimdns() {
 [ -d "$DIMDIR" ] || { echo "Error: not found $DIMDIR"; return 1; };
 [ -e "$DIMDNS" ] || { echo "Error: not found $DIMDNS"; return 1; };
 [ -d "$SYSDIR" ] || { echo "Error: not found $SYSDIR"; return 1; };
 case "$1" in
  print)
   print_unit;
   return 0;
   ;;
  install)
   is_root_user || return 1;
   print_unit > "$SYSDIR/$UNITID.service";
   ;;
  remove)
   is_root_user || return 1;
   systemctl disable $UNITID;
   [ -e "$SYSDIR/$UNITID.service" ] && rm -f > "$SYSDIR/$UNITID.service";
   ;;
  enable)
   is_root_user || return 1;
   systemctl enable $UNITID;
   ;;
  disable)
   is_root_user || return 1;
   systemctl disable $UNITID;
   ;;
  start)
   is_root_user || return 1;
   systemctl start $UNITID;
   ;;
  stop)
   is_root_user || return 1;
   systemctl stop $UNITID;
   ;;
  restart)
   is_root_user || return 1;
   systemctl restart $UNITID;
   ;;
  status)
   systemctl status $UNITID;
   ;;
  dim_ports_enable)
   is_root_user || return 1;
   dim_ports_enable;
   ;;
  dim_ports_view)
   is_root_user || return 1;
   dim_ports_view;
   ;;
  *)
   print_usage "${0##*/} {print|install|remove|enable|disable|start|stop|restart|status|dim_ports_enable|dim_ports_view}"
 esac;
 return 0;
}

systemd_dimdns $* || exit 1;
exit 0;
