program server;                             // Demo server to illustrate DIM features. 

{$APPTYPE CONSOLE}

uses Windows, SysUtils, _dim;

var                                         // DIM uses data:
 dns_version  : Integer = 0;                // To receive DNS version
 no_link      : Integer = -1;               // To mark "server die" event.
 InfoServices : array of Integer = nil;     // List of published inform. DIM services
 CmndServices : array of Integer = nil;     // List of published command DIM services
 Terminated   : Boolean = false;            // Terminator for main loop
 Info1        : record                      // User data set #1
  CallCount   : Integer;                    // Call counter
  TickCount   : Integer;                    // GetTickCount
  Now         : Double;                     // Current time
 end;
 Info2        : record                      // User data set #2
  FileTime    : TFileTime;                  // System time as file time
  DateTime    : array[byte] of char;        // Date and time string
 end;

 //
 // Thread safe printing: Print(s,1) => StdOut, Print(s,2) => StdErr
 //
function Print(const Msg:AnsiString; n:Integer=1):DWORD;
var h:THandle;
begin
 if n=1 then h:=GetStdHandle(STD_OUTPUT_HANDLE) else
 if n=2 then h:=GetStdHandle(STD_ERROR_HANDLE) else h:=0;
 if not WriteFile(h,PChar(Msg)^,Length(Msg),Result,nil) then Result:=0;
end;

function PrintLn(const Msg:AnsiString; n:Integer=1):DWORD;
const Delimeter=#13#10;
begin
 Result:=Print(Msg+Delimeter,n);
end;

 //
 // Callback to receive DNS version
 //
procedure on_got_dns_version(var tag:TDimLong; buff:Pointer; var size:Integer); cdecl;
begin
 try
  if (size=SizeOf(no_link)) and (Integer(buff^) = no_link) then begin
   PrintLn('DNS server is dead. Please restart DNS.EXE.',2);
   Exit;
  end;
  if size=SizeOf(dns_version) then begin
   PrintLn(Format('Got DNS version: %d, tag:%d',[Integer(buff^),tag]));
   dns_version:=Integer(buff^);
   Exit;
  end;
  PrintLn(Format('Invalid data size: %d, tag:%d',[size,tag]),2);
 except
  on E:Exception do PrintLn(E.Message,2);
 end;
end;

 //
 // Callback on server exit
 //
procedure on_server_exit(var code:Integer); cdecl;
var name:array[byte] of char;
begin
 try
  if dis_get_client(name)=0 then name:='?';
  PrintLn(Format('Server "%s" (code %d) exited.',[name,code]));
 except
  on E:Exception do PrintLn(E.Message,2);
 end;
end;

 //
 // Callback on client exit
 //
procedure on_client_exit(var code:Integer); cdecl;
var name:array[byte] of char;
begin
 try
  if dis_get_client(name)=0 then name:='?';
  PrintLn(Format('Client "%s" disconnected from service %d.',[name,code]));
 except
  on E:Exception do PrintLn(E.Message,2);
 end;
end;

 //
 // Callback on server update - provide published information services
 //
procedure on_serve(var tag:TDimLong; var buff:Pointer; var size:Integer; var first:Integer); cdecl;
var name:array[byte] of char; i:Integer;
begin
 size:=0;
 buff:=nil;
 try
  if dis_get_client(name)=0 then name:='?';
  PrintLn(Format('Serving: service %d, client "%s"',[tag,name]));
  case tag of
   1: begin // Serve dataset #1 : Update content, assign buff & size
       Inc(Info1.CallCount); Info1.TickCount:=GetTickCount; Info1.Now:=Now;
       PrintLn(Format('Provide data: %d, %d, %g',[Info1.CallCount,Info1.TickCount,Info1.Now]));
       size:=SizeOf(Info1);
       buff:=@Info1;
      end;
   2: begin // Serve dataset #2 : Update content, assign buff & size
       GetSystemTimeAsFileTime(Info2.FileTime); StrPCopy(Info2.DateTime,DateTimeToStr(Now));
       PrintLn(Format('Provide data: %d, %s',[Int64(Info2.FileTime),Info2.DateTime]));
       size:=SizeOf(Info2.FileTime)+StrLen(Info2.DateTime)+1;
       buff:=@Info2;
      end;
   else PrintLn(Format('Unknown service identifier %d requested.',[tag]));
  end;
  if first>0 then begin
   if dis_get_client(name)=0 then name:='?';
   PrintLn(Format('Client "%s" connected to service %d.',[name,tag]));
   dis_set_client_exit_handler(dis_get_conn_id,tag);
   if (dis_get_client_services(dis_get_conn_id)<>nil)
   then PrintLn('Client services: '+StringReplace(Trim(dis_get_client_services(dis_get_conn_id)),#10,' ',[rfReplaceAll]));
   Print('Client counter(s):');
   for i:=Low(InfoServices) to High(InfoServices) do
   Print(' '+IntToStr(dis_get_n_clients(InfoServices[i])));
   PrintLn('');
  end;
 except
  on E:Exception do PrintLn(E.Message,2);
 end;
end;

 //
 // Callback on client command - handle it
 //
procedure on_command(var tag:TDimLong; cmnd:Pointer; var size:Integer); cdecl;
begin
 try
  PrintLn(Format('Got command: %s, tag: %d',[PChar(cmnd),tag]));
  if(StrIComp(cmnd,'RESET')=0) then Info1.CallCount:=0;
  if(StrIComp(cmnd,'EXIT')=0) then Terminated:=true;
 except
  on E:Exception do PrintLn(E.Message,2);
 end;
end;

 //
 // Callback on Client Error
 //
procedure on_client_error(severity:Integer; error_code:Integer; error_message:PChar); cdecl;
begin
 try
  PrintLn(Format('%s - %s (%d) - %s',[FormatDateTime('yyyy.mm.dd-hh:nn:ss',Now),
                                      dim_severity_name(severity),error_code,error_message]));
  if (dic_get_error_services<>nil) and (strlen(dic_get_error_services)>0)
  then PrintLn('Error services: '+dic_get_error_services);
 except
  on E:Exception do PrintLn(E.Message,2);
 end;
end;

 //
 // Callback on Server Error
 //
procedure on_server_error(severity:Integer; error_code:Integer; error_message:PChar); cdecl;
begin
 try
  PrintLn(Format('%s - %s (%d) - %s',[FormatDateTime('yyyy.mm.dd-hh:nn:ss',Now),
                                      dim_severity_name(severity),error_code,error_message]));
  if (dis_get_error_services<>nil) and (strlen(dis_get_error_services)>0)
  then PrintLn('Error services: '+dis_get_error_services);
 except
  on E:Exception do PrintLn(E.Message,2);
 end;
end;

procedure Main;
var i:Integer; dns_node:array[byte] of char;
begin
 try
  //
  // 1. Initialize DIM, set DNS node & exit handlers.
  //
  dim_init;
  dis_disable_padding;
  dic_disable_padding;
  dic_add_error_handler(on_client_error);
  dis_add_error_handler(on_server_error);
  if dim_get_dns_node(dns_node)=0 then raise Exception.Create('Fail get DNS.');
  if StrLen(dns_node)=0 then dim_set_dns_node('localhost');
  if dim_get_dns_node(dns_node)=0 then raise Exception.Create('Fail get DNS.');
  if StrLen(dns_node)=0 then raise Exception.Create('Fail set DNS.');
  PrintLn('DIM_DNS_NODE='+dns_node);
  dis_add_client_exit_handler(on_client_exit);
  dis_add_exit_handler(on_server_exit);
  //
  // 2. Subscribe DNS server version to call it only once.
  //
  dic_info_service('DIS_DNS/VERSION_NUMBER',    // DNS server version service name
                   ONCE_ONLY,                   // Call it only once
                   10,                          // Timeout, seconds
                   nil, 0, on_got_dns_version,  // Callback to receive DNS version
                   1,                           // Tag to identify this service
                   @no_link, SizeOf(no_link)    // Data to send in case of timeout
                   );
  //
  // 3. Publish information services.
  //
  SetLength(InfoServices,2);
  InfoServices[0]:=dis_add_service('DEMO/INFO1', // Name of information service
                                   'I:2;D',      // Data format: two integers and double
                                   nil, 0,       // No constant buffer, use callback
                                   on_serve,     // Callback to provide service data
                                   1);           // Tag to identify this service
  InfoServices[1]:=dis_add_service('DEMO/INFO2', // Name of information service
                                   'I:2;C',      // Data format: two integers and chars
                                   nil, 0,       // No constant buffer, use callback
                                   on_serve,     // Callback to provide service data
                                    2);          // Tag to identify this service
  //
  // 4.Publish command services.
  //
  SetLength(CmndServices,2);
  CmndServices[0]:=dis_add_cmnd('DEMO/COMMAND1',  // Name of command service
                                'C',              // Data format
                                on_command,       // Callback to handle the command
                                1);               // Tag to identify this service
  CmndServices[2]:=dis_add_cmnd('DEMO/COMMAND2',  // Name of command service
                                'C',              // Data format
                                on_command,       // Callback to handle the command
                                2);               // Tag to identify this service
  //
  // 5. Start to serve published services.
  //
  if dis_start_serving('DEMO_TASK')=0 then begin
   PrintLn('Could not start serving!');
   Sleep(2000);
   Exit;
  end;
  //
  // 6. Main server loop:
  //    Update published information services each second
  //
  Info1.CallCount:=0;
  PrintLn('Start serving...');
  while not Terminated do begin 
   for i:=0 to Length(InfoServices)-1 do
   dis_update_service(InfoServices[i]);
   Sleep(1000);
  end;
  //
  // 7. Stop DIM.
  //
  PrintLn('Stop DIM...');
  for i:=0 to Length(InfoServices)-1 do dis_remove_service(InfoServices[i]);
  for i:=0 to Length(CmndServices)-1 do dis_remove_service(CmndServices[i]);
  dis_stop_serving; dim_stop;
  PrintLn('Goodbye.');
  Sleep(2000);
 except
  on E:Exception do PrintLn(E.Message,2);
 end;
end;

begin
 Main;
end.
