 {
 ***********************************************************************
 Daq Pascal application program E140_DRV.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @WinShow w      - Show window w
| @WinHide w      - Hide window w
| @WinDraw w|opt  - Draw window w with options
| @WinSelect w    - Show and select window w
| @Async cmd      - Execute asyncronous command cmd
| @Exit           - Same as @E140.Exit
| @E140.Exit      - Exit E140_DRV.EXE and restart
| @E140.Help      - Show E140_DRV.EXE help
| @E140.Errors=n  - Set E140 error counter to n.
| @E140.Memory    - Readout E140 memory usage.
| @E140.ProcessPriority=p - Set E140_DRV.EXE priotity:
|                 - Idle/Low/Normal/High/Realtime
| @E140.ThreadPriority=p  - Set E140 thread priority:
|                 - tpIdle,..,tpNormal,..,tpTimeCritical
| @E140.DI        - Readout E140 Digital Input(s)
| @E140.DO=m      - Write E140 Digital Output(s)
| @E140.AO=i,m    - Write E140 Analog Output code m.
| @E140.SimulateAdc=i,z,a,n - chan i, zero z,ampl a, noise n
|******************************************************
[]
 }
program E140_DRV;                { L-CARD E140 driver for CRW-DAQ   }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 snd_Clear         = 'BrokenGlass';  { Sound on clear curves        }
 E140SendTimeOut   = 100;        { timeout on message send          }
 E140TimerPeriod   = 1000;       { E140.Timer poll period           }
 E140AdcChans      = 32;         { E140 Max.num. of ADC channels    }
 E140AdcMinFreq    = 1;          { E140 Min. ADC frequency, kHz     }
 E140AdcMaxFreq    = 400;        { E140 Max. ADC frequency, kHz     }

var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 errorsend         : Integer;    { Error code for data send         }
 errorterm         : Integer;    { Error code for host death        }
 erroriobug        : Integer;    { Error code for I/O errors        }
 errorpolling      : Integer;    { Error code for convertion        }
 E140              : record      { E140 private data record         }
  DllHand          : Integer;    { DLL handle                       }
  DllPath          : String;     { E140_DLL.DLL path                }
  ExePath          : String;     { E140_DRV.EXE path                }
  SerNum           : String;     { E140 Serial number               }
  Buff             : String;     { E140 task input buffer           }
  Line             : String;     { Temporary variable               }
  Tid              : Integer;    { E140 task identifier             }
  IPipeSize        : Integer;    { StdInPipeSize                    }
  OPipeSize        : Integer;    { StdOutPipeSize                   }
  Watchdog         : Real;       { Watchdog timer                   }
  Memory           : Integer;    { Uses memory by E140.exe          }
  IErrors          : Real;       { IO Errors                        }
  TErrors          : Real;       { Terminated Errors                }
  SErrors          : Real;       { Send Errors                      }
  PErrors          : Real;       { Polling Errors                   }
  ICount           : Real;       { Input  data count                }
  OCount           : Real;       { Output data count                }
  LastDo           : Integer;    { Last sent DO value               }
  LastEd           : Integer;    { Last sent EDO value              }
  LastA0           : Integer;    { Last sent DAC 0 value            }
  LastA1           : Integer;    { Last sent DAC 1 value            }
  LastStart        : Integer;    { Last state of START button       }
  winPlot          : String;     { Window to plot curves            }
  winCtrl          : String;     { Window to control GUI            }
  devPreset        : Integer;    { Preset dialog device             }
  devSaveCrw       : Integer;    { CRW saver device reference       }
 end;                            {                                  }
 tagReadout        : Integer;    { For OnReadout event              }
 tagStart          : Integer;    { START button                     }
 tagClear          : Integer;    { CLEAR button                     }
 tagPreset         : Integer;    { PRESET button                    }
 tagNotify         : Integer;    { NOTIFY flag                      }
 tagGateAo         : Integer;    { Gate on AO-s, i.e. DAC           }
 tagGateDo         : Integer;    { Gate on DO-s                     }
 tagGateDi         : Integer;    { Gate on Di-s                     }
 tagWatchdog       : Integer;    { Watchdog timer, sec              }
 tagSaveCrw        : Integer;    { SAVE CRW button                  }
 tagSaveIni        : Integer;    { SAVE INI button                  }
 tagLoadIni        : Integer;    { LOAD INI button                  }
 tagDrum           : Integer;    { DRUM, progress indicatator       }
 tagDiff           : Integer;    { Diff/Single channels             }
 tagFreq           : Integer;    { Wanted frequency                 }
 tagGate           : array[0..E140AdcChans] of Integer; { Gates     }
 tagGain           : array[0..E140AdcChans] of Integer; { Gains     }
 tagFir            : array[0..E140AdcChans] of Integer; { FIRs      }
 tagClkSrc         : Integer;    { Clock source                     }
 tagClkOut         : Integer;    { Clock output                     }
 tagInpMod         : Integer;    { Input mode, synchronization      }
 tagAdType         : Integer;    { ADC synchro type, Lev/Front      }
 tagAdMode         : Integer;    { ADC synchro mode, Hi/Lo          }
 tagAdChan         : Integer;    { ADC synchro chan, 0..31          }
 tagAdTrsh         : Integer;    { Threshold, mV                    }
 tagCadjP          : Integer;    { Clock adjustment period          }
 tagCadjC          : Integer;    { Clock adjustment coeff.          }
 tagIErrors        : Integer;    { In/Out  errors counter,real      }
 tagTErrors        : Integer;    { Terminated errors,real           }
 tagSErrors        : Integer;    { Send errors,real                 }
 tagPErrors        : Integer;    { Polling errors,real              }
 RestartOnFail     : Boolean;    { RestartOnFailure flag            }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 
 {
 Fix bug with given counter tag, error code, local counter and increment n.
 If counter tag is present and has Real type, use this counter instead of FixError.
 }
 procedure FixBug(tag:Integer; code:Integer; var count:Real; n:Integer);
 const ForceErrors=True;
 begin
  if TypeTag(tag)=2
  then bNul(rSetTag(tag,rGetTag(tag)+n))
  else bNul(FixError(code));
  if ForceErrors then
  if TypeTag(tag)=2 then bNul(FixError(code));
  count:=count+n;
 end;
 {
 Report on host terminated.
 }
 procedure HostTerm(msg:String);
 begin
  if iAnd(DebugFlags,dfTrouble)>0 then
  if Length(msg)>0 then Writeln(DevName+' ! '+msg);
  FixBug(tagTErrors,errorterm,E140.TErrors,1);
  if RunCount=1 then errors:=errors+1;
 end;
 {
 Xor bit on click (local version)
 }
 procedure ClickBitXorLocal(tag,XorMask:Integer);
 begin
  if ClickTag=tag then begin
   bNul(iSetTagXor(tag,XorMask));
   bNul(Voice(snd_Click));
  end;
 end;
 {
 Clear E140 curves.
 }
 procedure E140_ClearCurves;
 var i:Integer;
 begin
  for i:=0 to NumAos-1 do
  if RefAo(i)<>_Nil then begin
   bNul(CrvLock(RefAo(i)));
   rNul(CrvDel(RefAo(i),1,CrvLen(RefAo(i))));
   bNul(CrvUnLock(RefAo(i)));
  end;
  for i:=0 to NumDos-1 do
  if RefDo(i)<>_Nil then begin
   bNul(CrvLock(RefDo(i)));
   rNul(CrvDel(RefDo(i),1,CrvLen(RefDo(i))));
   bNul(CrvUnLock(RefDo(i)));
  end;
 end;
 {
 Clear E140 table.
 }
 procedure E140_Clear(ForceFree:Boolean);
 var i:Integer;
 begin
  if ForceFree then begin
   bNul(iSetTag(tagStart,0));
   DLL_FREE(E140.DllHand);
  end;
  E140.Tid:=0;
  E140.Buff:='';
  E140.Line:='';
  E140.ICount:=0;
  E140.OCount:=0;
  E140.Memory:=0;
  E140.IErrors:=0;
  E140.TErrors:=0;
  E140.SErrors:=0;
  E140.PErrors:=0;
  E140.DllHand:=0;
  E140.DllPath:='';
  E140.ExePath:='';
  E140.SerNum:='';
  E140.LastStart:=0;
  E140.LastDo:=MaxInt;
  E140.LastEd:=MaxInt;
  E140.LastA0:=MaxInt;
  E140.LastA1:=MaxInt;
  E140.Watchdog:=_PlusInf;
 end;
 {
 Initialize E140 table.
 }
 procedure E140_Init;
 var i:Integer;
 begin
  {---Clear E140---}
  E140_Clear(false);
  {---Read FIFO size---}
  E140.IPipeSize:=Round(Max(64,Min(64*1024,Val(ReadIni('StdInPipe')))))*1024;
  E140.OPipeSize:=Round(Max(64,Min(64*1024,Val(ReadIni('StdOutPipe')))))*1024;
  {---Find E140 server executable---}
  E140.ExePath:=DaqFileRef(AdaptExeFileName(ExpEnv(ReadIni('EXE_FILE_PATH'))),'');
  if FileExists(E140.ExePath)
  then Success('EXE_FILE_PATH='+E140.ExePath)
  else Trouble('Could not find EXE_FILE_PATH: '+E140.ExePath);
  {---Find E140 DLL---}
  E140.DllPath:=DaqFileRef(AdaptDllFileName(ExpEnv(ReadIni('DLL_FILE_PATH'))),'');
  if FileExists(E140.DllPath)
  then Success('DLL_FILE_PATH='+E140.DllPath)
  else Trouble('Could not find DLL_FILE_PATH: '+E140.DllPath);
  {---Read serial number---}
  E140.SerNum:=Trim(ReadIni('@E140.SerNum'));
  Success('@E140.SerNum='+E140.SerNum);
  {---Initialize tags---}
  InitTag(tagReadout,   ReadIni('tagReadout'),  3);
  InitTag(tagStart,     ReadIni('tagStart'),    1);
  InitTag(tagClear,     ReadIni('tagClear'),    1);
  InitTag(tagPreset,    ReadIni('tagPreset'),   1);
  InitTag(tagNotify,    ReadIni('tagNotify'),   1);
  InitTag(tagGateAo,    ReadIni('tagGateAo'),   1);
  InitTag(tagGateDo,    ReadIni('tagGateDo'),   1);
  InitTag(tagGateDi,    ReadIni('tagGateDi'),   1);
  InitTag(tagWatchdog,  ReadIni('tagWatchdog'), 1);
  InitTag(tagSaveCrw,   ReadIni('tagSaveCrw'),  1);
  InitTag(tagSaveIni,   ReadIni('tagSaveIni'),  1);
  InitTag(tagLoadIni,   ReadIni('tagLoadIni'),  1);
  InitTag(tagDrum,      ReadIni('tagDrum'),     1);
  InitTag(tagDiff,      ReadIni('tagDiff'),     1);
  InitTag(tagFreq,      ReadIni('tagFreq'),     1);
  for i:=0 to E140AdcChans-1 do begin
   InitTag(tagGate[i],  StrReplace(ReadIni('tagGate##'), '##',Str(i),0), 1);
   InitTag(tagGain[i],  StrReplace(ReadIni('tagGain##'), '##',Str(i),0), 1);
   InitTag(tagFir[i],   StrReplace(ReadIni('tagFir##'),  '##',Str(i),0), 1);
  end;
  InitTag(tagClkSrc,    ReadIni('tagClkSrc'),   1);
  InitTag(tagClkOut,    ReadIni('tagClkOut'),   1);
  InitTag(tagInpMod,    ReadIni('tagInpMod'),   1);
  InitTag(tagAdType,    ReadIni('tagAdType'),   1);
  InitTag(tagAdMode,    ReadIni('tagAdMode'),   1);
  InitTag(tagAdChan,    ReadIni('tagAdChan'),   1);
  InitTag(tagAdTrsh,    ReadIni('tagAdTrsh'),   1);
  InitTag(tagCadjP,     ReadIni('tagCadjP'),    1);
  InitTag(tagCadjC,     ReadIni('tagCadjC'),    2);
  InitTag(tagIErrors,   ReadIni('tagIErrors'), -2);
  InitTag(tagTErrors,   ReadIni('tagTErrors'), -2);
  InitTag(tagSErrors,   ReadIni('tagSErrors'), -2);
  InitTag(tagPErrors,   ReadIni('tagPErrors'), -2);
  {---Initialize devices---}
  InitDevice(E140.devPreset,  ReadIni('devPreset'),  2);
  InitDevice(E140.devSaveCrw, ReadIni('devSaveCrw'), 2);
  {---Initialize windows---}
  E140.winCtrl:=ReadIni('winCtrl');
  E140.winPlot:=ReadIni('winPlot');
  {---Initialize RestartOnFailure flag---}
  RestartOnFail:=Val(ReadIni('RestartOnFailure'))>0;
 end;
 {
 Send message to E140 task.
 Wait for some time if transmitter FIFO is over.
 }
 procedure E140_Send(msg:string);
 var ms:real;
 begin
  if E140.Tid<>0 then
  if Length(msg)>0 then begin
   if task_txspace(E140.Tid)<Length(msg)+2 then begin
    ms:=msecnow;
    while(msecnow-ms<E140SendTimeOut) and (task_txspace(E140.Tid)<Length(msg)+2) do bNul(Sleep(1));
   end;
   if task_send(E140.Tid,msg+CrLf)>0 then begin
    if iAnd(DebugFlags,dfViewExp)>0 then ViewExp('DRV: '+msg);
    E140.OCount:=E140.OCount+Length(msg)+2;
   end else begin
    FixBug(tagSErrors,errorsend,E140.SErrors,1);
   end;
  end;
 end;
 {
 Return number of active channels.
 }
 function E140_ActiveChans:Integer;
 var i,n:Integer;
 begin
  n:=0;
  for i:=0 to E140AdcChans-1 do
  if iGetTag(tagGate[i])<>0 then n:=n+1;
  E140_ActiveChans:=n;
 end;
 {
 Normalize E140 ADC parameters.
 }
 procedure E140_NormalizeParams;
 var i,j,m,k:Integer;
 begin
  {
  Normalize Gates to be 2^N active channels.
  }
  m:=1;
  while (m<E140AdcChans) and (m<E140_ActiveChans) do m:=m*2;
  if m>E140AdcChans then m:=E140AdcChans;
  k:=E140AdcChans div 2;
  for i:=E140_ActiveChans to m do begin
   if iGetTag(tagDiff)>0 then
   for j:=k to E140AdcChans-1 do
   if iGetTag(tagGate[j])=0 then
   if iGetTag(tagGate[j-k])<>0 then
   if E140_ActiveChans<i then bNul(iSetTag(tagGate[j],1));
   for j:=0 to E140AdcChans-1 do
   if iGetTag(tagGate[j])=0 then
   if E140_ActiveChans<i then bNul(iSetTag(tagGate[j],1));
  end;
  //
  // Normalize Frequency to be 1..100 kHz, Watchdog>=0, CadjP>=0, 0<=CAdjC<=1
  //
  if iGetTag(tagFreq)<E140AdcMinFreq then bNul(iSetTag(tagFreq,E140AdcMinFreq));
  if iGetTag(tagFreq)>E140AdcMaxFreq then bNul(iSetTag(tagFreq,E140AdcMaxFreq));
  if iGetTag(tagWatchdog)<0 then bNul(iSetTag(tagWatchdog,0));
  if iGetTag(tagCadjP)<0 then bNul(iSetTag(tagCadjP,0));
  if rGetTag(tagCadjC)<0 then bNul(rSetTag(tagCadjC,0));
  if rGetTag(tagCadjC)>1 then bNul(rSetTag(tagCadjC,1));
 end;
 {
 Return string with E140 ADC params:
  Word Type Means
  1    int  ADC Clock source
            0:Internal, i.e. use on-board quartz clock source
            1:External, i.e. use external INT clock source (Digital,DB-37,pin19)
  2    int  Clock output (Digital,DB-37,pin19)
            0:Disable, i.e. INT will be in high-impedance state  
            1:Enable,  i.e. INT will output on-board quartz clock; works only when AdcClock=0
  3    int  InputMode, ADC START synchonization: 
            0:No-Sync,  i.e. No ADC START synchronization = START on software call
            1:TTL-Sync, i.e. START synchronized by INT signal (Analog,DB-37,pin20)
            2:TTL-Kadr, i.e. each Kadr synchronized by INT signal
            3:AIN-sync, i.e. synchronize ADC START by analog input
  4    int  Analog synchronization type (when InputMode=3)
            0: by level
            1: by change
  5    int  Analog synchronization mode (when InputMode=3)
            0: Start on HIGH level or RAISE UP
            1: Start on LOW  level or FALL DOWN
  6    int  Analog synchronization channel, 0..31
  7    int  Analog synchronization threshold, in ADC channels
  8    real ADC Rate, kHz
  9    real Inter-Kadr delay, ms
 10    int  Channels Quantity, i.e. number of active channels
 11    11..10+ChannelsQuantity - table of active channels, see E140 manual
 NOTE:
 At the moment many items are fixed and should not be changed.
 Maybe it will be done in future ...
 }
 function E140_AdcParams:String;
 var s:String; i,code,nChan:Integer;
  procedure AddInt(i:Integer);
  begin
   if Length(s)=0 then s:=Str(i) else s:=s+','+Str(i);
  end;
  procedure AddReal(r:Real);
  begin
   if Length(s)=0 then s:=Str(r) else s:=s+','+Str(r);
  end;
  function SynchroAdPorog:Integer;
  var chan,gain:Integer; mv,rang,code,r1,r2,c1,c2:Real;
  begin
   mv:=iGetTag(tagAdTrsh);
   chan:=iAnd(iGetTag(tagAdChan),E140AdcChans-1);
   gain:=iGetTag(tagGain[chan]);
   rang:=10000/iShift(1,2*gain);
   r1:=-rang; r2:=+rang;
   c1:=-8192; c2:=+8191;
   code:=c1+(c2-c1)*(mv-r1)/(r2-r1);
   code:=Max(c1,Min(c2,code));
   SynchroAdPorog:=Round(code);
  end;
 begin
  s:='';
  E140_NormalizeParams;
  AddInt(iGetTag(tagClkSrc));   //  1: ClkSource         0:Internal 1:External
  AddInt(iGetTag(tagClkOut));   //  2: EnableClkOutput   0:Disable  1:Enable
  AddInt(iGetTag(tagInpMod));   //  3: InputMode         0:No-Sync, 1:TTL-sync, 2:TTL-Kadr, 3:AIN-sync
  AddInt(iGetTag(tagAdType));   //  4: SynchroAdType     0:Level,   1:Front
  AddInt(iGetTag(tagAdMode));   //  5: SynchroAdMode     0:Higher,  2:Lower
  AddInt(iGetTag(tagAdChan));   //  6: SynchroAdChannel, 0..31
  AddInt(SynchroAdPorog);       //  7: SynchroAdPorog,   ADC counts
  AddReal(iGetTag(tagFreq));    //  8: AdcRate,kHz
  AddReal(0);                   //  9: InterKadrDelay
  AddInt(E140_ActiveChans);     // 10: ChannelsQuantity
  for i:=0 to E140AdcChans-1 do
  if iGetTag(tagGate[i])<>0 then begin
   code:=iAnd(i,E140AdcChans-1);
   code:=iOr(code,iShift(iGetTag(tagGain[i]),6));
   code:=iOr(code,iShift(Ord(iGetTag(tagDiff)=0),5));
   AddInt(code);
  end;   
  E140_AdcParams:=s;
  s:='';
 end;
 {
 Reset watchdog timer.
 }
 procedure E140_Watchdog_Reset;
 begin
  if E140.Tid=0                then E140.Watchdog:=_PlusInf else
  if iGetTag(tagStart)<=0      then E140.Watchdog:=_PlusInf else
  if iGetTag(tagWatchdog)<=0   then E140.Watchdog:=_PlusInf else
  if not task_wait(E140.Tid,0) then E140.Watchdog:=_PlusInf else
  E140.Watchdog:=mSecNow+1e3*iGetTag(tagWatchdog)
 end;
 {
 Get watchdog doom time.
 Use if mSecNow>E140_Watchdog then ...
 }
 function E140_Watchdog:Real;
 begin
  if E140.Watchdog<=0        then E140_Watchdog:=_PlusInf else
  if iGetTag(tagStart)<=0    then E140_Watchdog:=_PlusInf else
  if iGetTag(tagWatchdog)<=0 then E140_Watchdog:=_PlusInf else
  E140_Watchdog:=E140.Watchdog;
 end;
 {
 Send AO value if one enabled by GateAo tag.
 Remember last sent value to minimize traffic.
 }
 procedure E140_SendAo(arg:String);
 var i,data:Integer; v:Real;
 begin
  if iGetTag(tagGateAo)<>0 then begin
   i:=Val(ExtractWord(1,arg));
   v:=rVal(ExtractWord(2,arg));
   if not IsNan(v) then begin
    data:=Round(v);
    if i=0 then
    if data<>E140.LastA0 then begin
     E140_Send('@E140.AO='+Str(i)+','+Str(data));
     E140.LastA0:=data;
    end;
    if i=1 then
    if data<>E140.LastA1 then begin
     E140_Send('@E140.AO='+Str(i)+','+Str(data));
     E140.LastA1:=data;
    end;
   end;
  end;
 end;
 {
 Send DO value if one enabled.
 Remember last sent value to minimize traffic.
 "ON" and "OFF" uses to enable/disable DO I/O.
 After "OFF" DO go to high impedance state.
 }
 procedure E140_SendDo(arg:String);
 var v:Real; data:Integer;
 begin
  if iGetTag(tagGateDo)<>0 then begin
   v:=rVal(Trim(arg));
   if not IsNan(v) then begin
    data:=iAnd(Round(v),65535);
    if data<>E140.LastDo then begin
     E140_Send('@E140.DO='+Str(data));
     E140.LastDo:=data;
    end;
   end else
   if IsSameText(Trim(arg),'ON')
   or IsSameText(Trim(arg),'OFF') then begin
    data:=Ord(IsSameText(Trim(arg),'ON'));
    if data<>E140.LastEd then begin
     E140_Send('@E140.EDO='+Str(data));
     E140.LastEd:=data;
    end;
   end;
  end;
 end;
 {
 Send DI command if one enabled.
 }
 procedure E140_SendDi(arg:String);
 begin
  if RefDo(0)<>0 then
  if iGetTag(tagGateDi)<>0 then begin
   E140_Send('@E140.DI='+arg);
  end;
 end;
 {
 Send command if only command is enabled.
 }
 procedure E140_GateSend(CmdLine:String);
 var cmd:String;
 begin
  cmd:=ExtractWord(1,CmdLine);
  if IsSameText(cmd,'@E140.DI') then E140_SendDi(SkipWords(1,CmdLine)) else
  if IsSameText(cmd,'@E140.AO') then E140_SendAo(SkipWords(1,CmdLine)) else
  if IsSameText(cmd,'@E140.DO') then E140_SendDo(SkipWords(1,CmdLine)) else
  E140_Send(CmdLine);
  cmd:='';
 end;
 {
 Analyse data coming from E140 task stdout.
 }
 procedure E140_Process(var Data:string);
 var cmd,arg:String; Trigger:Boolean; i,n,q,tag:Integer; t,v:Real;
 begin
  E140.ICount:=E140.ICount+Length(Data)+2;
  if iAnd(DebugFlags,dfViewImp)<>0 then ViewImp('DRV: '+Data);
  {
  "@cmd=arg" or "@cmd args" commands:
  }
  cmd:='';
  arg:='';
  if GotCommand(Data,cmd,arg) then begin
   {
   @E140.OnReadout=arglist
   E140_DRV.EXE send this message on asynchronous data block read.
   That is main message to readout ADC data from shared memory buffer.
   This message coming each ~50 milliseconds from E140_DRV.EXE process StdOut.
   arglist[1] - Shared memory buffer name; arglist[2] - data block size, words;
   arglist[3] - data block offset, bytes;  arglist[4] - data block index in Buffer;
   arglist[5] - request number, 0..63;     arglist[6] - data blocks counter;
   arglist[7] - ADC samples counter.
   Cyclic request number uses to draw progress indicator ("drum").
   Save arglist to tagReadout and call E140_DLL.DLL by using DLL_POLL procedure.
   DLL uses tagReadout to extract data block name, index etc to process data.
   After ADC data processing call DI, DO transfer procedures.
   }
   if IsSameText(cmd,'@E140.OnReadout') then begin
    bNul(iSetTag(tagDrum,1+iAnd(Val(ExtractWord(5,arg)) div 4,3)));
    bNul(sSetTag(tagReadout,arg)); DLL_POLL(E140.DllHand);
    E140_SendDo(Str(DiWord(0,16))); E140_SendDi('');
    E140_Send('@Errors=0'); E140_Send('@Memory');
    E140_Watchdog_Reset;
    Data:='';
   end else
   {
   @E140.OnException=class-name,comment
   E140_DRV.EXE send this message on exception occured.
   Standard handling is to send @Exit to restart driver.
   }
   if IsSameText(cmd,'@E140.OnException') then begin
    if RestartOnFail then E140_Send('@Exit');
    Data:='';
   end else
   {
   @E140.DI=1,$FFFF
   E140_DRV.EXE send this message on DI read.
   1-st word is status (0/1),2-nd is DI data.
   }
   if IsSameText(cmd,'@E140.DI') then begin
    if Val(ExtractWord(1,arg))=1 then UpdateDo(0,Time,Val(ExtractWord(2,arg)));
    Data:='';
   end else
   {
   @Errors=n
   E140_DRV.EXE send this message to transfer error counter.
   }
   if IsSameText(cmd,'@Errors') then begin
    n:=Val(arg);
    if n>0 then FixBug(tagIErrors,erroriobug,E140.IErrors,n);
    Data:='';
   end else
   {
   @Memory=n
   E140_DRV.EXE send this message to transfer memory usage.
   }
   if IsSameText(cmd,'@Memory') then begin
    E140.Memory:=Val(arg);
    Data:='';
   end else
   {
   @Exit=n
   E140_DRV.EXE send this message on EXIT.
   }
   if IsSameText(cmd,'@Exit') then begin
    Success('Exit with code '+Trim(arg));
    //E140.TErrors:=E140.TErrors+1;
    Data:='';
   end else
   {
   Any other command
   }
   if Length(Data)>0 then begin
    Data:='';
   end;
  end;
  Data:='';
  cmd:='';
  arg:='';
 end;
 {
 Stop E140 task if one started.
 }
 procedure E140_Stop;
 var ms:Real; i,t,n:Integer;
 begin
  if E140.Tid>0 then begin
   if task_wait(E140.Tid,0) then begin
    Success('E140.exe termination will take some time.');
    Success('You should wait up to 10 sec...');
    t:=ReadIniSection(Text_New,28,ParamStr('DaqConfigFile'),ReadIni('ScriptOnStop'));
    for i:=Text_NumLn(t)-1 downto 0 do if Pos('@',Text_GetLn(t,i))<>1 then bNul(Text_DelLn(t,i));
    for i:=0 to Text_NumLn(t)-1 do E140_GateSend(Text_GetLn(t,i));
    E140_Send('@Exit');
    bNul(Text_Free(t));
    ms:=msecnow;
    while (msecnow-ms<10000) and task_wait(E140.Tid,100) do E140_Send(CRLF);
    if task_wait(E140.Tid,0) then bNul(task_kill(E140.Tid,0,1,0));
    if task_rxcount(E140.Tid)>0 then
    while Task_Readln(E140.Tid,E140.Line,E140.Buff) do E140_Process(E140.Line);
    Success('E140 exit code = '+str(task_result(E140.Tid)));
   end;
   bNul(task_free(E140.Tid));
   E140.LastStart:=0;
  end;
  DLL_FREE(E140.DllHand);
  E140.Tid:=0;
  E140.Buff:='';
  E140.Line:='';
  E140.LastDo:=MaxInt;
  E140.LastEd:=MaxInt;
  E140.LastA0:=MaxInt;
  E140.LastA1:=MaxInt;
  E140_Watchdog_Reset;
 end;
 {
 Kill E140 task if one started.
 }
 procedure E140_Kill;
 var ms:Real;
 begin
  if E140.Tid>0 then begin
   if task_wait(E140.Tid,0) then begin
    Success('E140.exe killing will take some time.');
    Success('You should wait up to 10 sec...');
    ms:=msecnow;
    E140_Send('@Exit');
    while (msecnow-ms<10000) and task_wait(E140.Tid,100) do E140_Send(CRLF);
    if task_wait(E140.Tid,0) then bNul(task_kill(E140.Tid,0,1,0));
    if task_rxcount(E140.Tid)>0 then
    while Task_Readln(E140.Tid,E140.Line,E140.Buff) do E140_Process(E140.Line);
    Success('E140 exit code = '+str(task_result(E140.Tid)));
   end;
  end;
  E140_Watchdog_Reset;
 end;
 {
 Finalize E140 table.
 }
 procedure E140_Free;
 begin
  E140_Stop;
  E140_Clear(True);
 end;
 {
 Start E140 server if one not started.
 }
 procedure E140_Start;
 var i,j,t:Integer;
 begin
  if E140.DllHand=0 then begin
   E140.DllHand:=DLL_INIT(E140.DllPath);
   if E140.DllHand<>0
   then Success('Load Ok DLL: '+E140.DllPath)
   else Trouble('Fail to load DLL: '+E140.DllPath);
  end;
  if E140.DllHand<>0 then
  if E140.Tid=0 then begin
   {
   Initialize separate user task, run it invisible...
   }
   E140.Tid:=task_init(E140.ExePath);
   if pos('?',task_ctrl(E140.Tid,'HomeDir='+ExtractFilePath(E140.ExePath))
             +task_ctrl(E140.Tid,'StdInPipeSize='+str(E140.IPipeSize))
             +task_ctrl(E140.Tid,'StdOutPipeSize='+str(E140.OPipeSize))
             +task_ctrl(E140.Tid,'Display=0')
          )>0
   then begin
    Trouble('User task setup error!');
    E140_Stop;
   end;
   {
   Run task if one was created...
   }
   if E140.Tid>0 then
   if task_run(E140.Tid) then begin
    Success('TaskId  = '+str(E140.Tid));
    Success('TaskPid = '+str(task_pid(E140.Tid)));
    Success('TaskRef = '+str(task_ref(E140.Tid)));
    Success('CmdLine = '+task_ctrl(E140.Tid,'CmdLine'));
    Success('HomeDir = '+task_ctrl(E140.Tid,'HomeDir'));
    Success('PipeIn  = '+task_ctrl(E140.Tid,'StdInPipeSize'));
    Success('PipeOut = '+task_ctrl(E140.Tid,'StdOutPipeSize'));
    Success('Display = '+task_ctrl(E140.Tid,'Display'));
   end else begin
    Trouble('Could not start E140!');
    E140_Stop;
   end;
   {
   Is it Ok with user task? Send preset parameters.
   Send all commands to self StdIn for future processing.
   }
   if E140.Tid>0 then
   if task_wait(E140.Tid,0) then begin
    E140.Line:='';
    E140_Send('@Help');
    E140_Send('@Memory');
    E140_Send('@Errors=0');
    E140_Send('@ProcessPriority=RealTime');
    E140_Send('@ThreadPriority=tpTimeCritical');
    if Val(ReadIni('Simulation'))=1
    then E140_Send('@E140.Simulation=1')
    else E140_Send('@E140.Simulation=0');
    E140_Send('@E140.SerNum= '+E140.SerNum);
    E140_Send('@E140.Open='+E140_AdcParams);
    t:=ReadIniSection(Text_New,28,ParamStr('DaqConfigFile'),ReadIni('ScriptOnStart'));
    for i:=Text_NumLn(t)-1 downto 0 do if Pos('@',Text_GetLn(t,i))<>1 then bNul(Text_DelLn(t,i));
    for i:=0 to Text_NumLn(t)-1 do E140_GateSend(Text_GetLn(t,i));
    bNul(Text_Free(t));
    E140.Line:='';
   end else HostTerm('Unexpected termination!');
  end;
  E140_Watchdog_Reset;
 end;
 {
 E140 polling.
 }
 procedure E140_Poll;
 var i,tag:Integer;
 begin
  { 
  E140 timer actions...
  }
  if SysTimer_Pulse(E140TimerPeriod)>0 then begin
   if iGetTag(tagStart)=0 then bNul(iSetTag(tagDrum,0));
  end;
  {
  Check watchdog timer...
  }
  if mSecNow>E140_Watchdog then begin
   Problem('Watchdog timeout!');
   E140_Watchdog_Reset;
   E140_Kill;
  end;
  {
  START button pressed...
  }
  if iGetTag(tagStart)<>E140.LastStart then begin
   if iGetTag(tagStart)<>0 then begin
    E140_SendDo('ON');
    E140_Send('@E140.Start='+E140_AdcParams);
    bNul(iSetTag(tagDrum,0));
    E140_Watchdog_Reset;
    if iGetTag(tagInpMod)>0 then E140.Watchdog:=_PlusInf;
   end else begin
    bNul(iSetTag(tagDrum,0));
    E140_Send('@E140.Stop');
    E140_SendDo('OFF');
    E140_Watchdog_Reset;
   end;
   E140.LastStart:=iGetTag(tagStart);
  end;
  {
  CLEAR button pressed...
  }
  if iGetTag(tagClear)<>0 then begin
   bNul(iSetTag(tagClear,0));
   if editstate=0 then begin
    if pos('?',edit('(Очистить кривые?')
              +edit(' Вы вызвали команду очистки кривых E140.')
              +edit(' ')
              +edit(' Убедитесь, что текущие данные записаны!')
              +edit(' Иначе все несохраненные данные пропадут!')
              +edit(' ')
              +edit(' Вы действительно хотите очистить кривые?')
              +edit(')YesNo YesNo_'+NameTag(tagClear)))>0
    then writeln('Could not initialize dialog!');
   end else writeln('Could not initialize dialog!');
  end;
  {
  PRESET button pressed...
  }
  if iGetTag(tagPreset)<>0 then begin
   bNul(DevMsg(RefInfo(E140.devPreset,'Name')+' Confirm=2'+CRLF)>0);
   bNul(Action(RefInfo(E140.devPreset,'Name')));
   bNul(iSetTag(tagPreset,0));
  end;
  {
  PRESET dialog notification
  }
  if iGetTag(tagNotify)<>0 then begin
   if iGetTag(tagNotify)=1 then begin // Ok
    E140_NormalizeParams;
   end;
   if iGetTag(tagNotify)=2 then begin // Cancel
    bNul(Voice(snd_Click));
   end;
   bNul(iSetTag(tagNotify,0));
  end;
  {
  SAVEINI button pressed...
  }
  if iGetTag(tagSaveIni)<>0 then begin
   bNul(iSetTag(tagSaveIni,0));
   iNul(CustomIniRw('W','',2));
  end;
  {
  LOADINI button pressed...
  }
  if iGetTag(tagLoadIni)<>0 then begin
   bNul(iSetTag(tagLoadIni,0));
   iNul(CustomIniRw('R','',2));
  end;
  {
  Edit tags...
  }
  if editstate=1 then begin
   {
   CLEAR button (after confirmation).
   }
   if IsSameText(ExtractWord(1,edit('?ans 0')),'YesNo_'+NameTag(tagClear)) then begin
    if Val(ExtractWord(2,edit('?ans 0')))=6 then begin
     bNul(Echo(DevName+' : '+GetDateTime(mSecNow)+' - Кривые очищены!'));
     bNul(WinSelect(ParamStr('MainConsole')));
     bNul(Voice(snd_Clear));
     E140_ClearCurves;
    end;
    sNul(edit(''));
   end;
  end;
  if editstate=1 then begin
   writeln('Unknown tag edition!');
   sNul(edit(''));
  end;
  if iand(editstate,9)<>0 then begin
   writeln('Dialog error detected!');
   sNul(edit(''));
  end;
  {
  GUI processing...
  }
  if ClickButton=1 then begin
   if ClickTag=tagStart then begin
    ClickBitXorLocal(tagStart,1);
   end;
   if ClickTag=tagPreset then begin
    if iGetTag(tagStart)=0
    then ClickBitXorLocal(tagPreset,1)
    else bNul(Voice(snd_Fails));
   end;
   if ClickTag=tagSaveCrw then begin
    if iGetTag(tagStart)=0
    then ClickBitXorLocal(tagSaveCrw,1)
    else bNul(Voice(snd_Fails));
    if iGetTag(tagSaveCrw)<>0 then bNul(WinSelect(ParamStr('MainConsole')));
   end;
   if ClickTag=tagClear then begin
    if iGetTag(tagStart)=0
    then ClickBitXorLocal(tagClear,1)
    else bNul(Voice(snd_Fails));
   end;
   if ClickTag=tagReadout then begin
    bNul(Voice(snd_Wheel));
    OpenConsole(1);
   end;
   for i:=0 to NumAos-1 do if RefAo(i)<>_Nil then
   if IsSameText(ClickParams('Curve'),CrvName(RefAo(i))) then begin
    if Length(E140.winPlot)>0 then begin
     bNul(WinSelect(E140.winPlot));
     bNul(WinDraw(E140.winPlot+'|SelectCurve='+ClickParams('Curve')));
     bNul(Voice(snd_Wheel));
    end;
   end;
   for i:=0 to NumDos-1 do if RefDo(i)<>_Nil then
   if IsSameText(ClickParams('Curve'),CrvName(RefDo(i))) then begin
    if Length(E140.winPlot)>0 then begin
     bNul(WinSelect(E140.winPlot));
     bNul(WinDraw(E140.winPlot+'|SelectCurve='+ClickParams('Curve')));
     bNul(Voice(snd_Wheel));
    end;
   end;
   for i:=0 to NumAis-1 do if RefAi(i)<>_Nil then
   if IsSameText(ClickParams('Curve'),CrvName(RefAi(i))) then begin
    if Length(E140.winPlot)>0 then begin
     bNul(WinSelect(E140.winPlot));
     bNul(WinDraw(E140.winPlot+'|SelectCurve='+ClickParams('Curve')));
     bNul(Voice(snd_Wheel));
    end;
   end;
  end;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  E140.Buff:='';    E140.Line:='';
  E140.winCtrl:=''; E140.winPlot:='';
  E140.ExePath:=''; E140.DllPath:='';
  E140_Clear(False);
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  {
  Initialize errors...
  }
  errorsend:=registererr(devname+': data send.');
  errorterm:=registererr(devname+': terminated.');
  erroriobug:=registererr(devname+': I/O error.');
  errorpolling:=registererr(devname+': polling.');
  {
  Clear and initialize variables...
  }
  E140_Init;
  {
  Run startup script...
  }
  RunStartupScript;
  {
  Load params on startup...
  }
  if Val(ReadIni('CustomIniAutoLoad'))=1 then iNul(CustomIniRw('R','',2));
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  if Val(ReadIni('CustomIniAutoSave'))=1 then iNul(CustomIniRw('W','',2));
  E140_Free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  {
  If E140 process is not still running,
  try to start E140 process periodically.
  }
  if E140.Tid=0 then
  if SysTimer_Pulse(E140TimerPeriod)>0 then E140_Start;
  {
  Communicate with E140 process if one still running...
  }
  if E140.Tid>0 then
  if task_wait(E140.Tid,0) then begin
   {
   If has data coming from Task StdOut, analyse it...
   }
   if task_rxcount(E140.Tid)>0 then
   while Task_Readln(E140.Tid,E140.Line,E140.Buff) do E140_Process(E140.Line);
   {
   E140 polling actions.
   }
   E140_Poll;
  end else begin
   HostTerm('E140 terminated, exit code = '+Str(task_result(E140.Tid)));
   E140_Stop;
  end;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; i,tag:Integer; v:Real;
 begin
  E140.ICount:=E140.ICount+Length(Data)+2;
  if iAnd(DebugFlags,dfViewImp)<>0 then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  if GotCommand(Data,cmd,arg) then begin
   {
   Example: @Exit=n
   Example: @E140.Exit
   Exit (with code n) and restart driver process.
   }
   if IsSameText(cmd,'@Exit')
   or IsSameText(cmd,'@E140.Exit') then begin
    E140_Send('@Exit='+arg);
    Data:='';
   end else
   {
   Example: @E140.Help
   Show help on driver commands.
   }
   if IsSameText(cmd,'@E140.Help') then begin
    E140_Send('@Help='+arg);
    Data:='';
   end else
   {
   Example: @E140.Errors=0
   Request to read driver's error counter.
   }
   if IsSameText(cmd,'@E140.Errors') then begin
    E140_Send('@Errors='+arg);
    Data:='';
   end else
   {
   Example: @E140.Memory
   Request to read driver's memory counter.
   }
   if IsSameText(cmd,'@E140.Memory') then begin
    E140_Send('@Memory='+arg);
    Data:='';
   end else
   {
   Example: @E140.Send=@Help
   Send arguments to E140 directly.
   }
   if IsSameText(cmd,'@E140.Send') then begin
    E140_Send(arg);
    Data:='';
   end else
   {
   Example: @E140.ProcessPriority=Realtime
   Set/get driver process priority.
   }
   if IsSameText(cmd,'@E140.ProcessPriority') then begin
    E140_Send('@ProcessPriority='+arg);
    Data:='';
   end else
   {
   Example: @E140.ThreadPriority=tpTimeCritical
   Set/get driver thread priority.
   }
   if IsSameText(cmd,'@E140.ThreadPriority') then begin
    E140_Send('@ThreadPriority='+arg);
    Data:='';
   end else
   {
   Example: @E140.DI
   Request to readout DI state.
   }
   if IsSameText(cmd,'@E140.DI') then begin
    E140_SendDi(arg);
    Data:='';
   end else
   {
   Example: @E140.DO=123  @E140.DO=ON   @E140.DO=OFF
   Set DO state.
   }
   if IsSameText(cmd,'@E140.DO') then begin
    E140_SendDo(arg);
    Data:='';
   end else
   {
   Example: @E140.AO=0,123
   Write data (123 counts) to DAC channel (0,1).
   }
   if IsSameText(cmd,'@E140.AO') then begin
    E140_SendAo(arg);
    Data:='';
   end else
   {
   Example: @E140.SerNum=1D234567
   Set E140 serial number to identify.
   }
   if IsSameText(cmd,'@E140.SerNum') then begin
    E140_Send('@E140.SerNum='+arg);
    Data:='';
   end else
   {
   Example: @E140.SimulateAdc=1,250,200,20
   Set simulation parameters.
   1   - ADC channel (0..31)
   250 - ADC level, mV
   200 - "saw" amplitude, mV
   20  - noise level, mV
   }
   if IsSameText(cmd,'@E140.SimulateAdc') then begin
    E140_Send('@E140.SimulateAdc='+arg);
    Data:='';
   end else
   {
   Example: @E140.ClearCurves
   }
   if IsSameText(cmd,'@E140.ClearCurves') then begin
    E140_ClearCurves;
    Data:='';
   end else
   {
   Example: @WinHide DEMO.GUI
   }
   if IsSameText(cmd,'@WinHide') then begin
    if Length(Trim(arg))>0 then bNul(WinHide(Trim(arg)));
    Data:='';
   end else
   {
   Example: @WinShow DEMO.GUI
   }
   if IsSameText(cmd,'@WinShow') then begin
    if Length(Trim(arg))>0 then bNul(WinShow(Trim(arg)));
    Data:='';
   end else
   {
   Example: @WinDraw DEMO.GUI
   }
   if IsSameText(cmd,'@WinDraw') then begin
    if Length(Trim(arg))>0 then bNul(WinDraw(Trim(arg)));
    Data:='';
   end else
   {
   Example: @WinSelect DEMO.GUI
   }
   if IsSameText(cmd,'@WinSelect') then begin
    if Length(Trim(arg))>0 then bNul(WinSelect(Trim(arg)));
    Data:='';
   end else
   {
   Example: @Async @WinSelect DEMO.GUI
   }
   if IsSameText(cmd,'@Async') then begin
    if Length(Trim(arg))>0 then DevSendCmd(devMySelf,Trim(arg));
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
