////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// L-CARD E14-140 driver for CRW-DAQ: shared memory data.                     //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20101112 - Created by A.K.                                                 //
// 20241208 - Modified for FPC.                                               //
// 20260123 - Many changes for 1st FPC release                                //
////////////////////////////////////////////////////////////////////////////////

unit _crw_e140_shm; //  E140 shared memory.

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math,
 _crw_alloc, _crw_ef, _crw_str, _crw_lcard;

const                             // Shared memory buffer:
 E140BuffFactor    = 2;           // To calculate buf.size
 E140NumBuffers    = 64;          // Number of I/O buffers
 E140BuffAdjust    = 32;          // Buf.length adjustment
 E140MinBuffLen    = 32;          // Minimal buffer length
 E140MaxBuffLen    = 1024*16;     // Maximal buffer length
 E140HeaderSize    = 1024*64;     // Size of shared header
 E140DefTimeout    = 1000;        // Default timeout in ms
 E140MinBuffTime   = 10;          // Minimal buff.time, ms
 E140DefBuffTime   = 50;          // Default buff.time, ms
 E140MaxBuffTime   = 1000;        // Maximal buff.time, ms
 E140ListBuffSize  = 1024*4;      // Buffer for device list
 E140BuffArrayLen  = E140MaxBuffLen*E140NumBuffers*E140BuffFactor;
 E140BuffArraySize = E140BuffArrayLen*SizeOf(SmallInt);

type
 pULONG   = ^ULONG;
 pSHORT   = ^SmallInt;
 pADC_PAR = ^ADC_PAR;

 //////////////////////////////////////////////////////////////
 // Next record uses for shared memory buffer to transfer data.
 //////////////////////////////////////////////////////////////
type
 PE140_SHARED_BUFFER = ^TE140_SHARED_BUFFER;
 TE140_SHARED_BUFFER = packed record
 public
  function CookiesBuff:PChar;
  function CookiesSize:SizeUInt;
 public
  case Integer of
   0: ( 
        Header : packed array[0..E140HeaderSize-1] of Char;
        Buffer : packed array[0..E140BuffArrayLen-1] of SmallInt;
       );
   1: (
        StartTime   : Double;
        Board       : LBoard;
        AdcInited   : ADC_PAR;
        AdcWanted   : ADC_PAR;
        ListLDev    : packed array[0..E140ListBuffSize-1] of Char;
        Cookies     : packed array[0..1] of Char;
       );
 end;

function LPlataSerNum(const Buf:PE140_SHARED_BUFFER):LongString; overload;
function LPlataSerNum(const Buf:TE140_SHARED_BUFFER):LongString; overload;
function GetE140SerialNumber(const Buf:PE140_SHARED_BUFFER):LongString; overload;
function GetE140SerialNumber(const Buf:TE140_SHARED_BUFFER):LongString; overload;
function GetE140Description(const Buf:PE140_SHARED_BUFFER; const Prefix:LongString=''):LongString; overload;
function GetE140Description(const Buf:TE140_SHARED_BUFFER; const Prefix:LongString=''):LongString; overload;
function GetE140Description(const sp:SLOT_PAR; const pd:PLATA_DESCR_U2; const Prefix:LongString=''):LongString; overload;
function GetE140AdcPars(const aAdc:ADC_PAR; const Prefix:LongString; Mode:Integer=0):LongString;

implementation

function TE140_SHARED_BUFFER.CookiesBuff:PChar;
begin
 Result:=@Cookies[0];
end;

function TE140_SHARED_BUFFER.CookiesSize:SizeUInt;
begin
 Result:=SizeOf(Header)-(@Cookies[0]-@Header[0]);
end;

function LPlataSerNum(const Buf:PE140_SHARED_BUFFER):LongString;
begin
 Result:=LPlataSerNum(Buf.Board.Sp,Buf.Board.Pd);
end;

function LPlataSerNum(const Buf:TE140_SHARED_BUFFER):LongString;
begin
 Result:=LPlataSerNum(Buf.Board.Sp,Buf.Board.Pd);
end;

function GetE140SerialNumber(const Buf:PE140_SHARED_BUFFER):LongString;
begin
 Result:=LPlataSerNum(Buf);
end;

function GetE140SerialNumber(const Buf:TE140_SHARED_BUFFER):LongString;
begin
 Result:=LPlataSerNum(Buf);
end;

function GetE140Description(const Buf:PE140_SHARED_BUFFER; const Prefix:LongString=''):LongString;
begin
 Result:=GetE140Description(Buf.Board.Sp,Buf.Board.Pd,Prefix);
end;

function GetE140Description(const Buf:TE140_SHARED_BUFFER; const Prefix:LongString=''):LongString;
begin
 Result:=GetE140Description(Buf.Board.Sp,Buf.Board.Pd,Prefix);
end;

function GetE140Description(const sp:SLOT_PAR; const pd:PLATA_DESCR_U2; const Prefix:LongString=''):LongString;
 procedure Add(s:LongString);
 begin
  if Length(Result)=0
  then Result:=Prefix+s
  else Result:=Result+EOL+Prefix+s; 
 end;
begin
 Result:='';
 Add('BoardType = '+LBoardTypeName(sp));
 Add('BoardName = '+ LPlataBrdName(sp,pd));
end;

function GetE140AdcPars(const aAdc:ADC_PAR; const Prefix:LongString; Mode:Integer=0):LongString;
var i:Integer; 
 procedure Add(s:LongString);
 var sn,sv:LongString;
 begin
  sn:=''; sv:='';
  if (ExtractNameValuePair(s,sn,sv)>0) then begin
   if HasFlags(Mode,1)
   then s:=Format('%s=%s',[sn,sv])
   else s:=Format('%-18s = %s',[sn,sv]); 
   if Length(Result)=0
   then Result:=Prefix+s
   else Result:=Result+EOL+Prefix+s; 
  end;
 end;
begin
 Result:='';
 with aAdc.t1 do begin
  Add(Format('s_Type             = %d',[s_Type]));
  Add(Format('FIFO               = %d',[FIFO]));
  Add(Format('IrqStep            = %d',[IrqStep]));
  Add(Format('Pages              = %d',[Pages]));
  Add(Format('AutoInit           = %d',[AutoInit]));
  Add(Format('dRate              = %g',[dRate]));
  Add(Format('dKadr              = %g',[dKadr]));
  Add(Format('dScale             = %g',[dScale]));
  Add(Format('Rate               = %d',[Rate]));
  Add(Format('Kadr               = %d',[Kadr]));
  Add(Format('Scale              = %d',[Scale]));
  Add(Format('FPDelay            = %d',[FPDelay]));
  Add(Format('SynchroType        = %d',[SynchroType]));
  Add(Format('SynchroSensitivity = %d',[SynchroSensitivity]));
  Add(Format('SynchroMode        = %d',[SynchroMode]));
  Add(Format('AdChannel          = %d',[AdChannel]));
  Add(Format('AdPorog            = %d',[AdPorog]));
  Add(Format('NCh                = %d',[NCh]));
  for i:=0 to Min(Integer(Nch)-1,High(Chn)) do 
  Add(Format('Chn[%d]            = %d',[i,Chn[i]]));
  Add(Format('IrqEna             = %d',[IrqEna]));
  Add(Format('AdcEna             = %d',[AdcEna]));
 end;
end;


///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_e140_shm;
begin
end;

procedure Free_crw_e140_shm;
begin
end;

initialization

 Init_crw_e140_shm;

finalization

 Free_crw_e140_shm;

end.

//////////////
// END OF FILE
//////////////

