////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// L-CARD E14-140 driver for CRW-DAQ: main driver class                       //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20101112 - Created by A.K.                                                 //
// 20241208 - Modified for FPC.                                               //
// 20260123 - Many changes for 1st FPC release                                //
////////////////////////////////////////////////////////////////////////////////

unit _crw_e140_api; //  E140 main driver class.

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math,
 _crw_alloc, _crw_ef, _crw_str, _crw_fio, _crw_rtc,
 _crw_ascio, _crw_sharm, _crw_proc,
 _crw_lcard, _crw_e140_shm;

 {
 List of possible LDevDriverStatus names.
 }
const
 LDevDriverStatusNames = 'INVALID,STOPPED,STARTED';

 {
 Get LCARD driver status: INVALID,STOPPED,STARTED.
 }
function LDevDriverStatus:LongString;

 {
 Start LCARD driver if one not started.
 }
function LDevDriverStart:Boolean;

 {
 List /dev/shm/ on Unix.
 }
function GetListOfDevShm:LongString;

 {
 Unlink shared memory device on Unix.
 }
function UnlinkDevShm(shmName:LongString):LongString;

 {
 Get list of LCARD devices.
 }
function LDevListDevices(Reset:Boolean=false):LongString;

 {
 Find slot number by serial number or return -1.
 }
function LPlataSlotBySerNum(const SerNum:LongString; Id:LongString='31:E140;30:E440'):Integer;

 {
 Compose/Split E140 Control Word (CW) in ADC_PAR.t1.chn[..]:
 CW Bit[0-4] - Channel;  Bit[5] - Diff/Single Type; Bit[6,7] - Gain.
 }
function E140CwCompose(nChan,nGain:Integer; nType:Integer=0):Integer;
function E140CwGetChan(cw:Integer):Integer; // Channel number
function E140CwGetGain(cw:Integer):Integer; // ADC Gain number
function E140CwGetType(cw:Integer):Integer; // 0/1=Diff/Single Type

 {
 For simulation.
 }
function TestE140AdcPars(nChan,nGain:Integer; Freq:Double):ADC_PAR;

 //////////////////////////////
 // Ring buffer for ADC polling
 //////////////////////////////
type
 TAdcPollRing = record       // For ADC Polling Loop
  DataSync : PULONG;         // Synchronization Variable Pointer
  DataBuff : PSmallIntArray; // ADC Data Buffer Pointer
  BuffLeng : ULONG;          // ADC Buffer Length
  HalfStep : Int64;          // Step of half buffers
  DataOffs : Int64;          // Data offset (flat)
  HalfFlag : Boolean;        // Half Buffer done
 public
  function  GetSync:ULONG;
  function  Valid:Boolean;
  function  HalfLeng:ULONG; inline;
  function  HalfBuff:PSmallIntArray;
  procedure Reset(const pAdc:pADC_PAR=nil);
  function  Init(const Adc:ADC_PAR):Boolean;
  function  HalfBuffHasCome:Boolean;
  procedure NextHalf;
 end;
 ////////////////
 // TAdcPollBlock
 ////////////////
type
 TAdcPollBlock = record      // ADC Polling Data Block
  DataPtr  : PSmallIntArray; // Points to Data Buffer 
  DataLen  : Integer;        // Data Buffer length in items
  Elapsed  : Double;         // Time elapsed since start, sec
  StepNum  : Int64;          // Step number of half-FIFO blocks
  FlatNum  : Int64;          // Flat Data Offset Number
 public
  function  Ready:Boolean;
  procedure Clear(Mode:Integer);
  function  ItemSize:Integer; inline;
  function  DataSize:Integer; inline;
 end;
 ///////////////////
 // TAdcRequestQueue
 ///////////////////
type
 TAdcRequestQueue = record              // Queue for I/O requests
  Number   : Integer;                   // 0..E140NumBuffers-1
  Blocks   : Int64;                     // Total number of blocks
  Samples  : Int64;                     // Total number of samples
  BuffLeng : Integer;                   // Maximal Buffer length
  Index    : array[0..E140NumBuffers-1] of Integer;        // Array index
  Offset   : array[0..E140NumBuffers-1] of Integer;        // Memory offset
  DataBuff : array[0..E140NumBuffers-1] of PSmallIntArray; // Data Pointers
 public
  procedure Clear;
  procedure NextStep;
  function  Valid:Boolean;
  procedure Init(Shared:PE140_SHARED_BUFFER; Leng:Integer=E140MaxBuffLen);
 end;
 ///////////////////////
 // E14-140 driver class
 ///////////////////////
type
 E140_FAILURE = class(Exception);
 TLDev_Reporter = procedure(const Msg:LongString); // Procedure to report errors
 TE140 = class(TMasterObject)
 private
  myBoard        : LBoard;                    // LCARD Device Board
  myHand         : THandle;                   // LCARD Device handle 
  myApi          : IDaqLDevice;               // LCARD device interface
  myBuffSize     : ULONG;                     // LCARD ADC buffer size
  mySharm        : TSharedMemory;             // Shared memory instance
  myShared       : PE140_SHARED_BUFFER;       // Shared memory pointer
  mySerNumWanted : LongString;                // L-CARD Serial number (wanted)
  mySerNumActual : LongString;                // L-CARD Serial number (readback)
  myReporter     : TLDev_Reporter;            // Error report routine
  myBuffTime     : Integer;                   // Buffer time to calc length, ms
  myRequest      : TAdcRequestQueue;          // Data for I/O requests
  myAdcRing      : TAdcPollRing;              // Ring buffer for ADC polling
  myAdcBlock     : TAdcPollBlock;             // Current block of ADC data
 private
  function  GetSlot:Integer;
  procedure SetReporter(aReporter:TLDev_Reporter);
  procedure PrintReport(const msg:LongString);
  procedure Success(const s:LongString); 
  procedure Failure(const s:LongString);
  procedure BugReport(const E:Exception; const msg:LongString='');
  procedure PrintDescr(const sp:SLOT_PAR; const pd:PLATA_DESCR_U2);
  procedure PrintADC(const aAdc:ADC_PAR);
  function  FindBufferLength(aBuffTime:Integer; const aAdc:ADC_PAR):Integer;
  function  GetSharedBuff:PE140_SHARED_BUFFER;
  function  GetSharedName:LongString;
  function  GetSerNumWanted:LongString;
  procedure SetSerNumWanted(const aSerNum:LongString);
  function  GetSerNumActual:LongString;
  function  GetBoardName:LongString;
  function  GetBoardType:Integer;
  function  GetBuffTime:Integer;
 protected
  function  ProcessBuffer(nWords:DWORD; nRequest:Integer; nSamples:Int64; nBlocks:Int64):DWORD; virtual;
 public
  constructor Create(aShare:LongString; aReporter:TLDev_Reporter=nil;
                     aSlot:Integer=-1; aAdc:pADC_PAR=nil; aBuffTime:Integer=0);
  destructor  Destroy; override;
 public
  property  Reporter     : TLDev_Reporter      write SetReporter;
  property  Report       : LongString          write PrintReport;
  property  SharedBuff   : PE140_SHARED_BUFFER read GetSharedBuff;
  property  SharedName   : LongString          read GetSharedName;
  property  SerNumWanted : LongString          read GetSerNumWanted write SetSerNumWanted;
  property  SerNumActual : LongString          read GetSerNumActual;
  property  BoardName    : LongString          read GetBoardName;
  property  BoardType    : Integer             read GetBoardType;
  property  BuffTime     : Integer             read GetBuffTime;
  property  Slot         : Integer             read GetSlot;
 public
  function  ReadSerialNumber(Save:Boolean=True):LongString;
  function  Open(aSlot:Integer; aAdc:pADC_PAR=nil):Boolean;
  function  Opened:Boolean;
  procedure Close;
  function  Start(aAdc:pADC_PAR=nil):Double;
  function  Started:Double;
  function  Stop:Double;
  function  Polling:Int64;
  function  PollAdcHandler(const Adc:ADC_PAR; var Block:TAdcPollBlock):Int64;
  function  DAC_SAMPLE(DacData:ULONG; DacChannel:WORD):Boolean;
  function  TTL_OUT(TtlOut:ULONG):Boolean;
  function  TTL_IN(var TtlIn:ULONG):Boolean;
  function  GetLastErrorInfo:LongString;
  function  SharedBufferInfo:LongString;
  function  IsAdcRunning(aTimeout:Integer=0):Boolean;
 public
  function  ValidateAdcPages(var Adc:ADC_PAR; StepMs:Integer=0):Boolean;
  function  MaxAdcRate:Double;
 public
  class var DefBuffTime:Integer;
  class function ValidateBuffTime(aBuffTime:Integer):Integer;
  class function ValidateAdcChans(var Adc:ADC_PAR):Boolean;
 end;

function NewE140(aShare:LongString; aReporter:TLDev_Reporter=nil; aSlot:Integer=-1; aAdc:pADC_PAR=nil; aBuffTime:Integer=0):TE140;
procedure Kill(var TheObject:TE140); overload;
 
 {
 E140 Extra constants from old Lusbapi.pas
 }
const
 MAX_CONTROL_TABLE_LENGTH_E140  = (128);
 DAC_CHANNELS_QUANTITY_E140     = ($02);
 DAC_CALIBR_COEFS_QUANTITY_E140 = (DAC_CHANNELS_QUANTITY_E140);
 TTL_LINES_QUANTITY_E140        = ($10); 
const // ADC input range indexes
 ADC_INPUT_RANGES_QUANTITY_E140 = ($04);
 ADC_INPUT_RANGE_10000mV_E140   = ($00);
 ADC_INPUT_RANGE_2500mV_E140    = ($01);
 ADC_INPUT_RANGE_625mV_E140     = ($02);
 ADC_INPUT_RANGE_156mV_E140     = ($03);
 ADC_CALIBR_COEFS_QUANTITY_E140 = (ADC_INPUT_RANGES_QUANTITY_E140);
 INVALID_ADC_INPUT_RANGE_E140   = ($04);
const // ADC clock source indexes
 INT_ADC_CLOCK_E140             = ($00);
 EXT_ADC_CLOCK_E140             = ($01);
 INVALID_ADC_CLOCK_E140         = ($02);
const 
 ADC_CLOCK_TRANS_DISABLED_E140  = ($00);
 ADC_CLOCK_TRANS_ENABLED_E140   = ($01);
 INVALID_ADC_CLOCK_TRANS_E140   = ($02);
const // ADC synchroization codes
 NO_SYNC_E140                   = ($00);
 TTL_START_SYNC_E140            = ($01);
 TTL_KADR_SYNC_E140             = ($02);
 ANALOG_SYNC_E140               = ($03);
 INVALID_SYNC_E140              = ($04);
const // DAC detection codes
 DAC_INACCESSIBLED_E140         = ($00);
 DAC_ACCESSIBLED_E140           = ($01);
 INVALID_DAC_OPTION_E140        = ($02);
const // ADC input ranges, V
 ADC_INPUT_RANGES_E140 : array [0..ADC_INPUT_RANGES_QUANTITY_E140-1] of Double
                       = ( 10.0, 10.0/4.0, 10.0/16.0, 10.0/64.0 );
const // DAC output range, V
 DAC_OUTPUT_RANGE_E140 = 5.0;
const // ADC data rate
 E140MaxAdcDataRate = 200; // E140 Maximal ADC data rate, kHz
 E440MaxAdcDataRate = 400; // E440 Maximal ADC data rate, kHz

const // Internal ADC buffer size
 E140BigBufferSize = E140BuffArraySize*2;
 
const // Block Marker - for debug only
 E140BlockMarkerFlag : Boolean = False;
 E140BlockMarkerVal1 : Integer = +8000;
 E140BlockMarkerVal2 : Integer = -8000;

implementation

 ////////////////////
 // Utility functions
 ////////////////////

function LDevDriverStatus:LongString;
var s:LongString;
begin
 Result:='INVALID'; s:='';
 if IsUnix then begin
  if RunCommand('unix ldev-cpl status',s) then
  if StartsText('ldev status:',s) then s:=ExtractWord(3,s,ScanSpaces);
  if (WordIndex(s,LDevDriverStatusNames,ScanSpaces)>0) then Result:=s;
 end;
 if IsWindows then begin
  s:=file_which('lcomp.dll');
  if IsNonEmptyStr(s) and FileExists(s) then Result:='STARTED';
 end;
end;

function LDevDriverStart:Boolean;
var ans:LongString;
begin
 Result:=False;
 ans:=LDevDriverStatus;
 if SameText(ans,'INVALID') then Exit;
 if SameText(ans,'STARTED') then Exit(True);
 if SameText(ans,'STOPPED') then begin
  if RunCommand(UnixSudoit('unix ldev-cpl start'),ans) then begin
   ans:=LDevDriverStatus;
   if SameText(ans,'STARTED') then Exit(true);
  end;
 end;
end;

function GetListOfDevShm:LongString;
var cmd,ans:LongString;
begin
 Result:='';
 cmd:=''; ans:='';
 if IsUnix then begin
  cmd:='ls /dev/shm/';
  if RunCommand(cmd,ans) then Result:=cmd+EOL+ans;
 end;
end;

function UnlinkDevShm(shmName:LongString):LongString;
var shm,cmd,ans:LongString;
begin
 Result:='';
 if not IsUnix then Exit;
 shmName:=ExtractFileName(shmName);
 if IsEmptyStr(shmName) then Exit;
 shm:='/dev/shm/'+shmName;
 cmd:='unlink '+shm; ans:='';
 if IsUnix and FileExists(shm) then begin
  if RunCommand('unlink '+shm,ans)
  then Result:='Exec: '+cmd+EOL+ans
  else Result:='Fail: '+cmd+EOL;
 end;
end;

const
 lsldevOut:LongString='';

function LDevListDevices(Reset:Boolean=false):LongString;
begin
 Result:='';
 if Reset then lsldevOut:='';
 if IsEmptyStr(lsldevOut) then
 if RunCommand('unix lsldev',lsldevOut)
 then lsldevOut:=ValidateEol(lsldevOut)
 else lsldevOut:='';
 Result:=lsldevOut;
end;

function LPlataSlotBySerNum(const SerNum:LongString; Id:LongString='31:E140;30:E440'):Integer;
var il,slot:Integer; Lines,Line,Iden,Sern:LongString;
begin
 Result:=-1;
 if (WordCount(Id,ScanSpaces)>1) then begin
  for il:=1 to WordCount(Id,ScanSpaces) do begin
   Iden:=Trim(ExtractWord(il,Id,ScanSpaces));
   slot:=LPlataSlotBySerNum(SerNum,Iden);
   if (slot<0) then continue;
   Result:=slot;
  end;
  Exit;
 end;
 Lines:=LDevListDevices;
 for il:=1 to WordCount(Lines,EolnDelims) do begin
  Line:=ExtractWord(il,Lines,EolnDelims);
  Iden:=ExtractWord(5,Line,ScanSpaces);
  Sern:=ExtractWord(8,Line,ScanSpaces);
  if not SameText(Iden,Id) then Continue;
  Slot:=StrToIntDef(ExtractWord(2,Line,ScanSpaces),-1);
  if (Slot<0) then Continue;
  if (SerNum='') then Exit(Slot);
  if SameText(SerNum,Sern) then Exit(Slot);
 end;
end;

function E140CwCompose(nChan,nGain:Integer; nType:Integer=0):Integer;
begin
 Result:=(nChan and 31) or ((nType and 1) shl 5) or ((nGain and 3) shl 6);
end;

function E140CwGetChan(cw:Integer):Integer;
begin
 Result:=(cw and 31);
end;

function E140CwGetGain(cw:Integer):Integer;
begin
 Result:=((cw shr 6) and 3);
end;

function E140CwGetType(cw:Integer):Integer;
begin
 Result:=((cw shr 5) and 1);
end;

function TestE140AdcPars(nChan,nGain:Integer; Freq:Double):ADC_PAR;
var Adc:ADC_PAR; i:Integer;
begin
 Adc:=Default(ADC_PAR);
 with Adc.t1 do begin
  s_Type           := L_ADC_PARAM;
  AutoInit         := 1;
  FIFO             := 0;
  IrqStep          := 0;
  Pages            := 0;
  SynchroType      := 0;
  SynchroMode      := 0;
  AdChannel        := 0;
  AdPorog          := 0;
  NCh              := nChan;
  for i:=0 to nChan-1 do Chn[i]:=E140CwCompose(i,nGain);
  dRate            := EnsureRange(Freq,0,E440MaxAdcDataRate);
  dKadr            := 0;
  AdcEna           := 1;
  IrqEna           := 1;
 end; 
 Result:=Adc;
end;

 //////////////////////////////
 // TAdcPollRing implementation
 //////////////////////////////
procedure TAdcPollRing.Reset(const pAdc:pADC_PAR=nil);
begin
 BuffLeng:=0; DataBuff:=nil; DataSync:=nil; 
 HalfFlag:=False; HalfStep:=0; DataOffs:=0;
 if Assigned(pAdc) then Init(pAdc^);
end;

function TAdcPollRing.HalfLeng:ULONG;
begin
 Result:=(BuffLeng div 2);
end;

function TAdcPollRing.GetSync:ULONG;
const z=High(ULONG);
begin
 if Assigned(DataSync)
 then Result:=LockedCompareExchange(DataSync^,z,z)
 else Result:=0;
end;

function TAdcPollRing.Valid:Boolean;
begin
 Result:=Assigned(DataBuff) and Assigned(DataSync) and (BuffLeng>0);
end;

function TAdcPollRing.Init(const Adc:ADC_PAR):Boolean;
begin
 BuffLeng:=Adc.t1.IrqStep*Adc.t1.Pages;
 Result:=Valid;
end;

function TAdcPollRing.HalfBuff:PSmallIntArray;
var i:Integer;
begin
 if HalfFlag then i:=HalfLeng else i:=0;
 Result:=@DataBuff[i];
end;

function TAdcPollRing.HalfBuffHasCome:Boolean;
var HFlag:Boolean;
begin
 HFlag:=(GetSync<=HalfLeng);
 Result:=(HFlag=HalfFlag);
end;

procedure TAdcPollRing.NextHalf;
begin
 HalfFlag:=not HalfFlag;
 Inc(DataOffs,HalfLeng);
 Inc(HalfStep);
end;

 ///////////////////////////////
 // TAdcPollBlock implementation
 ///////////////////////////////

function TAdcPollBlock.ItemSize:Integer;
begin
 Result:=SizeOf(DataPtr[0]);
end;

function TAdcPollBlock.DataSize:Integer;
begin
 Result:=DataLen*SizeOf(DataPtr[0]);
end;

function TAdcPollBlock.Ready:Boolean;
begin
 Result:=Assigned(DataPtr) and (DataLen>0);
end;

procedure TAdcPollBlock.Clear(Mode:Integer);
begin
 DataLen:=0; DataPtr:=nil; StepNum:=0; FlatNum:=0;
 if HasFlags(Mode,1) then Elapsed:=0;
end;

 ///////////////////
 // TAdcRequestQueue
 ///////////////////
function TAdcRequestQueue.Valid:Boolean;
begin
 if not InRange(BuffLeng,E140MinBuffLen,E140MaxBuffLen) then Exit(False);
 if not InRange(Number,0,E140NumBuffers-1) then Exit(False);
 if (DataBuff[Number]=nil) then Exit(False);
 Result:=True;
end;

procedure TAdcRequestQueue.Clear;
begin
 SafeFillChar(Index,SizeOf(Index),0);
 SafeFillChar(Offset,SizeOf(Offset),0);
 SafeFillChar(DataBuff,SizeOf(DataBuff),0);
 Number:=0; Blocks:=0; Samples:=0; BuffLeng:=0;
end;

procedure TAdcRequestQueue.Init(Shared:PE140_SHARED_BUFFER; Leng:Integer=E140MaxBuffLen);
var i:Integer;
begin
 Clear;
 if Assigned(Shared) and (Leng>0) then begin
  BuffLeng:=EnsureRange(Leng,E140MinBuffLen,E140MaxBuffLen);
  for i:=0 to E140NumBuffers-1 do begin
   DataBuff[i]:=@Shared.Buffer[i*BuffLeng];
   Offset[i]:=(PChar(DataBuff[i])-PChar(Shared));
   Index[i]:=i*BuffLeng;
  end;
 end;
end;

procedure TAdcRequestQueue.NextStep;
begin
 Number:=((Number+1) mod E140NumBuffers);
end;

 //////////////////////
 // TE140 help routines
 //////////////////////

function LSucceed(Code:ULONG):Boolean;
begin
 Result:=(Code=L_SUCCESS);
end;

function ValidHandle(h:THandle):Boolean;
begin
 Result:=(h<>0) and (h<>INVALID_HANDLE_VALUE);
end;

procedure DefaultReporter(const Msg:LongString);
begin
 StdOut.Put:=Msg;
end;

procedure FillTestAdcPar(var adcPar: ADC_PAR);
begin
 adcPar.t1.s_Type := L_ADC_PARAM;
 adcPar.t1.AutoInit := 1;
 adcPar.t1.dRate := 100.0;
 adcPar.t1.dKadr := 0;
 adcPar.t1.dScale := 0;
 adcPar.t1.SynchroType := 0;
 adcPar.t1.SynchroSensitivity := 0;
 adcPar.t1.SynchroMode := 0;
 adcPar.t1.AdChannel := 0;
 adcPar.t1.AdPorog := 0;
 adcPar.t1.NCh := 4;
 adcPar.t1.Chn[0] := $0;
 adcPar.t1.Chn[1] := $1;
 adcPar.t1.Chn[2] := $2;
 adcPar.t1.Chn[3] := $3;
 adcPar.t1.FIFO := 4096;
 adcPar.t1.IrqStep := 4096;
 adcPar.t1.Pages := 32;
 adcPar.t1.IrqEna := 1;
 adcPar.t1.AdcEna := 1;
end;

 ///////////////////////
 // TE140 implementation
 ///////////////////////

procedure TE140.SetReporter(aReporter:TLDev_Reporter);
begin
 if Assigned(Self) then myReporter:=aReporter;
end;

procedure TE140.PrintReport(const msg:LongString);
begin
 if Assigned(Self) then
 if Assigned(myReporter) then
 try
  myReporter(msg);
 except
  on E:Exception do LockedInc(StdIoErrorCount);
 end;
end;

function TE140.GetSharedBuff:PE140_SHARED_BUFFER;
begin
 if Assigned(Self)
 then Result:=myShared
 else Result:=nil;
end;

function TE140.GetSlot:Integer;
begin
 if Assigned(Self)
 then Result:=myBoard.Slot
 else Result:=-1;
end;

function TE140.GetSharedName:LongString;
begin
 if Assigned(Self)
 then Result:=mySharm.Name
 else Result:='';
end;

function TE140.GetSerNumWanted:LongString;
begin
 if Assigned(Self)
 then Result:=mySerNumWanted
 else Result:='';
end;

procedure TE140.SetSerNumWanted(const aSerNum:LongString);
begin
 if Assigned(Self) then mySerNumWanted:=Trim(aSerNum);
end;

function TE140.GetSerNumActual:LongString;
begin
 if Assigned(Self)
 then Result:=mySerNumActual
 else Result:='';
end;

function TE140.GetBoardName:LongString;
begin
 if Assigned(Self)
 then Result:=myBoard.BoardName
 else Result:='';
end;

function TE140.GetBoardType:Integer;
begin
 if Assigned(Self)
 then Result:=myBoard.BoardType
 else Result:=L_NONE;
end;

function TE140.GetBuffTime:Integer;
begin
 if Assigned(Self)
 then Result:=myBuffTime
 else Result:=0;
end;

function TE140.Started:Double;
begin
 if Assigned(Self) and Assigned(myShared)
 then Result:=myShared.StartTime
 else Result:=0;
end;

function TE140.Opened:Boolean;
begin
 if Assigned(Self)
 then Result:=Assigned(myApi) and ValidHandle(myHand)
 else Result:=False;
end;

function TE140.ReadSerialNumber(Save:Boolean=True):LongString;
var sp:SLOT_PAR; pd:PLATA_DESCR_U2;
begin
 Result:='';
 if Assigned(Self) then
 try
  sp:=Default(SLOT_PAR);
  pd:=Default(PLATA_DESCR_U2);
  if Opened then
  if Assigned(myApi) then
  if LSucceed(myApi.GetSlotParam(sp)) then
  if LSucceed(myApi.ReadPlataDescr(pd)) then begin
   if Save then begin myBoard.Sp:=sp; myBoard.Pd:=pd; end;
   Result:=LPlataSerNum(sp,pd);
  end;
 except
  on E:Exception do BugReport(E,'ReadSerialNumber');
 end;
end;

procedure TE140.Success(const s:LongString);
begin
 Report:='@E140.OnSuccess='+s;
end;
 
procedure TE140.Failure(const s:LongString);
begin
 Raise E140_Failure.Create(s);
end; 

procedure TE140.BugReport(const E:Exception; const msg:LongString);
begin
 if Assigned(E) then begin
  if Length(msg)>0
  then Report:=Format('@E140.OnException=%s,%s,"%s"',[E.ClassName,E.Message,msg])
  else Report:=Format('@E140.OnException=%s,%s',[E.ClassName,E.Message]);
 end;
 LockedInc(StdIoErrorCount);
end;

function TE140.Open(aSlot:Integer; aAdc:pADC_PAR=nil):Boolean;
var lnk,bio:LongString; ifac:LUnknown; icod,lcod:Integer;
begin
 Result:=false;
 try
  //
  // Check if one already started or opened
  //
  if Started>0 then Stop;
  if Opened then Close;
  //
  // Check device exists.
  //
  lnk:=LDeviceLink(aSlot);
  if LDeviceExists(lnk)
  then Success('Device exist '+lnk)
  else Failure('Error: Device is not exists '+lnk);
  //
  // Create Instance and query interface.
  //
  ifac:=LCreateInstance(aSlot);
  icod:=GetLastOsError;
  if Assigned(ifac)
  then Success(Format('Slot %d',[aSlot]))
  else Failure('Error: '+LErrorCodeName(icod));
  lcod:=ifac.QueryInterface(IID_ILDEV,myApi);
  if LSucceed(lcod) and Assigned(myApi)
  then Success('Interface '+GuidToString(IID_ILDEV))
  else Failure('Error: '+LErrorCodeName(lcod));
  //
  // Open device.
  //
  myHand:=myApi.OpenLDevice; 
  if ValidHandle(myHand)
  then Success('OpenLDevice')
  else Failure('Error: OpenLDevice '+LErrorCodeName(GetLastOsError)); 
  //
  // Detect board type
  //
  myBoard.Sp:=Default(SLOT_PAR);
  if LSucceed(myApi.GetSlotParam(myBoard.Sp))
  then Success('GetSlotParam')
  else Failure('Error: GetSlotParam');
  Success('Detected BoardType '+IntToStr(BoardType)+':'+BoardName);
  //
  // LoadBios for E440
  //
  if (BoardType=L_E440) then begin
   bio:='e440';
   if LSucceed(myApi.LoadBios(PChar(bio)))
   then Success('LoadBios '+bio)
   else Failure('Error: LoadBios '+bio);
  end;
  //
  // Test plata load succeed.
  //
  if LSucceed(myApi.PlataTest)
  then Success('PlataTest')
  else Failure('Error: PlataTest ');
  //
  // Read board name, serial number and other params.
  //
  mySerNumActual:=ReadSerialNumber;
  if SameText(BoardName,'E140') or SameText(BoardName,'E440')
  then Success('LBoardName '+BoardName+' ReadSerial '+SerNumActual)
  else Failure('Error: Device is not [E140,E440].');
  //
  // Device reset
  //
  if LSucceed(myApi.StopLDevice)
  then Success('StopLDevice')
  else Failure('StopLDevice'); 
  //
  // Check shared memory
  //
  if Assigned(myShared)
  then Success('SharedMemory = '+IntToStr(mySharm.Size)+', '+mySharm.Name)
  else Failure('SharedMemory');
  ZeroMemory(myShared,mySharm.Size);
  //
  // Read device driver parameters to shared memory
  //
  myShared.Board.Slot:=Slot;
  if LSucceed(myApi.GetSlotParam(myShared.Board.Sp))
  then Success('GetSlotParam')
  else Failure('GetSlotParam');
  if LSucceed(myApi.ReadPlataDescr(myShared.Board.Pd))
  then Success('GetPlataDescr')
  else Failure('GetPlataDescr');
  PrintDescr(myShared.Board.Sp,myShared.Board.Pd);
  //
  // Request ADC buffer
  //
  myBuffSize:=E140BigBufferSize;
  if LSucceed(myApi.RequestBufferStream(myBuffSize,L_STREAM_ADC))
  then Success('RequestBufferStream '+IntToStr(myBuffSize)+' bytes')
  else Failure('RequestBufferStream');
  //
  // Set/Get current ADC settings
  //
  if Assigned(aAdc) then begin
   myShared.AdcInited:=aAdc^;
   myShared.AdcWanted:=aAdc^;
   myShared.Board.Adc:=aAdc^;
   PrintAdc(myShared.AdcWanted);
   //
   // Read actual (accepted) ADC settings
   //
   if not LSucceed(myApi.FillDAQparameters(PDAQ_PAR(@myShared.Board.Adc)^))
   then Failure('FillDAQparameters')
   else Success('FillDAQparameters');
   PrintAdc(myShared.Board.Adc);
   myBoard.Adc:=myShared.Board.Adc;
  end;
  //
  // Done.
  //
  Result:=Opened;
  if Opened
  then Report:='Opened Slot '+IntToStr(aSlot)
  else Report:='Failed Slot '+IntToStr(aSlot);
 except
  on E:Exception do begin
   BugReport(E,'OpenSlot');
   Close;
  end;
 end;
end;

class function TE140.ValidateBuffTime(aBuffTime:Integer):Integer;
begin
 if (aBuffTime<=0) then aBuffTime:=DefBuffTime;
 if (aBuffTime<=0) then aBuffTime:=E140DefBuffTime;
 Result:=EnsureRange(aBuffTime,E140MinBuffTime,E140MaxBuffTime);
end;

constructor TE140.Create(aShare:LongString; aReporter:TLDev_Reporter;
                         aSlot:Integer; aAdc:pADC_PAR; aBuffTime:Integer);
begin
 Inherited Create;
 myBoard.Reset;
 myApi:=nil;
 myAdcRing.Reset;
 Reporter:=aReporter;
 myAdcBlock:=Default(TAdcPollBlock);
 myBuffTime:=ValidateBuffTime(aBuffTime);
 mySharm:=NewSharedMemory(aShare,Sizeof(myShared^));
 mySharm.Master:=@mySharm;
 myShared:=mySharm[0];
 if (aSlot>=0) then Open(aSlot,aAdc);
end;

destructor TE140.Destroy;
begin
 if Started>0 then Stop;
 if Opened then Close;
 Kill(mySharm);
 myBoard.Reset;
 myApi:=nil;
 mySerNumWanted:='';
 mySerNumActual:='';
 Inherited Destroy;
end;

procedure TE140.PrintDescr(const sp:SLOT_PAR; const pd:PLATA_DESCR_U2);
begin
 if Assigned(Self) then
 try
  Report:=GetE140Description(sp,pd,'@E140.OnSuccess= ');
 except
  on E:Exception do BugReport(E,'PrintDescr');
 end;
end;

procedure TE140.PrintADC(const aAdc:ADC_PAR);
begin
 if Assigned(Self) then
 try
  Report:=GetE140AdcPars(aAdc,'@E140.OnSuccess= ');
 except
  on E:Exception do BugReport(E,'PrintAdc');
 end;
end;

function TE140.MaxAdcRate:Double;
begin
 Result:=0;
 if Assigned(Self) then
 case BoardType of
  L_NONE: Result:=E440MaxAdcDataRate;
  L_E140: Result:=E140MaxAdcDataRate;
  L_E440: Result:=E440MaxAdcDataRate;
  else    Result:=0;
 end;
end;

function GetNumBitsSignificant(aValue:Integer):Integer;
begin
 Result:=0;
 while (aValue<>0) do begin
  aValue:=(aValue shr 1);
  Inc(Result);
 end;
end;

function TE140.ValidateAdcPages(var Adc:ADC_PAR; StepMs:Integer=0):Boolean;
var wl,nw,nb,ns,np,nl,n1,n2:Integer;
begin
 Result:=False;
 if not Assigned(Self) then Exit;
 if (StepMs<=0) then StepMs:=BuffTime;
 Adc.t1.dRate:=EnsureRange(Adc.t1.dRate,0,MaxAdcRate);
 // Wanted block length, i.e. num. points in block
 wl:=LeastPowerOfTwo(Round(Adc.t1.dRate*StepMs*2));
 nw:=EnsureRange(wl,E140MinBuffLen,E140MaxBuffLen);
 n1:=GetNumBitsSignificant(E140MinBuffLen)-1;
 n2:=GetNumBitsSignificant(E140MaxBuffLen)-1;
 for nb:=n2 downto n1 do begin
  ns:=(Integer(1) shl nb);
  np:=nw div ns;
  if (np>=4) then begin
   Adc.t1.IrqStep:=ns;
   Adc.t1.Pages:=np;
   Break;
  end;
 end;
 nl:=Adc.t1.IrqStep*Adc.t1.Pages;
 Result:=InRange(nl,E140MinBuffLen,E140MaxBuffLen);
end;

class function TE140.ValidateAdcChans(var Adc:ADC_PAR):Boolean;
const nlo=Low(Adc.t1.Chn); nhi=High(Adc.t1.Chn);
var lpt,i,j:Integer;
begin
 if (Adc.t1.NCh>0) then begin
  lpt:=LeastPowerOfTwo(Adc.t1.NCh);
  lpt:=EnsureRange(lpt,nlo+1,nhi+1);
  if (lpt>Adc.t1.NCh) then begin
   for i:=0 to Adc.t1.NCh-1 do begin
    j:=i+Adc.t1.NCh;
    if InRange(j,nlo,lpt-1)
    then Adc.t1.Chn[j]:=Adc.t1.Chn[i] 
    else Break;
   end;
   Adc.t1.Nch:=lpt;
  end;
 end;
 Result:=InRange(Adc.t1.NCh,nlo+1,nhi+1);
end;

function TE140.FindBufferLength(aBuffTime:Integer; const aAdc:ADC_PAR):Integer;
var Adc:ADC_PAR;
begin
 Result:=0;
 if Assigned(Self) then begin
  if (aBuffTime<=0) then Result:=E140MaxBuffLen else begin
   Adc:=aAdc; ValidateAdcPages(Adc,aBuffTime);
   Result:=Adc.t1.IrqStep*Adc.t1.Pages;
   if not InRange(Result,E140MinBuffLen,E140MaxBuffLen)
   then Failure(Format('Invalid buffer length %d',[Result]));
  end;
 end;
end;

const
 IsClearHeader : Boolean = False;
 IsClearBuffer : Boolean = True;

procedure TE140.Close;
begin
 if Assigned(Self) then
 try
  if Started>0 then Stop;
  if Assigned(myApi) then begin
   if ValidHandle(myHand) then begin
    myApi.StopLDevice;
    myApi.CloseLDevice;
   end;
   myApi:=nil;
  end;
  myBoard.Reset;
  mySerNumActual:='';
  myAdcBlock:=Default(TAdcPollBlock);
  myHand:=INVALID_HANDLE_VALUE;
  if Assigned(myShared) then begin
   if IsClearHeader then ZeroMemory(@myShared.Header,SizeOf(myShared.Header));
   if IsClearBuffer then ZeroMemory(@myShared.Buffer,SizeOf(myShared.Buffer));
  end;
 except
  on E:Exception do BugReport(E,'Close');
 end;
end;

function TE140.Stop:Double;
begin
 Result:=Started;
 if Result>0 then
 try
  //
  // Check interface
  //
  if Assigned(myApi)
  then Success('STOP')
  else Failure('STOP'); 
  //
  // Reset device
  //
  if LSucceed(myApi.StopLDevice)
  then Success('StopLDevice')
  else Failure('StopLDevice');
  //
  // Reset ADC ring buffer, clear StartTime
  //
  myAdcRing.Reset;
  myShared.StartTime:=0;
 except
  on E:Exception do begin
   BugReport(E,GetLastErrorInfo);
   Close;
  end;
 end;
end;

function TE140.Start(aAdc:pADC_PAR):Double;
var Leng:Integer; aTestAdcPar:ADC_PAR;
begin
 Result:=0;
 if Opened then
 try
  //
  // Stop if one already started
  //
  if Started>0 then Stop;
  //
  // Check interface
  //
  if Assigned(myApi)
  then Success('START')
  else Failure('START'); 
  //
  // Check shared memory
  //
  if Assigned(myShared)
  then Success('SharedMemory = '+IntToStr(mySharm.Size)+', '+mySharm.Name)
  else Failure('SharedMemory');
  //
  // Reset device
  //
  if LSucceed(myApi.StopLDevice)
  then Success('StopLDevice')
  else Failure('StopLDevice');
  //
  // Use Test params if aAdc=nil
  //
  aTestAdcPar:=Default(ADC_PAR);
  FillTestAdcPar(aTestAdcPar);
  if (aAdc=nil) then aAdc:=@aTestAdcPar;
  //
  // Write wanted ADC settings if one specified
  //
  if Assigned(aAdc) then begin
   myShared.AdcWanted:=aAdc^;
   ValidateAdcPages(myShared.AdcWanted);
   ValidateAdcChans(myShared.AdcWanted);
   PrintAdc(myShared.AdcWanted);
   myShared.Board.Adc:=myShared.AdcWanted;
   if not LSucceed(myApi.FillDaqParameters(PDAQ_PAR(@myShared.Board.Adc)^))
   then Failure('FillDaqParameters')
   else Success('FillDaqParameters');
   PrintAdc(myShared.Board.Adc);
   myBoard.Adc:=myShared.Board.Adc;
  end;
  //
  // Calculate wanted buffer size
  // It's near BuffTime ms of data taking
  //
  Leng:=FindBufferLength(BuffTime,myShared.Board.Adc);
  Success(Format('BuffTime = %d',[BuffTime]));
  Success(Format('BuffLeng = %d',[Leng]));
  //
  // SetParametersStream
  //
  myAdcRing.Reset(@myShared.Board.Adc);
  if LSucceed(myApi.SetParametersStream(PDAQ_PAR(@myShared.Board.Adc)^, myAdcRing.BuffLeng, myAdcRing.DataBuff, myAdcRing.DataSync,L_STREAM_ADC))
  then Success('SetParametersStream')
  else Failure('SetParametersStream');
  //
  // Prepare asynchronous I/O buffers
  //
  myRequest.Init(myShared,E140MaxBuffLen);
  myAdcBlock.Clear(1);
  //
  // Check ADC ring buffer.
  //
  if myAdcRing.Valid 
  then Success('AdcRing.BuffLeng = '+IntToStr(myAdcRing.BuffLeng))
  else Failure('AdcRing is invalid');
  //
  // Initialize device internals required to Start
  //
  if LSucceed(myApi.InitStartLDevice)
  then Success('InitStartLDevice')
  else Failure('InitStartLDevice');
  //
  // Start device
  //
  if LSucceed(myApi.StartLDevice)
  then Success('StartLDevice')
  else Failure('StartLDevice');
  //
  // Check ADC started.
  //
  if IsAdcRunning(1000)
  then Success('ADC Started.');
  //
  // Remember start time.
  //
  myShared.StartTime:=MSecNow;
  Result:=Started;
 except
  on E:Exception do begin
   BugReport(E,GetLastErrorInfo);
   Close;
  end;
 end;
end;

function TE140.IsAdcRunning(aTimeout:Integer=0):Boolean;
var sync:Integer; ms,Deadline:Double;
begin
 Result:=False;
 if Assigned(Self) then
 if Assigned(mySharm) then
 if Assigned(myShared) then
 if (myAdcRing.Valid) then begin
  if not Opened then Exit(False);
  if (aTimeout<=0) then Exit(Started>0);
  sync:=myAdcRing.GetSync;
  ms:=mSecNow; Deadline:=ms+aTimeout;
  while (mSecNow<=Deadline) do begin
   if (myAdcRing.GetSync<>sync)
   then Exit(True);
  end;
 end;
end;

function TE140.PollAdcHandler(const Adc:ADC_PAR; var Block:TAdcPollBlock):Int64;
var Passed,Elapsed:Double;
begin
 Result:=0;
 Block.Clear(0);
 if Assigned(Self) then
 if Assigned(myApi) then
 if Assigned(myShared) then
 try
  if IsAdcRunning then begin
   if not myAdcRing.Valid then Failure('Error: invalid AdcRing');
   if myAdcRing.HalfBuffHasCome then begin
    Elapsed:=(mSecNow-Started)*1e-3;
    Passed:=(Elapsed-Block.Elapsed)*1e3;
    Block.Elapsed:=Elapsed;
    Block.DataPtr:=myAdcRing.HalfBuff;
    Block.DataLen:=myAdcRing.HalfLeng;
    Block.StepNum:=myAdcRing.HalfStep;
    Block.FlatNum:=myAdcRing.DataOffs;
    if E140BlockMarkerFlag then begin
     Block.DataPtr[0]:=E140BlockMarkerVal1;
     Block.DataPtr[Block.DataLen-Adc.t1.nch]:=E140BlockMarkerVal2;
    end;
    StdOut.Put:=Format('AdcBlock DataLen=%d; StepNum=%d; DataOffs=%d; Elapsed=%1.3f sec; Step=%.3f ms;',
                 [Block.DataLen,Block.StepNum,Block.FlatNum,Block.Elapsed,Passed]);
    Result:=Block.FlatNum;
    myAdcRing.NextHalf;
   end;
  end;
 except
  on E:Exception do begin
   BugReport(E,GetLastErrorInfo);
   Close;
  end;
 end;
end;

function TE140.Polling:Int64;
var DataOffs:Int64;
begin
 Result:=0;
 if Assigned(Self) then
 if Assigned(myApi) then
 if Assigned(myShared) then
 try
 if Started>0 then begin
   DataOffs:=PollAdcHandler(myShared.Board.Adc,myAdcBlock);
   Result:=myRequest.Blocks;
   if myAdcBlock.Ready and myRequest.Valid then begin
    if InRange(myAdcBlock.DataLen,1,myRequest.BuffLeng) then begin
     SafeMove(myAdcBlock.DataPtr^,myRequest.DataBuff[myRequest.Number]^,myAdcBlock.DataSize);
     SafeFillChar(myAdcBlock.DataPtr^,myAdcBlock.DataSize,0);
    end else Failure(Format('Bad DataLen %d over %d.',[myAdcBlock.DataLen,myRequest.BuffLeng]));
    ProcessBuffer(myAdcBlock.DataLen,myRequest.Number,myRequest.Samples,myRequest.Blocks);
    Inc(myRequest.Blocks); 
    Result:=myRequest.Blocks;
    Inc(myRequest.Samples,myAdcBlock.DataLen);
    Success(Format('RequestNumber %d; DataOffs %d;',[myRequest.Number,DataOffs]));
    myRequest.NextStep;
   end;
  end;
 except
  on E:Exception do begin
   BugReport(E,GetLastErrorInfo);
   Close;
  end;
 end;
end;

function TE140.ProcessBuffer(nWords:DWORD; nRequest:Integer; nSamples:Int64; nBlocks:Int64):DWORD;
var sn:LongString; nw:DWORD; nr,ri,ro:Integer; nb,ns:Int64;
begin
 Result:=0;
 if Assigned(Self) then
 try
  sn:=mySharm.Name; nw:=nWords;
  nr:=nRequest; nb:=nBlocks; ns:=nSamples;
  ro:=myRequest.Offset[nr]; ri:=myRequest.Index[nr];
  Report:=Format('@E140.OnReadout=%s,%d,%d,%d,%d,%d,%d',
                [sn, nw, ro, ri, nr, nb, ns]);
  Result:=nw;
 except
  on E:Exception do BugReport(E,'ProcessBuffer');
 end;
end;

function TE140.DAC_SAMPLE(DacData:ULONG; DacChannel:WORD):Boolean;
begin
 Result:=False;
 if Opened then
 try
  if (BoardType in [L_E140]) then begin
   myBoard.Ap:=Default(ASYNC_PAR);
   myBoard.Ap.s_Type:=L_ASYNC_DAC_OUT;
   myBoard.Ap.Mode:=DacChannel;
   myBoard.Ap.Data[0]:=DacData;
   Result:=LSucceed(myApi.IoAsync(PDAQ_PAR(@myBoard.Ap)^));
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function TE140.TTL_OUT(TtlOut:ULONG):Boolean;
begin
 Result:=False;
 if Opened then
 try
  if (BoardType in [L_E140]) then begin
   myBoard.Ap:=Default(ASYNC_PAR);
   myBoard.Ap.s_Type:=L_ASYNC_TTL_OUT;
   myBoard.Ap.Data[0]:=TtlOut;
   Result:=LSucceed(myApi.IoAsync(PDAQ_PAR(@myBoard.Ap)^));
  end;
 except
  on E:Exception do BugReport(E,'TTL_OUT');
 end;
end;

function TE140.TTL_IN(var TtlIn:ULONG):Boolean;
begin
 Result:=False;
 if Opened then
 try
  if (BoardType in [L_E140]) then begin
   myBoard.Ap:=Default(ASYNC_PAR);
   myBoard.Ap.s_Type:=L_ASYNC_TTL_INP;
   Result:=LSucceed(myApi.IoAsync(PDAQ_PAR(@myBoard.Ap)^));
   TtlIn:=myBoard.Ap.Data[0];
  end;
 except
  on E:Exception do BugReport(E,'TTL_IN');
 end;
end;
 
function TE140.GetLastErrorInfo:LongString;
begin
 Result:='E140 error: NIL reference';
 if Assigned(Self) then
 if Assigned(myShared) then
 try
  if Assigned(myApi) then
  Result:=Format('E140 error %d',[myShared.Board.Errno]);
 except
  on E:Exception do Result:=E.Message;
 end;
end;
 
function TE140.SharedBufferInfo:LongString;
begin
 Result:='E140 error: NIL reference';
 if Assigned(Self) then
 if Assigned(mySharm) then
 if Assigned(myShared) then
 try
  Result:=Format('Name=%s;Path=%s;Size=%d;Capacity=%d;HeaderSize=%d;CookiesSize=%d;BufferSize=%d;',
  [mySharm.Name,mySharm.PathName,mySharm.Size,mySharm.Capacity,
   SizeOf(myShared.Header),myShared.CookiesSize,SizeOf(myShared.Buffer)]);
 except
  on E:Exception do Result:=E.Message;
 end;
end;

function NewE140(aShare:LongString; aReporter:TLDev_Reporter; aSlot:Integer; aAdc:pADC_PAR; aBuffTime:Integer):TE140;
begin
 Result:=nil;
 try
  Result:=TE140.Create(aShare,aReporter,aSlot,aAdc,aBuffTime);
  if not Assigned(Result.myShared) then Raise E140_FAILURE.Create('Shared memory fails!');
 except
  on E:Exception do begin
   Echo(Format('@E140.OnException=%s,%s',[E.ClassName,E.Message]));
   Kill(Result);
  end;
 end; 
end;

procedure Kill(var TheObject:TE140); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do Echo(Format('@E140.OnException=%s,%s',[E.ClassName,E.Message]));
 end; 
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_e140_api;
begin
 lsldevOut:='';
 TE140.DefBuffTime:=E140DefBuffTime;
end;

procedure Free_crw_e140_api;
begin
 lsldevOut:='';
end;

initialization

 Init_crw_e140_api;

finalization

 Free_crw_e140_api;

end.

//////////////
// END OF FILE
//////////////

