@echo off
title %~nx0
SetLocal EnableExtensions EnableDelayedExpansion
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: Copyright (c) 2001-2021 Alexey Kuryakin kouriakine@mail.ru ::
:: Under MIT License, see https://opensource.org/licenses/MIT ::
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: Utility to make UniHeater`s configuration.                 ::
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

chcp 65001

:SetSelfInfo
set SelfArgs=%*
set "SelfExt=%~x0"
set "SelfBase=%~n0"
set "SelfFile=%~f0"
set "SelfName=%~nx0"
for %%i in (".") do set "HomeDir=%%~fi"
for %%i in ("%~dp0\.") do set "SelfDir=%%~fi"

:ParseOptions
if /i "%~1" == "/?"         goto :PrintHelp
if /i "%~1" == "-h"         goto :PrintHelp
if /i "%~1" == "--help"     goto :PrintHelp
if /i "%~1" == "--version"  goto :PrintVersion

:ParseArguments
call :RequiredParamStr UH_PrefixName %1                     || exit /b 1    & rem Prefix string to identify UniHeater
call :RequiredParamInt UH_NumHeaters %2                     || exit /b 1    & rem Number of Unified Heaters, 1..32
call :RequiredParamInt UH_TileNumber %3                     || exit /b 1    & rem Tile number per Row, 1..UH_NumHeaters
call :RequiredParamStr UH_CfgFolder  %4                     || exit /b 1    & rem Target Directory for CFG files
call :RequiredParamStr UH_CrcFolder  %5                     || exit /b 1    & rem Target Directory for CRC files
call :RequiredParamStr UH_BmpFolder  %6                     || exit /b 1    & rem Target Directory for BMP files

:SetAddonVariables
call :GetPrefixFile UH_PrefixFile %UH_PrefixName%                           & rem File name prefix
call :Define UH_PrefixFile      %UH_PrefixName:.=_%                         & rem File name prefix
call :Define UH_HomeDir   %CRW_DAQ_SYS_HOME_DIR%\Resource\DaqSite\UniHeater & rem Source directory of UniHeater
call :Define UH_PrefixTSen      SENSOR.T                                    & rem Prefix of T sensors to bind
call :Define UH_PrefixPSen      SENSOR.P                                    & rem Prefix of P sensors to bind
call :Define UH_NumberTSen      %UH_NumHeaters%                             & rem Number of T sensors to bind
call :Define UH_NumberPSen      %UH_NumHeaters%                             & rem Number of P sensors to bind
call :Define UH_NickTSen        T                                           & rem Prefix of T sensors nicknames to display
call :Define UH_NickPSen        P                                           & rem Prefix of P sensors nicknames to display
call :Define UH_SmoothingWw     1e-4                                        & rem Smoothing Window Width, in TimeUnits
call :Define UH_SmoothingPw     0                                           & rem Smoothing Power, -1..9
call :Define UH_SmoothingK1     2                                           & rem Smoothing Param K1,  0..9
call :Define UH_SmoothingK2     2                                           & rem Smoothing Param K2,  0..9
call :Define UH_HistoryLen      1000                                        & rem History for curves
call :Define UH_EncryptMethod   RC6                                         & rem DEFAULT,BLOWFISH,GOST,RC2,RC4,RC5,RC6,BASE64,HEX,NONE

:CheckRequiredEntities
call :RequiredUnixUtils                                     || exit /b 1
call :RequiredAdminRights                                   || exit /b 1
call :RequiredParentCrwDaq                                  || exit /b 1
call :RequiredDefined UH_PrefixName                         || exit /b 1
call :RequiredDefined UH_PrefixFile                         || exit /b 1
call :RequiredDefined UH_NumHeaters                         || exit /b 1
call :RequiredDefined UH_TileNumber                         || exit /b 1
call :RequiredIntRange %UH_NumHeaters% 1 32                 || exit /b 1
call :RequiredIntRange %UH_TileNumber% 1 %UH_NumHeaters%    || exit /b 1
call :RequiredDefined UH_CfgFolder                          || exit /b 1
call :RequiredDefined UH_BmpFolder                          || exit /b 1
call :RequiredDefined UH_CrcFolder                          || exit /b 1
call :RequiredFile "%UH_CfgFolder%\"                        || exit /b 1
call :RequiredFile "%UH_BmpFolder%\"                        || exit /b 1
call :RequiredFile "%UH_CrcFolder%\"                        || exit /b 1
call :RequiredDefined UH_HomeDir                            || exit /b 1
call :RequiredFile "%UH_HomeDir%\"                          || exit /b 1
call :RequiredFile "%UH_HomeDir%\uniheater_make_crc.cmd"    || exit /b 1

:Main
call :SetErrors 0
call :SetErrorLevel 0
call :SetAnsiCodePage
pushd "%HomeDir%" && call :UniHeaterConfig
popd
call :HasNoErrors || call :ErrorPrintLn "%errors% error(s) found in %SelfName%."
call :RestoreCodePage
exit /b %ERRORLEVEL%
goto :EOF

:PrintVersion
echo %SelfBase% version 1.0
goto :EOF

:PrintHelp
call :PrintVersion
echo Copyright ^(c^) 2001-2021 Alexey Kuryakin kouriakine@mail.ru
echo Utility to make UniHeater`s configuration.
echo Usage:
echo  %SelfBase% [Options] [Arguments]
echo Options:
echo  -h,--help  - print help
echo  --version  - print version
echo Arguments:
echo  UH_PrefixName - Prefix name to identify Uniheater
echo  UH_NumHeaters - Number of Unified Heaters, 1..32
echo  UH_TileNumber - Tile number = columns per row, 1..UH_NumHeaters
echo  UH_CfgFolder  - Target directory for CFG files, like ..\Config
echo  UH_CrcFolder  - Target directory for CRC files, like ..\Circuits
echo  UH_BmpFolder  - Target directory for BMP files, like ..\Bitmaps
echo Example:
echo  call %SelfBase% DEMO 16 8 ..\Config ..\Circuits ..\Bitmaps
goto :EOF

:UniHeaterConfig
call :UniHeaterEnvironment                                                    || call :IncErrors 1
call :UniHeaterBindCfg  > "%UH_CfgFolder%\%UH_PrefixFile%_uniheater_bind.cfg" || call :IncErrors 1
call :UniHeaterCtrlCfg  > "%UH_CfgFolder%\%UH_PrefixFile%_uniheater_ctrl.cfg" || call :IncErrors 1
call :UniHeaterDisCfg   > "%UH_CfgFolder%\%UH_PrefixFile%_uniheater_dis.cfg"  || call :IncErrors 1
call :UniHeaterDicCfg   > "%UH_CfgFolder%\%UH_PrefixFile%_uniheater_dic.cfg"  || call :IncErrors 1
call :exec "%UH_HomeDir%\uniheater_make_crc.cmd" %UH_PrefixName% %UH_NumHeaters% %UH_TileNumber% "%UH_CrcFolder%" "%UH_BmpFolder%" || call :IncErrors 1
goto :EOF

:UniHeaterEnvironment
set "UH_HomeRef=~~\Resource\DaqSite\UniHeater"  & rem UniHeater home directory reference
set "UH_PREF=%UH_PrefixName%.UNIHEATER"         & rem UniHeater name prefix
set "UH_FILE=%UH_PREF:.=_%"                     & rem UniHeater file prefix
set "UH_PATH=%UH_PREF:.=/%"                     & rem UniHeater path prefix
goto :EOF

:UniHeaterBindCfg
call :UniHeaterSelector
goto :EOF

:UniHeaterCtrlCfg
    call :UniHeaterHeader
    call :UniHeaterBinding
    call :UniHeaterDataStorage      %UH_NumHeaters%
    call :UniHeaterTagList          %UH_NumHeaters%
    call :UniHeaterWinCtrl          %UH_NumHeaters%
    call :UniHeaterWinProp
    call :UniHeaterWinPlot          %UH_NumHeaters%
    call :UniHeaterCtrl             %UH_NumHeaters%
    call :UniHeaterCustomParam      %UH_NumHeaters%
    call :UniHeaterStartupScript    %UH_NumHeaters%    %UH_TileNumber%
    call :UniHeaterPulser           %UH_NumHeaters%
    call :UniHeaterSelect           %UH_NumHeaters%
rem call :UniHeaterDimDis           %UH_NumHeaters%
rem call :UniHeaterDimDic           %UH_NumHeaters%
goto :EOF

:UniHeaterDisCfg
    call :UniHeaterHeader
    call :UniHeaterBinding
    call :UniHeaterDataStorage      %UH_NumHeaters%
    call :UniHeaterTagList          %UH_NumHeaters%
    call :UniHeaterWinCtrl          %UH_NumHeaters%
    call :UniHeaterWinProp
    call :UniHeaterWinPlot          %UH_NumHeaters%
    call :UniHeaterCtrl             %UH_NumHeaters%
    call :UniHeaterCustomParam      %UH_NumHeaters%
    call :UniHeaterStartupScript    %UH_NumHeaters%    %UH_TileNumber%
    call :UniHeaterPulser           %UH_NumHeaters%
    call :UniHeaterSelect           %UH_NumHeaters%
    call :UniHeaterDimDis           %UH_NumHeaters%
rem call :UniHeaterDimDic           %UH_NumHeaters%
goto :EOF

:UniHeaterDicCfg
    call :UniHeaterHeader
rem call :UniHeaterBinding
    call :UniHeaterDataStorage      %UH_NumHeaters%
    call :UniHeaterTagList          %UH_NumHeaters%
    call :UniHeaterWinCtrl          %UH_NumHeaters%
    call :UniHeaterWinProp
    call :UniHeaterWinPlot          %UH_NumHeaters%
    call :UniHeaterCtrl             %UH_NumHeaters%
rem call :UniHeaterCustomParam      %UH_NumHeaters%
    call :UniHeaterStartupScript    %UH_NumHeaters%    %UH_TileNumber%
rem call :UniHeaterPulser           %UH_NumHeaters%
rem call :UniHeaterSelect           %UH_NumHeaters%
rem call :UniHeaterDimDis           %UH_NumHeaters%
    call :UniHeaterDimDic           %UH_NumHeaters%
goto :EOF

:UniHeaterHeader
echo ;******************************************************************************
echo ;*** ПОДСИСТЕМА УПРАВЛЕНИЯ НАГРЕВАТЕЛЯМИ UNIHEATER (Unified Heaters)         **
echo ;******************************************************************************
echo.
echo ;******************************************************************************
echo ; Описание подключения UniHeater к DAQ системе: (здесь i-номер нагревателя)
echo ; 1) Для исключения конфликтов имен устройств, кривых и тегов в DAQ-системе
echo ;    объекты UniHeater имеют имена вида "XXXX.UniHeater.PPPP", где:
echo ;    XXXX      - общий префикс, обычно связан с именем установки,
echo ;    UniHeater - имя подсистемы нагревателей (от Unified Heaters),
echo ;    PPPP      - имя объекта (окна, устройства, кривой, тега).
echo ;    Например: DEMO.INIHEATER.GATE_1
echo ; 2) Скопировать все файлы uniheater_*.BMP -^> Bitmaps, UniHeater_*.PAS -^> DaqPas,
echo ;    UniHeater_*.CFG -^> Config, UniHeater_*.CRC -^> Circuits.
echo ; 3) Отредактировать файл uniheater_.CFG - селектор сигналов.
echo ;    При помощи тегов-селекторов TNUMi,PNUMi этот модуль должен
echo ;    обеспечить запись каналов давления и температуры в кривые
echo ;    TREALi,PREALi. Используется программа UniHeater_Select.pas.
echo ;    Задаются также списки имен каналов TNAMES, PNAMES.
echo ;    Имя каждого канала не длиннее 4 символов. Список разделяется
echo ;    слешем, например, T1/T2/T3...
echo ; 4) Выходные кривые GATE_i, BLK_i надо использовать для управления
echo ;    нагревателями как сигнал включения и блокировки соответственно.
echo ; 5) При необходимости контроля по току надо обеспечить запись
echo ;    в тег ICTRL_i: 0=Ok, 1=Обрыв. По умолчанию не используется (всегда 0).
echo ; 6) При сложной логике блокировки внешняя программа может выставлять внешнюю
echo ;    блокировку EBLK_i, за которую эта программа должна целиком отвечать.
echo ; 7) Если часть каналов не используется, их при старте надо запретить
echo ;    записью 0 в тег HENABL_i. Запрещенная печь "умирает",  то есть
echo ;    не реагирует на нажатие кнопки включения и на блокировки.
echo ;    Каналы GATE_i,BLK_i можно просто не использовать.
echo ;    Каналы TNUM_i,PNUM_i надо завести на выходы UNIHEATER.SELECT, но входы
echo ;    можно не подключать.
echo ; 8) При необходимости внешнего управления (нагрев по графику)
echo ;    надо просто периодически записывать в тег TGOAL_i нужную
echo ;    целевую температуру.
echo ; 9) Переменная [^&XXXX.UniHeater.CTRL] UsesSpeakSrv = 1 включает речевые сообщения о
echo ;    блокировке и обрыве через синтезатор ^&SpeakSrv, вместо звуковых файлов.
echo ;******************************************************************************
echo ; 20000830 Alexey Kuryakin
echo ; 20131013 Alexey Kuryakin
echo ; 20140825 Alexey Kuryakin
echo ; 20211126 Alexey Kuryakin
echo ; 20211215 Alexey Kuryakin
echo ;******************************************************************************
echo.
goto :EOF

:UniHeaterDataStorage
echo ;*********************************
echo ;*** UniHeater: Declare Curves ***
echo ;*********************************
echo.
for /L %%n in (1,1,%1) do call :UniHeaterCurves %%n
goto :EOF
:UniHeaterCurves
echo [DataStorage] ; Declare Curves for Heater %1
echo %UH_PREF%.GATE_%1  = Curve 0 150 Black 15 1 ; Gate (meander) signal
echo %UH_PREF%.BLK_%1   = Curve 0 150 Black 15 1 ; Blocking (safety) signal
echo %UH_PREF%.TREAL_%1 = Curve 0 150 Black 15 1 ; Real (actual) Temperature of Heater
echo %UH_PREF%.PREAL_%1 = Curve 0 150 Black 15 1 ; Real (actual) Pressure of Heater
echo %UH_PREF%.QX_%1    = Curve 0 150 Black 15 1 ; Wanted  power [0..100%%]
echo %UH_PREF%.QMEAN_%1 = Curve 0 150 Black 15 1 ; Average power [0..100%%]
echo %UH_PREF%.TMEAN_%1 = Curve 0 150 Black 15 1 ; Average temperature T
echo %UH_PREF%.TRMSD_%1 = Curve 0 150 Black 15 1 ; Root mean square deviation of T
echo []
echo.
goto :EOF

:UniHeaterTagList
echo ;*******************************
echo ;*** UniHeater: Declare Tags ***
echo ;*******************************
echo.
call :UniHeaterCommonTags
for /L %%n in (0,1,%1) do call :UniHeaterTags %%n
goto :EOF
:UniHeaterTags
if "%~1" == "0" echo ;********** N.B. Fake Heater 0 uses to edit SETTINGS
echo [TagList] ; Declare Tags for Heater %1
echo %UH_PREF%.HNAME_%1   = string HEAT_%1 ; Name of Heater
echo %UH_PREF%.TNAME_%1   = string  ?     ; Temperature sensor name (by selector)
echo %UH_PREF%.PNAME_%1   = string  ?     ; Pressure sensor name (by selector)
echo %UH_PREF%.TNUM_%1    = integer %1     ; Temperature sensor number
echo %UH_PREF%.PNUM_%1    = integer %1     ; Pressure    sensor number
echo %UH_PREF%.MODE_%1    = integer 0     ; Program operation mode (heating/cooling)
echo %UH_PREF%.CONF_%1    = integer 5     ; Confirmation flags: Bit0:HEAT_ON, Bit1:HEAT_OFF, Bit2:LOADINI/SAVEINI
echo %UH_PREF%.HENABL_%1  = integer 1     ; Enable Heater flag
echo %UH_PREF%.TENABL_%1  = integer 1     ; Enable safety blocking by temperature T
echo %UH_PREF%.PENABL_%1  = integer 1     ; Enable safety blocking by pressure P
echo %UH_PREF%.WENABL_%1  = integer 1     ; Enable safety blocking by watchdog timer WDT
echo %UH_PREF%.PERIOD_%1  = real    3000  ; Meander cycle period in milliseconds
echo %UH_PREF%.TGOAL_%1   = real    0     ; Goal temperature
echo %UH_PREF%.TREAL_%1   = real    0     ; Real (actual) Temperature of Heater
echo %UH_PREF%.PREAL_%1   = real    0     ; Real (actual) Pressure    of Heater
echo %UH_PREF%.GATE_%1    = integer 0     ; Gate (meander) signal
echo %UH_PREF%.BLK_%1     = integer 0     ; Blocking (safety) signal
echo %UH_PREF%.BT_%1      = integer 0     ; Button Heater ON/OFF state
echo %UH_PREF%.ST_%1      = integer 0     ; Status of Heater, 0..6
echo %UH_PREF%.SB_%1      = integer 0     ; Status byte (bit mask)
echo %UH_PREF%.QX_%1      = real    0     ; Wanted power [0..100%%]
echo %UH_PREF%.Q1_%1      = real    50    ; Power on stage 1
echo %UH_PREF%.Q2_%1      = real    20    ; Power on stage 2
echo %UH_PREF%.Q3_%1      = real    0     ; Power on stage 3
echo %UH_PREF%.DT1_%1     = real    5     ; Delta (gap) for Q1/Q2
echo %UH_PREF%.DT2_%1     = real    2     ; Delta (gap) for Q3/0
echo %UH_PREF%.WDT_%1     = real    5     ; WatchDog Time in seconds
echo %UH_PREF%.TBLK_%1    = real    1600  ; Temperature level for blocking
echo %UH_PREF%.PBLK_%1    = real    1600  ; Pressure    level for blocking
echo %UH_PREF%.EBLK_%1    = integer 0     ; External blocking signal
echo %UH_PREF%.ICTRL_%1   = integer 0     ; Enable control of Current
echo %UH_PREF%.QMEAN_%1   = real    0     ; Average power [0..100%%]
echo %UH_PREF%.TMEAN_%1   = real    0     ; Average temperature T
echo %UH_PREF%.TRMSD_%1   = real    0     ; Root mean square deviation of T
echo %UH_PREF%.TSTAT_%1   = real    60    ; Time, sec to calculate statistics
echo []
echo.
goto :EOF
:UniHeaterCommonTags
echo [TagList] ; Declare Tags for UniHeater
echo %UH_PREF%.CLOCK    = string  ?     ; Host clock Date-Time
echo %UH_PREF%.SERVID   = string  ?     ; ServerId: pid@hostname
echo %UH_PREF%.SOUNDS   = integer 1     ; Enable UniHeater sound
echo []
echo.
goto :EOF



:UniHeaterWinCtrl
echo ;*********************************
echo ;*** UniHeater: Control Window ***
echo ;*********************************
echo.
echo [Windows]
echo %UH_PREF%.CTRL = Circuit_Window
echo [%UH_PREF%.CTRL]
echo Name = %UH_PREF%.CTRL
echo Circuit = ..\Circuits\%UH_PrefixFile%_uniheater_ctrl.crc
for /L %%n in (1,1,%1) do call :UniHeaterWinCtrlLinkSens %%n
goto :EOF

:UniHeaterWinCtrlLinkSens
echo ;--- Heater %1 ---
echo Link sensor %UH_PREF%.HENABL_%1 with tag   %UH_PREF%.HENABL_%1 device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.BT_%1     with tag   %UH_PREF%.ST_%1     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TNUM_%1   with tag   %UH_PREF%.TNUM_%1   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TNAME_%1  with tag   %UH_PREF%.TNAME_%1  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TGOAL_%1  with tag   %UH_PREF%.TGOAL_%1  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TREAL_%1  with curve %UH_PREF%.TREAL_%1  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.GATE_%1   with curve %UH_PREF%.GATE_%1   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.BLK_%1    with curve %UH_PREF%.BLK_%1    device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.PROP_%1   with tag   %UH_PREF%.HNAME_%1  device ^&%UH_PREF%.CTRL
goto :EOF



:UniHeaterWinProp
echo []
echo.
echo ;**********************************
echo ;*** UniHeater: Settings Window ***
echo ;**********************************
echo [Windows]
echo %UH_PREF%.SETTINGS = Circuit_Window
echo [%UH_PREF%.SETTINGS]
echo Name = %UH_PREF%.SETTINGS
echo Circuit = ..\Circuits\%UH_PrefixFile%_uniheater_settings.crc
echo Link sensor %UH_PREF%.HNAME_0   with tag %UH_PREF%.HNAME_0  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.HENABL_0  with tag %UH_PREF%.HENABL_0 device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.PBLK_0    with tag %UH_PREF%.PBLK_0   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.PENABL_0  with tag %UH_PREF%.PENABL_0 device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.PREAL_0   with tag %UH_PREF%.PREAL_0  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.PNAME_0   with tag %UH_PREF%.PNAME_0  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.PNUM_0    with tag %UH_PREF%.PNUM_0   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.WDT_0     with tag %UH_PREF%.WDT_0    device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.WENABL_0  with tag %UH_PREF%.WENABL_0 device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TBLK_0    with tag %UH_PREF%.TBLK_0   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TENABL_0  with tag %UH_PREF%.TENABL_0 device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TREAL_0   with tag %UH_PREF%.TREAL_0  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TNAME_0   with tag %UH_PREF%.TNAME_0  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TNUM_0    with tag %UH_PREF%.TNUM_0   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.Q1_0      with tag %UH_PREF%.Q1_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.Q2_0      with tag %UH_PREF%.Q2_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.Q3_0      with tag %UH_PREF%.Q3_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.DT1_0     with tag %UH_PREF%.DT1_0    device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.DT2_0     with tag %UH_PREF%.DT2_0    device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TGOAL_0   with tag %UH_PREF%.TGOAL_0  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.PERIOD_0  with tag %UH_PREF%.PERIOD_0 device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SOUNDS    with tag %UH_PREF%.SOUNDS   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SERVID    with tag %UH_PREF%.SERVID   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.CLOCK     with tag %UH_PREF%.CLOCK    device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.ACCEPT    with                        device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.CANCEL    with                        device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SAVEINI   with                        device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.LOADINI   with                        device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.MANUAL    with                        device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.CLOSE     with                        device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.MAKE.LM9  with                        device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.MODE_0    with tag %UH_PREF%.MODE_0   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.MODE_N    with tag %UH_PREF%.MODE_0   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.CONF_0_0  with tag %UH_PREF%.CONF_0   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.CONF_0_1  with tag %UH_PREF%.CONF_0   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.CONF_0_2  with tag %UH_PREF%.CONF_0   device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.QX_0      with tag %UH_PREF%.QX_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.QMEAN_0   with tag %UH_PREF%.QMEAN_0  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TMEAN_0   with tag %UH_PREF%.TMEAN_0  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TRMSD_0   with tag %UH_PREF%.TRMSD_0  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.TSTAT_0   with tag %UH_PREF%.TSTAT_0  device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SB_0_0    with tag %UH_PREF%.SB_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SB_0_1    with tag %UH_PREF%.SB_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SB_0_2    with tag %UH_PREF%.SB_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SB_0_3    with tag %UH_PREF%.SB_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SB_0_4    with tag %UH_PREF%.SB_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SB_0_5    with tag %UH_PREF%.SB_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SB_0_6    with tag %UH_PREF%.SB_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SB_0_7    with tag %UH_PREF%.SB_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SB_0_8    with tag %UH_PREF%.SB_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SB_0_9    with tag %UH_PREF%.SB_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SB_0_10   with tag %UH_PREF%.SB_0     device ^&%UH_PREF%.CTRL
echo Link sensor %UH_PREF%.SB_0_11   with tag %UH_PREF%.SB_0     device ^&%UH_PREF%.CTRL
echo []
echo.
goto :EOF

:UniHeaterWinPlot
echo ;************************************
echo ;*** UniHeater: Data Plot Windows ***
echo ;************************************
echo.
echo [Windows]
echo %UH_PREF%.T.PLOT = Curve_Window
echo [%UH_PREF%.T.PLOT]
echo AxisX = ^^R{Hours}___^^N^^CTime, 0, 0.05
echo AxisY = ^^CTemperature^^N^^L___{°C}, 0, 1000
for /l %%i in (1,1,%1) do echo CurveList = %UH_PREF%.TREAL_%%i
for /l %%i in (1,1,%1) do echo CurveList = %UH_PREF%.TMEAN_%%i
for /l %%i in (1,1,%1) do echo CurveList = %UH_PREF%.TRMSD_%%i
echo []
echo.
echo [Windows]
echo %UH_PREF%.P.PLOT = Curve_Window
echo [%UH_PREF%.P.PLOT]
echo AxisX = ^^R{Hours}___^^N^^CTime, 0, 0.05
echo AxisY = ^^CPressure^^N^^L___{Bar}, 0, 2500
for /l %%i in (1,1,%1) do echo CurveList = %UH_PREF%.PREAL_%%i
echo []
echo.
echo [Windows]
echo %UH_PREF%.Q.PLOT = Curve_Window
echo [%UH_PREF%.Q.PLOT]
echo AxisX = ^^R{Hours}___^^N^^CTime, 0, 0.05
echo AxisY = ^^CQX_-_Quantity_of_Power^^N^^L___{%%}, 0, 100
for /l %%i in (1,1,%1) do echo CurveList = %UH_PREF%.QX_%%i
for /l %%i in (1,1,%1) do echo CurveList = %UH_PREF%.QMEAN_%%i
echo []
echo.
echo [Windows]
echo %UH_PREF%.STATE.PLOT = Curve_Window
echo [%UH_PREF%.STATE.PLOT]
echo AxisX = ^^R{Hours}___^^N^^CTime, 0, 0.05
echo AxisY = ^^CState_of_GATE_and_BLK^^N^^L___{State}, 0, 1
for /l %%i in (1,1,%1) do echo CurveList = %UH_PREF%.GATE_%%i
for /l %%i in (1,1,%1) do echo CurveList = %UH_PREF%.BLK_%%i
echo []
echo.
goto :EOF


:UniHeaterCtrl
set /a "nai=%1+1"
set /a "ndo=%1+1"
set /a "nao=%nai%+%1+%1+%1"
set /a "naf=%nao%*32+32"
set /a "ndf=%ndo%*32+32"
set /a "nif=%1*4+32"
set /a "nof=%1*4+32"
echo ;***************************************
echo ;*** UniHeater: Main control program ***
echo ;***************************************
echo [DeviceList]
echo ^&%UH_PREF%.CTRL = device software program
echo [^&%UH_PREF%.CTRL]
echo Comment         = Стабилизатор температуры нагревателей UniHeater.
echo InquiryPeriod   = 1
echo DevicePolling   = 10, tpTimeCritical
echo ProgramSource   = %UH_HomeRef%\uniheater_ctrl.pas
echo UniHeaterManual = %UH_HomeRef%\uniheater.htm
echo DebugFlags      = 3
echo OpenConsole     = 2
echo StartupScript   = [^&%UH_PREF%.CTRL.StartupScript]
echo FinallyScript   = [^&%UH_PREF%.CTRL.FinallyScript]
echo StdInFifo       = %nif%
echo StdOutFifo      = %nof%
echo AnalogFifo      = %naf%
echo DigitalFifo     = %ndf%
echo EncryptMethod   = %UH_EncryptMethod%
echo TrustedUsers    = [TrustedUsers]
echo DIM_GuiClickTag = %UH_PREF%.DIMGUICLICK
echo AnalogInputs    = %nai%  ; Temperatures
for /L %%i in (1,1,%1) do call :UniHeaterCtrlAI_TREAL %%i
echo DigitalInputs   = %nai%  ; Pressures
for /L %%i in (1,1,%1) do call :UniHeaterCtrlDI_PREAL %%i
echo AnalogOutputs   = %nao%  ; Wanted Power and Average power
for /L %%i in (1,1,%1) do call :UniHeaterCtrlAO_QX %%i
for /L %%i in (1,1,%1) do call :UniHeaterCtrlAO_QMEAN %%i %1
for /L %%i in (1,1,%1) do call :UniHeaterCtrlAO_TMEAN %%i %1
for /L %%i in (1,1,%1) do call :UniHeaterCtrlAO_TRMSD %%i %1
echo DigitalOutputs  = %ndo%  ; Blocking signals
for /L %%i in (1,1,%1) do call :UniHeaterCtrlDO_BLK %%i
echo ;------ Save custom parameters to INI file
echo CustomIniTagList  = [%UH_PREF%.CustomParameters.TagList]
echo CustomIniFileRef  = ..\Data\%UH_FILE%.ini
echo CustomIniSection  = [%UH_PREF%.CustomParameters]
echo CustomIniBackups  = ..\Data\Custom
echo CustomIniAutoLoad = 1
echo CustomIniAutoSave = 0
echo []
echo.
echo [ConfigFileList]
echo ConfigFile = ..\Data\%UH_FILE%.ini
echo []
echo.
goto :EOF
:UniHeaterCtrlAI_TREAL
echo Link AnalogInput %1 with curve %UH_PREF%.TREAL_%1
goto :EOF
:UniHeaterCtrlDI_PREAL
echo Link DigitalInput %1 with curve %UH_PREF%.PREAL_%1  bit 0
goto :EOF
:UniHeaterCtrlAO_QX
echo Link AnalogOutput %1 with curve %UH_PREF%.QX_%1 history %UH_HistoryLen%
goto :EOF
:UniHeaterCtrlAO_QMEAN
set /a ao=%1+%2
echo Link AnalogOutput %ao% with curve %UH_PREF%.QMEAN_%1 history %UH_HistoryLen%
goto :EOF
:UniHeaterCtrlAO_TMEAN
set /a ao=%1+%2+%2
echo Link AnalogOutput %ao% with curve %UH_PREF%.TMEAN_%1 history %UH_HistoryLen%
goto :EOF
:UniHeaterCtrlAO_TRMSD
set /a ao=%1+%2+%2+%2
echo Link AnalogOutput %ao% with curve %UH_PREF%.TRMSD_%1 history %UH_HistoryLen%
goto :EOF
:UniHeaterCtrlDO_BLK
echo Link DigitalOutput %1 with curve %UH_PREF%.BLK_%1 history %UH_HistoryLen%
goto :EOF



:UniHeaterCustomParam
echo [%UH_PREF%.CustomParameters.TagList]
for /L %%n in (1,1,%1) do call :UniHeaterCustomParams %%n
echo []
echo.
goto :EOF
:UniHeaterCustomParams
echo ;--- Heater %1, custom parameters ---
echo TagList = %UH_PREF%.HNAME_%1
echo TagList = %UH_PREF%.MODE_%1
echo TagList = %UH_PREF%.CONF_%1
echo TagList = %UH_PREF%.HENABL_%1
echo TagList = %UH_PREF%.BT_%1
echo TagList = %UH_PREF%.TGOAL_%1
echo TagList = %UH_PREF%.TNUM_%1
echo TagList = %UH_PREF%.TENABL_%1
echo TagList = %UH_PREF%.TBLK_%1
echo TagList = %UH_PREF%.PENABL_%1
echo TagList = %UH_PREF%.PNUM_%1
echo TagList = %UH_PREF%.WENABL_%1
echo TagList = %UH_PREF%.WDT_%1
echo TagList = %UH_PREF%.DT1_%1
echo TagList = %UH_PREF%.DT2_%1
echo TagList = %UH_PREF%.PBLK_%1
echo TagList = %UH_PREF%.PERIOD_%1
echo TagList = %UH_PREF%.Q1_%1
echo TagList = %UH_PREF%.Q2_%1
echo TagList = %UH_PREF%.Q3_%1
echo TagList = %UH_PREF%.TSTAT_%1
goto :EOF



:UniHeaterStartupScript
set /a "ncols=%~2"                      & rem Number of columns
set /a "nrows= ( %1 + %2 - 1 ) / %2"    & rem Number of rows
set /a "cx=%ncols% * 125 + 15"          & rem Window size on X axis
set /a "cy=%nrows% * 119 + 57"          & rem Window size on Y axis
echo [^&%UH_PREF%.CTRL.StartupScript]
echo @WinShow   %UH_PREF%.CTRL
echo @WinDraw   %UH_PREF%.CTRL^|Top=0^|Left=0^|Width=!cx!^|Height=!cy!
echo @WinDraw   %UH_PREF%.CTRL^|Options=-Min,-Max,+Close,+Top,+Left,-Width,-Height,-HScroll,-VScroll,+StatusBar
echo @WinSelect %UH_PREF%.CTRL
echo @WinHide   %UH_PREF%.CTRL
echo @WinShow   %UH_PREF%.SETTINGS
echo @WinDraw   %UH_PREF%.SETTINGS^|Top=170^|Left=167^|Width=950^|Height=420
echo @WinDraw   %UH_PREF%.SETTINGS^|Options=-Min,-Max,-Close,+Top,+Left,-Width,-Height,-HScroll,-VScroll,+StatusBar
echo @WinSelect %UH_PREF%.SETTINGS
echo @WinHide   %UH_PREF%.SETTINGS
echo @WinShow   %UH_PREF%.T.PLOT
echo @WinDraw   %UH_PREF%.T.PLOT^|Top=317^|Left=167^|Width=1070^|Height=483
echo @WinDraw   %UH_PREF%.T.PLOT^|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar
echo @WinHide   %UH_PREF%.T.PLOT
echo @WinShow   %UH_PREF%.P.PLOT
echo @WinDraw   %UH_PREF%.P.PLOT^|Top=317^|Left=167^|Width=1070^|Height=483
echo @WinDraw   %UH_PREF%.P.PLOT^|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar
echo @WinHide   %UH_PREF%.P.PLOT
echo @WinShow   %UH_PREF%.Q.PLOT
echo @WinDraw   %UH_PREF%.Q.PLOT^|Top=317^|Left=167^|Width=1070^|Height=483
echo @WinDraw   %UH_PREF%.Q.PLOT^|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar
echo @WinHide   %UH_PREF%.Q.PLOT
echo @WinShow   %UH_PREF%.STATE.PLOT
echo @WinDraw   %UH_PREF%.STATE.PLOT^|Top=317^|Left=167^|Width=1070^|Height=483
echo @WinDraw   %UH_PREF%.STATE.PLOT^|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar
echo @WinHide   %UH_PREF%.STATE.PLOT
echo []
echo.
echo [^&%UH_PREF%.CTRL.FinallyScript]
echo @WinHide %UH_PREF%.CTRL
echo @WinHide %UH_PREF%.SETTINGS
echo @WinHide %UH_PREF%.T.PLOT
echo @WinHide %UH_PREF%.P.PLOT
echo @WinHide %UH_PREF%.Q.PLOT
echo @WinHide %UH_PREF%.STATE.PLOT
echo []
echo.
echo ;*******************************************************
echo ;*** UniHeater: CRON jobs to draw UniHeater windows  ***
echo ;*******************************************************
echo [^&CronSrv.StartupScript]
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.CTRL.HOME 0 0 0
echo @cron.job %UH_PREF%.CTRL.HOME @WinShow %UH_PREF%.CTRL
echo @cron.job %UH_PREF%.CTRL.HOME @WinDraw %UH_PREF%.CTRL^|Top=0^|Left=0^|Width=!cx!^|Height=!cy!
echo @cron.job %UH_PREF%.CTRL.HOME @WinDraw %UH_PREF%.CTRL^|Options=-Min,-Max,+Close,-Top,-Left,-Width,-Height,-HScroll,-VScroll,+StatusBar
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.CTRL.SHOW 0 0 0
echo @cron.job %UH_PREF%.CTRL.SHOW @WinShow %UH_PREF%.CTRL
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.CTRL.HIDE 0 0 0
echo @cron.job %UH_PREF%.CTRL.HIDE @WinHide %UH_PREF%.CTRL
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.CTRL.SELECT 0 0 0
echo @cron.job %UH_PREF%.CTRL.SELECT @WinSelect %UH_PREF%.CTRL
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.SETTINGS.HOME 0 0 0
echo @cron.job %UH_PREF%.SETTINGS.HOME @WinShow %UH_PREF%.SETTINGS
echo @cron.job %UH_PREF%.SETTINGS.HOME @WinDraw %UH_PREF%.SETTINGS^|Top=170^|Left=167^|Width=950^|Height=420
echo @cron.job %UH_PREF%.SETTINGS.HOME @WinDraw %UH_PREF%.SETTINGS^|Options=-Min,-Max,-Close,-Top,-Left,-Width,-Height,-HScroll,-VScroll,+StatusBar
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.SETTINGS.SHOW 0 0 0
echo @cron.job %UH_PREF%.SETTINGS.SHOW @WinShow %UH_PREF%.SETTINGS
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.SETTINGS.HIDE 0 0 0
echo @cron.job %UH_PREF%.SETTINGS.HIDE @WinHide %UH_PREF%.SETTINGS
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.SETTINGS.SELECT 0 0 0
echo @cron.job %UH_PREF%.SETTINGS.SELECT @WinSelect %UH_PREF%.SETTINGS
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.T.PLOT.HOME 0 0 0
echo @cron.job %UH_PREF%.T.PLOT.HOME @WinShow %UH_PREF%.T.PLOT
echo @cron.job %UH_PREF%.T.PLOT.HOME @WinDraw %UH_PREF%.T.PLOT^|Top=317^|Left=167^|Width=1070^|Height=483
echo @cron.job %UH_PREF%.T.PLOT.HOME @WinDraw %UH_PREF%.T.PLOT^|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar
echo @cron.tab %UH_PREF%.T.PLOT.SHOW 0 0 0
echo @cron.job %UH_PREF%.T.PLOT.SHOW @WinShow %UH_PREF%.T.PLOT
echo @cron.tab %UH_PREF%.T.PLOT.HIDE 0 0 0
echo @cron.job %UH_PREF%.T.PLOT.HIDE @WinHide %UH_PREF%.T.PLOT
echo @cron.tab %UH_PREF%.T.PLOT.SELECT 0 0 0
echo @cron.job %UH_PREF%.T.PLOT.SELECT @WinSelect %UH_PREF%.T.PLOT
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.P.PLOT.HOME 0 0 0
echo @cron.job %UH_PREF%.P.PLOT.HOME @WinShow %UH_PREF%.P.PLOT
echo @cron.job %UH_PREF%.P.PLOT.HOME @WinDraw %UH_PREF%.P.PLOT^|Top=317^|Left=167^|Width=1070^|Height=483
echo @cron.job %UH_PREF%.P.PLOT.HOME @WinDraw %UH_PREF%.P.PLOT^|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar
echo @cron.tab %UH_PREF%.P.PLOT.SHOW 0 0 0
echo @cron.job %UH_PREF%.P.PLOT.SHOW @WinShow %UH_PREF%.P.PLOT
echo @cron.tab %UH_PREF%.P.PLOT.HIDE 0 0 0
echo @cron.job %UH_PREF%.P.PLOT.HIDE @WinHide %UH_PREF%.P.PLOT
echo @cron.tab %UH_PREF%.P.PLOT.SELECT 0 0 0
echo @cron.job %UH_PREF%.P.PLOT.SELECT @WinSelect %UH_PREF%.P.PLOT
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.Q.PLOT.HOME 0 0 0
echo @cron.job %UH_PREF%.Q.PLOT.HOME @WinShow %UH_PREF%.Q.PLOT
echo @cron.job %UH_PREF%.Q.PLOT.HOME @WinDraw %UH_PREF%.Q.PLOT^|Top=317^|Left=167^|Width=1070^|Height=483
echo @cron.job %UH_PREF%.Q.PLOT.HOME @WinDraw %UH_PREF%.Q.PLOT^|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar
echo @cron.tab %UH_PREF%.Q.PLOT.SHOW 0 0 0
echo @cron.job %UH_PREF%.Q.PLOT.SHOW @WinShow %UH_PREF%.Q.PLOT
echo @cron.tab %UH_PREF%.Q.PLOT.HIDE 0 0 0
echo @cron.job %UH_PREF%.Q.PLOT.HIDE @WinHide %UH_PREF%.Q.PLOT
echo @cron.tab %UH_PREF%.Q.PLOT.SELECT 0 0 0
echo @cron.job %UH_PREF%.Q.PLOT.SELECT @WinSelect %UH_PREF%.Q.PLOT
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.STATE.PLOT.HOME 0 0 0
echo @cron.job %UH_PREF%.STATE.PLOT.HOME @WinShow %UH_PREF%.STATE.PLOT
echo @cron.job %UH_PREF%.STATE.PLOT.HOME @WinDraw %UH_PREF%.STATE.PLOT^|Top=317^|Left=167^|Width=1070^|Height=483
echo @cron.job %UH_PREF%.STATE.PLOT.HOME @WinDraw %UH_PREF%.STATE.PLOT^|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar
echo @cron.tab %UH_PREF%.STATE.PLOT.SHOW 0 0 0
echo @cron.job %UH_PREF%.STATE.PLOT.SHOW @WinShow %UH_PREF%.STATE.PLOT
echo @cron.tab %UH_PREF%.STATE.PLOT.HIDE 0 0 0
echo @cron.job %UH_PREF%.STATE.PLOT.HIDE @WinHide %UH_PREF%.STATE.PLOT
echo @cron.tab %UH_PREF%.STATE.PLOT.SELECT 0 0 0
echo @cron.job %UH_PREF%.STATE.PLOT.SELECT @WinSelect %UH_PREF%.STATE.PLOT
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.ALL.HOME 0 0 0
echo @cron.job %UH_PREF%.ALL.HOME @cron.run %UH_PREF%.CTRL.HOME
echo @cron.job %UH_PREF%.ALL.HOME @cron.run %UH_PREF%.SETTINGS.HOME
echo @cron.job %UH_PREF%.ALL.HOME @cron.run %UH_PREF%.T.PLOT.HOME
echo @cron.job %UH_PREF%.ALL.HOME @cron.run %UH_PREF%.P.PLOT.HOME
echo @cron.job %UH_PREF%.ALL.HOME @cron.run %UH_PREF%.Q.PLOT.HOME
echo @cron.job %UH_PREF%.ALL.HOME @cron.run %UH_PREF%.STATE.PLOT.HOME
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.ALL.SHOW 0 0 0
echo @cron.job %UH_PREF%.ALL.SHOW @cron.run %UH_PREF%.CTRL.SHOW
echo @cron.job %UH_PREF%.ALL.SHOW @cron.run %UH_PREF%.SETTINGS.SHOW
echo @cron.job %UH_PREF%.ALL.SHOW @cron.run %UH_PREF%.T.PLOT.SHOW
echo @cron.job %UH_PREF%.ALL.SHOW @cron.run %UH_PREF%.P.PLOT.SHOW
echo @cron.job %UH_PREF%.ALL.SHOW @cron.run %UH_PREF%.Q.PLOT.SHOW
echo @cron.job %UH_PREF%.ALL.SHOW @cron.run %UH_PREF%.STATE.PLOT.SHOW
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.ALL.HIDE 0 0 0
echo @cron.job %UH_PREF%.ALL.HIDE @cron.run %UH_PREF%.CTRL.HIDE
echo @cron.job %UH_PREF%.ALL.HIDE @cron.run %UH_PREF%.SETTINGS.HIDE
echo @cron.job %UH_PREF%.ALL.HIDE @cron.run %UH_PREF%.T.PLOT.HIDE
echo @cron.job %UH_PREF%.ALL.HIDE @cron.run %UH_PREF%.P.PLOT.HIDE
echo @cron.job %UH_PREF%.ALL.HIDE @cron.run %UH_PREF%.Q.PLOT.HIDE
echo @cron.job %UH_PREF%.ALL.HIDE @cron.run %UH_PREF%.STATE.PLOT.HIDE
echo ;---------------------------------------------------------------------
echo @cron.tab %UH_PREF%.ALL.SELECT 0 0 0
echo @cron.job %UH_PREF%.ALL.SELECT @cron.run %UH_PREF%.CTRL.SELECT
echo @cron.job %UH_PREF%.ALL.SELECT @cron.run %UH_PREF%.SETTINGS.SELECT
echo @cron.job %UH_PREF%.ALL.SELECT @cron.run %UH_PREF%.T.PLOT.SELECT
echo @cron.job %UH_PREF%.ALL.SELECT @cron.run %UH_PREF%.P.PLOT.SELECT
echo @cron.job %UH_PREF%.ALL.SELECT @cron.run %UH_PREF%.Q.PLOT.SELECT
echo @cron.job %UH_PREF%.ALL.SELECT @cron.run %UH_PREF%.STATE.PLOT.SELECT
echo ;---------------------------------------------------------------------
echo @async @cron.run %UH_PREF%.ALL.HOME
echo @async @cron.run %UH_PREF%.ALL.HIDE
echo ;---------------------------------------------------------------------
echo []
echo.
goto :EOF



:UniHeaterPulser
set /a "nai=%1+1"
set /a "ndo=%1+1"
set /a "ndf=%ndo%*32+32"
echo ;*******************************************************
echo ;*** Генератор скважности для нагревателей UniHeater ***
echo ;*******************************************************
echo [DeviceList]
echo ^&%UH_PREF%.PULSER = device software program
echo [^&%UH_PREF%.PULSER]
echo Comment        = Генератор импульсов скважности для нагревателей UniHeater.
echo InquiryPeriod  = 0
echo DevicePolling  = 4, tpTimeCritical
echo ProgramSource  = %UH_HomeRef%\uniheater_pulser.pas
echo DigitalFifo    = %ndf%
echo DebugFlags     = 3
echo OpenConsole    = 2
echo AnalogInputs   = %nai% ; Duty factor to control Heaters
for /L %%n in (1,1,%1) do call :UniHeaterPulserAI %%n
echo DigitalOutputs = %ndo% ; Heaters ON/OFF signals
for /L %%n in (1,1,%1) do call :UniHeaterPulserDO %%n
echo []
echo.
goto :EOF
:UniHeaterPulserAI
echo Link AnalogInput   %1  with curve %UH_PREF%.QX_%1
goto :EOF
:UniHeaterPulserDO
echo Link DigitalOutput %1  with curve %UH_PREF%.GATE_%1  history %UH_HistoryLen%
goto :EOF



:UniHeaterSelect
set /a "nao=%1+1"
set /a "naf=%nao%*32+32"
echo ;******************************************************************
echo ;*** Селектор каналов температуры для подсистемы UniHeater.     ***
echo ;*** Задаются кривые AnalogInputs и TNAMES - список имен для    ***
echo ;*** датчиков T, без пробелов, разделенных слэшем: T1/T2/...    ***
echo ;******************************************************************
echo [DeviceList]
echo ^&%UH_PREF%.SELECT_T = device software program
echo [^&%UH_PREF%.SELECT_T]
echo Comment        = Селектор датчиков температуры для UniHeater.
echo InquiryPeriod  = 1
echo DevicePolling  = 10, tpTimeCritical
echo ProgramSource  = %UH_HomeRef%\uniheater_select.pas
echo SelectorPrefix = %UH_PREF%.TNUM_
echo AnalogFifo     = %naf%
echo DebugFlags     = 3
echo OpenConsole    = 2
echo AnalogOutputs  = %nao%
for /L %%n in (1,1,%1) do call :UniHeaterSelectTAO %%n
echo []
echo.
echo ;******************************************************************
echo ;*** Селектор каналов давлений для подсистемы UniHeater.        ***
echo ;*** Задаются кривые AnalogInputs и PNAMES - список имен для    ***
echo ;*** датчиков P, без пробелов, разделенных слэшем: P1/P2/...    ***
echo ;******************************************************************
echo [DeviceList]
echo ^&%UH_PREF%.SELECT_P = device software program
echo [^&%UH_PREF%.SELECT_P]
echo Comment        = Селектор датчиков давления для UniHeater.
echo InquiryPeriod  = 1
echo DevicePolling  = 10, tpTimeCritical
echo ProgramSource  = %UH_HomeRef%\uniheater_select.pas
echo SelectorPrefix = %UH_PREF%.PNUM_
echo AnalogFifo     = %naf%
echo DebugFlags     = 3
echo OpenConsole    = 2
echo AnalogOutputs  = %nao%
for /L %%n in (1,1,%1) do call :UniHeaterSelectPAO %%n
echo []
echo.
goto :EOF
:UniHeaterSelectTAO
echo Link AnalogOutput  %1  with curve %UH_PREF%.TREAL_%1  history %UH_HistoryLen%
goto :EOF
:UniHeaterSelectPAO
echo Link AnalogOutput  %1  with curve %UH_PREF%.PREAL_%1  history %UH_HistoryLen%
goto :EOF



:UniHeaterBinding
echo ;******************************************************************
echo ;*** Подключение селекторов каналов датчиков T,P для UniHeater. ***
echo ;******************************************************************
echo [ConfigFileList] ; Подключение конфигурации селекторов UniHeater
echo ConfigFile = %UH_PrefixFile%_uniheater_bind.cfg
echo []
echo.
goto :EOF

:UniHeaterSelector
set /a "ntai=%UH_NumberTSen%+1"
set /a "npai=%UH_NumberPSen%+1"
echo ;******************************************************************
echo ;*** Селектор каналов датчиков T,P для подсистемы UniHeater.    ***
echo ;*** Этот файл настраивается индивидуально для каждой системы,  ***
echo ;*** т.к. кривые источников данных (датчиков T,P) меняются.     ***
echo ;******************************************************************
echo.
echo ;******************************************************************
echo ;*** Селектор каналов температуры для подсистемы UniHeater.     ***
echo ;*** Задаются кривые AnalogInputs и TNAMES - список имен для    ***
echo ;*** датчиков T, без пробелов, разделенных слэшем: T1/T2/...    ***
echo ;******************************************************************
echo [^&%UH_PREF%.SELECT_T]
echo UsesSmoothing  = 1
echo AnalogInputs   = %ntai%
set "tnames=!UH_NickTSen!1"
for /l %%n in (2,1,%UH_NumberTSen%) do set "tnames=!tnames!/!UH_NickTSen!%%n"
for /l %%n in (1,1,%UH_NumberTSen%) do call :UniHeaterComSelectTAI %%n
echo []
echo [TagList]
echo %UH_PREF%.TNAMES = string %tnames%
echo []
echo.
echo ;******************************************************************
echo ;*** Селектор каналов давлений для подсистемы UniHeater.        ***
echo ;*** Задаются кривые AnalogInputs и PNAMES - список имен для    ***
echo ;*** датчиков P, без пробелов, разделенных слэшем: P1/P2/...    ***
echo ;******************************************************************
echo [^&%UH_PREF%.SELECT_P]
echo UsesSmoothing  = 1
echo AnalogInputs   = %npai%
set "pnames=!UH_NickPSen!1"
for /l %%n in (2,1,%UH_NumberPSen%) do set "pnames=!pnames!/!UH_NickPSen!%%n"
for /l %%n in (1,1,%UH_NumberPSen%) do call :UniHeaterComSelectPAI %%n
echo []
echo [TagList]
echo %UH_PREF%.PNAMES = string %pnames%
echo []
goto :EOF
:UniHeaterComSelectTAI
echo Link AnalogInput %1 with curve !UH_PrefixTSen!%1 smoothing %UH_SmoothingWw% %UH_SmoothingPw% %UH_SmoothingK1% %UH_SmoothingK2%
goto :EOF
:UniHeaterComSelectPAI
echo Link AnalogInput %1 with curve !UH_PrefixPSen!%1 smoothing %UH_SmoothingWw% %UH_SmoothingPw% %UH_SmoothingK2% %UH_SmoothingK2%
goto :EOF




:UniHeaterDimGuiClick
set "devmsg=devPostMsg"
echo [TagList]
echo %UH_PREF%.DIMGUICLICK string ? ; Uses to Handle DIM GUI Clicks
echo []
goto :EOF

:UniHeaterDimDis
call :UniHeaterDimGuiClick
unix dimcfg ^
  -n section "[&%UH_PREF%.CTRL]" ^
  -n print    DimServerMode = 1 ^
  -n end ^
  -n dis_cmnd   %UH_PATH%/DIMGUICLICK ^
  -n tag        %UH_PREF%.DIMGUICLICK ^
  -n %devmsg% "&%UH_PREF%.CTRL @DIMGUICLICK=%%**" ^
  -n end ^
  -n dic_cmnd   %UH_PATH%/DIMGUICLICK ^
  -n tag        %UH_PREF%.DIMGUICLICK ^
  -n end ^
  -n dis_info   %UH_PATH%/CLOCK ^
  -n tag        %UH_PREF%.CLOCK ^
  -n end ^
  -n dis_info   %UH_PATH%/SERVID ^
  -n tag        %UH_PREF%.SERVID ^
  -n end ^
  -n dis_info   %UH_PATH%/TNAMES ^
  -n tag        %UH_PREF%.TNAMES ^
  -n end ^
  -n dis_info   %UH_PATH%/PNAMES ^
  -n tag        %UH_PREF%.PNAMES ^
  -n end ^
  -n dis_info   %UH_PATH%/SOUNDS ^
  -n tag        %UH_PREF%.SOUNDS ^
  -n end ^
  -n
for /l %%i in (1,1,%UH_NumHeaters%) do call :UniHeaterDimDisLoop %%i
goto :EOF
:UniHeaterDimDisLoop
unix dimcfg ^
  -n dis_info   %UH_PATH%/HENABL+HNAME_%1 ^
  -n tag        %UH_PREF%.HENABL_%1 ^
  -n tag        %UH_PREF%.HNAME_%1 ^
  -n end ^
  -n dis_info   %UH_PATH%/TENABL+TBLK+TNUM+TNAME_%1 ^
  -n tag        %UH_PREF%.TENABL_%1 ^
  -n tag        %UH_PREF%.TBLK_%1 ^
  -n tag        %UH_PREF%.TNUM_%1 ^
  -n tag        %UH_PREF%.TNAME_%1 ^
  -n end ^
  -n dis_info   %UH_PATH%/PENABL+PBLK+PNUM+PNAME_%1 ^
  -n tag        %UH_PREF%.PENABL_%1 ^
  -n tag        %UH_PREF%.PBLK_%1 ^
  -n tag        %UH_PREF%.PNUM_%1 ^
  -n tag        %UH_PREF%.PNAME_%1 ^
  -n end ^
  -n dis_info   %UH_PATH%/WENABL+WDT_%1 ^
  -n tag        %UH_PREF%.WENABL_%1 ^
  -n tag        %UH_PREF%.WDT_%1 ^
  -n end ^
  -n dis_info   %UH_PATH%/TGOAL+MODE+PERIOD+Q1+Q2+Q3+DT1+DT2_%1 ^
  -n tag        %UH_PREF%.TGOAL_%1 ^
  -n tag        %UH_PREF%.MODE_%1 ^
  -n tag        %UH_PREF%.PERIOD_%1 ^
  -n tag        %UH_PREF%.Q1_%1 ^
  -n tag        %UH_PREF%.Q2_%1 ^
  -n tag        %UH_PREF%.Q3_%1 ^
  -n tag        %UH_PREF%.DT1_%1 ^
  -n tag        %UH_PREF%.DT2_%1 ^
  -n end ^
  -n dis_info   %UH_PATH%/CONF_%1 ^
  -n tag        %UH_PREF%.CONF_%1 ^
  -n end ^
  -n dis_info   %UH_PATH%/TREAL_%1 ^
  -n tag        %UH_PREF%.TREAL_%1 ^
  -n end ^
  -n dis_info   %UH_PATH%/PREAL_%1 ^
  -n tag        %UH_PREF%.PREAL_%1 ^
  -n end ^
  -n dis_info   %UH_PATH%/BT+ST+SB+BLK+QX_%1 ^
  -n tag        %UH_PREF%.BT_%1 ^
  -n tag        %UH_PREF%.ST_%1 ^
  -n tag        %UH_PREF%.SB_%1 ^
  -n tag        %UH_PREF%.BLK_%1 ^
  -n tag        %UH_PREF%.QX_%1 ^
  -n end ^
  -n dis_info   %UH_PATH%/GATE_%1 ^
  -n tag        %UH_PREF%.GATE_%1 ^
  -n end ^
  -n dis_info   %UH_PATH%/EBLK+ICTRL_%1 ^
  -n tag        %UH_PREF%.EBLK_%1 ^
  -n tag        %UH_PREF%.ICTRL_%1 ^
  -n end ^
  -n dis_info   %UH_PATH%/TSTAT+QMEAN+TMEAN+TRMSD_%1 ^
  -n tag        %UH_PREF%.TSTAT_%1 ^
  -n tag        %UH_PREF%.QMEAN_%1 ^
  -n tag        %UH_PREF%.TMEAN_%1 ^
  -n tag        %UH_PREF%.TRMSD_%1 ^
  -n end ^
  -n
goto :EOF

:UniHeaterDimDic
call :UniHeaterDimGuiClick
unix dimcfg ^
  -n section "[&%UH_PREF%.CTRL]" ^
  -n print    DimClientMode = 1 ^
  -n end ^
  -n dic_cmnd   %UH_PATH%/DIMGUICLICK ^
  -n tag        %UH_PREF%.DIMGUICLICK ^
  -n end ^
  -n dic_info   %UH_PATH%/CLOCK ^
  -n tag        %UH_PREF%.CLOCK ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.CLOCK" ^
  -n end ^
  -n dic_info   %UH_PATH%/SERVID ^
  -n tag        %UH_PREF%.SERVID ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.SERVID" ^
  -n end ^
  -n dic_info   %UH_PATH%/TNAMES ^
  -n tag        %UH_PREF%.TNAMES ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.TNAMES" ^
  -n end ^
  -n dic_info   %UH_PATH%/PNAMES ^
  -n tag        %UH_PREF%.PNAMES ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.PNAMES" ^
  -n end ^
  -n dic_info   %UH_PATH%/SOUNDS ^
  -n tag        %UH_PREF%.SOUNDS ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.SOUNDS" ^
  -n end ^
  -n
for /l %%i in (1,1,%UH_NumHeaters%) do call :UniHeaterDimDicLoop %%i
call :UniHeaterDimDicRecv
goto :EOF
:UniHeaterDimDicLoop
unix dimcfg ^
  -n dic_info   %UH_PATH%/HENABL+HNAME_%1 ^
  -n tag        %UH_PREF%.HENABL_%1 ^
  -n tag        %UH_PREF%.HNAME_%1 ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.HENABL_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.HNAME_%1" ^
  -n end ^
  -n dic_info   %UH_PATH%/TENABL+TBLK+TNUM+TNAME_%1 ^
  -n tag        %UH_PREF%.TENABL_%1 ^
  -n tag        %UH_PREF%.TBLK_%1 ^
  -n tag        %UH_PREF%.TNUM_%1 ^
  -n tag        %UH_PREF%.TNAME_%1 ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.TENABL_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.TBLK_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.TNUM_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.TNAME_%1" ^
  -n end ^
  -n dic_info   %UH_PATH%/PENABL+PBLK+PNUM+PNAME_%1 ^
  -n tag        %UH_PREF%.PENABL_%1 ^
  -n tag        %UH_PREF%.PBLK_%1 ^
  -n tag        %UH_PREF%.PNUM_%1 ^
  -n tag        %UH_PREF%.PNAME_%1 ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.PENABL_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.PBLK_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.PNUM_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.PNAME_%1" ^
  -n end ^
  -n dic_info   %UH_PATH%/WENABL+WDT_%1 ^
  -n tag        %UH_PREF%.WENABL_%1 ^
  -n tag        %UH_PREF%.WDT_%1 ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.WENABL_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.WDT_%1" ^
  -n end ^
  -n dic_info   %UH_PATH%/TGOAL+MODE+PERIOD+Q1+Q2+Q3+DT1+DT2_%1 ^
  -n tag        %UH_PREF%.TGOAL_%1 ^
  -n tag        %UH_PREF%.MODE_%1 ^
  -n tag        %UH_PREF%.PERIOD_%1 ^
  -n tag        %UH_PREF%.Q1_%1 ^
  -n tag        %UH_PREF%.Q2_%1 ^
  -n tag        %UH_PREF%.Q3_%1 ^
  -n tag        %UH_PREF%.DT1_%1 ^
  -n tag        %UH_PREF%.DT2_%1 ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.TGOAL_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.MODE_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.PERIOD_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.Q1_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.Q2_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.Q3_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.DT1_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.DT2_%1" ^
  -n end ^
  -n dic_info   %UH_PATH%/CONF_%1 ^
  -n tag        %UH_PREF%.CONF_%1 ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.CONF_%1" ^
  -n end ^
  -n dic_info   %UH_PATH%/TREAL_%1 ^
  -n tag        %UH_PREF%.TREAL_%1 ^
  -n %devmsg% "&%UH_PREF%.RECV @DimTagUpdate=%UH_PREF%.TREAL_%1" ^
  -n end ^
  -n dic_info   %UH_PATH%/PREAL_%1 ^
  -n tag        %UH_PREF%.PREAL_%1 ^
  -n %devmsg% "&%UH_PREF%.RECV @DimTagUpdate=%UH_PREF%.PREAL_%1" ^
  -n end ^
  -n dic_info   %UH_PATH%/BT+ST+SB+BLK+QX_%1 ^
  -n tag        %UH_PREF%.BT_%1 ^
  -n tag        %UH_PREF%.ST_%1 ^
  -n tag        %UH_PREF%.SB_%1 ^
  -n tag        %UH_PREF%.BLK_%1 ^
  -n tag        %UH_PREF%.QX_%1 ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.BT_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.ST_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.SB_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.BLK_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.QX_%1" ^
  -n end ^
  -n dic_info   %UH_PATH%/GATE_%1 ^
  -n tag        %UH_PREF%.GATE_%1 ^
  -n %devmsg% "&%UH_PREF%.RECV @DimTagUpdate=%UH_PREF%.GATE_%1" ^
  -n end ^
  -n dic_info   %UH_PATH%/EBLK+ICTRL_%1 ^
  -n tag        %UH_PREF%.EBLK_%1 ^
  -n tag        %UH_PREF%.ICTRL_%1 ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.EBLK_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.ICTRL_%1" ^
  -n end ^
  -n dic_info   %UH_PATH%/TSTAT+QMEAN+TMEAN+TRMSD_%1 ^
  -n tag        %UH_PREF%.TSTAT_%1 ^
  -n tag        %UH_PREF%.QMEAN_%1 ^
  -n tag        %UH_PREF%.TMEAN_%1 ^
  -n tag        %UH_PREF%.TRMSD_%1 ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.TSTAT_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.QMEAN_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.TMEAN_%1" ^
  -n %devmsg% "&%UH_PREF%.CTRL @DimTagUpdate=%UH_PREF%.TRMSD_%1" ^
  -n end ^
  -n
goto :EOF

:UniHeaterDimDicRecv
set /a "nao=%UH_NumHeaters%*3"
set /a "naf=%nao%*32+32"
echo [TagList]
echo %UH_PREF%.TNAMES = string %tnames%
echo %UH_PREF%.PNAMES = string %pnames%
echo []
echo.
echo ;******************************
echo ; Receiver for %UH_PREF%
echo ;******************************
echo [DeviceList]
echo ^&%UH_PREF%.RECV = device software program
echo [^&%UH_PREF%.RECV]
echo Comment         = DIM RECEIVER FOR %UH_PREF%.
echo InquiryPeriod   = 1
echo DevicePolling   = 10, tpNormal
echo ProgramSource   = ~~\Resource\DaqSite\DimServer\_DIMRCV.PAS
echo AnalogFifo      = %naf%
echo DebugFlags      = 3
echo OpenConsole     = 2
echo EncryptMethod   = %UH_EncryptMethod%
echo TrustedUsers    = [TrustedUsers]
echo DIM_GuiClickTag = %UH_PREF%.DIMGUICLICK
echo AnalogOutputs   = %nao%
for /l %%i in (1,1,%UH_NumHeaters%) do call :LinkRecvAO %%i
echo []
echo.
goto :EOF
:LinkRecvAO
set /a "i1=(%1-1)*3+0"
set /a "i2=(%1-1)*3+1"
set /a "i3=(%1-1)*3+2"
echo Link AnalogOutput %i1% with curve %UH_PREF%.TREAL_%1 history %UH_HistoryLen%
echo Link AnalogOutput %i2% with curve %UH_PREF%.PREAL_%1 history %UH_HistoryLen%
echo Link AnalogOutput %i3% with curve %UH_PREF%.GATE_%1 history %UH_HistoryLen%
goto :EOF



::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: Common library routines                                                    ::
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

:Define
if "%~1" == "" goto :EOF & rem Environ Variable Name
if "%~2" == "" goto :EOF & rem Environ Variable Value
if not defined %~1 set "%~1=%~2"
goto :EOF

:exec
echo %*
call %*
goto :EOF

:GetPrefixFile
if "%~1" == "" goto :EOF
if "%~2" == "" goto :EOF
for /f "tokens=* delims=" %%i in ('echo %~2^|unix replacestr -l . _ 2^>nul') do set "%~1=%%i"
goto :EOF

:RequiredParamStr
if "%~1" == "" goto :EOF
if not "%~2" == "" set "%~1=%~2"
if defined %~1 exit /b 0
call :ErrorPrintLn "ERROR in %SelfName% : missed string param %~1. Call %SelfName% --help for details."
call :Sleep 5 . .
exit /b 1
goto :EOF

:RequiredParamInt
if "%~1" == "" goto :EOF
if not "%~2" == "" set /a "%~1=%~2"
if defined %~1 exit /b 0
call :ErrorPrintLn "ERROR in %SelfName% : missed integer param %~1. Call %SelfName% --help for details."
call :Sleep 5 . .
exit /b 1
goto :EOF

:RequiredAdminRights
net session 1>nul 2>nul && exit /b 0
call :ErrorPrintLn "ERROR in %SelfName% : Access denied. Admin rights required."
call :Sleep 5 . .
exit /b 1
goto :EOF

:RequiredUnixUtils
if defined UnixRoot call unix %* 1>nul 2>nul && exit /b 0
call :ErrorPrintLn "ERROR in %SelfName% : UnixUtils is not valid. Install UnixUtils first."
call :Sleep 5 . .
exit /b 1
goto :EOF

:RequiredParentCrwDaq
if not defined CRW_DAQ_SYS_EXE_PID  goto :FailureParentCrwDaq
if not defined CRW_DAQ_SYS_HOME_DIR goto :FailureParentCrwDaq
if not defined CRW_DAQ_SYS_EXE_FILE goto :FailureParentCrwDaq
if not exist %CRW_DAQ_SYS_HOME_DIR% goto :FailureParentCrwDaq
call :TaskExePidIsRunning "%CRW_DAQ_SYS_EXE_FILE%" "%CRW_DAQ_SYS_EXE_PID%" && exit /b 0
rem for %%i in ( %CRW_DAQ_SYS_EXE_FILE% ) do ( unix lsps --pid %CRW_DAQ_SYS_EXE_PID% --name %%~nxi --format pid,name 1>nul 2>nul && exit /b 0 )
:FailureParentCrwDaq
call :ErrorPrintLn "ERROR in %SelfName% : parent CRW-DAQ is not found. Run this script from CRW-DAQ."
call :Sleep 5 . .
exit /b 1
goto :EOF

:RequiredFile
if "%~1" == "" goto :EOF
if exist "%~f1" exit /b 0
call :ErrorPrintLn "ERROR in %SelfName% : not found %~f1."
call :Sleep 5 . .
exit /b 1
goto :EOF

:RequiredDefined
if not "%~1" == "" if defined %1 exit /b 0
call :ErrorPrintLn "ERROR in %SelfName% : undefined variable %~1. Call %SelfName% --help for details."
call :Sleep 5 . .
exit /b 1
goto :EOF

:RequiredIntRange
if "%~1" == "" goto :EOF
if "%~2" == "" goto :EOF
if "%~3" == "" goto :EOF
if %1 GEQ %2 if %1 LEQ %3 exit /b 0
call :ErrorPrintLn "ERROR in %SelfName% : invalid value %~1 (not in %~2..%~3 range). Call %SelfName% --help for details."
call :Sleep 5 . .
exit /b 1
goto :EOF

:EnableCloseMenu
unix setclosemenu -e
goto :EOF

:DisableCloseMenu
unix setclosemenu -d
goto :EOF

:StartTimer
unix cltimer -start
goto :EOF

:StopTimer
unix cltimer -stop
goto :EOF

:SetAnsiCodePage
for /f "tokens=1* delims=:" %%i in ('chcp') do set /a "CodePage=%%j"
if "%CodePage%" == "866" chcp 1251 > nul & rem Set RU ANSI CodePage
goto :EOF

:RestoreCodePage
if defined CodePage chcp %CodePage% > nul
goto :EOF

:DeleteFiles
if "%~1" == "" goto :EOF
if exist "%~1" del /f /q "%~1"
shift /1 & goto :DeleteFiles
goto :EOF

:TaskExePidIsRunning
if "%~1" == "" exit /b 1 & rem EXE name
if "%~2" == "" exit /b 1 & rem PID number
SetLocal EnableExtensions EnableDelayedExpansion
for /f "tokens=1,2*" %%i in ('tasklist /nh /fi "imagename eq %~nx1"') do if /i "%%i" == "%~nx1" if /i "%%j" == "%~2" exit /b 0
exit /b 1
goto :EOF

:Sleep
:: Usage = call :Sleep NumSeconds [StartMessage] [ProgressChar]
if "%~1" == "" goto :EOF
SetLocal EnableExtensions EnableDelayedExpansion
set /a "PingCount=%~1+1"
if not "%~2" == "" call :ErrorPrint %2
if not "%~3" == "" for /L %%i in (1,1,%1) do ( call ping -n 2 localhost 1>nul 2>nul & call :ErrorPrint %~3 )
if     "%~3" == "" call ping -n %PingCount% localhost 1>nul 2>nul
if not "%~2" == "" call :ErrorPrintLn
EndLocal
goto :EOF

:Print
if "%~1" == "" goto :EOF
SetLocal EnableExtensions EnableDelayedExpansion
if not "%~2" == "" <nul set /p "_=%~1 "
if     "%~2" == "" <nul set /p "_=%~1"
EndLocal & shift /1 & goto :Print
goto :EOF

:PrintLn
call :Print %*
echo.
goto :EOF

:ErrorPrint
1>&2 call :Print %*
goto :EOF

:ErrorPrintLn
1>&2 call :PrintLn %*
goto :EOF

:SetErrors
set /a "errors=%~1"
goto :EOF

:IncErrors
set /a "errors+=%~1"
goto :EOF

:HasNoErrors
if "%errors%" == "0" exit /b %ERRORLEVEL%
exit /b 1
goto :EOF

:SetErrorLevel
exit /b %1
goto :EOF

::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: END OF FILE                                                                ::
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
