 {
 Functions for FsmManager.
 procedure fsm_trouble(errno:Integer; msg:String);
 function  fsm_is_valid_name(name:String; typ:Integer):Boolean;
 function  fsm_is_valid_host(name:String):Boolean;
 function  fsm_extract_name(path:String; typ:Integer):String;
 function  fsm_dump(fsm:Integer; mode:Integer; filter:String):String;
 procedure fsm_add_iparam(ref:Integer; name:String; data:Integer);
 procedure fsm_add_fparam(ref:Integer; name:String; data:Real);
 procedure fsm_add_sparam(ref:Integer; name:String; data:String);
 procedure fsm_put_iparam(ref:Integer; name:String; data:Integer);
 procedure fsm_put_fparam(ref:Integer; name:String; data:Real);
 procedure fsm_put_sparam(ref:Integer; name:String; data:String);
 function  fsm_ask_iparam(ref:Integer; name:String):Integer;
 function  fsm_ask_fparam(ref:Integer; name:String):Real;
 function  fsm_ask_sparam(ref:Integer; name:String):String;
 function  fsm_get_cookies(ref:Integer):String;
 function  fsm_get_cookie(ref:Integer; name:String):String;
 procedure fsm_set_cookie(ref:Integer; name,value:String);
 procedure fsm_insert_in(ref:Integer; id:String);
 procedure fsm_remove_from(ref:Integer; id:String);
 procedure fsm_remove_all_from(ref:Integer);
 function  fsm_is_includes(ref:Integer; id:String):Boolean;
 function  fsm_validate_dns(ref:Integer; dns:String):String;
 function  fsm_parse_sml(fsm,txt:Integer; domain:String):Integer;
 function  fsm_read_sml(fsm:Integer; domain,sml:String):Integer;
 function  fsm_init(arg:String):Integer;
 procedure fsm_kill(var ref:Integer);
 procedure ClearFsmManager;
 procedure InitFsmManager;
 procedure FreeFsmManager;
 procedure PollFsmManager;
 }
 {
 FSM error handler.
 }
 procedure fsm_trouble(errno:Integer; msg:String);
 begin
  if (errno<>0) then bNul(fixerror(errno));
  if (fsm_verbose_mode<>0) then Problem(msg);
 end;
 {
 FSM name of (domain,object,state,action,param,option) is valid?
 }
 function fsm_is_valid_name(name:String; typ:Integer):Boolean;
 begin
  fsm_is_valid_name:=IsLexeme(name,fsm_name_rule(typ));
 end;
 {
 FSM hostname is valid?
 }
 function fsm_is_valid_host(name:String):Boolean;
 var i,l,n:Integer; c:Char;
 begin
  i:=1; n:=0;
  l:=Length(name);
  if (l<=253) then
  while (i<=l) do begin
   c:=StrFetch(name,i);
   if (c='.') and (i>1) and (i<l) then n:=n+1 else
   if (c='-') and (i>1) and (i<l) then n:=n+1 else
   if (c>='A') and (c<='Z') then n:=n+1 else
   if (c>='a') and (c<='z') then n:=n+1 else
   if (c>='0') and (c<='9') then n:=n+1 else i:=l;
   i:=i+1;
  end;
  fsm_is_valid_host:=(l>0) and (n=l);
 end;
 {
 FSM extract name of domain/object from path like DOMAIN::OBJECT/STATE/ACTION#PARAM.
 }
 function fsm_extract_name(path:String; typ:Integer):String;
 var p:Integer;
 begin
  if (typ=fsm_type_domain) then begin
   p:=Pos('::',path); if (p>0) then path:=Copy(path,1,p-1);
   p:=Pos('#',path);  if (p>0) then path:=Copy(path,1,p-1);
   if not IsLexeme(path,lex_Name) then path:='';
  end else
  if (typ=fsm_type_object) or (typ=fsm_type_class) or (typ=fsm_type_objectset) then begin
   p:=Pos('::',path); if (p>0) then path:=Copy(path,p+2) else path:='';
   p:=Pos('/',path);  if (p>0) then path:=Copy(path,1,p-1);
   p:=Pos('#',path);  if (p>0) then path:=Copy(path,1,p-1);
   if not IsLexeme(path,lex_Name) then path:='';
  end else
  if (typ=fsm_type_state) then begin
   p:=Pos('/',path);  if (p>0) then path:=Copy(path,p+1) else path:='';
   p:=Pos('/',path);  if (p>0) then path:=Copy(path,1,p-1);
   p:=Pos('#',path);  if (p>0) then path:=Copy(path,1,p-1);
   if not IsLexeme(path,lex_Name) then path:='';
  end else
  if (typ=fsm_type_action) then begin
   p:=Pos('/',path);  if (p>0) then path:=Copy(path,p+1) else path:='';
   p:=Pos('/',path);  if (p>0) then path:=Copy(path,p+1) else path:='';
   p:=Pos('#',path);  if (p>0) then path:=Copy(path,1,p-1);
   if not IsLexeme(path,lex_Name) then path:='';
  end else
  if (typ>=fsm_type_int) and (typ<=fsm_type_parameter) then begin
   p:=Pos('#',path);  if (p>0) then path:=Copy(path,p+1) else path:='';
   if not IsLexeme(path,lex_Name) then path:='';
  end else
  path:='';
  fsm_extract_name:=path;
 end;
 {
 FSM dump, as list of PATH => DECLARATION.
 }
 function fsm_dump(fsm:Integer; mode:Integer; filter:String):String;
 var i,typ,lines,list,wid,ref:Integer; line:String;
  function MatchFilter(var s:String):Boolean;
  var inv:Boolean; iw,nm,ne,p:Integer; wi:String;
  begin
   wi:='';
   nm:=0; ne:=0;
   if (filter='') then nm:=1 else begin
    for iw:=1 to WordCount(filter) do begin
     wi:=ExtractWord(iw,filter);
     if (StrFetch(wi,1)='^') then begin
      inv:=true; wi:=Copy(wi,2);
     end else inv:=false;
     p:=Pos(wi,UpCaseStr(s));
     if (p>0) then begin if inv then ne:=ne+1 else nm:=nm+1; end;
    end;
   end;
   wi:='';
   MatchFilter:=(nm>0) and (ne=0);
  end;
 begin
  line:='';
  list:=StringToText('');
  lines:=StringToText(fsm_ctrl(fsm,fsm_sym_catalog));
  filter:=UpCaseStr(filter);
  filter:=Trim(filter);
  wid:=0;
  for i:=0 to text_numln(lines)-1 do begin
   line:=text_getln(lines,i);
   wid:=imax(wid,length(line));
  end;
  for i:=0 to text_numln(lines)-1 do begin
   line:=text_getln(lines,i);
   if MatchFilter(line) then begin
    if (mode>0) then begin
     ref:=fsm_find(fsm,fsm_type_any,line);
     if (ref=0) then for typ:=fsm_type_int to fsm_type_any-1 do if (ref=0) then ref:=fsm_find(fsm,typ,line);
     line:=RightPad(line,wid,' ')+' => ';
     line:=line+fsm_ctrl(ref,fsm_sym_declaration);
    end;
    bNul(text_addln(list,line));
   end;
  end;
  fsm_dump:=TextToString(list);
  bNul(text_free(lines));
  bNul(text_free(list));
  line:='';
 end;
 {
 FSM add parameter int/float/string and set data.
 }
 procedure fsm_add_iparam(ref:Integer; name:String; data:Integer);
 begin
  if not fsm_set_iparam(fsm_add(ref,fsm_type_int,name),data)
  then fsm_trouble(fsm_errno_set_param,'fsm_add_iparam '+name);
 end;
 procedure fsm_add_fparam(ref:Integer; name:String; data:Real);
 begin
  if not fsm_set_fparam(fsm_add(ref,fsm_type_float,name),data)
  then fsm_trouble(fsm_errno_set_param,'fsm_add_fparam: '+name);
 end;
 procedure fsm_add_sparam(ref:Integer; name:String; data:String);
 begin
  if not fsm_set_sparam(fsm_add(ref,fsm_type_string,name),data)
  then fsm_trouble(fsm_errno_set_param,'fsm_add_sparam: '+name);
 end;
 {
 FSM put (set by name) parameter int/float/string with given data.
 }
 procedure fsm_put_iparam(ref:Integer; name:String; data:Integer);
 begin
  if not fsm_set_iparam(fsm_find(ref,fsm_type_int,name),data)
  then fsm_trouble(fsm_errno_set_param,'fsm_put_iparam '+name);
 end;
 procedure fsm_put_fparam(ref:Integer; name:String; data:Real);
 begin
  if not fsm_set_fparam(fsm_find(ref,fsm_type_float,name),data)
  then fsm_trouble(fsm_errno_set_param,'fsm_put_fparam: '+name);
 end;
 procedure fsm_put_sparam(ref:Integer; name:String; data:String);
 begin
  if not fsm_set_sparam(fsm_find(ref,fsm_type_string,name),data)
  then fsm_trouble(fsm_errno_set_param,'fsm_put_sparam: '+name);
 end;
 {
 FSM ask (get by name) int/float/string parameter value.
 }
 function fsm_ask_iparam(ref:Integer; name:String):Integer;
 var par:Integer; v:Integer;
 begin
  v:=0;
  par:=fsm_find(ref,fsm_type_int,name);
  if (par<>0) then v:=fsm_get_iparam(par) else fsm_trouble(fsm_errno_get_param,'fsm_ask_iparam: '+name);
  fsm_ask_iparam:=v;
 end;
 function fsm_ask_fparam(ref:Integer; name:String):Real;
 var par:Integer; v:Real;
 begin
  v:=0;
  par:=fsm_find(ref,fsm_type_float,name);
  if (par<>0) then v:=fsm_get_fparam(par) else fsm_trouble(fsm_errno_get_param,'fsm_ask_fparam: '+name);
  fsm_ask_fparam:=v;
 end;
 function fsm_ask_sparam(ref:Integer; name:String):String;
 var par:Integer; v:String;
 begin
  v:='';
  par:=fsm_find(ref,fsm_type_string,name);
  if (par<>0) then v:=fsm_get_sparam(par) else fsm_trouble(fsm_errno_get_param,'fsm_ask_sparam: '+name);
  fsm_ask_sparam:=v;
  v:='';
 end;
 {
 FSM internal use routine: update cookies (list of EOL separated name=value) with new (value).
 }
 function fsm_update_cookies(cookies,name,value:String):String;
 var txt,i,n,p:Integer; line:String;
 begin
  line:='';
  if IsLexeme(name,lex_Name) then begin
   txt:=StringToText(cookies);
   i:=0; n:=-1; // find n = index of name
   while (i<text_numln(txt)) and (n<0) do begin
    line:=text_getln(txt,i); p:=Pos('=',line);
    if (p>0) then if IsSameText(Trim(Copy(line,1,p-1)),name) then n:=i;
    i:=i+1;
   end;
   if (n>=0) and (value='') then bNul(text_delln(txt,n)) else
   if (n>=0) and (Ord(IsLexeme(value,lex_Word))>0) then bNul(text_putln(txt,n,LoCaseStr(name)+'='+value)) else
   if (n<0)  and (Ord(IsLexeme(value,lex_Word))>0) then bNul(text_addln(txt,LoCaseStr(name)+'='+value));
   cookies:=TextToString(txt);
   bNul(text_free(txt));
  end;
  line:='';
  fsm_update_cookies:=cookies;
 end;
 {
 FSM get/set item (name) from EOL separated list of cookies (name=value) stored in fsm_ctrl(ref,'cookie').
 }
 function fsm_get_cookies(ref:Integer):String;
 begin
  if (fsm_type(ref)>fsm_type_parameter)
  then fsm_get_cookies:=fsm_ctrl(ref,fsm_sym_cookie)
  else fsm_get_cookies:='';
 end;
 function fsm_get_cookie(ref:Integer; name:String):String;
 begin
  if (fsm_type(ref)>fsm_type_parameter)
  then fsm_get_cookie:=cookiescan(fsm_ctrl(ref,fsm_sym_cookie),name,0)
  else fsm_get_cookie:='';
 end;
 procedure fsm_set_cookie(ref:Integer; name,value:String);
 begin
  if IsLexeme(name,lex_Name) and (fsm_type(ref)>fsm_type_parameter) then begin
   if (cookiescan(fsm_ctrl(ref,fsm_sym_cookie),name,0)='') and (value<>'')
   then sNul(fsm_ctrl(ref,fsm_sym_cookie+'='+fsm_ctrl(ref,fsm_sym_cookie)+(LoCaseStr(name)+'='+value+EOL)))
   else sNul(fsm_ctrl(ref,fsm_sym_cookie+'='+fsm_update_cookies(fsm_ctrl(ref,fsm_sym_cookie),name,value)));
  end;
 end;
 {
 FSM internal operation (op=+/-) to Insert/Remove object/objectset (id) in objectset (ref).
 }
 procedure fsm_objset_op(ref:Integer; id:String; op:Char);
 var i,obj,wc,um,cl1,cl2:Integer; ul,wd:string;
  procedure  ClearLocals;
  begin
   ul:=''; wd:='';
  end;
  function TextListOp(list,item:String; op:Char):String;
  var txt:Integer;
  begin
   if (item<>'') then begin
    txt:=StringToText(StringReplace(list,',',EOL,rfReplaceAll));
    if (Text_IndexOf(txt,item)<0) and (op='+') then begin
     bNul(Text_AddLn(txt,item));
     list:=StringReplace(Trim(Text_ToString(txt)),EOL,',',rfReplaceAll);
    end;
    if (Text_IndexOf(txt,item)>=0) and (op='-') then begin
     iNul(Text_Remove(txt,item));
     list:=StringReplace(Trim(Text_ToString(txt)),EOL,',',rfReplaceAll);
    end;
    bNul(Text_Free(txt));
   end;
   TextListOp:=list;
  end;
 begin
  ClearLocals;
  if (fsm_type(ref)=fsm_type_objectset) then begin
   um:=val(fsm_ctrl(ref,fsm_sym_unionmode));
   ul:=fsm_ctrl(ref,fsm_sym_unionlist);
   id:=Trim(UpCaseStr(id));
   wc:=WordCount(id);
   if (wc>0) then
   if (wc=1) then begin
    if IsLexeme(id,lex_Name) then begin
     obj:=fsm_find(fsm_parent(ref),fsm_type_any,id); // object to be inserted/removed
     cl1:=fsm_find(fsm_parent(ref),fsm_type_class,fsm_ctrl(ref,fsm_sym_is_of_class));
     cl2:=fsm_find(fsm_parent(ref),fsm_type_class,fsm_ctrl(obj,fsm_sym_is_of_class));
     if (cl1=0) then cl2:=0;
     if (fsm_type(obj)=fsm_type_object) and (um=0) and (cl1=cl2) then begin
      sNul(fsm_ctrl(ref,fsm_sym_unionlist+'='+TextListOp(ul,id,op)));
     end;
     if (fsm_type(obj)=fsm_type_objectset) and (um=1) and (cl1=cl2) then begin
      sNul(fsm_ctrl(ref,fsm_sym_unionlist+'='+TextListOp(ul,id,op)));
     end;
    end;
   end else begin
    for i:=1 to WordCount(id) do begin
     wd:=ExtractWord(i,id);
     if IsLexeme(wd,lex_Name)
     then fsm_objset_op(ref,wd,op);
    end;
   end;
  end;
  ClearLocals;
 end;
 {
 Insert object/objectset (id) in objectset (ref).
 }
 procedure fsm_insert_in(ref:Integer; id:String);
 begin
  fsm_objset_op(ref,id,'+');
 end;
 {
 Remove object/objectset (id) from objectset (ref).
 }
 procedure fsm_remove_from(ref:Integer; id:String);
 begin
  fsm_objset_op(ref,id,'-');
 end;
 {
 Remove all objecs from objectset (ref).
 }
 procedure fsm_remove_all_from(ref:Integer);
 begin
  sNul(fsm_ctrl(ref,fsm_sym_unionlist+'='));
 end;
 {
 FSM objectset (ref) is includes object with name id?
 }
 function fsm_is_includes(ref:Integer; id:String):Boolean;
 var p:Integer;
 begin
  p:=0;
  if IsLexeme(id,lex_Name) then
  if (fsm_type(ref)=fsm_type_objectset) then begin
   p:=Pos(StrFmt(',%s,',UpCaseStr(id)),StrFmt(',%s,',StringReplace(fsm_ctrl(ref,fsm_sym_unionlist),EOL,',',rfReplaceAll)));
  end;
  fsm_is_includes:=(p>0);
 end;
 {
 FSM validate dns (empty -> dns_def, . -> dns_dot).
 }
 function fsm_validate_dns(fsm:Integer; dns:String):String;
 begin
  if (dns='')  then dns:=fsm_get_cookie(fsm,fsm_sym_hostname);
  if (dns='.') then dns:=fsm_get_cookie(fsm,fsm_sym_hostname);
  if not fsm_is_valid_host(dns) then dns:='';
  fsm_validate_dns:=dns;
 end;
 {
 FSM readback content as SML text.
 }
 function fsm_readback_sml(fsm:Integer):String;
 var dom,idom,ndom,obj,iobj,nobj,sta,ista,nsta,act,iact,nact,objset,iobjset,nobjset,fun,ifun,nfun:Integer;
  line,stab:String; lines,ncookies:Integer;
  procedure ClearLocals;
  begin
   line:=''; stab:='';
  end;
  procedure SetTabLevel(n:Integer);
  begin
   stab:=StringOfChar(' ',fsm_tablen*n);
  end;
  procedure FormatComment(var line:String; compos:Integer);
  var p:Integer;
  begin
   p:=Pos('!',line);
   if (p>0) and (p<compos) then
   if (Trim(Copy(line,1,p-1))<>'') then begin
    line:=RightPad(Copy(line,1,p-1),compos,' ')+' '+Copy(line,p);
   end;
  end;
  procedure AppendLine(line:String);
  begin
   bNul(text_addln(lines,line));
  end;
  procedure AppendEmptyLine;
  begin
   if (text_getln(lines,text_numln(lines)-1)<>'')
   then bNul(text_addln(lines,''));
  end;
  procedure AppendDeclaration(ref:Integer);
  begin
   if (fsm_type(ref)>fsm_type_parameter) then
   if (fsm_type(ref)=fsm_type_manager) then begin
    line:=stab+'!manager: '+fsm_name(ref);
    AppendLine(line);
   end else begin
    line:=stab+fsm_ctrl(ref,fsm_sym_declaration);
    FormatComment(line,fsm_rempos);
    AppendLine(line);
   end;
  end;
  procedure AppendBody(ref:Integer);
  var body:String;
  begin
   body:='';
   if (fsm_type(ref)>fsm_type_parameter) then begin
    body:=fsm_ctrl(ref,fsm_sym_body);
    if not IsEmptyStr(body) then begin
     iNul(TextAppendString(lines,body));
     AppendEmptyLine;
    end;
   end;
   body:='';
  end;
  procedure AppendCookie(s:String);
  var p:Integer;
  begin
   p:=Pos('=',s);
   if (p>1) and (p<Length(s)) then begin
    s:=stab+'!'+Copy(s,1,p-1)+': '+Copy(s,p+1);
    bNul(text_addln(lines,s));
    ncookies:=ncookies+1;
   end;
  end;
  procedure AppendCookies(ref:Integer);
  var i:Integer; cookie:String;
  begin
   cookie:='';
   ncookies:=0;
   cookie:=fsm_ctrl(ref,fsm_sym_cookie);
   for i:=0 to TextLineCount(cookie,EOL)-1 do begin
    AppendCookie(ExtractTextLine(i,cookie,EOL));
   end;
   if (fsm_type(ref)<>fsm_type_state) then AppendCookie(fsm_sym_color+'='+fsm_ctrl(ref,fsm_sym_color));
   if (fsm_type(ref)=fsm_type_manager) then AppendCookie(fsm_sym_defaultcolor+'='+fsm_ctrl(ref,fsm_sym_defaultcolor));
   if (fsm_type(ref)<>fsm_type_action) then AppendCookie(fsm_sym_visible+'='+fsm_ctrl(ref,fsm_sym_visible));
   if (ncookies>0) then AppendEmptyLine;
   cookie:='';
  end;
  procedure AppendParameters(ref:Integer; prefix:String);
  var par,ipar,npar:Integer;
  begin
   npar:=fsm_count(ref,fsm_type_parameter);
   for ipar:=0 to npar-1 do begin
    par:=fsm_items(ref,fsm_type_parameter,ipar);
    if (fsm_type(par)>0) then begin
     if (ipar=0) then line:=stab+prefix else line:=stab+StringOfChar(' ',Length(prefix));
     line:=line+fsm_ctrl(par,fsm_sym_declaration);
     if (ipar<npar-1) then line:=line+',';
     AppendLine(line);
    end;
    if (ipar=npar-1) then AppendEmptyLine;
   end;
  end;
 begin
  ClearLocals;
  lines:=Text_New;
  // FsmManager
  SetTabLevel(0);
  AppendDeclaration(fsm);
  AppendEmptyLine;
  AppendCookies(fsm);
  AppendParameters(fsm,'!parameters: ');
  // !domain:
  ndom:=fsm_count(fsm,fsm_type_domain);
  for idom:=0 to ndom-1 do begin
   dom:=fsm_items(fsm,fsm_type_domain,idom);
   if (fsm_type(dom)>0) then begin
    SetTabLevel(0);
    AppendDeclaration(dom);
    AppendEmptyLine;
    AppendCookies(dom);
    AppendParameters(dom,'!parameters: ');
    AppendBody(dom);
    // class: object:  objectset:
    nobj:=fsm_count(dom,fsm_type_parent);
    for iobj:=0 to nobj-1 do begin
     obj:=fsm_items(dom,fsm_type_parent,iobj);
     if (fsm_type(obj)=fsm_type_class) or (fsm_type(obj)=fsm_type_object) then begin
      SetTabLevel(0);
      AppendDeclaration(obj);
      AppendCookies(obj);
      AppendParameters(obj,'parameters: ');
      AppendBody(obj);
      // function:
      nfun:=fsm_count(obj,fsm_type_function);
      for ifun:=0 to nfun-1 do begin
       fun:=fsm_items(obj,fsm_type_function,ifun);
       if (fsm_type(fun)>0) then begin
        SetTabLevel(1);
        AppendDeclaration(fun);
        AppendCookies(fun);
        AppendBody(fun);
       end; // if (fun)
      end; // for ifun
      AppendEmptyLine;
      // state:
      nsta:=fsm_count(obj,fsm_type_state);
      for ista:=0 to nsta-1 do begin
       sta:=fsm_items(obj,fsm_type_state,ista);
       if (fsm_type(sta)>0) then begin
        SetTabLevel(1);
        AppendDeclaration(sta);
        AppendCookies(sta);
        AppendParameters(sta,'!parameters: ');
        AppendBody(sta);
        // action:
        nact:=fsm_count(sta,fsm_type_action);
        for iact:=0 to nact-1 do begin
         act:=fsm_items(sta,fsm_type_action,iact);
         if (fsm_type(act)=fsm_type_action) then begin
          SetTabLevel(2);
          AppendDeclaration(act);
          AppendCookies(act);
          AppendBody(act);
         end; // if (act)
        end; // for iact
       end; // if (sta)
      end; // for ista
      AppendEmptyLine;
     end; // if (obj)
     if (fsm_type(obj)=fsm_type_objectset) then begin
      SetTabLevel(0);
      AppendDeclaration(obj);
      AppendCookies(obj);
      AppendEmptyLine;
     end;
    end; // for iobj
    AppendEmptyLine;
   end; // if (dom)
  end; // for idom
  fsm_readback_sml:=TextToString(lines);
  bNul(text_free(lines));
  ClearLocals;
 end;
 {
 Convert comma separated params to normalized EOL text of lines: (int|float|string) name = value.
 }
 function fsm_normalize_params(arg:String; var params:String):Integer;
 var errors,i,p,lines,typ:Integer; c,cbeg,cend:Char; wt,wn,wv,line,delims:String; IsQuoted:Boolean;
  procedure ClearLocals;
  begin
   wt:=''; wn:=''; wv:=''; line:=''; delims:='';
  end;
 begin
  ClearLocals;
  errors:=0; params:=''; arg:=Trim(arg);
  // Remove brackets () if one exists
  cbeg:=StrFetch(arg,1); cend:=StrFetch(arg,Length(arg)); // 1st/last char
  if (cbeg='(') then arg:=Copy(arg,2,Length(arg)-1);      // Remove lead (
  if (cend=')') then arg:=Copy(arg,1,Length(arg)-1);      // Remove tail )
  if (cbeg='(') and (cend<>')') then errors:=errors+1;    // Unbalanced ()
  if (cend=')') and (cbeg<>'(') then errors:=errors+1;    // Unbalanced ()
  // Replace comma (,) to EOL, if one is not quoted
  i:=1; IsQuoted:=false;
  while (i<=Length(arg)) do begin
   c:=StrFetch(arg,i);
   if (c=QuoteMark) then IsQuoted:=not IsQuoted else
   if (c=',') and not IsQuoted then arg:=Copy(arg,1,i-1)+EOL+Copy(arg,i+1);
   i:=i+1;
  end;
  // Process text lines
  lines:=StringToText(arg);
  delims:=' '+EOL+Chr(_HT);
  for i:=0 to text_numln(lines)-1 do begin
   line:=Trim(text_getln(lines,i));
   if (line<>'') then begin
    wv:=''; wn:='';
    p:=Pos('=',line);
    if (p>0) then begin
     wv:=Trim(Copy(line,p+1));
     line:=Trim(Copy(line,1,p-1));
    end;
    wt:=ExtractWordDelims(1,line,delims);
    typ:=WordIndex(LoCaseStr(wt),fsm_scalar_type_list);
    if (typ>0) then begin
     wt:=LoCaseStr(wt);
     wn:=UpCaseStr(ExtractWordDelims(2,line,delims));
     if (WordCountDelims(line,delims)<>2) then errors:=errors+1;
    end else begin
     wn:=UpCaseStr(wt);
     wt:=''; // To be defined indirect from data value
     if (typ=0) then if IsLexeme(wv,lex_iparam) then begin typ:=1; wt:=ExtractWord(typ,fsm_scalar_type_list); end;
     if (typ=0) then if IsLexeme(wv,lex_fparam) then begin typ:=2; wt:=ExtractWord(typ,fsm_scalar_type_list); end;
     if (typ=0) then if IsLexeme(wv,lex_sparam) then begin typ:=3; wt:=ExtractWord(typ,fsm_scalar_type_list); end;
     if (typ=0) then if (wv='') then begin typ:=3; wt:=ExtractWord(typ,fsm_scalar_type_list); end;
     if (WordCountDelims(line,delims)<>1) then errors:=errors+1;
    end;
    if (typ=0) then errors:=errors+1;
    if (typ=1) and (wv='') then wv:='0';
    if (typ=2) and (wv='') then wv:='0.0';
    if (typ=3) and (wv='') then wv:=AnsiQuotedStr('',Quotemark);
    if not IsLexeme(wt,lex_name) then errors:=errors+1;
    if not IsLexeme(wn,lex_name) then errors:=errors+1;
    if (typ=1) then if not IsLexeme(wv,lex_iparam) then errors:=errors+1;
    if (typ=2) then if not IsLexeme(wv,lex_fparam) then errors:=errors+1;
    if (typ=3) then if not IsLexeme(wv,lex_sparam) then errors:=errors+1;
    if (errors=0) then begin
     if (typ=3) and (StrFetch(wv,1)<>QuoteMark) then wv:=AnsiQuotedStr(wv,QuoteMark);
     line:=wt+' '+wn+' = '+wv;
     if (params='')
     then params:=line
     else params:=params+EOL+line;
    end;
   end;
  end;
  ClearLocals;
  bNul(text_free(lines));
  fsm_normalize_params:=errors;
 end;
 {
 FSM internal routine.
 FSM clone (make a copy) from class/object (obj1) to object (obj2).
 }
 procedure fsm_clone_object(obj1,obj2:Integer);
 var ista,iact:Integer; sta1,sta2,act1,act2:Integer;
  procedure CopyProperty(src,dst:Integer; id:String);
  begin
   sNul(fsm_ctrl(dst,id+'='+fsm_ctrl(src,id)));
  end;
  procedure CopyProperties(src,dst:Integer);
  var i:Integer; list:String;
  begin
   list:=''; list:=fsm_ctrl(src,'*=');
   for i:=1 to WordCount(list) do CopyProperty(src,dst,ExtractWord(i,list));
   list:='';
  end;
  procedure CopyParameter(src,dst:Integer);
  begin
   if (fsm_type(src)=fsm_type_int) then fsm_add_iparam(dst,fsm_name(src),fsm_get_iparam(src));
   if (fsm_type(src)=fsm_type_float) then fsm_add_fparam(dst,fsm_name(src),fsm_get_fparam(src));
   if (fsm_type(src)=fsm_type_string) then fsm_add_sparam(dst,fsm_name(src),fsm_get_sparam(src));
  end;
  procedure CopyParameters(src,dst:Integer);
  var ipar:Integer;
  begin
   for ipar:=0 to fsm_count(src,fsm_type_parameter)-1 do CopyParameter(fsm_items(src,fsm_type_parameter,ipar),dst);
  end;
 begin
  if (fsm_type(obj1)=fsm_type_class) or  (fsm_type(obj1)=fsm_type_object) then
  if (fsm_type(obj2)=fsm_type_class) or  (fsm_type(obj2)=fsm_type_object) then begin
   // copy object states
   for ista:=0 to fsm_count(obj1,fsm_type_state)-1 do begin
    sta1:=fsm_items(obj1,fsm_type_state,ista);
    sta2:=fsm_add(obj2,fsm_type_state,fsm_name(sta1));
    // copy state actions
    for iact:=0 to fsm_count(sta1,fsm_type_action)-1 do begin
     act1:=fsm_items(sta1,fsm_type_action,iact);
     act2:=fsm_add(sta2,fsm_type_action,fsm_name(act1));
     CopyProperties(act1,act2);
     CopyParameters(act1,act2);
    end;
    CopyProperties(sta1,sta2);
    CopyParameters(sta1,sta2);
   end;
   CopyProperties(obj1,obj2);
   CopyParameters(obj1,obj2);
  end;
 end;
 {
 FSM with reference (fsm) parse SML text (txt), use domain name.
 }
 function fsm_parse_sml(fsm,txt:Integer; domain:String):Integer;
 var errors,body,context,iline,j,p,lines,typ:Integer;
     textline,sline,comm,sn,sv,decltype,declargs,declname,declopts:String;
     rexlinecomm,rexdecltype,rexdeclname,rexoptassoc,rexdeadstat:Integer;
     rexinitstat,rexisofclas,rexunionmod,rexunionlst,rexunioncsv:Integer;
     rexoptcooki:Integer;
     dom,obj,objset,sta,act,fun,par,iv:Integer; fv:Real;
  procedure ErrorFound(msg:String);
  begin
   errors:=errors+1;
   if (msg<>'') then Problem('fsm_parse_sml: '+msg);
  end;
  procedure ClearLocals;
  begin
   errors:=0; body:=0; context:=0;
   textline:=''; sline:=''; comm:=''; sn:=''; sv:='';
   decltype:=''; declargs:=''; declname:=''; declopts:='';
   rexlinecomm:=0; rexdecltype:=0; rexdeclname:=0; rexoptassoc:=0;
   rexdeadstat:=0; rexinitstat:=0; rexisofclas:=0; rexunionmod:=0;
   rexunionlst:=0; rexunioncsv:=0; rexoptcooki:=0;
   dom:=0; obj:=0; objset:=0; sta:=0; act:=0; fun:=0; par:=0; iv:=0; fv:=0;
  end;
  procedure InitLocals;
  begin
   body:=text_new;
   rexlinecomm:=regexp_init(0,'/^(.*?)([!#].*)$/i'); // Extract line and !comment or #comment.
   rexdecltype:=regexp_init(0,'/^\s*\b(class|object|state|action|objectset|function|parameters)\b\s*(:)(.*)$/i'); // Declaration
   rexdeclname:=regexp_init(0,'/^\s*([a-zA-Z_&]+[a-zA-Z0-9_&:\.-]*)(.*)$/i');
   rexoptassoc:=regexp_init(0,'/\s*/\s*(associated)\b/i');
   rexdeadstat:=regexp_init(0,'/\s*/\s*(dead_state)\b/i');
   rexinitstat:=regexp_init(0,'/\s*/\s*(initial_state)\b/i');
   rexisofclas:=regexp_init(0,'/\s*(is_of_class)\s+\b([a-zA-Z_]+\w*)\b/i');
   rexunionmod:=regexp_init(0,'/\s+(union)\b/i');
   rexunionlst:=regexp_init(0,'/\{([\w\,\s]*)\}/i');
   rexunioncsv:=regexp_init(0,'/\b([_a-zA-Z]+\w*)\b/ig');
   rexoptcooki:=regexp_init(0,'/\s*!\s*([a-zA-Z_]+\w*)\s*:\s*([^!#]+)\s*/img');
  end;
  procedure FreeLocals;
  begin
   bNul(text_free(body));
   bNul(regexp_free(rexlinecomm));
   bNul(regexp_free(rexdecltype));
   bNul(regexp_free(rexdeclname));
   bNul(regexp_free(rexoptassoc));
   bNul(regexp_free(rexdeadstat));
   bNul(regexp_free(rexinitstat));
   bNul(regexp_free(rexisofclas));
   bNul(regexp_free(rexunionmod));
   bNul(regexp_free(rexunionlst));
   bNul(regexp_free(rexunioncsv));
   bNul(regexp_free(rexoptcooki));
  end;
  procedure ChangeContext(con:Integer);
  begin
   if (fsm_type(context)>fsm_type_parameter)
   then sNul(fsm_ctrl(context,fsm_sym_body+'='+TextToString(body)));
   ClearText(body);
   context:=con;
  end;
  procedure AddParameters(ref:Integer; sv:String);
  var lines,typ,iv,j:Integer; fv:Real;
  begin
   lines:=StringToText(sv);
   for j:=0 to text_numln(lines)-1 do begin
    typ:=WordIndex(ExtractWord(1,text_getln(lines,j)),fsm_scalar_type_list);
    if (typ=1) then begin
     iv:=iValDef(ExtractWord(3,text_getln(lines,j)),0);
     fsm_add_iparam(ref,ExtractWord(2,text_getln(lines,j)),iv);
    end;
    if (typ=2) then begin
     fv:=rValDef(ExtractWord(3,text_getln(lines,j)),0);
     fsm_add_fparam(ref,ExtractWord(2,text_getln(lines,j)),fv);
    end;
    if (typ=3) then begin
     sv:=ExtractFirstParam(SkipWords(2,text_getln(lines,j)),QuoteMark);
     if (sv<>'') and (backslash_decode(sv)<>'') then sv:=backslash_decode(sv);
     fsm_add_sparam(ref,ExtractWord(2,text_getln(lines,j)),sv)
    end;
   end;
   bNul(text_free(lines));
  end;
  // Add cookies from comment line looks like: !color: Red !title: ERROR.
  // NB: !color: and !visible: cookies should be saved via fsm_ctrl(..).
  procedure AddCookies(ref:Integer);
  var k:Integer; sn,sv:String;
  begin
   sn:=''; sv:='';
   if (fsm_type(ref)>fsm_type_parameter) then
   for k:=1 to regexp_matchnum(rexoptcooki,0) do begin
    sn:=Trim(regexp_matchstr(rexoptcooki,k,1));
    sv:=Trim(regexp_matchstr(rexoptcooki,k,2));
    if (sn<>'') and (sv<>'') then
    if IsSameText(sn,fsm_sym_parameters) then begin
     if (fsm_type(ref)=fsm_type_manager)
     or (fsm_type(ref)=fsm_type_domain)
     or (fsm_type(ref)=fsm_type_state) then begin
      if (fsm_normalize_params(sv,sv)=0)
      then AddParameters(ref,sv)
      else ErrorFound('Invalid declaration of !parameters: '+declname);
     end;
    end else
    if IsSameText(sn,fsm_sym_color) then begin
     sNul(fsm_ctrl(ref,fsm_sym_color+'='+sv));
    end else
    if IsSameText(sn,fsm_sym_visible) then begin
     sNul(fsm_ctrl(ref,fsm_sym_visible+'='+sv));
    end else
    fsm_set_cookie(ref,sn,sv);
   end;
   sn:=''; sv:='';
  end;
 begin
  ClearLocals; InitLocals;
  if (fsm_type(fsm)=fsm_type_manager) and (txt<>0) and (text_numln(txt)>0) then begin
   dom:=fsm_find(fsm,fsm_type_domain,domain);
   if (fsm_type(dom)=fsm_type_domain) then begin
    context:=dom;
    for iline:=0 to text_numln(txt)-1 do begin
     // Get text line and extract !comment or #comment
     /////////////////////////////////////////////////
     textline:=text_getln(txt,iline);
     sline:=textline; comm:=''; sn:=''; sv:='';
     if (regexp_exec(rexlinecomm,sline)=1) then begin
      sline:=regexp_matchstr(rexlinecomm,1,1);
      comm:=regexp_matchstr(rexlinecomm,1,2);
     end;
     // Extract declaration of class,object,state,action,objectset,function,parameters
     /////////////////////////////////////////////////////////////////////////////////
     decltype:=''; declargs:=''; declname:=''; declopts:='';
     if (regexp_exec(rexdecltype,sline)=1) then begin
      decltype:=regexp_matchstr(rexdecltype,1,1);
      declargs:=regexp_matchstr(rexdecltype,1,3);
     end;
     if (regexp_exec(rexdeclname,declargs)=1) then begin
      declname:=regexp_matchstr(rexdeclname,1,1);
      declopts:=regexp_matchstr(rexdeclname,1,2);
     end;
     ///////////////////////////////////////////////////////////////////
     // No declaration - parse cookie (!name: value) or add line to body
     ///////////////////////////////////////////////////////////////////
     if (decltype='') then begin
      if (fsm_type(context)>fsm_type_parameter) then begin
       if IsEmptyStr(sline) and (regexp_exec(rexoptcooki,comm)>0) and (StrFetch(comm,1)='!') then begin
        if (StrFetch(comm,1)='!') then AddCookies(context);
       end else
       bNul(text_addln(body,textline));
      end;
     end else
     /////////////////////////////////////////////////////////////////////////
     // class: class-name [ /associated ]
     /////////////////////////////////////////////////////////////////////////
     if IsSameText(decltype,'class') then begin
      obj:=fsm_add(dom,fsm_type_class,declname);
      if (fsm_type(obj)=fsm_type_class) then begin
       if regexp_test(rexoptassoc,declopts) then sNul(fsm_ctrl(obj,fsm_sym_associated+'=1'));
       // !color: and another cookies
       if (regexp_exec(rexoptcooki,comm)>0) then begin
        if (StrFetch(comm,1)='!') then AddCookies(obj);
       end;
      end else ErrorFound('Invalid declaration of class: '+declname);
      objset:=0; sta:=0; act:=0; fun:=0; par:=0;
      ChangeContext(obj);
     end else
     //////////////////////////////////////////////////////////////////
     // object: object-name [ is_of_class class-name ] [ /associated ]
     //////////////////////////////////////////////////////////////////
     if IsSameText(decltype,'object') then begin
      obj:=fsm_add(dom,fsm_type_object,declname);
      if (fsm_type(obj)=fsm_type_object) then begin
       if regexp_test(rexoptassoc,declopts) then sNul(fsm_ctrl(obj,fsm_sym_associated+'=1'));
       if (regexp_exec(rexisofclas,declopts)=1) then begin
        iv:=fsm_find(dom,fsm_type_class,regexp_matchstr(rexisofclas,1,2));
        if (fsm_type(iv)=fsm_type_class) then begin
         fsm_clone_object(iv,obj);
         sNul(fsm_ctrl(obj,fsm_sym_is_of_class+'='+fsm_name(iv)));
        end else ErrorFound('Invalid reference is_of_class: '+regexp_matchstr(rexisofclas,1,2));
       end;
       // !color: and another cookies
       if (regexp_exec(rexoptcooki,comm)>0) then begin
        if (StrFetch(comm,1)='!') then AddCookies(obj);
       end;
      end else ErrorFound('Invalid declaration of object: '+declname);
      objset:=0; sta:=0; act:=0; fun:=0; par:=0;
      ChangeContext(obj);
     end else
     /////////////////////////////////////////////////////////////////////////
     // state: state-name [ /initial_state ] [ /dead_state ] [ !color: COLOR ]
     // Corresponds to object/class which was current before
     /////////////////////////////////////////////////////////////////////////
     if IsSameText(decltype,'state') then begin
      sta:=fsm_add(obj,fsm_type_state,declname);
      if (fsm_type(sta)=fsm_type_state) then begin
       // option /initial_state
       if regexp_test(rexinitstat,declopts) then begin
        sNul(fsm_ctrl(obj,fsm_sym_initial_state+'='+declname));
        iNul(fsm_set_state(obj,sta));
       end;
       // option /dead_state
       if regexp_test(rexdeadstat,declopts) then begin
        sNul(fsm_ctrl(obj,fsm_sym_dead_state+'='+declname));
       end;
       // !color: and another cookies
       if (regexp_exec(rexoptcooki,comm)>0) then begin
        if (StrFetch(comm,1)='!') then AddCookies(sta);
       end;
      end else ErrorFound('Invalid declaration of state: '+declname);
      objset:=0; act:=0; fun:=0; par:=0;
      ChangeContext(sta);
     end else
     //////////////////////////////////////////////////////////////////
     // action: NAME [ (PAR1, PAR2, ..) ]
     //////////////////////////////////////////////////////////////////
     if IsSameText(decltype,'action') then begin
      act:=fsm_add(sta,fsm_type_action,declname);
      if (fsm_type(act)=fsm_type_action) then begin
       // parse parameters ( ... )
       p:=Pos('(',declopts);
       if (p>0) then
       if (fsm_normalize_params(declopts,sv)=0)
       then AddParameters(act,sv)
       else ErrorFound('Invalid arguments of action: '+declname);
       // !visible: and another cookies
       if (regexp_exec(rexoptcooki,comm)>0) then begin
        if (StrFetch(comm,1)='!') then AddCookies(act);
       end;
      end else ErrorFound('Invalid declaration of action: '+declname);
      objset:=0; fun:=0; par:=0;
      ChangeContext(act);
     end else
     ///////////////////////////////////
     // objectset: NAME [OBJ1, OBJ2, ..]
     ///////////////////////////////////
     if IsSameText(decltype,'objectset') then begin
      objset:=fsm_add(dom,fsm_type_objectset,declname);
      if (fsm_type(objset)=fsm_type_objectset) then begin
       if (regexp_exec(rexisofclas,declopts)=1) then begin
        iv:=fsm_find(dom,fsm_type_class,regexp_matchstr(rexisofclas,1,2));
        if (fsm_type(iv)=fsm_type_class) or IsSameText(regexp_matchstr(rexisofclas,1,2),fsm_sym_void) then begin
         sNul(fsm_ctrl(objset,fsm_sym_is_of_class+'='+regexp_matchstr(rexisofclas,1,2)));
        end else ErrorFound('Invalid reference is_of_class: '+regexp_matchstr(rexisofclas,1,2));
       end;
       if (regexp_exec(rexunionmod,declopts)=1) then begin
        sNul(fsm_ctrl(objset,fsm_sym_unionmode+'=1'));
       end;
       if (regexp_exec(rexunionlst,declopts)=1) then begin
        sv:=regexp_matchstr(rexunionlst,1,1);
        if (regexp_exec(rexunioncsv,sv)>0) then begin
         sv:='';
         for j:=1 to regexp_matchnum(rexunioncsv,0) do
         if (regexp_matchstr(rexunioncsv,j,1)<>'') then begin
          if (sv<>'') then sv:=sv+',';
          sv:=sv+UpCaseStr(regexp_matchstr(rexunioncsv,j,1));
          //fsm_insert_in(objset,UpCaseStr(regexp_matchstr(rexunioncsv,j,1)));
         end;
         if (sv<>'') then sNul(fsm_ctrl(objset,fsm_sym_unionlist+'='+sv));
        end;
       end;
       // !color: and another cookies
       if (regexp_exec(rexoptcooki,comm)>0) then begin
        if (StrFetch(comm,1)='!') then AddCookies(objset);
       end;
      end else ErrorFound('Invalid declaration of objectset: '+declname);
      obj:=0; sta:=0; act:=0; fun:=0; par:=0;
      ChangeContext(objset);
     end else
     /////////////////////////////////////////////////////////////////////////
     // function: function-name [ (par1[, par2,..]) ]
     // Corresponds to object/class which was current before
     /////////////////////////////////////////////////////////////////////////
     if IsSameText(decltype,'function') then begin
      fun:=fsm_add(obj,fsm_type_function,declname);
      if (fsm_type(fun)=fsm_type_function) then begin
       // parse parameters ( ... )
       p:=Pos('(',declopts);
       if (p>0) then
       if (fsm_normalize_params(declopts,sv)=0)
       then AddParameters(fun,sv)
       else ErrorFound('Invalid arguments of function: '+declname);
       // !color: and another cookies
       if (regexp_exec(rexoptcooki,comm)>0) then begin
        if (StrFetch(comm,1)='!') then AddCookies(fun);
       end;
      end else ErrorFound('Invalid declaration of function: '+declname);
      objset:=0; sta:=0; act:=0; par:=0;
      ChangeContext(fun);
     end else
     //////////////////////////////
     // parameters: par1, par2, ...
     //////////////////////////////
     if IsSameText(decltype,'parameters') then begin
      if ((fsm_type(obj)=fsm_type_class) or (fsm_type(obj)=fsm_type_object)) and (fsm_normalize_params(declargs,sv)=0)
      then AddParameters(obj,sv)
      else ErrorFound('Invalid parameters of class/object: '+declargs);
      sta:=0; act:=0; par:=0;
      //ChangeContext(0);
     end;
    end; // for iline
    ChangeContext(0);
   end else ErrorFound('Invalid domain: '+domain);
  end else ErrorFound('Invalid fsm/text reference.');
  if (errors<>0) then fsm_trouble(fsm_errno_parse_sml,StrFmt('FSM found %d error(s) on parse SML.',errors));
  FreeLocals; ClearLocals;
  fsm_parse_sml:=errors;
 end;
 {
 FSM read from SML file (sml) in specified domain.
 }
 function fsm_read_sml(fsm:Integer; domain,sml:String):Integer;
 var errors,dom,txt:Integer; ms:Real;
  procedure ErrorFound(msg:String);
  begin
   errors:=errors+1;
   if (msg<>'') then Problem('fsm_read_sml: '+msg);
  end;
  procedure HandleTailComma(var s:String);
  var rex:Integer;
  begin
   rex:=regexp_init(0,'/[,]\s*'+regexp_escape(EOL)+'\s*/gs');
   if regexp_test(rex,s) then s:=regexp_replace(rex,s,',');
   bNul(regexp_free(rex));
  end;
  function ReadFileToString(filename:String):String;
  var buff:String;
  begin
   buff:='';
   filename:=Trim(filename);
   if FileExists(filename) then
   if (f_reset(filename,0)=0) then begin
    buff:=AdjustLineBreaks(f_read(Round(f_size))); bNul(f_close);
    if (IOResult<>0) then ErrorFound('Error read file '+filename);
   end else ErrorFound('Error open file '+filename);
   if fsm_handletailcomma then HandleTailComma(buff);
   ReadFileToString:=buff;
   buff:='';
  end;
 begin
  errors:=0; ms:=msecnow;
  sml:=Trim(sml); domain:=Trim(domain);
  dom:=fsm_find(fsm,fsm_type_domain,domain);
  if (fsm_type(dom)=fsm_type_domain) then begin
   if not IsEmptyStr(sml) then begin
    sml:=DaqFileRef(sml,'.sml');
    if FileExists(sml) then begin
     txt:=StringToText(ReadFileToString(sml));
     if (text_numln(txt)>0) then begin
      Success(StrFmt('SML file "%s"',sml)+StrFmt(' has %d line(s).',text_numln(txt)));
      fsm_set_cookie(dom,'SML_SOURCE_FILE',sml);
      errors:=errors+fsm_parse_sml(fsm,txt,domain);
      fsm_set_cookie(dom,'SML_READ_ERRORS',Str(errors));
      Success(StrFmt('SML file parsing took %1.0f ms.',msecnow-ms));
     end else ErrorFound('Could not read '+sml);
     bNul(text_free(txt));
    end else ErrorFound('Missed file '+sml);
   end else ErrorFound('Empty SML file name.');
  end else ErrorFound('Invalid domain '+domain);
  if (errors<>0) then fsm_trouble(fsm_errno_read_sml,StrFmt('FSM found %d error(s) on read SML.',errors));
  fsm_read_sml:=errors;
 end;
 {
 FSM initialization with argument list (arg).
  -dns n         - use DIM_DNS_NODE node (n)
  -dom d         - use domain name (d)
  -sml f         - load SML file (f)
  -verbose n     - use verbose mode (n)
  -listing n     - use listing mode (n)
  -name s        - use FSM name (s)
 Example:
  fsm:=fsm_init('-dns localhost -dom DEMO -sml run_con.sml');
  fsm:=fsm_init('-dns=localhost -dom=DEMO -sml=run_con.sml');
 }
 function fsm_init(arg:String):Integer;
 var fsm,ipar:Integer; dns,dom,sml,opt,par,lst:String;
  procedure ClearLocals;
  begin
   dns:=''; dom:=''; sml:=''; opt:=''; par:=''; lst:='';
  end;
  procedure ErrorFound(msg:String);
  begin
   fsm_trouble(fsm_errno_init,'fsm_init: '+msg);
  end;
  function WriteStringToFile(filename,data:String):Boolean;
  var flag:Boolean; n:Integer;
  begin
   flag:=false;
   filename:=Trim(filename);
   if (data<>'') then
   if DirExists(ExtractFilePath(filename)) then
   if (f_rewrite(filename,1)=0) then begin
    data:=AdjustLineBreaks(data);
    n:=f_write(data);
    bNul(f_close);
    if (IOResult<>0) or (n<>Length(data))
    then ErrorFound('Error write file '+filename)
    else flag:=true;
   end else ErrorFound('Error create file '+filename);
   WriteStringToFile:=flag;
  end;
 begin
  ClearLocals;
  fsm:=fsm_new;
  fsm_set_cookie(fsm,fsm_sym_hostname,ParamStr(fsm_sym_hostname));
  fsm_set_cookie(fsm,fsm_sym_computername,ParamStr(fsm_sym_computername));
  fsm_set_cookie(fsm,fsm_sym_dim_dns_node,fsm_validate_dns(fsm,GetEnv(UpcaseStr(fsm_sym_dim_dns_node))));
  ////////////////////////////////
  ipar:=1; // Parse argument line:
  while (ipar<=WordCount(arg)) do begin
   opt:=ExtractWord(ipar,arg);
   par:=ExtractWord(ipar+1,arg);
   if IsOption(opt,'') then begin
    if IsOption(opt,'-dns,--dns,-dns_node,--dns_node') then begin
     if (GetOptionValue(opt)<>'') then begin
      par:=Trim(GetOptionValue(opt));
      dns:=fsm_validate_dns(fsm,par);
      fsm_set_cookie(fsm,fsm_sym_dns_node,dns);
     end else
     if not IsOption(par,'') and (par<>'') then begin
      dns:=fsm_validate_dns(fsm,Trim(par));
      fsm_set_cookie(fsm,fsm_sym_dns_node,dns);
      ipar:=ipar+1;
     end;
    end else
    if IsOption(opt,'-dom,--dom,-domain,--domain') then begin
     if (GetOptionValue(opt)<>'') then begin
      dom:=Trim(GetOptionValue(opt));
     end else
     if not IsOption(par,'') and (par<>'') then begin
      dom:=Trim(par);
      ipar:=ipar+1;
     end;
     if fsm_is_valid_name(dom,fsm_type_domain) then begin
      if (fsm_add(fsm,fsm_type_domain,dom)=0)
      then ErrorFound('could not add domain '+dom);
     end;
    end else
    if IsOption(opt,'-sml,--sml,-sml_file,--sml_file') then begin
     if (GetOptionValue(opt)<>'') then begin
      sml:=Trim(GetOptionValue(opt));
     end else
     if not IsOption(par,'') and (par<>'') then begin
      sml:=Trim(par);
      ipar:=ipar+1;
     end;
     if not IsEmptyStr(sml) then begin
      if (fsm_read_sml(fsm,dom,sml)>0)
      then ErrorFound('error read '+sml);
     end;
    end else
    if IsOption(opt,'-verbose,--verbose,-verb,--verb') then begin
     if (GetOptionValue(opt)<>'') then begin
      par:=Trim(GetOptionValue(opt));
      fsm_verbose_mode:=iValDef(par,fsm_verbose_mode);
     end else begin
      fsm_verbose_mode:=iValDef(par,fsm_verbose_mode);
      ipar:=ipar+1;
     end;
    end else
    if IsOption(opt,'-lst,--lst,-listing,--listing') then begin
     if (GetOptionValue(opt)<>'') then begin
      par:=Trim(GetOptionValue(opt));
      fsm_listing_mode:=iValDef(par,fsm_listing_mode);
     end else begin
      fsm_listing_mode:=iValDef(par,fsm_listing_mode);
      ipar:=ipar+1;
     end;
    end else
    if IsOption(opt,'-name,--name') then begin
     if (GetOptionValue(opt)<>'') then begin
      par:=Trim(GetOptionValue(opt));
      sNul(fsm_ctrl(fsm,'name='+par));
     end else begin
      sNul(fsm_ctrl(fsm,'name='+par));
      ipar:=ipar+1;
     end;
    end else
    ErrorFound('invalid option '+opt);
   end;
   ipar:=ipar+1;
  end;
  {}
  if (fsm_listing_mode>0) then begin
   lst:=DevName+'_'+fsm_name(fsm);
   lst:=StringReplace(lst,'&','_',rfReplaceAll);
   lst:=StringReplace(lst,'.','_',rfReplaceAll);
   if (GetEnv('CRW_DAQ_CONFIG_TEMP_PATH')<>'')
   then lst:=AddPathDelim(GetEnv('CRW_DAQ_CONFIG_TEMP_PATH'))+AdaptFileName(lst)
   else lst:=AdaptFileName('..\temp\'+lst);
   lst:=DaqFileRef(lst,'.lst');
   par:=fsm_readback_sml(fsm);
   if WriteStringToFile(lst,par) then Success('Listing: '+lst);
  end;
  {}
  ClearLocals;
  fsm_init:=fsm;
 end;
 {
 FSM kill, i.e. free and zero reference.
 }
 procedure fsm_kill(var fsm:Integer);
 begin
  if (fsm<>0) then begin
   if (fsm_type(fsm)=fsm_type_manager)
   then bNul(fsm_free(fsm));
   fsm:=0;
  end;
 end;
 {
 Clear FsmManager.
 }
 procedure ClearFsmManager;
 begin
  fsm_verbose_mode:=0;
  fsm_listing_mode:=0;
  fsm_errno_get_param:=0;
  fsm_errno_set_param:=0;
  fsm_errno_parse_sml:=0;
  fsm_errno_read_sml:=0;
  fsm_errno_init:=0;
 end;
 {
 Initialize FsmManager.
 }
 procedure InitFsmManager;
 begin
  fsm_tablen:=4;
  fsm_rempos:=59;
  fsm_verbose_mode:=1;
  fsm_listing_mode:=1;
  fsm_handletailcomma:=true;
  ShouldPollFsmManager:=true;
  fsm_errno_init:=RegisterErr('FSM init');
  fsm_errno_read_sml:=RegisterErr('FSM read SML');
  fsm_errno_get_param:=RegisterErr('FSM get param');
  fsm_errno_set_param:=RegisterErr('FSM set param');
  fsm_errno_parse_sml:=RegisterErr('FSM parse SML');
  fsm_verbose_mode:=iValDef(ReadIni('fsm_verbose_mode'),fsm_verbose_mode);
  fsm_listing_mode:=iValDef(ReadIni('fsm_listing_mode'),fsm_listing_mode);
 end;
 {
 Finalize FsmManager.
 }
 procedure FreeFsmManager;
 begin
 end;
 {
 Poll FsmManager.
 }
 procedure PollFsmManager;
 begin
 end;
/////////////////
// END OF FILE //
/////////////////
