#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################
# CRW-DAQ script to edit DaqPascal program code text.
##########################################################

################################
# Current script identification.
################################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

function print_comment(){
 echo "***************************************************";
 echo "CRW-DAQ script to edit DaqPascal program code text.";
 echo "The script expected location: /opt/crwdaq/resource.";
 echo "Param[1]  - program source file full path name.    ";
 echo "Param[2]  - DAQ program device name.               ";
 echo "--sid s   - session identifier name (s).           ";
 echo "--row r   - row/line number (r) started from 1.    ";
 echo "--col c   - column   number (c) started from 1.    ";
 echo "--err n   - compiler code (n) to identify error.   ";
 echo "--msg s   - compiles message (s) to identify error.";
 echo "***************************************************";
};

function fatal(){
 1>&2 echo "$2";
 exit $1;
};

function check_environ(){
 if [ -z "$CRW_DAQ_SYS_HOME_DIR" ]; then
  fatal 1 "Error: script should be called from CRW-DAQ system only.";
 fi;
};

readonly datadir="$HOME/.local/share/daqgroup/daqpaseditor";
declare -i paramcount=0;
declare SourceFileName="";
declare DeviceName="";
declare sid="";
declare row="";
declare col="";
declare err="";
declare msg="";

#######################################################################################################
# get/set INI file parameter:
# git_ini -get inifile section key
# git_ini -set inifile section key value
#######################################################################################################
function git_ini(){
 local op="$1"; local ini="$2"; local sec="$3"; local key="$4"; local val="$5";
 case $op in
  -get|--get) if [ $# -eq 4 ] && [ -e "$ini" ];                then git config -f $ini $sec.$key;        else return 1; fi; ;;
  -set|--set) if [ $# -eq 5 ] && mkdir -p "$(dirname "$ini")"; then git config -f $ini $sec.$key "$val"; else return 1; fi; ;;
  *) return 1; ;;
 esac;
};

######################################################################################################
# git config section and key could not contain punctuation chars (except -), so we need to replace it.
######################################################################################################
function name_to_sec(){ if [ -n "$1" ]; then echo "$1" | sed -e 's/[[:punct:]]/-/g' -e 's/^-*//' -e 's/-*$//'; fi; };
function name_to_key(){ if [ -n "$1" ]; then echo "$1" | sed -e 's/[[:punct:]]/-/g' -e 's/^-*//' -e 's/-*$//'; fi; };

##############################
# read_ini inifile section key
# value '-' replace to empty
##############################
function read_ini(){
 if [ $# -ge 3 ]; then
  local ini="$1";
  local sec="$(name_to_sec "$2")";
  local key="$(name_to_key "$3")";
  if [ "x$ini" = "x" ]; then return 1; fi;
  if [ "x$sec" = "x" ]; then return 1; fi;
  if [ "x$key" = "x" ]; then return 1; fi;
  local val="$(git_ini -get "$ini" "$sec" "$key")";
  if [ "x$val" = "x" ]; then return 1; fi;
  if [ "$val" = "-" ]; then return 0; fi;
  echo -n "$val";
  return 0;
 fi;
 return 1;
};

####################################
# save_ini inifile section key value
# empty values replace to '-' marker
####################################
function save_ini(){
 if [ $# -ge 3 ]; then
  local ini="$1";
  local sec="$(name_to_sec "$2")";
  local key="$(name_to_key "$3")";
  local val="$4"; val="${val:--}";
  if [ "x$ini" = "x" ]; then return 1; fi;
  if [ "x$sec" = "x" ]; then return 1; fi;
  if [ "x$key" = "x" ]; then return 1; fi;
  if git_ini -set "$ini" "$sec" "$key" "$val"; then return 0; fi;
 fi;
 return 1;
};

################################
# save current state to INI file
################################
function save_current_state(){
 if [ -n "$datadir" ] && [ -n "$sid" ]; then
  if mkdir -p "$datadir"; then
   local ini="$datadir/$sid.ini";
   local sec="$(realpath $SourceFileName)";
   save_ini $ini $sec SourceFileName "$SourceFileName";
   save_ini $ini $sec DeviceName     "$DeviceName";
   save_ini $ini $sec sid            "$sid";
   save_ini $ini $sec row            "$row";
   save_ini $ini $sec col            "$col";
   save_ini $ini $sec err            "$err";
   save_ini $ini $sec msg            "$msg";
   save_ini $ini $sec TimeStamp      "$(date +%s.%N)";
   for v in $(printenv | grep 'CRW_DAQ' | cut -d '=' -f 1 | xargs); do
    save_ini $ini $sec $v "$(printenv $v)";
   done;
  fi;
 fi;
};

##################################
# tooltip report on compiler error
##################################
function report_compiler_error(){
 if [ -n "$row" ] && [ -n "$col" ] && [ -n "$err" ] && [ -n "$msg" ]; then
  local fn="$(basename $SourceFileName)";
  unix tooltip-notifier \
       head "$sid: @daq compile $DeviceName" \
       text "Error($err): «$msg».\nIn $fn at $row:$col." \
       preset stdError delay 60000;
 fi;
};

#######################################
# column_to_offset row col filename
# convert 1-based column to byte offset
#######################################
function column_to_offset(){
 local row="$1"; local col="$2"; local fn="$3";
 if [ -n "$fn" ] && [ -e "$fn" ]; then
  local line="$(cat "$fn" | sed -n "${row}p")";
  if [ -n "$line" ]; then
   local num=0;
   let num=$col-1;
   line="${line:0:$num}";
   echo -n "$line" | wc -c;
   return 0;
  fi;
 fi;
 echo "$col";
};

##############################
# handle positional parameters
##############################
function handle_param(){
 if [ -n "$1" ]; then
  let paramcount++;
  case $paramcount in
   1) SourceFileName="$1"; ;;
   2) DeviceName="$1"; ;;
   *) ;;
  esac;
 fi;
};

##############################
# parse command line arguments
##############################
function parse_cmdline(){
 local npar=0;
 while [ -n "$1" ]; do
  case $1 in
   -s|-sid|--sid) shift; sid="$1"; ;;
   -r|-row|--row) shift; row="$1"; ;;
   -c|-col|--col) shift; col="$1"; ;;
   -e|-err|--err) shift; err="$1"; ;;
   -m|-msg|--msg) shift; msg="$1"; ;;
   -*) fatal 1 "Error: invalid option $1."; ;;
   *) handle_param $1; ;;
  esac;
  shift;
 done;
};

#######################################
# find editor name and his command line
#######################################
declare editor="";
declare editor_cmd="";

function find_editor(){
 for item in $*; do
  if which $item >/dev/null 2>&1; then
   editor="$item";
   editor_cmd="$editor";
   if [ "$editor" = "kate" ]; then
    if [ -n "$sid" ]; then editor_cmd="$editor_cmd --start $sid"; fi;
    if [ -n "$row" ]; then editor_cmd="$editor_cmd --line $row"; fi;
    if [ -n "$col" ]; then editor_cmd="$editor_cmd --column $col"; fi;
    return 0;
   fi;
   if [ "$editor" = "geany" ]; then
    if [ -n "$sid" ]; then
     $scriptHOME/tools/geany/cp-config.sh;
     local cfg="$HOME/.config/daqgroup/geany/geany_$sid";
     if [ -e "$cfg/geany.conf" ]; then editor_cmd="$editor_cmd --config $cfg"; fi;
    fi;
    if [ -n "$row" ]; then editor_cmd="$editor_cmd --line $row"; fi;
    if [ -n "$col" ]; then
     # geany use --column as byte offset instead of 1-based char position
     # so we have to convert given column (col) to 0-based byte offset (num)
     local num="$(column_to_offset $row $col $SourceFileName)";
     editor_cmd="$editor_cmd --column $num";
    fi;
    return 0;
   fi;
  fi;
 done;
 return 1;
};

#########################
# report on missed editor
#########################
function missed_editor(){
 if [ -z "$editor" ]; then
  fatal 1 "Error: could not find editor.";
 fi;
};

######################
# execute found editor
######################
function launch_editor(){
 local cmdline="$editor_cmd $SourceFileName";
 echo "$cmdline"; echo "";
 exec  $cmdline;
};

######################
# main function to run
######################
function main(){
 print_comment;
 check_environ;
 parse_cmdline "$@";
 save_current_state;
 if find_editor geany kate; then
  report_compiler_error;
  launch_editor;
 else
  missed_editor;
 fi;
};

main "$@";

#############
# END OF FILE
#############
