#!/bin/bash

##########################################################
# Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru
##########################################################
# CRW-DAQ script to call on Logon.
##########################################################

function print_comment(){
 echo "****************************************************";
 echo "CRW-DAQ Initialization script. Run on CRW-DAQ logon.";
 echo "****************************************************";
};

################################
# Current script identification.
################################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

readonly logfile="$CRW_DAQ_VAR_TMP_DIR/$scriptbase.log";

readonly crwdaqEXE="$(realpath $CRW_DAQ_SYS_EXE_FILE)";

function fatal(){
 1>&2 echo "$2";
 exit $1;
};

function check_environ(){
 if [ -z "$CRW_DAQ_VAR_TMP_DIR" ] || [ -z "$CRW_DAQ_SYS_EXE_FILE" ]; then
  fatal 1 "Error: script should be called from CRW-DAQ system only.";
 fi;
};

function enter_logfile(){
 echo ""                                              >> $logfile;
 echo "$(date +%Y.%m.%d-%H:%M:%S): Enter $scriptfile" >> $logfile;
};

function leave_logfile(){
 echo "$(date +%Y.%m.%d-%H:%M:%S): Leave $scriptfile" >> $logfile;
 echo ""                                              >> $logfile;
};

function get_file_size(){
 if [ -e "$1" ]; then
  stat -c %s "$1";
 else
  echo "0";
 fi;
};

function delete_files(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then rm -fv "$1"; fi;
  shift;
 done;
};

function cleanup_dirs(){
 let KB=1024; let MB=$KB*$KB; let maxsize=32*$MB;
 echo "$(date +%Y.%m.%d-%H:%M:%S): Cleanup dirs: $maxsize $*";
 for dir in $*; do
  if [ -d "$dir" ]; then
   for fn in $(ls -1 "$dir"); do
    local file="$dir/$fn";
    if [ -f "$file" ]; then
     if [ "$file" = "$logfile" ]; then continue; fi;
     local size="$(get_file_size $file)";
     if [ $size -gt $maxsize ]; then
      old="$dir/obsolete";
      mkdir -p "$old";
      mv -fv "$file" "$old/$fn";
     fi;
    fi;
   done;
  fi;
 done;
};

function cleanup_files(){
 $scriptHOME/crwdaqlogroll.sh; return; # skip obsolete log files cleaner
 cleanup_dirs "$CRW_DAQ_VAR_TMP_DIR" "$CRW_DAQ_SYS_TMP_DIR" >> $logfile;
 delete_files $HOME/.local/share/daqgroup/daqpaseditor/crwdaq_$CRW_DAQ_SYS_SESSION_NB.ini >> $logfile;
};

#######################
# check file integrity.
#######################
function chk_integrity(){
 $scriptHOME/crwdaq.verify.sh;
};

#############################
# check user has sudo rights.
#############################
function is_iam_sudo(){
 unix sudoit -v </dev/null >/dev/null 2>&1;
};

###########################
# find binary file by name.
###########################
function whereis_bin(){
 whereis -b $1 | sed 's/.*://' | xargs;
};

##########################
# read capabilities of $1.
##########################
function read_caps(){
 local getcap="$(whereis_bin getcap)";
 if [ ! -e "$getcap" ]; then getcap="unix sudoit getcap"; fi;
 $getcap $1 | sed 's/.*=//' | xargs;
};

#######################################
# verify capabilities $1 of program $2.
#######################################
function verify_caps(){
 local setcap="$(whereis_bin setcap)";
 if [ ! -e "$setcap" ]; then setcap="unix sudoit setcap"; fi;
 $setcap -v $1 $2 >/dev/null 2>&1;
};

###################################################################
# check executable capabilities, set wanded capabilities of crwdaq.
# https://www.protokols.ru/WP/wp-content/uploads/2021/01/setcap.pdf
###################################################################
function chk_exec_caps(){
 if is_iam_sudo; then
  echo "User $USER has sudo rights.";
  if [[ -x $scriptHOME/crwdaq.setcap.sh ]]; then
   echo "Exec: crwdaq.setcap.sh";
   if $scriptHOME/crwdaq.setcap.sh; then
    echo "Done: crwdaq.setcap.sh";
    return;
   fi;
   echo "Fail: crwdaq.setcap.sh";
  fi;
  # if modern crwdaq.setcap.sh fail, use old as fallback:
  local wanted_caps="cap_net_bind_service,cap_sys_nice+ep";
  if verify_caps $wanted_caps $crwdaqEXE; then
   echo "$(basename $crwdaqEXE) already have: $wanted_caps";
  else
   echo "Try set caps $wanted_caps to $(basename $crwdaqEXE)";
   unix sudoit setcap $wanted_caps $crwdaqEXE;
  fi;
  local actual_caps="$(read_caps $crwdaqEXE)";
  echo "$(basename $crwdaqEXE) capabilities: $actual_caps";
 else
  echo "User $USER has NO sudo rights.";
 fi;
};

function list_groups(){ getent group | cut -d: -f1; };
function exist_group(){ if [[ -n "$1" ]] && getent group "$1" >/dev/null 2>&1; then return 0; else return 1; fi; };

##############################
# check user group membership.
##############################
function chk_usr_group(){
 local wanted_groups="dialout firebird";
 for w in $wanted_groups; do
  if exist_group "$w"; then true; else continue; fi;
  local n=0; for g in $(groups); do if [ "$g" = "$w" ]; then let n++; fi; done;
  if [ $n -eq 0 ]; then echo "Add $USER to group $w"; unix sudoit usermod -a -G $w $USER; fi;
 done;
 echo "$USER groups: $(groups)";
};

#####################################
# make directory for works (projects)
#####################################
function mkdir_works(){
 unix sudoit mkdir -v --mode=777 $(realpath $scriptHOME/../../../works);
};

####################
# run crwdaq watcher
####################
function run_watcher(){
 if unix grun $scriptHOME/crwdaqwatcher.sh $CRW_DAQ_SYS_EXE_PID $CRW_DAQ_SYS_SESSION_HEAD; then
  echo "Started crwdaqwatcher.sh";
 else
  echo "Failed crwdaqwatcher.sh";
 fi;
};

###################################
# print default handler for MIME $1
###################################
function default_app(){
 local mime="$1";
 if [ -n "$mime" ] && which gio >/dev/null 2>&1; then
  env LANG=en gio mime $mime | grep 'Default application ' | sed 's/.*://' | xargs;
 fi;
};

#########################################
# check $2 is default MIME handler for $1
# if it is not, set $2 as default handler
#########################################
function update_assoc(){
 local mime="$1"; local app="$2";
 if [ -n "$mime" ] && [ -n "$app" ] && which gio >/dev/null 2>&1; then
  local def="$(default_app "$mime")";
  if [ -z "$def" ] || [ "x$def" != "x$app" ]; then gio mime $mime $app; fi;
 fi;
};

#############################
# check/fix MIME associations
#############################
function chk_assoc(){
 if which CrossMachine >/dev/null 2>&1; then
  update_assoc application/x-diesel CrossMachine.desktop;
 fi;
 if unix root --version 2>&1; then
  update_assoc application/x-root RootOpen.desktop;
 fi;
};

##########################
# make some files writable
##########################
function chmod_x_files(){
 if [[ -n "$CRW_DAQ_SYS_HOME_DIR" ]] && [[ -d "$CRW_DAQ_SYS_HOME_DIR" ]]; then
  unix sudoit chmod -c  ugo+rw  $CRW_DAQ_SYS_HOME_DIR/resource/manual/crw-daq-colors.htm;
  unix sudoit chmod -c  ugo+rwx $(find $CRW_DAQ_SYS_HOME_DIR/demo -type d | xargs);
  unix sudoit chmod -cR ugo+rw  $CRW_DAQ_SYS_HOME_DIR/demo/*;
  unix sudoit chmod -c  ugo+rwx $CRW_DAQ_SYS_HOME_DIR/resource/daqsite/stdlib/bitmaps;
  unix sudoit chmod -c  ugo+rwx $CRW_DAQ_SYS_HOME_DIR/settings;
  unix sudoit chmod -cR ugo+rw  $CRW_DAQ_SYS_HOME_DIR/settings/*;
 fi;
};

##########################
# perform required actions
##########################
function perform_actions(){
 print_comment;
 cleanup_files;
 chk_integrity;
 chk_exec_caps;
 chk_usr_group;
 mkdir_works;
 run_watcher;
 chk_assoc;
 chmod_x_files;
};

################
# main function.
################
function main(){
 check_environ;
 enter_logfile;
 perform_actions | tee -a $logfile;
 leave_logfile;
};

main "$@";

#############
# END OF FILE
#############
