 {
 ***********************************************************************
 Daq Pascal application program WJ300A_DRV.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @PrintCmdTable   - Print Command Table state.
| @Setting id v    - Send command id with data v, id=(SRS).
| @Edit id         - edit parameter and write it
| @AssignTag t v   - Assign tag t to value v
| @DimTagUpdate    - Update tag from DIM
| @LoadIni         - load params from INI file.
| @SaveIni         - save params to   INI file.
| @GetWaveForm     - Receipt waveform
| @Autoset         - Automatically adjust oscilloscope
| @TrigRun         - Command sets the oscilloscope to run.
| @TrigStop        - Command sets the oscilloscope to stop
| @SingleStart     - Set trigger to single mode
| @SingleStop      - Ser trigger to auto mode
| @SetDirBase arg  - Set base directory name
| @SetDirExp arg   - Set experiment directory name
|********************************************************
[]
 }
program WJ300A_DRV;              { WJ300A driver                    }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 st_NoReq                 = 0;   { Status : No request sent         }
 st_WaitAns               = 1;   { Status : Waiting answer          }
 st_WaitGap               = 2;   { Status : Waiting time gap        }
 st_TimeOut               = 3;   { Status : TimeOut found           }
 {------------------------------}{                                  }
 MaxCmdNum                = 41;  { Max command id number:           }
 cm_ACQ                   = 1;   {                                  }
 cm_CPL_C1                = 2;   {                                  }
 cm_CPL_C2                = 3;   {                                  }
 cm_CPL_C3                = 4;   {                                  }
 cm_CPL_C4                = 5;   {                                  }
 cm_CLS                   = 6;   {                                  }
 cm_DATE                  = 7;   {                                  }
 cm_DTBORD                = 8;   {                                  }
 cm_DTFORM                = 9;   {                                  }
 cm_DTSTART               = 10;  {                                  }
 cm_DTPOINTS              = 11;  {                                  }
 cm_DTWAVE                = 12;  {                                  }
 cm_ESR                   = 13;  {                                  }
 cm_IDN                   = 14;  {                                  }
 cm_MLEN                  = 15;  {                                  }
 cm_OFST_C1               = 16;  {                                  }
 cm_OFST_C2               = 17;  {                                  }
 cm_OFST_C3               = 18;  {                                  }
 cm_OFST_C4               = 19;  {                                  }
 cm_OFST_M1               = 20;  {                                  }
 cm_RST                   = 21;  {                                  }
 cm_TCPL                  = 22;  {                                  }
 cm_TDIV                  = 23;  {                                  }
 cm_TESR                  = 24;  {                                  }
 cm_TLVL                  = 25;  {                                  }
 cm_TRA_C1                = 26;  {                                  }
 cm_TRA_C2                = 27;  {                                  }
 cm_TRA_C3                = 28;  {                                  }
 cm_TRA_C4                = 29;  {                                  }
 cm_TRA_M1                = 30;  {                                  }
 cm_TRDL                  = 31;  {                                  }
 cm_TRMD                  = 32;  {                                  }
 cm_TSLP                  = 33;  {                                  }
 cm_TSRC                  = 34;  {                                  }
 cm_TTYP                  = 35;  {                                  }
 cm_VDIV_C1               = 36;  {                                  }
 cm_VDIV_C2               = 37;  {                                  }
 cm_VDIV_C3               = 38;  {                                  }
 cm_VDIV_C4               = 39;  {                                  }
 cm_VDIV_M1               = 40;  {                                  }
 cm_WAVESRC               = 41;  {                                  }
 {------------------------------}{ Analog outputs:                  }
 ao_POLL_RATE             = 0;   { Polling rate, poll/sec           }
 ao_ERROR_CNT             = 1;   { Device error counter             }
 ao_ESR                   = 2;   {                                  }
 ao_OFST_C1               = 3;   {                                  }
 ao_OFST_C2               = 4;   {                                  }
 ao_OFST_C3               = 5;   {                                  }
 ao_OFST_C4               = 6;   {                                  }
 ao_OFST_M1               = 7;   {                                  }
 ao_TDIV                  = 8;   {                                  }
 ao_TLVL                  = 9;   {                                  }
 ao_TRDL                  = 10;  {                                  }
 ao_VDIV_C1               = 11;  {                                  }
 ao_VDIV_C2               = 12;  {                                  }
 ao_VDIV_C3               = 13;  {                                  }
 ao_VDIV_C4               = 14;  {                                  }
 ao_VDIV_M1               = 15;  {                                  }
 DelayOnStart             = 1000;{ Delay before start polling       }
 DfltDir                  = '..\..\WJ300A_DATA\WAVEFORMS\';{        }
 DimDeadline              = 5000; { Detect DIM server is dead       }
 stt_wfNone               = 0;   { Waveform request statuses:       }
 stt_wfSetSource          = 1;   {  Set data source                 }
 stt_wfChkSource          = 2;   {  Check data source               }
 stt_wfGetData            = 3;   {  Get wavedata                    }
 stt_wfIncSource          = 4;   {  Increment source number         }
 stt_wfGetSource          = 5;   {  Get source number               }
 stt_wfDone               = 6;   {  Waveform received success       }
 stt_trigNone             = 0;   { Trigger request status:          }
 stt_trigSetSingle        = 1;   {  Set edge trigger mode Single    }
 stt_trigReady            = 2;   {  Get edge trigger status Ready   }
 stt_trigDone             = 3;   {  trigger done                    }
 tm_wfSetSource           = 10000; { Data source setting timeout    }
 tm_trigSetSingle         = 15000; { Trigger mode setting timeout   }
 DataFlag                 = 129;
 HeaderLength             = 8;
 WaveformOffset           = 8;
 PrNormal                 = 1;   { CMD poll priority normal         }
 PrHigh                   = 2;   { CMD poll while movement          }
type
 {------------------------------}{ Declare uses program types:      }
 {$I _typ_StdLibrary}            { Include all Standard types,      }
 {------------------------------}{ And add User defined types       }
 TChanRec = record               { Channel data                     }
  COUP    : TTagRef;             {  Coupling                        }
  IMP     : TTagRef;             {  Impedance                       }
  OFFS    : TTagRef;             {  Offset                          }
  SCAL    : TTagRef;             {  Scale                           }
  TRA     : TTagRef;             {  Trace                           }
  WAVFORM : TTagRef;             {  Waveform                        }
 end;                            {                                  }
 {------------------------------}{                                  }
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 WJ300A            : record      { WJ300A data                      }
  Simulator        : Boolean;    { Simulator mode                   }
  ecTimeOut        : Integer;    { Error code: TimeOut              }
  IsWaveFormDataRequest : Boolean;{}
  PollPriority          : Integer;{}
  GetWaveFormStatus     : Integer;{}
  GetWaveFormSource     : Integer;{}
  GetWaveFormTimer      : Real;   {}
  CalcDevRef        : Integer;   {                                  }
  GetTrigStatus     : Integer;   {                                  }
  GetTrigStatusTimer: Real;      {                                  }
  TagPrefix         : String;    {                                  }
  FilePrefix        : String;    {                                  }
  Pipe             : record      { Pipe port data                   }
   PipeId          : Integer;    {  Pipe reference                  }
   PipeSIM         : Integer;    {  Local PIPE for Simulation       }
   Status          : Integer;    {  Status flag                     }
   TimeOut         : Integer;    {  TimeOut, ms                     }
   TimeGap         : Integer;    {  Time delay after poll, ms       }
   PollPeriod      : Integer;    { Poll period, ms                  }
   Req,Ans,Buf     : String;     {  Request, Answer                 }
   ReqTime,AnsTime : Real;       {  Request, Answer time, ms        }
   LastPoll        : Real;       { Time of last poll, ms            }
   PollRate        : Integer;    { Poll rate, poll/sec              }
  end;                           {                                  }
  Cmd              : record      { Command cycle data               }
   Num             : Integer;    {  Current command number          }
   Query           : array [1..MaxCmdNum] of Boolean; {             }
   Enabled         : array [1..MaxCmdNum] of Boolean; {             }
   Priority        : array [1..MaxCmdNum] of Integer; {             }
   Acronym         : array [1..MaxCmdNum] of String;  {             }
   OpData          : array [1..MaxCmdNum] of String;  {             }
   OpBuff          : array [1..MaxCmdNum] of String;  {             }
   AoNum           : array [1..MaxCmdNum] of Integer; {             }
   SimData         : array [1..MaxCmdNum] of String;  {             }
  end;                           {                                  }
  DIRBASE          : TTagRef;    { Base directory                   }
  DIREXP           : TTagRef;    { Experiment directory             }
  DEFAULTDIR       : TTagRef;    { Default base directory           }
  EXTRACTNUM       : TTagRef;    { Extraction number                }
  SAVESTAT         : TTagRef;    { Save status                      }
  CLOCK            : TTagRef;    { Server date time                 }
  SERVID           : TTagRef;    { Server identifier                }
  POLL             : record      { Poll parameters:                 }
   ENABLE          : TTagRef;    { Poll enable flag                 }
   RATE            : TTagRef;    {                                  }
  end;                           {                                  }
  ERROR            : record      {                                  }
   COUNT           : TTagRef;    {                                  }
  end;                           {                                  }
  C1,C2,C3,C4,M1   : TChanRec;   { Channels parametrs               }
  PARAM            : record      { Device parameters:               }
   ACQ             : TTagRef;    {                                  }
   DATE            : TTagRef;    {                                  }
   DTBORD          : TTagRef;    {                                  }
   DTFORM          : TTagRef;    {                                  }
   DTSTART         : TTagRef;    {                                  }
   DTPOINTS        : TTagRef;    {                                  }
   DTWAVE          : TTagRef;    {                                  }
   ESR             : TTagRef;    {                                  }
   IDN             : TTagRef;    {                                  }
   MLEN            : TTagRef;    {                                  }
   TCPL            : TTagRef;    {                                  }
   TDIV            : TTagRef;    {                                  }
   TESR            : TTagRef;    {                                  }
   TLVL            : TTagRef;    {                                  }
   TRDL            : TTagRef;    {                                  }
   TRG             : TTagRef;    {                                  }
   TRMD            : TTagRef;    {                                  }
   TSLP            : TTagRef;    {                                  }
   TSRC            : TTagRef;    {                                  }
   TTYP            : TTagRef;    {                                  }
   WAVESRC         : TTagRef;    {                                  }
  end;                           {                                  }
  SIM              : record      { Simulator data:                  }
   C1,C2,C3,C4,C5   : TChanRec;  { Channels parametrs               }
   PARAM            : record     { Device parameters:               }
    ACQ             : Integer;   {                                  }
    DATE            : Integer;   {                                  }
    DTBORD          : Integer;   {                                  }
    DTFORM          : Integer;   {                                  }
    DTSTART         : Integer;   {                                  }
    DTPOINTS        : Integer;   {                                  }
    DTWAVE          : Integer;   {                                  }
    ESR             : Integer;   {                                  }
    IDN             : Integer;   {                                  }
    MLEN            : Integer;   {                                  }
    TCPL            : Integer;   {                                  }
    TDIV            : Real;      {                                  }
    TESR            : Integer;   {                                  }
    TLVL            : Real;      {                                  }
    TRDL            : Real;      {                                  }
    TRG             : Integer;   {                                  }
    TRMD            : Integer;   {                                  }
    TSLP            : Integer;   {                                  }
    TSRC            : Integer;   {                                  }
    TTYP            : Integer;   {                                  }
    WAVESRC         : Integer;   {                                  }
   end;                          {                                  }
  end;                           {                                  }
  SelfId           : String;     { Self pid@hostname                }
 end;                            {                                  }
 cmd_PrintCmdTable : Integer;    { @PrintCmdTable                   }
 cmd_Setting       : Integer;    { @Setting                         }
 cmd_Edit          : Integer;    { @Edit                            }
 cmd_DimTagUpdate  : Integer;    { @DimTagUpdate                    }
 cmd_AssignTag     : Integer;    { @AssignTag                       }
 cmd_LoadIni       : Integer;    { @LoadIni                         }
 cmd_SaveIni       : Integer;    { @SaveIni                         }
 cmd_GetWaveForm   : Integer;    { @GetWaveForm                     }
 cmd_Autoset       : Integer;    { @Autoset                         }
 cmd_TrigRun       : Integer;    { @TrigRun                         }
 cmd_TrigStop      : Integer;    { @TrigStop                        }
 cmd_DimCmdMy      : Integer;    { System Command @DimCmdMy         }
 cmd_SingleStart   : Integer;    { @SingleStart                     }
 cmd_SingleStop    : Integer;    { @SingleStop                      }
 cmd_SetDirBase    : Integer;    { @SetDirBase                      }
 cmd_SetDirExp     : Integer;    { @SetDirExp                       }
 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 {
 Procedure to show sensor help
 }
 procedure SensorHelp(s:String);
 begin
  StdSensorHelpTooltip(s,15000);
 end;
 {
 Prefix for DIM @remote commands.
 }
 function DimRemote:String;
 var CanRemote:Boolean;
 begin
  CanRemote:=DIM_IsServerMode or DIM_IsClientMode;
  if (DIM_GuiClickTag=0) then CanRemote:=false;
  if (devDimSrv=0) then CanRemote:=false;
  if CanRemote
  then DimRemote:='@remote '
  else DimRemote:='';
 end;
 {
 Set base directory for save wavedata file
 }
 procedure SetDirBase(arg:String);
 begin
  if not IsEmptyStr(arg) then begin
   arg:=LoCaseStr(AdaptFileName(arg));
   bNul(sSetTag(WJ300A.DIRBASE.tag,AddPathDelim(arg)));
   Success('New base directory: '+AddPathDelim(arg));
  end else begin
   bNul(sSetTag(WJ300A.DIRBASE.tag,AddPathDelim(LoCaseStr(AdaptFileName(ReadIni('DefaultDirBase'))))));
  end;
 end;
 {
 Set experiment directory for save wavedata file
 }
 procedure SetDirExp(arg:String);
 begin
  if not IsEmptyStr(arg) then begin
   arg:=LoCaseStr(AdaptFileName(arg));
   bNul(sSetTag(WJ300A.DIREXP.tag,arg));
   bNul(sSetTag(WJ300A.EXTRACTNUM.tag,ExtractWordDelims(2,arg,'\/')));
   Success('New save directory: '+arg);
  end else bNul(sSetTag(WJ300A.DIREXP.tag,AddPathDelim(LoCaseStr(AdaptFileName(ReadIni('DefaultDirExp'))))));
 end;
 {
 Find command by id (Acronym).
 }
 function WJ300A_Cmd_Find(id:String):Integer;
 var i,n:Integer;
 begin
  n:=0;
  if (Length(id)>0) then
   for i:=1 to MaxCmdNum do if n=0 then
   if IsSameText(id,WJ300A.Cmd.Acronym[i])
   then n:=i;
  WJ300A_Cmd_Find:=n;
 end;
 {
 Check command number is valid.
 }
 function IsValidCmdNum(n:Integer):Boolean;
 begin
  IsValidCmdNum:=(n>=1) and (n<=MaxCmdNum);
 end;

 function IsQueryCmdNum(n:Integer):Boolean;
 begin
  if IsValidCmdNum(n)
  then IsQueryCmdNum:=WJ300A.Cmd.Query[n]
  else IsQueryCmdNum:=false;
 end;
 {
 Enable/disable command n.
 }
 procedure EnableCmdNum(n:Integer; Enabled:Boolean);
 begin
  if IsValidCmdNum(n) then WJ300A.Cmd.Enabled[n]:=Enabled;
 end;
 {
 Check if WJ300A polling enabled, with warning or not.
 }
 function IsPollingEnabled:Boolean;
 var flag:Boolean;
 begin
  flag:=iGetTag(WJ300A.POLL.ENABLE.tag)>0;
  if not flag then
  ShowTooltip('text "Для изменения параметров необходимо включить опрос" delay 15000 preset stdWarning');
  IsPollingEnabled:=flag;
 end;
 {
 Init command
 }
 procedure InitCmdItem(n,AoNum:Integer;Query,Enabled:Boolean;Priority:Integer; Acronym, SimData:String);
 begin
  if IsValidCmdNum(n) then begin
   WJ300A.Cmd.Acronym[n]:=Acronym;
   WJ300A.Cmd.Query[n]:=Query;
   WJ300A.Cmd.Enabled[n]:=Enabled;
   WJ300A.Cmd.Priority[n]:=Priority;
   WJ300A.Cmd.AoNum[n]:=AoNum;
   WJ300A.Cmd.SimData[n]:=SimData;
  end;
 end;
 {
 Clear commands.
 }
 procedure WJ300A_Clear_Cmd;
 var i:Integer;
 begin
  for i:=1 to MaxCmdNum do begin
   InitCmdItem(i,0,False,False,0,'','');
   WJ300A.Cmd.OpData[i]:='';
   WJ300A.Cmd.OpBuff[i]:='';
  end;
 end;
 {
 Initialize commands.
 }
 procedure WJ300A_Init_Cmd;
 begin
  WJ300A.Cmd.Num:=1;
  //          Command     AoNum       Query  Enabled Priority  Acronym     SimData
  InitCmdItem(cm_ACQ,     -1,         True,  True,   PrNormal, 'ACQ',      'NORMAL');               // cmd/q
  InitCmdItem(cm_CPL_C1,  -1,         True,  True,   PrNormal, 'C1:CPL',   'DC1M');                 // cmd/q
  InitCmdItem(cm_CPL_C2,  -1,         True,  True,   PrNormal, 'C2:CPL',   'DC1M');                 // cmd/q
  InitCmdItem(cm_CPL_C3,  -1,         True,  True,   PrNormal, 'C3:CPL',   'DC1M');                 // cmd/q
  InitCmdItem(cm_CPL_C4,  -1,         True,  True,   PrNormal, 'C4:CPL',   'DC1M');                 // cmd/q
  InitCmdItem(cm_CLS,     -1,         False, False,  PrNormal, '*CLS',     '00');                   // cmd
  InitCmdItem(cm_DATE,    -1,         True,  True,   PrNormal, 'DATE',     '28,JUN,1996,19,43,13'); // cmd/q
  InitCmdItem(cm_DTBORD,  -1,         True,  True,   PrNormal, 'DTBORD',   'H/L');                  // cmd/q
  InitCmdItem(cm_DTFORM,  -1,         True,  True,   PrNormal, 'DTFORM',   'WORD');                 // cmd/q
  InitCmdItem(cm_DTSTART, -1,         True,  True,   PrNormal, 'DTSTART',  '00');                   // cmd/q
  InitCmdItem(cm_DTPOINTS,-1,         True,  True,   PrNormal, 'DTPOINTS', '500000');               // cmd/q
  InitCmdItem(cm_DTWAVE,  -1,         True,  False,  PrHigh,   'DTFORM WORD;DTPOINTS 500000;DTSTART 0;DTWAVE', 'OFF');  // q
  InitCmdItem(cm_ESR,     ao_ESR,     True,  True,   PrNormal, '*ESR',     '00');                   // cmd/q
  InitCmdItem(cm_IDN,     -1,         True,  True,   PrNormal, '*IDN',     'SIMULATOR');            // q
  InitCmdItem(cm_MLEN,    -1,         True,  True,   PrNormal, 'MLEN',     '10K');                  // cmd/q
  InitCmdItem(cm_OFST_C1, ao_OFST_C1, True,  True,   PrNormal, 'C1:OFST',  '-4.44000E+00');         // cmd/q
  InitCmdItem(cm_OFST_C2, ao_OFST_C2, True,  True,   PrNormal, 'C2:OFST',  '+1.00000E+00');         // cmd/q
  InitCmdItem(cm_OFST_C3, ao_OFST_C3, True,  True,   PrNormal, 'C3:OFST',  '-50.0000E-03');         // cmd/q
  InitCmdItem(cm_OFST_C4, ao_OFST_C4, True,  True,   PrNormal, 'C4:OFST',  '-150.000E-03');         // cmd/q
  InitCmdItem(cm_OFST_M1, ao_OFST_M1, True,  True,   PrNormal, 'M1:OFST',  '+0.00000E+00');         // cmd/q
  InitCmdItem(cm_RST,     -1,         False, False,  PrNormal, '*RST',     'RST');                  // cmd
  InitCmdItem(cm_TCPL,    -1,         True,  True,   PrNormal, 'TCPL',     'DC');                   // cmd/q
  InitCmdItem(cm_TDIV,    ao_TDIV,    True,  True,   PrNormal, 'TDIV',     '+1.00000E-06');         // cmd/q
  InitCmdItem(cm_TESR,    -1,         True,  True,   PrNormal, 'TESR',     '1.00E-07');             // q
  InitCmdItem(cm_TLVL,    ao_TLVL,    True,  True,   PrNormal, 'TLVL',     '+2.52000E+00');         // cmd/q
  InitCmdItem(cm_TRA_C1,  -1,         True,  True,   PrNormal, 'C1:TRA',   'ON');                   // cmd/q
  InitCmdItem(cm_TRA_C2,  -1,         True,  True,   PrNormal, 'C2:TRA',   'OFF');                  // cmd/q
  InitCmdItem(cm_TRA_C3,  -1,         True,  True,   PrNormal, 'C3:TRA',   'ON');                   // cmd/q
  InitCmdItem(cm_TRA_C4,  -1,         True,  True,   PrNormal, 'C4:TRA',   'OFF');                  // cmd/q
  InitCmdItem(cm_TRA_M1,  -1,         True,  True,   PrNormal, 'M1:TRA',   'OFF');                  // cmd/q
  InitCmdItem(cm_TRDL,    ao_TRDL,    True,  True,   PrNormal, 'TRDL',     '+0.00000E+00');         // cmd/q
  InitCmdItem(cm_TRMD,    -1,         True,  True,   PrNormal, 'TRMD',     'AUTO');                 // cmd/q
  InitCmdItem(cm_TSLP,    -1,         True,  True,   PrNormal, 'TSLP',     'POS');                  // cmd/q
  InitCmdItem(cm_TSRC,    -1,         True,  True,   PrNormal, 'TSRC',     'CH2');                  // cmd/q
  InitCmdItem(cm_TTYP,    -1,         True,  True,   PrNormal, 'TTYP',     'EDGE');                 // cmd/q
  InitCmdItem(cm_VDIV_C1, ao_VDIV_C1, True,  True,   PrNormal, 'C1:VDIV',  '+2.00000E+00');         // cmd/q
  InitCmdItem(cm_VDIV_C2, ao_VDIV_C2, True,  True,   PrNormal, 'C2:VDIV',  '+2.00000E+00');         // cmd/q
  InitCmdItem(cm_VDIV_C3, ao_VDIV_C3, True,  True,   PrNormal, 'C3:VDIV',  '+100.000E-03');         // cmd/q
  InitCmdItem(cm_VDIV_C4, ao_VDIV_C4, True,  True,   PrNormal, 'C4:VDIV',  '+100.000E-03');         // cmd/q
  InitCmdItem(cm_VDIV_M1, ao_VDIV_M1, True,  True,   PrNormal, 'M1:VDIV',  '+100.000E-03');         // cmd/q
  InitCmdItem(cm_WAVESRC, -1,         True,  True,   PrHigh,   'WAVESRC',  'CH1');                  // cmd/q
 end;
 {
 Print command table
 }
 procedure PrintCmdTable;
 var n:Integer;
 begin
  Success('Command table:');
  Success(StrFmt('%-2s ','Cm')
         +StrFmt('%-2s ','Ao')
         +StrFmt('%-12s ','Acro')
         +StrFmt('%-2s ','Qu')
         +StrFmt('%-2s ','En'));
  for n:=1 to MaxCmdNum do
  Success(StrFmt('%-2d ',n)
         +StrFmt('%-2d ',WJ300A.Cmd.AoNum[n])
         +StrFmt('%-12s ',WJ300A.Cmd.Acronym[n])
         +StrFmt('%-2d ',Ord(WJ300A.Cmd.Query[n]))
         +StrFmt('%-2d ',Ord(WJ300A.Cmd.Enabled[n])));
 end;
 {
 Read line (Line) from pipe (pip).
 Temporary buffer (Buff) should be global lifetime variable with startup initialization.
 }
 function Pipe_WJ300AReadln(pip:Integer; var Line,Buff:String; isWaveform:boolean):Boolean;
 const MaxLeng = 2000020; // dot_count*4+waveform_header
 var LengthData,WaveformLength:Integer;
 begin
  Line:='';
  Pipe_WJ300AReadln:=false;
  if (Pipe_Pid(pip)<>0) then begin
   if (Length(Buff)<MaxLeng) and (Pipe_RxCount(pip)>0)
   then Buff:=Buff+Pipe_Recv(pip,MaxLeng-Length(Buff));
   LengthData:=val('$'+hex_encode(Copy(Buff,5,4)));
   if (LengthData>0) then begin
    WaveformLength:=0;
    if isWaveform then WaveformLength:=val(Copy(Buff,11,8))+WaveformOffset;
    Line:=Copy(Buff,1+HeaderLength,LengthData+WaveformLength);
    if (Length(Line)=(LengthData+WaveformLength)) then Pipe_WJ300AReadln:=True;
   end else begin
    if (Length(Buff)>=MaxLeng) then begin
     Trouble('Received line is too long!');
     Buff:='';
    end;
   end;
  end;
 end;
 {
 Return channel name by index
 }
 function GetChannelByID(index:integer):String;
 begin
  if index>0 then begin
   case index of
    1: GetChannelByID:='CH1';
    2: GetChannelByID:='CH2';
    3: GetChannelByID:='CH3';
    4: GetChannelByID:='CH4';
    5: GetChannelByID:='MATH';
   end;
  end else GetChannelByID:='NONE';
 end;
 {
 Switch WaveForm source....
 }
 procedure IncWaveFormSource;
 begin
  WJ300A.GetWaveFormSource:=WJ300A.GetWaveFormSource+1;
 end;
 {
 Switch channel tracktion state
 }
 function SwitchTraState(tag:Integer):String;
 begin
  if iGetTag(tag)>0 then SwitchTraState:='OFF' else SwitchTraState:='ON';
 end;
 {
 Convert channel tracktion state
 }
 function ConvertTraState(State:String):String;
 begin
  State:=extractword(1,State);
  if IsSameText(State,'ON')  then ConvertTraState:=dump('1');
  if IsSameText(State,'OFF') then ConvertTraState:=dump('0');
 end;
 {
 Convert trigger mode state to bit
 }
 function ConvertTRMD(State:String):String;
 begin
  State:=extractword(1,State);
  if IsSameText(State,'AUTO')   then ConvertTRMD:=dump('1');
  if IsSameText(State,'NORMAL') then ConvertTRMD:=dump('2');
  if IsSameText(State,'SINGLE') then ConvertTRMD:=dump('4');
  if IsSameText(State,'STOP')   then ConvertTRMD:=dump('8');
 end; 
 {
 Check switch status for channels
 }
 function IsChannelOn(var CHAN:TChanRec):Boolean;
 begin
  IsChannelOn:=(iGetTag(CHAN.TRA.tag)=1);
 end;
 {
 Returns enabled channel number
 }
 function GetWaveFormSourceNumber:Integer;
 var n:Integer;
 begin
  n:=0;
  if WJ300A.GetWaveFormSource=1 then if IsChannelOn(WJ300A.C1) then n:=WJ300A.GetWaveFormSource else IncWaveFormSource;
  if WJ300A.GetWaveFormSource=2 then if IsChannelOn(WJ300A.C2) then n:=WJ300A.GetWaveFormSource else IncWaveFormSource;
  if WJ300A.GetWaveFormSource=3 then if IsChannelOn(WJ300A.C3) then n:=WJ300A.GetWaveFormSource else IncWaveFormSource;
  if WJ300A.GetWaveFormSource=4 then if IsChannelOn(WJ300A.C4) then n:=WJ300A.GetWaveFormSource;
  GetWaveFormSourceNumber:=n;
 end;
 {
 Returns wave form source channel current number
 }
 function GetSelectedChannel:Integer;
 var Channel:String;
  procedure Cleanup;
  begin
   Channel:='';
  end;
 begin
  Cleanup;
  Channel:=sGetTag(WJ300A.PARAM.WAVESRC.tag);
  if IsSameText(Channel,'CH1') then GetSelectedChannel:=1
  else if IsSameText(Channel,'CH2') then GetSelectedChannel:=2
  else if IsSameText(Channel,'CH3') then GetSelectedChannel:=3
  else if IsSameText(Channel,'CH4') then GetSelectedChannel:=4;
  Cleanup;
 end;
 {
 Pipe connection with device
 }
 function WJ300A_PipeConnect:Boolean;
 var sPID,sPSIM:String;
  procedure Cleanup;
  begin
   sPID:='';
   sPSIM:='';
  end;
 begin
  WJ300A_PipeConnect:=false;
  Cleanup;
  if WJ300A.Simulator then begin
   //Pipe connection for simulation
   WJ300A.Pipe.PipeID:=Pipe_Init('pipe '+ReadIni('tagPrefix'));
   WJ300A.Pipe.PipeSIM:=Pipe_Init('pipe .\'+ReadIni('tagPrefix')+' TxPipeSize 66384');
   WJ300A_PipeConnect:=(Pipe_Run(WJ300A.Pipe.PipeID) and Pipe_Run(WJ300A.Pipe.PipeSIM));
   sPID :=str(WJ300A.Pipe.PipeID);
   sPSIM:=str(WJ300A.Pipe.PipeSIM);
   Details('WJ300A.Pipe.PipeID = '+sPID);
   Details('WJ300A.Pipe.PipeSIM = '+sPSIM);
  end else begin
   //Pipe connection for device control by TCP connection
   WJ300A.Pipe.PipeID:=Pipe_Init('Tcp Port '+ReadIni('TCPPort')+' Client '+LoCaseStr(ReadIni('TcpHostname'))+
   ' Polling '+ReadIni('Polling')+' Priority '+ReadIni('Priority')+
   ' TxPipeSize '+ReadIni('TxPipeSize')+' RxPipeSize '+ReadIni('RxPipeSize')+
   ' Timeout '+ReadIni('Timeout'));
   bNul(Pipe_Run(WJ300A.Pipe.PipeID));
   WJ300A_PipeConnect:=Pipe_Ref(WJ300A.Pipe.PipeID)>0;
   sPID:=str(WJ300A.Pipe.PipeID);
   Details('WJ300A.Pipe.PipeID = '+sPID);
  end;
  Cleanup;
 end;
 {
 Request cmd generation
 }
 function ReqCmd(n:Integer):String;
 var Request,OpData:String;
  procedure Cleanup;
  begin
   Request:='';
   OpData:='';
  end;
 begin
  Cleanup;
  if IsValidCmdNum(n) then begin
   Request:=WJ300A.Cmd.Acronym[n];
   OpData:=WJ300A.Cmd.OpData[n];
   if IsEmptyStr(OpData) then begin
    if (IsQueryCmdNum(n) and IsEmptyStr(OpData)) then Request:=Request+'?';
   end else Request:=Request+' '+OpData
  end;
  ReqCmd:=Request;
  Cleanup;
 end;
 {
 Add header to request
 }
 function AddHeader(Data:String;isWave:Boolean):String;
 begin
  if isWave then AddHeader:=dump(chr(DataFlag))+dump(chr(1))+dump(chr(1))+dump(chr(0))+dump(htonl(8))+Data
  else AddHeader:=dump(chr(DataFlag))+dump(chr(1))+dump(chr(1))+dump(chr(0))+dump(htonl(Length(Data)))+Data;
 end;
 {
 Update trigger status
 }
 procedure TrigStatusUpdate;
 var i:Integer;
 begin
  if WJ300A.GetTrigStatus=stt_trigNone then begin
  end else
  if WJ300A.GetTrigStatus=stt_trigSetSingle then begin
   if iGetTag(WJ300A.PARAM.TESR.tag)=0 then begin
    DevSendCmdLocal(DimRemote+'@Setting TRMD SINGLE');
    WJ300A.GetTrigStatus:=stt_trigReady;
   end else WJ300A.GetTrigStatus:=stt_trigDone;
  end else
  if WJ300A.GetTrigStatus=stt_trigReady then begin
   if iGetTag(WJ300A.PARAM.TESR.tag)=1 then WJ300A.GetTrigStatus:=stt_trigDone
   else if msElapsedSinceMarker(WJ300A.GetTrigStatusTimer)>tm_trigSetSingle then begin
    WJ300A.GetTrigStatus:=stt_trigSetSingle;
    WJ300A.GetTrigStatusTimer:=0;
   end;
  end else
  if WJ300A.GetTrigStatus=stt_trigDone then begin
   if iGetTag(WJ300A.PARAM.TESR.tag)=1 then begin
    DevSendCmdLocal(DimRemote+'@GetWaveForm 1');
    WJ300A.GetTrigStatus:=stt_trigNone;
   end;
  end;
 end;
 {
 Get WaveForm data
 }
 procedure GetWaveFormData;
 begin
  if WJ300A.IsWaveFormDataRequest then begin
   if WJ300A.GetWaveFormStatus=stt_wfNone then begin
   end else
   if WJ300A.GetWaveFormStatus=stt_wfSetSource then begin
    DevSendCmdLocal(DimRemote+'@Setting WAVESRC '+GetChannelByID(WJ300A.GetWaveFormSource));
    WJ300A.GetWaveFormTimer:=mSecNow;
    WJ300A.GetWaveFormStatus:=stt_wfChkSource;
   end else
   if WJ300A.GetWaveFormStatus=stt_wfChkSource then begin
    if GetSelectedChannel=WJ300A.GetWaveFormSource then WJ300A.GetWaveFormStatus:=stt_wfGetData else
    if msElapsedSinceMarker(WJ300A.GetWaveFormTimer)>tm_wfSetSource then begin
     WJ300A.GetWaveFormStatus:=stt_wfSetSource;
     WJ300A.GetWaveFormTimer:=0;
    end;
   end;
   if WJ300A.GetWaveFormStatus=stt_wfGetData then begin
    EnableCmdNum(cm_DTWAVE,true);
    WJ300A.GetWaveFormStatus:=stt_wfNone;
   end;
   if WJ300A.GetWaveFormStatus=stt_wfIncSource then begin
    IncWaveFormSource;
    WJ300A.GetWaveFormStatus:=stt_wfGetSource;
   end;
   if WJ300A.GetWaveFormStatus=stt_wfGetSource then begin
    WJ300A.GetWaveFormSource:=GetWaveFormSourceNumber;
    if WJ300A.GetWaveFormSource<>0
    then WJ300A.GetWaveFormStatus:=stt_wfSetSource
    else WJ300A.GetWaveFormStatus:=stt_wfDone;
   end;
   if WJ300A.GetWaveFormStatus=stt_wfDone then begin
    WJ300A.PollPriority:=PrNormal;
    WJ300A.GetWaveFormTimer:=0;
    WJ300A.GetWaveFormSource:=1;
    WJ300A.GetWaveFormStatus:=stt_wfNone;
    WJ300A.IsWaveFormDataRequest:=false;
    DevSendCmdLocal(DimRemote+'@AssignTag '+NameTag(WJ300A.PARAM.DTWAVE.tag)+' 0');
    DevSendCmd(WJ300A.CalcDevRef,'@CalculateData');
   end;
  end;
 end;
 {
 Main command cycle.
 }
 procedure WJ300A_CMD_CYCLE;
  function ValidateCmdNum(Num:Integer):Integer;
  begin
   if IsValidCmdNum(Num)
   then ValidateCmdNum:=Num
   else ValidateCmdNum:=1;
  end;
  function NextCmdNum(Num:Integer):Integer;
  var i:Integer;
  begin
   i:=0;
   while (i<MaxCmdNum) do begin
    Num:=ValidateCmdNum(Num+1);
    if WJ300A.Cmd.Enabled[Num] then
    if WJ300A.Cmd.Priority[Num]>=WJ300A.PollPriority
    then i:=MaxCmdNum
    else i:=i+1;
   end;
   NextCmdNum:=Num;
  end;
  procedure ClearRequest(Num,Status:Integer);
  begin
   WJ300A.Pipe.Req:='';
   WJ300A.Pipe.Ans:='';
   WJ300A.Pipe.Buf:='';
   WJ300A.Pipe.ReqTime:=0;
   WJ300A.Pipe.AnsTime:=0;
   WJ300A.Pipe.Status:=Status;
   WJ300A.Cmd.Num:=ValidateCmdNum(Num);
   if WJ300A.Pipe.PipeID<>0 then begin
    bNul(pipe_txclear(WJ300A.Pipe.PipeID));
    bNul(pipe_rxclear(WJ300A.Pipe.PipeID));
   end;
   if WJ300A.Pipe.PipeSIM<>0 then begin
    bNul(pipe_txclear(WJ300A.Pipe.PipeSIM));
    bNul(pipe_rxclear(WJ300A.Pipe.PipeSIM));
   end;
  end;
  procedure PrepareRequest(n:Integer);
  var req:String;
  begin
   if IsValidCmdNum(n) then begin
    req:=ReqCmd(n);
    WJ300A.Pipe.Req:=AddHeader(req,false);
   end;
  end;
  procedure UpdateCmdData(n,tag:Integer; data:String);
  var v:Real; CurveRef:Integer;
  begin
   if IsRefTag(tag) then
   if not IsEmptyStr(data) then
   if TypeTag(tag)<=2 then begin
    CurveRef:=RefAo(WJ300A.Cmd.AoNum[n]);
    v:=rVal(data);
    if IsNaN(v) then Trouble('Invalid tag update '+str(v)+' '+data) else
    if TypeTag(tag)=1 then bNul(iSetTag(tag,Round(v))) else
    if TypeTag(tag)=2 then bNul(rSetTag(tag,v));
    if IsRefCurve(CurveRef) then UpdateAo(WJ300A.Cmd.AoNum[n],time,v);
   end else
   if TypeTag(tag)=3 then begin
    bNul(sSetTag(tag,Trim(data)));
   end;
  end;
  {
  Процедура декодирования данных и их запись
  }
  procedure DecodeData(n:Integer; data:String);
  var i:Integer;
  begin
   if IsValidCmdNum(n) then begin
    if Length(data)>0 then begin
     if (n=cm_ACQ)      then UpdateCmdData(n,WJ300A.PARAM.ACQ.tag,data)         else
     if (n=cm_CPL_C1)   then UpdateCmdData(n,WJ300A.C1.COUP.tag,data)           else
     if (n=cm_CPL_C2)   then UpdateCmdData(n,WJ300A.C2.COUP.tag,data)           else
     if (n=cm_CPL_C3)   then UpdateCmdData(n,WJ300A.C3.COUP.tag,data)           else
     if (n=cm_CPL_C4)   then UpdateCmdData(n,WJ300A.C4.COUP.tag,data)           else
     if (n=cm_CLS)      then EnableCmdNum(n,false)                              else
     if (n=cm_DATE)     then UpdateCmdData(n,WJ300A.PARAM.DATE.tag,data)        else
     if (n=cm_DTBORD)   then UpdateCmdData(n,WJ300A.PARAM.DTBORD.tag,data)      else
     if (n=cm_DTFORM)   then UpdateCmdData(n,WJ300A.PARAM.DTFORM.tag,data)      else
     if (n=cm_DTSTART)  then UpdateCmdData(n,WJ300A.PARAM.DTSTART.tag,data)     else
     if (n=cm_DTPOINTS) then UpdateCmdData(n,WJ300A.PARAM.DTPOINTS.tag,data)    else
     if (n=cm_DTWAVE)   then begin
      if WJ300A.GetWaveFormSource=1 then bNul(sSetTag(WJ300A.C1.WAVFORM.tag,data));
      if WJ300A.GetWaveFormSource=2 then bNul(sSetTag(WJ300A.C2.WAVFORM.tag,data));
      if WJ300A.GetWaveFormSource=3 then bNul(sSetTag(WJ300A.C3.WAVFORM.tag,data));
      if WJ300A.GetWaveFormSource=4 then bNul(sSetTag(WJ300A.C4.WAVFORM.tag,data));
      EnableCmdNum(n,false);
      WJ300A.GetWaveFormStatus:=stt_wfIncSource;
     end else
     if (n=cm_ESR)      then UpdateCmdData(n,WJ300A.PARAM.ESR.tag,data)         else
     if (n=cm_IDN)      then begin
      UpdateCmdData(n,WJ300A.PARAM.IDN.tag,data);
      EnableCmdNum(n,false);
     end else
     if (n=cm_MLEN)     then UpdateCmdData(n,WJ300A.PARAM.MLEN.tag,data)        else
     if (n=cm_OFST_C1)  then UpdateCmdData(n,WJ300A.C1.OFFS.tag,data)           else
     if (n=cm_OFST_C2)  then UpdateCmdData(n,WJ300A.C2.OFFS.tag,data)           else
     if (n=cm_OFST_C3)  then UpdateCmdData(n,WJ300A.C3.OFFS.tag,data)           else
     if (n=cm_OFST_C4)  then UpdateCmdData(n,WJ300A.C4.OFFS.tag,data)           else
     if (n=cm_OFST_M1)  then UpdateCmdData(n,WJ300A.M1.OFFS.tag,data)           else
     if (n=cm_RST)      then EnableCmdNum(n,false)                              else
     if (n=cm_TCPL)     then UpdateCmdData(n,WJ300A.PARAM.TCPL.tag,data)        else
     if (n=cm_TDIV)     then UpdateCmdData(n,WJ300A.PARAM.TDIV.tag,data)        else
     if (n=cm_TESR)     then UpdateCmdData(n,WJ300A.PARAM.TESR.tag,data)        else
     if (n=cm_TLVL)     then UpdateCmdData(n,WJ300A.PARAM.TLVL.tag,data)        else
     if (n=cm_TRA_C1)   then UpdateCmdData(n,WJ300A.C1.TRA.tag,ConvertTraState(data)) else
     if (n=cm_TRA_C2)   then UpdateCmdData(n,WJ300A.C2.TRA.tag,ConvertTraState(data)) else
     if (n=cm_TRA_C3)   then UpdateCmdData(n,WJ300A.C3.TRA.tag,ConvertTraState(data)) else
     if (n=cm_TRA_C4)   then UpdateCmdData(n,WJ300A.C4.TRA.tag,ConvertTraState(data)) else
     if (n=cm_TRA_M1)   then UpdateCmdData(n,WJ300A.M1.TRA.tag,ConvertTraState(data)) else
     if (n=cm_TRDL)     then UpdateCmdData(n,WJ300A.PARAM.TRDL.tag,data)        else
     if (n=cm_TRMD)     then UpdateCmdData(n,WJ300A.PARAM.TRMD.tag,ConvertTRMD(data)) else
     if (n=cm_TSLP)     then UpdateCmdData(n,WJ300A.PARAM.TSLP.tag,data)        else
     if (n=cm_TSRC)     then UpdateCmdData(n,WJ300A.PARAM.TSRC.tag,data)        else
     if (n=cm_TTYP)     then UpdateCmdData(n,WJ300A.PARAM.TTYP.tag,data)        else
     if (n=cm_VDIV_C1)  then UpdateCmdData(n,WJ300A.C1.SCAL.tag,data)           else
     if (n=cm_VDIV_C2)  then UpdateCmdData(n,WJ300A.C2.SCAL.tag,data)           else
     if (n=cm_VDIV_C3)  then UpdateCmdData(n,WJ300A.C3.SCAL.tag,data)           else
     if (n=cm_VDIV_C4)  then UpdateCmdData(n,WJ300A.C4.SCAL.tag,data)           else
     if (n=cm_VDIV_M1)  then UpdateCmdData(n,WJ300A.M1.SCAL.tag,data)           else
     if (n=cm_WAVESRC)  then UpdateCmdData(n,WJ300A.PARAM.WAVESRC.tag,data);
    end;
   end;
  end;
  function HandleRequest(n:Integer; Ans:String):Boolean;
  var Handled:Boolean;
  begin
   Handled:=false;
   if IsValidCmdNum(n) and (Length(Ans)>0) then begin
    ViewImp('Pipe < '+Ans);
    DecodeData(n,Ans);
    Handled:=True;
   end;
   HandleRequest:=Handled;
  end;
  procedure PrintDetails;
  begin
   if iAnd(DebugFlags,dfDetails)>0 then Details('Run '+Str(RunCount)
    +', Cmd '+WJ300A.Cmd.Acronym[WJ300A.Cmd.Num]
    +', Enb '+Str(Ord(WJ300A.Cmd.Enabled[WJ300A.Cmd.Num]))
    +', Sta '+Str(WJ300A.Pipe.Status)
    +', Req '+Trim(WJ300A.Pipe.Req)
    +', Ans '+Trim(WJ300A.Pipe.Ans)
    +', Buf '+Trim(WJ300A.Pipe.Buf));
  end;
  procedure UpdateOpData;
  var n:Integer;
  begin
   for n:=1 to MaxCmdNum do
   if not IsEmptyStr(WJ300A.Cmd.OpBuff[n]) then begin
    WJ300A.Cmd.OpData[n]:=WJ300A.Cmd.OpBuff[n];
    EnableCmdNum(n,true);
    WJ300A.Cmd.OpBuff[n]:='';
   end;
  end;
  procedure ExecuteIdleActions;
  begin
   UpdateOpData;
  end;
 begin
  {
  Pipe port communication handler
  }
  if not DIM_IsClientMode then begin
   if iGetTag(WJ300A.POLL.ENABLE.tag)>0 then begin
    if pipe_connected(WJ300A.Pipe.PipeId)>0 then begin
     WJ300A.Cmd.Num:=ValidateCmdNum(WJ300A.Cmd.Num);
     if iAnd(DebugFlags,dfDetails)>0 then PrintDetails;
     if WJ300A.Pipe.Status=st_NoReq then begin
      ExecuteIdleActions;
      ClearRequest(WJ300A.Cmd.Num,st_NoReq);
      if mSecNow>=WJ300A.Pipe.LastPoll+WJ300A.Pipe.PollPeriod then begin
       if WJ300A.Cmd.Enabled[WJ300A.Cmd.Num] then PrepareRequest(WJ300A.Cmd.Num);
       if Length(WJ300A.Pipe.Req)>0 then begin
        if pipe_send(WJ300A.Pipe.PipeId,WJ300A.Pipe.Req)>0 then begin
         ViewExp('Pipe > '+copy(WJ300A.Pipe.Req,HeaderLength+1,length(WJ300A.Pipe.Req)-HeaderLength));
         WJ300A.Pipe.Status:=st_WaitAns;
         WJ300A.Pipe.ReqTime:=mSecNow;
         WJ300A.Pipe.LastPoll:=mSecNow;
        end else begin
         Trouble('Could not send request '+copy(WJ300A.Pipe.Req,HeaderLength+1,length(WJ300A.Pipe.Req)-HeaderLength));
         ClearRequest(NextCmdNum(WJ300A.Cmd.Num),st_NoReq);
        end;
       end else ClearRequest(NextCmdNum(WJ300A.Cmd.Num),st_NoReq);
      end;
     end else
     if WJ300A.Pipe.Status=st_WaitAns then begin
      if (not IsQueryCmdNum(WJ300A.Cmd.Num) or not IsEmptyStr(WJ300A.Cmd.OpData[WJ300A.Cmd.Num])) then begin
       WJ300A.Pipe.PollRate:=WJ300A.Pipe.PollRate+1;
       WJ300A.Pipe.Status:=st_WaitGap;
       WJ300A.Cmd.OpData[WJ300A.Cmd.Num]:='';
      end else begin
       if Pipe_WJ300AReadln(WJ300A.Pipe.PipeId,WJ300A.Pipe.Ans,WJ300A.Pipe.Buf,WJ300A.Cmd.Num=cm_DTWAVE) then begin
        bNul(HandleRequest(WJ300A.Cmd.Num,WJ300A.Pipe.Ans));
        WJ300A.Pipe.PollRate:=WJ300A.Pipe.PollRate+1;
        WJ300A.Pipe.Status:=st_WaitGap;
        WJ300A.Pipe.AnsTime:=mSecNow;
       end;
       if mSecNow>WJ300A.Pipe.ReqTime+WJ300A.Pipe.TimeOut then begin
        WJ300A.Pipe.Status:=st_TimeOut;
       end;
      end;
     end;
     if WJ300A.Pipe.Status=st_WaitGap then begin
      if mSecNow>=WJ300A.Pipe.AnsTime+WJ300A.Pipe.TimeGap then begin
       if not IsQueryCmdNum(WJ300A.Cmd.Num) then EnableCmdNum(WJ300A.Cmd.Num,false);
       ClearRequest(NextCmdNum(WJ300A.Cmd.Num),st_NoReq);
      end;
     end;
     if WJ300A.Pipe.Status=st_TimeOut then begin
      Failure(WJ300A.ecTimeOut,'TimeOut on command '+WJ300A.Cmd.Acronym[WJ300A.Cmd.Num]+' request '+Trim(WJ300A.Pipe.Req));
      ClearRequest(NextCmdNum(WJ300A.Cmd.Num),st_NoReq);
      bNul(sSetTag(WJ300A.PARAM.IDN.tag,'REFUSE'));
     end;
    end else begin
     ClearRequest(NextCmdNum(0),st_NoReq);
     bNul(sSetTag(WJ300A.PARAM.IDN.tag,'REFUSE'));
    end;
   end;
  end;
 end;
 {
 Menu Tools Starter to start editing.
 }
 procedure MenuToolsStarter;
 var i,n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Инструменты');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Просмотр справочной информации (HELP)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@BrowseHelp');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Открыть окно: '+ParamStr('CONSOLE '+DevName));
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@OpenConsole');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: нормальный  (3)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 3');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: ввод-вывод (15)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 15');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: детальный  (31)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 31');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сброс параметров устройства');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Setting *RST');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Загрузить Параметры из INI файла');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @LoadIni');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сохранить Параметры из INI файла');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @SaveIni');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Печать состояния таблицы команд');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@PrintCmdTable');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Перезапустить локальный драйвер '+DevName);
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SysEval @Daq Compile '+DevName);
   //////////////////////////////////////////
   if DIM_IsClientMode then begin
    n:=n+EditAddInputLn('Перезапустить удаленный драйвер '+DevName);
    n:=n+EditAddConfirm(EditGetLastInputLn);
    n:=n+EditAddCommand(DimRemote+'@SysEval @Daq Compile '+DevName);
   end;
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TOOLS'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu acquisition mode
 }
 procedure MenuAcquisitionMode(tag:integer);
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Режим измерения');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('нормальный');
   n:=n+EditAddCommand('@Setting ACQ NORMAL');
   n:=n+EditAddInputLn('максимальное значение');
   n:=n+EditAddCommand('@Setting ACQ PEAK');
   n:=n+EditAddInputLn('усредненный');
   n:=n+EditAddCommand('@Setting ACQ AVERAGE');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_ACQ'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu coupling mode
 }
 procedure MenuCoupling(tag:integer;acronym:String);
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Режим измерения');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('AC1M');
   n:=n+EditAddCommand('@Setting '+acronym+' AC1M');
   n:=n+EditAddInputLn('GND');
   n:=n+EditAddCommand('@Setting '+acronym+' GND');
   n:=n+EditAddInputLn('DC1M');
   n:=n+EditAddCommand('@Setting '+acronym+' DC1M');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_CPL'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu maximum memory length
 }
 procedure MenuMaximumMemoryLength(tag:integer);
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Количество точек');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('500');
   n:=n+EditAddCommand('@Setting MLEN 500');
   n:=n+EditAddInputLn('1 000');
   n:=n+EditAddCommand('@Setting MLEN 1K');
   n:=n+EditAddInputLn('10 000');
   n:=n+EditAddCommand('@Setting MLEN 10K');
   n:=n+EditAddInputLn('100 000');
   n:=n+EditAddCommand('@Setting MLEN 100K');
   n:=n+EditAddInputLn('500 000');
   n:=n+EditAddCommand('@Setting MLEN 500K');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_MLEN'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu trigger coupling mode
 }
 procedure MenuTriggerCouplingMode(tag:integer);
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Согласование триггера');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('AC');
   n:=n+EditAddCommand('@Setting TCPL AC');
   n:=n+EditAddInputLn('DC');
   n:=n+EditAddCommand('@Setting TCPL DC');
   n:=n+EditAddInputLn('HF');
   n:=n+EditAddCommand('@Setting TCPL HF');
   n:=n+EditAddInputLn('LF');
   n:=n+EditAddCommand('@Setting TCPL LF');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TCPL'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu trigger mode
 }
 procedure MenuTriggerMode(tag:integer);
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Режим триггера');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('автоматический');
   n:=n+EditAddCommand('@Setting TRMD AUTO');
   n:=n+EditAddInputLn('нормальный');
   n:=n+EditAddCommand('@Setting TRMD NORMAL');
   n:=n+EditAddInputLn('одиночный');
   n:=n+EditAddCommand('@Setting TRMD SINGLE');
   n:=n+EditAddInputLn('остановка');
   n:=n+EditAddCommand('@Setting TRMD STOP');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TRMD'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu trigger slope
 }
 procedure MenuTriggerSlope(tag:integer);
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Фронт триггера');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Передний фронт');
   n:=n+EditAddCommand('@Setting TSLP POS');
   n:=n+EditAddInputLn('Задний фронт');
   n:=n+EditAddCommand('@Setting TSLP NEG');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TSLP'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu trigger signal source
 }
 procedure MenuTriggerSignalSource(tag:integer);
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Источник сигнала триггера');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('CH1');
   n:=n+EditAddCommand('@Setting TSRC CH1');
   n:=n+EditAddInputLn('CH2');
   n:=n+EditAddCommand('@Setting TSRC CH2');
   n:=n+EditAddInputLn('CH3');
   n:=n+EditAddCommand('@Setting TSRC CH3');
   n:=n+EditAddInputLn('CH4');
   n:=n+EditAddCommand('@Setting TSRC CH4');
   n:=n+EditAddInputLn('EXT');
   n:=n+EditAddCommand('@Setting TSRC EXT');
   n:=n+EditAddInputLn('EXT10');
   n:=n+EditAddCommand('@Setting TSRC EXT10');
   n:=n+EditAddInputLn('LINE');
   n:=n+EditAddCommand('@Setting TSRC LINE');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TSRC'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu trigger type
 }
 procedure MenuTriggerType(tag:integer);
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Тип триггера');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('EDGE');
   n:=n+EditAddCommand('@Setting TTYP EDGE');
   n:=n+EditAddInputLn('PULSE');
   n:=n+EditAddCommand('@Setting TTYP PULSE');
   n:=n+EditAddInputLn('PERIOD');
   n:=n+EditAddCommand('@Setting TTYP PERIOD');
   n:=n+EditAddInputLn('COUNT');
   n:=n+EditAddCommand('@Setting TTYP COUNT');
   n:=n+EditAddInputLn('TV');
   n:=n+EditAddCommand('@Setting TTYP TV');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TTYP'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu handlers
 }
 procedure MenuHandlers;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_TOOLS'));
  EditMenuDefaultHandler(EditGetUID('MENU_ACQ'));
  EditMenuDefaultHandler(EditGetUID('MENU_CPL'));
  EditMenuDefaultHandler(EditGetUID('MENU_MLEN'));
  EditMenuDefaultHandler(EditGetUID('MENU_TCPL'));
  EditMenuDefaultHandler(EditGetUID('MENU_TRMD'));
  EditMenuDefaultHandler(EditGetUID('MENU_TSLP'));
  EditMenuDefaultHandler(EditGetUID('MENU_TSRC'));
  EditMenuDefaultHandler(EditGetUID('MENU_TTYP'));
 end;
 {
 GUI Handler to process user input...
 }
 procedure WJ300A_GUI_POLL;
 var ClickCurve,n:Integer; s:String;
  procedure Cleanup;
  begin
   DIM_GuiClickBuff:=''; s:='';
  end;
 begin
  {
  Handle user mouse/keyboard clicks...
  ClickWhat=(cw_Nothing,cw_MouseDown,cw_MouseUp,cw_MouseMove,cw_KeyDown,KeyUp)
  ClickButton=(VK_LBUTTON,VK_RBUTTON,VK_CANCEL,VK_MBUTTON,VK_BACK,VK_TAB,VK_CLEAR,VK_RETURN,...)
  }
  if ClickWhat<>0 then
  repeat
   {
   Copy GUI click to DIM buffer for remote execution.
   }
   DIM_GuiClickBuff:=DIM_GuiClickCopy;
   {
   Handle MouseDown/KeyDown
   }
   if (ClickWhat=cw_MouseDown) or (ClickWhat=cw_KeyDown) then begin
    {
    Handle Left mouse button click
    }
    if (ClickButton=VK_LBUTTON) then begin
     {
     Handle local clicks
     }
     if ClickIsLocal then begin
      {
      Handle sensor clicks...
      }
      if ClickSensor<>'' then begin
       bNul(Voice(snd_Click));
       if ClickSensor='TOOLS'       then MenuToolsStarter;
       if ClickSensor='CLS'         then EnableCmdNum(cm_CLS,true);
       if ClickSensor='RUN'         then DevSendCmdLocal(DimRemote+'@TrigRun');
       if ClickSensor='STOP'        then DevSendCmdLocal(DimRemote+'@TrigStop');
       if ClickSensor='TRG'         then DevSendCmdLocal(DimRemote+'@SingleStart');
       if ClickSensor='TRMD'        then MenuTriggerMode(WJ300A.PARAM.TRMD.tag);
      end;
      {
      Handle tag clicks...
      }
      if (ClickTag<>0) then begin
       if ClickTag=WJ300A.POLL.ENABLE.tag    then DevSendCmdLocal(DimRemote+'@AssignTag '+NameTag(ClickTag)+' '+Str(iXor(iGetTag(ClickTag),1)));
       if ClickTag=WJ300A.PARAM.ACQ.tag      then MenuAcquisitionMode(WJ300A.PARAM.ACQ.tag);
       if ClickTag=WJ300A.C1.COUP.tag        then MenuCoupling(WJ300A.C1.COUP.tag,'C1:CPL');
       if ClickTag=WJ300A.C2.COUP.tag        then MenuCoupling(WJ300A.C2.COUP.tag,'C2:CPL');
       if ClickTag=WJ300A.C3.COUP.tag        then MenuCoupling(WJ300A.C3.COUP.tag,'C3:CPL');
       if ClickTag=WJ300A.C4.COUP.tag        then MenuCoupling(WJ300A.C4.COUP.tag,'C4:CPL');
       if ClickTag=WJ300A.PARAM.DATE.tag     then DevSendCmdLocal('@Edit DATE');
       if ClickTag=WJ300A.PARAM.DTSTART.tag  then DevSendCmdLocal('@Edit DTSTART');
       if ClickTag=WJ300A.PARAM.DTPOINTS.tag then DevSendCmdLocal('@Edit DTPOINTS');
       if ClickTag=WJ300A.PARAM.MLEN.tag     then MenuMaximumMemoryLength(WJ300A.PARAM.MLEN.tag);
       if ClickTag=WJ300A.C1.OFFS.tag        then DevSendCmdLocal('@Edit OFST_C1');
       if ClickTag=WJ300A.C2.OFFS.tag        then DevSendCmdLocal('@Edit OFST_C2');
       if ClickTag=WJ300A.C3.OFFS.tag        then DevSendCmdLocal('@Edit OFST_C3');
       if ClickTag=WJ300A.C4.OFFS.tag        then DevSendCmdLocal('@Edit OFST_C4');
       if ClickTag=WJ300A.M1.OFFS.tag        then DevSendCmdLocal('@Edit OFST_M1');
       if ClickTag=WJ300A.PARAM.TCPL.tag     then MenuTriggerCouplingMode(WJ300A.PARAM.TCPL.tag);
       if ClickTag=WJ300A.PARAM.TDIV.tag     then DevSendCmdLocal('@Edit TDIV');
       if ClickTag=WJ300A.PARAM.TLVL.tag     then DevSendCmdLocal('@Edit TLVL');
       if ClickTag=WJ300A.C1.TRA.tag         then DevSendCmdLocal('@Setting C1:TRA '+SwitchTraState(WJ300A.C1.TRA.tag));
       if ClickTag=WJ300A.C2.TRA.tag         then DevSendCmdLocal('@Setting C2:TRA '+SwitchTraState(WJ300A.C2.TRA.tag));
       if ClickTag=WJ300A.C3.TRA.tag         then DevSendCmdLocal('@Setting C3:TRA '+SwitchTraState(WJ300A.C3.TRA.tag));
       if ClickTag=WJ300A.C4.TRA.tag         then DevSendCmdLocal('@Setting C4:TRA '+SwitchTraState(WJ300A.C4.TRA.tag));
       if ClickTag=WJ300A.M1.TRA.tag         then DevSendCmdLocal('@Setting M1:TRA '+SwitchTraState(WJ300A.M1.TRA.tag));
       if ClickTag=WJ300A.PARAM.TRDL.tag     then DevSendCmdLocal('@Edit TRDL');
       if ClickTag=WJ300A.PARAM.TSLP.tag     then MenuTriggerSlope(WJ300A.PARAM.TSLP.tag);
       if ClickTag=WJ300A.PARAM.TSRC.tag     then MenuTriggerSignalSource(WJ300A.PARAM.TSRC.tag);
       if ClickTag=WJ300A.PARAM.TTYP.tag     then MenuTriggerType(WJ300A.PARAM.TTYP.tag);
       if ClickTag=WJ300A.C1.SCAL.tag        then DevSendCmdLocal('@Edit VDIV_C1');
       if ClickTag=WJ300A.C2.SCAL.tag        then DevSendCmdLocal('@Edit VDIV_C2');
       if ClickTag=WJ300A.C3.SCAL.tag        then DevSendCmdLocal('@Edit VDIV_C3');
       if ClickTag=WJ300A.C4.SCAL.tag        then DevSendCmdLocal('@Edit VDIV_C4');
       if ClickTag=WJ300A.M1.SCAL.tag        then DevSendCmdLocal('@Edit VDIV_M1');
       if ClickTag=WJ300A.DIRBASE.tag        then DevSendCmdLocal('@Edit DIRBASE');
       if ClickTag=WJ300A.DIREXP.tag         then DevSendCmdLocal('@Edit DIREXP');
       if ClickTag=WJ300A.PARAM.DTWAVE.tag   then begin
        if iGetTagBitState(ClickTag,0) then n:=0 else n:=1;
        DevSendCmdLocal(DimRemote+'@AssignTag '+NameTag(ClickTag)+' '+Str(n));
        DevSendCmdLocal(DimRemote+'@GetWaveForm '+Str(n));
       end;
      end;
      {
      Select Plot & Tab windows by curve...
      }
      ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
      if IsRefCurve(ClickCurve) then begin
       iNul(WinSelectByCurve(ClickCurve,ClickCurve));
       bNul(Voice(snd_Wheel));
      end;
      {
      Console commands: @url_encoded_sensor ...
      }
      if LooksLikeCommand(ClickSensor) then begin
       DevPostCmdLocal(url_decode(ClickSensor));
       bNul(Voice(snd_Click));
      end;
     end;
     {
     Handle remote clicks comes from DIM via @DimGuiClick message.
     @DimGuiClick default handler decode and write events to FIFO,
     so we can find it as clicks and can handle it in usual way.
     }
     if ClickIsRemote then begin
      {
      Show time difference.
      }
      if DebugFlagEnabled(dfDetails) then
      Details('Remote Click Time Diff '+Str(mSecNow-rVal(ClickParams('When')))+' ms');
      {
      Handle remote console commands...
      }
      s:=Dim_GuiConsoleRecv(DevName,'');
      if LooksLikeCommand(s) then DevSendCmdLocal(s);
     end;
    end;
    {
    Handle Right mouse button click
    }
    if (ClickButton=VK_RBUTTON) then begin
     SensorHelp(Url_Decode(ClickParams('Hint')));
    end;
   end;
  until (ClickRead=0);
  {
  Edit handling...
  }
  if EditStateDone then begin
   if CheckEditTag(WJ300A.PARAM.DATE.tag,s)     then DevSendCmdLocal('@Setting DATE '+s);
   if CheckEditTag(WJ300A.PARAM.DTBORD.tag,s)   then DevSendCmdLocal('@Setting DTBORD '+s);
   if CheckEditTag(WJ300A.PARAM.DTFORM.tag,s)   then DevSendCmdLocal('@Setting DTFORM '+s);
   if CheckEditTag(WJ300A.PARAM.DTSTART.tag,s)  then DevSendCmdLocal('@Setting DTSTART '+s);
   if CheckEditTag(WJ300A.PARAM.DTPOINTS.tag,s) then DevSendCmdLocal('@Setting DTPOINTS '+s);
   if CheckEditTag(WJ300A.PARAM.MLEN.tag,s)     then DevSendCmdLocal('@Setting MLEN '+s);
   if CheckEditTag(WJ300A.C1.OFFS.tag,s)        then DevSendCmdLocal('@Setting C1:OFST '+s);
   if CheckEditTag(WJ300A.C2.OFFS.tag,s)        then DevSendCmdLocal('@Setting C2:OFST '+s);
   if CheckEditTag(WJ300A.C3.OFFS.tag,s)        then DevSendCmdLocal('@Setting C3:OFST '+s);
   if CheckEditTag(WJ300A.C4.OFFS.tag,s)        then DevSendCmdLocal('@Setting C4:OFST '+s);
   if CheckEditTag(WJ300A.M1.OFFS.tag,s)        then DevSendCmdLocal('@Setting M1:OFST '+s);
   if CheckEditTag(WJ300A.PARAM.TCPL.tag,s)     then DevSendCmdLocal('@Setting TCPL '+s);
   if CheckEditTag(WJ300A.PARAM.TDIV.tag,s)     then DevSendCmdLocal('@Setting TDIV '+s);
   if CheckEditTag(WJ300A.PARAM.TLVL.tag,s)     then DevSendCmdLocal('@Setting TLVL '+s);
   if CheckEditTag(WJ300A.PARAM.TRDL.tag,s)     then DevSendCmdLocal('@Setting TRDL '+s);
   if CheckEditTag(WJ300A.PARAM.TRMD.tag,s)     then DevSendCmdLocal('@Setting TRMD '+s);
   if CheckEditTag(WJ300A.PARAM.TSLP.tag,s)     then DevSendCmdLocal('@Setting TSLP '+s);
   if CheckEditTag(WJ300A.PARAM.TSRC.tag,s)     then DevSendCmdLocal('@Setting TSRC '+s);
   if CheckEditTag(WJ300A.PARAM.TTYP.tag,s)     then DevSendCmdLocal('@Setting TTYP '+s);
   if CheckEditTag(WJ300A.C1.SCAL.tag,s)        then DevSendCmdLocal('@Setting C1:VDIV '+s);
   if CheckEditTag(WJ300A.C2.SCAL.tag,s)        then DevSendCmdLocal('@Setting C2:VDIV '+s);
   if CheckEditTag(WJ300A.C3.SCAL.tag,s)        then DevSendCmdLocal('@Setting C3:VDIV '+s);
   if CheckEditTag(WJ300A.C4.SCAL.tag,s)        then DevSendCmdLocal('@Setting C4:VDIV '+s);
   if CheckEditTag(WJ300A.M1.SCAL.tag,s)        then DevSendCmdLocal('@Setting M1:VDIV '+s);
   if CheckEditTag(WJ300A.PARAM.WAVESRC.tag,s)  then DevSendCmdLocal('@Setting WAVESRC '+s);
   if CheckEditTag(WJ300A.DIRBASE.tag,s)        then SetDirBase(s);
   if CheckEditTag(WJ300A.DIREXP.tag,s)         then SetDirExp(s);
   MenuHandlers;
   {
   Warning, Information dialog completion.
   }
   if EditTestResultName('Warning') then EditReset;
   if EditTestResultName('Information') then EditReset;
  end;
  if EditStateDone then begin
   Problem('Uncompleted edit detected!');
   EditReset;
  end else
  if EditStateError then begin
   Problem('Edit error detected!');
   EditReset;
  end;
  Cleanup;
 end;
 {
 Handle message @AssignTag arg
 }
 procedure WJ300A_OnAssignTag(arg:String);
 var tag:Integer; w1,w2:String;
  procedure Cleanup;
  begin
   w1:=''; w2:='';
  end;
 begin
  Cleanup;
  if (arg<>'') then begin
   w1:=ExtractWord(1,arg);
   tag:=FindTag(w1);
   if (tag<>0) then begin
    w2:=ExtractWord(2,arg);
    if tag=WJ300A.POLL.ENABLE.tag  then UpdateTag(tag,w2,0,1) else
    if tag=WJ300A.PARAM.DTWAVE.tag then UpdateTag(tag,w2,0,1);
   end;
  end;
  Cleanup;
 end;
 {
 Handle message @DimUpdateTag arg
 }
 procedure WJ300A_OnDimUpdateTag(arg:String);
 var tag,typ:Integer; x,y:Real;
 begin
  if (arg<>'') then begin
   if DIM_IsClientMode and not DIM_IsServerMode then begin
    tag:=FindTag(ExtractWord(1,arg));
    if (tag<>0) then begin
     typ:=TypeTag(tag);
     if (typ=1) then y:=iGetTag(tag) else
     if (typ=2) then y:=rGetTag(tag) else y:=_Nan;
     x:=time;
     if not IsNan(y) then begin
      if tag=WJ300A.PARAM.ESR.tag   then UpdateAo(ao_ESR,x,y);
      if tag=WJ300A.C1.OFFS.tag     then UpdateAo(ao_OFST_C1,x,y);
      if tag=WJ300A.C2.OFFS.tag     then UpdateAo(ao_OFST_C2,x,y);
      if tag=WJ300A.C3.OFFS.tag     then UpdateAo(ao_OFST_C3,x,y);
      if tag=WJ300A.C4.OFFS.tag     then UpdateAo(ao_OFST_C4,x,y);
      if tag=WJ300A.M1.OFFS.tag     then UpdateAo(ao_OFST_M1,x,y);
      if tag=WJ300A.PARAM.TDIV.tag  then UpdateAo(ao_TDIV,x,y);
      if tag=WJ300A.PARAM.TLVL.tag  then UpdateAo(ao_TLVL,x,y);
      if tag=WJ300A.PARAM.TRDL.tag  then UpdateAo(ao_TRDL,x,y);
      if tag=WJ300A.C1.SCAL.tag     then UpdateAo(ao_VDIV_C1,x,y);
      if tag=WJ300A.C2.SCAL.tag     then UpdateAo(ao_VDIV_C2,x,y);
      if tag=WJ300A.C3.SCAL.tag     then UpdateAo(ao_VDIV_C3,x,y);
      if tag=WJ300A.C4.SCAL.tag     then UpdateAo(ao_VDIV_C4,x,y);
      if tag=WJ300A.M1.SCAL.tag     then UpdateAo(ao_VDIV_M1,x,y);
      if tag=WJ300A.ERROR.COUNT.tag then UpdateDo(ao_ERROR_CNT,x,y);
      if tag=WJ300A.POLL.RATE.tag   then UpdateDo(ao_POLL_RATE,x,y);
     end;
    end;
   end;
  end;
 end;
 //
 // Tags filling
 //
 procedure WJ300A_FillTags(InitVal:Real);
 begin
  WJ300A.POLL.ENABLE.val:=InitVal;
  WJ300A.PARAM.ACQ.val:=InitVal;
  WJ300A.C1.COUP.val:=InitVal;
  WJ300A.C2.COUP.val:=InitVal;
  WJ300A.C3.COUP.val:=InitVal;
  WJ300A.C4.COUP.val:=InitVal;
  WJ300A.PARAM.DATE.val:=InitVal;
  WJ300A.PARAM.DTBORD.val:=InitVal;
  WJ300A.PARAM.DTFORM.val:=InitVal;
  WJ300A.PARAM.DTSTART.val:=InitVal;
  WJ300A.PARAM.DTPOINTS.val:=InitVal;
  WJ300A.PARAM.DTWAVE.val:=InitVal;
  WJ300A.PARAM.ESR.val:=InitVal;
  WJ300A.PARAM.IDN.val:=InitVal;
  WJ300A.PARAM.MLEN.val:=InitVal;
  WJ300A.C1.OFFS.val:=InitVal;
  WJ300A.C2.OFFS.val:=InitVal;
  WJ300A.C3.OFFS.val:=InitVal;
  WJ300A.C4.OFFS.val:=InitVal;
  WJ300A.M1.OFFS.val:=InitVal;
  WJ300A.PARAM.TCPL.val:=InitVal;
  WJ300A.PARAM.TDIV.val:=InitVal;
  WJ300A.PARAM.TESR.val:=InitVal;
  WJ300A.PARAM.TLVL.val:=InitVal;
  WJ300A.C1.TRA.val:=InitVal;
  WJ300A.C2.TRA.val:=InitVal;
  WJ300A.C3.TRA.val:=InitVal;
  WJ300A.C4.TRA.val:=InitVal;
  WJ300A.M1.TRA.val:=InitVal;
  WJ300A.PARAM.TRDL.val:=InitVal;
  WJ300A.PARAM.TRG.val:=InitVal;
  WJ300A.PARAM.TRMD.val:=InitVal;
  WJ300A.PARAM.TSLP.val:=InitVal;
  WJ300A.PARAM.TSRC.val:=InitVal;
  WJ300A.PARAM.TTYP.val:=InitVal;
  WJ300A.C1.SCAL.val:=InitVal;
  WJ300A.C2.SCAL.val:=InitVal;
  WJ300A.C3.SCAL.val:=InitVal;
  WJ300A.C4.SCAL.val:=InitVal;
  WJ300A.M1.SCAL.val:=InitVal;
  WJ300A.C1.WAVFORM.val:=InitVal;
  WJ300A.C2.WAVFORM.val:=InitVal;
  WJ300A.C3.WAVFORM.val:=InitVal;
  WJ300A.C4.WAVFORM.val:=InitVal;
  WJ300A.PARAM.WAVESRC.val:=InitVal;
  WJ300A.ERROR.COUNT.val:=InitVal;
  WJ300A.POLL.RATE.val:=InitVal;
  WJ300A.DIRBASE.val:=InitVal;
  WJ300A.DIREXP.val:=InitVal;
 end;
 {
 Update DIM services
 }
 procedure DimUpdateState;
 begin
  if DIM_IsServerMode then begin
   // Enforce update each 10 sec
   if SysTimer_Pulse(10000)>0 then WJ300A_FillTags(-MaxReal);
   if ShouldRefresh(WJ300A.POLL.ENABLE.val,    GetStampOfTag(WJ300A.POLL.ENABLE.tag,0))>0    then DIM_UpdateTag(WJ300A.POLL.ENABLE.tag,'');
   if ShouldRefresh(WJ300A.PARAM.ACQ.val,      GetStampOfTag(WJ300A.PARAM.ACQ.tag,0))>0      then DIM_UpdateTag(WJ300A.PARAM.ACQ.tag,'');
   if ShouldRefresh(WJ300A.C1.COUP.val,        GetStampOfTag(WJ300A.C1.COUP.tag,0))>0        then DIM_UpdateTag(WJ300A.C1.COUP.tag,'');
   if ShouldRefresh(WJ300A.C2.COUP.val,        GetStampOfTag(WJ300A.C2.COUP.tag,0))>0        then DIM_UpdateTag(WJ300A.C2.COUP.tag,'');
   if ShouldRefresh(WJ300A.C3.COUP.val,        GetStampOfTag(WJ300A.C3.COUP.tag,0))>0        then DIM_UpdateTag(WJ300A.C3.COUP.tag,'');
   if ShouldRefresh(WJ300A.C4.COUP.val,        GetStampOfTag(WJ300A.C4.COUP.tag,0))>0        then DIM_UpdateTag(WJ300A.C4.COUP.tag,'');
   if ShouldRefresh(WJ300A.PARAM.DATE.val,     GetStampOfTag(WJ300A.PARAM.DATE.tag,0))>0     then DIM_UpdateTag(WJ300A.PARAM.DATE.tag,'');
   if ShouldRefresh(WJ300A.PARAM.DTBORD.val,   GetStampOfTag(WJ300A.PARAM.DTBORD.tag,0))>0   then DIM_UpdateTag(WJ300A.PARAM.DTBORD.tag,'');
   if ShouldRefresh(WJ300A.PARAM.DTFORM.val,   GetStampOfTag(WJ300A.PARAM.DTFORM.tag,0))>0   then DIM_UpdateTag(WJ300A.PARAM.DTFORM.tag,'');
   if ShouldRefresh(WJ300A.PARAM.DTSTART.val,  GetStampOfTag(WJ300A.PARAM.DTSTART.tag,0))>0  then DIM_UpdateTag(WJ300A.PARAM.DTSTART.tag,'');
   if ShouldRefresh(WJ300A.PARAM.DTPOINTS.val, GetStampOfTag(WJ300A.PARAM.DTPOINTS.tag,0))>0 then DIM_UpdateTag(WJ300A.PARAM.DTPOINTS.tag,'');
   if ShouldRefresh(WJ300A.PARAM.ESR.val,      GetStampOfTag(WJ300A.PARAM.ESR.tag,0))>0      then DIM_UpdateTag(WJ300A.PARAM.ESR.tag,'');
   if ShouldRefresh(WJ300A.PARAM.IDN.val,      GetStampOfTag(WJ300A.PARAM.IDN.tag,0))>0      then DIM_UpdateTag(WJ300A.PARAM.IDN.tag,'');
   if ShouldRefresh(WJ300A.PARAM.MLEN.val,     GetStampOfTag(WJ300A.PARAM.MLEN.tag,0))>0     then DIM_UpdateTag(WJ300A.PARAM.MLEN.tag,'');
   if ShouldRefresh(WJ300A.C1.OFFS.val,        GetStampOfTag(WJ300A.C1.OFFS.tag,0))>0        then DIM_UpdateTag(WJ300A.C1.OFFS.tag,'');
   if ShouldRefresh(WJ300A.C2.OFFS.val,        GetStampOfTag(WJ300A.C2.OFFS.tag,0))>0        then DIM_UpdateTag(WJ300A.C2.OFFS.tag,'');
   if ShouldRefresh(WJ300A.C3.OFFS.val,        GetStampOfTag(WJ300A.C3.OFFS.tag,0))>0        then DIM_UpdateTag(WJ300A.C3.OFFS.tag,'');
   if ShouldRefresh(WJ300A.C4.OFFS.val,        GetStampOfTag(WJ300A.C4.OFFS.tag,0))>0        then DIM_UpdateTag(WJ300A.C4.OFFS.tag,'');
   if ShouldRefresh(WJ300A.M1.OFFS.val,        GetStampOfTag(WJ300A.M1.OFFS.tag,0))>0        then DIM_UpdateTag(WJ300A.M1.OFFS.tag,'');
   if ShouldRefresh(WJ300A.PARAM.TCPL.val,     GetStampOfTag(WJ300A.PARAM.TCPL.tag,0))>0     then DIM_UpdateTag(WJ300A.PARAM.TCPL.tag,'');
   if ShouldRefresh(WJ300A.PARAM.TDIV.val,     GetStampOfTag(WJ300A.PARAM.TDIV.tag,0))>0     then DIM_UpdateTag(WJ300A.PARAM.TDIV.tag,'');
   if ShouldRefresh(WJ300A.PARAM.TESR.val,     GetStampOfTag(WJ300A.PARAM.TESR.tag,0))>0     then DIM_UpdateTag(WJ300A.PARAM.TESR.tag,'');
   if ShouldRefresh(WJ300A.PARAM.TLVL.val,     GetStampOfTag(WJ300A.PARAM.TLVL.tag,0))>0     then DIM_UpdateTag(WJ300A.PARAM.TLVL.tag,'');
   if ShouldRefresh(WJ300A.C1.TRA.val,         GetStampOfTag(WJ300A.C1.TRA.tag,0))>0         then DIM_UpdateTag(WJ300A.C1.TRA.tag,'');
   if ShouldRefresh(WJ300A.C2.TRA.val,         GetStampOfTag(WJ300A.C2.TRA.tag,0))>0         then DIM_UpdateTag(WJ300A.C2.TRA.tag,'');
   if ShouldRefresh(WJ300A.C3.TRA.val,         GetStampOfTag(WJ300A.C3.TRA.tag,0))>0         then DIM_UpdateTag(WJ300A.C3.TRA.tag,'');
   if ShouldRefresh(WJ300A.C4.TRA.val,         GetStampOfTag(WJ300A.C4.TRA.tag,0))>0         then DIM_UpdateTag(WJ300A.C4.TRA.tag,'');
   if ShouldRefresh(WJ300A.M1.TRA.val,         GetStampOfTag(WJ300A.M1.TRA.tag,0))>0         then DIM_UpdateTag(WJ300A.M1.TRA.tag,'');
   if ShouldRefresh(WJ300A.PARAM.TRDL.val,     GetStampOfTag(WJ300A.PARAM.TRDL.tag,0))>0     then DIM_UpdateTag(WJ300A.PARAM.TRDL.tag,'');
   if ShouldRefresh(WJ300A.PARAM.TRG.val,      GetStampOfTag(WJ300A.PARAM.TRG.tag,0))>0      then DIM_UpdateTag(WJ300A.PARAM.TRG.tag,'');
   if ShouldRefresh(WJ300A.PARAM.TRMD.val,     GetStampOfTag(WJ300A.PARAM.TRMD.tag,0))>0     then DIM_UpdateTag(WJ300A.PARAM.TRMD.tag,'');
   if ShouldRefresh(WJ300A.PARAM.TSLP.val,     GetStampOfTag(WJ300A.PARAM.TSLP.tag,0))>0     then DIM_UpdateTag(WJ300A.PARAM.TSLP.tag,'');
   if ShouldRefresh(WJ300A.PARAM.TSRC.val,     GetStampOfTag(WJ300A.PARAM.TSRC.tag,0))>0     then DIM_UpdateTag(WJ300A.PARAM.TSRC.tag,'');
   if ShouldRefresh(WJ300A.PARAM.TTYP.val,     GetStampOfTag(WJ300A.PARAM.TTYP.tag,0))>0     then DIM_UpdateTag(WJ300A.PARAM.TTYP.tag,'');
   if ShouldRefresh(WJ300A.C1.SCAL.val,        GetStampOfTag(WJ300A.C1.SCAL.tag,0))>0        then DIM_UpdateTag(WJ300A.C1.SCAL.tag,'');
   if ShouldRefresh(WJ300A.C2.SCAL.val,        GetStampOfTag(WJ300A.C2.SCAL.tag,0))>0        then DIM_UpdateTag(WJ300A.C2.SCAL.tag,'');
   if ShouldRefresh(WJ300A.C3.SCAL.val,        GetStampOfTag(WJ300A.C3.SCAL.tag,0))>0        then DIM_UpdateTag(WJ300A.C3.SCAL.tag,'');
   if ShouldRefresh(WJ300A.C4.SCAL.val,        GetStampOfTag(WJ300A.C4.SCAL.tag,0))>0        then DIM_UpdateTag(WJ300A.C4.SCAL.tag,'');
   if ShouldRefresh(WJ300A.M1.SCAL.val,        GetStampOfTag(WJ300A.M1.SCAL.tag,0))>0        then DIM_UpdateTag(WJ300A.M1.SCAL.tag,'');
   if ShouldRefresh(WJ300A.C1.WAVFORM.val,     GetStampOfTag(WJ300A.C1.WAVFORM.tag,0))>0     then DIM_UpdateTag(WJ300A.C1.WAVFORM.tag,'');
   if ShouldRefresh(WJ300A.C2.WAVFORM.val,     GetStampOfTag(WJ300A.C2.WAVFORM.tag,0))>0     then DIM_UpdateTag(WJ300A.C2.WAVFORM.tag,'');
   if ShouldRefresh(WJ300A.C3.WAVFORM.val,     GetStampOfTag(WJ300A.C3.WAVFORM.tag,0))>0     then DIM_UpdateTag(WJ300A.C3.WAVFORM.tag,'');
   if ShouldRefresh(WJ300A.C4.WAVFORM.val,     GetStampOfTag(WJ300A.C4.WAVFORM.tag,0))>0     then DIM_UpdateTag(WJ300A.C4.WAVFORM.tag,'');
   if ShouldRefresh(WJ300A.PARAM.WAVESRC.val,  GetStampOfTag(WJ300A.PARAM.WAVESRC.tag,0))>0  then DIM_UpdateTag(WJ300A.PARAM.WAVESRC.tag,'');
   if ShouldRefresh(WJ300A.ERROR.COUNT.val,    GetStampOfTag(WJ300A.ERROR.COUNT.tag,0))>0    then DIM_UpdateTag(WJ300A.ERROR.COUNT.tag,'');
   if ShouldRefresh(WJ300A.POLL.RATE.val,      GetStampOfTag(WJ300A.POLL.RATE.tag,0))>0      then DIM_UpdateTag(WJ300A.POLL.RATE.tag,'');
   if ShouldRefresh(WJ300A.DIRBASE.val,        GetStampOfTag(WJ300A.DIRBASE.tag,0))>0        then DIM_UpdateTag(WJ300A.DIRBASE.tag,'');
   if ShouldRefresh(WJ300A.DIREXP.val,         GetStampOfTag(WJ300A.DIREXP.tag,0))>0         then DIM_UpdateTag(WJ300A.DIREXP.tag,'');
   if ShouldRefresh(WJ300A.SAVESTAT.val,       GetStampOfTag(WJ300A.SAVESTAT.tag,0))>0       then DIM_UpdateTag(WJ300A.SAVESTAT.tag,'');
  end;
 end;
 {
 WJ300A tags initialization.
 }
 procedure WJ300A_InitTags(Prefix:String);
 begin
  if not IsEmptyStr(Prefix) then begin
   DIM_GuiClickInit(Prefix+'.DIMGUICLICK');
   InitTag(WJ300A.POLL.ENABLE.tag,    Prefix+'.POLL_ENABLE', 1);
   InitTag(WJ300A.POLL.RATE.tag,      Prefix+'.POLL_RATE',   2);
   InitTag(WJ300A.ERROR.COUNT.tag,    Prefix+'.ERROR_CNT',   2);
   InitTag(WJ300A.C1.COUP.tag,        Prefix+'.CPL_C1',      3);
   InitTag(WJ300A.C1.OFFS.tag,        Prefix+'.OFST_C1',     2);
   InitTag(WJ300A.C1.TRA.tag,         Prefix+'.TRA_C1',      1);
   InitTag(WJ300A.C1.SCAL.tag,        Prefix+'.VDIV_C1',     2);
   InitTag(WJ300A.C2.COUP.tag,        Prefix+'.CPL_C2',      3);
   InitTag(WJ300A.C2.OFFS.tag,        Prefix+'.OFST_C2',     2);
   InitTag(WJ300A.C2.TRA.tag,         Prefix+'.TRA_C2',      1);
   InitTag(WJ300A.C2.SCAL.tag,        Prefix+'.VDIV_C2',     2);
   InitTag(WJ300A.C3.COUP.tag,        Prefix+'.CPL_C3',      3);
   InitTag(WJ300A.C3.OFFS.tag,        Prefix+'.OFST_C3',     2);
   InitTag(WJ300A.C3.TRA.tag,         Prefix+'.TRA_C3',      1);
   InitTag(WJ300A.C3.SCAL.tag,        Prefix+'.VDIV_C3',     2);
   InitTag(WJ300A.C4.COUP.tag,        Prefix+'.CPL_C4',      3);
   InitTag(WJ300A.C4.OFFS.tag,        Prefix+'.OFST_C4',     2);
   InitTag(WJ300A.C4.TRA.tag,         Prefix+'.TRA_C4',      1);
   InitTag(WJ300A.C4.SCAL.tag,        Prefix+'.VDIV_C4',     2);
   InitTag(WJ300A.C1.WAVFORM.tag,     Prefix+'.WAVFORM_C1',  3);
   InitTag(WJ300A.C2.WAVFORM.tag,     Prefix+'.WAVFORM_C2',  3);
   InitTag(WJ300A.C3.WAVFORM.tag,     Prefix+'.WAVFORM_C3',  3);
   InitTag(WJ300A.C4.WAVFORM.tag,     Prefix+'.WAVFORM_C4',  3);
   InitTag(WJ300A.M1.OFFS.tag,        Prefix+'.OFST_M1',     2);
   InitTag(WJ300A.M1.TRA.tag,         Prefix+'.TRA_M1',      1);
   InitTag(WJ300A.M1.SCAL.tag,        Prefix+'.VDIV_M1',     2);
   InitTag(WJ300A.PARAM.ACQ.tag,      Prefix+'.ACQ',         3);
   InitTag(WJ300A.PARAM.DATE.tag,     Prefix+'.DATE',        3);
   InitTag(WJ300A.PARAM.DTBORD.tag,   Prefix+'.DTBORD',      3);
   InitTag(WJ300A.PARAM.DTFORM.tag,   Prefix+'.DTFORM',      3);
   InitTag(WJ300A.PARAM.DTSTART.tag,  Prefix+'.DTSTART',     1);
   InitTag(WJ300A.PARAM.DTPOINTS.tag, Prefix+'.DTPOINTS',    1);
   InitTag(WJ300A.PARAM.DTWAVE.tag,   Prefix+'.DTWAVE',      1);
   InitTag(WJ300A.PARAM.ESR.tag,      Prefix+'.ESR',         1);
   InitTag(WJ300A.PARAM.IDN.tag,      Prefix+'.IDN',         3);
   InitTag(WJ300A.PARAM.MLEN.tag,     Prefix+'.MLEN',        3);
   InitTag(WJ300A.PARAM.TCPL.tag,     Prefix+'.TCPL',        3);
   InitTag(WJ300A.PARAM.TDIV.tag,     Prefix+'.TDIV',        2);
   InitTag(WJ300A.PARAM.TESR.tag,     Prefix+'.TESR',        1);
   InitTag(WJ300A.PARAM.TLVL.tag,     Prefix+'.TLVL',        2);
   InitTag(WJ300A.PARAM.TRDL.tag,     Prefix+'.TRDL',        2);
   InitTag(WJ300A.PARAM.TRMD.tag,     Prefix+'.TRMD',        1);
   InitTag(WJ300A.PARAM.TSLP.tag,     Prefix+'.TSLP',        3);
   InitTag(WJ300A.PARAM.TSRC.tag,     Prefix+'.TSRC',        3);
   InitTag(WJ300A.PARAM.TTYP.tag,     Prefix+'.TTYP',        3);
   InitTag(WJ300A.PARAM.WAVESRC.tag,  Prefix+'.WAVESRC',     3);
   InitTag(WJ300A.DIRBASE.tag,        Prefix+'.DIRBASE',     3);
   InitTag(WJ300A.DIREXP.tag,         Prefix+'.DIREXP',      3);
   InitTag(WJ300A.DEFAULTDIR.tag,     Prefix+'.DEFAULTDIR',  3);
   InitTag(WJ300A.EXTRACTNUM.tag,     Prefix+'.EXTRACTNUM',  3);
   InitTag(WJ300A.SAVESTAT.tag,       Prefix+'.SAVESTAT',    1);
  end else Trouble('Tag initialized: tag prefix undefined...');
 end;
 {
 WJ300A cleanup.
 }
 procedure WJ300A_Clear;
 begin
  WJ300A_Clear_Cmd;
  WJ300A.Pipe.PipeId:=0;
  WJ300A.Pipe.Req:='';
  WJ300A.Pipe.Ans:='';
  WJ300A.Pipe.Buf:='';
  WJ300A.SelfId:='';
  WJ300A.TagPrefix:='';
  WJ300A.FilePrefix:='';
  bNul(sSetTag(WJ300A.EXTRACTNUM.tag,''));
 end;
 {
 WJ300A initialization.
 }
 procedure WJ300A_Init;
 begin
  {
  Register error codes
  }
  WJ300A.ecTimeOut:=RegisterErr(progname+': TimeOut');
  {
  Initialize variables
  }
  WJ300A.Pipe.ReqTime:=0;
  WJ300A.Pipe.AnsTime:=0;
  WJ300A.Pipe.LastPoll:=0;
  WJ300A.Pipe.Status:=st_NoReq;
  if DIM_IsServerMode then begin
   WJ300A.GetWaveFormTimer:=0;
   WJ300A.GetWaveFormSource:=1;
   WJ300A.GetTrigStatusTimer:=0;
  end;
  {
  Initialize command table.
  }
  WJ300A_Init_Cmd;
  {
  Initialize channel, tags and dir waveform
  }
  WJ300A_InitTags(ReadIni('tagPrefix'));
  bNul(sSetTag(WJ300A.DIRBASE.tag,AddPathDelim(DaqFileRef(LoCaseStr(AdaptFileName(ReadIni('DefaultDirBase'))),''))));
  bNul(sSetTag(WJ300A.DIREXP.tag,AddPathDelim(LoCaseStr(AdaptFileName(ReadIni('DefaultDirExp'))))));
  bNul(sSetTag(WJ300A.DEFAULTDIR.tag,AddPathDelim(DaqFileRef(LoCaseStr(AdaptFileName(ReadIni('DefaultDirBase'))),''))+AddPathDelim(LoCaseStr(AdaptFileName(ReadIni('DefaultDirExp'))))));
  bNul(sSetTag(WJ300A.EXTRACTNUM.tag,''));
  {
  Read ini file
  }
  WJ300A.Simulator:=iValDef(ReadIni('Simulator'),0)<>0; Success('Simulator='+Str(Ord(WJ300A.Simulator)));
  WJ300A.FilePrefix:=ReadIni('filePrefix');
  WJ300A.CalcDevRef:=RefFind('Device '+ReadIni('CalcDevName'));
  if WJ300A.CalcDevRef=0 then Trouble('Device not found') else Success('CalcDevName = '+ReadIni('CalcDevName'));
  if WJ300A.Simulator then Success('Run simulator mode.') else Success('Run as driver mode.');
  WJ300A.Pipe.TimeOut:=iValDef(ReadIni('PipeTimeOut'),1500);  Success('PipeTimeOut = '+Str(WJ300A.Pipe.TimeOut));
  WJ300A.Pipe.TimeGap:=iValDef(ReadIni('PipeTimeGap'),10);    Success('PipeTimeGap = '+Str(WJ300A.Pipe.TimeGap));
  WJ300A.Pipe.PollPeriod:=iValDef(ReadIni('PollPeriod'),100); Success('PollPeriod = '+Str(WJ300A.Pipe.PollPeriod));
  if DIM_IsServerMode then Success('DIM: Run as Server') else if DIM_IsClientMode then Success('DIM: Run as Client');
  {
  Initialize Pipe
  }
  bNul(WJ300A_PipeConnect);
  {
  Default settings
  }
  DevSendCmdLocal(DimRemote+'@Setting DTFORM WORD');
  DevSendCmdLocal(DimRemote+'@Setting DTBORD H/L');
 end;
 {
 WJ300A finalization.
 }
 procedure WJ300A_Free;
 begin
  { Pipe free for finalization application }
  bNul(pipe_free(WJ300A.Pipe.PipeID));
  bNul(pipe_free(WJ300A.Pipe.PipeSIM));
 end;
 {
 WJ300A Simulator: Read waveform from file
 }
 function ReadSimWave:String;
 var io,i,v:Integer; data:String;
  procedure Cleanup;
  begin
   data:='';
  end;
 begin
  Cleanup;
  if FileExists(DaqFileRef('..\DATA\simdat','.raw')) then io:=(f_reset(DaqFileRef('..\DATA\simdat','.raw'),0))
  else ShowTooltip('text "Неверно указан файл или его не существует" delay 10000 preset stdWarning');
  if io=0 then begin
   data:=f_read(round(f_size));
   ReadSimWave:=copy(data,(NthPos('#8',data,GetSelectedChannel)),(NthPos('#8',data,GetSelectedChannel+1)-(NthPos('#8',data,GetSelectedChannel))));
  end else ShowTooltip('text "Ошибка открытия файла" delay 10000 preset stdWarning');
  Cleanup;
 end;
 {
 WJ300A Simulator Command Cycle polling.
 }
 procedure WJ300A_SIM_POLL;
 var Req,AcronymReq:String; i:Integer;
 begin
  Req:='';
  Req:=copy(WJ300A.Pipe.Req,HeaderLength+1,length(WJ300A.Pipe.Req)-HeaderLength);
  if Length(Req)>0 then begin
   ViewImp('[SIM] Pipe < '+Req);
   if Length(Req)>0 then begin
    for i:=1 to MaxCmdNum do
      if IsSameText(Req,WJ300A.Cmd.Acronym[i]+'?') then
       WJ300A.Pipe.Ans:=WJ300A.Cmd.SimData[i]
      else if IsSameText(ExtractWord(1,Req),WJ300A.Cmd.Acronym[i]) then WJ300A.Cmd.SimData[i]:=ExtractWord(2,Req);
    { Одобряем все остальные запросы, без проверки }
    if WJ300A.Pipe.Ans<>'' then WJ300A.Pipe.Ans:=AddHeader(WJ300A.Pipe.Ans,false)+CRLF;
    { Отправка осциллограммы }
    if IsSameText(Req,'DTFORM WORD;DTPOINTS 500000;DTSTART 0;DTWAVE?') then WJ300A.Pipe.Ans:=AddHeader(ReadSimWave,true);
    {
    Send reply to requests
    }
    if not IsEmptyStr(WJ300A.Pipe.Ans) then if (pipe_send(WJ300A.Pipe.PipeSIM,WJ300A.Pipe.Ans)>0) then
    ViewExp('[SIM] Pipe > '+WJ300A.Pipe.Ans)
    else Trouble('[SIM] Could not send '+WJ300A.Pipe.Ans);
   end;
  end;
  WJ300A.Pipe.Req:='';
  WJ300A.Pipe.Ans:='';
  Req:='';
 end;
 {
 WJ300A Driver Command Cycle polling.
 }
 procedure WJ300A_CMD_POLL;
 begin
  WJ300A_GUI_POLL;
  if SysTimer_Pulse(1000)>0 then begin
   bNul(rSetTag(WJ300A.ERROR.COUNT.tag,GetErrCount(-1)));
   UpdateAo(ao_ERROR_CNT,time,GetErrCount(-1));
   bNul(rSetTag(WJ300A.POLL.RATE.tag,WJ300A.Pipe.PollRate));
   UpdateAo(ao_POLL_RATE,time,WJ300A.Pipe.PollRate);
   WJ300A.Pipe.PollRate:=0;
  end;
  if mSecNow-FixmSecNow>DelayOnStart then WJ300A_CMD_CYCLE;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  WJ300A_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  StdIn_SetScripts('','');
  StdIn_SetTimeouts(0,0,0,MaxInt);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  cmd_PrintCmdTable         := RegisterStdInCmd('@PrintCmdTable',  '');
  cmd_Setting               := RegisterStdInCmd('@Setting',        '');
  cmd_Edit                  := RegisterStdInCmd('@Edit',           '');
  cmd_DimTagUpdate          := RegisterStdInCmd('@DimTagUpdate',   '');
  cmd_AssignTag             := RegisterStdInCmd('@AssignTag',      '');
  cmd_LoadIni               := RegisterStdInCmd('@LoadIni',        '');
  cmd_SaveIni               := RegisterStdInCmd('@SaveIni',        '');
  cmd_GetWaveForm           := RegisterStdInCmd('@GetWaveForm',    '');
  cmd_Autoset               := RegisterStdInCmd('@Autoset',        '');
  cmd_TrigRun               := RegisterStdInCmd('@TrigRun',        '');
  cmd_TrigStop              := RegisterStdInCmd('@TrigStop',       '');
  cmd_DimCmdMy              := RegisterStdInCmd('@DimCmdMy',       '');
  cmd_SingleStart           := RegisterStdInCmd('@SingleStart',    '');
  cmd_SingleStop            := RegisterStdInCmd('@SingleStop',     '');
  cmd_SetDirBase            := RegisterStdInCmd('@SetDirBase',     '');
  cmd_SetDirExp             := RegisterStdInCmd('@SetDirExp',      '');
  if Val(ReadIni('CustomIniAutoLoad'))=1 then iNul(CustomIniRw('R','',2));
  WJ300A_Init;
  RunStartupScript;
  end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  if Val(ReadIni('CustomIniAutoSave'))=1 then iNul(CustomIniRW('W','',2));
  RunFinallyScript;
  WJ300A_Free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  if WJ300A.Simulator then WJ300A_SIM_POLL;
  WJ300A_CMD_POLL;
  TrigStatusUpdate;
  GetWaveFormData;
  DimUpdateState;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg, v:String; cmdid,tag,n:Integer;
  procedure Cleanup;
  begin
   cmd:='';
   arg:='';
   v:='';
  end;
  procedure SettingCmd(n:Integer; v:String);
  begin
   if IsValidCmdNum(n) then begin
    Success(cmd+' '+WJ300A.Cmd.Acronym[n]+' '+v);
    WJ300A.Cmd.OpBuff[n]:=v;
    EnableCmdNum(n,true);
   end;
  end;
  procedure UpdateSettingsAfterLoadIni;
  begin
   if (iGetTag(WJ300A.POLL.ENABLE.tag)<>0) then begin
   end;
  end;
 begin
  Cleanup;
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @Setting SETP 20
   }
   if (cmdid=cmd_Setting) then begin
    { if not WJ300A.Simulator then begin }
     n:=WJ300A_Cmd_Find(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      sNul(worddelims(' '));
      v:=ExtractWord(2,arg);
      sNul(WordDelims(Dump(0)));
      if (n=cm_ACQ)      then SettingCmd(n,v);
      if (n=cm_CPL_C1)   then SettingCmd(n,v);
      if (n=cm_CPL_C2)   then SettingCmd(n,v);
      if (n=cm_CPL_C3)   then SettingCmd(n,v);
      if (n=cm_CPL_C4)   then SettingCmd(n,v);
      if (n=cm_CLS)      then SettingCmd(n,v);
      if (n=cm_DATE)     then SettingCmd(n,v);
      if (n=cm_DTBORD)   then SettingCmd(n,v);
      if (n=cm_DTFORM)   then SettingCmd(n,v);
      if (n=cm_DTSTART)  then SettingCmd(n,v);
      if (n=cm_DTPOINTS) then SettingCmd(n,v);
      if (n=cm_ESR)      then SettingCmd(n,v);
      if (n=cm_MLEN)     then SettingCmd(n,v);
      if (n=cm_OFST_C1)  then SettingCmd(n,v);
      if (n=cm_OFST_C2)  then SettingCmd(n,v);
      if (n=cm_OFST_C3)  then SettingCmd(n,v);
      if (n=cm_OFST_C4)  then SettingCmd(n,v);
      if (n=cm_OFST_M1)  then SettingCmd(n,v);
      if (n=cm_RST)      then SettingCmd(n,v);
      if (n=cm_TCPL)     then SettingCmd(n,v);
      if (n=cm_TDIV)     then SettingCmd(n,v);
      if (n=cm_TLVL)     then SettingCmd(n,v);
      if (n=cm_TRA_C1)   then SettingCmd(n,v);
      if (n=cm_TRA_C2)   then SettingCmd(n,v);
      if (n=cm_TRA_C3)   then SettingCmd(n,v);
      if (n=cm_TRA_C4)   then SettingCmd(n,v);
      if (n=cm_TRA_M1)   then SettingCmd(n,v);
      if (n=cm_TRDL)     then SettingCmd(n,v);
      if (n=cm_TRMD)     then SettingCmd(n,v);
      if (n=cm_TSLP)     then SettingCmd(n,v);
      if (n=cm_TSRC)     then SettingCmd(n,v);
      if (n=cm_TTYP)     then SettingCmd(n,v);
      if (n=cm_VDIV_C1)  then SettingCmd(n,v);
      if (n=cm_VDIV_C2)  then SettingCmd(n,v);
      if (n=cm_VDIV_C3)  then SettingCmd(n,v);
      if (n=cm_VDIV_C4)  then SettingCmd(n,v);
      if (n=cm_VDIV_M1)  then SettingCmd(n,v);
      if (n=cm_WAVESRC)  then SettingCmd(n,v);
     end;
    { end; }
    Data:='';
   end else
   {
   @Edit WAV.POIN
   }
   if (cmdid=cmd_Edit) then begin
    if IsPollingEnabled then begin
     if IsSameText(arg,'ACQ')      then StartEditTag(WJ300A.PARAM.ACQ.tag,'Режим измерения:');
     if IsSameText(arg,'CPL_C1')   then StartEditTag(WJ300A.C1.COUP.tag,'Режим &&&я:');
     if IsSameText(arg,'CPL_C2')   then StartEditTag(WJ300A.C2.COUP.tag,'Режим &&:');
     if IsSameText(arg,'CPL_C3')   then StartEditTag(WJ300A.C3.COUP.tag,'Режим &&:');
     if IsSameText(arg,'CPL_C4')   then StartEditTag(WJ300A.C4.COUP.tag,'Режим &&:');
     if IsSameText(arg,'DATE')     then StartEditTag(WJ300A.PARAM.DATE.tag,'Установка времени:');
     if IsSameText(arg,'DTBORD')   then StartEditTag(WJ300A.PARAM.DTBORD.tag,'Порядок бит:');
     if IsSameText(arg,'DTFORM')   then StartEditTag(WJ300A.PARAM.DTFORM.tag,'Формат данных:');
     if IsSameText(arg,'DTSTART')  then StartEditTag(WJ300A.PARAM.DTSTART.tag,'Начальная точка:');
     if IsSameText(arg,'DTPOINTS') then StartEditTag(WJ300A.PARAM.DTPOINTS.tag,'Количнство точек:');
     if IsSameText(arg,'MLEN')     then StartEditTag(WJ300A.PARAM.MLEN.tag,'Количество точек:');
     if IsSameText(arg,'OFST_C1')  then StartEditTag(WJ300A.C1.OFFS.tag,'Вертикальное смещение:');
     if IsSameText(arg,'OFST_C2')  then StartEditTag(WJ300A.C2.OFFS.tag,'Вертикальное смещение:');
     if IsSameText(arg,'OFST_C3')  then StartEditTag(WJ300A.C3.OFFS.tag,'Вертикальное смещение:');
     if IsSameText(arg,'OFST_C4')  then StartEditTag(WJ300A.C4.OFFS.tag,'Вертикальное смещение:');
     if IsSameText(arg,'OFST_M1')  then StartEditTag(WJ300A.M1.OFFS.tag,'Вертикальное смещение:');
     if IsSameText(arg,'TCPL')     then StartEditTag(WJ300A.PARAM.TCPL.tag,'Согласование триггера:');
     if IsSameText(arg,'TDIV')     then StartEditTag(WJ300A.PARAM.TDIV.tag,'Развертка /10:');
     if IsSameText(arg,'TLVL')     then StartEditTag(WJ300A.PARAM.TLVL.tag,'Уровень триггера:');
     if IsSameText(arg,'TRDL')     then StartEditTag(WJ300A.PARAM.TRDL.tag,'Задержка триггера:');
     if IsSameText(arg,'TRMD')     then StartEditTag(WJ300A.PARAM.TRMD.tag,'Режим триггера:');
     if IsSameText(arg,'TSLP')     then StartEditTag(WJ300A.PARAM.TSLP.tag,'Фронт триггера:');
     if IsSameText(arg,'TSRC')     then StartEditTag(WJ300A.PARAM.TSRC.tag,'Источник триггера:');
     if IsSameText(arg,'TTYP')     then StartEditTag(WJ300A.PARAM.TTYP.tag,'Тип триггера:');
     if IsSameText(arg,'VDIV_C1')  then StartEditTag(WJ300A.C1.SCAL.tag,'Вертикальная чувствительность:');
     if IsSameText(arg,'VDIV_C2')  then StartEditTag(WJ300A.C2.SCAL.tag,'Вертикальная чувствительность:');
     if IsSameText(arg,'VDIV_C3')  then StartEditTag(WJ300A.C3.SCAL.tag,'Вертикальная чувствительность:');
     if IsSameText(arg,'VDIV_C4')  then StartEditTag(WJ300A.C4.SCAL.tag,'Вертикальная чувствительность:');
     if IsSameText(arg,'VDIV_M1')  then StartEditTag(WJ300A.M1.SCAL.tag,'Вертикальная чувствительность:');
     if IsSameText(arg,'WAVESRC')  then StartEditTag(WJ300A.PARAM.WAVESRC.tag,'Источник осциллограммы:');
     if IsSameText(arg,'DIRBASE')  then StartEditTag(WJ300A.DIRBASE.tag,'DIRBASE:');
     if IsSameText(arg,'DIREXP')   then StartEditTag(WJ300A.DIREXP.tag,'DIREXP:');
    end;
    Data:='';
   end else
   {
   @PrintCmdTable
   }
   if (cmdid=cmd_PrintCmdTable) then begin
    PrintCmdTable;
    Data:='';
   end else
   {
   @DimTagUpdate tag
   }
   if (cmdid=cmd_DimTagUpdate) then begin
    WJ300A_OnDimUpdateTag(arg);
    Data:='';
   end else
   {
   @AssignTag
   }
   if (cmdid=cmd_AssignTag) then begin
    WJ300A_OnAssignTag(arg);
    Data:='';
   end else
   {
   @GetWaveForm
   }
   if (cmdid=cmd_GetWaveForm) then begin
    if ((not IsEmptyStr(arg)) and (Val(arg)=0))
    then WJ300A.GetWaveFormStatus:=stt_wfDone
    else begin
     WJ300A.IsWaveFormDataRequest:=true;
     WJ300A.GetWaveFormStatus:=stt_wfGetSource;
     WJ300A.PollPriority:=PrHigh;
     GetWaveFormData;
    end;
    Data:='';
   end else
   {
   @Autoset
   }
   if (cmdid=cmd_Autoset) then begin
    DevSendCmdLocal(DimRemote+'@Setting TRMD AUTO');
    Data:='';
   end else
   {
   @TrigRun
   }
   if (cmdid=cmd_TrigRun) then begin
    DevSendCmdLocal(DimRemote+'@Setting TRMD AUTO');
    Data:='';
   end else
   {
   @TrigStop
   }
   if (cmdid=cmd_TrigStop) then begin
    DevSendCmdLocal(DimRemote+'@Setting TRMD STOP');
    Data:='';
   end else
   {
   @LoadIni
   }
   if (cmdid=cmd_LoadIni) then begin
    if not WJ300A.Simulator and not DIM_IsClientMode then begin
     iNul(CustomIniRW('R',arg,2*Ord(not IsEmptyStr(arg))));
     UpdateSettingsAfterLoadIni;
    end;
    Data:='';
   end else
   {
   @SaveIni
   }
   if (cmdid=cmd_SaveIni) then begin
    if not WJ300A.Simulator and not DIM_IsClientMode then begin
     iNul(CustomIniRW('W',arg,2*Ord(not IsEmptyStr(arg))));
    end;
    Data:='';
   end else
   {
   @DimCmdMy
   }
   if (cmdid=cmd_DimCmdMy) then begin
     if LooksLikeCommand(trim(base64_decode(arg))) then begin
     DevSendCmdLocal(trim(base64_decode(arg)));
     Success('@DIMCMDMY='+cmd+' arg<'+trim(base64_decode(arg))+'>');
    end;
    Data:='';
   end else
   {
   @SingleStart
   }
   if (cmdid=cmd_SingleStart) then begin
    WJ300A.GetTrigStatus:=stt_trigSetSingle;
    Data:='';
   end else
   {
   @SingleStop
   }
   if (cmdid=cmd_SingleStop) then begin
    DevSendCmdLocal(DimRemote+'@Setting TRMD STOP');
    Data:='';
   end else
   {
   @SetDirBase
   }
   if (cmdid=cmd_SetDirBase) then begin
    SetDirBase(arg);
    Data:='';
   end else
   {
   @SetDirExp
   }
   if (cmdid=cmd_SetDirExp) then begin
    SetDirExp(arg);
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  Cleanup;
 end;
{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{<== Please never change this code ***}
{***************************************************}
