////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq data analysis plugin.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241030 - Sample created by A.K.                                          //
// 20241104 - Translated from DPR source by A.K.                              //
////////////////////////////////////////////////////////////////////////////////

{
[Manual.Rus]
Эта утилита служит для расчета статистических характеристик "области интересов" ROI.
"Область интересов" должна быть задана в окне - источнике перед вызовом утилиты при помощи двух маркеров ROI.
Утилита расчитывает среднее значение точек, попавших в прямоугольник ROI, а также стандартное отклонение от среднего для кривых в окне.
Результат отображается в Главной Консоли.
[]
[Manual.Eng]
Use this plugin to calculate statistics of "Region Of Interest" ROI.
Set "Region Of Interest" before call plugin. Uses ROI markers in source data window.
Utility calculate average value of points located inside ROI rectangle and standard deviation.
Result will be shown in Main Console.
[]
[Arguments.Rus]
Имя окна = Результат:"STAT ROI"
Заголовок = ^CЗаголовок^N^L  Y
Легенда = ^RX  ^N^CЛегенда
[]
[Arguments.Eng]
Caption = Result:"STAT ROI"
Title = ^CTitle^N^L  Y
Legend = ^RX  ^N^CLegend
[]
}

library _curve_stat_roi;

{$I _crw_sysdef}

{$IFDEF FPC}{$mode Delphi}{$ENDIF}{$H+}

{$R *.res}

uses
 _crw_sharemem, // NB: THIS UNIT MUST BE FIRST !!!
 {$IFDEF UNIX} cthreads, dl, {$ENDIF}
 {$IFDEF WINDOWS} windows, {$ENDIF}
 sysutils, classes, math, graphics,
 _crw_crwapi;

//////////////////////////////////////////////////
{$I _crw_plugin_declare} // Declare CRWDAQ_PLUGIN.
//////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:TCrwApi):Integer;
//////////////////////////////////////////////////
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
var
 Roi : TRect2D;
 i, c, cFirst, cLast : Integer;
 norm, averx, avery, sigmx, sigmy : Double;
 procedure Refreshment(Delta:Integer);
 const LastTicks : Cardinal = 0;
 begin
  if LastTicks=0 then LastTicks:=GetTickCount;
  if abs(GetTickCount-LastTicks) > Delta then begin
   with CrwApi,GuiApi do begin ApplicationProcessMessages; UpdateSystemConsole; end;
   LastTicks:=GetTickCount;
  end;
 end;
begin
 Result:=0;					
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);				
  RedirectStdOut(Output);
  if Target <> ForDataAnalysis 
  then Raise EDanApi.Create(RusEng('Неверное значение Target!',
                                   'Invalid Target!'));
  if not WindowExists(swin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - источник!',
                                   'Source window not found!'));
  if not WindowExists(twin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - приемник!',
                                   'Target window not found!'));
  if CurvesCount[swin]=0 
  then Raise EDanApi.Create(RusEng('Нет данных для обработки!',
                                   'No input data curves found!'));
  Roi:=WindowRoi[swin];
  if isNAN(Roi.A.X+Roi.A.Y)
  then Raise EDanApi.Create(RusEng('Левый маркер РОИ не определен!',
                                   'Left ROI marker is not defined!'));
  if isNAN(Roi.B.X+Roi.B.Y)
  then Raise EDanApi.Create(RusEng('Правый маркер РОИ не определен!',
                                   'Right ROI marker is not defined!'));
  Roi:=Rect2DValidate(Roi);
  WindowRoi[twin]:=WindowRoi[swin];
  WindowCaption[twin]:=GetArgumentAsString(RusEng('Имя окна','Caption'));
  WindowTitle[twin]:=GetArgumentAsString(RusEng('Заголовок','Title'));
  WindowLegend[twin]:=GetArgumentAsString(RusEng('Легенда','Legend'));
  if SelectedCurve[swin]>0 then begin
   cFirst:=SelectedCurve[swin];
   cLast:=SelectedCurve[swin];
  end else begin
   cFirst:=1;
   cLast:=CurvesCount[swin];
  end;
  Echo('');
  Echo(GetDateStr(mSecNow,'.',true)+'-'+GetTimeStr(mSecNow)+' -> _CURVE_STAT_ROI result:');
  Echo(Format('ROI.X is: %g <= X <= %g',[ROI.A.X,ROI.B.X]));
  Echo(Format('ROI.Y is: %g <= Y <= %g',[ROI.A.Y,ROI.B.Y]));
  for c:=cFirst to cLast do begin
   norm:=0; averx:=0; avery:=0;
   for i:=0 to CurveLength[c]-1 do
   if Rect2DContainsPoint(Roi,CurvePoint[c,i]) then begin
    averx:=averx+CurvePoint[c,i].x;
    avery:=avery+CurvePoint[c,i].y;
    Refreshment(100);
    norm:=norm+1;
   end;
   averx:=averx/norm;
   avery:=avery/norm;
   norm:=0; sigmx:=0; sigmy:=0;
   for i:=0 to CurveLength[c]-1 do
   if Rect2DContainsPoint(Roi,CurvePoint[c,i]) then begin
    sigmx:=sigmx+sqr(averx-CurvePoint[c,i].x);
    sigmy:=sigmy+sqr(avery-CurvePoint[c,i].y);
    Refreshment(100);
    norm:=norm+1;
   end;
   sigmx:=sqrt(sigmx/(norm-1));
   sigmy:=sqrt(sigmy/(norm-1));
   Echo(Format('Curve[%d] = %s:',[c,CurveName[c]]));
   Echo(Format('x: %g ± %g',[averx,sigmx]));
   Echo(Format('y: %g ± %g',[avery,sigmy]));
  end;
  if SelectedCurve[swin]>0 then SelectedCurve[twin]:=1 else SelectedCurve[twin]:=0;
 except
  on E:Exception do begin
   Result:=-1;
   if WindowExists(twin) then CurvesCount[twin]:=0; 
   Echo(E.Message);
   if UsesBlaster then Voice('EXCEPTION');
   Error(E.Message);
  end;
 end;
end;

//////////////////////////////////////////////////
{$I _crw_plugin_exports} // Exports CRWDAQ_PLUGIN.
//////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID; //
//////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////
 
 
