////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq data analysis plugin.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241030 - Sample created by A.K.                                          //
// 20241104 - Translated from DPR source by A.K.                              //
////////////////////////////////////////////////////////////////////////////////

{
[Manual.Rus]
Утилита применяет к кривой пороговый медианный фильтр.
Медианный фильтр применяют для устранения "выбросов" или импульсных помех.
Надо задать полуширину, а также абсолютный и относительный триггер фильтра.
Фильтр вычисляет медиану на ширине окна сглаживания,
но применяет ее только тогда, когда она отличается от текущего значения
на величину, превышающую триггер. Относительный триггер означает, что
триггер вычисляется как заданная доля от диапазона изменения сигнала.
[]
[Manual.Eng]
This utility apply median filter with trigger for given curve.
It uses to erase impulse noise.
User should specify half width, absolute and relative trigger values.
[]
[Arguments.Rus]
Имя окна = Результат:"Пороговый медианный фильтр"
Заголовок = ^CЗаголовок^N^L  Y
Легенда = ^RX  ^N^CЛегенда
Полуширина = 1
Абс.Триггер = 0
Отн.Триггер = 0.1
[]
[Arguments.Eng]
Caption = Result:"Median filter with trigger"
Title = ^CTitle^N^L  Y
Legend = ^RX  ^N^CLegend
HalfWidth = 1
Abs.Trigger = 0
Rel.Trigger = 0.1
[]
}

library _curve_median_filter_wt;

{$I _crw_sysdef}

{$IFDEF FPC}{$mode Delphi}{$ENDIF}{$H+}

{$R *.res}

uses
 _crw_sharemem, // NB: THIS UNIT MUST BE FIRST !!!
 {$IFDEF UNIX} cthreads, dl, {$ENDIF}
 {$IFDEF WINDOWS} windows, {$ENDIF}
 sysutils, classes, math, graphics,
 _crw_crwapi;

//////////////////////////////////////////////////
{$I _crw_plugin_declare} // Declare CRWDAQ_PLUGIN.
//////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:TCrwApi):Integer;
//////////////////////////////////////////////////
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
 CheckFlags = cfInvalid + cfNoData + cfTooSmall + cfNotSortedX + cfDuplicatesX + cfNanInf;
var
 HalfWidth, c1, c2, cFirst, cLast : Integer;
 AbsEps, RelEps : Double;
 procedure Refreshment(Delta:Integer);
 const LastTicks : Cardinal = 0;
 begin
  if LastTicks=0 then LastTicks:=GetTickCount;
  if abs(GetTickCount-LastTicks) > Delta then begin
   with CrwApi,GuiApi do begin ApplicationProcessMessages; UpdateSystemConsole; end;
   LastTicks:=GetTickCount;
  end;
 end;
begin
 Result:=0;					
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);				
  RedirectStdOut(Output);
  if Target <> ForDataAnalysis 
  then Raise EDanApi.Create(RusEng('Неверное значение Target!',
                                   'Invalid Target!'));
  if not WindowExists(swin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - источник!',
                                   'Source window not found!'));
  if not WindowExists(twin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - приемник!',
                                   'Target window not found!'));
  if CurvesCount[swin]=0 
  then Raise EDanApi.Create(RusEng('Нет данных для обработки!',
                                   'No input data curves found!'));
  WindowRoi[twin]:=WindowRoi[swin];
  WindowCaption[twin]:=GetArgumentAsString(RusEng('Имя окна','Caption'));
  WindowTitle[twin]:=GetArgumentAsString(RusEng('Заголовок','Title'));
  WindowLegend[twin]:=GetArgumentAsString(RusEng('Легенда','Legend'));
  if not GetArgumentAsInteger(RusEng('Полуширина','HalfWidth'),HalfWidth) or
     not GetArgumentAsDouble(RusEng('Абс.Триггер','Abs.Trigger'),AbsEps) or
     not GetArgumentAsDouble(RusEng('Отн.Триггер','Rel.Trigger'),RelEps)
  then Raise EDanApi.Create(RusEng('Ошибка задания аргументов!',
                                   'Invalid input arguments!'));
  if SelectedCurve[swin]>0 then begin
   cFirst:=SelectedCurve[swin];
   cLast:=SelectedCurve[swin];
  end else begin
   if YesNo(RusEng('Выполнить для всех кривых в окне?',
                   'Execute for all curves in window?'))<>mrYes
   then Raise EDanApi.Create(RusEng('Пользователь прервал!','User break!'));
   cFirst:=1;
   cLast:=CurvesCount[swin];
  end;
  for c1:=cFirst to cLast do begin
   if CurveFlags(c1,0,1E-10,3) and CheckFlags <> 0
   then Raise EDanApi.Create(Format(RusEng('Кривая %d непригодна для фильтрации!',
                                           'Curve %d is not convenient for filter!'),[c1]));
   c2:=CreateCurve(twin);
   CurveAssign(c2,c1);
   CurveMedianFiltr(c2,HalfWidth,AbsEps,RelEps);
   if CurveFlags(c2,0,1E-10,3) and CheckFlags <> 0
   then Raise EDanApi.Create(Format(RusEng('Ошибка фильтрации кривой %d !',
                                           'Error on curve %d filter !'),[c1]));
   Refreshment(100);
  end;
  if SelectedCurve[swin]>0 then SelectedCurve[twin]:=1 else SelectedCurve[twin]:=0;
 except
  on E:Exception do begin
   Result:=-1;
   if WindowExists(twin) then CurvesCount[twin]:=0; 
   Echo(E.Message);
   if UsesBlaster then Voice('EXCEPTION');
   Error(E.Message);
  end;
 end;			
end;

//////////////////////////////////////////////////
{$I _crw_plugin_exports} // Exports CRWDAQ_PLUGIN.
//////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID; //
//////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////
 
 
