////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq data analysis plugin.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241030 - Sample created by A.K.                                          //
// 20241104 - Translated from DPR source by A.K.                              //
////////////////////////////////////////////////////////////////////////////////

{
[Manual.Rus]
Эта утилита служит для интерполяции кривой на новую сетку X.
Для этого надо задать в полях ввода Script#1..Script#9 начало
и конец интервала (a,b), число точек n.
Кривая будет переинтерполирована на указанную сетку.
[]
[Manual.Eng]
This plugin uses to interpolate curve to new grid along X.
You have to specify in Script#1..Script#9 lines left and right
interval bounds (a,b) and number of points n.
Curve will be interpolated to this new grid.
[]
[Arguments.Rus]
Имя окна = Результат:"Интерполяция кривой"
Заголовок = ^CЗаголовок^N^L  Y
Легенда = ^RX  ^N^CЛегенда
Script#1 = a = a
Script#2 = b = b
Script#3 = n = n
Script#4 =
Script#5 =
Script#6 =
Script#7 =
Script#8 =
Script#9 =
[]
[Arguments.Eng]
Caption = Result:"Interpolate curve"
Title = ^CTitle^N^L  Y
Legend = ^RX  ^N^CLegend
Script#1 = a = a
Script#2 = b = b
Script#3 = n = n
Script#4 =
Script#5 =
Script#6 =
Script#7 =
Script#8 =
Script#9 =
[]
}

library _curve_fit_to_new_grid;

{$I _crw_sysdef}

{$IFDEF FPC}{$mode Delphi}{$ENDIF}{$H+}

{$R *.res}

uses
 _crw_sharemem, // NB: THIS UNIT MUST BE FIRST !!!
 {$IFDEF UNIX} cthreads, dl, {$ENDIF}
 {$IFDEF WINDOWS} windows, {$ENDIF}
 sysutils, classes, math, graphics,
 _crw_crwapi;

//////////////////////////////////////////////////
{$I _crw_plugin_declare} // Declare CRWDAQ_PLUGIN.
//////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:TCrwApi):Integer;
//////////////////////////////////////////////////
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
 CheckFlags = cfInvalid + cfNoData + cfTooSmall + cfNotSortedX + cfDuplicatesX + cfNanInf;
var
 n, a, b : Double;
 p : TPoint2D;
 i, c1, c2, cFirst, cLast : Integer;
 Interpreter : TScriptInterpreter;
 s : ShortString;
 procedure Refreshment(Delta:Integer);
 const LastTicks : Cardinal = 0;
 begin
  if LastTicks=0 then LastTicks:=GetTickCount;
  if abs(GetTickCount-LastTicks) > Delta then begin
   with CrwApi,GuiApi do begin ApplicationProcessMessages; UpdateSystemConsole; end;
   LastTicks:=GetTickCount;
  end;
 end;
begin
 Result:=0;					
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);				
  RedirectStdOut(Output);
  if Target <> ForDataAnalysis 
  then Raise EDanApi.Create(RusEng('Неверное значение Target!',
                                   'Invalid Target!'));
  if not WindowExists(swin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - источник!',
                                   'Source window not found!'));
  if not WindowExists(twin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - приемник!',
                                   'Target window not found!'));
  if CurvesCount[swin]=0 
  then Raise EDanApi.Create(RusEng('Нет данных для обработки!',
                                   'No input data curves found!'));
  WindowRoi[twin]:=WindowRoi[swin];
  WindowCaption[twin]:=GetArgumentAsString(RusEng('Имя окна','Caption'));
  WindowTitle[twin]:=GetArgumentAsString(RusEng('Заголовок','Title'));
  WindowLegend[twin]:=GetArgumentAsString(RusEng('Легенда','Legend'));
  if SelectedCurve[swin]>0 then begin
   cFirst:=SelectedCurve[swin];
   cLast:=SelectedCurve[swin];
  end else begin
   if YesNo(RusEng('Выполнить для всех кривых в окне?',
                   'Execute for all curves in window?'))<>mrYes
   then Raise EDanApi.Create(RusEng('Пользователь прервал!','User break!'));
   cFirst:=1;
   cLast:=CurvesCount[swin];
  end;
  Interpreter:=CreateScriptInterpreter;
  try
   for i:=1 to 9 do begin
    s:=GetArgumentAsString('Script#'+IntToStr(i));
    if s<>'' then Interpreter.Script:=Interpreter.Script+s+CRLF;
   end;
   for c1:=cFirst to cLast do begin
    if CurveFlags(c1,0,1E-10,3) and CheckFlags <> 0
    then Raise EDanApi.Create(Format(RusEng('Кривая %d непригодна для интерполяции!',
                                            'Curve %d is not convenient for interpolation!'),[c1]));
    a:=CurveLimits[c1].A.X;
    b:=CurveLimits[c1].B.X;
    n:=CurveLength[c1];
    if Interpreter.SetValue('a',a) and
       Interpreter.SetValue('b',b) and
       Interpreter.SetValue('n',n) and
       (Interpreter.RunScript=0)   and
       Interpreter.GetValue('a',a) and
       Interpreter.GetValue('b',b) and
       Interpreter.GetValue('n',n) and
       not isNAN(a)                and
       not isNAN(b)                and
       not isNAN(n)                and
       (n>1)                       and
       (n<1E6)
    then begin
     c2:=CreateCurve(twin,CurveName[c1],CurveColor[c1],CurveStyle[c1],CurveComment[c1]);
     CurveLength[c2]:=Round(n);
     for i:=0 to CurveLength[c2]-1 do begin
      p.x:=a+(b-a)*i/(CurveLength[c2]-1);
      p.y:=CurveInterpolateAt(c1,p.x);
      CurvePoint[c2,i]:=p;
      Refreshment(100);
     end;
    end else Raise EDanApi.Create(RusEng('Ошибка интерпретации!','Interpreter error!'));
   end;
   if SelectedCurve[swin]>0 then SelectedCurve[twin]:=1 else SelectedCurve[twin]:=0;
  finally
   Interpreter.Free;
  end;
 except
  on E:Exception do begin
   Result:=-1;
   if WindowExists(twin) then CurvesCount[twin]:=0; 
   Echo(E.Message);
   if UsesBlaster then Voice('EXCEPTION');
   Error(E.Message);
  end;
 end;
end;

//////////////////////////////////////////////////
{$I _crw_plugin_exports} // Exports CRWDAQ_PLUGIN.
//////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID; //
//////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////
 
