unit uhistory;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, StdCtrls, lazUTF8;

type

  PStep = ^TStep;

  TStep = record           
    BytePos : SizeInt;
    InsStr  : string;
    DelStr  : string;
  end;

  { THistory }

  THistory = class
  private
    FMemo        : TMemo;
    FSteps       : TList;         // History Records
    FIndex       : Integer;       // Based 0
    FOldOnChange : TNotifyEvent;
    FInEdit      : Boolean;
    FPrevContent : string;

    function    GetStep(AIndex: Integer): PStep;  
    function    CurStep: PStep;

    procedure   AddStep(ABytePos: SizeInt; AInsStr, ADelStr: string);
    procedure   DelStep(Index: Integer);

    procedure   MemoOnChange(Sender: TObject);

    function    StrDiff(out BytePos: SizeInt; out InsStr, DelStr: string): Boolean;
  public
    constructor Create(AMemo: TMemo);
    destructor  Destroy; override;

    function    CanUndo: Boolean;
    function    CanRedo: Boolean;
    procedure   Undo;
    procedure   Redo;
  end;

implementation

{ THistory }

function THistory.GetStep(AIndex: Integer): PStep;
begin
  Result := PStep(FSteps[AIndex]);
end;

function THistory.CurStep: PStep;
begin
  Result := GetStep(FIndex);
end;
      
procedure THistory.AddStep(ABytePos: SizeInt; AInsStr, ADelStr: string);
begin
  DelStep(FIndex + 1);

  FSteps.Add(new(PStep));
  Inc(FIndex);
  with CurStep^ do begin
    BytePos  := ABytePos;
    InsStr    := AInsStr;
    DelStr    := ADelStr;
  end;
end;

procedure THistory.DelStep(Index: Integer);
var
  i: Integer;
begin
  for i := FSteps.Count - 1 downto Index do begin
    GetStep(i)^.InsStr := '';
    GetStep(i)^.DelStr := '';
    dispose(GetStep(i));
    FSteps.Delete(i);
  end;
  FIndex := Index - 1;
end;

procedure THistory.MemoOnChange(Sender: TObject);
var
  InsStr, DelStr: string;
  BytePos: SizeInt;
begin
  if FInEdit and StrDiff(BytePos, InsStr, DelStr) then begin
    AddStep(BytePos, InsStr, DelStr);
    FPrevContent := FMemo.Text;
  end;

  if Assigned(FOldOnChange) then
    FOldOnChange(Sender);
end;

function THistory.StrDiff(out BytePos: SizeInt; out InsStr, DelStr: string): Boolean;
var
  DiffLen: SizeInt;
begin
  DiffLen := Length(FMemo.Text) - Length(FPrevContent);
  // UTF8CharToByteIndex based 0
  BytePos := UTF8CharToByteIndex(PChar(FMemo.Text), Length(FMemo.Text), FMemo.SelStart) + 1;

  if DiffLen > 0 then begin
    BytePos := BytePos - DiffLen;
    InsStr := Copy(FMemo.Text, BytePos, DiffLen); // Copy based 1
    DelStr := '';
  end else if DiffLen < 0 then begin
    InsStr := '';
    DelStr := Copy(FPrevContent, BytePos, -DiffLen); // Copy based 1
  end else begin
    Result := False;
    Exit;
  end;
  Result := True;
end;

constructor THistory.Create(AMemo: TMemo);
begin
  inherited Create;
  FSteps := TList.Create;
  FMemo := AMemo;
  FOldOnChange := FMemo.OnChange;
  FMemo.OnChange := @MemoOnChange; 
  FPrevContent := FMemo.Text;
  FIndex := -1;
  FInEdit := True;
end;

destructor THistory.Destroy;
begin
  FInEdit := False;
  FMemo.OnChange := FOldOnChange;
  FMemo := nil;
  FPrevContent := '';
  DelStep(0);
  FSteps.Free;
  inherited Destroy;
end;

function THistory.CanUndo: Boolean;
begin
  Result := FIndex >= 0;
end;

function THistory.CanRedo: Boolean;
begin
  Result := FIndex < FSteps.Count - 1;
end;

procedure THistory.Undo;
var
  NewSelStart: SizeInt;
begin
  if FIndex < 0 then Exit;
  FInEdit := False;

  // FPrevContent == FMemo.Text
  with CurStep^ do begin
    if InsStr <> '' then begin
      Delete(FPrevContent, BytePos, Length(InsStr));
      NewSelStart := FMemo.SelStart - UTF8LengthFast(InsStr);
    end else if DelStr <> '' then begin
      Insert(DelStr, FPrevContent, BytePos);
      NewSelStart := FMemo.SelStart + UTF8LengthFast(DelStr);
    end else begin
      writeln('Invalid history data!');
    end;
  end;

  FMemo.Text := FPrevContent; 
  FMemo.SelStart := NewSelStart;
  FInEdit := True;

  Dec(FIndex);
end;

procedure THistory.Redo;
var
  NewSelStart: SizeInt;
begin            
  if FIndex >= FSteps.Count - 1 then Exit;
  FInEdit := False;
                  
  Inc(FIndex);
  // FPrevContent == FMemo.Text
  with CurStep^ do begin
    if InsStr <> '' then begin
      Insert(InsStr, FPrevContent, BytePos);
      NewSelStart := FMemo.SelStart + UTF8LengthFast(InsStr);
    end else if DelStr <> '' then begin
      Delete(FPrevContent, BytePos, Length(DelStr));
      NewSelStart := FMemo.SelStart - UTF8LengthFast(DelStr);
    end else begin
      writeln('Invalid history data!');
    end;
  end;

  FMemo.Text := FPrevContent;
  FMemo.SelStart := NewSelStart;
  FInEdit := True;
end;

end.

