package flashx.textLayout.conversion
{
   import flashx.textLayout.elements.TextFlow;
   import flashx.textLayout.elements.FlowGroupElement;
   import flashx.textLayout.elements.ParagraphElement;
   import flashx.textLayout.tlf_internal;
   import flashx.textLayout.elements.SpanElement;
   import flashx.textLayout.elements.FlowLeafElement;
   import flashx.textLayout.elements.BreakElement;
   import flashx.textLayout.elements.TabElement;
   import flashx.textLayout.elements.GlobalSettings;
   import flashx.textLayout.elements.IConfiguration;
   import flashx.textLayout.property.Property;
   import flashx.textLayout.elements.ContainerFormattedElement;
   import flashx.textLayout.elements.FlowElement;
   import flashx.textLayout.elements.ParagraphFormattedElement;
   
   use namespace tlf_internal;
   
   [ExcludeClass]
   class BaseTextLayoutImporter implements ITextImporter
   {
      
      private static const anyPrintChar:RegExp = /[^\t\n\r ]/g;
      
      private static const dblSpacePattern:RegExp = /[ ]{2,}/g;
      
      private static const tabNewLinePattern:RegExp = /[\t\n\r]/g;
       
      private var _ns:Namespace;
      
      private var _errors:Vector.<String>;
      
      private var _textFlowNamespace:Namespace;
      
      private var _throwOnError:Boolean;
      
      protected var _config:flashx.textLayout.conversion.ImportExportConfiguration;
      
      protected var _textFlowConfiguration:IConfiguration;
      
      private var _impliedPara:ParagraphElement = null;
      
      function BaseTextLayoutImporter(textFlowConfiguration:IConfiguration, nsValue:Namespace, config:flashx.textLayout.conversion.ImportExportConfiguration)
      {
         super();
         this._textFlowConfiguration = textFlowConfiguration;
         this._ns = nsValue;
         this._config = config;
      }
      
      protected static function stripWhitespace(insertString:String) : String
      {
         return insertString.replace(tabNewLinePattern," ");
      }
      
      public static function parseTextFlow(importFilter:BaseTextLayoutImporter, xmlToParse:XML, parent:Object = null) : TextFlow
      {
         return importFilter.createTextFlowFromXML(xmlToParse,null);
      }
      
      public static function parsePara(importFilter:BaseTextLayoutImporter, xmlToParse:XML, parent:FlowGroupElement) : void
      {
         var paraElem:ParagraphElement = importFilter.createParagraphFromXML(xmlToParse);
         if(importFilter.addChild(parent,paraElem))
         {
            importFilter.parseFlowGroupElementChildren(xmlToParse,paraElem);
            if(paraElem.numChildren == 0)
            {
               paraElem.addChild(new SpanElement());
            }
         }
      }
      
      protected static function copyAllStyleProps(dst:FlowLeafElement, src:FlowLeafElement) : void
      {
         dst.format = src.format;
         dst.styleName = src.styleName;
         dst.userStyles = src.userStyles;
         dst.id = src.id;
      }
      
      public static function parseSpan(importFilter:BaseTextLayoutImporter, xmlToParse:XML, parent:FlowGroupElement) : void
      {
         var child:XML = null;
         var elemName:String = null;
         var s:SpanElement = null;
         var brElem:BreakElement = null;
         var tabElem:TabElement = null;
         var firstSpan:SpanElement = importFilter.createSpanFromXML(xmlToParse);
         var elemList:XMLList = xmlToParse[0].children();
         if(elemList.length() == 0)
         {
            importFilter.addChild(parent,firstSpan);
            return;
         }
         for each(child in elemList)
         {
            elemName = Boolean(child.name())?child.name().localName:null;
            if(elemName == null)
            {
               if(firstSpan.parent == null)
               {
                  firstSpan.text = child.toString();
                  importFilter.addChild(parent,firstSpan);
               }
               else
               {
                  s = new SpanElement();
                  copyAllStyleProps(s,firstSpan);
                  s.text = child.toString();
                  importFilter.addChild(parent,s);
               }
            }
            else if(elemName == "br")
            {
               brElem = importFilter.createBreakFromXML(child);
               if(brElem)
               {
                  copyAllStyleProps(brElem,firstSpan);
                  importFilter.addChild(parent,brElem);
               }
               else
               {
                  importFilter.reportError(GlobalSettings.resourceStringFunction("unexpectedXMLElementInSpan",[elemName]));
               }
            }
            else if(elemName == "tab")
            {
               tabElem = importFilter.createTabFromXML(child);
               if(tabElem)
               {
                  copyAllStyleProps(tabElem,firstSpan);
                  importFilter.addChild(parent,tabElem);
               }
               else
               {
                  importFilter.reportError(GlobalSettings.resourceStringFunction("unexpectedXMLElementInSpan",[elemName]));
               }
            }
            else
            {
               importFilter.reportError(GlobalSettings.resourceStringFunction("unexpectedXMLElementInSpan",[elemName]));
            }
         }
      }
      
      public static function parseBreak(importFilter:BaseTextLayoutImporter, xmlToParse:XML, parent:FlowGroupElement) : void
      {
         var breakElem:BreakElement = importFilter.createBreakFromXML(xmlToParse);
         importFilter.addChild(parent,breakElem);
      }
      
      public static function parseTab(importFilter:BaseTextLayoutImporter, xmlToParse:XML, parent:FlowGroupElement) : void
      {
         var tabElem:TabElement = importFilter.createTabFromXML(xmlToParse);
         if(tabElem)
         {
            importFilter.addChild(parent,tabElem);
         }
      }
      
      protected static function extractAttributesHelper(curAttrs:Object, importer:TLFormatImporter) : Object
      {
         if(curAttrs == null)
         {
            return importer.result;
         }
         if(importer.result == null)
         {
            return curAttrs;
         }
         var workAttrs:Object = new importer.classType(curAttrs);
         workAttrs.apply(importer.result);
         return workAttrs;
      }
      
      protected function clear() : void
      {
         if(this.errors)
         {
            this.errors.splice(0,this.errors.length);
         }
         this._textFlowNamespace = null;
         this._impliedPara = null;
      }
      
      public function importToFlow(source:Object) : TextFlow
      {
         this.clear();
         if(this._throwOnError)
         {
            return this.importToFlowCanThrow(source);
         }
         var rslt:TextFlow = null;
         var savedErrorHandler:Function = Property.errorHandler;
         try
         {
            Property.errorHandler = this.importPropertyErrorHandler;
            rslt = this.importToFlowCanThrow(source);
         }
         catch(e:Error)
         {
            reportError(e.toString());
         }
         Property.errorHandler = savedErrorHandler;
         return rslt;
      }
      
      protected function importPropertyErrorHandler(p:Property, value:Object) : void
      {
         this.reportError(Property.createErrorString(p,value));
      }
      
      private function importToFlowCanThrow(source:Object) : TextFlow
      {
         if(source is String)
         {
            return this.importFromString(String(source));
         }
         if(source is XML)
         {
            return this.importFromXML(XML(source));
         }
         return null;
      }
      
      protected function importFromString(source:String) : TextFlow
      {
         var xmlTree:XML = null;
         var originalSettings:Object = XML.settings();
         try
         {
            XML.ignoreProcessingInstructions = false;
            XML.ignoreWhitespace = false;
            xmlTree = new XML(source);
         }
         finally
         {
            XML.setSettings(originalSettings);
         }
         return this.importFromXML(xmlTree);
      }
      
      protected function importFromXML(xmlSource:XML) : TextFlow
      {
         return this.parseContent(xmlSource[0]);
      }
      
      protected function parseContent(rootStory:XML) : TextFlow
      {
         var child:XML = rootStory..TextFlow[0];
         if(child)
         {
            return parseTextFlow(this,rootStory);
         }
         return null;
      }
      
      public function get ns() : Namespace
      {
         return this._ns;
      }
      
      protected function checkNamespace(xmlToParse:XML) : Boolean
      {
         var elementNS:Namespace = xmlToParse.namespace();
         if(!this._textFlowNamespace)
         {
            if(elementNS != this.ns)
            {
               this.reportError(GlobalSettings.resourceStringFunction("unexpectedNamespace",[elementNS.toString()]));
               return false;
            }
            this._textFlowNamespace = elementNS;
         }
         else if(elementNS != this._textFlowNamespace)
         {
            this.reportError(GlobalSettings.resourceStringFunction("unexpectedNamespace",[elementNS.toString()]));
            return false;
         }
         return true;
      }
      
      public function parseAttributes(xmlToParse:XML, formatImporters:Array) : void
      {
         var importer:IFormatImporter = null;
         var item:XML = null;
         var propertyName:String = null;
         var propertyValue:String = null;
         var imported:Boolean = false;
         for each(importer in formatImporters)
         {
            importer.reset();
         }
         for each(item in xmlToParse.attributes())
         {
            propertyName = item.name().localName;
            propertyValue = item.toString();
            imported = false;
            for each(importer in formatImporters)
            {
               if(importer.importOneFormat(propertyName,propertyValue))
               {
                  imported = true;
                  break;
               }
            }
            if(!imported)
            {
               this.handleUnknownAttribute(xmlToParse.name().localName,propertyName);
            }
         }
      }
      
      public function createTextFlowFromXML(xmlToParse:XML, newFlow:TextFlow = null) : TextFlow
      {
         return null;
      }
      
      public function createParagraphFromXML(xmlToParse:XML) : ParagraphElement
      {
         return null;
      }
      
      public function createSpanFromXML(xmlToParse:XML) : SpanElement
      {
         return null;
      }
      
      public function createBreakFromXML(xmlToParse:XML) : BreakElement
      {
         this.parseAttributes(xmlToParse,null);
         return new BreakElement();
      }
      
      public function createTabFromXML(xmlToParse:XML) : TabElement
      {
         this.parseAttributes(xmlToParse,null);
         return new TabElement();
      }
      
      public function parseFlowChildren(xmlToParse:XML, parent:FlowGroupElement) : void
      {
         this.parseFlowGroupElementChildren(xmlToParse,parent);
      }
      
      public function parseFlowGroupElementChildren(xmlToParse:XML, parent:FlowGroupElement, exceptionElements:Object = null, chainedParent:Boolean = false) : void
      {
         var child:XML = null;
         var txt:String = null;
         var strip:Boolean = false;
         var span:SpanElement = null;
         for each(child in xmlToParse.children())
         {
            if(child.nodeKind() == "element")
            {
               this.parseObject(child.name().localName,child,parent,exceptionElements);
            }
            else if(child.nodeKind() == "text")
            {
               txt = child.toString();
               strip = false;
               if(parent is ContainerFormattedElement)
               {
                  strip = txt.search(anyPrintChar) == -1;
               }
               if(!strip)
               {
                  span = new SpanElement();
                  span.text = txt;
                  this.addChild(parent,span);
               }
            }
         }
         if(Boolean(!chainedParent) && Boolean(parent is ContainerFormattedElement))
         {
            this.resetImpliedPara();
         }
      }
      
      public function createParagraphFlowFromXML(xmlToParse:XML, newFlow:TextFlow = null) : TextFlow
      {
         return null;
      }
      
      tlf_internal function parseObject(name:String, xmlToParse:XML, parent:FlowGroupElement, exceptionElements:Object = null) : void
      {
         if(!this.checkNamespace(xmlToParse))
         {
            return;
         }
         var info:FlowElementInfo = this._config.lookup(name);
         if(!info)
         {
            if(Boolean(exceptionElements == null) || Boolean(exceptionElements[name] === undefined))
            {
               this.handleUnknownElement(name,xmlToParse,parent);
            }
         }
         else
         {
            info.parser(this,xmlToParse,parent);
         }
      }
      
      protected function handleUnknownElement(name:String, xmlToParse:XML, parent:FlowGroupElement) : void
      {
         this.reportError(GlobalSettings.resourceStringFunction("unknownElement",[name]));
      }
      
      protected function handleUnknownAttribute(elementName:String, propertyName:String) : void
      {
         this.reportError(GlobalSettings.resourceStringFunction("unknownAttribute",[propertyName,elementName]));
      }
      
      protected function getElementInfo(xmlToParse:XML) : FlowElementInfo
      {
         return this._config.lookup(xmlToParse.name().localName);
      }
      
      protected function GetClass(xmlToParse:XML) : Class
      {
         var info:FlowElementInfo = this._config.lookup(xmlToParse.name().localName);
         return Boolean(info)?info.flowClass:null;
      }
      
      tlf_internal function createImpliedParagraph() : ParagraphElement
      {
         return this.createParagraphFromXML(<p/>);
      }
      
      tlf_internal function addChild(parent:FlowGroupElement, child:FlowElement) : Boolean
      {
         if(child is ParagraphFormattedElement)
         {
            this.resetImpliedPara();
         }
         else if(parent is ContainerFormattedElement)
         {
            if(!this._impliedPara)
            {
               this._impliedPara = this.createImpliedParagraph();
               parent.addChild(this._impliedPara);
            }
            var parent:FlowGroupElement = this._impliedPara;
         }
         if(this._throwOnError)
         {
            parent.addChild(child);
         }
         else
         {
            try
            {
               parent.addChild(child);
            }
            catch(e:*)
            {
               reportError(e);
               return false;
            }
         }
         return true;
      }
      
      tlf_internal function resetImpliedPara() : void
      {
         if(this._impliedPara)
         {
            this.onResetImpliedPara(this._impliedPara);
            this._impliedPara = null;
         }
      }
      
      protected function onResetImpliedPara(para:ParagraphElement) : void
      {
      }
      
      public function get errors() : Vector.<String>
      {
         return this._errors;
      }
      
      public function get throwOnError() : Boolean
      {
         return this._throwOnError;
      }
      
      public function set throwOnError(value:Boolean) : void
      {
         this._throwOnError = value;
      }
      
      protected function reportError(error:String) : void
      {
         if(this._throwOnError)
         {
            throw new Error(error);
         }
         if(!this._errors)
         {
            this._errors = new Vector.<String>();
         }
         this._errors.push(error);
      }
   }
}
