////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWKIT.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Draw a LED bar (filled rectangle) to BMP file.                             //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 2019xxxx - Created by A.K.                                                 //
// 20230623 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

program ledbmp; // LED bitmap

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$IFNDEF FPC}{$APPTYPE CONSOLE}{$ENDIF}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics, interfaces, forms,
 _crw_alloc, _crw_fio, _crw_colors;

type EWriteError=class(Exception);
type EBadArgument=class(Exception);

const stdin  = 0;
const stdout = 1;
const stderr = 2;

procedure Print(n:Integer; const S:LongString);
var h:THandle;
begin
 if Length(S)>0 then begin
  case n of
   stdout: h:=GetStdHandle(STD_OUTPUT_HANDLE);
   stderr: h:=GetStdHandle(STD_ERROR_HANDLE);
   else    h:=0;
  end;
  if (h<>0) then if FileWrite(h,PChar(S)^,Length(S))<0 then Raise EWriteError.Create('Fail write stream '+IntToStr(h));
 end;
end;

procedure Echo(s:LongString);
begin
 Print(stdout,s+EOL);
end;

procedure BugReport(E:Exception; Sender:TObject=nil; Note:LongString='');
var s,t:LongString;
begin
 if Assigned(E) then begin
  s:=Format('Exception "%s" - "%s"',[E.ClassName,E.Message]);
  SetString(t,PChar(s),Length(s));
  {$IFDEF WINDOWS}
  CharToOem(PChar(s),PChar(t));
  {$ENDIF WINDOWS}
  Print(stderr,t+EOL);
 end;
 ExitCode:=1;
end;

procedure PrintColors(Prefix:LongString='');
var i,n:Integer;
begin
 n:=1;
 if Assigned(KnownColorsList) then begin
  for i:=0 to KnownColorsList.Count-1 do n:=Max(n,Length(Prefix+KnownColorsList[i]));
  for i:=0 to KnownColorsList.Count-1 do Echo(Format('%-*s = $%.6x',[n,Prefix+KnownColorsList[i],TColor(KnownColorsList.Objects[i])]));
 end;
end;

function BaseName:LongString;
var i:Integer;
begin
 Result:=ExtractFileName(ParamStr(0));
 for i:=Length(Result) downto 1 do if Result[i]='.' then begin Result:=Copy(Result,1,i-1); break; end;
end;

procedure PrintHelp;
var exe:LongString;
begin
 Exe:=BaseName;
 Echo('Copyright(c) 2019-2022 Alexey Kuryakin kouriakine@mail.ru under MIT.');
 Echo(exe+' v 1.2 is the tool to create LED BMP (bitmaps for text fields)');
 Echo('with given text width, font size, color depth, fill color, font name');
 Echo('Usage:');
 Echo(' '+exe+' [-o] [w s d c f a t]  - create bitmap field for text');
 Echo(' with specified options [-o] and/or next parameters:');
 Echo('  w - required text width in chars');
 Echo('  s - required font size in pt units; by default 10pt');
 Echo('  d - required color depth: {1,4,8,15,16,24} bit; by default 24 bit');
 Echo('  c - required fill color, RGB number as hex $BBGGRR or named color');
 Echo('      call "'+exe+' --colors"  to print list of known named colors;');
 Echo('      by default uses color White = $FFFFFF');
 Echo('  f - font name; by default "PT Mono"');
 Echo('  a - caption text string; optional');
 Echo('  t - text color; optional');
 Echo(' Options may have two equivalent forms: short and long');
 Echo('  -oX (short) and --option=X (long) with argument X');
 Echo(' Options are:');
 Echo('  -h --help            - print help');
 Echo('  -v --verbose         - set verbose mode to view details');
 Echo('  -q --quiet           - set quiet (silent) mode to skip printing');
 Echo('     --colors          - print list of colors from X11,HTML,Win32');
 Echo('                       - in format Name = $Value ($BBGGRR hex RGB)');
 Echo('  -w --text-width=W    - set text width W, chars');
 Echo('  -s --font-size=S     - set font size S, pt units');
 Echo('  -d --color-depth=D   - set color depth D: {1,4,8,15,16,24} bit');
 Echo('  -c --fill-color=C    - set fill color number or name C');
 Echo('  -f --font-name=F     - set font name F');
 Echo('  -o --out-file=O      - set output file name O');
 Echo('  -x --x-space=X       - set horizontal space X');
 Echo('  -y --y-space=Y       - set vertical space Y');
 Echo('  -a --caption=A       - set caption text A');
 Echo('  -t --text-color=T    - set text color T');
 Echo('  -b --bold            - use bold font');
 Echo('  -i --italic          - use italic font');
 Echo('  -u --underline       - use underline font');
 Echo('Example:');
 Echo(' '+exe+' 7 10 8 Aqua "PT Mono"');
 Echo(' '+exe+' -w7 -s10 -d8 -cAqua -f"PT Mono"');
 Echo(' '+exe+' --text-width=7 --font-size=10 --color-depth=8 --fill-color=Aqua --font-name="PT Mono"');
 Echo(' - creates bitmap file '+exe+'_7_10_8_aqua_ptmono.bmp');
end;

function CheckOpt(const ShortOpt,LongOpt,Param:LongString; var Arg:LongString):Boolean;
begin
 Result:=False; Arg:='';
 if (ShortOpt<>'') and (Pos(ShortOpt,Param)=1) then begin
  Arg:=Param; Delete(Arg,1,Length(ShortOpt));
  Result:=True;
  Exit;
 end;
 if (LongOpt<>'') and (Pos(LongOpt,Param)=1) then begin
  Arg:=Param; Delete(Arg,1,Length(LongOpt));
  Result:=True;
  Exit;
 end;
end;

procedure Main;
var TextWidth,FontSize,ColorDepth,FillColor,TextColor,ParamNum,ParamMax,ArgNum,xSpace,ySpace,px,py:Integer;
    Caption,FontName,OutFile,Param,Arg:LongString; bmp:TBitmap; Quiet,Verbose:Boolean;
    FontStyle:TFontStyles;
begin
 if (ParamCount<1) then PrintHelp else
 try
  // Initialize and set defaults
  TextWidth:=0; FontSize:=10; ColorDepth:=24; FillColor:=clWhite; FontName:='PT Mono';
  TextColor:=clNone; xSpace:=0; ySpace:=0; OutFile:=''; FontStyle:=[];
  Quiet:=false; Verbose:=false;
  // Parse command line options and argiments
  ParamNum:=1; ArgNum:=1; ParamMax:=ParamCount;
  while (ParamNum<=ParamMax) do begin
   Param:=ParamStr(ParamNum);
   if Pos('-',Param)=1 then begin
    if CheckOpt('-h','--help',Param,Arg) then begin
     PrintHelp;
     Exit;
    end else
    if CheckOpt('','--colors',Param,Arg) then begin
     PrintColors;
     Exit;
    end else
    if CheckOpt('-v','--verbose',Param,Arg) then begin
     Verbose:=True;
    end else
    if CheckOpt('-q','--quiet',Param,Arg) then begin
     Quiet:=True;
    end else
    if CheckOpt('-b','--bold',Param,Arg) then begin
     FontStyle:=FontStyle+[fsBold];
    end else
    if CheckOpt('-i','--italic',Param,Arg) then begin
     FontStyle:=FontStyle+[fsItalic];
    end else
    if CheckOpt('-u','--underline',Param,Arg) then begin
     FontStyle:=FontStyle+[fsUnderline];
    end else
    if CheckOpt('-w','--text-width=',Param,Arg) then begin
     if (Arg<>'') then TextWidth:=StrToInt(Arg);
    end else
    if CheckOpt('-s','--font-size=',Param,Arg) then begin
     if (Arg<>'') then FontSize:=StrToInt(Arg);
    end else
    if CheckOpt('-d','--color-depth=',Param,Arg) then begin
     if (Arg<>'') then ColorDepth:=StrToInt(Arg);
    end else
    if CheckOpt('-c','--fill-color=',Param,Arg) then begin
     if (Arg<>'') then FillColor:=StringToColor(Arg);
    end else
    if CheckOpt('-f','--font-name=',Param,Arg) then begin
     if (Trim(Arg)<>'') then FontName:=Arg;
    end else
    if CheckOpt('-t','--text-color=',Param,Arg) then begin
     if (Arg<>'') then TextColor:=StringToColor(Arg);
    end else
    if CheckOpt('-a','--caption=',Param,Arg) then begin
     if (Trim(Arg)<>'') then Caption:=Arg;
    end else
    if CheckOpt('-x','--x-space=',Param,Arg) then begin
     if (Arg<>'') then xSpace:=StrToInt(Arg);
    end else
    if CheckOpt('-y','--y-space=',Param,Arg) then begin
     if (Arg<>'') then ySpace:=StrToInt(Arg);
    end else
    if CheckOpt('-o','--out-file=',Param,Arg) then begin
     if (Arg<>'') then OutFile:=ChangeFileExt(Arg,'.bmp');
    end else
    Raise EBadArgument.Create('Invalid option: '+Param);
   end else begin
    // Read arguments
    Arg:=Param;
    case ArgNum of
     1: if (Arg<>'') then TextWidth:=StrToInt(Arg);
     2: if (Arg<>'') then FontSize:=StrToInt(Arg);
     3: if (Arg<>'') then ColorDepth:=StrToInt(Arg);
     4: if (Arg<>'') then FillColor:=StringToColor(Arg);
     5: if (Arg<>'') then FontName:=Arg;
     6: if (Arg<>'') then Caption:=Arg;
     7: if (Arg<>'') then TextColor:=StringToColor(Arg);
    end;
    inc(ArgNum);
   end;
   inc(ParamNum);
  end;
  TextWidth:=Max(TextWidth,Length(Caption));
  if (Caption='') then Caption:=StringOfChar('0',TextWidth) else begin
   if TextWidth>Length(Caption) then Caption:=Caption+StringOfChar(' ',(TextWidth-Length(Caption)) div 2);
   if TextWidth>Length(Caption) then Caption:=StringOfChar(' ',TextWidth-Length(Caption))+Caption;
  end;
  if (TextWidth<=0) then Raise EBadArgument.Create('Text width is not valid: '+IntToStr(TextWidth));
  if (FontSize<=0) then Raise EBadArgument.Create('Font size is not valid: '+IntToStr(FontSize));
  if not (ColorDepth in [1,4,8,15,16,24]) then Raise EBadArgument.Create('Invalid Color Depth: out of set [1,4,8,15,16,24].');
  if (OutFile='') then OutFile:=Format('%s_%d_%d_%d_%s_%s.bmp',[BaseName,TextWidth,FontSize,ColorDepth,ColorToString(FillColor),StringReplace(FontName,' ','',[rfReplaceAll])]);
  OutFile:=LowerCase(OutFile);
  // Verbose information
  if Verbose and not Quiet then begin
   Echo('Options:');
   Echo(Format('--text-width  = %d',[TextWidth]));
   Echo(Format('--font-size   = %d',[FontSize]));
   Echo(Format('--color-depth = %d',[ColorDepth]));
   Echo(Format('--fill-color  = %s',[ColorToString(FillColor)]));
   Echo(Format('--font-name   = "%s"',[FontName]));
   Echo(Format('--caption     = "%s"',[Caption]));
   Echo(Format('--x-space     = %d',[xSpace]));
   Echo(Format('--y-space     = %d',[ySpace]));
   Echo(Format('--text-color  = %s',[ColorToString(TextColor)]));
   Echo(Format('--out-file    = %s',[OutFile]));
  end;
  // Create & fill bitmap
  bmp:=TBitmap.Create;
  try
   bmp.Canvas.Font.Name:=FontName;
   bmp.Canvas.Font.Size:=FontSize;
   bmp.Canvas.Font.Style:=FontStyle;
   bmp.Width:=bmp.Canvas.TextWidth(Caption)+2*xSpace;
   bmp.Height:=bmp.Canvas.TextHeight(Caption)+2*ySpace;
   case ColorDepth of
    1  : bmp.PixelFormat:=pf1bit;
    4  : bmp.PixelFormat:=pf4bit;
    8  : bmp.PixelFormat:=pf8bit;
    15 : bmp.PixelFormat:=pf15bit;
    16 : bmp.PixelFormat:=pf16bit;
    24 : bmp.PixelFormat:=pf24bit;
    else raise EBadArgument.Create('Invalid Color Depth: out of set {1,4,8,15,16,24}.');
   end;
   bmp.Canvas.Brush.Color:=FillColor;
   bmp.Canvas.Brush.Style:=bsSolid;
   bmp.Canvas.Pen.Color:=FillColor;
   bmp.Canvas.Pen.Mode:=pmCopy;
   bmp.Canvas.Pen.Style:=psSolid;
   bmp.Canvas.Pen.Width:=2;
   bmp.Canvas.Rectangle(0,0,bmp.Width,bmp.Height);
   Caption:=Trim(Caption);
   if (TextColor<>clNone) and (Caption<>'') then begin
    bmp.Canvas.Font.Color:=TextColor;
    px:=(bmp.Width-bmp.Canvas.TextWidth(Caption)) div 2;
    py:=(bmp.Height-bmp.Canvas.TextHeight(Caption)) div 2;
    bmp.Canvas.TextOut(px,py,Caption);
   end;
   bmp.SaveToFile(OutFile);
   if not Quiet then Echo(Format('%s - %d x %d x %d bitmap',[OutFile,bmp.Width,bmp.Height,ColorDepth]));
  finally
   FreeAndNil(bmp);
  end;
 except
  on E:Exception do BugReport(E,nil,'Main');
 end;
end;

begin
 try
  Main;
 except
  on E:Exception do BugReport(E,nil,'ledbmp');
 end;
end.

//////////////
// END OF FILE
//////////////

