////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Spectr Window.                                                        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231130 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_spectrwindow; // Form Spectr Window

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf, Printers,
 Form_CrwDaqSysChild, Form_CurveWindow,
 Form_ListBoxSelection, Form_TextEditDialog,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_sort,
 _crw_ef, _crw_ee, _crw_zm, _crw_curves,
 _crw_riff, _crw_colors,
 _crw_appforms, _crw_apptools, _crw_apputils;

const
  MaxBufPixels      = 1024*2;
  MinHorz           = 16;
  MaxHorz           = MaxInt;
  MinVert           = 16;
  MaxVert           = MaxInt;
  PeakMarkerMargin  = 1;
  PeakMarkerSize    = 12;
  DefaultSpectrFWHM = 5;

const
  clSpectrHistogram  : TColor = clLime;
  clSpectrGround     : TColor = clBlack;
  clSpectrDisabled   : TColor = clSilver;
  clSpectrROIMarker  : TColor = clYellow;
  clSpectrROISelect  : TColor = clAqua;
  clSpectrPeakMarker : TColor = clRed;
  clSpectrPeakForm   : TColor = clWhite;
  clSpectrPeakGnd    : TColor = clRed;

type
  TFormSpectrWindow = class;
  TSpectrDisplayX   = class(TMasterObject)
  private
    mySpecWin   : TFormSpectrWindow;
    myDisplay   : TPaintBox;
    myScale     : TPoint2D;
    myXorMarker : Integer;
    myBufPixel  : packed array[0..MaxBufPixels-1] of record
     Count      : Integer;
     ScreenY    : Integer;
     Color      : TColor;
     SummY      : Double;
    end;
  public
    constructor Create(TheSpecWin:TFormSpectrWindow; TheDisplay:TPaintBox);
    procedure   DrawBufPixel(IsHist:Boolean; aWidth:Integer);
    procedure   SpectrToBuf(StartIndex,StopIndex:Integer; IsLg:Boolean);
    function    ConvertX(X:Double):Integer;
    function    ConvertY(Value:Double; IsLog:Boolean):Integer;
    function    InvConvertX(I:Integer):Double;
    function    InvConvertY(I:Integer; IsLog:Boolean):Double;
  end;
  TSpectrDisplayExpand = class(TSpectrDisplayX)
  public
    procedure   Draw;
    procedure   XorVLine(WhereX:Integer);
    procedure   DrawMarker(WhereX:Integer);
    procedure   ReplaceMarker(NewPos:Integer);
    procedure   ReCalcScale;
    procedure   DrawPeakMarker(Chanel,Amplitude:Double; MarkerColor:TColor);
    procedure   DrawROIMarkers(Color:TColor);
  end;
  TSpectrDisplayFull = class(TSpectrDisplayX)
    procedure   Draw;
    procedure   ExpandInit;
    procedure   ReCalcScale;
    procedure   DrawPeakMarker(Chanel,Amplitude:double; MarkerColor:TColor);
    procedure   DrawROIMarkers(Color:TColor);
  end;
  TFormSpectrWindow = class(TFormCrwDaqSysChild)
    PanelFull: TPanel;
    PaintBoxFull: TPaintBox;
    PanelExpand: TPanel;
    PaintBoxExpand: TPaintBox;
    ActionSpectrViewRangeAuto: TAction;
    ActionSpectrViewRangeIncHorz: TAction;
    ActionSpectrViewRangeDecHorz: TAction;
    ActionSpectrViewRangeIncVert: TAction;
    ActionSpectrViewRangeDecVert: TAction;
    ActionSpectrViewAsHistogram: TAction;
    ActionSpectrViewInLogScale: TAction;
    ActionSpectrEditMarkerLeftSmall: TAction;
    ActionSpectrEditMarkerRightSmall: TAction;
    ActionSpectrEditMarkerLeftLarge: TAction;
    ActionSpectrEditMarkerRightLarge: TAction;
    ActionSpectrEditMarkerHome: TAction;
    ActionSpectrEditMarkerEnd: TAction;
    ActionSpectrEditRoiLeft: TAction;
    ActionSpectrEditRoiRight: TAction;
    ActionSpectrEditRoiSelect: TAction;
    ActionSpectrEditClear: TAction;
    ActionSpectrEditNote: TAction;
    ActionSpectrEditClone: TAction;
    ToolButtonSpectrEditRoiLeft: TToolButton;
    ToolButtonSpectrEditRoiRight: TToolButton;
    ToolButtonSpectrEditRoiSelect: TToolButton;
    ToolButtonSpectrHistogramSeparator: TToolButton;
    ToolButtonSpectrViewAsHistogram: TToolButton;
    ToolButtonSpectrViewInLogScale: TToolButton;
    ToolButtonSpectrViewRangeAutoSeparator: TToolButton;
    ToolButtonSpectrViewRangeAuto: TToolButton;
    ToolButtonSpectrViewRangeIncHorz: TToolButton;
    ToolButtonSpectrViewRangeDecHirz: TToolButton;
    ToolButtonSpectrViewRangeIncVert: TToolButton;
    ToolButtonSpectrViewRangeDecVert: TToolButton;
    ToolButtonSpectrEditClearSeparator: TToolButton;
    ToolButtonSpectrEditClear: TToolButton;
    ToolButtonSpectrEditNote: TToolButton;
    ToolButtonSpectrEditClone: TToolButton;
    MenuSpectrViewRange: TMenuItem;
    MenuSpectrViewRangeAuto: TMenuItem;
    MenuSpectrViewRangeIncHorz: TMenuItem;
    MenuSpectrViewRangeDecHorz: TMenuItem;
    MenuSpectrViewRangeIncVert: TMenuItem;
    MenuSpectrViewRangeDecVert: TMenuItem;
    MenuSpectrViewAsHistogram: TMenuItem;
    MenuSpectrViewInLogScale: TMenuItem;
    MenuSpectrEdit: TMenuItem;
    MenuSpectrEditMarker: TMenuItem;
    MenuSpectrEditMarkerLeftSmall: TMenuItem;
    MenuSpectrEditMarkerRightSmall: TMenuItem;
    MenuSpectrEditMarkerLeftLarge: TMenuItem;
    MenuSpectrEditMarkerRightLarge: TMenuItem;
    MenuSpectrEditMarkerHome: TMenuItem;
    MenuSpectrEditMarkerEnd: TMenuItem;
    MenuSpectrEditRoi: TMenuItem;
    MenuSpectrEditRoiLeft: TMenuItem;
    MenuSpectrEditRoiRight: TMenuItem;
    MenuSpectrEditRoiSelect: TMenuItem;
    MenuSpectrEditClear: TMenuItem;
    MenuSpectrEditNote: TMenuItem;
    MenuSpectrEditClone: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure PaintBoxFullPaint(Sender: TObject);
    procedure PaintBoxExpandPaint(Sender: TObject);
    procedure PaintBoxFullMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure PaintBoxExpandMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure PaintBoxExpandMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure PaintBoxExpandMouseUp(Sender: TObject; Button: TMouseButton;  Shift: TShiftState; X, Y: Integer);
    procedure PaintBoxExpandDblClick(Sender: TObject);
    procedure ActionSpectrViewRangeAutoExecute(Sender: TObject);
    procedure ActionSpectrViewRangeIncHorzExecute(Sender: TObject);
    procedure ActionSpectrViewRangeDecHorzExecute(Sender: TObject);
    procedure ActionSpectrViewRangeIncVertExecute(Sender: TObject);
    procedure ActionSpectrViewRangeDecVertExecute(Sender: TObject);
    procedure ActionSpectrViewAsHistogramExecute(Sender: TObject);
    procedure ActionSpectrViewInLogScaleExecute(Sender: TObject);
    procedure ActionSpectrEditMarkerHomeExecute(Sender: TObject);
    procedure ActionSpectrEditMarkerEndExecute(Sender: TObject);
    procedure ActionSpectrEditMarkerLeftSmallExecute(Sender: TObject);
    procedure ActionSpectrEditMarkerRightSmallExecute(Sender: TObject);
    procedure ActionSpectrEditMarkerLeftLargeExecute(Sender: TObject);
    procedure ActionSpectrEditMarkerRightLargeExecute(Sender: TObject);
    procedure ActionSpectrEditRoiLeftExecute(Sender: TObject);
    procedure ActionSpectrEditRoiRightExecute(Sender: TObject);
    procedure ActionSpectrEditRoiSelectExecute(Sender: TObject);
    procedure ActionSpectrEditClearExecute(Sender: TObject);
    procedure ActionSpectrEditNoteExecute(Sender: TObject);
    procedure ActionSpectrEditCloneExecute(Sender: TObject);
  private
    { Private declarations }
    myExpandView : TSpectrDisplayExpand;
    myFullView   : TSpectrDisplayFull;
    myXorRoiSel  : TXorSelector;
    mySpectr     : TCurve;
    myOwnsSpectr : Boolean;
    myMarker     : Integer;
    myMarkerL    : Integer;
    myMarkerR    : Integer;
    myLastMarker : Integer;
    myBegX       : Integer;
    myEndX       : Integer;
    myHorz       : Integer;
    myVert       : Integer;
    myROIL       : Integer;
    myROIR       : Integer;
    myIsLog      : Boolean;
    myIsHist     : Boolean;
    function  GetExpandView:TSpectrDisplayExpand;
    function  GetFullView:TSpectrDisplayFull;
    function  GetSpectrSize:Integer;
    function  GetSpectrValue(aIndex:Integer):Double;
    procedure SetSpectrValue(aIndex:Integer; aValue:Double);
    function  GetSpectrColor(i:Integer):TColor;
    function  GetSpectrName:LongString;
    function  GetSpectrIsEmpty:Boolean;
    function  GetMarker:Integer;
    procedure SetMarker(aMarker:Integer);
    function  GetMarkerL:Integer;
    procedure SetMarkerL(aMarker:Integer);
    function  GetMarkerR:Integer;
    procedure SetMarkerR(aMarker:Integer);
    procedure ValidateMarkers;
    function  GetIntegral:Double;
    function  GetBegX:Integer;
    function  GetEndX:Integer;
    function  GetHorz:Integer;
    procedure SetHorz(aHorz:Integer);
    function  GetVert:Integer;
    procedure SetVert(aVert:Integer);
    function  GetROIL:Integer;
    function  GetROIR:Integer;
    function  GetIsLog:Boolean;
    procedure SetIsLog(aIsLog:Boolean);
    function  GetIsHist:Boolean;
    procedure SetIsHist(aIsHist:Boolean);
    function  GetNote:TText;
 public
    { Public declarations }
    property  ExpandView             : TSpectrDisplayExpand read GetExpandView;
    property  FullView               : TSpectrDisplayFull   read GetFullView;
    property  SpectrSize             : Integer     read GetSpectrSize;
    property  SpectrValue[i:Integer] : Double      read GetSpectrValue    write SetSpectrValue; default;
    property  SpectrColor[i:Integer] : TColor      read GetSpectrColor;
    property  SpectrName             : LongString  read GetSpectrName;
    property  SpectrIsEmpty          : Boolean     read GetSpectrIsEmpty;
    property  Marker                 : Integer     read GetMarker        write SetMarker;
    property  MarkerL                : Integer     read GetMarkerL       write SetMarkerL;
    property  MarkerR                : Integer     read GetMarkerR       write SetMarkerR;
    property  Integral               : Double      read GetIntegral;
    property  BegX                   : Integer     read GetBegX;
    property  EndX                   : Integer     read GetEndX;
    property  Horz                   : Integer     read GetHorz          write SetHorz;
    property  Vert                   : Integer     read GetVert          write SetVert;
    property  ROIL                   : Integer     read GetROIL;
    property  ROIR                   : Integer     read GetROIR;
    property  IsLog                  : Boolean     read GetIsLog         write SetIsLog;
    property  IsHist                 : Boolean     read GetIsHist        write SetIsHist;
    property  Note                   : TText       read GetNote;
 public
    procedure AssignSpectr(aSpectr:TCurve; aOwnsSpectr:Boolean);
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
    procedure UpdateCommands; override;
    procedure PrintView; override;
    procedure FileSave; override;
    procedure FileSaveAs; override;
    procedure DrawView; override;
    procedure DrawFul; virtual;
    procedure DrawExp; virtual;
    procedure MarkerChanged; virtual;
    procedure UpdateMarkerView(aMarker:Integer);
    function  FormatMarker(aMarker:Integer):LongString; virtual;
    procedure Config(CfgFile,Section:LongString); virtual;
    procedure Animate; virtual;
    function  EnCalibrExist:Boolean; virtual;
    function  EnCalibr(Chan:Double):Double; virtual;
    procedure EnCalibrChanged; virtual;
    function  HwCalibrExist:Boolean; virtual;
    function  HwCalibr(Chan:Double):Double; virtual;
    procedure HwCalibrChanged; virtual;
    procedure SpectrClear; virtual;
    procedure AutoRange;
    procedure IncHorz;
    procedure DecHorz;
    procedure IncVert;
    procedure DecVert;
    procedure SetMarkerLR(L,R:Integer);
    procedure ROIMark;
    procedure ROIUnMark;
    procedure ROIMarkUnMark;
    procedure EditNote;
    function  SaveCalibrations(FileName:LongString):Boolean; virtual;
    function  LoadCalibrations(FileName:LongString):Boolean; virtual;
    function  SaveSPD(FileName : LongString;
                      DataSec  : LongString = '[SpectrData]';
                      NoteSec  : LongString = '[SpectrNote]'
                                   ) : Boolean;
    function  LoadSPD(FileName : LongString;
                      DataSec  : LongString = '[SpectrData]';
                      NoteSec  : LongString = '[SpectrNote]'
                                   ) : Boolean;
    function  GetRoiIntegral(L,R,N:Integer):Double;
  end;
  TSpectrWindowList = class(TObjectStorage)
  private
    function   GetWindow(i:Integer):TFormSpectrWindow;
    procedure  SetWindow(i:Integer; aWindow:TFormSpectrWindow);
  public
    property   Window[i:Integer]:TFormSpectrWindow read GetWindow write SetWindow; default;
  end;
  TSpecWindowConstructor = function(const aCaption    : LongString;
                                          aSpectr     : TCurve;
                                          aOwnsSpectr : Boolean ) : TFormSpectrWindow;

const
  regSpecWin = 'SpecWin';
  FullSpectrWindowList : TSpectrWindowList = nil;

procedure RegisterSpectrWindowConstructor(aConstructor : TSpecWindowConstructor;
                                    const aClassName   : LongString);

function  NewSpectrWindow(const aClassName  : LongString;
                                aCaption    : LongString;
                                aSpectr     : TCurve;
                                aOwnsSpectr : Boolean) : TFormSpectrWindow;

function  NewSpecWin(const aCaption    : LongString;
                           aSpectr     : TCurve;
                           aOwnsSpectr : Boolean):TFormSpectrWindow;
procedure Kill(var TheObject:TFormSpectrWindow); overload;

function  ActiveSpectrWindow:TFormSpectrWindow;

function  NewSpectrWindowList(aOwnsObjects : Boolean = true;
                              aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                              aStep        : LongInt = DefaultTObjectStorageStep
                                         ) : TSpectrWindowList;
procedure Kill(var TheObject:TSpectrWindowList); overload;

procedure ReadSpectrWindowsPalette(TheIniFile:LongString);

implementation

{$R *.lfm}

 {
 *******************************************************************************
 Spectr window descendants registry
 *******************************************************************************
 }
type
 TSpecRegItem = class(TMasterObject)
 private
  Cons : TSpecWindowConstructor;
  Name : String[31];
 end;

const
 RegList:TObjectStorage=nil;

function RegFind(const s:LongString):TSpecRegItem;
var i:Integer;
begin
 Result:=nil;
 if RegList.Ok then
 if IsNonEmptyStr(s) then
 for i:=0 to RegList.Count-1 do
 if (RegList[i] is TSpecRegItem) then
 with TSpecRegItem(RegList[i]) do
 if SameText(UnifyAlias(Name),UnifyAlias(s)) then begin
  Result:=TSpecRegItem(RegList[i]);
  Break;
 end;
end;

procedure RegisterSpectrWindowConstructor(aConstructor : TSpecWindowConstructor;
                                    const aClassName   : LongString);
var Item : TSpecRegItem;
begin
 if Assigned(aConstructor) and IsNonEmptyStr(aClassName) then begin
  Item:=RegFind(aClassName);
  if RegList.Ok and not Item.Ok then begin
   Item:=TSpecRegItem.Create;
   RegList.Add(Item);
  end;
  if Item.Ok then begin
   Item.Cons:=aConstructor;
   Item.Name:=aClassName;
  end;
 end;
end;

function  NewSpectrWindow(const aClassName  : LongString;
                                aCaption    : LongString;
                                aSpectr     : TCurve;
                                aOwnsSpectr : Boolean) : TFormSpectrWindow;
var Item : TSpecRegItem;
begin
 Result:=nil;
 Item:=RegFind(aClassName);
 if Item.Ok then
 if Assigned(Item.Cons) then
 if Assigned(aSpectr) and (aSpectr.Count>=MinHorz) and aSpectr.IsStatic then begin
  if IsEmptyStr(aCaption) then begin
   aCaption:=RusEng('Спектр','Spectr');
   if IsNonEmptyStr(aSpectr.Name) then aCaption:=aCaption+' '+aSpectr.Name;
  end;
  Result:=Item.Cons(aCaption, aSpectr, aOwnsSpectr);
 end;
end;

 {
 *******************************************************************************
 TSpectrWindowList implementation
 *******************************************************************************
 }
function TSpectrWindowList.GetWindow(i:Integer):TFormSpectrWindow;
begin
 Result:=TFormSpectrWindow(Items[i]);
end;

procedure TSpectrWindowList.SetWindow(i:Integer; aWindow:TFormSpectrWindow);
begin
 Items[i]:=aWindow;
end;

function NewSpectrWindowList(aOwnsObjects : Boolean = true;
                             aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                             aStep        : LongInt = DefaultTObjectStorageStep
                                        ) : TSpectrWindowList;
begin
 Result:=TSpectrWindowList.Create(aOwnsObjects,aCapacity,aStep);
end;

procedure Kill(var TheObject:TSpectrWindowList); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

 {
 *******************************************************************************
 General purpose routines
 *******************************************************************************
 }
function  NewSpecWin(const aCaption    : LongString;
                           aSpectr     : TCurve;
                           aOwnsSpectr : Boolean ) : TFormSpectrWindow;
begin
 Application.CreateForm(TFormSpectrWindow, Result);
 if Result.Ok then
 with Result do
 try
  LockDraw;
  if aCaption=''
  then Caption:=RusEng('Спектрометрическое окно','Spectrometry window')
  else Caption:=aCaption;
  AssignSpectr(aSpectr,aOwnsSpectr);
 finally
  UnlockDraw;
 end;
end;

procedure Kill(var TheObject:TFormSpectrWindow); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

function ActiveSpectrWindow:TFormSpectrWindow;
var Child:TForm;
begin
 Result:=nil;
 try
  if SdiMan.FindActiveChild(Child,sf_SdiChild,sf_SdiControl) then
  if (Child is TFormSpectrWindow) then Result:=TFormSpectrWindow(Child);
 except
  on E:Exception do BugReport(E,nil,'ActiveSpectrWindow');
 end;
end;

procedure ReadSpectrWindowsPalette(TheIniFile:LongString);
 function ReadColor(const ColName:LongString; DefColor:TColor):TColor;
 var s:LongString;
 begin
  Result:=DefColor;
  try
   s:='';
   if ReadIniFileAlpha(TheIniFile,'[Spectr Windows Palette]',ColName+'%a',s)
   or ReadIniFileAlpha(SysIniFile,'[Spectr Windows Palette]',ColName+'%a',s)
   then Result:=StringToColor(s);
  except
   on E:Exception do Result:=DefColor;
  end;
 end;
begin
 TheIniFile:=UnifyFileAlias(TheIniFile);
 clSpectrHistogram  := ReadColor('clSpectrHistogram',  clSpectrHistogram);
 clSpectrGround     := ReadColor('clSpectrGround',     clSpectrGround);
 clSpectrDisabled   := ReadColor('clSpectrDisabled',   clSpectrDisabled);
 clSpectrROIMarker  := ReadColor('clSpectrROIMarker',  clSpectrROIMarker);
 clSpectrROISelect  := ReadColor('clSpectrROISelect',  clSpectrROISelect);
 clSpectrPeakMarker := ReadColor('clSpectrPeakMarker', clSpectrPeakMarker);
 clSpectrPeakForm   := ReadColor('clSpectrPeakForm',   clSpectrPeakForm);
 clSpectrPeakGnd    := ReadColor('clSpectrPeakGnd',    clSpectrPeakGnd);
end;

 {
 *******************************************************************************
 TSpectrDisplayX implementation
 *******************************************************************************
 }
constructor TSpectrDisplayX.Create(TheSpecWin:TFormSpectrWindow; TheDisplay:TPaintBox);
begin
 inherited Create;
 mySpecWin:=TheSpecWin;
 myDisplay:=TheDisplay;
 myScale:=Point2D(1,1);
 myXorMarker:=0;
 SafeFillChar(myBufPixel,sizeof(myBufPixel),0);
end;

procedure TSpectrDisplayX.DrawBufPixel(IsHist:Boolean; aWidth:Integer);
var ix:Integer; R:TRect2I;
begin
 if Assigned(Self) then
 for ix:=0 to min(myDisplay.Width,MaxBufPixels)-1 do
 with myBufPixel[ix] do
 if Count>0 then begin
  if IsHist then begin
   R.A.X:=ix;  R.A.Y:=ScreenY;
   R.B.X:=ix;  R.B.Y:=myDisplay.Height;
   if aWidth>1 then begin
    RectGrow(R,aWidth,0);
    DrawBar(myDisplay.Canvas, R, Color);
   end else begin
    DrawLine(myDisplay.Canvas, R.A, R.B, Color, psSolid, pmCopy, 1);
   end;
  end else begin
   R.A.X:=ix;  R.A.Y:=ScreenY;
   R.B.X:=ix;  R.B.Y:=ScreenY;
   if aWidth>1 then begin
    RectGrow(R, aWidth, aWidth);
    DrawBar(myDisplay.Canvas, R, Color);
   end else begin
    myDisplay.Canvas.Pixels[R.A.X, R.A.Y]:=Color;
   end;
  end;
 end;
end;

procedure TSpectrDisplayX.SpectrToBuf(StartIndex,StopIndex:Integer; IsLg:Boolean);
var i,ix,ixcount:Integer;
begin
 if Assigned(Self) then begin
  ixcount:=min(myDisplay.Width,MaxBufPixels);
  SafeFillChar(myBufPixel,sizeof(myBufPixel),0);
  if (mySpecWin.SpectrSize>0) then begin
   StartIndex:=max(0,StartIndex);
   StopIndex:=min(mySpecWin.SpectrSize-1,StopIndex);
   for i:=StartIndex to StopIndex do begin
    ix:=ConvertX(i-StartIndex);
    if InRange(ix,0,ixcount-1) then with myBufPixel[ix] do begin
     inc(Count);
     SummY:=SummY+mySpecWin.SpectrValue[i];
     Color:=mySpecWin.SpectrColor[i];
    end;
   end;
   for ix:=0 to ixcount-1 do with myBufPixel[ix] do
   if Count>0 then ScreenY:=max(0,ConvertY(max(SummY,0)/Count,IsLg));
  end;
 end;
end;

function  TSpectrDisplayX.ConvertX(X:Double):Integer;
begin
 if Assigned(Self) then Result:=Round(X * myScale.X) else Result:=0;
end;

function  TSpectrDisplayX.ConvertY(Value:Double; IsLog:Boolean):Integer;
var Y:Double;
begin
 if Assigned(Self) then begin
  if IsLog then Y:=myScale.Y*Ln(Value+1.0) else Y:=myScale.Y*Value;
  Result:=myDisplay.Height-1-Round(max(0,min(myDisplay.Height-1,Y)));
 end else Result:=0;
end;

function  TSpectrDisplayX.InvConvertX(I:Integer):Double;
begin
 if Assigned(Self) then Result:=I/myScale.X else Result:=0;
end;

function  TSpectrDisplayX.InvConvertY(I:Integer; IsLog:Boolean):Double;
begin
 if Assigned(Self) then begin
  I:=myDisplay.Height-1-I;
  if IsLog then Result:=exp(I/myScale.Y) else Result:=I/myScale.Y;
 end else Result:=0;
end;

 {
 *******************************************************************************
 TSpectrDisplayExpand implementation
 *******************************************************************************
 }
procedure TSpectrDisplayExpand.Draw;
var aWidth:Integer;
begin
 if Assigned(Self) then begin
  DrawBar(myDisplay.Canvas, Rect2I(myDisplay.ClientRect), clSpectrGround);
  if mySpecWin.SpectrSize>0 then begin
   ReCalcScale;
   SpectrToBuf(mySpecWin.BegX, mySpecWin.EndX, mySpecWin.IsLog);
   aWidth:=myDisplay.Width div (4 * max(1,mySpecWin.EndX-mySpecWin.BegX));
   aWidth:=max(1,min(5,aWidth));
   DrawBufPixel(mySpecWin.IsHist,aWidth);
   DrawROIMarkers(clSpectrROIMarker);
   mySpecWin.DrawExp;
   DrawMarker(ConvertX(mySpecWin.Marker-mySpecWin.BegX));
  end;
 end;
end;

procedure TSpectrDisplayExpand.ReCalcScale;
begin
 if Assigned(Self) then
 if (mySpecWin.SpectrSize>0) then begin
  myScale.X:=myDisplay.Width/mySpecWin.Horz;
  if mySpecWin.IsLog
  then myScale.Y:=myDisplay.Height/Ln(High(mySpecWin.Vert))
  else myScale.Y:=myDisplay.Height/mySpecWin.Vert;
 end;
end;

procedure TSpectrDisplayExpand.XorVLine(WhereX:Integer);
begin
 if Assigned(Self) then
 DrawLine(myDisplay.Canvas, Point2I(WhereX, 0),
                            Point2I(WhereX, myDisplay.Height-1),
                            clWhite, psSolid, pmNot, 1);
end;

procedure TSpectrDisplayExpand.DrawMarker(WhereX:Integer);
begin
 if Assigned(Self) then begin
  myXorMarker:=WhereX;
  XorVLine(myXorMarker);
 end;
end;

procedure TSpectrDisplayExpand.ReplaceMarker(NewPos:Integer);
begin
 if Assigned(Self) then
 if (NewPos<>myXorMarker) then begin
  DrawMarker(myXorMarker);
  DrawMarker(NewPos);
  mySpecWin.MarkerChanged;
 end;
end;

procedure TSpectrDisplayExpand.DrawPeakMarker(Chanel,Amplitude:Double; MarkerColor:TColor);
var P:TPoint2I;
begin
 if Assigned(Self) then
 if (mySpecWin.SpectrSize>0) then
 if InRange(Chanel,mySpecWin.BegX,mySpecWin.EndX) then begin
  P.X:=ConvertX(Chanel-mySpecWin.BegX);
  P.Y:=max(ConvertY(Amplitude,mySpecWin.IsLog),0)-PeakMarkerMargin;
  DrawLine(myDisplay.Canvas, P, Point2I(P.X, P.Y - PeakMarkerSize), MarkerColor);
 end;
end;

procedure TSpectrDisplayExpand.DrawROIMarkers(Color:TColor);
var MarkL,MarkR:TPoint2I;
begin
 if Assigned(Self) then
 if (mySpecWin.SpectrSize>0) then begin
  {левый маркер ROI}
  MarkL.X:=max(0, ConvertX(mySpecWin.MarkerL-mySpecWin.BegX)-1);
  MarkL.Y:=ConvertY(mySpecWin.SpectrValue[mySpecWin.MarkerL], mySpecWin.IsLog);
  DrawLine(myDisplay.Canvas, Point2I(MarkL.X, MarkL.Y-8), Point2I(MarkL.X,   MarkL.Y),   Color);
  DrawLine(myDisplay.Canvas, Point2I(MarkL.X, MarkL.Y-8), Point2I(MarkL.X-3, MarkL.Y-8), Color);
  {правый маркер ROI}
  MarkR.X:=ConvertX(mySpecWin.MarkerR-mySpecWin.BegX)+1;
  MarkR.Y:=ConvertY(mySpecWin.SpectrValue[mySpecWin.MarkerR], mySpecWin.IsLog);
  DrawLine(myDisplay.Canvas, Point2I(MarkR.X, MarkR.Y-8), Point2I(MarkR.X,   MarkR.Y),   Color);
  DrawLine(myDisplay.Canvas, Point2I(MarkR.X, MarkR.Y-8), Point2I(MarkR.X+3, MarkR.Y-8), Color);
 end;
end;

 {
 *******************************************************************************
 TSpectrDisplayFull implementation
 *******************************************************************************
 }
procedure TSpectrDisplayFull.Draw;
var RectExp,R:TRect2I;
begin
 if Assigned(Self) then
 if (mySpecWin.SpectrSize>0) then begin
  ReCalcScale;
  SpectrToBuf(0,mySpecWin.SpectrSize-1,true);
  RectExp:=Rect2I(myDisplay.ClientRect);
  RectExp.A.X:=Round(myScale.X*mySpecWin.BegX);
  RectExp.B.X:=Round(myScale.X*mySpecWin.EndX);
  R:=Rect2I(myDisplay.ClientRect);
  R.B.X:=RectExp.A.X;
  DrawBar(myDisplay.Canvas, R, clSpectrDisabled);
  R:=Rect2I(myDisplay.ClientRect);
  R.A.X:=RectExp.B.X;
  DrawBar(myDisplay.Canvas, R, clSpectrDisabled);
  DrawBar(myDisplay.Canvas, RectExp, clSpectrGround);
  DrawBufPixel(mySpecWin.IsHist,1);
  DrawROIMarkers(clSpectrROIMarker);
  mySpecWin.DrawFul;
 end else begin
  R:=Rect2I(0, 0, myDisplay.Width, myDisplay.Height);
  DrawBar(myDisplay.Canvas, R, clSpectrGround);
 end;
end;

procedure TSpectrDisplayFull.ExpandInit;
begin
 if Assigned(Self) then
 if (mySpecWin.SpectrSize>0) then begin
  mySpecWin.myBegX:=max(0, mySpecWin.Marker-mySpecWin.Horz shr 1);
  mySpecWin.myEndX:=mySpecWin.BegX+mySpecWin.Horz-1;
  if mySpecWin.EndX > mySpecWin.SpectrSize-1 then begin
   mySpecWin.myEndX:=mySpecWin.SpectrSize-1;
   mySpecWin.myBegX:=max(0, mySpecWin.EndX-mySpecWin.Horz-1);
  end;
 end;
end;

procedure TSpectrDisplayFull.ReCalcScale;
begin
 if Assigned(Self) then
 if (mySpecWin.SpectrSize>0) then begin
  myScale.X:=myDisplay.Width/mySpecWin.SpectrSize;
  myScale.Y:=myDisplay.Height/Ln(High(mySpecWin.Vert));
 end;
end;

procedure TSpectrDisplayFull.DrawPeakMarker(Chanel,Amplitude:double; MarkerColor:TColor);
var P:TPoint;
begin
 if Assigned(Self) then
 if (mySpecWin.SpectrSize>0) then
 if InRange(Chanel,0,mySpecWin.SpectrSize-1) then begin
  P.X:=ConvertX(Chanel);
  P.Y:=max(ConvertY(Amplitude,True),0)-PeakMarkerMargin;
  DrawLine(myDisplay.Canvas, Point2I(P.X, P.Y), Point2I(P.X, P.Y-PeakMarkerSize), MarkerColor);
 end;
end;

procedure TSpectrDisplayFull.DrawROIMarkers(Color:TColor);
var MarkL,MarkR:TPoint;
begin
 if Assigned(Self) then
 if (mySpecWin.SpectrSize>0) then begin
  {левый маркер ROI}
  MarkL.X:=max(0, ConvertX(mySpecWin.MarkerL)-1);
  MarkL.Y:=ConvertY(mySpecWin.SpectrValue[mySpecWin.MarkerL], true);
  DrawLine(myDisplay.Canvas, Point2I(MarkL.X, MarkL.Y-8), Point2I(MarkL.X,   MarkL.Y),   Color);
  DrawLine(myDisplay.Canvas, Point2I(MarkL.X, MarkL.Y-8), Point2I(MarkL.X-3, MarkL.Y-8), Color);
  {правый маркер ROI}
  MarkR.X:=ConvertX(mySpecWin.MarkerR)+1;
  MarkR.Y:=ConvertY(mySpecWin.SpectrValue[mySpecWin.MarkerR], true);
  DrawLine(myDisplay.Canvas, Point2I(MarkR.X, MarkR.Y-8), Point2I(MarkR.X,   MarkR.Y),   Color);
  DrawLine(myDisplay.Canvas, Point2I(MarkR.X, MarkR.Y-8), Point2I(MarkR.X+3, MarkR.Y-8), Color);
 end;
end;

 {
 *******************************************************************************
 TFormSpectrWindow implementation
 *******************************************************************************
 }
function  TFormSpectrWindow.GetExpandView:TSpectrDisplayExpand;
begin
 if Assigned(Self) then Result:=myExpandView else Result:=nil;
end;

function  TFormSpectrWindow.GetFullView:TSpectrDisplayFull;
begin
 if Assigned(Self) then Result:=myFullView else Result:=nil;
end;

function TFormSpectrWindow.GetSpectrSize:Integer;
begin
 if Assigned(Self) then Result:=mySpectr.Count else Result:=0;
end;

function TFormSpectrWindow.GetSpectrValue(aIndex:Integer):Double;
begin
 if Assigned(Self) then Result:=mySpectr[aIndex].Y else Result:=0;
end;

procedure TFormSpectrWindow.SetSpectrValue(aIndex:Integer; aValue:Double);
begin
 if Assigned(Self) then begin
  mySpectr.Lock;
  mySpectr[aIndex]:=Point2D(mySpectr[aIndex].X,aValue);
  mySpectr.Unlock;
 end;
end;

function TFormSpectrWindow.GetSpectrColor(i:Integer):TColor;
begin
 Result:=clSpectrGround;
 if Assigned(Self) then
 if (i>=myROIL) and (i<=myROIR) and (myROIL<myROIR)
 then Result:=clSpectrROISelect
 else Result:=clSpectrHistogram;
end;

function TFormSpectrWindow.GetSpectrName:LongString;
begin
 if Assigned(Self) then Result:=mySpectr.Name else Result:='';
end;

function TFormSpectrWindow.GetSpectrIsEmpty:Boolean;
var i:Integer;
begin
 Result:=true;
 for i:=0 to SpectrSize-1 do
 if (SpectrValue[i]<>0) then begin
  Result:=false;
  break;
 end;
end;

function TFormSpectrWindow.GetMarker:Integer;
begin
 if Assigned(Self) then Result:=myMarker else Result:=0;
end;

procedure TFormSpectrWindow.SetMarker(aMarker:Integer);
begin
 if Ok then
 if InRange(aMarker,0,SpectrSize-1) then begin
  if InRange(aMarker,BegX,EndX) then begin
   myMarker:=aMarker;
   myExpandView.ReplaceMarker(myExpandView.ConvertX(myMarker-BegX));
  end else
  try
   LockDraw;
   myMarker:=aMarker;
   myFullView.ExpandInit;
  finally
   UnlockDraw;
  end;
 end;
end;

function TFormSpectrWindow.GetMarkerL:Integer;
begin
 if Assigned(Self) then Result:=myMarkerL else Result:=0;
end;

procedure TFormSpectrWindow.SetMarkerL(aMarker:Integer);
begin
 if (SpectrSize>0) then
 try
  LockDraw;
  myMarkerL:=aMarker;
  ValidateMarkers;
 finally
  UnlockDraw;
 end;
end;

function TFormSpectrWindow.GetMarkerR:Integer;
begin
 if Assigned(Self) then Result:=myMarkerR else Result:=0;
end;

procedure TFormSpectrWindow.SetMarkerR(aMarker:Integer);
begin
 if SpectrSize>0 then
 try
  LockDraw;
  myMarkerR:=aMarker;
  ValidateMarkers;
 finally
  UnlockDraw;
 end;
end;

procedure TFormSpectrWindow.ValidateMarkers;
begin
 if Ok then begin
  myMarkerL:=max(0,min(myMarkerL,SpectrSize-1));
  myMarkerR:=max(0,min(myMarkerR,SpectrSize-1));
  if (MarkerL>MarkerR) then ExchangeVar(myMarkerL,myMarkerR);
 end;
end;

function TFormSpectrWindow.GetIntegral:Double;
var i:Integer;
begin
 Result:=0;
 for i:=MarkerL to MarkerR do Result:=Result+SpectrValue[i];
end;

function TFormSpectrWindow.GetRoiIntegral(L,R,N:Integer):Double;
var i:Integer;
begin
 Result:=0;
 if (L<R) and (L>=0) and (R<SpectrSize) then
 case N of
  0: for i:=L to R do Result:=Result+SpectrValue[i];
  1: Result:=0.5*(SpectrValue[L]+SpectrValue[R])*(R-L+1);
  2: Result:=GetRoiIntegral(L,R,0)-GetRoiIntegral(L,R,1);
 end;
end;

function TFormSpectrWindow.GetBegX:Integer;
begin
 if Assigned(Self) then Result:=myBegX else Result:=0;
end;

function TFormSpectrWindow.GetEndX:Integer;
begin
 if Assigned(Self) then Result:=myEndX else Result:=0;
end;

function TFormSpectrWindow.GetHorz:Integer;
begin
 if Assigned(Self) then Result:=myHorz else Result:=0;
end;

procedure TFormSpectrWindow.SetHorz(aHorz:Integer);
begin
 if Assigned(Self) then
 try
  LockDraw;
  myHorz:=max(MinHorz,min(MaxHorz,aHorz));
  myFullView.ExpandInit;
 finally
  UnlockDraw;
 end;
end;

function TFormSpectrWindow.GetVert:Integer;
begin
 if Assigned(Self) then Result:=myVert else Result:=0;
end;

procedure TFormSpectrWindow.SetVert(aVert:Integer);
begin
 if Assigned(Self) then
 try
  LockDraw;
  myVert:=max(MinVert,min(MaxVert,aVert));
 finally
  UnlockDraw;
 end;
end;

function TFormSpectrWindow.GetROIL:Integer;
begin
 if Assigned(Self) then Result:=myROIL else Result:=0;
end;

function TFormSpectrWindow.GetROIR:Integer;
begin
 if Assigned(Self) then Result:=myROIR else Result:=0;
end;

function TFormSpectrWindow.GetIsLog:Boolean;
begin
 if Assigned(Self) then Result:=myIsLog else Result:=false;
end;

procedure TFormSpectrWindow.SetIsLog(aIsLog:Boolean);
begin
 if Ok then
 try
  LockDraw;
  myIsLog:=aIsLog;
 finally
  UnlockDraw;
 end;
end;

function TFormSpectrWindow.GetIsHist:Boolean;
begin
 if Assigned(Self) then Result:=myIsHist else Result:=false;
end;

procedure TFormSpectrWindow.SetIsHist(aIsHist:Boolean);
begin
 if Ok then
 try
  LockDraw;
  myIsHist:=aIsHist;
 finally
  UnlockDraw;
 end;
end;

function TFormSpectrWindow.GetNote:TText;
begin
 if Assigned(Self) then Result:=mySpectr.Comment else Result:=nil;
end;

procedure TFormSpectrWindow.AssignSpectr(aSpectr:TCurve; aOwnsSpectr:Boolean);
begin
 if Assigned(Self) then
 try
  LockDraw;
  if myOwnsSpectr then Kill(mySpectr);
  mySpectr:=aSpectr;
  myOwnsSpectr:=aOwnsSpectr;
  myHorz:=max(MinHorz,SpectrSize);
  myBegX:=0;
  myEndX:=Horz-1;
  myVert:=1024;
  myMarker:=Horz div 2;
  myMarkerL:=Horz div 10;
  myMarkerR:=EndX-(Horz div 10);
  myROIL:=0;
  myROIR:=0;
  myLastMarker:=-1;
  myIsLog:=false;
  myIsHist:=true;
  myFullView.ExpandInit;
  AutoRange;
 finally
  UnlockDraw;
 end;
end;

procedure TFormSpectrWindow.UpdateCommands;
var Exposed:Boolean;
begin
 inherited UpdateCommands;
 Exposed:=FormIsExposed(Self);
 SetEnabledActions(False,[]);
 SetEnabledActions(Exposed,[
                   ActionFileSave,
                   ActionFileSaveAs,
                   ActionFilePrint]);
 SetEnabledActions(Exposed and (SpectrSize>0),[
                   ActionSpectrEditMarkerLeftSmall,
                   ActionSpectrEditMarkerRightSmall,
                   ActionSpectrEditMarkerLeftLarge,
                   ActionSpectrEditMarkerRightLarge,
                   ActionSpectrEditMarkerHome,
                   ActionSpectrEditMarkerEnd,
                   ActionSpectrEditRoiLeft,
                   ActionSpectrEditRoiRight,
                   ActionSpectrEditRoiSelect,
                   ActionSpectrViewAsHistogram,
                   ActionSpectrViewInLogScale,
                   ActionSpectrViewRangeAuto,
                   ActionSpectrViewRangeIncHorz,
                   ActionSpectrViewRangeDecHorz,
                   ActionSpectrViewRangeIncVert,
                   ActionSpectrViewRangeDecVert,
                   ActionSpectrEditClear,
                   ActionSpectrEditNote]);
 ActionSpectrEditRoiSelect.Checked:=(ROIL<ROIR);
 ActionSpectrViewAsHistogram.Checked:=IsHist;
 ActionSpectrViewInLogScale.Checked:=IsLog;
end;

procedure TFormSpectrWindow.PrintView;
var Bmp:TBitmap; Key:Integer; mm:TMainMenu; tb,sb:Boolean;
var Params:LongString; fsp:TPoint;
begin
 fsp:=ClientToScreen(Point(0,0));
 Params:='@set Panel.Font   Name:PT_Mono\Size:12\Color:Navy\Style:[Bold]'+EOL
        +'@set ListBox.Font Name:PT_Mono\Size:12\Color:Black\Style:[Bold]'+EOL
        +'@set Form.Left '+IntToStr(fsp.x)+' relative Screen'+EOL
        +'@set Form.Top  '+IntToStr(fsp.y)+' relative Screen'+EOL;
 Key:=ListBoxMenu(RusEng('Файл\Печать','File\Print'),
                  RusEng('Как печатать','How to print'),
                  RusEng('Скопировать изображение в Буфер Обмена (цвет.)'+EOL+
                         'Скопировать изображение в Буфер Обмена (ч./б.)'+EOL+
                         'Скопировать изображение в Буфер Обмена (серый)'+EOL+
                         'Напечатать на Принтере '+Printer.PrinterName,
                         'Copy bitmap to Clipboard (color)'+EOL+
                         'Copy bitmap to Clipboard (b./w.)'+EOL+
                         'Copy bitmap to Clipboard (gray)'+EOL+
                         'Print hardcopy on Printer '+Printer.PrinterName),
                         myPrintViewKey,Params);
 if (Key>=0) then
 try
  myPrintViewKey:=Key;
  tb:=ToolBar.Visible;
  sb:=StatusBar.Visible;
  mm:=Menu;
  try
   if HasFlags(HidesOnPrint,hop_ToolBar) then ToolBar.Visible:=False;
   if HasFlags(HidesOnPrint,hop_StatusBar) then StatusBar.Visible:=False;
   if HasFlags(HidesOnPrint,hop_Menu) then Menu:=nil;
   if HasFlags(HidesOnPrint,hop_DrawView) then DrawView;
   if HasFlags(HidesOnPrint,hop_ProcMess) then SafeApplicationProcessMessages;
   case Key of
    0..2: begin
           Bmp:=GetPrintableImage;
           if Assigned(Bmp) then
           try
            if (Key=1) then ConvertBmpToBlackAndWhite(Bmp,clSpectrGround,clSpectrGround,1);
            if (Key=2) then ConvertBmpToBlackAndWhite(Bmp,clSpectrGround,clSpectrGround,3);
            CopyFormBmpToClipboard(Bmp,true);
           finally
            Kill(Bmp);
           end;
          end;
    3:    if HasPrintersDialog then Print;
   end;
  finally
   if HasFlags(HidesOnPrint,hop_Menu) then Menu:=mm;
   if HasFlags(HidesOnPrint,hop_ToolBar) then ToolBar.Visible:=tb;
   if HasFlags(HidesOnPrint,hop_StatusBar) then StatusBar.Visible:=sb;
  end;
  if HasFlags(HidesOnPrint,hop_DrawView) then DrawView;
  if HasFlags(HidesOnPrint,hop_ProcMess) then SafeApplicationProcessMessages;
 except
  on E:Exception do BugReport(E,Self,'PrintView');
 end;
end;

procedure TFormSpectrWindow.FileSave;
begin
 FileSaveAs;
end;

procedure TFormSpectrWindow.FileSaveAs;
const FName:LongString='';
var Ext:LongString;
begin
 SaveDialog.Title:=RusEng('Файл\Сохранить','File\Save');
 SaveDialog.Filter:=RusEng('Файлы сректров      (*.spd)|*.spd|'+
                           'Текстовые файлы     (*.txt)|*.txt|'+
                           'Все остальные файлы   (*.*)|*.*|',
                           'Spectr files        (*.spd)|*.spd|'+
                           'Text files          (*.txt)|*.txt|'+
                           'All other files       (*.*)|*.*|');
 if IsEmptyStr(SaveDialog.FileName) then SaveDialog.FileName:=FName;
 if IsEmptyStr(SaveDialog.FileName)
 then OpenDialogSelectType(SaveDialog,'*.spd')
 else OpenDialogSelectType(SaveDialog,SaveDialog.FileName);
 if GuardOpenDialog(SaveDialog).Execute then begin
  FName:=UnifyFileAlias(SaveDialog.FileName);
  Ext:=UnifyAlias(SaveDialog.DefaultExt);
  if IsEmptyStr(Ext) or IsWildCard(Ext)
  then FName:=UnifyFileAlias(FName,ua_FileDefLow)
  else FName:=UnifyFileAlias(ForceExtension(FName,Ext),ua_FileDefLow);
  if FileExists(FName) and
     (YesNo(RusEng('Файл существует!'+EOL+'Хотите перезаписать файл?',
                   'File exists!'+EOL+'Do you want overwrite this file?')+EOL+FName)<>mrYes)
  then FName:='';
  if IsNonEmptyStr(FName) then
  if not SaveSPD(FName) then
  SystemConsole.Activate;
 end;
end;

procedure TFormSpectrWindow.DrawView;
begin
 if Ok and IsFormViewable then begin
  DebugLogReport_DrawView;
  PaintBoxFull.Repaint;
  PaintBoxExpand.Repaint;
  UpdateMarkerView(Marker);
 end;
end;

procedure TFormSpectrWindow.DrawFul;
begin
end;

procedure TFormSpectrWindow.DrawExp;
begin
end;

procedure TFormSpectrWindow.MarkerChanged;
begin
 UpdateMarkerView(Marker);
end;


procedure TFormSpectrWindow.UpdateMarkerView(aMarker:Integer);
begin
 if Ok then begin
  StatusBar.SimpleText:=FormatMarker(aMarker);
  StatusBar.Update;
 end;
end;

function TFormSpectrWindow.FormatMarker(aMarker:Integer):LongString;
begin
 if SpectrSize>0
 then Result:=Format('  %s:  Chn=%-5d  Cnt=%-10.0f',[mySpectr.Name, aMarker, SpectrValue[aMarker]])
 else Result:='';
end;

procedure TFormSpectrWindow.Config(CfgFile,Section:LongString);
begin
end;

procedure TFormSpectrWindow.Animate;
begin
end;

function TFormSpectrWindow.EnCalibrExist:Boolean;
begin
 Result:=false;
end;

function TFormSpectrWindow.EnCalibr(Chan:Double):Double;
begin
 Result:=Chan;
end;

procedure TFormSpectrWindow.EnCalibrChanged;
var i:Integer;
begin
 UpdateMarkerView(Marker);
 for i:=0 to SpectrSize-1 do mySpectr[i]:=Point2D(EnCalibr(i),SpectrValue[i]);
end;

function TFormSpectrWindow.HwCalibrExist:Boolean;
begin
 Result:=false;
end;

function TFormSpectrWindow.HwCalibr(Chan:Double):Double;
begin
 Result:=DefaultSpectrFWHM;
end;

procedure TFormSpectrWindow.HwCalibrChanged;
begin
 UpdateMarkerView(Marker);
end;

procedure TFormSpectrWindow.SpectrClear;
var i:Integer;
begin
 if Ok then
 try
  LockDraw;
  for i:=0 to SpectrSize-1 do SpectrValue[i]:=0;
  Note.Text:='';
 finally
  UnlockDraw;
 end;
end;

procedure TFormSpectrWindow.AutoRange;
var i,W:LongInt; MaxVal:Double;
begin
 if SpectrSize>0 then
 try
  LockDraw;
  IsLog:=false;
  MaxVal:=0;
  for i:=BegX to EndX do MaxVal:=max(MaxVal,SpectrValue[i]);
  for i:=4 to 31 do begin
   W:=LongInt(1) shl i;
   if (MaxVal<W) then begin
    MaxVal:=W;
    Break;
   end;
  end;
  Vert:=Round(MaxVal);
 finally
  UnlockDraw;
 end;
end;

procedure TFormSpectrWindow.IncHorz;
begin
 if SpectrSize>0 then
 try
  LockDraw;
  Horz:=min(SpectrSize,max(MinHorz,Horz shl 1));
 finally
  UnlockDraw;
 end;
end;

procedure TFormSpectrWindow.DecHorz;
begin
 if SpectrSize>0 then
 try
  LockDraw;
  Horz:=min(SpectrSize,max(MinHorz,Horz shr 1));
 finally
  UnlockDraw;
 end;
end;

procedure TFormSpectrWindow.IncVert;
begin
 if SpectrSize>0 then
 try
  LockDraw;
  IsLog:=false;
  if Vert <= High(Vert) shr 1 then Vert:=Vert shl 1;
 finally
  UnlockDraw;
 end;
end;

procedure TFormSpectrWindow.DecVert;
begin
 if SpectrSize>0 then
 try
  LockDraw;
  IsLog:=false;
  if Vert >= MinVert shl 1 then Vert:=Vert shr 1;
 finally
  UnlockDraw;
 end;
end;

procedure TFormSpectrWindow.SetMarkerLR(L,R:Integer);
begin
 if SpectrSize>0 then
 try
  LockDraw;
  L:=max(0,min(L,SpectrSize-1));
  R:=max(0,min(R,SpectrSize-1));
  if L>R then ExchangeVar(L,R);
  myMarkerL:=L;
  myMarkerR:=R;
  ValidateMarkers;
 finally
  UnlockDraw;
 end;
end;

procedure TFormSpectrWindow.ROIMark;
begin
 if Ok then
 try
  LockDraw;
  myROIL:=MarkerL;
  myROIR:=MarkerR;
 finally
  UnlockDraw;
 end;
end;

procedure TFormSpectrWindow.ROIUnMark;
begin
 if Ok then
 try
  LockDraw;
  myROIL:=0;
  myROIR:=0;
 finally
  UnlockDraw;
 end;
end;

procedure TFormSpectrWindow.ROIMarkUnMark;
begin
 if (ROIL=0) and (ROIR=0) then ROIMark else ROIUnMark;
end;

procedure TFormSpectrWindow.EditNote;
begin
 if SpectrSize>0 then
 try
  mySpectr.Comment.Text:=TextEditDialog(RusEng('Редактирование комментария спектра ','Edit comment of spectr ')+mySpectr.Name,
                                        RusEng('Отредактируйте:','Please, edit:'),mySpectr.Comment.Text);
 except
  on E:Exception do BugReport(E,Self,'EditNote');
 end;
end;

function TFormSpectrWindow.SaveCalibrations(FileName:LongString):Boolean;
begin
 Result:=True;
end;

function TFormSpectrWindow.LoadCalibrations(FileName:LongString):Boolean;
begin
 Result:=True;
end;

function TFormSpectrWindow.SaveSPD(FileName : LongString;
                                   DataSec  : LongString = '[SpectrData]';
                                   NoteSec  : LongString = '[SpectrNote]'
                                          ) : Boolean;
var
 i : Integer;
 p : TText;
begin
 Result:=false;
 if Ok then
 if (SpectrSize>0) then
 try
  p:=NewText;
  try
   FileName:=UnifyFileAlias(FileName,ua_FileDefLow);
   if IsNonEmptyStr(DataSec) then p.Addln(DataSec);
   for i:=0 to SpectrSize-1 do p.Addln(Format('%d'#9'%.0f',[i,SpectrValue[i]]));
   if (Note.Count>0) and IsNonEmptyStr(NoteSec) then begin
    p.Addln(NoteSec);
    p.Concat(Note);
   end;
   if p.WriteFile(FileName)<>0
   then Raise EWriteError.CreateFmt(RusEng('Ошибка записи "%s".','Error writing "%s".'),[FileName]);
   Result:=SaveCalibrations(FileName);
  finally
   Kill(p);
  end;
 except
  on E:Exception do BugReport(E,Self,'SaveSPD');
 end;
end;

function TFormSpectrWindow.LoadSPD(FileName : LongString;
                                   DataSec  : LongString = '[SpectrData]';
                                   NoteSec  : LongString = '[SpectrNote]'
                                          ) : Boolean;
var
 p   : TText;
 i   : Integer;
 chn : Double;
 cnt : Double;
begin
 Result:=false;
 if Ok then
 if SpectrSize>0 then
 try
  FileName:=UnifyFileAlias(FileName);
  if not FileExists(FileName)
  then Raise EReadError.CreateFmt(RusEng('Файл не найден "%s".','File not found "%s".'),[FileName]);
  p:=NewText;
  try
   SpectrClear;
   if IsNonEmptyStr(DataSec) then
   p.Text:=ExtractTextSection(FileName,DataSec,efAsIs+efDelCom);
   if p.Count=0 then p.Text:=ExtractTextSection(FileName,'[]',efAsIs+efDelCom);
   if p.Count=0 then Raise EReadError.CreateFmt(RusEng('Не могу прочитать "%s".','Could not read "%s".'),[FileName]);
   for i:=0 to p.Count-1 do
   if Str2Real(ExtractWord(1,p[i],ScanSpaces),chn) and
      Str2Real(ExtractWord(2,p[i],ScanSpaces),cnt)
   then begin
    if (chn<0) or (chn>=SpectrSize) or (cnt<0) or (cnt>MaxInt)
    then Raise EReadError.CreateFmt(RusEng('Неверные данные (%g, %g).','Invalid data (%g, %g).'),[chn,cnt]);
    SpectrValue[Round(chn)]:=cnt;
   end;
   if IsEmptyStr(NoteSec)
   then Note.Text:=''
   else Note.Text:=ExtractTextSection(FileName,NoteSec,efAsIs);
   Result:=LoadCalibrations(FileName);
   AutoRange;
  finally
   Kill(p);
  end;
 except
  on E:Exception do BugReport(E,Self,'LoadSPD');
 end;
end;

procedure TFormSpectrWindow.FormCreate(Sender: TObject);
begin
 inherited;
 try
  LockDraw;
  myExpandView:=TSpectrDisplayExpand.Create(Self,PaintBoxExpand);
  myFullView:=TSpectrDisplayFull.Create(Self,PaintBoxFull);
  myXorRoiSel:=NewXorSelector;
  myXorRoiSel.Master:=@myXorRoiSel;
  myXorRoiSel.Mode:=sm_VertLine;
  myXorRoiSel.Width:=3;
  mySpectr:=nil;
  myOwnsSpectr:=false;
  AssignSpectr(nil,false);
 finally
  UnlockDraw;
 end;
 ActiveControl:=PanelExpand;
 UpdateMenu(MenuSpectrViewRange,
            RusEng('Масштаб','Range'),
            RusEng('Меню функций масштабирования.','Visible range menu.'),
            0);
 UpdateMenu(MenuSpectrViewRangeAuto,
            RusEng('Автоматически','Auto'),
            RusEng('Установить масштаб автоматически.','Autorange to see all.'),
            VK_F8);
 UpdateMenu(MenuSpectrViewRangeIncHorz,
            RusEng('Горизонтальное увеличение','Horz zoom in'),
            RusEng('Увеличить масштаб по горизонтали.','Horizontal zoom in.'),
            VK_ADD);
 UpdateMenu(MenuSpectrViewRangeDecHorz,
            RusEng('Горизонтальное уменьшение','Horz zoom out'),
            RusEng('Уменьшить масштаб по горизонтали.','Horizontal zoom out.'),
            VK_SUBTRACT);
 UpdateMenu(MenuSpectrViewRangeIncVert,
            RusEng('Вертикальное увеличение','Vert zoom in'),
            RusEng('Увеличить масштаб по вертикали.','Vertical zoom in.'),
            VK_NEXT);
 UpdateMenu(MenuSpectrViewRangeDecVert,
            RusEng('Вертикальное уменьшение','Vert zoom out'),
            RusEng('Уменьшить масштаб по вертикали.','Vertival zoom out.'),
            VK_PRIOR);
 UpdateMenu(MenuSpectrViewAsHistogram,
            RusEng('Гистограмма','View as histogram'),
            RusEng('Показывать спектр как гистограмму или точками.','Show spectr as bars or points.'),
            0);
 UpdateMenu(MenuSpectrViewInLogScale,
            RusEng('Лог/Лин шкала','Lin/Log scale'),
            RusEng('Логарифмическая или линейная шкала.','Logarithm/linear scale.'),
            0);
 UpdateMenu(MenuSpectrEdit,
            RusEng('Правка','Edit')+MenuRightSpace,
            RusEng('Меню функций выбора/редактирования.','Editor operations.'),
            0);
 UpdateMenu(MenuSpectrEditMarker,
            RusEng('Маркер','Marker'),
            RusEng('Меню функций перемещения маркера.','Marker movement menu.'),
            0);
 UpdateMenu(MenuSpectrEditMarkerLeftSmall,
            RusEng('Шаг влево','Step left'),
            RusEng('Переместить маркер на 1 влево.','Small marker step to left.'),
            ShortCut(VK_LEFT,[]));
 UpdateMenu(MenuSpectrEditMarkerRightSmall,
            RusEng('Шаг вправо','Step right'),
            RusEng('Переместить маркер на 1 вправо.','Small marker step to right.'),
            ShortCut(VK_RIGHT,[]));
 UpdateMenu(MenuSpectrEditMarkerLeftLarge,
            RusEng('Прыжок влево','Jump left'),
            RusEng('Большой шаг маркера влево.','Large marker step to left.'),
            ShortCut(VK_LEFT,[ssAlt]));
 UpdateMenu(MenuSpectrEditMarkerRightLarge,
            RusEng('Прыжок вправо','Jump right'),
            RusEng('Большой шаг маркера вправо.','Large marker step to right.'),
            ShortCut(VK_RIGHT,[ssAlt]));
 UpdateMenu(MenuSpectrEditMarkerHome,
            RusEng('К левому краю',''),
            RusEng('Переместить маркер к левому краю.','Move marker to left side.'),
            ShortCut(VK_LEFT,[ssCtrl]));
 UpdateMenu(MenuSpectrEditMarkerEnd,
            RusEng('К правому краю','Right side'),
            RusEng('Переместить маркер к правому краю.','Move marker to right side.'),
            ShortCut(VK_RIGHT,[ssCtrl]));
 UpdateMenu(MenuSpectrEditRoi,
            RusEng('РОИ','ROI'),
            RusEng('Меню работы с РегиОном Интереса(РОИ).','Menu to work with ROI.'),
            0);
 UpdateMenu(MenuSpectrEditRoiLeft,
            RusEng('Левый маркер','Left marker'),
            RusEng('Установить левый маркер РОИ.','Set left ROI marker.'),
            0);
 UpdateMenu(MenuSpectrEditRoiRight,
            RusEng('Правый маркер','Right marker'),
            RusEng('Установить правый маркер РОИ.','Set right ROI marker.'),
            0);
 UpdateMenu(MenuSpectrEditRoiSelect,
            RusEng('Выделить РОИ','Select ROI'),
            RusEng('Выделить РегиОн Интереса между маркерами.','Select ROI inside of left/right markers.'),
            0);
 UpdateMenu(MenuSpectrEditClear,
            RusEng('Очистить спектр','Clear spectr'),
            RusEng('Занулить все данные в спектре.','Clear all data of spectr.'),
            0);
 UpdateMenu(MenuSpectrEditNote,
            RusEng('Править комментарий','Edit spectr comment'),
            RusEng('Редактировать комментарий к спектру.','Edit spectr comment.'),
            0);
 UpdateMenu(MenuSpectrEditClone,
            RusEng('Клонировать','Clone'),
            RusEng('Создать копию спектра в виде графика.','Clone spectr as curve.'),
            0);
end;

procedure TFormSpectrWindow.FormDestroy(Sender: TObject);
begin
 Kill(TObject(myExpandView));
 Kill(TObject(myFullView));
 Kill(myXorRoiSel);
 AssignSpectr(nil,false);
 inherited;
end;

procedure TFormSpectrWindow.AfterConstruction;
begin
 inherited AfterConstruction;
 FullSpectrWindowList.Add(Self);
 AddonSdiFlags(sf_SdiSpectrWin);
end;

procedure TFormSpectrWindow.BeforeDestruction;
begin
 FullSpectrWindowList.Remove(Self);
 inherited BeforeDestruction;
end;

procedure TFormSpectrWindow.PaintBoxFullPaint(Sender: TObject);
begin
 inherited;
 myFullView.Draw;
end;

procedure TFormSpectrWindow.PaintBoxExpandPaint(Sender: TObject);
begin
 inherited;
 myExpandView.Draw;
end;

procedure TFormSpectrWindow.PaintBoxFullMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
 inherited;
 if SpectrSize>0 then begin
  case Button of
   mbLeft : try
             LockDraw;
             Marker:=Round(myFullView.InvConvertX(X));
             myFullView.ExpandInit;
            finally
             UnlockDraw;
            end;
  end;
 end;
end;

procedure TFormSpectrWindow.PaintBoxExpandMouseDown(Sender: TObject;  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
 MarkerN : Integer;
 AdjustX : Integer;
begin
 inherited;
 if SpectrSize>0 then begin
  case Button of
   mbLeft : begin
             MarkerN:=BegX+Round(myExpandView.InvConvertX(X));
             AdjustX:=myExpandView.ConvertX(MarkerN-BegX);
             Marker:=MarkerN;
             myXorRoiSel.StartXY(PaintBoxExpand.Canvas,AdjustX,Y);
            end;
  end;
 end;
end;

procedure TFormSpectrWindow.PaintBoxExpandMouseMove(Sender: TObject;  Shift: TShiftState; X, Y: Integer);
var
 MarkerN : Integer;
 AdjustX : Integer;
begin
 if (SpectrSize>0) and myXorRoiSel.IsStart then begin
  MarkerN:=BegX+Round(myExpandView.InvConvertX(X));
  AdjustX:=myExpandView.ConvertX(MarkerN-BegX);
  UpdateMarkerView(MarkerN);
  myXorRoiSel.ReplaceXY(PaintBoxExpand.Canvas,AdjustX,Y);
 end;
end;

procedure TFormSpectrWindow.PaintBoxExpandMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
 inherited;
 if (SpectrSize>0) then begin
  case Button of
   mbLeft : if myXorRoiSel.Stop(PaintBoxExpand.Canvas) then begin
             Marker:=BegX+Round(myExpandView.InvConvertX(X));
             if abs(RectSizeX(myXorRoiSel.Selection))>2 then
             with myXorRoiSel.Selection do
             SetMarkerLR(BegX+Round(myExpandView.InvConvertX(A.X)),
                         BegX+Round(myExpandView.InvConvertX(B.X)));
            end;
  end;
 end;
end;

procedure TFormSpectrWindow.PaintBoxExpandDblClick(Sender: TObject);
var P:TPoint2I;
begin
 if (SpectrSize>0) then begin
  P:=Point2I(PaintBoxExpand.ScreenToClient(Mouse.CursorPos));
  Marker:=BegX+Round(myExpandView.InvConvertX(P.X));
  if myLastMarker=-1 then begin
   myLastMarker:=Marker;
   SetMarkerLR(myLastMarker,Marker);
  end else begin
   SetMarkerLR(myLastMarker,Marker);
   myLastMarker:=-1;
  end;
 end;
end;

procedure TFormSpectrWindow.ActionSpectrViewRangeAutoExecute(Sender: TObject);
begin
 inherited;
 AutoRange;
end;

procedure TFormSpectrWindow.ActionSpectrViewRangeIncHorzExecute(Sender: TObject);
begin
 IncHorz;
end;

procedure TFormSpectrWindow.ActionSpectrViewRangeDecHorzExecute(Sender: TObject);
begin
 DecHorz;
end;

procedure TFormSpectrWindow.ActionSpectrViewRangeIncVertExecute(Sender: TObject);
begin
 IncVert;
end;

procedure TFormSpectrWindow.ActionSpectrViewRangeDecVertExecute(Sender: TObject);
begin
 DecVert;
end;

procedure TFormSpectrWindow.ActionSpectrViewAsHistogramExecute(Sender: TObject);
begin
 if Ok then
 with ActionSpectrViewAsHistogram do begin
  Checked:=not Checked;
  IsHist:=Checked;
 end;
end;

procedure TFormSpectrWindow.ActionSpectrViewInLogScaleExecute(Sender: TObject);
begin
 with ActionSpectrViewInLogScale do begin
  Checked:=not Checked;
  IsLog:=Checked;
 end;
end;

procedure TFormSpectrWindow.ActionSpectrEditMarkerLeftSmallExecute(Sender: TObject);
begin
 inherited;
 Marker:=Marker-1;
end;

procedure TFormSpectrWindow.ActionSpectrEditMarkerRightSmallExecute(Sender: TObject);
begin
 inherited;
 Marker:=Marker+1;
end;

procedure TFormSpectrWindow.ActionSpectrEditMarkerLeftLargeExecute(Sender: TObject);
begin
 inherited;
 Marker:=Marker-(Horz shr 3);
end;

procedure TFormSpectrWindow.ActionSpectrEditMarkerRightLargeExecute(Sender: TObject);
begin
 inherited;
 Marker:=Marker+(Horz shr 3);
end;

procedure TFormSpectrWindow.ActionSpectrEditMarkerHomeExecute(Sender: TObject);
begin
 inherited;
 Marker:=BegX;
end;

procedure TFormSpectrWindow.ActionSpectrEditMarkerEndExecute(Sender: TObject);
begin
 inherited;
 Marker:=EndX;
end;

procedure TFormSpectrWindow.ActionSpectrEditRoiLeftExecute(Sender: TObject);
begin
 MarkerL:=Marker;
end;

procedure TFormSpectrWindow.ActionSpectrEditRoiRightExecute(Sender: TObject);
begin
 MarkerR:=Marker;
end;

procedure TFormSpectrWindow.ActionSpectrEditRoiSelectExecute(Sender: TObject);
begin
 with ActionSpectrEditRoiSelect do begin
  Checked:=not Checked;
  if Checked then RoiMark else RoiUnmark;
 end;
end;

procedure TFormSpectrWindow.ActionSpectrEditClearExecute(Sender: TObject);
begin
 if Ok then
 if YesNo(RusEng('Вы действительно хотите очистить спектр?',
                 'Do you really want to clear spectr?'))=mrYes
 then SpectrClear;
end;

procedure TFormSpectrWindow.ActionSpectrEditNoteExecute(Sender: TObject);
begin
 EditNote;
end;

procedure TFormSpectrWindow.ActionSpectrEditCloneExecute(Sender: TObject);
var Win:TFormCurveWindow; i:Integer;
begin
 inherited;
 if Ok then
 if SpectrSize>0 then begin
  Win:=NewCurveWindow(RusEng('Спектр: ','Spectr: ')+SpectrName,
                      RusEng(^C+ASCII_CR+^L'  [Счет]',^C+ASCII_CR+^L+'  [Count]'),
                      RusEng(^R'[кэВ]  '+ASCII_CR+^C,^R'[keV]  '+ASCII_CR+^C));
  Win.AddCurve(NewCurve(SpectrSize,SpectrName,clBlack,$1F));
  for i:=0 to SpectrSize-1 do Win.Curves[0][i]:=Point2d(EnCalibr(i),SpectrValue[i]);
  Win.Width:=400;
  Win.Height:=300;
  Win.AutoRange;
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_spectrwindow;
begin
 FullSpectrWindowList:=NewSpectrWindowList(false);
 FullSpectrWindowList.Master:=@FullSpectrWindowList;
 RegList:=NewObjectStorage;
 RegList.Master:=@RegList;
 RegisterSpectrWindowConstructor(NewSpecWin, regSpecWin);
end;

procedure Free_form_spectrwindow;
begin
 ResourceLeakageLog(Format('%-60s = %d',['FullSpectrWindowList.Count', FullSpectrWindowList.Count]));
 Kill(FullSpectrWindowList);
 Kill(RegList);
end;

initialization

 Init_form_spectrwindow;

finalization

 Free_form_spectrwindow;

end.

//////////////
// END OF FILE
//////////////

