////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Spectr Pik Window.                                                    //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231203 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_spectrpikwindow; // Form Spectr Pik Window

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 Form_SpectrWindow, Form_SpectrDaqWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_sort,
 _crw_ef, _crw_ee, _crw_zm, _crw_curves,
 _crw_riff, _crw_calib, _crw_lsqpoly,
 _crw_daqsys, _crw_daqdev, _crw_peaksearch,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormSpectrPikWindow = class(TFormSpectrDaqWindow)
    ActionSpectrPeakSearchControl: TAction;
    ToolButtonSpectrPeakControlSeparator: TToolButton;
    ToolButtonSpectrPeakSearchControl: TToolButton;
    MenuSpectrPeakSearchControl: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ActionSpectrPeakSearchControlExecute(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    Peak : packed record
     Control      : TMasterForm;
     isViewPos    : Boolean;       {надо отображать положения пиков?}
     isViewForm   : Boolean;       {рисовать форму пика?}
     isViewGround : Boolean;       {рисовать фон?}
     isViewSingle : Boolean;       {рисовать один пик-ближайший к курсору}
     NumPeaks     : Integer;       {число обнаруженных пиков}
     NBeg         : LongInt;       {0..MaxCount-1,начальный канал спектра}
     NEnd         : LongInt;       {0..MaxCount-1,конечный канал спектра}
     Spectr       : TDoubleVector; {массив спектра}
     C_Peak       : TDoubleVector; {массив расчетного спектра}
     C_Smooth     : TDoubleVector; {массив сглаженного спектра}
     Channel      : TDoubleVector; {положение пика-канал}
     Energy       : TDoubleVector; {положение пика-энергия}
     Amplitude    : TDoubleVector; {амплитуда пика}
     FWHM         : TDoubleVector; {ПШПВ пика}
     Area         : TDoubleVector; {площадь под пиком}
     Ground       : TDoubleVector; {фон под пиком}
     LeftBound    : TDoubleVector; {признак границы слева}
     LeftGround   : TDoubleVector; {фон слева от пика}
     RightBound   : TDoubleVector; {признак границы справа}
     RightGround  : TDoubleVector; {фон справа от пика}
     ChanError    : TDoubleVector; {ошибка определения канала}
     AreaError    : TDoubleVector; {ошибка определения площади}
     FwhmError    : TDoubleVector; {ошибка определения полуширины}
     StatError    : TDoubleVector; {стат. пика}
     Derive       : TDoubleVector; {массив производных}
     Gauss        : TDoubleVector; {временный массив}
    end;
    procedure PeakSort;
    procedure PeakClear(Confirm:Boolean=false);
    function  PeakFindNearest(aChannel:Double):Integer;
    function  PeakSave(FName:LongString):Boolean;
    function  PeakLoad(FName:LongString):Boolean;
    procedure PeakValidateEnergy;
    procedure PeakValidateArea;
    function  PeakSumOfGaussians(Chan:Double; ib,ie:Integer):Double;
    procedure PeakMove(dir:Integer);
    procedure PeakNext;
    procedure PeakPrev;
    procedure PeakSearch;
    procedure PeakGetVisibleRange(out ib,ie:Integer);
    procedure DrawFul; override;
    procedure DrawPeaks(ib,ie:Integer; chb,che:Double; const Ground:TPolynom;
                        cPeakGround,cPeakForm:TColor; isGround,isForm:Boolean);
    procedure DrawExp; override;
    procedure MarkerChanged; override;
    procedure SpectrClear; override;
    procedure Animate; override;
    procedure PeakControlOpen;
    procedure PeakControlUpdate(Flags:Integer);
  end;

const
 regSpecPikWin = 'SpecPikWin';
 MaxNumPeaks   = 256;

function  NewSpecPikWin(const aCaption    : LongString;
                              aSpectr     : TCurve;
                              aOwnsSpectr : Boolean):TFormSpectrWindow;
procedure Kill(var TheObject:TFormSpectrPikWindow); overload;
procedure LoadPeakInParams(FileName,Section:LongString);

implementation

{$R *.lfm}

uses
 Form_SpectrPikControl;

 {
 *******************************************************************************
 General purpose utilites
 *******************************************************************************
 }
function  NewSpecPikWin(const aCaption    : LongString;
                              aSpectr     : TCurve;
                              aOwnsSpectr : Boolean ) : TFormSpectrWindow;
begin
 Application.CreateForm(TFormSpectrPikWindow, Result);
 if Result.Ok then
 with Result do
 try
  LockDraw;
  if aCaption=''
  then Caption:=RusEng('Спектрометрическое окно','Spectrometry window')
  else Caption:=aCaption;
  AssignSpectr(aSpectr,aOwnsSpectr);
 finally
  UnlockDraw;
 end;
end;

procedure Kill(var TheObject:TFormSpectrPikWindow); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

procedure LoadPeakInParams(FileName,Section:LongString);
var P:TPeakInParamsArray absolute PeakInParams;
var i:Integer;
begin
 FileName:=UnifyFileAlias(FileName);
 for i:=Low(P) to High(P) do
 ReadIniFileDouble(FileName,Section,ExtractWord(1+i,PeakInFormat,[';'])+'%f',P[i]);
end;

 {
 *******************************************************************************
 TFormSpectrPikWindow implementation
 *******************************************************************************
 }
procedure TFormSpectrPikWindow.FormCreate(Sender: TObject);
begin
 inherited;
 Peak.Control:=nil;
 Peak.isViewPos:=true;
 Peak.isViewForm:=false;
 Peak.isViewGround:=false;
 Peak.isViewSingle:=false;
 Peak.NumPeaks:=0;
 Peak.NBeg:=0;
 Peak.Nend:=0;
 Peak.Spectr:=NewDoubleVector(0);
 Peak.C_Peak:=NewDoubleVector(0);
 Peak.C_Smooth:=NewDoubleVector(0);
 Peak.Channel:=NewDoubleVector(MaxNumPeaks);
 Peak.Energy:=NewDoubleVector(MaxNumPeaks);
 Peak.Amplitude:=NewDoubleVector(MaxNumPeaks);
 Peak.FWHM:=NewDoubleVector(MaxNumPeaks);
 Peak.Area:=NewDoubleVector(MaxNumPeaks);
 Peak.Ground:=NewDoubleVector(MaxNumPeaks);
 Peak.LeftBound:=NewDoubleVector(MaxNumPeaks);
 Peak.LeftGround:=NewDoubleVector(MaxNumPeaks);
 Peak.RightBound:=NewDoubleVector(MaxNumPeaks);
 Peak.RightGround:=NewDoubleVector(MaxNumPeaks);
 Peak.ChanError:=NewDoubleVector(MaxNumPeaks);
 Peak.AreaError:=NewDoubleVector(MaxNumPeaks);
 Peak.FwhmError:=NewDoubleVector(MaxNumPeaks);
 Peak.StatError:=NewDoubleVector(MaxNumPeaks);
 Peak.Derive:=NewDoubleVector(MaxNumPeaks);
 Peak.Gauss:=NewDoubleVector(MaxNumPeaks);
 UpdateMenu(MenuSpectrPeakSearchControl,
            RusEng('Поиск пиков','Peak search'),
            RusEng('Открыть диалог поска пиков.','Open peak search dialog.'),
            0);
end;

procedure TFormSpectrPikWindow.FormDestroy(Sender: TObject);
begin
 Kill(Peak.Control);
 Kill(Peak.Spectr);
 Kill(Peak.C_Peak);
 Kill(Peak.C_Smooth);
 Kill(Peak.Channel);
 Kill(Peak.Energy);
 Kill(Peak.Amplitude);
 Kill(Peak.FWHM);
 Kill(Peak.Area);
 Kill(Peak.Ground);
 Kill(Peak.LeftBound);
 Kill(Peak.LeftGround);
 Kill(Peak.RightBound);
 Kill(Peak.RightGround);
 Kill(Peak.ChanError);
 Kill(Peak.AreaError);
 Kill(Peak.FwhmError);
 Kill(Peak.StatError);
 Kill(Peak.Derive);
 Kill(Peak.Gauss);
 inherited;
end;

procedure TFormSpectrPikWindow.PeakSort;
 procedure SwapItems(v:TDoubleVector; i,j:Integer);
 var temp:Double;
 begin
  temp:=v[i];
  v[i]:=v[j];
  v[j]:=temp;
 end;
 procedure Sort(l,r:Integer);
 var i,j:Integer; c:Double;
 begin
  i:=l;
  j:=r;
  c:=Peak.Channel[(l+r) shr 1];
  repeat
    while (Peak.Channel[i]<c) do inc(i);
    while (c<Peak.Channel[j]) do dec(j);
    if (i<=j) then begin
     SwapItems(Peak.Channel,i,j);
     SwapItems(Peak.Energy,i,j);
     SwapItems(Peak.Amplitude,i,j);
     SwapItems(Peak.FWHM,i,j);
     SwapItems(Peak.Area,i,j);
     SwapItems(Peak.Ground,i,j);
     SwapItems(Peak.LeftBound,i,j);
     SwapItems(Peak.LeftGround,i,j);
     SwapItems(Peak.RightBound,i,j);
     SwapItems(Peak.RightGround,i,j);
     SwapItems(Peak.ChanError,i,j);
     SwapItems(Peak.AreaError,i,j);
     SwapItems(Peak.FwhmError,i,j);
     SwapItems(Peak.StatError,i,j);
     SwapItems(Peak.Derive,i,j);
     SwapItems(Peak.Gauss,i,j);
     inc(i);
     dec(j);
    end;
  until i>j;
  if (l<j) then Sort(l,j);
  if (i<r) then Sort(i,r);
 end;
begin
 if Ok then
 if (Peak.NumPeaks>1) then
 try
  Sort(0,Peak.NumPeaks-1);
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

procedure TFormSpectrPikWindow.PeakClear(Confirm:Boolean=false);
var i:Integer;
begin
 if Ok then
 if (Peak.NumPeaks>0) then
 if not Confirm or (YesNo(RusEng('Вы действительно хотите очистить список пиков?',
                                 'Do you want to clear peak list?'))=mrYes)
 then
 try
  LockDraw;
  try
   Peak.NumPeaks:=0;
   for i:=0 to Peak.Spectr.Length-1 do Peak.Spectr[i]:=0;
   for i:=0 to Peak.C_Peak.Length-1 do Peak.C_Peak[i]:=0;
   for i:=0 to Peak.C_Smooth.Length-1 do Peak.C_Smooth[i]:=0;
   for i:=0 to Peak.Channel.Length-1 do Peak.Channel[i]:=0;
   for i:=0 to Peak.Energy.Length-1 do Peak.Energy[i]:=0;
   for i:=0 to Peak.Amplitude.Length-1 do Peak.Amplitude[i]:=0;
   for i:=0 to Peak.FWHM.Length-1 do Peak.FWHM[i]:=0;
   for i:=0 to Peak.Area.Length-1 do Peak.Area[i]:=0;
   for i:=0 to Peak.Ground.Length-1 do Peak.Ground[i]:=0;
   for i:=0 to Peak.LeftBound.Length-1 do Peak.LeftBound[i]:=0;
   for i:=0 to Peak.LeftGround.Length-1 do Peak.LeftGround[i]:=0;
   for i:=0 to Peak.RightBound.Length-1 do Peak.RightBound[i]:=0;
   for i:=0 to Peak.RightGround.Length-1 do Peak.RightGround[i]:=0;
   for i:=0 to Peak.ChanError.Length-1 do Peak.ChanError[i]:=0;
   for i:=0 to Peak.AreaError.Length-1 do Peak.AreaError[i]:=0;
   for i:=0 to Peak.FwhmError.Length-1 do Peak.FwhmError[i]:=0;
   for i:=0 to Peak.StatError.Length-1 do Peak.StatError[i]:=0;
   for i:=0 to Peak.Derive.Length-1 do Peak.Derive[i]:=0;
   for i:=0 to Peak.Gauss.Length-1 do Peak.Gauss[i]:=0;
  except
   on E:Exception do Daq.Report(E.Message);
  end;
 finally
  UnlockDraw;
  PeakControlUpdate(1+2);
 end;
end;

function TFormSpectrPikWindow.PeakFindNearest(aChannel:Double):Integer;
var i:Integer; mx,ch:Double;
begin
 Result:=0;
 if Ok then
 try
  mx:=_plusinf;
  for i:=0 to Peak.NumPeaks-1 do begin
   ch:=abs(aChannel-Peak.Channel[i]);
   if (mx>ch) then begin
    mx:=ch;
    Result:=i;
   end;
  end;
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

function TFormSpectrPikWindow.PeakSave(FName:LongString):Boolean;
var i:Integer; f:System.Text;
begin
 Result:=false;
 if Ok then
 if (Peak.NumPeaks>0) then begin
  FName:=UnifyFileAlias(FName);
  System.Assign(f,FName);
  try
   System.Rewrite(f);
   {заголовок}
   System.Write(f,'Channel'     :14,ASCII_HT);
   System.Write(f,'Energy'      :14,ASCII_HT);
   System.Write(f,'Amplitude'   :14,ASCII_HT);
   System.Write(f,'FWHM'        :14,ASCII_HT);
   System.Write(f,'Area'        :14,ASCII_HT);
   System.Write(f,'Ground'      :14,ASCII_HT);
   System.Write(f,'LeftBound'   :14,ASCII_HT);
   System.Write(f,'LeftGround'  :14,ASCII_HT);
   System.Write(f,'RightBound'  :14,ASCII_HT);
   System.Write(f,'RightGround' :14,ASCII_HT);
   System.Write(f,'ChanError'   :14,ASCII_HT);
   System.Write(f,'AreaError'   :14,ASCII_HT);
   System.Write(f,'FwhmError'   :14,ASCII_HT);
   System.Write(f,'StatError'   :14,ASCII_HT);
   System.Writeln(f);
   {таблица}
   for i:=0 to Peak.NumPeaks-1 do begin
    System.Write(f,Peak.Channel[i]      :14:3,ASCII_HT);
    System.Write(f,Peak.Energy[i]       :14:3,ASCII_HT);
    System.Write(f,Peak.Amplitude[i]    :14:3,ASCII_HT);
    System.Write(f,Peak.FWHM[i]         :14:3,ASCII_HT);
    System.Write(f,Peak.Area[i]         :14:3,ASCII_HT);
    System.Write(f,Peak.Ground[i]       :14:3,ASCII_HT);
    System.Write(f,Peak.LeftBound[i]    :14:3,ASCII_HT);
    System.Write(f,Peak.LeftGround[i]   :14:3,ASCII_HT);
    System.Write(f,Peak.RightBound[i]   :14:3,ASCII_HT);
    System.Write(f,Peak.RightGround[i]  :14:3,ASCII_HT);
    System.Write(f,Peak.ChanError[i]    :14:3,ASCII_HT);
    System.Write(f,Peak.AreaError[i]    :14:3,ASCII_HT);
    System.Write(f,Peak.FwhmError[i]    :14:3,ASCII_HT);
    System.Write(f,Peak.StatError[i]    :14:3,ASCII_HT);
    System.Writeln(f);
   end;
   Result:=(IOResult=0);
  except
   on E:Exception do Daq.Report(E.Message);
  end;
  SmartFileClose(f);
 end;
end;

function TFormSpectrPikWindow.PeakLoad(FName:LongString):Boolean;
var f:System.Text; s,h:LongString;
 function ReadItem(const aName:LongString; aVector:TDoubleVector; Strict:Boolean=true):Boolean;
 var p:Integer; v:Double;
 begin
  Result:=false;
  p:=WordIndex(UnifyAlias(aName),UnifyAlias(h),ScanSpaces);
  if (Peak.NumPeaks<=MaxNumPeaks) then begin
   if (p>0) and Str2Real(ExtractWord(p,s,ScanSpaces),v) then begin
    aVector[Peak.NumPeaks]:=v;
    Result:=true;
   end else begin
    if not Strict then begin
     aVector[Peak.NumPeaks]:=0;
     Result:=true;
    end;
   end;
  end;
 end;
begin
 Result:=false;
 if Ok then
 try
  LockDraw;
  PeakClear;
  FName:=UnifyAlias(FName);
  System.Assign(f,FName);
  try
   System.Reset(f);
   System.Readln(f,h);
   {
   чтение таблицы:если есть поле в заголовке (позиция>0)
   то берем слово из этой позиции и пытаемся его прочитать...
   }
   while not eof(f) and (Peak.NumPeaks<MaxNumPeaks) do begin
    System.Readln(f,s);
    if (IOResult<>0) then begin
     PeakClear;
     break;
    end;
    if ReadItem('Channel',Peak.Channel) and
       ReadItem('Energy',Peak.Energy) and
       ReadItem('Amplitude',Peak.Amplitude) and
       ReadItem('FWHM',Peak.FWHM) and
       ReadItem('Area',Peak.Area) and
       ReadItem('Ground',Peak.Ground) and
       ReadItem('LeftBound',Peak.LeftBound) and
       ReadItem('LeftGround',Peak.LeftGround) and
       ReadItem('RightBound',Peak.RightBound) and
       ReadItem('RightGround',Peak.RightGround) and
       ReadItem('ChanError',Peak.ChanError,false) and
       ReadItem('AreaError',Peak.AreaError,false) and
       ReadItem('FwhmError',Peak.FwhmError,false) and
       ReadItem('StatError',Peak.StatError,false)
    then inc(Peak.NumPeaks)
    else begin
     PeakClear;
     break;
    end;
   end;
   Result:=(IOResult=0) and (Peak.NumPeaks>0);
  except
   on E:Exception do Daq.Report(E.Message);
  end;
  SmartFileClose(f);
 finally
  UnlockDraw;
  PeakControlUpdate(1+2);
 end;
end;


procedure TFormSpectrPikWindow.PeakValidateEnergy;
var i:Integer;
begin
 if Ok then
 try
  for i:=0 to Peak.NumPeaks-1 do Peak.Energy[i]:=EnCalibr(Peak.Channel[i]);
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

procedure TFormSpectrPikWindow.PeakValidateArea;
var i:Integer; GaussFactor:Double;
begin
 if Ok then
 try
  GaussFactor:=sqrt(pi/alfa);
  // for i:=0 to Peak.NumPeaks-1 do Peak.Area[i]:=Peak.Amplitude[i]*Peak.FWHM[i]*GaussFactor;
  for i:=0 to Peak.NumPeaks-1 do Peak.Area[i]:=Peak.Amplitude[i]*HwCalibr(Peak.Channel[i])*GaussFactor;
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

function TFormSpectrPikWindow.PeakSumOfGaussians(Chan:Double; ib,ie:Integer):Double;
var i:Integer;
begin
 Result:=0;
 if Ok then
 try
  if (ib<0) then ib:=0;
  if (ie>=Peak.NumPeaks) then ie:=Peak.NumPeaks-1;
  for i:=ib to ie do
  if (abs(Chan-Peak.Channel[i])<=Peak.FWHM[i]*10) then
  Result:=Gaussian(Peak.Amplitude[i],Chan,Peak.Channel[i],Peak.FWHM[i],Result);
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

procedure TFormSpectrPikWindow.PeakMove(dir:Integer);
var i,ch,mr:Integer;
begin
 if Ok then
 if (Peak.NumPeaks>0) then
 try
  mr:=Marker;
  if (dir>0) then
  for i:=0 to Peak.NumPeaks-1 do begin
   ch:=Round(Peak.Channel[i]);
   if (ch>mr) then begin
    mr:=ch;
    break;
   end;
  end
  else
  for i:=Peak.NumPeaks-1 downto 0 do begin
   ch:=Round(Peak.Channel[i]);
   if (ch<mr) then begin
    mr:=ch;
    break;
   end;
  end;
  try
   if Peak.isViewSingle then LockDraw;
   Marker:=mr;
  finally
   if Peak.isViewSingle then UnLockDraw;
  end;
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

procedure TFormSpectrPikWindow.PeakNext;
begin
 PeakMove(1);
end;

procedure TFormSpectrPikWindow.PeakPrev;
begin
 PeakMove(-1);
end;

function EvsChan(c:Double; Custom:Pointer):Double;
begin
 EvsChan:=TFormSpectrWindow(Custom).EnCalibr(c);
end;

function FWHMvsChan(c:Double; Custom:Pointer):Double;
begin
 FWHMvsChan:=TFormSpectrWindow(Custom).HwCalibr(c);
end;

procedure TFormSpectrPikWindow.PeakSearch;
var i,Code:Integer; Summ,Vmax:Double;
begin
 if Ok then
 try
  LockDraw;
  PeakClear;
  try
   Peak.NBeg:=MarkerL;
   Peak.NEnd:=MarkerR;
   if (Peak.NBeg>=0) and (Peak.NEnd<=SpectrSize) and (Peak.NEnd-Peak.NBeg>9) then begin
    Peak.Spectr.Length:=SpectrSize;
    Peak.C_Peak.Length:=SpectrSize;
    Peak.C_Smooth.Length:=SpectrSize;
    Summ:=0;
    Vmax:=0;
    for i:=0 to SpectrSize-1 do begin
     Peak.Spectr[i]:=SpectrValue[i];
     Summ:=Summ+SpectrValue[i];
     Vmax:=max(Vmax,SpectrValue[i]);
    end;
    if (Summ>100) and (Vmax>10) then begin
     Code:=PeakIn(Peak.Spectr.Length, {размер массивов спектра}
                  MaxNumPeaks,        {макс.размер массивов пиков}
                  Peak.Spectr,        {0..MaxCount-1,массив спектра}
                  Peak.NBeg,          {0..MaxCount-1,начальный канал спектра}
                  Peak.Nend,          {0..MaxCount-1,конечный канал спектра}
                  EvsChan,            {калибровка каналов по энергии}
                  FWHMvsChan,         {калибровка каналов по полуширине}
                  Self,               {для передачи в EvsChan,FWHMvsChan}
                  Peak.NumPeaks,      {число обнаруженных пиков}
                  Peak.Channel,       {0..MaxPeak-1,положение пика}
                  Peak.StatError,     {0..MaxPeak-1,стат. пика}
                  Peak.FWHM,          {0..MaxPeak-1,ПШПВ пика}
                  Peak.Amplitude,     {0..MaxPeak-1,амплитуда пика}
                  Peak.Ground,        {0..MaxPeak-1,фон под пиком}
                  Peak.LeftGround,    {0..MaxPeak-1,фон слева от пика}
                  Peak.RightGround,   {0..MaxPeak-1,фон справа от пика}
                  Peak.LeftBound,     {0..MaxPeak-1,признак границы слева}
                  Peak.RightBound,    {0..MaxPeak-1,признак границы справа}
                  Peak.C_Peak,        {0..MaxCount-1,массив рассчетного спектра}
                  Peak.Derive,        {0..MaxPeak-1,массив производных}
                  Peak.Gauss,         {0..MaxPeak-1,временный массив}
                  Peak.C_Smooth);     {0..MaxCount-1}
     if (Code=seOk) then begin
      PeakValidateEnergy; {энергия находится из канала при помощи калибровки}
      PeakValidateArea;   {площадь под пиком вычисляем по амплитуде и полуширине}
      PeakSort;           {сортировка пиков по росту номера канала}
     end else begin
      Daq.ConsoleEcho(RusEng('Ошибка поиска пиков ','Peak search error ')+d2s(Code)+EOL+PeakInErrorMsg);
      Daq.OpenConsole;
      PeakClear;
     end;
     PeakControlOpen;
    end else begin
     Daq.ConsoleEcho(RusEng('Мала статистика для поиска пиков!','Not enough statistics to search peaks.'));
     Daq.OpenConsole;
    end;
   end else begin
    Daq.ConsoleEcho(RusEng('Недопустимый интервал РОИ!','Invalid ROI interval!'));
    Daq.OpenConsole;
   end;
  except
   on E:Exception do Daq.Report(E.Message);
  end;
 finally
  UnLockDraw;
 end;
end;

 {
 найти диапазон отображаемых пиков
 это либо все пики 0..NumPeaks-1, либо один пик (ib=ie)
 }
procedure TFormSpectrPikWindow.PeakGetVisibleRange(out ib,ie:Integer);
begin
 if Ok then begin
  ib:=0;
  ie:=Peak.NumPeaks-1;
  if Peak.isViewSingle then begin
   ib:=PeakFindNearest(Marker);
   ie:=ib;
  end;
 end else begin
  ib:=0; ie:=0;
 end; 
end;

 {
 Процедура рисует в окне общего обзора маркеры положения пиков
 }
procedure TFormSpectrPikWindow.DrawFul;
var i,ib,ie:Integer;
begin
 if Ok and (Peak.NumPeaks>0) and Peak.isViewPos then
 try 
  PeakGetVisibleRange(ib,ie);
  for i:=ib to ie do
  FullView.DrawPeakMarker(Peak.Channel[i],Peak.Amplitude[i]+Peak.Ground[i],clSpectrPeakMarker);
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

 {
 Процедура рисования пиков с полиномиальным фоном Ground.
 Рисуется сумма гауссианов в интервале индексов ib,ie на интервале каналов
 chb,che так, чтобы шаг вычисления функции по горизонтали был около 1 пикселя.
 }
procedure TFormSpectrPikWindow.DrawPeaks(ib,ie:Integer; chb,che:Double; const Ground:TPolynom;
                                         cPeakGround,cPeakForm:TColor; isGround,isForm:Boolean);
var i,n:Integer; a1,a2,g1,g2:TPoint2I; x0,x,dx,y,h,fon:Double;
begin
 if Ok then
 try
  {коррекция интервала так, чтобы учесть только видимую часть}
  chb:=max(chb,BegX);
  che:=min(che,SpectrSize-1);
  {пики видимы?}
  if (chb<che) then begin
   {находим ширину картинки в пикселях}
   n:=ExpandView.ConvertX(che-BegX)-ExpandView.ConvertX(chb-BegX)+1;
   h:=(che-chb)/n;
   x0:=chb;
   g1:=Point2I(0,0);
   a1:=Point2I(0,0);
   for i:=0 to n do begin
    dx:=i*h;
    x:=x0+dx;
    fon:=Ground.Get(x);
    y:=PeakSumOfGaussians(x,ib,ie)+fon;
    a2.x:=ExpandView.ConvertX(x-BegX);
    a2.y:=ExpandView.ConvertY(y,isLog);
    g2.x:=a2.x;
    g2.y:=ExpandView.ConvertY(fon,isLog);
    if (i>0) then begin
     if isGround then DrawLine(PaintBoxExpand.Canvas,g1,g2,cPeakGround);
     if isForm   then DrawLine(PaintBoxExpand.Canvas,a1,a2,cPeakForm);
    end;
    a1:=a2;
    g1:=g2;
   end;
  end;
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

 {
 Процедура рисует в окне подробного обзора список пиков
 }
procedure TFormSpectrPikWindow.DrawExp;
var i,ib,ie:Integer; Poly:TPolynom; Ch,FW,Fon:Double;
var b,g:array[0..1] of Double;
begin
 if (SpectrSize>0) and (Peak.NumPeaks>0) then
 try
  {нарисовать положения пиков}
  if Peak.isViewPos then begin
   PeakGetVisibleRange(ib,ie);
   for i:=ib to ie do
   ExpandView.DrawPeakMarker(Peak.Channel[i], Peak.Amplitude[i]+Peak.Ground[i], clSpectrPeakMarker);
  end;
  if Peak.isViewForm or Peak.isViewGround then begin
   PeakGetVisibleRange(ib,ie);
   for i:=ib to ie do begin
    Ch:=Peak.Channel[i];
    FW:=Peak.FWHM[i];
    Fon:=Peak.Ground[i];
    B[0]:=Peak.LeftBound[i];
    G[0]:=Peak.LeftGround[i];
    B[1]:=Peak.RightBound[i];
    G[1]:=Peak.RightGround[i];
    {линейный фон допустим? инициализация полинома в зависимости от этого}
    if (B[0]<0) or (B[1]<=B[0]) or (G[0]<0) or (G[1]<0) then begin
     Poly:=NewPolynom(0,0,1);
     Poly.Find(Ch,Fon,1);
    end else begin
     Poly:=NewPolynom(1,0,1);
     Poly.Find(B,G,2);
    end;
    DrawPeaks(i,i,Ch-2*FW,Ch+2*FW,Poly,clSpectrPeakGnd,clSpectrPeakForm,Peak.isViewGround,Peak.isViewForm);
    Kill(Poly);
   end;
  end;
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

procedure TFormSpectrPikWindow.MarkerChanged;
begin
 inherited MarkerChanged;
 if Peak.Control.Ok then
 if Peak.Control.Visible then PeakControlUpdate(2);
end;

procedure TFormSpectrPikWindow.SpectrClear;
begin
 try
  LockDraw;
  PeakClear;
  PeakControlUpdate(1+2);
  inherited SpectrClear;
 finally
  UnlockDraw;
 end;
end;

procedure TFormSpectrPikWindow.Animate;
begin
 inherited;
 LoadPeakInParams(Daq.ConfigFile,'[PeakSearchPreset]');
end;

procedure TFormSpectrPikWindow.PeakControlOpen;
begin
 if Ok then
 if (Peak.Control is TFormSpectrPikControl) then begin
  Peak.Control.Show;
  Peak.Control.WindowState:=wsNormal;
  Peak.Control.BringToFront;
  PeakControlUpdate(1+2+$10);
 end else begin
  Peak.Control:=NewFormSpectrPikControl(Self);
  Peak.Control.Master:=@Peak.Control;
  Peak.Control.Show;
  Peak.Control.BringToFront;
  PeakControlUpdate(1+2+$10);
 end;
end;

procedure TFormSpectrPikWindow.PeakControlUpdate(Flags:Integer);
begin
 if Ok then
 if (Peak.Control is TFormSpectrPikControl)
 then TFormSpectrPikControl(Peak.Control).UpdateControls(Flags);
end;

procedure TFormSpectrPikWindow.ActionSpectrPeakSearchControlExecute(Sender: TObject);
begin
 inherited;
 PeakControlOpen;
 PeakControlUpdate($20);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_spectrpikwindow;
begin
 RegisterSpectrWindowConstructor(NewSpecPikWin, regSpecPikWin);
end;

procedure Free_form_spectrpikwindow;
begin
end;

initialization

 Init_form_spectrpikwindow;

finalization

 Free_form_spectrpikwindow;

end.

//////////////
// END OF FILE
//////////////

