////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form RFA Mendeleev Table.                                                  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231130 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_rfamendeleevtable; // Form RFA Mendeleev Table

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, CheckLst,
 lcltype, lclintf,
 Form_ListBoxSelection,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_sort,
 _crw_ef, _crw_ee, _crw_zm, _crw_curves,
 _crw_calib, _crw_lsqpoly, _crw_simplex,
 _crw_rfadata, _crw_stopion, _crw_sesman,
 _crw_daqsys, _crw_daqdev,
 _crw_appforms, _crw_apptools, _crw_apputils;

  {
  TRfaParams описывает набор общих параметров, необходимых для РФА.
  }
type
 TRfaParams = packed record
  SelElem   : Byte;              { Текущий активный элемент в диалоге     }
  SelElems  : TByteSet;          { Множество выбранных элементов          }
  SelLines  : TRfaFieldsSet;     { Множество выбранных линий              }
  SelEnable : Boolean;           { Разрешен ли выбор элементов            }
  EnMin     : Double;            { Минимальная энергия                    }
  EnMax     : Double;            { Энергия возбуждения                    }
  Anod      : Byte;              { Тип анода                              }
  MulFWHM   : Double;            { Множитель для полуширин пиков          }
  Detector  : packed record      { Параметры детектора RFA                }
   Allow         : Boolean;      { Нужен учет эффективности регистрации ? }
   Element       : Byte;         { Элемент детектора, обычно Si           }
   WorkThickness : Double;       { Толщина Si кристалла детектора в см    }
   DeadThickness : Double;       { Толщина мертвой зоны Si детектора в см }
   AirDensity    : Double;       { Плотность воздуха                      }
   Layers        : packed array[1..4] of packed record {Слои на пути луча }
    Element      : Byte;         { Из какого материала слой               }
    Thickness    : Double;       { Толщина слоя в сантиметрах             }
   end;
  end;
  ReperAtom      : Byte;         { Опорный элемент,кол-во которого задано }
  ReperConc      : Double;       { Концентрация опорного элемента         }
  ReperUnit      : PureString;   { Единицы измерения концентрации         }
 end;

const
 AtomId_None = 0;
 AtomId_Air  = RfaMaxAtomID+1;
 DefaultRfaParams : TRfaParams = (
  SelElem         : 1;
  SelElems        : [1..95];
  SelLines        : [rf_EKA,rf_EKB,rf_ELA12,rf_ELL,rf_ELB1,rf_ELB2,rf_ELG1];
  SelEnable       : true;
  EnMin           : 1;
  EnMax           : 20;
  Anod            : 3;
  MulFWHM         : 1;
  Detector        : (
   Allow          : true;
   Element        : AtomId_Si;
   WorkThickness  : 0.2;
   DeadThickness  : 0.00001;
   AirDensity     : 0.0012;
   Layers         : ((Element : AtomId_Air;  Thickness   : 3),
                     (Element : AtomId_Be;   Thickness   : 0.0008),
                     (Element : AtomId_Au;   Thickness   : 0.000002),
                     (Element : AtomId_None; Thickness   : 0)
                    );
  );
 ReperAtom        : 0;
 ReperConc        : 1;
 ReperUnit        : '';
 );

 {
 TRfaLine описывает линию ХРИ при РФА анализе.
 }
type
 TRfaLine = class(TMasterObject)
 public
  AtomID    : Byte;                    { Порядковый номер атома                             }
  LineID    : TRfaFields;              { Идентификатор типа линии                           }
  Energy    : Double;                  { Энергия линии                                      }
  OI        : Double;                  { Относительная интенсивность из базы данных         }
  AI        : Double;                  { Относительная интенсивность с учетом эффективности }
  II        : Double;                  { Относительная интенсивность с учетом всех эффектов }
  Level     : Integer;                 { Уровень старшинства линии (по вероятности)         }
  Enabled   : Boolean;                 { Флаг разрешения - участвует ли линия в вычислениях }
 public
  constructor Create(aAtomID:Byte; aLineID:TRfaFields; const RfaParams:TRfaParams);
  destructor  Destroy; override;
  function    Intensity : Double;      { Интенсивность с учетом всех эффектов и флага Enable}
  function    Name      : LongString;  { Имя линии для отображения в списке                 }
  function    NameElem  : LongString;  { Символ элемента                                    }
 end;

function  NewRfaLine(aAtomID:Byte; aLineID:TRfaFields; const RfaParams:TRfaParams):TRfaLine;
procedure Kill(var TheObject:TRfaLine); overload;

 {
 TRfaLineList описывает список линий ХРИ, принадлежащих одному пику спектра.
 }
type
 TRfaLineList = class(TObjectStorage)
 private
  function  GetLine(index:Integer):TRfaLine;
  procedure SetLine(index:Integer; aLine:TRfaLine);
 public
  property  Lines[i:Integer]:TRfaLine read GetLine write SetLine; default;
  function  GetNumEnabledLines : Integer;
  function  GetText:LongString;
  procedure SearchForPeak(PeakEnergy,PeakFWHM:Double; const RfaParams:TRfaParams);
 end;

function  NewRfaLineList:TRfaLineList;
procedure Kill(var TheObject:TRfaLineList); overload;

 {
 TRfaPeak описывает спектрометрический пик со списком линий ХРИ, которые ему
 принадлежат.
 }
type
 TRfaPeak = class(TMasterObject)
 public
  Ch    : Double;                     { Канал                }
  En    : Double;                     { Энергия,keV          }
  Sq    : Double;                     { Площадь под пиком    }
  Fw    : Double;                     { Ширина на полувысоте }
  Lines : TRfaLineList;               { Список линий ХРИ     }
  constructor Create(C,E,S,F:Double);
  destructor  Destroy; override;
 end;

function  NewRfaPeak(C,E,S,F:Double):TRfaPeak;
procedure Kill(var TheObject:TRfaPeak); overload;

 {
 TRfaPeakList описывает список пиков. Это, собственно, исходные данные для РФА.
 }
type
 TRfaPeakList = class(TObjectStorage)
 private
  function  GetPeak(Index:Integer):TRfaPeak;
  procedure SetPeak(Index:Integer; aPeak:TRfaPeak);
 public
  Details : TText;
  constructor Create;
  destructor  Destroy; override;
  property    Peaks[i:Integer] : TRfaPeak read GetPeak write SetPeak; default;
  procedure   ViewDetails;
  procedure   Read(FileName:LongString);
  procedure   FindAllEnabledLines(const RfaParams:TRfaParams);
  function    GetNumNonIsolPeaks:Integer;
  function    FindPeaksEnergyMax(const RfaParams:TRfaParams):Double;
  function    FindPeaksEnergyMin(const RfaParams:TRfaParams):Double;
  function    LinePresent(AtomID:Byte;LineID:TRfaFields):Boolean;
  procedure   ReCalcCountLines(var CountLine,CountLineEnable:Integer);
  function    LineCanBe(Line:TRfaLine; PeakEMin,PeakEMax:Double;
                  const RfaParams:TRfaParams; out Msg:LongString):Boolean;
  procedure   AnalisSopLine(const RfaParams:TRfaParams);
  procedure   ViewDublicates;
  procedure   FirstAnalis(const RfaParams:TRfaParams);
  procedure   RemoveBadLines;
  function    TotalArea:Double;
  procedure   ConvertFwhmToKev;
 end;

function  NewRfaPeakList:TRfaPeakList;
procedure Kill(var TheObject:TRfaPeakList); overload;

 {
 TRfaSimplex описывает данные и методы для создания и решения задачи линейного
 программирования, возникающей при анализе линий ХРИ.
 }
type
 TRfaSimplex = class(TMasterObject)
 public
  ReperAtom      : Byte;           { Реперный элемент                        }
  ReperConc      : Double;         { Концентрация реперного элемента         }
  ReperUnit      : LongString;     { Единицы измерения концентрации          }
  NumPeaks       : Integer;        { Число пиков в списке                    }
  PeakEn         : TDoubleVector;  { 1..NumPeaks Энергии пиков               }
  PeakSq         : TDoubleVector;  { 1..NumPeaks Площади пиков               }
  PeakFw         : TDoubleVector;  { 1..NumPeaks Шинины пиков                }
  ErrorCode      : Integer;        { Код ошибки,см. константы siXXXX         }
  NumRows        : Integer;        { Число ограничений в симплекс-задаче     }
  NumCells       : Integer;        { Число переменных в симплекс-задаче      }
  Problem        : TDoubleVector;  { Вектор(N) коэффициентов целевой функции }
  MatrCoef       : TDoubleMatrix;  { Матрица(M,N) ограничений на переменные  }
  Signums        : TLongIntVector; { Вектор(M) знаков :1->=, 0-=, -1-<=      }
  RightSide      : TDoubleVector;  { Вектор(M) правых частей ограничений     }
  Solution       : TDoubleMatrix;  { Матрица(?,N) из строк-решений задачи    }
  SolutionText   : TText;          { Текст - отчет о решении                 }
  CellOfElement  : TLongIntVector; { 1..RfaMaxAtomID; RowOfElement[AtomID] - номер столбца, соответствующий элементу AtomID или 0 }
  ElementInCell  : TLongIntVector; { 1..RfaMaxAtomID; ElementInCell[Cell]  - элемент, соответствующий столбцу Cell или 0          }
  RowOfPeak      : TLongIntVector; { RowOfPeak[N] - номер строки, соответствующий пику N или 0 }
  PeakInRow      : TLongIntVector; { PeakInRow[N] - пик, соответствующий строке N или 0        }
  EvalTime       : Double;         { Измеряет время вычислений               }
  AvailElems     : TByteSet;       { Элементы участвующие в анализе          }
  PeakFile       : LongString;     { Файл с описанием пиков                  }
  ResultFile     : LongString;     { Файл с текстом решения                  }
  DetailFile     : LongString;     { Файл с подробностями минимизации        }
  constructor Create;
  destructor  Destroy; override;
  procedure   Clear;
  procedure   ConstructLinearProgrammingTask(PeakList:TRfaPeakList; const RfaParams:TRfaParams);
  procedure   CheckSolutionCell(Cell:Integer; var JustElement,PresentElement:Boolean);
  procedure   GetReper(var Concentration,Multiplier:Double; var Error:Integer);
  procedure   ReadSolutionFile;
  procedure   SaveSolution(var OutPutFile:Text);
  procedure   SolveLinearProgrammingTask;
  procedure   AnalizeNow(PeakList:TRfaPeakList; const RfaParams:TRfaParams);
 end;

function  NewRfaSimplex:TRfaSimplex;
procedure Kill(var TheObject:TRfaSimplex); overload;

 {
 Многостраничная форма описывает диалог, через который можно наблюдать базу
 данных РФА и проводить анализ.
 }
type
  TFormRfaMendeleevTable = class(TMasterForm)
    PageControl: TPageControl;
    TabSheetMenTab: TTabSheet;
    TabSheetDetector: TTabSheet;
    TabSheetRfa: TTabSheet;
    TabSheetPeak: TTabSheet;
    TabSheetAnalize: TTabSheet;
    TabSheetResult: TTabSheet;
    PanelMendTab: TPanel;
    PaintBoxMendTab: TPaintBox;
    StatusBar: TStatusBar;
    RadioGroupCategory: TRadioGroup;
    MemoRfaTab: TMemo;
    CheckBoxSelEnable: TCheckBox;
    GroupBoxKSeria: TGroupBox;
    CheckBoxKa: TCheckBox;
    CheckBoxKa1: TCheckBox;
    CheckBoxKa2: TCheckBox;
    CheckBoxKb: TCheckBox;
    CheckBoxKb1: TCheckBox;
    CheckBoxKb2: TCheckBox;
    CheckBoxKb3: TCheckBox;
    GroupBoxLSeria: TGroupBox;
    CheckBoxLa1: TCheckBox;
    CheckBoxLb1: TCheckBox;
    CheckBoxLb2: TCheckBox;
    CheckBoxLg1: TCheckBox;
    CheckBoxLL: TCheckBox;
    LabelSelEnable: TLabel;
    GroupBoxEminEmax: TGroupBox;
    EditEmin: TEdit;
    LabelEmin: TLabel;
    LabelEmax: TLabel;
    EditEmax: TEdit;
    ComboBoxAnod: TComboBox;
    LabelAnod: TLabel;
    GroupBoxDetectorParams: TGroupBox;
    LabelDetectorElement: TLabel;
    ComboBoxDetectorElement: TComboBox;
    LabelDetectorWorkThickness: TLabel;
    EditDetectorWorkThickness: TEdit;
    LabelDetectorDeadThickness: TLabel;
    EditDetectorDeadThickness: TEdit;
    LabelLayer1Element: TLabel;
    ComboBoxLayer1Element: TComboBox;
    LabelLayer1Thickness: TLabel;
    EditLayer1Thickness: TEdit;
    LabelLayer2Element: TLabel;
    ComboBoxLayer2Element: TComboBox;
    LabelLayer2Thickness: TLabel;
    EditLayer2Thickness: TEdit;
    LabelLayer3Element: TLabel;
    ComboBoxLayer3Element: TComboBox;
    LabelLayer3Thickness: TLabel;
    EditLayer3Thickness: TEdit;
    LabelLayer4Element: TLabel;
    ComboBoxLayer4Element: TComboBox;
    LabelLayer4Thickness: TLabel;
    EditLayer4Thickness: TEdit;
    GroupBoxDetectorEfficiency: TGroupBox;
    LabelEfficiencyAt: TLabel;
    EditEfficiencyAt: TEdit;
    LabelEfficiencyKeV: TLabel;
    ButtonEfficiencyEqual: TButton;
    EditEfficiencyIs: TEdit;
    CheckBoxAllowDetectorEfficiency: TCheckBox;
    LabelMulFWHM1: TLabel;
    LabelMulFWHM2: TLabel;
    EditMulFWHM: TEdit;
    GroupBoxPeakEnHw: TGroupBox;
    LabelPeakEn: TLabel;
    EditPeakEn: TEdit;
    LabelPeakFwhm: TLabel;
    EditPeakFwhm: TEdit;
    ButtonPeakFindLines: TButton;
    GroupBoxPeakLines: TGroupBox;
    MemoPeakLines: TMemo;
    GroupBoxRfaPeaks: TGroupBox;
    GroupBoxRfaLines: TGroupBox;
    GroupBoxRfaButtons: TGroupBox;
    ListBoxRfaPeaks: TListBox;
    CheckListBoxRfaLines: TCheckListBox;
    ButtonLoadPeaks: TButton;
    ButtonSearchLines: TButton;
    OpenDialogPik: TOpenDialog;
    SaveDialogRep: TSaveDialog;
    ButtonViewDetails: TButton;
    GroupBoxRfaReport: TGroupBox;
    PanelResultButtons: TPanel;
    MemoRfaReport: TMemo;
    ButtonAnalizeNow: TButton;
    ButtonSaveReport: TButton;
    OpenDialogRep: TOpenDialog;
    ButtonLoadReport: TButton;
    LabelReperAtom: TLabel;
    ComboBoxReperAtom: TComboBox;
    LabelReperConc: TLabel;
    EditReperConc: TEdit;
    LabelReperUnit: TLabel;
    EditReperUnit: TEdit;
    ButtonInvLines: TButton;
    LabelDetectorAirDensity: TLabel;
    EditDetectorAirDensity: TEdit;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure RadioGroupCategoryClick(Sender: TObject);
    procedure CheckBoxSelEnableClick(Sender: TObject);
    procedure PaintBoxMendTabMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure PaintBoxMendTabMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure PaintBoxMendTabMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure CheckBoxLinesClick(Sender: TObject);
    procedure ComboBoxAnodChange(Sender: TObject);
    procedure ApplyChanges(Sender: TObject);
    procedure CancelChanges(Sender: TObject);
    procedure AnyOneKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure ButtonEfficiencyEqualClick(Sender: TObject);
    procedure ButtonPeakFindLinesClick(Sender: TObject);
    procedure ButtonLoadPeaksClick(Sender: TObject);
    procedure ListBoxRfaPeaksClick(Sender: TObject);
    procedure ButtonSearchLinesClick(Sender: TObject);
    procedure ButtonViewDetailsClick(Sender: TObject);
    procedure CheckListBoxRfaLinesClickCheck(Sender: TObject);
    procedure ButtonAnalizeNowClick(Sender: TObject);
    procedure ButtonSaveReportClick(Sender: TObject);
    procedure ButtonLoadReportClick(Sender: TObject);
    procedure ButtonInvLinesClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    PeakList        : TRfaPeakList;
    RfaParams       : TRfaParams;
    RfaSimplex      : TRfaSimplex;
    ElemSelector    : TXorSelector;
    ButtonOfElement : packed array[1..RfaMaxAtomID] of TSpeedButton;
    procedure ButtonOfElementClick(Sender: TObject);
    procedure UpdateControls(Flags:Integer);
    function  LoadPeaksFrom(FileName:LongString):Boolean;
    procedure FindAvailableLines;
    procedure AnalizeNow;
    procedure LoadRfaParams;
    procedure SaveRfaParams;
    procedure AnalizeFile(PikFile:LongString);
  end;

function FormRfaMendeleevTableExecute(PikFile:LongString=''):Integer;


implementation

{$R *.lfm}

procedure DaqReport(msg:LongString);
begin
 Daq.Report(msg);
end;

 {
 *******************************************************************************
 Внутренние вспомогательные процедуры.
 *******************************************************************************
 }
const // Флаги UpdateContols
 idUpdate   = $00000001;
 idChange   = $00000002;
 idPeaks    = $00000010;
 idLines    = $00000020;
 idReport   = $00000040;
 idCheckLn  = $00000080;
 idInvLines = $00000100;

const // Идентификаторы используемых линий РФА
 RfaAvailLines  = [rf_EKA,  rf_EKB,  rf_EKA1, rf_EKA2,
                   rf_EKB1, rf_EKB2, rf_EKB3, rf_ELA12,
                   rf_ELB1, rf_ELB2, rf_ELG1, rf_ELL];

const // Список линий РФА по порядку убывания вероятности в виде одной строки.
 RfaAvailLinesList = chr(ord(rf_EKA))+chr(ord(rf_EKB))+chr(ord(rf_EKA1))+chr(ord(rf_EKA2))+
                     chr(ord(rf_EKB1))+chr(ord(rf_EKB2))+chr(ord(rf_EKB3))+chr(ord(rf_ELA12))+
                     chr(ord(rf_ELB1))+chr(ord(rf_ELB2))+chr(ord(rf_ELG1))+chr(ord(rf_ELL));
{RfaMaxLines       = Length(RfaAvailLinesList);}
 RfaFullLinesList  : PureString = RfaAvailLinesList;

procedure SaveRfaParameters(const RfaParams:TRfaParams; const FileName:LongString; isAppend:Boolean);
var
 i : Integer;
 p : TText;
 f : TRfaFields;
begin
 p:=NewText;
 try
  p.Addln('[RFA Parameters]');
  p.Addln(Format('SelElem = %d',[RfaParams.SelElem]));
  for i:=1 to RfaMaxAtomID do p.Addln(Format('Elem_%-2s = %d',[Rfa.AtomSymbol[i],ord(i in RfaParams.SelElems)]));
  for f:=Low(f) to High(f) do if f in RfaAvailLines then p.Addln(Format('Line_%-3s = %d',[Rfa.LineName[f],ord(f in RfaParams.SelLines)]));
  p.Addln(Format('SelEnable = %d', [ord(RfaParams.SelEnable)]));
  p.Addln(Format('EnMin = %g', [RfaParams.EnMin]));
  p.Addln(Format('EnMax = %g', [RfaParams.EnMax]));
  p.Addln(Format('Anod = %d', [RfaParams.Anod]));
  p.Addln(Format('MulFWHM = %g', [RfaParams.MulFWHM]));
  p.Addln(Format('Detector.Allow = %d', [ord(RfaParams.Detector.Allow)]));
  p.Addln(Format('Detector.Element = %d', [RfaParams.Detector.Element]));
  p.Addln(Format('Detector.WorkThickness = %g',   [RfaParams.Detector.WorkThickness]));
  p.Addln(Format('Detector.DeadThickness = %g',   [RfaParams.Detector.DeadThickness]));
  p.Addln(Format('Detector.AirDensity = %g',      [RfaParams.Detector.AirDensity]));
  p.Addln(Format('Detector.Layer1Element = %d',   [RfaParams.Detector.Layers[1].Element]));
  p.Addln(Format('Detector.Layer1Thickness = %g', [RfaParams.Detector.Layers[1].Thickness]));
  p.Addln(Format('Detector.Layer2Element = %d',   [RfaParams.Detector.Layers[2].Element]));
  p.Addln(Format('Detector.Layer2Thickness = %g', [RfaParams.Detector.Layers[2].Thickness]));
  p.Addln(Format('Detector.Layer3Element = %d',   [RfaParams.Detector.Layers[3].Element]));
  p.Addln(Format('Detector.Layer3Thickness = %g', [RfaParams.Detector.Layers[3].Thickness]));
  p.Addln(Format('Detector.Layer4Element = %d',   [RfaParams.Detector.Layers[4].Element]));
  p.Addln(Format('Detector.Layer4Thickness = %g', [RfaParams.Detector.Layers[4].Thickness]));
  p.Addln(Format('ReperAtom = %d',[RfaParams.ReperAtom]));
  p.Addln(Format('ReperConc = %g',[RfaParams.ReperConc]));
  p.Addln(Format('ReperUnit = %s',[RfaParams.ReperUnit]));
  if p.WriteFile(FileName,isAppend)<>0 then DaqReport('Error writing file '+FileName);
 except
  on E:Exception do DaqReport(E.Message);
 end;
 Kill(p);
end;

procedure LoadRfaParameters(var RfaParams:TRfaParams; FileName:LongString);
var i:Integer; d:LongInt; f:Double; r:TRfaFields; s,Sec:LongString;
begin
 try
  FileName:=UnifyFileAlias(FileName);
  sec:='[RFA Parameters]'; d:=0; f:=0; s:='';
  if ReadIniFileLongInt(FileName,Sec,'SelElem%d',d) then RfaParams.SelElem:=d;
  for i:=1 to RfaMaxAtomID do
  if ReadIniFileLongInt(FileName,Sec,'Elem_'+Trim(Rfa.AtomSymbol[i])+'%d',d) then
  if d=1 then include(RfaParams.SelElems,i) else exclude(RfaParams.SelElems,i);
  for r:=low(r) to High(r) do
  if r in RfaAvailLines then
  if ReadIniFileLongInt(FileName,Sec,'Line_'+Trim(Rfa.LineName[r])+'%d',d) then
  if d=1 then include(RfaParams.SelLines,r) else exclude(RfaParams.SelLines,r);
  if ReadIniFileLongInt(FileName,Sec,'SelEnable%d',d) then RfaParams.SelEnable:=(d=1);
  if ReadIniFileDouble(FileName,Sec,'EnMin%f',f) then RfaParams.EnMin:=f;
  if ReadIniFileDouble(FileName,Sec,'EnMax%f',f) then RfaParams.EnMax:=f;
  if ReadIniFileLongInt(FileName,Sec,'Anod%d',d) then RfaParams.Anod:=d;
  if ReadIniFileDouble(FileName,Sec,'MulFWHM%f',f) then RfaParams.MulFWHM:=f;
  if ReadIniFileLongInt(FileName,Sec,'Detector.Allow%d',d)          then RfaParams.Detector.Allow:=(d=1);
  if ReadIniFileLongInt(FileName,Sec,'Detector.Element%d',d)        then RfaParams.Detector.Element:=d;
  if ReadIniFileDouble(FileName,Sec,'Detector.WorkThickness%f',f)   then RfaParams.Detector.WorkThickness:=f;
  if ReadIniFileDouble(FileName,Sec,'Detector.DeadThickness%f',f)   then RfaParams.Detector.DeadThickness:=f;
  if ReadIniFileDouble(FileName,Sec,'Detector.AirDensity%f',f)      then RfaParams.Detector.AirDensity:=f;
  if ReadIniFileLongInt(FileName,Sec,'Detector.Layer1Element%d',d)  then RfaParams.Detector.Layers[1].Element:=d;
  if ReadIniFileDouble(FileName,Sec,'Detector.Layer1Thickness%f',f) then RfaParams.Detector.Layers[1].Thickness:=f;
  if ReadIniFileLongInt(FileName,Sec,'Detector.Layer2Element%d',d)  then RfaParams.Detector.Layers[2].Element:=d;
  if ReadIniFileDouble(FileName,Sec,'Detector.Layer2Thickness%f',f) then RfaParams.Detector.Layers[2].Thickness:=f;
  if ReadIniFileLongInt(FileName,Sec,'Detector.Layer3Element%d',d)  then RfaParams.Detector.Layers[3].Element:=d;
  if ReadIniFileDouble(FileName,Sec,'Detector.Layer3Thickness%f',f) then RfaParams.Detector.Layers[3].Thickness:=f;
  if ReadIniFileLongInt(FileName,Sec,'Detector.Layer4Element%d',d)  then RfaParams.Detector.Layers[4].Element:=d;
  if ReadIniFileDouble(FileName,Sec,'Detector.Layer4Thickness%f',f) then RfaParams.Detector.Layers[4].Thickness:=f;
  if ReadIniFileLongInt(FileName,Sec,'ReperAtom%d',d) then RfaParams.ReperAtom:=d;
  if ReadIniFileDouble(FileName,Sec,'ReperConc%f',f) then RfaParams.ReperConc:=f;
  if ReadIniFileAlpha(FileName,Sec,'ReperUnit%a',s) then RfaParams.ReperUnit:=s;
 except
  on E:Exception do DaqReport(E.Message);
 end;
end;

function RfaFindDetectorEfficiency(Energy:Double; const RfaParams:TRfaParams):Double;
var
 i : Integer;
 { найти затухание слоя из элемента AtomID толщины Thickness }
 function LayerFactor(AtomID:Byte; Thickness:Double):Double;
 var
  Mu      : Double;
  Lambda  : Double;
  Density : Double;
 begin
  Result:=1;
  case AtomID of
   1..RfaMaxAtomID :
    if Rfa.Accessible[AtomID] then begin
     Mu:=Rfa.FindCrossSection(AtomID,rf_TOTAL,Energy);
     Density:=Rfa[AtomID,rf_ATOM_DENSITY];
     Lambda:= - Mu * Density * Thickness;
     if Lambda<-300 then Result:=0 else Result:=exp(Lambda);
    end;
   AtomId_Air:
    if Rfa.Accessible[AtomId_N] and Rfa.Accessible[AtomId_O] then begin
     Mu:=0.77*Rfa.FindCrossSection(AtomId_N,rf_TOTAL,Energy)+
         0.23*Rfa.FindCrossSection(AtomId_O,rf_TOTAL,Energy);
     Density:=RfaParams.Detector.AirDensity;
     Lambda:= - Mu * Density * Thickness;
     if Lambda<-300 then Result:=0 else Result:=exp(Lambda);
    end;
  end;
 end;
begin
 Result:=1;
 {если не надо учитывать эффективность,возвратим 1}
 with RfaParams.Detector do if Allow then begin
  {учтем затухание на каждом слое}
  for i:=Low(Layers) to High(Layers) do
  Result:=Result*LayerFactor(Layers[i].Element,Layers[i].Thickness);
  {учтем затухание и мертвую зону детектора}
  Result:=Result*(1-LayerFactor(Element,WorkThickness));
  Result:=Result*LayerFactor(Element,DeadThickness);
 end;
end;

 {
 CheckKaKb устраняет противоречия линиий Ka,Ka1,Ka2; Kb,Kb1,Kb2,Kb3.
 Возможно либо рассмотрение дуплета Ka1,Ka2 как набора линий, либо как одной
 линии Ka, поэтому  в системе TLinesEnable не должно линий Ka1,Ka2, если есть
 линия Ka. Процедура удаляет линии Ka1,Ka2 при наличии линии Ka. То же касается
 Kb и триплета Kb1,Kb2,Kb3.
 }
procedure CheckKaKb(var LinesEnabled:TRfaFieldsSet);
begin
 if rf_EKA in LinesEnabled then LinesEnabled:=LinesEnabled-[rf_EKA1,rf_EKA2];
 if rf_EKB in LinesEnabled then LinesEnabled:=LinesEnabled-[rf_EKB1,rf_EKB2,rf_EKB3];
end;

 {
 Функция возвращает строку длины RfaMaxLines, которая содержит порядок
 старшинства линий, которые присутствуют в списке LinesEnabled.
 Линии ранжированы по порядку убывания вероятности.
 Конец строки заполнен нулями.
 Уровень Level старшинства линии LineId находится как
  Level:=pos(chr(LineId),GetLinesOrder(LinesEnable))
 }
function RfaGetLinesOrder(LinesEnabled:TRfaFieldsSet):LongString;
var i:Integer;
begin
 Result:='';
 for i:=1 to Length(RfaFullLinesList) do
 if TRfaFields(RfaFullLinesList[i]) in LinesEnabled
 then Result:=Result+RfaFullLinesList[i];
 while Length(Result)<Length(RfaFullLinesList) do Result:=Result+#0;
end;

function RfaGetList(AtomID:Byte):LongString;
var p:TText; i:TRfaFields;
begin
 p:=NewText;
 if Rfa.Accessible[AtomID] then
 for i:=rf_FIRST_FIELD to rf_LAST_FIELD do
 if Length(Rfa.FieldName[i])>0
 then p.Addln(Format('%-14s = %s',[Rfa.FieldName[i],Rfa.AsString[AtomID,i]]));
 Result:=p.Text;
 Kill(p);
end;

function RfaGetCrossList(AtomID:Byte):LongString;
var p:TText; i:Integer;
begin
 p:=NewText;
 if Rfa.Accessible[AtomID] then
 if Rfa.CrossSectionTableCount[AtomId]>0 then begin
  p.Addln(Format('%-11s %-11s %-11s %-11s %-11s',['Energy','Photo','Coher','Incoher','Total']));
  for i:=0 to Rfa.CrossSectionTableCount[AtomId]-1 do
  p.Addln(Format('%-11.7g %-11.7g %-11.7g %-11.7g %-11.7g',
                 [Rfa.CrossSectionTable[AtomId,rf_ENERGY,i],
                  Rfa.CrossSectionTable[AtomId,rf_PHOTO,i],
                  Rfa.CrossSectionTable[AtomId,rf_COHER,i],
                  Rfa.CrossSectionTable[AtomId,rf_INCOHER,i],
                  Rfa.CrossSectionTable[AtomId,rf_TOTAL,i]]));
 end;
 Result:=p.Text;
 Kill(p);
end;

 {
 *******************************************************************************
 TRfaLine implementation
 *******************************************************************************
 }
constructor TRfaLine.Create(aAtomID:Byte; aLineID:TRfaFields; const RfaParams:TRfaParams);
begin
 inherited Create;
 if Rfa.Accessible[aAtomID] and (aLineID in RfaAvailLines) then begin
  AtomID:=aAtomID;
  LineID:=aLineID;
  Energy:=Rfa.XRayLine[AtomID,LineID].Energy;
  OI:=Rfa.XRayLine[AtomID,LineID].Height;
  AI:=OI*max(0,RfaWhat_Omega(AtomID,LineID,RfaParams.EnMax));
  II:=AI*Rfa.FindCrossSection(AtomID,rf_PHOTO,RfaParams.EnMax)*RfaFindDetectorEfficiency(Energy,RfaParams);
  Level:=Pos(Chr(ord(LineID)),RfaGetLinesOrder(RfaParams.SelLines));
  Enabled:=true;
 end;
end;

destructor TRfaLine.Destroy;
begin
 inherited Destroy;
end;

function TRfaLine.Intensity:Double;
begin
 Result:=II*ord(Enabled);
end;

function TRfaLine.NameElem:LongString;
begin
 Result:=Format('%-2s',[Rfa.AtomSymbol[AtomID]]);
end;

function TRfaLine.Name:LongString;
begin
 Result:=Format('%-2s %-3s (%8.6g KeV)',[Rfa.AtomSymbol[AtomID],Rfa.LineName[LineID],Energy]);
end;

function NewRfaLine(aAtomID:Byte; aLineID:TRfaFields; const RfaParams:TRfaParams):TRfaLine;
begin
 if Rfa.Accessible[aAtomID] and (aLineID in RfaAvailLines)
 then Result:=TRfaLine.Create(aAtomID,aLineID,RfaParams)
 else Result:=nil;
end;

procedure Kill(var TheObject:TRfaLine); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E);
 end; 
end;

 {
 *******************************************************************************
 методы TRfaLineList
 *******************************************************************************
 }
function  TRfaLineList.GetLine(index:Integer):TRfaLine;
begin
 Result:=TRfaLine(Items[index]);
end;

procedure TRfaLineList.SetLine(index:Integer; aLine:TRfaLine);
begin
 Items[index]:=aLine;
end;

function TRfaLineList.GetNumEnabledLines:Integer;
var
 i : Integer;
begin
 Result:=0;
 for i:=0 to Count-1 do if Self[i].Ok and Self[i].Enabled then inc(Result);
end;

function TRfaLineList.GetText:LongString;
const
 EnSt : array[Boolean] of string[1]=('-','+');
var
 p : TText;
 i : Integer;
begin
 p:=NewText;
 for i:=0 to Count-1 do
 if Self[i].Ok then with Self[i] do begin
  p.Addln(Format('%-3d  %s  %-2s  %-3s (Z=%-3d; E=%9.4f; OI=%9.4f; AI=%9.4f; I=%9.4f)',
          [i+1, EnSt[Enabled], NameElem, Rfa.LineName[LineID], AtomID, Energy, OI, AI, Intensity]))
 end else p.Addln('NIL');
 Result:=p.Text;
 Kill(p);
end;

procedure TRfaLineList.SearchForPeak(PeakEnergy,PeakFWHM:Double; const RfaParams:TRfaParams);
var
 AtomID  : Integer;
 LineID  : TRfaFields;
 LineNum : Integer;
 EnMin   : Double;
 EnMax   : Double;
 Energy  : Double;
 TheLine : TRfaLine;
begin
 Count:=0;
 EnMin:=PeakEnergy-PeakFWHM*RfaParams.MulFWHM;
 EnMax:=PeakEnergy+PeakFWHM*RfaParams.MulFWHM;
 EnMin:=max(MachEps,EnMin);
 for AtomID:=1 to RfaMaxAtomID do
 if AtomID in RfaParams.SelElems then begin
  if Rfa.Accessible[AtomID] then
  for LineNum:=1 to Length(RfaFullLinesList) do begin
   LineID:=TRfaFields(RfaFullLinesList[LineNum]);
   if LineID in RfaParams.SelLines then begin
    Energy:=Rfa[AtomID,LineId];
    if (Energy >= EnMin) and (Energy <= EnMax) then begin
     TheLine:=NewRfaLine(AtomID,LineID,RfaParams);
     if TheLine.Ok then Add(TheLine);
    end;
   end;
  end;
 end;
end;

function NewRfaLineList:TRfaLineList;
begin
 Result:=TRfaLineList.Create;
end;

procedure Kill(var TheObject:TRfaLineList); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E);
 end; 
end;

 {
 *******************************************************************************
 TRfaPeak implementation
 *******************************************************************************
 }
constructor TRfaPeak.Create(C,E,S,F:Double);
begin
 inherited Create;
 Lines:=NewRfaLineList;
 Ch:=C;
 En:=E;
 Sq:=S;
 Fw:=F;
end;

destructor TRfaPeak.Destroy;
begin
 Kill(Lines);
 inherited Destroy;
end;

function NewRfaPeak(C,E,S,F:Double):TRfaPeak;
begin
 Result:=TRfaPeak.Create(C,E,S,F);
end;

procedure Kill(var TheObject:TRfaPeak); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E);
 end; 
end;

 {
 *******************************************************************************
 TRfaPeakList implementation
 *******************************************************************************
 }
constructor TRfaPeakList.Create;
begin
 inherited Create;
 Details:=NewText;
end;

destructor TRfaPeakList.Destroy;
begin
 Kill(Details);
 inherited Destroy;
end;

function  TRfaPeakList.GetPeak(Index:Integer):TRfaPeak;
begin
 Result:=TRfaPeak(Items[Index]);
end;

procedure TRfaPeakList.SetPeak(Index:Integer; aPeak:TRfaPeak);
begin
 Items[Index]:=aPeak;
end;

procedure TRfaPeakList.ViewDetails;
begin
 ListBoxMenu(RusEng('Детали анализа пиков','Peak analize details'),'',Details.Text);
end;

procedure TRfaPeakList.Read(FileName:LongString);
var
 f   : System.Text;
 s   : LongString;
 Ch  : Double;
 En  : Double;
 Ar  : Double;
 Fw  : Double;
 ich : Integer;
 ien : Integer;
 iar : Integer;
 ifw : Integer;
begin
 Count:=0;
 FileName:=UnifyFileAlias(FileName);
 if FileExists(FileName) then
 try
  if SameText(ExtractFileExt(FileName),'.pik') then begin
   SetInOutRes(0);
   System.Assign(f,FileName);
   System.Reset(f);
   System.Readln(f,s);
   s:=UpCaseStr(s);
   ich:=WordIndex('CHANNEL',s,ScanSpaces);
   ien:=WordIndex('ENERGY',s,ScanSpaces);
   iar:=WordIndex('AREA',s,ScanSpaces);
   if iar=0 then iar:=WordIndex('SQUARE',s,ScanSpaces);
   ifw:=WordIndex('FWHM',s,ScanSpaces);
   if ifw=0 then ifw:=WordIndex('FWHP',s,ScanSpaces);
   if (ich>0) and (ien>0) and (ifw>0) and (iar>0) then
   while not eof(f) do begin
    System.Readln(f,s);
    if IOResult<>0 then begin
     Echo(RusEng('Ошибка чтения ','Error reading ')+FileName);
     break;
    end;
    if Str2Real(ExtractWord(ich,s,ScanSpaces),Ch) and
       Str2Real(ExtractWord(ien,s,ScanSpaces),En) and
       Str2Real(ExtractWord(iar,s,ScanSpaces),Ar) and
       Str2Real(ExtractWord(ifw,s,ScanSpaces),Fw)
    then Add(NewRfaPeak(Ch,En,Ar,Fw));
   end else Echo(RusEng('Неверный заголовок файла ','Invalid header file ')+FileName);
   System.Close(f);
   if IOResult<>0 then Echo(RusEng('Ошибка чтения ','Error reading ')+FileName);
   ConvertFwhmToKev;
  end else Echo(RusEng('Неверное расширение файла ','Invalid file extension ')+FileName);
 except
  on E:Exception do DaqReport(E.Message);
 end else Echo(RusEng('Не найден файл ','File not found ')+FileName);
end;

procedure TRfaPeakList.FindAllEnabledLines(const RfaParams:TRfaParams);
var
 i : Integer;
begin
 for i:=0 to Count-1 do Self[i].Lines.SearchForPeak(Self[i].En,Self[i].Fw,RfaParams);
end;

function TRfaPeakList.GetNumNonIsolPeaks:Integer;
var
 i : Integer;
 j : Integer;
begin
 Result:=0;
 for i:=0 to Count-1 do
 for j:=0 to Self[i].Lines.Count-1 do
 if Self[i].Lines[j].Enabled then begin
  inc(Result);
  break;
 end;
end;

function  TRfaPeakList.FindPeaksEnergyMax(const RfaParams:TRfaParams):Double;
var
 i : Integer;
begin
 Result:=0;
 for i:=0 to Count-1 do Result:=max(Result,Self[i].En+abs(Self[i].Fw*RfaParams.MulFWHM));
end;

function  TRfaPeakList.FindPeaksEnergyMin(const RfaParams:TRfaParams):Double;
var
 i : Integer;
begin
 Result:=1E30;
 for i:=0 to Count-1 do Result:=min(Result,Self[i].En-abs(Self[i].Fw*RfaParams.MulFWHM));
end;

function TRfaPeakList.LinePresent(AtomID:Byte;LineID:TRfaFields):Boolean;
var
 i     : Integer;
 j     : Integer;
 aLine : TRfaLine;
begin
 LinePresent:=false;
 for i:=0 to Count-1 do
 for j:=0 to Self[i].Lines.Count-1 do begin
  aLine:=Self[i].Lines[j];
  if aLine.Enabled and (aLine.AtomID=AtomID) and (aLine.LineID=LineID) then begin
   LinePresent:=true;
   exit;
  end;
 end;
end;

procedure TRfaPeakList.ReCalcCountLines(var CountLine,CountLineEnable:Integer);
var
 i : Integer;
begin
 CountLine:=0;
 CountLineEnable:=0;
 for i:=0 to Count-1 do begin
  inc(CountLine,Self[i].Lines.Count);
  inc(CountLineEnable,Self[i].Lines.GetNumEnabledLines);
 end;
end;

function TRfaPeakList.LineCanBe(Line:TRfaLine; PeakEMin,PeakEMax:Double;
                          const RfaParams:TRfaParams; out Msg:LongString):Boolean;
var
 Order    : Integer;
 HiEn     : Double;
 HiOI     : Double;
 HiElt    : Double;
 HiLineID : TRfaFields;
begin
 Msg:='';
 LineCanBe:=true;
 if Line.Level=1 then exit; {это самая старшая линия!}
 if not Rfa.Accessible[Line.AtomID] then LineCanBe:=false else
 for Order:=Line.Level-1 downto 1 do begin
  {цикл по старшим линиям того же элемента}
  HiLineID:=TRfaFields(RfaGetLinesOrder(RfaParams.SelLines)[Order]);
  if HiLineID in RfaAvailLines then begin
   HiEn:=Rfa.XRayLine[Line.AtomID,HiLineID].Energy;
   HiOI:=Rfa.XRayLine[Line.AtomID,HiLineID].Height;
   HiElt:=RfaWhat_Omega(Line.AtomID,HiLineID,RfaParams.EnMax);
   {выполнено условие возбуждения?}
   if (HiEn>=RfaParams.EnMin) and (HiEn<=RfaParams.EnMax) and (HiOI>0) and (HiElt>0)
    and (HiEn>=PeakEMin) and (HiEn<=PeakEMax)
   then begin
    {старшая линия возможна по условию возбуждения, но не присутствует}
    if not LinePresent(Line.AtomID,HiLineID) then begin
     Msg:=RusEng('удалена, так как не найдена ','removed, because not found ')+
                 Rfa.LineName[HiLineID]+' ('+Format('%8.6g',[HiEn])+' KeV)';
     LineCanBe:=false;
     break;
    end;
   end;
  end;
 end;
end;

procedure TRfaPeakList.AnalisSopLine(const RfaParams:TRfaParams);
var
 i        : Integer;
 j        : Integer;
 Msg      : LongString;
 List     : TText;
 LineIs   : Boolean;
 PeakEMin : Double;
 PeakEMax : Double;
begin
 PeakEMin:=FindPeaksEnergyMin(RfaParams);
 PeakEMax:=FindPeaksEnergyMax(RfaParams);
 List:=NewText;
 LineIs:=true;
 while LineIs do begin
  LineIs:=False;
  for i:=0 to Count-1 do
  for j:=0 to Self[i].Lines.Count-1 do
  if Self[i].Lines[j].Enabled then
  if not LineCanBe(Self[i].Lines[j],PeakEMin,PeakEmax,RfaParams,Msg) then begin
   List.Addln(Pad(Self[i].Lines[j].Name,16)+' => '+Msg);
   Self[i].Lines.Delete(j);
   LineIs:=true;
   break;
  end;
 end;
 if List.Count>0 then begin
  Details.Addln(RusEng('Анализ сопряженных линий:','Analize conjugate lines:'));
  Details.Addln(RusEng('Список удаленных линий ','List of removed lines:')+'('+d2s(List.Count)+'):');
  Details.Concat(List);
 end;
 Kill(List);
end;

procedure TRfaPeakList.ViewDublicates;
var i,i1,j1,i2,j2:Integer; List:TText; Exist:Boolean;
var Peak1,Peak2:TRfaPeak; Line1,Line2:TRfaLine;
var Name1,Name2:LongString;
begin
 List:=NewText;
 for i1:=0 to Count-1 do begin
  Peak1:=Self[i1];
  for j1:=0 to Peak1.Lines.Count-1 do begin
   Line1:=Peak1.Lines[j1];
   if Line1.Enabled then
   for i2:=0 to Count-1 do begin
    Peak2:=Self[i2];
    if Peak2<>Peak1 then
    for j2:=0 to Peak2.Lines.Count-1 do begin
     Line2:=Peak2.Lines[j2];
     if (Line2.Enabled) and (Line2<>Line1) then begin
      if (Line1.AtomID=Line2.AtomID) and (Line1.LineID=Line2.LineID) then begin
       Name1:=Line1.NameElem+' '+Rfa.LineName[Line1.LineID]+RusEng(' (Пик ',' (Peak ')+Pad(d2s(i1+1),2)+')';
       Name2:=Line2.NameElem+' '+Rfa.LineName[Line2.LineID]+RusEng(' (Пик ',' (Peak ')+Pad(d2s(i2+1),2)+')';
       Exist:=false;
       for i:=0 to List.Count-1 do
       if List[i]=Name1 then begin
        Exist:=true;
        break;
       end;
       if not Exist then List.Addln(Name1);
       Exist:=false;
       for i:=0 to List.Count-1 do
       if List[i]=Name2 then begin
        Exist:=true;
        break;
       end;
       if not Exist then List.Addln(Name2);
      end;
     end;
    end;
   end;
  end;
 end;
 if List.Count>0 then begin
  Details.Addln(RusEng('Следующие линии дублируются в разных пиках:',
                       'Next lines have dublicates in other peaks:'));
  Details.Concat(List);
 end;
 Kill(List);
end;

procedure TRfaPeakList.FirstAnalis(const RfaParams:TRfaParams);
begin
 try
  Details.Count:=0;
  FindAllEnabledLines(RfaParams);
  RemoveBadLines;
  AnalisSopLine(RfaParams);
  ViewDublicates;
  //ViewDetails;
 except
  on E:Exception do DaqReport(E.Message);
 end;
end;

procedure TRfaPeakList.RemoveBadLines;
var
 i    : Integer;
 j    : Integer;
 Flag : Boolean;
 List : TText;
begin
 List:=NewText;
 for i:=0 to Count-1 do begin
  Flag:=true;
  while Flag do begin
   Flag:=false;
   for j:=0 to Self[i].Lines.Count-1 do
   if Self[i].Lines[j].AI<=0 then begin
    List.Addln(Self[i].Lines[j].Name);
    Self[i].Lines.Delete(j);
    Flag:=true;
    break;
   end;
  end;
 end;
 if List.Count>0 then begin
  Details.Addln(Format(RusEng('Следующие линии с AI=0 удалены из списка (%d):',
                              'Next lines with AI=0 removed from list (%d):'),
                              [List.Count]));
  Details.Concat(List);
 end;
 Kill(List);
end;

function  TRfaPeakList.TotalArea:Double;
var
 i : Integer;
begin
 Result:=0;
 for i:=0 to Count-1 do Result:=Result+Self[i].Sq;
end;

procedure TRfaPeakList.ConvertFwhmToKev;
var
 i      : Integer;
 x      : PDoubleArray;
 y      : PDoubleArray;
 minx   : Double;
 maxx   : Double;
 ChToEn : TPolynom;
 function  FwhmInKev(Channel,Fwhm:Double):Double;
 begin
  FwhmInKev:=0.5*(ChToEn.Get(Channel+Fwhm)-ChToEn.Get(Channel-Fwhm));
 end;
begin
 if Count=0 then exit;
 x:=Allocate(Count*sizeof(Double));
 y:=Allocate(Count*sizeof(Double));
 minx:=1E30;
 maxx:=-1E30;
 for i:=0 to Count-1 do begin
  x[i]:=Self[i].Ch;
  y[i]:=Self[i].En;
  maxx:=max(maxx,x[i]);
  minx:=min(minx,x[i]);
 end;
 ChToEn:=NewPolynom(3,0.5*(maxx+minx),0.5*(maxx-minx));
 if not ChToEn.Find(x[0],y[0],Count) then begin
  DaqReport(RusEng('Ошибка вычисления FWHM калибровки!','Error evaluate FWHM calibration!'));
  Kill(ChToEn);
  ChToEn:=NewPolynom(1,0,1);
 end;
 Deallocate(Pointer(x));
 Deallocate(Pointer(y));
 for i:=0 to Count-1 do Self[i].Fw:=FwhmInKev(Self[i].Ch,Self[i].Fw);
 Kill(ChToEn);
end;

function NewRfaPeakList:TRfaPeakList;
begin
 Result:=TRfaPeakList.Create;
end;

procedure Kill(var TheObject:TRfaPeakList); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E);
 end; 
end;

 {
 *******************************************************************************
 TRfaSimplex implementation
 *******************************************************************************
 }
constructor TRfaSimplex.Create;
begin
 inherited Create;
 ReperAtom:=0;
 ReperConc:=0;
 ReperUnit:='';
 NumPeaks:=0;
 PeakEn:=nil;
 PeakSq:=nil;
 PeakFw:=nil;
 ErrorCode:=0;
 NumRows:=0;
 NumCells:=0;
 Problem:=nil;
 MatrCoef:=nil;
 Signums:=nil;
 RightSide:=nil;
 Solution:=nil;
 SolutionText:=nil;
 CellOfElement:=nil;
 ElementInCell:=nil;
 RowOfPeak:=nil;
 PeakInRow:=nil;
 EvalTime:=0;
 AvailElems:=[];
 PeakFile:='*.pik';
 ResultFile:=SessionManager.VarTmpFile('rfa_result.out');
 DetailFile:=SessionManager.VarTmpFile('rfa_detail.out');
end;

destructor TRfaSimplex.Destroy;
begin
 ReperAtom:=0;
 ReperConc:=0;
 ReperUnit:='';
 Clear;
 PeakFile:='';
 ResultFile:='';
 DetailFile:='';
 inherited Destroy;
end;

procedure TRfaSimplex.Clear;
begin
 if Ok then begin
  NumPeaks:=0;
  Kill(PeakEn);
  Kill(PeakSq);
  Kill(PeakFw);
  ErrorCode:=0;
  NumRows:=0;
  NumCells:=0;
  Kill(Problem);
  Kill(MatrCoef);
  Kill(Signums);
  Kill(RightSide);
  Kill(Solution);
  Kill(SolutionText);
  Kill(CellOfElement);
  Kill(ElementInCell);
  Kill(RowOfPeak);
  Kill(PeakInRow);
  EvalTime:=0;
  AvailElems:=[];
 end;
end;

 {
 Формирование матриц для задачи линейного программирования на основании списка пиков и линий.
 }
procedure TRfaSimplex.ConstructLinearProgrammingTask(PeakList:TRfaPeakList; const RfaParams:TRfaParams);
var
 Peak           : TRfaPeak;
 Line           : TRfaLine;
 AtomID         : Integer;
 Row            : Integer;
 Cell           : Integer;
 LineNum        : Integer;
 PeakNum        : Integer;
 Summ           : Double;
 PeakSquare     : Double;
 LineIntensity  : Double;
 PeakHaveLine   : Boolean;
 ElementPresent : Boolean;
begin
 if Ok then begin
  {
  Проверить, есть ли пики для анализа
  }
  if Trouble(not Assigned(PeakList) or (PeakList.Count=0),
             RusEng('Нет пиков для анализа!','No peaks to analize!')) then exit;
  {
  Удалить предыдущие результаты, сохранить опорный элемент
  }
  Clear;
  ReperAtom:=RfaParams.ReperAtom;
  ReperConc:=RfaParams.ReperConc;
  ReperUnit:=RfaParams.ReperUnit;
  {
  Найти набор элементов, линии которого есть в списке пиков
  }
  AvailElems:=[];
  for PeakNum:=1 to PeakList.Count do begin
   Peak:=PeakList[PeakNum-1];
   if Peak.Ok then
   for LineNum:=1 to Peak.Lines.Count do begin
    Line:=Peak.Lines[LineNum-1];
    if Line.Ok then AvailElems:=AvailElems+[Line.AtomID];
   end;
  end;
  {
  Запомнить основные параметры пиков
  }
  NumPeaks:=PeakList.Count;
  PeakEn:=NewDoubleVector(PeakList.Count,1);
  PeakSq:=NewDoubleVector(PeakList.Count,1);
  PeakFw:=NewDoubleVector(PeakList.Count,1);
  for PeakNum:=1 to PeakList.Count do begin
   Peak:=PeakList[PeakNum-1];
   if Peak.Ok then begin
    PeakEn[PeakNum]:=Peak.En;
    PeakSq[PeakNum]:=Peak.Sq;
    PeakFw[PeakNum]:=Peak.Fw;
   end;
  end;
  {
  Найти теоретические интенсивности для всех линий
  Эта процедура может варьироваться, в данной версии
  интенсивности вычисляются в TRfaLine.Create
  }
  //PeakList.EvaluateIntensity(RfaParams);
  {
  Найти число переменных задачи, то есть число элементов,
  имеющих в списке какие-либо разрешенные линии ненулевой
  интенсивности у пиков с ненулевой (допустимой) площадью
  находится также и соответствие столбец-элемент
  }
  NumCells:=0;
  CellOfElement:=NewLongIntVector(RfaMaxAtomID,1);
  ElementInCell:=NewLongIntVector(RfaMaxAtomID,1);
  for AtomID:=1 to RfaMaxAtomID do begin
   CellOfElement[AtomID]:=0;
   ElementInCell[AtomID]:=0;
   { Определим,присутствуют ли допустимые линии элемента }
   ElementPresent:=false;
   for PeakNum:=1 to PeakList.Count do begin
    Peak:=PeakList[PeakNum-1];
    if Peak.Ok then begin
     PeakSquare:=Peak.Sq;
     if PeakSquare>0 then
     for LineNum:=1 to Peak.Lines.Count do begin
      Line:=Peak.Lines[LineNum-1];
      if Line.Ok
      then if Line.Enabled
      then if Line.AtomID=AtomID
      then if Line.Intensity>0
      then begin
       ElementPresent:=true;
       break;
      end;
     end;
    end;
    if ElementPresent then break;
   end;
   { Если элемент присутствует-запомним его столбец }
   if ElementPresent then begin
    inc(NumCells);
    CellOfElement[AtomID]:=NumCells;
    ElementInCell[NumCells]:=AtomID;
   end;
  end;
  if Trouble(NumCells=0,RusEng('Не могу анализировать (NumCells=0).','Could not analize (NumCells=0).')) then begin
   Clear;
   exit;
  end;
  {
  Найти число ограничений задачи, то есть число пиков,
  у которых есть линии ненулевой интенсивности
  Кроме того, номера этих пиков запоминаются для дальнейших вычислений
  для этих пиков точно известно,что они имеют площадь >0 и линии
  с интенсивностью > 0
  }
  RowOfPeak:=NewLongIntVector(PeakList.Count,1);
  PeakInRow:=NewLongIntVector(PeakList.Count,1);
  if Trouble((RowOfPeak=nil) or (PeakInRow=nil),RusEng('Мало памяти!','Out of memory!')) then begin
   Clear;
   exit;
  end;
  NumRows:=0;
  for PeakNum:=1 to PeakList.Count do begin
   RowOfPeak[PeakNum]:=0;
   PeakInRow[PeakNum]:=0;
   Peak:=PeakList[PeakNum-1];
   { Определим,имеет ли пик допустимые линии }
   PeakHaveLine:=false;
   if Peak.Ok then begin
    PeakSquare:=Peak.Sq;
    if PeakSquare>0 then
    for LineNum:=1 to Peak.Lines.Count do begin
     Line:=Peak.Lines[LineNum-1];
     if Line.Ok
     then if Line.Enabled
     then if Line.Intensity>0
     then begin
      PeakHaveLine:=true;
      break;
     end;
    end;
   end;
   { Если в пике есть допустимые линии, запомним его }
   if PeakHaveLine then begin
    inc(NumRows);
    RowOfPeak[PeakNum]:=NumRows;
    PeakInRow[NumRows]:=PeakNum;
   end;
  end;
  if Trouble(NumRows=0,RusEng('Не могу анализировать (NumRows=0).','Could not analize (NumRows=0).')) then begin
   Clear;
   exit;
  end;
  {
  Выделить массивы для данных для симплекс - алгоритма.
  После выделения памяти в массивах стоят нули - это существенно!
  }
  Problem:=NewDoubleVector(NumCells,1);
  MatrCoef:=NewDoubleMatrix(NumRows,NumCells,1);
  Signums:=NewLongIntVector(NumRows,1);
  RightSide:=NewDoubleVector(NumRows,1);
  if Trouble((Problem=nil) or (MatrCoef=nil) or (Signums=nil) or (RightSide=nil),
             RusEng('Мало памяти!','Out of memory!'))
  then begin
   Clear;
   exit;
  end;
  {
  Заполнить массив знаков знаками <=
  Заполнить массив правых частей ограничений площадами пиков
  Построить также матрицу ограничений: MatrCoef[i,j] должен характеризовать
  вклад элемента j в площадь пика с номером i и равен сумме интенсивностей
  линий этого элемента, попадающих в этот пик, так что сумма членов одной
  строки равна теоретической плошади пика, которая не должна превысить
  измеренную
  }
  for Row:=1 to NumRows do begin
   Peak:=PeakList[PeakInRow[Row]-1];
   Signums[Row]:=-1;
   PeakSquare:=Peak.Sq;
   RightSide[Row]:=PeakSquare;
   for LineNum:=1 to Peak.Lines.Count do begin
    Line:=Peak.Lines[LineNum-1];
    if Assigned(Line) then begin
     Cell:=CellOfElement[Line.AtomID];
     if Line.Enabled and (Cell>0) then begin
      LineIntensity:=Line.Intensity;
      if LineIntensity>0 then begin
       MatrCoef[Row,Cell]:=MatrCoef[Row,Cell]+LineIntensity;
      end;
     end;
    end;
   end;
  end;
  {
  Построить вектор Problem, описывающий целевую функцию
  В данном случае - полная площадь линий - то есть коэффициент целевой
  функции при элементе равен сумме строк матрицы в столбце этого элемента
  В дальнейшем возможно придание весов пикам и элементам
  }
  for Cell:=1 to NumCells do begin
   Summ:=0;
   for Row:=1 to NumRows do Summ:=Summ+MatrCoef[Row,Cell];
   Problem[Cell]:=-Summ;
  end;
 end;
end;

 {
 Проверка столбца Cell решения.
 PresentElement устанавливается в true, если среди решений симплекс-задачи
 (то есть среди строк Solution) есть такие, для которых в столбце Cell есть
 ненулевое значение, другими словами, присутствует ли элемент в решении.
 JustElement установлен в true, если во всех решениях в столбце Cell стоит одно
 и то же число, то есть элемент определен точно. Значение false говорит о том,
 что данный элемент не удалось изолировать.
 }
procedure TRfaSimplex.CheckSolutionCell(Cell:Integer; var JustElement,PresentElement:Boolean);
const
 Eps = 1E-6;
var
 Row     : Integer;
 Delta   : Double;
 Average : Double;
begin
 if Ok then begin
  JustElement:=true;
  PresentElement:=false;
  for Row:=1 to Solution.Rows do begin
   if Solution[Row,Cell]>0 then PresentElement:=true;
   Delta:=abs(Solution[Row,Cell]-Solution[1,Cell]);
   Average:=max(Solution[Row,Cell],Solution[1,Cell]);
   if Delta>Average*Eps then JustElement:=false;
  end;
 end;
end;

 {
 Проверяет и возвращает концентрацию опорного (реперного) элемента
 и коэффициент перевода условной концентрации в реальную.
 Должна вызываться после нахождения решения SolveLinearProgrammingTask.
 Концентрация равна 1 при ошибке или ReperConc если все в порядке
 Множитель равен 1 при ошибке или ReperConc/(условн.конц.репера)
 если все в порядке
 Значения флага ошибки Error:
  0: все в порядке
  1: реперный атом отсутствует
  2: реперный элемент равен нулю
  3: реперный элемент определен неоднозначно
  4: концентрация недопустима (<=0)
 }
procedure TRfaSimplex.GetReper(var Concentration,Multiplier:Double; var Error:Integer);
var
 Just    : Boolean;
 Present : Boolean;
begin
 if Ok then begin
  Concentration:=1;
  Multiplier:=1;
  Error:=1; Just:=false; Present:=false;
  if (ReperAtom<1) or (ReperAtom>RfaMaxAtomID) then exit;
  if CellOfElement[ReperAtom]=0 then exit;
  CheckSolutionCell(CellOfElement[ReperAtom],Just,Present);
  Error:=2;
  if not Present then exit;
  Error:=3;
  if not Just then exit;
  Error:=4;
  if ReperConc<=0 then exit;
  Error:=0;
  Concentration:=ReperConc;
  Multiplier:=Concentration/Solution[1,CellOfElement[ReperAtom]];
 end;
end;

 {
 Чтение решения из файла в список для отображения
 }
procedure TRfaSimplex.ReadSolutionFile;
begin
 if Ok then begin
  Kill(SolutionText);
  SolutionText:=NewText;
  if not FileExists(ResultFile) then begin
   SolutionText.Addln(RusEng('Не найден ','Not found ')+ResultFile);
   exit;
  end;
  if Trouble(SolutionText.ReadFile(ResultFile)<>0,RusEng('Ошибка чтения ','Error reading ')+ResultFile)
  then SolutionText.Addln(RusEng('Ошибка чтения ','Error reading ')+ResultFile);
 end;
end;

 {
 Сохранение решения в файле
 }
procedure TRfaSimplex.SaveSolution(var OutPutFile:Text);
const
 Delimer = ' ';
 {
 Вывод решений - точно известных, или варьируемых,
                 присутствующих или отсутствующих (нулевых)
 true  true    - вывод точно известных ненулевых
 true  false   - вывод точно известных нулевых
 false true    - вывод варьируемых (а потому ненулевых)
 false true    - не может быть
 }
 procedure WriteResult(Just,Present:Boolean);
 var
  Row                : Integer;
  Cell               : Integer;
  Count              : Integer;
  ReperError         : Integer;
  JustElement        : Boolean;
  PresentElement     : Boolean;
  ReperFactor        : Double;
  ReperConcentration : Double;
 begin
  JustElement:=false;PresentElement:=false;
  ReperConcentration:=0;ReperFactor:=0;ReperError:=0;
  GetReper(ReperConcentration,ReperFactor,ReperError);
  Count:=0;
  for Cell:=1 to NumCells do begin
   {
   Определяем, точно ли определен элемент и присутствует ли он
   }
   CheckSolutionCell(Cell,JustElement,PresentElement);
   {
   Теперь вывод, если есть запрос на этот тип элементов
   }
   if (Present=PresentElement) and (Just=JustElement) then begin
    inc(Count);
    { Напишем имя элемента }
    write(OutPutFile,Rfa.AtomSymbol[ElementInCell[Cell]]:2);
    { Затем, если элемент присутствует, его решение или решения }
    if PresentElement then begin
     if JustElement
     then write(OutPutFile,Delimer,Solution[1,Cell]*ReperFactor:14:5)
     else for Row:=1 to Solution.Rows do
     write(OutPutFile,Delimer,Solution[Row,Cell]*ReperFactor:14:5);
    end;
    writeln(OutPutFile);
   end;
  end;
  if Count=0 then writeln(OutPutFile,'Нет!');
 end;
 {
 Процедура печати элементов, не участвовавших в анализе
 }
 procedure WriteAbsentElements;
 var
  AtomId : Integer;
  Count  : Integer;
 begin
  writeln(OutPutFile,
  'Следующие элементы не анализировались из-за отсутствия ненулевых линий:');
  Count:=0;
  for AtomId:=1 to RfaMaxAtomId do
  if (AtomID in AvailElems) and (CellOfElement[AtomID]=0) then begin
   inc(Count);
   writeln(OutPutFile,Rfa.AtomSymbol[AtomID]);
   if Count=0 then writeln(OutPutFile,'Нет!');
  end;
 end;
 {
 Печать информации о распределении площадей
 Вычисляет и выдает площадь пиков, теоретическую площадь пиков
 как сумму площадей линий в пике и их отношение в процентах.
 Внимание!
 Здесь следует учесть,что вообще говоря Simplex переупорядочивает матрицу
 ограничений,так что соответствие строка-пик может потеряться.
 Этого не происходит потому что все ограничения имеют один тип <=.
 }
 procedure WriteAreaInformation;
 const
  w = 14;
  d = 4;
 var
  Row            : Integer;
  Cell           : Integer;
  SolNum         : Integer;
  PeakNum        : Integer;
  PeakArea       : Double;
  TheorPeakArea  : Double;
  TheorLineArea  : Double;
  TotalArea      : Double;
  TheorTotalArea : Double;
  PerCent        : Double;
 begin
  writeln(OutPutFile);
  writeln(OutPutFile,'А теперь сравнение пиков и решения по площади.');
  for SolNum:=1 to Solution.Rows do begin
   writeln(OutPutFile,'Распределение площадей, решение ',SolNum,' :');
   writeln(OutPutFile,' N',Delimer,
                      'Пик':w,Delimer,
                      'Решение':w,Delimer,
                      'Проценты':w);
   TotalArea:=0;
   TheorTotalArea:=0;
   PerCent:=0;
   for PeakNum:=1 to NumPeaks do begin
    Row:=RowOfPeak[PeakNum];
    TheorPeakArea:=0;
    if Row>0 then
    for Cell:=1 to NumCells do begin
     TheorLineArea:=MatrCoef[Row,Cell]*Solution[SolNum,Cell];
     TheorPeakArea:=TheorPeakArea+TheorLineArea;
    end;
    PeakArea:=PeakSq[PeakNum];
    if PeakArea>0 then PerCent:=100*TheorPeakArea/PeakArea;
    TotalArea:=TotalArea+PeakArea;
    TheorTotalArea:=TheorTotalArea+TheorPeakArea;
    writeln(OutPutFile,PeakNum:2,Delimer,
                       PeakArea:w:d,Delimer,
                       TheorPeakArea:w:d,Delimer,
                       PerCent:w:d);
   end;
   PerCent:=0;
   if TotalArea>0 then PerCent:=100*TheorTotalArea/TotalArea;
   writeln(OutPutFile,'Общая площадь:');
   writeln(OutPutFile,'  ',Delimer,
                      TotalArea:w:d,Delimer,
                      TheorTotalArea:w:d,Delimer,
                      PerCent:w:d);
  end;
 end;
 {
 Печать информации об опорном элементе
 }
 procedure WriteReperInformation;
 var
  ReperError         : Integer;
  ReperFactor        : Double;
  ReperConcentration : Double;
 begin
  write(OutPutFile,'Реперный элемент: ');
  ReperConcentration:=0;ReperFactor:=0;ReperError:=0;
  GetReper(ReperConcentration,ReperFactor,ReperError);
  case ReperError of
   0:writeln(OutputFile,Rfa.AtomSymbol[ReperAtom],' = ',ReperConcentration:11:5);
   1:writeln(OutPutFile,'отсутствует.');
   2:writeln(OutPutFile,Rfa.AtomSymbol[ReperAtom],' равен нулю!');
   3:writeln(OutPutFile,Rfa.AtomSymbol[ReperAtom],' определен неоднозначно!');
   4:writeln(OutPutFile,Rfa.AtomSymbol[ReperAtom],' концентрация недопустима!');
  end;
  if ReperError=0 then begin
   write(OutPutFile,'Концентрации определены в абсолютных единицах');
   if ReperUnit<>''
   then  write(OutPutFile,': ',ReperUnit)
   else write(OutPutFile,'.');
   writeln(OutPutFile);
  end else begin
   writeln(OutPutFile,'Концентрации определены в относительных единицах.');
  end;
 end;
 {
 Печать информации о распределении площадей линий по пикам
 }
 procedure WritePeakDetails;
 var
  Row            : Integer;
  Cell           : Integer;
  SolNum         : Integer;
  Count          : Integer;
  JustElement    : Boolean;
  PresentElement : Boolean;
 begin
  writeln(OutPutFile);
  writeln(OutPutFile,'А теперь детальная раскладка площадей пиков по элементам.');
  for Row:=1 to NumRows do begin
   writeln(OutPutFile,'Пик ',PeakInRow[Row],' площадью ',RightSide[Row]:11:3,' содержит:');
   Count:=0; JustElement:=false; PresentElement:=false;
   for Cell:=1 to NumCells do begin
    {
    Определяем,точно ли определен элемент и присутствует ли он, а также дает ли он вклад в данный пик
    }
    CheckSolutionCell(Cell,JustElement,PresentElement);
    PresentElement:=(MatrCoef[Row,Cell]<>0) and PresentElement;
    {
    Затем, если элемент присутствует, выпишем его решение или решения
    }
    if PresentElement then begin
     inc(Count);
     { Напишем имя элемента }
     write(OutPutFile,Rfa.AtomSymbol[ElementInCell[Cell]]:2);
     if JustElement
     then write(OutPutFile,Delimer,Solution[1,Cell]*MatrCoef[Row,Cell]:14:3)
     else for SolNum:=1 to Solution.Rows do
     write(OutPutFile,Delimer,Solution[SolNum,Cell]*MatrCoef[Row,Cell]:14:3);
     writeln(OutPutFile);
    end;
   end;
   if Count=0 then writeln(OutPutFile,'Нет линий!');
  end;
 end;
begin
 if Ok then begin
  writeln(OutPutFile,'Решение задачи определения концентраций ',GetDateStr(msecnow,'.',true),'-',GetTimeStr(msecnow,':'));
  writeln(OutPutFile,'Источник: '+PeakFile);
  writeln(OutPutFile,'Время вычислений: ',EvalTime:5:2,' секунд');
  case ErrorCode of
    siOk               :
     begin { Решение получено,успех  }
      if Solution<>nil then begin
       WriteReperInformation;
       write(OutPutFile,'Решение существует');
       if Solution.Rows=1 then begin
        writeln(OutPutFile,' и единственно:');
        WriteResult(true,true);
        writeln(OutPutFile,'Следующие элементы оказались нулевыми:');
        WriteResult(true,false);
       end else begin
        writeln(OutPutFile,' но не единственное!');
        writeln(OutPutFile,'Их количество определяется выпуклой оболочкой ',
                            Solution.Rows,' решений:');
        writeln(OutPutFile,'Следующие элементы могут варьироваться.');
        WriteResult(false,true);
        writeln(OutPutFile,'Следующие элементы определяются однозначно:');
        WriteResult(true,true);
        writeln(OutPutFile,'Следующие элементы оказались нулевыми:');
        WriteResult(true,false);
       end;
       WriteAbsentElements;
       WriteAreaInformation;
       WritePeakDetails;
      end else writeln(OutPutFile,'Нет памяти!');
     end;
    siSolutionInfinite:
     writeln(OutPutFile,'Решение неограничено!');
    siConditionsError:
     writeln(OutPutFile,'Ограничения несовместны!');
    siOutOfMemory:
     writeln(OutPutFile,'Нет памяти для массивов!');
    siInvalidInput:
     writeln(OutPutFile,'Недопустимые входные данные!');
    siInvalidSignum:
     writeln(OutPutFile,'Ошибка в векторе знаков!');
    siInvalidRange:
     writeln(OutPutFile,'Ошибка размерности входных матриц!');
    siException:
     writeln(OutPutFile,'Возникло исключение!');
   else writeln(OutPutFile,'Неизвестная ошибка номер ',ErrorCode);
  end;
 end;
end;

 {
 Процедура решает задачу линейного программирования, подготовленную процедурой
 ConstructLinearProgrammingTask. Промежуточные вычисления и решение сваливаем
 в файл DetailFile. В файл ResultFile сваливаем только решение.
 }
procedure TRfaSimplex.SolveLinearProgrammingTask;
var
 Timer      : Double;
 PrintMode  : Word;
 OutPutFile : Text;
begin
 if Ok then begin
  PrintMode:=pfInput+pfResult+pfIterBasis+pfIterFun+pfIterSigm+pfEcho;
  Assign(OutPutFile,DetailFile);
  Rewrite(OutPutFile);
  Timer:=MSecNow;
  Solution:=Simplex(Problem,MatrCoef,Signums,RightSide,ErrorCode,PrintMode,OutPutFile);
  EvalTime:=(MSecNow-Timer)/1000;
  SaveSolution(OutPutFile);
  Close(OutPutFile);
  Trouble(IOResult<>0,'Error saving file '+DetailFile);
  {
  сохранить отчет о решении в файле обмена
  }
  Assign(OutPutFile,ResultFile);
  Rewrite(OutPutFile);
  SaveSolution(OutPutFile);
  Close(OutPutFile);
  Trouble(IOResult<>0,'Error saving file '+ResultFile);
  {
  прочитать отчет о решении в список для показа
  }
  ReadSolutionFile;
 end;
end;

procedure TRfaSimplex.AnalizeNow(PeakList:TRfaPeakList; const RfaParams:TRfaParams);
begin
 try
  ConstructLinearProgrammingTask(PeakList,RfaParams);
  SolveLinearProgrammingTask;
 except
  on E:Exception do DaqReport(E.Message);
 end;
end;

function NewRfaSimplex:TRfaSimplex;
begin
 Result:=TRfaSimplex.Create;
end;

procedure Kill(var TheObject:TRfaSimplex); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

 {
 *******************************************************************************
 TFormRfaMendeleevTable implementation
 *******************************************************************************
 }
function NewFormRfaMendeleevTable:TFormRfaMendeleevTable;
begin
 if Rfa.Active
 then Application.CreateForm(TFormRfaMendeleevTable, Result)
 else Result:=nil;
end;

function FormRfaMendeleevTableExecute(PikFile:LongString=''):Integer;
var Form:TFormRfaMendeleevTable;
begin
 Result:=mrCancel;
 Form:=NewFormRfaMendeleevTable;
 if Form.Ok then begin
  Form.LoadRfaParams;
  if IsNonEmptyStr(PikFile)
  then PikFile:=UnifyFileAlias(PikFile);
  if FileExists(PikFile) then begin
   Form.OpenDialogPik.FileName:=PikFile;
   Form.AnalizeFile(PikFile);
  end;
  Result:=mrVoice(Form.ShowModal);
  Form.SaveRfaParams;
 end;
 Kill(TForm(Form));
end;

procedure TFormRfaMendeleevTable.FormCreate(Sender: TObject);
 procedure InitElementButtons;
 const
  X0=5; Y0=5; DW=8; DH=8;
 var
  i,x,y,W,H:Integer; Btn:TSpeedButton;
 begin
  PageControl.PageIndex:=0;
  W:=((PanelMendTab.Width - X0 - DW) div 18) - DW;
  H:=((PanelMendTab.Height - Y0 - DW) div 9) - DH;
  for i:=1 to RfaMaxAtomID do if Rfa.Accessible[i] then begin
   x:=X0+DW+round(Rfa[i,rf_ELEMPOS_X]*(W+DW));
   y:=Y0+DW+round(Rfa[i,rf_ELEMPOS_Y]*(H+DH));
   Btn:=TSpeedButton.Create(Self);
   Btn.Caption:=Rfa.AtomSymbol[i];
   Btn.AllowAllUp:=True;
   Btn.GroupIndex:=i;
   Btn.Down:=(i in RfaParams.SelElems);
   Btn.Left:=x;
   Btn.Top:=y;
   Btn.Width:=W;
   Btn.Height:=H;
   Btn.Parent:=PanelMendTab;
   Btn.OnClick:=ButtonOfElementClick;
   Btn.Font.Color:=Rfa.AtomColor[i];
   Btn.Font.Style:=Btn.Font.Style+[fsBold];
   ButtonOfElement[i]:=Btn;
   Btn.Update;
  end;
 end;
 procedure InitDetectorElements;
 var p:TText; i:Integer;
 begin
  p:=NewText;
  p.Addln(RusEng('НЕТ','NONE'));
  for i:=1 to RfaMaxAtomID do p.Addln(Rfa.AtomSymbol[i]);
  p.Addln(RusEng('Воздух','Air'));
  ComboBoxDetectorElement.Items.Text:=p.Text;
  ComboBoxLayer1Element.Items.Text:=p.Text;
  ComboBoxLayer2Element.Items.Text:=p.Text;
  ComboBoxLayer3Element.Items.Text:=p.Text;
  ComboBoxLayer4Element.Items.Text:=p.Text;
  Kill(p);
 end;
 procedure InitComboBoxReperAtom;
 var i:Integer;
 begin
  ComboBoxReperAtom.Items.Clear;
  ComboBoxReperAtom.Items.Add(RusEng('НЕТ','NONE'));
  for i:=1 to RfaMaxAtomID do
  ComboBoxReperAtom.Items.Add(Rfa.AtomSymbol[i]);
 end;
begin
 inherited;
 PeakList:=NewRfaPeakList;
 RfaParams:=DefaultRfaParams;
 RfaSimplex:=NewRfaSimplex;
 ElemSelector:=NewXorSelector;
 ElemSelector.Mode:=sm_Rect;
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 Caption:=RusEng('База данных РФА','RFA database');
 PageControl.ParentFont:=true;
 PageControl.Pages[0].Caption:=RusEng('Таблица Менделеева','Mendeleev table');
 PageControl.Pages[1].Caption:=RusEng('Детектор','Detector');
 PageControl.Pages[2].Caption:=RusEng('Таблица РФА','RFA table');
 PageControl.Pages[3].Caption:=RusEng('Линии пика','Peak lines');
 PageControl.Pages[4].Caption:=RusEng('Анализ','Analize');
 PageControl.Pages[5].Caption:=RusEng('Отчет','Report');
 CheckBoxSelEnable.Caption:=RusEng('Выбор','Select');
 LabelSelEnable.Caption:=RusEng('элемента','elements');
 GroupBoxKSeria.Caption:=RusEng('K-серия','K-seria');
 GroupBoxLSeria.Caption:=RusEng('L-серия','L-seria');
 GroupBoxEminEmax.Caption:=RusEng('Энергия,КэВ','Energy,KeV');
 LabelAnod.Caption:=RusEng('Анод','Anode');
 LabelMulFWHM1.Caption:='FWHM';
 LabelMulFWHM2.Caption:=RusEng('фактор','factor');
 RadioGroupCategory.Caption:=RusEng('Категория информации','Information category');
 RadioGroupCategory.Items.Text:=RusEng('Линии РФА'+EOL+'Сечения РФА','RFA lines'+EOL+'Cross sections');
 LabelDetectorElement.Caption:=RusEng('Детектор','Detector');
 GroupBoxDetectorParams.Caption:=RusEng('Параметры детектора','Detector parameters');
 CheckBoxAllowDetectorEfficiency.Caption:=RusEng('Учитывать эффективность детектора','Allow detector efficiency');
 LabelDetectorWorkThickness.Caption:=RusEng('рабочая зона, см','work zone, sm');
 LabelDetectorDeadThickness.Caption:=RusEng('мертвая зона, см','dead zone, sm');
 LabelDetectorAirDensity.Caption:=RusEng('плотность воздуха','air density');
 LabelLayer1Element.Caption:=RusEng('Слой 1','Layer 1');
 LabelLayer2Element.Caption:=RusEng('Слой 2','Layer 2');
 LabelLayer3Element.Caption:=RusEng('Слой 3','Layer 3');
 LabelLayer4Element.Caption:=RusEng('Слой 4','Layer 4');
 LabelLayer1Thickness.Caption:=RusEng('толщиной, см','with thickness, sm');
 LabelLayer2Thickness.Caption:=RusEng('толщиной, см','with thickness, sm');
 LabelLayer3Thickness.Caption:=RusEng('толщиной, см','with thickness, sm');
 LabelLayer4Thickness.Caption:=RusEng('толщиной, см','with thickness, sm');
 GroupBoxDetectorEfficiency.Caption:=RusEng('Эффективность регистрации','Detector efficiency');
 LabelEfficiencyAt.Caption:=RusEng('Эффективность при','Efficiency at');
 LabelEfficiencyKeV.Caption:=RusEng('КэВ','KeV');
 GroupBoxPeakEnHw.Caption:=RusEng('Параметры пика','Peak parameters');
 GroupBoxPeakLines.Caption:=RusEng('Линии пика','Peak lines');
 LabelPeakEn.Caption:=RusEng('Энергия, КэВ','Energy, KeV');
 ButtonPeakFindLines.Caption:=RusEng('Найти линии','Find lines');
 GroupBoxRfaPeaks.Caption:=RusEng('Список пиков','Peak list');
 GroupBoxRfaLines.Caption:=RusEng('Список линий','Line list');
 GroupBoxRfaButtons.Caption:=RusEng('Команды','Commands');
 ButtonLoadPeaks.Caption:=RusEng('Читать пики','Load peaks');
 ButtonSearchLines.Caption:=RusEng('Поиск линий','Search lines');
 ButtonViewDetails.Caption:=RusEng('Детали поиска','View details');
 ButtonAnalizeNow.Caption:=RusEng('Пуск анализа','Analize now');
 LabelReperAtom.Caption:=RusEng('Опорный атом','Base atom');
 LabelReperConc.Caption:=RusEng('количеством','amount');
 LabelReperUnit.Caption:=RusEng('Единицы измерения','Measure in units');
 ButtonSaveReport.Caption:=RusEng('Записать отчет','Save report');
 ButtonLoadReport.Caption:=RusEng('Прочитать отчет','Load report');
 GroupBoxRfaReport.Caption:=RusEng('Отчет Рентгеновского Флюоресцентного Анализа пиков (РФА)','Analize report');
 LocateFormToCenterOfScreen(Self);
 InitElementButtons;
 InitDetectorElements;
 InitComboBoxReperAtom;
 UpdateControls(idUpdate);
 HideControls([ButtonSearchLines,ButtonAnalizeNow,ButtonViewDetails,ButtonInvLines]);
 RadioGroupCategory.ItemIndex:=0;
end;

procedure TFormRfaMendeleevTable.FormDestroy(Sender: TObject);
begin
 Kill(PeakList);
 Kill(RfaSimplex);
 Kill(ElemSelector);
 inherited;
end;

procedure TFormRfaMendeleevTable.ButtonOfElementClick(Sender: TObject);
var
 AtomID : Byte;
begin
 if (Sender is TSpeedButton) then begin
  AtomID:=Rfa.FindAtom((Sender as TSpeedButton).Caption);
  if Rfa.Accessible[AtomID] then begin
   RfaParams.SelElem:=AtomID;
   if RfaParams.SelEnable then UpdateControls(idChange);
   UpdateControls(idUpdate);
  end;
 end;
end;

procedure TFormRfaMendeleevTable.RadioGroupCategoryClick(Sender: TObject);
begin
 UpdateControls(idUpdate);
end;

procedure TFormRfaMendeleevTable.CheckBoxSelEnableClick(Sender: TObject);
begin
 RfaParams.SelEnable:=CheckBoxSelEnable.Checked;
 UpdateControls(idUpdate);
end;

procedure TFormRfaMendeleevTable.PaintBoxMendTabMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
 if Button = mbLeft then ElemSelector.StartXY(PaintBoxMendTab.Canvas,X,Y);
end;

procedure TFormRfaMendeleevTable.PaintBoxMendTabMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
begin
 ElemSelector.ReplaceXY(PaintBoxMendTab.Canvas,X,Y);
end;

procedure TFormRfaMendeleevTable.PaintBoxMendTabMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
 i : Integer;
 R : TRect2I;
 B : TRect2I;
begin
 if Button = mbLeft then
 if ElemSelector.Stop(PaintBoxMendTab.Canvas) then
 if RfaParams.SelEnable then begin
  R:=RectValidate(ElemSelector.Selection);
  for i:=Low(ButtonOfElement) to High(ButtonOfElement) do begin
   if Assigned(ButtonOfElement[i]) then
   with ButtonOfElement[i] do B:=Rect2I(Left,Top,Left+Width,Top+Height);
   if RectContainsPoint(R,RectCenter(B)) then begin
    if i in RfaParams.SelElems
    then exclude(RfaParams.SelElems,i)
    else include(RfaParams.SelElems,i);
   end;
  end;
  UpdateControls(idUpdate);
 end;
end;

procedure TFormRfaMendeleevTable.UpdateControls(Flags:Integer);
var
 v      : Double;
 i,n    : Integer;
 AtomID : Byte;
 procedure SetLine(CheckBox:TCheckBox; idLine:TRfaFields; var Lines:TRfaFieldsSet);
 begin
  CheckWithoutClick(CheckBox, idLine in Lines);
 end;
 procedure GetLine(CheckBox:TCheckBox; idLine:TRfaFields; var Lines:TRfaFieldsSet);
 begin
  if CheckBox.Checked then Include(Lines,idLine) else Exclude(Lines,idLine);
 end;
begin
 if Ok then
 try
  if Flags and idUpdate <> 0 then begin
   AtomID:=RfaParams.SelElem;
   if Rfa.Accessible[AtomID] then begin
    StatusBar.SimpleText:=Format(' %-2s %-2s %-12s %-11s Period %s Group %-2s Weight %-7.3f Density %-7.5g',
                          [Rfa.AsString[AtomId,rf_ATOM_ID], Rfa.AsString[AtomId,rf_ATOM_SYMBOL],
                           Rfa.AsString[AtomId,rf_ATOM_NAME_ENG], Rfa.AsString[AtomId,rf_ATOM_NAME_RUS],
                           Rfa.AsString[AtomId,rf_ATOM_PERIOD], Rfa.AsString[AtomId,rf_ATOM_GROUP],
                           Rfa[AtomId,rf_ATOM_WEIGHT], Rfa[AtomId,rf_ATOM_DENSITY]]);
    case RadioGroupCategory.ItemIndex of
     0 :  SmartUpdate(MemoRfaTab,RfaGetList(AtomID));
     1 :  SmartUpdate(MemoRfaTab,RfaGetCrossList(AtomID));
     else SmartUpdate(MemoRfaTab,'');
    end;
   end else begin
    StatusBar.SimpleText:=Format(RusEng('Элемент %d не найден.','Element %d not found.'),[AtomID]);
    SmartUpdate(MemoRfaTab,'');
   end;
   StatusBar.Update;
   CheckWithoutClick(CheckBoxSelEnable,RfaParams.SelEnable);
   for AtomID:=1 to RfaMaxAtomID do begin
    if Assigned(ButtonOfElement[AtomID]) then
    ButtonOfElement[AtomID].Down:=(AtomID in RfaParams.SelElems);
   end;
   SetLine(CheckBoxKa,  rf_EKA,   RfaParams.SelLines);
   SetLine(CheckBoxKa1, rf_EKA1,  RfaParams.SelLines);
   SetLine(CheckBoxKa2, rf_EKA2,  RfaParams.SelLines);
   SetLine(CheckBoxKb,  rf_EKB,   RfaParams.SelLines);
   SetLine(CheckBoxKb1, rf_EKB1,  RfaParams.SelLines);
   SetLine(CheckBoxKb2, rf_EKB2,  RfaParams.SelLines);
   SetLine(CheckBoxKb3, rf_EKB3,  RfaParams.SelLines);
   SetLine(CheckBoxLa1, rf_ELA12, RfaParams.SelLines);
   SetLine(CheckBoxLb1, rf_ELB1,  RfaParams.SelLines);
   SetLine(CheckBoxLb2, rf_ELB2,  RfaParams.SelLines);
   SetLine(CheckBoxLg1, rf_ELG1,  RfaParams.SelLines);
   SetLine(CheckBoxLL,  rf_ELL,   RfaParams.SelLines);
   EditEmin.Text:=Format('%.5g',[RfaParams.EnMin]);
   EditEmax.Text:=Format('%.5g',[RfaParams.EnMax]);
   if EditEmax.Text=Format('%.5g',[8.39])   then ComboBoxAnod.ItemIndex:=0 else
   if EditEmax.Text=Format('%.5g',[17.48])  then ComboBoxAnod.ItemIndex:=1 else
   if EditEmax.Text=Format('%.5g',[22.104]) then ComboBoxAnod.ItemIndex:=2 else
   ComboBoxAnod.ItemIndex:=-1;
   EditMulFWHM.Text:=Format('%.5g',[RfaParams.MulFWHM]);
   CheckWithoutClick(CheckBoxAllowDetectorEfficiency, RfaParams.Detector.Allow);
   ComboBoxDetectorElement.ItemIndex:=RfaParams.Detector.Element;
   ComboBoxLayer1Element.ItemIndex:=RfaParams.Detector.Layers[1].Element;
   ComboBoxLayer2Element.ItemIndex:=RfaParams.Detector.Layers[2].Element;
   ComboBoxLayer3Element.ItemIndex:=RfaParams.Detector.Layers[3].Element;
   ComboBoxLayer4Element.ItemIndex:=RfaParams.Detector.Layers[4].Element;
   EditDetectorWorkThickness.Text:=Format('%g',[RfaParams.Detector.WorkThickness]);
   EditDetectorDeadThickness.Text:=Format('%g',[RfaParams.Detector.DeadThickness]);
   EditDetectorAirDensity.Text:=Format('%g',[RfaParams.Detector.AirDensity]);
   EditLayer1Thickness.Text:=Format('%g',[RfaParams.Detector.Layers[1].Thickness]);
   EditLayer2Thickness.Text:=Format('%g',[RfaParams.Detector.Layers[2].Thickness]);
   EditLayer3Thickness.Text:=Format('%g',[RfaParams.Detector.Layers[3].Thickness]);
   EditLayer4Thickness.Text:=Format('%g',[RfaParams.Detector.Layers[4].Thickness]);
   ComboBoxReperAtom.ItemIndex:=RfaParams.ReperAtom;
   EditReperConc.Text:=Format('%g',[RfaParams.ReperConc]);
   EditReperUnit.Text:=Format('%s',[RfaParams.ReperUnit]);
  end;
  if Flags and idChange <> 0 then begin
   for AtomID:=1 to RfaMaxAtomID do begin
    if Assigned(ButtonOfElement[AtomID]) then
    if ButtonOfElement[AtomID].Down
    then include(RfaParams.SelElems,AtomID)
    else exclude(RfaParams.SelElems,AtomID);
   end;
   GetLine(CheckBoxKa,  rf_EKA,   RfaParams.SelLines);
   GetLine(CheckBoxKa1, rf_EKA1,  RfaParams.SelLines);
   GetLine(CheckBoxKa2, rf_EKA2,  RfaParams.SelLines);
   GetLine(CheckBoxKb,  rf_EKB,   RfaParams.SelLines);
   GetLine(CheckBoxKb1, rf_EKB1,  RfaParams.SelLines);
   GetLine(CheckBoxKb2, rf_EKB2,  RfaParams.SelLines);
   GetLine(CheckBoxKb3, rf_EKB3,  RfaParams.SelLines);
   GetLine(CheckBoxLa1, rf_ELA12, RfaParams.SelLines);
   GetLine(CheckBoxLb1, rf_ELB1,  RfaParams.SelLines);
   GetLine(CheckBoxLb2, rf_ELB2,  RfaParams.SelLines);
   GetLine(CheckBoxLg1, rf_ELG1,  RfaParams.SelLines);
   GetLine(CheckBoxLL,  rf_ELL,   RfaParams.SelLines);
   if Str2Real(EditEmin.Text,v)
   then RfaParams.EnMin:=v
   else DaqReport(Format(RusEng('Ошибка "%s"','Error "%s"'),[EditEmin.Text]));
   if Str2Real(EditEmax.Text,v)
   then RfaParams.EnMax:=v
   else DaqReport(Format(RusEng('Ошибка "%s"','Error "%s"'),[EditEmax.Text]));
   if Str2Real(EditMulFWHM.Text,v)
   then RfaParams.MulFWHM:=v
   else DaqReport(Format(RusEng('Ошибка "%s"','Error "%s"'),[EditMulFWHM.Text]));
   RfaParams.Detector.Allow:=CheckBoxAllowDetectorEfficiency.Checked;
   RfaParams.Detector.Element:=ComboBoxDetectorElement.ItemIndex;
   RfaParams.Detector.Layers[1].Element:=ComboBoxLayer1Element.ItemIndex;
   RfaParams.Detector.Layers[2].Element:=ComboBoxLayer2Element.ItemIndex;
   RfaParams.Detector.Layers[3].Element:=ComboBoxLayer3Element.ItemIndex;
   RfaParams.Detector.Layers[4].Element:=ComboBoxLayer4Element.ItemIndex;
   if Str2Real(EditDetectorWorkThickness.Text,v)
   then RfaParams.Detector.WorkThickness:=v
   else DaqReport(Format(RusEng('Ошибка "%s"','Error "%s"'),[EditDetectorWorkThickness.Text]));
   if Str2Real(EditDetectorDeadThickness.Text,v)
   then RfaParams.Detector.DeadThickness:=v
   else DaqReport(Format(RusEng('Ошибка "%s"','Error "%s"'),[EditDetectorDeadThickness.Text]));
   if Str2Real(EditDetectorAirDensity.Text,v)
   then RfaParams.Detector.AirDensity:=v
   else DaqReport(Format(RusEng('Ошибка "%s"','Error "%s"'),[EditDetectorAirDensity.Text]));
   if Str2Real(EditLayer1Thickness.Text,v)
   then RfaParams.Detector.Layers[1].Thickness:=v
   else DaqReport(Format(RusEng('Ошибка "%s"','Error "%s"'),[EditLayer1Thickness.Text]));
   if Str2Real(EditLayer2Thickness.Text,v)
   then RfaParams.Detector.Layers[2].Thickness:=v
   else DaqReport(Format(RusEng('Ошибка "%s"','Error "%s"'),[EditLayer2Thickness.Text]));
   if Str2Real(EditLayer3Thickness.Text,v)
   then RfaParams.Detector.Layers[3].Thickness:=v
   else DaqReport(Format(RusEng('Ошибка "%s"','Error "%s"'),[EditLayer3Thickness.Text]));
   if Str2Real(EditLayer4Thickness.Text,v)
   then RfaParams.Detector.Layers[4].Thickness:=v
   else DaqReport(Format(RusEng('Ошибка "%s"','Error "%s"'),[EditLayer4Thickness.Text]));
   RfaParams.ReperAtom:=ComboBoxReperAtom.ItemIndex;
   if Str2Real(EditReperConc.Text,v)
   then RfaParams.ReperConc:=v
   else DaqReport(Format(RusEng('Ошибка "%s"','Error "%s"'),[EditReperConc.Text]));
   RfaParams.ReperUnit:=EditReperUnit.Text;
   { при всех изменениях надо очистить пики, линии и результаты анализа }
   PeakList.Count:=0;
   RfaSimplex.Clear;
   GroupBoxRfaPeaks.Caption:=RusEng('Список пиков','Peak list');
   GroupBoxRfaLines.Caption:=RusEng('Список линий','Line list');
   ListBoxRfaPeaks.Clear;
   CheckListBoxRfaLines.Clear;
   HideControls([ButtonSearchLines,ButtonAnalizeNow,ButtonViewDetails,ButtonInvLines]);
   SmartUpdate(MemoRfaReport,'');
  end;
  if Flags and idPeaks <> 0 then begin
   GroupBoxRfaPeaks.Caption:=Format(RusEng('Список пиков [%s]','Peak list [%s]'),
                                    [ExtractFileNameExt(RfaSimplex.PeakFile)]);
   n:=ListBoxRfaPeaks.ItemIndex;
   ListBoxRfaPeaks.Items.Clear;
   for i:=0 to PeakList.Count-1 do begin
    ListBoxRfaPeaks.Items.Add(Format('%-2d En=%-7.3f Hw=%-6.3f Sq=%-9.2f',
                              [i+1, PeakList[i].En, PeakList[i].Fw, PeakList[i].Sq]));
   end;
   ListBoxRfaPeaks.ItemIndex:=max(0,min(n,ListBoxRfaPeaks.Items.Count-1));
   SmartUpdate(MemoRfaReport,'');
  end;
  if Flags and idLines <> 0 then begin
   n:=ListBoxRfaPeaks.ItemIndex;
   GroupBoxRfaLines.Caption:=Format(RusEng('Список линий пика %d','Line list of peak %d'),[n+1]);
   if (n>=0) and (n<PeakList.Count) then begin
    CheckListBoxRfaLines.Items.Clear;
    for i:=0 to PeakList[n].Lines.Count-1 do
    with PeakList[n].Lines[i] do begin
     CheckListBoxRfaLines.Items.Add(Format('%s %s (E=%7.3f I=%9.4f)',
                                    [NameElem,Rfa.LineName[LineID],Energy,Intensity]));
     CheckListBoxRfaLines.Checked[i]:=Enabled;
    end;
   end;
   SmartUpdate(MemoRfaReport,'');
  end;
  if Flags and idReport <> 0 then begin
   SmartUpdate(MemoRfaReport,RfaSimplex.SolutionText.Text);
  end;
  if Flags and idCheckLn <> 0 then begin
   n:=ListBoxRfaPeaks.ItemIndex;
   if (n>=0) and (n<PeakList.Count) then begin
    if PeakList[n].Lines.Count=CheckListBoxRfaLines.Items.Count then
    for i:=0 to PeakList[n].Lines.Count-1 do
    PeakList[n].Lines[i].Enabled:=CheckListBoxRfaLines.Checked[i];
   end;
  end;
  if Flags and idInvLines <> 0 then begin
   n:=ListBoxRfaPeaks.ItemIndex;
   if (n>=0) and (n<PeakList.Count) then
   for i:=0 to PeakList[n].Lines.Count-1 do
   PeakList[n].Lines[i].Enabled:=not PeakList[n].Lines[i].Enabled;
  end;
 except
  on E:Exception do DaqReport(E.Message);
 end;
end;

procedure TFormRfaMendeleevTable.CheckBoxLinesClick(Sender: TObject);
begin
 UpdateControls(idChange);
 CheckKaKb(RfaParams.SelLines);
 UpdateControls(idUpdate);
end;

procedure TFormRfaMendeleevTable.ComboBoxAnodChange(Sender: TObject);
begin
 case ComboBoxAnod.ItemIndex of
  0 : RfaParams.EnMax:=8.39;
  1 : RfaParams.EnMax:=17.48;
  2 : RfaParams.EnMax:=22.104;
 end;
 UpdateControls(idUpdate);
end;

procedure TFormRfaMendeleevTable.ApplyChanges(Sender: TObject);
begin
 UpdateControls(idChange);
 UpdateControls(idUpdate);
end;

procedure TFormRfaMendeleevTable.CancelChanges(Sender: TObject);
begin
 UpdateControls(idUpdate);
end;

procedure TFormRfaMendeleevTable.AnyOneKeyDown(Sender: TObject;  var Key: Word; Shift: TShiftState);
begin
 case Key of
  VK_RETURN : ApplyChanges(Sender);
  VK_ESCAPE : CancelChanges(Sender);
 end;
end;

procedure TFormRfaMendeleevTable.ButtonEfficiencyEqualClick(Sender: TObject);
var
 En : Double;
begin
 if Str2Real(EditEfficiencyAt.Text,En)
 then EditEfficiencyIs.Text:=Format('%g',[RfaFindDetectorEfficiency(En,RfaParams)]);
end;

procedure TFormRfaMendeleevTable.ButtonPeakFindLinesClick(Sender: TObject);
var Energy,FWHM:Double; Lines:TRfaLineList;
begin
 if Str2Real(EditPeakEn.Text,Energy) and Str2Real(EditPeakFwhm.Text,FWHM) then begin
  GroupBoxPeakLines.Caption:=Format(RusEng(' Линии пика E=%-.5g +/- %-.5g ',
                                           ' Peak lines E=%-.5g +/- %-.5g '),
                                           [Energy,FWHM*RfaParams.MulFWHM]);
  Lines:=NewRfaLineList;
  Lines.SearchForPeak(Energy, Fwhm, RfaParams);
  MemoPeakLines.Text:=Lines.GetText;
  Kill(Lines);
 end else DaqReport(RusEng('Неверный числовой формат!','Invalid numerical format!'));
end;

function TFormRfaMendeleevTable.LoadPeaksFrom(FileName:LongString):Boolean;
begin
 LoadPeaksFrom:=false;
 HideControls([ButtonSearchLines,ButtonAnalizeNow,ButtonViewDetails,ButtonInvLines]);
 PeakList.Count:=0;
 FileName:=UnifyFileAlias(FileName);
 PeakList.Read(FileName);
 RfaSimplex.PeakFile:=FExpand(FileName);
 UpdateControls(idPeaks+idLines);
 if PeakList.Count>0 then begin
  ShowControls([ButtonSearchLines]);
  FindAvailableLines;
  LoadPeaksFrom:=true;
 end;
 UpdateControls(idPeaks+idLines);
end;

procedure TFormRfaMendeleevTable.FindAvailableLines;
begin
 PeakList.FirstAnalis(RfaParams);
 ShowControls([ButtonAnalizeNow,ButtonViewDetails,ButtonInvLines]);
 UpdateControls(idPeaks+idLines);
end;

procedure TFormRfaMendeleevTable.ButtonLoadPeaksClick(Sender: TObject);
begin
 OpenDialogPik.Title:=RusEng('Ввести список пиков из файла','Load peak list from file');
 if GuardOpenDialog(OpenDialogPik).Execute then LoadPeaksFrom(OpenDialogPik.FileName);
end;

procedure TFormRfaMendeleevTable.ListBoxRfaPeaksClick(Sender: TObject);
begin
 UpdateControls(idLines);
end;

procedure TFormRfaMendeleevTable.ButtonSearchLinesClick(Sender: TObject);
begin
 FindAvailableLines;
end;

procedure TFormRfaMendeleevTable.ButtonViewDetailsClick(Sender: TObject);
begin
 PeakList.ViewDetails;
end;

procedure TFormRfaMendeleevTable.CheckListBoxRfaLinesClickCheck(Sender: TObject);
begin
 UpdateControls(idCheckLn);
 UpdateControls(idPeaks+idLines);
end;

procedure TFormRfaMendeleevTable.AnalizeNow;
begin
 RfaSimplex.AnalizeNow(PeakList,RfaParams);
 UpdateControls(idReport);
 PageControl.PageIndex:=5;
end;


procedure TFormRfaMendeleevTable.ButtonAnalizeNowClick(Sender: TObject);
begin
 if PeakList.Count>0
 then AnalizeNow
 else DaqReport(RusEng('Нет пиков для анализа!','No peaks to analize!'));
end;

procedure TFormRfaMendeleevTable.ButtonSaveReportClick(Sender: TObject);
var
 p : TText;
begin
 if Ok and (MemoRfaReport.Lines.Count>0) then begin
  SaveDialogRep.Title:=RusEng('Записать файл отчета','Save report file');
  SaveDialogRep.Filter:=RusEng('Отчеты (*.rep)|*.rep','Reports (*.rep)|*.rep');
  SaveDialogRep.FileName:=ForceExtension(RfaSimplex.PeakFile,'.rep');
  if IsEmptyStr(SaveDialogRep.FileName)
  then OpenDialogSelectType(SaveDialogRep,'*.rep')
  else OpenDialogSelectType(SaveDialogRep,SaveDialogRep.FileName);
  if GuardOpenDialog(SaveDialogRep).Execute then begin
   p:=NewText;
   p.Text:=MemoRfaReport.Text;
   if p.WriteFile(SaveDialogRep.FileName)<>0
   then DaqReport(RusEng('Ошибка записи ','Error writing ')+SaveDialogRep.FileName);
   Kill(p);
  end;
 end;
end;

procedure TFormRfaMendeleevTable.ButtonLoadReportClick(Sender: TObject);
var
 p : TText;
begin
 if Ok then begin
  OpenDialogRep.Title:=RusEng('Открыть файл отчета','Open report file');
  OpenDialogRep.Filter:=RusEng('Отчеты (*.rep)|*.rep','Reports (*.rep)|*.rep');
  OpenDialogRep.FileName:=ForceExtension(RfaSimplex.PeakFile,'.rep');
  if IsEmptyStr(OpenDialogRep.FileName)
  then OpenDialogSelectType(OpenDialogRep,'*.rep')
  else OpenDialogSelectType(OpenDialogRep,OpenDialogRep.FileName);
  if GuardOpenDialog(OpenDialogRep).Execute then begin
   SmartUpdate(MemoRfaReport,'');
   p:=NewText;
   if p.ReadFile(OpenDialogRep.FileName)<>0
   then DaqReport(RusEng('Ошибка записи ','Error writing ')+OpenDialogRep.FileName)
   else SmartUpdate(MemoRfaReport,p.Text);
   Kill(p);
  end;
 end;
end;

procedure TFormRfaMendeleevTable.ButtonInvLinesClick(Sender: TObject);
begin
 UpdateControls(idInvLines);
 UpdateControls(idLines);
end;

procedure TFormRfaMendeleevTable.LoadRfaParams;
var cfg:LongString;
begin
 cfg:='';
 if Ok then
 if ReadIniFilePath(SysIniFile,'[Rfa]','FileOfCurrentParameters',HomeDir,cfg) then begin
  LoadRfaParameters(RfaParams,cfg);
  UpdateControls(idUpdate);
 end;
end;

procedure TFormRfaMendeleevTable.SaveRfaParams;
var cfg:LongString;
begin
 cfg:='';
 if Ok then
 if ReadIniFilePath(SysIniFile,'[Rfa]','FileOfCurrentParameters',HomeDir,cfg)
 then SaveRfaParameters(RfaParams,cfg,false);
end;

procedure TFormRfaMendeleevTable.AnalizeFile(PikFile:LongString);
begin
 PikFile:=UnifyFileAlias(PikFile);
 if Ok then
 if FileExists(PikFile) then begin
  LoadPeaksFrom(PikFile);
  AnalizeNow;
 end else DaqReport(RusEng('Не найден ','Could not find ')+PikFile);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_rfamendeleevtable;
begin
end;

procedure Free_form_rfamendeleevtable;
begin
end;

initialization

 Init_form_rfamendeleevtable;

finalization

 Free_form_rfamendeleevtable;

end.

//////////////
// END OF FILE
//////////////

