////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Модуль дает доступ к базе данных RFA в файле rfadata.ini.                  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231127 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit _crw_rfadata; // RFA database access

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_ef,
 _crw_appforms, _crw_apptools, _crw_apputils;

 ///////////////////////////////////////////////////////////////////////////////
 // Идентификаторы элементов таблицы Менделеева.                              //
 ///////////////////////////////////////////////////////////////////////////////
const
 AtomId_H  = 1;    AtomId_He = 2;     AtomId_Li = 3;     AtomId_Be = 4;     AtomId_B  = 5;
 AtomId_C  = 6;    AtomId_N  = 7;     AtomId_O  = 8;     AtomId_F  = 9;     AtomId_Ne = 10;
 AtomId_Na = 11;   AtomId_Mg = 12;    AtomId_Al = 13;    AtomId_Si = 14;    AtomId_P  = 15;
 AtomId_S  = 16;   AtomId_Cl = 17;    AtomId_Ar = 18;    AtomId_K  = 19;    AtomId_Ca = 20;
 AtomId_Sc = 21;   AtomId_Ti = 22;    AtomId_V  = 23;    AtomId_Cr = 24;    AtomId_Mn = 25;
 AtomId_Fe = 26;   AtomId_Co = 27;    AtomId_Ni = 28;    AtomId_Cu = 29;    AtomId_Zn = 30;
 AtomId_Ga = 31;   AtomId_Ge = 32;    AtomId_As = 33;    AtomId_Se = 34;    AtomId_Br = 35;
 AtomId_Kr = 36;   AtomId_Rb = 37;    AtomId_Sr = 38;    AtomId_Y  = 39;    AtomId_Zr = 40;
 AtomId_Nb = 41;   AtomId_Mo = 42;    AtomId_Tc = 43;    AtomId_Ru = 44;    AtomId_Rh = 45;
 AtomId_Pd = 46;   AtomId_Ag = 47;    AtomId_Cd = 48;    AtomId_In = 49;    AtomId_Sn = 50;
 AtomId_Sb = 51;   AtomId_Te = 52;    AtomId_I  = 53;    AtomId_Xe = 54;    AtomId_Cs = 55;
 AtomId_Ba = 56;   AtomId_La = 57;    AtomId_Ce = 58;    AtomId_Pr = 59;    AtomId_Nd = 60;
 AtomId_Pm = 61;   AtomId_Sm = 62;    AtomId_Eu = 63;    AtomId_Gd = 64;    AtomId_Tb = 65;
 AtomId_Dy = 66;   AtomId_Ho = 67;    AtomId_Er = 68;    AtomId_Tm = 69;    AtomId_Yb = 70;
 AtomId_Lu = 71;   AtomId_Hf = 72;    AtomId_Ta = 73;    AtomId_W  = 74;    AtomId_Re = 75;
 AtomId_Os = 76;   AtomId_Ir = 77;    AtomId_Pt = 78;    AtomId_Au = 79;    AtomId_Hg = 80;
 AtomId_Tl = 81;   AtomId_Pb = 82;    AtomId_Bi = 83;    AtomId_Po = 84;    AtomId_At = 85;
 AtomId_Rn = 86;   AtomId_Fr = 87;    AtomId_Ra = 88;    AtomId_Ac = 89;    AtomId_Th = 90;
 AtomId_Pa = 91;   AtomId_U  = 92;    AtomId_Np = 93;    AtomId_Pu = 94;    AtomId_Am = 95;
 AtomId_Cm = 96;   AtomId_Bk = 97;    AtomId_Cf = 98;    AtomId_Es = 99;    AtomId_Fm = 100;
 AtomId_Md = 101;  AtomId_No = 102;   AtomId_Lr = 103;   AtomId_Ku = 104;   AtomId_Ns = 105;
 RfaMinAtomId = AtomId_H;
 RfaMaxAtomId = AtomId_Ns;

 ///////////////////////////////////////////////////////////////////////////////
 // Идентификаторы полей базы данных РФА                                      //
 // Линии ХРИ идентифицируются по полю энергии линии, например, rf_EKA1.      //
 // Следующим элементом всегда идет интенсивность линии, например,  rf_IKA1.  //
 ///////////////////////////////////////////////////////////////////////////////
type
 TRfaFields=(                                           // Поля базы данных РФА:
  rf_ATOM_ID,        rf_ATOM_SYMBOL,                    // Номер и символ элемента
  rf_ATOM_NAME_RUS,  rf_ATOM_NAME_ENG,                  // Название на русском/английском
  rf_ATOM_GROUP,     rf_ATOM_PERIOD,                    // Группа и период
  rf_ATOM_COLOR,     rf_CROSS_SECTION,                  // Цвет и таблица сечений
  rf_ATOM_WEIGHT,    rf_ATOM_DENSITY,                   // Атомный вес и плотность
  rf_EKAB,  rf_EL1AB,  rf_EL2AB,  rf_EL3AB,             // Энергии абсорбции для K,L1,L2,L3-уровней линий ХРИ
  rf_EKA,   rf_IKA,                                     // Линия ХРИ K a1 + a2
  rf_EKA1,  rf_IKA1,                                    // Линия ХРИ K a1
  rf_EKA2,  rf_IKA2,                                    // Линия ХРИ K a2
  rf_EKB,   rf_IKB,                                     // Линия ХРИ K b1 + b2
  rf_EKB1,  rf_IKB1,                                    // Линия ХРИ K b1
  rf_EKB2,  rf_IKB2,                                    // Линия ХРИ K b2
  rf_EKB3,  rf_IKB3,                                    // Линия ХРИ K b3
  rf_ELA12, rf_ILA12,                                   // Линия ХРИ L3 a1 + a2
  rf_ELA1,  rf_ILA1,                                    // Линия ХРИ L3 a1
  rf_ELA2,  rf_ILA2,                                    // Линия ХРИ L3 a2
  rf_ELB1,  rf_ILB1,                                    // Линия ХРИ L2 b1
  rf_ELB2,  rf_ILB2,                                    // Линия ХРИ L3 b2
  rf_ELB3,  rf_ILB3,                                    // Линия ХРИ L1 b3
  rf_ELB4,  rf_ILB4,                                    // Линия ХРИ L1 b4
  rf_ELB5,  rf_ILB5,                                    // Линия ХРИ L3 b5
  rf_ELB6,  rf_ILB6,                                    // Линия ХРИ L3 b6
  rf_ELB10, rf_ILB10,                                   // Линия ХРИ L1 b10
  rf_ELB15, rf_ILB15,                                   // Линия ХРИ L3 b15
  rf_ELB17, rf_ILB17,                                   // Линия ХРИ L2 b17
  rf_ELG1,  rf_ILG1,                                    // Линия ХРИ L2 g1
  rf_ELG2,  rf_ILG2,                                    // Линия ХРИ L1 g2
  rf_ELG3,  rf_ILG3,                                    // Линия ХРИ L1 g3
  rf_ELG4,  rf_ILG4,                                    // Линия ХРИ L1 g4
  rf_ELG4_, rf_ILG4_,                                   // Линия ХРИ L1 g4_
  rf_ELG5,  rf_ILG5,                                    // Линия ХРИ L2 g5
  rf_ELG6,  rf_ILG6,                                    // Линия ХРИ L2 g6
  rf_ELG8,  rf_ILG8,                                    // Линия ХРИ L2 g6
  rf_ELG11, rf_ILG11,                                   // Линия ХРИ L1 g11
  rf_ELmu,  rf_ILmu,                                    // Линия ХРИ L2 Mu
  rf_ELnu,  rf_ILnu,                                    // Линия ХРИ L2 Nu
  rf_ELL,   rf_ILL,                                     // Линия ХРИ L3 L
  rf_RSRV01, rf_RSRV02, rf_RSRV03, rf_RSRV04,           // Зарезервировано на будущее
  rf_RSRV05, rf_RSRV06, rf_RSRV07, rf_RSRV08,           // Зарезервировано на будущее
  rf_RSRV09, rf_RSRV10, rf_RSRV11, rf_RSRV12,           // Зарезервировано на будущее
  rf_RSRV13, rf_RSRV14, rf_RSRV15, rf_RSRV16,           // Зарезервировано на будущее
  rf_RSRV17, rf_RSRV18, rf_RSRV19, rf_RSRV20,           // Зарезервировано на будущее
  rf_RSRV21, rf_RSRV22, rf_RSRV23, rf_RSRV24,           // Зарезервировано на будущее
  rf_RSRV25, rf_RSRV26, rf_RSRV27, rf_RSRV28,           // Зарезервировано на будущее
  rf_RSRV29, rf_RSRV30, rf_RSRV31, rf_RSRV32,           // Зарезервировано на будущее
  rf_RSRV33, rf_RSRV34, rf_RSRV35, rf_RSRV36,           // Зарезервировано на будущее
  rf_RSRV37, rf_RSRV38, rf_RSRV39, rf_RSRV40,           // Зарезервировано на будущее
  rf_RSRV41, rf_RSRV42, rf_RSRV43, rf_RSRV44,           // Зарезервировано на будущее
  rf_RSRV45, rf_RSRV46, rf_RSRV47, rf_RSRV48,           // Зарезервировано на будущее
  rf_RSRV49, rf_RSRV50, rf_RSRV51, rf_RSRV52,           // Зарезервировано на будущее
  rf_RSRV53, rf_RSRV54, rf_RSRV55, rf_RSRV56,           // Зарезервировано на будущее
  rf_RSRV57, rf_RSRV58, rf_RSRV59, rf_RSRV60,           // Зарезервировано на будущее
  rf_RSRV61, rf_RSRV62, rf_RSRV63, rf_RSRV64,           // Зарезервировано на будущее
  rf_RSRV65, rf_RSRV66, rf_RSRV67, rf_RSRV68,           // Зарезервировано на будущее
  rf_RSRV69, rf_RSRV70, rf_RSRV71, rf_RSRV72,           // Зарезервировано на будущее
  rf_RSRV73, rf_RSRV74, rf_RSRV75, rf_RSRV76,           // Зарезервировано на будущее
  rf_RSRV77, rf_RSRV78, rf_RSRV79, rf_RSRV80,           // Зарезервировано на будущее
  rf_RSRV81, rf_RSRV82, rf_RSRV83, rf_RSRV84,           // Зарезервировано на будущее
  rf_RSRV85, rf_RSRV86, rf_RSRV87, rf_RSRV88,           // Зарезервировано на будущее
  rf_RSRV89, rf_RSRV90, rf_RSRV91, rf_RSRV92,           // Зарезервировано на будущее
  rf_RSRV93, rf_RSRV94, rf_RSRV95, rf_RSRV96,           // Зарезервировано на будущее
  rf_RSRV97, rf_RSRV98, rf_RSRV99, rf_RSRV100,          // Зарезервировано на будущее
  rf_KEMISSION,   rf_L31EMISSION, rf_L32EMISSION,       // Факторы эмиссии линий ХРИ
  rf_L33EMISSION, rf_L21EMISSION, rf_L22EMISSION,       // Факторы эмиссии линий ХРИ
  rf_ATOMS_MM1,                                         // атомная масса наиболее распространенного изотопа
  rf_ATOMS_M1,                                          // атомный вес наиболее распространенного изотопа
  rf_ATOMS_M2,                                          // атомный вес естественной смеси изотопов
  rf_ATOMS_RHO,                                         // плотность г/куб.см
  rf_ATOMS_ATRHO,                                       // атомная плотность, 10**22 атом/куб.см
  rf_ATOMS_VFERMI,                                      // скорость Ферми в единицах скорости Бора
  rf_ATOMS_LFCTR,                                       // множитель длины экранирования
  rf_STRAG_S1, rf_STRAG_S2, rf_STRAG_S3, rf_STRAG_S4,   // Коэффициенты для вычисления страгглинга
  rf_PSTOP_P1, rf_PSTOP_P2, rf_PSTOP_P3, rf_PSTOP_P4,   // Коэффициенты протонного торможения
  rf_PSTOP_P5, rf_PSTOP_P6, rf_PSTOP_P7, rf_PSTOP_P8,   // Коэффициенты протонного торможения
  rf_ELEMPOS_X, rf_ELEMPOS_Y,                           // Положения кнопок для отображения таблицы Менделеева
  rf_INVALID_FIELD                                      // Такого поля быть не должно
 );
 TRfaFieldsSet = set of TRfaFields;                     // Для наборов полей базы данных
 TRfaCrossSectionFields = (                             // Поля таблиц сечений
  rf_ENERGY,                                            // Энергия в кэв
  rf_PHOTO,                                             // Сечение фотоэффекта
  rf_COHER,                                             // Сечение когерентного рассеяния
  rf_INCOHER,                                           // Сечение некогерентного рассеяния
  rf_TOTAL                                              // Полное сечение
 );
 TXRayLine = packed record Energy,Height:Double; end;   // Описывает линию ХРИ

const
 rf_FIRST_FIELD        = Low(TRfaFields);               // Первое допустимое поле
 rf_LAST_FIELD         = Pred(rf_INVALID_FIELD);        // Последнее допустимое поле

 //
 // Различные наборы, идентифицирующие линии ХРИ.
 // Shorten - наборы линий, где тонкая структура линий сжата, например, Ka вместо Ka1,Ka2.
 // Perfect - наборы линий, где тонкая структура линий учитывается, например, Ka1,Ka2 вместо Ka.
 // Без индекса - наборы всех линий, содержат дублирование, например, Ka и Ka1,Ka2.
 //
const
 rf_K_Series_Shorten   = [rf_EKA,            rf_EKB];
 rf_K_Series_Perfect   = [rf_EKA1, rf_EKA2,  rf_EKB1, rf_EKB2, rf_EKB3];
 rf_L1_Series_Shorten  = [rf_ELB3, rf_ELB4,  rf_ELB10, rf_ELG2, rf_ELG3, rf_ELG4, rf_ELG4_, rf_ELG11];
 rf_L1_Series_Perfect  = [rf_ELB3, rf_ELB4,  rf_ELB10, rf_ELG2, rf_ELG3, rf_ELG4, rf_ELG4_, rf_ELG11];
 rf_L2_Series_Shorten  = [rf_ELB1, rf_ELB17, rf_ELG1, rf_ELG5, rf_ELG6, rf_ELG8, rf_ELmu, rf_ELnu];
 rf_L2_Series_Perfect  = [rf_ELB1, rf_ELB17, rf_ELG1, rf_ELG5, rf_ELG6, rf_ELG8, rf_ELmu, rf_ELnu];
 rf_L3_Series_Shorten  = [rf_ELA12,          rf_ELB2, rf_ELB5, rf_ELB6, rf_ELB15, rf_ELL];
 rf_L3_Series_Perfect  = [rf_ELA1,  rf_ELA2, rf_ELB2, rf_ELB5, rf_ELB6, rf_ELB15, rf_ELL];
 rf_K_Series           = rf_K_Series_Shorten  + rf_K_Series_Perfect;
 rf_L1_Series          = rf_L1_Series_Shorten + rf_L1_Series_Perfect;
 rf_L2_Series          = rf_L2_Series_Shorten + rf_L2_Series_Perfect;
 rf_L3_Series          = rf_L3_Series_Shorten + rf_L3_Series_Perfect;
 rf_L_Series_Shorten   = rf_L1_Series_Shorten + rf_L2_Series_Shorten + rf_L3_Series_Shorten;
 rf_L_Series_Perfect   = rf_L1_Series_Perfect + rf_L2_Series_Perfect + rf_L3_Series_Perfect;
 rf_L_Series           = rf_L_Series_Shorten + rf_L_Series_Perfect;
 rf_All_Series_Shorten = rf_K_Series_Shorten + rf_L_Series_Shorten;
 rf_All_Series_Perfect = rf_K_Series_Perfect + rf_L_Series_Perfect;
 rf_All_Series         = rf_K_Series + rf_L_Series;

 ///////////////////////////////////////////////////////////////////////////////
 // Объект инкапсулирует функции для доступа к базе данных РФА.               //
 ///////////////////////////////////////////////////////////////////////////////
type
 TRfaTable = class(TMasterObject)
 private
  myActive       : Boolean;
  myTable        : packed array[RfaMinAtomId..RfaMaxAtomId] of packed record
   CROSS_SECTION : TDoubleMatrix;
   ATOM_NAME_ENG : string[12];
   ATOM_NAME_RUS : string[32];
   ATOM_SYMBOL   : string[2];
   ATOM_GROUP    : string[2];
   FIELDS        : packed array[TRfaFields] of Single;
  end;
  procedure ClearAll;
  function  GetActive:Boolean;
  function  GetFeasible(AtomId:Integer):Boolean;
  function  GetAccessible(AtomId:Integer):Boolean;
  function  GetAsDouble(AtomId:Integer; FieldId:TRfaFields):Double;
  function  GetAsString(AtomId:Integer; FieldId:TRfaFields):LongString;
  function  GetXRayLine(AtomId:Integer; LineId:TRfaFields):TXRayLine;
  function  GetAtomSymbol(AtomId:Integer):LongString;
  function  GetFieldSection(FieldId:TRfaFields):LongString;
  function  GetFieldName(FieldId:TRfaFields):LongString;
  function  GetLineName(LineId:TRfaFields):LongString;
  function  GetAtomColor(AtomID:Integer):TColor;
  function  GetCrossSectionTableCount(AtomId:Integer):Integer;
  function  GetCrossSectionTable(AtomId:Integer; aChan:TRfaCrossSectionFields; aIndex:Integer):Double;
 public
  constructor Create;
  destructor  Destroy; override;
              // Возвращает True, если объект создан и база данных загружена.
  property    Active                                      : Boolean     read GetActive;
              // Возвращает True, если атомный номер довустим, независимо от того, загружена ли база данных.
  property    Feasible[AtomId:Integer]                    : Boolean     read GetFeasible;
              // Возвращает True, если атомный номер доступен, то есть если он допустим и база данных загружена.
  property    Accessible[AtomId:Integer]                  : Boolean     read GetAccessible;
              // Дает доступ к полям базы данных как к числу; нечисловые поля возвращают длину строки/таблицы.
  property    AsDouble[AtomId:Integer;FieldId:TRfaFields] : Double      read GetAsDouble; default;
              // Дает доступ к полям базы данных как к строкам; числовые поля преобразуются в строку.
  property    AsString[AtomId:Integer;FieldId:TRfaFields] : LongString read GetAsString;
              // Возвращает энергию и интенсивность линии ХРИ. Возвращает ноль, если LineId неверное.
  property    XRayLine[AtomId:Integer;LineId:TRfaFields]  : TXRayLine   read GetXRayLine;
              // Возвращает химический символ атома.
  property    AtomSymbol[AtomId:Integer]                  : LongString read GetAtomSymbol;
              // Возвращает символьное имя поля.
  property    FieldName[FieldId:TRfaFields]               : LongString read GetFieldName;
              // Возвращает название линии.
  property    LineName[LineId:TRfaFields]                 : LongString read GetLineName;
              // Возвращает цвет для рисования элемента.
  property    AtomColor[AtomId:Integer]                   : TColor      read GetAtomColor;
              // Возвращает длину таблицы сечений для данного элемента.
  property    CrossSectionTableCount[AtomId:Integer]      : Integer     read GetCrossSectionTableCount;
              // Дает доступ к таблице сечений.
  property    CrossSectionTable[AtomId:Integer; Chan:TRfaCrossSectionFields; aIndex:Integer] : Double  read GetCrossSectionTable;
              // Находит номер атома по его химическому символу.
  function    FindAtom(const aSymbol:LongString):Integer;
              // Находит линию ХРИ по ее названию.
  function    FindLine(const aLineName:LongString):TRfaFields;
              // Вычисляет сечение, используя логарифмическую интерполяцию таблицы сечений.
  function    FindCrossSection(AtomID:Integer; CrossId:TRfaCrossSectionFields; Energy:Double):Double;
              // Загружает базу данных.
  function    Load(IniFile:LongString):Boolean;
 end;

 ///////////////////////////////////////////////////////////////////////////////
 // Доступ, инициализация и завершение базы данных подсистемы Rfa             //
 ///////////////////////////////////////////////////////////////////////////////
function  Rfa:TRfaTable;
procedure Init_Rfa_SubSystem;
procedure Done_Rfa_SubSystem;

IMPLEMENTATION

function NormalizeName(const Name:LongString):LongString;
var s:WideString;
begin
 s:=StrToWide(Trim(Name));
 s:=UpperCase(Copy(s,1,1))+LowerCase(Copy(s,2,Length(s)-1));
 Result:=WideToStr(s); s:='';
end;

 //////////////////////////////
 // TRfaTable implementation //
 //////////////////////////////
constructor TRfaTable.Create;
begin
 inherited;
 myActive:=false;
 SafeFillChar(myTable,sizeof(myTable),0);
 ClearAll;
end;

destructor TRfaTable.Destroy;
begin
 ClearAll;
 inherited;
end;

procedure TRfaTable.ClearAll;
var
 AtomId : Integer;
begin
 myActive:=false;
 for AtomId:=Low(myTable) to High(myTable) do with myTable[AtomId] do begin
  Kill(CROSS_SECTION);
  ATOM_NAME_ENG:='';
  ATOM_NAME_RUS:='';
  ATOM_SYMBOL:='';
  ATOM_GROUP:='';
  SafeFillChar(FIELDS,sizeof(FIELDS),0);
  FIELDS[rf_ATOM_ID]:=AtomId;
 end;
end;

function TRfaTable.GetActive:Boolean;
begin
 Result:=Assigned(Self) and myActive;
end;

function TRfaTable.GetFeasible(AtomId:Integer):Boolean;
begin
 Result:=AtomId in [Low(myTable)..High(myTable)];
end;

function TRfaTable.GetAccessible(AtomId:Integer):Boolean;
begin
 Result:=Active and Feasible[AtomId];
end;

function  TRfaTable.GetAsDouble(AtomId:Integer; FieldId:TRfaFields):Double;
begin
 if Accessible[AtomId] and (FieldId in [rf_FIRST_FIELD..rf_LAST_FIELD])
 then Result:=myTable[AtomId].FIELDS[FieldId]
 else Result:=0;
end;

function TRfaTable.GetAsString(AtomId:Integer; FieldId:TRfaFields):LongString;
begin
 Result:='';
 if Accessible[AtomId] then with myTable[AtomId] do
 case FieldId of
  rf_ATOM_ID                    : Result:=Format('%.0f',[FIELDS[FieldId]]);
  rf_ATOM_SYMBOL                : Result:=NormalizeName(ATOM_SYMBOL);
  rf_ATOM_NAME_RUS              : Result:=NormalizeName(ATOM_NAME_RUS);
  rf_ATOM_NAME_ENG              : Result:=NormalizeName(ATOM_NAME_ENG);
  rf_ATOM_GROUP                 : Result:=NormalizeName(ATOM_GROUP);
  rf_ATOM_PERIOD                : Result:=Format('%.0f',[FIELDS[FieldId]]);
  rf_ATOM_COLOR                 : Result:=Format('%.0f',[FIELDS[FieldId]]);
  rf_CROSS_SECTION              : Result:=Format('%.0f',[FIELDS[FieldId]]);
  rf_ATOM_WEIGHT..rf_LAST_FIELD : Result:=Format('%.7g',[FIELDS[FieldId]]);
 end;
end;

function TRfaTable.GetXRayLine(AtomId:Integer; LineId:TRfaFields):TXRayLine;
begin
 with Result do
 if Accessible[AtomId] and (LineId in rf_All_Series) then begin
  Energy:=myTable[AtomID].FIELDS[LineId];
  Height:=myTable[AtomID].FIELDS[Succ(LineId)];
  if Energy<=0 then Height:=0;
 end else begin
  Energy:=0;
  Height:=0;
 end;
end;

function TRfaTable.GetAtomSymbol(AtomId:Integer):LongString;
begin
 Result:=AsString[AtomID,rf_ATOM_SYMBOL];
end;

function TRfaTable.GetFieldSection(FieldId:TRfaFields):LongString;
begin
 case FieldId of
  rf_ATOM_ID..rf_L22EMISSION   : Result:=UnifySection(FieldName[FieldId]);
  rf_ATOMS_MM1..rf_ATOMS_LFCTR : Result:=UnifySection('ATOMS.ZBL');
  rf_STRAG_S1..rf_STRAG_S4     : Result:=UnifySection('STRAG.YOW');
  rf_PSTOP_P1..rf_PSTOP_P8     : Result:=UnifySection('PSTOP.ZBL');
  rf_ELEMPOS_X..rf_ELEMPOS_Y   : Result:=UnifySection('ELEMPOS');
  else                           Result:='';
 end;
end;

function TRfaTable.GetFieldName(FieldId:TRfaFields):LongString;
begin
 Result:='';
 case FieldId of
  rf_ATOM_ID       : Result:='ATOM_ID';
  rf_ATOM_SYMBOL   : Result:='ATOM_SYMBOL';
  rf_ATOM_NAME_RUS : Result:='ATOM_NAME_RUS';
  rf_ATOM_NAME_ENG : Result:='ATOM_NAME_ENG';
  rf_ATOM_GROUP    : Result:='ATOM_GROUP';
  rf_ATOM_PERIOD   : Result:='ATOM_PERIOD';
  rf_ATOM_COLOR    : Result:='ATOM_COLOR';
  rf_CROSS_SECTION : Result:='CROSS_SECTION';
  rf_ATOM_WEIGHT   : Result:='ATOM_WEIGHT';
  rf_ATOM_DENSITY  : Result:='ATOM_DENSITY';
  rf_EKAB          : Result:='EKAB';
  rf_EL1AB         : Result:='EL1AB';
  rf_EL2AB         : Result:='EL2AB';
  rf_EL3AB         : Result:='EL3AB';
  rf_EKA           : Result:='EKA';
  rf_IKA           : Result:='IKA';
  rf_EKA1          : Result:='EKA1';
  rf_IKA1          : Result:='IKA1';
  rf_EKA2          : Result:='EKA2';
  rf_IKA2          : Result:='IKA2';
  rf_EKB           : Result:='EKB';
  rf_IKB           : Result:='IKB';
  rf_EKB1          : Result:='EKB1';
  rf_IKB1          : Result:='IKB1';
  rf_EKB2          : Result:='EKB2';
  rf_IKB2          : Result:='IKB2';
  rf_EKB3          : Result:='EKB3';
  rf_IKB3          : Result:='IKB3';
  rf_ELA12         : Result:='ELA12';
  rf_ILA12         : Result:='ILA12';
  rf_ELA1          : Result:='ELA1';
  rf_ILA1          : Result:='ILA1';
  rf_ELA2          : Result:='ELA2';
  rf_ILA2          : Result:='ILA2';
  rf_ELB1          : Result:='ELB1';
  rf_ILB1          : Result:='ILB1';
  rf_ELB2          : Result:='ELB2';
  rf_ILB2          : Result:='ILB2';
  rf_ELB3          : Result:='ELB3';
  rf_ILB3          : Result:='ILB3';
  rf_ELB4          : Result:='ELB4';
  rf_ILB4          : Result:='ILB4';
  rf_ELB5          : Result:='ELB5';
  rf_ILB5          : Result:='ILB5';
  rf_ELB6          : Result:='ELB6';
  rf_ILB6          : Result:='ILB6';
  rf_ELB10         : Result:='ELB10';
  rf_ILB10         : Result:='ILB10';
  rf_ELB15         : Result:='ELB15';
  rf_ILB15         : Result:='ILB15';
  rf_ELB17         : Result:='ELB17';
  rf_ILB17         : Result:='ILB17';
  rf_ELG1          : Result:='ELG1';
  rf_ILG1          : Result:='ILG1';
  rf_ELG2          : Result:='ELG2';
  rf_ILG2          : Result:='ILG2';
  rf_ELG3          : Result:='ELG3';
  rf_ILG3          : Result:='ILG3';
  rf_ELG4          : Result:='ELG4';
  rf_ILG4          : Result:='ILG4';
  rf_ELG4_         : Result:='ELG4_';
  rf_ILG4_         : Result:='ILG4_';
  rf_ELG5          : Result:='ELG5';
  rf_ILG5          : Result:='ILG5';
  rf_ELG6          : Result:='ELG6';
  rf_ILG6          : Result:='ILG6';
  rf_ELG8          : Result:='ELG8';
  rf_ILG8          : Result:='ILG8';
  rf_ELG11         : Result:='ELG11';
  rf_ILG11         : Result:='ILG11';
  rf_ELmu          : Result:='ELMU';
  rf_ILmu          : Result:='ILMU';
  rf_ELnu          : Result:='ELNU';
  rf_ILnu          : Result:='ILNU';
  rf_ELL           : Result:='ELL';
  rf_ILL           : Result:='ILL';
  rf_KEMISSION     : Result:='KEMISSION';
  rf_L31EMISSION   : Result:='L31EMISSION';
  rf_L32EMISSION   : Result:='L32EMISSION';
  rf_L33EMISSION   : Result:='L33EMISSION';
  rf_L21EMISSION   : Result:='L21EMISSION';
  rf_L22EMISSION   : Result:='L22EMISSION';
  rf_ATOMS_MM1     : Result:='ATOMS_MM1';
  rf_ATOMS_M1      : Result:='ATOMS_M1';
  rf_ATOMS_M2      : Result:='ATOMS_M2';
  rf_ATOMS_RHO     : Result:='ATOMS_RHO';
  rf_ATOMS_ATRHO   : Result:='ATOMS_ATRHO';
  rf_ATOMS_VFERMI  : Result:='ATOMS_VFERMI';
  rf_ATOMS_LFCTR   : Result:='ATOMS_LFCTR';
  rf_STRAG_S1      : Result:='STRAG_S1';
  rf_STRAG_S2      : Result:='STRAG_S2';
  rf_STRAG_S3      : Result:='STRAG_S3';
  rf_STRAG_S4      : Result:='STRAG_S4';
  rf_PSTOP_P1      : Result:='PSTOP_P1';
  rf_PSTOP_P2      : Result:='PSTOP_P2';
  rf_PSTOP_P3      : Result:='PSTOP_P3';
  rf_PSTOP_P4      : Result:='PSTOP_P4';
  rf_PSTOP_P5      : Result:='PSTOP_P5';
  rf_PSTOP_P6      : Result:='PSTOP_P6';
  rf_PSTOP_P7      : Result:='PSTOP_P7';
  rf_PSTOP_P8      : Result:='PSTOP_P8';
  rf_ELEMPOS_X     : Result:='ELEMPOS_X';
  rf_ELEMPOS_Y     : Result:='ELEMPOS_Y';
 end;
end;

function TRfaTable.GetLineName(LineId:TRfaFields):LongString;
begin
 if LineId in rf_All_Series
 then Result:=NormalizeName(Copy(FieldName[LineId],2,25))
 else Result:='';
end;

function TRfaTable.GetAtomColor(AtomID:Integer):TColor;
begin
 case Round(Self[AtomID,rf_ATOM_COLOR]) of
  1  : Result:=RGB(255, 0,   0  );     // редкоземельные
  2  : Result:=RGB(0,   0,   170);     // металлы
  3  : Result:=RGB(0,   0,   0  );     // неметаллы
  4  : Result:=RGB(0,   127, 0  );     // лантаниды
  5  : Result:=RGB(127, 127, 0  );     // актиниды
  else Result:=clDkGray;               // элемент не в базе 
 end;
end;

function  TRfaTable.GetCrossSectionTableCount(AtomId:Integer):Integer;
begin
 if Accessible[AtomId]
 then Result:=myTable[AtomId].CROSS_SECTION.Columns
 else Result:=0;
end;

function TRfaTable.GetCrossSectionTable(AtomId:Integer; aChan:TRfaCrossSectionFields; aIndex:Integer):Double;
begin
 if Accessible[AtomId]
 then Result:=myTable[AtomId].CROSS_SECTION[ord(aChan),aIndex]
 else Result:=0;
end;

function TRfaTable.FindAtom(const aSymbol:LongString):Integer;
var
 AtomId : Integer;
begin
 Result:=0;
 if Active then
 if IsNonEmptyStr(aSymbol) then
 for AtomId:=RfaMinAtomId to RfaMaxAtomId do
 if Accessible[AtomId] then
 if SameText(UnifyAlias(aSymbol),UnifyAlias(AtomSymbol[AtomID])) then begin
  Result:=AtomId;
  Break;
 end;
end;

function TRfaTable.FindLine(const aLineName:LongString):TRfaFields;
var
 LineId : TRfaFields;
begin
 Result:=rf_INVALID_FIELD;
 if Active then
 if IsNonEmptyStr(aLineName) then
 for LineId:=Low(LineId) to High(LineId) do
 if SameText(UnifyAlias(LineName[LineId]),UnifyAlias(aLineName)) then begin
  Result:=LineId;
  Break;
 end;
end;

function TRfaTable.FindCrossSection(AtomID:Integer; CrossId:TRfaCrossSectionFields; Energy:Double):Double;
var
 X1 : Double;
 Y1 : Double;
 X2 : Double;
 Y2 : Double;
 i  : Integer;
 j  : Integer;
 k  : Integer;
begin
 Result:=0;
 if Accessible[AtomID] and (CrossSectionTableCount[AtomID]>1) then begin
  i:=0;
  j:=CrossSectionTableCount[AtomID]-1;
  while j>i+1 do begin
   k:=(i+j) shr 1;
   if Energy<CrossSectionTable[AtomID,rf_ENERGY,k] then j:=k else i:=k;
  end;
  X1:=CrossSectionTable[AtomID, rf_ENERGY, i];
  Y1:=CrossSectionTable[AtomID, CrossId,   i];
  X2:=CrossSectionTable[AtomID, rf_ENERGY, i+1];
  Y2:=CrossSectionTable[AtomID, CrossId,   i+1];
  if (X1>0) and (X2>0) and (Y1>0) and (Y2>0) and (Energy>0)
  then Result:=Max(0,exp(ln(Y1)+ln(Y2/Y1)*ln(Energy/X1)/ln(X2/X1)));
 end;
end;

function TRfaTable.Load(IniFile:LongString):Boolean;
var
 FieldId   : TRfaFields;
 SectionID : TRfaFields;
 Counters  : packed array[TRfaFields] of Integer;
 Expected  : packed array[TRfaFields] of Integer;
 f         : Text;
 w         : Word;
 rif       : packed record
  i        : Integer;
  f        : packed array[0..9] of Double;
 end;
 ris       : packed record
  i        : Integer;
  s        : PureString;
 end;
 s         : LongString;
 function Scan(const str,format:LongString; var data):Boolean;
 var
  buf : TParsingBuffer;
 begin
  Scan:=ScanVarRecord(svConfig,StrCopyBuff(buf,str),format,data)<>nil;
 end;
begin
 Result:=false;
 if Assigned(Self) then
 try
  // Очистка всех полей, инициализация счетчиков для проверки чтения таблиц...
  // Учитывается, что некоторых таблиц сечений нет...
  ClearAll;
  for FieldID:=rf_FIRST_FIELD to rf_LAST_FIELD do Counters[FieldId]:=0;
  for FieldID:=rf_FIRST_FIELD to rf_LAST_FIELD do Expected[FieldId]:=RfaMaxAtomId*ord(Length(FieldName[FieldId])>0);
  Expected[rf_CROSS_SECTION]:=87;
  // Проверить и открыть файл...
  IniFile:=UnifyFileAlias(IniFile);
  if not FileExists(IniFile)
  then Raise EReadError.CreateFmt('File "%s" not found.',[IniFile]);
  Assign(f,IniFile);
  try
   Reset(f);
   SectionID:=rf_INVALID_FIELD;
   while not eof(f) and (IOResult=0) do begin
    // Прочитать строку, удалить комментарий и пробелы
    Readln(f,s);
    if pos(';',s)>0 then Delete(s,pos(';',s),length(s)-pos(';',s)+1);
    s:=Trim(s);
    s:=UpCaseStr(s);
    // Идентификация секции
    if StrFetch(s,1)='[' then begin
     SectionId:=rf_INVALID_FIELD;
     for FieldId:=rf_FIRST_FIELD to rf_LAST_FIELD do
     if s=Rfa.GetFieldSection(FieldId) then begin
      SectionId:=FieldId;
      Break;
     end;
     Continue;
    end;
    if not (SectionId in [rf_FIRST_FIELD..rf_LAST_FIELD]) then Continue;
    FieldId:=SectionId;
    SafeFillChar(rif,SizeOf(rif),0);
    SafeFillChar(ris,SizeOf(ris),0);
    case FieldId of
     // Поля ATOM_ID..ATOM_COLOUR находятся в одноименных секциях
     // 1 столбец - номер элемента
     // 2 столбец - значение параметра
     rf_ATOM_ID..rf_ATOM_COLOR:
      if Scan(s,'%i;%a',ris) then
      if Feasible[ris.i] then with myTable[ris.i] do
      case FieldId of
       rf_ATOM_ID        : if Str2Word(ris.s,w) and (w=ris.i)
                           then inc(Counters[FieldId]);
       rf_ATOM_SYMBOL    : begin
                            ATOM_SYMBOL:=ris.s;
                            FIELDS[FieldId]:=Length(ATOM_SYMBOL);
                            inc(Counters[FieldId]);
                           end;
       rf_ATOM_NAME_RUS  : begin
                            ATOM_NAME_RUS:=ris.s;
                            FIELDS[FieldId]:=Length(ATOM_NAME_RUS);
                            inc(Counters[FieldId]);
                           end;
       rf_ATOM_NAME_ENG  : begin
                            ATOM_NAME_ENG:=ris.s;
                            FIELDS[FieldId]:=Length(ATOM_NAME_ENG);
                            inc(Counters[FieldId]);
                           end;
       rf_ATOM_GROUP     : begin
                            ATOM_GROUP:=ris.s;
                            FIELDS[FieldId]:=Length(ATOM_GROUP);
                            inc(Counters[FieldId]);
                           end;
       rf_ATOM_PERIOD    : if Str2Word(ris.s,w) then begin
                            FIELDS[FieldId]:=w;
                            inc(Counters[FieldId]);
                           end;
       rf_ATOM_COLOR     : if Str2Word(ris.s,w) then begin
                            FIELDS[FieldId]:=w;
                            inc(Counters[FieldId]);
                           end;
      end;
     // Таблица сечений элемента Z находится в секции CROSS_SECTION
     // 1 столбец - ATOM_ID
     // 2 столбец - ENERGY
     // 3 столбец - PHOTO
     // 4 столбец - COHER
     // 5 столбец - INCOHER
     // 6 столбец - TOTAL
     rf_CROSS_SECTION:
      if Scan(s,'%i;%f;%f;%f;%f;%f',rif) then
      if Feasible[rif.i] then with myTable[rif.i] do begin
       if Assigned(CROSS_SECTION) then begin
        CROSS_SECTION.Columns:=CROSS_SECTION.Columns+1;
       end else begin
        CROSS_SECTION:=NewDoubleMatrix(ord(High(TRfaCrossSectionFields))-
                                       ord(Low(TRfaCrossSectionFields))+1,1);
        CROSS_SECTION.Master:=@CROSS_SECTION;
        CROSS_SECTION.Exceptions:=false;
        inc(Counters[rf_CROSS_SECTION]);
       end;
       CROSS_SECTION[ord(rf_Energy),  CROSS_SECTION.Columns-1]:=rif.f[0];
       CROSS_SECTION[ord(rf_PHOTO),   CROSS_SECTION.Columns-1]:=rif.f[1];
       CROSS_SECTION[ord(rf_COHER),   CROSS_SECTION.Columns-1]:=rif.f[2];
       CROSS_SECTION[ord(rf_INCOHER), CROSS_SECTION.Columns-1]:=rif.f[3];
       CROSS_SECTION[ord(rf_TOTAL),   CROSS_SECTION.Columns-1]:=rif.f[4];
       FIELDS[FieldId]:=CROSS_SECTION.Columns;
      end;
     // Поля ATOM_WEIGHT..L22EMISSION находятся в одноименных секциях
     // 1 столбец - номер элемента
     // 2 столбец - значение параметра
     rf_ATOM_WEIGHT..rf_L22EMISSION:
      if Scan(s,'%i;%f',rif) then
      if Feasible[rif.i] then with myTable[rif.i] do begin
       FIELDS[FieldId]:=rif.f[0];
       inc(Counters[FieldId]);
      end;
     // Поля ATOMS_MM1..ATOMS_LFCTR находятся в секции ATOMS.ZBL
     // 1 столбец - номер элемента
     // 2..8 столбец - значения параметров
     rf_ATOMS_MM1:
      if Scan(s,'%i;%f;%f;%f;%f;%f;%f;%f',rif) then
      if Feasible[rif.i] then with myTable[rif.i] do begin
       for FieldId:=rf_ATOMS_MM1 to rf_ATOMS_LFCTR do begin
        FIELDS[FieldId]:=rif.f[ord(FieldId)-ord(rf_ATOMS_MM1)];
        inc(Counters[FieldId]);
       end;
       {коррекция единиц AtRho}
       FIELDS[rf_ATOMS_ATRHO]:=FIELDS[rf_ATOMS_ATRHO]*1.0e22;
      end;
     // Поля STRAG_S1..STRAG_S4 находятся в секции STRAG.YOW
     // 1 столбец - номер элемента
     // 2..5 столбец - значения параметров
     rf_STRAG_S1:
      if Scan(s,'%i;%f;%f;%f;%f',rif) then
      if Feasible[rif.i] then with myTable[rif.i] do
      for FieldId:=rf_STRAG_S1 to rf_STRAG_S4 do begin
       FIELDS[FieldId]:=rif.f[ord(FieldId)-ord(rf_STRAG_S1)];
       inc(Counters[FieldId]);
      end;
     // Поля PSTOP_P1..PSTOP_P8 находятся в секции PSTOP.ZBL
     // 1 столбец - номер элемента
     // 2..9 столбец - значения параметров
     rf_PSTOP_P1:
      if Scan(s,'%i;%f;%f;%f;%f;%f;%f;%f;%f',rif) then
      if Feasible[rif.i] then with myTable[rif.i] do
      for FieldId:=rf_PSTOP_P1 to rf_PSTOP_P8 do begin
       FIELDS[FieldId]:=rif.f[ord(FieldId)-ord(rf_PSTOP_P1)];
       inc(Counters[FieldId]);
      end;
     // Поля ELEMPOS находятся в секции ELEMPOS
     // 1 столбец - номер элемента
     // 2..3 столбец - значения параметров
     rf_ELEMPOS_X:
      if Scan(s,'%i;%f;%f',rif) then
      if Feasible[rif.i] then with myTable[rif.i] do begin
       FIELDS[rf_ELEMPOS_X]:=rif.f[0];
       inc(Counters[rf_ELEMPOS_X]);
       FIELDS[rf_ELEMPOS_Y]:=rif.f[1];
       inc(Counters[rf_ELEMPOS_Y]);
      end;
    end;
   end;
  finally
   Close(f);
  end;
  {
  Проверить что все поля прочитаны
  }
  for FieldId:=rf_FIRST_FIELD to rf_LAST_FIELD do
  if Counters[FieldId]<Expected[FieldId]
  then Raise EReadError.CreateFmt('Error read "%s" field.',[FieldName[FieldId]]);
  {
  Теперь все хорошо
  }
  myActive:=true;
  Result:=Active;
 except
  on E:Exception do BugReport(E,Self,'Load');
 end;
end;

 ///////////////////////////////////////////////////////////////////////////////
 // Доступ, инициализация и завершение базы данных подсистемы Rfa             //
 ///////////////////////////////////////////////////////////////////////////////
const
 TheRfa : TRfaTable = nil;

function Rfa:TRfaTable;
begin
 if not Assigned(TheRfa) then begin
  TheRfa:=TRfaTable.Create;
  TheRfa.Master:=@TheRfa;
 end;
 Result:=TheRfa;
end;

procedure Init_Rfa_SubSystem;
begin
 if SubSystemTurnOn('[Rfa]') then begin
  UpdateStatusLine(RusEng('Инициализирую RFA...','Initialize RFA...'));
  DebugOut(stdfDebug,'');
  DebugOut(stdfDebug,'Init RFA database.');
  DebugOut(stdfDebug,'******************');
  if Rfa.Load(SubSystemIniFile('[Rfa]')) then begin
   Echo(RusEng('База данных RFA : Загружена.','RFA database : Ok.'));
   DebugOut(stdfDebug,'Load success '+SubSystemIniFile('[Rfa]'))
  end else begin
   Echo(RusEng('База данных RFA : Сбой.','RFA database : Fails.'));
   DebugOut(stdfDebug,'Load failure '+SubSystemIniFile('[Rfa]'));
  end;
 end;
end;

procedure Done_Rfa_SubSystem;
begin
 if Rfa.Ok then begin
  UpdateStatusLine(RusEng('Завершаю RFA...','Finalize RFA...'));
  Rfa.Free;
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_rfadata;
begin
end;

procedure Free_crw_rfadata;
begin
 Kill(TObject(TheRfa));
end;

initialization

 Init_crw_rfadata;

finalization

 Free_crw_rfadata;

end.

//////////////
// END OF FILE
//////////////

