////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Модуль идентификации пиков для спектрометрии                               //
// Автор: Чулков Виталий                                                      //
// Модифицировал: Курякин А.В.                                                //
//  00-02-05 Ввел защиту по границам массивов                                 //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231201 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit _crw_peaksearch; // Peak Search

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 _crw_alloc, _crw_fpu, _crw_rtc,
 _crw_str, _crw_fio, _crw_dynar;

 {
 Исключение в случае ошибки индекса в процедуре поиска пиков. 
 }
type
 EPeakInFailure = class(ESoftException);

 {
 Тип для функций калибровки
 }
type
 TCalibFun = function(x:Double; Custom:Pointer):Double;

 {
 Функция поиска пиков
 ********************
 Организованы два прохода спектра
 Используется фильтр, предназначенный для идентификации слабых пиков
 Спектр просматривается через Count - первый раз с конца(Znak=-1),
 затем с начала спектра. Учитывается влияние соседних пиков, расчетный
 спектр накапливается в C_Peak.
 Во время сканирования по спектру осуществляется динамическое
 сжатие по числу точек Nps из условия, что ПШПВ должна быть равна 5.
 Пики идентифицируются по первой производной от исходного спектра,
 а отбираются по критерию Стьюдента для производных скорректированного
 спектра, порог Tinp, а также по площадям и амплитудам.
 Производные в окрестности пика (IMin-Znak, IMax+Znak) сохраняются
 в массиве Deriv.
 На входе :
 **********
  Tinp           - статистический порог идентификации, обычно 3-6;
  Ns,Nend        - начальный и конечный каналы интервала спектра;
  MaxPeak        - размер выходных массивов;
  FwMin,FwMax    - коэффициены при калибровочном значении ПШПВ.
                   При умножении получаются допустимые интервалы изменения ПШПВ,
                   используемые для контроля, если <=0, то нет контроля.
  Delta_Peak_Old - Критерий совпадения двух пиков, если они ближе
                   Delta_Peak_Old*FWHMK, то пик один и тот же.
  Tkf,Tkp        - Коэффициенты определяются при настройке:
                   если ST < Tkf*tinp, то безусловно фон;
                   если ST > Tkf*tinp, то безусловно пик.
  DeltaFon_Gauss - Коэффициент к стастистической погрешности фона, если
                   вклад от пика меньше этой величины, то счет принимается
                   за фон.
 На выходе:
 **********
  Numpks         - число найденных пиков
  Peak           - центры пиков
  St             - статистические значимости
  Ampl           - амплитуды пиков
  Fon            - фон под пиком
  Fwhm           - ширины на полувысоте
  Gran_L,Fon_L   - массивы признаков левой границы комплекса
                   если Gran_L>0 и Fon_L>0, то это левая грань комплекса
  Gran_R,Fon_R   - массивы признаков правой границы комплекса
                   если Gran_R>0 и Fon_L>0, то это правая грань комплекса
                   комплексы объединяют группы связанных пиков, которые
                   должны описываться и фитироваться вместе (мультиплеты)
 Внутри:
 *******
  Gaus           - Счет в идентифицированных пиках.
  C_Peak         - Расчетный спектр, только пики.
 **************************************************************************
 }
function PeakIn(
     {
     входные данные
     }
     MaxCount   : integer;           {размер массивов спектра}
     MaxPeak    : integer;           {размер массивов пиков}
     Spectr     : TDoubleVector;     {0..MaxCount-1,массив спектра}
     Nstart     : integer;           {0..MaxCount-1,начальный канал спектра}
     Nend       : integer;           {0..MaxCount-1,конечный канал спектра}
     EvsChan    : TCalibFun;         {калибровка каналов по энергии}
     FWHMvsChan : TCalibFun;         {калибровка каналов по полуширине}
     Custom     : Pointer;
     {
     выходные данные
     }
 var NumPeaks   : integer;           {число обнаруженных пиков}
     PeakPos    : TDoubleVector;     {0..MaxPeak,положение пика}
     Stat       : TDoubleVector;     {0..MaxPeak,стат. пика}
     Fwhmag     : TDoubleVector;     {0..MaxPeak,ПШПВ пика}
     Amplitude  : TDoubleVector;     {0..MaxPeak,амплитуда пика}
     Fone       : TDoubleVector;     {0..MaxPeak,фон под пиком}
     Fon_Left   : TDoubleVector;     {0..MaxPeak,фон слева от пика}
     Fon_Right  : TDoubleVector;     {0..MaxPeak,фон справа от пика}
     Gran_Left  : TDoubleVector;     {0..MaxPeak,признак границы слева}
     Gran_Right : TDoubleVector;     {0..MaxPeak,признак границы справа}
     {
     временные данные
     }
     Cnt_Peak   : TDoubleVector;     {0..MaxCount-1,массив рассчетного спектра}
     Derive     : TDoubleVector;     {0..MaxPeak,массив производных}
     Gauss      : TDoubleVector;     {0..MaxPeak,временный массив}
     Smooth     : TDoubleVector      {0..Maxcount-1}
     )          : integer;           {возвращает код ошибки}

 {
 коды результата работы PeakIn
 }
const
 seOk              = 0; { все в порядке                      }
 seInvalidInput    = 1; { недопустимые входные данные        }
 seIndexOutOfRange = 2; { выход индекса за границу диапазона }
 seTooManyPeaks    = 3; { переполнение списка пиков          }
 seUnknownError    = 4; { все остальные исключения           }

const
 PeakInErrorMsg    : string = '';

 {
 Параметры поиска пиков
 }
const
 PeakInParams : packed record
  Tinp              : Double;
  FwMin             : Double;
  FwMax             : Double;
  Correct_Deriv     : Double;
  Delta_Peak_Old    : Double;
  Tkf               : Double;
  Tkp               : Double;
  DeltaFon_Gauss    : Double;
  Gauss_Ampl_Border : Double;
  Border_Koef_T     : Double;
 end=(
  Tinp              : 4;
  FwMin             : 0.4;
  FwMax             : 2.5;
  Correct_Deriv     : 1;
  Delta_Peak_Old    : 0.8;
  Tkf               : 0.5;
  Tkp               : 3;
  DeltaFon_Gauss    : 1;
  Gauss_Ampl_Border : 0.2;
  Border_Koef_T     : 0.4
  );
type
 TPeakInParamsArray = packed array[0..(SizeOf(PeakInParams) div SizeOf(Double))-1] of Double;

 {
 Форматка для диалога
 }
const
 PeakInFormat=
  'Tinp;FwMin;FwMax;Correct_Deriv;Delta_Peak_Old;Tkf;Tkp;DeltaFon_Gauss;Gauss_Ampl_Border;Border_Koef_T';

 {
 Фильтр для идентификации малых пиков
 }
const
 NFilt  = 5;
 Sqar   = 10;
 Filtr  : packed array[1..NFilt] of double = (2, 1, 0, -1, -2);

 {
 Константа для вычисления ширины на полувысоте
 }
const
 alfa = 2.76; { alfa=2.77258872223978=4*ln(2), откуда взялась 2.76? }

 {
 Элементарный гауссовский пик
 Amplitude*exp(-alfa*sqr((Channel-Center)/fwhm))+Ground
 sigma = sqtr(Fwhm*Fwhm/2/alfa
 }
function Gaussian(Amplitude,Channel,Center,FWHM,Ground:double):double;

implementation

{$HINTS OFF}
{$WARNINGS OFF}

function Gaussian(Amplitude,Channel,Center,FWHM,Ground:double):double;
begin
 Gaussian:=Amplitude*Exp(-alfa*sqr((Channel-Center)/FWHM))+Ground;
end;

function PeakIn;
 {
 Функция проверяет индекс и генерирует исключение если индекс не в диапазоне
 Применяется для глухой защиты от возможных глупостей
 }
 procedure CheckIndex(Index,MinIndex,MaxIndex:integer);
 begin
  if (Index<MinIndex) or (Index>MaxIndex)
  then raise EPeakInFailure.CreateFmt('Index %d out of range [%d,%d]',
                                      [Index,MinIndex,MaxIndex]);
 end;
 {
 Следующие функции обеспечивают доступ к массивам спектра.
 Это надо по причинам:
  1 Из-за разницы в индексации (внешние массивы индексируются с 0,
    внутренние индексируются с 1), также делается коррекция индексов на
    входе и выходе.
  2 Для доступа к большим массивам (более 64К)
  3 Для глухой защиты от ошибок (чтобы не было выхода за границу массивов)
 Для обращения к элементам массивов используем функции XXX_Put, XXX,
 к примеру: C_Peak_Put(i, C_Peak(i) + 1 );
 }
 function Peak(i:integer):double;
 begin
  CheckIndex(i,1,MaxPeak);
  Peak:=PeakPos[i-1];
 end;
 procedure Peak_Put(i:integer; val:double);
 begin
  CheckIndex(i,1,MaxPeak);
  PeakPos[i-1]:=val;
 end;
 function St(i:integer):double;
 begin
  CheckIndex(i,1,MaxPeak);
  St:=Stat[i-1];
 end;
 procedure St_Put(i:integer; val:double);
 begin
  CheckIndex(i,1,MaxPeak);
  Stat[i-1]:=val;
 end;
 function Fwhm(i:integer):double;
 begin
  CheckIndex(i,1,MaxPeak);
  fwhm:=Fwhmag[i-1];
 end;
 procedure Fwhm_Put(i:integer; val:double);
 begin
  CheckIndex(i,1,MaxPeak);
  Fwhmag[i-1]:=val;
 end;
 function Ampl(i:integer):double;
 begin
  CheckIndex(i,1,MaxPeak);
  Ampl:=Amplitude[i-1];
 end;
 procedure Ampl_Put(i:integer; val:double);
 begin
  CheckIndex(i,1,MaxPeak);
  Amplitude[i-1]:=val;
 end;
 function Fon(i:integer):double;
 begin
  CheckIndex(i,1,MaxPeak);
  Fon:=Fone[i-1];
 end;
 procedure Fon_Put(i:integer; val:double);
 begin
  CheckIndex(i,1,MaxPeak);
  Fone[i-1]:=val;
 end;
 function Fon_L(i:integer):double;
 begin
  CheckIndex(i,1,MaxPeak);
  Fon_L:=Fon_Left[i-1];
 end;
 procedure Fon_L_Put(i:integer; val:double);
 begin
  CheckIndex(i,1,MaxPeak);
  Fon_Left[i-1]:=val;
 end;
 function Fon_R(i:integer):double;
 begin
  CheckIndex(i,1,MaxPeak);
  Fon_R:=Fon_Right[i-1];
 end;
 procedure Fon_R_Put(i:integer; val:double);
 begin
  CheckIndex(i,1,MaxPeak);
  Fon_Right[i-1]:=val;
 end;
 function Deriv(i:integer):double;
 begin
  CheckIndex(i,1,MaxPeak);
  Deriv:=Derive[i-1];
 end;
 procedure Deriv_Put(i:integer; val:double);
 begin
  CheckIndex(i,1,MaxPeak);
  Derive[i-1]:=val;
 end;
 function Gaus(i:integer):double;
 begin
  CheckIndex(i,1,MaxPeak);
  Gaus:=Gauss[i-1];
 end;
 procedure Gaus_Put(i:integer; val:double);
 begin
  CheckIndex(i,1,MaxPeak);
  Gauss[i-1]:=val;
 end;
 function Gran_L(i:integer):double;
 begin
  CheckIndex(i,1,MaxPeak);
  Gran_L:=Gran_Left[i-1];
 end;
 procedure Gran_L_Put(i:integer; val:double);
 begin
  CheckIndex(i,1,MaxPeak);
  Gran_Left[i-1]:=val;
 end;
 function Gran_R(i:integer):double;
 begin
  CheckIndex(i,1,MaxPeak);
  Gran_R:=Gran_Right[i-1];
 end;
 procedure Gran_R_Put(i:integer; val:double);
 begin
  CheckIndex(i,1,MaxPeak);
  Gran_Right[i-1]:=val;
 end;
 function Count(i:integer):Double;
 begin
  CheckIndex(i,1,MaxCount);
  Count:=Spectr[i-1];
 end;
 function C_Peak(i:integer):Double;
 begin
  CheckIndex(i,1,MaxCount);
  C_Peak:=Cnt_Peak[i-1];
 end;
 procedure C_Peak_Put(i:integer; Val:Double);
 begin
  CheckIndex(i,1,MaxCount);
  Cnt_Peak[i-1]:=Val;
 end;
 function C_Smooth(i:integer):Double;
 begin
  CheckIndex(i,1,MaxCount);
  C_Smooth:=Smooth[i-1];
 end;
 procedure C_Smooth_Put(i:integer; Val:Double);
 begin
  CheckIndex(i,1,MaxCount);
  Smooth[i-1]:=Val;
 end;
 {
 }
label
 NextPeak,Quit;
var
 {
 double variables
 }
 Sigma, Sum_Drv, Deriv0, Deriv1, Deriv2, Fwhm_K, Fw_Max, Fw_Min, Fwhm_J, Sum,
 Drv, Drv_Old, Drv_Rez, Peak_J, Max_J, Min_J, Ampl_J, Fon_J,
 Fon_Ampl, Ampl_Theor, CFon_L, CFon_R, CFon, T, S_Gauss, S_Peak, S_Deriv,
 Delta, S_Fon, C1, C2, X1, X2: double;
 {
 integer variables
 }
 Znak, IMin, KMin, IMax, KMax, N_Deriv, NSkan, I_First, I_Last, Nps, I_K,
 intFwhm, Num,  Centr_Peak, iFL, iFR, Interval_Peak_L, I_Stop,
 Interval_Peak_R, Interval_Peak_J, Nlf, Nrf, I_D,
 Int05,  Swich, II1,II2, Jp, Ip, K1p, K2p, I1, I2,
 IK, JJ, iip, Chan_Max, J_Old, NPeak, Mp, J_Left, J_Right,
 Flag_Border_L, Flag_Border_R,  Zona_Change, K_Deriv_L, K_Deriv_R,
 L_H_05, R_H_05, NSmoothing : integer;
 {
 boolean variables
 }
 Flag_L,Flag_R : boolean;
 {
 Сжатие спектра, вычисление производных и дисперсий
 }
 procedure Proizv;
 var Sum_Cor,Sum_Nps,Sum_Nps_Cor,F_N:double; Lp,LLp:integer;
 begin
  Sum:=0;            { Для нескорректированного спектра }
  Sum_Cor:=0;        { Для скорректированного спектра }
  Lp:=1;             { Счетчик точек фильтра, l=NFilt}
  LLp:=1;            { Счетчик точек при сжатии спектра <= Nps }
  Sum_Nps:=0;        { Для нескорректированного спектра }
  Sum_Nps_Cor:=0;    { Для скорректированного спектра }
  while Lp<=NFilt do begin
   iip:=((Lp-1)*Nps+LLp-1)*Znak;
   Num:=Ip+iip;                           { Канал в Count  }
   Sum_Nps:=Sum_Nps+Count(Num);
   { Без коррекции, либо с ней}
   if Nskan=2 then Sum_Nps:=Sum_Nps-PeakInParams.Correct_Deriv*C_Peak(Num);
   Sum_Nps_Cor:=Sum_Nps_Cor+Count(Num)-C_Peak(Num);   { С коррекцией}
   if LLp=Nps then begin
    F_N:=Filtr[Lp]/Nps;
    Sum:=Sum+F_N*Sum_Nps;
    Sum_Cor:=Sum_Cor+F_N*Sum_Nps_Cor;
    Inc(Lp);
    LLp:=1;
    Sum_Nps:=0;
    Sum_Nps_Cor:=0;
   end else inc(LLp);
  end;
  Deriv2:=Sum_Cor;
  Drv:=Sum;
 end;{Proizv}
 {
 Вычисление Max в пике и его центра
 }
 procedure Maxim;
 var S1,S2:double;
 begin
  Peak_J:=(KMin+KMax)*0.5;
  Centr_Peak:=round(Peak_J);
  S1:=Count(Centr_Peak)*Centr_Peak;
  S2:=Count(Centr_Peak);
  Max_J:=Count(Centr_Peak)-C_Peak(Centr_Peak);{Точка с Max счетом}
  Chan_Max:=Centr_Peak;
  Min_J:=Max_J;
  iip:=1;
  {
  Ищем Max в Count на интервале KMin KMax и Min на интервале +- IntFwhm
  }
  while iip<=IntFwhm do begin
   II1:=Centr_Peak-iip*Znak; { В сторону KMin}
   II2:=Centr_Peak+iip*Znak; { В сторону KMax}
   inc(iip);
   if (II1-I_First)*Znak>=0 then begin
    if Count(II1)<Min_J then Min_J:=Count(II1);
    if ((II1-KMin)*Znak>=0) then begin
     if Count(II1)-C_Peak(II1)>Max_J then begin
      Max_J:=Count(II1)-C_Peak(II1);
      Chan_Max:=II1;
     end;
     S1:=S1+Count(II1)*II1;
     S2:=S2+Count(II1);
    end;
   end;
   if (I_Last-II2)*Znak>=0 then begin
    if Count(II2)<Min_J then Min_J:=Count(II2);
    if ((KMax-II2)*Znak>=0) then begin
     if Count(II2)-C_Peak(II2)>Max_J then begin
      Max_J:=Count(II2)-C_Peak(II2);
      Chan_Max:=II2;
     end;
     S1:=S1+Count(II2)*II2;
     S2:=S2+Count(II2);
    end;
   end;
  end; {While}
  Fon_J:=Min_J;                                   { Оценка фона}
  Peak_J:=(Peak_J+Chan_Max)*0.5;
  if (KMax-KMin)*Znak > 2 then Peak_J:=(Peak_J+S1/S2)*0.5;                                    { Центр тяжести}
  Centr_Peak:=round(Peak_J);
  Ampl_J:=Max_J-Fon_J;                            { Оценка амплитуды}
 end; {Maxim}
 {
 Анализ на повторное обнаружение пика, ближе Fwhm_K/2
 Поиск соседей: слева есть - J_Left>0, сосед справа есть - J_Right>0
 Определение интервала-Interval_Peak_J,если дальше, то фон.
 }
 procedure Analiz;
 begin
  J_Left:=0;  { Соседа слева нет }
  J_Right:=0; { Соседа справа нет, если > 0, то это номер в массиве Peak }
  J_Old:=0;             { Пик найден впервые }
  if Znak=1 then begin  { Второе сканирование }
   NPeak:=NumPeaks;      {NPeak - номер правого либо совпадающего пика}
   Jp:=0;
   while Jp=0 do begin
    if NPeak=0 then begin          { Справа пиков нет }
     if NumPeaks>0 then J_Left:=1; { Сосед только слева }
     Jp:=1;
    end else begin { Соседние пики есть }
     if (Peak_J -Peak(NPeak)) > Fwhm_K*PeakInParams.Delta_Peak_Old then begin
      { и левый далеко }
      dec(NPeak);
      continue;
     end;
     Jp:=1;                     { Справа или рядом слева }
     if abs(Peak_J-Peak(NPeak))<Fwhm_K*PeakInParams.Delta_Peak_Old  then begin
      { Есть пик рядом, т.е. пик  обнаружен повторно }
      J_Old:=NPeak ;
      Peak_J:=Peak(NPeak);
      Centr_Peak:=round(Peak_J);
      if Fon_J>Fon(Npeak) then Fon_J:=Fon(Npeak);
      if Ampl_J<Ampl(Npeak) then Ampl_J:=Ampl(Npeak);
      if NPeak>1 then J_Right:=NPeak-1;{ и он справа }
     end else J_Right:=NPeak;     { Сосед далеко справа }
     if NPeak<>NumPeaks then J_Left:=NPeak+1; { Сосед слева }
    end;
   end; {while}
  end else begin              { Первое сканирование }
   if Jp>1 then begin
    J_Right:=Jp-1;
    {Повторное обнаружение}
    if (Peak(Jp-1)-Peak_J)<Fwhm_K*PeakInParams.Delta_Peak_Old then begin
     J_Old:=Jp-1;
     Peak_J:=0.5*(Peak_J+Peak(J_Old));
     Centr_Peak:=round(Peak_J);
     if Fon_J>Fon(J_Old) then Fon_J:=Fon(J_Old);
     if Ampl_J<Ampl(J_Old) then Ampl_J:=Ampl(J_Old);
     Fwhm_J:=(Fwhm_J+Fwhm(J_Old))*0.5+abs(Peak_J-Peak(J_Old));
     if J_Old>1 then J_Right:=J_Old-1;
    end;
   end;
  end;
  if J_Old=0 then begin
   if (J_Left>0) and (Peak_J-Peak(J_Left)<2*Fwhm_K) and (Fon(J_Left)<Fon_J)
   then Fon_J:=Fon(J_Left);
   if (J_Right>0) and (Peak(J_Right)-Peak_J<2*Fwhm_K) and (Fon(J_Right)<Fon_J)
   then Fon_J:=Fon(J_Right);
   if (J_Left*J_RIght>0) and (Peak_J-Peak(J_Left)<2*Fwhm_K) and
      (Peak(J_Right)-Peak_J<2*Fwhm_K) and (Fon(J_Left)<Fon(J_Right)) and
      (Fon(J_Left)<Fon_J)
   then Fon_J:=Fon(J_Left);
   Ampl_J:=Max_J-Fon_J;
  end;
  {Интервал, занимаемый пиком Interval_Peak_J}
  Fon_Ampl:=sqrt(Fon_J)*PeakInParams.DeltaFon_Gauss/Ampl_J;
  if Fon_Ampl>0.5 then Fon_Ampl:=0.5;
  Interval_Peak_J:=round(Fwhm_K*sqrt(-Ln(Fon_Ampl)/alfa));
  if Interval_Peak_J<Int05 then Interval_Peak_J:=Int05;
 end;{Analiz}
 {
 Поиск границы зоны аппроксимации, Flag_Border =
  1 фоновый участок >= IntFwhm и канал, отстоящий от пика > Interwal_Peak_j
  2 фон < IntWfhm, либо сосед близко, но канал отстоит от пика > Interwal_Peak_j
  3 граница есть канал на наибольшем нефлуктуирующем участке
  4 граница  есть  канал с минимальным счетом,
  5 признаков фона нет - сомнение в наличии пика.
 }
 procedure Border(
             { Входные данные }
             L_R         : integer; { Признак границы -1 левая  1 правая}
             J_LR        : integer; { Номер соседнего пика в данных}
             NLR         : integer; { Оценка числа фоновых точек}
             K_Deriv     : integer; { Канал экстремума производной}
             { Выходные данные }
      var    H_05        : integer;
      var    K_Border    : integer;
      var    Flag_Border : integer;
      var    CFon        : double
                 );
 var
  T_Border,Test, Test_Old, C_Average, C_Min, C_Min_Old: double;
  N_Fon, Ip, K_Min, N_Fluct, N_Fon_Old, No_Bound,
  Control_Int, N_Test: integer;
  Flag_Peak, Flag_05, Flag_Write : boolean;
 begin
  {
  Зона была разбита
  Фон из результата аппроксимации, если таковой не было, то ранее найденный
  }
  if Zona_Change=1 then exit;
  {
  Ищем значимую флуктуацию
  }
  T_border:=PeakInParams.Tinp*PeakInParams.Border_Koef_T;
  Ip:=K1p-L_R;
  C_Min:=Count(K_Deriv);
  K_Min:=K_Deriv;
  N_Fon:=0;     {Счетчик фоновых точек}
  N_Fon_old:=0;     {Число точек на наибольшем фоновом интервале}
  N_Fluct:=0;     {Счетчик точек флуктуаций}
  K_Border:=K_Min;
  H_05:=K_Min;
  Flag_Border:=5;
  Control_Int:=Nps*2; {Длина фонового интервала}
  Flag_Peak:=false;
  Flag_05:=false;       { Флаг половины высоты пика}
  Flag_Write:=false;       { Флаг записи границы по 3 либо вход в цикл}
  No_Bound:=abs(K_Deriv-Centr_Peak);      { Нельзя ставить границу}
  if No_Bound<PeakInParams.Delta_Peak_Old
  then No_Bound:=trunc(PeakInParams.Delta_Peak_Old*Fwhm_K/2);
  { Внутри центр +- Control_Int}
  Test_Old:=0;
  Test:=0;
  while L_R*Ip<=L_R*K2p do begin
   {
   C_Smooth(Ip)^:=abs((Count(Ip-2)^-8*Count(Ip-1)^-Count(Ip+2)^+8*Count(Ip+1)^))/
                  sqrt(64*(Count(Ip-2)^+Count(Ip+2)^)+Count(Ip-1)^+Count(Ip+1)^);
   }
   Ip:=Ip+L_R;
   if abs(Test)>T_Border then Test_Old:=Test;
   Test:=(-2*Count(Ip-2)-Count(Ip-1)+2*Count(Ip+2)+Count(Ip+1))/
             sqrt(4*(Count(Ip-2)+Count(Ip+2))+Count(Ip-1)+Count(Ip+1));
   if Flag_Write then begin   { Новый поиск}
    C_Min:=Count(Ip);
    K_Min:=Ip;
    C1:=(Count(IP-1)+Count(IP+1))/2;
    if (abs(Count(IP-1)-Count(IP+1))<3*sqrt(C1)) and (C1-C_Min>3*sqrt(C1))
    then C_Min:=C1; {Т.е. был выброс}
    Flag_Write:=false;
   end else begin
    if (Count(Ip) > 0) and (Count(Ip)<C_Min) then begin
     C_Min:=Count(Ip);
     K_Min:=Ip;
     C1:=(Count(IP-1)+Count(IP+1))/2;
     if (abs(Count(IP-1)-Count(IP+1))<3*sqrt(C1)) and (C1-C_Min>3*sqrt(C1))
     then C_Min:=C1; {Т.е. был выброс}
    end;
   end;
   if not Flag_05 then begin
    if Count(Ip)<Ampl_J*0.5+Fon_J+C_Peak(Ip) then begin
     Flag_05:=true;
     H_05:=Ip;
    end else { Влияет соседний пик? }
    if (abs(k1p-Ip)>Int05) and (Count(Ip)>Max_J) then Flag_05:=true;
   end;
   if abs(Ip-Centr_Peak)<=No_Bound then begin
    if C_Smooth(Ip)<T_Border then begin     { Исключаем случайную флуктуацию}
     Inc(N_Fon);
     if (N_Fon=abs(Ip-K1p)+1) and (N_Fon=No_Bound)
     then Flag_Peak:=true; { Сомнение  в наличии пика}
    end;
    if abs(Ip-Centr_Peak)=No_Bound then N_Fon:=0;
    continue;
   end;
   if abs(Test)<T_Border then begin  { Флуктуаций нет}
    N_Fluct:=0;              { Счетчик числа каналов в  флуктуации}
    Inc(N_Fon);
    if (N_Fon>=IntFwhm) and (L_R*Ip>L_R*(Centr_Peak+L_R*Interval_Peak_J))
    then begin
     K_Border:=Ip;
     CFon:=C_Average;
     Flag_Border:=1;
     NLR:=N_Fon;
     break;                { Граница найдена}
    end;
    if N_Fon=1 then C_Average:=Count(Ip);
    C_Average:=(C_Average+Count(Ip))*0.5;
   end else begin                           { Есть  флуктуации}
    Inc(N_Fluct);                          { Длина флуктуации}
    if (N_Fluct = 0) or (Test*Test_Old<=0)
    then N_Test:=1 {Интервал, монотонного поведения}
    else Inc(N_Test);
    if N_Fon>0 then begin
     if N_Fon>N_Fon_old then  begin
      { Перед этим была последняя точка в самом  длинном интервале}
      K_Border:=Ip-L_R;
      N_Fon_Old:=N_Fon;
      NLR:=N_Fon;
      CFon:=C_Min-C_Peak(K_Min);
      if (CFon<0) or (C_Peak(K_Min)<0) then CFon:=C_Min;
      Flag_Border:=4;
      C_Min_Old:=C_Min;
      if N_Fon>Control_Int then begin
       Flag_Border:=3;
       Flag_Write:=true;  {Разрешаем новый поиск минимума C_Min}
       if (L_R*Ip>L_R*(Centr_Peak+L_R*Interval_Peak_J)) and
          (C_Peak(Ip)>C_Peak(Ip-L_R)) and{на возрастающем участке}
          (Test>Test_Old)
       then begin
        CFon:=(C_Average+C_Min)/2;
        Flag_Border:=2;
        N_Fon:=0;
        break;
       end;
      end;
     end else begin      { N_Fon <= N_Fon_Old}
      if C_Min<C_Min_Old then begin
       { Перед этим была последняя точка в равном, но с меньшим счетом интервале}
       K_Border:=K_Min;
       C_Min_Old:=C_Min;
       CFon:=C_Min-C_Peak(K_Min);
       if (CFon<0) or (C_Peak(K_Min)<0) then CFon:=C_Min;
      end;
     end;
    end;
    if (abs(Test)>2*T_Border) and (C_Smooth(Ip)>C_Smooth(Ip-L_R))
    then break;
    if (N_Fon_Old>0) and (abs(Centr_Peak-Ip)>Interval_Peak_J) and
       (N_Fluct>1) and (C_Smooth(Ip)>C_Smooth(Ip-L_R))
    then break;
    if (N_Fon_Old>0) and (N_Test>NPS) and (C_Smooth(Ip)>C_Smooth(Ip-L_R))
    then break; {Т.е. производная изменила знак}
    {
    Предел поиска границы
    }
    {Диагностируем нефизичный переход в спектре}
    if (Count(Ip)=0) and (Count(Ip-L_R)>10) then break;
    N_Fon:=0;
   end; { if флутуаций}
   if (C_Peak(Ip)>C_Peak(Ip-L_R)) and
      (C_Peak(Ip)/Ampl_J>PeakInParams.Gauss_Ampl_Border)
   then break;
  end;{ while}
  if N_Fon>0 then begin   { Вышли на фоновом интервале}
   if Flag_Border>3 then begin
    CFon:=C_Min-C_Peak(K_Min);
    if (CFon<0) or (C_Peak(K_Min)<0) then CFon:=C_Min;
    K_Border:=K_Min;
    Flag_Border:=4;
    NLR:=N_Fon;
   end;
   if NLR>0 then begin
    Flag_Border:=3;
    if N_Fon>N_Fon_Old then begin
     K_Border:=Ip;
     CFon:=C_Min-C_Peak(K_Min);
     if (CFon<0) or (C_Peak(K_Min)<0) then CFon:=C_Min;
     if L_R*Ip>L_R*(Centr_Peak+L_R*Interval_Peak_J) then begin
      CFon:=C_Average;
      Flag_Border:=2;
     end;
    end else begin
     if (N_Fon_Old>0) and (C_Min<CFon) then begin
      K_Border:=Ip;
      CFon:=C_Min-C_Peak(K_Min);
     end;
    end;
   end;
  end;
  if Flag_Peak then Flag_Border:=-Flag_Border;
  if (Flag_Border=3) and (NLR>0) and (J_LR=0) then begin
   CFon:=(CFon+2*(Count(K_Border)-C_Peak(K_Border))+Count(K_Border-L_R)-
          C_Peak(K_Border-L_R))/4;
   if C_Peak(K_Border-L_R)<0
   then CFon:=(CFon+2*Count(K_Border)+Count(K_Border-L_R))/4;
  end;
  if (J_LR=0) and (Flag_Border=5) then begin
   CFon:=C_Min-C_Peak(K_Min);
   if (CFon<0) or (C_Peak(K_Min)<0) then CFon:=C_Min;
   K_Border:=K_Min;
  end;
  if (Flag_Border>2) and (C_Peak(K_Border)/Ampl_J<3*Count(K_Border))
  then begin
   CFon:=(CFon+2*(Count(K_Border)-C_Peak(K_Border))+
          Count(K_Border-L_R)-C_Peak(K_Border-L_R))/4;
   if C_Peak(K_Border)<0
   then CFon:=(CFon+2*Count(K_Border)+Count(K_Border-L_R))/4;
  end;
  if CFon<0 then CFon:=C_Min;
  if CFon<1 then CFon:=1;
 end; { Border}
 {
 Фон слева от пика.
 }
 procedure Fon_Left_Side;
 var Kp_Fon:integer;
 begin
  Flag_L:=false;        { Пик далеко он границы}
  { Предел левой границы интрвала для поиска фона}
  K2p:=Centr_Peak-Interval_Peak_J-3*IntFwhm;
  Kp_Fon:=K2p+IntFwhm;
  if K2p<=NStart then begin
   Flag_L:=true;       { Пик близко к границе}
   K2p:=NStart+2;
   Kp_Fon:=K2p;        { Левая граница фонового интервала}
  end;
  if J_Left>0 then begin
   Fon_Ampl:=sqrt(Fon(J_Left))*PeakInParams.DeltaFon_Gauss/Ampl(J_Left);
   if Fon_Ampl>0.5 then Fon_Ampl:=0.5;
   Interval_Peak_L:=round(Fwhm_K*sqrt(-Ln(Fon_Ampl)/alfa));
   if Interval_Peak_L<IntFwhm then Interval_Peak_L:=IntFwhm;
   { Interval_Peak_L - интервал, где значим пик слева }
   K2p:=round(Peak(J_Left))+Int05;
   Kp_Fon:=K2p; { Левая граница фонового интервала}
   if round(Peak_J-Peak(J_Left))>Interval_Peak_J+Interval_Peak_L+IntFwhm*2
   then J_Left:=0;{ т.к. пик достаточно далеко}
  end;
  K1p:=Centr_Peak-1;
  {
  Поиск границы и фона
  }
  Nlf:=Centr_Peak-Interval_Peak_J-Kp_Fon;
  Zona_Change:=0;
  CFon_L:=Fon_J;
  Border (-1, J_Left, Nlf, K_Deriv_L, L_H_05, IFL, Flag_Border_L, CFon_L);
  if (J_Left>0) and (Flag_Border_L>3) and (abs(Fon_R(J_Left))<CFon_L)
  then  CFon_L:=abs(Fon_R(J_Left));
  if J_Old>0 then begin
   if abs(Fon_L(J_Old))<CFon_L then CFon_L:=abs(Fon_L (J_Old));
   if (J_Left=0) and (abs(round(Gran_L(J_Old)))<IFL)
   then IFL:=abs(round(Gran_L(J_Old)));
  end;
 end;{Fon_Left_Side}
 {
 Фон справа от пика.
 }
 procedure Fon_Right_Side;
 var Kp_Fon:integer;
 begin
  Flag_R:=false;           { Пик далеко он границы}
  { Предел правой границы интервала поиска фона }
  K2p:=Centr_Peak+Interval_Peak_J+3*IntFwhm;
  if K2p>=Nend then begin
   Flag_R:=true;          { Пик близко к границе}
   K2p:=Nend;
  end;
  Kp_Fon:=K2p;             { Правая граница фонового участка}
  { Ищем правую границу интервала поиска фона}
  if J_Right>0 then begin
   Fon_Ampl:=sqrt(Fon(J_Right))*PeakInParams.DeltaFon_Gauss/Ampl(J_Right);
   if Fon_Ampl>0.5 then Fon_Ampl:=0.5;
   Interval_Peak_R:=round(Fwhm_K*sqrt(-Ln(Fon_Ampl)/alfa));
   if Interval_Peak_R<IntFwhm then Interval_Peak_R:=IntFwhm;
   K2p:=round(Peak(J_Right))-Int05;
   Kp_Fon:=K2p; { Правая граница фонового участка}
   if round(Peak(J_Right)-Peak_J)>Interval_Peak_J+Interval_Peak_R+IntFwhm*2
   then J_Right:=0; { т.е. пик достаточно далеко}
  end;
  if (Nskan=2) and (J_Old>0) then begin { Берем старые значения}
   CFon_R:=Fon_R(J_Old);
   IFR:=round(abs(Gran_R(J_Old)));
  end else begin { Вычисляем }
   K1p:=Centr_Peak+1;
   Nrf:=Kp_Fon-Centr_Peak-Interval_Peak_J;
   Zona_Change:=0;
   CFon_R:=Fon_J;
   Border (1, J_Right, Nrf, K_Deriv_R, R_H_05, IFR, Flag_Border_R, CFon_R);
   if (J_Right>0) and (Flag_Border_R>3) and (abs(Fon_L(J_Right))<CFon_R)
   then CFon_R:=abs(Fon_L(J_Right));
  end;
 end;{Fon_Right_Side}
 {
 Фон для пика
 }
 procedure Fon_for_Peak;
 begin
  if Nskan=1 then begin
   CFon:=CFon_R;
   if Flag_Border_L=1
   then CFon:=(CFon_L+CFon_R)*0.5 { Слева фоновый интервал}
   else if CFon_L<CFon_R then begin
    CFon:=CFon_L;
    if Flag_Border_R<3 then CFon:=(CFon_l+CFon_R)*0.5;
   end;
  end else begin { Nskan = 2}
   CFon:=CFon_L;
   if Flag_Border_L<3 then begin  { Слева хороший фоновый интервал}
    if (Flag_Border_R<3) or (CFon_R<CFon_L) then CFon:=(CFon_l+CFon_R)*0.5;
   end else begin{ Flag_Border_L > 2  Слева не хороший фоновый интервал}
    if Flag_Border_R = 1 then begin
     if CFon_L<CFon_R then CFon:=(CFon_L+CFon_R)*0.5 else CFon:=CFon_R;
    end else begin { Нет фоновых интервалов}
     if (C_Peak(IFL)/CFon_L<0.1) and (CFon_L<CFon_R) then CFon:=CFon_L;
     if (C_Peak(IFR)/CFon_R<0.1) and (CFon_R<CFon_L) then CFon:=CFon_R;
     if (C_Peak(IFL)/CFon_L<0.1) and (C_Peak(IFR)/CFon_R<0.1)
     then CFon:=(CFon_L+CFon_R)*0.5;
    end;
   end;
   if (J_Old>0) and (Fon(J_Old)<CFon) then CFon:=Fon(J_Old);
  end;
  if CFon < 1 then CFon:=1;
 end;{Fon_For_Peak}
 {
 Корректируем  амплитуду центр и Fwhm_J.
 }
 procedure Correction(var Centr_Peak:integer; var Ampl_J,Fwhm_J:double);
 var
  Sigma_P, Sigma_L_P, Sigma_L_M, Sigma_R_P, Sigma_R_M,
  Delta_L_P, Delta_L_M, Delta_R_P, Delta_R_M, Gauss_L_P, Gauss_L_M,
  Gauss_R_P, Gauss_R_M, Centr_L_M, Centr_L_P, Centr_R_P, Centr_R_M, Delta_A,
  S_Gauss, S_Peak, L_M, L_P, R_M, R_P, Delta_Peak, Delta_Peak_Old, Delta_Fwhm, Delta_Fw,
  Delta_Y, sqrt1, sqrt2, hi_new, hi_t, hi_old, Threshold,
  Number_Sigma: double;
  Minus_L, Minus_R, IG, JJ, Ip, II1, II2, IG1, IG2, N_L, N_R : integer;
  Flag_0, Flag_L, Flag_R : boolean;
 begin
  IG:=(MaxPeak-1) div 2 + 1;      {Рассотреть размер???}
  JJ:=0;                            {Счетчик корректировок}
  Delta_Peak:=0;
  if Fwhm_J<Fwhm_K then Fwhm_J:=Fwhm_K;
  while JJ<9  do begin
   Inc(JJ);
   Minus_L:=0;    { Количество точек слева и справа превышения}
   Minus_R:=0;    {          расчета над экспериментом}
   Sigma_P:=0;    { Дефект отсчетов  в пике)}
   Sigma_L_P:=0;    { Дефект отсчетов + (по левому склону)}
   Sigma_L_M:=0;    { Дефект отсчетов - (по левому склону)}
   Sigma_R_P:=0;    { Дефект отсчетов + (по левому склону)}
   Sigma_R_M:=0;    { Дефект отсчетов - (по левому склону)}
   Centr_L_M:=0;    { Смещение центра (момент)}
   Centr_L_P:=0;
   Centr_R_M:=0;
   Centr_R_P:=0;
   Delta_L_P:=0;    { Дефект площади + (по левому склону)}
   Delta_L_M:=0;    { Дефект площади - (по левому склону)}
   Delta_R_P:=0;    { Дефект площади + (по правому склону)}
   Delta_R_M:=0;    { Дефект площади - (по правому склону)}
   Gauss_L_P:=0;    { Доля пика от sqrt(2*пи)*А*сигма)}
   Gauss_L_M:=0;    { Доля пика от sqrt(2*пи)*А*сигма)}
   Gauss_R_P:=0;    { Доля пика от sqrt(2*пи)*А*сигма)}
   Gauss_R_M:=0;    { Доля пика от sqrt(2*пи)*А*сигма)}
   Delta_A:=0;    { Деффект амплитуды в центре}
   hi_new:=0;    { хи-квадрат}
   Number_Sigma:=1;    { Значимое отклонение в сигмах от расчета}
   N_L:=0;    { Счетчик каналов, где Delta_Y > 0}
   N_R:=0;
   hi_old:=hi_new;
   Delta_Peak_Old:=Delta_Peak; {Расхождение в центрe}
   Flag_0:=false;
   Gaus_Put(IG, Gaussian(1,Centr_Peak,Peak_J,Fwhm_J,0));
   C1:=Count(Centr_Peak)-CFon-C_Peak(Centr_Peak);
   Delta_Y:=C1-Ampl_j*Gaus(IG);
   if Delta_Y < 0  then  Flag_0:=true;
   C1:=Count(Chan_Max)-CFon-C_Peak(Centr_Peak);
   S_Gauss:=Ampl_J*Gaus(IG);   { Доля площади гауссиана}
   S_Peak:=C1;
   Sigma_P:=C1/sqrt(Count(Chan_Max));
   Delta_Y:=C1-Ampl_j*Gaus(IG);
   hi_new:=Delta_Y*Delta_Y/Count(Chan_Max);
   if Delta_Y < 0  then Delta_A:=Delta_Y;{Деффект амплитуды}
   Flag_L:=false;
   Flag_R:=false;
   Ip:=0;
   while Ip<=NPS*4 do begin       {Поиск расхождений }
    Inc(Ip);
    II1:=Centr_Peak-Ip;
    II2:=Centr_Peak+Ip;
    IG1:=IG-Ip;
    IG2:=IG+Ip;                {Расхождение на левом склоне}
    Gaus_Put(IG1, Gaussian(1,II1,Peak_J,Fwhm_J,0));
    Sqrt1:=sqrt(Count(II1));
    if sqrt1 < 1 then sqrt1:=1;
    if (2*Ip+1 > Fwhm_K) and (Ampl_J*Gaus(IG1) < sqrt1) then  begin
     Ip:=Ip-1;
     break;
    end;
    Delta_Y:=Count(II1)-CFon-C_Peak(II1);
    if (Ip-1)*2+2 <= Fwhm_k then begin
     S_Gauss:=S_Gauss+Ampl_J*Gaus(IG1);   { Доля площади гауссиана}
     S_Peak:=S_Peak+Delta_Y;
    end;
    Delta_Y:=Delta_Y-Ampl_j*Gaus(IG1);
    hi_new:=hi_new+Delta_Y*Delta_Y/Count(II1);
    if Delta_Y < 0 then begin      {Расхождение есть -}
     Inc(Minus_L);
     Gauss_L_M:=Gauss_L_M+Gaus(IG1);   { Доля площади гауссиана}
     Delta_L_M:=Delta_L_M+Delta_Y;
     Sigma_L_M:=Sigma_L_M+Delta_Y/sqrt1;
     if (-Delta_Y/sqrt1) >= Number_Sigma then Flag_L:=true;
     Centr_L_M:=Centr_L_M-Delta_Y*(2*Peak_J-II1)/sqrt1;
     {Момет: сила-Delta_Y/sqrt1; плечо 2*Peak_J-II1, т.е. по другую сторону}
    end else begin                 {Расхождение есть +}
     Inc(N_L);
     Sigma_P:=Sigma_P+Delta_Y/sqrt1;
     Gauss_L_P:=Gauss_L_P+Gaus(IG1);   { Доля площади гауссиана}
     Delta_L_P:=Delta_L_P+Delta_Y;
     Sigma_L_P:=Sigma_L_P+Delta_Y/sqrt1;
     Centr_L_P:=Centr_L_P+Delta_Y*II1/sqrt1;
    end; {Расхождение на правом склоне }
    Gaus_Put(IG2, Gaussian(1,II2,Peak_J,Fwhm_J,0));
    Delta_Y:=Count(II2)-CFon-C_Peak(II2);
    if Ip*2+1 <= Fwhm_k then begin
     S_Gauss:=S_Gauss+Ampl_J*Gaus(IG2);   { Доля площади гауссиана}
     S_Peak:=S_Peak+Delta_Y;
    end;
    sqrt2:=sqrt(Count(II2));
    if sqrt2 < 1 then sqrt2:=1;
    if Delta_Y > 0 then Sigma_P:=Sigma_P+Delta_Y/sqrt2;
    Delta_Y:=Delta_Y-Ampl_j*Gaus(IG2);
    hi_new:=hi_new+Delta_Y*Delta_Y/Count(II2);
    if Delta_Y < 0 then begin  {Расхождение есть -}
     Inc(Minus_R);
     Gauss_R_M:=Gauss_R_M+Gaus(IG2);   { Доля площади гауссиана}
     Delta_R_M:=Delta_R_M+Delta_Y;
     Sigma_R_M:=Sigma_R_M+Delta_Y/sqrt2;
     if (-Delta_Y/sqrt2) >= Number_Sigma then Flag_R:=true;
     Centr_R_M:=Centr_R_M-Delta_Y*(2*Peak_J-II2)/sqrt2;
    end else begin             {Расхождение есть +}
     Inc(N_R);
     Sigma_P:=Sigma_P+Delta_Y/sqrt2;
     Gauss_R_P:=Gauss_R_P+Gaus(IG2);   { Доля площади гауссиана}
     Delta_R_P:=Delta_R_P+Delta_Y;
     Sigma_R_P:=Sigma_R_P+Delta_Y/sqrt2;
     Centr_R_P:=Centr_R_P+Delta_Y*II2/sqrt2;
    end;
   end;   { While ip}
   {If (JJ > 1) and (hi_old-hi_new < 0) then break; }
   {
   Корректируем амплитуду по центру
   }
   if (-Delta_A > sqrt(Count(Centr_Peak))) and
      (Sigma_L_P+Sigma_R_P  < -Sigma_L_M-Sigma_R_M)
   then begin
    Ampl_J:=Ampl_j+Delta_A;
    if Ampl_J < PeakInParams.Tkf*PeakInParams.Tinp*
       sqrt(CFon+C_Peak(Centr_Peak))
    then begin
     Ampl_J:=-1;
     Exit;
    end;
    continue;
   end;
   {
   Корректируем Fwhm_J  и центр
   }
   L_M:=0;
   L_P:=0;
   R_M:=0;
   R_P:=0;
   Delta_Peak:=0;
   Delta_Fwhm:=0;
   if Minus_L > 0 then begin
    L_M:=Sigma_L_M/Minus_L;
    L_P:=Sigma_L_P/Minus_L;
   end;
   if Minus_R > 0 then begin
    R_M:=Sigma_R_M/Minus_R;
    R_P:=Sigma_R_P/Minus_R;
   end;
   Threshold:=-0.3;
   Delta_Fwhm:=0;
   if ((JJ > 1) and (hi_old > hi_new)) or {Идет согласование}
      (L_M < Threshold) or (R_M < Threshold) or Flag_L or Flag_R
   then begin
    {
    деффект Fwhm - из соотношения: относительный деффект площади равен
          относительному деффекту Fwhm
    }
    if J_Left+J_Right = 0 then begin
     if L_P+R_P = 0
     then Delta_Fwhm:=(Delta_L_M+Delta_R_M)*sqrt(alfa/3.141)/Ampl_j
     else Delta_Fwhm:=(Delta_L_M+Delta_R_M-Threshold)*sqrt(alfa/3.141)/Ampl_j;
     if Delta_Fwhm > 0 then  Delta_Fwhm:=0;
    end else begin   { есть соседи}
     if J_Left > 0 then begin
      if (R_M  < Threshold) or Flag_R or (R_P = 0)
      then Delta_Fwhm:=Delta_R_M *sqrt(alfa/3.141)/Ampl_j;
      if (L_M  < Threshold) or Flag_L or (L_P = 0) then begin
       if Ampl_J > Ampl(J_Left)
       then Delta_Fwhm:=Delta_Fwhm+Delta_L_M *sqrt(alfa/3.141)/Ampl_j
       else begin
        Delta_Fw:=Delta_L_M*sqrt(alfa/3.141)/Ampl(J_Left);
        Fwhm_Put(J_Left, Fwhm(J_Left)+Delta_Fw);
        if Fwhm(J_Left) < Fwhm_K then Fwhm_Put(J_Left, Fwhm_K);
       end;
      end;
     end;
     if J_Right > 0 then begin
      if (L_M  < Threshold) or Flag_L or (L_P = 0)
      then Delta_Fwhm:=Delta_L_M *sqrt(alfa/3.141)/Ampl_j;
      if (R_M  < Threshold) or Flag_R or (R_P = 0) then begin
       if Ampl_J >Ampl(J_Right)
       then Delta_Fwhm:=Delta_Fwhm+Delta_L_M *sqrt(alfa/3.141)/Ampl_j
       else begin  {соседний пик больше}
        Delta_Fw:=Delta_R_M*sqrt(alfa/3.141)/Ampl(J_Right);
        Fwhm_Put(J_Right, Fwhm(J_Right)+Delta_Fw);
        if Fwhm(J_Right) < Fwhm_K then  Fwhm_Put(J_Right, Fwhm_K);
       end;
      end;
     end;
    end;
    {
    Смещаем центр. Центр пика - как центр тяжести всей известной
    площади (в старом центре) и отрицательного деффекта
    }
    if NSkan = 1 then begin
     if (L_M < Threshold) and (R_P > -Threshold)
     then Delta_Peak:=((Sigma_P-Sigma_R_P)*Peak_J+Centr_L_M+Centr_R_M)/
                      ( Sigma_P-Sigma_R_P- Sigma_L_M-Sigma_R_M)-Peak_J;
     if (R_M < Threshold) and (L_M = 0) and (L_P > -Threshold)
     then Delta_Peak:=(Sigma_P*Peak_J+Centr_R_M)/(Sigma_P-Sigma_R_M)-Peak_J;
    end else begin {Nskan = 2}
     if ((L_M > Threshold) and (R_M < Threshold)) or (R_M < -1)
     then Delta_Peak:=((Sigma_P-Sigma_L_P+Sigma_R_P)*Peak_J+Centr_R_M)/
                      ( Sigma_P-Sigma_L_P+Sigma_R_P-Sigma_R_M)-Peak_J;
     if ((L_M < Threshold) and (R_M > Threshold)) or (L_M < -1)
     then Delta_Peak:=((Sigma_P-Sigma_R_P+Sigma_L_P)*Peak_J+Centr_L_M)/
                      ( Sigma_P-Sigma_R_P+Sigma_L_P-Sigma_L_M)-Peak_J;
    end;
    Peak_J:=Peak_J+Delta_Peak;
    Centr_Peak:=round(Peak_J);
    Fwhm_J:=Fwhm_J+Delta_Fwhm;
    if (JJ > 1) and (Minus_L+Minus_R > NPS) and
       (Flag_L or Flag_R or Flag_0 or (N_L = 0) or (N_R = 0)) and
       (abs(Delta_Fwhm) < 0.1*Fwhm_J) and
       (Delta_Peak < 0.1)
    then begin
     Ampl_J:=Ampl_j-Number_Sigma*sqrt(Ampl_J);
     if Ampl_J < 3*sqrt(CFon+C_Peak(Centr_Peak)) then begin
      Ampl_J:=-1;
      Exit;
     end;
     continue;
    end;
    if Fwhm_J < Fw_Min then Exit;
   end; {if случай наличия деффекта}
   if ((abs(Delta_Peak) < 0.1) or (Delta_Peak*Delta_Peak_Old < 0)) and
      (abs(Delta_Fwhm) < 0.1*Fwhm_J)
   then break;
   Ip:=Ip-3;
   hi_t:=3.3+(1.518+(-0.0060406+0.000030577*Ip)*Ip)*Ip;
   if (JJ > 1) and (hi_new < hi_t) and (abs(Delta_Fwhm) < 0.1*Fwhm_J)
   then break;
  end; {While JJ}
  if Fwhm_J < Fw_Min then Exit;
  if S_Peak < 1 then S_Peak:=1;
  if (Ampl_J< 3*sqrt(CFon+C_Peak(Centr_Peak))) or ((t<PeakInParams.TkF*PeakInParams.Tinp) and (
     (
     S_Peak < PeakInParams.TkF*
               PeakInParams.Tinp*sqrt((CFon+C_Peak(Centr_Peak))*Fwhm_K)
     )
     or
     (
     S_Peak < PeakInParams.TkF*sqrt(3.14/alfa)*Ampl_J*Fwhm_K
     )  ))
     or
     (
     Fwhm_J < PeakInParams.TkF*Fwhm_K
     )
     or
     (
     (J_Left  > 0) and (abs(Peak_J -Peak(J_Left)) < Fwhm_K*
                                            PeakInParams.Delta_Peak_Old)
     )
     or
     (
     (J_Right > 0) and (abs(Peak_J -Peak(J_Right)) < Fwhm_K*
                                            PeakInParams.Delta_Peak_Old)
     )
  then begin
   Ampl_J:=-1;
   Exit;
  end;
  I_D:=trunc(Fwhm_K*sqrt(Ln(Ampl_J/0.1*sqrt(0.1*CFon))/alfa));
  if 2*I_D > (MaxPeak-1) then I_D:=trunc((MaxPeak-1)/2);
  if I_D < 2*IntFwhm then I_D:=2*IntFwhm;
  if Centr_Peak-NStart < I_D then I_D:=Centr_Peak-NStart;
  if NEnd-Centr_Peak   < I_D then I_D:=NEnd-Centr_Peak;
  {
  Расчетный спектр в окрестности iip, где амплитуда*EXP > sqrt(Fon)-C_Peak
  }
  C_Peak_Put(Centr_Peak,C_Peak(Centr_Peak)+Ampl_J*Gaus(IG));
  iip:=1;
  while iip<=I_D do begin      { Вычисление гауссианов на флангах}
   II1:=Centr_Peak-iip;
   II2:=Centr_Peak+iip;
   IG1:=IG-iip;
   IG2:=IG+iip;
   Gaus_Put(IG1, Gaussian(1,II1,Peak_J,Fwhm_J,0));
   Gaus_Put(IG2, Gaussian(1,II2,Peak_J,Fwhm_J,0));
   C_Peak_Put(II1, C_Peak(II1)+Ampl_J*Gaus(IG1));
   C_Peak_Put(II2, C_Peak(II2)+Ampl_J*Gaus(IG2));
   if (Ampl_J*Gaus(IG1)<1 ) or (Ampl_J*Gaus(IG2)<1) then break;
   Inc(iip);
  end;
 end; {Correction}
 {
 Начало основной процедуры
 }
begin { *************** Peakin *************************************** }
 {
 Проверяем входные данные на грубые глупости
 }
 PeakIn:=seInvalidInput;
 PeakInErrorMsg:='PeakIn: Invalid input data.';
 if (MaxCount>0) and
    (MaxPeak>0) and
    (Nstart>0) and
    (Nend<MaxCount) and
    (Nstart<Nend) and
    Assigned(Spectr) and
    Assigned(EvsChan) and
    Assigned(FWHMvsChan) and
    Assigned(PeakPos) and
    Assigned(Stat) and
    Assigned(Fwhmag) and
    Assigned(Amplitude) and
    Assigned(Fone) and
    Assigned(Fon_Left) and
    Assigned(Fon_Right) and
    Assigned(Gran_Left) and
    Assigned(Gran_Right) and
    Assigned(Cnt_Peak) and
    Assigned(Derive) and
    Assigned(Gauss) and
    Assigned(Smooth)
 then
 {
 Поиск пиков в критической секции с проверкой индексов гарантирует безопасность
 при обращении к массивам
 }
 try
  PeakIn:=seOk;
  PeakInErrorMsg:='';
  Flag_L:=false; Flag_R:=false;
  FakeNOP(Flag_L and Flag_R); // To supress compiler hints.
  {
  индексация массивов внутри процедуры начинается с 1, а на входе с 0!
  }
  inc(NStart);
  inc(NEnd);
  {
  Зануление расчетного спектра
  }
  for Ip:=Nstart to NEnd do begin
   C_Peak_Put(Ip,0);
   C_Smooth_Put(Ip,Count(Ip));
  end;
  { Спектр сглаженных вторых производных, отнесенных к дисперсиям}
  Fwhm_K:=FWHMvsChan(NStart,Custom);
  NSmoothing:=trunc(Fwhm_K);
  if NSmoothing<1 then NSmoothing:=1;
  for Ip:=1 to NSmoothing do begin
   X1:=C_Smooth(NStart);
   C_Smooth_Put(NStart, (2*C_Smooth(NStart)+C_Smooth(NStart+1))/3);
   for iip:=Nstart+1 to NEnd-1 do begin
    X2:=C_Smooth(iip);
    C_Smooth_Put(iip, (X1+2*C_Smooth(iip)+C_Smooth(iip+1))/4);
    X1:=X2;
   end;
   C_Smooth_Put(NEnd, (2*C_Smooth(NEnd)+X1)/3);
  end;
  {
  Первый проход по спектру
  }
  NSkan:=1;
  {
  Сканирование с конца спектра
  }
  Znak:=-1;
  NumPeaks:=0;
  I_First:=Nend-2;
  I_Last:=NStart+2;
  while Nskan<=2 do begin       { Определим границу цикла по спектру:  I_Stop}
   Fwhm_K:=FWHMvsChan(I_Last-1,Custom);{индексация внутри - с 1, снаружи - с 0}
   Nps:=trunc(Fwhm_K/NFilt);
   I_Stop:=I_Last-((NFilt-1)*Nps+Nps)*Znak;
   {
   Счетчик найденных пиков
   }
   Jp:=1;
   Ip:=I_First-Znak;       {Индекс в спектре}
   {
   Расстояние от пика до границы фона
   }
   {
   Swich - ключ при вычислении производных:
     -2 первая точка спектра либо после пика,
      0 начало пика (min производная),
      1 конец пика  (max производная)
      2 пик найден.
   }
   Swich:=-2;
   while Ip<>I_Stop do begin
    Ip:=Ip+Znak;
    if Ip=I_Stop then break;
    Deriv1:=Deriv2;
    {
    Следующая точка Count , определение Nps и калибровочной FWHM
    }
    Fwhm_K:=FWHMvsChan(Ip-1,Custom);{индексация внутри - с 1, снаружи - с 0}
    Int05:=round(Fwhm_K*0.5);
    IntFwhm:=round(Fwhm_K);
    if Fwhm_J>Fwhm_K then intFwhm:=round(Fwhm_J);
    if (Nskan=2) and (PeakInParams.Correct_Deriv=1) and (J_Old>0) then begin
     Ip:=round(Peak(J_Old))+round(PeakInParams.Delta_Peak_Old*Fwhm_K);
     Swich:=-2;
     while J_Old>1 do begin
      dec(J_Old);
      if Ip>round(Peak(J_Old))-Int05
      then Ip:=round(Peak(J_Old))+round(PeakInParams.Delta_Peak_Old*Fwhm_K)
      else break;
     end;
     if Ip>=I_Stop then break;
     J_Old:=0;
    end;
    Nps:=trunc(Fwhm_K/NFilt);
    if Nps<1 then Nps:=1;
    I_K:=NFilt*Nps div 2 * Znak ;
    {
    Сжатие, вычисление производных и дисперсий
    }
    Proizv;    { В Deriv2 -скоррректированная, в Drv в зависимости от ключа}
    {
    Массив Count  просмотрен?
    }
    {
    Первая точка?
    }
    if Swich<0 then goto NextPeak;
    {
    Поиск Min производной, т.е. начала пика
    }
    if Swich=0 then begin
     if Drv<=Drv_Old then begin          { Нет начала }
      Drv_Old:=Drv;
      Mp:=0;
      continue;
     end else begin                      { Начало пика найдено }
      Swich:=1;
      IMin:=Ip-Znak ;
      KMin:=IMin+I_K;
      Deriv_Put(1, Deriv0);                { За одну до минимума }
      Deriv_Put(2, Deriv1);
      Deriv_Put(3, Deriv2);
      N_Deriv:=3;                          {  Число запомненных производных }
      Drv_Old:=Drv;
      Mp:=0;
      continue;
     end;
    end;
    {
    Поиск Max производной, т.е. конца пика
    }
    if Swich>0 then begin
     inc(N_Deriv);
     if N_Deriv>MaxPeak  then begin
      PeakIn:=seTooManyPeaks;
      PeakInErrorMsg:='PeakIn: Too many peaks found.';
      goto Quit;
     end;                                { Выход за пределы массива}
     Deriv_Put(N_Deriv, Deriv2);         { Запоминание производных }
     if Drv>=Drv_Old then begin          { Нет максимума }
      Drv_Old:=Drv;
      Mp:=0;
      continue;
     end;
     if Mp=0 then Drv_Rez:=Drv;          { Это Drv_Old для следующего пика }
     inc(Mp);                            { Возможен Max, начало контроля }
     if Mp<Nps then continue;            { защита от шума}
    end;
    {  Найден максимум }
    IMax:=Ip-Mp*Znak ;
    KMax:=IMax+I_K;
    K_Deriv_L:=KMax;
    K_Deriv_R:=KMin;
    if Nskan=2 then begin
     K_Deriv_L:=KMin;
     K_Deriv_R:=KMax;
    end;
    L_H_05:=K_Deriv_L;
    R_H_05:=K_Deriv_R;
    {
    Подготовка к поиску следующего пика
    }
    N_Deriv:=N_Deriv-Mp+1;
    Deriv1:=Deriv(N_Deriv-1);
    Deriv2:=Deriv(N_Deriv);
    Drv:=Drv_Rez;
    {
    Контроль параметров пика
    Контроль Kmax- KMin с целью расширить, т.к. в случае малого   пика
    на склоне большого эта величина будет меньше сигмы
    }
    K1p:=KMax;
    K2p:=KMin;
    if (K1p-K2p)*Znak<=Nps then begin  {Возможно, близко пик  }
     if Count(KMax)<Count(KMax+Znak) then K1p:=K1p+Znak;
     if Count(KMin)<Count(KMin-Znak) then K2p:=K2p-Znak;
    end;
    Fw_Max:=PeakInParams.FwMax*Fwhm_K;
    Fw_Min:=PeakInParams.FwMin*Fwhm_K;
    Fwhm_J:=(K1p-K2p)*Znak*1.18;      {Т.к. ПШПВ=2.354*сигма}
    {
    Критерий отбора пиков по ПШПВ
    }
    if ((PeakInParams.FwMax>0) and (Fwhm_J>Fw_Max))       { Очень широкий }
    {or ((PeakInParams.FwMin > 0) and (Fwhm_J < Fw_Min))} { или очень узкий}
      or (Count(KMin)*Count(KMax)=0)                      { или выпал канал }
    then goto NextPeak;
    if Fwhm_J<Fwhm_K then Fwhm_J:=Fwhm_k;
    {
    Вычисление Max в пике и его центра
    }
    Maxim;
    {
    Анализ на повторное обнаружение пика, ближе Fwhm_K/2
    Сосед слева есть : J_Left>0,Сосед справа : J_Right>0
    }
    Analiz;
    {
    Оценка амплитуды  и фона
    Фон как среднее в Count -C_Peak  на фоновом инт-ле (Fwhm_K);
    если инт-ла нет, то по минимуму в Count   и ср. по
    трем точкам около минимума в Count -C_Peak(CFon)^
    }
    {
    Фон слева от пика.
    }
    Fon_Left_Side;
    if (Nskan=1) and (J_Old>0) then goto NextPeak;
    {
    Фон справа от пика.
    }
    if Nskan=2 then begin
     if J_Old > 0 then begin {взяты старые значения и декодируем Flag_Border_}
      Flag_Border_L:=5;
      if Gran_L(J_Old)>0 then Flag_Border_L:=3;
      if Fon_L(J_Old)>0 then Flag_Border_L:=2;
      if (Gran_L(J_Old)>0) and (Fon_L(J_Old)>0) then Flag_Border_L:=1;
      Flag_Border_R:=5;
      CFon_R:=abs(Fon_R(J_Old));
      IFR:=abs(round(Gran_R(J_Old)));
      if Gran_R(J_Old)>0 then Flag_Border_R:=3;
      if Fon_R (J_Old)>0 then Flag_Border_R:=2;
      if (Gran_R(J_Old)>0) and (Fon_R(J_Old)>0) then Flag_Border_R:=1;
     end else Fon_Right_Side;
    end else Fon_Right_Side;
    if (Flag_Border_L<0) and (Flag_Border_R<0) then goto NextPeak;
    Flag_Border_L:=abs(Flag_Border_L);
    Flag_Border_R:=abs(Flag_Border_R);
    if (J_Old=0) and (PeakInParams.FwMax>0) and (R_H_05-L_H_05>round(Fw_Max))
    then goto NextPeak;
    {
    Фон для пика
    }
    Fon_for_Peak;
    if J_Old>0 then Ampl_J:=Ampl(J_Old)-CFon+Fon(J_Old);
    if J_Old=0 then begin      { т.е. только для впервые найденных пиков}
     Ampl_J:=Max_J-CFon;
     if (Nskan=1) and (Ampl_J>10*sqrt(CFon)) then begin
      I1:=Centr_Peak-Nps;
      I2:=Centr_Peak+Nps;
      S_Peak:=0;
      S_Gauss:=0;
      for iip:=I1 to I2 do begin
       S_Peak:=S_Peak+Count(iip)-CFon;
       if C_Peak(IIP)>0 then S_Peak:=S_Peak-C_Peak(IIP);
       S_Gauss:=S_Gauss+Gaussian(1,iip,Peak_J,Fwhm_K,0);
      end;
      if Ampl_J<S_Peak/S_Gauss then Ampl_J:=S_Peak/S_Gauss;
     end;
     Ampl_Theor:=(Count(Centr_Peak-Nps)-C_Peak(Centr_Peak-Nps)+
                  Count(Centr_Peak+Nps)-C_Peak(Centr_Peak+Nps)-CFon*2)*0.87;
     if Ampl_J+CFon>Count(Centr_Peak) then Ampl_J:=Ampl_J-sqrt(Ampl_j);
     if Ampl_j<PeakInParams.Tkf*PeakInParams.Tinp*sqrt(CFon)
     then goto NextPeak;
     if Ampl_Theor<0 then goto NextPeak;
     {
     Это расчетная амплитуда по точкам расположенным от центра на FWHM/5,
     если Ampl_Theor существенно меньше Ampl_J, то это выброс, т.е.
     это проверка на гауссовую форму пика
     }
     {
     Уточняем влияние соседних пиков
     }
     if J_Left>0 then begin
      Fon_Ampl:=sqrt(Fon(J_Left))*PeakInParams.DeltaFon_Gauss/Ampl(J_Left);
      if Fon_Ampl>0.5 then Fon_Ampl:=0.5;
      Interval_Peak_L:=round(Fwhm_K*sqrt(-Ln(Fon_Ampl)/alfa));
      if Peak_J-Peak(J_Left)>Interval_Peak_L+Interval_Peak_J+Fwhm_K
      then J_Left:=0;
     end;
     if J_Right>0 then begin
      Fon_Ampl:=sqrt(Fon(J_Right))*PeakInParams.DeltaFon_Gauss/Ampl(J_Right);
      if Fon_Ampl>0.5 then Fon_Ampl:=0.5;
      Interval_Peak_R:=round(Fwhm_K*sqrt(-Ln(Fon_Ampl)/alfa));
      if Peak(J_Right)-Peak_J>Interval_Peak_R+Interval_Peak_J+Fwhm_K
      then J_Right:=0;
     end;
     { Отличаем от ступеньки}
     C1:=0.5*(Count(K_Deriv_L)+Count(K_Deriv_L-1)-
              C_Peak(K_Deriv_L)-C_Peak(K_Deriv_L-1))-CFon;
     C2:=0.5*(Count(K_Deriv_R)+Count(K_Deriv_R+1)-
              C_Peak(K_Deriv_R)-C_Peak(K_Deriv_L+1))-CFon;
     if (J_Left=0) and (Ampl_J-C1<2*sqrt(Max_J)) and
        (Max_J-C_Peak(Centr_Peak)-Count(IFL)+C_Peak(IFL)<2*sqrt(Max_J))
     then goto NextPeak;
     if (J_Right=0) and (Ampl_J-C2<2*sqrt(Max_J)) and
        (Max_J-C_Peak(Centr_Peak)-Count(IFR)+C_Peak(IFR)<2*sqrt(Max_J))
     then goto NextPeak;
     {
     Ищем соотношение счета на склонах пика к его амплитуде
     }
     X1:=C1/Ampl_J;
     X2:=C2/Ampl_J;
     S_Peak:=0;
     S_Deriv:=0;
     I1:=Centr_Peak-intFwhm;
     I2:=Centr_Peak+intFwhm;
     if I1 < NStart then I1:=NStart;
     if I2 > NEnd   then I2:=NEnd;
     for iip:=I1 to I2 do begin
      S_Peak:=S_Peak+Count(iip)-C_Peak(iip);
      if (iip>=K_Deriv_L) and (IIP<=K_Deriv_R)
      then S_Deriv:=S_Deriv+Count(iip)-C_Peak(iip);
     end;
     S_Deriv:=S_Deriv-CFon*(K_Deriv_R-K_Deriv_R+1);
     S_Fon:=CFon*Fwhm_J;
     S_Peak:=S_Peak-S_Fon;
     S_Fon:=sqrt(S_Fon);
     {
     Критерий St'udenta
     }
     IK:=trunc(N_Deriv/2);
     Sum_Drv:=0;
     Sigma:=sqrt(2*Sqar*CFon);
     for iip:=1 to IK do begin
      Delta:=Deriv(N_Deriv-iip+1)-Deriv(iip);
      if Delta>0 then Sum_Drv:=Sum_Drv+Delta;
     end;
     T:=Sum_Drv/Sigma;
     {
     Исключаем пики, если площадь < Tkf*Tinp*sqrt(S_fon)
     либо амплитуда < Tkf*Tinp*sqrt(CFon)
     }
     if T<PeakInParams.Tkf*PeakInParams.Tinp then goto NextPeak;
     if (Nskan=1) and (Ampl_J>Ampl_Theor) and
        ((Ampl_Theor < PeakInParams.Tkf*PeakInParams.Tinp*sqrt(CFon)) or
        (S_Peak < PeakInParams.Tkf*PeakInParams.Tinp*S_Fon) or
        (S_Peak-S_Deriv < sqrt(CFon*(I2-I1-K_Deriv_R+K_Deriv_R))))
     then goto NextPeak;
     if (S_Peak-Ampl_J*Nps)<S_Fon then goto NextPeak;
     if Nskan=1 then begin     { Первое сканирование}
      { Выброс? }
      if (Fwhm_J<Fwhm_K) and ((X1<0.2) or (X2<0.2)) then goto NextPeak;
      if (J_Right=0) and ((X1/X2>5) or (X2/X1>5) or (T<PeakInParams.Tinp))
      then goto NextPeak;
     end else begin            {Только при втором проходе}
      {
      Исключаются ложные выбросы и ступеньки
      }
      { Выброс? }
      if (Fwhm_J<PeakInParams.Tkf*Fwhm_K) and ((X1<0.1) or (X2<0.1))
      then goto NextPeak;
      { Ступенька? }
      if (J_Left+J_Right=0) and ((X1>2) or (X2>2) or (X1/X2>5) or (X2/X1>5))
      then goto NextPeak;
      { Выброс? }
      if (T<PeakInParams.Tinp) and
         ((Fwhm_J<0.7*Fwhm_K) and (X1<0.2) and (X2 < 0.2))
      then goto NextPeak;
      C1:=0.5*(Count(KMin)+Count(KMin-Znak))-CFon-C_Peak(KMin);
      C2:=0.5*(Count(KMax)+Count(KMax+Znak))-CFon-C_Peak(KMax);
      X1:=C1/Ampl_J;
      X2:=C2/Ampl_J;
      if (X1>1) and (X2>1) then goto NextPeak;
      if (J_Left=0) and (Count(IFL)>Count(Centr_Peak)) and
         (J_Right=0) and (Count(IFR)>Count(Centr_Peak))
      then goto NextPeak;
      if (T<PeakInParams.Tinp) and ((Ampl_J<PeakInParams.Tinp*sqrt(CFon)) or
           (S_Peak<PeakInParams.Tinp*S_Fon))
      then goto NextPeak;
      if (T< PeakInParams.TkP*PeakInParams.Tinp)
         and (Ampl_J < PeakInParams.Tkf*PeakInParams.Tinp*sqrt(CFon))
         and (S_Peak < PeakInParams.TkF*PeakInParams.Tinp*S_Fon)
         and (Fwhm_J < 0.7*Fwhm_K)
      then goto NextPeak;
     end; {if Nskan}
     { Пик найден }
     Correction(Centr_Peak,AMPL_J,Fwhm_J);
     if Ampl_J<PeakInParams.Tkf*PeakInParams.Tinp*sqrt(CFon)
     then goto NextPeak;
     if Fwhm_J<Fw_Min then goto NextPeak;
    end; { J_Old}
    {
    Кодируется информация о фоне с учетом соседних:
    Нет соседа или Flag_Border_L (R) = 1   то CFon_L (R) > 0 и IfL (R) > 0
    Есть сосед и:  Flag_Border_L (R) = 1,2 то CFon_L (R) > 0 и IfL (R) < 0;
                   Flag_Border_L (R) = 3   то CFon_L (R) < 0 и IfL (R) > 0;
                   Flag_Border_L (R) = 4,5 то CFon_L (R) < 0 и IfL (R) < 0;
    Уточняем влияние соседних пиков
    }
    if J_Left>0 then begin
     Fon_Ampl:=sqrt(Fon(J_Left))*PeakInParams.DeltaFon_Gauss/Ampl(J_Left);
     if Fon_Ampl>0.5 then Fon_Ampl:=0.5;
     Interval_Peak_L:=round(Fwhm_K*sqrt(-Ln(Fon_Ampl)/alfa));
     if Peak_J-Peak(J_Left)>Interval_Peak_L+Interval_Peak_J+Fwhm_K
     then J_Left:=0;
    end;
    if J_Right>0 then begin
     Fon_Ampl:=sqrt(Fon(J_Right))*PeakInParams.DeltaFon_Gauss/Ampl(J_Right);
     if Fon_Ampl>0.5 then Fon_Ampl:=0.5;
     Interval_Peak_R:=round(Fwhm_K*sqrt(-Ln(Fon_Ampl)/alfa));
     if Peak(J_Right)-Peak_J>Interval_Peak_R+Interval_Peak_J+Fwhm_K
     then J_Right:=0;
    end;
    { Приводим знаки в соответствие с указанной выше таблицей}
    if J_Left>0 then begin
     if IFL<abs(Gran_R(J_Left)) then begin {Перехлест границ, доверие новым}
      Gran_R_Put(J_Left, IFL);             {Знак установится ниже}
      if abs(Fon_R(J_Left))>CFon_L then Fon_R_Put(J_Left, CFon_L);
     end;
     if Flag_Border_L<>3 then begin
      IFL:=-IFL;
      Gran_R_Put(J_Left, -abs(Gran_R(J_Left)));
     end;
     if Flag_Border_L > 2 then begin
      CFon_L:=-CFon_L;
      Fon_R_Put(J_Left,-abs(Fon_R(J_Left)));
     end;
     {Проверяем соответствие знаков}
     IIP:=round(abs(Gran_R(J_Left)));
     C1:=C_Peak(iip)+C_Peak(abs(IFL)); {Вклад от пиков в фоновой точке}
     if (C1/Ampl(J_Left)>PeakInParams.Gauss_Ampl_Border) or
        (C1/Ampl_J>PeakInParams.Gauss_Ampl_Border)
     then begin {Запрещаем второй вариант кодировки фона}
      CFon_L:=-abs(CFon_L);
      Fon_R_Put(J_Left, -abs(Fon_R(J_Left)));
     end else begin {Разрешаем второй вариант кодировки фона}
      CFon_L:=abs(CFon_L);
      Fon_R_Put(J_Left, abs(Fon_R(J_Left)));
     end;
    end else begin
     IFL:=abs(IFL);
     CFon_L:=abs(CFon_L);
    end;
    if J_Right>0 then begin
     if IFR>abs(Gran_L(J_Right)) then begin {Перехлест границ}
      Gran_L_Put(J_Right, IFR);             {Знак установится ниже}
      if abs(Fon_L(J_Right))>CFon_R then Fon_L_Put(J_Right, CFon_R);
     end;
     if Flag_Border_R<>3 then begin
      IFR:=-IFR;
      Gran_L_Put(J_Right, -abs(Gran_L(J_Right)));
     end;
     if Flag_Border_R>2 then begin
      CFon_R:=-CFon_R;
      Fon_L_Put(J_Right, -abs(Fon_L(J_Right)));
     end;
     {Проверяем соответствие знаков}
     IIP:=round(abs(Gran_L(J_Right)));
     C1:=C_Peak(iip)+C_Peak(abs(IFR)); {Вклад от пиков в фоновой точке}
     if (C1/Ampl(J_Right)>PeakInParams.Gauss_Ampl_Border) or
        (C1/Ampl_J>PeakInParams.Gauss_Ampl_Border)
     then begin     {Запрещаем второй вариант кодировки фона}
      CFon_R:=-abs(CFon_R);
      Fon_L_Put(J_Right, -abs(Fon_L(J_Right)));
     end else begin {Разрешаем второй вариант кодировки фона}
      CFon_R:=abs(CFon_R);
      Fon_L_Put(J_Right, abs(Fon_L(J_Right)));
     end;
    end else begin
     IFR:=abs(IFR);
     CFon_R:=abs(CFon_R);
    end;
    {
    Заполнение выходных массивов
    }
    if J_Old>0 then begin             { Обнаружен повторно }
     JJ:=J_Old;
    end else begin                    { Peak_J обнаружен впервые }
     if (Nskan=1) then begin          { Первый проход}
      JJ:=Jp;
      inc(Jp);
     end else begin                  { Второй проход}
      if NumPeaks+1>MaxPeak then begin
       PeakIn:=seTooManyPeaks;
       PeakInErrorMsg:='PeakIn: Too many peaks found.';
       goto Quit;
      end;
      JJ:=NPeak+1;                   { Место куда будет вставлен пик}
      inc(NumPeaks);
      if JJ<>NumPeaks then begin
       iip:=NumPeaks;
       repeat                        { Готовим место в NPeak+1}
        dec(iip);
        Peak_Put(iip+1, Peak(iip));
        Fon_Put(iip+1, Fon(iip));
        Ampl_Put(iip+1, Ampl(iip));
        Fwhm_Put(iip+1, Fwhm(iip));
        ST_Put(iip+1, ST(iip));
        Gran_L_Put(iip+1, Gran_L(iip));
        Gran_R_Put(iip+1, Gran_R(iip));
        Fon_L_Put(iip+1, Fon_L(iip));
        Fon_R_Put(iip+1, Fon_R(iip));
       until iip=NPeak+1;
      end;
     end; { Nskan}
    end; {J_Old}
    if J_Old=0 then begin   { Для впервые обнаруженных пиков}
     ST_Put(JJ, T);
     Peak_Put(jj, Peak_J);
     Fwhm_Put(jj, Fwhm_J);
    end;
    Ampl_Put(jj, Ampl_J);
    Fon_Put(JJ, CFon);
    Gran_L_Put(jj, iFL);
    Fon_L_Put(jj, CFon_L);
    Gran_R_Put(jj, iFR);
    Fon_R_Put(jj, CFon_R);
    if (PeakInParams.Correct_Deriv>0) and (J_Old=0) then Swich:=2;
    if Nskan=1 then inc(NumPeaks);
    {
    переполнение?
    }
    if NumPeaks = MaxPeak then begin
     PeakIn:=seTooManyPeaks;
     PeakInErrorMsg:='PeakIn: Too many peaks found.';
     goto Quit;
    end;
    {
    Переход к следующему пику
    }
 NextPeak:
    if Swich<0 then Swich:=Swich+1;
    if Swich=2 then begin
     Swich:=-2;
     if PeakInParams.Correct_Deriv>0 then Ip:=Ip-Znak*2;
     { Возвращаемся для восстановления производных}
    end;
    if Swich=1 then Swich:=0;
    Drv_Old:=Drv; { Предыдущее значение для поиска Min, Max }
    Mp:=0;        { Счет точек зоны, где должен сохраняться MAX }
   end; {While (Ip)  конец цикла по спектру}
   {
   Массив Count просмотрeн :
    Закончено первое сканирование    --> Подготовка ко второму
    Закончено последнее сканирование --> Выход
   }
   if Nskan=1 then begin
    NumPeaks:=Jp-1;
    Nskan:=2;
    Znak:=1;
    I_First:=NStart+1;
    I_Last:=Nend-1;
   end else break;
  end; {While (Nskan<=2)}
 Quit:
  {
  Модуль поправки на фон и амплитуду:
  за истимый фон принимается Gran[ip] > 0 и Fon[ip] > 0
  }
  jp:=1;
  JJ:=0;
  for ip:=1 to NumPeaks do begin
   if (Gran_L(ip)<=0) or (Fon_L(ip)<=0) then continue;
   x1:=(Fon_R(jp)-Fon_L(ip))/(Gran_R(jp)-Gran_L(ip));
   for iip:=jp to ip do begin
    x2:=Fon_L(ip)+x1*(Peak(iip)-Gran_L(ip)); {Новый фон iip-го пика}
    Ampl_j:=Ampl(iip)+Fon(iip)-x2;
    if Ampl_J<0 then Ampl_J:=1;
    if Ampl_J<0 then JJ:=JJ+1 else begin
     Ampl_Put(iip-JJ, Ampl_J);
     Fon_Put(iip-JJ, x2);
     ST_Put(iip-JJ, ST(iip));
     Peak_Put(iip-JJ, Peak(iip));
     Fwhm_Put(iip-JJ, Fwhm(iip));
     Gran_L_Put(iip-JJ, Gran_L(iip));
     Fon_L_Put(iip-JJ, Fon_L(iip));
     Gran_R_Put(iip-JJ, Gran_R(iip));
     Fon_R_Put(iip-JJ, Fon_R(iip));
    end;
   end;
   jp:=ip+1;
  end;
  NumPeaks:=NumPeaks-JJ;
  {
  завершение : учтем, что пики на выходе должны индексироваться с 0
  а не с 1 как внутри процедуры
  }
  for iip:=1 to NumPeaks do begin
   Peak_Put(iip, Peak(iip)-1);
   if Gran_L(iip) > 0 then Gran_L_Put(iip, Gran_L(iip)-1);
   if Gran_L(iip) < 0 then Gran_L_Put(iip, Gran_L(iip)+1);
   if Gran_R(iip) > 0 then Gran_R_Put(iip, Gran_R(iip)-1);
   if Gran_R(iip) < 0 then Gran_R_Put(iip, Gran_R(iip)+1);
  end;
 except
  on E: Exception do begin
   if E is EMatrixFailure then PeakIn:=seIndexOutOfRange else
   if E is EPeakInFailure then PeakIn:=seIndexOutOfRange else
   PeakIn:=seUnknownError;
   PeakInErrorMsg:='PeakIn: '+E.Message;
  end;
 end;
end; { ******************** PeakIn ********************************* }

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_peaksearch;
begin
 PeakInErrorMsg:='';
end;

procedure Free_crw_peaksearch;
begin
 PeakInErrorMsg:='';
end;

initialization

 Init_crw_peaksearch;

finalization

 Free_crw_peaksearch;

end.

//////////////
// END OF FILE
//////////////

